/*	$NetBSD: rk3588_cru.c,v 1.3 2024/02/07 04:20:27 msaitoh Exp $	*/

/*-
 * Copyright (c) 2022 Ryo Shimizu
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: rk3588_cru.c,v 1.3 2024/02/07 04:20:27 msaitoh Exp $");

#include <sys/param.h>
#include <sys/device.h>

#include <dev/fdt/fdtvar.h>

#include <arm/rockchip/rk_cru.h>
#include <arm/rockchip/rk3588_cru.h>

#define PLL_CON(base, n)	(0x0000 + (base) + (n) * 4)
#define MODE_CON(base, n)	(0x0280 + (base) + (n) * 4)
#define CLKSEL_CON(base, n)	(0x0300 + (base) + (n) * 4)
#define CLKGATE_CON(base, n)	(0x0800 + (base) + (n) * 4)
#define SOFTRST_CON(base, n)	(0x0a00 + (base) + (n) * 4)
/* base of above *_CON() macro */
#define PHP			0x00008000
#define PMU			0x00030000
#define BIGCORE0		0x00050000
#define BIGCORE1		0x00052000
#define DSU			0x00058000

#define RK3588_PHYREF_ALT_GATE	0x0c38

static int rk3588_cru_match(device_t, cfdata_t, void *);
static void rk3588_cru_attach(device_t, device_t, void *);

static const struct device_compatible_entry compat_data[] = {
	{ .compat = "rockchip,rk3588-cru" },
	DEVICE_COMPAT_EOL
};

CFATTACH_DECL_NEW(rk3588_cru, sizeof(struct rk_cru_softc),
    rk3588_cru_match, rk3588_cru_attach, NULL, NULL);

#define RK3588_CLK_CORE_L_SEL_MASK	__BITS(6,5)
#define RK3588_CLK_DSU_SEL_DF_MASK	__BIT(15)
#define RK3588_CLK_DSU_DF_SRC_MASK	__BITS(6,5)
#define RK3588_CLK_DSU_DF_DIV_MASK	__BITS(4,0)
#define RK3588_ACLKM_DSU_DIV_MASK	__BITS(5,1)
#define RK3588_ACLKS_DSU_DIV_MASK	__BITS(10,6)
#define RK3588_ACLKMP_DSU_DIV_MASK	__BITS(15,11)
#define RK3588_PERIPH_DSU_DIV_MASK	__BITS(4,0)
#define RK3588_ATCLK_DSU_DIV_MASK	__BITS(4,0)
#define RK3588_GICCLK_DSU_DIV_MASK	__BITS(9,5)

#define RK3588_CORE_L_SEL_CORE(regoff, apllcore)			\
	{								\
		.reg = CLKSEL_CON(DSU, 6 + (regoff)),			\
		.mask = RK3588_CLK_CORE_L_SEL_MASK,			\
		.val = __SHIFTIN((apllcore), RK3588_CLK_CORE_L_SEL_MASK)\
	}

#define RK3588_CORE_L_SEL_DSU(seldsu, divdsu)				\
	{								\
		.reg = CLKSEL_CON(DSU, 0),				\
		.mask =							\
		    RK3588_CLK_DSU_DF_SRC_MASK |			\
		    RK3588_CLK_DSU_DF_DIV_MASK |			\
		    RK3588_CLK_DSU_SEL_DF_MASK,				\
		.val =							\
		    __SHIFTIN((seldsu), RK3588_CLK_DSU_DF_SRC_MASK) |	\
		    __SHIFTIN((divdsu) - 1, RK3588_CLK_DSU_DF_DIV_MASK) |\
		    __SHIFTIN(0, RK3588_CLK_DSU_SEL_DF_MASK)		\
	}

#define RK3588_CORE_L_SEL_ACLKS(aclkm, aclkmp, aclks)			\
	{								\
		.reg = CLKSEL_CON(DSU, 1),				\
		.mask =							\
		    RK3588_ACLKM_DSU_DIV_MASK |				\
		    RK3588_ACLKMP_DSU_DIV_MASK |			\
		    RK3588_ACLKS_DSU_DIV_MASK,				\
		.val =							\
		    __SHIFTIN((aclkm) - 1, RK3588_ACLKM_DSU_DIV_MASK) |	\
		    __SHIFTIN((aclkmp) - 1, RK3588_ACLKMP_DSU_DIV_MASK)|\
		    __SHIFTIN((aclks) - 1, RK3588_ACLKS_DSU_DIV_MASK)	\
	}

#define RK3588_CORE_L_SEL_PERI(periph)					\
	{								\
		.reg = CLKSEL_CON(DSU, 2),				\
		.mask = RK3588_PERIPH_DSU_DIV_MASK,			\
		.val = __SHIFTIN((periph) - 1, RK3588_PERIPH_DSU_DIV_MASK)\
	}

#define RK3588_CORE_L_SEL_GIC_ATCLK(gicclk, atclk)			\
	{								\
		.reg = CLKSEL_CON(DSU, 3),				\
		.mask =							\
		    RK3588_GICCLK_DSU_DIV_MASK |			\
		    RK3588_ATCLK_DSU_DIV_MASK,				\
		.val =							\
		    __SHIFTIN((gicclk) - 1, RK3588_GICCLK_DSU_DIV_MASK) |\
		    __SHIFTIN((atclk) - 1, RK3588_ATCLK_DSU_DIV_MASK)	\
	}

#define RK3588_ARMCLK_L_RATE(targetrate, apllcore, seldsu, divdsu,	\
    atclk, gicclk, aclkmp, aclkm, aclks, periph)			\
	{								\
		.rate = (targetrate),					\
		.divs = {						\
			RK3588_CORE_L_SEL_DSU((seldsu), (divdsu)),	\
			RK3588_CORE_L_SEL_ACLKS((aclkm), (aclkmp), (aclks)),\
			RK3588_CORE_L_SEL_PERI((periph)),		\
			RK3588_CORE_L_SEL_GIC_ATCLK((gicclk), (atclk)),	\
		},							\
		.pre_muxs = {						\
			RK3588_CORE_L_SEL_CORE(0, 0),			\
			RK3588_CORE_L_SEL_CORE(1, 0),			\
			RK3588_CORE_L_SEL_DSU(3, 2),			\
		},							\
		.post_muxs = {						\
			RK3588_CORE_L_SEL_CORE(0, (apllcore)),		\
			RK3588_CORE_L_SEL_CORE(1, (apllcore)),		\
			RK3588_CORE_L_SEL_DSU((seldsu), (divdsu))	\
		},							\
	}

static const struct rk_cru_cpu_rate armclk_l_rates[] = {
	RK3588_ARMCLK_L_RATE(2208000000, 1, 3, 1, 4, 4, 4, 4, 4, 4),
	RK3588_ARMCLK_L_RATE(2184000000, 1, 3, 1, 4, 4, 4, 4, 4, 4),
	RK3588_ARMCLK_L_RATE(2088000000, 1, 3, 1, 4, 4, 4, 4, 4, 4),
	RK3588_ARMCLK_L_RATE(2040000000, 1, 3, 1, 4, 4, 4, 4, 4, 4),
	RK3588_ARMCLK_L_RATE(2016000000, 1, 3, 1, 4, 4, 4, 4, 4, 4),
	RK3588_ARMCLK_L_RATE(1992000000, 1, 3, 1, 4, 4, 4, 4, 4, 4),
	RK3588_ARMCLK_L_RATE(1896000000, 1, 3, 1, 4, 4, 4, 4, 4, 4),
	RK3588_ARMCLK_L_RATE(1800000000, 1, 3, 1, 4, 4, 4, 4, 4, 4),
	RK3588_ARMCLK_L_RATE(1704000000, 0, 3, 1, 3, 3, 3, 3, 3, 3),
	RK3588_ARMCLK_L_RATE(1608000000, 0, 3, 1, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1584000000, 0, 3, 1, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1560000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1536000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1512000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1488000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1464000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1440000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1416000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1392000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1368000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1344000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1320000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1296000000, 0, 2, 2, 3, 3, 3, 2, 3, 3),
	RK3588_ARMCLK_L_RATE(1272000000, 0, 2, 2, 2, 2, 2, 2, 2, 2),
	RK3588_ARMCLK_L_RATE(1248000000, 0, 2, 2, 2, 2, 2, 2, 2, 2),
	RK3588_ARMCLK_L_RATE(1224000000, 0, 2, 2, 2, 2, 2, 2, 2, 2),
	RK3588_ARMCLK_L_RATE(1200000000, 0, 2, 2, 2, 2, 2, 2, 2, 2),
	RK3588_ARMCLK_L_RATE(1104000000, 0, 2, 2, 2, 2, 2, 2, 2, 2),
	RK3588_ARMCLK_L_RATE(1008000000, 0, 2, 2, 2, 2, 2, 2, 2, 2),
	RK3588_ARMCLK_L_RATE( 912000000, 0, 2, 2, 2, 2, 2, 1, 2, 2),
	RK3588_ARMCLK_L_RATE( 816000000, 0, 2, 2, 2, 2, 2, 1, 2, 2),
	RK3588_ARMCLK_L_RATE( 696000000, 0, 2, 2, 2, 2, 2, 1, 2, 2),
	RK3588_ARMCLK_L_RATE( 600000000, 0, 2, 1, 1, 1, 1, 1, 1, 1),
	RK3588_ARMCLK_L_RATE( 408000000, 0, 2, 1, 1, 1, 1, 1, 1, 1),
	RK3588_ARMCLK_L_RATE( 312000000, 0, 2, 1, 1, 1, 1, 1, 1, 1),
	RK3588_ARMCLK_L_RATE( 216000000, 0, 2, 1, 1, 1, 1, 1, 1, 1),
	RK3588_ARMCLK_L_RATE(  96000000, 0, 2, 1, 1, 1, 1, 1, 1, 1),
};

#define RK3588_CLK_CORE_B_SEL_MASK	__BITS(14,13)
#define RK3588_CLK_CORE_B_GPLL_DIV_MASK	__BITS(5,1)

#define RK3588_ARMCLK_B_RATE(_rate, _bigcore, _apllcore)		\
	{								\
		.rate = (_rate),					\
		.divs[0] = {						\
			.reg = CLKSEL_CON(_bigcore, 0),			\
			.mask = RK3588_CLK_CORE_B_SEL_MASK |		\
			    RK3588_CLK_CORE_B_GPLL_DIV_MASK,		\
			.val = __SHIFTIN((_apllcore),			\
			    RK3588_CLK_CORE_B_SEL_MASK) |		\
			    __SHIFTIN(0,				\
			    RK3588_CLK_CORE_B_GPLL_DIV_MASK)		\
		},							\
		.divs[1] = {						\
			.reg = CLKSEL_CON(_bigcore, 1),			\
			.mask = RK3588_CLK_CORE_B_SEL_MASK,		\
			.val = __SHIFTIN((_apllcore),			\
			    RK3588_CLK_CORE_B_SEL_MASK)			\
		}							\
	}

static const struct rk_cru_cpu_rate armclk_b01_rates[] = {
	RK3588_ARMCLK_B_RATE(2496000000, BIGCORE0, 1),
	RK3588_ARMCLK_B_RATE(2400000000, BIGCORE0, 1),
	RK3588_ARMCLK_B_RATE(2304000000, BIGCORE0, 1),
	RK3588_ARMCLK_B_RATE(2208000000, BIGCORE0, 1),
	RK3588_ARMCLK_B_RATE(2184000000, BIGCORE0, 1),
	RK3588_ARMCLK_B_RATE(2088000000, BIGCORE0, 1),
	RK3588_ARMCLK_B_RATE(2040000000, BIGCORE0, 1),
	RK3588_ARMCLK_B_RATE(2016000000, BIGCORE0, 1),
	RK3588_ARMCLK_B_RATE(1992000000, BIGCORE0, 1),
	RK3588_ARMCLK_B_RATE(1896000000, BIGCORE0, 1),
	RK3588_ARMCLK_B_RATE(1800000000, BIGCORE0, 1),
	RK3588_ARMCLK_B_RATE(1704000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1608000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1584000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1560000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1536000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1512000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1488000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1464000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1440000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1416000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1392000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1368000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1344000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1320000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1296000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1272000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1248000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1224000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1200000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1104000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(1008000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE( 912000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE( 816000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE( 696000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE( 600000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE( 408000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE( 312000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE( 216000000, BIGCORE0, 0),
	RK3588_ARMCLK_B_RATE(  96000000, BIGCORE0, 0),
};

static const struct rk_cru_cpu_rate armclk_b23_rates[] = {
	RK3588_ARMCLK_B_RATE(2496000000, BIGCORE1, 1),
	RK3588_ARMCLK_B_RATE(2400000000, BIGCORE1, 1),
	RK3588_ARMCLK_B_RATE(2304000000, BIGCORE1, 1),
	RK3588_ARMCLK_B_RATE(2208000000, BIGCORE1, 1),
	RK3588_ARMCLK_B_RATE(2184000000, BIGCORE1, 1),
	RK3588_ARMCLK_B_RATE(2088000000, BIGCORE1, 1),
	RK3588_ARMCLK_B_RATE(2040000000, BIGCORE1, 1),
	RK3588_ARMCLK_B_RATE(2016000000, BIGCORE1, 1),
	RK3588_ARMCLK_B_RATE(1992000000, BIGCORE1, 1),
	RK3588_ARMCLK_B_RATE(1896000000, BIGCORE1, 1),
	RK3588_ARMCLK_B_RATE(1800000000, BIGCORE1, 1),
	RK3588_ARMCLK_B_RATE(1704000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1608000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1584000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1560000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1536000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1512000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1488000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1464000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1440000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1416000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1392000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1368000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1344000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1320000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1296000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1272000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1248000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1224000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1200000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1104000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(1008000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE( 912000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE( 816000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE( 696000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE( 600000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE( 408000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE( 312000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE( 216000000, BIGCORE1, 0),
	RK3588_ARMCLK_B_RATE(  96000000, BIGCORE1, 0),
};

static struct rk_cru_pll_rate rk3588_pll_rates[] = {
	RK3588_PLL_RATE(2520000000, 2, 210, 0, 0),
	RK3588_PLL_RATE(2496000000, 2, 208, 0, 0),
	RK3588_PLL_RATE(2472000000, 2, 206, 0, 0),
	RK3588_PLL_RATE(2448000000, 2, 204, 0, 0),
	RK3588_PLL_RATE(2424000000, 2, 202, 0, 0),
	RK3588_PLL_RATE(2400000000, 2, 200, 0, 0),
	RK3588_PLL_RATE(2376000000, 2, 198, 0, 0),
	RK3588_PLL_RATE(2352000000, 2, 196, 0, 0),
	RK3588_PLL_RATE(2328000000, 2, 194, 0, 0),
	RK3588_PLL_RATE(2304000000, 2, 192, 0, 0),
	RK3588_PLL_RATE(2280000000, 2, 190, 0, 0),
	RK3588_PLL_RATE(2256000000, 2, 376, 1, 0),
	RK3588_PLL_RATE(2232000000, 2, 372, 1, 0),
	RK3588_PLL_RATE(2208000000, 2, 368, 1, 0),
	RK3588_PLL_RATE(2184000000, 2, 364, 1, 0),
	RK3588_PLL_RATE(2160000000, 2, 360, 1, 0),
	RK3588_PLL_RATE(2136000000, 2, 356, 1, 0),
	RK3588_PLL_RATE(2112000000, 2, 352, 1, 0),
	RK3588_PLL_RATE(2088000000, 2, 348, 1, 0),
	RK3588_PLL_RATE(2064000000, 2, 344, 1, 0),
	RK3588_PLL_RATE(2040000000, 2, 340, 1, 0),
	RK3588_PLL_RATE(2016000000, 2, 336, 1, 0),
	RK3588_PLL_RATE(1992000000, 2, 332, 1, 0),
	RK3588_PLL_RATE(1968000000, 2, 328, 1, 0),
	RK3588_PLL_RATE(1944000000, 2, 324, 1, 0),
	RK3588_PLL_RATE(1920000000, 2, 320, 1, 0),
	RK3588_PLL_RATE(1896000000, 2, 316, 1, 0),
	RK3588_PLL_RATE(1872000000, 2, 312, 1, 0),
	RK3588_PLL_RATE(1848000000, 2, 308, 1, 0),
	RK3588_PLL_RATE(1824000000, 2, 304, 1, 0),
	RK3588_PLL_RATE(1800000000, 2, 300, 1, 0),
	RK3588_PLL_RATE(1776000000, 2, 296, 1, 0),
	RK3588_PLL_RATE(1752000000, 2, 292, 1, 0),
	RK3588_PLL_RATE(1728000000, 2, 288, 1, 0),
	RK3588_PLL_RATE(1704000000, 2, 284, 1, 0),
	RK3588_PLL_RATE(1680000000, 2, 280, 1, 0),
	RK3588_PLL_RATE(1656000000, 2, 276, 1, 0),
	RK3588_PLL_RATE(1632000000, 2, 272, 1, 0),
	RK3588_PLL_RATE(1608000000, 2, 268, 1, 0),
	RK3588_PLL_RATE(1584000000, 2, 264, 1, 0),
	RK3588_PLL_RATE(1560000000, 2, 260, 1, 0),
	RK3588_PLL_RATE(1536000000, 2, 256, 1, 0),
	RK3588_PLL_RATE(1512000000, 2, 252, 1, 0),
	RK3588_PLL_RATE(1488000000, 2, 248, 1, 0),
	RK3588_PLL_RATE(1464000000, 2, 244, 1, 0),
	RK3588_PLL_RATE(1440000000, 2, 240, 1, 0),
	RK3588_PLL_RATE(1416000000, 2, 236, 1, 0),
	RK3588_PLL_RATE(1392000000, 2, 232, 1, 0),
	RK3588_PLL_RATE(1320000000, 2, 220, 1, 0),
	RK3588_PLL_RATE(1200000000, 2, 200, 1, 0),
	RK3588_PLL_RATE(1188000000, 2, 198, 1, 0),
	RK3588_PLL_RATE(1100000000, 3, 550, 2, 0),
	RK3588_PLL_RATE(1008000000, 2, 336, 2, 0),
	RK3588_PLL_RATE(1000000000, 3, 500, 2, 0),
	RK3588_PLL_RATE( 983040000, 4, 655, 2, 23592),
	RK3588_PLL_RATE( 955520000, 3, 477, 2, 49806),
	RK3588_PLL_RATE( 903168000, 6, 903, 2, 11009),
	RK3588_PLL_RATE( 900000000, 2, 300, 2, 0),
	RK3588_PLL_RATE( 850000000, 3, 425, 2, 0),
	RK3588_PLL_RATE( 816000000, 2, 272, 2, 0),
	RK3588_PLL_RATE( 786432000, 2, 262, 2, 9437),
	RK3588_PLL_RATE( 786000000, 1, 131, 2, 0),
	RK3588_PLL_RATE( 785560000, 3, 392, 2, 51117),
	RK3588_PLL_RATE( 722534400, 8, 963, 2, 24850),
	RK3588_PLL_RATE( 600000000, 2, 200, 2, 0),
	RK3588_PLL_RATE( 594000000, 2, 198, 2, 0),
	RK3588_PLL_RATE( 408000000, 2, 272, 3, 0),
	RK3588_PLL_RATE( 312000000, 2, 208, 3, 0),
	RK3588_PLL_RATE( 216000000, 2, 288, 4, 0),
	RK3588_PLL_RATE( 100000000, 3, 400, 5, 0),
	RK3588_PLL_RATE(  96000000, 2, 256, 5, 0),
};

static const char *mux_pll_parents[] = {
    "xin24m", "xin32k" };
static const char *mux_armclkl_parents[] = {
    "xin24m", "gpll", "lpll" };
static const char *mux_armclkb01_parents[] = {
    "xin24m", "gpll", "b0pll" };
static const char *mux_armclkb23_parents[] = {
    "xin24m", "gpll", "b1pll" };
static const char *b0pll_b1pll_lpll_gpll_parents[]= {
    "b0pll", "b1pll", "lpll", "gpll" };
static const char *gpll_24m_parents[] = {
    "gpll", "xin24m" };
static const char *gpll_aupll_parents[] = {
    "gpll", "aupll" };
static const char *gpll_lpll_parents[] = {
    "gpll", "lpll" };
static const char *gpll_cpll_parents[] = {
    "gpll", "cpll" };
static const char *gpll_spll_parents[] = {
    "gpll", "spll" };
static const char *gpll_cpll_24m_parents[] = {
    "gpll", "cpll", "xin24m"};
static const char *gpll_cpll_aupll_parents[] = {
    "gpll", "cpll", "aupll"};
static const char *gpll_cpll_npll_parents[] = {
    "gpll", "cpll", "npll"};
static const char *gpll_cpll_npll_v0pll_parents[]= {
    "gpll", "cpll", "npll", "v0pll"};
static const char *gpll_cpll_24m_spll_parents[] = {
    "gpll", "cpll", "xin24m", "spll" };
static const char *gpll_cpll_aupll_spll_parents[]= {
    "gpll", "cpll", "aupll", "spll" };
static const char *gpll_cpll_aupll_npll_parents[]= {
    "gpll", "cpll", "aupll", "npll" };
static const char *gpll_cpll_v0pll_aupll_parents[]= {
    "gpll", "cpll", "v0pll", "aupll" };
static const char *gpll_cpll_v0pll_spll_parents[]= {
    "gpll", "cpll", "v0pll", "spll" };
static const char *gpll_cpll_aupll_npll_spll_parents[]= {
    "gpll", "cpll", "aupll", "npll", "spll" };
static const char *gpll_cpll_npll_aupll_spll_parents[]= {
    "gpll", "cpll", "npll", "aupll", "spll" };
static const char *gpll_cpll_dmyaupll_npll_spll_parents[] =
    { "gpll", "cpll", "dummy_aupll", "npll", "spll" };
static const char *gpll_cpll_npll_1000m_parents[]= {
    "gpll", "cpll", "npll", "clk_1000m_src" };
static const char *mux_24m_spll_gpll_cpll_parents[]= {
    "xin24m", "spll", "gpll", "cpll" };
static const char *mux_24m_32k_parents[] = {
    "xin24m", "xin32k" };
static const char *mux_24m_100m_parents[] = {
    "xin24m", "clk_100m_src" };
static const char *mux_200m_100m_parents[] = {
    "clk_200m_src", "clk_100m_src" };
static const char *mux_100m_50m_24m_parents[] = {
    "clk_100m_src", "clk_50m_src", "xin24m" };
static const char *mux_150m_50m_24m_parents[] = {
    "clk_150m_src", "clk_50m_src", "xin24m" };
static const char *mux_150m_100m_24m_parents[] = {
    "clk_150m_src", "clk_100m_src", "xin24m" };
static const char *mux_200m_150m_24m_parents[] = {
    "clk_200m_src", "clk_150m_src", "xin24m" };
static const char *mux_150m_100m_50m_24m_parents[]= {
    "clk_150m_src", "clk_100m_src", "clk_50m_src", "xin24m" };
static const char *mux_200m_100m_50m_24m_parents[]= {
    "clk_200m_src", "clk_100m_src", "clk_50m_src", "xin24m" };
static const char *mux_300m_200m_100m_24m_parents[]= {
    "clk_300m_src", "clk_200m_src", "clk_100m_src", "xin24m" };
static const char *mux_700m_400m_200m_24m_parents[]= {
    "clk_700m_src", "clk_400m_src", "clk_200m_src", "xin_osc0_func" };
static const char *mux_500m_250m_100m_24m_parents[]= {
    "clk_500m_src", "clk_250m_src", "clk_100m_src", "xin_osc0_func" };
static const char *mux_500m_300m_100m_24m_parents[]= {
    "clk_500m_src", "clk_300m_src", "clk_100m_src", "xin_osc0_func" };
static const char *mux_400m_200m_100m_24m_parents[]= {
    "clk_400m_src", "clk_200m_src", "clk_100m_src", "xin_osc0_func" };
static const char *clk_i2s2_2ch_parents[] = {
    "clk_i2s2_2ch_src", "clk_i2s2_2ch_frac", "i2s2_mclkin", "xin12m" };
static const char *clk_i2s3_2ch_parents[] = {
    "clk_i2s3_2ch_src", "clk_i2s3_2ch_frac", "i2s3_mclkin", "xin12m" };
static const char *clk_i2s0_8ch_tx_parents[] = {
    "clk_i2s0_8ch_tx_src", "clk_i2s0_8ch_tx_frac", "i2s0_mclkin", "xin12m" };
static const char *clk_i2s0_8ch_rx_parents[] = {
    "clk_i2s0_8ch_rx_src", "clk_i2s0_8ch_rx_frac", "i2s0_mclkin", "xin12m" };
static const char *clk_i2s1_8ch_tx_parents[] = {
    "clk_i2s1_8ch_tx_src", "clk_i2s1_8ch_tx_frac", "i2s1_mclkin", "xin12m" };
static const char *clk_i2s1_8ch_rx_parents[] = {
    "clk_i2s1_8ch_rx_src", "clk_i2s1_8ch_rx_frac", "i2s1_mclkin", "xin12m" };
static const char *clk_i2s4_8ch_tx_parents[] = {
    "clk_i2s4_8ch_tx_src", "clk_i2s4_8ch_tx_frac", "i2s4_mclkin", "xin12m" };
static const char *clk_i2s5_8ch_tx_parents[] = {
    "clk_i2s5_8ch_tx_src", "clk_i2s5_8ch_tx_frac", "i2s5_mclkin", "xin12m" };
static const char *clk_i2s6_8ch_tx_parents[] = {
    "clk_i2s6_8ch_tx_src", "clk_i2s6_8ch_tx_frac", "i2s6_mclkin", "xin12m" };
static const char *clk_i2s6_8ch_rx_parents[] = {
    "clk_i2s6_8ch_rx_src", "clk_i2s6_8ch_rx_frac", "i2s6_mclkin", "xin12m" };
static const char *clk_i2s7_8ch_rx_parents[] = {
    "clk_i2s7_8ch_rx_src", "clk_i2s7_8ch_rx_frac", "i2s7_mclkin", "xin12m" };
static const char *clk_i2s8_8ch_tx_parents[] = {
    "clk_i2s8_8ch_tx_src", "clk_i2s8_8ch_tx_frac", "i2s8_mclkin", "xin12m" };
static const char *clk_i2s9_8ch_rx_parents[] = {
    "clk_i2s9_8ch_rx_src", "clk_i2s9_8ch_rx_frac", "i2s9_mclkin", "xin12m" };
static const char *clk_i2s10_8ch_rx_parents[] = {
    "clk_i2s10_8ch_rx_src", "clk_i2s10_8ch_rx_frac", "i2s10_mclkin", "xin12m" };
static const char *clk_spdif0_parents[] = {
    "clk_spdif0_src", "clk_spdif0_frac", "xin12m" };
static const char *clk_spdif1_parents[] = {
    "clk_spdif1_src", "clk_spdif1_frac", "xin12m" };
static const char *clk_spdif2_dp0_parents[] = {
    "clk_spdif2_dp0_src", "clk_spdif2_dp0_frac", "xin12m" };
static const char *clk_spdif3_parents[] = {
    "clk_spdif3_src", "clk_spdif3_frac", "xin12m" };
static const char *clk_spdif4_parents[] = {
    "clk_spdif4_src", "clk_spdif4_frac", "xin12m" };
static const char *clk_spdif5_dp1_parents[] = {
    "clk_spdif5_dp1_src", "clk_spdif5_dp1_frac", "xin12m" };
static const char *clk_uart0_parents[] = {
    "clk_uart0_src", "clk_uart0_frac", "xin24m" };
static const char *clk_uart1_parents[] = {
    "clk_uart1_src", "clk_uart1_frac", "xin24m" };
static const char *clk_uart2_parents[] = {
    "clk_uart2_src", "clk_uart2_frac", "xin24m" };
static const char *clk_uart3_parents[] = {
    "clk_uart3_src", "clk_uart3_frac", "xin24m" };
static const char *clk_uart4_parents[] = {
    "clk_uart4_src", "clk_uart4_frac", "xin24m" };
static const char *clk_uart5_parents[] = {
    "clk_uart5_src", "clk_uart5_frac", "xin24m" };
static const char *clk_uart6_parents[] = {
    "clk_uart6_src", "clk_uart6_frac", "xin24m" };
static const char *clk_uart7_parents[] = {
    "clk_uart7_src", "clk_uart7_frac", "xin24m" };
static const char *clk_uart8_parents[] = {
    "clk_uart8_src", "clk_uart8_frac", "xin24m" };
static const char *clk_uart9_parents[] = {
    "clk_uart9_src", "clk_uart9_frac", "xin24m" };
static const char *clk_gmac0_ptp_ref_parents[] = {
    "cpll", "clk_gmac0_ptpref_io" };
static const char *clk_gmac1_ptp_ref_parents[] = {
    "cpll", "clk_gmac1_ptpref_io" };
static const char *aclk_hdcp1_root_parents[] = {
    "gpll", "cpll", "clk_hdmitrx_refsrc" };
static const char *dclk_vop0_parents[] = {
    "dclk_vop0_src", "clk_hdmiphy_pixel0", "clk_hdmiphy_pixel1" };
static const char *dclk_vop1_parents[] = {
    "dclk_vop1_src", "clk_hdmiphy_pixel0", "clk_hdmiphy_pixel1" };
static const char *dclk_vop2_parents[] = {
    "dclk_vop2_src", "clk_hdmiphy_pixel0", "clk_hdmiphy_pixel1" };
static const char *pmu_200m_100m_parents[] = {
    "clk_pmu1_200m_src", "clk_pmu1_100m_src" };
static const char *pmu_300m_24m_parents[] = {
    "clk_300m_src", "xin24m" };
static const char *pmu_400m_24m_parents[] = {
    "clk_400m_src", "xin24m" };
static const char *pmu_100m_50m_24m_src_parents[]= {
    "clk_pmu1_100m_src", "clk_pmu1_50m_src", "xin24m" };
static const char *pmu_24m_32k_100m_src_parents[]= {
    "xin24m", "32k", "clk_pmu1_100m_src" };
static const char *hclk_pmu1_root_parents[] = {
    "clk_pmu1_200m_src", "clk_pmu1_100m_src", "clk_pmu1_50m_src", "xin24m" };
static const char *hclk_pmu_cm0_root_parents[] = {
    "clk_pmu1_400m_src", "clk_pmu1_200m_src", "clk_pmu1_100m_src", "xin24m" };
static const char *mclk_pdm0_parents[] = {
    "clk_pmu1_300m_src", "clk_pmu1_200m_src" };
static const char *mux_24m_ppll_spll_parents[] = {
    "xin24m", "ppll", "spll" };
static const char *mux_24m_ppll_parents[] = {
    "xin24m", "ppll" };
static const char *aclk_vop_sub_src_parents[] = {
    "aclk_vop_root", "aclk_vop_div2_src" };
static const char *clk_ref_pipe_phy0_parents[] = {
    "clk_ref_pipe_phy0_osc_src", "clk_ref_pipe_phy0_pll_src" };
static const char *clk_ref_pipe_phy1_parents[] = {
    "clk_ref_pipe_phy1_osc_src", "clk_ref_pipe_phy1_pll_src" };
static const char *clk_ref_pipe_phy2_parents[] = {
    "clk_ref_pipe_phy2_osc_src", "clk_ref_pipe_phy2_pll_src" };
static const char *i2s0_8ch_mclkout_parents[] = {
    "mclk_i2s0_8ch_tx", "mclk_i2s0_8ch_rx", "xin12m" };
static const char *i2s1_8ch_mclkout_parents[] = {
    "mclk_i2s1_8ch_tx", "mclk_i2s1_8ch_rx", "xin12m" };
static const char *i2s2_2ch_mclkout_parents[] = {
    "mclk_i2s2_2ch", "xin12m" };
static const char *i2s3_2ch_mclkout_parents[] = {
    "mclk_i2s3_2ch", "xin12m" };
static const char *i2s6_8ch_mclkout_parents[] = {
    "mclk_i2s6_8ch_tx", "mclk_i2s6_8ch_rx", "xin12m" };


static struct rk_cru_clk rk3588_cru_clks[] = {
	/* PLLs */
	RK3588_PLL(RK3588_PLL_B0PLL, "b0pll", mux_pll_parents,
	    PLL_CON(BIGCORE0, 0),
	    CLKSEL_CON(BIGCORE0, 0),
	    __BIT(6),
	    __BIT(15),
	    rk3588_pll_rates),
	RK3588_PLL(RK3588_PLL_B1PLL, "b1pll", mux_pll_parents,
	    PLL_CON(BIGCORE1, 8),
	    CLKSEL_CON(BIGCORE1, 0),
	    __BIT(6),
	    __BIT(15),
	    rk3588_pll_rates),

	RK3588_PLL(RK3588_PLL_LPLL, "lpll", mux_pll_parents,
	    PLL_CON(DSU, 16),
	    CLKSEL_CON(DSU, 5),
	    __BIT(14),
	    __BIT(15),
	    rk3588_pll_rates),
	RK3588_PLL(RK3588_PLL_V0PLL, "v0pll", mux_pll_parents,
	    PLL_CON(0, 88),
	    MODE_CON(0, 0),
	    __BIT(4),
	    __BIT(15),
	    rk3588_pll_rates),
	RK3588_PLL(RK3588_PLL_AUPLL, "aupll", mux_pll_parents,
	    PLL_CON(0, 96),
	    MODE_CON(0, 0),
	    __BIT(6),
	    __BIT(15),
	    rk3588_pll_rates),
	RK3588_PLL(RK3588_PLL_CPLL, "cpll", mux_pll_parents,
	    PLL_CON(0, 104),
	    MODE_CON(0, 0),
	    __BIT(8),
	    __BIT(15),
	    rk3588_pll_rates),
	RK3588_PLL(RK3588_PLL_GPLL, "gpll", mux_pll_parents,
	    PLL_CON(0, 112),
	    MODE_CON(0, 0),
	    __BIT(2),
	    __BIT(15),
	    rk3588_pll_rates),
	RK3588_PLL(RK3588_PLL_NPLL, "npll", mux_pll_parents,
	    PLL_CON(0, 120),
	    MODE_CON(0, 0),
	    __BIT(0),
	    __BIT(15),
	    rk3588_pll_rates),
	RK3588_PLL(RK3588_PLL_PPLL, "ppll", mux_pll_parents,
	    PLL_CON(PHP, 128),
	    MODE_CON(0, 0),
	    __BIT(10),
	    __BIT(15),
	    rk3588_pll_rates),

	/* big,little cores */
	RK_CPU_CORE4(RK3588_ARMCLK_L, "armclk_l", mux_armclkl_parents,
	    CLKSEL_CON(DSU, 5),		/* mux_reg */
	    __BITS(15,14), 2, 1,	/* mux_mask, mux_main, mux_alt */
	    CLKSEL_CON(DSU, 6),		/* div0_reg */
	    __BITS(4,0),		/* div0_mask */
	    CLKSEL_CON(DSU, 6),		/* div1_reg */
	    __BITS(11,7),		/* div1_mask */
	    CLKSEL_CON(DSU, 7),		/* div2_reg */
	    __BITS(4,0),		/* div2_mask */
	    CLKSEL_CON(DSU, 7),		/* div3_reg */
	    __BITS(11,7),		/* div3_mask */
	    armclk_l_rates),
	RK_CPU_CORE2(RK3588_ARMCLK_B01, "armclk_b01", mux_armclkb01_parents,
	    CLKSEL_CON(BIGCORE0, 0),	/* mux_reg */
	    __BITS(7,6), 2, 1,		/* mux_mask, mux_main, mux_alt */
	    CLKSEL_CON(BIGCORE0, 0),	/* div0_reg */
	    __BITS(12,8),		/* div0_mask */
	    CLKSEL_CON(BIGCORE0, 1),	/* div1_reg */
	    __BITS(4,0),		/* div1_mask */
	    armclk_b01_rates),
	RK_CPU_CORE2(RK3588_ARMCLK_B23, "armclk_b23", mux_armclkb23_parents,
	    CLKSEL_CON(BIGCORE1, 0),	/* reg */
	    __BITS(7,6), 2, 1,		/* mux_mask, mux_main, mux_alt */
	    CLKSEL_CON(BIGCORE1, 0),	/* div0_reg */
	    __BITS(12,8),		/* div0_mask */
	    CLKSEL_CON(BIGCORE1, 1),	/* div1_reg */
	    __BITS(4,0),		/* div1_mask */
	    armclk_b23_rates),

	RK_COMPOSITE_NODIV(RK3588_PCLK_BIGCORE0_ROOT, "pclk_bigcore0_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(BIGCORE0, 2), __BITS(1,0),
	    CLKGATE_CON(BIGCORE0, 0), __BIT(14),
	    0),
	RK_GATE(RK3588_PCLK_BIGCORE0_PVTM, "pclk_bigcore0_pvtm",
	    "pclk_bigcore0_root",
	    CLKGATE_CON(BIGCORE0, 1), 0),
	RK_GATE(RK3588_CLK_BIGCORE0_PVTM, "clk_bigcore0_pvtm", "xin24m",
	    CLKGATE_CON(BIGCORE0, 0), 12),
	RK_GATE(RK3588_CLK_CORE_BIGCORE0_PVTM, "clk_core_bigcore0_pvtm",
	    "armclk_b01",
	    CLKGATE_CON(BIGCORE0, 0), 13),

	RK_COMPOSITE_NODIV(RK3588_PCLK_BIGCORE1_ROOT, "pclk_bigcore1_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(BIGCORE1, 2), __BITS(1,0),
	    CLKGATE_CON(BIGCORE1, 0), __BIT(14),
	    0),
	RK_GATE(RK3588_PCLK_BIGCORE1_PVTM, "pclk_bigcore1_pvtm",
	    "pclk_bigcore1_root",
	    CLKGATE_CON(BIGCORE1, 1), 0),
	RK_GATE(RK3588_CLK_BIGCORE1_PVTM, "clk_bigcore1_pvtm", "xin24m",
	    CLKGATE_CON(BIGCORE1, 0), 12),
	RK_GATE(RK3588_CLK_CORE_BIGCORE1_PVTM, "clk_core_bigcore1_pvtm",
	    "armclk_b23",
	    CLKGATE_CON(BIGCORE1, 0), 13),

	RK_COMPOSITE(RK3588_CLK_50M_SRC, "clk_50m_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 0), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 0), __BIT(0),
	    0),
	RK_COMPOSITE(RK3588_CLK_100M_SRC, "clk_100m_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 0), __BITS(11,11), __BITS(10,6),
	    CLKGATE_CON(0, 0), __BIT(1),
	    0),
	RK_COMPOSITE(RK3588_CLK_150M_SRC, "clk_150m_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 1), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 0), __BIT(2),
	    0),
	RK_COMPOSITE(RK3588_CLK_200M_SRC, "clk_200m_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 1), __BITS(11,11), __BITS(10,6),
	    CLKGATE_CON(0, 0), __BIT(3),
	    0),
	RK_COMPOSITE(RK3588_CLK_250M_SRC, "clk_250m_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 2), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 0), __BIT(4),
	    0),
	RK_COMPOSITE(RK3588_CLK_300M_SRC, "clk_300m_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 2), __BITS(11,11), __BITS(10,6),
	    CLKGATE_CON(0, 0), __BIT(5),
	    0),
	RK_COMPOSITE(RK3588_CLK_350M_SRC, "clk_350m_src",
	    gpll_spll_parents,
	    CLKSEL_CON(0, 3), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 0), __BIT(6),
	    0),
	RK_COMPOSITE(RK3588_CLK_400M_SRC, "clk_400m_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 3), __BITS(11,11), __BITS(10,6),
	    CLKGATE_CON(0, 0), __BIT(7),
	    0),
	RK_COMPOSITE_HALF(RK3588_CLK_450M_SRC, "clk_450m_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 4), __BITS(5,5),
	    __BITS(4,0),
	    CLKGATE_CON(0, 0), __BIT(8),
	    0),
	RK_COMPOSITE(RK3588_CLK_500M_SRC, "clk_500m_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 4), __BITS(11,11), __BITS(10,6),
	    CLKGATE_CON(0, 0), __BIT(9),
	    0),
	RK_COMPOSITE(RK3588_CLK_600M_SRC, "clk_600m_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 5), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 0), __BIT(10),
	    0),
	RK_COMPOSITE(RK3588_CLK_650M_SRC, "clk_650m_src",
	    gpll_lpll_parents,
	    CLKSEL_CON(0, 5), __BITS(11,11), __BITS(10,6),
	    CLKGATE_CON(0, 0), __BIT(11),
	    0),
	RK_COMPOSITE(RK3588_CLK_700M_SRC, "clk_700m_src",
	    gpll_spll_parents,
	    CLKSEL_CON(0, 6), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 0), __BIT(12),
	    0),
	RK_COMPOSITE(RK3588_CLK_800M_SRC, "clk_800m_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 6), __BITS(11,11), __BITS(10,6),
	    CLKGATE_CON(0, 0), __BIT(13),
	    0),
	RK_COMPOSITE_HALF(RK3588_CLK_1000M_SRC, "clk_1000m_src",
	    gpll_cpll_npll_v0pll_parents,
	    CLKSEL_CON(0, 7), __BITS(6,5),
	    __BITS(4,0),
	    CLKGATE_CON(0, 0), __BIT(14),
	    0),
	RK_COMPOSITE(RK3588_CLK_1200M_SRC, "clk_1200m_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 7), __BITS(12,12), __BITS(11,7),
	    CLKGATE_CON(0, 0), __BIT(15),
	    0),
	RK_COMPOSITE_NODIV(RK3588_ACLK_TOP_M300_ROOT, "aclk_top_m300_root",
	    mux_300m_200m_100m_24m_parents,
	    CLKSEL_CON(0, 9), __BITS(1,0),
	    CLKGATE_CON(0, 1), __BIT(10),
	    0),
	RK_COMPOSITE_NODIV(RK3588_ACLK_TOP_M500_ROOT, "aclk_top_m500_root",
	    mux_500m_300m_100m_24m_parents,
	    CLKSEL_CON(0, 9), __BITS(3,2),
	    CLKGATE_CON(0, 1), __BIT(11),
	    0),
	RK_COMPOSITE_NODIV(RK3588_ACLK_TOP_M400_ROOT, "aclk_top_m400_root",
	    mux_400m_200m_100m_24m_parents,
	    CLKSEL_CON(0, 9), __BITS(5,4),
	    CLKGATE_CON(0, 1), __BIT(12),
	    0),
	RK_COMPOSITE_NODIV(RK3588_ACLK_TOP_S200_ROOT, "aclk_top_s200_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 9), __BITS(7,6),
	    CLKGATE_CON(0, 1), __BIT(13),
	    0),
	RK_COMPOSITE_NODIV(RK3588_ACLK_TOP_S400_ROOT, "aclk_top_s400_root",
	    mux_400m_200m_100m_24m_parents,
	    CLKSEL_CON(0, 9), __BITS(9,8),
	    CLKGATE_CON(0, 1), __BIT(14),
	    0),
	RK_COMPOSITE(RK3588_ACLK_TOP_ROOT, "aclk_top_root",
	    gpll_cpll_aupll_parents,
	    CLKSEL_CON(0, 8), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(0, 1), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_TOP_ROOT, "pclk_top_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(0, 8), __BITS(8,7),
	    CLKGATE_CON(0, 1), __BIT(1),
	    0),
	RK_COMPOSITE(RK3588_ACLK_LOW_TOP_ROOT, "aclk_low_top_root",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 8), __BITS(14,14), __BITS(13,9),
	    CLKGATE_CON(0, 1), __BIT(2),
	    0),
	RK_COMPOSITE(RK3588_CLK_MIPI_CAMARAOUT_M0, "clk_mipi_camaraout_m0",
	    mux_24m_spll_gpll_cpll_parents,
	    CLKSEL_CON(0, 18), __BITS(9,8), __BITS(7,0),
	    CLKGATE_CON(0, 5), __BIT(9),
	    0),
	RK_COMPOSITE(RK3588_CLK_MIPI_CAMARAOUT_M1, "clk_mipi_camaraout_m1",
	    mux_24m_spll_gpll_cpll_parents,
	    CLKSEL_CON(0, 19), __BITS(9,8), __BITS(7,0),
	    CLKGATE_CON(0, 5), __BIT(10),
	    0),
	RK_COMPOSITE(RK3588_CLK_MIPI_CAMARAOUT_M2, "clk_mipi_camaraout_m2",
	    mux_24m_spll_gpll_cpll_parents,
	    CLKSEL_CON(0, 20), __BITS(9,8), __BITS(7,0),
	    CLKGATE_CON(0, 5), __BIT(11),
	    0),
	RK_COMPOSITE(RK3588_CLK_MIPI_CAMARAOUT_M3, "clk_mipi_camaraout_m3",
	    mux_24m_spll_gpll_cpll_parents,
	    CLKSEL_CON(0, 21), __BITS(9,8), __BITS(7,0),
	    CLKGATE_CON(0, 5), __BIT(12),
	    0),
	RK_COMPOSITE(RK3588_CLK_MIPI_CAMARAOUT_M4, "clk_mipi_camaraout_m4",
	    mux_24m_spll_gpll_cpll_parents,
	    CLKSEL_CON(0, 22), __BITS(9,8), __BITS(7,0),
	    CLKGATE_CON(0, 5), __BIT(13),
	    0),
	RK_COMPOSITE(RK3588_MCLK_GMAC0_OUT, "mclk_gmac0_out",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 15), __BITS(7,7), __BITS(6,0),
	    CLKGATE_CON(0, 5), __BIT(3),
	    0),
	RK_COMPOSITE(RK3588_REFCLKO25M_ETH0_OUT, "refclko25m_eth0_out",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 15), __BITS(15,15), __BITS(14,8),
	    CLKGATE_CON(0, 5), __BIT(4),
	    0),
	RK_COMPOSITE(RK3588_REFCLKO25M_ETH1_OUT, "refclko25m_eth1_out",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 16), __BITS(7,7), __BITS(6,0),
	    CLKGATE_CON(0, 5), __BIT(5),
	    0),
	RK_COMPOSITE(RK3588_CLK_CIFOUT_OUT, "clk_cifout_out",
	    gpll_cpll_24m_spll_parents,
	    CLKSEL_CON(0, 17), __BITS(9,8), __BITS(7,0),
	    CLKGATE_CON(0, 5), __BIT(6),
	    0),
	RK_GATE(RK3588_PCLK_MIPI_DCPHY0, "pclk_mipi_dcphy0", "pclk_top_root",
	    CLKGATE_CON(0, 3), 14),
	RK_GATE(RK3588_PCLK_MIPI_DCPHY1, "pclk_mipi_dcphy1", "pclk_top_root",
	    CLKGATE_CON(0, 4), 3),
	RK_GATE(RK3588_PCLK_CSIPHY0, "pclk_csiphy0", "pclk_top_root",
	    CLKGATE_CON(0, 1), 6),
	RK_GATE(RK3588_PCLK_CSIPHY1, "pclk_csiphy1", "pclk_top_root",
	    CLKGATE_CON(0, 1), 8),
	RK_GATE(RK3588_PCLK_CRU, "pclk_cru", "pclk_top_root",
	    CLKGATE_CON(0, 5), 0),
	RK_COMPOSITE(0, "sclk_dsu",
	    b0pll_b1pll_lpll_gpll_parents,
	    CLKSEL_CON(DSU, 0), __BITS(13,12), __BITS(4,0),
	    CLKGATE_CON(DSU, 0), __BIT(4),
	    0),
	RK_COMPOSITE_NOMUX(0, "atclk_dsu", "sclk_dsu",
	    CLKSEL_CON(DSU, 3), __BITS(4,0),
	    CLKGATE_CON(DSU, 1), __BIT(0),
	    0),
	RK_COMPOSITE_NOMUX(0, "gicclk_dsu", "sclk_dsu",
	    CLKSEL_CON(DSU, 3), __BITS(9,5),
	    CLKGATE_CON(DSU, 1), __BIT(1),
	    0),
	RK_COMPOSITE_NOMUX(0, "aclkmp_dsu", "sclk_dsu",
	    CLKSEL_CON(DSU, 1), __BITS(15,11),
	    CLKGATE_CON(DSU, 0), __BIT(12),
	    0),
	RK_COMPOSITE_NOMUX(0, "aclkm_dsu", "sclk_dsu",
	    CLKSEL_CON(DSU, 1), __BITS(5,1),
	    CLKGATE_CON(DSU, 0), __BIT(8),
	    0),
	RK_COMPOSITE_NOMUX(0, "aclks_dsu", "sclk_dsu",
	    CLKSEL_CON(DSU, 1), __BITS(10,6),
	    CLKGATE_CON(DSU, 0), __BIT(9),
	    0),
	RK_COMPOSITE_NOMUX(0, "periph_dsu", "sclk_dsu",
	    CLKSEL_CON(DSU, 2), __BITS(4,0),
	    CLKGATE_CON(DSU, 0), __BIT(13),
	    0),
	RK_COMPOSITE_NOMUX(0, "cntclk_dsu", "periph_dsu",
	    CLKSEL_CON(DSU, 2), __BITS(9,5),
	    CLKGATE_CON(DSU, 0), __BIT(14),
	    0),
	RK_COMPOSITE_NOMUX(0, "tsclk_dsu", "periph_dsu",
	    CLKSEL_CON(DSU, 2), __BITS(14,10),
	    CLKGATE_CON(DSU, 0), __BIT(15),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_DSU_S_ROOT, "pclk_dsu_s_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(DSU, 4), __BITS(12,11),
	    CLKGATE_CON(DSU, 2), __BIT(2),
	    0),
	RK_COMPOSITE(RK3588_PCLK_DSU_ROOT, "pclk_dsu_root",
	    b0pll_b1pll_lpll_gpll_parents,
	    CLKSEL_CON(DSU, 4), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(DSU, 1), __BIT(3),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_DSU_NS_ROOT, "pclk_dsu_ns_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(DSU, 4), __BITS(8,7),
	    CLKGATE_CON(DSU, 1), __BIT(4),
	    0),
	RK_GATE(RK3588_PCLK_LITCORE_PVTM, "pclk_litcore_pvtm",
	    "pclk_dsu_ns_root",
	    CLKGATE_CON(DSU, 2), 6),
	RK_GATE(RK3588_PCLK_DBG, "pclk_dbg", "pclk_dsu_root",
	    CLKGATE_CON(DSU, 1), 7),
	RK_GATE(RK3588_PCLK_DSU, "pclk_dsu", "pclk_dsu_root",
	    CLKGATE_CON(DSU, 1), 6),
	RK_GATE(RK3588_PCLK_S_DAPLITE, "pclk_s_daplite", "pclk_dsu_ns_root",
	    CLKGATE_CON(DSU, 1), 8),
	RK_GATE(RK3588_PCLK_M_DAPLITE, "pclk_m_daplite", "pclk_dsu_root",
	    CLKGATE_CON(DSU, 1), 9),
	RK_GATE(RK3588_CLK_LITCORE_PVTM, "clk_litcore_pvtm", "xin24m",
	    CLKGATE_CON(DSU, 2), 0),
	RK_GATE(RK3588_CLK_CORE_LITCORE_PVTM, "clk_core_litcore_pvtm",
	    "armclk_l",
	    CLKGATE_CON(DSU, 2), 1),
	RK_COMPOSITE_NODIV(RK3588_HCLK_AUDIO_ROOT, "hclk_audio_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 24), __BITS(1,0),
	    CLKGATE_CON(0, 7), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_AUDIO_ROOT, "pclk_audio_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(0, 24), __BITS(3,2),
	    CLKGATE_CON(0, 7), __BIT(1),
	    0),
	RK_GATE(RK3588_HCLK_I2S2_2CH, "hclk_i2s2_2ch", "hclk_audio_root",
	    CLKGATE_CON(0, 7), 12),
	RK_GATE(RK3588_HCLK_I2S3_2CH, "hclk_i2s3_2ch", "hclk_audio_root",
	    CLKGATE_CON(0, 7), 13),
	RK_COMPOSITE(RK3588_CLK_I2S2_2CH_SRC, "clk_i2s2_2ch_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 28), __BITS(9,9), __BITS(8,4),
	    CLKGATE_CON(0, 7), __BIT(14),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S2_2CH_FRAC, "clk_i2s2_2ch_frac",
	    "clk_i2s2_2ch_src",
	    CLKGATE_CON(0, 7),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S2_2CH, "clk_i2s2_2ch", clk_i2s2_2ch_parents,
	    CLKSEL_CON(0, 30), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S2_2CH, "mclk_i2s2_2ch", "clk_i2s2_2ch",
	    CLKGATE_CON(0, 8), 0),
	RK_MUX(RK3588_I2S2_2CH_MCLKOUT, "i2s2_2ch_mclkout",
	    i2s2_2ch_mclkout_parents,
	    CLKSEL_CON(0, 30), __BITS(2,2)),
	RK_COMPOSITE(RK3588_CLK_I2S3_2CH_SRC, "clk_i2s3_2ch_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 30), __BITS(8,8), __BITS(7,3),
	    CLKGATE_CON(0, 8), __BIT(1),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S3_2CH_FRAC, "clk_i2s3_2ch_frac",
	    "clk_i2s3_2ch_src",
	    CLKGATE_CON(0, 8),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S3_2CH, "clk_i2s3_2ch", clk_i2s3_2ch_parents,
	    CLKSEL_CON(0, 32), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S3_2CH, "mclk_i2s3_2ch", "clk_i2s3_2ch",
	    CLKGATE_CON(0, 8), 3),
	RK_GATE(RK3588_CLK_DAC_ACDCDIG, "clk_dac_acdcdig", "mclk_i2s3_2ch",
	    CLKGATE_CON(0, 8), 4),
	RK_MUX(RK3588_I2S3_2CH_MCLKOUT, "i2s3_2ch_mclkout",
	    i2s3_2ch_mclkout_parents,
	    CLKSEL_CON(0, 32), __BITS(2,2)),
	RK_GATE(RK3588_PCLK_ACDCDIG, "pclk_acdcdig", "pclk_audio_root",
	    CLKGATE_CON(0, 7), 11),
	RK_GATE(RK3588_HCLK_I2S0_8CH, "hclk_i2s0_8ch", "hclk_audio_root",
	    CLKGATE_CON(0, 7), 4),
	RK_COMPOSITE(RK3588_CLK_I2S0_8CH_TX_SRC, "clk_i2s0_8ch_tx_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 24), __BITS(9,9), __BITS(8,4),
	    CLKGATE_CON(0, 7), __BIT(5),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S0_8CH_TX_FRAC, "clk_i2s0_8ch_tx_frac",
	   "clk_i2s0_8ch_tx_src",
	    CLKGATE_CON(0, 7),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S0_8CH_TX, "clk_i2s0_8ch_tx",
	    clk_i2s0_8ch_tx_parents,
	    CLKSEL_CON(0, 26), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S0_8CH_TX, "mclk_i2s0_8ch_tx", "clk_i2s0_8ch_tx",
	    CLKGATE_CON(0, 7), 7),
	RK_COMPOSITE(RK3588_CLK_I2S0_8CH_RX_SRC, "clk_i2s0_8ch_rx_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 26), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 7), __BIT(8),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S0_8CH_RX_FRAC, "clk_i2s0_8ch_rx_frac",
	    "clk_i2s0_8ch_rx_src",
	    CLKGATE_CON(0, 7),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S0_8CH_RX, "clk_i2s0_8ch_rx",
	    clk_i2s0_8ch_rx_parents,
	    CLKSEL_CON(0, 28), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S0_8CH_RX, "mclk_i2s0_8ch_rx", "clk_i2s0_8ch_rx",
	    CLKGATE_CON(0, 7), 10),
	RK_MUX(RK3588_I2S0_8CH_MCLKOUT, "i2s0_8ch_mclkout",
	    i2s0_8ch_mclkout_parents,
	    CLKSEL_CON(0, 28), __BITS(3,2)),
	RK_GATE(RK3588_HCLK_PDM1, "hclk_pdm1", "hclk_audio_root",
	    CLKGATE_CON(0, 9), 6),
	RK_COMPOSITE(RK3588_MCLK_PDM1, "mclk_pdm1",
	    gpll_cpll_aupll_parents,
	    CLKSEL_CON(0, 36), __BITS(8,7), __BITS(6,2),
	    CLKGATE_CON(0, 9), __BIT(7),
	    0),
	RK_GATE(RK3588_HCLK_SPDIF0, "hclk_spdif0", "hclk_audio_root",
	    CLKGATE_CON(0, 8), 14),
	RK_COMPOSITE(RK3588_CLK_SPDIF0_SRC, "clk_spdif0_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 32), __BITS(8,8), __BITS(7,3),
	    CLKGATE_CON(0, 8), __BIT(15),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_SPDIF0_FRAC, "clk_spdif0_frac",
	    "clk_spdif0_src",
	    CLKGATE_CON(0, 9),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_SPDIF0, "clk_spdif0", clk_spdif0_parents,
	    CLKSEL_CON(0, 34), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_SPDIF0, "mclk_spdif0", "clk_spdif0",
	    CLKGATE_CON(0, 9), 1),
	RK_GATE(RK3588_HCLK_SPDIF1, "hclk_spdif1", "hclk_audio_root",
	    CLKGATE_CON(0, 9), 2),
	RK_COMPOSITE(RK3588_CLK_SPDIF1_SRC, "clk_spdif1_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 34), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 9), __BIT(3),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_SPDIF1_FRAC, "clk_spdif1_frac",
	    "clk_spdif1_src",
	    CLKGATE_CON(0, 9),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_SPDIF1, "clk_spdif1", clk_spdif1_parents,
	    CLKSEL_CON(0, 36), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_SPDIF1, "mclk_spdif1", "clk_spdif1",
	    CLKGATE_CON(0, 9), 5),
	RK_COMPOSITE(RK3588_ACLK_AV1_ROOT, "aclk_av1_root",
	    gpll_cpll_aupll_parents,
	    CLKSEL_CON(0, 163), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(0, 68), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_AV1_ROOT, "pclk_av1_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 163), __BITS(8,7),
	    CLKGATE_CON(0, 68), __BIT(3),
	    0),
	RK_COMPOSITE(RK3588_ACLK_BUS_ROOT, "aclk_bus_root",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 38), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 10), __BIT(0),
	    0),
	RK_GATE(RK3588_PCLK_MAILBOX0, "pclk_mailbox0", "pclk_top_root",
	    CLKGATE_CON(0, 16), 11),
	RK_GATE(RK3588_PCLK_MAILBOX1, "pclk_mailbox1", "pclk_top_root",
	    CLKGATE_CON(0, 16), 12),
	RK_GATE(RK3588_PCLK_MAILBOX2, "pclk_mailbox2", "pclk_top_root",
	    CLKGATE_CON(0, 16), 13),
	RK_GATE(RK3588_PCLK_PMU2, "pclk_pmu2", "pclk_top_root",
	    CLKGATE_CON(0, 19), 3),
	RK_GATE(RK3588_PCLK_PMUCM0_INTMUX, "pclk_pmucm0_intmux",
	    "pclk_top_root",
	    CLKGATE_CON(0, 19), 4),
	RK_GATE(RK3588_PCLK_DDRCM0_INTMUX, "pclk_ddrcm0_intmux",
	    "pclk_top_root",
	    CLKGATE_CON(0, 19), 5),
	RK_GATE(RK3588_PCLK_PWM1, "pclk_pwm1", "pclk_top_root",
	    CLKGATE_CON(0, 15), 3),
	RK_COMPOSITE_NODIV(RK3588_CLK_PWM1, "clk_pwm1",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(0, 59), __BITS(13,12),
	    CLKGATE_CON(0, 15), __BIT(4),
	    0),
	RK_GATE(RK3588_CLK_PWM1_CAPTURE, "clk_pwm1_capture", "xin24m",
	    CLKGATE_CON(0, 15), 5),
	RK_GATE(RK3588_PCLK_PWM2, "pclk_pwm2", "pclk_top_root",
	    CLKGATE_CON(0, 15), 6),
	RK_COMPOSITE_NODIV(RK3588_CLK_PWM2, "clk_pwm2",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(0, 59), __BITS(15,14),
	    CLKGATE_CON(0, 15), __BIT(7),
	    0),
	RK_GATE(RK3588_CLK_PWM2_CAPTURE, "clk_pwm2_capture", "xin24m",
	    CLKGATE_CON(0, 15), 8),
	RK_GATE(RK3588_PCLK_PWM3, "pclk_pwm3", "pclk_top_root",
	    CLKGATE_CON(0, 15), 9),
	RK_COMPOSITE_NODIV(RK3588_CLK_PWM3, "clk_pwm3",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(0, 60), __BITS(1,0),
	    CLKGATE_CON(0, 15), __BIT(10),
	    0),
	RK_GATE(RK3588_CLK_PWM3_CAPTURE, "clk_pwm3_capture", "xin24m",
	    CLKGATE_CON(0, 15), 11),
	RK_GATE(RK3588_PCLK_BUSTIMER0, "pclk_bustimer0", "pclk_top_root",
	    CLKGATE_CON(0, 15), 12),
	RK_GATE(RK3588_PCLK_BUSTIMER1, "pclk_bustimer1", "pclk_top_root",
	    CLKGATE_CON(0, 15), 13),
	RK_COMPOSITE_NODIV(RK3588_CLK_BUS_TIMER_ROOT, "clk_bus_timer_root",
	    mux_24m_100m_parents,
	    CLKSEL_CON(0, 60), __BITS(2,2),
	    CLKGATE_CON(0, 15), __BIT(14),
	    0),
	RK_GATE(RK3588_CLK_BUSTIMER0, "clk_bustimer0", "clk_bus_timer_root",
	    CLKGATE_CON(0, 15), 15),
	RK_GATE(RK3588_CLK_BUSTIMER1, "clk_bustimer1", "clk_bus_timer_root",
	    CLKGATE_CON(0, 16), 0),
	RK_GATE(RK3588_CLK_BUSTIMER2, "clk_bustimer2", "clk_bus_timer_root",
	    CLKGATE_CON(0, 16), 1),
	RK_GATE(RK3588_CLK_BUSTIMER3, "clk_bustimer3", "clk_bus_timer_root",
	    CLKGATE_CON(0, 16), 2),
	RK_GATE(RK3588_CLK_BUSTIMER4, "clk_bustimer4", "clk_bus_timer_root",
	    CLKGATE_CON(0, 16), 3),
	RK_GATE(RK3588_CLK_BUSTIMER5, "clk_bustimer5", "clk_bus_timer_root",
	    CLKGATE_CON(0, 16), 4),
	RK_GATE(RK3588_CLK_BUSTIMER6, "clk_bustimer6", "clk_bus_timer_root",
	    CLKGATE_CON(0, 16), 5),
	RK_GATE(RK3588_CLK_BUSTIMER7, "clk_bustimer7", "clk_bus_timer_root",
	    CLKGATE_CON(0, 16), 6),
	RK_GATE(RK3588_CLK_BUSTIMER8, "clk_bustimer8", "clk_bus_timer_root",
	    CLKGATE_CON(0, 16), 7),
	RK_GATE(RK3588_CLK_BUSTIMER9, "clk_bustimer9", "clk_bus_timer_root",
	    CLKGATE_CON(0, 16), 8),
	RK_GATE(RK3588_CLK_BUSTIMER10, "clk_bustimer10", "clk_bus_timer_root",
	    CLKGATE_CON(0, 16), 9),
	RK_GATE(RK3588_CLK_BUSTIMER11, "clk_bustimer11", "clk_bus_timer_root",
	    CLKGATE_CON(0, 16), 10),
	RK_GATE(RK3588_PCLK_WDT0, "pclk_wdt0", "pclk_top_root",
	    CLKGATE_CON(0, 15), 0),
	RK_GATE(RK3588_TCLK_WDT0, "tclk_wdt0", "xin24m",
	    CLKGATE_CON(0, 15), 1),
	RK_GATE(RK3588_PCLK_CAN0, "pclk_can0", "pclk_top_root",
	    CLKGATE_CON(0, 11), 8),
	RK_COMPOSITE(RK3588_CLK_CAN0, "clk_can0",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 39), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 11), __BIT(9),
	    0),
	RK_GATE(RK3588_PCLK_CAN1, "pclk_can1", "pclk_top_root",
	    CLKGATE_CON(0, 11), 10),
	RK_COMPOSITE(RK3588_CLK_CAN1, "clk_can1",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 39), __BITS(11,11), __BITS(10,6),
	    CLKGATE_CON(0, 11), __BIT(11),
	    0),
	RK_GATE(RK3588_PCLK_CAN2, "pclk_can2", "pclk_top_root",
	    CLKGATE_CON(0, 11), 12),
	RK_COMPOSITE(RK3588_CLK_CAN2, "clk_can2",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 40), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 11), __BIT(13),
	    0),
	RK_GATE(RK3588_ACLK_DECOM, "aclk_decom", "aclk_bus_root",
	    CLKGATE_CON(0, 17), 6),
	RK_GATE(RK3588_PCLK_DECOM, "pclk_decom", "pclk_top_root",
	    CLKGATE_CON(0, 17), 7),
	RK_COMPOSITE(RK3588_DCLK_DECOM, "dclk_decom",
	    gpll_spll_parents,
	    CLKSEL_CON(0, 62), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 17), __BIT(8),
	    0),
	RK_GATE(RK3588_ACLK_DMAC0, "aclk_dmac0", "aclk_bus_root",
	    CLKGATE_CON(0, 10), 5),
	RK_GATE(RK3588_ACLK_DMAC1, "aclk_dmac1", "aclk_bus_root",
	    CLKGATE_CON(0, 10), 6),
	RK_GATE(RK3588_ACLK_DMAC2, "aclk_dmac2", "aclk_bus_root",
	    CLKGATE_CON(0, 10), 7),
	RK_GATE(RK3588_ACLK_GIC, "aclk_gic", "aclk_bus_root",
	    CLKGATE_CON(0, 10), 3),
	RK_GATE(RK3588_PCLK_GPIO1, "pclk_gpio1", "pclk_top_root",
	    CLKGATE_CON(0, 16), 14),
	RK_COMPOSITE(RK3588_DBCLK_GPIO1, "dbclk_gpio1",
	    mux_24m_32k_parents,
	    CLKSEL_CON(0, 60), __BITS(8,8), __BITS(7,3),
	    CLKGATE_CON(0, 16), __BIT(15),
	    0),
	RK_GATE(RK3588_PCLK_GPIO2, "pclk_gpio2", "pclk_top_root",
	    CLKGATE_CON(0, 17), 0),
	RK_COMPOSITE(RK3588_DBCLK_GPIO2, "dbclk_gpio2",
	    mux_24m_32k_parents,
	    CLKSEL_CON(0, 60), __BITS(14,14), __BITS(13,9),
	    CLKGATE_CON(0, 17), __BIT(1),
	    0),
	RK_GATE(RK3588_PCLK_GPIO3, "pclk_gpio3", "pclk_top_root",
	    CLKGATE_CON(0, 17), 2),
	RK_COMPOSITE(RK3588_DBCLK_GPIO3, "dbclk_gpio3",
	    mux_24m_32k_parents,
	    CLKSEL_CON(0, 61), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 17), __BIT(3),
	    0),
	RK_GATE(RK3588_PCLK_GPIO4, "pclk_gpio4", "pclk_top_root",
	    CLKGATE_CON(0, 17), 4),
	RK_COMPOSITE(RK3588_DBCLK_GPIO4, "dbclk_gpio4",
	    mux_24m_32k_parents,
	    CLKSEL_CON(0, 61), __BITS(11,11), __BITS(10,6),
	    CLKGATE_CON(0, 17), __BIT(5),
	    0),
	RK_GATE(RK3588_PCLK_I2C1, "pclk_i2c1", "pclk_top_root",
	    CLKGATE_CON(0, 10), 8),
	RK_GATE(RK3588_PCLK_I2C2, "pclk_i2c2", "pclk_top_root",
	    CLKGATE_CON(0, 10), 9),
	RK_GATE(RK3588_PCLK_I2C3, "pclk_i2c3", "pclk_top_root",
	    CLKGATE_CON(0, 10), 10),
	RK_GATE(RK3588_PCLK_I2C4, "pclk_i2c4", "pclk_top_root",
	    CLKGATE_CON(0, 10), 11),
	RK_GATE(RK3588_PCLK_I2C5, "pclk_i2c5", "pclk_top_root",
	    CLKGATE_CON(0, 10), 12),
	RK_GATE(RK3588_PCLK_I2C6, "pclk_i2c6", "pclk_top_root",
	    CLKGATE_CON(0, 10), 13),
	RK_GATE(RK3588_PCLK_I2C7, "pclk_i2c7", "pclk_top_root",
	    CLKGATE_CON(0, 10), 14),
	RK_GATE(RK3588_PCLK_I2C8, "pclk_i2c8", "pclk_top_root",
	    CLKGATE_CON(0, 10), 15),
	RK_COMPOSITE_NODIV(RK3588_CLK_I2C1, "clk_i2c1",
	    mux_200m_100m_parents,
	    CLKSEL_CON(0, 38), __BITS(6,6),
	    CLKGATE_CON(0, 11), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_CLK_I2C2, "clk_i2c2",
	    mux_200m_100m_parents,
	    CLKSEL_CON(0, 38), __BITS(7,7),
	    CLKGATE_CON(0, 11), __BIT(1),
	    0),
	RK_COMPOSITE_NODIV(RK3588_CLK_I2C3, "clk_i2c3",
	    mux_200m_100m_parents,
	    CLKSEL_CON(0, 38), __BITS(8,8),
	    CLKGATE_CON(0, 11), __BIT(2),
	    0),
	RK_COMPOSITE_NODIV(RK3588_CLK_I2C4, "clk_i2c4",
	    mux_200m_100m_parents,
	    CLKSEL_CON(0, 38), __BITS(9,9),
	    CLKGATE_CON(0, 11), __BIT(3),
	    0),
	RK_COMPOSITE_NODIV(RK3588_CLK_I2C5, "clk_i2c5",
	    mux_200m_100m_parents,
	    CLKSEL_CON(0, 38), __BITS(10,10),
	    CLKGATE_CON(0, 11), __BIT(4),
	    0),
	RK_COMPOSITE_NODIV(RK3588_CLK_I2C6, "clk_i2c6",
	    mux_200m_100m_parents,
	    CLKSEL_CON(0, 38), __BITS(11,11),
	    CLKGATE_CON(0, 11), __BIT(5),
	    0),
	RK_COMPOSITE_NODIV(RK3588_CLK_I2C7, "clk_i2c7",
	    mux_200m_100m_parents,
	    CLKSEL_CON(0, 38), __BITS(12,12),
	    CLKGATE_CON(0, 11), __BIT(6),
	    0),
	RK_COMPOSITE_NODIV(RK3588_CLK_I2C8, "clk_i2c8",
	    mux_200m_100m_parents,
	    CLKSEL_CON(0, 38), __BITS(13,13),
	    CLKGATE_CON(0, 11), __BIT(7),
	    0),
	RK_GATE(RK3588_PCLK_OTPC_NS, "pclk_otpc_ns", "pclk_top_root",
	    CLKGATE_CON(0, 18), 9),
	RK_GATE(RK3588_CLK_OTPC_NS, "clk_otpc_ns", "xin24m",
	    CLKGATE_CON(0, 18), 10),
	RK_GATE(RK3588_CLK_OTPC_ARB, "clk_otpc_arb", "xin24m",
	    CLKGATE_CON(0, 18), 11),
	RK_GATE(RK3588_CLK_OTP_PHY_G, "clk_otp_phy_g", "xin24m",
	    CLKGATE_CON(0, 18), 13),
	RK_GATE(RK3588_CLK_OTPC_AUTO_RD_G, "clk_otpc_auto_rd_g", "xin24m",
	    CLKGATE_CON(0, 18), 12),
	RK_GATE(RK3588_PCLK_SARADC, "pclk_saradc", "pclk_top_root",
	    CLKGATE_CON(0, 11), 14),
	RK_COMPOSITE(RK3588_CLK_SARADC, "clk_saradc",
	    gpll_24m_parents,
	    CLKSEL_CON(0, 40), __BITS(14,14), __BITS(13,6),
	    CLKGATE_CON(0, 11), __BIT(15),
	    0),
	RK_GATE(RK3588_PCLK_SPI0, "pclk_spi0", "pclk_top_root",
	    CLKGATE_CON(0, 14), 6),
	RK_GATE(RK3588_PCLK_SPI1, "pclk_spi1", "pclk_top_root",
	    CLKGATE_CON(0, 14), 7),
	RK_GATE(RK3588_PCLK_SPI2, "pclk_spi2", "pclk_top_root",
	    CLKGATE_CON(0, 14), 8),
	RK_GATE(RK3588_PCLK_SPI3, "pclk_spi3", "pclk_top_root",
	    CLKGATE_CON(0, 14), 9),
	RK_GATE(RK3588_PCLK_SPI4, "pclk_spi4", "pclk_top_root",
	    CLKGATE_CON(0, 14), 10),
	RK_COMPOSITE_NODIV(RK3588_CLK_SPI0, "clk_spi0",
	    mux_200m_150m_24m_parents,
	    CLKSEL_CON(0, 59), __BITS(3,2),
	    CLKGATE_CON(0, 14), __BIT(11),
	    0),
	RK_COMPOSITE_NODIV(RK3588_CLK_SPI1, "clk_spi1",
	    mux_200m_150m_24m_parents,
	    CLKSEL_CON(0, 59), __BITS(5,4),
	    CLKGATE_CON(0, 14), __BIT(12),
	    0),
	RK_COMPOSITE_NODIV(RK3588_CLK_SPI2, "clk_spi2",
	    mux_200m_150m_24m_parents,
	    CLKSEL_CON(0, 59), __BITS(7,6),
	    CLKGATE_CON(0, 14), __BIT(13),
	    0),
	RK_COMPOSITE_NODIV(RK3588_CLK_SPI3, "clk_spi3",
	    mux_200m_150m_24m_parents,
	    CLKSEL_CON(0, 59), __BITS(9,8),
	    CLKGATE_CON(0, 14), __BIT(14),
	    0),
	RK_COMPOSITE_NODIV(RK3588_CLK_SPI4, "clk_spi4",
	    mux_200m_150m_24m_parents,
	    CLKSEL_CON(0, 59), __BITS(11,10),
	    CLKGATE_CON(0, 14), __BIT(15),
	    0),
	RK_GATE(RK3588_ACLK_SPINLOCK, "aclk_spinlock", "aclk_bus_root",
	    CLKGATE_CON(0, 18), 6),
	RK_GATE(RK3588_PCLK_TSADC, "pclk_tsadc", "pclk_top_root",
	    CLKGATE_CON(0, 12), 0),
	RK_COMPOSITE(RK3588_CLK_TSADC, "clk_tsadc",
	    gpll_24m_parents,
	    CLKSEL_CON(0, 41), __BITS(8,8), __BITS(7,0),
	    CLKGATE_CON(0, 12), __BIT(1),
	    0),
	RK_GATE(RK3588_PCLK_UART1, "pclk_uart1", "pclk_top_root",
	    CLKGATE_CON(0, 12), 2),
	RK_GATE(RK3588_PCLK_UART2, "pclk_uart2", "pclk_top_root",
	    CLKGATE_CON(0, 12), 3),
	RK_GATE(RK3588_PCLK_UART3, "pclk_uart3", "pclk_top_root",
	    CLKGATE_CON(0, 12), 4),
	RK_GATE(RK3588_PCLK_UART4, "pclk_uart4", "pclk_top_root",
	    CLKGATE_CON(0, 12), 5),
	RK_GATE(RK3588_PCLK_UART5, "pclk_uart5", "pclk_top_root",
	    CLKGATE_CON(0, 12), 6),
	RK_GATE(RK3588_PCLK_UART6, "pclk_uart6", "pclk_top_root",
	    CLKGATE_CON(0, 12), 7),
	RK_GATE(RK3588_PCLK_UART7, "pclk_uart7", "pclk_top_root",
	    CLKGATE_CON(0, 12), 8),
	RK_GATE(RK3588_PCLK_UART8, "pclk_uart8", "pclk_top_root",
	    CLKGATE_CON(0, 12), 9),
	RK_GATE(RK3588_PCLK_UART9, "pclk_uart9", "pclk_top_root",
	    CLKGATE_CON(0, 12), 10),
	RK_COMPOSITE(RK3588_CLK_UART1_SRC, "clk_uart1_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 41), __BITS(14,14), __BITS(13,9),
	    CLKGATE_CON(0, 12), __BIT(11),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_UART1_FRAC, "clk_uart1_frac",
	    "clk_uart1_src",
	    CLKGATE_CON(0, 12),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_UART1, "clk_uart1", clk_uart1_parents,
	    CLKSEL_CON(0, 43), __BITS(1,0)),
	RK_GATE(RK3588_SCLK_UART1, "sclk_uart1", "clk_uart1",
	    CLKGATE_CON(0, 12), 13),
	RK_COMPOSITE(RK3588_CLK_UART2_SRC, "clk_uart2_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 43), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 12), __BIT(14),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_UART2_FRAC, "clk_uart2_frac",
	    "clk_uart2_src",
	    CLKGATE_CON(0, 12),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_UART2, "clk_uart2", clk_uart2_parents,
	    CLKSEL_CON(0, 45), __BITS(1,0)),
	RK_GATE(RK3588_SCLK_UART2, "sclk_uart2", "clk_uart2",
	    CLKGATE_CON(0, 13), 0),
	RK_COMPOSITE(RK3588_CLK_UART3_SRC, "clk_uart3_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 45), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 13), __BIT(1),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_UART3_FRAC, "clk_uart3_frac",
	    "clk_uart3_src",
	    CLKGATE_CON(0, 13),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_UART3, "clk_uart3", clk_uart3_parents,
	    CLKSEL_CON(0, 47), __BITS(1,0)),
	RK_GATE(RK3588_SCLK_UART3, "sclk_uart3", "clk_uart3",
	    CLKGATE_CON(0, 13), 3),
	RK_COMPOSITE(RK3588_CLK_UART4_SRC, "clk_uart4_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 47), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 13), __BIT(4),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_UART4_FRAC, "clk_uart4_frac",
	    "clk_uart4_src",
	    CLKGATE_CON(0, 13),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_UART4, "clk_uart4", clk_uart4_parents,
	    CLKSEL_CON(0, 49), __BITS(1,0)),
	RK_GATE(RK3588_SCLK_UART4, "sclk_uart4", "clk_uart4",
	    CLKGATE_CON(0, 13), 6),
	RK_COMPOSITE(RK3588_CLK_UART5_SRC, "clk_uart5_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 49), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 13), __BIT(7),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_UART5_FRAC, "clk_uart5_frac",
	    "clk_uart5_src",
	    CLKGATE_CON(0, 13),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_UART5, "clk_uart5", clk_uart5_parents,
	    CLKSEL_CON(0, 51), __BITS(1,0)),
	RK_GATE(RK3588_SCLK_UART5, "sclk_uart5", "clk_uart5",
	    CLKGATE_CON(0, 13), 9),
	RK_COMPOSITE(RK3588_CLK_UART6_SRC, "clk_uart6_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 51), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 13), __BIT(10),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_UART6_FRAC, "clk_uart6_frac",
	    "clk_uart6_src",
	    CLKGATE_CON(0, 13),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_UART6, "clk_uart6", clk_uart6_parents,
	    CLKSEL_CON(0, 53), __BITS(1,0)),
	RK_GATE(RK3588_SCLK_UART6, "sclk_uart6", "clk_uart6",
	    CLKGATE_CON(0, 13), 12),
	RK_COMPOSITE(RK3588_CLK_UART7_SRC, "clk_uart7_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 53), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 13), __BIT(13),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_UART7_FRAC, "clk_uart7_frac",
	    "clk_uart7_src",
	    CLKGATE_CON(0, 13),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_UART7, "clk_uart7", clk_uart7_parents,
	    CLKSEL_CON(0, 55), __BITS(1,0)),
	RK_GATE(RK3588_SCLK_UART7, "sclk_uart7", "clk_uart7",
	    CLKGATE_CON(0, 13), 15),
	RK_COMPOSITE(RK3588_CLK_UART8_SRC, "clk_uart8_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 55), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 14), __BIT(0),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_UART8_FRAC, "clk_uart8_frac",
	    "clk_uart8_src",
	    CLKGATE_CON(0, 14),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_UART8, "clk_uart8", clk_uart8_parents,
	    CLKSEL_CON(0, 57), __BITS(1,0)),
	RK_GATE(RK3588_SCLK_UART8, "sclk_uart8", "clk_uart8",
	    CLKGATE_CON(0, 14), 2),
	RK_COMPOSITE(RK3588_CLK_UART9_SRC, "clk_uart9_src",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 57), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 14), __BIT(3),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_UART9_FRAC, "clk_uart9_frac",
	    "clk_uart9_src",
	    CLKGATE_CON(0, 14),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_UART9, "clk_uart9", clk_uart9_parents,
	    CLKSEL_CON(0, 59), __BITS(1,0)),
	RK_GATE(RK3588_SCLK_UART9, "sclk_uart9", "clk_uart9",
	    CLKGATE_CON(0, 14), 5),
	RK_COMPOSITE_NODIV(RK3588_ACLK_CENTER_ROOT, "aclk_center_root",
	    mux_700m_400m_200m_24m_parents,
	    CLKSEL_CON(0, 165), __BITS(1,0),
	    CLKGATE_CON(0, 69), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_ACLK_CENTER_LOW_ROOT, "aclk_center_low_root",
	    mux_500m_250m_100m_24m_parents,
	    CLKSEL_CON(0, 165), __BITS(3,2),
	    CLKGATE_CON(0, 69), __BIT(1),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_CENTER_ROOT, "hclk_center_root",
	    mux_400m_200m_100m_24m_parents,
	    CLKSEL_CON(0, 165), __BITS(5,4),
	    CLKGATE_CON(0, 69), __BIT(2),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_CENTER_ROOT, "pclk_center_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 165), __BITS(7,6),
	    CLKGATE_CON(0, 69), __BIT(3),
	    0),
	RK_GATE(RK3588_ACLK_DMA2DDR, "aclk_dma2ddr", "aclk_center_root",
	    CLKGATE_CON(0, 69), 5),
	RK_GATE(RK3588_ACLK_DDR_SHAREMEM, "aclk_ddr_sharemem",
	    "aclk_center_low_root",
	    CLKGATE_CON(0, 69), 6),
	RK_COMPOSITE_NODIV(RK3588_ACLK_CENTER_S200_ROOT,
	    "aclk_center_s200_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 165), __BITS(9,8),
	    CLKGATE_CON(0, 69), __BIT(8),
	    0),
	RK_COMPOSITE_NODIV(RK3588_ACLK_CENTER_S400_ROOT,
	    "aclk_center_s400_root", mux_400m_200m_100m_24m_parents,
	    CLKSEL_CON(0, 165), __BITS(11,10),
	    CLKGATE_CON(0, 69), __BIT(9),
	    0),
	RK_GATE(RK3588_FCLK_DDR_CM0_CORE, "fclk_ddr_cm0_core",
	    "hclk_center_root",
	    CLKGATE_CON(0, 69), 14),
	RK_COMPOSITE_NODIV(RK3588_CLK_DDR_TIMER_ROOT, "clk_ddr_timer_root",
	    mux_24m_100m_parents,
	    CLKSEL_CON(0, 165), __BITS(12,12),
	    CLKGATE_CON(0, 69), __BIT(15),
	    0),
	RK_GATE(RK3588_CLK_DDR_TIMER0, "clk_ddr_timer0", "clk_ddr_timer_root",
	    CLKGATE_CON(0, 70), 0),
	RK_GATE(RK3588_CLK_DDR_TIMER1, "clk_ddr_timer1", "clk_ddr_timer_root",
	    CLKGATE_CON(0, 70), 1),
	RK_GATE(RK3588_TCLK_WDT_DDR, "tclk_wdt_ddr", "xin24m",
	    CLKGATE_CON(0, 70), 2),
	RK_COMPOSITE(RK3588_CLK_DDR_CM0_RTC, "clk_ddr_cm0_rtc",
	    mux_24m_32k_parents,
	    CLKSEL_CON(0, 166), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 70), __BIT(4),
	    0),
	RK_GATE(RK3588_PCLK_WDT, "pclk_wdt", "pclk_center_root",
	    CLKGATE_CON(0, 70), 7),
	RK_GATE(RK3588_PCLK_TIMER, "pclk_timer", "pclk_center_root",
	    CLKGATE_CON(0, 70), 8),
	RK_GATE(RK3588_PCLK_DMA2DDR, "pclk_dma2ddr", "pclk_center_root",
	    CLKGATE_CON(0, 70), 9),
	RK_GATE(RK3588_PCLK_SHAREMEM, "pclk_sharemem", "pclk_center_root",
	    CLKGATE_CON(0, 70), 10),
	RK_COMPOSITE(RK3588_CLK_GPU_SRC, "clk_gpu_src",
	    gpll_cpll_aupll_npll_spll_parents,
	    CLKSEL_CON(0, 158), __BITS(7,5), __BITS(4,0),
	    CLKGATE_CON(0, 66), __BIT(1),
	    0),
	RK_GATE(RK3588_CLK_GPU, "clk_gpu", "clk_gpu_src",
	    CLKGATE_CON(0, 66), 4),
	RK_GATE(RK3588_CLK_GPU_COREGROUP, "clk_gpu_coregroup", "clk_gpu_src",
	    CLKGATE_CON(0, 66), 6),
	RK_COMPOSITE_NOMUX(RK3588_CLK_GPU_STACKS, "clk_gpu_stacks",
	    "clk_gpu_src",
	    CLKSEL_CON(0, 159), __BITS(4,0),
	    CLKGATE_CON(0, 66), __BIT(7),
	    0),
	RK_GATE(RK3588_CLK_GPU_PVTM, "clk_gpu_pvtm", "xin24m",
	    CLKGATE_CON(0, 67), 0),
	RK_GATE(RK3588_CLK_CORE_GPU_PVTM, "clk_core_gpu_pvtm", "clk_gpu_src",
	    CLKGATE_CON(0, 67), 1),
	RK_COMPOSITE(RK3588_ACLK_ISP1_ROOT, "aclk_isp1_root",
	    gpll_cpll_aupll_spll_parents,
	    CLKSEL_CON(0, 67), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(0, 26), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_ISP1_ROOT, "hclk_isp1_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 67), __BITS(8,7),
	    CLKGATE_CON(0, 26), __BIT(1),
	    0),
	RK_COMPOSITE(RK3588_CLK_ISP1_CORE, "clk_isp1_core",
	    gpll_cpll_aupll_spll_parents,
	    CLKSEL_CON(0, 67), __BITS(15,14), __BITS(13,9),
	    CLKGATE_CON(0, 26), __BIT(2),
	    0),
	RK_GATE(RK3588_CLK_ISP1_CORE_MARVIN, "clk_isp1_core_marvin",
	    "clk_isp1_core",
	    CLKGATE_CON(0, 26), 3),
	RK_GATE(RK3588_CLK_ISP1_CORE_VICAP, "clk_isp1_core_vicap",
	    "clk_isp1_core",
	    CLKGATE_CON(0, 26), 4),
	RK_COMPOSITE_NODIV(RK3588_HCLK_NPU_ROOT, "hclk_npu_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 73), __BITS(1,0),
	    CLKGATE_CON(0, 29), __BIT(0),
	    0),
	RK_COMPOSITE(RK3588_CLK_NPU_DSU0, "clk_npu_dsu0",
	    gpll_cpll_aupll_npll_spll_parents,
	    CLKSEL_CON(0, 73), __BITS(9,7), __BITS(6,2),
	    CLKGATE_CON(0, 29), __BIT(1),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_NPU_ROOT, "pclk_npu_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(0, 74), __BITS(2,1),
	    CLKGATE_CON(0, 29), __BIT(4),
	    0),
	RK_GATE(RK3588_ACLK_NPU1, "aclk_npu1", "clk_npu_dsu0",
	    CLKGATE_CON(0, 27), 0),
	RK_GATE(RK3588_HCLK_NPU1, "hclk_npu1", "hclk_npu_root",
	    CLKGATE_CON(0, 27), 2),
	RK_GATE(RK3588_ACLK_NPU2, "aclk_npu2", "clk_npu_dsu0",
	    CLKGATE_CON(0, 28), 0),
	RK_GATE(RK3588_HCLK_NPU2, "hclk_npu2", "hclk_npu_root",
	    CLKGATE_CON(0, 28), 2),
	RK_COMPOSITE_NODIV(RK3588_HCLK_NPU_CM0_ROOT, "hclk_npu_cm0_root",
	    mux_400m_200m_100m_24m_parents,
	    CLKSEL_CON(0, 74), __BITS(6,5),
	    CLKGATE_CON(0, 30), __BIT(1),
	    0),
	RK_GATE(RK3588_FCLK_NPU_CM0_CORE, "fclk_npu_cm0_core",
	    "hclk_npu_cm0_root",
	    CLKGATE_CON(0, 30), 3),
	RK_COMPOSITE(RK3588_CLK_NPU_CM0_RTC, "clk_npu_cm0_rtc",
	    mux_24m_32k_parents,
	    CLKSEL_CON(0, 74), __BITS(12,12), __BITS(11,7),
	    CLKGATE_CON(0, 30), __BIT(5),
	    0),
	RK_GATE(RK3588_PCLK_NPU_PVTM, "pclk_npu_pvtm", "pclk_npu_root",
	    CLKGATE_CON(0, 29), 12),
	RK_GATE(RK3588_PCLK_NPU_GRF, "pclk_npu_grf", "pclk_npu_root",
	    CLKGATE_CON(0, 29), 13),
	RK_GATE(RK3588_CLK_NPU_PVTM, "clk_npu_pvtm", "xin24m",
	    CLKGATE_CON(0, 29), 14),
	RK_GATE(RK3588_CLK_CORE_NPU_PVTM, "clk_core_npu_pvtm", "clk_npu_dsu0",
	    CLKGATE_CON(0, 29), 15),
	RK_GATE(RK3588_ACLK_NPU0, "aclk_npu0", "clk_npu_dsu0",
	    CLKGATE_CON(0, 30), 6),
	RK_GATE(RK3588_HCLK_NPU0, "hclk_npu0", "hclk_npu_root",
	    CLKGATE_CON(0, 30), 8),
	RK_GATE(RK3588_PCLK_NPU_TIMER, "pclk_npu_timer", "pclk_npu_root",
	    CLKGATE_CON(0, 29), 6),
	RK_COMPOSITE_NODIV(RK3588_CLK_NPUTIMER_ROOT, "clk_nputimer_root",
	    mux_24m_100m_parents,
	    CLKSEL_CON(0, 74), __BITS(3,3),
	    CLKGATE_CON(0, 29), __BIT(7),
	    0),
	RK_GATE(RK3588_CLK_NPUTIMER0, "clk_nputimer0", "clk_nputimer_root",
	    CLKGATE_CON(0, 29), 8),
	RK_GATE(RK3588_CLK_NPUTIMER1, "clk_nputimer1", "clk_nputimer_root",
	    CLKGATE_CON(0, 29), 9),
	RK_GATE(RK3588_PCLK_NPU_WDT, "pclk_npu_wdt", "pclk_npu_root",
	    CLKGATE_CON(0, 29), 10),
	RK_GATE(RK3588_TCLK_NPU_WDT, "tclk_npu_wdt", "xin24m",
	    CLKGATE_CON(0, 29), 11),
	RK_COMPOSITE_NODIV(RK3588_HCLK_NVM_ROOT, "hclk_nvm_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 77), __BITS(1,0),
	    CLKGATE_CON(0, 31), __BIT(0),
	    0),
	RK_COMPOSITE(RK3588_ACLK_NVM_ROOT, "aclk_nvm_root",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 77), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 31), __BIT(1),
	    0),
	RK_GATE(RK3588_ACLK_EMMC, "aclk_emmc", "aclk_nvm_root",
	    CLKGATE_CON(0, 31), 5),
	RK_COMPOSITE(RK3588_CCLK_EMMC, "cclk_emmc",
	    gpll_cpll_24m_parents,
	    CLKSEL_CON(0, 77), __BITS(15,14), __BITS(13,8),
	    CLKGATE_CON(0, 31), __BIT(6),
	    0),
	RK_COMPOSITE(RK3588_BCLK_EMMC, "bclk_emmc",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 78), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 31), __BIT(7),
	    0),
	RK_GATE(RK3588_TMCLK_EMMC, "tmclk_emmc", "xin24m",
	    CLKGATE_CON(0, 31), 8),
	RK_COMPOSITE(RK3588_SCLK_SFC, "sclk_sfc",
	    gpll_cpll_24m_parents,
	    CLKSEL_CON(0, 78), __BITS(13,12), __BITS(11,6),
	    CLKGATE_CON(0, 31), __BIT(9),
	    0),
	RK_COMPOSITE(RK3588_CLK_GMAC0_PTP_REF, "clk_gmac0_ptp_ref",
	    clk_gmac0_ptp_ref_parents,
	    CLKSEL_CON(0, 81), __BITS(6,6), __BITS(5,0),
	    CLKGATE_CON(0, 34), __BIT(10),
	    0),
	RK_COMPOSITE(RK3588_CLK_GMAC1_PTP_REF, "clk_gmac1_ptp_ref",
	    clk_gmac1_ptp_ref_parents,
	    CLKSEL_CON(0, 81), __BITS(13,13), __BITS(12,7),
	    CLKGATE_CON(0, 34), __BIT(11),
	    0),
	RK_COMPOSITE(RK3588_CLK_GMAC_125M, "clk_gmac_125m",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 83), __BITS(15,15), __BITS(14,8),
	    CLKGATE_CON(0, 35), __BIT(5),
	    0),
	RK_COMPOSITE(RK3588_CLK_GMAC_50M, "clk_gmac_50m",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 84), __BITS(7,7), __BITS(6,0),
	    CLKGATE_CON(0, 35), __BIT(6),
	    0),
	RK_COMPOSITE(RK3588_ACLK_PCIE_ROOT, "aclk_pcie_root",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 80), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 32), __BIT(6),
	    0),
	RK_COMPOSITE(RK3588_ACLK_PHP_ROOT, "aclk_php_root",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 80), __BITS(13,13), __BITS(12,8),
	    CLKGATE_CON(0, 32), __BIT(7),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_PHP_ROOT, "pclk_php_root",
	    mux_150m_50m_24m_parents,
	    CLKSEL_CON(0, 80), __BITS(1,0),
	    CLKGATE_CON(0, 32), __BIT(0),
	    0),
	RK_GATE(RK3588_ACLK_PHP_GIC_ITS, "aclk_php_gic_its", "aclk_pcie_root",
	    CLKGATE_CON(0, 34), 6),
	RK_GATE(RK3588_ACLK_PCIE_BRIDGE, "aclk_pcie_bridge", "aclk_pcie_root",
	    CLKGATE_CON(0, 32), 8),
	RK_GATE(RK3588_ACLK_MMU_PCIE, "aclk_mmu_pcie", "aclk_pcie_bridge",
	    CLKGATE_CON(0, 34), 7),
	RK_GATE(RK3588_ACLK_MMU_PHP, "aclk_mmu_php", "aclk_php_root",
	    CLKGATE_CON(0, 34), 8),
	RK_GATE(RK3588_ACLK_PCIE_4L_DBI, "aclk_pcie_4l_dbi", "aclk_php_root",
	    CLKGATE_CON(0, 32), 13),
	RK_GATE(RK3588_ACLK_PCIE_2L_DBI, "aclk_pcie_2l_dbi", "aclk_php_root",
	    CLKGATE_CON(0, 32), 14),
	RK_GATE(RK3588_ACLK_PCIE_1L0_DBI, "aclk_pcie_1l0_dbi", "aclk_php_root",
	    CLKGATE_CON(0, 32), 15),
	RK_GATE(RK3588_ACLK_PCIE_1L1_DBI, "aclk_pcie_1l1_dbi", "aclk_php_root",
	    CLKGATE_CON(0, 33), 0),
	RK_GATE(RK3588_ACLK_PCIE_1L2_DBI, "aclk_pcie_1l2_dbi", "aclk_php_root",
	    CLKGATE_CON(0, 33), 1),
	RK_GATE(RK3588_ACLK_PCIE_4L_MSTR, "aclk_pcie_4l_mstr", "aclk_mmu_pcie",
	    CLKGATE_CON(0, 33), 2),
	RK_GATE(RK3588_ACLK_PCIE_2L_MSTR, "aclk_pcie_2l_mstr", "aclk_mmu_pcie",
	    CLKGATE_CON(0, 33), 3),
	RK_GATE(RK3588_ACLK_PCIE_1L0_MSTR, "aclk_pcie_1l0_mstr",
	    "aclk_mmu_pcie",
	    CLKGATE_CON(0, 33), 4),
	RK_GATE(RK3588_ACLK_PCIE_1L1_MSTR, "aclk_pcie_1l1_mstr",
	    "aclk_mmu_pcie",
	    CLKGATE_CON(0, 33), 5),
	RK_GATE(RK3588_ACLK_PCIE_1L2_MSTR, "aclk_pcie_1l2_mstr",
	    "aclk_mmu_pcie",
	    CLKGATE_CON(0, 33), 6),
	RK_GATE(RK3588_ACLK_PCIE_4L_SLV, "aclk_pcie_4l_slv", "aclk_php_root",
	    CLKGATE_CON(0, 33), 7),
	RK_GATE(RK3588_ACLK_PCIE_2L_SLV, "aclk_pcie_2l_slv", "aclk_php_root",
	    CLKGATE_CON(0, 33), 8),
	RK_GATE(RK3588_ACLK_PCIE_1L0_SLV, "aclk_pcie_1l0_slv", "aclk_php_root",
	    CLKGATE_CON(0, 33), 9),
	RK_GATE(RK3588_ACLK_PCIE_1L1_SLV, "aclk_pcie_1l1_slv", "aclk_php_root",
	    CLKGATE_CON(0, 33), 10),
	RK_GATE(RK3588_ACLK_PCIE_1L2_SLV, "aclk_pcie_1l2_slv", "aclk_php_root",
	    CLKGATE_CON(0, 33), 11),
	RK_GATE(RK3588_PCLK_PCIE_4L, "pclk_pcie_4l", "pclk_php_root",
	    CLKGATE_CON(0, 33), 12),
	RK_GATE(RK3588_PCLK_PCIE_2L, "pclk_pcie_2l", "pclk_php_root",
	    CLKGATE_CON(0, 33), 13),
	RK_GATE(RK3588_PCLK_PCIE_1L0, "pclk_pcie_1l0", "pclk_php_root",
	    CLKGATE_CON(0, 33), 14),
	RK_GATE(RK3588_PCLK_PCIE_1L1, "pclk_pcie_1l1", "pclk_php_root",
	    CLKGATE_CON(0, 33), 15),
	RK_GATE(RK3588_PCLK_PCIE_1L2, "pclk_pcie_1l2", "pclk_php_root",
	    CLKGATE_CON(0, 34), 0),
	RK_GATE(RK3588_CLK_PCIE_AUX0, "clk_pcie_aux0", "xin24m",
	    CLKGATE_CON(0, 34), 1),
	RK_GATE(RK3588_CLK_PCIE_AUX1, "clk_pcie_aux1", "xin24m",
	    CLKGATE_CON(0, 34), 2),
	RK_GATE(RK3588_CLK_PCIE_AUX2, "clk_pcie_aux2", "xin24m",
	    CLKGATE_CON(0, 34), 3),
	RK_GATE(RK3588_CLK_PCIE_AUX3, "clk_pcie_aux3", "xin24m",
	    CLKGATE_CON(0, 34), 4),
	RK_GATE(RK3588_CLK_PCIE_AUX4, "clk_pcie_aux4", "xin24m",
	    CLKGATE_CON(0, 34), 5),
	RK_GATE(RK3588_CLK_PIPEPHY0_REF, "clk_pipephy0_ref", "xin24m",
	    CLKGATE_CON(0, 37), 0),
	RK_GATE(RK3588_CLK_PIPEPHY1_REF, "clk_pipephy1_ref", "xin24m",
	    CLKGATE_CON(0, 37), 1),
	RK_GATE(RK3588_CLK_PIPEPHY2_REF, "clk_pipephy2_ref", "xin24m",
	    CLKGATE_CON(0, 37), 2),
	RK_GATE(RK3588_PCLK_GMAC0, "pclk_gmac0", "pclk_php_root",
	    CLKGATE_CON(0, 32), 3),
	RK_GATE(RK3588_PCLK_GMAC1, "pclk_gmac1", "pclk_php_root",
	    CLKGATE_CON(0, 32), 4),
	RK_GATE(RK3588_ACLK_GMAC0, "aclk_gmac0", "aclk_mmu_php",
	    CLKGATE_CON(0, 32), 10),
	RK_GATE(RK3588_ACLK_GMAC1, "aclk_gmac1", "aclk_mmu_php",
	    CLKGATE_CON(0, 32), 11),
	RK_GATE(RK3588_CLK_PMALIVE0, "clk_pmalive0", "xin24m",
	    CLKGATE_CON(0, 37), 4),
	RK_GATE(RK3588_CLK_PMALIVE1, "clk_pmalive1", "xin24m",
	    CLKGATE_CON(0, 37), 5),
	RK_GATE(RK3588_CLK_PMALIVE2, "clk_pmalive2", "xin24m",
	    CLKGATE_CON(0, 37), 6),
	RK_GATE(RK3588_ACLK_SATA0, "aclk_sata0", "aclk_mmu_php",
	    CLKGATE_CON(0, 37), 7),
	RK_GATE(RK3588_ACLK_SATA1, "aclk_sata1", "aclk_mmu_php",
	    CLKGATE_CON(0, 37), 8),
	RK_GATE(RK3588_ACLK_SATA2, "aclk_sata2", "aclk_mmu_php",
	    CLKGATE_CON(0, 37), 9),
	RK_COMPOSITE(RK3588_CLK_RXOOB0, "clk_rxoob0",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 82), __BITS(7,7), __BITS(6,0),
	    CLKGATE_CON(0, 37), __BIT(10),
	    0),
	RK_COMPOSITE(RK3588_CLK_RXOOB1, "clk_rxoob1",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 82), __BITS(15,15), __BITS(14,8),
	    CLKGATE_CON(0, 37), __BIT(11),
	    0),
	RK_COMPOSITE(RK3588_CLK_RXOOB2, "clk_rxoob2",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 83), __BITS(7,7), __BITS(6,0),
	    CLKGATE_CON(0, 37), __BIT(12),
	    0),
	RK_GATE(RK3588_ACLK_USB3OTG2, "aclk_usb3otg2", "aclk_mmu_php",
	    CLKGATE_CON(0, 35), 7),
	RK_GATE(RK3588_SUSPEND_CLK_USB3OTG2, "suspend_clk_usb3otg2", "xin24m",
	    CLKGATE_CON(0, 35), 8),
	RK_GATE(RK3588_REF_CLK_USB3OTG2, "ref_clk_usb3otg2", "xin24m",
	    CLKGATE_CON(0, 35), 9),
	RK_COMPOSITE(RK3588_CLK_UTMI_OTG2, "clk_utmi_otg2",
	    mux_150m_50m_24m_parents,
	    CLKSEL_CON(0, 84), __BITS(13,12), __BITS(11,8),
	    CLKGATE_CON(0, 35), __BIT(10),
	    0),
	RK_GATE(RK3588_PCLK_PCIE_COMBO_PIPE_PHY0, "pclk_pcie_combo_pipe_phy0",
	    "pclk_top_root",
	    CLKGATE_CON(PHP, 0), 5),
	RK_GATE(RK3588_PCLK_PCIE_COMBO_PIPE_PHY1, "pclk_pcie_combo_pipe_phy1",
	    "pclk_top_root",
	    CLKGATE_CON(PHP, 0), 6),
	RK_GATE(RK3588_PCLK_PCIE_COMBO_PIPE_PHY2, "pclk_pcie_combo_pipe_phy2",
	    "pclk_top_root",
	    CLKGATE_CON(PHP, 0), 7),
	RK_GATE(RK3588_PCLK_PCIE_COMBO_PIPE_PHY, "pclk_pcie_combo_pipe_phy",
	    "pclk_top_root",
	    CLKGATE_CON(PHP, 0), 8),
	RK_COMPOSITE(RK3588_CLK_RGA3_1_CORE, "clk_rga3_1_core",
	    gpll_cpll_aupll_spll_parents,
	    CLKSEL_CON(0, 174), __BITS(15,14), __BITS(13,9),
	    CLKGATE_CON(0, 76), __BIT(6),
	    0),
	RK_COMPOSITE(RK3588_ACLK_RGA3_ROOT, "aclk_rga3_root",
	    gpll_cpll_aupll_parents,
	    CLKSEL_CON(0, 174), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(0, 76), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_RGA3_ROOT, "hclk_rga3_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 174), __BITS(8,7),
	    CLKGATE_CON(0, 76), __BIT(1),
	    0),
	RK_GATE(RK3588_HCLK_RGA3_1, "hclk_rga3_1", "hclk_rga3_root",
	    CLKGATE_CON(0, 76), 4),
	RK_GATE(RK3588_ACLK_RGA3_1, "aclk_rga3_1", "aclk_rga3_root",
	    CLKGATE_CON(0, 76), 5),
	RK_COMPOSITE_NODIV(0, "hclk_rkvdec0_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 89), __BITS(1,0),
	    CLKGATE_CON(0, 40), __BIT(0),
	    0),
	RK_COMPOSITE(0, "aclk_rkvdec0_root",
	    gpll_cpll_aupll_spll_parents,
	    CLKSEL_CON(0, 89), __BITS(8,7), __BITS(6,2),
	    CLKGATE_CON(0, 40), __BIT(1),
	    0),
	RK_COMPOSITE(RK3588_ACLK_RKVDEC_CCU, "aclk_rkvdec_ccu",
	    gpll_cpll_aupll_spll_parents,
	    CLKSEL_CON(0, 89), __BITS(15,14), __BITS(13,9),
	    CLKGATE_CON(0, 40), __BIT(2),
	    0),
	RK_COMPOSITE(RK3588_CLK_RKVDEC0_CA, "clk_rkvdec0_ca",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 90), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 40), __BIT(7),
	    0),
	RK_COMPOSITE(RK3588_CLK_RKVDEC0_HEVC_CA, "clk_rkvdec0_hevc_ca",
	    gpll_cpll_npll_1000m_parents,
	    CLKSEL_CON(0, 90), __BITS(12,11), __BITS(10,6),
	    CLKGATE_CON(0, 40), __BIT(8),
	    0),
	RK_COMPOSITE(RK3588_CLK_RKVDEC0_CORE, "clk_rkvdec0_core",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 91), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 40), __BIT(9),
	    0),
	RK_COMPOSITE_NODIV(0, "hclk_rkvdec1_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 93), __BITS(1,0),
	    CLKGATE_CON(0, 41), __BIT(0),
	    0),
	RK_COMPOSITE(0, "aclk_rkvdec1_root",
	    gpll_cpll_aupll_npll_parents,
	    CLKSEL_CON(0, 93), __BITS(8,7), __BITS(6,2),
	    CLKGATE_CON(0, 41), __BIT(1),
	    0),
	RK_COMPOSITE(RK3588_CLK_RKVDEC1_CA, "clk_rkvdec1_ca",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 93), __BITS(14,14), __BITS(13,9),
	    CLKGATE_CON(0, 41), __BIT(6),
	    0),
	RK_COMPOSITE(RK3588_CLK_RKVDEC1_HEVC_CA, "clk_rkvdec1_hevc_ca",
	    gpll_cpll_npll_1000m_parents,
	    CLKSEL_CON(0, 94), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(0, 41), __BIT(7),
	    0),
	RK_COMPOSITE(RK3588_CLK_RKVDEC1_CORE, "clk_rkvdec1_core",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 94), __BITS(12,12), __BITS(11,7),
	    CLKGATE_CON(0, 41), __BIT(8),
	    0),
	RK_COMPOSITE_NODIV(0, "hclk_sdio_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 172), __BITS(1,0),
	    CLKGATE_CON(0, 75), __BIT(0),
	    0),
	RK_COMPOSITE(RK3588_CCLK_SRC_SDIO, "cclk_src_sdio",
	    gpll_cpll_24m_parents,
	    CLKSEL_CON(0, 172), __BITS(9,8), __BITS(7,2),
	    CLKGATE_CON(0, 75), __BIT(3),
	    0),
	RK_COMPOSITE(RK3588_ACLK_USB_ROOT, "aclk_usb_root",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 96), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 42), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_USB_ROOT, "hclk_usb_root",
	    mux_150m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 96), __BITS(7,6),
	    CLKGATE_CON(0, 42), __BIT(1),
	    0),
	RK_GATE(RK3588_SUSPEND_CLK_USB3OTG0, "suspend_clk_usb3otg0", "xin24m",
	    CLKGATE_CON(0, 42), 5),
	RK_GATE(RK3588_REF_CLK_USB3OTG0, "ref_clk_usb3otg0", "xin24m",
	    CLKGATE_CON(0, 42), 6),
	RK_GATE(RK3588_SUSPEND_CLK_USB3OTG1, "suspend_clk_usb3otg1", "xin24m",
	    CLKGATE_CON(0, 42), 8),
	RK_GATE(RK3588_REF_CLK_USB3OTG1, "ref_clk_usb3otg1", "xin24m",
	    CLKGATE_CON(0, 42), 9),
	RK_COMPOSITE(RK3588_ACLK_VDPU_ROOT, "aclk_vdpu_root",
	    gpll_cpll_aupll_parents,
	    CLKSEL_CON(0, 98), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(0, 44), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_ACLK_VDPU_LOW_ROOT, "aclk_vdpu_low_root",
	    mux_400m_200m_100m_24m_parents,
	    CLKSEL_CON(0, 98), __BITS(8,7),
	    CLKGATE_CON(0, 44), __BIT(1),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_VDPU_ROOT, "hclk_vdpu_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 98), __BITS(10,9),
	    CLKGATE_CON(0, 44), __BIT(2),
	    0),
	RK_COMPOSITE(RK3588_ACLK_JPEG_DECODER_ROOT, "aclk_jpeg_decoder_root",
	    gpll_cpll_aupll_spll_parents,
	    CLKSEL_CON(0, 99), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(0, 44), __BIT(3),
	    0),
	RK_GATE(RK3588_HCLK_IEP2P0, "hclk_iep2p0", "hclk_vdpu_root",
	    CLKGATE_CON(0, 45), 4),
	RK_COMPOSITE(RK3588_CLK_IEP2P0_CORE, "clk_iep2p0_core",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 99), __BITS(12,12), __BITS(11,7),
	    CLKGATE_CON(0, 45), __BIT(6),
	    0),
	RK_GATE(RK3588_HCLK_JPEG_ENCODER0, "hclk_jpeg_encoder0",
	    "hclk_vdpu_root",
	    CLKGATE_CON(0, 44), 11),
	RK_GATE(RK3588_HCLK_JPEG_ENCODER1, "hclk_jpeg_encoder1",
	    "hclk_vdpu_root",
	    CLKGATE_CON(0, 44), 13),
	RK_GATE(RK3588_HCLK_JPEG_ENCODER2, "hclk_jpeg_encoder2",
	    "hclk_vdpu_root",
	    CLKGATE_CON(0, 44), 15),
	RK_GATE(RK3588_HCLK_JPEG_ENCODER3, "hclk_jpeg_encoder3",
	    "hclk_vdpu_root",
	    CLKGATE_CON(0, 45), 1),
	RK_GATE(RK3588_HCLK_JPEG_DECODER, "hclk_jpeg_decoder", "hclk_vdpu_root",
	    CLKGATE_CON(0, 45), 3),
	RK_GATE(RK3588_HCLK_RGA2, "hclk_rga2", "hclk_vdpu_root",
	    CLKGATE_CON(0, 45), 7),
	RK_GATE(RK3588_ACLK_RGA2, "aclk_rga2", "aclk_vdpu_root",
	    CLKGATE_CON(0, 45), 8),
	RK_COMPOSITE(RK3588_CLK_RGA2_CORE, "clk_rga2_core",
	    gpll_cpll_npll_aupll_spll_parents,
	    CLKSEL_CON(0, 100), __BITS(7,5), __BITS(4,0),
	    CLKGATE_CON(0, 45), __BIT(9),
	    0),
	RK_GATE(RK3588_HCLK_RGA3_0, "hclk_rga3_0", "hclk_vdpu_root",
	    CLKGATE_CON(0, 45), 10),
	RK_GATE(RK3588_ACLK_RGA3_0, "aclk_rga3_0", "aclk_vdpu_root",
	    CLKGATE_CON(0, 45), 11),
	RK_COMPOSITE(RK3588_CLK_RGA3_0_CORE, "clk_rga3_0_core",
	    gpll_cpll_npll_aupll_spll_parents,
	    CLKSEL_CON(0, 100), __BITS(15,13), __BITS(12,8),
	    CLKGATE_CON(0, 45), __BIT(12),
	    0),
	RK_GATE(RK3588_HCLK_VPU, "hclk_vpu", "hclk_vdpu_root",
	    CLKGATE_CON(0, 44), 9),
	RK_COMPOSITE_NODIV(RK3588_HCLK_RKVENC1_ROOT, "hclk_rkvenc1_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 104), __BITS(1,0),
	    CLKGATE_CON(0, 48), __BIT(0),
	    0),
	RK_COMPOSITE(RK3588_ACLK_RKVENC1_ROOT, "aclk_rkvenc1_root",
	    gpll_cpll_npll_parents,
	    CLKSEL_CON(0, 104), __BITS(8,7), __BITS(6,2),
	    CLKGATE_CON(0, 48), __BIT(1),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_RKVENC0_ROOT, "hclk_rkvenc0_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 102), __BITS(1,0),
	    CLKGATE_CON(0, 47), __BIT(0),
	    0),
	RK_COMPOSITE(RK3588_ACLK_RKVENC0_ROOT, "aclk_rkvenc0_root",
	    gpll_cpll_npll_parents,
	    CLKSEL_CON(0, 102), __BITS(8,7), __BITS(6,2),
	    CLKGATE_CON(0, 47), __BIT(1),
	    0),
	RK_GATE(RK3588_HCLK_RKVENC0, "hclk_rkvenc0", "hclk_rkvenc0_root",
	    CLKGATE_CON(0, 47), 4),
	RK_GATE(RK3588_ACLK_RKVENC0, "aclk_rkvenc0", "aclk_rkvenc0_root",
	    CLKGATE_CON(0, 47), 5),
	RK_COMPOSITE(RK3588_CLK_RKVENC0_CORE, "clk_rkvenc0_core",
	    gpll_cpll_aupll_npll_parents,
	    CLKSEL_CON(0, 102), __BITS(15,14), __BITS(13,9),
	    CLKGATE_CON(0, 47), __BIT(6),
	    0),
	RK_COMPOSITE(RK3588_CLK_RKVENC1_CORE, "clk_rkvenc1_core",
	    gpll_cpll_aupll_npll_parents,
	    CLKSEL_CON(0, 104), __BITS(15,14), __BITS(13,9),
	    CLKGATE_CON(0, 48), __BIT(6),
	    0),
	RK_COMPOSITE(RK3588_ACLK_VI_ROOT, "aclk_vi_root",
	    gpll_cpll_npll_aupll_spll_parents,
	    CLKSEL_CON(0, 106), __BITS(7,5), __BITS(4,0),
	    CLKGATE_CON(0, 49), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_VI_ROOT, "hclk_vi_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 106), __BITS(9,8),
	    CLKGATE_CON(0, 49), __BIT(1),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_VI_ROOT, "pclk_vi_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(0, 106), __BITS(11,10),
	    CLKGATE_CON(0, 49), __BIT(2),
	    0),
	RK_COMPOSITE_NODIV(RK3588_ICLK_CSIHOST01, "iclk_csihost01",
	    mux_400m_200m_100m_24m_parents,
	    CLKSEL_CON(0, 108), __BITS(15,14),
	    CLKGATE_CON(0, 51), __BIT(10),
	    0),
	RK_GATE(RK3588_ICLK_CSIHOST0, "iclk_csihost0", "iclk_csihost01",
	    CLKGATE_CON(0, 51), 11),
	RK_GATE(RK3588_ICLK_CSIHOST1, "iclk_csihost1", "iclk_csihost01",
	    CLKGATE_CON(0, 51), 12),
	RK_GATE(RK3588_PCLK_CSI_HOST_0, "pclk_csi_host_0", "pclk_vi_root",
	    CLKGATE_CON(0, 50), 4),
	RK_GATE(RK3588_PCLK_CSI_HOST_1, "pclk_csi_host_1", "pclk_vi_root",
	    CLKGATE_CON(0, 50), 5),
	RK_GATE(RK3588_PCLK_CSI_HOST_2, "pclk_csi_host_2", "pclk_vi_root",
	    CLKGATE_CON(0, 50), 6),
	RK_GATE(RK3588_PCLK_CSI_HOST_3, "pclk_csi_host_3", "pclk_vi_root",
	    CLKGATE_CON(0, 50), 7),
	RK_GATE(RK3588_PCLK_CSI_HOST_4, "pclk_csi_host_4", "pclk_vi_root",
	    CLKGATE_CON(0, 50), 8),
	RK_GATE(RK3588_PCLK_CSI_HOST_5, "pclk_csi_host_5", "pclk_vi_root",
	    CLKGATE_CON(0, 50), 9),
	RK_GATE(RK3588_ACLK_FISHEYE0, "aclk_fisheye0", "aclk_vi_root",
	    CLKGATE_CON(0, 49), 14),
	RK_GATE(RK3588_HCLK_FISHEYE0, "hclk_fisheye0", "hclk_vi_root",
	    CLKGATE_CON(0, 49), 15),
	RK_COMPOSITE(RK3588_CLK_FISHEYE0_CORE, "clk_fisheye0_core",
	    gpll_cpll_aupll_spll_parents,
	    CLKSEL_CON(0, 108), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(0, 50), __BIT(0),
	    0),
	RK_GATE(RK3588_ACLK_FISHEYE1, "aclk_fisheye1", "aclk_vi_root",
	    CLKGATE_CON(0, 50), 1),
	RK_GATE(RK3588_HCLK_FISHEYE1, "hclk_fisheye1", "hclk_vi_root",
	    CLKGATE_CON(0, 50), 2),
	RK_COMPOSITE(RK3588_CLK_FISHEYE1_CORE, "clk_fisheye1_core",
	    gpll_cpll_aupll_spll_parents,
	    CLKSEL_CON(0, 108), __BITS(13,12), __BITS(11,7),
	    CLKGATE_CON(0, 50), __BIT(3),
	    0),
	RK_COMPOSITE(RK3588_CLK_ISP0_CORE, "clk_isp0_core",
	    gpll_cpll_aupll_spll_parents,
	    CLKSEL_CON(0, 107), __BITS(12,11), __BITS(10,6),
	    CLKGATE_CON(0, 49), __BIT(9),
	    0),
	RK_GATE(RK3588_CLK_ISP0_CORE_MARVIN, "clk_isp0_core_marvin",
	    "clk_isp0_core",
	    CLKGATE_CON(0, 49), 10),
	RK_GATE(RK3588_CLK_ISP0_CORE_VICAP, "clk_isp0_core_vicap",
	    "clk_isp0_core",
	    CLKGATE_CON(0, 49), 11),
	RK_GATE(RK3588_ACLK_ISP0, "aclk_isp0", "aclk_vi_root",
	    CLKGATE_CON(0, 49), 12),
	RK_GATE(RK3588_HCLK_ISP0, "hclk_isp0", "hclk_vi_root",
	    CLKGATE_CON(0, 49), 13),
	RK_COMPOSITE(RK3588_DCLK_VICAP, "dclk_vicap",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 107), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 49), __BIT(6),
	    0),
	RK_GATE(RK3588_ACLK_VICAP, "aclk_vicap", "aclk_vi_root",
	    CLKGATE_CON(0, 49), 7),
	RK_GATE(RK3588_HCLK_VICAP, "hclk_vicap", "hclk_vi_root",
	    CLKGATE_CON(0, 49), 8),
	RK_COMPOSITE(RK3588_ACLK_VO0_ROOT, "aclk_vo0_root",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 116), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 55), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_VO0_ROOT, "hclk_vo0_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 116), __BITS(7,6),
	    CLKGATE_CON(0, 55), __BIT(1),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_VO0_S_ROOT, "hclk_vo0_s_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 116), __BITS(9,8),
	    CLKGATE_CON(0, 55), __BIT(2),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_VO0_ROOT, "pclk_vo0_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(0, 116), __BITS(11,10),
	    CLKGATE_CON(0, 55), __BIT(3),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_VO0_S_ROOT, "pclk_vo0_s_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(0, 116), __BITS(13,12),
	    CLKGATE_CON(0, 55), __BIT(4),
	    0),
	RK_GATE(RK3588_PCLK_DP0, "pclk_dp0", "pclk_vo0_root",
	    CLKGATE_CON(0, 56), 4),
	RK_GATE(RK3588_PCLK_DP1, "pclk_dp1", "pclk_vo0_root",
	    CLKGATE_CON(0, 56), 5),
	RK_GATE(RK3588_PCLK_S_DP0, "pclk_s_dp0", "pclk_vo0_s_root",
	    CLKGATE_CON(0, 56), 6),
	RK_GATE(RK3588_PCLK_S_DP1, "pclk_s_dp1", "pclk_vo0_s_root",
	    CLKGATE_CON(0, 56), 7),
	RK_GATE(RK3588_CLK_DP0, "clk_dp0", "aclk_vo0_root",
	    CLKGATE_CON(0, 56), 8),
	RK_GATE(RK3588_CLK_DP1, "clk_dp1", "aclk_vo0_root",
	    CLKGATE_CON(0, 56), 9),
	RK_GATE(RK3588_HCLK_HDCP_KEY0, "hclk_hdcp_key0", "hclk_vo0_s_root",
	    CLKGATE_CON(0, 55), 11),
	RK_GATE(RK3588_PCLK_HDCP0, "pclk_hdcp0", "pclk_vo0_root",
	    CLKGATE_CON(0, 55), 14),
	RK_GATE(RK3588_ACLK_TRNG0, "aclk_trng0", "aclk_vo0_root",
	    CLKGATE_CON(0, 56), 0),
	RK_GATE(RK3588_PCLK_TRNG0, "pclk_trng0", "pclk_vo0_root",
	    CLKGATE_CON(0, 56), 1),
	RK_GATE(RK3588_PCLK_VO0GRF, "pclk_vo0grf", "pclk_vo0_root",
	    CLKGATE_CON(0, 55), 10),
	RK_COMPOSITE(RK3588_CLK_I2S4_8CH_TX_SRC, "clk_i2s4_8ch_tx_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 118), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 56), __BIT(11),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S4_8CH_TX_FRAC, "clk_i2s4_8ch_tx_frac",
	    "clk_i2s4_8ch_tx_src",
	    CLKGATE_CON(0, 56),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S4_8CH_TX, "clk_i2s4_8ch_tx",
	    clk_i2s4_8ch_tx_parents,
	    CLKSEL_CON(0, 120), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S4_8CH_TX, "mclk_i2s4_8ch_tx", "clk_i2s4_8ch_tx",
	    CLKGATE_CON(0, 56), 13),
	RK_COMPOSITE(RK3588_CLK_I2S8_8CH_TX_SRC, "clk_i2s8_8ch_tx_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 120), __BITS(8,8), __BITS(7,3),
	    CLKGATE_CON(0, 56), __BIT(15),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S8_8CH_TX_FRAC, "clk_i2s8_8ch_tx_frac",
	    "clk_i2s8_8ch_tx_src",
	    CLKGATE_CON(0, 57),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S8_8CH_TX, "clk_i2s8_8ch_tx",
	    clk_i2s8_8ch_tx_parents,
	    CLKSEL_CON(0, 122), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S8_8CH_TX, "mclk_i2s8_8ch_tx", "clk_i2s8_8ch_tx",
	    CLKGATE_CON(0, 57), 1),
	RK_COMPOSITE(RK3588_CLK_SPDIF2_DP0_SRC, "clk_spdif2_dp0_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 122), __BITS(8,8), __BITS(7,3),
	    CLKGATE_CON(0, 57), __BIT(3),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_SPDIF2_DP0_FRAC, "clk_spdif2_dp0_frac",
	    "clk_spdif2_dp0_src",
	    CLKGATE_CON(0, 57),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_SPDIF2_DP0, "clk_spdif2_dp0", clk_spdif2_dp0_parents,
	    CLKSEL_CON(0, 124), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_SPDIF2_DP0, "mclk_spdif2_dp0", "clk_spdif2_dp0",
	    CLKGATE_CON(0, 57), 5),
	RK_GATE(RK3588_MCLK_SPDIF2, "mclk_spdif2", "clk_spdif2_dp0",
	    CLKGATE_CON(0, 57), 6),
	RK_COMPOSITE(RK3588_CLK_SPDIF5_DP1_SRC, "clk_spdif5_dp1_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 124), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 57), __BIT(8),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_SPDIF5_DP1_FRAC, "clk_spdif5_dp1_frac",
	    "clk_spdif5_dp1_src",
	    CLKGATE_CON(0, 57),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_SPDIF5_DP1, "clk_spdif5_dp1", clk_spdif5_dp1_parents,
	    CLKSEL_CON(0, 126), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_SPDIF5_DP1, "mclk_spdif5_dp1", "clk_spdif5_dp1",
	    CLKGATE_CON(0, 57), 10),
	RK_GATE(RK3588_MCLK_SPDIF5, "mclk_spdif5", "clk_spdif5_dp1",
	    CLKGATE_CON(0, 57), 11),
	RK_COMPOSITE_NOMUX(RK3588_CLK_AUX16M_0, "clk_aux16m_0", "gpll",
	    CLKSEL_CON(0, 117), __BITS(7,0),
	    CLKGATE_CON(0, 56), __BIT(2),
	    0),
	RK_COMPOSITE_NOMUX(RK3588_CLK_AUX16M_1, "clk_aux16m_1", "gpll",
	    CLKSEL_CON(0, 117), __BITS(15,8),
	    CLKGATE_CON(0, 56), __BIT(3),
	    0),
	RK_COMPOSITE_HALF(RK3588_CLK_HDMITRX_REFSRC, "clk_hdmitrx_refsrc",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 157), __BITS(7,7),
	    __BITS(6,2),
	    CLKGATE_CON(0, 65), __BIT(9),
	    0),
	RK_COMPOSITE(RK3588_ACLK_HDCP1_ROOT, "aclk_hdcp1_root",
	    aclk_hdcp1_root_parents,
	    CLKSEL_CON(0, 128), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(0, 59), __BIT(0),
	    0),
	RK_COMPOSITE(RK3588_ACLK_HDMIRX_ROOT, "aclk_hdmirx_root",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 128), __BITS(12,12), __BITS(11,7),
	    CLKGATE_CON(0, 59), __BIT(1),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_VO1_ROOT, "hclk_vo1_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 128), __BITS(14,13),
	    CLKGATE_CON(0, 59), __BIT(2),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_VO1_S_ROOT, "hclk_vo1_s_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 129), __BITS(1,0),
	    CLKGATE_CON(0, 59), __BIT(3),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_VO1_ROOT, "pclk_vo1_root",
	    mux_150m_100m_24m_parents,
	    CLKSEL_CON(0, 129), __BITS(3,2),
	    CLKGATE_CON(0, 59), __BIT(4),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_VO1_S_ROOT, "pclk_vo1_s_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(0, 129), __BITS(5,4),
	    CLKGATE_CON(0, 59), __BIT(5),
	    0),
	RK_COMPOSITE(RK3588_ACLK_VOP_ROOT, "aclk_vop_root",
	    gpll_cpll_dmyaupll_npll_spll_parents,
	    CLKSEL_CON(0, 110), __BITS(7,5), __BITS(4,0),
	    CLKGATE_CON(0, 52), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_ACLK_VOP_LOW_ROOT, "aclk_vop_low_root",
	    mux_400m_200m_100m_24m_parents,
	    CLKSEL_CON(0, 110), __BITS(9,8),
	    CLKGATE_CON(0, 52), __BIT(1),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_VOP_ROOT, "hclk_vop_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 110), __BITS(11,10),
	    CLKGATE_CON(0, 52), __BIT(2),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_VOP_ROOT, "pclk_vop_root",
	    mux_100m_50m_24m_parents,
	    CLKSEL_CON(0, 110), __BITS(13,12),
	    CLKGATE_CON(0, 52), __BIT(3),
	    0),
	RK_COMPOSITE(RK3588_ACLK_VO1USB_TOP_ROOT, "aclk_vo1usb_top_root",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 170), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(0, 74), __BIT(0),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_VO1USB_TOP_ROOT, "hclk_vo1usb_top_root",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 170), __BITS(7,6),
	    CLKGATE_CON(0, 74), __BIT(2),
	    0),
	RK_MUX(RK3588_ACLK_VOP_SUB_SRC, "aclk_vop_sub_src",
	    aclk_vop_sub_src_parents,
	    CLKSEL_CON(0, 115), __BITS(9,9)),
	RK_GATE(RK3588_PCLK_EDP0, "pclk_edp0", "pclk_vo1_root",
	    CLKGATE_CON(0, 62), 0),
	RK_GATE(RK3588_CLK_EDP0_24M, "clk_edp0_24m", "xin24m",
	    CLKGATE_CON(0, 62), 1),
	RK_COMPOSITE_NODIV(RK3588_CLK_EDP0_200M, "clk_edp0_200m",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 140), __BITS(2,1),
	    CLKGATE_CON(0, 62), __BIT(2),
	    0),
	RK_GATE(RK3588_PCLK_EDP1, "pclk_edp1", "pclk_vo1_root",
	    CLKGATE_CON(0, 62), 3),
	RK_GATE(RK3588_CLK_EDP1_24M, "clk_edp1_24m", "xin24m",
	    CLKGATE_CON(0, 62), 4),
	RK_COMPOSITE_NODIV(RK3588_CLK_EDP1_200M, "clk_edp1_200m",
	    mux_200m_100m_50m_24m_parents,
	    CLKSEL_CON(0, 140), __BITS(4,3),
	    CLKGATE_CON(0, 62), __BIT(5),
	    0),
	RK_GATE(RK3588_HCLK_HDCP_KEY1, "hclk_hdcp_key1", "hclk_vo1_s_root",
	    CLKGATE_CON(0, 60), 4),
	RK_GATE(RK3588_PCLK_HDCP1, "pclk_hdcp1", "pclk_vo1_root",
	    CLKGATE_CON(0, 60), 7),
	RK_GATE(RK3588_ACLK_HDMIRX, "aclk_hdmirx", "aclk_hdmirx_root",
	    CLKGATE_CON(0, 61), 9),
	RK_GATE(RK3588_PCLK_HDMIRX, "pclk_hdmirx", "pclk_vo1_root",
	    CLKGATE_CON(0, 61), 10),
	RK_GATE(RK3588_CLK_HDMIRX_REF, "clk_hdmirx_ref", "aclk_hdcp1_root",
	    CLKGATE_CON(0, 61), 11),
	RK_COMPOSITE(RK3588_CLK_HDMIRX_AUD_SRC, "clk_hdmirx_aud_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 138), __BITS(8,8), __BITS(7,0),
	    CLKGATE_CON(0, 61), __BIT(12),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_HDMIRX_AUD_FRAC, "clk_hdmirx_aud_frac",
	    "clk_hdmirx_aud_src",
	    CLKGATE_CON(0, 61),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_GATE(RK3588_CLK_HDMIRX_AUD, "clk_hdmirx_aud", "clk_hdmirx_aud_mux",
	    CLKGATE_CON(0, 61), 14),
	RK_GATE(RK3588_PCLK_HDMITX0, "pclk_hdmitx0", "pclk_vo1_root",
	    CLKGATE_CON(0, 60), 11),
	RK_COMPOSITE(RK3588_CLK_HDMITX0_EARC, "clk_hdmitx0_earc",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 133), __BITS(6,6), __BITS(5,1),
	    CLKGATE_CON(0, 60), __BIT(15),
	    0),
	RK_GATE(RK3588_CLK_HDMITX0_REF, "clk_hdmitx0_ref", "aclk_hdcp1_root",
	    CLKGATE_CON(0, 61), 0),
	RK_GATE(RK3588_PCLK_HDMITX1, "pclk_hdmitx1", "pclk_vo1_root",
	    CLKGATE_CON(0, 61), 2),
	RK_COMPOSITE(RK3588_CLK_HDMITX1_EARC, "clk_hdmitx1_earc",
	    gpll_cpll_parents,
	    CLKSEL_CON(0, 136), __BITS(6,6), __BITS(5,1),
	    CLKGATE_CON(0, 61), __BIT(6),
	    0),
	RK_GATE(RK3588_CLK_HDMITX1_REF, "clk_hdmitx1_ref", "aclk_hdcp1_root",
	    CLKGATE_CON(0, 61), 7),
	RK_GATE(RK3588_ACLK_TRNG1, "aclk_trng1", "aclk_hdcp1_root",
	    CLKGATE_CON(0, 60), 9),
	RK_GATE(RK3588_PCLK_TRNG1, "pclk_trng1", "pclk_vo1_root",
	    CLKGATE_CON(0, 60), 10),
	RK_GATE(0, "pclk_vo1grf", "pclk_vo1_root",
	    CLKGATE_CON(0, 59), 12),
	RK_GATE(RK3588_PCLK_S_EDP0, "pclk_s_edp0", "pclk_vo1_s_root",
	    CLKGATE_CON(0, 59), 14),
	RK_GATE(RK3588_PCLK_S_EDP1, "pclk_s_edp1", "pclk_vo1_s_root",
	    CLKGATE_CON(0, 59), 15),
	RK_GATE(RK3588_PCLK_S_HDMIRX, "pclk_s_hdmirx", "pclk_vo1_s_root",
	    CLKGATE_CON(0, 65), 8),
	RK_COMPOSITE(RK3588_CLK_I2S10_8CH_RX_SRC, "clk_i2s10_8ch_rx_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 155), __BITS(8,8), __BITS(7,3),
	    CLKGATE_CON(0, 65), __BIT(5),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S10_8CH_RX_FRAC, "clk_i2s10_8ch_rx_frac",
	    "clk_i2s10_8ch_rx_src",
	    CLKGATE_CON(0, 65),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S10_8CH_RX, "clk_i2s10_8ch_rx",
	    clk_i2s10_8ch_rx_parents,
	    CLKSEL_CON(0, 157), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S10_8CH_RX, "mclk_i2s10_8ch_rx",
	    "clk_i2s10_8ch_rx",
	    CLKGATE_CON(0, 65), 7),
	RK_COMPOSITE(RK3588_CLK_I2S7_8CH_RX_SRC, "clk_i2s7_8ch_rx_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 129), __BITS(11,11), __BITS(10,6),
	    CLKGATE_CON(0, 60), __BIT(1),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S7_8CH_RX_FRAC, "clk_i2s7_8ch_rx_frac",
	    "clk_i2s7_8ch_rx_src",
	    CLKGATE_CON(0, 60),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S7_8CH_RX, "clk_i2s7_8ch_rx",
	    clk_i2s7_8ch_rx_parents,
	    CLKSEL_CON(0, 131), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S7_8CH_RX, "mclk_i2s7_8ch_rx", "clk_i2s7_8ch_rx",
	    CLKGATE_CON(0, 60), 3),
	RK_COMPOSITE(RK3588_CLK_I2S9_8CH_RX_SRC, "clk_i2s9_8ch_rx_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 153), __BITS(12,12), __BITS(11,7),
	    CLKGATE_CON(0, 65), __BIT(1),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S9_8CH_RX_FRAC, "clk_i2s9_8ch_rx_frac",
	    "clk_i2s9_8ch_rx_src",
	    CLKGATE_CON(0, 65),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S9_8CH_RX, "clk_i2s9_8ch_rx",
	    clk_i2s9_8ch_rx_parents,
	    CLKSEL_CON(0, 155), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S9_8CH_RX, "mclk_i2s9_8ch_rx", "clk_i2s9_8ch_rx",
	    CLKGATE_CON(0, 65), 3),
	RK_COMPOSITE(RK3588_CLK_I2S5_8CH_TX_SRC, "clk_i2s5_8ch_tx_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 140), __BITS(10,10), __BITS(9,5),
	    CLKGATE_CON(0, 62), __BIT(6),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S5_8CH_TX_FRAC, "clk_i2s5_8ch_tx_frac",
	    "clk_i2s5_8ch_tx_src",
	    CLKGATE_CON(0, 62),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S5_8CH_TX, "clk_i2s5_8ch_tx",
	    clk_i2s5_8ch_tx_parents,
	    CLKSEL_CON(0, 142), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S5_8CH_TX, "mclk_i2s5_8ch_tx", "clk_i2s5_8ch_tx",
	    CLKGATE_CON(0, 62), 8),
	RK_COMPOSITE(RK3588_CLK_I2S6_8CH_TX_SRC, "clk_i2s6_8ch_tx_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 144), __BITS(8,8), __BITS(7,3),
	    CLKGATE_CON(0, 62), __BIT(13),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S6_8CH_TX_FRAC, "clk_i2s6_8ch_tx_frac",
	    "clk_i2s6_8ch_tx_src",
	    CLKGATE_CON(0, 62),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S6_8CH_TX, "clk_i2s6_8ch_tx",
	    clk_i2s6_8ch_tx_parents,
	    CLKSEL_CON(0, 146), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S6_8CH_TX, "mclk_i2s6_8ch_tx", "clk_i2s6_8ch_tx",
	    CLKGATE_CON(0, 62), 15),
	RK_COMPOSITE(RK3588_CLK_I2S6_8CH_RX_SRC, "clk_i2s6_8ch_rx_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 146), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 63), __BIT(0),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S6_8CH_RX_FRAC, "clk_i2s6_8ch_rx_frac",
	    "clk_i2s6_8ch_rx_src",
	    CLKGATE_CON(0, 63),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S6_8CH_RX, "clk_i2s6_8ch_rx",
	    clk_i2s6_8ch_rx_parents,
	    CLKSEL_CON(0, 148), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S6_8CH_RX, "mclk_i2s6_8ch_rx", "clk_i2s6_8ch_rx",
	    CLKGATE_CON(0, 63), 2),
	RK_MUX(RK3588_I2S6_8CH_MCLKOUT, "i2s6_8ch_mclkout",
	    i2s6_8ch_mclkout_parents,
	    CLKSEL_CON(0, 148), __BITS(3,2)),
	RK_COMPOSITE(RK3588_CLK_SPDIF3_SRC, "clk_spdif3_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 148), __BITS(9,9), __BITS(8,4),
	    CLKGATE_CON(0, 63), __BIT(5),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_SPDIF3_FRAC, "clk_spdif3_frac",
	    "clk_spdif3_src",
	    CLKGATE_CON(0, 63),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_SPDIF3, "clk_spdif3", clk_spdif3_parents,
	    CLKSEL_CON(0, 150), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_SPDIF3, "mclk_spdif3", "clk_spdif3",
	    CLKGATE_CON(0, 63), 7),
	RK_COMPOSITE(RK3588_CLK_SPDIF4_SRC, "clk_spdif4_src",
	    gpll_aupll_parents,
	    CLKSEL_CON(0, 150), __BITS(7,7), __BITS(6,2),
	    CLKGATE_CON(0, 63), __BIT(9),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_SPDIF4_FRAC, "clk_spdif4_frac",
	    "clk_spdif4_src",
	    CLKGATE_CON(0, 63),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_SPDIF4, "clk_spdif4", clk_spdif4_parents,
	    CLKSEL_CON(0, 152), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_SPDIF4, "mclk_spdif4", "clk_spdif4",
	    CLKGATE_CON(0, 63), 11),
	RK_COMPOSITE(RK3588_MCLK_SPDIFRX0, "mclk_spdifrx0",
	    gpll_cpll_aupll_parents,
	    CLKSEL_CON(0, 152), __BITS(8,7), __BITS(6,2),
	    CLKGATE_CON(0, 63), __BIT(13),
	    0),
	RK_COMPOSITE(RK3588_MCLK_SPDIFRX1, "mclk_spdifrx1",
	    gpll_cpll_aupll_parents,
	    CLKSEL_CON(0, 152), __BITS(15,14), __BITS(13,9),
	    CLKGATE_CON(0, 63), __BIT(15),
	    0),
	RK_COMPOSITE(RK3588_MCLK_SPDIFRX2, "mclk_spdifrx2",
	    gpll_cpll_aupll_parents,
	    CLKSEL_CON(0, 153), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(0, 64), __BIT(1),
	    0),
	RK_GATE(RK3588_CLK_HDMIHDP0, "clk_hdmihdp0", "xin24m",
	    CLKGATE_CON(0, 73), 12),
	RK_GATE(RK3588_CLK_HDMIHDP1, "clk_hdmihdp1", "xin24m",
	    CLKGATE_CON(0, 73), 13),
	RK_GATE(RK3588_PCLK_HDPTX0, "pclk_hdptx0", "pclk_top_root",
	    CLKGATE_CON(0, 72), 5),
	RK_GATE(RK3588_PCLK_HDPTX1, "pclk_hdptx1", "pclk_top_root",
	    CLKGATE_CON(0, 72), 6),
	RK_GATE(RK3588_PCLK_USBDPPHY0, "pclk_usbdpphy0", "pclk_top_root",
	    CLKGATE_CON(0, 72), 2),
	RK_GATE(RK3588_PCLK_USBDPPHY1, "pclk_usbdpphy1", "pclk_top_root",
	    CLKGATE_CON(0, 72), 4),
	RK_GATE(RK3588_HCLK_VOP, "hclk_vop", "hclk_vop_root",
	    CLKGATE_CON(0, 52), 8),
	RK_GATE(RK3588_ACLK_VOP, "aclk_vop", "aclk_vop_sub_src",
	    CLKGATE_CON(0, 52), 9),
	RK_COMPOSITE(RK3588_DCLK_VOP0_SRC, "dclk_vop0_src",
	    gpll_cpll_v0pll_aupll_parents,
	    CLKSEL_CON(0, 111), __BITS(8,7), __BITS(6,0),
	    CLKGATE_CON(0, 52), __BIT(10),
	    0),
	RK_COMPOSITE(RK3588_DCLK_VOP1_SRC, "dclk_vop1_src",
	    gpll_cpll_v0pll_aupll_parents,
	    CLKSEL_CON(0, 111), __BITS(15,14), __BITS(13,9),
	    CLKGATE_CON(0, 52), __BIT(11),
	    0),
	RK_COMPOSITE(RK3588_DCLK_VOP2_SRC, "dclk_vop2_src",
	    gpll_cpll_v0pll_aupll_parents,
	    CLKSEL_CON(0, 112), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(0, 52), __BIT(12),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_COMPOSITE_NODIV(RK3588_DCLK_VOP0, "dclk_vop0",
	    dclk_vop0_parents,
	    CLKSEL_CON(0, 112), __BITS(8,7),
	    CLKGATE_CON(0, 52), __BIT(13),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_COMPOSITE_NODIV(RK3588_DCLK_VOP1, "dclk_vop1",
	    dclk_vop1_parents,
	    CLKSEL_CON(0, 112), __BITS(10,9),
	    CLKGATE_CON(0, 53), __BIT(0),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_COMPOSITE_NODIV(RK3588_DCLK_VOP2, "dclk_vop2",
	    dclk_vop2_parents,
	    CLKSEL_CON(0, 112), __BITS(12,11),
	    CLKGATE_CON(0, 53), __BIT(1),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_COMPOSITE(RK3588_DCLK_VOP3, "dclk_vop3",
	    gpll_cpll_v0pll_aupll_parents,
	    CLKSEL_CON(0, 113), __BITS(8,7), __BITS(6,0),
	    CLKGATE_CON(0, 53), __BIT(2),
	    0),
	RK_GATE(RK3588_PCLK_DSIHOST0, "pclk_dsihost0", "pclk_vop_root",
	    CLKGATE_CON(0, 53), 4),
	RK_GATE(RK3588_PCLK_DSIHOST1, "pclk_dsihost1", "pclk_vop_root",
	    CLKGATE_CON(0, 53), 5),
	RK_COMPOSITE(RK3588_CLK_DSIHOST0, "clk_dsihost0",
	    gpll_cpll_v0pll_spll_parents,
	    CLKSEL_CON(0, 114), __BITS(8,7), __BITS(6,0),
	    CLKGATE_CON(0, 53), __BIT(6),
	    0),
	RK_COMPOSITE(RK3588_CLK_DSIHOST1, "clk_dsihost1",
	    gpll_cpll_v0pll_spll_parents,
	    CLKSEL_CON(0, 115), __BITS(8,7), __BITS(6,0),
	    CLKGATE_CON(0, 53), __BIT(7),
	    0),
	RK_GATE(RK3588_CLK_VOP_PMU, "clk_vop_pmu", "xin24m",
	    CLKGATE_CON(0, 53), 8),
	RK_GATE(RK3588_ACLK_VOP_DOBY, "aclk_vop_doby", "aclk_vop_root",
	    CLKGATE_CON(0, 53), 10),
	RK_GATE(RK3588_CLK_USBDP_PHY0_IMMORTAL, "clk_usbdp_phy0_immortal",
	    "xin24m",
	    CLKGATE_CON(0, 2), 8),
	RK_GATE(RK3588_CLK_USBDP_PHY1_IMMORTAL, "clk_usbdp_phy1_immortal",
	    "xin24m",
	    CLKGATE_CON(0, 2), 15),
	RK_GATE(RK3588_CLK_REF_PIPE_PHY0_OSC_SRC, "clk_ref_pipe_phy0_osc_src",
	    "xin24m",
	    CLKGATE_CON(0, 77), 0),
	RK_GATE(RK3588_CLK_REF_PIPE_PHY1_OSC_SRC, "clk_ref_pipe_phy1_osc_src",
	    "xin24m",
	    CLKGATE_CON(0, 77), 1),
	RK_GATE(RK3588_CLK_REF_PIPE_PHY2_OSC_SRC, "clk_ref_pipe_phy2_osc_src",
	    "xin24m",
	    CLKGATE_CON(0, 77), 2),
	RK_COMPOSITE_NOMUX(RK3588_CLK_REF_PIPE_PHY0_PLL_SRC,
	    "clk_ref_pipe_phy0_pll_src", "ppll",
	    CLKSEL_CON(0, 176), __BITS(5,0),
	    CLKGATE_CON(0, 77), __BIT(3),
	    0),
	RK_COMPOSITE_NOMUX(RK3588_CLK_REF_PIPE_PHY1_PLL_SRC,
	    "clk_ref_pipe_phy1_pll_src", "ppll",
	    CLKSEL_CON(0, 176), __BITS(11,6),
	    CLKGATE_CON(0, 77), __BIT(4),
	    0),
	RK_COMPOSITE_NOMUX(RK3588_CLK_REF_PIPE_PHY2_PLL_SRC,
	    "clk_ref_pipe_phy2_pll_src", "ppll",
	    CLKSEL_CON(0, 177), __BITS(5,0),
	    CLKGATE_CON(0, 77), __BIT(5),
	    0),
	RK_MUX(RK3588_CLK_REF_PIPE_PHY0, "clk_ref_pipe_phy0",
	    clk_ref_pipe_phy0_parents,
	    CLKSEL_CON(0, 177), __BITS(6,6)),
	RK_MUX(RK3588_CLK_REF_PIPE_PHY1, "clk_ref_pipe_phy1",
	    clk_ref_pipe_phy1_parents,
	    CLKSEL_CON(0, 177), __BITS(7,7)),
	RK_MUX(RK3588_CLK_REF_PIPE_PHY2, "clk_ref_pipe_phy2",
	    clk_ref_pipe_phy2_parents,
	    CLKSEL_CON(0, 177), __BITS(8,8)),
	RK_COMPOSITE(RK3588_CLK_PMU1_300M_SRC, "clk_pmu1_300m_src",
	    pmu_300m_24m_parents,
	    CLKSEL_CON(PMU, 0), __BITS(15,15), __BITS(14,10),
	    CLKGATE_CON(PMU, 0), __BIT(3),
	    0),
	RK_COMPOSITE(RK3588_CLK_PMU1_400M_SRC, "clk_pmu1_400m_src",
	    pmu_400m_24m_parents,
	    CLKSEL_CON(PMU, 1), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(PMU, 0), __BIT(4),
	    0),
	RK_COMPOSITE_NOMUX(RK3588_CLK_PMU1_50M_SRC, "clk_pmu1_50m_src",
	    "clk_pmu1_400m_src",
	    CLKSEL_CON(PMU, 0), __BITS(3,0),
	    CLKGATE_CON(PMU, 0), __BIT(0),
	    0),
	RK_COMPOSITE_NOMUX(RK3588_CLK_PMU1_100M_SRC, "clk_pmu1_100m_src",
	    "clk_pmu1_400m_src",
	    CLKSEL_CON(PMU, 0), __BITS(6,4),
	    CLKGATE_CON(PMU, 0), __BIT(1),
	    0),
	RK_COMPOSITE_NOMUX(RK3588_CLK_PMU1_200M_SRC, "clk_pmu1_200m_src",
	    "clk_pmu1_400m_src",
	    CLKSEL_CON(PMU, 0), __BITS(9,7),
	    CLKGATE_CON(PMU, 0), __BIT(2),
	    0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_PMU1_ROOT, "hclk_pmu1_root",
	    hclk_pmu1_root_parents,
	    CLKSEL_CON(PMU, 1), __BITS(7,6),
	    CLKGATE_CON(PMU, 0), __BIT(5),
	    0),
	RK_COMPOSITE_NODIV(RK3588_PCLK_PMU1_ROOT, "pclk_pmu1_root",
	    pmu_100m_50m_24m_src_parents,
	    CLKSEL_CON(PMU, 1), __BITS(9,8),
	    CLKGATE_CON(PMU, 0), __BIT(7),
	    0),
	RK_GATE(RK3588_PCLK_PMU0_ROOT, "pclk_pmu0_root", "pclk_pmu1_root",
	    CLKGATE_CON(PMU, 5), 0),
	RK_COMPOSITE_NODIV(RK3588_HCLK_PMU_CM0_ROOT, "hclk_pmu_cm0_root",
	    hclk_pmu_cm0_root_parents,
	    CLKSEL_CON(PMU, 1), __BITS(11,10),
	    CLKGATE_CON(PMU, 0), __BIT(8),
	    0),
	RK_GATE(RK3588_CLK_PMU0, "clk_pmu0", "xin24m",
	    CLKGATE_CON(PMU, 5), 1),
	RK_GATE(RK3588_PCLK_PMU0, "pclk_pmu0", "pclk_pmu0_root",
	    CLKGATE_CON(PMU, 5), 2),
	RK_GATE(RK3588_PCLK_PMU0IOC, "pclk_pmu0ioc", "pclk_pmu0_root",
	    CLKGATE_CON(PMU, 5), 4),
	RK_GATE(RK3588_PCLK_GPIO0, "pclk_gpio0", "pclk_pmu0_root",
	    CLKGATE_CON(PMU, 5), 5),
	RK_COMPOSITE_NODIV(RK3588_DBCLK_GPIO0, "dbclk_gpio0",
	    mux_24m_32k_parents,
	    CLKSEL_CON(PMU, 17), __BITS(0,0),
	    CLKGATE_CON(PMU, 5), __BIT(6),
	    0),
	RK_GATE(RK3588_PCLK_I2C0, "pclk_i2c0", "pclk_pmu0_root",
	    CLKGATE_CON(PMU, 2), 1),
	RK_COMPOSITE_NODIV(RK3588_CLK_I2C0, "clk_i2c0",
	    pmu_200m_100m_parents,
	    CLKSEL_CON(PMU, 3), __BITS(6,6),
	    CLKGATE_CON(PMU, 2), __BIT(2),
	    0),
	RK_GATE(RK3588_HCLK_I2S1_8CH, "hclk_i2s1_8ch", "hclk_pmu1_root",
	    CLKGATE_CON(PMU, 2), 7),
	RK_COMPOSITE_NOMUX(RK3588_CLK_I2S1_8CH_TX_SRC, "clk_i2s1_8ch_tx_src",
	    "cpll",
	    CLKSEL_CON(PMU, 5), __BITS(6,2),
	    CLKGATE_CON(PMU, 2), __BIT(8),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S1_8CH_TX_FRAC, "clk_i2s1_8ch_tx_frac",
	    "clk_i2s1_8ch_tx_src",
	    CLKGATE_CON(PMU, 2),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S1_8CH_TX, "clk_i2s1_8ch_tx",
	    clk_i2s1_8ch_tx_parents,
	    CLKSEL_CON(PMU, 7), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S1_8CH_TX, "mclk_i2s1_8ch_tx", "clk_i2s1_8ch_tx",
	    CLKGATE_CON(PMU, 2), 10),
	RK_COMPOSITE_NOMUX(RK3588_CLK_I2S1_8CH_RX_SRC, "clk_i2s1_8ch_rx_src",
	    "cpll",
	    CLKSEL_CON(PMU, 7), __BITS(6,2),
	    CLKGATE_CON(PMU, 2), __BIT(11),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_I2S1_8CH_RX_FRAC, "clk_i2s1_8ch_rx_frac",
	    "clk_i2s1_8ch_rx_src",
	    CLKGATE_CON(PMU, 2),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_I2S1_8CH_RX, "clk_i2s1_8ch_rx",
	    clk_i2s1_8ch_rx_parents,
	    CLKSEL_CON(PMU, 9), __BITS(1,0)),
	RK_GATE(RK3588_MCLK_I2S1_8CH_RX, "mclk_i2s1_8ch_rx", "clk_i2s1_8ch_rx",
	    CLKGATE_CON(PMU, 2), 13),
	RK_MUX(RK3588_I2S1_8CH_MCLKOUT, "i2s1_8ch_mclkout",
	    i2s1_8ch_mclkout_parents,
	    CLKSEL_CON(PMU, 9), __BITS(3,2)),
	RK_GATE(RK3588_PCLK_PMU1, "pclk_pmu1", "pclk_pmu0_root",
	    CLKGATE_CON(PMU, 1), 0),
	RK_GATE(RK3588_CLK_DDR_FAIL_SAFE, "clk_ddr_fail_safe", "clk_pmu0",
	    CLKGATE_CON(PMU, 1), 1),
	RK_GATE(RK3588_CLK_PMU1, "clk_pmu1", "clk_pmu0",
	    CLKGATE_CON(PMU, 1), 3),
	RK_GATE(RK3588_HCLK_PDM0, "hclk_pdm0", "hclk_pmu1_root",
	    CLKGATE_CON(PMU, 2), 14),
	RK_COMPOSITE_NODIV(RK3588_MCLK_PDM0, "mclk_pdm0",
	    mclk_pdm0_parents,
	    CLKSEL_CON(PMU, 9), __BITS(4,4),
	    CLKGATE_CON(PMU, 2), __BIT(15),
	    0),
	RK_GATE(RK3588_HCLK_VAD, "hclk_vad", "hclk_pmu1_root",
	    CLKGATE_CON(PMU, 3), 0),
	RK_GATE(RK3588_FCLK_PMU_CM0_CORE, "fclk_pmu_cm0_core",
	    "hclk_pmu_cm0_root",
	    CLKGATE_CON(PMU, 0), 13),
	RK_COMPOSITE(RK3588_CLK_PMU_CM0_RTC, "clk_pmu_cm0_rtc",
	    mux_24m_32k_parents,
	    CLKSEL_CON(PMU, 2), __BITS(5,5), __BITS(4,0),
	    CLKGATE_CON(PMU, 0), __BIT(15),
	    0),
	RK_GATE(RK3588_PCLK_PMU1_IOC, "pclk_pmu1_ioc", "pclk_pmu0_root",
	    CLKGATE_CON(PMU, 1), 5),
	RK_GATE(RK3588_PCLK_PMU1PWM, "pclk_pmu1pwm", "pclk_pmu0_root",
	    CLKGATE_CON(PMU, 1), 12),
	RK_COMPOSITE_NODIV(RK3588_CLK_PMU1PWM, "clk_pmu1pwm",
	    pmu_100m_50m_24m_src_parents,
	    CLKSEL_CON(PMU, 2), __BITS(10,9),
	    CLKGATE_CON(PMU, 1), __BIT(13),
	    0),
	RK_GATE(RK3588_CLK_PMU1PWM_CAPTURE, "clk_pmu1pwm_capture", "xin24m",
	    CLKGATE_CON(PMU, 1), 14),
	RK_GATE(RK3588_PCLK_PMU1TIMER, "pclk_pmu1timer", "pclk_pmu0_root",
	    CLKGATE_CON(PMU, 1), 8),
	RK_COMPOSITE_NODIV(RK3588_CLK_PMU1TIMER_ROOT, "clk_pmu1timer_root",
	    pmu_24m_32k_100m_src_parents,
	    CLKSEL_CON(PMU, 2), __BITS(8,7),
	    CLKGATE_CON(PMU, 1), __BIT(9),
	    0),
	RK_GATE(RK3588_CLK_PMU1TIMER0, "clk_pmu1timer0", "clk_pmu1timer_root",
	    CLKGATE_CON(PMU, 1), 10),
	RK_GATE(RK3588_CLK_PMU1TIMER1, "clk_pmu1timer1", "clk_pmu1timer_root",
	    CLKGATE_CON(PMU, 1), 11),
	RK_COMPOSITE_NOMUX(RK3588_CLK_UART0_SRC, "clk_uart0_src", "cpll",
	    CLKSEL_CON(PMU, 3), __BITS(11,7),
	    CLKGATE_CON(PMU, 2), __BIT(3),
	    0),
	RK_COMPOSITE_FRAC(RK3588_CLK_UART0_FRAC, "clk_uart0_frac",
	    "clk_uart0_src",
	    CLKGATE_CON(PMU, 2),
	    RK_COMPOSITE_SET_RATE_PARENT),
	RK_MUX(RK3588_CLK_UART0, "clk_uart0", clk_uart0_parents,
	    CLKSEL_CON(PMU, 5), __BITS(1,0)),
	RK_GATE(RK3588_SCLK_UART0, "sclk_uart0", "clk_uart0",
	    CLKGATE_CON(PMU, 2), 5),
	RK_GATE(RK3588_PCLK_UART0, "pclk_uart0", "pclk_pmu0_root",
	    CLKGATE_CON(PMU, 2), 6),
	RK_GATE(RK3588_PCLK_PMU1WDT, "pclk_pmu1wdt", "pclk_pmu0_root",
	    CLKGATE_CON(PMU, 1), 6),
	RK_COMPOSITE_NODIV(RK3588_TCLK_PMU1WDT, "tclk_pmu1wdt",
	    mux_24m_32k_parents,
	    CLKSEL_CON(PMU, 2), __BITS(6,6),
	    CLKGATE_CON(PMU, 1), __BIT(7),
	    0),
	RK_COMPOSITE(RK3588_CLK_CR_PARA, "clk_cr_para",
	    mux_24m_ppll_spll_parents,
	    CLKSEL_CON(PMU, 15), __BITS(6,5), __BITS(4,0),
	    CLKGATE_CON(PMU, 4), __BIT(11),
	    0),
	RK_COMPOSITE(RK3588_CLK_USB2PHY_HDPTXRXPHY_REF,
	    "clk_usb2phy_hdptxrxphy_ref", mux_24m_ppll_parents,
	    CLKSEL_CON(PMU, 14), __BITS(14,14), __BITS(13,9),
	    CLKGATE_CON(PMU, 4), __BIT(7),
	    0),
	RK_COMPOSITE(RK3588_CLK_USBDPPHY_MIPIDCPPHY_REF,
	    "clk_usbdpphy_mipidcpphy_ref", mux_24m_ppll_spll_parents,
	    CLKSEL_CON(PMU, 14), __BITS(8,7), __BITS(6,0),
	    CLKGATE_CON(PMU, 4), __BIT(3),
	    0),
	RK_GATE(RK3588_CLK_PHY0_REF_ALT_P, "clk_phy0_ref_alt_p", "ppll",
	    RK3588_PHYREF_ALT_GATE, 0),
	RK_GATE(RK3588_CLK_PHY0_REF_ALT_M, "clk_phy0_ref_alt_m", "ppll",
	    RK3588_PHYREF_ALT_GATE, 1),
	RK_GATE(RK3588_CLK_PHY1_REF_ALT_P, "clk_phy1_ref_alt_p", "ppll",
	    RK3588_PHYREF_ALT_GATE, 2),
	RK_GATE(RK3588_CLK_PHY1_REF_ALT_M, "clk_phy1_ref_alt_m", "ppll",
	    RK3588_PHYREF_ALT_GATE, 3),
	RK_GATE(RK3588_HCLK_SPDIFRX0, "hclk_spdifrx0", "hclk_vo1",
	    CLKGATE_CON(0, 63), 12),
	RK_GATE(RK3588_HCLK_SPDIFRX1, "hclk_spdifrx1", "hclk_vo1",
	    CLKGATE_CON(0, 63), 14),
	RK_GATE(RK3588_HCLK_SPDIFRX2, "hclk_spdifrx2", "hclk_vo1",
	    CLKGATE_CON(0, 64), 0),
	RK_GATE(RK3588_HCLK_SPDIF4, "hclk_spdif4", "hclk_vo1",
	    CLKGATE_CON(0, 63), 8),
	RK_GATE(RK3588_HCLK_SPDIF3, "hclk_spdif3", "hclk_vo1",
	    CLKGATE_CON(0, 63), 4),
	RK_GATE(RK3588_HCLK_I2S6_8CH, "hclk_i2s6_8ch", "hclk_vo1",
	    CLKGATE_CON(0, 63), 3),
	RK_GATE(RK3588_HCLK_I2S5_8CH, "hclk_i2s5_8ch", "hclk_vo1",
	    CLKGATE_CON(0, 62), 12),
	RK_GATE(RK3588_HCLK_I2S9_8CH, "hclk_i2s9_8ch", "hclk_vo1",
	    CLKGATE_CON(0, 65), 0),
	RK_GATE(RK3588_HCLK_I2S7_8CH, "hclk_i2s7_8ch", "hclk_vo1",
	    CLKGATE_CON(0, 60), 0),
	RK_GATE(RK3588_HCLK_I2S10_8CH, "hclk_i2s10_8ch", "hclk_vo1",
	    CLKGATE_CON(0, 65), 4),
	RK_GATE(RK3588_ACLK_HDCP1, "aclk_hdcp1", "aclk_hdcp1_pre",
	    CLKGATE_CON(0, 60), 5),
	RK_GATE(RK3588_HCLK_HDCP1, "hclk_hdcp1", "hclk_vo1",
	    CLKGATE_CON(0, 60), 6),
	RK_GATE(RK3588_HCLK_SPDIF5_DP1, "hclk_spdif5_dp1", "hclk_vo0",
	    CLKGATE_CON(0, 57), 7),
	RK_GATE(RK3588_HCLK_SPDIF2_DP0, "hclk_spdif2_dp0", "hclk_vo0",
	    CLKGATE_CON(0, 57), 2),
	RK_GATE(RK3588_HCLK_I2S8_8CH, "hclk_i2s8_8ch", "hclk_vo0",
	    CLKGATE_CON(0, 56), 14),
	RK_GATE(RK3588_HCLK_I2S4_8CH, "hclk_i2s4_8ch", "hclk_vo0",
	    CLKGATE_CON(0, 56), 10),
	RK_GATE(RK3588_ACLK_HDCP0, "aclk_hdcp0", "aclk_hdcp0_pre",
	    CLKGATE_CON(0, 55), 12),
	RK_GATE(RK3588_HCLK_HDCP0, "hclk_hdcp0", "hclk_vo0",
	    CLKGATE_CON(0, 55), 13),
	RK_GATE(RK3588_HCLK_RKVENC1, "hclk_rkvenc1", "hclk_rkvenc1_pre",
	    CLKGATE_CON(0, 48), 4),
	RK_GATE(RK3588_ACLK_RKVENC1, "aclk_rkvenc1", "aclk_rkvenc1_pre",
	    CLKGATE_CON(0, 48), 5),
	RK_GATE(RK3588_ACLK_VPU, "aclk_vpu", "aclk_vdpu_low_pre",
	    CLKGATE_CON(0, 44), 8),
	RK_GATE(RK3588_ACLK_IEP2P0, "aclk_iep2p0", "aclk_vdpu_low_pre",
	    CLKGATE_CON(0, 45), 5),
	RK_GATE(RK3588_ACLK_JPEG_ENCODER0, "aclk_jpeg_encoder0",
	    "aclk_vdpu_low_pre",
	    CLKGATE_CON(0, 44), 10),
	RK_GATE(RK3588_ACLK_JPEG_ENCODER1, "aclk_jpeg_encoder1",
	    "aclk_vdpu_low_pre",
	    CLKGATE_CON(0, 44), 12),
	RK_GATE(RK3588_ACLK_JPEG_ENCODER2, "aclk_jpeg_encoder2",
	    "aclk_vdpu_low_pre",
	    CLKGATE_CON(0, 44), 14),
	RK_GATE(RK3588_ACLK_JPEG_ENCODER3, "aclk_jpeg_encoder3",
	    "aclk_vdpu_low_pre",
	    CLKGATE_CON(0, 45), 0),
	RK_GATE(RK3588_ACLK_JPEG_DECODER, "aclk_jpeg_decoder",
	    "aclk_jpeg_decoder_pre",
	    CLKGATE_CON(0, 45), 2),
	RK_GATE(RK3588_ACLK_USB3OTG1, "aclk_usb3otg1", "aclk_usb",
	    CLKGATE_CON(0, 42), 7),
	RK_GATE(RK3588_HCLK_HOST0, "hclk_host0", "hclk_usb",
	    CLKGATE_CON(0, 42), 10),
	RK_GATE(RK3588_HCLK_HOST_ARB0, "hclk_host_arb0", "hclk_usb",
	    CLKGATE_CON(0, 42), 11),
	RK_GATE(RK3588_HCLK_HOST1, "hclk_host1", "hclk_usb",
	    CLKGATE_CON(0, 42), 12),
	RK_GATE(RK3588_HCLK_HOST_ARB1, "hclk_host_arb1", "hclk_usb",
	    CLKGATE_CON(0, 42), 13),
	RK_GATE(RK3588_ACLK_USB3OTG0, "aclk_usb3otg0", "aclk_usb",
	    CLKGATE_CON(0, 42), 4),
	RK_GATE(RK3588_HCLK_SDIO, "hclk_sdio", "hclk_sdio_pre",
	    CLKGATE_CON(0, 75), 2),
	RK_GATE(RK3588_HCLK_RKVDEC1, "hclk_rkvdec1", "hclk_rkvdec1_pre",
	    CLKGATE_CON(0, 41), 2),
	RK_GATE(RK3588_ACLK_RKVDEC1, "aclk_rkvdec1", "aclk_rkvdec1_pre",
	    CLKGATE_CON(0, 41), 3),
	RK_GATE(RK3588_HCLK_RKVDEC0, "hclk_rkvdec0", "hclk_rkvdec0_pre",
	    CLKGATE_CON(0, 40), 3),
	RK_GATE(RK3588_ACLK_RKVDEC0, "aclk_rkvdec0", "aclk_rkvdec0_pre",
	    CLKGATE_CON(0, 40), 4),
	RK_GATE(RK3588_CLK_PCIE4L_PIPE, "clk_pcie4l_pipe",
	    "clk_pipe30phy_pipe0_i",
	    CLKGATE_CON(0, 39), 0),
	RK_GATE(RK3588_CLK_PCIE2L_PIPE, "clk_pcie2l_pipe",
	    "clk_pipe30phy_pipe2_i",
	    CLKGATE_CON(0, 39), 1),
	RK_GATE(RK3588_CLK_PIPEPHY0_PIPE_G, "clk_pipephy0_pipe_g",
	    "clk_pipephy0_pipe_i",
	    CLKGATE_CON(0, 38), 3),
	RK_GATE(RK3588_CLK_PIPEPHY1_PIPE_G, "clk_pipephy1_pipe_g",
	    "clk_pipephy1_pipe_i",
	    CLKGATE_CON(0, 38), 4),
	RK_GATE(RK3588_CLK_PIPEPHY2_PIPE_G, "clk_pipephy2_pipe_g",
	    "clk_pipephy2_pipe_i",
	    CLKGATE_CON(0, 38), 5),
	RK_GATE(RK3588_CLK_PIPEPHY0_PIPE_ASIC_G, "clk_pipephy0_pipe_asic_g",
	    "clk_pipephy0_pipe_i",
	    CLKGATE_CON(0, 38), 6),
	RK_GATE(RK3588_CLK_PIPEPHY1_PIPE_ASIC_G, "clk_pipephy1_pipe_asic_g",
	    "clk_pipephy1_pipe_i",
	    CLKGATE_CON(0, 38), 7),
	RK_GATE(RK3588_CLK_PIPEPHY2_PIPE_ASIC_G, "clk_pipephy2_pipe_asic_g",
	    "clk_pipephy2_pipe_i",
	    CLKGATE_CON(0, 38), 8),
	RK_GATE(RK3588_CLK_PIPEPHY2_PIPE_U3_G, "clk_pipephy2_pipe_u3_g",
	    "clk_pipephy2_pipe_i",
	    CLKGATE_CON(0, 38), 9),
	RK_GATE(RK3588_CLK_PCIE1L2_PIPE, "clk_pcie1l2_pipe",
	    "clk_pipephy0_pipe_g",
	    CLKGATE_CON(0, 38), 13),
	RK_GATE(RK3588_CLK_PCIE1L0_PIPE, "clk_pcie1l0_pipe",
	    "clk_pipephy1_pipe_g",
	    CLKGATE_CON(0, 38), 14),
	RK_GATE(RK3588_CLK_PCIE1L1_PIPE, "clk_pcie1l1_pipe",
	    "clk_pipephy2_pipe_g",
	    CLKGATE_CON(0, 38), 15),
	RK_GATE(RK3588_HCLK_SFC, "hclk_sfc", "hclk_nvm",
	    CLKGATE_CON(0, 31), 10),
	RK_GATE(RK3588_HCLK_SFC_XIP, "hclk_sfc_xip", "hclk_nvm",
	    CLKGATE_CON(0, 31), 11),
	RK_GATE(RK3588_HCLK_EMMC, "hclk_emmc", "hclk_nvm",
	    CLKGATE_CON(0, 31), 4),
	RK_GATE(RK3588_ACLK_ISP1, "aclk_isp1", "aclk_isp1_pre",
	    CLKGATE_CON(0, 26), 5),
	RK_GATE(RK3588_HCLK_ISP1, "hclk_isp1", "hclk_isp1_pre",
	    CLKGATE_CON(0, 26), 7),
	RK_GATE(RK3588_PCLK_AV1, "pclk_av1", "pclk_av1_pre",
	    CLKGATE_CON(0, 68), 5),
	RK_GATE(RK3588_ACLK_AV1, "aclk_av1", "aclk_av1_pre",
	    CLKGATE_CON(0, 68), 2),

#if 0
	notyet
#define RK3588_SDIO_CON0		0x0c24
#define RK3588_SDIO_CON1		0x0c28
#define RK3588_SDMMC_CON0		0x0c30
#define RK3588_SDMMC_CON1		0x0c34
	SCLK_SDIO_DRV, "sdio_drv","cclk_src_sdio", RK3588_SDIO_CON0
	SCLK_SDIO_SAMPLE, "sdio_sample", "cclk_src_sdio", RK3588_SDIO_CON1
	SCLK_SDMMC_DRV, "sdmmc_drv", "scmi_cclk_sd", RK3588_SDMMC_CON0
	SCLK_SDMMC_SAMPLE, "sdmmc_sample", "scmi_cclk_sd", RK3588_SDMMC_CON1
#endif

};

static void
rk3588_cru_init(struct rk_cru_softc *sc)
{
}

static int
rk3588_cru_match(device_t parent, cfdata_t cf, void *aux)
{
	struct fdt_attach_args * const faa = aux;
	return of_compatible_match(faa->faa_phandle, compat_data);
}

static void
rk3588_cru_attach(device_t parent, device_t self, void *aux)
{
	struct rk_cru_softc * const sc = device_private(self);
	struct fdt_attach_args * const faa = aux;

	sc->sc_dev = self;
	sc->sc_phandle = faa->faa_phandle;
	sc->sc_bst = faa->faa_bst;
	sc->sc_clks = rk3588_cru_clks;
	sc->sc_nclks = __arraycount(rk3588_cru_clks);

	sc->sc_grf_soc_status = 0x0480;	/* XXX */
	sc->sc_softrst_base = SOFTRST_CON(0, 0);	/* XXX */

	if (rk_cru_attach(sc) != 0)
		return;

	aprint_naive("\n");
	aprint_normal(": RK3588 CRU\n");

	rk3588_cru_init(sc);

	rk_cru_print(sc);
}
