/*
 * NFS client-side locking for NetBSD (userland part).
 * Edgar Fu, Mathematisches Institut der Universtitt Bonn, July 2006.
 * Do with this whatever you like, as long as you are either me or
 * - acknowledge that I wrote it for NetBSD in the first place, and
 * - don't blame me if it doesn't do what you like or expect.
 * This program does exactly what it does. If that's not what you expect it
 * or would like it to do, don't complain with me, the NetBSD Foundation,
 * my employer's sister-in-law or anybody else, but rewrite it to your taste.
 */

#define ASYNC 0
#define THREADING 1
#define THREADDEBUG 1

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <errno.h>
#include <string.h>
#include <sys/queue.h>
#include <sys/socket.h>
#include <rpc/rpc.h>
#include <rpcsvc/nlm_prot.h>
#include <nfs/nfs_lock.h>
#if THREADING
#include <pthread.h>
#endif
#include "lockd.h"
#include "lock_proc.h"
#include "lock_common.h"
#include "lock_client.h"

char hostname[256];

/*
 * List of lock requests in progress (so as to be able to handle results)
 * or granted (so as to be able to reclaim them after a server crash).
 * Granted locks will have wait set to 2, so we a) don't re-answer to the
 * kernel if we reclaim and b) can spit out errors if re-claiming fails.
 */
#if THREADING
/*
 * To prevent mutual queue/element deadlocking:
 *   Don't try to lock the queue when you've locked an element.
 * To prevent a lock being destroyed while someone's waiting for it:
 *   Don't try to lock an element unless you've locked the queue.
 *     That means dont't try to lock an element that's not on the queue.
 *     Since you've locked the queue, the element can't disappear from it.
 *   Don't destroy an element that's on the queue.
 *     So you can't destroy an element somebody's trying to lock.
 *   Don't take an element off the queue unless you hold a lock on the element.
 *     So you don't unlink an element someone else holds a lock on.
 */
#endif
TAILQ_HEAD(reclaim_head, reclaim_ent);
struct reclaim_ent{
        TAILQ_ENTRY(reclaim_ent) link;
#if THREADING
	pthread_t thread;
	pthread_mutex_t mutex;
#endif
        struct nfslock_req req;
	int fragment; /* adjustlocks() may split locks */
};
struct reclaim_head reclaim = TAILQ_HEAD_INITIALIZER(reclaim);
#if THREADING
pthread_mutex_t reclaim_mutex;
#endif

/*
 * The idea behind cur_ent and newreq()/handlereq() is that we only expose
 * the nfslock_req type, not the (internal) relaim_ent type.
 */
struct reclaim_ent *cur_ent;

struct nfslock_req *newreq(void) {
	syslog(LOG_DEBUG, "newreq");
	if (cur_ent != NULL) syslog(LOG_WARNING, "newreq: cur_ent != NULL");
	cur_ent = malloc(sizeof(*cur_ent));
	if (cur_ent == NULL) return NULL;
	/* don't initialize the lock, this is done when putting on the list */
	cur_ent->fragment = 0;
	return &cur_ent->req;
}

#if THREADING
#if THREADDEBUG
/* XXX is ({}) gcc? */
#define INITLOCK(ent) \
	({\
		syslog(LOG_DEBUG, "initlock(%d)", ent->req.serial);\
		pthread_mutex_init(&ent->mutex, NULL);\
	})
#else
#define INITLOCK(ent) \
	pthread_mutex_init(&ent->mutex, NULL)
#endif
#define FREEENT(ent) freeent(ent)
static void freeent(struct reclaim_ent *ent)
{
	if (ent) {
#if THREADDEBUG
		syslog(LOG_DEBUG, "freeent(%d)", ent->req.serial);
#endif
		if (pthread_mutex_destroy(&ent->mutex))
			syslog(LOG_WARNING, "freeent: mutex: %m");
		free(ent);
#if THREADDEBUG
	} else {
		syslog(LOG_DEBUG, "freeent(NULL)");

#endif
	}
}
#else
#define INITLOCK(ent) 0
#define FREEENT(ent) free(ent)
#endif

#if THREADING
#if THREADDEBUG
#define LOCKLIST \
	do {\
		syslog(LOG_DEBUG, "locklist(%p)", pthread_self());\
		if (pthread_mutex_lock(&reclaim_mutex))\
			syslog(LOG_WARNING, "LOCKLIST: %m");\
		syslog(LOG_DEBUG, "locklist(%p) OK", pthread_self());\
	} while (0)
#define UNLOCKLIST \
	do {\
		syslog(LOG_DEBUG, "unlocklist(%p)", pthread_self());\
		if (pthread_mutex_unlock(&reclaim_mutex))\
			syslog(LOG_WARNING, "UNLOCKLIST: %m");\
	} while (0)
#define LOCKENT(ent) \
	do {\
		syslog(LOG_DEBUG, "lockent(%p, %d)", pthread_self(), ent->req.serial);\
		if (pthread_mutex_lock(&ent->mutex))\
			syslog(LOG_WARNING, "LOCKENT(%d): %m", ent->req.serial);\
		syslog(LOG_DEBUG, "lockent(%p, %d) OK", pthread_self(), ent->req.serial);\
	} while (0)
#define UNLOCKENT(ent) \
	do {\
		syslog(LOG_DEBUG, "unlockent(%p, %d)", pthread_self(), ent->req.serial);\
		if (pthread_mutex_unlock(&ent->mutex))\
			syslog(LOG_WARNING, "UNLOCKENT(%d): %m", ent->req.serial);\
	} while (0)
#else
#define LOCKLIST \
	if (pthread_mutex_lock(&reclaim_mutex))\
		syslog(LOG_WARNING, "LOCKLIST: %m")
#define UNLOCKLIST \
	if (pthread_mutex_unlock(&reclaim_mutex))\
		syslog(LOG_WARNING, "UNLOCKLIST: %m")
#define LOCKENT(ent) \
	if (pthread_mutex_lock(&ent->mutex))\
		syslog(LOG_WARNING, "LOCKENT(%d): %m", ent->req.serial)
#define UNLOCKENT(ent) \
	if (pthread_mutex_unlock(&ent->mutex))\
		syslog(LOG_WARNING, "UNLOCKENT(%d): %m", ent->req.serial)
#endif /* THREADDEBUG */
#define REMOVEENT(ent) removeent(ent)
/*
 * Remove an entry from the list if another thread hasn't done so yet.
 * Entered/exited with lock on the list held, but not on the entry.
 */
static struct reclaim_ent *removeent(struct reclaim_ent *ent)
{
	struct reclaim_ent *p;

#if THREADDEBUG
	syslog(LOG_DEBUG, "removeent(%d)", ent->req.serial);
#endif
	TAILQ_FOREACH(p, &reclaim, link)
		if (p == ent) {
			LOCKENT(ent);
			TAILQ_REMOVE(&reclaim, ent, link);
			UNLOCKENT(ent);
			syslog(LOG_DEBUG, "removeent: %d", ent->req.serial);
			return ent;
		}
#if THREADDEBUG
	syslog(LOG_DEBUG, "removeent: NULL");
#endif
	return NULL;
}
#else
#define LOCKLIST
#define UNLOCKLIST
#define LOCKENT(ent)
#define UNLOCKENT(ent)
/* XXX is ({ ... }) a gcc extension? */
#define REMOVEENT(ent) ({TAILQ_REMOVE(&reclaim, ent, link); ent;})
#endif

#if THREADING
/* get_client in lock_proc.c is probably not thread safe */
/* XXX what if the main thread calls it? */
static pthread_mutex_t lock_proc_mutex;
#if THREADDEBUG
#define LOCKPROC \
	do {\
		syslog(LOG_DEBUG, "lockproc(%p)", pthread_self());\
		if (pthread_mutex_lock(&lock_proc_mutex))\
			syslog(LOG_WARNING, "LOCKPROC: %m");\
		syslog(LOG_DEBUG, "lockproc(%p) OK", pthread_self());\
	} while (0)
#define UNLOCKPROC \
	do {\
		syslog(LOG_DEBUG, "unlockproc(%p)", pthread_self());\
		if (pthread_mutex_unlock(&lock_proc_mutex))\
			syslog(LOG_WARNING, "UNLOCKPROC: %m");\
	} while (0)
#else
#define LOCKPROC \
	if (pthread_mutex_lock(&lock_proc_mutex))\
		syslog(LOG_WARNING, "LOCKPROC: %m")
#define UNLOCKPROC \
	if (pthread_mutex_unlock(&lock_proc_mutex))\
		syslog(LOG_WARNING, "UNLOCKPROC: %m")
#endif
#else
#define LOCKPROC
#define UNLOCKPROC
#endif

void handleent(struct reclaim_ent *ent);

void handlereq(void)
{
#if THREADING
	struct nfslock_rep rep;
#endif

	if (cur_ent == NULL) {
		syslog(LOG_WARNING, "handlereq: cur_ent == NULL");
		return;
	}
#if THREADING
	rep.errnum = pthread_create(&cur_ent->thread, NULL, (void *(*)(void *))handleent, (void *)cur_ent);
	if (rep.errnum) {
		syslog(LOG_NOTICE, "handlereq: pthread_create: %m");
		return;
		rep.serial = cur_ent->req.serial;
		rep.conflict = 0;
		rep.offset = rep.len = 0;
		rep.owner = 0;
		sendrep(&rep);
		freeent(cur_ent);
	}
#else
	handleent(cur_ent);
#endif
	cur_ent = NULL;
}

static struct reclaim_ent *findserial(int serial);
static void adjustlocks(struct nfslock_req *req, const char *msg);
#if ASYNC
static int call_test(struct reclaim_ent *ent, int locked);
static int call_lock(struct reclaim_ent *ent, int locked, int reclaim);
static int call_cancel(struct reclaim_ent *ent, int locked);
static int call_unlock(struct reclaim_ent *ent, int locked);
#else
static int call_test(struct reclaim_ent *ent, int locked, enum nlm4_stats *stat, struct nlm4_holder *holder);
static int call_lock(struct reclaim_ent *ent, int locked, int reclaim, enum nlm4_stats *stat);
static int call_cancel(struct reclaim_ent *ent, int locked, enum nlm4_stats *stat);
static int call_unlock(struct reclaim_ent *ent, int locked, enum nlm4_stats *stat);
#endif
static int clntstat2errno(enum clnt_stat stat);
static int nlmstat2errno(enum nlm4_stats stat);

void handleent(struct reclaim_ent *ent)
{
	struct reclaim_ent *old_ent;
	struct nfslock_rep rep;
	int ret;
#if !ASYNC
	enum nlm4_stats stat;
	struct nlm4_holder holder;
#endif
	
#if THREADING
	/* No need to lock ent since it's not on the list. */
#endif
	syslog(LOG_DEBUG, "handleent(%d)", ent->req.op);
	rep.serial = ent->req.serial;
	rep.errnum = 0;
	rep.conflict = 0;
	rep.offset = rep.len = 0;
	rep.owner = 0;
	switch (ent->req.op) {
		case 0: /* NULL */
			sendrep(&rep);
			free(ent);
			break;
		case 1: /* TEST */
#if ASYNC
			ret = call_test(ent, 0);
#else
#if 0
			holder.oh.n_bytes = NULL;
#endif
			ret = call_test(ent, 0, &stat, &holder);
#endif
			if (ret) {
				rep.errnum = clntstat2errno(ret);
				sendrep(&rep);
			}
#if !ASYNC
			rep.errnum = nlmstat2errno(stat);
			if (stat == nlm4_denied) {
				rep.errnum = 0; /* denied status is not an error if command was test */
				rep.conflict = holder.exclusive ? 2 : 1;
				rep.owner = (pid_t)holder.svid;
				rep.offset = holder.l_offset;
				rep.len = holder.l_len;
			}
			sendrep(&rep);
#endif /* !ASYNC */
#if 0
			if (holder.oh.n_bytes) free(holder.oh.n_bytes);
#endif
			free(ent);
			break;
		case 2: /* LOCK */
#if ASYNC
			/* must put the request on the list before making the call */
			ret = INITLOCK(ent);
			if (ret) {
				syslog(LOG_NOTICE, "lock/async: lock initialization failed: %m");
				rep.errnum = ret;
				sendrep(&rep);
				free(ent);
				break;
			}
			LOCKLIST;
			LOCKENT(ent); /* so nothing changes until the RPC */
			TAILQ_INSERT_TAIL(&reclaim, ent, link);
			UNLOCKLIST;
			/* list unlocked, entry locked */
			ret = call_lock(ent, 1, 0); /* unlocks ent */
			if (ret) {
				LOCKLIST;
				FREEENT(REMOVEENT(ent));
				UNLOCKLIST;
				rep.errnum = clntstat2errno(ret);
				sendrep(&rep);
				break;
			}
#else
			/* may make the call without the entry on the list */
			ret = call_lock(ent, 0, 0, &stat);
			if (ret) {
				rep.errnum = clntstat2errno(ret);
				sendrep(&rep);
				free(ent);
			} else {
				rep.errnum = nlmstat2errno(stat);
				switch (stat) {
					case nlm4_granted:
						ent->req.wait = 2;
						ret = INITLOCK(ent);
						if (ret) {
							syslog(LOG_NOTICE, "lock/granted: lock initialization failed: %m");
							rep.errnum = ret;
							free(ent);
						} else {
							LOCKLIST;
							adjustlocks(&ent->req, "lock");
							TAILQ_INSERT_TAIL(&reclaim, ent, link);
							UNLOCKLIST;
						}
						sendrep(&rep);
						break;
					case nlm4_blocked:
						if (ent->req.wait) {
							syslog(LOG_DEBUG, "lock/blocked/wait");
							/* don't reply, process will wait for grant */
							ret = INITLOCK(ent);
							if (ret) {
								syslog(LOG_NOTICE, "lock/blocked: lock initialization failed: %m");
								rep.errnum = ret;
								free(ent);
							} else {
								LOCKLIST;
								TAILQ_INSERT_TAIL(&reclaim, ent, link);
								UNLOCKLIST;
							}
						} else {
							/* strange */
							syslog(LOG_WARNING, "lock/blocked/nowait");
							sendrep(&rep);
							free(ent);
						}
						break;
					case nlm4_denied:
						if (ent->req.wait) {
							/* strange */
							syslog(LOG_WARNING, "lock/denied/wait");
							sendrep(&rep);
							free(ent);
							/* don't reply, process will wait */
							ret = INITLOCK(ent);
							if (ret) {
								syslog(LOG_NOTICE, "lock/denied: lock initialization failed: %m");
								rep.errnum = ret;
								free(ent);
							} else {
								LOCKLIST;
								TAILQ_INSERT_TAIL(&reclaim, ent, link);
								UNLOCKLIST;
							}
						} else {
							syslog(LOG_DEBUG, "lock/denied/nowait");
							/* Posix semantics want EACCES, not EWOULDBLOCK */
							/* Yes I know these two are in fact identical */
							if (ent->req.posix) rep.errnum = EAGAIN;
							sendrep(&rep);
							free(ent);
						}
				 		break;
					default:
						sendrep(&rep);
						free(ent);
						break;
				}
			}
#endif /* ASYNC */
			break;
		case 3: /* CANCEL */
			old_ent = findserial(ent->req.serial);
			if (old_ent) {
				/* list and old_ent locked */
				syslog(LOG_DEBUG, "cancel %d removes recovery list entry", ent->req.serial);
				UNLOCKENT(old_ent);
				FREEENT(REMOVEENT(old_ent));
				UNLOCKLIST;
			}
			/* list unlocked, entry not on list */
#if ASYNC
			(void)call_cancel(ent, 0);
#else
			(void)call_cancel(ent, 0, &stat);
#endif
			free(ent);
			/* no reply */
			break;
		case 4: /* UNLOCK */
#if ASYNC
			ret = call_unlock(ent, 0);
#else
			ret = call_unlock(ent, 0, &stat);
#endif
			rep.errnum = clntstat2errno(ret);
			sendrep(&rep);
			if (ret == 0) {
				LOCKLIST;
				adjustlocks(&ent->req, "unlock");
				UNLOCKLIST;
			}
			free(ent);
			break;
		default:/* OOPS */
			syslog(LOG_WARNING, "handleent: unknown op %d", ent->req.op);
			free(ent);
			break;
	}
}

void client_notify(char *hostname, int state)
{
	struct reclaim_ent *ent;
#if !ASYNC
	enum nlm4_stats stat;
#endif
	
	syslog(LOG_INFO, "starting lock recovery");
	LOCKLIST;
	/*
	 * We run through the list and make RPCs with the lock held.
	 * However, this is grace time.
	 */
	TAILQ_FOREACH(ent, &reclaim, link) {
		LOCKENT(ent); /* so it doesn't change until the RPC */
		if (ent->fragment)
			syslog(LOG_DEBUG, "trying to recover lock %d, fragment %d", ent->req.serial,  ent->fragment);
		else
			syslog(LOG_DEBUG, "trying to recover lock %d", ent->req.serial);
#if ASYNC
		(void)call_lock(ent, 1, 1); /* unlocks ent */
#else
		(void)call_lock(ent, 1, 1, &stat); /* unlocks ent */
#endif
	}
	UNLOCKLIST;
}

void client_poll(void)
{
}

void client_init(void)
{
	gethostname(hostname, sizeof(hostname));
#if THREADING
	if (pthread_mutex_init(&reclaim_mutex, NULL)) {
		syslog(LOG_ERR, "client_init: initializing reclaim list mutex failed: %m");
		exit(1);
	}
	if (pthread_mutex_init(&lock_proc_mutex, NULL)) {
		syslog(LOG_ERR, "client_init: initializing locking proc mutex failed: %m");
		exit(1);
	}
#endif
	syslog(LOG_DEBUG, "client: %s/%s", ASYNC ? "async" : "sync", THREADING ? "threading" : "nothreading");
}

void client_sigchild(int s)
{
	(void)s;
}

/*
 * Look up an entry in the reclaim list matching a given lock.
 * Needed to identify the entry a granted call refers to.
 */
static struct reclaim_ent *findlock(const struct netbuf *caller, int exclusive, const struct nlm4_lock *lock)
{
	struct reclaim_ent *ent;

	LOCKLIST;
	TAILQ_FOREACH(ent, &reclaim, link) {
		LOCKENT(ent);
		if (ent->req.mount_nam.ss_len == caller->len 
		 && memcmp(&ent->req.mount_nam, &caller->buf, ent->req.mount_nam.ss_len) == 0
		 && ent->req.exclusive == exclusive
		 && ent->req.handle_len == lock->fh.n_len
		 && memcmp(&ent->req.handle, &lock->fh.n_bytes, ent->req.handle_len) == 0
		/* XXX compare oh? */
		 && (int)ent->req.owner == lock->svid
		 && ent->req.offset == lock->l_offset
		 && ent->req.len == lock->l_len) {
			return ent;
		}
		UNLOCKENT(ent);
	}
	UNLOCKLIST;
	return NULL;
}

/*
 * Look up an entry in the reclaim list matching a given serial.
 * Needed on cancel (process died) to remove entry from recovery list.
 * Entered with list unlocked.
 * If found, exit with both list and entry locked.
 */
static struct reclaim_ent *findserial(int serial)
{
	struct reclaim_ent *ent;

	LOCKLIST;
	TAILQ_FOREACH(ent, &reclaim, link) {
		LOCKENT(ent);
		if (ent->req.serial == serial && ent->fragment == 0) {
			return ent;
		}
		UNLOCKENT(ent);
	}
	UNLOCKLIST;
	return NULL;
}

#if ASYNC
/*
 * Find entry matching a given cookie.
 * Needed for xxxres() routines.
 * Only lockres()/unlockres() use this since testres() can do with
 * the serial alone. See cookie2serial().
 * If found, returns with list and entry locked.
 */
static struct reclaim_ent *findcookie(const netobj *cookie)
{
	struct reclaim_ent *ent;
	
	if (cookie->n_len != sizeof(ent->req.serial)) return NULL;
	LOCKLIST;
	TAILQ_FOREACH(ent, &reclaim, link) {
		LOCKENT(ent);
		if (memcmp(cookie->n_bytes, &ent->req.serial, sizeof(ent->req.serial)) == 0) {
			return ent;
		}
		UNLOCKENT(ent);
	}
	UNLOCKLIST;
	return NULL;
}

/*
 * Extract request serial from cookie.
 */
static int cookie2serial(const netobj *cookie, int *serial)
{
	if (cookie->n_len != sizeof(int)) return 1;
	memcpy(serial, cookie->n_bytes, sizeof(int));
	return 0;
}
#endif

/*
 * Encode a request serial as a cookie.
 */
static void serial2cookie(const int *serial, netobj *cookie)
{
	cookie->n_len = sizeof(int);
	cookie->n_bytes = (char *)serial;
}

/*
 * Convert an NLM nlm_stats (actually nlm4_stats) to an errno.
 */
static int nlmstat2errno(enum nlm4_stats stat)
{
	switch (stat) {
		case nlm4_granted: return 0;
		case nlm4_denied: return EACCES;
		/* XXX typo in rpcsvc/nlm_prot.x 
		 * case nlm4_denied_nolocks: return ENOLCK;
		 */
		case nlm4_denied_nolock: return ENOLCK;
		case nlm4_blocked: return EWOULDBLOCK;
		case nlm4_denied_grace_period: return EAGAIN;
		case nlm4_deadlck: return EDEADLK;
		case nlm4_rofs: return EROFS;
		case nlm4_stale_fh: return ESTALE;
		case nlm4_fbig: return EOVERFLOW;
		case nlm4_failed: return ENOLCK;
		default: return ENOLCK;
	}
}

/*
 * Convert an RPC clnt_stat to an errno.
 */
static int clntstat2errno(enum clnt_stat stat)
{
	switch (stat) {
		case RPC_SUCCESS:		return 0;
		case RPC_CANTENCODEARGS:	return EBADRPC;
		case RPC_CANTDECODERES:		return EBADRPC;
		case RPC_CANTSEND:		return ECONNABORTED /* XXX */;
		case RPC_CANTRECV:		return ECONNABORTED /* XXX */;
		case RPC_TIMEDOUT:		return ETIMEDOUT;
		case RPC_VERSMISMATCH:		return ERPCMISMATCH;
		case RPC_AUTHERROR:		return EAUTH;
		case RPC_PROGUNAVAIL:		return EPROGUNAVAIL;
		case RPC_PROGVERSMISMATCH:	return EPROGMISMATCH;
		case RPC_PROCUNAVAIL:		return EPROCUNAVAIL;
		case RPC_CANTDECODEARGS:	return EBADRPC;
		case RPC_SYSTEMERROR:		return EBADRPC;
		case RPC_UNKNOWNHOST:		return EDESTADDRREQ /* XXX */;
		case RPC_UNKNOWNPROTO:		return ENOPROTOOPT;
		case RPC_PMAPFAILURE:		return EPROGUNAVAIL;
		case RPC_PROGNOTREGISTERED:	return EPROGUNAVAIL;
		case RPC_FAILED:		return EBADRPC;
		default:			return EBADRPC;
	}
}

/*
 * Figure out which NLM protocol to use for a given NFS protocol level.
 */
static int nfs2nlm(int nfs_vers)
{
	if (nfs_vers > 3) return 0;
	if (nfs_vers == 3) return NLM_VERS4;
	return NLM_VERS;
}

/*
 * Set timeout according to NFS mount options.
 */
static void settimeo(const struct nfslock_req *req, struct timeval *timeo)
{
#if 1
	if (req->mount_soft) {
		timeo->tv_sec = req->mount_retry * req->mount_timeo;
		timeo->tv_usec = 0;
	} else {
		/* XXX this is not infinite, but how do I know the maximum a timeval can hold? */
		timeo->tv_sec = 23 + ( 59 + 59 * 60 ) * 60; /* just under one day */
		timeo->tv_usec = 999999;
	}
#else
	timeo->tv_sec = 3;
	timeo->tv_usec = 0;
#endif
}

/*
 * All call_xxxx() routines have a parameter telling them whether they are called
 * with the lock on ent held and therefore should release it before the actual RPC.
 * Only call_lock currently actually needs this.
 * XXX There is some duplication of NLM3<->NLM4 conversion from lock_proc.c here.
 * Need a local copy of the request serial since its address goes into the cookie.
 * Using the adress inside the request structure would not be thread-safe.
 */

/* XXX Why does rpc/clnt.h cast to caddr_t in clnt_call but not in clnt_freeres? */
#undef clnt_freeres
#define clnt_freeres(rh,xres,resp) CLNT_FREERES(rh,xres,(caddr_t)(void *)resp)

#if 0
/*
 * Copy netobj. Allocate memory if dest ptr is NULL.
 */
static int netobjcpy(netobj *dest, const netobj *src)
{
	if (dest->n_bytes == NULL) {
		dest->n_bytes = malloc(src->n_len);
		if dest->bytes == NULL return ENOMEM;
	} else {
		if src->n_bytes > dest->n_bytes return ENOBUFS;
	}
	dest->nlen = src->n_len;
	memcpy(dest->n_bytes, src->n_bytes, src->n_len);
	return 0;
}
#endif

/*
 * The spec says oh (object holder) is an opaque object identifying the host,
 * or a process on the host, making the request. But the host already goes
 * into caller_name and the PID into svid.
 */
static int dummy_oh = 0xefef;

/*
 * Make an NLM_TEST/NLM4_TEST (or NLM_TEST_MSG/NLM4_TEST_MSG) RPC.
 * Nothing interesting. Just convert nfslock_req to nlm_testargs.
 */
#if ASYNC
static int call_test(struct reclaim_ent *ent, int locked)
#else
static int call_test(struct reclaim_ent *ent, int locked, enum nlm4_stats *stat, struct nlm4_holder *holder)
#endif
{
	CLIENT *client;
	int nlm_vers;
	struct timeval timeo;
	struct nlm_testargs testargs;
	struct nlm4_testargs testargs4;
#if !ASYNC
	struct nlm_testres testres;
	struct nlm4_testres testres4;
#endif
	int serial;
	int ret;

	nlm_vers = nfs2nlm(ent->req.mount_version);
	LOCKPROC;
	client = get_client((struct sockaddr *)&ent->req.mount_nam, nlm_vers);
	if (client == NULL) {
		syslog(LOG_WARNING, "call_test: get_client failed");
		ret = rpc_createerr.cf_stat;
		UNLOCKPROC;
		return ret;
	}
	UNLOCKPROC;
	settimeo(&ent->req, &timeo);
	serial = ent->req.serial;

	switch (nlm_vers) {
		case 0:
			syslog(LOG_WARNING, "call_test: mount_version=%d", ent->req.mount_version);
			return RPC_FAILED;
		case NLM_VERS:
			serial2cookie(&serial, &testargs.cookie);
			testargs.exclusive = ent->req.exclusive;
			testargs.alock.caller_name = hostname;
			testargs.alock.fh.n_len = ent->req.handle_len;
			testargs.alock.fh.n_bytes = (char *)&ent->req.handle;
			/* XXX what to put in oh and svid for a test? */
			testargs.alock.oh.n_len = 0;
			testargs.alock.oh.n_bytes = NULL;
			testargs.alock.svid = 0;
			/* XXX test for offset/len too large */
			testargs.alock.l_offset = ent->req.offset;
			testargs.alock.l_len = ent->req.len;
			if (locked) UNLOCKENT(ent);
#if ASYNC
			syslog(LOG_DEBUG, "call_test: NLM_TEST_MSG");
			ret = clnt_call(client, NLM_TEST_MSG, xdr_nlm_testargs, &testargs, xdr_void, NULL, timeo);
#else
			testres.cookie.n_len = 0;
			testres.cookie.n_bytes = NULL;
			testres.stat.nlm_testrply_u.holder.oh.n_len = 0;
			testres.stat.nlm_testrply_u.holder.oh.n_bytes = NULL;
			syslog(LOG_DEBUG, "call_test: NLM_TEST");
			ret = clnt_call(client, NLM_TEST, xdr_nlm_testargs, &testargs, xdr_nlm_testres, &testres, timeo);
			if (ret) {
				syslog(LOG_NOTICE, "call_test: clnt_call: %s", clnt_sperrno(ret));
			} else {
				syslog(LOG_DEBUG, "call_test: clnt_call OK");
				*stat = (enum nlm4_stats) testres.stat.stat;
				if (testres.stat.stat == nlm_denied) {
					holder->exclusive = testres.stat.nlm_testrply_u.holder.exclusive;
					holder->svid = testres.stat.nlm_testrply_u.holder.svid;
#if 0
					if (netobjcpy(holder->oh, testres.stat.nlm_testrply_u.holder.oh)) {
						syslog(LOG_NOTICE, "call_test: netobjcpy failed");
						ret = RPC_FAILED;
					}
#else
					holder->oh.n_len = 0;
					holder->oh.n_bytes = NULL;
#endif
					holder->l_offset = testres.stat.nlm_testrply_u.holder.l_offset;
					holder->l_len = testres.stat.nlm_testrply_u.holder.l_len;
				}
				if (!clnt_freeres(client, xdr_nlm_testres, &testres)) {
					syslog(LOG_NOTICE, "call_test: freeres failed");
					ret = RPC_FAILED;
				}
			}
#endif
			break;
		case NLM_VERS4:
			serial2cookie(&serial, &testargs4.cookie);
			testargs4.exclusive = ent->req.exclusive;
			testargs4.alock.caller_name = hostname;
			testargs4.alock.fh.n_len = ent->req.handle_len;
			testargs4.alock.fh.n_bytes = (char *)&ent->req.handle;
			/* XXX what to put in oh and svid for a test? */
			testargs4.alock.oh.n_len = 0;
			testargs4.alock.oh.n_bytes = NULL;
			testargs4.alock.svid = 0;
			testargs4.alock.l_offset = ent->req.offset;
			testargs4.alock.l_len = ent->req.len;
			if (locked) UNLOCKENT(ent);
#if ASYNC
			syslog(LOG_DEBUG, "call_test: NLM4_TEST_MSG");
			ret = clnt_call(client, NLM4_TEST_MSG, xdr_nlm4_testargs, &testargs4, xdr_void, NULL, timeo);
#else
			testres4.cookie.n_len = 0;
			testres4.cookie.n_bytes = NULL;
			testres4.stat.nlm4_testrply_u.holder.oh.n_len = 0;
			testres4.stat.nlm4_testrply_u.holder.oh.n_bytes = NULL;
			syslog(LOG_DEBUG, "call_test: NLM4_TEST");
			ret = clnt_call(client, NLM4_TEST, xdr_nlm4_testargs, &testargs4, xdr_nlm4_testres, &testres4, timeo);
			if (ret) {
				syslog(LOG_NOTICE, "call_test: clnt_call: %s", clnt_sperrno(ret));
			} else {
				syslog(LOG_DEBUG, "call_test: clnt_call OK");
				*stat = testres4.stat.stat;
				if (testres4.stat.stat == nlm4_denied) {
					/* must copy all fields individually, too */
					holder->exclusive = testres4.stat.nlm4_testrply_u.holder.exclusive;
					holder->svid = testres4.stat.nlm4_testrply_u.holder.svid;
#if 0
					if (netobjcpy(holder->oh, testres4.stat.nlm4_testrply_u.holder.oh)) {
						syslog(LOG_NOTICE, "call_test: netobjcpy failed");
						ret = RPC_FAILED;
					}
#else
					holder->oh.n_len = 0;
					holder->oh.n_bytes = NULL;
#endif
					holder->l_offset = testres4.stat.nlm4_testrply_u.holder.l_offset;
					holder->l_len = testres4.stat.nlm4_testrply_u.holder.l_len;
				}
				if (!clnt_freeres(client, xdr_nlm4_testres, &testres4)) {
					syslog(LOG_NOTICE, "call_test: freeres failed");
					ret = RPC_FAILED;
				}
			}
#endif
			break;
		default:
			syslog(LOG_WARNING, "call_test: nlm_vers=%d", nlm_vers);
			return RPC_FAILED;
	}
	return ret;
}

/*
 * Make an NLM_LOCK/NLM4_LOCK (or NLM_LOCK_MSG/NLM4_LOCK_MSG) RPC.
 * Not interesting either. Convert nfslock_req to nlm_lockargs.
 */
#if ASYNC
static int call_lock(struct reclaim_ent *ent, int locked, int reclaim)
#else
static int call_lock(struct reclaim_ent *ent, int locked, int reclaim, enum nlm4_stats *stat)
#endif
{
	CLIENT *client;
	int nlm_vers;
	struct timeval timeo;
	struct nlm_lockargs lockargs;
	struct nlm4_lockargs lockargs4;
#if !ASYNC
	struct nlm_res res;
	struct nlm4_res res4;
#endif
	int serial;
	int ret;

	nlm_vers = nfs2nlm(ent->req.mount_version);
	LOCKPROC;
	client = get_client((struct sockaddr *)&ent->req.mount_nam, nlm_vers);
	if (client == NULL) {
		syslog(LOG_WARNING, "call_lock: get_client failed");
		ret = rpc_createerr.cf_stat;
		UNLOCKPROC;
		return ret;
	}
	UNLOCKPROC;
	settimeo(&ent->req, &timeo);
	serial = ent->req.serial;

	switch (nlm_vers) {
		case 0:
			syslog(LOG_WARNING, "call_lock: mount_version=%d", ent->req.mount_version);
			return RPC_FAILED;
		case NLM_VERS:
			serial2cookie(&serial, &lockargs.cookie);
			lockargs.block = ent->req.wait;
			lockargs.exclusive = ent->req.exclusive;
			lockargs.alock.caller_name = hostname;
			lockargs.alock.fh.n_len = ent->req.handle_len;
			lockargs.alock.fh.n_bytes = (char *)&ent->req.handle;
			/* XXX what do we want to put into oh? */
			lockargs.alock.oh.n_len = sizeof(dummy_oh);
			lockargs.alock.oh.n_bytes = (char *)&dummy_oh;
			lockargs.alock.svid = ent->req.owner;
			/* XXX test for offset/len too large */
			lockargs.alock.l_offset = ent->req.offset;
			lockargs.alock.l_len = ent->req.len;
			lockargs.reclaim = reclaim;
			lockargs.state = sm_state;
			if (locked) UNLOCKENT(ent);
#if ASYNC
			syslog(LOG_DEBUG, "call_lock: NLM_LOCK_MSG");
			ret = clnt_call(client, NLM_LOCK_MSG, xdr_nlm_lockargs, &lockargs, xdr_void, NULL, timeo);
#else
			res.cookie.n_len = 0;
			res.cookie.n_bytes = NULL;
			syslog(LOG_DEBUG, "call_lock: NLM_LOCK");
			ret = clnt_call(client, NLM_LOCK, xdr_nlm_lockargs, &lockargs, xdr_nlm_res, &res, timeo);
			if (ret) {
				syslog(LOG_NOTICE, "call_lock: clnt_call: %s", clnt_sperrno(ret));
			} else {
				syslog(LOG_DEBUG, "call_lock: clnt_call OK");
				*stat = (enum nlm4_stats) res.stat.stat;
				if (!clnt_freeres(client, xdr_nlm_res, &res)) {
					syslog(LOG_NOTICE, "call_lock: freeres failed");
					ret = RPC_FAILED;
				}
			}
#endif
			break;
		case NLM_VERS4:
			serial2cookie(&serial, &lockargs4.cookie);
			lockargs4.block = ent->req.wait;;
			lockargs4.exclusive = ent->req.exclusive;
			lockargs4.alock.caller_name = hostname;
			lockargs4.alock.fh.n_len = ent->req.handle_len;
			lockargs4.alock.fh.n_bytes = (char *)&ent->req.handle;
			/* XXX what do we want to put into oh? */
			lockargs4.alock.oh.n_len = sizeof(dummy_oh);
			lockargs4.alock.oh.n_bytes = (char *)&dummy_oh;
			lockargs4.alock.svid = ent->req.owner;
			lockargs4.alock.l_offset = ent->req.offset;
			lockargs4.alock.l_len = ent->req.len;
			lockargs4.reclaim = reclaim;
			lockargs4.state = sm_state;
			if (locked) UNLOCKENT(ent);
#if ASYNC
			syslog(LOG_DEBUG, "call_lock: NLM4_LOCK_MSG");
			ret = clnt_call(client, NLM4_LOCK_MSG, xdr_nlm4_lockargs, &lockargs4, xdr_void, NULL, timeo);
#else
			res4.cookie.n_len = 0;
			res4.cookie.n_bytes = NULL;
			syslog(LOG_DEBUG, "call_lock: NLM4_LOCK");
			ret = clnt_call(client, NLM4_LOCK, xdr_nlm4_lockargs, &lockargs4, xdr_nlm4_res, &res4, timeo);
			if (ret) {
				syslog(LOG_NOTICE, "call_lock: clnt_call: %s", clnt_sperrno(ret));
			} else {
				syslog(LOG_DEBUG, "call_lock: clnt_call OK");
				*stat = res4.stat.stat;
				if (!clnt_freeres(client, xdr_nlm4_res, &res4)) {
					syslog(LOG_NOTICE, "call_lock: freeres failed");
					ret = RPC_FAILED;
				}
			}
#endif
			break;
		default:
			syslog(LOG_WARNING, "call_lock: nlm_vers=%d", nlm_vers);
			return RPC_FAILED;
	}
	return ret;
}

/*
 * Make an NLM_CANCEL/NLM4_CANCEL (or NLM_CANCEL_MSG/NLM4_CANCEL_MSG) RPC.
 * Boringly convert nfslock_req to nlm_cancarg.
 */
#if ASYNC
static int call_cancel(struct reclaim_ent *ent, int locked)
#else
static int call_cancel(struct reclaim_ent *ent, int locked, enum nlm4_stats *stat)
#endif
{
	CLIENT *client;
	int nlm_vers;
	struct timeval timeo;
	struct nlm_cancargs cancargs;
	struct nlm4_cancargs cancargs4;
#if !ASYNC
	struct nlm_res res;
	struct nlm4_res res4;
#endif
	int serial;
	int ret;

	nlm_vers = nfs2nlm(ent->req.mount_version);
	LOCKPROC;
	client = get_client((struct sockaddr *)&ent->req.mount_nam, nlm_vers);
	if (client == NULL) {
		syslog(LOG_WARNING, "call_cancel: get_client failed");
		ret = rpc_createerr.cf_stat;
		UNLOCKPROC;
		return ret;
	}
	UNLOCKPROC;
	settimeo(&ent->req, &timeo);
	serial = ent->req.serial;

	switch (nlm_vers) {
		case 0:
			syslog(LOG_WARNING, "call_cancel: mount_version=%d", ent->req.mount_version);
			return RPC_FAILED;
		case NLM_VERS:
			serial2cookie(&serial, &cancargs.cookie);
			cancargs.block = ent->req.wait;
			cancargs.exclusive = ent->req.exclusive;
			cancargs.alock.caller_name = hostname;
			cancargs.alock.fh.n_len = ent->req.handle_len;
			cancargs.alock.fh.n_bytes = (char *)&ent->req.handle;
			cancargs.alock.oh.n_len = sizeof(dummy_oh);
			cancargs.alock.oh.n_bytes = (char *)&dummy_oh;
			cancargs.alock.svid = ent->req.owner;
			/* XXX test for offset/len too large */
			cancargs.alock.l_offset = ent->req.offset;
			cancargs.alock.l_len = ent->req.len;
			if (locked) UNLOCKENT(ent);
#if ASYNC
			syslog(LOG_DEBUG, "call_cancel: NLM_CANCEL_MSG");
			ret = clnt_call(client, NLM_CANCEL_MSG, xdr_nlm_cancargs, &cancargs, xdr_void, NULL, timeo);
#else
			res.cookie.n_len = 0;
			res.cookie.n_bytes = NULL;
			syslog(LOG_DEBUG, "call_cancel: NLM_CANCEL");
			ret = clnt_call(client, NLM_CANCEL, xdr_nlm_cancargs, &cancargs, xdr_nlm_res, &res, timeo);
			if (ret) {
				syslog(LOG_NOTICE, "call_cancel: clnt_call: %s", clnt_sperrno(ret));
			} else {
				syslog(LOG_DEBUG, "call_cancel: clnt_call OK");
				*stat = (enum nlm4_stats) res.stat.stat;
				if (!clnt_freeres(client, xdr_nlm_res, &res)) {
					syslog(LOG_NOTICE, "call_cancel: freeres failed");
					ret = RPC_FAILED;
				}
			}
#endif
			break;
		case NLM_VERS4:
			serial2cookie(&serial, &cancargs4.cookie);
			cancargs4.block = ent->req.wait;
			cancargs4.exclusive = ent->req.exclusive;
			cancargs4.alock.caller_name = hostname;
			cancargs4.alock.fh.n_len = ent->req.handle_len;
			cancargs4.alock.fh.n_bytes = (char *)&ent->req.handle;
			cancargs4.alock.oh.n_len = sizeof(dummy_oh);
			cancargs4.alock.oh.n_bytes = (char *)&dummy_oh;
			cancargs4.alock.svid = ent->req.owner;
			cancargs4.alock.l_offset = ent->req.offset;
			cancargs4.alock.l_len = ent->req.len;
			if (locked) UNLOCKENT(ent);
#if ASYNC
			syslog(LOG_DEBUG, "call_cancel: NLM4_CANCEL_MSG");
			ret = clnt_call(client, NLM4_CANCEL_MSG, xdr_nlm4_cancargs, &cancargs4, xdr_void, NULL, timeo);
#else
			res4.cookie.n_len = 0;
			res4.cookie.n_bytes = NULL;
			syslog(LOG_DEBUG, "call_cancel: NLM4_CANCEL");
			ret = clnt_call(client, NLM4_CANCEL_MSG, xdr_nlm4_cancargs, &cancargs4, xdr_nlm4_res, &res4, timeo);
			if (ret) {
				syslog(LOG_NOTICE, "call_cancel: clnt_call: %s", clnt_sperrno(ret));
			} else {
				syslog(LOG_DEBUG, "call_cancel: clnt_call OK");
				*stat = res4.stat.stat;
				if (!clnt_freeres(client, xdr_nlm4_res, &res4)) {
					syslog(LOG_NOTICE, "call_cancel: freeres failed");
					ret = RPC_FAILED;
				}
			}
#endif
			break;
		default:
			syslog(LOG_WARNING, "call_cancel: nlm_vers=%d", nlm_vers);
			return RPC_FAILED;
	}
	return ret;
}

/*
 * Make an NLM_UNLOCK/NLM4_UNLOCK (or NLM_UNLOCK_MSG/NLM4_UNLOCK_MSG) RPC.
 * Just as boring as the above three routines, but with nlm_unlockargs.
 */
#if ASYNC
static int call_unlock(struct reclaim_ent *ent, int locked)
#else
static int call_unlock(struct reclaim_ent *ent, int locked, enum nlm4_stats *stat)
#endif
{
	CLIENT *client;
	int nlm_vers;
	struct timeval timeo;
	struct nlm_unlockargs unlockargs;
	struct nlm4_unlockargs unlockargs4;
#if !ASYNC
	struct nlm_res res;
	struct nlm4_res res4;
#endif
	int serial;
	int ret;

	nlm_vers = nfs2nlm(ent->req.mount_version);
	LOCKPROC;
	client = get_client((struct sockaddr *)&ent->req.mount_nam, nlm_vers);
	if (client == NULL) {
		syslog(LOG_WARNING, "call_unlock: get_client failed");
		ret = rpc_createerr.cf_stat;
		UNLOCKPROC;
		return ret;
	}
	UNLOCKPROC;
	settimeo(&ent->req, &timeo);
	serial = ent->req.serial;

	switch (nlm_vers) {
		case 0:
			syslog(LOG_WARNING, "call_unlock: mount_version=%d", ent->req.mount_version);
			return RPC_FAILED;
		case NLM_VERS:
			serial2cookie(&serial, &unlockargs.cookie);
			unlockargs.alock.caller_name = hostname;
			unlockargs.alock.fh.n_len = ent->req.handle_len;
			unlockargs.alock.fh.n_bytes = (char *)&ent->req.handle;
			unlockargs.alock.oh.n_len = sizeof(dummy_oh);
			unlockargs.alock.oh.n_bytes = (char *)&dummy_oh;
			unlockargs.alock.svid = ent->req.owner;
			/* XXX test for offset/len too large */
			unlockargs.alock.l_offset = ent->req.offset;
			unlockargs.alock.l_len = ent->req.len;
			if (locked) UNLOCKENT(ent);
#if ASYNC
			syslog(LOG_DEBUG, "call_unlock: NLM_UNLOCK_MSG");
			ret = clnt_call(client, NLM_UNLOCK_MSG, xdr_nlm_unlockargs, &unlockargs, xdr_void, NULL, timeo);
#else
			res.cookie.n_len = 0;
			res.cookie.n_bytes = NULL;
			syslog(LOG_DEBUG, "call_unlock: NLM_UNLOCK");
			ret = clnt_call(client, NLM_UNLOCK, xdr_nlm_unlockargs, &unlockargs, xdr_nlm_res, &res, timeo);
			if (ret) {
				syslog(LOG_NOTICE, "call_unlock: clnt_call: %s", clnt_sperrno(ret));
			} else {
				syslog(LOG_DEBUG, "call_unlock: clnt_call OK");
				*stat = (enum nlm4_stats) res.stat.stat;
				if (!clnt_freeres(client, xdr_nlm_res, &res)) {
					syslog(LOG_NOTICE, "call_unlock: freeres failed");
					ret = RPC_FAILED;
				}
			}
#endif
			break;
		case NLM_VERS4:
			serial2cookie(&serial, &unlockargs4.cookie);
			unlockargs4.alock.caller_name = hostname;
			unlockargs4.alock.fh.n_len = ent->req.handle_len;
			unlockargs4.alock.fh.n_bytes = (char *)&ent->req.handle;
			unlockargs4.alock.oh.n_len = sizeof(dummy_oh);
			unlockargs4.alock.oh.n_bytes = (char *)&dummy_oh;
			unlockargs4.alock.svid = ent->req.owner;
			unlockargs4.alock.l_offset = ent->req.offset;
			unlockargs4.alock.l_len = ent->req.len;
			if (locked) UNLOCKENT(ent);
#if ASYNC
			syslog(LOG_DEBUG, "call_unlock: NLM4_UNLOCK_MSG");
			ret = clnt_call(client, NLM4_UNLOCK_MSG, xdr_nlm4_unlockargs, &unlockargs4, xdr_void, NULL, timeo);
#else
			res4.cookie.n_len = 0;
			res4.cookie.n_bytes = NULL;
			syslog(LOG_DEBUG, "call_unlock: NLM4_UNLOCK");
			ret = clnt_call(client, NLM4_UNLOCK, xdr_nlm4_unlockargs, &unlockargs4, xdr_nlm4_res, &res4, timeo);
			if (ret) {
				syslog(LOG_NOTICE, "call_unlock: clnt_call: %s", clnt_sperrno(ret));
			} else {
				syslog(LOG_DEBUG, "call_unlock: clnt_call OK");
				*stat = res4.stat.stat;
				if (!clnt_freeres(client, xdr_nlm4_res, &res4)) {
					syslog(LOG_NOTICE, "call_unlock: freeres failed");
					ret = RPC_FAILED;
				}
			}
#endif
			break;
		default:
			syslog(LOG_WARNING, "call_unlock: nlm_vers=%d", nlm_vers);
			return RPC_FAILED;
	}
	return ret;
}

/*
 * Handle an NLM_TEST_RES (or NLM4_TEST_RES) RPC.
 * Answer to the kernel according to the returned state.
 */
void testres(netobj cookie, const nlm4_testrply *testrply)
{
#if ASYNC
	struct nfslock_rep rep;
	
	if (cookie2serial(&cookie, &rep.serial)) {
		syslog(LOG_WARNING, "testres(%d): can't extract serial", testrply->stat);
		return;
	}
	syslog(LOG_DEBUG, "testres(%d): serial %d", testrply->stat, rep.serial);
	if (testrply->stat == nlm4_denied) {
		/* rep.serial set above */
		rep.errnum = 0; /* denied status is not an error if command was test */
		rep.conflict = testrply->nlm4_testrply_u.holder.exclusive ? 2 : 1;
		rep.owner = (pid_t)testrply->nlm4_testrply_u.holder.svid;
		rep.offset = testrply->nlm4_testrply_u.holder.l_offset;
		rep.len = testrply->nlm4_testrply_u.holder.l_len;
	} else {
		/* rep.serial set above */
		rep.errnum = nlmstat2errno(testrply->stat);
		rep.conflict = 0;
		rep.offset = rep.len = 0;
		rep.owner = 0;
	}
	sendrep(&rep);
#else
	syslog(LOG_NOTICE, "testres/sync");
#endif
}

/*
 * Handle an NLM_LOCK_RES (or NLM4_LOCK_RES) RPC.
 * This is the trickiest part of all (next is handling an NLM_GRANTED).
 * First, we must have a matching request in the reclaim queue.
 * Then, what to do depends on whether the lock was granted or deferred
 * and the matching entry's wait field. A value of 2 means we have already
 * answered (in the positive) via the kernel to the requesting process.
 * If the lock was granted immediately, answer to the kernel in case wait was
 * 0 or 1 (i.e., we haven't answererd yet) and record that by setting wait to 2.
 * If a lock was granted who's wait field is 2, we have already (positively)
 * answered earlier, so this must be grant during lock recovery, so don't answer.
 * If the lock was not granted immediately, only answer (in the negative) to the
 * kernel if the wait field is 0 and delete the entry.
 * If the wait field is 1, just do nothing; the process wants to wait.
 * If wait is already 2, this must be a failed lock recovery.
 * In case the status is anything else (i.e., neither granted nor blocked nor
 * denied), the server failed. Answer (in the negative) to the kernel for wait
 * values of 0 and 1, removing the entry from the recovery queue, and complain
 * for a value of 2, which means a failed recovery.
 */
void lockres(netobj cookie, nlm4_stats stat)
{
#if ASYNC
	struct reclaim_ent *ent;
	struct nfslock_rep rep;
	
	ent = findcookie(&cookie);
	if (ent) {
		/* list and entry locked */
		syslog(LOG_DEBUG, "lockres(%d): entry %d, wait=%d", stat, ent->req.serial, ent->req.wait);
		rep.serial = ent->req.serial;
		rep.errnum = nlmstat2errno(stat);
		rep.conflict = 0;
		rep.offset = rep.len = 0;
		rep.owner = 0;
		switch (stat) {
			case nlm4_granted:
				/* list unlocked, entry locked */
				switch (ent->req.wait) {
					case 0: /* non-waiting */
					case 1: /* waiting */
						ent->req.wait = 2;
						UNLOCKENT(ent);
						adjustlocks(&ent->req, "lockres");
						UNLOCKLIST;
						sendrep(&rep);
						break;
					case 2: /* already answered, must be recovery */
						UNLOCKLIST;
						if (ent->fragment)
							syslog(LOG_NOTICE, "recovered lock %d, fragment %d", ent->req.serial, ent->fragment);
						else
							syslog(LOG_NOTICE, "recovered lock %d", ent->req.serial);
						break;
					default:/* oops */
						UNLOCKLIST;
						syslog(LOG_WARNING, "lockres/granted, wait=%d", ent->req.wait);
						break;
				}
				break;
			case nlm4_blocked:
				/* list and entry locked */
				switch (ent->req.wait) {
					case 0:	/* strange */
						syslog(LOG_WARNING, "lockres/blocked/nowait");
						UNLOCKENT(ent);
						FREEENT(REMOVEENT(ent));
						UNLOCKLIST;
						sendrep(&rep);
						break;
					case 1: /* waiting: don't reply, will wait for grant */
						UNLOCKENT(ent);
						UNLOCKLIST;
						break;
					case 2: /* already answered, must be recovery */
						if (ent->fragment)
							syslog(LOG_INFO, "blocked recovery of lock %d, fragment %d", ent->req.serial, ent->fragment);
						else
							syslog(LOG_INFO, "blocked recovery of lock %d", ent->req.serial);
						UNLOCKENT(ent);
						UNLOCKLIST;
						break;
					default:/* oops */
						syslog(LOG_WARNING, "lockres/blocked: wait=%d", ent->req.wait);
						UNLOCKENT(ent);
						UNLOCKLIST;
						break;
				}
				break;
			case nlm4_denied:
				/* list and entry locked */
				switch (ent->req.wait) {
					case 0:	/* answer and remove */
						/* Posix semantics want EAGAIN, not EWOULDBLOCK */
						if (ent->req.posix) rep.errnum = EAGAIN;
						/* put entry off the list */
						UNLOCKENT(ent);
						FREEENT(REMOVEENT(ent));
						UNLOCKLIST;
						sendrep(&rep);
						break;
					case 1: /* strange */
						syslog(LOG_WARNING, "lockres/denied/wait");
						UNLOCKENT(ent);
						UNLOCKLIST;
						break;
					case 2: /* already answered, must be recovery */
						if (ent->fragment)
							syslog(LOG_INFO, "denied recovery of lock %d, ffragment %d", ent->req.serial, ent->fragment);
						else
							syslog(LOG_INFO, "denied recovery of lock %d", ent->req.serial);
						UNLOCKENT(ent);
						UNLOCKLIST;
						break;
					default:/* oops */
						syslog(LOG_WARNING, "lockres/denied: wait=%d", ent->req.wait);
						UNLOCKENT(ent);
						FREEENT(REMOVEENT(ent));
						UNLOCKLIST;
						break;
				}
				break;
			default: /* some error */
				/* list and entry locked */
				switch (ent->req.wait) {
					case 0: /* non-waiting */
					case 1: /* waiting */
						UNLOCKENT(ent);
						FREEENT(REMOVEENT(ent));
						UNLOCKLIST;
						sendrep(&rep);
						break;
					case 2: /* must be recovery */
						UNLOCKENT(ent);
						UNLOCKLIST;
						if (ent->fragment)
							syslog(LOG_WARNING, "error %d recovering lock %d, fragment %d", stat, ent->req.serial, ent->fragment);
						else
							syslog(LOG_WARNING, "error %d recovering lock %d", stat, ent->req.serial);
						/* leave on list, better luck next crash */
						break;
					default: /* oops */
						UNLOCKENT(ent);
						FREEENT(REMOVEENT(ent));
						UNLOCKLIST;
						syslog(LOG_WARNING, "lockres/granted, wait=%d", ent->req.wait);
						break;
				}
				break;
		}
	} else {
		/* nothing locked */
		syslog(LOG_WARNING, "lockres: cookie not found");
	}
#else
	syslog(LOG_NOTICE, "lockres/sync");
#endif
}

/*
 * Handle an NLM_CANCEL_RES (or NLM4_CANCEL_RES) RPC.
 * We don't need to handle or reply anything since the cancel
 * can't have originated from userland.
 */
void cancelres(netobj cookie, nlm4_stats stat)
{
#if ASYNC
	syslog(LOG_DEBUG, "cancelres(%d)", stat);
	/* This space intentionally left blank */
#else
	syslog(LOG_NOTICE, "cancelres/sync");
#endif
}

/*
 * Handle an NLM_UNLOCK_RES (or NLM4_UNLOCK_RES) RPC.
 * Just reply to the kernel.
 */
void unlockres(netobj cookie, nlm4_stats stat)
{
#if ASYNC
	struct reclaim_ent *ent;
	struct nfslock_rep rep;

	ent = findcookie(&cookie);
	if (ent) {
		/* list and entry locked */
		syslog(LOG_DEBUG, "unlockres(%d): entry %d, wait=%d", stat, ent->req.serial, ent->req.wait);
		UNLOCKENT(ent);
		adjustlocks(&ent->req, "unlockres");
		UNLOCKLIST;
		rep.serial = ent->req.serial;
		rep.conflict = 0;
		rep.offset = rep.len = 0;
		rep.owner = 0;
		rep.errnum = nlmstat2errno(stat);
		sendrep(&rep);
	}
#else
	syslog(LOG_NOTICE, "cancelres/sync");
#endif
}

/*
 * Handle an NLM_GRANTED (or NLM4_GRANTED) RPC.
 * This is the second-to-trickiest after handling NLM_LOCK_RES.
 * First, we must have a matching request in the reclaim queue.
 * What to do depends on that entry's wait field.
 * If wait is 0, that's strange since we don't expect a GRANTED.
 * If wait is 1, it means a lock a process has been waiting for has been
 *   granted, so reply to the kernel and log the fact we have replied by
 *   setting wait to 2.
 * If wait is 2, this must be a reclaimed lock we couldn't re-establish
 *   immediately.
 */
nlm_stats granted(const struct netbuf *caller, int exclusive, const nlm4_lock *lock)
{
	struct reclaim_ent *ent;
	struct nfslock_rep rep;
	
	ent = findlock(caller, exclusive, lock);
	if (ent) {
		syslog(LOG_DEBUG, "granted: entry %d, wait=%d", ent->req.serial, ent->req.wait);
		rep.serial = ent->req.serial;
        	rep.errnum = 0;
        	rep.conflict = 0;
        	rep.offset = rep.len = 0;
        	rep.owner = 0;
		switch (ent->req.wait) {
			case 0: /* granted for non-blocking request? */
				UNLOCKENT(ent);
				UNLOCKLIST;
				syslog(LOG_NOTICE, "granted: wait=0");
				break;
			case 1: /* blocking request */
				UNLOCKENT(ent);
				adjustlocks(&ent->req, "granted");
				UNLOCKLIST;
				ent->req.wait = 2;
				sendrep(&rep);
				break;
			case 2: /* already answered, must be lock recovery */
				UNLOCKENT(ent);
				UNLOCKLIST;
				if (ent->fragment)
					syslog(LOG_INFO, "granted recovery of lock %d, fragment %d", ent->req.serial, ent->fragment);
				else
					syslog(LOG_INFO, "granted recovery of lock %d", ent->req.serial);
				break;
			default:/* oops */
				UNLOCKENT(ent);
				UNLOCKLIST;
				syslog(LOG_WARNING, "granted: wait=%d", ent->req.wait);
				break;
		}
		return nlm_granted;
	} else {
		syslog(LOG_WARNING, "granted: lock not found");
		return nlm_denied;
	}
}

/*
 * Adjust locks in the recovery list.
 * Remove any overlaps with this request, which is either a lock or unlock.
 * Entered (and exited) with the list locked, but not on any element.
 */
static void adjustlocks(struct nfslock_req *req, const char *msg)
{
	struct reclaim_ent *ent, *new_ent;
	int ret;

	/* don't use FOREACH since we modify the list */
        for (ent = TAILQ_FIRST(&reclaim); ent != NULL; ent = TAILQ_NEXT(ent, link)) {
		LOCKENT(ent);
		if (&ent->req != req /* don't try to adjust yourself! */
		 && ent->req.owner == req->owner
		 && ent->req.handle_len == req->handle_len
		 && memcmp(&ent->req.handle, &req->handle, ent->req.handle_len) == 0
		 && ent->req.mount_nam.ss_len == req->mount_nam.ss_len 
		 && memcmp(&ent->req.mount_nam, &req->mount_nam, ent->req.mount_nam.ss_len) == 0) {
			if (req->offset <= ent->req.offset) {
				/* new lock begins no later than old one */
				if (req->len == 0 || (ent->req.len != 0 && req->offset + req->len >= ent->req.offset + ent->req.len)) {
					/* new lock ends no earlier than old one */
					/* so new lock totally encompasses the old one, remove it */
					if (ent->fragment)
						syslog(LOG_DEBUG, "%s %d removes %d/%d", msg, req->serial, ent->req.serial, ent->fragment);
					else
						syslog(LOG_DEBUG, "%s %d removes %d", msg, req->serial, ent->req.serial);
					UNLOCKENT(ent);
					FREEENT(REMOVEENT(ent));
				} else {
					/* new lock ends earlier than old one */
					/* so shorten the old lock at the head */
					if (ent->fragment)
						syslog(LOG_DEBUG, "%s %d shortens %d/%d at the head", msg, req->serial, ent->req.serial, ent->fragment);
					else
						syslog(LOG_DEBUG, "%s %d shortens %d at the head", msg, req->serial, ent->req.serial);
					/* this is positive due to the second test, or ent->req.len is zero */
					ent->req.len = ent->req.len ? (ent->req.offset + ent->req.len) - (req->offset - req->len) : 0;
					/* req.len is non-zero here */
					ent->req.offset = req->offset + req->len;
				}
			} else {
				/* new lock begins later than old one */
				if (req->len == 0 || (ent->req.len != 0 && req->offset + req->len >= ent->req.offset + ent->req.len)) {
					/* new lock ends no earlier than old one */
					/* so shorten the old lock at the tail */
					if (ent->fragment)
						syslog(LOG_DEBUG, "%s %d shortens %d/%d at the tail", msg, req->serial, ent->req.serial, ent->fragment);
					else
						syslog(LOG_DEBUG, "%s %d shortens %d at the tail", msg, req->serial, ent->req.serial);
					/* this is positive due to the first test */
					ent->req.len = req->offset - ent->req.offset;
				} else {
					/* new lock ends earlier than old one */
					/* so old lock strictly encompasses the new one, we have to split */
					if (ent->fragment)
						syslog(LOG_DEBUG, "%s %d splits %d/%d", msg, req->serial, ent->req.serial, ent->fragment);
					else
						syslog(LOG_DEBUG, "%s %d splits %d", msg, req->serial, ent->req.serial);
					new_ent = malloc(sizeof(*new_ent));
					if (new_ent == NULL) {
						syslog(LOG_WARNING, "out of memory splitting lock");
					} else {
						ret = INITLOCK(new_ent);
						if (ret) {
							syslog(LOG_WARNING, "can't init thread lock splitting lock: %s", strerror(ret));
						} else {
							if (ent->fragment == 0) {
								ent->fragment = 1;
								new_ent->fragment = 2;
							} else {
								new_ent->fragment = ent->fragment + 1;
							}
							new_ent->req = ent->req;
							/* calculate the second fragment first */
							/* either ent->req.len is zero or this is positive due to the second test */
							new_ent->req.len = ent->req.len ? (ent->req.offset + ent->req.len) - (req->offset + req->len) : 0;
							/* req->len is non-zero due to the second test */
							new_ent->req.offset = req->offset + req->len;
							/* now destructively calculate the first fragment */
							/* this is positive due to the first test */
							ent->req.len = req->offset - ent->req.offset;
							/* insert second fragment immediately after the first one */
							TAILQ_INSERT_AFTER(&reclaim, ent, new_ent, link);
							ent = new_ent; /* skip inserted entry */
						}
					}
				}
			}
		}
		UNLOCKENT(ent);
	}
}
