/******************************************************************************
 * Top contributors (to current version):
 *   Andrew Reynolds, Tim King, Abdalrhman Mohamed
 *
 * This file is part of the cvc5 project.
 *
 * Copyright (c) 2009-2022 by the authors listed in the file AUTHORS
 * in the top-level source directory and their institutional affiliations.
 * All rights reserved.  See the file COPYING in the top-level source
 * directory for licensing information.
 * ****************************************************************************
 *
 * Implementation of command objects.
 */

#include "smt/command.h"

#include <exception>
#include <iostream>
#include <iterator>
#include <sstream>
#include <utility>
#include <vector>

#include "api/cpp/cvc5.h"
#include "base/check.h"
#include "base/modal_exception.h"
#include "base/output.h"
#include "expr/node.h"
#include "expr/symbol_manager.h"
#include "expr/type_node.h"
#include "options/io_utils.h"
#include "options/main_options.h"
#include "options/options.h"
#include "options/printer_options.h"
#include "options/smt_options.h"
#include "printer/printer.h"
#include "proof/unsat_core.h"
#include "smt/model.h"
#include "util/smt2_quote_string.h"
#include "util/utility.h"

using namespace std;

namespace cvc5 {

using namespace internal;

std::string sexprToString(cvc5::Term sexpr)
{
  // if sexpr is a constant string, return the stored constant string. We don't
  // call Term::toString as its result depends on the output language.
  // Notice that we only check for string constants. The sexprs generated by the
  // parser don't contains other constants, so we can ignore them.
  if (sexpr.isStringValue())
  {
    // convert std::wstring to std::string
    std::wstring wstring = sexpr.getStringValue();
    return std::string(wstring.cbegin(), wstring.cend());
  }

  // if sexpr is not a spec constant, make sure it is an array of sub-sexprs
  Assert(sexpr.getKind() == cvc5::SEXPR);

  std::stringstream ss;
  auto it = sexpr.begin();

  // recursively print the sub-sexprs
  ss << '(' << sexprToString(*it);
  ++it;
  while (it != sexpr.end())
  {
    ss << ' ' << sexprToString(*it);
    ++it;
  }
  ss << ')';

  return ss.str();
}

const int CommandPrintSuccess::s_iosIndex = std::ios_base::xalloc();
const CommandSuccess* CommandSuccess::s_instance = new CommandSuccess();
const CommandInterrupted* CommandInterrupted::s_instance =
    new CommandInterrupted();

std::ostream& operator<<(std::ostream& out, const Command& c)
{
  c.toStream(out,
             options::ioutils::getNodeDepth(out),
             options::ioutils::getDagThresh(out),
             options::ioutils::getOutputLang(out));
  return out;
}

ostream& operator<<(ostream& out, const Command* c)
{
  if (c == NULL)
  {
    out << "null";
  }
  else
  {
    out << *c;
  }
  return out;
}

std::ostream& operator<<(std::ostream& out, const CommandStatus& s)
{
  s.toStream(out, options::ioutils::getOutputLang(out));
  return out;
}

ostream& operator<<(ostream& out, const CommandStatus* s)
{
  if (s == NULL)
  {
    out << "null";
  }
  else
  {
    out << *s;
  }
  return out;
}

/* -------------------------------------------------------------------------- */
/* class CommandPrintSuccess                                                  */
/* -------------------------------------------------------------------------- */

void CommandPrintSuccess::applyPrintSuccess(std::ostream& out)
{
  out.iword(s_iosIndex) = d_printSuccess;
}

bool CommandPrintSuccess::getPrintSuccess(std::ostream& out)
{
  return out.iword(s_iosIndex);
}

void CommandPrintSuccess::setPrintSuccess(std::ostream& out, bool printSuccess)
{
  out.iword(s_iosIndex) = printSuccess;
}

std::ostream& operator<<(std::ostream& out, CommandPrintSuccess cps)
{
  cps.applyPrintSuccess(out);
  return out;
}

/* -------------------------------------------------------------------------- */
/* class Command                                                              */
/* -------------------------------------------------------------------------- */

Command::Command() : d_commandStatus(nullptr), d_muted(false) {}

Command::Command(const Command& cmd)
{
  d_commandStatus =
      (cmd.d_commandStatus == NULL) ? NULL : &cmd.d_commandStatus->clone();
  d_muted = cmd.d_muted;
}

Command::~Command()
{
  if (d_commandStatus != NULL && d_commandStatus != CommandSuccess::instance())
  {
    delete d_commandStatus;
  }
}

bool Command::ok() const
{
  // either we haven't run the command yet, or it ran successfully
  return d_commandStatus == NULL
         || dynamic_cast<const CommandSuccess*>(d_commandStatus) != NULL;
}

bool Command::fail() const
{
  return d_commandStatus != NULL
         && dynamic_cast<const CommandFailure*>(d_commandStatus) != NULL;
}

bool Command::interrupted() const
{
  return d_commandStatus != NULL
         && dynamic_cast<const CommandInterrupted*>(d_commandStatus) != NULL;
}

void Command::invoke(cvc5::Solver* solver, SymbolManager* sm, std::ostream& out)
{
  invoke(solver, sm);
  if (!(isMuted() && ok()))
  {
    printResult(out);
  }
}

std::string Command::toString() const
{
  std::stringstream ss;
  toStream(ss);
  return ss.str();
}

void CommandStatus::toStream(std::ostream& out, Language language) const
{
  Printer::getPrinter(language)->toStream(out, this);
}

void Command::printResult(std::ostream& out) const
{
  if (d_commandStatus != nullptr)
  {
    out << *d_commandStatus;
  }
}

void Command::resetSolver(cvc5::Solver* solver)
{
  std::unique_ptr<Options> opts = std::make_unique<Options>();
  opts->copyValues(*solver->d_originalOptions);
  // This reconstructs a new solver object at the same memory location as the
  // current one. Note that this command does not own the solver object!
  // It may be safer to instead make the ResetCommand a special case in the
  // CommandExecutor such that this reconstruction can be done within the
  // CommandExecutor, who actually owns the solver.
  solver->~Solver();
  new (solver) cvc5::Solver(std::move(opts));
}

Node Command::termToNode(const cvc5::Term& term) { return term.getNode(); }

std::vector<Node> Command::termVectorToNodes(
    const std::vector<cvc5::Term>& terms)
{
  return cvc5::Term::termVectorToNodes(terms);
}

TypeNode Command::sortToTypeNode(const cvc5::Sort& sort)
{
  return sort.getTypeNode();
}

std::vector<TypeNode> Command::sortVectorToTypeNodes(
    const std::vector<cvc5::Sort>& sorts)
{
  return cvc5::Sort::sortVectorToTypeNodes(sorts);
}

TypeNode Command::grammarToTypeNode(cvc5::Grammar* grammar)
{
  return grammar == nullptr ? TypeNode::null()
                            : sortToTypeNode(grammar->resolve());
}

/* -------------------------------------------------------------------------- */
/* class EmptyCommand                                                         */
/* -------------------------------------------------------------------------- */

EmptyCommand::EmptyCommand(std::string name) : d_name(name) {}
std::string EmptyCommand::getName() const { return d_name; }
void EmptyCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  /* empty commands have no implementation */
  d_commandStatus = CommandSuccess::instance();
}

Command* EmptyCommand::clone() const { return new EmptyCommand(d_name); }
std::string EmptyCommand::getCommandName() const { return "empty"; }

void EmptyCommand::toStream(std::ostream& out,
                            int toDepth,
                            size_t dag,
                            Language language) const
{
  Printer::getPrinter(language)->toStreamCmdEmpty(out, d_name);
}

/* -------------------------------------------------------------------------- */
/* class EchoCommand                                                          */
/* -------------------------------------------------------------------------- */

EchoCommand::EchoCommand(std::string output) : d_output(output) {}

std::string EchoCommand::getOutput() const { return d_output; }

void EchoCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  /* we don't have an output stream here, nothing to do */
  d_commandStatus = CommandSuccess::instance();
}

void EchoCommand::invoke(cvc5::Solver* solver,
                         SymbolManager* sm,
                         std::ostream& out)
{
  out << cvc5::internal::quoteString(d_output) << std::endl;
  Trace("dtview::command") << "* ~COMMAND: echo |" << d_output << "|~"
                           << std::endl;
  d_commandStatus = CommandSuccess::instance();
  printResult(out);
}

Command* EchoCommand::clone() const { return new EchoCommand(d_output); }

std::string EchoCommand::getCommandName() const { return "echo"; }

void EchoCommand::toStream(std::ostream& out,
                           int toDepth,
                           size_t dag,
                           Language language) const
{
  Printer::getPrinter(language)->toStreamCmdEcho(out, d_output);
}

/* -------------------------------------------------------------------------- */
/* class AssertCommand                                                        */
/* -------------------------------------------------------------------------- */

AssertCommand::AssertCommand(const cvc5::Term& t) : d_term(t) {}

cvc5::Term AssertCommand::getTerm() const { return d_term; }
void AssertCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    solver->assertFormula(d_term);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* AssertCommand::clone() const { return new AssertCommand(d_term); }

std::string AssertCommand::getCommandName() const { return "assert"; }

void AssertCommand::toStream(std::ostream& out,
                             int toDepth,
                             size_t dag,
                             Language language) const
{
  Printer::getPrinter(language)->toStreamCmdAssert(out, termToNode(d_term));
}

/* -------------------------------------------------------------------------- */
/* class PushCommand                                                          */
/* -------------------------------------------------------------------------- */

PushCommand::PushCommand(uint32_t nscopes) : d_nscopes(nscopes) {}

void PushCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    solver->push(d_nscopes);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* PushCommand::clone() const { return new PushCommand(d_nscopes); }
std::string PushCommand::getCommandName() const { return "push"; }

void PushCommand::toStream(std::ostream& out,
                           int toDepth,
                           size_t dag,
                           Language language) const
{
  Printer::getPrinter(language)->toStreamCmdPush(out, d_nscopes);
}

/* -------------------------------------------------------------------------- */
/* class PopCommand                                                           */
/* -------------------------------------------------------------------------- */

PopCommand::PopCommand(uint32_t nscopes) : d_nscopes(nscopes) {}

void PopCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    solver->pop(d_nscopes);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* PopCommand::clone() const { return new PopCommand(d_nscopes); }
std::string PopCommand::getCommandName() const { return "pop"; }

void PopCommand::toStream(std::ostream& out,
                          int toDepth,
                          size_t dag,
                          Language language) const
{
  Printer::getPrinter(language)->toStreamCmdPop(out, d_nscopes);
}

/* -------------------------------------------------------------------------- */
/* class CheckSatCommand                                                      */
/* -------------------------------------------------------------------------- */

CheckSatCommand::CheckSatCommand() {}

void CheckSatCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  Trace("dtview::command") << "* ~COMMAND: " << getCommandName() << "~"
                           << std::endl;
  try
  {
    d_result = solver->checkSat();
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

cvc5::Result CheckSatCommand::getResult() const { return d_result; }

void CheckSatCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    Trace("dtview::command") << "* RESULT: " << d_result << std::endl;
    out << d_result << endl;
  }
}

Command* CheckSatCommand::clone() const
{
  CheckSatCommand* c = new CheckSatCommand();
  c->d_result = d_result;
  return c;
}

std::string CheckSatCommand::getCommandName() const { return "check-sat"; }

void CheckSatCommand::toStream(std::ostream& out,
                               int toDepth,
                               size_t dag,
                               Language language) const
{
  Printer::getPrinter(language)->toStreamCmdCheckSat(out);
}

/* -------------------------------------------------------------------------- */
/* class CheckSatAssumingCommand                                              */
/* -------------------------------------------------------------------------- */

CheckSatAssumingCommand::CheckSatAssumingCommand(cvc5::Term term)
    : d_terms({term})
{
}

CheckSatAssumingCommand::CheckSatAssumingCommand(
    const std::vector<cvc5::Term>& terms)
    : d_terms(terms)
{
}

const std::vector<cvc5::Term>& CheckSatAssumingCommand::getTerms() const
{
  return d_terms;
}

void CheckSatAssumingCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  Trace("dtview::command") << "* ~COMMAND: (check-sat-assuming ( " << d_terms
                           << " )~" << std::endl;
  try
  {
    d_result = solver->checkSatAssuming(d_terms);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

cvc5::Result CheckSatAssumingCommand::getResult() const
{
  Trace("dtview::command") << "* ~RESULT: " << d_result << "~" << std::endl;
  return d_result;
}

void CheckSatAssumingCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    out << d_result << endl;
  }
}

Command* CheckSatAssumingCommand::clone() const
{
  CheckSatAssumingCommand* c = new CheckSatAssumingCommand(d_terms);
  c->d_result = d_result;
  return c;
}

std::string CheckSatAssumingCommand::getCommandName() const
{
  return "check-sat-assuming";
}

void CheckSatAssumingCommand::toStream(std::ostream& out,
                                       int toDepth,
                                       size_t dag,
                                       Language language) const
{
  Printer::getPrinter(language)->toStreamCmdCheckSatAssuming(
      out, termVectorToNodes(d_terms));
}

/* -------------------------------------------------------------------------- */
/* class DeclareSygusVarCommand */
/* -------------------------------------------------------------------------- */

DeclareSygusVarCommand::DeclareSygusVarCommand(const std::string& id,
                                               cvc5::Term var,
                                               cvc5::Sort sort)
    : DeclarationDefinitionCommand(id), d_var(var), d_sort(sort)
{
}

cvc5::Term DeclareSygusVarCommand::getVar() const { return d_var; }
cvc5::Sort DeclareSygusVarCommand::getSort() const { return d_sort; }

void DeclareSygusVarCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  d_commandStatus = CommandSuccess::instance();
}

Command* DeclareSygusVarCommand::clone() const
{
  return new DeclareSygusVarCommand(d_symbol, d_var, d_sort);
}

std::string DeclareSygusVarCommand::getCommandName() const
{
  return "declare-var";
}

void DeclareSygusVarCommand::toStream(std::ostream& out,
                                      int toDepth,
                                      size_t dag,
                                      Language language) const
{
  Printer::getPrinter(language)->toStreamCmdDeclareVar(
      out, termToNode(d_var), sortToTypeNode(d_sort));
}

/* -------------------------------------------------------------------------- */
/* class SynthFunCommand                                                      */
/* -------------------------------------------------------------------------- */

SynthFunCommand::SynthFunCommand(const std::string& id,
                                 cvc5::Term fun,
                                 const std::vector<cvc5::Term>& vars,
                                 cvc5::Sort sort,
                                 bool isInv,
                                 cvc5::Grammar* g)
    : DeclarationDefinitionCommand(id),
      d_fun(fun),
      d_vars(vars),
      d_sort(sort),
      d_isInv(isInv),
      d_grammar(g)
{
}

cvc5::Term SynthFunCommand::getFunction() const { return d_fun; }

const std::vector<cvc5::Term>& SynthFunCommand::getVars() const
{
  return d_vars;
}

cvc5::Sort SynthFunCommand::getSort() const { return d_sort; }
bool SynthFunCommand::isInv() const { return d_isInv; }

const cvc5::Grammar* SynthFunCommand::getGrammar() const { return d_grammar; }

void SynthFunCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  sm->addFunctionToSynthesize(d_fun);
  d_commandStatus = CommandSuccess::instance();
}

Command* SynthFunCommand::clone() const
{
  return new SynthFunCommand(
      d_symbol, d_fun, d_vars, d_sort, d_isInv, d_grammar);
}

std::string SynthFunCommand::getCommandName() const
{
  return d_isInv ? "synth-inv" : "synth-fun";
}

void SynthFunCommand::toStream(std::ostream& out,
                               int toDepth,
                               size_t dag,
                               Language language) const
{
  std::vector<Node> nodeVars = termVectorToNodes(d_vars);
  Printer::getPrinter(language)->toStreamCmdSynthFun(
      out,
      termToNode(d_fun),
      nodeVars,
      d_isInv,
      d_grammar == nullptr ? TypeNode::null() : grammarToTypeNode(d_grammar));
}

/* -------------------------------------------------------------------------- */
/* class SygusConstraintCommand */
/* -------------------------------------------------------------------------- */

SygusConstraintCommand::SygusConstraintCommand(const cvc5::Term& t,
                                               bool isAssume)
    : d_term(t), d_isAssume(isAssume)
{
}

void SygusConstraintCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    if (d_isAssume)
    {
      solver->addSygusAssume(d_term);
    }
    else
    {
      solver->addSygusConstraint(d_term);
    }
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

cvc5::Term SygusConstraintCommand::getTerm() const { return d_term; }

Command* SygusConstraintCommand::clone() const
{
  return new SygusConstraintCommand(d_term, d_isAssume);
}

std::string SygusConstraintCommand::getCommandName() const
{
  return d_isAssume ? "assume" : "constraint";
}

void SygusConstraintCommand::toStream(std::ostream& out,
                                      int toDepth,
                                      size_t dag,
                                      Language language) const
{
  if (d_isAssume)
  {
    Printer::getPrinter(language)->toStreamCmdAssume(out, termToNode(d_term));
  }
  else
  {
    Printer::getPrinter(language)->toStreamCmdConstraint(out,
                                                         termToNode(d_term));
  }
}

/* -------------------------------------------------------------------------- */
/* class SygusInvConstraintCommand */
/* -------------------------------------------------------------------------- */

SygusInvConstraintCommand::SygusInvConstraintCommand(
    const std::vector<cvc5::Term>& predicates)
    : d_predicates(predicates)
{
}

SygusInvConstraintCommand::SygusInvConstraintCommand(const cvc5::Term& inv,
                                                     const cvc5::Term& pre,
                                                     const cvc5::Term& trans,
                                                     const cvc5::Term& post)
    : SygusInvConstraintCommand(std::vector<cvc5::Term>{inv, pre, trans, post})
{
}

void SygusInvConstraintCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    solver->addSygusInvConstraint(
        d_predicates[0], d_predicates[1], d_predicates[2], d_predicates[3]);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

const std::vector<cvc5::Term>& SygusInvConstraintCommand::getPredicates() const
{
  return d_predicates;
}

Command* SygusInvConstraintCommand::clone() const
{
  return new SygusInvConstraintCommand(d_predicates);
}

std::string SygusInvConstraintCommand::getCommandName() const
{
  return "inv-constraint";
}

void SygusInvConstraintCommand::toStream(std::ostream& out,
                                         int toDepth,
                                         size_t dag,
                                         Language language) const
{
  Printer::getPrinter(language)->toStreamCmdInvConstraint(
      out,
      termToNode(d_predicates[0]),
      termToNode(d_predicates[1]),
      termToNode(d_predicates[2]),
      termToNode(d_predicates[3]));
}

/* -------------------------------------------------------------------------- */
/* class CheckSynthCommand                                                    */
/* -------------------------------------------------------------------------- */

void CheckSynthCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    d_result = d_isNext ? solver->checkSynthNext() : solver->checkSynth();
    d_commandStatus = CommandSuccess::instance();
    d_solution.clear();
    // check whether we should print the status
    if (!d_result.hasSolution()
        || options::sygusOut() == options::SygusSolutionOutMode::STATUS_AND_DEF
        || options::sygusOut() == options::SygusSolutionOutMode::STATUS)
    {
      if (d_result.hasSolution())
      {
        d_solution << "feasible" << std::endl;
      }
      else if (d_result.hasNoSolution())
      {
        d_solution << "infeasible" << std::endl;
      }
      else
      {
        d_solution << "fail" << std::endl;
      }
    }
    // check whether we should print the solution
    if (d_result.hasSolution()
        && options::sygusOut() != options::SygusSolutionOutMode::STATUS)
    {
      std::vector<cvc5::Term> synthFuns = sm->getFunctionsToSynthesize();
      d_solution << "(" << std::endl;
      Printer* p = Printer::getPrinter(Language::LANG_SYGUS_V2);
      for (cvc5::Term& f : synthFuns)
      {
        cvc5::Term sol = solver->getSynthSolution(f);
        std::vector<cvc5::Term> formals;
        if (sol.getKind() == cvc5::LAMBDA)
        {
          formals.insert(formals.end(), sol[0].begin(), sol[0].end());
          sol = sol[1];
        }
        cvc5::Sort rangeSort = f.getSort();
        if (rangeSort.isFunction())
        {
          rangeSort = rangeSort.getFunctionCodomainSort();
        }
        p->toStreamCmdDefineFunction(d_solution,
                                     f.toString(),
                                     termVectorToNodes(formals),
                                     sortToTypeNode(rangeSort),
                                     termToNode(sol));
      }
      d_solution << ")" << std::endl;
    }
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

cvc5::SynthResult CheckSynthCommand::getResult() const { return d_result; }
void CheckSynthCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    out << d_solution.str();
  }
}

Command* CheckSynthCommand::clone() const { return new CheckSynthCommand(); }

std::string CheckSynthCommand::getCommandName() const
{
  return d_isNext ? "check-synth-next" : "check-synth";
}

void CheckSynthCommand::toStream(std::ostream& out,
                                 int toDepth,
                                 size_t dag,
                                 Language language) const
{
  if (d_isNext)
  {
    Printer::getPrinter(language)->toStreamCmdCheckSynthNext(out);
  }
  else
  {
    Printer::getPrinter(language)->toStreamCmdCheckSynth(out);
  }
}

/* -------------------------------------------------------------------------- */
/* class ResetCommand                                                         */
/* -------------------------------------------------------------------------- */

void ResetCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    sm->reset();
    resetSolver(solver);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* ResetCommand::clone() const { return new ResetCommand(); }
std::string ResetCommand::getCommandName() const { return "reset"; }

void ResetCommand::toStream(std::ostream& out,
                            int toDepth,
                            size_t dag,
                            Language language) const
{
  Printer::getPrinter(language)->toStreamCmdReset(out);
}

/* -------------------------------------------------------------------------- */
/* class ResetAssertionsCommand                                               */
/* -------------------------------------------------------------------------- */

void ResetAssertionsCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    sm->resetAssertions();
    solver->resetAssertions();
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* ResetAssertionsCommand::clone() const
{
  return new ResetAssertionsCommand();
}

std::string ResetAssertionsCommand::getCommandName() const
{
  return "reset-assertions";
}

void ResetAssertionsCommand::toStream(std::ostream& out,
                                      int toDepth,
                                      size_t dag,
                                      Language language) const
{
  Printer::getPrinter(language)->toStreamCmdResetAssertions(out);
}

/* -------------------------------------------------------------------------- */
/* class QuitCommand                                                          */
/* -------------------------------------------------------------------------- */

void QuitCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  d_commandStatus = CommandSuccess::instance();
}

Command* QuitCommand::clone() const { return new QuitCommand(); }
std::string QuitCommand::getCommandName() const { return "exit"; }

void QuitCommand::toStream(std::ostream& out,
                           int toDepth,
                           size_t dag,
                           Language language) const
{
  Printer::getPrinter(language)->toStreamCmdQuit(out);
}

/* -------------------------------------------------------------------------- */
/* class CommandSequence                                                      */
/* -------------------------------------------------------------------------- */

CommandSequence::CommandSequence() : d_index(0) {}
CommandSequence::~CommandSequence()
{
  for (unsigned i = d_index; i < d_commandSequence.size(); ++i)
  {
    delete d_commandSequence[i];
  }
}

void CommandSequence::addCommand(Command* cmd)
{
  d_commandSequence.push_back(cmd);
}

void CommandSequence::clear() { d_commandSequence.clear(); }
void CommandSequence::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  for (; d_index < d_commandSequence.size(); ++d_index)
  {
    d_commandSequence[d_index]->invoke(solver, sm);
    if (!d_commandSequence[d_index]->ok())
    {
      // abort execution
      d_commandStatus = d_commandSequence[d_index]->getCommandStatus();
      return;
    }
    delete d_commandSequence[d_index];
  }

  AlwaysAssert(d_commandStatus == NULL);
  d_commandStatus = CommandSuccess::instance();
}

void CommandSequence::invoke(cvc5::Solver* solver,
                             SymbolManager* sm,
                             std::ostream& out)
{
  for (; d_index < d_commandSequence.size(); ++d_index)
  {
    d_commandSequence[d_index]->invoke(solver, sm, out);
    if (!d_commandSequence[d_index]->ok())
    {
      // abort execution
      d_commandStatus = d_commandSequence[d_index]->getCommandStatus();
      return;
    }
    delete d_commandSequence[d_index];
  }

  AlwaysAssert(d_commandStatus == NULL);
  d_commandStatus = CommandSuccess::instance();
}

Command* CommandSequence::clone() const
{
  CommandSequence* seq = new CommandSequence();
  for (const_iterator i = begin(); i != end(); ++i)
  {
    seq->addCommand((*i)->clone());
  }
  seq->d_index = d_index;
  return seq;
}

CommandSequence::const_iterator CommandSequence::begin() const
{
  return d_commandSequence.begin();
}

CommandSequence::const_iterator CommandSequence::end() const
{
  return d_commandSequence.end();
}

CommandSequence::iterator CommandSequence::begin()
{
  return d_commandSequence.begin();
}

CommandSequence::iterator CommandSequence::end()
{
  return d_commandSequence.end();
}

std::string CommandSequence::getCommandName() const { return "sequence"; }

void CommandSequence::toStream(std::ostream& out,
                               int toDepth,
                               size_t dag,
                               Language language) const
{
  Printer::getPrinter(language)->toStreamCmdCommandSequence(out,
                                                            d_commandSequence);
}

/* -------------------------------------------------------------------------- */
/* class DeclarationSequence                                                  */
/* -------------------------------------------------------------------------- */

void DeclarationSequence::toStream(std::ostream& out,
                                   int toDepth,
                                   size_t dag,
                                   Language language) const
{
  Printer::getPrinter(language)->toStreamCmdDeclarationSequence(
      out, d_commandSequence);
}

/* -------------------------------------------------------------------------- */
/* class DeclarationDefinitionCommand                                         */
/* -------------------------------------------------------------------------- */

DeclarationDefinitionCommand::DeclarationDefinitionCommand(
    const std::string& id)
    : d_symbol(id)
{
}

std::string DeclarationDefinitionCommand::getSymbol() const { return d_symbol; }

/* -------------------------------------------------------------------------- */
/* class DeclareFunctionCommand                                               */
/* -------------------------------------------------------------------------- */

DeclareFunctionCommand::DeclareFunctionCommand(const std::string& id,
                                               cvc5::Term func,
                                               cvc5::Sort sort)
    : DeclarationDefinitionCommand(id), d_func(func), d_sort(sort)
{
}

cvc5::Term DeclareFunctionCommand::getFunction() const { return d_func; }
cvc5::Sort DeclareFunctionCommand::getSort() const { return d_sort; }

void DeclareFunctionCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  // mark that it will be printed in the model
  sm->addModelDeclarationTerm(d_func);
  d_commandStatus = CommandSuccess::instance();
}

Command* DeclareFunctionCommand::clone() const
{
  DeclareFunctionCommand* dfc =
      new DeclareFunctionCommand(d_symbol, d_func, d_sort);
  return dfc;
}

std::string DeclareFunctionCommand::getCommandName() const
{
  return "declare-fun";
}

void DeclareFunctionCommand::toStream(std::ostream& out,
                                      int toDepth,
                                      size_t dag,
                                      Language language) const
{
  Printer::getPrinter(language)->toStreamCmdDeclareFunction(
      out, d_symbol, sortToTypeNode(d_func.getSort()));
}

/* -------------------------------------------------------------------------- */
/* class DeclarePoolCommand                                               */
/* -------------------------------------------------------------------------- */

DeclarePoolCommand::DeclarePoolCommand(const std::string& id,
                                       cvc5::Term func,
                                       cvc5::Sort sort,
                                       const std::vector<cvc5::Term>& initValue)
    : DeclarationDefinitionCommand(id),
      d_func(func),
      d_sort(sort),
      d_initValue(initValue)
{
}

cvc5::Term DeclarePoolCommand::getFunction() const { return d_func; }
cvc5::Sort DeclarePoolCommand::getSort() const { return d_sort; }
const std::vector<cvc5::Term>& DeclarePoolCommand::getInitialValue() const
{
  return d_initValue;
}

void DeclarePoolCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  // Notice that the pool is already declared by the parser so that it the
  // symbol is bound eagerly. This is analogous to DeclareSygusVarCommand.
  // Hence, we do nothing here.
  d_commandStatus = CommandSuccess::instance();
}

Command* DeclarePoolCommand::clone() const
{
  DeclarePoolCommand* dfc =
      new DeclarePoolCommand(d_symbol, d_func, d_sort, d_initValue);
  return dfc;
}

std::string DeclarePoolCommand::getCommandName() const
{
  return "declare-pool";
}

void DeclarePoolCommand::toStream(std::ostream& out,
                                  int toDepth,
                                  size_t dag,
                                  Language language) const
{
  Printer::getPrinter(language)->toStreamCmdDeclarePool(
      out,
      d_func.toString(),
      sortToTypeNode(d_sort),
      termVectorToNodes(d_initValue));
}

/* -------------------------------------------------------------------------- */
/* class DeclareOracleFunCommand */
/* -------------------------------------------------------------------------- */

DeclareOracleFunCommand::DeclareOracleFunCommand(const std::string& id,
                                                 Sort sort)
    : d_id(id), d_sort(sort), d_binName("")
{
}
DeclareOracleFunCommand::DeclareOracleFunCommand(const std::string& id,
                                                 Sort sort,
                                                 const std::string& binName)
    : d_id(id), d_sort(sort), d_binName(binName)
{
}

const std::string& DeclareOracleFunCommand::getIdentifier() const
{
  return d_id;
}

Sort DeclareOracleFunCommand::getSort() const { return d_sort; }

const std::string& DeclareOracleFunCommand::getBinaryName() const
{
  return d_binName;
}

void DeclareOracleFunCommand::invoke(Solver* solver, SymbolManager* sm)
{
  std::vector<Sort> args;
  Sort ret;
  if (d_sort.isFunction())
  {
    args = d_sort.getFunctionDomainSorts();
    ret = d_sort.getFunctionCodomainSort();
  }
  else
  {
    ret = d_sort;
  }
  // will call solver declare oracle function when available in API
  d_commandStatus = CommandSuccess::instance();
}

Command* DeclareOracleFunCommand::clone() const
{
  DeclareOracleFunCommand* dfc =
      new DeclareOracleFunCommand(d_id, d_sort, d_binName);
  return dfc;
}

std::string DeclareOracleFunCommand::getCommandName() const
{
  return "declare-oracle-fun";
}

void DeclareOracleFunCommand::toStream(std::ostream& out,
                                       int toDepth,
                                       size_t dag,
                                       Language language) const
{
  Printer::getPrinter(language)->toStreamCmdDeclareOracleFun(
      out, d_id, sortToTypeNode(d_sort), d_binName);
}

/* -------------------------------------------------------------------------- */
/* class DeclareSortCommand                                                   */
/* -------------------------------------------------------------------------- */

DeclareSortCommand::DeclareSortCommand(const std::string& id,
                                       size_t arity,
                                       cvc5::Sort sort)
    : DeclarationDefinitionCommand(id), d_arity(arity), d_sort(sort)
{
}

size_t DeclareSortCommand::getArity() const { return d_arity; }
cvc5::Sort DeclareSortCommand::getSort() const { return d_sort; }
void DeclareSortCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  // mark that it will be printed in the model, if it is an uninterpreted
  // sort (arity 0)
  if (d_arity == 0)
  {
    sm->addModelDeclarationSort(d_sort);
  }
  d_commandStatus = CommandSuccess::instance();
}

Command* DeclareSortCommand::clone() const
{
  return new DeclareSortCommand(d_symbol, d_arity, d_sort);
}

std::string DeclareSortCommand::getCommandName() const
{
  return "declare-sort";
}

void DeclareSortCommand::toStream(std::ostream& out,
                                  int toDepth,
                                  size_t dag,
                                  Language language) const
{
  Printer::getPrinter(language)->toStreamCmdDeclareType(out,
                                                        sortToTypeNode(d_sort));
}

/* -------------------------------------------------------------------------- */
/* class DefineSortCommand                                                    */
/* -------------------------------------------------------------------------- */

DefineSortCommand::DefineSortCommand(const std::string& id, cvc5::Sort sort)
    : DeclarationDefinitionCommand(id), d_params(), d_sort(sort)
{
}

DefineSortCommand::DefineSortCommand(const std::string& id,
                                     const std::vector<cvc5::Sort>& params,
                                     cvc5::Sort sort)
    : DeclarationDefinitionCommand(id), d_params(params), d_sort(sort)
{
}

const std::vector<cvc5::Sort>& DefineSortCommand::getParameters() const
{
  return d_params;
}

cvc5::Sort DefineSortCommand::getSort() const { return d_sort; }
void DefineSortCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  d_commandStatus = CommandSuccess::instance();
}

Command* DefineSortCommand::clone() const
{
  return new DefineSortCommand(d_symbol, d_params, d_sort);
}

std::string DefineSortCommand::getCommandName() const { return "define-sort"; }

void DefineSortCommand::toStream(std::ostream& out,
                                 int toDepth,
                                 size_t dag,
                                 Language language) const
{
  Printer::getPrinter(language)->toStreamCmdDefineType(
      out, d_symbol, sortVectorToTypeNodes(d_params), sortToTypeNode(d_sort));
}

/* -------------------------------------------------------------------------- */
/* class DefineFunctionCommand                                                */
/* -------------------------------------------------------------------------- */

DefineFunctionCommand::DefineFunctionCommand(const std::string& id,
                                             cvc5::Sort sort,
                                             cvc5::Term formula)
    : DeclarationDefinitionCommand(id),
      d_formals(),
      d_sort(sort),
      d_formula(formula)
{
}

DefineFunctionCommand::DefineFunctionCommand(
    const std::string& id,
    const std::vector<cvc5::Term>& formals,
    cvc5::Sort sort,
    cvc5::Term formula)
    : DeclarationDefinitionCommand(id),
      d_formals(formals),
      d_sort(sort),
      d_formula(formula)
{
}

const std::vector<cvc5::Term>& DefineFunctionCommand::getFormals() const
{
  return d_formals;
}

cvc5::Sort DefineFunctionCommand::getSort() const { return d_sort; }

cvc5::Term DefineFunctionCommand::getFormula() const { return d_formula; }

void DefineFunctionCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    bool global = sm->getGlobalDeclarations();
    cvc5::Term fun =
        solver->defineFun(d_symbol, d_formals, d_sort, d_formula, global);
    sm->getSymbolTable()->bind(d_symbol, fun, global);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* DefineFunctionCommand::clone() const
{
  return new DefineFunctionCommand(d_symbol, d_formals, d_sort, d_formula);
}

std::string DefineFunctionCommand::getCommandName() const
{
  return "define-fun";
}

void DefineFunctionCommand::toStream(std::ostream& out,
                                     int toDepth,
                                     size_t dag,
                                     Language language) const
{
  Printer::getPrinter(language)->toStreamCmdDefineFunction(
      out,
      d_symbol,
      termVectorToNodes(d_formals),
      sortToTypeNode(d_sort),
      termToNode(d_formula));
}

/* -------------------------------------------------------------------------- */
/* class DefineFunctionRecCommand                                             */
/* -------------------------------------------------------------------------- */

DefineFunctionRecCommand::DefineFunctionRecCommand(
    cvc5::Term func, const std::vector<cvc5::Term>& formals, cvc5::Term formula)
{
  d_funcs.push_back(func);
  d_formals.push_back(formals);
  d_formulas.push_back(formula);
}

DefineFunctionRecCommand::DefineFunctionRecCommand(
    const std::vector<cvc5::Term>& funcs,
    const std::vector<std::vector<cvc5::Term>>& formals,
    const std::vector<cvc5::Term>& formulas)
    : d_funcs(funcs), d_formals(formals), d_formulas(formulas)
{
}

const std::vector<cvc5::Term>& DefineFunctionRecCommand::getFunctions() const
{
  return d_funcs;
}

const std::vector<std::vector<cvc5::Term>>&
DefineFunctionRecCommand::getFormals() const
{
  return d_formals;
}

const std::vector<cvc5::Term>& DefineFunctionRecCommand::getFormulas() const
{
  return d_formulas;
}

void DefineFunctionRecCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    bool global = sm->getGlobalDeclarations();
    solver->defineFunsRec(d_funcs, d_formals, d_formulas, global);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* DefineFunctionRecCommand::clone() const
{
  return new DefineFunctionRecCommand(d_funcs, d_formals, d_formulas);
}

std::string DefineFunctionRecCommand::getCommandName() const
{
  return "define-fun-rec";
}

void DefineFunctionRecCommand::toStream(std::ostream& out,
                                        int toDepth,
                                        size_t dag,
                                        Language language) const
{
  std::vector<std::vector<Node>> formals;
  formals.reserve(d_formals.size());
  for (const std::vector<cvc5::Term>& formal : d_formals)
  {
    formals.push_back(termVectorToNodes(formal));
  }

  Printer::getPrinter(language)->toStreamCmdDefineFunctionRec(
      out, termVectorToNodes(d_funcs), formals, termVectorToNodes(d_formulas));
}
/* -------------------------------------------------------------------------- */
/* class DeclareHeapCommand                                                   */
/* -------------------------------------------------------------------------- */
DeclareHeapCommand::DeclareHeapCommand(cvc5::Sort locSort, cvc5::Sort dataSort)
    : d_locSort(locSort), d_dataSort(dataSort)
{
}

cvc5::Sort DeclareHeapCommand::getLocationSort() const { return d_locSort; }
cvc5::Sort DeclareHeapCommand::getDataSort() const { return d_dataSort; }

void DeclareHeapCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  solver->declareSepHeap(d_locSort, d_dataSort);
}

Command* DeclareHeapCommand::clone() const
{
  return new DeclareHeapCommand(d_locSort, d_dataSort);
}

std::string DeclareHeapCommand::getCommandName() const
{
  return "declare-heap";
}

void DeclareHeapCommand::toStream(std::ostream& out,
                                  int toDepth,
                                  size_t dag,
                                  Language language) const
{
  Printer::getPrinter(language)->toStreamCmdDeclareHeap(
      out, sortToTypeNode(d_locSort), sortToTypeNode(d_dataSort));
}

/* -------------------------------------------------------------------------- */
/* class SimplifyCommand                                                      */
/* -------------------------------------------------------------------------- */

SimplifyCommand::SimplifyCommand(cvc5::Term term) : d_term(term) {}
cvc5::Term SimplifyCommand::getTerm() const { return d_term; }
void SimplifyCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    d_result = solver->simplify(d_term);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

cvc5::Term SimplifyCommand::getResult() const { return d_result; }
void SimplifyCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    out << d_result << endl;
  }
}

Command* SimplifyCommand::clone() const
{
  SimplifyCommand* c = new SimplifyCommand(d_term);
  c->d_result = d_result;
  return c;
}

std::string SimplifyCommand::getCommandName() const { return "simplify"; }

void SimplifyCommand::toStream(std::ostream& out,
                               int toDepth,
                               size_t dag,
                               Language language) const
{
  Printer::getPrinter(language)->toStreamCmdSimplify(out, termToNode(d_term));
}

/* -------------------------------------------------------------------------- */
/* class GetValueCommand                                                      */
/* -------------------------------------------------------------------------- */

GetValueCommand::GetValueCommand(cvc5::Term term) : d_terms()
{
  d_terms.push_back(term);
}

GetValueCommand::GetValueCommand(const std::vector<cvc5::Term>& terms)
    : d_terms(terms)
{
  PrettyCheckArgument(
      terms.size() >= 1, terms, "cannot get-value of an empty set of terms");
}

const std::vector<cvc5::Term>& GetValueCommand::getTerms() const
{
  return d_terms;
}
void GetValueCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    std::vector<cvc5::Term> result = solver->getValue(d_terms);
    Assert(result.size() == d_terms.size());
    for (int i = 0, size = d_terms.size(); i < size; i++)
    {
      cvc5::Term request = d_terms[i];
      cvc5::Term value = result[i];
      result[i] = solver->mkTerm(cvc5::SEXPR, {request, value});
    }
    d_result = solver->mkTerm(cvc5::SEXPR, {result});
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.what());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

cvc5::Term GetValueCommand::getResult() const { return d_result; }
void GetValueCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    options::ioutils::Scope scope(out);
    options::ioutils::applyDagThresh(out, 0);
    out << d_result << endl;
  }
}

Command* GetValueCommand::clone() const
{
  GetValueCommand* c = new GetValueCommand(d_terms);
  c->d_result = d_result;
  return c;
}

std::string GetValueCommand::getCommandName() const { return "get-value"; }

void GetValueCommand::toStream(std::ostream& out,
                               int toDepth,
                               size_t dag,
                               Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetValue(
      out, termVectorToNodes(d_terms));
}

/* -------------------------------------------------------------------------- */
/* class GetAssignmentCommand                                                 */
/* -------------------------------------------------------------------------- */

GetAssignmentCommand::GetAssignmentCommand() {}
void GetAssignmentCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    std::map<cvc5::Term, std::string> enames = sm->getExpressionNames();
    std::vector<cvc5::Term> terms;
    std::vector<std::string> names;
    for (const std::pair<const cvc5::Term, std::string>& e : enames)
    {
      terms.push_back(e.first);
      names.push_back(e.second);
    }
    // Must use vector version of getValue to ensure error is thrown regardless
    // of whether terms is empty.
    std::vector<cvc5::Term> values = solver->getValue(terms);
    Assert(values.size() == names.size());
    std::vector<cvc5::Term> sexprs;
    for (size_t i = 0, nterms = terms.size(); i < nterms; i++)
    {
      // Treat the expression name as a variable name as opposed to a string
      // constant to avoid printing double quotes around the name.
      cvc5::Term name = solver->mkVar(solver->getBooleanSort(), names[i]);
      sexprs.push_back(solver->mkTerm(cvc5::SEXPR, {name, values[i]}));
    }
    d_result = solver->mkTerm(cvc5::SEXPR, sexprs);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.what());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

cvc5::Term GetAssignmentCommand::getResult() const { return d_result; }
void GetAssignmentCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    out << d_result << endl;
  }
}

Command* GetAssignmentCommand::clone() const
{
  GetAssignmentCommand* c = new GetAssignmentCommand();
  c->d_result = d_result;
  return c;
}

std::string GetAssignmentCommand::getCommandName() const
{
  return "get-assignment";
}

void GetAssignmentCommand::toStream(std::ostream& out,
                                    int toDepth,
                                    size_t dag,
                                    Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetAssignment(out);
}

/* -------------------------------------------------------------------------- */
/* class GetModelCommand                                                      */
/* -------------------------------------------------------------------------- */

GetModelCommand::GetModelCommand() {}
void GetModelCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    std::vector<cvc5::Sort> declareSorts = sm->getModelDeclareSorts();
    std::vector<cvc5::Term> declareTerms = sm->getModelDeclareTerms();
    d_result = solver->getModel(declareSorts, declareTerms);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.what());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

void GetModelCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    out << d_result;
  }
}

Command* GetModelCommand::clone() const
{
  GetModelCommand* c = new GetModelCommand;
  c->d_result = d_result;
  return c;
}

std::string GetModelCommand::getCommandName() const { return "get-model"; }

void GetModelCommand::toStream(std::ostream& out,
                               int toDepth,
                               size_t dag,
                               Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetModel(out);
}

/* -------------------------------------------------------------------------- */
/* class BlockModelCommand */
/* -------------------------------------------------------------------------- */

BlockModelCommand::BlockModelCommand(modes::BlockModelsMode mode) : d_mode(mode)
{
}
void BlockModelCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    solver->blockModel(d_mode);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.what());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* BlockModelCommand::clone() const
{
  BlockModelCommand* c = new BlockModelCommand(d_mode);
  return c;
}

std::string BlockModelCommand::getCommandName() const { return "block-model"; }

void BlockModelCommand::toStream(std::ostream& out,
                                 int toDepth,
                                 size_t dag,
                                 Language language) const
{
  Printer::getPrinter(language)->toStreamCmdBlockModel(out, d_mode);
}

/* -------------------------------------------------------------------------- */
/* class BlockModelValuesCommand */
/* -------------------------------------------------------------------------- */

BlockModelValuesCommand::BlockModelValuesCommand(
    const std::vector<cvc5::Term>& terms)
    : d_terms(terms)
{
  PrettyCheckArgument(terms.size() >= 1,
                      terms,
                      "cannot block-model-values of an empty set of terms");
}

const std::vector<cvc5::Term>& BlockModelValuesCommand::getTerms() const
{
  return d_terms;
}
void BlockModelValuesCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    solver->blockModelValues(d_terms);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.what());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* BlockModelValuesCommand::clone() const
{
  BlockModelValuesCommand* c = new BlockModelValuesCommand(d_terms);
  return c;
}

std::string BlockModelValuesCommand::getCommandName() const
{
  return "block-model-values";
}

void BlockModelValuesCommand::toStream(std::ostream& out,
                                       int toDepth,
                                       size_t dag,
                                       Language language) const
{
  Printer::getPrinter(language)->toStreamCmdBlockModelValues(
      out, termVectorToNodes(d_terms));
}

/* -------------------------------------------------------------------------- */
/* class GetProofCommand                                                      */
/* -------------------------------------------------------------------------- */

GetProofCommand::GetProofCommand() {}
void GetProofCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    d_result = solver->getProof();
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.what());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

void GetProofCommand::printResult(std::ostream& out) const
{
  if (ok())
  {
    out << d_result;
  }
  else
  {
    this->Command::printResult(out);
  }
}

Command* GetProofCommand::clone() const
{
  GetProofCommand* c = new GetProofCommand();
  return c;
}

std::string GetProofCommand::getCommandName() const { return "get-proof"; }

void GetProofCommand::toStream(std::ostream& out,
                               int toDepth,
                               size_t dag,
                               Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetProof(out);
}

/* -------------------------------------------------------------------------- */
/* class GetInstantiationsCommand                                             */
/* -------------------------------------------------------------------------- */

GetInstantiationsCommand::GetInstantiationsCommand() : d_solver(nullptr) {}
bool GetInstantiationsCommand::isEnabled(cvc5::Solver* solver,
                                         const cvc5::Result& res)
{
  return (res.isSat()
          || (res.isUnknown()
              && res.getUnknownExplanation()
                     == cvc5::UnknownExplanation::INCOMPLETE))
         || res.isUnsat();
}
void GetInstantiationsCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    d_solver = solver;
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

void GetInstantiationsCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    out << d_solver->getInstantiations();
  }
}

Command* GetInstantiationsCommand::clone() const
{
  GetInstantiationsCommand* c = new GetInstantiationsCommand();
  // c->d_result = d_result;
  c->d_solver = d_solver;
  return c;
}

std::string GetInstantiationsCommand::getCommandName() const
{
  return "get-instantiations";
}

void GetInstantiationsCommand::toStream(std::ostream& out,
                                        int toDepth,
                                        size_t dag,
                                        Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetInstantiations(out);
}

/* -------------------------------------------------------------------------- */
/* class GetInterpolCommand                                                   */
/* -------------------------------------------------------------------------- */

GetInterpolantCommand::GetInterpolantCommand(const std::string& name,
                                             Term conj)
    : d_name(name), d_conj(conj), d_sygus_grammar(nullptr)
{
}
GetInterpolantCommand::GetInterpolantCommand(const std::string& name,
                                             Term conj,
                                             Grammar* g)
    : d_name(name), d_conj(conj), d_sygus_grammar(g)
{
}

Term GetInterpolantCommand::getConjecture() const { return d_conj; }

const Grammar* GetInterpolantCommand::getGrammar() const
{
  return d_sygus_grammar;
}

Term GetInterpolantCommand::getResult() const { return d_result; }

void GetInterpolantCommand::invoke(Solver* solver, SymbolManager* sm)
{
  try
  {
    // we must remember the name of the interpolant, in case
    // get-interpolant-next is called later.
    sm->setLastSynthName(d_name);
    if (d_sygus_grammar == nullptr)
    {
      d_result = solver->getInterpolant(d_conj);
    }
    else
    {
      d_result = solver->getInterpolant(d_conj, *d_sygus_grammar);
    }
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

void GetInterpolantCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    options::ioutils::Scope scope(out);
    options::ioutils::applyDagThresh(out, 0);
    if (!d_result.isNull())
    {
      out << "(define-fun " << d_name << " () Bool " << d_result << ")"
          << std::endl;
    }
    else
    {
      out << "fail" << std::endl;
    }
  }
}

Command* GetInterpolantCommand::clone() const
{
  GetInterpolantCommand* c =
      new GetInterpolantCommand(d_name, d_conj, d_sygus_grammar);
  c->d_result = d_result;
  return c;
}

std::string GetInterpolantCommand::getCommandName() const
{
  return "get-interpolant";
}

void GetInterpolantCommand::toStream(std::ostream& out,
                                     int toDepth,
                                     size_t dag,
                                     Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetInterpol(
      out, d_name, termToNode(d_conj), grammarToTypeNode(d_sygus_grammar));
}

/* -------------------------------------------------------------------------- */
/* class GetInterpolNextCommand */
/* -------------------------------------------------------------------------- */

GetInterpolantNextCommand::GetInterpolantNextCommand() {}

Term GetInterpolantNextCommand::getResult() const { return d_result; }

void GetInterpolantNextCommand::invoke(Solver* solver, SymbolManager* sm)
{
  try
  {
    // Get the name of the interpolant from the symbol manager
    d_name = sm->getLastSynthName();
    d_result = solver->getInterpolantNext();
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

void GetInterpolantNextCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    options::ioutils::Scope scope(out);
    options::ioutils::applyDagThresh(out, 0);
    if (!d_result.isNull())
    {
      out << "(define-fun " << d_name << " () Bool " << d_result << ")"
          << std::endl;
    }
    else
    {
      out << "fail" << std::endl;
    }
  }
}

Command* GetInterpolantNextCommand::clone() const
{
  GetInterpolantNextCommand* c = new GetInterpolantNextCommand;
  c->d_result = d_result;
  return c;
}

std::string GetInterpolantNextCommand::getCommandName() const
{
  return "get-interpolant-next";
}

void GetInterpolantNextCommand::toStream(std::ostream& out,
                                         int toDepth,
                                         size_t dag,
                                         Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetInterpolNext(out);
}

/* -------------------------------------------------------------------------- */
/* class GetAbductCommand                                                     */
/* -------------------------------------------------------------------------- */

GetAbductCommand::GetAbductCommand(const std::string& name, cvc5::Term conj)
    : d_name(name), d_conj(conj), d_sygus_grammar(nullptr)
{
}
GetAbductCommand::GetAbductCommand(const std::string& name,
                                   cvc5::Term conj,
                                   cvc5::Grammar* g)
    : d_name(name), d_conj(conj), d_sygus_grammar(g)
{
}

cvc5::Term GetAbductCommand::getConjecture() const { return d_conj; }

const cvc5::Grammar* GetAbductCommand::getGrammar() const
{
  return d_sygus_grammar;
}

std::string GetAbductCommand::getAbductName() const { return d_name; }
cvc5::Term GetAbductCommand::getResult() const { return d_result; }

void GetAbductCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    // we must remember the name of the abduct, in case get-abduct-next is
    // called later.
    sm->setLastSynthName(d_name);
    if (d_sygus_grammar == nullptr)
    {
      d_result = solver->getAbduct(d_conj);
    }
    else
    {
      d_result = solver->getAbduct(d_conj, *d_sygus_grammar);
    }
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

void GetAbductCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    options::ioutils::Scope scope(out);
    options::ioutils::applyDagThresh(out, 0);
    if (!d_result.isNull())
    {
      out << "(define-fun " << d_name << " () Bool " << d_result << ")"
          << std::endl;
    }
    else
    {
      out << "fail" << std::endl;
    }
  }
}

Command* GetAbductCommand::clone() const
{
  GetAbductCommand* c = new GetAbductCommand(d_name, d_conj, d_sygus_grammar);
  c->d_result = d_result;
  return c;
}

std::string GetAbductCommand::getCommandName() const { return "get-abduct"; }

void GetAbductCommand::toStream(std::ostream& out,
                                int toDepth,
                                size_t dag,
                                Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetAbduct(
      out, d_name, termToNode(d_conj), grammarToTypeNode(d_sygus_grammar));
}

/* -------------------------------------------------------------------------- */
/* class GetAbductNextCommand */
/* -------------------------------------------------------------------------- */

GetAbductNextCommand::GetAbductNextCommand() {}

cvc5::Term GetAbductNextCommand::getResult() const { return d_result; }

void GetAbductNextCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    // Get the name of the abduct from the symbol manager
    d_name = sm->getLastSynthName();
    d_result = solver->getAbductNext();
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

void GetAbductNextCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    options::ioutils::Scope scope(out);
    options::ioutils::applyDagThresh(out, 0);
    if (!d_result.isNull())
    {
      out << "(define-fun " << d_name << " () Bool " << d_result << ")"
          << std::endl;
    }
    else
    {
      out << "fail" << std::endl;
    }
  }
}

Command* GetAbductNextCommand::clone() const
{
  GetAbductNextCommand* c = new GetAbductNextCommand;
  c->d_result = d_result;
  return c;
}

std::string GetAbductNextCommand::getCommandName() const
{
  return "get-abduct-next";
}

void GetAbductNextCommand::toStream(std::ostream& out,
                                    int toDepth,
                                    size_t dag,
                                    Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetAbductNext(out);
}

/* -------------------------------------------------------------------------- */
/* class GetQuantifierEliminationCommand                                      */
/* -------------------------------------------------------------------------- */

GetQuantifierEliminationCommand::GetQuantifierEliminationCommand()
    : d_term(), d_doFull(true)
{
}
GetQuantifierEliminationCommand::GetQuantifierEliminationCommand(
    const cvc5::Term& term, bool doFull)
    : d_term(term), d_doFull(doFull)
{
}

cvc5::Term GetQuantifierEliminationCommand::getTerm() const { return d_term; }
bool GetQuantifierEliminationCommand::getDoFull() const { return d_doFull; }
void GetQuantifierEliminationCommand::invoke(cvc5::Solver* solver,
                                             SymbolManager* sm)
{
  try
  {
    if (d_doFull)
    {
      d_result = solver->getQuantifierElimination(d_term);
    }
    else
    {
      d_result = solver->getQuantifierEliminationDisjunct(d_term);
    }
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

cvc5::Term GetQuantifierEliminationCommand::getResult() const
{
  return d_result;
}
void GetQuantifierEliminationCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    out << d_result << endl;
  }
}

Command* GetQuantifierEliminationCommand::clone() const
{
  GetQuantifierEliminationCommand* c =
      new GetQuantifierEliminationCommand(d_term, d_doFull);
  c->d_result = d_result;
  return c;
}

std::string GetQuantifierEliminationCommand::getCommandName() const
{
  return d_doFull ? "get-qe" : "get-qe-disjunct";
}

void GetQuantifierEliminationCommand::toStream(std::ostream& out,
                                               int toDepth,
                                               size_t dag,
                                               Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetQuantifierElimination(
      out, termToNode(d_term), d_doFull);
}

/* -------------------------------------------------------------------------- */
/* class GetUnsatAssumptionsCommand                                           */
/* -------------------------------------------------------------------------- */

GetUnsatAssumptionsCommand::GetUnsatAssumptionsCommand() {}

void GetUnsatAssumptionsCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    d_result = solver->getUnsatAssumptions();
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.what());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

std::vector<cvc5::Term> GetUnsatAssumptionsCommand::getResult() const
{
  return d_result;
}

void GetUnsatAssumptionsCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    container_to_stream(out, d_result, "(", ")\n", " ");
  }
}

Command* GetUnsatAssumptionsCommand::clone() const
{
  GetUnsatAssumptionsCommand* c = new GetUnsatAssumptionsCommand;
  c->d_result = d_result;
  return c;
}

std::string GetUnsatAssumptionsCommand::getCommandName() const
{
  return "get-unsat-assumptions";
}

void GetUnsatAssumptionsCommand::toStream(std::ostream& out,
                                          int toDepth,
                                          size_t dag,
                                          Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetUnsatAssumptions(out);
}

/* -------------------------------------------------------------------------- */
/* class GetUnsatCoreCommand                                                  */
/* -------------------------------------------------------------------------- */

GetUnsatCoreCommand::GetUnsatCoreCommand() : d_sm(nullptr) {}
void GetUnsatCoreCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    d_sm = sm;
    d_result = solver->getUnsatCore();

    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.what());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

void GetUnsatCoreCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    if (options::printUnsatCoresFull())
    {
      // use the assertions
      UnsatCore ucr(termVectorToNodes(d_result));
      ucr.toStream(out);
    }
    else
    {
      // otherwise, use the names
      std::vector<std::string> names;
      d_sm->getExpressionNames(d_result, names, true);
      UnsatCore ucr(names);
      ucr.toStream(out);
    }
  }
}

const std::vector<cvc5::Term>& GetUnsatCoreCommand::getUnsatCore() const
{
  // of course, this will be empty if the command hasn't been invoked yet
  return d_result;
}

Command* GetUnsatCoreCommand::clone() const
{
  GetUnsatCoreCommand* c = new GetUnsatCoreCommand;
  c->d_sm = d_sm;
  c->d_result = d_result;
  return c;
}

std::string GetUnsatCoreCommand::getCommandName() const
{
  return "get-unsat-core";
}

void GetUnsatCoreCommand::toStream(std::ostream& out,
                                   int toDepth,
                                   size_t dag,
                                   Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetUnsatCore(out);
}

/* -------------------------------------------------------------------------- */
/* class GetDifficultyCommand */
/* -------------------------------------------------------------------------- */

GetDifficultyCommand::GetDifficultyCommand() : d_sm(nullptr) {}
void GetDifficultyCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    d_sm = sm;
    d_result = solver->getDifficulty();

    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.what());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

void GetDifficultyCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    out << "(" << std::endl;
    for (const std::pair<const cvc5::Term, cvc5::Term>& d : d_result)
    {
      out << "(";
      // use name if it has one
      std::string name;
      if (d_sm->getExpressionName(d.first, name, true))
      {
        out << name;
      }
      else
      {
        out << d.first;
      }
      out << " " << d.second << ")" << std::endl;
    }
    out << ")" << std::endl;
  }
}

const std::map<cvc5::Term, cvc5::Term>& GetDifficultyCommand::getDifficultyMap()
    const
{
  return d_result;
}

Command* GetDifficultyCommand::clone() const
{
  GetDifficultyCommand* c = new GetDifficultyCommand;
  c->d_sm = d_sm;
  c->d_result = d_result;
  return c;
}

std::string GetDifficultyCommand::getCommandName() const
{
  return "get-difficulty";
}

void GetDifficultyCommand::toStream(std::ostream& out,
                                    int toDepth,
                                    size_t dag,
                                    Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetDifficulty(out);
}

/* -------------------------------------------------------------------------- */
/* class GetLearnedLiteralsCommand */
/* -------------------------------------------------------------------------- */

GetLearnedLiteralsCommand::GetLearnedLiteralsCommand() {}
void GetLearnedLiteralsCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    d_result = solver->getLearnedLiterals();

    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.what());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

void GetLearnedLiteralsCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    out << "(" << std::endl;
    for (const cvc5::Term& lit : d_result)
    {
      out << lit << std::endl;
    }
    out << ")" << std::endl;
  }
}

const std::vector<cvc5::Term>& GetLearnedLiteralsCommand::getLearnedLiterals()
    const
{
  return d_result;
}

Command* GetLearnedLiteralsCommand::clone() const
{
  GetLearnedLiteralsCommand* c = new GetLearnedLiteralsCommand;
  c->d_result = d_result;
  return c;
}

std::string GetLearnedLiteralsCommand::getCommandName() const
{
  return "get-learned-literals";
}

void GetLearnedLiteralsCommand::toStream(std::ostream& out,
                                         int toDepth,
                                         size_t dag,
                                         Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetLearnedLiterals(out);
}

/* -------------------------------------------------------------------------- */
/* class GetAssertionsCommand                                                 */
/* -------------------------------------------------------------------------- */

GetAssertionsCommand::GetAssertionsCommand() {}
void GetAssertionsCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    stringstream ss;
    const vector<cvc5::Term> v = solver->getAssertions();
    ss << "(\n";
    copy(v.begin(), v.end(), ostream_iterator<cvc5::Term>(ss, "\n"));
    ss << ")\n";
    d_result = ss.str();
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

std::string GetAssertionsCommand::getResult() const { return d_result; }
void GetAssertionsCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else
  {
    out << d_result;
  }
}

Command* GetAssertionsCommand::clone() const
{
  GetAssertionsCommand* c = new GetAssertionsCommand();
  c->d_result = d_result;
  return c;
}

std::string GetAssertionsCommand::getCommandName() const
{
  return "get-assertions";
}

void GetAssertionsCommand::toStream(std::ostream& out,
                                    int toDepth,
                                    size_t dag,
                                    Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetAssertions(out);
}

/* -------------------------------------------------------------------------- */
/* class SetBenchmarkLogicCommand                                             */
/* -------------------------------------------------------------------------- */

SetBenchmarkLogicCommand::SetBenchmarkLogicCommand(std::string logic)
    : d_logic(logic)
{
}

std::string SetBenchmarkLogicCommand::getLogic() const { return d_logic; }
void SetBenchmarkLogicCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    solver->setLogic(d_logic);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* SetBenchmarkLogicCommand::clone() const
{
  return new SetBenchmarkLogicCommand(d_logic);
}

std::string SetBenchmarkLogicCommand::getCommandName() const
{
  return "set-logic";
}

void SetBenchmarkLogicCommand::toStream(std::ostream& out,
                                        int toDepth,
                                        size_t dag,
                                        Language language) const
{
  Printer::getPrinter(language)->toStreamCmdSetBenchmarkLogic(out, d_logic);
}

/* -------------------------------------------------------------------------- */
/* class SetInfoCommand                                                       */
/* -------------------------------------------------------------------------- */

SetInfoCommand::SetInfoCommand(const std::string& flag,
                               const std::string& value)
    : d_flag(flag), d_value(value)
{
}

const std::string& SetInfoCommand::getFlag() const { return d_flag; }
const std::string& SetInfoCommand::getValue() const { return d_value; }
void SetInfoCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    solver->setInfo(d_flag, d_value);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiUnsupportedException&)
  {
    // As per SMT-LIB spec, silently accept unknown set-info keys
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.getMessage());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* SetInfoCommand::clone() const
{
  return new SetInfoCommand(d_flag, d_value);
}

std::string SetInfoCommand::getCommandName() const { return "set-info"; }

void SetInfoCommand::toStream(std::ostream& out,
                              int toDepth,
                              size_t dag,
                              Language language) const
{
  Printer::getPrinter(language)->toStreamCmdSetInfo(out, d_flag, d_value);
}

/* -------------------------------------------------------------------------- */
/* class GetInfoCommand                                                       */
/* -------------------------------------------------------------------------- */

GetInfoCommand::GetInfoCommand(std::string flag) : d_flag(flag) {}
std::string GetInfoCommand::getFlag() const { return d_flag; }
void GetInfoCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    std::vector<cvc5::Term> v;
    v.push_back(solver->mkString(":" + d_flag));
    v.push_back(solver->mkString(solver->getInfo(d_flag)));
    d_result = sexprToString(solver->mkTerm(cvc5::SEXPR, {v}));
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiUnsupportedException&)
  {
    d_commandStatus = new CommandUnsupported();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.getMessage());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

std::string GetInfoCommand::getResult() const { return d_result; }
void GetInfoCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else if (d_result != "")
  {
    out << d_result << endl;
  }
}

Command* GetInfoCommand::clone() const
{
  GetInfoCommand* c = new GetInfoCommand(d_flag);
  c->d_result = d_result;
  return c;
}

std::string GetInfoCommand::getCommandName() const { return "get-info"; }

void GetInfoCommand::toStream(std::ostream& out,
                              int toDepth,
                              size_t dag,
                              Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetInfo(out, d_flag);
}

/* -------------------------------------------------------------------------- */
/* class SetOptionCommand                                                     */
/* -------------------------------------------------------------------------- */

SetOptionCommand::SetOptionCommand(const std::string& flag,
                                   const std::string& value)
    : d_flag(flag), d_value(value)
{
}

const std::string& SetOptionCommand::getFlag() const { return d_flag; }
const std::string& SetOptionCommand::getValue() const { return d_value; }
void SetOptionCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    solver->setOption(d_flag, d_value);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiUnsupportedException&)
  {
    d_commandStatus = new CommandUnsupported();
  }
  catch (cvc5::CVC5ApiRecoverableException& e)
  {
    d_commandStatus = new CommandRecoverableFailure(e.getMessage());
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

Command* SetOptionCommand::clone() const
{
  return new SetOptionCommand(d_flag, d_value);
}

std::string SetOptionCommand::getCommandName() const { return "set-option"; }

void SetOptionCommand::toStream(std::ostream& out,
                                int toDepth,
                                size_t dag,
                                Language language) const
{
  Printer::getPrinter(language)->toStreamCmdSetOption(out, d_flag, d_value);
}

/* -------------------------------------------------------------------------- */
/* class GetOptionCommand                                                     */
/* -------------------------------------------------------------------------- */

GetOptionCommand::GetOptionCommand(std::string flag) : d_flag(flag) {}
std::string GetOptionCommand::getFlag() const { return d_flag; }
void GetOptionCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  try
  {
    d_result = solver->getOption(d_flag);
    d_commandStatus = CommandSuccess::instance();
  }
  catch (cvc5::CVC5ApiUnsupportedException&)
  {
    d_commandStatus = new CommandUnsupported();
  }
  catch (exception& e)
  {
    d_commandStatus = new CommandFailure(e.what());
  }
}

std::string GetOptionCommand::getResult() const { return d_result; }
void GetOptionCommand::printResult(std::ostream& out) const
{
  if (!ok())
  {
    this->Command::printResult(out);
  }
  else if (d_result != "")
  {
    out << d_result << endl;
  }
}

Command* GetOptionCommand::clone() const
{
  GetOptionCommand* c = new GetOptionCommand(d_flag);
  c->d_result = d_result;
  return c;
}

std::string GetOptionCommand::getCommandName() const { return "get-option"; }

void GetOptionCommand::toStream(std::ostream& out,
                                int toDepth,
                                size_t dag,
                                Language language) const
{
  Printer::getPrinter(language)->toStreamCmdGetOption(out, d_flag);
}

/* -------------------------------------------------------------------------- */
/* class DatatypeDeclarationCommand                                           */
/* -------------------------------------------------------------------------- */

DatatypeDeclarationCommand::DatatypeDeclarationCommand(
    const cvc5::Sort& datatype)
    : d_datatypes()
{
  d_datatypes.push_back(datatype);
}

DatatypeDeclarationCommand::DatatypeDeclarationCommand(
    const std::vector<cvc5::Sort>& datatypes)
    : d_datatypes(datatypes)
{
}

const std::vector<cvc5::Sort>& DatatypeDeclarationCommand::getDatatypes() const
{
  return d_datatypes;
}

void DatatypeDeclarationCommand::invoke(cvc5::Solver* solver, SymbolManager* sm)
{
  d_commandStatus = CommandSuccess::instance();
}

Command* DatatypeDeclarationCommand::clone() const
{
  return new DatatypeDeclarationCommand(d_datatypes);
}

std::string DatatypeDeclarationCommand::getCommandName() const
{
  return "declare-datatypes";
}

void DatatypeDeclarationCommand::toStream(std::ostream& out,
                                          int toDepth,
                                          size_t dag,
                                          Language language) const
{
  Printer::getPrinter(language)->toStreamCmdDatatypeDeclaration(
      out, sortVectorToTypeNodes(d_datatypes));
}

}  // namespace cvc5
