package test

import (
	"context"
	"fmt"
	"log/slog"
	"os"
	"path/filepath"
	"testing"
	"time"

	"git.sr.ht/~whynothugo/ImapGoose/internal/daemon"
	"git.sr.ht/~whynothugo/ImapGoose/internal/imap"
	imap2 "github.com/emersion/go-imap/v2"
)

// TestBasicSync tests that a message created on the IMAP server appears in the local Maildir within 100ms.
func TestBasicSync(t *testing.T) {
	dovecot := setupDovecot(t)

	stateDir := t.TempDir()
	localMailDir := t.TempDir()

	cfg := dovecot.MakeConfig("test", localMailDir, stateDir)

	ctx, stopImapGoose := context.WithCancel(context.Background())
	defer stopImapGoose()

	logger := slog.New(slog.NewTextHandler(os.Stdout, &slog.HandlerOptions{
		Level: slog.LevelDebug,
	}))

	d := daemon.New(cfg, logger)

	// Start ImapGoose daemon in background.
	go func() { _ = d.Run(ctx) }()

	// Give daemon plenty of time to start and settle.
	time.Sleep(50 * time.Millisecond)

	// Connect to IMAP server to create messages "externally".
	imapClient, err := imap.Connect(
		context.Background(),
		fmt.Sprintf("127.0.0.1:%d", dovecot.Port),
		"",
		"",
		true, // plaintext
		nil,
		logger,
	)
	if err != nil {
		// FIXME: externalise to function?
		t.Logf("Reading Dovecot logs from %s", dovecot.BaseDir)
		if logData, readErr := os.ReadFile(filepath.Join(dovecot.BaseDir, "dovecot.log")); readErr == nil {
			t.Logf("Dovecot log:\n%s", string(logData))
		}
		if logData, readErr := os.ReadFile(filepath.Join(dovecot.BaseDir, "dovecot-info.log")); readErr == nil {
			t.Logf("Dovecot info log:\n%s", string(logData))
		}
		t.Fatalf("Failed to connect to IMAP server: %v", err)
	}
	defer func() { _ = imapClient.Close() }()

	// Select INBOX.
	_, err = imapClient.SelectMailbox(context.Background(), "INBOX", nil)
	if err != nil {
		// Print Dovecot logs for debugging.
		t.Logf("Reading Dovecot logs from %s", dovecot.BaseDir)
		if logData, readErr := os.ReadFile(filepath.Join(dovecot.BaseDir, "dovecot.log")); readErr == nil {
			t.Logf("Dovecot log:\n%s", string(logData))
		}
		if logData, readErr := os.ReadFile(filepath.Join(dovecot.BaseDir, "dovecot-info.log")); readErr == nil {
			t.Logf("Dovecot info log:\n%s", string(logData))
		}
		t.Fatalf("Failed to select INBOX: %v", err)
	}

	// Create a test message.
	testMessage := []byte("From: test@example.com\r\nTo: user@example.com\r\nSubject: Test Message\r\n\r\nThis is a test message.\r\n")
	uid, err := imapClient.AppendMessage(context.Background(), "INBOX", []imap2.Flag{}, testMessage)
	if err != nil {
		t.Fatalf("Failed to append message: %v", err)
	}

	t.Logf("Created message with UID %d", uid)

	// Wait up to 2 seconds for the message to appear in local Maildir.
	// NOTIFY + worker dispatch + download can take a few milliseconds.
	inboxPath := filepath.Join(localMailDir, "INBOX")
	deadline := time.Now().Add(500 * time.Millisecond)

	var found bool
	for time.Now().Before(deadline) {
		// TODO: should we use a watcher?
		for _, subdir := range []string{"cur", "new"} {
			dirPath := filepath.Join(inboxPath, subdir)
			entries, err := os.ReadDir(dirPath)
			if err != nil {
				if os.IsNotExist(err) {
					continue
				}
				t.Fatalf("Failed to read directory %s: %v", dirPath, err)
			}

			if len(entries) > 0 {
				found = true
				t.Logf("Found %d messages in %s", len(entries), dirPath)
				break
			}
		}

		if found {
			break
		}

		time.Sleep(10 * time.Millisecond)
	}

	if !found {
		t.Fatal("Message did not appear in local Maildir within timeout")
	}

	// Compare message body. Probably not needed.
	for _, subdir := range []string{"cur", "new"} {
		dirPath := filepath.Join(inboxPath, subdir)
		entries, err := os.ReadDir(dirPath)
		if err != nil {
			continue
		}

		for _, entry := range entries {
			content, err := os.ReadFile(filepath.Join(dirPath, entry.Name()))
			if err != nil {
				t.Fatalf("Failed to read message file: %v", err)
			}

			if string(content) != string(testMessage) {
				t.Errorf("Message content mismatch:\nExpected: %s\nGot: %s", testMessage, content)
			}

			t.Logf("Successfully verified message content")
		}
	}
}
