package daemon

import (
	"context"
	"fmt"
	"log/slog"
	"os"
	"os/signal"
	"sync"
	"syscall"

	"git.sr.ht/~whynothugo/ImapGoose/internal/config"
	"git.sr.ht/~whynothugo/ImapGoose/internal/notify"
	"git.sr.ht/~whynothugo/ImapGoose/internal/status"
	"git.sr.ht/~whynothugo/ImapGoose/internal/watcher"
)

// Daemon manages the ImapGoose synchronisation daemon.
type Daemon struct {
	config *config.Config
	logger *slog.Logger
}

// New creates a new daemon instance.
func New(cfg *config.Config, logger *slog.Logger) *Daemon {
	return &Daemon{
		config: cfg,
		logger: logger,
	}
}

// Run starts the daemon and blocks until shutdown.
func (d *Daemon) Run(ctx context.Context) error {
	d.logger.Info("Starting ImapGoose daemon", "accounts", len(d.config.Accounts))

	// TODO: maybe this should happen in main(), since it leaks the goroutine.
	// Create a context that cancels on signal.
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()

	// Handle shutdown signals.
	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, os.Interrupt, syscall.SIGTERM)

	go func() {
		sig := <-sigChan
		d.logger.Info("Received shutdown signal", "signal", sig)
		cancel()
	}()

	// Start watchers for each account.
	var wg sync.WaitGroup

	for _, account := range d.config.Accounts {
		wg.Add(1)
		go func(acc config.Account) {
			defer wg.Done()
			if err := d.runAccount(ctx, acc); err != nil && ctx.Err() == nil {
				// Only log errors if not due to cancellation.
				d.logger.Error("Account error", "account", acc.Name, "error", err)
			}
		}(account)
	}

	// Wait for all accounts to finish.
	wg.Wait()

	d.logger.Info("Daemon stopped")
	return nil
}

// runAccount runs synchronisation for a single account using worker pool.
func (d *Daemon) runAccount(ctx context.Context, account config.Account) error {
	accountLogger := d.logger.With("account", account.Name)
	accountLogger.Info("Starting account sync")

	var statusRepo *status.Repository
	if account.StateDir != "" {
		statusRepo = status.NewWithStateDir(account.Name, account.StateDir)
	} else {
		statusRepo = status.New(account.Name)
	}
	if err := statusRepo.Init(); err != nil {
		return fmt.Errorf("failed to initialize status repository: %w", err)
	}
	defer func() {
		_ = statusRepo.Close() // Best effort close
	}()

	version, err := statusRepo.GetVersion()
	if err != nil {
		return fmt.Errorf("failed to get status repository version: %w", err)
	}
	if version != 2 {
		return fmt.Errorf("status repository must be version 2, use `imapgoose -m` to migrate")
	}

	// Resolve password from password-cmd.
	if err := account.ResolvePassword(); err != nil {
		return fmt.Errorf("failed to resolve password: %w", err)
	}

	localQueue := make(chan watcher.LocalEvent, 100)
	remoteQueue := make(chan notify.RemoteEvent, 100)
	listener := notify.NewListener(account, accountLogger)

	var wg sync.WaitGroup

	// Dispatcher coordinates tasks with mailbox-level serialization.
	dispatcher := NewDispatcher(localQueue, remoteQueue, accountLogger)
	wg.Add(1)
	go func() {
		defer wg.Done()
		if err := dispatcher.Run(ctx); err != nil && ctx.Err() == nil {
			accountLogger.Error("Dispatcher error", "error", err)
		}
	}()

	// Create ready channels for synchronizing listener and watcher startup.
	// These channels receive nil on success, error on failure.
	listenerReady := make(chan error)
	watcherReady := make(chan error)

	// Start NOTIFY listener for IMAP changes.
	wg.Add(1)
	go func() {
		defer wg.Done()
		listener.Run(ctx, remoteQueue, listenerReady)
	}()

	// Start filesystem watcher for local Maildir changes.
	fsWatcher := watcher.New(account.LocalPath, localQueue, accountLogger)
	wg.Add(1)
	go func() {
		defer wg.Done()
		fsWatcher.Run(ctx, watcherReady)
	}()

	// Wait for both listener and watcher to be ready before proceeding.
	// This ensures they've both completed setup and queued initial tasks.
	accountLogger.Info("Waiting for listener and watcher to be ready")

	// At this point, both the listener and the watcher have queued all mailboxes
	// which they have found. The dispatcher will de-duplicate, so mailboxes present
	// in both sides will only have one task.
	accountLogger.Info("Listener and watcher ready, starting workers.")

	numWorkers := account.MaxConnections - 1
	workerQueue := dispatcher.WorkerQueue()
	for i := 0; i < numWorkers; i++ {
		wg.Add(1)
		worker := NewWorker(i, account, account.LocalPath, statusRepo, accountLogger)
		go func() {
			defer wg.Done()
			if err := worker.Run(ctx, workerQueue); err != nil && ctx.Err() == nil {
				accountLogger.Error("Worker error", "error", err)
			}
		}()
	}

	// If these fail initial setup (even after workers started), bail.
	if err := <-listenerReady; err != nil {
		return fmt.Errorf("listener failed to start: %w", err)
	}
	accountLogger.Debug("Listener ready")
	if err := <-watcherReady; err != nil {
		return fmt.Errorf("watcher failed to start: %w", err)
	}
	accountLogger.Debug("Watcher ready")

	// TODO: if we notified service readiness, we'd do it here.

	// Wait for shutdown signal.
	<-ctx.Done()

	// Close input queues to signal dispatcher and workers to finish.
	close(localQueue)
	close(remoteQueue)

	// Wait for all workers, dispatcher, and listener to exit.
	wg.Wait()

	accountLogger.Info("Account sync stopped")
	return nil
}
