package maildir

import (
	"crypto/rand"
	"encoding/hex"
	"fmt"
	"log/slog"
	"os"
	"path/filepath"
	"slices"
	"strings"
	"time"

	"github.com/emersion/go-imap/v2"
)

// Maildir represents a Maildir directory.
type Maildir struct {
	path    string
	mailbox string
	logger  *slog.Logger
}

// Message represents a message in a Maildir.
type Message struct {
	Filename string // Absolute path to the message file
	Flags    []imap.Flag
}

// New creates a new Maildir instance.
func New(basePath, mailbox string, logger *slog.Logger) *Maildir {
	return &Maildir{
		path:    basePath,
		mailbox: mailbox,
		logger:  logger,
	}
}

// Init initializes the Maildir structure.
func (m *Maildir) Init() error {
	maildirPath := m.getMaildirPath()

	// Create the standard Maildir subdirectories.
	for _, subdir := range []string{"tmp", "new", "cur"} {
		dir := filepath.Join(maildirPath, subdir)
		if err := os.MkdirAll(dir, 0700); err != nil {
			return fmt.Errorf("failed to create %s: %w", dir, err)
		}
	}

	return nil
}

// getMaildirPath returns the full path to the Maildir directory.
func (m *Maildir) getMaildirPath() string {
	return filepath.Join(m.path, m.mailbox)
}

// GetLegacyMaildirPath returns the v1 path for a mailbox.
// Each mailbox is a separate flat directory under the root path.
func GetLegacyMaildirPath(basePath, mailbox string) string {
	// Flat structure: {root}/{mailbox}
	// INBOX → {root}/INBOX
	// Sent → {root}/Sent
	return filepath.Join(basePath, strings.ReplaceAll(mailbox, "/", "."))
}

// ValidateMailboxName checks if a mailbox name is valid for nested Maildir.
// Returns error if any component is "cur", "new", or "tmp" (reserved).
func ValidateMailboxName(mailbox string) error {
	parts := strings.Split(mailbox, "/")
	for _, part := range parts {
		switch part {
		case "cur", "new", "tmp":
			return fmt.Errorf("mailbox name contains prohibited component %q: %s", part, mailbox)
		}
	}
	return nil
}

// List returns all messages in the Maildir, keyed by filename (basename only).
func (m *Maildir) List() (map[string]Message, error) {
	messages := make(map[string]Message)

	curPath := filepath.Join(m.getMaildirPath(), "cur")
	if err := m.readDir(curPath, messages); err != nil {
		return nil, err
	}

	newPath := filepath.Join(m.getMaildirPath(), "new")
	if err := m.readDir(newPath, messages); err != nil {
		return nil, err
	}

	return messages, nil
}

// ResolveAbsPath resolves the absolute path of a message by its basename.
// Checks both cur/ and new/ directories.
//
// Matches on the unique portion of the Maildir filename (before ":2,"), so if
// a message's flags have changed, the absolute path to its new location is
// returned.
//
// Returns empty string if not found, error if stat fails for other reasons.
func (m *Maildir) ResolveAbsPath(basename string) (string, error) {
	maildirPath := m.getMaildirPath()

	curPath := filepath.Join(maildirPath, "cur", basename)
	if _, err := os.Stat(curPath); err == nil {
		return curPath, nil
	} else if !os.IsNotExist(err) {
		return "", fmt.Errorf("failed to stat file in cur: %w", err)
	}

	newPath := filepath.Join(maildirPath, "new", basename)
	if _, err := os.Stat(newPath); err == nil {
		return newPath, nil
	} else if !os.IsNotExist(err) {
		return "", fmt.Errorf("failed to stat file in new: %w", err)
	}

	// Slow fallback: match on unique portion.
	m.logger.Debug("Falling back to slow message matching", "basename", basename)

	uniquePart := basename
	if idx := strings.Index(basename, ":2,"); idx != -1 {
		uniquePart = basename[:idx]
	}

	curDir := filepath.Join(maildirPath, "cur")
	if entries, err := os.ReadDir(curDir); err == nil {
		for _, entry := range entries {
			if entry.IsDir() {
				continue
			}
			filename := entry.Name()
			// Check if this file has the same unique portion.
			fileUniquePart := filename
			if idx := strings.Index(filename, ":2,"); idx != -1 {
				fileUniquePart = filename[:idx]
			}
			if fileUniquePart == uniquePart {
				return filepath.Join(curDir, filename), nil
			}
		}
	}

	newDir := filepath.Join(maildirPath, "new")
	if entries, err := os.ReadDir(newDir); err == nil {
		for _, entry := range entries {
			if entry.IsDir() {
				continue
			}
			filename := entry.Name()
			// Check if this file has the same unique portion.
			fileUniquePart := filename
			if idx := strings.Index(filename, ":2,"); idx != -1 {
				fileUniquePart = filename[:idx]
			}
			if fileUniquePart == uniquePart {
				return filepath.Join(newDir, filename), nil
			}
		}
	}

	return "", os.ErrNotExist
}

// GetFlagsByAbsPath retrieves the flags for a single message by absolute path.
func (m *Maildir) GetFlagsByAbsPath(absPath string) (flags []imap.Flag, exists bool, err error) {
	if _, err := os.Stat(absPath); err != nil {
		if os.IsNotExist(err) {
			return nil, false, nil
		}
		return nil, false, fmt.Errorf("failed to stat file: %w", err)
	}

	_, flags = m.parseFilename(filepath.Base(absPath))
	return flags, true, nil
}

// readDir reads messages from a directory.
func (m *Maildir) readDir(dir string, messages map[string]Message) error {
	entries, err := os.ReadDir(dir)
	if err != nil {
		if os.IsNotExist(err) {
			return nil
		}
		return fmt.Errorf("failed to read directory %s: %w", dir, err)
	}

	for _, entry := range entries {
		if entry.IsDir() {
			continue
		}

		filename := entry.Name()
		_, flags := m.parseFilename(filename)

		messages[filename] = Message{
			Filename: filepath.Join(dir, filename),
			Flags:    flags,
		}
	}

	return nil
}

// ExtractTentativeUID extracts the U= parameter (UID hint) from a filename.
// Returns nil if the filename does not contain a U= parameter.
func (m *Maildir) ExtractTentativeUID(filename string) *imap.UID {
	uid, _ := m.parseFilename(filename)
	return uid
}

// parseFilename extracts UID and flags from a Maildir filename.
// Format: unique_id:2,flags or unique_id,U=UID:2,flags
// Returns nil for UID if not present in filename.
func (m *Maildir) parseFilename(filename string) (*imap.UID, []imap.Flag) {
	var uid *imap.UID
	var flags []imap.Flag

	// Extract UID from filename if present (format: ,U=123).
	if idx := strings.Index(filename, ",U="); idx != -1 {
		uidPart := filename[idx+3:]
		if colonIdx := strings.Index(uidPart, ":"); colonIdx != -1 {
			uidPart = uidPart[:colonIdx]
		}
		if commaIdx := strings.Index(uidPart, ","); commaIdx != -1 {
			uidPart = uidPart[:commaIdx]
		}
		var parsedUID uint32
		if n, _ := fmt.Sscanf(uidPart, "%d", &parsedUID); n == 1 {
			u := imap.UID(parsedUID)
			uid = &u
		}
	}

	// Extract flags (format: :2,flags).
	if idx := strings.Index(filename, ":2,"); idx != -1 {
		flagChars := filename[idx+3:]
		flags = m.parseFlags(flagChars)
	}

	return uid, flags
}

// parseFlags converts Maildir flag characters to IMAP flags.
func (m *Maildir) parseFlags(flagChars string) []imap.Flag {
	var flags []imap.Flag

	for _, ch := range flagChars {
		switch ch {
		case 'S':
			flags = append(flags, imap.FlagSeen)
		case 'R':
			flags = append(flags, imap.FlagAnswered)
		case 'F':
			flags = append(flags, imap.FlagFlagged)
		case 'T':
			flags = append(flags, imap.FlagDeleted)
		case 'D':
			flags = append(flags, imap.FlagDraft)
		}
	}

	return flags
}

// Add adds a message to the Maildir and returns the filename (basename only).
// Messages with the Seen flag are placed in cur/, unseen messages in new/.
func (m *Maildir) Add(uid imap.UID, flags []imap.Flag, content []byte) (string, error) {
	// Generate unique filename.
	filename := m.generateFilename(uid, flags)

	// Write to tmp first.
	tmpPath := filepath.Join(m.getMaildirPath(), "tmp", filename)
	if err := os.WriteFile(tmpPath, content, 0600); err != nil {
		return "", fmt.Errorf("failed to write message to tmp: %w", err)
	}

	var targetDir string
	if slices.Contains(flags, imap.FlagSeen) {
		targetDir = "cur"
	} else {
		targetDir = "new"
	}

	targetPath := filepath.Join(m.getMaildirPath(), targetDir, filename)
	if err := os.Link(tmpPath, targetPath); err != nil {
		_ = os.Remove(tmpPath) // Best effort cleanup
		return "", fmt.Errorf("failed to link message to %s: %w", targetDir, err)
	}

	if err := os.Remove(tmpPath); err != nil {
		// File is already in target directory, so this is non-fatal.
		m.logger.Warn("Failed to remove tmp file after link", "path", tmpPath, "error", err)
	}

	return filename, nil
}

// Delete removes a message from the Maildir given its absolute path.
func (m *Maildir) Delete(absPath string) error {
	return os.Remove(absPath)
}

// UpdateFlags updates the flags for a message by adding and removing specific flags.
// Returns the final flags and new filename (basename only) after applying changes.
func (m *Maildir) UpdateFlags(absPath string, toAdd, toRemove []imap.Flag) (newFilename string, newFlags []imap.Flag, err error) {
	oldFilename := filepath.Base(absPath)

	currentFlags := m.parseFlags(oldFilename[strings.Index(oldFilename, ":2,")+3:])
	if idx := strings.Index(oldFilename, ":2,"); idx == -1 {
		// No flags section - treat as empty flags.
		currentFlags = []imap.Flag{}
	}

	// This is O(N²), but typically messages have 0-2 flags.
	newFlags = slices.DeleteFunc(currentFlags, func(flag imap.Flag) bool {
		return slices.Contains(toRemove, flag)
	})
	for _, addFlag := range toAdd {
		found := slices.Contains(newFlags, addFlag)
		if !found {
			newFlags = append(newFlags, addFlag)
		}
	}

	if idx := strings.Index(oldFilename, ":2,"); idx != -1 {
		// Preserve the unique part, replace only the flags.
		uniquePart := oldFilename[:idx]
		flagStr := m.formatFlags(newFlags)
		newFilename = fmt.Sprintf("%s:2,%s", uniquePart, flagStr)
	} else {
		// No flags section found, add one.
		flagStr := m.formatFlags(newFlags)
		newFilename = fmt.Sprintf("%s:2,%s", oldFilename, flagStr)
	}

	newPath := filepath.Join(filepath.Dir(absPath), newFilename)

	// Rename the file.
	if err := os.Rename(absPath, newPath); err != nil {
		return "", nil, fmt.Errorf("failed to update flags: %w", err)
	}

	return newFilename, newFlags, nil
}

// generateFilename creates a Maildir filename with UID and flags.
// Example output: 1234567890.a1b2c3d4,U=42:2,SF
//
// Format breakdown:
// - 1234567890 - Unix timestamp (before first dot)
// - .a1b2c3d4 - 8 hex chars from 4 random bytes (after first dot)
// - ,U=42 - UID metadata (extension, allowed by spec)
// - :2,SF - Standard Maildir info section with flags
//
// See: https://cr.yp.to/proto/maildir.html
func (m *Maildir) generateFilename(uid imap.UID, flags []imap.Flag) string {
	// Generate unique part using timestamp and random bytes.
	timestamp := time.Now().Unix()
	randBytes := make([]byte, 4)
	_, _ = rand.Read(randBytes) // Best effort random generation
	unique := fmt.Sprintf("%d.%s", timestamp, hex.EncodeToString(randBytes))

	// Add UID.
	unique = fmt.Sprintf("%s,U=%d", unique, uid)

	// Add flags.
	flagStr := m.formatFlags(flags)

	return fmt.Sprintf("%s:2,%s", unique, flagStr)
}

// formatFlags converts IMAP flags to Maildir flag characters.
func (m *Maildir) formatFlags(flags []imap.Flag) string {
	var chars []rune

	for _, flag := range flags {
		switch flag {
		case imap.FlagSeen:
			chars = append(chars, 'S')
		case imap.FlagAnswered:
			chars = append(chars, 'R')
		case imap.FlagFlagged:
			chars = append(chars, 'F')
		case imap.FlagDeleted:
			chars = append(chars, 'T')
		case imap.FlagDraft:
			chars = append(chars, 'D')
		}
	}

	// Maildir flags must be in alphabetical order.
	slices.Sort(chars)

	return string(chars)
}

// ReadFile reads the content of a message file.
func (m *Maildir) ReadFile(filename string) ([]byte, error) {
	return os.ReadFile(filename)
}
