package notify

import (
	"context"
	"fmt"
	"log/slog"
	"time"

	"git.sr.ht/~whynothugo/ImapGoose/internal/config"
	"git.sr.ht/~whynothugo/ImapGoose/internal/imap"
	imap2 "github.com/emersion/go-imap/v2"
	"github.com/emersion/go-imap/v2/imapclient"
)

// RemoteEvent represents a remote IMAP change notification on a specific mailbox.
type RemoteEvent struct {
	Mailbox string
}

// Listener monitors NOTIFY events for all mailboxes on an account.
type Listener struct {
	account config.Account
	logger  *slog.Logger
	queue   chan<- RemoteEvent
}

// NewListener creates a new NOTIFY listener.
func NewListener(account config.Account, logger *slog.Logger) *Listener {
	return &Listener{
		account: account,
		logger:  logger,
	}
}

// Run starts the NOTIFY listener to monitor all mailboxes.
//
// Connects to IMAP server, establishes NOTIFY monitoring, and handles reconnection.
// Sends nil to ready channel once NOTIFY monitoring has started successfully.
// Sends error to ready channel if initial setup fails.
// Returns only upon context cancellation.
//
// Sends remote events for all mailboxes after establishing NOTIFY (both initially
// and after reconnection).
func (l *Listener) Run(ctx context.Context, queue chan<- RemoteEvent, ready chan<- error) {
	l.logger.Info("Starting NOTIFY listener")
	l.queue = queue

	// Watch PERSONAL (all mailboxes in user's namespace) for new messages and expunges.
	// Not SELECTED; the NOTIFY listener never selects a mailbox.
	options := &imap2.NotifyOptions{
		Items: []imap2.NotifyItem{
			{
				MailboxSpec: imap2.NotifyMailboxSpecPersonal,
				Events: []imap2.NotifyEvent{
					imap2.NotifyEventMessageNew,
					imap2.NotifyEventMessageExpunge,
				},
			},
		},
	}

	client, notifyCmd, err := l.ensureConnectedAndNotify(ctx, nil, options)
	if err != nil {
		ready <- err
		return
	}
	l.logger.Info("NOTIFY listener ready")

	if err := l.enqueueMailboxes(ctx, client); err != nil {
		ready <- fmt.Errorf("failed to enumerate mailboxes: %w", err)
		return
	}

	cancelKeepAlive := l.startKeepAlive(ctx, client)
	defer cancelKeepAlive() // Ensure keep-alive stops on exit

	ready <- nil
	close(ready)

	// Retry loop with exponential backoff.
	backoff := time.Second
	maxBackoff := 5 * time.Minute

	for {
		if client != nil {
			err = client.WaitForNotifyEnd(ctx, notifyCmd)
			if ctx.Err() != nil {
				l.logger.Info("NOTIFY listener stopped")
				err = client.Close()
				if err != nil {
					l.logger.Error("error closing notify connection", "error", err)
				}
				return
			}
			l.logger.Warn("NOTIFY ended, reconnecting", "error", err)
		}

		select {
		case <-time.After(backoff):
			// Exponential backoff.
			backoff *= 2
			if backoff > maxBackoff {
				backoff = maxBackoff
			}
		case <-ctx.Done():
			l.logger.Info("NOTIFY listener stopped")
			return
		}

		// Stop keep-alive before (possibly) reconnecting.
		cancelKeepAlive()

		client, notifyCmd, err = l.ensureConnectedAndNotify(ctx, client, options)
		if err != nil {
			l.logger.Error("Failed to reconnect and re-initialize NOTIFY", "error", err, "backoff", backoff)
			client = nil // Ensure we retry connection next iteration.
			continue
		}

		// Catch up on changes that happened while not listening.
		if err := l.enqueueMailboxes(ctx, client); err != nil {
			l.logger.Error("Failed to enumerate mailboxes after reconnection", "error", err)
			continue
		}

		cancelKeepAlive = l.startKeepAlive(ctx, client) // Restart for (new?) connection.

		// Reset backoff on successful reconnection.
		backoff = time.Second
		l.logger.Info("NOTIFY re-established")
	}
}

// ensureConnectedAndNotify ensures client is connected and NOTIFY is initialized.
func (l *Listener) ensureConnectedAndNotify(ctx context.Context, client *imap.Client, options *imap2.NotifyOptions) (*imap.Client, *imapclient.NotifyCommand, error) {
	if client != nil {
		if err := client.Noop(ctx); err != nil {
			l.logger.Warn("Connection is dead", "error", err)
			_ = client.Close() // Best effort close
			client = nil
		}
	}

	if client == nil {
		var err error
		client, err = l.connect(ctx)
		if err != nil {
			return nil, nil, fmt.Errorf("failed to connect: %w", err)
		}
	}

	notifyCmd, err := client.InitNotify(options)
	if err != nil {
		_ = client.Close() // Best effort close
		return nil, nil, fmt.Errorf("failed to initialize NOTIFY: %w", err)
	}

	return client, notifyCmd, nil
}

// connect establishes a connection to the IMAP server with NOTIFY handler.
func (l *Listener) connect(ctx context.Context) (*imap.Client, error) {
	l.logger.Info("Connecting to IMAP server", "server", l.account.Server)

	// Create handler that enqueues remote events for STATUS updates.
	handler := &imapclient.UnilateralDataHandler{
		Status: func(data *imap2.StatusData) {
			l.logger.Debug("STATUS event", "mailbox", data.Mailbox, "numMessages", data.NumMessages, "uidNext", data.UIDNext)
			// FIXME: this might block (briefly), but using the ctx in scope seems wrong.
			l.queue <- RemoteEvent{Mailbox: data.Mailbox}
			l.logger.Debug("Queued remote event for STATUS", "mailbox", data.Mailbox)
		},
	}

	client, err := imap.Connect(ctx, l.account.Server, l.account.Username, l.account.Password, l.account.Plaintext, handler, l.logger)
	if err != nil {
		return nil, fmt.Errorf("failed to connect: %w", err)
	}

	l.logger.Info("Connected to IMAP server")
	return client, nil
}

// enqueueMailboxes discovers and enqueues full sync tasks for all mailboxes.
func (l *Listener) enqueueMailboxes(ctx context.Context, client *imap.Client) error {
	mailboxes, err := client.ListMailboxes(ctx)
	if err != nil {
		return fmt.Errorf("failed to list mailboxes: %w", err)
	}

	l.logger.Info("Queueing mailboxes for sync", "count", len(mailboxes))
	for _, mailbox := range mailboxes {
		l.logger.Info("Queued full sync with local rescan", "mailbox", mailbox)
		select {
		case l.queue <- RemoteEvent{Mailbox: mailbox}:
		case <-ctx.Done():
			l.logger.Debug("Context cancelled while enqueuing mailboxes")
			return ctx.Err()
		}
	}

	return nil
}

// startKeepAlive starts a goroutine that sends NOOP commands every 15 minutes.
// Prevents server from disconnecting NOTIFY connection after 30 minutes of inactivity.
// Returns a cancel function to stop the keep-alive goroutine.
func (l *Listener) startKeepAlive(ctx context.Context, client *imap.Client) context.CancelFunc {
	keepAliveCtx, cancel := context.WithCancel(ctx)

	go func() {
		ticker := time.NewTicker(15 * time.Minute)
		defer ticker.Stop()

		l.logger.Debug("Keep-alive started.")

		for {
			select {
			case <-ticker.C:
				l.logger.Debug("Sending keep-alive NOOP.")
				if err := client.Noop(keepAliveCtx); err != nil {
					l.logger.Warn("Keep-alive NOOP failed.", "error", err)
				} else {
					l.logger.Debug("Keep-alive NOOP sent successfully.")
				}
			case <-keepAliveCtx.Done():
				l.logger.Debug("Keep-alive stopped")
				return
			}
		}
	}()

	return cancel
}
