package sync

import (
	"context"
	"fmt"
	"log/slog"
	"path/filepath"
	"sync"

	"git.sr.ht/~whynothugo/ImapGoose/internal/imap"
	"git.sr.ht/~whynothugo/ImapGoose/internal/maildir"
	"git.sr.ht/~whynothugo/ImapGoose/internal/status"
	imap2 "github.com/emersion/go-imap/v2"
	"golang.org/x/sync/semaphore"
)

// Syncer handles synchronisation between IMAP and local Maildir.
type Syncer struct {
	client  *imap.Client
	maildir *maildir.Maildir
	status  *status.Repository
	mailbox string
	logger  *slog.Logger
}

// New creates a new Syncer.
func New(client *imap.Client, maildir *maildir.Maildir, status *status.Repository, mailbox string, logger *slog.Logger) *Syncer {
	return &Syncer{
		client:  client,
		maildir: maildir,
		status:  status,
		mailbox: mailbox,
		logger:  logger,
	}
}

// Sync performs synchronization based on the requested scan flags.
// If absPaths is provided, performs targeted file-level sync for those files.
// Otherwise, conditionally scans remote (IMAP) and/or local (maildir) based on flags.
func (s *Syncer) Sync(ctx context.Context, absPaths []string, scanLocal, scanRemote bool) error {
	s.logger.Info("Starting sync", "mailbox", s.mailbox, "scanLocal", scanLocal, "scanRemote", scanRemote)

	if len(absPaths) > 0 {
		s.logger.Debug("Performing file-level sync", "paths", absPaths)
		if err := s.syncFiles(ctx, absPaths); err != nil {
			return fmt.Errorf("failed to sync files: %w", err)
		}
	} // else: we should never have absPaths and scanLocal!

	if scanRemote {
		if err := s.scanRemote(ctx); err != nil {
			return fmt.Errorf("failed to scan remote: %w", err)
		}
	}

	if scanLocal {
		if err := s.scanLocal(ctx); err != nil {
			return fmt.Errorf("failed to scan local: %w", err)
		}
	}

	s.logger.Info("Sync completed", "mailbox", s.mailbox)
	return nil
}

// scanRemote scans the IMAP server for changes and syncs remote to local.
// Uses QRESYNC if MODSEQ is available, otherwise performs full scan.
// Selects the IMAP mailbox.
func (s *Syncer) scanRemote(ctx context.Context) error {
	s.logger.Info("Scanning remote", "mailbox", s.mailbox)

	lastModSeq, err := s.status.GetHighestModSeq(s.mailbox)
	if err != nil {
		return fmt.Errorf("failed to get stored HIGHESTMODSEQ: %w", err)
	}

	storedUIDValidity, hasUIDValidity, err := s.status.GetUIDValidity(s.mailbox)
	if err != nil {
		return fmt.Errorf("failed to get stored UIDValidity: %w", err)
	}

	// Build QRESYNC options if we have stored state.
	var qresync *imap2.SelectQResyncOptions
	if hasUIDValidity && lastModSeq > 0 {
		qresync = &imap2.SelectQResyncOptions{
			UIDValidity: storedUIDValidity,
			ModSeq:      lastModSeq,
		}
		s.logger.Debug("Using QRESYNC", "lastModSeq", lastModSeq)
	} else {
		s.logger.Info("No MODSEQ found, performing full remote scan")
	}

	result, err := s.client.SelectMailbox(ctx, s.mailbox, qresync)
	if err != nil {
		return fmt.Errorf("failed to select mailbox: %w", err)
	}

	if err := s.status.SetUIDValidity(s.mailbox, result.UIDValidity); err != nil {
		return fmt.Errorf("UIDValidity validation failed: %w", err)
	}

	if qresync != nil && lastModSeq == result.HighestModSeq {
		s.logger.Debug("No remote changes detected", "highestModSeq", result.HighestModSeq)
		return nil
	}

	if qresync != nil {
		s.logger.Info("Processing remote changes", "lastModSeq", lastModSeq, "highestModSeq", result.HighestModSeq)
		return s.scanRemoteIncremental(ctx, result, lastModSeq)
	} else {
		return s.scanRemoteFull(ctx, result)
	}
}

func (s *Syncer) scanLocal(ctx context.Context) error {
	statusUIDs, err := s.status.GetUIDs(ctx, s.mailbox)
	if err != nil {
		return fmt.Errorf("failed to reload status: %w", err)
	}

	localMessages, err := s.maildir.List()
	if err != nil {
		return fmt.Errorf("failed to list local messages: %w", err)
	}

	if err := s.syncLocalToRemote(ctx, localMessages, statusUIDs); err != nil {
		return fmt.Errorf("failed to sync local to remote: %w", err)
	}
	return nil
}

// scanRemoteIncremental performs incremental sync using QRESYNC.
// PRECONDITION: the IMAP mailbox must be selected.
func (s *Syncer) scanRemoteIncremental(ctx context.Context, selectResult *imap.SelectResult, lastModSeq uint64) error {
	changedMessages, err := s.client.FetchChangesSince(ctx, lastModSeq)
	if err != nil {
		return fmt.Errorf("failed to fetch changed messages: %w", err)
	}

	var downloaded, deleted, flagChanges int

	// Process changed messages (new + flag changes).
	for uid, msg := range changedMessages {
		statusMsg, err := s.status.GetByUID(ctx, s.mailbox, uid)
		if err != nil {
			return fmt.Errorf("failed to check status for UID %d: %w", uid, err)
		}

		if statusMsg == nil {
			// Not in status: new message.
			if err = s.downloadMessage(ctx, uid); err != nil {
				return fmt.Errorf("error downloading message: %w", err)
			}
			downloaded++
		} else if !flagsEqual(statusMsg.Flags, msg.Flags) {
			// Present in status: flag change only.
			err := s.updateLocalFlags(ctx, uid, statusMsg.Filename, statusMsg.Flags, msg.Flags)
			if err != nil {
				return fmt.Errorf("error updating local flags: %w", err)
			}
			flagChanges++
		}
	}

	for _, uid := range selectResult.VanishedUIDs {
		if err := ctx.Err(); err != nil {
			return err
		}

		ok, err := s.deleteLocalMessage(ctx, uid)
		if err != nil {
			return fmt.Errorf("failed to delete message: %w", err)
		} else if ok {
			deleted++
		}
	}

	s.logger.Info("Remote changes processed", "downloaded", downloaded, "flag_changes", flagChanges, "deleted", deleted)

	if err := s.status.SetHighestModSeq(s.mailbox, selectResult.HighestModSeq); err != nil {
		return fmt.Errorf("failed to store HIGHESTMODSEQ: %w", err)
	}

	return nil
}

// scanRemoteFull performs a full sync reading all messages.
// PRECONDITION: the IMAP mailbox must be selected.
func (s *Syncer) scanRemoteFull(ctx context.Context, selectResult *imap.SelectResult) error {
	s.logger.Info("Performing full remote scan")

	statusUIDs, err := s.status.GetUIDs(ctx, s.mailbox)
	if err != nil {
		return fmt.Errorf("failed to load status: %w", err)
	}

	remoteMessages, err := s.client.ListMessages(ctx)
	if err != nil {
		return fmt.Errorf("failed to list remote messages: %w", err)
	}

	s.logger.Info("Loaded repositories", "remote", len(remoteMessages), "status", len(statusUIDs))

	if err := s.syncRemoteToLocal(ctx, remoteMessages, statusUIDs); err != nil {
		return fmt.Errorf("failed to sync remote to local: %w", err)
	}

	if err := s.status.SetHighestModSeq(s.mailbox, selectResult.HighestModSeq); err != nil {
		return fmt.Errorf("failed to store HIGHESTMODSEQ: %w", err)
	}

	return nil
}

// syncFiles performs a targeted sync for multiple files that changed locally.
// Selects the IMAP mailbox.
func (s *Syncer) syncFiles(ctx context.Context, absPaths []string) error {
	s.logger.Info("Starting file sync", "mailbox", s.mailbox, "count", len(absPaths), "paths", absPaths)

	selectResult, err := s.client.SelectMailbox(ctx, s.mailbox, nil)
	if err != nil {
		return fmt.Errorf("failed to select mailbox: %w", err)
	}

	if err := s.status.SetUIDValidity(s.mailbox, selectResult.UIDValidity); err != nil {
		return fmt.Errorf("UIDValidity validation failed: %w", err)
	}

	for _, absPath := range absPaths {
		if err := ctx.Err(); err != nil {
			return err
		}

		if err := s.syncSingleFile(ctx, absPath); err != nil {
			return fmt.Errorf("failed to sync file %s: %w", absPath, err)
		}
	}

	s.logger.Info("File sync completed", "mailbox", s.mailbox, "count", len(absPaths))
	return nil
}

// syncSingleFile syncs a single file. Assumes mailbox is already selected.
// PRECONDITION: the IMAP mailbox must be selected.
func (s *Syncer) syncSingleFile(ctx context.Context, absPath string) error {
	basename := filepath.Base(absPath)

	flags, exists, err := s.maildir.GetFlagsByAbsPath(absPath)
	if err != nil {
		return fmt.Errorf("failed to get local message: %w", err)
	}

	// Status repository is keyed by basename (not relative path).
	statusMsg, err := s.status.GetByFilename(ctx, s.mailbox, basename)
	if err != nil {
		return fmt.Errorf("failed to check status for filename %s: %w", basename, err)
	}

	if exists && statusMsg != nil && flagsEqual(statusMsg.Flags, flags) {
		// Message exists on both sides and flags up to date.
		s.logger.Debug("No changes detected for file", "path", absPath)
		return nil
	}

	if exists && statusMsg == nil {
		// File exists locally but not in status: upload to IMAP.
		s.logger.Info("New local file detected, uploading", "path", absPath)

		localMsg := maildir.Message{
			Filename: absPath,
			Flags:    flags,
		}
		if err = s.uploadMessage(ctx, basename, localMsg); err != nil {
			return fmt.Errorf("error uploading message: %w", err)
		}

		return nil
	}

	if exists && statusMsg != nil {
		// File exists locally and in status: check for flag changes.
		if !flagsEqual(statusMsg.Flags, flags) {
			err := s.updateRemoteFlags(ctx, statusMsg.UID, basename, statusMsg.Flags, flags)
			if err != nil {
				return fmt.Errorf("error updating remove flags: %w", err)
			}
		} else {
			s.logger.Debug("No changes detected for file", "path", absPath)
		}
		return nil
	}

	if !exists && statusMsg != nil {
		// File doesn't exist locally but in status: delete from IMAP.
		s.logger.Info("Local file deleted, removing from remote", "path", absPath, "uid", statusMsg.UID)
		if err := s.deleteRemoteMessage(ctx, statusMsg.UID); err != nil {
			return err
		}
		return nil
	}

	// File doesn't exist locally or in status: nothing to do (already cleaned up).
	// XXX: I think this should not happen??
	s.logger.Debug("File not found locally or in status, nothing to do", "path", absPath)
	return nil
}

// syncRemoteToLocal implements the syncto(R, L, S) algorithm.
// PRECONDITION: the IMAP mailbox must be selected.
func (s *Syncer) syncRemoteToLocal(
	ctx context.Context,
	remote map[imap2.UID]imap.Message,
	statusUIDs map[imap2.UID]bool,
) error {
	var downloaded, deleted, flagsUpdated int

	// Step 1: Download messages in (remote - status) concurrently.
	sem := semaphore.NewWeighted(16) // 16 = max concurrency.
	resultCh := make(chan error, len(remote))
	var wg sync.WaitGroup

	for uid := range remote {
		if !statusUIDs[uid] {
			if err := sem.Acquire(ctx, 1); err != nil {
				return err
			}

			wg.Add(1)
			go func(uid imap2.UID) {
				defer wg.Done()
				defer sem.Release(1)
				resultCh <- s.downloadMessage(ctx, uid)
			}(uid)
		}
	}

	go func() {
		wg.Wait()
		close(resultCh)
	}()
	for err := range resultCh {
		if err != nil {
			return err
		}
		downloaded++
	}

	// Update flags for existing messages.
	for uid, remoteMsg := range remote {
		if statusUIDs[uid] {
			statusMsg, err := s.status.GetByUID(ctx, s.mailbox, uid)
			if err != nil {
				return fmt.Errorf("failed to get status for UID %d: %w", uid, err)
			}

			if statusMsg != nil {
				if !flagsEqual(statusMsg.Flags, remoteMsg.Flags) {
					err := s.updateLocalFlags(ctx, uid, statusMsg.Filename, statusMsg.Flags, remoteMsg.Flags)
					if err != nil {
						return fmt.Errorf("error updating local flags: %w", err)
					}
					flagsUpdated++
				}
			}
		}
	}

	// Step 2: Delete local messages in (status - remote).
	for uid := range statusUIDs {
		if err := ctx.Err(); err != nil {
			return err
		}

		if _, existsInRemote := remote[uid]; !existsInRemote {
			ok, err := s.deleteLocalMessage(ctx, uid)
			if err != nil {
				return fmt.Errorf("failed to delete message: %w", err)
			} else if ok {
				deleted++
			}
		}
	}

	s.logger.Info("Syncing remote to local completed", "downloaded", downloaded, "deleted", deleted, "flags_updated", flagsUpdated)
	return nil
}

// syncLocalToRemote implements the syncto(L, R, S) algorithm.
// Selects the IMAP mailbox.
func (s *Syncer) syncLocalToRemote(
	ctx context.Context,
	local map[string]maildir.Message,
	statusUIDs map[imap2.UID]bool,
) error {
	var uploaded, deleted, flagsUpdated int

	sem := semaphore.NewWeighted(16) // 16 = max concurrency.
	uploadResultCh := make(chan error, len(local))
	flagResultCh := make(chan error, len(local))
	deleteResultCh := make(chan error, len(statusUIDs))
	var wg sync.WaitGroup

	selectResult, err := s.client.SelectMailbox(ctx, s.mailbox, nil)
	if err != nil {
		return fmt.Errorf("failed to select mailbox: %w", err)
	}

	if err := s.status.SetUIDValidity(s.mailbox, selectResult.UIDValidity); err != nil {
		return fmt.Errorf("UIDValidity validation failed: %w", err)
	}

	// Step 1: Upload new messages and update flags concurrently.
	for filename, localMsg := range local {
		statusMsg, err := s.status.GetByFilename(ctx, s.mailbox, filename)
		if err != nil {
			return fmt.Errorf("failed to check status for filename %s: %w", filename, err)
		}

		if statusMsg == nil {
			// New message — upload to remote.
			if err := sem.Acquire(ctx, 1); err != nil {
				return err
			}

			wg.Add(1)
			go func(filename string, localMsg maildir.Message) {
				defer wg.Done()
				defer sem.Release(1)
				uploadResultCh <- s.uploadMessage(ctx, filename, localMsg)
			}(filename, localMsg)
		} else if !flagsEqual(statusMsg.Flags, localMsg.Flags) {
			// Existing message with flag changes — Update remote.
			if err := sem.Acquire(ctx, 1); err != nil {
				return err
			}

			wg.Add(1)
			go func(uid imap2.UID, newFilename string, oldFlags, newFlags []imap2.Flag) {
				defer wg.Done()
				defer sem.Release(1)
				flagResultCh <- s.updateRemoteFlags(ctx, uid, newFilename, oldFlags, newFlags)
			}(statusMsg.UID, filename, statusMsg.Flags, localMsg.Flags)
		}
	}

	// Step 2: Delete remote messages in (status - local) concurrently.
	for uid := range statusUIDs {
		// Look up the filename for this UID.
		statusMsg, err := s.status.GetByUID(ctx, s.mailbox, uid)
		if err != nil {
			return fmt.Errorf("failed to get status for UID %d: %w", uid, err)
		}

		// Check if this file still exists locally.
		var existsInLocal bool
		if statusMsg != nil {
			_, existsInLocal = local[statusMsg.Filename]
		}

		if !existsInLocal {
			if err := sem.Acquire(ctx, 1); err != nil {
				return err
			}

			wg.Add(1)
			go func(uid imap2.UID) {
				defer wg.Done()
				defer sem.Release(1)
				deleteResultCh <- s.deleteRemoteMessage(ctx, uid)
			}(uid)
		}
	}

	// Hint: this completes before the below loop finishes.
	go func() {
		wg.Wait()
		close(uploadResultCh)
		close(flagResultCh)
		close(deleteResultCh)
	}()

	// Collect results from all three channels concurrently.
	for uploadResultCh != nil || flagResultCh != nil || deleteResultCh != nil {
		select {
		case err, ok := <-uploadResultCh:
			if !ok {
				uploadResultCh = nil
			} else if err != nil {
				return err
			} else {
				uploaded++
			}
		case err, ok := <-flagResultCh:
			if !ok {
				flagResultCh = nil
			} else if err != nil {
				return err
			} else {
				flagsUpdated++
			}
		case err, ok := <-deleteResultCh:
			if !ok {
				deleteResultCh = nil
			} else if err != nil {
				return err
			} else {
				deleted++
			}
		}
	}

	s.logger.Info("Sync local to remote completed", "uploaded", uploaded, "deleted", deleted, "flags_updated", flagsUpdated)
	return nil
}
