/*
 * Copyright 1993, 1995 Christopher Seiwald.
 *
 * This file is part of Jam - see jam.c for Copyright information.
 */

# include "jam.h"
# include "filesys.h"
# include "regexp.h"
# include "lists.h"
# include "variable.h"

# ifdef USE_PATHUNIX

/*
 * pathunix.c - manipulate file names on UNIX, NT, OS2, AmigaOS
 *
 * External routines:
 *
 *	file_parse() - split a file name into dir/base/suffix/member
 *	file_build() - build a filename given dir/base/suffix/member
 *	file_parent() - make a FILENAME point to its parent dir
 *
 * File_parse() and file_build() just manipuate a string and a structure;
 * they do not make system calls.
 *
 * 04/08/94 (seiwald) - Coherent/386 support added.
 * 12/26/93 (seiwald) - handle dir/.suffix properly in file_build()
 * 12/19/94 (mikem) - solaris string table insanity support
 * 12/21/94 (wingerd) Use backslashes for pathnames - the NT way.
 * 02/14/95 (seiwald) - parse and build /xxx properly
 * 02/23/95 (wingerd) Compilers on NT can handle "/" in pathnames, so we
 *                    should expect hdr searches to come up with strings
 *                    like "thing/thing.h". So we need to test for "/" as
 *                    well as "\" when parsing pathnames.
 * 03/16/95 (seiwald) - fixed accursed typo on line 69.
 * 05/03/96 (seiwald) - split from filent.c, fileunix.c
 * 12/20/96 (seiwald) - when looking for the rightmost . in a file name,
 *		      don't include the archive member name.
 */

/*
 * file_parse() - split a file name into dir/base/suffix/member
 */

void
file_parse( 
	char	*file,
	FILENAME *f )
{
	char *p, *q;
	char *end;
	
	memset( (char *)f, 0, sizeof( *f ) );

	/* Look for <grist> */

	if( file[0] == '<' && ( p = strchr( file, '>' ) ) )
	{
	    f->f_grist.ptr = file;
	    f->f_grist.len = p - file;
	    file = p + 1;
	}

	/* Look for dir/ */

	p = strrchr( file, '/' );

# ifndef UNIX
# ifndef AMIGA
	/* On NT, look for dir\ as well */
	{
	    char *p1 = strrchr( file, '\\' );
	    p = p1 > p ? p1 : p;
	}
# endif
# endif

	if( p )
	{
	    f->f_dir.ptr = file;
	    f->f_dir.len = p - file;
	
	    /* Special case for / - dirname is /, not "" */

	    if( !f->f_dir.len )
		f->f_dir.len = 1;

# ifndef UNIX
# ifndef AMIGA
	    /* Special case for D:/ - dirname is D:/, not "D:" */

	    if( f->f_dir.len == 2 && file[1] == ':' )
		f->f_dir.len = 3;
# endif
# endif

	    file = p + 1;
	}

	end = file + strlen( file );

	/* Look for (member) */

	if( ( p = strchr( file, '(' ) ) && end[-1] == ')' )
	{
	    f->f_member.ptr = p + 1;
	    f->f_member.len = end - p - 2;
	    end = p;
	} 

	/* Look for .suffix */
	/* This would be memrchr() */

	p = 0;
	q = file;

	while( q = (char *)memchr( q, '.', end - q ) )
	    p = q++;

	if( p )
	{
	    f->f_suffix.ptr = p;
	    f->f_suffix.len = end - p;
	    end = p;
	}

	/* Leaves base */

	f->f_base.ptr = file;
	f->f_base.len = end - file;
}

/*
 * file_build() - build a filename given dir/base/suffix/member
 */

void
file_build(
	FILENAME *f,
	char	*file,
	int	binding )
{
	/* Start with the grist.  If the current grist isn't */
	/* surrounded by <>'s, add them. */

	if( f->f_grist.len )
	{
	    if( f->f_grist.ptr[0] != '<' ) *file++ = '<';
	    memcpy( file, f->f_grist.ptr, f->f_grist.len );
	    file += f->f_grist.len;
	    if( file[-1] != '>' ) *file++ = '>';
	}

	/* Don't prepend root if it's . or directory is rooted */

# if PATH_DELIM == '/'

	if( f->f_root.len 
	    && !( f->f_root.len == 1 && f->f_root.ptr[0] == '.' )
	    && !( f->f_dir.len && f->f_dir.ptr[0] == '/' ) )

# else /* unix */

	if( f->f_root.len 
	    && !( f->f_root.len == 1 && f->f_root.ptr[0] == '.' )
	    && !( f->f_dir.len && f->f_dir.ptr[0] == '/' )
	    && !( f->f_dir.len && f->f_dir.ptr[0] == '\\' )
	    && !( f->f_dir.len && f->f_dir.ptr[1] == ':' ) )

# endif /* unix */

	{
	    memcpy( file, f->f_root.ptr, f->f_root.len );
	    file += f->f_root.len;
	    *file++ = PATH_DELIM;
	}

	if( f->f_dir.len )
	{
	    memcpy( file, f->f_dir.ptr, f->f_dir.len );
	    file += f->f_dir.len;
	}

	/* UNIX: Put / between dir and file */
	/* NT:   Put \ between dir and file */

	if( f->f_dir.len && ( f->f_base.len || f->f_suffix.len ) )
	{
	    /* UNIX: Special case for dir \ : don't add another \ */
	    /* NT:   Special case for dir / : don't add another / */

# if PATH_DELIM == '\\'
	    if( !( f->f_dir.len == 3 && f->f_dir.ptr[1] == ':' ) )
# endif
		if( !( f->f_dir.len == 1 && f->f_dir.ptr[0] == PATH_DELIM ) )
		    *file++ = PATH_DELIM;
	}

	if( f->f_base.len )
	{
	    memcpy( file, f->f_base.ptr, f->f_base.len );
	    file += f->f_base.len;
	}

	if( f->f_suffix.len )
	{
	    memcpy( file, f->f_suffix.ptr, f->f_suffix.len );
	    file += f->f_suffix.len;
	}

	if( f->f_member.len )
	{
	    *file++ = '(';
	    memcpy( file, f->f_member.ptr, f->f_member.len );
	    file += f->f_member.len;
	    *file++ = ')';
	}
	*file = 0;
}

/*
 *	file_parent() - make a FILENAME point to its parent dir
 */

void
file_parent( FILENAME *f )
{
	/* just set everything else to nothing */

	f->f_base.ptr =
	f->f_suffix.ptr =
	f->f_member.ptr = "";

	f->f_base.len = 
	f->f_suffix.len = 
	f->f_member.len = 0;
}


#define MAXPATTERN      10

/*
** Don't make the buffers too large. The package description is to be found at 
** the start of the file somewhere.
*/

#define BUFSIZE       1024 

void file_classfile( FILENAME * fn)
{

  LIST * packagescan;
  char buf1[BUFSIZE];
  char packagename[BUFSIZE];
  char buf2[BUFSIZE];
  char * c1;
  char * c2;
  FILE * f;
  regexp * re[MAXPATTERN];
  int rec = 0;
  int i;
  int found = 0;

  packagescan = var_get( "PACKAGESCAN" );
  if (! packagescan)
    return;

  while (rec < MAXPATTERN && packagescan)
  {
    re[rec++] = regcomp( packagescan->string );
    packagescan = list_next( packagescan );
  }

  /*
  ** Build the filename from the grist, directory, basename and suffix.
  */

  memset(buf1, 0, BUFSIZE);
  c1 = buf1;

  /*
  ** Add the grist if any. Maybe there is a better way to do
  ** all this but I haven't looked around long enough...
  */

  if (fn->f_grist.len) {
    for (i = 1; i < fn->f_grist.len; i++) {
      if (fn->f_grist.ptr[i] == '!') {
        *c1++ = '/';
      }
      else {
        *c1++ = fn->f_grist.ptr[i];
      }
    }
    *c1++ = '/';
  }

  /*
  ** Add the directory component.
  */

  if (fn->f_dir.len) {
    for (i = 0; i < fn->f_dir.len; i++) {
      *c1++ = fn->f_dir.ptr[i];
    }
    *c1++ = '/';
  }

  /*
  ** Add the basename component.
  */

  if (fn->f_base.len) {
    for (i = 0; i < fn->f_base.len; i++) {
      *c1++ = fn->f_base.ptr[i];
    }
  }

  /*
  ** Add the suffix component.
  */

  if (fn->f_suffix.len) {
    for (i = 0; i < fn->f_suffix.len; i++) {
      *c1++ = fn->f_suffix.ptr[i];
    }
  }

//  printf("opening file '%s' grist = %s %d\n", buf1, fn->f_grist.ptr, fn->f_grist.len);  
  if (! ( f = fopen(buf1, "r") ) ) {
//    printf("failed\n");
    fn->f_grist.ptr = fn->f_base.ptr = fn->f_suffix.ptr = fn->f_member.ptr = fn->f_dir.ptr = "";
    fn->f_grist.len = fn->f_base.len = fn->f_suffix.len = fn->f_member.len = fn->f_dir.len = 0;
    return;
  }
    
  while ( fgets( buf2, sizeof(buf2), f ) )
  {
    for (i = 0; i < rec; i++) {
      if ( regexec( re[i], buf2 ) && re[i]->startp[1] ) {
        re[i]->endp[1][0] = '\0';
        memset(buf1, 0, sizeof(buf1));
        memcpy(buf1, re[i]->startp[1], strlen(re[i]->startp[1]));
        found = 1;
        goto have_it;
      }
    }
  }

have_it:
  
  fclose(f);

  if (found) {
    memset(buf2, 0, BUFSIZE);
    c2 = buf2;
    c1 = buf1;

    /*
    ** Create a version where all whitespace is removed and
    ** the dots are replaced with slashes.
    */
    
    while (*c1) {
      if (*c1 != ' ') {
        if (*c1 == '.') {
          *c2++ = '/';
        }
        else {
          *c2++ = *c1;
        }
      }
      c1 += 1;
    }

    *c2++ = '/';

    /*
    ** Create a generated class filename based on our concoctation.
    */

    memcpy(buf2 + strlen(buf2), fn->f_base.ptr, fn->f_base.len);
    memcpy(buf2 + strlen(buf2), ".class", 6);
    file_parse(buf2, fn);

  }
  else {
  
    /*
    ** Nothing found, just change the .java into a .class extension
    */
    
    fn->f_grist.ptr = fn->f_base.ptr = fn->f_suffix.ptr = fn->f_member.ptr = fn->f_dir.ptr = "";
    fn->f_grist.len = fn->f_base.len = fn->f_suffix.len = fn->f_member.len = fn->f_dir.len = 0;
  }

}
# endif /* unix, NT, OS/2, AmigaOS */
