/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * $Id: bsdav_audio_hw.c,v 1.9 2005/11/20 00:35:16 jakemsr Exp $
 */

#include "includes.h"

#include <sys/ioctl.h>

#include <err.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "bsdav.h"


int
bsdav_get_aud_formats(int fd)
{
#if defined(HAVE_SUN_AUDIO)
audio_encoding_t audenc;
int	 i, j;

	for (i = 0; ; i++) {
		audenc.index = i;
		if (ioctl(fd, AUDIO_GETENC, &audenc) < 0)
			break;
		for (j = 0; bsdav_aud_fmts[j].name; j++) {
			if (audenc.encoding == bsdav_aud_fmts[j].audio_enc) {
				bsdav_aud_fmts[j].audio_idx = i;
				break;
			}
		}
	}

	return(0);
#elif defined(HAVE_OSS_AUDIO)
int fmts;
int i;

	if (ioctl(fd, SNDCTL_DSP_GETFMTS, &fmts) < 0) {
		warn("SNDCTL_DSP_GETFMTS");
		return(1);
	}

	for (i = 0; bsdav_aud_fmts[i].name; i++)
		if (fmts & bsdav_aud_fmts[i].audio_enc)
			bsdav_aud_fmts[i].audio_idx = i;

	return(0);
#endif
}


int
bsdav_list_audio_formats(char *dev, int fd)
{
#if defined(HAVE_SUN_AUDIO)
audio_device_t adev;
#endif
char	*dev_name;
int	 i;
int	 dev_was_open = 0;


	if (fd < 0) {
		if ((fd = open(dev, O_RDONLY)) < 0) {
			warn("%s", dev);
			close(fd);
			return(1);
		}
	} else
		dev_was_open = 1;

	if (bsdav_get_aud_formats(fd) != 0) {
		warnx("get audio formats failed");
		return(1);
	}


#if defined(HAVE_SUN_AUDIO)
	/* get the name of the device */
	if (ioctl(fd, AUDIO_GETDEV, &adev) < 0) {
		warn("AUDIO_GETDEV");
		return(1);
	}
	dev_name = adev.name;
#elif defined(HAVE_OSS_AUDIO)
	dev_name = "";
#endif

	if (dev_was_open != 1)
		close(fd);

	printf("Audio Formats\n");
	printf("  %s: %s\n", dev, dev_name);
	printf("    %-12s  %16s\n", "name", "bits per sample");
	for (i = 0; bsdav_aud_fmts[i].name; i++) {
		if (bsdav_aud_fmts[i].audio_idx != -1)
			printf("    %12s  %16d\n", bsdav_aud_fmts[i].name,
			    bsdav_aud_fmts[i].bps);
	}

	return(0);
}


int
bsdav_audio_init(int fd, int mode, int fmt, int chan, int srate)
{
#if defined(HAVE_SUN_AUDIO)
audio_info_t sa_if;
audio_info_t ga_if;
struct audio_prinfo s_prinfo;
struct audio_prinfo g_prinfo;


	AUDIO_INITINFO(&sa_if);
	AUDIO_INITINFO(&ga_if);
	if (mode == BSDAV_AUDMODE_REC) {
		s_prinfo = sa_if.record;
		g_prinfo = ga_if.record;
	} else {
		s_prinfo = sa_if.play;
		g_prinfo = ga_if.play;
	}

	if (mode == BSDAV_AUDMODE_REC) {
		sa_if.mode = AUMODE_RECORD;
	} else {
		sa_if.mode = AUMODE_PLAY_ALL;
	}
	if (ioctl(fd, AUDIO_SETINFO, &sa_if) < 0) {
		warn("AUDIO_SETINFO");
		return(1);
	}

	bsdav_get_aud_formats(fd);

	s_prinfo.encoding = bsdav_aud_fmts[fmt].audio_enc;
	s_prinfo.precision = bsdav_aud_fmts[fmt].bps;

	if (mode == BSDAV_AUDMODE_REC)
		sa_if.record = s_prinfo;
	else
		sa_if.play = s_prinfo;

	if (ioctl(fd, AUDIO_SETINFO, &sa_if) < 0) {
		warn("AUDIO_SETINFO");
		return(1);
	}
	if (ioctl(fd, AUDIO_GETINFO, &ga_if) < 0) {
		warn("AUDIO_GETINFO");
		return(1);
	}

	if (mode == BSDAV_AUDMODE_REC)
		g_prinfo = ga_if.record;
	else
		g_prinfo = ga_if.play;

	/* Make sure the settings stuck. */
	if ((s_prinfo.encoding != g_prinfo.encoding) ||
	    (s_prinfo.precision != g_prinfo.precision)) {
		warnx("unable to set encoding");
		return(1);
	}


	/* channels */
	s_prinfo.channels = chan;
	if (mode == BSDAV_AUDMODE_REC)
		sa_if.record = s_prinfo;
	else
		sa_if.play = s_prinfo;
	if (ioctl(fd, AUDIO_SETINFO, &sa_if) < 0) {
		warn("AUDIO_SETINFO");
		return(1);
	}
	if (ioctl(fd, AUDIO_GETINFO, &ga_if) < 0) {
		warn("AUDIO_GETINFO");
		return(1);
	}
	if (mode == BSDAV_AUDMODE_REC)
		g_prinfo = ga_if.record;
	else
		g_prinfo = ga_if.play;

	if (s_prinfo.channels != g_prinfo.channels) {
		warnx("unable to set channels");
		return(1);
	}


	/* sample rate */
	s_prinfo.sample_rate = srate;
	if (mode == BSDAV_AUDMODE_REC)
		sa_if.record = s_prinfo;
	else
		sa_if.play = s_prinfo;
	if (ioctl(fd, AUDIO_SETINFO, &sa_if) < 0) {
		warn("AUDIO_SETINFO");
		return(1);
	}
	if (ioctl(fd, AUDIO_GETINFO, &ga_if) < 0) {
		warn("AUDIO_GETINFO");
		return(1);
	}
	if (mode == BSDAV_AUDMODE_REC)
		g_prinfo = ga_if.record;
	else
		g_prinfo = ga_if.play;

	if (s_prinfo.sample_rate != g_prinfo.sample_rate) {
		warnx("unable to set audio sample rate");
		return(1);
	}


	/* may be samples of different encoding still in the buffer */
	if (ioctl(fd, AUDIO_FLUSH) < 0) {
		warn("AUDIO_FLUSH");
		return(1);
	}

#elif defined(HAVE_OSS_AUDIO)

	/* mode */
	/* doesn't matter? */

	/* format */
	if (ioctl(fd, SNDCTL_DSP_SETFMT, &bsdav_aud_fmts[fmt].audio_enc) < 0) {
		warn("SNDCTL_DSP_SETFMT");
		return(1);
	}

	/* channels */
	if (ioctl(fd, SNDCTL_DSP_CHANNELS, &chan) < 0) {
		warn("SNDCTL_DSP_CHANNELS");
		return(1);
	}

	/* sample rate */
	if (ioctl(fd, SOUND_PCM_WRITE_RATE, &srate) < 0) {
		warn("SOUND_PCM_WRITE_RATE");
		return(1);
	}

#endif
	return(0);
}


size_t
bsdav_get_audhw_buf_size(int fd, int mode)
{
size_t buf_size;

#if defined(HAVE_SUN_AUDIO)
audio_info_t audio_if;

	AUDIO_INITINFO(&audio_if);
	if (ioctl(fd, AUDIO_GETINFO, &audio_if) < 0) {
		warn("AUDIO_GETINFO");
		return(0);
	}
	buf_size = (mode == BSDAV_AUDMODE_PLAY) ?
	    audio_if.play.buffer_size : audio_if.record.buffer_size;

#elif defined(HAVE_OSS_AUDIO)
audio_buf_info abinfo;

	if (mode == BSDAV_AUDMODE_PLAY) {
		if (ioctl(fd, SNDCTL_DSP_GETISPACE, &abinfo) < 0) {
			warn("SNDCTL_DSP_GETISPACE");
			return(0);
		}
	} else {
		if (ioctl(fd, SNDCTL_DSP_GETOSPACE, &abinfo) < 0) {
			warn("SNDCTL_DSP_GETOSPACE");
			return(0);
		}
	}
	buf_size = abinfo.fragstotal * abinfo.fragsize;

#endif
	return(buf_size);
}


size_t
bsdav_get_audhw_buf_pos(int fd, int mode)
{
size_t pos;

#if defined(HAVE_SUN_AUDIO)
audio_info_t audio_if;

	AUDIO_INITINFO(&audio_if);
	if (ioctl(fd, AUDIO_GETINFO, &audio_if) < 0) {
		warn("AUDIO_GETINFO");
		return(0);
	}
	pos = (mode == BSDAV_AUDMODE_PLAY) ?
	    audio_if.play.seek : audio_if.record.seek;

#elif defined(HAVE_OSS_AUDIO)
audio_buf_info abinfo;

	if (mode == BSDAV_AUDMODE_PLAY) {
		if (ioctl(fd, SNDCTL_DSP_GETISPACE, &abinfo) < 0) {
			warn("SNDCTL_DSP_GETISPACE");
			return(0);
		}
	} else {
		if (ioctl(fd, SNDCTL_DSP_GETOSPACE, &abinfo) < 0) {
			warn("SNDCTL_DSP_GETOSPACE");
			return(0);
		}
	}
	pos = abinfo.fragments * abinfo.fragsize;

#endif
	return(pos);
}
