/* Copyright (c) 1991 by John Atwood deVries II. */
/* For copying and distribution information, see the file COPYING. */
/* $Id$ */

#include <sys/param.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <db.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#if defined(__DragonFly__)
#include <libutil.h>
#else
#include <util.h>
#endif
#include "externs.h"
#include "userdb.h"
#include "version.h"

#if defined(__DragonFly__) && !defined(O_SYNC)
#define O_SYNC O_FSYNC
#endif

static void	open_logs (void);
void	close_logs (void);
void	cycle_logs (int);
void	icbexit (int);

int
main (int argc, char **argv)
{
	int c;
	int noforkflag = 0;
	int nodbflag = 0;
	int port = DEFAULTPORT;
	int i;
	long TheTime;
	char *bindhost = (char *) NULL;
	struct rlimit rlp;

	userdb = NULL;

	restart = 0;

	setbuf(stdout, (char *) 0);

	while ((c = getopt(argc, argv, "p:fDRb:")) != EOF)
		switch (c) {
		    case 'b':
			bindhost = optarg;
			break;
		    case 'D':
			nodbflag++;
			break;
		    case 'R':
			restart++;
			break;
		    case 'f':
			noforkflag++;
			break;
		    case 'p':
			port = atoi(optarg);
			break;
		    case '?':
		    default:
			fputs("usage: icbd [-b host] [-p port] [-cRf]", stderr);
			fputs("-D	 don't use user database", stderr);
			fputs("-f	 don't fork", stderr);
			fputs("-p port	 bind to port instead of default", stderr);
			fputs("-b host	 bind socket to \"host\"", stderr);
			exit(1);
			break;
		}

#ifdef RLIMIT_NOFILE
	getrlimit(RLIMIT_NOFILE, &rlp);
	rlp.rlim_cur = MAX_USERS + 1;
	if (setrlimit(RLIMIT_NOFILE, &rlp) < 0) {
		perror("setrlimit");
		exit(1);
	}
#endif

	if (restart == 0) {
		open_logs();

		/* get our hostname */
		if (gethostname(thishost,MAXHOSTNAMELEN) < 0)
			error("gethostname failed: %s", strerror(errno));

#ifdef	BIND_HOSTNAME
		/* if bindhost is null (unset), let BIND_HOSTNAME set it */
		if ( (char *)NULL == bindhost )
		{
			bindhost = BIND_HOSTNAME;
			strlcpy (thishost, BIND_HOSTNAME, MAXHOSTNAMELEN);
		}
#endif	/* BIND_HOSTNAME */

		if ( (char *) NULL != bindhost )
			if ( '\0' != *bindhost )
				strlcpy (thishost, bindhost, MAXHOSTNAMELEN);

		if (makenewport(bindhost, port) < 0) {
			error("makenewport failed: %s", strerror(errno));
			exit(1);
		}

		if (noforkflag == 0)
			daemon(0,0);

		pidfile("icbd");

		if (nodbflag == 0)
			if ((userdb = dbopen(USERDB, O_RDWR, 0600, DB_HASH, NULL)) == NULL) {
				error("dbopen of USERDB failed");
			}

		/* initialize out tables  */
		init_groups();
		clear_users();

		/* invent our resident automagical user */
		TheTime = time(NULL);
		fill_user_entry(NICKSERV, "server", thishost, "Server", "", "ICB", TheTime, 0, 1);
		u_tab[NICKSERV].login = 1;
		u_tab[NICKSERV].t_on = TheTime;
		u_tab[NICKSERV].t_recv = TheTime;
		strlcpy(u_tab[NICKSERV].realname, "registered", MAX_NICKLEN);
		fill_group_entry(0, "ICB", "...here to serve you!", SUPERSECRET, RESTRICTED, NICKSERV, QUIET);
		nickwritetime(NICKSERV, 0);

		info("ICB revision %s on %s.", VERSION, thishost);
		info("There are %d users possible.", MAX_USERS);
		info("Of those, %d are real.", MAX_REAL_USERS);

		for (i = 0; i < MAX_REAL_USERS; i++) {
			S_kill[i] = 0;
			S_bricks[i] = INIT_BRICKS;
		}
	
		setpolldelay(POLL_DELAY);

		setpolltimeout(IDLE_POLLWAIT);

		trapsignals();
	} else {
		open_logs();

		/* get our hostname */
		if (gethostname(thishost,MAXHOSTNAMELEN) < 0) {
			error("gethostname failed: %s", strerror(errno));
		}

		setpolldelay(POLL_DELAY);
		setpolltimeout(IDLE_POLLWAIT);

		init_groups();
		clear_users();
		trapsignals(); 

		MDB("[RESTART] Reloading server data");
		icbload(0);

		for (i = 0; i < MAX_USERS; i++)
			if (u_tab[i].login == 1)
				sendimport(i, "Restart", "Server is back up.");
	}

#ifdef RLIMIT_NOFILE
        getrlimit(RLIMIT_NOFILE, &rlp);
        info("Max fds set to %qd", rlp.rlim_cur);
#endif

	/* start the serve loop */
	serverserve();

	/* all's well that ends */
	if (userdb)
		userdb->close(userdb);

	exit(0);
	/* NOTREACHED */
}

static void
open_logs(void)
{
	if((icbd_log = open(ICBDLOG,O_WRONLY|O_SYNC|O_CREAT|O_APPEND, 0644)) < 0) {
		fprintf(stderr, "icbd: could not open log file.\n");
		perror("ICBDLOG open");
		exit(1);
	}
}

void
close_logs(void)
{
	close(icbd_log);
}

/*ARGSUSED*/
void
cycle_logs(int sig)
{
	close_logs();
	open_logs();
}

/*ARGSUSED*/
void
icbexit(int sig)
{
	int user;

	info("someone told us to exit");
	for (user=0; user < MAX_REAL_USERS; user++)
		if (u_tab[user].login == 1)
			disconnectuser(user);

	if(userdb)
		userdb->close(userdb);

	exit(0);
}
