;;; jde.el -- Integrated Development Environment for Java.
;; $Revision: 1.286 $ $Date: 2002/12/30 05:25:24 $ 

;; Author: Paul Kinnucan <pkinnucan@attbi.com>
;; Maintainer: Paul Kinnucan
;; Keywords: java, tools

;; Copyright (C) 1997, 1998, 1999, 2000, 2001, 2002 Paul Kinnucan.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This is one of a set of packages that make up the 
;; Java Development Environment (JDE) for Emacs. See the
;; JDE User's Guide for more information.

;; The latest version of the JDE is available at
;; <URL:http://jde.sunsite.dk>.
;; <URL:http://www.geocities.com/SiliconValley/Lakes/1506/>

;; Please send any comments, bugs, or upgrade requests to
;; Paul Kinnucan at paulk@mathworks.com.

;;; Code:


;;;###autoload
(defconst jde-version "2.3.2"
  "JDE version number.")

(defconst jde-semantic-min-version "1.4beta13"
  "Semantic minimum version")
(defconst jde-semantic-max-version "2.0"
  "Semantic maximum version")
(defconst jde-speedbar-min-version "0.14beta2"
  "Speedbar minimum version")
(defconst jde-speedbar-max-version "0.16"
  "Speedbar maximum version")
(defconst jde-eieio-min-version "0.17beta3"
  "Eieio minimum version")
(defconst jde-eieio-max-version "0.18"
  "Eieio maximum version")

(defconst jde-xemacsp (string-match "XEmacs" (emacs-version))
  "Non-nil if we are running in the XEmacs environment.")

(defconst jde-xemacs20p (and jde-xemacsp (>= emacs-major-version 20)))
 
(defconst jde-emacs21p (and (string-match "\\bEmacs\\b" (emacs-version))
			    (>= emacs-major-version 21)))

(require 'jde-autoload)
(require 'jde-help)
(require 'semantic-load)
(require 'easymenu)
(require 'cl)
(require 'font-lock)
(require 'cc-mode)
(require 'cus-edit)
(require 'comint)
(require 'jde-compile)
(require 'jde-db)
(require 'jde-bug)
(require 'jde-jdb)
(require 'jde-run)
(require 'jde-gen)
(require 'compile)
(require 'imenu)
(require 'browse-url)
(require 'beanshell)
(require 'jde-wiz)
(require 'jde-java-grammar)
(require 'jde-complete)
(require 'jde-which-method)
(require 'jde-java-font-lock)
(require 'jde-import)
(require 'executable)  ;; in XEmacs' sh-script package
(require 'efc)
(require 'jde-open-source)

(if (not (fboundp 'custom-set-default))
    (defalias 'custom-set-default 'set-default))


(defgroup jde nil
  "Java Development Environment"
  :group 'tools
  :prefix "jde-")

(defgroup jde-project nil
  "JDE Project Optio
 ns"
  :group 'jde
  :prefix "jde-")

(defcustom jde-check-version-flag t
  "*Non-nil means to check versions of semantic, eieio, and speedbar.
That is if they meet the requirements for this version of the JDE.
If nil only check if semantic, eieio, and speedbar are available.
See also the function `jde-check-versions'."
  :group 'jde
  :type 'boolean)

;; (makunbound 'jde-key-bindings)
(defcustom jde-key-bindings
  (list 
   (cons "[?\C-c ?\C-v ?\C-a]" 'jde-run-menu-run-applet)
   (cons "[?\C-c ?\C-v ?\C-b]" 'jde-build)
   (cons "[?\C-c ?\C-v ?\C-c]" 'jde-compile)
   (cons "[?\C-c ?\C-v ?\C-d]" 'jde-debug)
   (cons "[?\C-c ?\C-v ?\C-f]" 'jde-find)
   (cons "[?\C-c ?\C-v ?\C-g]" 'jde-open-class-at-point)
   (cons "[?\C-c ?\C-v ?\C-k]" 'bsh)
   (cons "[?\C-c ?\C-v ?\C-l]" 'jde-gen-println)
   (cons "[?\C-c ?\C-v ?\C-n]" 'jde-help-browse-jdk-doc)
   (cons "[?\C-c ?\C-v ?\C-p]" 'jde-save-project)
   (cons "[?\C-c ?\C-v ?\C-q]" 'jde-wiz-update-class-list)
   (cons "[?\C-c ?\C-v ?\C-r]" 'jde-run)
   (cons "[?\C-c ?\C-v ?\C-s]" 'speedbar-frame-mode)
   (cons "[?\C-c ?\C-v ?\C-t]" 'jde-db-menu-debug-applet)
   (cons "[?\C-c ?\C-v ?\C-w]" 'jde-help-symbol)
   (cons "[?\C-c ?\C-v ?\C-x]" 'jde-show-superclass-source)
   (cons "[?\C-c ?\C-v ?\C-y]" 'jde-open-class-at-point)
   (cons "[?\C-c ?\C-v ?\C-z]" 'jde-import-find-and-import)
   (cons "[?\C-c ?\C-v ?e]"    'jde-wiz-extend-abstract-class)
   (cons "[?\C-c ?\C-v ?f]"    'jde-gen-try-finally-wrapper)
   (cons "[?\C-c ?\C-v ?i]"    'jde-wiz-implement-interface)
   (cons "[?\C-c ?\C-v ?j]"    'jde-javadoc-autodoc-at-line)
   (cons "[?\C-c ?\C-v ?o]"    'jde-wiz-override-method)
   (cons "[?\C-c ?\C-v ?t]"    'jde-gen-try-catch-wrapper)
   (cons "[?\C-c ?\C-v ?\C-[]" 'jde-run-etrace-prev)
   (cons "[?\C-c ?\C-v ?\C-]]" 'jde-run-etrace-next)
   (cons "[(control c) (control v) (control ?.)]" 'jde-complete)
   (cons "[(control c) (control v) ?.]" 'jde-complete-in-line)
   )
  "*Specifies key bindings for the JDE.
The value of this variable is an association list. The car of
each element specifies a key sequence. The cdr specifies 
an interactive command that the key sequence executes. To enter
a key with a modifier, type C-q followed by the desired modified
keystroke. For example, to enter C-s (Control s) as the key to be
bound, type C-q C-s in the key field in the customization buffer.
You can use the notation [f1], [f2], etc., to specify function keys."
  :group 'jde-project
  :type '(repeat
	  (cons :tag "Key binding"
		(string :tag "Key")
		(function :tag "Command")))
  :set '(lambda (sym val)
	  (mapc
	   (lambda (buf)
	     (save-excursion
	       (set-buffer buf)
	       (when (boundp 'jde-mode-map)
		 ;; Unmap existing key bindings
		 (if (and (boundp 'jde-key-bindings)
			  jde-key-bindings)
		     (mapc 
		      (lambda (binding)
			(let ((key (car binding)))
			  (if (string-match "\\[.+]"key)
			      (setq key (car (read-from-string key))))
			  (local-unset-key key)))
		      jde-key-bindings))
		 ;; Map new key bindings.
		 (mapc 
		  (lambda (binding)
		    (let ((key (car binding))
			  (fcn (cdr binding)))
		      (if (string-match "\\[.+]" key)
			  (setq key (car (read-from-string key))))
		      (define-key (current-local-map) key fcn)))
		  val))))
	   (jde-get-java-source-buffers))
	  (set-default sym val)))

(defcustom jde-launch-beanshell-on-demand-p t
  "If non-nil, the JDE launches the Beanshell the first time it is needed.
Otherwise, the JDE launches the Beanshell, if it is not already running,
whenever you open a Java source file."
  :group 'jde-project
  :type 'boolean)

(defcustom jde-project-context-switching-enabled-p t
  "*Enable project context switching.
If non-nil, the JDE reloads a buffer's project file when you switch to the buffer from
another buffer belonging to another project. You can disable this feature if you prefer
to load project files manually. The debugger uses this variable to disable context-switching
temporarily when stepping through code."
  :group 'jde-project
  :type 'boolean)

(defun jde-toggle-project-switching ()
  "Toggles project switching on or off."
  (interactive)
  (setq jde-project-context-switching-enabled-p 
	(not jde-project-context-switching-enabled-p)))

(defcustom jde-java-environment-variables '("JAVA_VERSION" "JAVA_HOME")
  "This variable specifies the names of environment variables used to
specify the version and location of the JDK to be used by the JDE.
If set, the `jde-jdk' customization variable overrides the 
java enviroment variables."
  :group 'jde-project
  :type '(list
	  (string :tag "Java Version")
	  (string :tag "Java Home")))

(defun jde-set-jdk-dir-type (sym val)
  (if val
      (let ((type
	     (list
	      (quote radio-button-choice)
		  )))
	    (loop for jdk in val do
		  (setq 
		   type 
		   (append
		    type
		    (list (list (quote item) (car jdk))))))
	    (put 'jde-jdk 
		 'custom-type 
		 (list (quote list) type))
	    (put 'jde-jdk 'customized-value nil)
	    (put 'jde-jdk
		 'standard-value
		 (list (list (quote list) (car (car val)))))
	    (customize-set-value 'jde-jdk (list (car (car val)))))
    (progn
      (put 'jde-jdk 'custom-type 'symbol)
      (put 'jde-jdk 'standard-value nil)
      (put 'jde-jdk 'customized-value nil)
      (customize-set-value 'jde-jdk nil)))
  (set-default sym val))

;; (makunbound 'jde-jdk-registry)
(defcustom jde-jdk-registry nil
  "Specifies the versions and locations of the JDKs installed on your
system.  For each JDK to be registered, enter the version number
(e.g., 1.4.0) of the JDK in the Version field. Enter the path of the
JDK's root directory (e.g., c:/jdk1.3.1 or $JAVA_HOME) in the Path
field. Setting this variable determines the choices offered by the
`jde-jdk' variable. You should therefore customize this variable
first."
  :group 'jde-project
  :type '(repeat
	  (cons 
	   :tag "JDK"
	   (string :tag "Version")
	   (string :tag "Path")))
  :set 'jde-set-jdk-dir-type)

;; (makunbound 'jde-jdk)
(defcustom jde-jdk nil
  "Specifies the version of the JDK to be used to develop the current
project. The version must be one of the versions listed in the
`jde-jdk-registry'. If you specify nil (the default), the JDE uses the
JDK specified by the Java version environment variable (see
`jde-java-enviroment-variables', if set; otherwise, the first JDK
located on the system command path specified by te PATH environment
variable.

You must customize `jde-jdk-registry' first, then `jde-jdk'. After you
have customized jde-jdk-registry, the customization buffer for`
jde-jdk' presents you with a set of radio buttons, one for each
registered JDK.  Select the button of the JDK that you want to use for
the current project."
  :group 'jde-project
  :type 'symbol

  (if (or (featurep 'xemacs) (< emacs-major-version 21))
      :set
    :set-after)

  (if (or (featurep 'xemacs) (< emacs-major-version 21))
      'custom-set-default
    '(jde-jdk-registry)))
 


(defun jde-get-jdk-dir () 
  "Get the root directory of the JDK currently being used by the
JDE. The directory is the directory of the version of the JDK
specified by `jde-jdk'. If none is specified, this function returns
the value of the Java home environment variable (see
`jde-java-environment-variables') or the first JDK directory on the
system path, i.e., the directory that contains java on Unix systems or
java.exe on Windows systems.  If neither `jde-jdk' nor the system path
nor the Java home environment variable specify a JDK directory, this
function displays an error message."
  (interactive)
  (if jde-jdk
      (let* ((jdk (assoc (car jde-jdk) jde-jdk-registry))
	     (jdk-dir (cdr jdk)))
        (when (null jdk)
          (error (format
                  "No mapping in the jde-jdk-registry found for JDK version %s"
                  (car jde-jdk))))
	(if (not (string= jdk-dir ""))
	    (progn
	      (setq jdk-dir (substitute-in-file-name jdk-dir))
	      (if (not (file-exists-p jdk-dir))
		  (error 
		   (format "The path specified for JDK %s does not exist: %s"
			   jde-jdk
			   jdk-dir)))))
	jdk-dir)
    (let ((jdk-dir (getenv (nth 1 jde-java-environment-variables))))
      (if jdk-dir
	  (progn
	    (setq jdk-dir (substitute-in-file-name jdk-dir))
	    (if (not (file-exists-p jdk-dir))
		(error 
		 (format "The path specified by %s does not exist: %s"
			 (nth 1 jde-java-environment-variables)
			 jdk-dir))))
	(progn 
	  (setq jdk-dir 
		(executable-find 
		 (if (eq system-type 'windows-nt) "java.exe" "java")))
	  (if jdk-dir
	      (setq jdk-dir 
		    (expand-file-name 
		     ".." 
		     (file-name-directory jdk-dir)))
	    (error "Cannot find the JDK directory. See `jde-jdk'."))))
      jdk-dir)))


(defun jde-get-jdk-prog (progname)
   "Returns the full path of the program passed in.  By default, assume
   it's in the bin directory under `jde-get-jdk-dir', but if not,
   look in the environment's command path."
   (let* ((progname-str 
	   (if (symbolp progname)
	       (symbol-name progname) progname))
          (full-progname 
	   (if (eq system-type 'windows-nt)
	       (concat progname-str ".exe")
	     progname-str))
          (progpath 
	   (expand-file-name 
	    (concat
	     (if (eq system-type 'darwin) "Home/bin/" "bin/")
	     full-progname)
	    (jde-get-jdk-dir))))
     (if (file-exists-p progpath)
       progpath
       (executable-find full-progname))))

(defun jde-get-tools-jar ()
  "Gets the correct tools.jar or equivalent"
  (expand-file-name (if (eq system-type 'darwin)
                      "Classes/classes.jar"
                      "lib/tools.jar")
                    (jde-get-jdk-dir)))

(defvar jde-java-version-cache nil
"Cache to hold the version of Java being used.")

(defun jde-java-version-via-java ()
  "Get the version of the java vm on the
system command path."
  (if (not jde-java-version-cache)
      (let ((buf (get-buffer-create "java version"))
	    proc)
	(save-excursion
	  (set-buffer buf)
	  (setq proc
		(start-process
		 "java version" buf "java" "-version"))
	  (accept-process-output proc 10)
	  (goto-char (point-min))
	  (re-search-forward "[1-9][.][1-9]" (point-max) t)
	  (setq jde-java-version-cache (match-string 0)))
	(kill-buffer buf)))
  jde-java-version-cache)

(defun jde-java-version ()
  "Get the version of Java used by the JDE."
  (interactive)
  (let ((java-version (if jde-jdk (car jde-jdk)
			(getenv 
			 (nth 0 jde-java-environment-variables)))))
    (if (not java-version)
	(if jde-java-version-cache
	    (setq java-version jde-java-version-cache)
	  (if (bsh-running-p)
	      (progn
		(setq jde-java-version-cache
		      (jde-jeval-r "jde.util.JdeUtilities.getJavaVersion();"))
		(setq java-version jde-java-version-cache))
	    (setq java-version (jde-java-version-via-java)))))	    
    (if (interactive-p)
      (message java-version)
      java-version)))

(defun jde-java-major-version ()
  "Returns an integer representing
the major version of the JDK being used
by the current project."
  (let ((version (jde-java-version)))
    (string-match "\\([0-9]+\\)\\.\\([0-9]+\\)"
		version)
    (string-to-int
     (substring 
     version 
     (match-beginning 1)
     (match-end 1)))))

(defun jde-java-minor-version ()
  "Returns an integer representing
the minor version of the JDK being used
by the current project."
  (let ((version (jde-java-version)))
    (string-match "\\([0-9]+\\)\\.\\([0-9]+\\)"
		version)
    (string-to-int
     (substring 
     version 
     (match-beginning 2)
     (match-end 2)))))


;;(makunbound 'jde-jdk-doc-url)
(defcustom jde-jdk-doc-url ""
  "*URL of JDK documentation. 
This can point to a remote or local copy of the documentation. If the value
of this variable is the null string, the JDE looks for the JDK documentation
in the docs subdirectory of the directory returned by `jde-get-jdk-dir'."
  :group 'jde-project
  :type 'file)

;;(makunbound 'jde-global-classpath)
(defcustom jde-global-classpath nil
  "Specify a common classpath for compile, run, and debug commands.
Use this variable if you want to the JDE to use the same classpath for
compiling, running,and debugging an application. Note that the value
of this variable is a list of strings, each of which specifies a
path. The JDE converts this list to a colon- or semicolon-separated
list before inserting in the compiler or vm command line. 

The path may start with a tilde (~) or period (.) and may include
environment variables. The JDEE replaces a ~ with your home directory.
If `jde-resolve-relative-paths-p' is nonnil, the JDEE replaces the
. with the path of the current project file. The JDEE replaces each
instance of an environment variable with its value before inserting it
into the command line.

You can specify different classpaths for compiling, running and
debugging applicaitons. Use `jde-compile-option-classpath' to specify
the compilation classpath, `jde-run-option-classpath' to specify the
run classpath, and/or `jde-db-option-classpath' to specify the debug
classpath. You can use these variables together. For example, suppose
that you need to use one classpath for compilation and other for
running and debugging. You could do this by setting
`jde-compile-option-classpath' to the compile classpath and
`jde-global-classpath' to the run and debug classpath. If you set
`jde-global-classpath', the JDE uses it to construct the classpath for
any operation for which you do not set the operation-specific
classpath variable (e.g., `jde-compile-option-classpath'). 

If you do not set `jde-global-classpath', the JDE uses the operation-specific
classpath if it is set. If neither the global nor the
operation-specific classpath is set, the JDE does not generate a
-classpath argument for the operation, e.g., compile or run a Java
class. In this case, the operation uses the value of the CLASSPATH variable
if specified."
  :group 'jde-project
  :type '(repeat (file :tag "Path")))

(defcustom jde-quote-classpath t
  "*Quote the classpath argument.
Set this option on when using the bash shell with Windows 95 or NT.
The semicolons in the classpath confuse the shell."
  :group 'jde-project
  :type 'boolean)

(defcustom jde-expand-classpath-p t
  "Replace each occurence of a directory named `jde-lib-directory-names'
 in the classpath with paths to the jar and zip files in that directory."
  :group 'jde-project
  :type 'boolean)

;; (makunbound 'jde-lib-directory-names)
(defcustom jde-lib-directory-names (list "^lib" "^jar")
  "Regular expressions that matches names of jar/zip directories for 
the current project. See `jde-expand-classpath-p' and
`jde-expand-classpath' for more information"
  :group 'jde-project
  :type '(repeat (string :tag "Name")))

;; (makunbound 'jde-sourcepath)
(defcustom jde-sourcepath nil
  "*List of source directory paths.  The JDE uses this list to locate
source files corresponding to class files.  When entering paths in the
custom buffer, enter each path as a separate item in a separate edit
field. Do NOT put more than one path in the same edit field. You'll
only confuse JDE.  Paths may contain environment variables."
  :group 'jde-project
  :type '(repeat (file :tag "Path")))


(defcustom jde-project-name "default"
"Specifies name of project to which the current buffer belongs."
  :group 'jde-project
  :type 'string)

(defcustom jde-project-file-name "prj.el"
  "*Specify name of JDE project file.
When it loads a Java source file, the JDE looks for a lisp file of
this name (the default is prj.el in the source file hierarchy. If it
finds such a file, it loads the file. You can use this file to set the
classpath, compile options, and other JDE options on a
project-by-project basis."
  :group 'jde-project
  :type 'string)

(defcustom jde-project-hooks nil
  "Specifies a list of functions to be run when a project
becomes active. The JDE runs the project hooks after
the jde-mode hooks."
  :group 'jde-project
  :type '(repeat (function :tag "Function")))

;; (makunbound 'jde-build-function)
(defcustom jde-build-function '(jde-make)
  "*Function that will be invoked by the `jde-build' command.
The `jde-make' function uses a make 
program to rebuild the project. The `jde-ant-build' function
uses the Apache Ant program to build the project. You may also
specify a custom function to use. The custom function must
be an interactive function that can be called by 
`call-interactively'."
  :group 'jde-project
  :type '(list
	  (radio-button-choice
	   :format "%t \n%v"
	   :tag "Function: "
	   :entry-format " %b %v"
	   (const jde-make)
	   (const jde-ant-build)
	   (function my-custom-build-function))))

;;(makunbound 'jde-debugger)
(defcustom jde-debugger (list "jdb")
  "Specify the debugger you want to use to debug Java programs.
Select JDEbug, if you want to use the JDE's builtin debugger.  Select
jdb, if you want to use the default version of jdb for the JDK used by
the current project (see `jde-jdk'). Select old jdb, if you are using
JDK 1.2.2 or later and want to use the the old (e.g., pre-JPDA)
version of jdb instead of the new (JPDA-based) version of jdb."
  :group 'jde-project
  :type '(list 
	  (radio-button-choice
	  (item "JDEbug")
	  (item "jdb")
	  (item "old jdb")))
  :set '(lambda (sym val)
	  (mapc 
	   (lambda (buff)
	     (save-excursion	       
	       (set-buffer buff)
	       (if (string= (car val) "JDEbug")
		   (progn
		     (jde-jdb-minor-mode -1)
		     (jde-bug-minor-mode 1))
		 (progn
		   (jde-jdb-minor-mode 1)
		   (jde-bug-minor-mode -1)))))
	   (jde-get-java-source-buffers))
	  (set-default sym val)))

(defcustom jde-enable-senator t
  "Enable senator minor mode.
This mode provides Java-aware buffer navigation and searching
commands."
  :group 'jde-project
  :type 'boolean
  :set '(lambda (sym val)
          ;; Starting with version 1.4 beta 12 Semantic can globally
          ;; enable `senator-minor-mode'.  So don't override the
          ;; global setting in JDE's buffers.
          (or (and (boundp 'global-senator-minor-mode)
                   global-senator-minor-mode)
              (mapc 
               (lambda (buff)
                 (save-excursion	       
                   (set-buffer buff)
                   (senator-minor-mode (if val 1 -1))))
               (jde-get-java-source-buffers)))
	  (set-default sym val)))

(defcustom jde-enable-abbrev-mode nil
"*Enable expansion of abbreviations in jde-mode.
See `jde-mode-abbreviations' for more information."
  :group 'jde-project
  :type 'boolean)

(defcustom jde-mode-abbreviations
  (list 
   (cons "ab" "abstract")
   (cons "bo" "boolean")
   (cons "br" "break")
   (cons "by" "byte")
   (cons "byv" "byvalue")
   (cons "cas" "cast")
   (cons "ca" "catch")
   (cons "ch" "char")
   (cons "cl" "class")
   (cons "co" "const")
   (cons "con" "continue")
   (cons "de" "default")
   (cons "dou" "double")
   (cons "el" "else")
   (cons "ex" "extends")
   (cons "fa" "false")
   (cons "fi" "final")
   (cons "fin" "finally")
   (cons "fl" "float")
   (cons "fo" "for")
   (cons "fu" "future")
   (cons "ge" "generic")
   (cons "go" "goto")
   (cons "impl" "implements")
   (cons "impo" "import")
   (cons "ins" "instanceof")
   (cons "in" "int")
   (cons "inte" "interface")
   (cons "lo" "long")
   (cons "na" "native")
   (cons "ne" "new")
   (cons "nu" "null")
   (cons "pa" "package")
   (cons "pri" "private")
   (cons "pro" "protected")
   (cons "pu" "public")
   (cons "re" "return")
   (cons "sh" "short")
   (cons "st" "static")
   (cons "su" "super")
   (cons "sw" "switch")
   (cons "sy" "synchronized")
   (cons "th" "this")
   (cons "thr" "throw")
   (cons "throw" "throws")
   (cons "tra" "transient")
   (cons "tr" "true")
   (cons "vo" "void")
   (cons "vol" "volatile")
   (cons "wh" "while")
   )
"*Abbreviations used for Java keywords.
To use these abbreviations, you must enable abbrev-mode (see
`jde-enable-abbrev-mode'). To use an abbreviation, enter the
abbreviation followed by a white-space character. To suppress
expansion, enter C-q white-space."
   :group 'jde-project
  :type '(repeat 
	  (cons :tag "jde-mode abbreviation"
		(string :tag "Abbreviation")
		(string :tag "Expansion"))))

(defcustom jde-devel-debug nil
  "If true, use the JDEE Java classes in the jde/java/classes
directory instead of the jde.jar. This variable is intended for
use in testing the JDEE's java classes."
  :group 'jde-project
  :type 'boolean)

(defvar jde-mode-abbrev-table (make-abbrev-table)
  "Abbrev table for use in JDE-mode buffers.")


(defun jde-init-abbrev-table ()
  "Load the abbrev table.
Load it with a set of abbrevs that invoke an anonymous function that
does the expansion only if point is not in a quoted string or a
comment."

  ;; Note the use of lexical-let - must have the common lisp packages
  ;; around, since the anonymous function needs the closure provided by
  ;; lexical-let.
  (interactive)
  ;; (setq local-abbrev-table (make-abbrev-table))
  (mapc 
   (lambda (x)
     (lexical-let
	 ((abbrev (car x))		; this is the abbrev, lexically scoped
	  (expansion (cdr x)))		; this is the expansion
       (define-abbrev 
	 local-abbrev-table
	 abbrev
	 ""
	 (lambda ()
	   (if (jde-parse-comment-or-quoted-p)
	       (insert abbrev)		; insert the abbrev in quote/comment
	     (insert expansion)))       ; proceed with expansion elsewhere
	 0)))
   jde-mode-abbreviations)

  (if jde-gen-cflow-enable
      (jde-gen-load-abbrev-templates))

  (setq abbrevs-changed nil))

;; The next two functions contributed by s.nicolas@videotron.ca
(defun jde-abbrev-mode ()
"*Toggle abbreviation mode in JDE without altering project settings.
See `jde-mode-abbreviations' for more information."
 (interactive)
  (setq jde-enable-abbrev-mode (not jde-enable-abbrev-mode))
  (setq abbrev-mode jde-enable-abbrev-mode)
  (when jde-enable-abbrev-mode
     ;; Define abbreviations.a
    (jde-init-abbrev-table))
  (if jde-enable-abbrev-mode
    (message "abbreviation mode on")
    (message "abbreviation mode off")))

(defun jde-show-abbrevs ()
"*Shows a popup menu containing all available expansions.
See `jde-mode-abbreviations' for more information."
  (interactive)
   (let* ((expansions
          (mapcar
            (lambda(x) (cons (cdr x) (car x)))
              jde-mode-abbreviations))
         (expansion (car (imenu--mouse-menu expansions (if jde-xemacsp nil
t) "Abbreviations"))))
  (insert expansion)))


(defvar jde-classpath-separator (if (member system-type '(cygwin32 cygwin))
				    ";" path-separator)
  "The separator to use in a classpath.
This is usually the same as `path-separator'")

;;;###autoload
(defun jde-set-global-classpath (classpath)
  "Set the value of `jde-global-classpath'.
It specifies the -classpath argument for the Java compiler and
interpreter."
  (interactive 
   "sEnter classpath: ")
  (custom-set-variables 
   '(jde-global-classpath (split-string classpath jde-classpath-separator) t)))	
		
(defun jde-show-run-options ()
  "Show the JDE Run Options panel."
  (interactive)
  (customize-apropos "jde-run-options" 'groups))

(defun jde-show-debug-options ()
  "Show the JDE Debug Options panel."
  (interactive)
  (customize-apropos "jde-db-options" 'groups))

(defun jde-show-project-options ()
  "Show the JDE Debug Options panel."
  (interactive)
  (customize-apropos "jde-project" 'groups))

(defun jde-show-autocode-options ()
  "Show the JDE Autocode panel."
  (interactive)
  (customize-apropos "jde-gen" 'groups))

(defun jde-show-wiz-options ()
  "Show the JDE Wizards Options panel."
  (interactive)
  (customize-apropos "jde-wiz" 'groups))

(defun jde-show-complete-options ()
  "Show the JDE Complete Options panel."
  (interactive)
  (customize-apropos "jde-complete" 'groups))

;;;###autoload
(defun jde-build ()
  "Rebuild the entire project.
This command invokes the function defined by `jde-build-function'."
  (interactive)
  (call-interactively (car jde-build-function)))

; (define-derived-mode 
;   jde-mode java-mode "JDE"
;   "Major mode for developing Java applications and applets.
;   \\{jde-mode-map}"
;   (jde-mode-internal)
; )

;; The following is the expansion of the above macro.
;; We include the expansion to permit automatic
;; loading of the JDE.
(derived-mode-init-mode-variables 'jde-mode)
(put 'jde-mode 'derived-mode-parent 'java-mode)

;;;###autoload
(defun jde-mode ()
  "Major mode for developing Java applications and applets.
\\{jde-mode-map}"
  (interactive)
  (condition-case err
      (progn 
        (jde-check-versions)
        (java-mode)
        (if (get 'java-mode 'special)
            (put 'jde-mode 'special t))
        (setq major-mode 'jde-mode)
        (setq mode-name "JDE")
        (derived-mode-set-keymap 'jde-mode)
        (derived-mode-set-syntax-table 'jde-mode)
        (derived-mode-set-abbrev-table 'jde-mode)

        ;; Define buffer-local variables.
        (make-local-variable 'jde-project-name)
        (make-local-variable 'jde-run-applet-document)

        (setq jde-current-project 
              (or (jde-find-project-file default-directory)
                  ""));; Avoid setting twice the startup values!

        ;; Load the project file for this buffer. The project file
        ;; defines JDE options for a project.
        (if (and (not (jde-debugger-running-p)) jde-project-context-switching-enabled-p)
            (jde-load-project-file))

        ;; Enable support for automatic project switching.
        ;; This feature loads the appropriate project settings whenever
        ;; a user switches from a Java buffer belonging to one project
        ;; to a buffer belonging to another.
        (make-local-hook 'post-command-hook)
        (add-hook 'post-command-hook
                  'jde-detect-java-buffer-activation nil t)
  
	(if jde-xemacsp
	    (jde-insert-menu-in-xemacs-menubar))

        ;; Define underscore as a word constituent. This is needed
        ;; to support coding styles the begin fields with an underscore.
        (modify-syntax-entry ?_ "w")

        (when jde-enable-abbrev-mode
          ;; Define abbreviations.
          (jde-init-abbrev-table)
          (abbrev-mode 1))

        ;; Reset the key bindings in case jde-mode-keymap
        ;; was not bound at startup.
        (custom-initialize-reset 'jde-key-bindings nil)

        (if (and
             jde-setnu-mode-enable
             (< (point-max) jde-setnu-mode-threshold))
            (setnu-mode 1))

        (make-local-variable 'mode-line-format)
        (setq mode-line-format jde-mode-line-format)
  
        ;; When looking for a tag that has multiple matches
        ;; in the TAGS file, prefer (find first) the
        ;; occurrence in the _current_ buffer.
        ;; Contributed by Charles Rich, Mitsubishi Electric Research Laboratories,
        ;; Cambridge, MA>
        (make-local-variable 'tags-table-format-hooks)
        (setq tags-table-format-hooks '(jde-etags-recognize-tags-table
                                        recognize-empty-tags-table))

        (if (and
             (not jde-launch-beanshell-on-demand-p)
             (not (bsh-running-p)))
            (bsh-internal))

        (jde-wiz-set-bsh-project)

        ;; Setup Semantic stuff needed by the JDEE when Seantic is ready to
        ;; parse!
        (add-hook 'semantic-init-hooks  'jde-parse-semantic-default-setup)

	;; Install debug menu.
	(if (string= (car jde-debugger) "JDEbug")
	    (jde-bug-minor-mode 1)
	  (jde-jdb-minor-mode 1))

        ;; The next form must be the last executed
        ;; by jde-mode.
        (derived-mode-run-hooks 'jde-mode))    
    (error
     (message "%s" (error-message-string err)))))


(defun jde-check-versions () 
  "Check for correct versions of semantic and eieio.
Signal an error if they don't meet the requirements for this version
of the JDE.  If `jde-check-version-flag' is nil only check if
semantic, eieio, and speedbar are available."
  (let ((current-semantic-version semantic-version)
        (current-eieio-version eieio-version)
        (not-installed "%s is not installed.")
        (wrong-version
         "The wrong version of %s is installed, please install %s."))
    
    (if (null current-semantic-version) 
        (error (format not-installed "Semantic")))
    (if (null current-eieio-version) 
        (error (format not-installed "Eieio")))
    
    (when jde-check-version-flag
      
      ;;Checking Semantic version
      (if (not (jde-check-version current-semantic-version
                                  jde-semantic-min-version
                                  jde-semantic-max-version))
          (error (format wrong-version
                         "Semantic" 
                         jde-semantic-min-version)))
                       
      ;;Checking Eieio version
      (if (string= current-eieio-version "0.17beta1")
          (error (format wrong-version
                         "Eieio"
                         jde-eieio-min-version)))

      (if (not (jde-check-version current-eieio-version
                                  jde-eieio-min-version
                                  jde-eieio-max-version))
          (error (format wrong-version
                         "Eieio" 
                         jde-eieio-min-version))))))

(defun jde-check-version (current-version min-version max-version)
  "Return non-nil if CURRENT-VERSION >= MIN-VERSION or <= MAX-VERSION."
  (and (or (jde-earlier-versionp current-version 
                                 max-version)
           (string= current-version
                    max-version))
       (or (jde-earlier-versionp min-version
                                 current-version)
           (string= current-version
                    min-version))))

(defun jde-earlier-versionp (ver1 ver2) 
  "Return non-nil if VER1 is earlier than VER2"
  (let ((ver1-betap (string-match "beta" ver1))
        (ver2-betap (string-match "beta" ver2)))
    (if (or (and ver1-betap ver2-betap)
            (and (not ver1-betap) (not ver2-betap)))
        (string< ver1 ver2)
      (if ver1-betap
          (progn
            (or (string= ver2 (substring ver1  0 ver1-betap))
                (string< (substring ver1 0 ver1-betap) ver2)))
          (progn
            (string< ver1 (substring ver2 0 ver2-betap)))))))
      
(defcustom jde-log-max 500
  "*Maximum number of lines to keep in the JDE log buffer.
If nil, disable logging.  If t, don't truncate the buffer."
  :group 'jde-project
  :type '(choice (integer :tag "Number of lines to keep")
		 (boolean :tag "Disable/Unlimited")))

(defun jde-log-msg (msg &rest args)
  "Log message MSG to the *jde-log* buffer.
Optional ARGS are used to `format' MSG.
Does nothing if `jde-log-max' is nil."
  (if jde-log-max
      (save-match-data
	(save-excursion
	  (set-buffer (get-buffer-create "*jde-log*"))
	  (goto-char (point-max))
	  (insert (apply 'format msg args))
	  (insert "\n")
	  (if (integerp jde-log-max)
	      (let ((line-cnt 0))
		(while (search-backward "\n" nil t)
		  (setq line-cnt (1+ line-cnt)))
		(goto-char (point-min))
		(while (> line-cnt jde-log-max)
		  (delete-region (point) (search-forward "\n" nil t))
		  (setq line-cnt (1- line-cnt)))))))))

(defun jde-log-msg-t (msg &rest args)
  "Log message MSG to the *jde-log* buffer, and return t.
Optional ARGS are used to `format' MSG.
Does nothing but return t if `jde-log-max' is nil."
  (jde-log-msg msg args)
  t)

(defun jde-log-msg-nil (msg &rest args)
  "Log message MSG to the *jde-log* buffer, and return nil.
Optional ARGS are used to `format' MSG.
Does nothing but return nil if `jde-log-max' is nil."
  (jde-log-msg msg args)
  nil)

;; Make jde-mode the default mode for Java source code buffers.
;; Prepend the jde-mode entry so that it shadows the java-mode
;; entry already in the list.
;;;###autoload
(add-to-list 'auto-mode-alist '("\\.java\\'" . jde-mode))

(defcustom jde-menu-definition 
  (list "JDE"
	["Compile"           jde-compile t]
	;; ["Run App"           jde-run (not (jde-run-application-running-p))]
	["Run App"           jde-run t]
	["Debug App"         jde-debug t]
	"-"
	;;["-"                 ignore nil]
	["Run Applet"        jde-run-menu-run-applet t]
	["Debug Applet"      jde-jdb-menu-debug-applet t]
	"-"  
	["Build"             jde-build t]
	(list "Find"
	      ["Expression"    jde-find 
	                          (and 
				   (executable-find
				    (if (eq system-type 'windows-nt) "find.exe" "find"))
				   (executable-find
				    (if (eq system-type 'windows-nt) "grep.exe" "grep")))]
	      ["Expression..."  jde-find-dlg
	                          (and 
				   (executable-find
				    (if (eq system-type 'windows-nt) "find.exe" "find"))
				   (executable-find
				    (if (eq system-type 'windows-nt) "grep.exe" "grep")))]
	       ["Symbol Definition" jde-open-class-at-point t]
	       ["Class"  jde-show-class-source t]
	       ["Super Class"  jde-show-superclass-source t]
	       ["Interface"  jde-show-interface-source t]
	      )
	(list "Interpreter"
	      ["Start"         bsh t]
	      ["Exit"          bsh-exit t]
	 )
        (list "Documentation"
	      ["Add"             jde-javadoc-autodoc-at-line (jde-javadoc-enable-menu-p)]
	      ["Check This"      jde-javadoc-checkdoc-at-line (jde-javadoc-enable-menu-p)]
	      ["Check All"       jde-javadoc-checkdoc t]
	      ["Generate"        jde-javadoc-make t]
	)
        "-" 
	(list "Code Generation"
	      (list "Templates"
		    ["Get/Set Pair..."  jde-gen-get-set t]
		    ["Println..."       jde-gen-println t]
		    (list "Listener"
			  ["Action"          jde-gen-action-listener t]
			  ["Window"          jde-gen-window-listener t]
			  ["Mouse"           jde-gen-mouse-listener t]
			  )
		    ["Other..."        jde-gen-code t]
		    )
	      (list "Wizards"
		    ["Import class..."             jde-import-find-and-import t]
		    ["Override Method"             jde-wiz-override-method t]
		    ["Implement Interface..."      jde-wiz-implement-interface t]
		    ["Generate Get/Set Methods"    jde-wiz-get-set-methods t]
                    ["Update Package Statement"    jde-package-update t]
		    ["Implement Event Source..."   jde-wiz-implement-event-source t]
		    ["Extend Abstract Class..."    jde-wiz-extend-abstract-class t]
		    ["Delegate Methods..."         jde-wiz-delegate t]
		    "-"
		    ["Update Class List"   jde-wiz-update-class-list t]
		    )
	      )
	["Speedbar"          jde-show-speedbar t]
	["Check Style"  jde-checkstyle]
	(list "Project"
	      (vector "Auto Switch" 
                      'jde-toggle-project-switching
                      (if jde-xemacsp :active :enable) t 
                      :style 'radio 
                      :selected 'jde-project-context-switching-enabled-p)
	      (list "Options"
		    ["General"         jde-show-project-options t]
		    ["Compile"         jde-compile-show-options-buffer t]
		    ["Run"             jde-show-run-options t]
		    ["Debug"           jde-show-debug-options t]
		    ["Autocode"        jde-show-autocode-options t]
		    ["Javadoc"         jde-javadoc-customize t]
		    ["Make"            jde-make-show-options t]
		    ["Ant"             jde-ant-show-options t]
		    ["Complete"        jde-show-complete-options t]
		    ["Wiz"             jde-show-wiz-options t]
		    )
	      (list "Project File"
		    ["Create New" jde-create-new-project t]
		    ["Save"     jde-save-project t]
		    ["Load"     jde-load-project-file t]
		    ["Load All" jde-load-all-project-files t]
		    )
	      )
	(list "Help"
	      ["JDEE Users Guide"      jde-show-help t]
	      ["JDK"                   jde-help-browse-jdk-doc t]
	      ["JDEE Key Bindings"     jde-keys t]
	      "-"
	      ["Class..."              jde-help-class t]
	      ["Class Member..."       jde-help-class-member t]
	      ["Symbol at Point"       jde-help-symbol t]
	      "-"
	      ["Submit problem report" jde-submit-problem-report t]
	      "-"
	      (concat "JDE " jde-version)
	      )
	)
  "*The JDE main menu"
  :group 'jde-project
  :type 'sexp
  :set '(lambda (sym val)
	  (set-default sym val)
	  ; Define JDE menu for FSF Emacs.
	  (if (or (not jde-xemacsp) (featurep 'infodock))
	      (easy-menu-define jde-menu 
				jde-mode-map
				"Menu for JDE."
				val))
	  (if (and jde-xemacsp 
                   (eq major-mode 'jde-mode))
	      (jde-insert-menu-in-xemacs-menubar))))


(defun jde-insert-menu-in-xemacs-menubar ()
  "Insert JDE menu in the XEmacs menu bar."
  (if (and 
       (not (featurep 'infodock))
       (not (memq 'infodock c-emacs-features))
       (boundp 'current-menubar)
       current-menubar)
      (if (fboundp 'add-submenu)
	  (add-submenu nil jde-menu-definition)
	(add-menu nil "JDE" (cdr jde-menu-definition)))))

(defcustom jde-new-buffer-menu 
  (list
   "JDE New"
   ["Class..."         jde-gen-class-buffer t]
   ["Interface..."     jde-gen-interface-buffer t]
   ["Console..."       jde-gen-console-buffer t]
   (list
    "EJB"
    ["Session Bean"    jde-ejb-session-bean-buffer t]
    ["Entity Bean"     jde-ejb-entity-bean-buffer t])
   ["Other..."         jde-gen-buffer t]
   )
  "*The JDE New buffer menu"
  :group 'jde-project
  :type 'sexp
  :set '(lambda (sym val)
	  (set-default sym val)
	  (if jde-xemacsp
	      (unless (featurep 'infodock)
		(add-submenu '("File") val "Insert File..."))
	    (let* ((mb (assq 'menu-bar global-map))
		   (files (assq 'files mb))
		   (menu (if (fboundp 'easy-menu-create-menu)
			     (easy-menu-create-menu 
			      (car val) (cdr val))))
		   (menu-name (car val)))
	      (define-key-after (cdr (cdr files)) [jde-new]
		(cons menu-name menu)
		'open-file)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                            ;;
;; Classpaths                                                                 ;;
;;                                                                            ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun jde-cygpath (path &optional direction)
  "Converts a path from cygwin to DOS form if DIRECTION is nil.
Otherwise, it converts the path to cygwin form.  Requires that cygpath
be in your path."
  (interactive "sPath: ")
  (if (executable-find "cygpath")
      (save-excursion
	(let ((buf-name "*cygwin-output*")
	      (output-type (if direction "-u" "-w")))
	  (shell-command 
	   (concat "cygpath " output-type " -p '" path "'") buf-name)
	  (set-buffer buf-name)
	  (let ((output (buffer-substring (point-min) (point-max))))
	    (kill-buffer buf-name)
	    (substitute ?\/ ?\\ (remove ?\n output)))))
    (error "Cannot find cygpath executable.")))

(defvar jde-cygwin-root-cache nil
  "Cache of converted cygwin root directory paths.")

(defun jde-cygwin-path-converter-cygpath (path)
  (interactive "sPath: ")
  (if (string-match "^[a-zA-Z]:" path)
      path
    (cond
     ((string-match "^/\\(cygdrive\\)?/\\([a-zA-Z]\\)/" path)
      (concat 
       (substring
	path
	(match-beginning 2)
	(match-end 2))
       ":/"
       (substring path (match-end 0))))
     ((string-match "^/[^/]*" path)
       (let* ((root (substring 
		    path (match-beginning 0) (match-end 0)))
	      (rest (substring path (match-end 0)))
	      (converted-root (cdr (assoc root jde-cygwin-root-cache))))
	 (if (not converted-root)
	   (progn
             (setq converted-root (jde-cygpath root))
	     (if converted-root
		 (add-to-list 'jde-cygwin-root-cache 
			      (cons root converted-root))
	       (error "Cannot convert %s" path))))	 
	 (if (string= rest "")
	     converted-root
	   (concat converted-root rest))))
     (t
      (error "Cannot convert %s" path)))))
	 	 

(defun jde-cygwin-path-converter-internal (path)
  "Convert cygwin style PATH to a form acceptable to java vm.  Basically
converts paths of the form: '//C/dir/file' or '/cygdrive/c/dir/file' to
'c:/dir/file'.  This function will not modify standard unix style paths
unless they begin with '//[a-z]/' or '/cygdrive/[a-z]/'."
  (interactive "sPath: ")
  (if (fboundp 'mswindows-cygwin-to-win32-path)
      (substitute ?/ ?\\ (mswindows-cygwin-to-win32-path path))
    (let* ((path-re "/\\(cygdrive\\)?/\\([a-zA-Z]\\)/")
	   (subexpr 2)
	   (index1 (* 2 subexpr))
	   (index2 (1+ index1)))
      (if (string-match (concat "^" path-re) path)
	  (let ((new-path
		 (concat (substring path 
				    (nth index1 (match-data)) 
				    (nth index2 (match-data)))
			 ":/" 
			 (substring path (match-end 0)))))
	    (while (string-match (concat ":" path-re) new-path)
	      (setq new-path
		    (concat
		     (substring new-path 0 (match-beginning 0))
		     ";"
		     (substring new-path 
				(nth index1 (match-data)) 
				(nth index2 (match-data)))
		     ":/" 
		     (substring new-path (match-end 0)))))
	    (substitute ?\\ ?\/ new-path))
	path))))

(defcustom jde-cygwin-path-converter '(jde-cygwin-path-converter-internal)
  "Function to use to convert cygwin paths to DOS paths.  
Choose jde-cygwin-path-converter-internal, jde-cygwin-path-converter-cygpath,
or \"custom-function.\" jde-cygwin-path-converter-cygpath handles all
cygwin-style paths, including mount points, e.g.,/bin. 
jde-cygwin-path-converter-internal does not handle mount
paths. However, it is much faster as it does not require running a
subprocess every time the JDE needs to convert a path. Choose
\"custom-function\" if you want the JDE to use a function that you
supply. Replace \"custom-function\" with the name of the function that
you want to use."
  :group 'jde-project
  :type  '(list
	   (radio-button-choice :format "%t \n%v"
			       :tag "Converter: "
			       :entry-format "  %b %v"
			       (const jde-cygwin-path-converter-internal)
			       (const jde-cygwin-path-converter-cygpath)
			       (function custom-function))))
		       

(defun jde-convert-cygwin-path (path &optional separator)
  "Convert cygwin style PATH to a form acceptable to java vm, using
the conversion function specified by `jde-cygwin-path-converter'."
  (interactive "sPath: ")
  (funcall (car jde-cygwin-path-converter) 
	   (if separator (substitute ?\: (string-to-char separator) path) path)))

(defcustom jde-resolve-relative-paths-p t
  "If this variable is non-nil, the JDE converts relative paths to 
absolute paths. The JDE does this by appending the relative path to the path
of the project file for the current source buffer, if such
a file exists. Otherwise, the JDE appends the relative path to the path
of the current directory."
  :group 'jde-project
  :type 'boolean)

(defun jde-normalize-path (path &optional symbol) 
  "This function performs the following transformation on PATH:

  * Replaces environment variables of the form $VAR or ${VAR} with
    their values. Note that you must use the Unix notation for
    environment variables on the native Windows versions of Emacs and
    XEmacs.

  * Replaces the tilde character with the value of the home directory,
    typically specified by the HOME environment variable.

  * Converts Cygwin style paths to DOS notation on Windows.

  * Converts relative paths to absolute paths if
    `jde-resolve-relative-paths-p' is non-nil.  Paths are resolved
    according to the location of the deepest project file found, or if
    optional SYMBOL is non-nil, paths are resolved to the location of
    the deepest project file found that defines SYMBOL.

Note: PATH can either be a path string or a symbol corresponding to a
variable that holds a path string, in which case the optional arg
SYMBOL is unnecessary."
  (if (symbolp path)
      (setq symbol path
	    path (symbol-value symbol)))
  (let* ((directory-sep-char ?/)
	 (p (substitute-in-file-name path))
	(len (length p)))
    (if (and
	 jde-resolve-relative-paths-p
	 (> len 0)
	 (eq (aref p 0) ?.))
	(let* (prj-file-path
	       (dir (file-name-directory (or (buffer-file-name)
					     default-directory))))
	  ;; find the deepest originating project for the symbol
	  ;; based on the current directory, and resolve to that
	  ;; project's directory
	  (if symbol
	      (let ((prjs (get symbol 'jde-project))
		    (sort-fn
		     (lambda (x1 x2) 
		       (let* ((dir1 (file-name-directory (car x1)))
			      (dir2 (file-name-directory (car x2)))
			      match1 match2)
			 (if (null dir1)
			     (null dir2)
			   (if (null dir2)
			       t
			     (setq match1 (compare-strings
					   dir1 0 (length dir1)
					   dir 0 (length dir1)))
			     (setq match2 (compare-strings
					   dir2 0 (length dir2)
					   dir 0 (length dir2))))
			   (cond
			    ((not (eq match1 t))
			     (if (eq match2 t)
				 nil
			       (> (length dir1) (length dir2))))
			    ((not (eq match2 t))
			     t)
			    ((> (length dir1) (length dir2)))))))))
		(setq prjs (sort prjs sort-fn))
		(setq prj-file-path (caar prjs)))
	    (setq prj-file-path 
		  (jde-find-project-file dir)))
	  (if prj-file-path
	      (setq dir (file-name-directory prj-file-path))
	    (setq dir default-directory))
	  (if (and (> len 1)
		   (eq (aref p 1) ?.))
	      ;; path actually begins with `..', so normalize to one
	      ;; directory up
	      (save-match-data
		(string-match "\\.+/?" p)
		(setq p (expand-file-name (substring p (match-end 0))
					  (expand-file-name (concat dir "../")))))
	    (setq p (expand-file-name p dir))))
      ;; Do tilde expansion but not relative path expansion when
      ;; jde-resolve-relative-paths-p is false.
      (if (not 
	   (or
	    (string= p ".")
	    (string-match "[.]/" p)))
	  (setq p (expand-file-name p))))
    (setq p (jde-convert-cygwin-path p))
    p))

(defun jde-expand-classpath (classpath &optional symbol)
  "If `jde-expand-classpath-p' is nonnil, replaces paths to 
directories that match `jde-lib-directory-names' with paths to jar or
zip files in those directories. This function assumes that the
existing paths are already normalized."
  (if jde-expand-classpath-p
      (let (paths)
	(loop for path in classpath do
	      (if (and 
		   (file-exists-p path)
		   (file-directory-p path)
		   (let ((dir-name (file-name-nondirectory path)))
		     (member-if 
		      (lambda (lib-name)
			(string-match lib-name dir-name))
		      jde-lib-directory-names)))
		  (progn
		    (setq paths 
			  (append 
			   paths
			   ;; Need to convert paths to
                           ;; DOS style on Cygwin.
			   (mapcar
			    (lambda (path)
			      (jde-normalize-path path symbol))
			    (directory-files path t "\\.jar$"))))
		    (setq paths
			  (append 
			   paths 
			   (mapcar
			    (lambda (path)
			    (jde-normalize-path path symbol))		    
			   (directory-files path t "\\.zip$")))))
		(setq paths (append paths (list path)))))
	paths)
    classpath))


(defun jde-build-classpath (paths &optional symbol quote-path-p)
  "Builds a classpath from PATHS.  PATHS is a either list of paths or
a symbol whose value is a list of paths, in which case the optional
arg SYMBOL is unnecessary. If QUOTE-PATH-P is nonnil, quote paths
that contain spaces."
  (if (symbolp paths)
      (setq symbol paths
	    paths (symbol-value symbol)))
  (mapconcat
   (lambda (path)
     path)
    (jde-expand-classpath 
     (mapcar
      (lambda (path)
	(jde-normalize-path path symbol))
      paths)
     symbol)
   jde-classpath-separator))

(defun jde-global-classpath ()
  "Builds a classpath string from the path entries in
`jde-global-classpath'."
  (jde-build-classpath 'jde-global-classpath))


(defun jde-build-path-arg (arg path-list &optional quote symbol)
"Build a command-line path argument from a list of paths."
  (let ((path (jde-build-classpath path-list symbol)))
    (if quote
        (setq path (concat "\"" path "\"")))
    (setq path (concat arg " " path))))


(defun jde-build-classpath-arg (path-list &optional quote symbol)
"Build a classpath from a list of paths."
 (jde-build-path-arg "-classpath" path-list quote symbol))

(defun jde-root-dir-p (dir)
  "Return nonnil if DIR is a root directory." 
  (let ((parent (expand-file-name  "../" dir)))
    (cond 
     ((and
       (fboundp 'ange-ftp-ftp-name)
       (ange-ftp-ftp-name dir))
      (ange-ftp-get-file-entry parent))
     ((eq system-type 'windows-nt)
      ;; If the current directory tree is on a 
      ;; virtual drive created by the subst command
      ;;
      ;;  (not (file-exists-p parent)) 
      ;;  
      ;; fails. Hence, the following hack contributed
      ;; by Nat Goodspeed.
      (or 
       (string= parent "//") ; for paths like //host/d/prj/src
       (string= parent "\\\\") ; for paths like \\host\d\prj\src
       (string= (substring parent -3) "../"))) ; for paths like d:/prj/src
     ((member system-type '(cygwin32 cygwin))
      (or (string= (file-truename dir) (file-truename "/"))
	  (string= parent "//") ; for paths like //host/d/prj/src
	  (string= parent "\\\\") ; for paths like \\host\d\prj\src
	  (and (> (length parent) 3) ; for paths like d:/prj/src
	       (string= (substring parent -3) "../"))
	  (not (file-exists-p (file-truename dir)))))
     (t
      (or (or (not (file-readable-p dir))
	      (not (file-readable-p parent)))
	  (and 
	   (string= (file-truename dir) "/")
	   (string= (file-truename parent) "/")))))))

(defun jde-get-global-classpath () 
  "Return the value of `jde-global-classpath', if defined, otherwise
the value of the CLASSPATH environment variable converted to a list."
  (if jde-global-classpath 
      jde-global-classpath
    (let ((cp (getenv "CLASSPATH")))
      (if (stringp cp)
	  (split-string cp jde-classpath-separator)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                            ;;
;; Project Files                                                              ;;
;;                                                                            ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconst jde-project-file-version "1.0"
  "*The current JDE project file version number.")

(defvar jde-loaded-project-file-version nil
  "*Temporary var that holds the project file version of the project
being loaded.")

(defun jde-project-file-version (ver)
  (setq jde-loaded-project-file-version ver))

(defun jde-find-project-file (dir)
  "Finds the next project file upwards in the directory tree
from DIR. Returns nil if it cannot find a project file in DIR
or an ascendant directory."
  (let* ((directory-sep-char ?/) ;; Override NT/XEmacs setting
	 (file (find jde-project-file-name
		     (directory-files dir) :test 'string=)))
    (if file
	(expand-file-name file dir)
      (if (not (jde-root-dir-p dir))
	  (jde-find-project-file (expand-file-name "../" dir))))))

(defun jde-find-project-files (dir)
  "Return all the project files in the current directory tree,
starting with the topmost."
  (let* ((directory-sep-char ?/) ;; Override NT/XEmacs setting
	 (file (jde-find-project-file dir))
	 current-dir files)
    (while file
      (setq files (append (list file) files))
      (setq current-dir (file-name-directory file))
      (setq 
       file
       (if (not (jde-root-dir-p current-dir))
	   (jde-find-project-file
	    (expand-file-name "../" current-dir)))))
    files))
      
(defvar jde-loading-project nil
  "Used by project loading system.")

(defun jde-load-project-file ()
  "Load the project file(s) for the Java source file in the current
buffer. Search for all the project file first in the directory
tree containing the current source buffer. If any files are found,
first reset all variables to their startup values. Then load
the project files starting with the topmost in the tree.
If no project files are found, set the JDE variables to their
Emacs startup values."
  (interactive)
  (let ((prj-files (jde-find-project-files default-directory)))
    (if prj-files
	(progn
	  (jde-set-variables-init-value)
	  (loop for file in prj-files do
	    (setq jde-loading-project file)
	    (jde-log-msg "jde-load-project-file: Loading %s" file)
	    ;; reset project file version
	    (setq jde-loaded-project-file-version nil)
	    (load-file file)
	    (setq jde-loading-project nil))
	  (run-hooks 'jde-project-hooks))
      (jde-set-variables-init-value t))))


(defun jde-load-all-project-files ()
  (interactive)
  "Loads the project file associated with each Java source buffer."
  (mapc
   (lambda (java-buffer)
     (save-excursion
       (set-buffer java-buffer)
       (message "Loading project file for %s ..." 
		(buffer-file-name java-buffer))
       (jde-load-project-file)))
   (jde-get-java-source-buffers)))

;;;###autoload
(defun jde-open-project-file ()
  "Opens the project file for the Java source file in the
current buffer."
  (interactive)
  (let ((prj-file (jde-find-project-file default-directory)))
    (if prj-file
	(find-file prj-file)
      (message "%s" "Project file not found."))))


(defun jde-save-delete (symbol buffer)
  "Delete the call to SYMBOL from project file in BUFFER.
Leave point at the location of the call, or after the last expression."
  (save-excursion
    (set-buffer buffer)
    (goto-char (point-min))
    (catch 'found
      (while t
	(let ((sexp (condition-case nil
			(read (current-buffer))
		      (end-of-file (throw 'found nil)))))
	  (when (and (listp sexp)
		     (eq (car sexp) symbol))
	    (delete-region (save-excursion
			     (backward-sexp)
			     (point))
			   (point))
	    (throw 'found nil)))))
    (unless (bolp)
      (princ "\n"))))

(defun jde-symbol-p (symbol)
  "Returns non-nil if SYMBOL is a JDE variable."
  (and (get symbol 'custom-type)
       (or (string-match "^bsh-" (symbol-name symbol))
	   (string-match "^jde-" (symbol-name symbol)))))

(defvar jde-symbol-list nil
  "*A list of jde variables which are processed by `jde-save-project'.")

(defun jde-symbol-list (&optional force-update)
  "Return a list of variables to be processed by `jde-save-project'.
The first time this is called, the list is saved in `jde-symbol-list'.
If nonnil, FORCE-UPDATE forces regeneration of `jde-symbol-list'. 
This is useful for updating customization variables defined by 
packages loaded after startup of the JDEE."
  (if force-update
      (setq jde-symbol-list nil))
  (unless jde-symbol-list 
    (mapatoms
     (lambda (symbol)
       (if (jde-symbol-p symbol)
	   (setq jde-symbol-list (cons symbol jde-symbol-list))))))
  jde-symbol-list)

(defun jde-set-project-name (name)
  (put 'jde-project-name 'customized-value (list name))
  (setq jde-project-name name))

(defun jde-put-project (symbol project value)
  "Stores a new value for SYMBOL in PROJECT, or overwrites any
existing value."
  (let ((proj-alist (get symbol 'jde-project)))
    (if (null proj-alist)
	(put symbol 'jde-project (list (cons project (list value))))
      (if (assoc project proj-alist)
	  (setcdr (assoc project proj-alist) (list value))
	(put symbol 'jde-project (pushnew (cons project (list value)) proj-alist))))))

(defun jde-get-project (symbol project)
  "Gets the value for SYMBOL that is associated with PROJECT, or nil
if none.  To test if SYMBOL has any value for PROJECT, use
`jde-project-present-p'."
  (car-safe (cdr-safe (assoc project (get symbol 'jde-project)))))

(defun jde-project-present-p (symbol project)
  "Returns non-nil if SYMBOL has a value for PROJECT."
  (assoc project (get symbol 'jde-project)))

(defun jde-save-open-buffer (project)
  "Creates a new buffer or opens an existing buffer for PROJECT."
  (let ((auto-insert nil)	; turn off auto-insert when
	buffer standard-output)	; creating a new file
    (setq buffer (find-file-noselect project))
    (setq standard-output buffer)
    (save-excursion
      (set-buffer buffer)
      (goto-char (point-min))
      (jde-save-delete 'jde-project-file-version buffer)
      (delete-blank-lines)
      (jde-save-delete 'jde-set-variables buffer)
      (delete-blank-lines)
      (jde-save-delete 'jde-set-project-name buffer)
      (delete-blank-lines))
    (princ "(jde-project-file-version ")
    (prin1 jde-project-file-version)
    (princ ")\n")
    (princ "(jde-set-variables")
    (jde-log-msg "jde-save-open-buffer: Opening buffer for %s" project)
    buffer))

(defun jde-save-close-buffer (project)
  "Saves and closes the buffer associated with PROJECT."
  (let* ((buffer 
	  (if jde-xemacsp
	      (get-file-buffer project)
	    (find-buffer-visiting project)))
	 (standard-output buffer))
    (if buffer
      (progn
	(princ ")\n")
	(save-excursion
	  (set-buffer buffer)
	  (save-buffer))
	(jde-log-msg "jde-save-close-buffer: Closing buffer for %s" project)
	(kill-buffer buffer))
      (jde-log-msg "jde-save-close-buffer: Unable to find buffer for %s" project))))

(defun jde-save-variable (symbol projects)
  "Saves all of the values of SYMBOL for each project file mentioned
in PROJECTS."
  (mapc
   (lambda (project)
     (if (and (not (string= (car project) "default"))
	      (member (car project) projects))
	 (let ((buffer 
		(if jde-xemacsp
		    (get-file-buffer (car project))
		  (find-buffer-visiting (car project))))
	       standard-output)
	   (if (null buffer)
	       (setq standard-output (setq buffer (jde-save-open-buffer (car project))))
	     (setq standard-output buffer))
	   (jde-log-msg "jde-save-variable: Saving %S in %s" symbol (car project))
	   (princ "\n '(")
	   (princ symbol)
	   (princ " ")
	   (prin1 (custom-quote (car (cdr project))))
	   (princ ")"))))
   (get symbol 'jde-project)))

(defun jde-save-needs-saving-p (symbol projects)
  "Function used internally by the project saving mechanism to
determine whether or not to save a symbol in a project file.  If there
are settings to be saved, this function also resolves which project
should receive the customized values."
  (unless (= (length projects) 0)
    (let ((value (symbol-value symbol))
	  val-to-save
	  current-proj proj-iter)
      (setq current-proj (car projects))
      (cond
       ;; CASE: current value changed from saved value in current
       ;; project
       ((and (jde-project-present-p symbol current-proj)
	     (not (equal value (jde-get-project symbol current-proj))))
	(jde-log-msg "jde-save-needs-saving-p: changed value for %S in project `%s'"
		     symbol current-proj)
	(jde-put-project symbol current-proj value)
	t)
       ;; CASE: no value for symbol in current project - check all
       ;; parent projects (plus default) to see if value has changed
       ((and (not (jde-project-present-p symbol current-proj))
	     (progn
	       (setq val-to-save value)
	       (setq proj-iter (cdr projects))
	       (while (and proj-iter
			   (not (jde-project-present-p symbol (car proj-iter))))
		 (setq proj-iter (cdr proj-iter)))
	       (if proj-iter
		   (not (equal value
			       (jde-get-project symbol (car proj-iter))))
		 (setq proj-iter (list "default"))
		 (setq val-to-save (eval (car (get symbol 'customized-value))))
		 (and (not (null (get symbol 'customized-value)))
		      (not (equal val-to-save
				  (and (jde-project-present-p symbol (car proj-iter))
				       (jde-get-project symbol (car proj-iter)))))))))
	(jde-log-msg "jde-save-needs-saving-p: override value %S from parent `%s' in project `%s'"
		     symbol (car proj-iter) current-proj)
	(jde-put-project symbol current-proj val-to-save)
	t)
       ;; CASE: current value same as value in the deepest project that
       ;; holds that value - re-save it
       ((progn
	  (setq proj-iter projects)
	  (while (and proj-iter
		      (not (jde-project-present-p symbol (car proj-iter))))
	    (setq proj-iter (cdr proj-iter)))
	  (if proj-iter
	      (equal value (jde-get-project symbol (car proj-iter)))))
	(jde-log-msg "jde-save-needs-saving-p: original value for %S in project `%s'"
		     symbol (car proj-iter))
	t)))))

(defun jde-save-project-internal (projects)
  (let ((projects-reversed (nreverse projects)))
    (jde-log-msg "jde-save-project-internal: projects: %S" projects-reversed)
    (mapc 'jde-save-open-buffer projects-reversed)
    (mapc (lambda (symbol)
	    (if (jde-save-needs-saving-p symbol projects-reversed)
		(jde-save-variable symbol projects-reversed)))
	  (jde-symbol-list))
    (mapc 'jde-save-close-buffer projects-reversed)))

;;;###autoload
(defun jde-save-project ()
  "Saves source file buffer options in one or more project files.
This command provides an easy way to create and update a project file
for a Java project. Simply open a source file, set the desired
options, using the JDE Options menu, then save the settings in the
project file, using this command.  Now, whenever you open a source
file from the same directory tree, the saved settings will be restored
for that file."
  (interactive)
  (let* ((directory-sep-char ?/) ;; Override NT/XEmacs setting
	(project-file-paths (jde-find-project-files default-directory)))
    (if (not project-file-paths)
	(setq project-file-paths
	      (list (expand-file-name jde-project-file-name
				      (read-file-name "Save in directory: "
						      default-directory
						      default-directory)))))
    (jde-save-project-internal project-file-paths)))

;;;###autoload
(defun jde-create-new-project (new-dir)
  "Creates a new JDE project file in directory NEW-DIR, saving any
current customized variables.  If a project file already exists in the
given directory, the project is simply re-saved.  This functions the
same as `jde-save-project' when no project files can be found for the
current source file.  But, if there already exist projects somewhere
along the path, this command unconditionally creates a project file in
the directory specified, thus allowing the user to create and maintain
hierarchical projects."
  (interactive "DCreate new project in directory: ")
  (let* ((directory-sep-char ?/) ;; Override NT/XEmacs setting
	 (prj-file (expand-file-name jde-project-file-name new-dir))
	 (projects (jde-find-project-files new-dir)))
    (if (not (member prj-file projects))
	;; create empty project file if none found
	(let* ((auto-insert nil)	; disable auto-insert
	       (standard-output (find-file-noselect prj-file))	
	       (message-log-max nil))	; disable message log
	  (princ "(jde-project-file-version ")
	  (prin1 jde-project-file-version)
	  (princ ")\n(jde-set-variables)\n")
	  (save-excursion
	    (set-buffer standard-output)
	    (save-buffer))
	  (kill-buffer standard-output)
	  (setq projects (nconc projects (list prj-file)))))
    (jde-save-project-internal projects)))

(defun jde-set-variables (&rest args)
  "Initialize JDE customization variables.  

Takes a variable number of arguments. Each argument 
should be of the form:

  (SYMBOL VALUE)

The value of SYMBOL is set to VALUE.
"
  (while args 
    (let ((entry (car args)))
      (if (listp entry)
	  (let* ((symbol (nth 0 entry))
		 (value (nth 1 entry))
		 (customized (nth 2 entry))
		 (set (or (and (local-variable-if-set-p symbol nil) 'set)
			  (get symbol 'custom-set)
			  'set-default)))
	    (if (or customized
		    jde-loaded-project-file-version)
		(put symbol 'customized-value (list value)))
	    (if jde-loading-project
		(progn
		  (jde-log-msg "jde-set-variables: Loading %S from project %s" symbol
			       jde-loading-project)
		  (jde-put-project symbol
				   jde-loading-project
				   (eval value)))
	      (jde-log-msg "jde-set-variables: Loading %S from unknown project" symbol))
	    (when (default-boundp symbol)
	      ;; Something already set this, overwrite it
	      (funcall set symbol (eval value)))
	    (setq args (cdr args)))))))

(defun jde-set-variables-init-value (&optional msg)
  "Set each JDE variable to the value it has at Emacs startup."
  (interactive)
  (if (or (interactive-p) msg)
      (message "Setting JDE variables to startup values..."))
  (mapcar 
   (lambda (symbol) 
     (let ((val-to-set (eval (car (or (get symbol 'saved-value)
				      (get symbol 'standard-value)))))
	   (set (or (get symbol 'custom-set) 'set-default)))
       (if (or (get symbol 'customized-value)
	       (get symbol 'jde-project))
	   (funcall set symbol val-to-set))
       (put symbol 'customized-value nil)
       (put symbol 'jde-project nil)
       (jde-put-project symbol "default" val-to-set)))
   (jde-symbol-list)))

;; Code to update JDE customization variables when a user switches
;; from a Java source buffer belonging to one project to a buffer
;; belonging to another.

(setq jde-current-project "")

(defun jde-reload-project-file ()
  "If project context-switching is enabled (see
`jde-project-context-switching-enabled-p'), reloads the project file
for a newly activated Java buffer when the new buffer's project
differs from the old buffer's."
  (interactive)
  (let ((project-file-path (jde-find-project-file default-directory)))
    (if (not project-file-path) (setq project-file-path ""))
    (if (and 
	 jde-project-context-switching-enabled-p
	 (not (jde-debugger-running-p))
	 (not (string= 
	       (file-truename jde-current-project) (file-truename project-file-path))))
	(progn
	  (setq jde-current-project project-file-path)
	  (jde-load-project-file)
	  (jde-wiz-set-bsh-project)))))

(defun jde-update-autoloaded-symbols ()
  "Regenerate `jde-symbol-list' and reload
the project files for the current project. Insert
this function at the end of autoloaded JDEE packages
to register and  initialize customization variables 
defined by the current project's project file."
  (jde-symbol-list t)
  (jde-load-project-file))

(defun jde-debugger-running-p () 
  (and 
   (jde-dbs-debugger-running-p)
   (jde-dbs-get-target-process)))

(defcustom jde-entering-java-buffer-hook 
  '(jde-reload-project-file 
    jde-which-method-update-on-entering-buffer)
  "*Lists functions to run when entering a Java source buffer."
  :group 'jde-project
  :type 'hook)

(defvar jde-current-buffer (current-buffer)
  "*Internal JDE variable that holds the current active buffer.")

(defun jde-detect-java-buffer-activation ()
  "Detects when a user activates a buffer.
If the activated buffer is a Java buffer, runs the 
`jde-entering-java-buffer-hook' hooks."
  (let ((curr-buff (current-buffer)))
    (if (not (equal curr-buff jde-current-buffer))
	(progn
	  (setq jde-current-buffer curr-buff)
	  (if (eq major-mode 'jde-mode)
	      (run-hooks 'jde-entering-java-buffer-hook))))))

(defun jde-count-open-java-buffers ()
  "Returns non-nil if any java buffers are open."
  (count 
   ".java"
   (buffer-list)
   :test
   (lambda (file-type buffer)
     (let ((file-name (buffer-file-name buffer)))
       (if file-name
	   (string-match file-type file-name))))))
	 

(defun jde-remove-jde-hook ()
  "Removes `jde-detect-java-buffer-activation-hook' when
all Java source buffers have been closed."
  (unless (> (jde-count-open-java-buffers) 1)
  (remove-hook 'post-command-hook 'jde-detect-java-buffer-activation)))

(add-hook 'kill-buffer-hook 'jde-remove-jde-hook)


;; JDE help
(defun jde-find-jde-data-directory ()
  "Return the path of the JDE data directory.
Returns the path of the directory containing the
JDE java and documentation directories;  nil if the 
directory cannot be found. If XEmacs, returns the location of
the data directory in the XEmacs distribution hierarchy. On all other Emacs versions, 
the JDE expects to find the documentation and Java class directories
in the same directory that contains the JDE lisp directory."
  (let ((directory-sep-char ?/)
	dir)
    (flet ((find-data-dir 
	    ()
	    (expand-file-name 
	     "../" 
	     (file-name-directory (locate-library "jde")))))
      (if jde-xemacsp
	  (progn
	    (setq dir (locate-data-directory "jde"))
	    (if (not dir)
		(setq dir (find-data-dir))))
	(setq dir (find-data-dir))))
      dir))

(defun jde-find-jde-doc-directory ()
  "Return the path of the JDE documentation directory.
Returns  nil if the directory cannot be found. At some
point, XEmacs will include the JDE. Versions of XEmacs
that include JDE will store the JDE doc in a data
directory called jde. On all other Emacs versions, the JDE
expects to find the documentation in a subdirectory 
named doc of the directory that contains the file
jde.el."
  (jde-find-jde-data-directory))

;;;###autoload
(defun jde-show-help ()
  "Displays the JDE User's Guide in a browser."
  (interactive)
  (let* ((jde-dir (jde-find-jde-doc-directory))
         (jde-help
          (if jde-dir
	      (expand-file-name "doc/html/jde-ug/jde-ug.html" jde-dir))))      
    (if (and
         jde-help
         (file-exists-p jde-help))
        (browse-url (concat "file://" (jde-convert-cygwin-path jde-help))
                    (if (boundp 'browse-url-new-window-flag)
			'browse-url-new-window-flag
		      browse-url-new-window-p))
      (signal 'error '("Cannot find JDE help file.")))))

(defun jde-debug ()
"*Runs the debugger specified by `jde-debugger'."
  (interactive)
  (if (string= (car jde-debugger) "JDEbug")
      (jde-bug-debug-app)
    (jde-jdb)))

;;
;; Problem reporting functions contributed by Phillip Lord <plord@hgmp.mrc.ac.uk>.
;;
(defvar jde-problem-report-mail-address "pkinnucan@attbi.com" )

(defun jde-submit-problem-report()
  "Submit a problem report for the JDE" 
  (interactive)
  (require 'reporter)
  (and 
   (y-or-n-p "Do you want to submit a problem report on the JDE? ")
   (progn
     (message "Preparing problem report...")
     ;;prepare the basic buffer
     (reporter-submit-bug-report
      jde-problem-report-mail-address
      (format "JDE version %s\nRequired packages: semantic-%s eieio-%s speedbar-%s\n"
	      jde-version semantic-version eieio-version speedbar-version)
      (jde-problem-report-list-all-variables)
      nil
      'jde-problem-report-post-hooks
      "Please enter the details of your bug report here" )
     (message "Preparing bug report...done"))))


(defun jde-problem-report-post-hooks()
  "Function run the reporter package done its work.
It looks for a JDEBug buffer and inserts the contents of that, and then prompts 
for insertion of the .emacs file"
  (save-excursion 
    (goto-char (point-max))
    (let* ((debug-buffer (get-buffer "*JDEbug*"))
	   (messages-buffer 
	    (get-buffer
	     (if jde-xemacsp " *Message-Log*" "*Messages*")))
	   (backtrace-buffer (get-buffer "*Backtrace*"))
	   (jde-log-buffer (get-buffer "*jde-log*"))
	   (process 
	    (let ((proc (jde-dbs-get-target-process)))
	      (if (not proc)
		  (let ((dead-proc-alist 
			 (oref jde-dbs-the-process-morgue proc-alist)))
		    (if dead-proc-alist
			(setq proc (cdr (car dead-proc-alist))))))
	      proc))
	   (cli-buffer (if (and process (slot-boundp process 'cli-buf))
			   (oref process cli-buf)))
	   (locals-buffer (if (and process (slot-boundp process 'locals-buf))
			      (oref process locals-buf))))
     

      ;;insert the contents of the debug buffer if it is there. 
      (if debug-buffer
	  (progn
	    (insert-string "\n\n\nThe contents of the *JDEBug* buffer were\n\n")
	    (insert-buffer debug-buffer)
	    (goto-char (point-max))
	    (insert-string "\n\n\nEnd Insert *JDEbug* buffer" ))
	(insert-string "\n\n\nThere was no *JDEBug* buffer" ))

      ;;insert the contents of the CLI buffer if it exists.
      (if cli-buffer
	  (progn
	    (insert-string "\n\n\nThe contents of the CLI buffer are\n\n")
	    (insert-buffer cli-buffer)
	    (goto-char (point-max))
	    (insert-string "\n\n\nEnd Insert CLI buffer" ))
	(insert-string "\n\n\nThere is no CLI buffer" ))


      ;;insert the contents of the locals buffer if it exists.
      (if locals-buffer
	  (progn
	    (insert-string "\n\n\nThe contents of the locals buffer are\n\n")
	    (insert-buffer locals-buffer)
	    (goto-char (point-max))
	    (insert-string "\n\n\nEnd Insert locals buffer" ))
	(insert-string "\n\n\nThere is no locals buffer" ))

      ;;insert the contents of the backtrace buffer if it is there. 
      (if backtrace-buffer
	  (progn
	    (insert-string "\n\n\nThe contents of the *Backtrace* buffer were\n\n")
	    (insert-buffer backtrace-buffer)
	    (goto-char (point-max))
	    (insert-string "\n\n\nEnd Insert *Backtrace* buffer" ))
	(insert-string "\n\n\nThere was no *Backtrace* buffer" ))


      ;;insert the contents of the messages buffer if it is there. 
      (if messages-buffer
	  (progn
	    (insert-string "\n\n\nThe contents of the *Messages* buffer were\n\n")
	    (insert-buffer messages-buffer)
	    (goto-char (point-max))
	    (insert-string "\n\n\nEnd Insert *Messages* buffer" ))
	(insert-string "\n\n\nThere was no *Messages* buffer" ))

      ;;insert the contents of the jde-log buffer if it is there. 
      (if jde-log-buffer
	  (progn
	    (insert-string "\n\n\nThe contents of the *jde-log* buffer were\n\n")
	    (insert-buffer jde-log-buffer)
	    (goto-char (point-max))
	    (insert-string "\n\n\nEnd Insert *jde-log* buffer" ))
	(insert-string "\n\n\nThere was no *jde-log* buffer" )))

    (when process-environment
      (insert-string "\n\n\nProcess environment: \n\n")
      (insert-string (mapconcat (lambda (var) var) process-environment "\n")))

    (let ((buf (get-buffer-create "*Insert .emacs*"))
	  (mail-buf (current-buffer)))
      
      (set-buffer buf)
      (widget-insert "It is requested that you send the entire contents of your .emacs file.\n")
      (widget-insert "This is because it has been found that those parts of the .emacs file\n" )
      (widget-insert "which appear not to be JDE related often do in fact contain the cause of\n")
      (widget-insert "reported bugs.\n\n")
      (widget-insert "If you do not want to send the contents of your .emacs or you load a large\n" )
      (widget-insert "large number of files from your full .emacs file, then please attempt to\n" )
      (widget-insert "replicate the bug using the minimal .emacs file suggested in the JDE\n" )
      (widget-insert "documentation, and note that you have done this in this bug report\n" )
      (widget-insert "If you choose to do neither of these things we may not be able to\n" )
      (widget-insert "or necessarily want to help determine the cause of the problem!\n" )
      (switch-to-buffer "*Insert .emacs*")
      
      (set-buffer mail-buf)
      (goto-char (point-max))
      (if (y-or-n-p "Insert your .emacs file into the problem report? " )
	  (progn
	    (insert-string "\n\n\nThe contents of the .emacs file was\n\n\n")
	    (insert-file "~/.emacs")
	    (goto-char (point-max))
	    (insert-string "\n\n\n=====end inserted .emacs file"))
	(insert-string "\n\n\nThe user choose not to insert their .emacs file\n" ))
      ;;clean up the prompt buffer
      (kill-buffer buf))))

(defun jde-problem-report-list-all-variables()
  "List all variables starting with `jde' or `bsh'."
  (let ((vars))
    (mapatoms
     (lambda (symbol)
       (when 
	   (and (or 
		 (string-match "bsh-" (symbol-name symbol))
		 (string-match "jde-" (symbol-name symbol)))
		(get symbol 'custom-type))
	 (setq vars (cons symbol vars)))))
    vars))


;; Line numbering support.
(eval-when (compile)
  (require 'setnu))

(defvar jde-setnu-deletion-check t "deletion check")
(make-variable-buffer-local 'jde-setnu-deletion-check)

(defun jde-setnu-after-change (start end length)
 "When in setnu-mode, toggles setnu-mode off and on."
   (if setnu-mode
       (if (or
	    (and
	     (> length 0)
	     jde-setnu-deletion-check)
	    (string-match 
		  "[\n\r]" 
		  (buffer-substring-no-properties start end)))
	   (run-with-timer 
	    0.001 nil
	    ;; setnu toggler      
	   (lambda () (setnu-mode) (setnu-mode))))
     (setq jde-setnu-deletion-check nil)))

(defun jde-setnu-before-change (start end) 
  "Determines whether any newlines were deleted."
   (if setnu-mode
       (if (> end start) 
	   (setq jde-setnu-deletion-check 
		 (string-match "[\n\r]" (buffer-substring-no-properties start end))))))


(defcustom jde-setnu-mode-threshold 20000
 "Maximum number of bytes in a file (buffer) that can result in
automatic line numbering."
 :group 'jde-project
 :type 'integer)

(defcustom jde-setnu-mode-enable nil
 "Enable numbering of lines in Java source buffers."
 :group 'jde-project
 :type 'boolean
 :set '(lambda (sym val)
	 (if val
	     (progn
	       (require 'setnu)
	       (add-hook 
		'after-change-functions 
		'jde-setnu-after-change)
	       (add-hook 
		'before-change-functions 
		'jde-setnu-before-change)
	       (mapc
		(lambda (buf)
		  (save-excursion
		    (set-buffer buf)
		    (if (and
			 (not setnu-mode)
			 (< (point-max) jde-setnu-mode-threshold))
			(setnu-mode 1))))
		  (jde-get-java-source-buffers)))
	   (progn
	     (mapc 
	      (lambda (buf)
		(save-excursion
		  (set-buffer buf)
		  (if (and (boundp 'setnu-mode)
			   setnu-mode)
		      (setnu-mode))))
	      (jde-get-java-source-buffers))))	 
	 (set-default sym val)))

;; jde-describe-map is Ehud Karni's describe map with jde prepended.
(defun jde-keymap-test (var)           ; internal function for keymap checking
       (and (boundp var)
            (keymapp (symbol-value var))))

(defun jde-describe-map (map)          ; display map binding
 "Display binding for MAP which must be a quoted keymap variable"
  (interactive
       (let ((map (intern (completing-read "Key map: " obarray 'jde-keymap-test 1))))
           (list map)))
       (let ((val (symbol-value map)))
           (or (keymapp val)
               (error "%s is not a keymap !" (symbol-name map)))
           (with-output-to-temp-buffer "*Help*"
               (princ (format "Binding for keymap %s is:\n" (symbol-name map)))
               (princ (substitute-command-keys "\\{val}" ))
               (print-help-return-message))))

(defun jde-keys ()
  "Displays JDE key bindings. Use `jde-bug-keys' to display JDEbug keybindings ."
  (interactive)
  (jde-describe-map 'jde-mode-map))


;; Contributed by John Ciolfi, jciolfi@mathworks.com.
(defun jde-compile-file-if-necessary (file)
  "Compile the JDE file FILE if necessary.
This is done if FILE.el is newer than FILE.elc or if FILE.elc doesn't exist."
  (if (string= (file-name-extension file) "el")
      (let* ((root (file-name-sans-extension file))
	     (elc-file (concat root ".elc")))
	(if (or (not (file-exists-p elc-file))
		(file-newer-than-file-p file  elc-file))
	    (progn
	      (message (format "Byte-compiling %s..." 
			       (file-name-nondirectory file)))
	      (byte-compile-file file))))))

;;;###autoload
(defun jde-compile-jde ()
  "Byte-compile all uncompiled files of jde."

  ;; Be sure to have . in load-path since a number of files in jde
  ;; depend on other files and we always want the newer one even if
  ;; a previous version of jde exists.

  (interactive)
  (require 'jde-compat)
  (let ((load-path (append '(".") load-path))
	(jde-lisp-directory (expand-file-name "lisp" (jde-find-jde-data-directory))))
    (save-excursion 
      (mapcar 
       (function jde-compile-file-if-necessary)
       (directory-files jde-lisp-directory t)))))

;; Provided for XEmacs compatibility.
(if (not (fboundp 'subst-char-in-string))
    (defun subst-char-in-string (fromchar tochar string &optional inplace)
      "Replace FROMCHAR with TOCHAR in STRING each time it occurs.
Unless optional argument INPLACE is non-nil, return a new string."
      (let ((i (length string))
	    (newstr (if inplace string (copy-sequence string))))
	(while (> i 0)
	  (setq i (1- i))
	  (if (eq (aref newstr i) fromchar)
	      (aset newstr i tochar)))
	newstr)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                            ;; 
;; Find command                                                               ;;
;;                                                                            ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcustom jde-find-case-sensitive nil
  "*Specifies whether the jde-find command performs a case-sensitive search.
If non-nil, the search is case-sensitive; otherwise, the search ignores case."
  :group 'jde-project
  :type 'boolean
)

;; (makunbound 'jde-find-granularity)
(defcustom jde-find-granularity '("Character")
  "*Specifies the granularity of the expression search
conducted by `jde-find': Character (expression starting
on any character), Word (match words only), Line
(match lines only)."
  :group 'jde-project
  :type  '(list
	   (radio-button-choice
	    :format "%t \n%v"
	    :tag "Search granularity:"
	    (item "Character")
	    (item "Word")
	    (item "Line"))))


(defcustom jde-find-file-regexp '("*.java")
  "*Specifies the regular expression that the jde-find command uses
to select files to be searched. You can use any regular expression  
supported by the -name option of the GNU find command."
  :group 'jde-project
  :type '(repeat (string :tag "Find regexp"))
)

(defclass jde-find-dialog (efc-dialog)
  ((expr-field     :initarg :expr-field
		   :documentation "Edit field for expression to find.")
   (expression     :initarg :expression
		   :type string
		   :initform ""
		   :documentation "Regular expression to find.")
   (dir-fields     :initarg :dir-fields
		   :documentation "List of directory path fields.")
   (dirs           :initarg :dirs
		   :type list
		   :documentation "Directories to search recursively.")
   (file-fields    :initarg :file-fields
		   :documentation "Regular expression fields for files to search.")
   (files          :initarg :files
		   :type list
		   :initform ("*.java")
		   :documentation "Files to search.")
   (ignore-case-cb :initarg :ignore-case
		   :documentation "Ignore case check box.")
   (ignore-case-p  :initarg :ignore-case-p
		   :type boolean
		   :initform t
		   :documentation "If true, ignore case when searching.")
   (grain-rbs      :initarg :grain-rbs
		   :documentation "Granularity radio buttons.")
   (grain          :initarg :grain
		   :type string
		   :initform "Character"
		   :documentation "Search granularity: Character, Word, Line")
   (ok             :initarg :ok
		   :type boolean
		   :initform nil
		   :documentation "True if user clicked the OK button.")
   (the-dialog     :type (or null jde-find-dialog)
		   :allocation :class
		   :initform nil
		   :documentation
		   "The only instance of the find expression dialog buffer."))
  "Dialog displayed by `jde-find' command.")

(defmethod initialize-instance ((this jde-find-dialog) &rest fields)
  "Find options dialog constructor."
  (oset this title "Find Dialog")
  (call-next-method))

(defmethod efc-dialog-create ((this jde-find-dialog))
 
  (widget-insert "Find Expression Options\n\n")

  (oset this expr-field
 	(widget-create
	 (list
	  'text
	  :tab-order 1
	  :format "%t %v"
	  :tag "Expression:"
	  :value (oref this expression))))

  (widget-insert "\n")

  (oset this dir-fields
	(widget-create
	 (list
	  'repeat 
	  :tag "Directories to search recursively"
	  :value (if (slot-boundp this 'dirs)
		     (oref this dirs)
		   (mapcar
		    (lambda (p)
		      (jde-normalize-path p 'jde-sourcepath))
		    jde-sourcepath))
	  (list 'file :tag "Path"))))
  (widget-insert "\n")

  (oset this file-fields
	(widget-create
	 (list
	  'repeat 
	  :tag "File types to search"
	  :value (oref this files)
	  (list 'file :tag "File regexp"))))

  (widget-insert "\n")

  (oset this ignore-case-cb
	(widget-create
	 (list 'checkbox 
	       :format "%[%v%] %t" 
	       :tag "Ignore case"
	       :value (oref this ignore-case-p)
	       )))

  (widget-insert "\n\n")

  (oset this grain-rbs
 	(widget-create
 	 (list
	  'radio-button-choice
	  :format "%t\n%v"
	  :tag "Search granularity:"
	  :value (oref this grain)
	  :args (list
		 (list 'item "Character")
		 (list 'item "Word")
		 (list 'item "Line")))))
		       
  (widget-insert "\n"))


(defmethod efc-dialog-show ((this jde-find-dialog))
  "Shows the options dialog buffer. After showing the dialog buffer,
this method invokes recursive-edit to emulate the behavior of a modal
dialog. This suspends the current command until the user has selected
an option or canceled the dialog. See `efc-dialog-ok' and
`efc-dialog-cancel' for more information."
  (call-next-method)
  (recursive-edit))


(defmethod efc-dialog-ok ((this jde-find-dialog))
  "Invoked when the user selects the OK button on the options
dialog. Sets the :dirs field of THIS to the search paths chosen by the
user, kills the dialog buffer, and exits recursive-edit mode."
  
  (oset this 
	expression
	(widget-value (oref this expr-field)))

  (oset this 
	dirs
	(widget-value (oref this dir-fields)))

  (oset this
	files
	(widget-value (oref this file-fields)))

  (oset this 
	ignore-case-p
	(widget-value (oref this ignore-case-cb)))

  (oset this
	grain
	(widget-value (oref this grain-rbs)))

  (oset this ok t)

  (delete-window)
  (kill-buffer (oref this buf))
  (pop-to-buffer (oref this initbuf))
  (set-buffer (oref this initbuf))

  (exit-recursive-edit))

(defmethod efc-dialog-cancel ((this jde-find-dialog))
  "Invoked when the user clicks the dialog's Cancel button.  Invokes
the default cancel method, sets the :selection field of THIS to nil,
and then exits recursive edit mode."
  (call-next-method)
  (oset this ok nil)
  (exit-recursive-edit))



(defvar jde-find-root-history nil
  "History of directory trees searched in this session.")

(defvar jde-find-regexp-history nil
  "History of search expressions used in this session.")

(defun jde-find-grep-internal (regexp files &optional dirs no-case grain)
  "Find a regular expression REGEXP in files of type FILES in
 DIRS, where DIRS is a string of space-separated paths of
directories to search recursively. If NO-CASE is nonnil, ignore
case. GRAIN is a string that indicates the granularity of the search,
i.e., match any \"Character\" string, a \"Word\" only, or a \"Line\"
only."
  (if (not (executable-find 
	    ;; Hack required by faulty XEmacs implementation of executable-find.
	    (if (eq system-type 'windows-nt) "grep.exe" "grep")))
      (error "This command requires the Unix grep utility."))
  (if (not (executable-find 
	    (if (eq system-type 'windows-nt) "find.exe" "find")))
      (error (list "This command requires the Unix find utility.")))
  (let* ((directories-option
	  (if dirs dirs "."))
	 (case-sensitive-option 
	  (if no-case  "-i" ""))
	 (granularity-option
	  (cond
	   ((and grain (string= grain "Word"))
	    "-w")
	   ((and grain (string= grain "Line"))
	    "-x")
	   (t
	    " ")))	    
	  (file-regexp-option 
	   (mapconcat 
	    (lambda (x) 
	      (format "-name \"%s\"" x)) 
	    files
	    " -or "))
	  (cmd 
	   (format "find %s %s -type f | xargs grep %s %s -n \"%s\" /dev/null" 
		  directories-option
		  file-regexp-option
		  case-sensitive-option
		  granularity-option
		  regexp)))
    (grep cmd)))


(defun jde-find (&optional regexp)
  "Find a regular expression REGEXP in all of the files in the
current JDE project. Tests each of the following path variables
`jde-sourcepath', `jde-compile-option-sourcepath',
`jde-compile-option-classpath', or `jde-global-classpath' and uses the
directories specified by the first path variable that has a nonnil
value. The `jde-find-case-sensitive' variable controls case
sensitivity, `jde-find-granularity' determines the granularity of the
search (character, word, line), and `jde-find-file-regexp' determines
the type of files to be searched. Use `jde-find-dlg' if you want to
set case sensitivity, granularity, or file types interactively. This
command requires that the Unix grep and find utilities be installed on
your system in the Emacs command path. The Cygwin package contains
Windows versions of both utilities."
  (interactive)
  (let ((regexp 
	 (if (and (boundp 'regexp) regexp) 
	     regexp
	   (read-from-minibuffer 
	    "Search for regexp: "
	    (if (boundp 'jde-find-regexp-history) 
		(car jde-find-regexp-history)
	      nil)
	    nil nil 'jde-find-regexp-history)))
	(search-path 
	 (read-from-minibuffer 
	   "Search directories: "
	   (cons 
	    (mapconcat 
	     (lambda (x) x)
	     (cond
	      (jde-sourcepath
	       (mapcar 
		(lambda (path)
		  (jde-normalize-path path 'jde-sourcepath))
		jde-sourcepath))
	      (jde-compile-option-sourcepath 
	       (mapcar 
		(lambda (path)
		  (jde-normalize-path path 'jde-compile-option-sourcepath))
		jde-compile-option-sourcepath))
	      (jde-compile-option-classpath 
	       (mapcar 
		(lambda (path)
		  (jde-normalize-path path 'jde-compile-option-classpath))
		jde-compile-option-classpath))
	      (jde-global-classpath 
	       (mapcar 
		(lambda (path)
		  (jde-normalize-path path 'jde-global-classpath))
		jde-global-classpath))
	      (t
	       (list default-directory)))	
	     " ")
	    0)
	   nil nil 'jde-find-root-history)))
    (jde-find-grep-internal
     regexp
     jde-find-file-regexp
     search-path
     (not jde-find-case-sensitive)
     (car jde-find-granularity))))

(defun jde-find-dlg ()
  "Displays a dialog buffer that allows you to set all search options
interactively. Pressing the dialog's OK button initiates the
search. Use `jde-find' if you need to set only the expression to be
found and the directories to be searched and prefer using the
minibuffer."
  (interactive)
  (let ((dialog 
	 (progn
	   (if (not (oref 'jde-find-dialog the-dialog))
	       (oset-default 'jde-find-dialog the-dialog (jde-find-dialog "find dialog")))
	   (oref 'jde-find-dialog the-dialog))))
    (efc-dialog-show dialog)
    (when (oref dialog ok)
      (jde-find-grep-internal
       (oref dialog expression)
       (oref dialog files)
       (mapconcat
	'jde-normalize-path
	(oref dialog dirs)
	" ")
       (oref dialog ignore-case-p)
       (oref dialog grain)))))

(defun jde-create-prj-values-str ()
  "Create Java expression that updates the JDEE's class list
to include all the classes on `jde-global-classpath' or it
that is not defined the classpath specified by the CLASSPATH
environment variable."
  (let* ((directory-sep-char ?/)  ;; Override NT/XEmacs setting
	 (classpath 
	  (if jde-global-classpath
	      (jde-build-path-arg nil jde-global-classpath t)
	    (if (getenv "CLASSPATH")
		(concat " \"" (getenv "CLASSPATH") "\"")
	      "\"\""))))		    
    (format "jde.util.JdeUtilities.setProjectValues(\"%s\", %s);"
	    jde-current-project 
	    classpath)))

(defun jde-jeval (java-statement &optional eval-return)
  (if (not (bsh-running-p))
      (bsh-eval (jde-create-prj-values-str)))
  (bsh-eval java-statement eval-return))

(defun jde-jeval-r (java-statement)
  (jde-jeval java-statement t))


(defun jde-show-speedbar ()
  "Show the speedbar after first checking whether the correct
version of speedar is installed."
  (interactive)
  (require 'speedbar)
  (if (and
       jde-check-version-flag
       (or 
	(not (boundp 'speedbar-version))
	(not speedbar-version)
	(not (jde-check-version speedbar-version
				jde-speedbar-min-version
				jde-speedbar-max-version))))
      (error 
       (format 
	"The JDEE does not support the version of Speedbar installed on your system." 
	jde-speedbar-min-version)))
  (speedbar-frame-mode))
    
(defun jde-autoload-update ()
  "Updates autoload definitions in jde-autoload.el."
  (interactive)
  (let* ((default-directory (concat (jde-find-jde-data-directory) "lisp/"))
	 (generated-autoload-file (concat default-directory "jde-autoload.el")))
    (mapc 'update-file-autoloads (directory-files "." nil "\\.el$"))))

(provide 'jde)

;; Change History 

;;
;; $Log: jde.el,v $
;; Revision 1.286  2002/12/30 05:25:24  paulk
;; Minor tweaks for JDEE 2.3.2
;;
;; Revision 1.285  2002/12/26 06:43:49  paulk
;; - Removed the obsolete function jde-set-compiler.
;; - Fixed jde-set-global-classpath to set jde-global-classpath correctly.
;;
;; Revision 1.284  2002/12/21 08:58:36  paulk
;; Fix docstring for jde-global-classpath.
;;
;; Revision 1.283  2002/12/19 06:35:59  paulk
;; Changed to permit autoloading of jde-package.el file.
;;
;; Revision 1.282  2002/12/06 03:47:33  ahyatt
;; Changes to support Mac OS X, which does not use tools.jar
;;
;; Revision 1.281  2002/12/03 04:49:21  paulk
;; Fixes infinite recursion bug when a project file specifies
;; jde-ant-build as the jde-build-function.
;;
;; Revision 1.280  2002/11/30 04:48:10  paulk
;; Bumped revision number to 2.3.0.
;;
;; Revision 1.279  2002/11/27 06:06:09  paulk
;; Replaced the alias jde-javadoc-generate-javadoc-template with
;; its real name: jde-javadoc-autodoc-at-line.
;;
;; Revision 1.278  2002/11/26 06:11:37  paulk
;; Fixed regression bug introduced in JDEE 2.2.9.2 that caused symbols to
;; be listed more than once in the jde-symbol-list variable under some
;; circumstances. This in turn caused a symbol to be set multiple times
;; in a prj.el file.
;;
;; Revision 1.277  2002/11/25 04:56:34  paulk
;; Updated jde-create-prj-values-str to use the CLASSPATH environment variable if
;; jde-global-classpath is nil.
;;
;; Revision 1.276  2002/11/23 07:58:39  ahyatt
;; Display helpful error message when no correct mapping is found in the
;; jde-jdk-registry
;;
;; Revision 1.275  2002/11/22 07:44:44  paulk
;; Correct the docstring for jde-get-jdk-dir.
;;
;; Revision 1.274  2002/11/21 04:05:38  paulk
;; The following changes provide the infrastructure to ensure registration
;; and initialization of customization variables defined by autoloaded
;; JDEE packages:
;;   - Provide an argument to jde-symbol-list to force regeneration of the list.
;;   - Define jde-update-autoloaded-symbols function.
;;
;; Revision 1.273  2002/11/14 06:07:56  paulk
;; Bumped JDEE version to 2.2.9.1 to accommodate some minor bug fixes.
;;
;; Revision 1.272  2002/11/06 04:52:45  paulk
;; Fixed bug that I introduced into jde-get-jdk-prog.
;;
;; Revision 1.271  2002/11/05 07:27:55  paulk
;; Define jde-get-jdk-prog function. Thanks to Andrew Hyatt.
;;
;; Revision 1.270  2002/10/25 05:02:42  paulk
;; Use file-truename in jde-reload-project-file to eliminate symbolic links. Thanks to
;; "mark a. foltz" <mfoltz@ai.mit.edu>.
;;
;; Revision 1.269  2002/10/22 05:04:16  paulk
;; Remove require for jde-make package (now autoloaded) and add a menu item for displaying the jde-make customization buffer to the JDEE menu.
;;
;; Revision 1.268  2002/10/21 04:49:34  paulk
;; - Remove require for ant package. This package is now autloaded.
;; - Defer version checking for speedbar until it is used.
;;
;; Revision 1.267  2002/10/16 05:04:24  paulk
;; Fixes regression where the JDEE was not honoring jde-project-context-switching-enabled-p when loading Java files. Thanks to Andy Piper.
;;
;; Revision 1.266  2002/10/11 06:17:56  paulk
;; Use add-to-list instead of setq to add the jde to auto-mode-alist. This avoids multiple entries. Thanks to Klaus Berndl for suggesting this change.
;;
;; Revision 1.265  2002/10/11 05:53:19  paulk
;; Added more packages to the list of packages that are demand loaded. This is intended to reduce the startup time for the JDEE.
;;
;; Revision 1.264  2002/09/30 05:21:07  paulk
;; Removed require forms for the following packages: jde-ejb, jde-javadoc,
;; and jde-checkstyle. These packages are now loaded only if the user
;; access one of the commands that they define.
;;
;; Revision 1.263  2002/09/26 06:17:34  paulk
;; Added templates to the JDE New menu for creating EJB session and entity beans.
;; The templates were created by David T. Smith.
;;
;; Revision 1.262  2002/09/16 05:05:56  paulk
;; Cygwin Emacs compatibility fix. Check for Cygwin Emacs when processing paths. Thanks
;; to Klaus Berndl.
;;
;; Revision 1.261  2002/09/11 03:40:05  paulk
;; * Adds jde-devel-debug variable. This variable is intended to enable
;;   features that facilitate debugging the JDEE itself.
;;
;; * Bumps JDEE's version number 2.2.9beta12
;;
;; Revision 1.260  2002/09/06 12:58:05  paulk
;; Adds a Check Style command to the JDE menu. This command requires that
;; checkstyle-all.jar be in the JDEE's java/lib directory.
;;
;; Revision 1.259  2002/09/02 05:16:55  paulk
;; Fixed a bug in jde-normalize-path whereby it was expanding a relative path
;; even when jde-resolve-relative-paths-p was false.
;;
;; Revision 1.258  2002/08/30 20:58:26  jslopez
;; Adds key bindings for jde-gen-try-finally-wrapper, jde-gen-try-catch-wrapper,
;; jde-wiz-extend-abstract-class, and jde-wiz-override-method.
;;
;; Revision 1.257  2002/08/27 05:28:02  paulk
;; Fixed cygwin support bug in jde-root-dir-p. Thanks to Andy Piper.
;;
;; Revision 1.256  2002/08/07 06:36:14  paulk
;; Removed code intended to eliminate spurious warnings when byte-compiling the JDEE. The
;; removed code now resides in a separate package, jde-compat.el. Thanks to Andy Piper
;; for suggesting this restructuring. Also fixed a number of compiler warnings caused
;; by genuine bugs.
;;
;; Revision 1.255  2002/06/21 05:15:19  paulk
;; Conditioned the :set-after option out of jde-jdk for versions of Emacs
;; earlier than Emacs 21. This is so users will not need Emacs 21 to run
;; the JDEE. Thanks to YAMANO Yuji.
;;
;; Revision 1.254  2002/06/21 04:52:27  paulk
;; XEmacs compatibility fix: set directory sep character to be a forward slash
;; for project files. This fixes a project file save bug that corrupted
;; project files under some circumstances. Thanks to Hanson Char <HChar@realm.com>
;; for diagnosing this bug.
;;
;; Revision 1.253  2002/06/17 07:41:37  paulk
;; Updated version number.
;;
;; Revision 1.252  2002/06/17 07:24:07  paulk
;; Updated the JDEE's applet debugging command to
;; work with its new jdb interface.
;;
;; Revision 1.251  2002/06/12 06:58:42  paulk
;; XEmacs compatibility fix: removed unsupported :set-after clause from
;; the definition of jde-jdk. The result may be that JDK registry settings
;; may not persist in some cases on XEmacs. I will restore this clause
;; as soon as XEmacs supports it.
;;
;; Revision 1.250  2002/06/04 20:13:26  mnl
;; JDE menus are now defined by custom-variables. This allows especially
;; project specific menus that are automatically switched when changing
;; projects (as is usual with jde-... custom variables).
;;
;; Revision 1.249  2002/05/31 19:02:27  mnl
;; Added new template to generate a new interface from scratch.
;;
;; Revision 1.248  2002/05/29 04:47:30  paulk
;; jde-cygwin-path-converter-internal now substitutes forward slashes for
;; backslashes in paths returned by mswindows-cygwin-to-win32-path.
;;
;; Revision 1.247  2002/05/26 12:44:16  jslopez
;; Rebinds C-C C-v C-y from jde-show-class-source to
;; jde-open-class-at-point.
;;
;; Revision 1.246  2002/05/23 05:28:47  paulk
;; Patch jde-root-dir-p to support the Cygwin version of Xemacs 21.4.6, which
;; accepts native Windows path names. Thanks to Michael Lipp.
;;
;; Revision 1.245  2002/05/20 05:03:31  paulk
;; Used defcustom's :set-after option to specify that jde-jdk should be
;; set after jde-jdk-registry. This is necessary to prevent jde-jdk-registry
;; from resetting jde-jdk to the first registered jdk rather than the JDK
;; selected by the user.
;;
;; Revision 1.244  2002/05/11 23:02:40  jslopez
;; Removes control characters.
;;
;; Revision 1.243  2002/04/18 04:21:07  paulk
;; Rebound completion key bindings.
;;
;; Revision 1.242  2002/04/16 03:17:05  jslopez
;; Integrates jde-open-source.
;;
;; Revision 1.241  2002/04/15 02:59:28  jslopez
;; Updates jde-complete-select for jde-complete.
;;
;; Revision 1.240  2002/04/02 06:44:54  paulk
;; Rebinds C-v C-c C-. to jde-complete-select.
;;
;; Revision 1.239  2002/03/31 07:49:47  paulk
;; Renamed jde-db-source-directories. The new name is jde-sourcepath.
;;
;; Revision 1.238  2002/03/24 05:34:55  paulk
;; Patched jde-save-close-buffer and jde-save-variable to use find-buffer-visiting
;; to find the buffer containing the prj.elf file instead of looping through
;; the buffer list.
;;
;; Revision 1.237  2002/03/22 05:28:59  paulk
;; Updated doc for jde-build-function variable.
;; Updated my email address.
;;
;; Revision 1.236  2002/03/21 12:34:53  paulk
;; Fixes the jde-build-function variable to default to jde-make.
;;
;; Revision 1.235  2002/03/21 12:30:51  paulk
;; Removed jde-java-build-function. It never worked reliably.
;;
;; Revision 1.234  2002/03/12 04:40:44  paulk
;; Corrected minimum eieio version number.
;; Removed initarg for class slots to silence eieio warning.
;;
;; Revision 1.233  2002/03/12 04:14:12  paulk
;; Updated min and max version numbers for required packages.
;;
;; Revision 1.232  2002/03/05 10:30:32  paulk
;; Replaces jde-lib-directory-name with jde-lib-directory-names. Allows
;; you to specify multiple jar/zip directories in a classpath.
;;
;; Revision 1.231  2002/02/25 20:12:39  jslopez
;; Updates to reflect changes made to beanshell.el
;;
;; Revision 1.230  2002/02/21 05:32:04  paulk
;; Adds the following enhancements to the JDEE's expression
;; search functionality.
;;
;;   - A new customization variable, jde-find-granularity,
;;     that allows you to specify the granularity of the
;;     search (character, word, or line).
;;
;;   - A new command, JDE->Search->Expression... (jde-find-dlg),
;;     that allows you to set all search options interactively
;;     in a popup dialog buffer.
;;
;; Revision 1.229  2002/02/17 13:42:20  paulk
;; jde-submit-problem-report command now includes versions of required packages.
;;
;; Revision 1.228  2002/02/15 02:48:49  jslopez
;; Updates jde-jeval to pass accept 3 arguments.
;;
;; Revision 1.227  2002/01/22 05:38:28  paulk
;; - Bumped jde-version to 2.2.0beta9.
;; - Updated jde-semantic-min/max-version and eieio-min/max-version
;; - Modified jde-build function to use call-interactively to call
;;   the build function.
;;
;; Revision 1.226  2002/01/19 06:42:22  paulk
;; Minor updates.
;;
;; Revision 1.225  2002/01/18 12:49:42  paulk
;; Added a new hook, jde-project-hooks. This variable lists functions to be run
;; when the JDEE switches to a new project.
;;
;; Revision 1.224  2002/01/15 05:53:58  paulk
;; Now requires comint.
;;
;; Revision 1.223  2001/12/19 08:00:43  paulk
;; Update jde-semantic-max-version to allow 1.4beta13.
;;
;; Revision 1.222  2001/12/04 05:39:22  paulk
;; Fixed JDEbug menu related bug.
;;
;; Revision 1.221  2001/11/30 10:46:08  jslopez
;; Fixes the definition of jde-emacs21p.
;; Changes the name from jde-emacs21-p to jde-emacs21p.
;;
;; Revision 1.220  2001/11/30 04:22:13  jslopez
;; Fixes another compilation warning...
;;
;; Revision 1.219  2001/11/30 04:14:34  jslopez
;; Fixes regression compilation warning in Emacs20.
;;
;; Revision 1.218  2001/11/30 03:53:37  jslopez
;; Adds constant jde-emacs21-p.
;; Fixes compilations warnings in emacs 21.
;;
;; Revision 1.217  2001/11/28 08:25:29  paulk
;; Updated jde-cygwin-path-converter-cygpath to minimize calls to cygpath command. This function now invokes cygpath only if the cygwin path does not begin with /cygdrive and has a root that has not previously been converted.
;;
;; Revision 1.216  2001/11/25 20:56:45  paulk
;; Added functions jde-java-major-version and jde-java-minor-version.
;;
;; Revision 1.215  2001/11/24 15:00:48  paulk
;; Expanded doc strings for jde-jdk-registry and jde-jdk to clarify their usage.
;;
;; Revision 1.214  2001/11/13 07:31:26  paulk
;; Fixed jde-java-version to return the version of Java on the system command path
;; when the user does not specify the version and the Beanshell is not running.
;; This eliminates the infinite recursion caused by starting the Beanshell to
;; find the version, which is needed to start the Beanshell.
;;
;; Revision 1.213  2001/11/12 12:09:30  jslopez
;; Adds a new option `jde-check-version-flag'
;; to customize the behavior of the new
;; `jde-check-versions' feature.
;;
;; When `jde-check-version-flag' is non-nil
;; `jde-check-versions' will check for correct
;; versions of the CEDET libraries required
;; by the JDE.
;; If nil it only checks if these libraries
;; are available.
;;
;; Fixes some doc strings to follow
;; Emacs conventions and removed the set up
;; of the `jde-which-method-idle-timer'
;; in `jde-mode' because it has been
;; moved in jde-java-grammar.el with the
;; other stuff which need Semantic.
;;
;; Submitted by David Ponce
;;
;; Revision 1.212  2001/11/11 23:00:26  jslopez
;; Fixing typo.
;;
;; Revision 1.211  2001/11/11 18:10:44  jslopez
;; Adds version checking to jde for speedbar, semantic, and eieio.
;; If the wrong version of one of the packages is installed, it
;; shows an error to the user and stops loading jde.
;;
;; Revision 1.210  2001/11/11 07:18:17  paulk
;; Moves all the `jde-mode' code depending on Semantic
;; into `jde-parse-semantic-default-setup' which is now run as a
;; `semantic-init-hooks' ensuring that at this point the buffer is ready
;; for parsing.
;;
;; This solves a nasty problem of synchronization between `jde-mode' and
;; Semantic initialization.  This problem was particularly annoying in
;; XEmacs where `jde-mode' buffers were not parsed the first time.  In
;; some rare conditions the problem occurred with Emacs too.
;;
;; Thanks to David Ponce.
;;
;; Revision 1.209  2001/11/08 06:14:23  paulk
;; Updated to support semantic 1.4 beta 12. Thanks to David Ponce.
;;
;; Revision 1.208  2001/11/04 04:43:53  paulk
;; Fixed bug in jde-java-version. Thanks to SEBASTIEN VAUCLAIR <sebastien.vauclair@epfl.ch>.
;;
;; Revision 1.207  2001/10/31 08:58:43  paulk
;; Emacs 21 compatibility fix: added support for browse-url-new-window-flag variable (replaces browse-url-new-window-p in Emacs 21).
;;
;; Revision 1.206  2001/10/30 06:44:48  paulk
;; Added the Event Source wizard to the JDE menu.
;;
;; Revision 1.205  2001/10/27 22:30:38  paulk
;; Eliminated jde-mode-internal as its only purpose was to facilitate debugging jde-mode. Moved the invocation of the hook functions to be the last form in jde-mode. Thanks to David Ponce for noticing this latter bug.
;;
;; Revision 1.204  2001/10/22 02:45:35  paulk
;; Bumped version number to 2.2.9beta5.
;;
;; Revision 1.203  2001/10/21 14:49:54  jslopez
;; Added require jde-ant.
;; Added jde-show-complete-options
;; Added jde-show-wiz-options
;; Added jde-show-ant-options
;; Added wiz, complete, and ant to the menu under JDE -> Project -> Options
;;
;; Revision 1.202  2001/10/21 14:19:11  jslopez
;; Added require jde-ant; now the use of ant does not require any
;; extra customization.
;;
;; Revision 1.201  2001/10/19 09:53:21  paulk
;; Cygwin/XEmacs compatibility fix: Now uses
;; mswindows-cygwin-to-win32-path function if defined.
;; Thanks to Andy Piper.
;;
;; Revision 1.200  2001/10/16 05:06:44  paulk
;; Fixed bug in jde-java-version command. Thanks to Jim Davidson.
;;
;; Revision 1.199  2001/10/15 03:37:11  paulk
;; - XEmacs compatibility fix. Fixed bug that caused the JDE
;;   to append "(jde-save-variable " to every line of a prj.el
;;   file. This bug was due to a conditional expression that
;;   depended on the file sep char being a forward slash when
;;   by default it is a backslash on XEmacs.
;;
;; - XEmacs cygwin compatibility fix. Fixed bug in the
;;   code that builds classpaths that garbled the resulting
;;   classpath.
;;
;; - Fixed all compiler warnings in both Emacs and XEmacs.
;;
;; Revision 1.198  2001/10/04 11:18:27  paulk
;; Fixes bug in jde-get-jdk-dir that caused it to return
;; an incorrect path when looking for the directory on
;; the PATH environment variable.
;;
;; Revision 1.197  2001/10/01 12:08:38  paulk
;; Fixes bugs in jde-java-version function.
;;
;; Revision 1.196  2001/09/30 05:31:47  paulk
;; Changed name of customization variable jde-db-debugger to jde-debugger to be consistent with jde-compiler.
;;
;; Revision 1.195  2001/09/28 04:38:18  paulk
;; Added the following variables: jde-jdk-registry and jde-java-environment-variables.
;;
;; Revision 1.194  2001/09/27 14:57:44  jslopez
;; Updated the menu to reflect new method name, jde-help-browse-jdk-doc.
;;
;; Revision 1.193  2001/09/16 17:47:33  paulk
;; - Replaced jde-jdk-directory with jde-jdk. The new variable specifies both the version
;; and the location of the JDK currently being used by the JDE.
;;
;; - Semantic setup code moved to jde-java-grammar.el. Thanks to David Ponce.
;;
;; - In `jde-mode-internal', `jde-current-project' set to "" instead of nil
;; when there is no project file.  This avoid setting twice the startup
;; values because `jde-reload-project-file' assumes that
;; `jde-current-project' is "" when not found. Thanks to David Ponce.
;;
;; Revision 1.192  2001/09/05 06:09:50  paulk
;; Added the function/command jde-get-jdk-dir. This command gets the root
;; directory of the JDK currently being used by the JDE. The JDK used
;; by the JDE is either the directory specified by jde-jdk-directory
;; or by the path environment variable.
;;
;; Revision 1.191  2001/09/05 04:44:30  paulk
;; * Add a new command jde-java-version that gets the version of Java being used by the JDE. Note that this command uses the BeanShell.
;;
;; Revision 1.190  2001/08/30 04:12:49  paulk
;; - Moved jde-browse-jdk-doc to the jde-help package.
;; - Added jde-jdk-directory customization variable.
;;
;; Revision 1.189  2001/08/30 01:29:56  paulk
;; Adds support for compile server. Thanks to Javier Lopez.
;;
;; Revision 1.188  2001/08/29 12:27:39  paulk
;; Fixed compile buffer fontification problem caused by  copying global post-command-hook into local post-command-hook in jde-mode-internal. Thanks to David Ponce.
;;
;; Revision 1.187  2001/08/24 06:01:04  paulk
;; Made jde-compile-finish-flush-completion-cache more robust.
;;
;; Revision 1.186  2001/08/14 05:45:42  paulk
;; Updated version number.
;;
;; Revision 1.185  2001/08/08 06:00:00  paulk
;; * Added some more API help commands to the JDE menu.
;; * XEmacs compatibility fix for error that occurs when the JDE tries to erase the completion cache after compiling a file.
;;
;; Revision 1.184  2001/08/04 05:34:18  paulk
;; Reorganized JDE menu somewhat.
;;
;; Revision 1.183  2001/08/04 03:46:25  paulk
;; Adds JDE->Wizards->Extend Abstract Class menu item.
;;
;; Revision 1.182  2001/07/31 05:07:40  paulk
;; Adds JDE->Wizards->Generate Get/Set Methods. Thanks to Javier Lopez and Sandip Chitale.
;;
;; Revision 1.181  2001/07/29 07:01:38  paulk
;; Fixes JDE->Help->JDE User Guide function so that it works on XEmacs.
;;
;; Revision 1.180  2001/07/18 02:41:27  paulk
;; NT/XEmacs compatibility fix: jde-find-project-file now returns paths with forward slashes as directory sep characters on NT/XEmacs.
;;
;; Revision 1.179  2001/07/17 05:42:15  paulk
;; Bump version number.
;;
;; Revision 1.178  2001/07/16 13:36:30  paulk
;; Adds jde-enable-senator variable.
;;
;; Revision 1.177  2001/07/16 13:23:48  paulk
;; Fixed bug in jde-save-needs-saving-p that caused the JDE not to save variables whose customized value is nil.
;;
;; Revision 1.176  2001/07/08 04:37:28  paulk
;; Added compatibility fixes for NT/XEmacs use of backslash as the
;; default directory sep character. Thanks to William Griswold <wgg@cs.ucsd.edu>
;;
;; Revision 1.175  2001/07/07 04:47:22  paulk
;; Adds the customization variable, jde-launch-beanshell-on-demand.
;;
;; Revision 1.174  2001/07/06 02:13:20  paulk
;; Provide support for user-supplied completion templates.
;;
;; Revision 1.173  2001/06/27 04:41:32  paulk
;; Adds the customization variable, jde-build-function, which allows selection of one a standard set of functions (jde-java-build, jde-ant-build or jde-make) or specification of a custom function.
;;
;; Removes the customization variable, jde-build-use-make.
;;
;; Thanks to Jason Stell jason.stell@globalone.net.
;;
;; Revision 1.172  2001/06/07 04:08:12  paulk
;; Fixed bug in jde-jeval.
;;
;; Revision 1.171  2001/05/31 05:14:39  paulk
;; Provide support for per-project caching of class data in the Beanshell. Thanks to Matt Conway.
;;
;; Revision 1.170  2001/05/23 03:33:56  paulk
;; Added require for semantic-load. Thanks to David Ponce.
;;
;; Revision 1.169  2001/05/22 02:54:14  paulk
;; Adds jde-find command. Thanks to "Guillaume Berche" <guillaume.berche@eloquant.com>.
;;
;; Revision 1.168  2001/05/19 02:36:01  paulk
;; Updated to support semantic 1.4. Thanks to David Ponce.
;;
;; Revision 1.167  2001/04/27 04:52:10  paulk
;; Fixes bug in jde-normalize-path's handling of paths that begin with ../. Thanks to "Nick Sieger" <nsieger@bitstream.net> for this fix.
;;
;; Revision 1.166  2001/04/26 08:46:27  paulk
;; jde-lib-directory-name can now be a regular expression that matches the name of directories that contain zip or jar files.
;;
;; Revision 1.165  2001/04/25 03:23:26  paulk
;; Fix Project->Auto Switch menu to work on XEmacs.
;;
;; Revision 1.164  2001/04/24 08:13:48  paulk
;; Adds a new menu item, JDE->Project->Auto Switch, that toggles project context switching on or off.
;;
;; Revision 1.163  2001/04/24 08:01:21  paulk
;; Define new function jde-toggle-project-switching.
;;
;; Revision 1.162  2001/04/20 20:52:13  paulk
;; Bug fix: JDE no longer overwrites local version of after-change-functions. Thanks to David Ponce.
;;
;; Revision 1.161  2001/04/19 04:40:57  paulk
;; Updated version number.
;;
;; Revision 1.160  2001/04/17 04:18:03  paulk
;; XEmacs compatibility fix: pass nil third argument to local-variable-if-set-p. Thanks to Nick Sieger.
;;
;; Revision 1.159  2001/04/16 05:41:27  paulk
;; - jde-build-classpath now optionally takes a symbol representing a string of paths.
;; - Normalized JDK doc path.
;;   Thanks to Nick Sieger for both enhancements.
;;
;; Revision 1.158  2001/04/11 03:13:55  paulk
;; This is a major update to the JDE project file system to support hierarchical
;; project files. The enhanced  code remembers which project files set
;; options and will save them back to the appropriate project
;; file when the project is saved.
;;
;; When the values of variables are newly customized, the new value is saved in
;; the deepest project file found relative to the buffer where
;; `jde-save-project' is invoked.  Thus, the buffer where you save the project
;; controls in whic project file newly customized values are saved.
;;
;; The implementation boils down to associating a new property `jde-project'
;; with each symbol that has been saved to a project file.   The property's
;; content is an alist of cons cells that look like ("<project file path>"
;; <saved value>).
;;
;; A  side-effect of the changes is where do relative paths resolve to now
;; that path-related options can be set by multiple project files?  The project
;; system solves that problem by resolving relative paths relative to the project
;; file that defines the option.
;;
;; Thanks to "Nick Sieger" <nsieger@bitstream.net> for contributing this
;; enhancement.
;;
;; Revision 1.157  2001/04/11 02:33:42  paulk
;; The JDE now replaces occurrences of lib directories in  classpaths
;; with the jar and zip files that those directories
;; contain. This means that you no longer have to include
;; jar and zip files explicitly in classpaths. See
;; jde-expand-classpath-p and jde-lib-directory-name for details.
;;
;; Revision 1.156  2001/03/30 18:04:28  paulk
;; Fix jde-save-variables to save overrides of variables customized in the user's .emacs file. Thanks to Nick Sieger <nsieger@bitstream.net> for providing this fix.
;;
;; Revision 1.155  2001/03/30 08:46:29  paulk
;; Implements hierarchical project files. The JDE saves only variables that have been customized for the current session in project files. The JDE now loads project files in the following manner. It first sets all the JDE variables to their Emacs startup values (i.e., the default value or the value saved in your .emacs file). It then loads all the project files in the directory tree containing the current source buffer, starting with the topmost file. Thanks to Nick Sieger for contributing to this enhancement.
;;
;; Revision 1.154  2001/03/29 06:19:15  paulk
;; - jde-save-project now prompts you to enter the directory in which to save a new project file.
;;
;; - jde-save-project-in prompts you to enter the directory in which to save a new or existing project file.
;;
;; Revision 1.153  2001/03/29 04:16:16  paulk
;; Adds support for relative paths. The JDE now converts relative paths to absolute paths by appending the relative path to the path of the project file for the current source buffer.
;;
;; Revision 1.152  2001/03/29 02:44:32  paulk
;; Replaced jde-find-exec with executable-find, which is defined by executable.el available with both the Emacs and XEmacs distributions.
;;
;; Revision 1.151  2001/03/27 17:47:45  paulk
;; Eliminate dependency on which package by including the function jde-find-exec and replacing references to the which command with jde-find-exec. Thanks to klaus.berndl@sdm.de for suggesting this change and providing the implementation of jde-find-exec.
;;
;; Revision 1.150  2001/03/23 09:04:06  paulk
;; Now update the method in the mode line during idle times instead of after every keystroke. Thanks to Steven Monnier for suggesting this enhancement.
;;
;; Revision 1.149  2001/02/27 12:53:09  paulk
;; Now initializes jde-mode-abbrev-table to ensure compatability with cc-mode 5.28. Thanks to David Ponce for this fix.
;;
;; Revision 1.148  2001/02/25 04:14:55  paulk
;; - Update docstring for jde-global-classpath.
;; - Removed jde-path-to-string function because split-string does the same job.
;;
;; Revision 1.147  2001/02/21 05:53:13  paulk
;; Added require for new jde-package package.
;;
;; Revision 1.146  2001/02/20 05:09:04  paulk
;; The JDE now expands paths that begin with a tilde but not a period (.).
;;
;; Revision 1.145  2001/02/17 07:27:01  paulk
;; - Fixed regression bug in context-switching that resulted from setting
;;   jde-current-project to the path of the current prj.el file.
;;
;; - No longer expand classpaths to avoid expanding relative path (.) notation.
;;
;; Revision 1.144  2001/02/03 08:14:26  paulk
;; Changed declarations for all path variables from string to file. This means that you can now use completion (M tab) to complete paths in customization buffers.
;;
;; Revision 1.143  2001/02/03 07:32:29  paulk
;; Made quote argument optional in jde-build-path-arg and jde-build-classpath-arg.
;;
;; Revision 1.142  2001/01/16 04:25:30  paulk
;; Adds jde-abbrev-mode and jde-show-abbrev commands. Thanks to s.nicolas@videotron.ca.
;; .
;; .
;; .
;; Old entries deleted to save space.
;;
;; Revision 1.8  1997/06/18 17:20:00  paulk
;; Initial checkin.
;;

;; End of jde.el
