/*	$Id: kyopon.c,v 1.14 2008/01/17 14:13:01 itohy Exp $	*/

/*
 * Copyright (c) 2004 Zoroyoshi.
 * Copyright (c) 2004 raktajino.
 * Copyright (c) 2004, 2005 Fujidana.
 * Copyright (c) 2005-2006, 2008 ITOH Yasufumi.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <inttypes.h>	/* int*_t, uint*_t */
#include <errno.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/uio.h>
#ifndef NO_UKYOPON
#include <dev/usb/ukyopon.h>
#endif
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <libgen.h>	/* basename() */

#include "utils.h"

#define YES	1
#define NO	0
struct kyopon {
	int fd;
	struct termios oterm;
	const char *devname;
	int loggedin;
};

#define KYOPON_READ_TIMEOUT	80	/* x 0.1s, 255 (25.5s) max */

/*----------------------------------------------------------------------
 * low-level interface
 */

void kyopon_init __P((struct kyopon *, const char *));
int kyopon_open __P((struct kyopon *));
void kyopon_close __P((struct kyopon *));
int kyopon_send_command_str __P((struct kyopon *, int /*cmd*/, int /*subcmd*/,
	const char *));
int kyopon_send_command __P((struct kyopon *, int /*cmd*/, int /*subcmd*/,
	const void *, size_t));
int kyopon_receive_data __P((struct kyopon *, uint8_t *, size_t, int));
int kyopon_login_with_passwd __P((struct kyopon *, const char *));
void kyopon_logout __P((struct kyopon *));

void
kyopon_init(k, devpath)
	struct kyopon *k;
	const char *devpath;
{

	k->fd = -1;
	k->loggedin = NO;
	k->devname = devpath;
}

/*
 * open AIR-EDGE PHONE
 */
int
kyopon_open(k)
	struct kyopon *k;
{
	struct termios mode;
#ifndef NO_UKYOPON
	struct ukyopon_identify id;
#endif

	/*
	 * open data transfer port (/dev/ttyU*)
	 */
	if ((k->fd = open(k->devname, O_RDWR | O_NONBLOCK)) == -1) {
		perror(k->devname);
		return -1;
	}

#ifndef NO_UKYOPON
	/*
	 * make sure the device is the data port of Kyocera PHS handset
	 */
	if (ioctl(k->fd, UKYOPON_IDENTIFY, &id) != 0 ||
	    strcmp(id.ui_name, UKYOPON_NAME) != 0 ||
	    id.ui_porttype != UKYOPON_PORT_DATA) {
		fprintf(stderr, "kyopon_open: %s: not a ukyopon data port\n",
		    k->devname);
		goto error1;
	}
#endif

	if (ioctl(k->fd, TIOCEXCL) == -1 ||	/* exclusive use */
	    fcntl(k->fd, F_SETFL, 0) == -1 ||	/* clear non-blocking mode */
	    tcgetattr(k->fd, &mode) == -1)	/* get current tty mode */
		goto error;

	/* change tty to raw mode */
	k->oterm = mode;
	cfmakeraw(&mode);
	mode.c_cflag |= CLOCAL;	/* status lines are not the same as MODEMs */
	mode.c_cc[VMIN] = 0;
	mode.c_cc[VTIME] = KYOPON_READ_TIMEOUT;
	if (tcsetattr(k->fd, TCSANOW, &mode) == -1)
		goto error;

	k->loggedin = NO;
	return 0;

error:	perror(k->devname);
error1:	(void)close(k->fd);
	k->fd = -1;

	return -1;
}

/*
 * close AIR-EDGE PHONE
 */
void
kyopon_close(k)
	struct kyopon *k;
{

	if (k->fd != -1) {
		kyopon_logout(k);
#if 0	/* must not */
		tcdrain(k->fd);
		tcsetattr(k->fd, TCSAFLUSH, &k->oterm);
#endif
		close(k->fd);
		k->fd = -1;
	}
}

/*
 * send a command to AIR-EDGE PHONE
 */
int
kyopon_send_command_str(k, cmd, subcmd, str)
	struct kyopon *k;
	int cmd, subcmd;
	const char *str;
{

	return kyopon_send_command(k, cmd, subcmd, str, strlen(str));
}

int
kyopon_send_command(k, cmd, subcmd, param, len)
	struct kyopon *k;
	int cmd, subcmd;
	const void *param;
	size_t len;
{
	uint8_t hdr[12];
	int32_t lenwithhdr;
	ssize_t nbyte;
	struct iovec iov[2];

	lenwithhdr = len + 4;
	hdr[0] = 0xe1;
	hdr[1] = 0x01;
	hdr[2] = cmd >> 8;
	hdr[3] = cmd;
	hdr[4] = lenwithhdr;
	hdr[5] = lenwithhdr >> 8;
	hdr[6] = lenwithhdr >> 16;
	hdr[7] = lenwithhdr >> 24;
	hdr[8] = 0;
	hdr[9] = subcmd >> 8;
	hdr[10] = 0;
	hdr[11] = subcmd;

	if (param == NULL)
		len = 0;	/* header only */

	iov[0].iov_base = hdr;
	iov[0].iov_len = sizeof hdr;
	iov[1].iov_base = (void *)param;
	iov[1].iov_len = len;
	if ((size_t)(nbyte = writev(k->fd, iov, param == NULL ? 1 : 2))
	    != sizeof hdr + len) {
		if (nbyte < 0)
			perror("writev");
		else
			fprintf(stderr, "short write %ld < %lu\n",
			    (long)nbyte, (unsigned long)(sizeof hdr + len));
		return -1;
	}

	return 0;
}

/*
 * receive data from AIR-EDGE PHONE
 */
int
kyopon_receive_data(k, buf, bufsz, has_additional_data)
	struct kyopon *k;
	uint8_t *buf;
	size_t bufsz;
	int has_additional_data;
{
	size_t buflen, size;
	ssize_t nbyte;
	size_t n;
	uint32_t len;

	assert(bufsz > 13);
	buf[12] = 0xff;
	buf[13] = 0x00;

	buflen = 0;
	while (buflen < 12) {
		if ((nbyte = read(k->fd, buf + buflen, 12 - buflen)) == -1) {
			perror("read");
			return -1;
		}
		if (nbyte == 0) {
			fprintf(stderr, "kyopon_receive_data: unexpected EOF\n");
			return -1;
		}
		buflen += nbyte;

		/* search for header mark */
		for (n = 0; n + 1 < buflen; n++) {
			if (buf[n] == 0xe1 && buf[n + 1] == 0x02) {
				/* move header mark to the top of buf */
				if (n > 0) {
					buflen -= n;
					memmove(buf, buf + n, buflen);
				}
				break;
			}
		}
	}

	/* 12byte header has been read */

	len = (buf[4] | buf[5] << 8 | buf[6] << 16 | buf[7] << 24) + 8;

	if (has_additional_data) {
		if (len < 12) {
			fprintf(stderr,
			    "kyopon_receive_data: format error: len %u < 12\n",
			    len);
			return -1;
		}

		size = (len > bufsz) ? bufsz : len;
		while (buflen < size) {
			if ((nbyte = read(k->fd, buf + buflen, size - buflen))
			    == -1) {
				perror("read");
				return -1;
			}
			if (nbyte == 0) {
				fprintf(stderr, "kyopon_receive_data: unexpected EOF\n");
				return -1;
			}
			buflen += nbyte;
		}
	}

	return len;
}

/*
 * log into AIR-EDGE PHONE
 */
int
kyopon_login_with_passwd(k, passwd)
	struct kyopon *k;
	const char *passwd;
{
#define KYOPON_BUFSZ	4096
	uint8_t buf[KYOPON_BUFSZ];

	if (kyopon_send_command_str(k, 0, 0x0101, "1") == -1)
		return -1;
	if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
		return -1;

	if (buf[12] != 'O' || buf[13] != 'K') {
		fprintf(stderr,
		    "kyopon_login_with_passwd: not in main screen\n");
		return -1;
	}
	k->loggedin = YES;

	if (kyopon_send_command_str(k, 1, 0x0201, passwd) == -1)
		return -1;
	if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
		return -1;

	if (buf[12] != 'O' || buf[13] != 'K') {
		fprintf(stderr, "login failed\n");
		return -1;
	}

	return 0;
}

/*
 * log outof AIR-EDGE PHONE
 */
void
kyopon_logout(k)
	struct kyopon *k;
{

	if (k->loggedin) {
		/*
		 * XXX?
		 * logging out too quick may cause logout failure
		 */
		usleep(400000);

		kyopon_send_command_str(k, 0xff, 0x0102, "");
		k->loggedin = NO;
	}
}

#ifdef NEED_FGETLN
char *
fgetln(fp, len)
	FILE *fp;
	size_t *len;
{
	static char buf[8192];

	if (fgets(buf, sizeof buf, fp) == NULL)
		return NULL;

	/* XXX should check whole line is read */

	*len = strlen(buf);
	return buf;
}
#endif

/*----------------------------------------------------------------------
 * binary data handlings
 */

int kyopon_get_fields __P((uint8_t *, size_t, uint8_t *[], size_t, int));

int
kyopon_get_fields(bin, blen, fields, nfields, bookmark_bug_workaround)
	uint8_t *bin;		/* must have blen+1 bytes for nul termination */
	uint8_t *fields[];	/* fields[0]: #1, fields[1]: #2, ... */
	size_t blen, nfields;
	int bookmark_bug_workaround;
{
	uint8_t *p = bin;
	unsigned idx;
	uint32_t len;

	/* init */
	for (idx = 0; idx < nfields; idx++)
		fields[idx] = NULL;

	/* header: 0x1c */
	if (blen <= 0 || *p++ != 0x1c) {
		fprintf(stderr, "kyopon_get_fields: unknown format\n");
		return -1;
	}

	/* field: 0x1e field-idx ?? length(4bytes) data... */
	while (p + 7 <= bin + blen && p[0] == 0x1e) {
		*p = '\0';	/* terminate previous field */

		idx = p[1];
		len = p[3] | p[4] << 8 | p[5] << 16 | p[6] << 24;
		if (len == 0x01000000)
			len = p[2];
		p += 7;
		if (idx - 1 /* unsigned underflow OK */ >= nfields) {
			fprintf(stderr,
			    "kyopon_get_fields: index %u out of range\n", idx);
		} else if (p + len > bin + blen) {
			if (bookmark_bug_workaround)
				return 0;
			fprintf(stderr, "kyopon_get_fields: index %u overrun\n",
			    idx);
			return -1;
		} else {
			fields[idx - 1] = p;
		}

		p += len;
	}

	*p = '\0';	/* terminate last field */
	if (bin + blen - p)
		fprintf(stderr, "kyopon_get_fields: %lu bytes unused\n",
		    (unsigned long)(bin + blen - p));

	return 0;
}

/*----------------------------------------------------------------------
 * address data handlings
 */

struct address {
	size_t	size;
	uint8_t	*data;
};

static void address_dtor __P((void *));
int address_file_read __P((struct vararray *, const char *));
int address_file_write __P((struct vararray *, const char *));

#define COMPARE_FULL(str)	\
	lend - ln == sizeof(str) - 1 && strncmp(ln, str, sizeof(str) - 1) == 0
#define COMPARE_SUB(str)	\
	(p = ln + sizeof(str) - 1) <= lend && strncmp(ln, str, sizeof(str) - 1) == 0

static void
address_dtor(p)
	void *p;
{
	struct address *a = p;

	free(a->data);
}

int
address_file_read(addrlist, fn)
	struct vararray *addrlist;	/* must be initialized */
	const char *fn;
{
	FILE *fp;
	char *ln, *lend;
	size_t len;
	int ioerr, ret, in_vcard, lno;
	struct address entry;
	struct varbuf buf;

	if ((fp = fopen(fn, "r")) == NULL) {
		perror(fn);
		return -1;
	}

	varbuf_init(&buf);
	ret = 0;
	lno = 0;
	in_vcard = 0;
	while ((ln = fgetln(fp, &len)) != NULL) {
		lno++;
		lend = ln + len;
		while (ln < lend && (lend[-1] == '\n' || lend[-1] == '\r'))
			--lend;
		if (ln == lend || ln[0] == '#')
			continue;

		if (COMPARE_FULL("BEGIN:VCARD")) {
			if (in_vcard) {
				fprintf(stderr, "%s:%d: missing END:VCARD\n",
				    fn, lno);
				varbuf_reinit(&buf);
			}
			in_vcard = 1;
			varbuf_add(&buf, ln, lend - ln);
			varbuf_add(&buf, "\r\n", 2);
		} else if (COMPARE_FULL("END:VCARD")) {
			if (!in_vcard) {
				fprintf(stderr, "%s:%d: missing BEGIN:VCARD\n",
				    fn, lno);
				varbuf_reinit(&buf);
			} else {
				varbuf_add(&buf, ln, lend - ln);
				varbuf_add(&buf, "\r\n", 2);
				entry.data = varbuf_getbuf(&buf, &entry.size);
				vararray_add(addrlist, &entry);
				varbuf_init(&buf);
			}
			in_vcard = 0;
		} else {
			if (!in_vcard) {
				fprintf(stderr, "%s:%d: missing BEGIN:VCARD\n",
				    fn, lno);
			} else {
				varbuf_add(&buf, ln, lend - ln);
				varbuf_add(&buf, "\r\n", 2);
			}
		}
	}
	if (in_vcard) {
		fprintf(stderr, "%s:%d: missing END:VCARD\n", fn, lno);
	}
	varbuf_reinit(&buf);

	ioerr = 0;
	if (ferror(fp)) {
		perror(fn);
		ioerr = 1;
	}
	if (fclose(fp)) {
		if (ioerr == 0) {
			perror(fn);
			ioerr = 1;
		}
	}
	return ioerr ? -1 : ret;
}

int
address_file_write(addrlist, fn)
	struct vararray *addrlist;	/* must be initialized */
	const char *fn;
{
	FILE *fp;
	size_t n;
	int ioerr;
	struct address *entry;
	const uint8_t *p, *pend;

	if ((fp = fopen(fn, "w")) == NULL) {
		perror(fn);
		return -1;
	}

	fprintf(fp, "# kyopon address book\n");
	for (n = 0; n < addrlist->nitem; n++) {
		entry = vararray_item(addrlist, n);
		putc('\n', fp);
		for (p = entry->data, pend = p + entry->size; p < pend; p++)
			if (*p != '\r')
				putc(*p, fp);
	}

	ioerr = 0;
	if (ferror(fp)) {
		perror(fn);
		ioerr = 1;
	}
	if (fclose(fp)) {
		if (ioerr == 0) {
			perror(fn);
			ioerr = 1;
		}
	}
	return ioerr ? -1 : 0;
}

/*----------------------------------------------------------------------
 * bookmark data handlings
 */

struct bookmark {
	uint8_t	*name, *uri, *mno, *fno;
};

static void bookmark_dtor __P((void *));
static int bookmark_compare __P((const void *, const void *));
int bookmark_file_read __P((struct vararray *, const char *));
int bookmark_file_write __P((struct vararray *, const char *));
static int kyopon_bookmark_get __P((uint8_t *, size_t, struct bookmark *));
static size_t kypon_bookmark_put __P((uint8_t *, struct bookmark *));

static void
bookmark_dtor(p)
	void *p;
{
	struct bookmark *bm = p;

	freestash(bm->name);
	freestash(bm->uri);
	freestash(bm->mno);
	freestash(bm->fno);
}

/* for sorting */
static int
bookmark_compare(arg1, arg2)
	const void *arg1, *arg2;
{
	const struct bookmark *bm1 = arg1, *bm2 = arg2;
	int result;

	if ((result = atoi(bm1->fno) - atoi(bm2->fno)) != 0)
		return result;
	return atoi(bm1->mno) - atoi(bm2->mno);
}

/*
 * bookmark file format:
 *	BEGIN:BOOKMARK
 *	NAME:<bookmark name>
 *	LOCATION:<address>
 *	NO:<memory number>
 *	FOLDER:<folder number>
 *	END:BOOKMARK
 */

int
bookmark_file_read(bmlist, fn)
	struct vararray *bmlist;	/* must be initialized */
	const char *fn;
{
	FILE *fp;
	char *ln, *lend, *p;
	size_t len;
	int ioerr, ret, in_bm, lno;
	struct bookmark bm;

	if ((fp = fopen(fn, "r")) == NULL) {
		perror(fn);
		return -1;
	}

	ret = 0;
	lno = 0;
	in_bm = 0;
	while ((ln = fgetln(fp, &len)) != NULL) {
		lno++;
		lend = ln + len;
		while (ln < lend && (lend[-1] == '\n' || lend[-1] == '\r'))
			--lend;
		if (ln == lend || ln[0] == '#')
			continue;

		if (COMPARE_FULL("BEGIN:BOOKMARK")) {
			if (in_bm)
				fprintf(stderr, "%s:%d: missing END:BOOKMARK\n",
				    fn, lno);
			in_bm = 1;
			bm.name = bm.uri = bm.mno = bm.fno = NULL;
		} else if (COMPARE_SUB("NAME:")) {
			if ((bm.name = stash2(p, lend - p)) == NULL) {
				perror("malloc");
				ret = -1;
			}
		} else if (COMPARE_SUB("LOCATION:")) {
			if ((bm.uri = stash2(p, lend - p)) == NULL) {
				perror("malloc");
				ret = -1;
			}
		} else if (COMPARE_SUB("NO:")) {
			if ((bm.mno = stash2(p, lend - p)) == NULL) {
				perror("malloc");
				ret = -1;
			}
		} else if (COMPARE_SUB("FOLDER:")) {
			if ((bm.fno = stash2(p, lend - p)) == NULL) {
				perror("malloc");
				ret = -1;
			}
		} else if (COMPARE_FULL("END:BOOKMARK")) {
			if (!in_bm) {
				fprintf(stderr,
				    "%s:%d: missing BEGIN:BOOKMARK\n",
				    fn, lno);
			} else {
				vararray_add(bmlist, &bm);
			}
			in_bm = 0;
		} else {
			fprintf(stderr, "%s:%d: syntax error\n", fn, lno);
		}

	}

	ioerr = 0;
	if (ferror(fp)) {
		perror(fn);
		ioerr = 1;
	}
	if (fclose(fp)) {
		if (ioerr == 0) {
			perror(fn);
			ioerr = 1;
		}
	}
	return ioerr ? -1 : ret;
}

int
bookmark_file_write(bmlist, fn)
	struct vararray *bmlist;	/* must be initialized */
	const char *fn;
{
	FILE *fp;
	size_t n;
	int ioerr;
	struct bookmark *bm;

	if ((fp = fopen(fn, "w")) == NULL) {
		perror(fn);
		return -1;
	}

	fprintf(fp, "# kyopon bookmarks\n");
	for (n = 0; n < bmlist->nitem; n++) {
		bm = vararray_item(bmlist, n);
		fprintf(fp, "\
\n\
BEGIN:BOOKMARK\n\
NAME:%s\n\
LOCATION:%s\n\
NO:%s\n\
FOLDER:%s\n\
END:BOOKMARK\n",
		    bm->name, bm->uri, bm->mno, bm->fno);
	}

	ioerr = 0;
	if (ferror(fp)) {
		perror(fn);
		ioerr = 1;
	}
	if (fclose(fp)) {
		if (ioerr == 0) {
			perror(fn);
			ioerr = 1;
		}
	}
	return ioerr ? -1 : 0;
}

static int
kyopon_bookmark_get(bin, blen, bm)
	uint8_t *bin;		/* input */
	size_t blen;		/* input length */
	struct bookmark *bm;	/* output */
{
	uint8_t *fields[4];

	if (kyopon_get_fields(bin, blen, fields, 4, YES) == -1)
		return -1;

	bm->name = stash(fields[0]);
	bm->uri = stash(fields[1]);
	bm->mno = stash(fields[2]);
	bm->fno = stash(fields[3]);

	return 0;
}

static size_t
kypon_bookmark_put(bin, bm)
	uint8_t *bin;		/* output */
	struct bookmark *bm;	/* input */
{
	uint8_t *p = bin;
	uint32_t len;

	*p++ = 0x1c;

#define PUT_BMK(member, idx, x) \
	*p++ = 0x1e; *p++ = idx; *p++ = x; \
	len = strlen(bm->member); \
	*p++ = len; *p++ = len >> 8; *p++ = len >> 16; *p++ = len >> 24; \
	memcpy(p, bm->member, len); p += len;

	PUT_BMK(name, 1, 2)
	PUT_BMK(uri, 2, 1)
	PUT_BMK(mno, 3, 1)
	PUT_BMK(fno, 4, 2)

#undef PUT_BMK

	return p - bin;
}

#define KYOPON_LEN_TIMESTAMP	16
enum kyopon_file_type {
	KYOPON_FILETYPE_IMAGE, KYOPON_FILETYPE_MUSIC,
	KYOPON_FILETYPE_NET, KYOPON_FILETYPE_CARD,
	KYOPON_FILETYPE_OTHER,
	KYOPON_FILETYPE_INVALID = -1
};
const char *const filetypes[] = { "image", "music", "net", "card", "other" };
const char *const filesfxs[] = {
			/* note: substring must appear earlier (.jpe->.jpeg) */
	/* image */	".jpe.jpg.jpeg.bmp.gif.png",
	/* music */	".mid.midi.dxm.adp",
	/* net */	".htm.html.shtml.css.xht.xhtm.xhtml.wml.js.xml",
	/* card */	".vcf"
};

struct kdirentry {
	uint8_t *name;
	uint8_t *ftype;
	uint32_t size;
	uint8_t timestamp[KYOPON_LEN_TIMESTAMP + 1];
};

static void kdirentry_dtor __P((void *));
static int kyopon_kdirentry_get __P((uint8_t *, size_t, struct kdirentry *));

static void
kdirentry_dtor(p)
	void *p;
{
	struct kdirentry *d = p;

	free(d->name);
	free(d->ftype);
}

static int
kyopon_kdirentry_get(bin, blen, d)
	uint8_t *bin;		/* input */
	size_t blen;		/* input length */
	struct kdirentry *d;	/* output */
{
	const u_int8_t *p, *q;
	const u_int8_t *bend = bin + blen;

	/*			    tab
	 *	name		     |	ftype	size	timestamp
	 *	050318_2014~001.jpg	image	17978	2005/03/18/20/14
	 */
	for (q = bin; q < bend; q++)
		if (*q == '\t')
			break;
	d->name = stash2(bin, q - bin);

	for (p = ++q; q < bend; q++)
		if (*q == '\t')
			break;
	if ((d->ftype = malloc(q - p + 1)) == NULL) {
		perror("malloc");
		return -1;
	}
	strncpy(d->ftype, p, q - p);
	d->ftype[q - p] = '\0';

	for (p = ++q; q < bend; q++)
		if (*q == '\t')
			break;
	d->size = atoi(p);

	p = q + 1;
	if (bend - p != KYOPON_LEN_TIMESTAMP) {
		fprintf(stderr, "kyopon_kdirentry_get: unexpected format\n");
		return -1;
	}
	memcpy(d->timestamp, p, KYOPON_LEN_TIMESTAMP);
	d->timestamp[KYOPON_LEN_TIMESTAMP] = '\0';

	return 0;
}

/*----------------------------------------------------------------------
 * high-level interface
 */

#define NAME_FILE_SUFFIX	".r"
#define KYOPON_FILENAME_LEN	76		/* ?? */

int kyopon_read_address __P((struct kyopon *, struct vararray *));
int kyopon_write_address __P((struct kyopon *, struct vararray *));
int kyopon_read_bookmarks __P((struct kyopon *, struct vararray *));
int kyopon_write_bookmarks __P((struct kyopon *, struct vararray *));
int kyopon_list_datafiles __P((struct kyopon *, struct vararray *));
int kyopon_file_get __P((struct kyopon *, FILE *, const char *,
	const uint8_t *));
int kyopon_read_file __P((struct kyopon *, const char * /*kyfn*/,
	const char * /*fn*/));
int kyopon_read_datafiles __P((struct kyopon *, struct vararray *));
int kyopon_file_put __P((struct kyopon *, FILE *, size_t, const char *,
	enum kyopon_file_type));
enum kyopon_file_type kyopon_filename __P((const char *, char *));
int kyopon_write_file __P((struct kyopon *, const char * /*kyfn*/,
	const char * /*fn*/));
int kyopon_write_datafiles __P((struct kyopon *, int, char *[]));
int kyopon_read_mail __P((struct kyopon *, int));
int kyopon_write_mail __P((struct kyopon *, int, char *[]));

static ssize_t do_write __P((int, const void *, size_t));
static char * kyopon_mail_getline __P((FILE *));
static void usage __P((const char *));

int main __P((int, char *[]));

/*
 * read address book from AIR-EDGE PHONE
 */
int
kyopon_read_address(k, addrlist)
	struct kyopon *k;
	struct vararray *addrlist;
{
	uint8_t buf[KYOPON_BUFSZ];
	int count, n, datalen;
	struct progress prgr;
	struct address addrentry;

	vararray_removeall(addrlist);

	/* get number of entries in the addreess book */
	if (kyopon_send_command_str(k, 2, 0x0301, "") == -1)
		goto out;
	if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
		goto out;
	count = atoi(buf + 12);

	if (count > 0) {
		progress_init(&prgr, count, "Reading address book");

		/* start reading address book */
		if (kyopon_send_command_str(k, 3, 0x0302, "") == -1)
			goto out;

		for (n = 0; n < count; n++) {
			progress_setval(&prgr, n + 1);

			/* read an entry */
			datalen = kyopon_receive_data(k, buf, sizeof buf, YES);
			if (datalen == -1) {
				progress_end(&prgr);
				goto out;
			}
			if (buf[3] != 0x03 || buf[9] != 0x03 ||
			    buf[11] != 0x02) {
				progress_end(&prgr);
				fprintf(stderr, "kyopon_read_address: format error\n");
				goto out;
			}
			if (kyopon_send_command_str(k, 0x10, 0x0910, "0") == -1)
				goto out;

			if ((addrentry.data = malloc(datalen - 12)) == NULL) {
				perror("malloc");
				goto out;
			}
			memcpy(addrentry.data, buf + 12, datalen - 12);
			addrentry.size = datalen - 12;
			vararray_add(addrlist, &addrentry);

#if 0			/* XXX FOR TEST */
			{
				char fn[256];
				FILE *fp;

				sprintf(fn, "A%u.vcd", n);
				if ((fp = fopen(fn, "wb")) == NULL)
					perror(fn);
				else {
					if (fwrite(buf + 12, datalen - 12, 1, fp) != 1)
						perror("fwrite");
					if (fclose(fp))
						perror("fclose");
				}
			}
#endif
		}

		progress_end(&prgr);
	}
	return 0;

out:	return -1;
}

/*
 * write address book to AIR-EDGE PHONE
 */
int
kyopon_write_address(k, addrlist)
	struct kyopon *k;
	struct vararray *addrlist;
{
	size_t count, n;
	struct progress prgr;
	uint8_t buf[KYOPON_BUFSZ];
	struct address *addrentry;

#define KYOPON_MAX_ADDRESS	500
	/* check max number of entries */
	if ((count = addrlist->nitem) > KYOPON_MAX_ADDRESS) {
		fprintf(stderr, "kyopon_write_address: too many entries: %lu > %u\n",
			(unsigned long)addrlist->nitem,
			KYOPON_MAX_ADDRESS);
		return -1;
	}

	progress_init(&prgr, count, "Downloading address book");

	/* remove address book */
	if (kyopon_send_command_str(k, 3, 0x0304, "10") == -1)
		goto out;
	if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
		goto out;

	if (count > 0) {
		/* start adding entries */
		if (kyopon_send_command_str(k, 3, 0x0305, "") == -1)
			goto out;
		if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
			goto out;

		for (n = 0; n < count; n++) {
			progress_setval(&prgr, n + 1);

			/* send an entry */
			addrentry = vararray_item(addrlist, n);
			if (kyopon_send_command(k, 3, 0x0303,
			    addrentry->data, addrentry->size) == -1)
				goto out1;
			if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
				goto out1;
			if (buf[12] != '0') {
				progress_end(&prgr);
				fprintf(stderr, "kyopon_write_address: failed: code %c\n",
					buf[12]);
				goto out1;
			}
		}
		if (kyopon_send_command_str(k, 3, 0x0306, "") == -1)
			goto out;
		if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
			goto out;
	}
	progress_end(&prgr);
	return 0;

out1:	progress_end(&prgr);
	(void)kyopon_send_command_str(k, 3, 0x0306, "");
	(void)kyopon_receive_data(k, buf, sizeof buf, YES);
out:	return -1;
}

/*
 * read bookmarks from AIR-EDGE PHONE
 */
int
kyopon_read_bookmarks(k, bookmarklist)
	struct kyopon *k;
	struct vararray *bookmarklist;
{
	uint8_t buf[KYOPON_BUFSZ];
	int count, n, datalen;
	struct progress prgr;
	struct bookmark bm, *lastbm;

	vararray_removeall(bookmarklist);

	/* get number of bookmarks */
	if (kyopon_send_command_str(k, 2, 0x0801, "") == -1)
		goto out;
	if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
		goto out;
	count = atoi(buf + 12);

	if (count > 0) {
		progress_init(&prgr, count, "Reading bookmarks");

		/* start reading bookmarks */
		if (kyopon_send_command_str(k, 3, 0x0802, "") == -1)
			goto out;

		for (n = 0; n < count; n++) {
			progress_setval(&prgr, n + 1);

			/* read a bookmark */
			datalen = kyopon_receive_data(k, buf, sizeof buf, YES);
			if (datalen == -1)
				goto out1;
			if (buf[3] != 0x03 || buf[9] != 0x08 ||
			    buf[11] != 0x02) {
				progress_end(&prgr);
				fprintf(stderr,
				    "kyopon_read_bookmarks: format error\n");
				goto out;
			}
			if (kyopon_send_command_str(k, 0x10, 0x0910, "0") == -1)
				goto out1;

#if 0			/* XXX FOR TEST */
			{
				char fn[256];
				FILE *fp;

				sprintf(fn, "B%u.bmk", n);
				if ((fp = fopen(fn, "wb")) == NULL)
					perror(fn);
				else {
					if (fwrite(buf + 12, datalen - 12, 1, fp) != 1)
						perror("fwrite");
					if (fclose(fp))
						perror("fclose");
				}
			}
#endif

			if (kyopon_bookmark_get(buf + 12, datalen - 12, &bm)
			    == -1)
				goto out1;

#if 1	/* bug workaround */
			/*
			 * AH-K3001V bug workaround
			 *
			 * When the name field is empty, the folder number
			 * field is not sent properly.
			 */
			/* Use URI field as the name */
			if (bm.name == NULL)
				if ((bm.name = stash(bm.uri)) == NULL)
					goto out1;
			/* Use folder number of the previous bookmark if any */
			if (bm.fno == NULL) {
				/* bug workaround */
				lastbm = vararray_last(bookmarklist);
				if (lastbm) {
					if ((bm.fno = stash(lastbm->fno)) == NULL)
						goto out1;
				} else
					bm.fno = stash("0");
			}
#endif
			vararray_add(bookmarklist, &bm);
		}

		progress_end(&prgr);
	}
	return 0;

out1:	progress_end(&prgr);
out:	return -1;
}

/*
 * write bookmarks to AIR-EDGE PHONE
 */
int
kyopon_write_bookmarks(k, bookmarklist)
	struct kyopon *k;
	struct vararray *bookmarklist;
{
	size_t count, n;
	struct progress prgr;
	uint8_t buf[KYOPON_BUFSZ];
	struct bookmark *bm;
	size_t bmsize;

#define KYOPON_MAX_BOOKMARK	100
	/* check max number of entries */
	if ((count = bookmarklist->nitem) > KYOPON_MAX_BOOKMARK) {
		fprintf(stderr, "kyopon_write_bookmarks: too many entries: %lu > %u\n",
			(unsigned long)bookmarklist->nitem,
			KYOPON_MAX_BOOKMARK);
		return -1;
	}

	progress_init(&prgr, count, "Downloading bookmarks");

	/* remove bookmarks */
	if (kyopon_send_command_str(k, 3, 0x0804, "13") == -1)
		goto out;
	if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
		goto out;

	/* add bookmarks */
	if (count > 0) {
		/* sort bookmarks */	/* XXX needed? */
		vararray_qsort(bookmarklist, bookmark_compare);

		/* start adding entries */
		if (kyopon_send_command_str(k, 3, 0x0805, "") == -1)
			goto out;
		if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
			goto out;

		for (n = 0; n < count; n++) {
			progress_setval(&prgr, n + 1);

			/* send a bookmark */
			bm = vararray_item(bookmarklist, n);
			bmsize = kypon_bookmark_put(buf, bm);

			if (kyopon_send_command(k, 3, 0x0803, buf, bmsize)== -1)
				goto out1;
			if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
				goto out1;
			if (buf[12] != '0') {
				progress_end(&prgr);
				fprintf(stderr, "kyopon_write_bookmarks: failed: code %c\n",
					buf[12]);
				goto out1;
			}
		}
		if (kyopon_send_command_str(k, 3, 0x0806, "") == -1)
			goto out;
		if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
			goto out;
	}
	progress_end(&prgr);
	return 0;

out1:	progress_end(&prgr);
	(void)kyopon_send_command_str(k, 3, 0x0806, "");
	(void)kyopon_receive_data(k, buf, sizeof buf, YES);
out:	return -1;
}

/*
 * list files in data folder of AIR-EDGE PHONE
 */
int
kyopon_list_datafiles(k, filelist)
	struct kyopon *k;
	struct vararray *filelist;
{
	uint8_t buf[KYOPON_BUFSZ];
	size_t count, n;
	int datalen;
	struct kdirentry d;
	struct progress prgr;

	vararray_removeall(filelist);

	if (kyopon_send_command_str(k, 2, 0x0601, "") == -1)
		goto out;
	if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
		goto out;
	count = atoi(buf + 12);

	progress_init(&prgr, count, "Getting file list");

	if (count > 0) {
		if (kyopon_send_command_str(k, 3, 0x0607, "") == -1)
			goto out2;

		for (n = 0; n < count; n++) {
			progress_setval(&prgr, n + 1);

			datalen = kyopon_receive_data(k, buf, sizeof buf, YES);
			if (datalen == -1)
				goto out2;
			if (datalen < 16 || buf[3] != 0x03 || buf[9] != 0x06 ||
			    buf[11] != 0x07) {
				fprintf(stderr, "kyopon_list_datafiles: format error\n");
				goto out2;
			}
			if (kyopon_send_command_str(k, 0x10, 0x0910, "0") == -1)
				goto out2;

			kyopon_kdirentry_get(buf + 12, datalen - 12, &d);
			vararray_add(filelist, &d);
		}
	}

	progress_end(&prgr);
	return 0;

out2:	progress_end(&prgr);
out:	return -1;
}

int
kyopon_file_get(k, fp, name, typestr)
	struct kyopon *k;
	FILE *fp;
	const char *name;
	const uint8_t *typestr;
{
	uint8_t buf[KYOPON_BUFSZ];
	uint8_t *p;
	ssize_t filelen, nr;
	size_t rest;
	int inhdr;
	int nretry = 10;

again:
	sprintf(buf, "%s\t%s\t", name, typestr);
	if (kyopon_send_command_str(k, 3, 0x0602, buf) == -1)
		return -1;
	if ((filelen = kyopon_receive_data(k, buf, sizeof buf, NO)) < 12)
		return -1;

#if 0
	fprintf(stderr, "filelen = %u\n", filelen);
#endif

	/*
	 * name \t type \t data....
	 * |<------ filelen ------>|
	 */
	inhdr = 2;
	for (rest = filelen - 12; rest > 0; rest -= nr) {
		if ((nr = read(k->fd, buf,
		    (rest < sizeof buf) ? rest : sizeof buf)) == -1) {
			perror("read");
			return -1;
		}
		if (nr == 0) {
			/*
			 * Read timed out.  Probably because some data
			 * were lost.  Try again.
			 */
			if (--nretry) {
#if 0
				fprintf(stderr, "kyopon_file_get: retry\n");
#endif
				rewind(fp);
				goto again;
			}
			fprintf(stderr, "kyopon_file_get: unexpected EOF\n");
			return -1;
		}
		p = buf;
		while (inhdr) {
			while (p < buf + nr)
				if (*p++ == '\t') {
					inhdr--;
					break;
				}
			if (p == buf + nr)
				goto continue_reading;
		}
		if (fwrite(p, nr - (p - buf), 1, fp) != 1) {
			perror("fwrite");
			return -1;
		}
	continue_reading:;
	}

	if (inhdr) {
		fprintf(stderr, "%s: not found\n", name);
		return -1;
	}

	return 0;
}

/*
 * read a file from AIR-EDGE PHONE
 */
int
kyopon_read_file(k, kyfn, fn)
	struct kyopon *k;
	const char *kyfn, *fn;
{
	FILE *fp;

	if ((fp = fopen(fn, "wb")) == NULL) {
		perror(fn);
		return -1;
	} else {
		if (kyopon_file_get(k, fp, kyfn,
		    filetypes[KYOPON_FILETYPE_OTHER]))
			remove(fn);
		if (fclose(fp))
			perror("fclose");
	}
	return 0;
}

/*
 * read files from data folder of AIR-EDGE PHONE
 */
int
kyopon_read_datafiles(k, flist)
	struct kyopon *k;
	struct vararray *flist;
{
	size_t n;
	struct kdirentry *d;
	char fn[256];
	FILE *fp;
	struct progress prgr;

	if (flist->nitem == 0)
		return 0;

	progress_init(&prgr, flist->nitem, "Reading files");
	for (n = 0; n < flist->nitem; n++) {
		progress_setval(&prgr, n + 1);

		d = vararray_item(flist, n);
#if 0
		printf("%u: %-6s %8u %s\n", n, d->ftype, d->size,
		    d->timestamp);
#endif
		sprintf(fn, "D%lu.dat%s", (unsigned long)n, NAME_FILE_SUFFIX);
		if ((fp = fopen(fn, "wb")) == NULL)
			perror(fn);
		else {
			fprintf(fp, "%s\n%s\n%s\n",
			    d->name, d->ftype, d->timestamp);
			if (fclose(fp))
				perror("fclose");

			sprintf(fn, "D%lu.dat", (unsigned long)n);
			if ((fp = fopen(fn, "wb")) == NULL)
				perror(fn);
			else {
				if (kyopon_file_get(k, fp, d->name, d->ftype))
					remove(fn);
				if (fclose(fp))
					perror("fclose");
			}
		}
	}
	progress_end(&prgr);
	return 0;
}

static ssize_t
do_write(fd, buf, nbytes)
	int fd;
	const void *buf;
	size_t nbytes;
{
	size_t cnt;
	ssize_t nw;
	ssize_t r = 0;
	const char *p;

	for (nw = 0, p = buf, cnt = nbytes; cnt; nw += r, p += r, cnt -= r)
		if ((r = write(fd, p, cnt)) <= 0)
			break;

	return (r < 0 && nw == 0) ? -1 : nw;
}

int
kyopon_file_put(k, fp, filelen, name, type)
	struct kyopon *k;
	FILE *fp;
	size_t filelen;
	const char *name;
	enum kyopon_file_type type;
{
	uint8_t buf[KYOPON_BUFSZ];
	size_t rest;
	ssize_t nw;
	int hdrsz;
	int error = 0;

	hdrsz = sprintf(buf, "%s\t%s\t", name, filetypes[type]);
	if (kyopon_send_command(k, 3, 0x0603, NULL, filelen + hdrsz) == -1)
		return -1;

	if (do_write(k->fd, buf, hdrsz) != hdrsz) {
		fprintf(stderr, "kyopon_file_put: write failed (header)\n");
		error = -1;
	}
	for (rest = filelen; rest > 0; rest -= nw) {
		nw = rest < sizeof buf ? rest : sizeof buf;
		if (fread(buf, nw, 1, fp) != 1) {
			fprintf(stderr, "kyopon_file_put: fread failed\n");
			error = -1;
		}
		if (do_write(k->fd, buf, nw) != nw) {
			fprintf(stderr, "kyopon_file_put: write failed (data)\n");
			error = -1;
		}
	}
	if (kyopon_receive_data(k, buf, sizeof buf, YES) == -1)
		error = -1;
	if (buf[12] != '0') {
		fprintf(stderr, "kyopon_file_put: write failed\n");
		error = -1;
	}

	return error;
}

/* make file name and file type */
enum kyopon_file_type
kyopon_filename(sfn, kyoponfn)
	const char *sfn;
	char *kyoponfn;		/* output */
{
	FILE *fp;
	char *rfn, *p;
	const char *q, *sfx;
	int i, sfxlen;

	/*
	 * open file which contains filename for kyopon
	 */
	if ((rfn = alloca(strlen(sfn) + sizeof NAME_FILE_SUFFIX)) == NULL) {
		perror("alloca");
		return KYOPON_FILETYPE_INVALID;
	}
	sprintf(rfn, "%s%s", sfn, NAME_FILE_SUFFIX);
	if ((fp = fopen(rfn, "r")) != NULL) {
		if (fgets(kyoponfn, KYOPON_FILENAME_LEN + 1, fp) != NULL) {
			if ((p = strchr(kyoponfn, '\r')) != NULL)
				*p = '\0';
			if ((p = strchr(kyoponfn, '\n')) != NULL)
				*p = '\0';
		} else {
			if (ferror(fp))
				perror(rfn);
			else
				fprintf(stderr, "%s: unexpected EOF\n", rfn);
			return KYOPON_FILETYPE_INVALID;
		}
		fclose(fp);
	} else {
		if (errno != ENOENT) {
			perror(rfn);
			return KYOPON_FILETYPE_INVALID;
		}
		kyoponfn[0] = '\0';
	}

	if (kyoponfn[0] == '\0') {
		/*
		 * use source filename
		 */
		if ((q = strrchr(sfn, '/')) != NULL)
			q++;
		else
			q = sfn;
		/*
		 * XXX max length depends on the char type (ASCII/kanji)
		 */
		if (strlen(q) > KYOPON_FILENAME_LEN) {
			fprintf(stderr, "%s: filename too long\n", q);
			return KYOPON_FILETYPE_INVALID;
		}
		if (q[0] == '\0') {
			fprintf(stderr, "%s: can't generate filename\n", sfn);
			return KYOPON_FILETYPE_INVALID;
		}
		strcpy(kyoponfn, q);
	}

	if ((sfx = strrchr(kyoponfn, '.')) != NULL) {
		sfxlen = strlen(sfx);
		for (i = 0; i < KYOPON_FILETYPE_OTHER; i++) {
			if ((p = strstr(filesfxs[i], sfx)) &&
			    (p[sfxlen] == '.' || p[sfxlen] == '\0'))
				return i;
		}
	}
	return KYOPON_FILETYPE_OTHER;
}

/*
 * write a file to AIR-EDGE PHONE
 */
int
kyopon_write_file(k, kyfn, fn)
	struct kyopon *k;
	const char *kyfn, *fn;
{
	FILE *fp;
	struct stat st;

	if ((fp = fopen(fn, "rb")) == NULL) {
		perror(fn);
		goto out;
	}
	if (fstat(fileno(fp), &st)) {
		perror(fn);
		goto out2;
	}

#if 0
	fprintf(stderr, "write: size %u\n", (unsigned)st.st_size);
#endif
	kyopon_file_put(k, fp, (size_t)st.st_size, kyfn, KYOPON_FILETYPE_OTHER);

	fclose(fp);
	return 0;

out2:	fclose(fp);
out:	return -1;
}

/*
 * write files to data folder of AIR-EDGE PHONE
 */
int
kyopon_write_datafiles(k, nfiles, files)
	struct kyopon *k;
	int nfiles;
	char *files[];
{
	FILE *fp;
	int n;
	enum kyopon_file_type type;
	struct stat st;
	struct progress prgr;
	char kyoponfn[KYOPON_FILENAME_LEN + 1];

	if (nfiles <= 0)
		return 0;

	progress_init(&prgr, nfiles, "Downloading files");
	for (n = 0; n < nfiles; n++) {
		if ((type = kyopon_filename(files[n], kyoponfn)) ==
		    KYOPON_FILETYPE_INVALID)
			continue;

		progress_setval(&prgr, n + 1);
#if 0
		fprintf(stderr, "write: %s -> %s (%s)\n",
		    files[n], kyoponfn, filetypes[type]);
#endif
		if ((fp = fopen(files[n], "rb")) == NULL) {
			perror(files[n]);
			continue;
		}
		if (fstat(fileno(fp), &st)) {
			perror(files[n]);
			goto next;
		}
#if 0
		fprintf(stderr, "write: size %u\n", (unsigned)st.st_size);
#endif
		kyopon_file_put(k, fp, (size_t)st.st_size, kyoponfn, type);
	next:
		fclose(fp);
	}

	progress_end(&prgr);

	return 0;
}

/*
 * read mail from AIR-EDGE PHONE
 */
int
kyopon_read_mail(k, light_mail)
	struct kyopon *k;
	int light_mail;
{
	uint8_t *buf, *btmp, *b;
	size_t bufsz;
	int count, n;
	ssize_t len, nr;
	size_t rest;
	char fn[80];
	uint8_t *fields[7];
	FILE *fp;
	struct progress prgr;

	if ((buf = malloc(bufsz = KYOPON_BUFSZ)) == NULL) {
		perror("malloc");
		return -1;
	}

	/*
	 * Light mail / E-mail
	 */
	if (kyopon_send_command_str(k, 2, light_mail? 0x0401 : 0x0501, "") == -1)
		goto out2;
	if (kyopon_receive_data(k, buf, bufsz, YES) == -1)
		goto out2;
	count = atoi(buf + 12);

	progress_init(&prgr, count,
	    light_mail? "Reading light mail" : "Reading E-mail");

	if (count > 0) {
		if (kyopon_send_command_str(k, 3,
		    light_mail? 0x0402 : 0x0502, "") == -1)
			goto out3;

		for (n = 0; n < count; n++) {
			progress_setval(&prgr, n + 1);

			if ((len = kyopon_receive_data(k, buf, bufsz, NO)) < 12)
				goto out3;
			if (buf[3] != 0x03 ||
			    buf[9] != (light_mail? 0x04 : 0x05) ||
			    buf[11] != 0x02) {
				fprintf(stderr,
				    "kyopon_read_mail: %smail: format error\n",
				    light_mail? "light " : "E-");
				goto out3;
			}

			len -= 12;

			/*
			 * make sure we have enough buffer
			 * (+1 for nul termination)
			 */
			if ((size_t)len + 1 > bufsz) {
				if ((btmp = realloc(buf, len + 1)) == NULL) {
					perror("realloc");
					goto out3;
				} else {
					buf = btmp;
					bufsz = len + 1;
				}
			}

			for (rest = len, b = buf; rest > 0;
			    rest -= nr, b += nr) {
				if ((nr = read(k->fd, b, rest)) == -1) {
					perror("read");
					goto out3;
				}
				if (nr == 0) {
					fprintf(stderr, "kyopon_read_mail: unexpected EOF\n");
					goto out3;
				}
			}

			if (kyopon_send_command_str(k, 0x10, 0x0910, "0") == -1)
				goto out3;

			if (kyopon_get_fields(buf, len, fields,
			    light_mail? 7 : 5, NO) == -1)
				goto out3;

			sprintf(fn, "%c%u.txt", light_mail? 'L' : 'M', n);
			if ((fp = fopen(fn, "w")) == NULL) {
				perror(fn);
				goto out3;
			}
			if (light_mail) {
				fprintf(fp,
				    "X-KYOPON-LIGHT: %s,%s,%s,%s\nDATE: %s\nTO: %s\n",
				    fields[0], fields[1], fields[2], fields[3],
				    fields[4], fields[5]);
				b = fields[6];
			} else {	/* E-mail */
				fprintf(fp, "X-KYOPON-MAIL: %s,%s,%s,%s\n",
				    fields[0], fields[1], fields[2], fields[3]);
				b = fields[4];
			}

			for ( ; *b; b++) {
				if (*b != '\r')
					putc(*b, fp);
			}
			if (ferror(fp)) {
				perror(fn);
				fclose(fp);
				goto out3;
			}
			if (fclose(fp)) {
				perror(fn);
				goto out3;
			}
		}
	}

	progress_end(&prgr);
	free(buf);
	return 0;

out3:	progress_end(&prgr);
out2:	free(buf);
	return -1;
}

static char *
kyopon_mail_getline(fp)
	FILE *fp;
{
	size_t len;
	char *ln, *lend;

	if ((ln = fgetln(fp, &len)) == NULL)
		return NULL;
	lend = ln + len;
	if (ln >= lend || lend[-1] != '\n')
		return NULL;
	*--lend = '\0';

	return ln;
}

/*
 * write mail to AIR-EDGE PHONE
 */
int
kyopon_write_mail(k, nfiles, files)
	struct kyopon *k;
	int nfiles;
	char *files[];
{
	FILE *fp;
	int light_mail;
	int n;
	struct progress prgr;
	struct varbuf buf;
	char *ln;
	uint32_t len;
	unsigned p1, p2, p3;
	char p4[5], lmail_date[20], lmail_to[20];
	char str[128];
	int c;
	unsigned char ch;
	uint8_t *body;
	size_t bodylen;

	if (nfiles <= 0)
		return 0;

	varbuf_init(&buf);
	progress_init(&prgr, nfiles, "Downloading mail");
	for (n = 0; n < nfiles; n++) {
		progress_setval(&prgr, n + 1);

		body = NULL;

		if ((fp = fopen(files[n], "r")) == NULL) {
			perror(files[n]);
			continue;
		}

		/* line #1 */
		if ((ln = kyopon_mail_getline(fp)) == NULL) {
		ferr:
			if (ferror(fp))
				perror(files[n]);
			else if (feof(fp))
				fprintf(stderr, "kyopon_write_mail: unexpected EOF\n");
			goto next;
		}
		if (sscanf(ln, "X-KYOPON-LIGHT: %u , %u , %u , %4s",
		    &p1, &p2, &p3, p4) == 4) {
			light_mail = YES;
		} else if (sscanf(ln, "X-KYOPON-MAIL: %u , %u , %u , %4s",
		    &p1, &p2, &p3, p4) == 4) {
			light_mail = NO;
		} else {
		fmterr:
			fprintf(stderr,
			    "\n%s: not in light mail nor in E-mail format\n",
			    files[n]);
			goto next;
		}

		if (strcmp(p4, "ON") != 0 && strcmp(p4, "OFF") != 0)
			goto fmterr;

		if (light_mail) {
			/* line #2 */
			if ((ln = kyopon_mail_getline(fp)) == NULL)
				goto ferr;
			if (sscanf(ln, "DATE: %19s", lmail_date) != 1)
				goto fmterr;

			/* line #3 */
			if ((ln = kyopon_mail_getline(fp)) == NULL)
				goto ferr;
			if (sscanf(ln, "TO: %19s", lmail_to) != 1) {
				/* check for empty "TO:" */
				if (strncmp(ln, "TO:", 3) != 0)
					goto fmterr;
				lmail_to[0] = '\0';
			}
		}

		/*
		 * read body, converting LF to CRLF
		 */
		while ((c = getc(fp)) != EOF) {
			switch (c) {
			case '\r':
				break;	/* ignore */
			case '\n':
				if (varbuf_add(&buf, "\r\n", 2) == -1)
					goto out;
				break;
			default:
				ch = c;
				if (varbuf_add(&buf, &ch, 1) == -1)
					goto out;
				break;
			}
		}
		if (ferror(fp))
			goto ferr;

		if ((body = varbuf_getbuf(&buf, &bodylen)) == NULL)
			goto out;

		/* reuse buf for binary */
		varbuf_init(&buf);

		if (varbuf_add(&buf, "\034", 1) == -1)	/* header */
			goto out;

#define PARAMNUM(parno, xx, num)	\
		len = sprintf(str + 7, "%u", (num));		\
		str[0] = 0x1e;					\
		str[1] = (parno);				\
		str[2] = (xx);					\
		str[3] = len;					\
		str[4] = len >> 8;				\
		str[5] = len >> 16;				\
		str[6] = len >> 24;				\
		if (varbuf_add(&buf, str, 7 + len) == -1)	\
			goto out;
#define PARAMSTR(parno, xx, s)	\
		len = strlen(s);				\
		str[0] = 0x1e;					\
		str[1] = (parno);				\
		str[2] = (xx);					\
		str[3] = len;					\
		str[4] = len >> 8;				\
		str[5] = len >> 16;				\
		str[6] = len >> 24;				\
		if (varbuf_add(&buf, str, 7) == -1)		\
			goto out;				\
		if (varbuf_add(&buf, (s), len) == -1)		\
			goto out;

		PARAMNUM(1, 1, p1);
		PARAMNUM(2, 1, p2);
		PARAMNUM(3, 1, p3);
		PARAMSTR(4, 1, p4);

		if (light_mail) {
			PARAMSTR(5, 1, lmail_date);
			PARAMSTR(6, 1, lmail_to);
			str[1] = 7;
			str[2] = 2;
		} else {
			str[1] = 5;
			str[2] = 3;
		}
		str[0] = 0x1e;
		str[3] = bodylen;
		str[4] = bodylen >> 8;
		str[5] = bodylen >> 16;
		str[6] = bodylen >> 24;
		if (varbuf_add(&buf, str, 7) == -1)
			goto out;

		if (kyopon_send_command(k, 3, light_mail? 0x0403 : 0x0503,
		    NULL, buf.size + bodylen) == -1)
			goto out;

		if (do_write(k->fd, buf.buf, buf.size) != (ssize_t)buf.size) {
			fprintf(stderr,
			    "kyopon_write_mail: write failed (header)\n");
			goto out;
		}
		if (do_write(k->fd, body, bodylen) != (ssize_t)bodylen) {
			fprintf(stderr,
			    "kyopon_write_mail: write failed (body)\n");
			goto out;
		}

		if (kyopon_receive_data(k, str, sizeof str, YES) == -1)
			goto out;
		if (str[12] != '0') {
			fprintf(stderr, "kyopon_write_mail: write failed\n");
			goto out;
		}

	next:
		freestash(body);
		varbuf_reinit(&buf);
		fclose(fp);
#undef PARAMNUM
#undef PARAMSTR
	}

	progress_end(&prgr);
	return 0;

out:
	progress_end(&prgr);
	freestash(body);
	varbuf_reinit(&buf);
	fclose(fp);

	return -1;
}

static void
usage(c)
	const char *c;	/* command name */
{

	fprintf(stderr, "\
usage:	%s [-f device] -r             read all possible data\n\
	%s [-f device] -a             read address book (addressbook.txt)\n\
	%s [-f device] -b             read bookmarks    (bookmarks.txt)\n\
	%s [-f device] -i             read browser info (browser.ini)\n\
	%s [-f device] -d             read data files   (D*.dat, D*.dat%s)\n\
	%s [-f device] -m             read mail         (L*.txt, M*.txt)\n\
	%s [-f device] -A <file>      write (replace) address book\n\
	%s [-f device] -B <file>      write (replace) bookmarks\n\
	%s [-f device] -I <file>      write (replace) browser info\n\
	%s [-f device] -D <file> ...  write (add or replace) data files\n\
	%s [-f device] -M <file> ...  write (add) mail\n\
",
	c, c, c, c, c, NAME_FILE_SUFFIX, c, c, c, c, c, c);

	exit(2);
}

#ifndef DEFAULT_UKYOPON_DEV
#define DEFAULT_UKYOPON_DEV	"/dev/ttyU1"	/* default device path */
#endif

#define FLAG_READ_ADDRESS	0x0001
#define FLAG_READ_BOOKMARKS	0x0002
#define FLAG_READ_BROWSERINI	0x0004
#define FLAG_READ_DATA		0x0008
#define FLAG_READ_MAIL		0x0010
#define FLAG_READ_ALL		0x001f	/* all "READ" flags */

#define FLAG_WRITE_ADDRESS	0x0100
#define FLAG_WRITE_BOOKMARKS	0x0200
#define FLAG_WRITE_BROWSERINI	0x0400
#define FLAG_WRITE_DATA		0x0800
#define FLAG_WRITE_MAIL		0x1000

int
main(argc, argv)
	int argc;
	char *argv[];
{
	struct kyopon k;
	struct vararray addrlist;
	struct vararray bookmarklist;
	struct vararray filelist;
	char *passwd;
	int ch;
	unsigned flag = 0;
	const char *fn_address = 0, *fn_bookmarks = 0, *fn_browserini = 0;
	const char *devpath = DEFAULT_UKYOPON_DEV;
	char *myname = basename(argv[0]);

	vararray_init(&addrlist, sizeof(struct address), address_dtor);
	vararray_init(&bookmarklist, sizeof(struct bookmark), bookmark_dtor);
	vararray_init(&filelist, sizeof(struct kdirentry), kdirentry_dtor);

	while ((ch = getopt(argc, argv, "f:rabidmA:B:I:DM")) != -1) {
		switch (ch) {
		case 'f':		/* spacify device name */
			devpath = optarg;
			break;

		case 'r':		/* read all */
			flag |= FLAG_READ_ALL;
			break;
		case 'a':		/* read address book */
			flag |= FLAG_READ_ADDRESS;
			break;
		case 'b':		/* read bookmarks */
			flag |= FLAG_READ_BOOKMARKS;
			break;
		case 'i':		/* read browser.ini */
			flag |= FLAG_READ_BROWSERINI;
			break;
		case 'd':		/* read data files */
			flag |= FLAG_READ_DATA;
			break;
		case 'm':		/* read mail */
			flag |= FLAG_READ_MAIL;
			break;

		case 'A':		/* download address book */
			flag |= FLAG_WRITE_ADDRESS;
			fn_address = optarg;
			break;
		case 'B':		/* download bookmarks */
			flag |= FLAG_WRITE_BOOKMARKS;
			fn_bookmarks = optarg;
			break;
		case 'I':		/* download browser.ini */
			flag |= FLAG_WRITE_BROWSERINI;
			fn_browserini = optarg;
			break;
		case 'D':		/* download data files */
			flag |= FLAG_WRITE_DATA;
			break;
		case 'M':		/* download mail */
			flag |= FLAG_WRITE_MAIL;
			break;

		case '?':
		default:
			usage(myname);
			/* NOTREACHED */
			break;
		}
	}

	argc -= optind;
	argv += optind;

	switch (flag & (FLAG_WRITE_DATA|FLAG_WRITE_MAIL)) {
	case 0:
		if (argc != 0 || flag == 0)
			usage(myname);
		break;

	case FLAG_WRITE_DATA:
	case FLAG_WRITE_MAIL:
		if (argc == 0) {
			fprintf(stderr, "%s: missing arguments\n",
			    myname);
			return 2;
		}
		break;		/* OK */

	default:
		fprintf(stderr,
		    "%s: only one of -D or -M can be specified\n",
		    myname);
		return 2;
	}

	kyopon_init(&k, devpath);

	passwd = getpass("handset password: ");

	if (kyopon_open(&k) == -1)
		return 1;

	if (kyopon_login_with_passwd(&k, passwd) != 0)
		goto error;

	/* strike out passwd */
	strcpy(passwd, "    ");

	/*
	 * read
	 */
	if (flag & FLAG_READ_ADDRESS) {
		if (kyopon_read_address(&k, &addrlist) != 0)
			goto error;
		if (address_file_write(&addrlist, "addressbook.txt") != 0)
			goto error;
	}
	if (flag & FLAG_READ_BOOKMARKS) {
		if (kyopon_read_bookmarks(&k, &bookmarklist) != 0)
			goto error;
		if (bookmark_file_write(&bookmarklist, "bookmarks.txt") != 0)
			goto error;
	}
	if (flag & FLAG_READ_BROWSERINI) {
		if (isatty(1))
			printf("Reading browser.ini\n");
		if (kyopon_read_file(&k, "..\\browser.ini", "browser.ini") != 0)
			goto error;
	}
	if (flag & FLAG_READ_DATA) {
		if (kyopon_list_datafiles(&k, &filelist) != 0)
			goto error;
		if (kyopon_read_datafiles(&k, &filelist) != 0)
			goto error;
	}
	if (flag & FLAG_READ_MAIL) {
		/* light mail */
		if (kyopon_read_mail(&k, YES) == -1)
			goto error;
		/* E-mail */
		if (kyopon_read_mail(&k, NO) == -1)
			goto error;
	}

	/*
	 * download
	 */
	if (flag & FLAG_WRITE_ADDRESS) {
		if (address_file_read(&addrlist, fn_address) != 0)
			goto error;
		if (kyopon_write_address(&k, &addrlist) != 0)
			goto error;
	}
	if (flag & FLAG_WRITE_BOOKMARKS) {
		if (bookmark_file_read(&bookmarklist, fn_bookmarks) != 0)
			goto error;
		if (kyopon_write_bookmarks(&k, &bookmarklist) != 0)
			goto error;
	}
	if (flag & FLAG_WRITE_BROWSERINI) {
		if (isatty(1))
			printf("Downloading browser.ini\n");
		if (kyopon_write_file(&k, "..\\browser.ini", fn_browserini) != 0)
			goto error;
	}
	if (flag & FLAG_WRITE_DATA) {
		if (kyopon_write_datafiles(&k, argc, argv) != 0)
			goto error;
	}
	if (flag & FLAG_WRITE_MAIL) {
		if (kyopon_write_mail(&k, argc, argv) != 0)
			goto error;
	}

	kyopon_close(&k);
	return 0;

error:
	kyopon_close(&k);
	return 1;
}
