/* C version of pgraf metacode generator */
#define NAME "pgraf3.out"
#define AI2 ny*ix + iy
#define BADDATA  1.e30 /*data greater than this is ignored in scale and dcon*/
#define LARGE 1.e20
#define SMALL 1.e-20
#define XPT 1500
#define YPT 1500
#define VSCALE 20       /* scaling factor for automatic vector scaling */

#include <stdio.h>
#include <math.h>

static FILE *pgraf;
static float axl[100],ayl[100],axh[100],ayh[100],baddata; 
static float md();
static void windono(),wingen(),axgen();

/* open metacode file */
gopen()
{
  int loop;

  if ((pgraf = fopen(NAME,"w")) == NULL) exit(1);
  for(loop = 0; loop < 100; loop++) {
    axh[loop] = ayh[loop] = 1;
    axl[loop] = ayl[loop] = 0;
  }
  baddata = BADDATA;                 /* default baddata value */
}
 
/* close metacode file */
gclose()
{
  fclose(pgraf);
}
 
/* clear screen */
gclear()
{
  fprintf(pgraf," i\n");
}
 
/* wait for operator response */
gpause()
{
  fprintf(pgraf," f\n");
}

/* set bad data limit */
badset(value)
float value;
{
  fprintf(pgraf," x\n");
  fprintf(pgraf,"%g\n",value);
  baddata = value;
}
 
static void windowno(nw,xl,xh,yl,yh)
int nw;
float xl,xh,yl,yh;
{
  wingen(nw,xl,yl,xh,yh,0);
}

/* define a window */
window(nw,xl,xh,yl,yh)
int nw;
float xl,xh,yl,yh;
{
  wingen(nw,xl,yl,xh,yh,1);
}

static void wingen(nw,xl,yl,xh,yh,draw)
int nw,draw;
float xl,xh,yl,yh; 
{
  fprintf(pgraf," w\n");
  fprintf(pgraf,"%d %g %g %g %g %d\n",nw,xl,yl,xh,yh,draw);
}
 
/* draw axes with labels */
axes(nw,sxl,sxh,ntx,xlabel,syl,syh,nty,ylabel)
int nw,ntx,nty;
float sxl,sxh,syl,syh;
char *xlabel,*ylabel;
{
  axgen(nw,sxl,sxh,ntx,xlabel,syl,syh,nty,ylabel,1,1);
}

/* draw invisible axes */
axesnoxy(nw,sxl,sxh,ntx,syl,syh,nty)
int nw;
float sxl,sxh,syl, syh;
{
  axgen(nw,sxl,sxh,ntx," ",syl,syh,nty," ",0,0);
} 

axesnox(nw,sxl,sxh,ntx,syl,syh,nty,ylabel)
int nw,nty;
float sxl,sxh,syl,syh;
char *ylabel;
{
  axgen(nw,sxl,sxh,ntx," ",syl,syh,nty,ylabel,0,1);
}

axesnoy(nw,sxl,sxh,ntx,xlabel,syl,syh,nty)
int nw,ntx;
float sxl,sxh,syl,syh;
char *xlabel;
{
  axgen(nw,sxl,sxh,ntx,xlabel,syh,syl,nty," ",1,0);
}

/* basic axis drawing routine */
static void axgen(nw,sxl,sxh,ntx,xlabel,syl,syh,nty,ylabel,drx,dry)
int nw,ntx,nty,drx,dry;
float sxl,sxh,syl,syh;
char *xlabel,*ylabel;
{
  int i;
  char *temp;

/* remove newlines from the strings */
  temp = xlabel;
  while (*temp != '\0') {
    if ((*temp == '\n') || (*temp == '\r')) *temp = ' ';
    temp++;
  }
  temp = ylabel;
  while (*temp != '\0') {
    if ((*temp == '\n') || (*temp == '\r')) *temp = ' ';
    temp++;
  }
  axl[nw] = sxl;
  ayl[nw] = syl;
  axh[nw] = sxh;
  ayh[nw] = syh;
  fprintf(pgraf," a\n");  
  fprintf(pgraf,"%d %g %g %g %g %d %d %d %d\n",
       nw,sxl,sxh,syl,syh,ntx,nty,drx,dry);
  fprintf(pgraf,"%s\n%s\n",xlabel,ylabel);
}

/* conto convert to conto2 */
conto(nw,nx,ny,dz,ltype,v)
int nw,nx,ny,ltype;
float dz, v[];
{
  float xloc[XPT],yloc[YPT];
  int ix,iy;
  if (nx > XPT || ny > YPT) {
    fprintf(stderr,"pgraf: data size %d > %d or %d > %d too big\n",
         nx,XPT,ny,YPT);
    exit(1);
  }
  if(nx == 1) xloc[0] = (axh[nw]+axl[nw])/2.;  
  else  for(ix = 0; ix<nx; ix++)
    xloc[ix] = (axh[nw]-axl[nw])*ix/( nx - 1.) + axl[nw];

  /* space the points evenly  */ 
  if(ny == 1) yloc[0] = (ayh[nw]+ayl[nw])/2.; 
    else for(iy = 0; iy<ny ; iy++)      /* for the default axes values  */     
    yloc[iy] = (ayh[nw]-ayl[nw])*iy/(1.*ny -1.) + ayl[nw];
  conto2(nw,nx,ny,dz,ltype,v,xloc,yloc);
}

/* make a contour plot */
conto2(nw,nx,ny,dz,ltype,v,xloc,yloc)
int nw,nx,ny,ltype;
float dz,v[],xloc[],yloc[];
{
  int ix,iy;
  fprintf(pgraf," c\n");
  fprintf(pgraf,"%d %d %d %g %d\n",nw,nx,ny,dz,ltype);
  for(ix = 0;ix<nx;ix++)
    fprintf(pgraf,"%g\n",xloc[ix]);
  for(iy = 0;iy<ny;iy++ ){
    fprintf(pgraf,"%g\n",yloc[iy]);
  }
  for (iy = 0; iy < ny; iy++) {
    for (ix = 0; ix < nx; ix++) {
      fprintf(pgraf,"%g\n",v[AI2]);
    }
  }
}
 
/* compute a reasonable contour interval */
double dcon(nx,ny,nz,v)
int nx,ny,nz;
float v[];
{
  int i,n;
  float min,max;
  min = 1.e30;
  max = -min;
  n = nx*ny;
  for (i = 0; i < n; i++) {
    if(fabs((v[i]))< baddata) { 
      if(max < v[i]) max = v[i];
      if (min > v[i]) min = v[i];
    }
  }
  return((max - min)/nz);
}

/* fill plot old style */
fill(nw,nx,ny,dvl,dvh,ltype,v)
int nw,nx,ny,ltype;
float dvl,dvh,v[];
{
  int ix,iy;
  float xloc[XPT],yloc[YPT];
  if (nx > XPT || ny > YPT) {
    fprintf(stderr,"pgraf: data size %d > %d or %d > %d too big\n",
         nx,XPT,ny,YPT);
    exit(1);
  }
  if(nx == 1) xloc[0] = (axh[nw]+axl[nw])/2. ; 
  else for(ix = 0; ix<nx; ix++)
    xloc[ix] = (axh[nw]-axl[nw])*ix/(1.*nx - 1.) + axl[nw];
  if(ny == 1) yloc[0] = (ayh[nw]+ayl[nw])/2.;
  else for(iy = 0; iy<ny; iy++)
    yloc[iy] = (ayh[nw]-ayl[nw])*iy/(1.*ny -1.) + ayl[nw];
  fill2(nw,nx,ny,dvl,dvh,ltype,v,xloc,yloc);
}

/* make a fill plot */
fill2(nw,nx,ny,dvl,dvh,ltype,v,xloc,yloc)
int nw,nx,ny,ltype;
float dvl,dvh,v[],xloc[],yloc[];
{
  int ix,iy;
  fprintf(pgraf," d\n");
  fprintf(pgraf,"%d %d %d %g %g %d\n",nw,nx,ny,dvl,dvh,ltype);
  for(ix = 0;ix<nx;ix++)
    fprintf(pgraf,"%g\n",xloc[ix]);
  for(iy = 0; iy<ny;iy++)
    fprintf(pgraf,"%g\n",yloc[iy]);
  for (iy = 0; iy < ny; iy++) {
    for (ix = 0; ix < nx; ix++) {
      fprintf(pgraf,"%g\n",v[AI2]);
    }
  }
}

/* grid old version */
grid(nw,nx,ny,jx,jy,v)
int nw,nx,ny,jx,jy;
float v[];
{
  int ix,iy;
  float xloc[XPT],yloc[YPT];
  if (nx > XPT || ny > YPT) {
    fprintf(stderr,"pgraf: data size %d > %d or %d > %d too big\n",
         nx,XPT,ny,YPT);
    exit(1);
  }
  if (nx == 1) {
    xloc[0] = (axh[nw] + axl[nw])/2.;
  }
  else {
    for(ix = 0; ix < nx; ix++) {
      xloc[ix] = (axh[nw] - axl[nw])*ix/(nx - 1.) + axl[nw];
    }
  }
  if (ny == 1) {
    yloc[0] = (ayh[nw] + ayl[nw])/2.;
  }
  else {
    for(iy = 0; iy < ny; iy++) {
      yloc[iy] = (ayh[nw] - ayl[nw])*iy/(ny - 1.) + ayl[nw];
    }
  }
  grid2(nw,nx,ny,jx,jy,v,xloc,yloc);
}

/* make a grid of numbers */
grid2(nw,nx,ny,jx,jy,v,xloc,yloc)
int nw,nx,ny,jx,jy;
float v[],xloc[],yloc[];
{
  int ix,iy;
  float max;

/* safety change on jx and jy */
  if (jx <= 0) jx = 1;
  if (jy <= 0) jy = 1;
  fprintf(pgraf," g\n");
  fprintf(pgraf,"%d %d %d %d %d\n",nw,nx,ny,jx,jy);
  for(ix = 1 + jx/2; ix < nx - 1; ix += jx)
    fprintf(pgraf,"%g\n",xloc[ix]);
  for(iy = 1 + jy/2; iy < ny - 1; iy += jy)
    fprintf(pgraf,"%g\n",yloc[iy]);
  max = 0.;
  for (ix = 0; ix < nx*ny; ix++) {
    if (fabs(v[ix]) > max) max = fabs(v[ix]);
  }
  for (iy = 1 + jy/2; iy < ny - 1; iy += jy) {
    for (ix = 1 + jx/2; ix < nx - 1; ix += jx) {
      if (max < 10.) fprintf(pgraf,"%5.2f\n",v[AI2]);
      else if (max < 100.) fprintf(pgraf,"%5.1f\n",v[AI2]);
      else fprintf(pgraf,"%5.0f\n",v[AI2]);
    }
  }
}
 
/* make a label */
label(nw,sx,sy,text)
int nw;
float sx,sy;
char *text;
{
  int i;
  char *temp;
/* remove newlines from the string */
  temp = text;
  while (*temp != '\0') {
    if ((*temp == '\n') || (*temp == '\r')) *temp = ' ';
    temp++;
  }
  fprintf(pgraf," b\n");
  fprintf(pgraf,"%d %g %g\n",nw,sx,sy);
  fprintf(pgraf,"%s\n",text);
}

/* make a label in perspective */
plabel(nw,sx,sy,sz,text)
int nw;
float sx,sy,sz;
char *text;
{
  char *temp;
/* remove newlines from the string */
  temp = text;
  while (*temp != '\0') {
    if ((*temp == '\n') || (*temp == '\r')) *temp = ' ';
    temp++;
  }
  fprintf(pgraf," e\n");
  fprintf(pgraf,"%d %g %g %g\n",nw,sx,sy,sz);
  fprintf(pgraf,"%s\n",text);
}
 
/* draw a line */
line(nw,ltype,num,x,y)
int nw,ltype,num;
float x[],y[];
{
  int i;
  fprintf(pgraf," l\n");
  fprintf(pgraf,"%d %d %d\n",nw,ltype,num);
  for (i = 0; i < num; i++)
    fprintf(pgraf,"%g %g\n",x[i],y[i]);
}

/* make a mark */
mark(nw,mtype,x,y,size)
int nw,mtype;
float x,y,size;
{
  fprintf(pgraf," m\n");
  fprintf(pgraf,"%d %d %g %g %g\n",nw,mtype,x,y,size);
}
 
/* scale the range of graphs */
scale(n,coarse,v,max,min)
int n;
float coarse,v[],*max,*min;
{
  float md();
  int i,j;
  j =1;
  *max = -LARGE;
  *min = LARGE;
  for (i = 0; i < n; i++) {
    if(fabs((v[i]))<baddata){ 
      if (v[i] > *max) *max = v[i];
      if (v[i] < *min) *min = v[i];
    }
    else j++;
  }
  if (md(*max,coarse) != 0.) {
    i = 1. + *max/coarse;
    *max = i*coarse;
  }
  if (md(*min,coarse) != 0.) {
    i = *min/coarse - 1.;
    *min = i*coarse;
  }  
  if (fabs(*min - *max) < SMALL) {
    *min -= coarse/2.;
    *max += coarse/2.;
  }
}

static float md(x,y)
float x,y;
{
  float b,a;
  long i;
  if (y != 0.) {
        b = x/y;
	if (fabs(b) > 1.e8) return(0.);
	i = x/y;
	a = x - y * i;
	return(a);
  }
  else  return(0.);
}

/* scale vectors */
vscale(nw,nx,ny,x,y,sx,sy)
int nx,ny;
float x[],y[],*sx,*sy;
{
  int ix,iy;
  float max,tm;
  max = -LARGE;
  for(ix=0;ix<nx;ix++){
    for(iy=0;iy<ny;iy++){
/* check for baddata problems */
      if((fabs(x[AI2])< baddata && fabs(y[AI2])< baddata)){
/* get magnitude of the vectors */
        tm = sqrt( x[AI2]*x[AI2] + y[AI2]*y[AI2]);
/* compare with the magnitude of the biggest one */
        if (tm>max) max = tm;
      }
    }
  }
 /* try to avoid deviding by 0 */
 if ( max < SMALL) max = SMALL;
 tm =sqrt((axh[nw]-axl[nw])*(axh[nw]-axl[nw])+(ayh[nw]-ayl[nw])
         *(ayh[nw]-ayl[nw]));
 *sx = *sy = tm/(VSCALE*sqrt(2.0)*max); 
 if(max == - LARGE) *sx = *sy = baddata;
 return(1);
} 

/* vector plot old method */
vector(nw,nx,ny,sx,sy,vx,vy)
int nw,nx,ny;
float sx,sy,vx[],vy[];
{
  int ix,iy;
  float xloc[XPT],yloc[YPT];
  if (nx > XPT || ny > YPT) {
    fprintf(stderr,"pgraf: data size %d > %d or %d > %d too big\n",
         nx,XPT,ny,YPT);
    exit(1);
  }
  if(nx ==1) xloc[0] = (axh[nw]+axl[nw])/2.;
  else for(ix = 0; ix<nx; ix++)
    xloc[ix] = (axh[nw]-axl[nw])*ix/(1.*nx - 1.) + axl[nw];
  if(ny ==1) yloc[0] = (ayh[nw]+ayl[nw])/2.;
  else for(iy = 0; iy<ny; iy++)
    yloc[iy] = (ayh[nw]-ayl[nw])*iy/(1.*ny - 1.) + ayl[nw];
  vector2(nw,nx,ny,sx,sy,vx,vy,xloc,yloc);
}

/* make a vector plot */
vector2(nw,nx,ny,sx,sy,vx,vy,xloc,yloc)
int nw,nx,ny;
float sx,sy,vx[],vy[],xloc[],yloc[];
{
  int ix,iy;
  fprintf(pgraf," v\n");
  fprintf(pgraf,"%d %d %d %g %g\n",nw,nx,ny,sx,sy);
  for(ix = 0; ix<nx; ix++)
    fprintf(pgraf,"%g\n",xloc[ix]);
  for(iy = 0; iy<ny;iy++)
    fprintf(pgraf,"%g\n",yloc[iy]);
  for (iy = 0; iy < ny; iy++) {
    for (ix = 0; ix < nx; ix++) {
      fprintf(pgraf,"%g %g\n",vx[AI2],vy[AI2]);
    }
  }
}
 
/* define a perspective view */
pview(nw,xp,yp,zp,xf,yf,zf,zoom)
int nw;
float xp,yp,zp,xf,yf,zf,zoom;
{
  fprintf(pgraf," p\n");
  fprintf(pgraf,"%d %g %g %g %g %g %g %g\n",nw,xp,yp,zp,xf,yf,zf,zoom);
}

/* draw 3-D axes in perspective */
paxes(nw,sxl,sxh,xstretch,xlabel,syl,syh,ystretch,ylabel,
     szl,szh,zstretch,zlabel)
int nw;
float sxl,sxh,xstretch;
char *xlabel;
float syl,syh,ystretch;
char *ylabel;
float szl,szh,zstretch;
char *zlabel;
{

/* write the stuff out */
  fprintf(pgraf," h\n");
  fprintf(pgraf,"%d %g %g %g %g %g %g %g %g %g\n",
       nw,sxl,sxh,xstretch,syl,syh,ystretch,szl,szh,zstretch);
  fprintf(pgraf,"%s\n",xlabel);
  fprintf(pgraf,"%s\n",ylabel);
  fprintf(pgraf,"%s\n",zlabel);
}

/* make a surface plot of a 2-D function */
psurface(nw,nx,ny,ltype,v,xvals,yvals)
int nw,nx,ny,ltype;
float *v,*xvals,*yvals;
{
  int ix,iy;
  fprintf(pgraf," j\n");
  fprintf(pgraf,"%d %d %d %d\n",nw,nx,ny,ltype);
  for (ix = 0; ix < nx; ix++) {
    fprintf(pgraf,"%g\n",xvals[ix]);
  }
  for (iy = 0; iy < ny; iy++) {
    fprintf(pgraf,"%g\n",yvals[iy]);
  }
  for (iy = 0; iy < ny; iy++) {
    for (ix = 0; ix < nx; ix++) {
      fprintf(pgraf,"%g\n",v[AI2]);
    }
  }
}

/* draw 3-D structure in perspective */
pline(nw,ltype,num,x,y,z)
int nw,ltype,num;
float x[],y[],z[];
{
  int l;
  fprintf(pgraf," s\n");
  fprintf(pgraf,"%d %d %d\n",nw,ltype,num);
  for (l = 0; l < num; l++)
    fprintf(pgraf,"%g %g %g\n",x[l],y[l],z[l]);
}

/* pconto -- generate a contour line in perspective */
#define SMALL 1.e-20
static float x[2],y[2],z[2],axa,aya,dx,dy,vc[5],px,py,denom;

pconto(nw,ltype,nx,ny,cval,v,orig,xa,ya)
int nw,ltype,nx,ny;
float cval,v[],orig[3],xa[3],ya[3];
{
  int ix,iy,i,index;
/* find lengths of axis vectors */
  axa = aya = 0.;
  for (i = 0; i < 3; i++) {
    axa += xa[i]*xa[i];
    aya += ya[i]*ya[i];
  }
  axa = sqrt(axa);
  aya = sqrt(aya);
/* find cell sizes */
  dx = axa/(nx - 1);
  dy = aya/(ny - 1);
/* loop on cells */
  for (ix = 1; ix < nx; ix++) {
    for (iy = 1; iy < ny; iy++) {
/* get corner values */
      vc[0] = vc[4] = v[(ix - 1)*ny + iy - 1];
      vc[1] = v[(ix - 1)*ny + iy];
      vc[2] = v[ix*ny + iy];
      vc[3] = v[ix*ny + iy - 1];
/* loop on cell sides */
      index = 0;
      for (i = 0; i < 4; i++) {
/* do we have a between? */
        if (((cval >= vc[i]) && (cval <= vc[i + 1])) ||
            ((cval <= vc[i]) && (cval >= vc[i + 1]))) {
/* if so, find its location on cell side */
          denom = vc[i + 1] - vc[i];
          if ((denom >= 0.) && (denom < SMALL)) denom = SMALL;
          if ((denom < 0.) && (denom > -SMALL)) denom = -SMALL;
          if (i == 0) {
            px = 0.;
            py = (cval - vc[0])/denom;
          }
          else if (i == 1) {
            px = (cval - vc[1])/denom;
            py = 1.;
          }
          else if (i == 2) {
            px = 1.;
            py = (vc[3] - cval)/denom;
          }
          else if (i == 3) {
            px = (vc[4] - cval)/denom;
            py = 0.;
          }
          px = dx*(px + ix - 1)/axa;
          py = dy*(py + iy - 1)/aya;
/* then find its position in 3-D space */
          x[index] = orig[0] + px*xa[0] + py*ya[0];
          y[index] = orig[1] + px*xa[1] + py*ya[1];
          z[index] = orig[2] + px*xa[2] + py*ya[2];
/* if its the first point, increment index */
          if (index == 0) index = 1;
/* otherwise, plot a line segment and reset index */
          else {
            index = 0;
            pline(nw,ltype,2,x,y,z);
          }
        }
      }
    }
  }
}
