/*	$NetBSD$	*/

/*-
 * Copyright (c) 2015 Taylor R. Campbell
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__RCSID("$NetBSD$");

#include "namespace.h"

#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/ioctl_pb.h>

#include <limits.h>
#include <stdlib.h>

#include <proplib/plistref.h>

#include <pb.h>
#include <pb_decode.h>
#include <pb_encode.h>
#include <pb_prop_decode.h>
#include <pb_prop_encode.h>

int
ioctl_pb_init(int fd, unsigned long cmd, struct pb_msg req,
    struct pb_msg res)
{

	pb_init(res);
	if (ioctl_pb(fd, cmd, req, res) == -1) {
		pb_destroy(res);
		return -1;
	}

	return 0;
}

int
ioctl_pb(int fd, unsigned long cmd, struct pb_msg req, struct pb_msg res)
{
	struct plistref pref;
	size_t size, buflen;
	uint64_t s64;
	unsigned char *buf = NULL;
	int error;

	error = pb_encoding_size(req, &size);
	if (error)
		goto out;
	s64 = size;

	if (round_page(size) > SIZE_MAX - round_page(PB_IOCTL_HDRSIZE)) {
		error = ENOMEM;
		goto out;
	}
	buflen = round_page(PB_IOCTL_HDRSIZE + size);
	buf = malloc(buf, buflen);
	if (buf == NULL) {
		error = ENOMEM;
		goto out;
	}

	error = pb_encode_to_memory(req, buf + PB_IOCTL_HDRSIZE, size);
	if (error)
		goto out;

retry:	(void)memcpy(buf, PB_IOCTL_HDR_REQ, PB_IOCTL_HDR_NTYPE);
	(void)memcpy(buf + PB_IOCTL_HDR_NTYPE, &s64, sizeof s64);
	pref.pref_plist = buf;
	pref.pref_len = buflen;
	if (ioctl(fd, cmd, &pref) == -1) {
		error = errno;
		goto out;
	}
	if (memcmp(buf, PB_IOCTL_HDR_TRUNC, PB_IOCTL_HDR_NTYPE) == 0) {
		uint64_t buflen0;
		void *buf0;

		(void)memcpy(&buflen0, buf + PB_IOCTL_HDR_NTYPE,
		    sizeof(buflen0));
		if (buflen0 < buflen) {
			error = EIO;
			goto out;
		}
		if (SIZE_MAX < buflen0) {
			error = ENOMEM;
			goto out;
		}
		buf0 = realloc(buf, buflen0);
		if (buf0 == NULL) {
			error = ENOMEM;
			goto out;
		}
		buf = buf0;
		buflen = buflen0;
		goto retry;
	} else if (memcmp(buf, PB_IOCTL_HDR_RES, PB_IOCTL_HDR_NTYPE) != 0) {
		error = EIO;
		goto out;
	}

	(void)memcpy(&s64, buf + PB_IOCTL_HDR_NTYPE, sizeof(s64));
	if (buflen < s64) {
		error = EIO;
		goto out;
	}
	size = s64;
	error = pb_decode_from_memory(res, buf + PB_IOCTL_HDRSIZE, size);
	if (error)
		goto out;

out:	if (buf)
		free(buf);
	if (error) {
		errno = error;
		return -1;
	}
	return 0;
}
