/*-
 * Copyright (c) 2015 Taylor R. Campbell
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <err.h>
#include <errno.h>
#include <inttypes.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <pb.h>
#include <pb_prop.h>
#include <pb_prop_encode.h>
#include <pb_prop_decode.h>
#include <prop/proplib.h>

#include "disk.pb.h"

static const struct pb_prop_record_field disk_info__geom__proprecord_fields[] = {
	{
		.pbprf_key = "sectors-per-unit",
		.pbprf_field = {
			.pbpf_msgdesc = &disk_info__geom__msgdesc,
			.pbpf_fieldno = 0,
		},
	},
	{
		.pbprf_key = "sector-size",
		.pbprf_field = {
			.pbpf_msgdesc = &disk_info__geom__msgdesc,
			.pbpf_fieldno = 1,
		},
	},
	{
		.pbprf_key = "sectors-per-track",
		.pbprf_field = {
			.pbpf_msgdesc = &disk_info__geom__msgdesc,
			.pbpf_fieldno = 2,
		},
	},
	{
		.pbprf_key = "tracks-per-cylinder",
		.pbprf_field = {
			.pbpf_msgdesc = &disk_info__geom__msgdesc,
			.pbpf_fieldno = 3,
		},
	},
	{
		.pbprf_key = "cylinders-per-unit",
		.pbprf_field = {
			.pbpf_msgdesc = &disk_info__geom__msgdesc,
			.pbpf_fieldno = 4,
		},
	},
};

static const struct pb_prop_msgdesc disk_info__geom__prop = {
	.pbpm_msgdesc = &disk_info__geom__msgdesc,
	.pbpm_t = PB_PROP_MSG_RECORD,
	.pbpm_u = { .record = {
		.pbpr_fields = disk_info__geom__proprecord_fields,
		.pbpr_nfields = sizeof(disk_info__geom__proprecord_fields) /
		    sizeof(disk_info__geom__proprecord_fields[0]),
	} },
};

static const struct pb_prop_record_field disk_info__proprecord_fields[] = {
	{
		.pbprf_key = "geometry",
		.pbprf_field = {
			.pbpf_msgdesc = &disk_info__msgdesc,
			.pbpf_fieldno = 0,
			.pbpf_u = { .message = &disk_info__geom__prop },
		},
	},
	{
		.pbprf_key = "type",
		.pbprf_field = {
			.pbpf_msgdesc = &disk_info__msgdesc,
			.pbpf_fieldno = 1,
		},
	},
};

static const struct pb_prop_msgdesc disk_info__prop = {
	.pbpm_msgdesc = &disk_info__msgdesc,
	.pbpm_t = PB_PROP_MSG_RECORD,
	.pbpm_u = { .record = {
		.pbpr_fields = disk_info__proprecord_fields,
		.pbpr_nfields = sizeof(disk_info__proprecord_fields) /
		    sizeof(disk_info__proprecord_fields[0]),
	} },
};

int
main(int argc, char **argv)
{
	prop_dictionary_t dict;
	prop_object_t obj;
	struct disk_info dkinfo;
	char *plist;
	int error;

	if (argc != 3 || strcmp(argv[1], "-d") != 0)
		errx(1, "Usage: %s -d <prop>\n", getprogname());

	dict = prop_dictionary_internalize_from_file(argv[2]);
	if (dict == NULL)
		errx(1, "failed to read prop dict");

	pb_init(disk_info(&dkinfo));
	error = pb_prop_decode(disk_info(&dkinfo), &disk_info__prop, dict);
	if (error) {
		errno = error;
		err(1, "decode prop dict");
	}

	(void)printf("disk_info {\n");
	(void)printf("  geom = geom {\n");
	(void)printf("    sectors-per-unit = 0x%"PRIx64"\n",
	    dkinfo.geometry.sectors_per_unit);
	(void)printf("    sector-size = 0x%"PRIx32"\n",
	    dkinfo.geometry.sector_size);
	if (dkinfo.geometry.sectors_per_track.present)
		(void)printf("    sectors-per-track = 0x%"PRIx32"\n",
		    dkinfo.geometry.sectors_per_track.value);
	if (dkinfo.geometry.tracks_per_cylinder.present)
		(void)printf("    tracks-per-cylinder = 0x%"PRIx32"\n",
		    dkinfo.geometry.tracks_per_cylinder.value);
	if (dkinfo.geometry.cylinders_per_unit.present)
		(void)printf("    cylinders-per-unit = 0x%"PRIx64"\n",
		    dkinfo.geometry.cylinders_per_unit.value);
	(void)printf("  }\n");
	if (dkinfo.type.present)
		(void)printf("  type = \"%s\"\n",
		    pb_string_ptr(&dkinfo.type.value));
	(void)printf("}\n");

	prop_object_release(dict);
	dict = NULL;

	error = pb_prop_encode(disk_info(&dkinfo), &disk_info__prop, &obj);
	if (error) {
		errno = error;
		err(1, "encode prop dict");
	}
	assert(prop_object_type(obj) == PROP_TYPE_DICTIONARY);
	dict = obj;
	plist = prop_dictionary_externalize(dict);
	if (plist == NULL)
		errx(1, "failed to encode prop dict");
	(void)printf("%s", plist);
	free(plist);

	prop_object_release(dict);
	pb_destroy(disk_info(&dkinfo));

	return 0;
}
