/*
 * sio_fs.h
 *
 * Filesystem-like API for SIO
 *
 * Authors: Khalil Amiri, CMU SCS/ECE, July 18 1997
 *          Sean Levy, CMU SCS, July 1999
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1996,1997,1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */

#ifndef __SIO_FS_H__
#define __SIO_FS_H__

#include <nasd/nasd_options.h>
#include <nasd/nasd_common.h>
#include <nasd/nasd_threadstuff.h>

/*
 * Basic system types.  under c9x, provided by inttypes.h.
 */
#if __alpha
typedef	long		int64m_t;
typedef	int		int32m_t;
typedef	unsigned int	u_int32m_t;
typedef	unsigned int	u_int32_t;
#define SIO_64d_FMT		"ld"
#define SIO_64u_FMT		"lu"
#define SIO_64x_FMT		"lx"
#define SIO_64X_FMT		"lX"
#define SIO_NASDID_FMT		"016lx"
#endif /* __alpha */

#if __i386__
typedef long long int	int64m_t;
typedef int		int32m_t;
typedef unsigned int	u_int32m_t;
#ifndef __GLIBC__
/* u_int32_t is defined in <sys/types.h> on glibc-based systems. */
typedef unsigned int	u_int32_t;
#endif /* __GLIBC__ */
#define SIO_64d_FMT		"Ld"
#define SIO_64u_FMT		"Lu"
#define SIO_64x_FMT		"Lx"
#define SIO_64X_FMT		"LX"
#define SIO_NASDID_FMT		"016Lx"
#endif /* __i386__ */

#ifndef TRUE
#define TRUE (1)
#endif /* TRUE */
#ifndef FALSE
#define FALSE (0)
#endif /* FALSE */

/*
 * SIO types
 */
typedef u_int32m_t	sio_async_flags_t;
typedef int		sio_async_handle_t;
typedef u_int32m_t	sio_caching_mode_t;
typedef u_int32m_t	sio_control_flags_t;
typedef u_int32m_t	sio_control_op_t;
typedef u_int32m_t	sio_count_t;
typedef u_int32m_t	sio_extension_id_t;
typedef u_int32m_t	sio_hint_class_t;
typedef u_int32m_t	sio_hint_flags_t;
typedef u_int32m_t	sio_layout_algorithm_t;
typedef u_int32m_t	sio_layout_flags_t;
typedef u_int32m_t	sio_mode_t;
typedef int64m_t	sio_offset_t;
typedef u_int32m_t	sio_return_t;
typedef int32m_t	sio_size_t;
typedef int64m_t	sio_transfer_len_t;

typedef struct {
  sio_transfer_len_t count;
  sio_return_t	status;
} sio_async_status_t;

typedef struct {
  sio_control_flags_t flags;
  sio_control_op_t op_code;
  void		*op_data;
  sio_return_t	result;
} sio_control_t;

typedef struct {
  sio_offset_t	offset;
  sio_size_t	size;
  sio_size_t	stride;
  sio_count_t	element_cnt;
} sio_file_io_list_t;

typedef struct {
  sio_hint_flags_t flag;
  sio_file_io_list_t *io_list;
  sio_count_t	list_len;
  void		*arg;
  sio_size_t	arg_len;
} sio_hint_t;

typedef struct {
  sio_size_t	size;
  void		*data;
} sio_label_t;

typedef struct {
  sio_layout_flags_t flags;
  sio_count_t	stripe_width;
  sio_size_t	stripe_depth;
  sio_layout_algorithm_t algorithm;
  void		*algorithm_data;
  /* XXX should have algorithm_data_len */
} sio_layout_t;

typedef struct {
  void		*addr;
  sio_size_t	size;
  sio_size_t	stride;
  sio_count_t	element_cnt;
} sio_mem_io_list_t;


/*
 * SIO return codes
 */

#define	SIO_SUCCESS				0

#define	SIO_ERR_ALREADY_EXISTS			1
#define	SIO_ERR_CONTROL_FAILED			2
#define	SIO_ERR_CONTROL_NOT_ATTEMPTED		3
#define	SIO_ERR_CONTROL_NOT_ON_TEST		4
#define	SIO_ERR_CONTROL_WOULD_HAVE_SUCCEEDED	5
#define	SIO_ERR_CONTROLS_CLASH			6
#define	SIO_ERR_FILE_NOT_FOUND			7
#define	SIO_ERR_FILE_OPEN			8
#define	SIO_ERR_INCORRECT_MODE			9
#define	SIO_ERR_INVALID_CLASS			10
#define	SIO_ERR_INVALID_DESCRIPTOR		11
#define	SIO_ERR_INVALID_EXTENSION		12
#define	SIO_ERR_INVALID_FILE_LIST		13
#define	SIO_ERR_INVALID_FILENAME		14
#define	SIO_ERR_INVALID_HANDLE			15
#define	SIO_ERR_INVALID_ITERATION		16
#define	SIO_ERR_INVALID_MEMORY_LIST		17
#define	SIO_ERR_INVALID_PARTICIPANT		18
#define	SIO_ERR_IO_CANCELED			19
#define	SIO_ERR_IO_FAILED			20
#define	SIO_ERR_IO_IN_PROGRESS			21
#define	SIO_ERR_MAX_ASYNC_OUTSTANDING_EXCEEDED	22
#define	SIO_ERR_MAX_COLL_ITERATIONS_EXCEEDED	23
#define	SIO_ERR_MAX_COLL_OUTSTANDING_EXCEEDED	24
#define	SIO_ERR_MAX_COLL_PARTICIPANTS_EXCEEDED	25
#define	SIO_ERR_MAX_OPEN_EXCEEDED		26
#define SIO_ERR_MIXED_COLL_AND_ASYNC		27
#define	SIO_ERR_NO_SPACE			28
#define	SIO_ERR_ONLY_AT_CREATE			29
#define	SIO_ERR_ONLY_AT_OPEN			30
#define	SIO_ERR_OP_UNSUPPORTED			31
#define	SIO_ERR_UNEQUAL_LISTS			32
#define	SIO_ERR_VEND_INVALID_FLAGS		33
#define	SIO_ERR_VEND_ASYNC_ALREADY_WAITING	34
#define	SIO_ERR_VEND_EOF                	35

#define	SIO_ERR_VEND_OSERR_FIRST		0x10000
#define	SIO_ERR_VEND_OSERR_LAST			0x1ffff

/*
 * SIO control operations
 */

#define	SIO_CTL_GetSize			1
#define	SIO_CTL_SetSize			2
#define	SIO_CTL_GetAllocation		3
#define	SIO_CTL_GetPreallocation	4
#define	SIO_CTL_SetPreallocation	5
#define	SIO_CTL_GetCachingMode		6
#define	SIO_CTL_SetCachingMode		7
#define	SIO_CTL_Propagate		8
#define	SIO_CTL_Refresh			9
#define	SIO_CTL_Sync			10
#define	SIO_CTL_GetLayout		11
#define	SIO_CTL_SetLayout		12
#define	SIO_CTL_GetLabel		13
#define	SIO_CTL_SetLabel		14
#define	SIO_CTL_GetConsistencyUnit	15


/*
 * SIO range constants
 */

#define SIO_MAX_ASYNC_OUTSTANDING	32
#define	SIO_MAX_COUNT			0xffffffff
#define	SIO_MAX_LABEL_LEN		1024
#define	SIO_MAX_NAME_LEN		255
#define	SIO_MAX_OFFSET			0x7fffffffffffffff
#define	SIO_MAX_OPEN			256
#define	SIO_MAX_SIZE			0x7fffffff
#define	SIO_MAX_TRANSFER_LEN		0x7fffffffffffffff

/*
 * Other SIO constants
 */

#define	SIO_ASYNC_BLOCKING		0x0001
#define	SIO_ASYNC_NONBLOCKING		0x0002

#define	SIO_ASYNC_DUMMY_HANDLE		0

#define	SIO_CACHING_NONE		0
#define	SIO_CACHING_STRONG		1
#define	SIO_CACHING_WEAK		2

#define	SIO_CONTROL_MANDATORY		0x0001
#define	SIO_CONTROL_OPTIONAL		0x0002

#define	SIO_EXT_ABSENT			0
#define	SIO_EXT_PRESENT			1
#define SIO_EXT_MAYBE			2

#define	SIO_HINT_CLASS_ORDERED		0
#define	SIO_HINT_CLASS_UNORDERED	1

#define	SIO_HINT_READ			0x0001
#define	SIO_HINT_WRITE			0x0002
#define	SIO_HINT_CANCEL_ALL		0x0004
#define	SIO_HINT_CANCEL_NEXT		0x0008
#define SIO_HINT_CANCEL_MATCHING	0x0010
#define	SIO_HINT_SEQUENTIAL		0x0020
#define	SIO_HINT_REVERSE		0x0040
#define	SIO_HINT_RANDOM_PARTIAL		0x0080
#define	SIO_HINT_RANDOM_COMPLETE	0x0100
#define	SIO_HINT_NO_FURTHER_USE		0x0200
#define	SIO_HINT_WILL_USE		0x0400

#define	SIO_LAYOUT_ALGORITHM_SIMPLE_STRIPING	0

#define	SIO_LAYOUT_WIDTH		0x0001
#define	SIO_LAYOUT_DEPTH		0x0002
#define	SIO_LAYOUT_ALGORITHM		0x0004

#define	SIO_MODE_READ			0x0001
#define	SIO_MODE_WRITE			0x0002
#define	SIO_MODE_CREATE			0x0004


/*
 * Function prototypes
 */

sio_return_t	sio_async_cancel_all(sio_async_handle_t *, sio_count_t);
sio_return_t	sio_async_sg_read(int, const sio_file_io_list_t *,
		    sio_count_t, const sio_mem_io_list_t *, sio_count_t,
		    sio_async_handle_t *);
sio_return_t	sio_async_sg_write(int, const sio_file_io_list_t *,
		    sio_count_t, const sio_mem_io_list_t *, sio_count_t,
		    sio_async_handle_t *);
sio_return_t	sio_async_status_any(sio_async_handle_t *, sio_count_t,
		    sio_count_t *, sio_async_status_t *, sio_async_flags_t);
sio_return_t	sio_close(int);
sio_return_t	sio_control(int, sio_control_t *, sio_count_t);
const char	*sio_error_string(sio_return_t);
sio_return_t	sio_hint(int, sio_hint_class_t,
		    const sio_hint_t *, sio_count_t);
sio_return_t	sio_hint_by_name(const char *, sio_hint_class_t,
		    const sio_hint_t *, sio_count_t);
sio_return_t	sio_open(int *, const char *, sio_mode_t, sio_control_t *,
		    sio_count_t);
sio_return_t	sio_query_extension(sio_extension_id_t);
sio_return_t	sio_rename(const char *, const char *);
sio_return_t	sio_sg_read(int, const sio_file_io_list_t *, sio_count_t,
		    const sio_mem_io_list_t *, sio_count_t,
		    sio_transfer_len_t *);
sio_return_t	sio_sg_write(int, const sio_file_io_list_t *, sio_count_t,
		    const sio_mem_io_list_t *, sio_count_t,
		    sio_transfer_len_t *);
sio_return_t	sio_test(const char *, sio_mode_t, sio_control_t *,
		    sio_count_t);
sio_return_t	sio_unlink(const char *);

#endif /* __SIO_FS_H__ */

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
