/*
 * nasd_edrfs_client_dce.c
 *
 * DCE dispatch functions for NASD EDRFS client
 *
 * Author: Jim Zelenka
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>

#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE

#include <dce/dce_error.h>
#include <nasd/nasd_freelist.h>
#include <nasd/nasd_edrfs_client.h>
#include <nasd/nasd_edrfs_client_dce.h>
#include <nasd/nasd_edrfs_server.h>
#include <nasd/nasd_error.h>
#include <nasd/nasd_edrfs_rpc.h>
#include <nasd/nasd_marshall.h>
#include <sys/errno.h>

#ifdef _KERNEL

#include <dce/ker/pthread_exc.h>

/* this doesnt exist in the kernel DCE header files for whatever reason */
#define exc_get_status(e,s) ( \
   (e)->kind == _exc_kind_status ? \
        (*(s) = (e)->match.value, 0) : \
        -1)

#endif /* _KERNEL */


/* pipe stuff */
void
nasd_edrfs_client_readdir_alloc(
  rpc_ss_pipe_state_t    raw_state,
  unsigned int           bsize,
  nasd_byte_t          **buf,
  unsigned int          *bcount)
{
  nasd_edrfs_readdir_pipe_state_t *state;

  state = (nasd_edrfs_readdir_pipe_state_t *)raw_state;

  /* we never alloc more than a single structure. such is life. */

  *buf = (nasd_byte_t *)&(state->partial_buf[state->partial_buf_offset]);
  *bcount =
    NASD_MIN(bsize, (NASD_EDRFS_DIRENT_T_MAX - state->partial_buf_offset));
  state->partial_buf_offset += *bcount;
}

void
nasd_edrfs_client_readdir_push(
  rpc_ss_pipe_state_t  *raw_state,
  idl_char             *buf,
  unsigned int          count)
{
  nasd_edrfs_readdir_pipe_state_t *state;

  state  = (nasd_edrfs_readdir_pipe_state_t *) raw_state;

  if (count == 0) {
    /* no bytes to grab. time to go home. */
    return;
  }

  /*
   * This code assumes we only get pushed buffers we allocated.
   * Verify that.
   */
{
  unsigned long buf_addr = (unsigned long)buf;
  unsigned long pb_addr = (unsigned long)&state->partial_buf[0];
  unsigned long pb_last = pb_addr + NASD_EDRFS_DIRENT_T_MAX - 1;

  NASD_ASSERT(buf_addr >= pb_addr);
  NASD_ASSERT(buf_addr <= pb_last);
}

  if (state->partial_buf_offset != NASD_EDRFS_DIRENT_T_MAX) {
    /* not a whole entry */
    return;
  }

  state->partial_buf_offset = 0;

  state->dir_ents_pushed++;

  if (state->dir_buf_ents_filled >= state->dir_buf_ents)
    return;

  nasd_edrfs_dirent_t_unmarshall(state->partial_buf,
    &state->dir_buf[state->dir_buf_ents_filled]);

  state->dir_buf_ents_filled++;
}



nasd_status_t
nasd_edrfscli_dce_init(
  nasd_shutdown_list_t  *sl)
{
  return(NASD_SUCCESS);
}


nasd_status_t
nasd_edrfsdce_bind(
  nasd_edrfs_handle_t   handle,
  char                 *server_name,
  char                 *portnum,
  int                   binding_type,
  void                 *binding_param,
  int                   binding_param_len)
{
  nasd_edrfscli_handle_dce_t *rpch;
  nasd_status_t rc;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  if (binding_type == NASD_BIND_DCE_DIRECT_TCP) {
    rc = nasd_dce_direct_bind_to_server((unsigned_char_t *)"ncadg_ip_tcp",
      (unsigned_char_t *)server_name, (unsigned_char_t *)portnum,
      &rpch->h);
  } else if (binding_type == NASD_BIND_DCE_DIRECT_UDP) {
    rc = nasd_dce_direct_bind_to_server((unsigned_char_t *)"ncadg_ip_udp",
      (unsigned_char_t *)server_name, (unsigned_char_t *)portnum,
      &rpch->h);
  }
  else {
    rc = NASD_BAD_HANDLE_TYPE;
  }
  if (rc) {
    return(rc);
  }

  return(NASD_SUCCESS);
}

nasd_status_t
nasd_edrfsdce_unbind(
  nasd_edrfs_handle_t  handle)
{
  nasd_edrfscli_handle_dce_t *rpch;
  int ret;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  ret = nasd_dce_direct_unbind_server(&rpch->h);
  if (ret)
    return(NASD_FAIL);

  return(NASD_SUCCESS);
}

void
nasd_edrfsdce_null(
  nasd_edrfs_handle_t   handle,
  nasd_res_t           *res,
  error_status_t       *op_status)
{
  nasd_edrfscli_handle_dce_t *rpch;
  nasd_res_otw_t res_otw;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_null_fm(rpch->h, res_otw, op_status);
    nasd_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_mount(
  nasd_edrfs_handle_t       handle,
  nasd_edrfs_mount_args_t  *args,
  nasd_edrfs_mount_res_t   *res,
  error_status_t           *op_status)
{
  nasd_edrfs_mount_args_otw_t args_otw;
  nasd_edrfs_mount_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  nasd_edrfs_mount_args_t_marshall(args, args_otw);

  TRY {
    nasd_mount_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_mount_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}



void
nasd_edrfsdce_fsstat(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_fsstat_args_t  *args,
  nasd_edrfs_fsstat_res_t   *res,
  error_status_t            *op_status)
{
  nasd_edrfs_fsstat_args_otw_t args_otw;
  nasd_edrfs_fsstat_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_fsstat_args_t_marshall(args, args_otw);
    nasd_fsstat_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_fsstat_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_fsinfo(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_fsinfo_args_t  *args,
  nasd_edrfs_fsinfo_res_t   *res,
  error_status_t            *op_status)
{
  nasd_edrfs_fsinfo_args_otw_t args_otw;
  nasd_edrfs_fsinfo_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_fsinfo_args_t_marshall(args, args_otw);
    nasd_fsinfo_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_fsinfo_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_lookup_otw_provided(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_lookup_args_t  *args,
  nasd_edrfs_lookup_res_t   *res,
  nasd_otw_base_t           *args_otw_p,
  nasd_otw_base_t           *res_otw_p,
  error_status_t            *op_status)
{
  nasd_edrfscli_handle_dce_t *rpch;

  nasd_edrfs_lookup_args_t_marshall(args, args_otw_p);

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_lookup_fm(rpch->h, args_otw_p, res_otw_p, op_status);
    nasd_edrfs_lookup_res_t_unmarshall(res_otw_p, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_lookup(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_lookup_args_t  *args,
  nasd_edrfs_lookup_res_t   *res,
  error_status_t            *op_status)
{
  nasd_edrfs_lookup_args_otw_t args_otw;
  nasd_edrfs_lookup_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  nasd_edrfs_lookup_args_t_marshall(args, args_otw);

  TRY {
    nasd_lookup_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_lookup_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_readdir(
  nasd_edrfs_handle_t         handle,
  nasd_edrfs_readdir_args_t  *args,
  nasd_edrfs_dirent_t        *out_entries,
  nasd_edrfs_readdir_res_t   *res,
  error_status_t             *op_status)
{
  nasd_edrfs_readdir_args_otw_t args_otw;
  nasd_edrfs_readdir_pipe_state_t statep;
  nasd_edrfs_readdir_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;
  nasd_byte_pipe_t pipe;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  statep.dir_buf = out_entries;
  statep.dir_buf_ents = args->in_count;
  statep.dir_buf_ents_filled = 0;
  statep.partial_buf_offset = 0;

  pipe.alloc = (void (*)())nasd_edrfs_client_readdir_alloc;
  pipe.push = (void (*)())nasd_edrfs_client_readdir_push;
  pipe.pull = NULL;
  pipe.state = (rpc_ss_pipe_state_t)&statep;

  nasd_edrfs_readdir_args_t_marshall(args, args_otw);

  TRY {
    nasd_readdir_fm(rpch->h, args_otw, pipe, res_otw, op_status);
    nasd_edrfs_readdir_res_t_unmarshall(res_otw, res);

    if (statep.partial_buf_offset != 0) {
      /* we have a partial buffer hanging around */
      res->nasd_status = NASD_BAD_TRANSFER_STATE;
    }
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_access(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_access_args_t  *args,
  nasd_edrfs_access_res_t   *res,
  error_status_t            *op_status)
{
  nasd_edrfs_access_args_otw_t args_otw;
  nasd_edrfs_access_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_access_args_t_marshall(args, args_otw);
    nasd_access_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_access_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_setattr(
  nasd_edrfs_handle_t         handle,
  nasd_edrfs_setattr_args_t  *args,
  nasd_edrfs_setattr_res_t   *res,
  error_status_t             *op_status)
{
  nasd_edrfs_setattr_args_otw_t args_otw;
  nasd_edrfs_setattr_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_setattr_args_t_marshall(args, args_otw);
    nasd_setattr_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_setattr_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_create_otw_provided(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_create_args_t  *args,
  nasd_edrfs_create_res_t   *res,
  nasd_otw_base_t           *args_otw_p,
  nasd_otw_base_t           *res_otw_p,
  error_status_t            *op_status)
{
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_create_args_t_marshall(args, args_otw_p);
    nasd_create_fm(rpch->h, args_otw_p, res_otw_p, op_status);
    nasd_edrfs_create_res_t_unmarshall(res_otw_p, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_create(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_create_args_t  *args,
  nasd_edrfs_create_res_t   *res,
  error_status_t            *op_status)
{
  nasd_edrfs_create_args_otw_t args_otw;
  nasd_edrfs_create_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_create_args_t_marshall(args, args_otw);
    nasd_create_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_create_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_symlink(
  nasd_edrfs_handle_t         handle,
  nasd_edrfs_symlink_args_t  *args,
  nasd_edrfs_symlink_res_t   *res,
  error_status_t             *op_status)
{
  nasd_edrfs_symlink_args_otw_t args_otw;
  nasd_edrfs_symlink_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_symlink_args_t_marshall(args, args_otw);
    nasd_symlink_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_symlink_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_remove(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_remove_args_t  *args,
  nasd_edrfs_remove_res_t   *res,
  error_status_t            *op_status)
{
  nasd_edrfs_remove_args_otw_t args_otw;
  nasd_edrfs_remove_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_remove_args_t_marshall(args, args_otw);
    nasd_remove_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_remove_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_mkdir(
  nasd_edrfs_handle_t       handle,
  nasd_edrfs_mkdir_args_t  *args,
  nasd_edrfs_mkdir_res_t   *res,
  error_status_t           *op_status)
{
  nasd_edrfs_mkdir_args_otw_t args_otw;
  nasd_edrfs_mkdir_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_mkdir_args_t_marshall(args, args_otw);
    nasd_mkdir_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_mkdir_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_rmdir(
  nasd_edrfs_handle_t       handle,
  nasd_edrfs_rmdir_args_t  *args,
  nasd_edrfs_rmdir_res_t   *res,
  error_status_t           *op_status)
{
  nasd_edrfs_rmdir_args_otw_t args_otw;
  nasd_edrfs_rmdir_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_rmdir_args_t_marshall(args, args_otw);
    nasd_rmdir_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_rmdir_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_newcookie(
  nasd_edrfs_handle_t           handle,
  nasd_edrfs_newcookie_args_t  *args,
  nasd_edrfs_newcookie_res_t   *res,
  error_status_t               *op_status)
{
  nasd_edrfs_newcookie_args_otw_t args_otw;
  nasd_edrfs_newcookie_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_newcookie_args_t_marshall(args, args_otw);
    nasd_newcookie_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_newcookie_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_rename(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_rename_args_t  *args,
  nasd_edrfs_rename_res_t   *res,
  error_status_t            *op_status)
{
  nasd_edrfs_rename_args_otw_t args_otw;
  nasd_edrfs_rename_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_edrfs_rename_args_t_marshall(args, args_otw);
    nasd_rename_fm(rpch->h, args_otw, res_otw, op_status);
    nasd_edrfs_rename_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}


void
nasd_edrfsdce_getstats(
  nasd_edrfs_handle_t         handle,
  nasd_edrfs_getstats_res_t  *res,
  error_status_t             *op_status)
{
  nasd_edrfs_getstats_res_otw_t res_otw;
  nasd_edrfscli_handle_dce_t *rpch;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_getstats_fm(rpch->h, res_otw, op_status);
    nasd_edrfs_getstats_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}

void
nasd_edrfsdce_resetstats(
  nasd_edrfs_handle_t   handle,
  nasd_res_t           *res,
  error_status_t       *op_status)
{
  nasd_edrfscli_handle_dce_t *rpch;
  nasd_res_otw_t res_otw;

  rpch = (nasd_edrfscli_handle_dce_t *)handle->rpc_specific_handle;

  TRY {
    nasd_resetstats_fm(rpch->h, res_otw, op_status);
    nasd_res_t_unmarshall(res_otw, res);
  }
  CATCH_ALL {
    res->nasd_status = NASD_RPC_TRAP;
    exc_get_status(THIS_CATCH,(int *)op_status);
  } ENDTRY
}

void
nasd_edrfscli_dce_error_string(
  nasd_edrfs_handle_t     handle,
  nasd_rpc_status_t       status,
  nasd_error_string_t     str,
  char                   *file,
  int                     line)
{
  int dce_ps;

  dce_error_inq_text(status, (unsigned char *)str, &dce_ps);
  if (dce_ps) {
    sprintf(str, "Unknown DCE status 0x%x", status);
  }
}

nasd_edrfscli_rpcmod_tab_t nasd_edrfscli_dce_udp_mod = {
  NULL,
  NASD_RPCMOD_INIT_FAIL,

  nasd_edrfsdce_bind,
  nasd_edrfsdce_unbind,

  nasd_edrfsdce_null,
  nasd_edrfsdce_mount,
  nasd_edrfsdce_fsstat,
  nasd_edrfsdce_fsinfo,
  nasd_edrfsdce_lookup,
  nasd_edrfsdce_readdir,
  nasd_edrfsdce_access,
  nasd_edrfsdce_setattr,
  nasd_edrfsdce_create,
  nasd_edrfsdce_symlink,
  nasd_edrfsdce_remove,
  nasd_edrfsdce_mkdir,
  nasd_edrfsdce_rmdir,
  nasd_edrfsdce_newcookie,
  nasd_edrfsdce_rename,
  nasd_edrfsdce_getstats,
  nasd_edrfsdce_resetstats,

  nasd_edrfsdce_lookup_otw_provided,
  nasd_edrfsdce_create_otw_provided,

  nasd_edrfscli_dce_error_string,

  NASD_BIND_DCE_DIRECT_UDP,
  "DCE-UDP",
  sizeof(nasd_edrfscli_handle_dce_t),

  NULL
};

nasd_edrfscli_rpcmod_tab_t nasd_edrfscli_dce_tcp_mod = {
  NULL,
  NASD_RPCMOD_INIT_FAIL,

  nasd_edrfsdce_bind,
  nasd_edrfsdce_unbind,

  nasd_edrfsdce_null,
  nasd_edrfsdce_mount,
  nasd_edrfsdce_fsstat,
  nasd_edrfsdce_fsinfo,
  nasd_edrfsdce_lookup,
  nasd_edrfsdce_readdir,
  nasd_edrfsdce_access,
  nasd_edrfsdce_setattr,
  nasd_edrfsdce_create,
  nasd_edrfsdce_symlink,
  nasd_edrfsdce_remove,
  nasd_edrfsdce_mkdir,
  nasd_edrfsdce_rmdir,
  nasd_edrfsdce_newcookie,
  nasd_edrfsdce_rename,
  nasd_edrfsdce_getstats,
  nasd_edrfsdce_resetstats,

  nasd_edrfsdce_lookup_otw_provided,
  nasd_edrfsdce_create_otw_provided,

  nasd_edrfscli_dce_error_string,

  NASD_BIND_DCE_DIRECT_TCP,
  "DCE-TCP",
  sizeof(nasd_edrfscli_handle_dce_t),

  NULL
};

void
nasd_edrfscli_dce_register(
  int  counter_lock_held)
{
  nasd_edrfscli_mod_register(&nasd_edrfscli_dce_udp_mod, counter_lock_held);
  nasd_edrfscli_mod_register(&nasd_edrfscli_dce_tcp_mod, counter_lock_held);
}

#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
