/*
 * nasd_edrfs_dumpattr.c
 *
 * Read a NASD object's attributes and interpret them for EDRFS
 *
 * Author: Nat Lanza
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <nasd/nasd_pdrive.h>
#include <nasd/nasd_pdrive_client.h>
#include <nasd/nasd_pdrive_client_kpdev.h>
#include <nasd/nasd_security.h>
#include <nasd/nasd_edrfs_types.h>
#include <nasd/nasd_edrfs_types_marshall.h>
#include <nasd/nasd_mem.h>

#include <pwd.h>
#include <grp.h>
#include <sys/types.h>

char *progname;

int                        binding_type;
int                        binding_args_len;
void                      *binding_args;
char                      *binding_port = NASD_PDRIVE_PORT;
nasd_drive_param_kpdev_t   kpdev_args;
int                        nondefault_binding = 0;

nasd_sec_keyring_t         keys;
nasd_error_string_t        error_text;

void usage() {
  int i;

  fprintf(stderr, "usage: %s [options] servername ppartnum nasdid master_password\n", progname);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -t don't print NASD stats\n");
  fprintf(stderr, "  -k use kernel device\n");
  fprintf(stderr, "  -l use colocated drive\n");
  fprintf(stderr, "  -M use message queues\n");
  fprintf(stderr, "  -s security_level\n");
  for(i = 0; i <= NASD_MAX_SECURITY_LEVEL; i++) {
    fprintf(stderr, "     %d %s\n", i, nasd_sec_level_string(i));
  }
  
  fflush(stderr);
  exit(1);
}


int main(int argc, char **argv) {
  nasd_p_getattr_dr_args_t  args;
  nasd_p_getattr_dr_res_t   res;
  nasd_attribute_t         *attr;
  nasd_edrfs_attributes_t   edrfsattr;
  nasd_security_param_t     security_param;
  nasd_cookie_t             cookie;
  nasd_sec_keyring_t        keys;
  nasd_rpc_status_t         op_status;
  nasd_status_t rc;
  nasd_drive_handle_t       handle;
  nasd_timespec_t           tm;
  nasd_uint16               protection;
  char                     *server_name;
  char                     *master_password;
  int                       security_level = 0;
  int                       partnum;
  char                      c;
  struct passwd            *passwd;
  struct group             *group;
  int                      printnasd = 1;

  progname = argv[0];

  binding_type = NASD_BIND_DEFAULT;
  binding_args = NULL;
  binding_args_len = 0;

  while (nasd_getopt(argc, argv, "klMs:t", &c)) {
    switch (c) {
    case 't':
      printnasd = 0;
      break;
    case 'k':
      if (nondefault_binding) { usage(); }
      nondefault_binding = 1;
      binding_type = NASD_BIND_KPDEV_DEFAULT;
      binding_args = &kpdev_args;
      binding_args_len = sizeof(kpdev_args);
      strcpy(kpdev_args.devname, "/dev/nasdkp0");
      break;
    case 'l':
      if (nondefault_binding) { usage(); }
      nondefault_binding = 1;
      binding_type = NASD_BIND_COLOCATE;
      binding_args = &kpdev_args;
      binding_args_len = sizeof(kpdev_args);
      strcpy(kpdev_args.devname, "/dev/nasdkp0");
      break;
    case 'M':
      if (nondefault_binding) { usage(); }
      nondefault_binding = 1;
      binding_type = NASD_BIND_MSGQ;
      break;
    case 's':
      if (sscanf(nasd_optarg, "%d", &security_level) != 1) {
	usage();
      }
      break;
    default:
      fprintf(stderr, "Unknown option '%c'\n", nasd_optopt);
      usage();
    }
  }
  
  if (nasd_optind >= argc) { usage(); }
  server_name = argv[nasd_optind];
  nasd_optind++;

  memset((char *) &args, 0, sizeof(nasd_p_getattr_dr_args_t));

  if (nasd_optind >= argc) { usage(); }
  if (sscanf(argv[nasd_optind], "%d", &partnum) != 1) { usage(); }
  nasd_optind++;
  args.in_partnum = partnum;

  if (nasd_optind >= argc) { usage(); }
  rc = nasd_str_to_nasd_id(argv[nasd_optind], &args.in_identifier);
  nasd_optind++;
  if (rc != NASD_SUCCESS) { usage(); }

  if (nasd_optind >= argc) { usage(); }
    master_password=argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind < argc) { usage(); }

  rc = nasd_sec_seclevel_to_protection(security_level, &protection);
  if (rc != NASD_SUCCESS) {
    fprintf(stderr, "ERROR: invalid security level %d\n",
	    security_level);
    usage();
  }

  rc = nasd_cl_p_init();
  if (rc != NASD_SUCCESS) {
    fprintf(stderr, "ERROR: cannot init client library, rc=0x%x (%s)\n",
	    rc, nasd_error_string(rc));
    exit(1);
  }
  
  rc = nasd_bind_to_drive(server_name, binding_port, binding_type, 
			  binding_args, binding_args_len, &handle);
  if (rc != NASD_SUCCESS) {
    fprintf(stderr, "ERROR: cannot bind to server %s error 0x%x (%s)\n",
	    server_name, rc, nasd_error_string(rc));
    fflush(stderr);
    exit(1);
  }

  /* generate capability */
  nasd_sec_password_to_keys(master_password, args.in_partnum, &keys);
  nasd_drive_handle_get_time(handle, &tm);
  tm.ts_sec += (60*60);
  nasd_sec_build_capability(args.in_partnum, args.in_identifier,
                            NASD_ACCESS_RIGHTS_GETATTR, 0, tm.ts_sec,
                            protection, NASD_BLACK_CAPABILITY, 0, 0, 7,
                            keys.black_key, &cookie);

  /* fill in security param */
  security_param.type = NASD_BLACK_CAPABILITY;
  security_param.partnum = args.in_partnum;
  security_param.actual_protection = protection;

  /* do RPC */
  nasd_cl_p_getattr_dr(handle, cookie.key, &security_param,
		       &cookie.capability, &args, &res, &op_status);
  rc = res.nasd_status;

  if ((rc != NASD_SUCCESS) || (op_status != 0)) {
    fprintf(stderr, "ERROR: getattr went bad, rc=0x%x (%s) op_status=0x%x (%s)\n",
	    rc, nasd_error_string(rc), op_status,
	    nasd_cl_error_string(handle, op_status, error_text));
    exit(1);
  }

  attr = &res.out_attribute;
  nasd_edrfs_attributes_t_unmarshall((nasd_otw_base_t *) attr->fs_specific,
                                     &edrfsattr);

  if ( printnasd ) {
    nasd_printf("Object ID: 0x%" NASD_ID_FMT "\n", args.in_identifier);
    nasd_printf("\nNASD Attributes:\n");
    nasd_printstat(attr);
  }

  nasd_printf("\nEDRFS Attributes:\n");

  switch (edrfsattr.type) {
  case NASD_EDRFS_TYPE_DIR:
    printf("  type:  DIR\n");
    break;
  case NASD_EDRFS_TYPE_REG:
    printf("  type:  FILE\n");
    break;
  case NASD_EDRFS_TYPE_LNK:
    printf("  type:  LINK\n");
    break;
  default:
    printf("  type:  ???? (%d)\n", edrfsattr.type);
  }

  printf("  mode:  %06o\n", edrfsattr.mode);
  printf("  nlink: %d\n", edrfsattr.nlink);

  passwd = getpwuid(edrfsattr.uid);
  group  = getgrgid(edrfsattr.gid);

  printf("  uid:   %d (%s)\n", edrfsattr.uid,
	 (passwd == NULL)? "????" : passwd->pw_name);

  printf("  gid:   %d (%s)\n", edrfsattr.gid,
	 (group == NULL)? "????" : group->gr_name);

  rc = nasd_unbind_drive(&handle);
  if (rc != NASD_SUCCESS) {
    fprintf(stderr, "ERROR: nasd_unbind_drive failed, rc=0x%x (%s)\n",
	    rc, nasd_error_string(rc));
    fflush(stderr);
    exit(2);
  }

  printf("  clean: %d\n", edrfsattr.clean);

  nasd_cl_p_shutdown();
  exit(0);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
