/*
 * nasd_edrfs_tee.c
 *
 * Read from standard input, and write to standard output and an EDRFS
 * object. If no object is specified, we create a new object and return its
 * ID to standard output before echoing the input.
 *
 * Author: Ted Wong */
/*
 * Copyright (c) of Carnegie Mellon University 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <string.h>
#include <libgen.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <nasd/nasd_pdrive.h>
#include <nasd/nasd_pdrive_client.h>
#include <nasd/nasd_pdrive_client_kpdev.h>
#include <nasd/nasd_security.h>
#include <nasd/nasd_edrfs_client.h>
#include <nasd/nasd_mem.h>

#define TEE_BUFFER_LEN 4096

typedef struct {
  nasd_edrfs_handle_t server_h;
  char *server_name;

  nasd_drive_handle_t drive_h;
  char drive_name[64];
  char drive_port[16];

  int binding_type;
  void *binding_param;
  int binding_param_len;

  char *root_path;
  nasd_edrfs_identifier_t root_id;
  nasd_edrfs_credential_t root_credential;
  nasd_cookie_t root_cookie;
} tee_mount_info_t;

typedef struct {
  char *fullname;
  char *name;
  nasd_edrfs_identifier_t dir_id;
  nasd_edrfs_identifier_t id;
  nasd_attribute_t attr;
  nasd_cookie_t cookie;
} tee_file_info_t;

char *progname;

nasd_drive_param_kpdev_t   kpdev_args;

void
tee_shutdown(tee_mount_info_t *m_p)
{
  nasd_status_t rc;

  /* Clean up and exit. */

  if (m_p->drive_h != NULL && (rc = nasd_unbind_drive(&m_p->drive_h))) {
    fprintf(stderr, "ERROR: failed unbinding from drive %s, rc=0x%x (%s)\n",
            m_p->drive_name, rc, nasd_error_string(rc));
  }
  if (m_p->server_h != NULL &&
      (rc = nasd_unbind_edrfs_server(&m_p->server_h))) {
    fprintf(stderr, "ERROR: failed unbinding from server %s, rc=0x%x (%s)\n",
            m_p->server_name, rc, nasd_error_string(rc));
  }

  nasd_edrfscli_shutdown();
  nasd_cl_p_shutdown();
  fflush(stderr);
}

void
usage()
{
  int i;
  fprintf(stderr, "USAGE: %s [options] servername filename\n", progname);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -k : use kernel device\n");
  fprintf(stderr, "  -r : read back what gets written out to disk\n");
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
  fprintf(stderr, "  -T use DCE-TCP\n");
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
  fprintf(stderr, "  -v : be verbose\n");
  fflush(stderr);
  exit(1);
}

int
main(
     int     argc,
     char  **argv)
{
  int new_file_flag = 0;
  int nondefault_binding = 0;
  int read_flag = 0;
  int verbose_flag = 0;
  char name[1024], c;
  char *p;
  nasd_edrfscli_error_string_t edrfserr_str;
  nasd_rpc_status_t status;
  nasd_status_t rc;
  nasd_uint16 protection;
  tee_file_info_t finfo;
  tee_mount_info_t minfo;

  progname = basename(argv[0]);

  minfo.server_h = NULL;
  minfo.drive_h = NULL;

  minfo.binding_type = NASD_BIND_DEFAULT;
  minfo.binding_param = NULL;
  minfo.binding_param_len = 0;

  minfo.root_credential.uid = 0;
  minfo.root_credential.gid = 0;

  while (nasd_getopt(argc, argv, "krTv", &c)) {
    switch(c) {
    case 'k':
      if (nondefault_binding) {
        usage();
      }
      nondefault_binding = 1;
      minfo.binding_type = NASD_BIND_KPDEV_DEFAULT;
      minfo.binding_param = &kpdev_args;
      minfo.binding_param_len = sizeof(kpdev_args);
      strcpy(kpdev_args.devname, "/dev/nasdkp0");
      break;
    case 'r':
      read_flag = 1;
      break;
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
    case 'T':
      if (nondefault_binding) {
        usage();
      }
      nondefault_binding = 1;
      minfo.binding_type = NASD_BIND_DCE_DIRECT_TCP;
      break;
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
    case 'v':
      verbose_flag = 1;
      break;
    default:
      fprintf(stderr, "Unknown option '%c'\n", c);
      usage();
    }
  }

  if (nasd_optind >= argc) {
    usage();
  }
  minfo.server_name = argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind >= argc) {
    usage();
  }
  strncpy(name, argv[nasd_optind], 1024);
  finfo.fullname = argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind < argc) {
    usage();
  }

  /* Initialize drive and file system clients. */

  rc = nasd_cl_p_init();
  if (rc) {
    fprintf(stderr, "ERROR: cannot init client library, rc=0x%x (%s)\n",
            rc, nasd_error_string(rc));
    exit(1);
  }

  rc = nasd_edrfscli_init();
  if (rc) {
    fprintf(stderr, "ERROR: cannot init EDRFS client library, rc=0x%x (%s)\n",
            rc, nasd_error_string(rc));
    exit(1);
  }

  rc = nasd_bind_to_edrfs_server(minfo.server_name,
                                 NASD_EDRFS_SERVER_PORT,
                                 minfo.binding_type,
                                 minfo.binding_param,
                                 minfo.binding_param_len,
                                 &minfo.server_h);
  if (rc) {
    fprintf(stderr, "ERROR: failed binding to EDRFS server %s, rc=0x%x (%s)\n",
            minfo.server_name, rc, nasd_error_string(rc));
    exit(1);
  }

  /* Mount the file system. */

  p = name;
  if (*p != '/') {
    fprintf(stderr, "ERROR: file name must have a fully-specified path\n");
    tee_shutdown(&minfo);
    exit(1);
  }
  p = strtok(p + 1, "/");
  if (p == NULL) {
    fprintf(stderr, "ERROR: file name must include the filesystem name\n");
    tee_shutdown(&minfo);
    exit(1);
  }
  p--;
  minfo.root_path = p;
  {
    struct in_addr drive_addr;
    nasd_edrfs_mount_args_t args;
    nasd_edrfs_mount_res_t res;

    args.in_credential = minfo.root_credential;
    strncpy(args.in_dirpath, minfo.root_path, NASD_EDRFS_MAX_NAME_LEN);
    nasd_edrfscli_mount(minfo.server_h, &args, &res, &status);
    if (status || res.nasd_status) {
      fprintf(stderr,
              "ERROR: failed mounting EDRFS, "
              "rc=0x%x (%s) nasd_status=0x%x (%s)\n",
              status, nasd_error_string(status),
              res.nasd_status, nasd_error_string(res.nasd_status));
      tee_shutdown(&minfo);
      exit(1);
    }
    minfo.root_id = res.out_identifier;
    minfo.root_cookie = res.out_cookie;

    if (verbose_flag) {
      printf("Mounted root dir %s, ID is 0x%" NASD_ID_FMT "\n",
             minfo.root_path, minfo.root_id);
    }

    /* Get a drive handle to do writes. */

    drive_addr.s_addr = res.out_drivelist[0].network_address;
    strncpy(minfo.drive_name, inet_ntoa(drive_addr), sizeof(minfo.drive_name));
    sprintf(minfo.drive_port, "%u", res.out_drivelist[0].port_number);
    rc = nasd_bind_to_drive(minfo.drive_name,
                            minfo.drive_port,
                            minfo.binding_type,
                            minfo.binding_param,
                            minfo.binding_param_len,
                            &minfo.drive_h);
    if (rc) {
      fprintf(stderr,
              "ERROR: failed binding to drive %s port %s, rc=0x%x (%s)\n",
               minfo.drive_name, minfo.drive_port, rc, nasd_error_string(rc));
      tee_shutdown(&minfo);
      exit(1);
    }

    if (verbose_flag) {
      printf("Mounted drive %s port %s\n", minfo.drive_name, minfo.drive_port);
    }
  }

  /* Walk the directory tree. */

  {
    char *last_path = NULL;
    nasd_cookie_t cookie = minfo.root_cookie;
    nasd_edrfs_identifier_t dir_id = minfo.root_id;
    nasd_cookie_t last_cookie;
    nasd_edrfs_identifier_t last_dir_id;

    while((p = strtok(NULL, "/"))) {
      nasd_edrfs_lookup_args_t args;
      nasd_edrfs_lookup_res_t res;

      args.in_cookie = cookie;
      args.in_identifier = dir_id;
      args.in_credential = minfo.root_credential;
      strcpy(args.in_dirpath, p);
      nasd_edrfscli_lookup(minfo.server_h, &args, &res, &status);

      if(status ||
         (res.nasd_status && res.nasd_status != NASD_EDRFS_BAD_NAME) ||
         (res.nasd_status == NASD_EDRFS_BAD_NAME && new_file_flag)) {
        fprintf(stderr, "ERROR: lookup %s in dir 0x%" NASD_ID_FMT
                ": status=0x%x (%s) nasd_status=0x%x (%s)\n",
                p, dir_id.nasd_identifier,
                status, nasd_edrfscli_error_string(minfo.server_h, status, edrfserr_str),
                res.nasd_status, nasd_error_string(res.nasd_status));
        tee_shutdown(&minfo);
        exit(1);
      }
      else if (res.nasd_status == NASD_EDRFS_BAD_NAME) {
        new_file_flag = 1;
      }
      if(verbose_flag) {
        printf("lookup %s in dir 0x%" NASD_ID_FMT " successful\n", p,
               dir_id.nasd_identifier);
      }
      last_path = p;
      last_dir_id = dir_id;
      last_cookie = cookie;
      cookie = res.out_cookie;
      dir_id = res.out_identifier;
    }
    if (last_path == NULL) {
      fprintf(stderr, "ERROR: unable to parse path \"%s\" for last element\n",
              finfo.fullname);
      fflush(stderr);
      exit(1);
    }
    finfo.name = last_path;
    if (!new_file_flag) {
      finfo.id = dir_id;
    }
    finfo.dir_id = last_dir_id;
    finfo.cookie = last_cookie;
  }

  /* If the file is not new, truncate it by removing it (ugh). */

  if (!new_file_flag) {
    nasd_edrfs_remove_args_t args;
    nasd_edrfs_remove_res_t res;

    args.in_directory = finfo.dir_id;
    args.in_credential = minfo.root_credential;
    args.in_cookie = minfo.root_cookie;
    strncpy(args.in_dirpath, finfo.name, NASD_EDRFS_MAX_NAME_LEN);

    nasd_edrfscli_remove(minfo.server_h, &args, &res, &status);
    if (status || res.nasd_status) {
      fprintf(stderr,
              "ERROR: failed truncating (removing) file %s, "
              "rc=0x%x (%s) nasd_status=0x%x (%s)\n",
              finfo.name,
              status, nasd_error_string(status),
              res.nasd_status, nasd_error_string(res.nasd_status));
      tee_shutdown(&minfo);
      exit(1);
    }
  }

  {
    nasd_edrfs_create_args_t args;
    nasd_edrfs_create_res_t res;

    args.in_directory = finfo.dir_id;
    args.in_credential = minfo.root_credential;
    args.in_cookie = minfo.root_cookie;
    strncpy(args.in_dirpath, finfo.name, NASD_EDRFS_MAX_NAME_LEN);

    nasd_edrfscli_create(minfo.server_h, &args, &res, &status);
    if (status || res.nasd_status) {
      fprintf(stderr,
              "ERROR: failed creating file %s, "
              "rc=0x%x (%s) nasd_status=0x%x (%s)\n",
              finfo.name,
              status, nasd_error_string(status),
              res.nasd_status, nasd_error_string(res.nasd_status));
      tee_shutdown(&minfo);
      exit(1);
    }
    finfo.id = res.out_identifier;
    finfo.attr = res.out_attribute;
    finfo.cookie = res.out_cookie;
  }

  /* OK, we now have a valid file (new or otherwise). Start copying input
     from stdin into the file and to stdout. */

  if (verbose_flag) {
    printf("File name is %s, ID is 0x%" NASD_ID_FMT "\n",
           finfo.name, finfo.id);
  }

  {
    off_t off = 0;

    while (!feof(stdin)) {
      char buffer[TEE_BUFFER_LEN];
      nasd_len_t len_wrt_to_file;
      nasd_security_param_t sp;
      size_t len_rd, len_wrt_to_stdout;

      len_rd = fread(buffer, sizeof(char), sizeof(buffer), stdin);
      if (len_rd < TEE_BUFFER_LEN &&
          !feof(stdin)) {
        fprintf(stderr,
                "ERROR: failed reading from stdin, errno=0x%x (%s) \n",
                errno, strerror(errno));
        tee_shutdown(&minfo);
        exit(1);
      }

      sp.type = finfo.cookie.capability.type;
      sp.partnum = finfo.id.partnum;
      sp.actual_protection = finfo.cookie.capability.min_protection;

      {
        nasd_p_smpl_op_dr_args_t args;
        nasd_p_fastwrite_dr_res_t res;

        args.in_identifier = finfo.id.nasd_identifier;
        args.in_offset = off;
        args.in_len = len_rd;
        args.in_partnum = finfo.id.partnum;

        nasd_cl_p_write_simple_dr(minfo.drive_h,
                                  finfo.cookie.key,
                                  &sp,
                                  &finfo.cookie.capability,
                                  &args,
                                  buffer,
                                  &res,
                                  &status);
        if (status || res.nasd_status) {
          fprintf(stderr,
                  "ERROR: failed writing to %s obj 0x%" NASD_ID_FMT ", "
                  "rc=0x%x (%s) nasd_status=0x%x (%s)\n",
                  finfo.fullname, finfo.id.nasd_identifier,
                  status, nasd_error_string(status),
                  res.nasd_status, nasd_error_string(res.nasd_status));
          tee_shutdown(&minfo);
          exit(1);
        }
        len_wrt_to_file = res.out_datalen;
      }

      /* See if we can read back what we just wrote out! */

      if (read_flag) {
        nasd_p_smpl_op_dr_args_t args;
        nasd_p_fastread_dr_res_t res;
        nasd_security_param_t sp;

        args.in_identifier = finfo.id.nasd_identifier;
        args.in_offset = off;
        args.in_len = sizeof(buffer);
        args.in_partnum = finfo.id.partnum;

        nasd_cl_p_read_simple_dr(minfo.drive_h,
                                 finfo.cookie.key,
                                 &sp,
                                 &finfo.cookie.capability,
                                 &args,
                                 buffer,
                                 &res,
                                 &status);
        if (status || res.nasd_status) {
          fprintf(stderr,
                  "ERROR: failed reading from file %s obj 0x%" NASD_ID_FMT ", "
                  "rc=0x%x (%s) nasd_status=0x%x (%s)\n",
                  finfo.fullname, finfo.id.nasd_identifier,
                  status, nasd_error_string(status),
                  res.nasd_status, nasd_error_string(res.nasd_status));
          tee_shutdown(&minfo);
          exit(1);
        }
        if (res.out_datalen != len_wrt_to_file) {
          fprintf(stderr,
                  "ERROR: failed reading from file: non-matching lengths\n");
          tee_shutdown(&minfo);
          exit(1);
        }
      }

      off += len_rd;

      len_wrt_to_stdout = fwrite(buffer, sizeof(char), len_rd, stdout);
      if (len_wrt_to_stdout != len_rd) {
        fprintf(stderr,
                "ERROR: failed writing to stdout, errno=0x%x (%s) \n",
                errno, strerror(errno));
        tee_shutdown(&minfo);
        exit(1);
      }
    }
  }

  tee_shutdown(&minfo);

  exit(0);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
