/*
 * pcrmpdrspeed.c
 *
 * Create the objects necessary for an mpdrspeed run.
 *
 * pcrmpdrspeed creates a tiny filesystem in a
 * partition.  The filesystem consists of a header with a magic number,
 * followed by a list of the object ID's in the "filesystem".  
 * 
 * This filesystem is used in the "-m" option of mpdrspeed (where we test 
 * multiple clients reading from different objects so that one client's
 * reads do not populate the cache for other clients)
 *
 * Author: David Rochberg
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1997,1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <nasd/nasd_drive_types.h>
#include <nasd/nasd_pdrive.h>
#include <nasd/nasd_pdrive_client.h>
#include <nasd/nasd_security.h>
#include <nasd/nasd_mem.h>

#include <nasd/mpdrspeed_common.h>


nasd_error_string_t error_text;
char *progname;
nasd_sec_keyring_t keys;

int	size = 1048576;
int	nobjects = 16;
char	*buf;

#if NASD_SECURITY_ON > 0
/* protection level for all operations */
#define SECURITY_PROTECTION	NASD_SECURE_OPS

#define BUILD_CAPABILITY(_h_,_powers_,_partnum_,_ni_,_sp_,_cookie_)	\
{									\
  nasd_timespec_t __tm;							\
  nasd_drive_handle_get_time(_h_,&__tm);				\
  __tm.ts_sec+=(60*60);							\
  nasd_sec_build_capability(partnum,_ni_,_powers_,0,__tm.ts_sec,	\
                            SECURITY_PROTECTION,NASD_BLACK_CAPABILITY,	\
			    0,0,7,					\
                            keys.black_key,_cookie_);			\
  (_sp_)->type = NASD_BLACK_CAPABILITY;		 			\
  (_sp_)->partnum = _partnum_;						\
  (_sp_)->actual_protection = SECURITY_PROTECTION;			\
}
#else /* NASD_SECURITY_ON > 0 */
#define BUILD_CAPABILITY(_h_,_args_,_powers_,_partnum_,_sp_,_cookie_)
#define SECURITY_PROTECTION	NASD_NO_PROTECTION
#endif /* NASD_SECURITY_ON > 0 */


void
usage()
{
  fprintf(stderr, "USAGE: %s [options] servername master_password\n", progname);
  fprintf(stderr, "  -p  partition number\n");
  fflush(stderr);
  exit(1);
}



void
check_status(char *string,nasd_drive_handle_t h,nasd_status_t *nasd_status,nasd_rpc_status_t *rpc_status)
{
  nasd_error_string_t           err_str;
  if (rpc_status && *rpc_status) {
    fprintf(stderr, "%s: ERROR: status=0x%x (%s) nasd_status=0x%x (%s)\n",
	    string,
	    *rpc_status, nasd_cl_error_string(h, *rpc_status, err_str),
	    *nasd_status, nasd_error_string(*nasd_status));
    fflush(stderr);
    exit(2);
  }

  if (nasd_status && *nasd_status) {
    fprintf(stderr, "%s: NASD ERROR: %s\n", string, nasd_error_string(*nasd_status));
    fflush(stderr);
    exit(2);
  }
}



nasd_identifier_t
get_first_object(
  nasd_drive_handle_t   drive,
  int                   partnum,
  char                 *name)
{
  nasd_ctrl_part_info_t ptinfo;
  nasd_status_t         rc;
  nasd_rpc_status_t     status;
  nasd_cookie_t		cookie;
  nasd_security_param_t sec_param;
  

  /* If the drive is freshly started with no partitions, it seems to
     not want to accept I/Os */
  nasd_cl_p_null_dr(drive, &rc, &status);
  check_status("Initial null RPC in get_first_object",drive,&rc,&status);

  BUILD_CAPABILITY(drive, NASD_ACCESS_RIGHTS_GETINFO, partnum,
		   NASD_CTRL_PART_INFO, &sec_param, &cookie);
  rc = nasd_cl_p_ctrl_get_part_info(drive, cookie.key, &sec_param,
				    &cookie.capability, partnum, &ptinfo);

  if ((rc == NASD_CTRL_ID_CHECK_FAILED) && (ptinfo.ctrl_id == 0)) {
    /* got back a page of zeroes - no such partition */
    fprintf(stderr, "ERROR: drive %s partition %d does not exist\n", name,
	    partnum);
    fflush(stderr);
    exit(1);
  }
  if (rc) {
    fprintf(stderr, "ERROR: got 0x%x (%s) getting partition info\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    exit(1);
  }
  return ptinfo.first_obj;
  
}

int
main(argc, argv)
  int     argc;
  char  **argv;
{
  char			       *server_name, *master_password;
  nasd_cookie_t			create_cookie;
  nasd_cookie_t			write_cookie;
  nasd_security_param_t		sp;
  nasd_p_create_dr_args_t	create_args;
  nasd_error_string_t		err_str;
  nasd_p_create_dr_res_t	res;
  nasd_rpc_status_t		status;
  nasd_drive_handle_t		h;
  nasd_timespec_t		tm;
  mpdrspeed_header_t		header;
  char				*pattern="Plus, we have no use for weapons grade nuclear materials, preferring instead to carry out our plans for world domination through total control of the world's data storage.";
  int				patlen;
  nasd_identifier_t		firstobj;
  nasd_len_t			out_len;
  int				i;
  nasd_status_t			rc;
  int				partnum;
  char				c;
  int				binding_type = NASD_BIND_DEFAULT;
  nasd_p_smpl_op_dr_args_t wr_args;
  nasd_p_fastwrite_dr_res_t wr_res;

  progname = argv[0];

  bzero((char *)&create_args, sizeof(create_args));
  partnum = 1;

  while (nasd_getopt(argc, argv, "p:b:s:n:T", &c)) {
    switch(c) {
      case 's':
      case 'b':
	if (sscanf(nasd_optarg, "%d",&size) != 1) 
	  usage();
	break;
      case 'n':
	if (sscanf(nasd_optarg, "%d",&nobjects) != 1) 
	  usage();
	if (nobjects > MPDRSPEED_MAX_OBJS) {
	  fprintf (stderr,"Only %d objects supported in mpdrspeedfs\n",
		   MPDRSPEED_MAX_OBJS);
	  exit(2);
	}
	break;
      case 'p':
	if (sscanf(nasd_optarg, "%d", &partnum) != 1)
	  usage();
	break;
      case 'T':
	binding_type = NASD_BIND_DCE_DIRECT_TCP;
	break;
      default:
        fprintf(stderr, "Unknown option '%c'\n", nasd_optopt);
        usage();
    }
  }
  if (nasd_optind >= argc)
    usage();
  server_name = argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind >= argc)
    usage();
  master_password = argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind < argc)
    usage();

  if (partnum < 0)
    usage();


  NASD_Malloc(buf, size, (char *));
  if (buf == NULL) {
    fprintf(stderr,"Could not allocate %d bytes for buffer\n",size);
    exit(1);
  }
  patlen = strlen(pattern);

  /* fill the buffer with pattern */
  i = 0;
  while(i < size) {
    int to_copy;

    to_copy = NASD_MIN(patlen, size - i);
    bcopy(pattern, buf+i, to_copy);
    i += to_copy;
  }

  rc = nasd_cl_p_init();
  if (rc) {
    printf("ERROR (%s:%d): cannot init client library, rc=%ld\n",
      __FILE__, __LINE__, (long)rc);
    return(rc);
  }

  rc = nasd_bind_to_drive(server_name, NASD_PDRIVE_PORT,
    binding_type, NULL, 0, &h);
  if (rc) {
    fprintf(stderr, "ERROR: cannot bind to server %s\n", server_name);
    fflush(stderr);
    exit(1);
  }

  nasd_sec_password_to_keys(master_password, partnum, &keys);
  
  bzero(&header,sizeof(header)); 
  strcpy(header.magic, MPDRSPEED_MULTI_MAGIC);
 
  bzero(&create_args,sizeof(nasd_p_create_dr_args_t));
  for (i = 0; i< nobjects;i++) {
    create_args.in_fieldmask=NASD_ATTR_AV;
    create_args.in_attribute.av=7; /* Goodness knows where this 7 came from */
    create_args.in_partnum = partnum;
    sp.partnum = partnum;
    sp.type = NASD_BLACK_KEY;
    sp.actual_protection = SECURITY_PROTECTION;
    nasd_cl_p_create_dr(h, keys.black_key, &sp, NULL, &create_args,
			&res, &status);
    check_status("Creating object",h,&res.nasd_status,&status);
    printf("Creating object 0x%" NASD_ID_FMT "\n",res.out_identifier);
    header.objects[i] = res.out_identifier;

    BUILD_CAPABILITY(h, NASD_ACCESS_RIGHTS_WRITE, partnum,
		     header.objects[i], &sp, &write_cookie);
    
    wr_args.in_partnum = partnum;
    wr_args.in_identifier = header.objects[i];
    wr_args.in_offset = 0;
    wr_args.in_len = size;
    nasd_cl_p_write_simple_dr(h, write_cookie.key, &sp,
			      &write_cookie.capability,
			      &wr_args, buf, &wr_res, &status);
    out_len = wr_res.out_datalen;
    rc = wr_res.nasd_status;
    check_status("Writing data",h,&rc,&status);
    if (out_len != size ) {
      fprintf(stderr,"Wrote only %ld bytes while writing data in iteration %d\n",out_len,i);
      exit(1);
    }
  }

  firstobj = get_first_object(h,partnum,server_name);
  /*  bzero(&write_cookie,sizeof(write_cookie));
  bzero(&wr_args, sizeof(wr_args));*/

  BUILD_CAPABILITY(h, NASD_ACCESS_RIGHTS_WRITE, partnum, firstobj, &sp,
		   &write_cookie);

  wr_args.in_partnum = partnum;
  wr_args.in_identifier = firstobj;
  wr_args.in_offset = 0;
  wr_args.in_len = sizeof(header); 

  nasd_cl_p_write_simple_dr(h, write_cookie.key, &sp,
			    &write_cookie.capability,
			    &wr_args, &header, &wr_res, &status);
  out_len = wr_res.out_datalen;
  rc = wr_res.nasd_status;
  check_status("Writing header",h,&rc,&status);

  if (out_len != sizeof(header) ) {
    fprintf(stderr,"Wrote only %ld bytes while writing header, sizeof(header) = %d \n",out_len,sizeof(header));
    exit(1);
  }

  nasd_cl_p_sync_dr(h, &rc, &status);
  check_status("Forcing drive writeback", h, &rc, &status);
  exit(0);
}
