/*
 * prshutdown.c
 *
 * Remotely shut-down a NASD drive
 *
 * Author: Jim Zelenka
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <nasd/nasd_pdrive.h>
#include <nasd/nasd_pdrive_client.h>
#include <nasd/nasd_pdrive_client_kpdev.h>

char *progname;

int                        binding_type;
int                        binding_args_len;
void                      *binding_args;
char                      *binding_port = NASD_PDRIVE_PORT;
nasd_drive_param_kpdev_t   kpdev_args;
int                        nondefault_binding = 0;

void
usage()
{
  fprintf(stderr, "USAGE: %s [options] servername master_password\n", progname);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -l use colocated interface\n");
  fprintf(stderr, "  -M use message queues\n");
  fprintf(stderr, "  -s use security\n");
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
  fprintf(stderr, "  -T use DCE-TCP\n");
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */

  fflush(stderr);
  exit(1);
}

int
main(argc, argv)
  int     argc;
  char  **argv;
{
  nasd_sec_keyring_t keys;
  nasd_security_param_t sec_param;
  nasd_p_rshutdown_dr_args_t rsd_args;
  nasd_p_rshutdown_dr_res_t rsd_res;
  nasd_error_string_t error_text;
  nasd_rpc_status_t status;
  nasd_drive_handle_t h;
  char *server_name;
  nasd_status_t rc;
  char c;
  nasd_uint16 protection;

  progname = argv[0];

  protection = NASD_NO_PROTECTION;
  binding_type = NASD_BIND_DEFAULT;
  binding_args = NULL;
  binding_args_len = 0;

  while(nasd_getopt(argc, argv, "lMsT", &c)) {
    switch(c) {
      case 'l':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_COLOCATE;
        binding_args = &kpdev_args;
        binding_args_len = sizeof(kpdev_args);
        strcpy(kpdev_args.devname, "/dev/nasdkp0");
        break;
      case 'M':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_MSGQ;
        break;
      case 's':
        protection = NASD_INTEGRITY_OPS;
        break;
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
      case 'T':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_DCE_DIRECT_TCP;
        break;
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
      default:
        fprintf(stderr, "Unknown option '%c'\n", nasd_optopt);
        usage();
    }
  }

  if (nasd_optind >= argc)
    usage();
  server_name = argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind >= argc)
    usage();

  /* partnum doesn't matter, since we're using the drive key */
  nasd_sec_password_to_keys(argv[nasd_optind], 0, &keys);
  nasd_optind++;

  if (nasd_optind < argc)
    usage();

  rc = nasd_cl_p_init();
  if (rc) {
    fprintf(stderr,
      "ERROR: cannot initialize client library, error 0x%x (%s)\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    exit(1);
  }

  rc = nasd_bind_to_drive(server_name, binding_port,
    binding_type, binding_args, binding_args_len, &h);
  if (rc) {
    fprintf(stderr, "ERROR: cannot bind to server %s error 0x%x (%s)\n",
      server_name, rc, nasd_error_string(rc));
    fflush(stderr);
    exit(1);
  }

  bzero((char *)&rsd_args, sizeof(rsd_args));

  /* Fill in capability and security param */

  sec_param.type = NASD_DRIVE_KEY;
  sec_param.partnum = 0;
  sec_param.actual_protection = protection;

  rsd_args.in_flags = 0;

  nasd_cl_p_rshutdown_dr(h, keys.drive_key, &sec_param, NULL,
                         &rsd_args, &rsd_res, &status);
  rc = rsd_res.nasd_status;

  if (status || rc) {
    fprintf(stderr, "ERROR: status=0x%x (%s) nasd_status=0x%x (%s)\n",
      status, nasd_cl_error_string(h, status, error_text),
      rc, nasd_error_string(rc));
    fflush(stderr);
    exit(2);
  }

  printf("Shutdown request complete.\n");
  fflush(stdout);

  rc = nasd_unbind_drive(&h);
  if (rc) {
    fprintf(stderr, "ERROR: got 0x%x (%s) from nasd_unbind_drive()\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    exit(2);
  }

  nasd_cl_p_shutdown();

  exit(0);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
