;;########################################################################
;; dashobj1.lsp
;; Contains data and datasheet menu functions and datasheet constructor
;; Copyright (c) 1994-8 by Forrest W. Young
;;########################################################################

;;########################################################################
;; data menu functions
;;########################################################################

(defun open-data (&optional file)
"Args: (&optional file)
Opens a data object contained in a file, and displays the datasheet and the about-these-data window. The file's name must end with .lsp. If the optional string argument FILE is included, the data object is loaded from FILE, otherwise a dialog is presented to select the file. The string need not end with .lsp.  Returns the object-id of the data object."
  (let ((result (load-data file)))
    (when result (browse-data result)
          (send *datasheet* :show-window)
          (show-about-these-data))
    result))

(defun new-data ()
"Args: None
Creates new data object and a datasheet for the new object."
  (let* ((txt1 (send text-item-proto :new "Create New Data Named:"))
         (name (send edit-text-item-proto :new "NewData" :text-length 14))
         (txt2 (send text-item-proto :new "The New Data Will Be:"))
         (type (send choice-item-proto :new
                     '("Multivariate Data" "Matrix Data")))
         (OK   (send modal-button-proto :new "OK" :action #'(lambda () 
                    (list (send name :text) (send type :value)))))
         (cancel (send modal-button-proto :new "Cancel"))
         (dialog (send modal-dialog-proto :new
                       (list txt1 name txt2 type (list OK cancel)) 
                       :default-button OK))
         (return (send dialog :modal-dialog))
         (choice nil) 
         (size (list (first (first (send *vista* :help-layout-sizeloc)))
                     (- (second (first (send *vista* :workmap-layout-sizeloc))) 
                        #+msdos 115
                        #+macintosh 105
                        )))
         (loc  (list (first (second (send *vista* :help-layout-sizeloc)))
                     (+ (second (second (send *vista* :workmap-layout-sizeloc))) 
                       #+msdos 115
                       #+macintosh 65
                        )))
         )
    (when return
          (setf choice (second return))
          (case choice 
            (0 
             (show-datasheet (data (first return)
                :variables '("Var0") :labels '("Obs0") :data '(nil))
                             :size size  :frame-location loc :new-data t))
            (1 (show-datasheet (data (first return)
                :variables '("Var0" "Var1")  
                :data '(nil nil nil nil)
                :shapes '("Symmetric") 
                :matrices '("Mat0")) :size size :frame-location loc :new-data t))
            (t (show-datasheet (data (first return)
                :variables '("Response" "Way1") 
                :types '("Numeric" "Category")
                :labels '("Obs0") :data '(nil nil)) :size size :frame-location loc )))
          (show-about-these-data (format nil "To enter information about these data type:~%  (about-these-data \"INFORMATION\")~%in the listener window. Replace INFORMATION with the information about the data that you wish to enter. If you wish to do this later you can enter the information this way whenever the data icon is highlighted."))
          (send *datasheet* :show-window)
          )))

(defun show-about-these-data (&optional string)
  (let ((locx (first (send *datasheet* :location)))
        (locy (second (send *datasheet* :location)))
        (sizx (first (first (send *vista* :help-layout-sizeloc))))
        (sizy (second (first (send *vista* :help-layout-sizeloc))))
        )

#+macintosh (setf locx (max locx
                            (- (first (send *vista* :desktop-sizes)) 
                               10 sizx)))
#+macintosh (setf locy (max 20 (- locy 87 window-decoration-height)))

#-macintosh (setf sizx (+ sizx 8))
#-macintosh (setf locx (- (first (send *vista* :desktop-sizes)) -4 sizx))
#-macintosh (setf locy (max 30 (- locy 106 window-decoration-height)))

    (about-these-data string
                      :location (list locx locy) 
                      :size 
#+macintosh            (list sizx 85)
#-macintosh            (list (- sizx 8) 86)
                      )
    (send *datasheet* :show-window)
    ))

(defun edit-data (&rest args) 
  (show-datasheet args))

(defun edit-datasheet (&rest args)
  (show-datasheet args))

(defun show-datasheet (&optional data-object &key new-data size location frame-location)
"Args: &optional DATA-OBJECT &key NEW-DATA
Alias for edit-data and edit-datasheet.
Shows an editable datasheet for DATA-OBJECT (or current-data if not specified). NEW-DATA is t for new data, nil otherwise" 
  (when (not data-object) (setf data-object current-data)) 
  (when (send data-object :ways) 
        (setf data-object 
              (create-data (strcat "MV-"(send data-object :name)))))
  (let ((dash nil)
        (b4x nil)
        (dsob (send data-object :datasheet-object)))
    (cond 
      ((not dsob)
       (setf dsob (datasheet data-object :editable t 
                             :size size :location location 
                             :frame-location frame-location))
       )
      (t 
       (send dsob :editable t)
       (send dsob :enable-menu-items t)
       (send dsob :hot-cell nil)
       (send dsob :hot-cell-ready nil)
       (send dsob :hot-cell-string nil)
       (send dsob :scroll 0 0)
       (setf b4x (first (send dsob :size)))
(FORMAT T "~d~%" (list "Size" size))
       (if size
           (apply #'send dsob :size size)
           (send dsob :set-window-size 
                 (send dsob :field-width)
                 (send dsob :field-height)
                 (send dsob :label-width)
                 (send dsob :nvar)
                 (send dsob :nobs) t))
     ;  (when (and (> (+ (send dsob :field-width)
     ;                   (first (send dsob :size)) 
     ;                   (first (send dsob :location)))
     ;                (first screen-size))
     ;             (> (first (send dsob :size)) b4x))
     ;        (apply #'send dsob :location (- (send dsob :location) 
     ;                                (list (send dsob :field-width) 0)))
     ;        (apply #'send dsob :size (+ (send dsob :size)
     ;                                    (list (send dsob :field-width) 0))))
       (send dsob :redraw)
       (send dsob :show-window)))
    (when dsob
          (send dsob :enable-vista-menus&tools nil)
          (send dsob :new-data new-data)
          (send dsob :edited nil)
          (when (not (send dsob :help-menu-installed))
                (if (equal (send (select (send *help-menu* :items) 
                      (- (length (send *help-menu* :items)) 1)) :title) "-")
                    (setf dash nil)
                    (send *help-menu* :append-items 
                          (send dash-item-proto :new)))
                (send dsob :add-plot-help-item)
                (send dsob :help-menu-installed t)))
    (send (first (send (send dsob :menu) :items)) :enabled nil)
    t))

(defun browse-data (&optional data-object) 
"Args: DATA-OBJECT
Function to show a non-editable datasheet for browsing DATA-OBJECT"
  (when (not data-object) (setf data-object current-data)) 
  (cond 
    ((not (send data-object :datasheet-object))
     (datasheet data-object :editable nil)
     )
    (t
     (let ((dsob (send data-object :datasheet-object)))
       (send dsob :editable nil)
       (send (first (send (send dsob :menu) :items)) :enabled t)
       (when (not (send dsob :help-menu-installed))
             (send dsob :add-plot-help-item)
             (send dsob :help-menu-installed t))
       (send dsob :show-window))))
    (send (send data-object :datasheet-object) :enable-menu-items nil)
  t)

(defun datasheet (data-object &key editable size location frame-location
                              ncolumns ndecimals (show t))
"Args: DATA-OBJECT &KEY EDITABLE SIZE LOCATION FRAME-LOCATION NCOLUMNS NDECIMALS (SHOW T)
Constructor function for constructing datasheet for editing or browsing DATA-OBJECT"
  (let* ((object (send datasheet-proto :new 2
                       :title (strcat (send data-object :name) " DataSheet")
                       :size (- (send *vista* :desktop-sizes)
                                      (send *workmap* :location)
                                #+msdos(list 35 90)
                                #-msdos(list 30 105)
                                )
                       :show nil))
#+msdos  (loc (+ (list 35 90) (send *workmap* :location)))
#-msdos  (loc (+ (list 30 105) (send *workmap* :location)))
         (nvar   (send data-object :nvar))
         (nobs   (send data-object :nobs))
         (dsmenu (send menu-proto :new "DataSheet"))
         (saveda (send menu-item-proto :new "Save Data ..."
                       :action #'(lambda ()
                                   (setcd data-object)
                                   (save-data)
                                   )))
         (editda (send expert-menu-item-proto :new "Make Editable"
                       :action #'(lambda ()
                                   (setcd data-object)
                                   (send object :make-editable)) ;'edit-data
                       :enabled (not editable)))
         (setfw  (send menu-item-proto :new "Width of Columns"
                       :action #'(lambda ()
                                   (setcd data-object)
                                   (send object :set-fw-dialog))))
         (setdec (send menu-item-proto :new "Number of Decimals"
                       :action #'(lambda ()
                                   (setcd data-object)
                                   (send object :set-dec-dialog))))
         (new1st (cond 
                   ((send data-object :matrices)
                    (send menu-item-proto :new "New Rows and Columns"
                          :action #'(lambda ()
                                      (setcd data-object)
                                      (new-rows-and-columns))))
                   (t
                    (send menu-item-proto :new "New Observations"
                          :action #'(lambda ()
                                      (setcd data-object)
                                      (new-observations))))))
         (new2nd (cond
                   ((send data-object :matrices)
                    (send menu-item-proto :new "New Matrices"
                          :action #'(lambda ()
                                      (setcd data-object)
                                      (new-matrices))))
                   (t
                    (send menu-item-proto :new "New Variables"
                       :action #'(lambda ()
                                   (setcd data-object)
                                   (new-variables))))))
         (newlab (cond
                   ((send data-object :matrices)
                    (send menu-item-proto :new "Change Matrix Names"
                          :action #'(lambda ()
                                      (setcd data-object)
                                      (change-matrix-names))))
                   (t
                    (send menu-item-proto :new "Switch Label Variable"
                          :action #'(lambda ()
                                      (setcd data-object)
                                      (switch-label-variable))))))
         (tw     (send object :text-width "9"))
         (data   (cond
                   ((send data-object :matrices)
                    (send data-object :data))
                   ((send data-object :ways)
                    (combine (send data-object :data)))
                   (t
                    (send data-object :active-data '(numeric ordinal)))))
         (nonil  (remove nil (remove "NIL" data :test #'equal) :test #'equal))
         (ndigit (cond
                   (nonil (ceiling (log (max (abs nonil)) 10)))
                   (t 6))) 
         (ndciml 2)
         (ncols  (max 8 (+ ndigit ndciml)))
         (dcimlw (send object :text-width "."))
         (signw  (send object :text-width "-"))
         (fw     (+ (* tw ncols) dcimlw signw 6))
         (fh     (+ (send object :text-ascent) 
                    (send object :text-descent) 3))
         (lw     nil)
         (dsobj-args (send data-object :datasheet-arguments))) 
    (if ndecimals 
        (setf ndciml ndecimals)
        (if dsobj-args
            (setf ndciml (third dsobj-args))))
    (if ncolumns 
        (setf ncols ncolumns)
        (if dsobj-args
            (setf ncols (fourth dsobj-args))))
    (setf *make-editable-menu-item* editda)
    (send object :nvar nvar)
    (when (send data-object :matrices)
          (send object :nmat (send data-object :nmat)))
    (send object :variable-strings 
          (copy-list (send data-object :variables)))
    (send object :matrix-strings 
          (copy-list (send data-object :matrices)))
    (send object :type-strings (copy-list (send data-object :types))) 
    (send object :create-label-strings data-object)
    (setf fw     (+ (* tw ncols) dcimlw signw 6))
    (setf lw (max (* 10 tw) fw (+ 6 (max 
             (map-elements #'send object :text-width 
                           (send object :label-strings))))))
    (when dsobj-args
          (if (some #'< (- screen-size 
                           (+ (first dsobj-args) (second dsobj-args)))
                    (list 0 0))
              (setf dsobj-args nil)))
    (if size
        (apply #'send object :size size)
        (if dsobj-args
            (apply #'send object :size (first dsobj-args))
            (send object :size 
                  (min ;(select (send object :size) 0)
                       (- (first (send *vista* :desktop-sizes))
#+macintosh               10
                          (first (send *workmap* :location)) 100)
                       (+ 1 lw (* fw (1+ nvar)) (if editable fw 0)
                          msdos-fiddle border-thickness
#+macintosh               window-decoration-width -3
                          ))
                  (min ;(select (send object :size) 1) 
                       (- (second (send *vista* :desktop-sizes))
#+macintosh               10
                          (second (send *workmap* :location)) 125)
                       (+ (if editable fh 0) 1 (* fh (+ 4 nobs)))))))
    (if (and (not frame-location) (not location) )
        (if dsobj-args
            (setf location (second dsobj-args))
            (setf location 
                  (list 
                   (max (first loc) 
                        (- (first (send *vista* :desktop-sizes))
                           (first (send object :frame-size))))
                   (second loc)))))
#+msdos(if frame-location
           (apply #'send object :location frame-location)
           (apply #'send object :frame-location location))
#-msdos(if frame-location
           (apply #'send object :frame-location frame-location)
           (apply #'send object :location location))
    (send dsmenu :append-items editda saveda (send dash-item-proto :new)
          setfw setdec (send dash-item-proto :new)
          new1st new2nd (send dash-item-proto :new) newlab)
    (defmeth dsmenu :install ()
      (if (not (equal *current-data* data-object)) (setcd data-object))
      (call-next-method))
    (send object :data-object data-object)
    (send object :nobs nobs) 
    (send object :newvar 0)
    (send object :newobs 0)
    (send object :newmat 0)
    (send object :field-width fw) 
    (send object :field-height fh)
    (send object :label-width  lw)
    (send object :number-of-decimals ndciml)
    (send object :number-of-columns ncols)
    (send object :create-data-matrix-strings)
    (send object :editable editable)
    (send object :redraw-now t)
    (send object :menu dsmenu)
    (when show 
          (send object :show-window)
          (send data-object :datasheet-open t)
          (send object :showing t))
    (send object :set-window-size fw fh lw nvar nobs 
          (and (not size) (not dsobj-args)))
    (send object :h-scroll-incs fw (* 4 fw))
    (send object :v-scroll-incs fh (* 10 fh))
    (send object :add-plot-help-item)
    (send object :help-menu-installed t)
    (setf *datasheet* object)
    (send data-object :datasheet-object object)
    
    object))

;;########################################################################
;; datasheet menu functions
;;########################################################################

(defun new-observation ()
  (send *datasheet* :expand-mv-datasheet nil t 1))

(defun new-observations (&optional (n 0))
  (when (= n 0) 
        (send *datasheet* :redraw-now nil)
        (setf n (first (get-value-dialog 
                        "Number of New Observations" :initial 1))))
  (cond 
    (n 
     (when (< n 1) (error "You Must Add at Least 1 New Observation."))
     (send *datasheet* :expand-mv-datasheet nil t n))
    (t
     (send *datasheet* :redraw-now t)
     (send *datasheet* :redraw))))

(defun new-variable ()
  (send *datasheet* :expand-mv-datasheet t nil 1))

(defun new-variables (&optional (n 0))
  (when (= n 0) 
        (send *datasheet* :redraw-now nil)
        (setf n (first (get-value-dialog 
                        "Number of New Variables" :initial 1)))
        )
  (cond 
    (n
     (when (< n 1) (error "You Must Add at Least 1 New Variable."))
     (send *datasheet* :expand-mv-datasheet t nil n))
    (t
     (send *datasheet* :redraw-now t)
     (send *datasheet* :redraw))))


(defun new-row-and-column ()
  (send *datasheet* :expand-mat-datasheet t nil 1))

(defun new-rows-and-columns (&optional (n 0))
  (when (= n 0) 
        (send *datasheet* :redraw-now nil)
        (setf n (first (get-value-dialog 
                        "Number of New Rows and Columns" :initial 1))))
  (when n 
        (when (< n 1) (error "You Must Add at Least 1 New Row and Column."))
        (send *datasheet* :expand-mat-datasheet t nil n)))

(defun new-matrix ()
  (send *datasheet* :expand-mat-datasheet nil t 1))

(defun new-matrices (&optional (n 0))
  (when (= n 0) 
        (send *datasheet* :redraw-now nil)
        (setf n (first (get-value-dialog 
                        "Number of New Matrices" :initial 1))))
  (when n 
        (when (< n 1) (error "You Must Add at Least 1 New Matrix."))
        (send *datasheet* :expand-mat-datasheet nil t n)))
