use anyhow::Context;
use forgejo_api::structs::{CreateIssueCommentOption, IssueListIssuesQuery};

use crate::actions::issue::display_issue;

use crate::actions::text_manipulation::select_prompt_for;
use crate::actions::GeneralArgs;
use crate::render::spinner::spin_until_ready;
use crate::render::ui::fuzzy_select_with_key;
use crate::types::context::BergContext;
use crate::types::git::OwnerRepo;

use clap::Parser;

/// Add a comment to selected issue
#[derive(Parser, Debug)]
pub struct CommentIssueArgs {}

impl CommentIssueArgs {
    pub async fn run(self, general_args: GeneralArgs) -> anyhow::Result<()> {
        let _ = general_args;
        let ctx = BergContext::new(self, general_args).await?;
        let OwnerRepo { owner, repo } = ctx.owner_repo()?;
        let issues_list = spin_until_ready(ctx.client.issue_list_issues(
            owner.as_str(),
            repo.as_str(),
            IssueListIssuesQuery::default(),
        ))
        .await
        .map_err(anyhow::Error::from)?;

        let selected_issue =
            fuzzy_select_with_key(&issues_list, select_prompt_for("issue"), display_issue)?;

        let (nr, title) = {
            (
                selected_issue.id.context("Selected issue had no id")?,
                selected_issue
                    .title
                    .as_ref()
                    .cloned()
                    .context("Selected issue had no title")?,
            )
        };

        let body = get_comment_input(&ctx, title)?;
        let comment = ctx
            .client
            .issue_create_comment(
                owner.as_str(),
                repo.as_str(),
                nr as u64,
                CreateIssueCommentOption {
                    body,
                    updated_at: None,
                },
            )
            .await?;

        tracing::debug!("{comment:?}");

        Ok(())
    }
}

fn get_comment_input(
    ctx: &BergContext<CommentIssueArgs>,
    issue_title: impl AsRef<str>,
) -> anyhow::Result<String> {
    ctx.editor_for(
        "a comment",
        format!("Write a comment for issue \"{}\"", issue_title.as_ref()).as_str(),
    )
}
