use forgejo_api::structs::{Release, RepoListReleasesQuery};

use crate::actions::GeneralArgs;
use crate::render::json::JsonToStdout;
use crate::render::option::option_display;
use crate::render::spinner::spin_until_ready;
use crate::types::context::BergContext;
use crate::types::git::OwnerRepo;

use clap::Parser;

/// List all releases in the current repository
#[derive(Parser, Debug)]
pub struct ListReleaseArgs {
    /// Number of releases to be displayed
    #[arg(short, long, value_name = "N")]
    pub count: Option<u32>,
}

impl ListReleaseArgs {
    pub async fn run(self, general_args: GeneralArgs) -> anyhow::Result<()> {
        let ctx = BergContext::new(self, general_args).await?;
        let OwnerRepo { repo, owner } = ctx.owner_repo()?;
        let (_, releases_list) = spin_until_ready(ctx.client.repo_list_releases(
            owner.as_str(),
            repo.as_str(),
            RepoListReleasesQuery {
                page: ctx.args.count.as_ref().map(|_| 1),
                limit: ctx.args.count,
                ..Default::default()
            },
        ))
        .await?;

        match general_args.output_mode {
            crate::types::output::OutputMode::Pretty => {
                present_releases_list(&ctx, releases_list);
            }
            crate::types::output::OutputMode::Json => {
                releases_list.print_json()?;
            }
        }

        Ok(())
    }
}

fn present_releases_list(ctx: &BergContext<ListReleaseArgs>, releases: Vec<Release>) {
    let releases_empty = releases.is_empty();

    let mut table = ctx.make_table().add_table((!releases_empty).then(|| {
        let mut inner_table = ctx.make_table();
        inner_table
            .set_header(vec!["ID", "Name", "Tag", "Published At"])
            .add_rows(releases.into_iter().map(|release| {
                let Release {
                    id,
                    name,
                    tag_name,
                    published_at,
                    ..
                } = release;
                vec![
                    option_display(&id),
                    option_display(&name),
                    option_display(&tag_name),
                    option_display(&published_at),
                ]
            }));
        inner_table
    }));

    table.set_header(vec![format!(
        "Releases{}",
        if releases_empty {
            " (empty)"
        } else {
            Default::default()
        }
    )]);

    println!("{table}", table = table.show());
}
