use crate::render::json::JsonToStdout;
use crate::render::option::option_display;
use crate::render::table::TableWrapper;
use crate::{actions::GeneralArgs, types::context::BergContext};
use anyhow::Ok;
use clap::Parser;
use clap::ValueEnum;
use forgejo_api::structs::{Repository, UserCurrentListReposQuery};
use strum::{Display, EnumIter, EnumString, IntoStaticStr};

/// List user current active repositories
#[derive(Debug, Parser, Clone, Copy)]
pub struct RepoListArgs {
    /// The amount of repositories that should be displayed
    #[arg(short, long, default_value_t = 19)]
    pub limit: u32,

    /// Order the displayed repositories by different criteria
    #[arg(short, long, value_enum, name = "ORDER BY", default_value_t = OrderBy::Name)]
    pub order_by: OrderBy,
}

#[derive(
    Debug,
    Clone,
    Copy,
    Default,
    PartialEq,
    Eq,
    Display,
    EnumString,
    ValueEnum,
    EnumIter,
    IntoStaticStr,
)]
pub enum OrderBy {
    #[default]
    Name,
    Id,
    Newest,
    Oldest,
    Recentupdate,
    Leastupdate,
    Reversealphabetically,
    Alphabetically,
    Reversesize,
    Size,
    Reversegitsize,
    Gitsize,
    Reverselfssize,
    Lfssize,
    Moststars,
    Feweststars,
    Mostforks,
    Fewestforks,
}

impl RepoListArgs {
    pub async fn run(self, general_args: GeneralArgs) -> anyhow::Result<()> {
        let ctx = BergContext::new(self, general_args).await?;

        let (_, repos) = ctx
            .client
            .user_current_list_repos(UserCurrentListReposQuery {
                limit: Some(ctx.args.limit),
                order_by: Some(ctx.args.order_by.to_string().to_lowercase()),
                page: None,
            })
            .await?;

        match general_args.output_mode {
            crate::types::output::OutputMode::Pretty => {
                present_repos(ctx, repos).await?;
            }
            crate::types::output::OutputMode::Json => {
                repos.print_json()?;
            }
        }

        Ok(())
    }
}

async fn present_repos(
    ctx: BergContext<RepoListArgs>,
    repos: Vec<Repository>,
) -> anyhow::Result<()> {
    let mut table: TableWrapper = ctx.make_table();

    table
        .set_header(vec!["Name", "Description", "Size", "Info", "Updated"])
        .add_rows(repos.iter().map(|repo| {
            let Repository {
                name,
                description,
                updated_at,
                private,
                archived,
                fork,
                size,
                ..
            } = repo;

            let repo_visibility = if private.unwrap_or(false) {
                "Private"
            } else {
                "Public"
            };
            let is_archived = archived.unwrap_or(false);
            let is_fork = fork.unwrap_or(false);

            let mut info_arr = vec![repo_visibility];

            if is_archived {
                info_arr.push("Archived");
            }

            if is_fork {
                info_arr.push("Fork");
            }

            vec![
                option_display(name),
                option_display(description),
                option_display(size),
                info_arr.join(", "),
                option_display(updated_at),
            ]
        }));

    println!("{tbl}", tbl = table.show());

    Ok(())
}
