use crate::actions::keys::ssh::ssh_key_identifier;
use crate::actions::text_manipulation::select_prompt_for;
use crate::render::ui::confirm_with_prompt;
use crate::types::context::BergContext;
use crate::{actions::GlobalArgs, render::ui::fuzzy_select_with_key};

use clap::Parser;
use forgejo_api::structs::UserCurrentListKeysQuery;
use miette::{Context, IntoDiagnostic};

/// Remove a ssh key from your profile
#[derive(Parser, Debug)]
pub struct RemoveSshArgs {}

impl RemoveSshArgs {
    pub async fn run(self, global_args: GlobalArgs) -> miette::Result<()> {
        if global_args.non_interactive {
            return Err(miette::miette!(
                "Non interactive version of this command is not supported at the moment"
            )
            .context("It is too dangerous to mess something up"));
        }

        let ctx = BergContext::new(self, global_args).await?;

        let options = UserCurrentListKeysQuery { fingerprint: None };

        let (_, ssh_keys) = ctx
            .client
            .user_current_list_keys(options)
            .await
            .into_diagnostic()?;

        let selected_ssh_key = fuzzy_select_with_key(
            ssh_keys.as_slice(),
            select_prompt_for("SSH Key to remove"),
            ssh_key_identifier,
        )
        .context("No key selected!")?;

        let key_id = selected_ssh_key.id.context(
            "Key has no ID. Something is very wrong. Remove it in the web UI if possible!",
        )?;

        tracing::debug!("Key has ID: {key_id}");

        let key_name = ssh_key_identifier(selected_ssh_key);

        if !confirm_with_prompt(format!("Are you sure you want to remove {key_name}?"))
            .context("Confirmation dialouge aborted!")?
        {
            return Ok(());
        }

        ctx.client
            .user_current_delete_key(key_id)
            .await
            .into_diagnostic()
            .context("Removing SSH public key failed!")?;

        println!("Successfully removed the SSH key to from your user profile!");

        Ok(())
    }
}
