use crate::{
    actions::GlobalArgs,
    types::{
        context::BergContext,
        git::{MaybeOwnerRepo, OwnerRepo},
    },
};

use clap::Parser;
use miette::{Context, IntoDiagnostic};

/// Delete a repository
#[derive(Parser, Debug)]
pub struct RepoDeleteArgs {
    /// Repository to be deleted
    #[arg(value_name = "REPO or OWNER/REPO")]
    pub owner_and_repo: MaybeOwnerRepo,
}

impl RepoDeleteArgs {
    pub async fn run(self, global_args: GlobalArgs) -> miette::Result<()> {
        let _ = global_args;
        let ctx = BergContext::new(self, global_args).await?;

        let owner_repo = match ctx.args.owner_and_repo {
            MaybeOwnerRepo::ImplicitOwner(repo) => {
                let owner = ctx
                    .client
                    .user_get_current()
                    .await
                    .into_diagnostic()?
                    .login
                    .context("Current user has no username on targeted forgejo instance")?;
                OwnerRepo { owner, repo }
            }
            MaybeOwnerRepo::ExplicitOwner(owner_repo) => owner_repo,
        };

        ctx.client
            .repo_delete(owner_repo.owner.as_str(), owner_repo.repo.as_str())
            .await
            .map_err(|e| {
                miette::miette!(
                    help = format!(
                        "Make sure the repository {repo_display} exists!",
                        repo_display = owner_repo.to_string()
                    ),
                    "{e}"
                )
                .wrap_err("Deleting repository failed!")
            })?;
        Ok(())
    }
}
