#include <stdio.h>
#include <signal.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <linux/kd.h>
#include <linux/serial.h>
#include <sys/ioctl.h>
#include "cuecat_lib.h"
#include "cuecat_at2xt.h"


/* Scancode and scancode masks */
#define ESC_SCANCODE 0x01
#define CR_SCANCODE  0x1c
#define KEY_RELEASE  0x80
#define DEFAULT_SERIAL_DEV_FILE "/dev/ttyS0"



/* Global vars */
int fd;
int previous_kbd_mode;
struct termios old_term_settings;
struct termios new_term_settings;
struct serial_struct old_serinfo;
struct serial_struct new_serinfo;
char serial_dev_filename[FILENAME_MAX]=DEFAULT_SERIAL_DEV_FILE;
char serial;
char help;
const unsigned char at2xt[128]=AT2XT_VALS;



/* Function prototypes */
void terminate_program(int x);
void restore_console(void);
void restore_serial_port(void);
int getfd(void);
static int open_tty(char *fnam);
static int is_a_console(int fd);
char parse_cmdline(int argc,char *argv[]);
void setup_signal_handlers(void);
static int scancode_at_to_xt(int scancode);



int main(int argc,char *argv[])
{
  int invalid_parameter;
  struct barcode_decoder_state barcode_decoder_state;
  unsigned char scancode;
  char cuecat_id[CUECAT_DEVICE_ID_LEN+1];		/* Cuecat device ID */
  char barcode_type[CUECAT_BARCODE_TYPE_LEN+1];
  char barcode[256];
  int user_abort=0;

  /* Parse the command line parameters */
  if((invalid_parameter=parse_cmdline(argc,argv)))
  {
    /* Invalid parameter */
    printf("%s : error : invalid or unknow parameter \"%s\"\n",
           argv[0],argv[invalid_parameter]);
    printf("Try %s -h for help\n",argv[0]);

    return(-1);
  }

  /* Does the user want help ? */
  if(help)
  {
    printf("\n");
    printf("%s options :\n",argv[0]);
    printf("\n");
    printf("  -s [dev file] : use CueCat with serial pod connected on\n");
    printf("                  the specified device file. If not specified,\n");
    printf("                  default is %s\n",DEFAULT_SERIAL_DEV_FILE);
    printf("  -h            : this help screen\n");
    printf("\n");

    return(0);
  }

  /* Get a console or serial port descriptor */
  fd=getfd();

  /* Did the user want to use a serial pod ? */
  if(!serial)
  {
    /* Give a quick word of advise */
    printf("Press ESC to end the program ...\n\n");

    /* Get the current state of the keyboard */
    if (ioctl(fd, KDGKBMODE, &previous_kbd_mode))
    {
      perror("KDGKBMODE");
      close(fd);
      exit(1);
    }

    /* Display the current state of the keyboard */
    printf("The keyboard was in ");
    switch(previous_kbd_mode)
    {
      case K_RAW:       printf("raw")      ;break;
      case K_XLATE:     printf("xlate")    ;break;
      case K_MEDIUMRAW: printf("mediumraw");break;
      case K_UNICODE:   printf("unicode")  ;break;
      default:          printf("unknown")  ;break;
    }
    printf(" state.\n\n");

    /* Setup the signal handler to trap any signal that might kill us before
       we can restore the keyboard */
    setup_signal_handlers();

    /* Set the keyboard in raw scancode mode */
    if (tcgetattr(fd,&old_term_settings)==-1)
    {
      perror("tcgetattr");
      close(fd);
      exit(1);
    }
    if (tcgetattr(fd,&new_term_settings)==-1)
    {
      perror("tcgetattr");
      close(fd);
      exit(1);
    }

    new_term_settings.c_lflag &= ~ (ICANON | ECHO | ISIG);
    new_term_settings.c_iflag = 0;
    new_term_settings.c_cc[VTIME] = 1;	/* 0.1 sec intercharacter timeout */

    if (tcsetattr(fd,TCSAFLUSH,&new_term_settings)==-1)
    {
      perror("tcsetattr");
      close(fd);
      exit(1);
    }
    if (ioctl(fd,KDSKBMODE,K_RAW))
    {
      perror("KDSKBMODE");
      close(fd);
      exit(1);
    }
  }
  else					/* The user wants to use a serial pod */
  {
    /* Give a quick word of advise */
    printf("Press CTRL-C to end the program ...\n\n");

    /* Get the current state of the serial port */
    if(ioctl(fd,TIOCGSERIAL,&old_serinfo))
    {
      perror("TIOCGSERIAL");
      close(fd);
      exit(1);
    }
    if(tcgetattr(fd,&old_term_settings)==-1)
    {
      perror("tcgetattr");
      close(fd);
      exit(1);
    }

    new_serinfo=old_serinfo;
    new_term_settings=old_term_settings;

    /* Setup the signal handler to trap any signal that might kill us before
       we can restore the serial port as it was before */
    setup_signal_handlers();

    /* set SPD_CUST and divisor */
    new_serinfo.flags&=~ASYNC_SPD_MASK;
    new_serinfo.flags|=ASYNC_SPD_CUST;

    /* we measured the cuecat's baudrate  to be about 8000 the closest we can
       get to this with a basebaud of 115200 is to use a divisor of 14 */
    new_serinfo.custom_divisor=14;

    if (ioctl(fd,TIOCSSERIAL,&new_serinfo))
    {
      perror("TIOCSSERIAL");
      close(fd);
      exit(1);
    }

    cfmakeraw(&new_term_settings);
    cfsetospeed(&new_term_settings, B38400);

    new_term_settings.c_cc[VTIME] = 0;	/* Wait indefinately ... */
    new_term_settings.c_cc[VMIN] = 1;	/* ... for 1 character */

    if(tcsetattr(fd,TCSAFLUSH,&new_term_settings)==-1)
    {
      perror("tcsetattr");
      close(fd);
      exit(1);
    }
  }

  /* Read scancodes and attempt to convert them until we get an ESCape code */
  do
  {
    /* Read a new scancode */
    if(read(fd,&scancode,1)!=1)
    {
      fprintf(stderr,"Error reading a scancode ...\n");
      restore_console();
      exit(1);
    }

    /* If we use a serial pod, turn the AT scancode into an XT one */
    if(serial && (!(scancode=scancode_at_to_xt(scancode))))
      continue;

    /* If we watch for scancodes from the keyboard, watch for ESCape to exit */
    if(!serial &&
       ((scancode==ESC_SCANCODE) | (scancode==(ESC_SCANCODE | KEY_RELEASE))))
      user_abort=1;

    /* Try to convert the current scancode sequence into barcode information */
    if(cuecat_decode_barcode(&barcode_decoder_state,
                             scancode,cuecat_id,
                             barcode_type,barcode)==BARCODE_DECODED)
    {
      printf("Barcode read by cuecat #%s :\n",cuecat_id);
      printf("Barcode type : %s\n",barcode_type);
      printf("Barcode : %s\n\n",barcode);
      fflush(stdout);
    }
  }
  while(!user_abort);

  /* Put the console or the serial console back like we found it */
  if(!serial)
    restore_console();
  else
    restore_serial_port();

  return(0);
}



/* Signal handler, to stop the program gracefully and avoid screwing up the
   console */
void terminate_program(int x)
{
  fprintf(stderr,"Caught signal %d, restoring ", x);
  if(!serial)
  {
    printf("console ...\n");
    restore_console();
  }
  else
  {
    printf("serial port ...\n");
    restore_serial_port();
  }
  exit(1);
}



/* Restore the keyboard and console as they where when we started */
void restore_console()
{
  if(ioctl(fd,KDSKBMODE,previous_kbd_mode))
  {
    perror("KDSKBMODE");
    close(fd);
    exit(1);
  }
  if(tcsetattr(fd,0,&old_term_settings)==-1)
  {
    perror("tcsetattr");
    close(fd);
    exit(1);
  }
  close(fd);
}



/* Restore the keyboard and console as they where when we started */
void restore_serial_port()
{
  if(ioctl(fd,TIOCSSERIAL,&old_serinfo)==-1)
  {
    perror("TIOCSSERIAL");
    close(fd);
    exit(1);
  }
  close(fd);
}



/* Get a console or serial port descriptor */
int getfd()
{
  int fd;

  /* Does the user want to use a serial pod or the console ? */
  if(!serial)
  {
    fd=open_tty("/dev/tty");
    if(fd>=0)
      return(fd);

    fd=open_tty("/dev/tty0");
    if(fd>=0)
      return(fd);

    fd=open_tty("/dev/console");
    if(fd>=0)
      return(fd);

    for(fd=0;fd<3;fd++)
      if(is_a_console(fd))
        return(fd);

    fprintf(stderr,"Couldnt get a file descriptor referring to the console\n");
  }
  else						/* Try to open a serial port */
  {
    fd=open_tty(serial_dev_filename);
    if(fd>=0)
      return(fd);

    fprintf(stderr,
            "Couldnt get a file descriptor referring to the serial port\n");
  }

  exit(1);		/* total failure */
}



/* Open a tty */
static int open_tty(char *fnam)
{
  int fd;

  fd=open(fnam,O_RDONLY);
  if(fd<0 && errno==EACCES)
    fd=open(fnam,O_WRONLY);
  if(fd<0 || (!serial && !is_a_console(fd)))
    return(-1);
  return(fd);
}



/* Determine if a file descriptor is a console */
static int is_a_console(int fd)
{
  char arg;

  arg=0;
  return(ioctl(fd,KDGKBTYPE,&arg)==0
         && ((arg==KB_101) || (arg==KB_84)));
}



/* Parse the command line parameters */
char parse_cmdline(int argc,char *argv[])
{
  int i,j;
  char got_dash_s;
  char got_parameter;

  /* Parse all the arguments */
  got_dash_s=0;
  for(i=1;i<argc;i++)
  {
    got_parameter=0;
    j=0;
    if(!got_dash_s)
    {
      while(argv[i][j])
      {
        if(argv[i][j]=='-')
        {
          switch(argv[i][j+1])
          {
          case 0:			/* '-' with no letter means nothing */
          case ' ':			/* Neither does "- " */
            return(i);
            break;
  
          case 'h':			/* Help wanted */
            help=1;
            break;
  
          case 's':			/* New serial device filename follows
					   or not */
            serial=1;
            got_dash_s=1;
            break;
          }

          j++;
        }
        else
          if(argv[i][j]!=' ')
            return(i);
  
        j++;
      }
    }
    else				/* The argument is the cfg filename */
    {
      strcpy(serial_dev_filename,argv[i]);
      got_dash_s=0;
    }
  }

  return(0);
}



/* Setup a signal handler for all signals that we need to trap to restore the
   console or the serial port before exiting */
void setup_signal_handlers(void)
{
  signal(SIGHUP,terminate_program);
  signal(SIGINT,terminate_program);
  signal(SIGQUIT,terminate_program);
  signal(SIGILL,terminate_program);
  signal(SIGTRAP,terminate_program);
  signal(SIGABRT,terminate_program);
  signal(SIGIOT,terminate_program);
  signal(SIGFPE,terminate_program);
  signal(SIGKILL,terminate_program);
  signal(SIGUSR1,terminate_program);
  signal(SIGSEGV,terminate_program);
  signal(SIGUSR2,terminate_program);
  signal(SIGPIPE,terminate_program);
  signal(SIGTERM,terminate_program);
#ifdef SIGSTKFLT
  signal(SIGSTKFLT,terminate_program);
#endif
  signal(SIGCHLD,terminate_program);
  signal(SIGCONT,terminate_program);
  signal(SIGSTOP,terminate_program);
  signal(SIGTSTP,terminate_program);
  signal(SIGTTIN,terminate_program);
  signal(SIGTTOU,terminate_program);
}



/* Turn an AT scancode into a XT one, and also watch for and convert keypresses
   and keyreleases */
static int scancode_at_to_xt(int scancode)
{
  static int break_bit;

  if(scancode==0xf0)
  {
    break_bit=0x80;
    return(0);
  }

  scancode=at2xt[scancode];
  scancode|=break_bit;
  break_bit=0;

  return(scancode);
}
