#include "prjlibs-c/standards.h"
#include <sys/stat.h>
#include <unistd.h>
#include <limits.h>
#include <time.h>
#include <errno.h>

#include <skalibs/stddjb.h>
#include "prjlibs-c/constants.h"
#include "prjlibs-c/diewarn.h"
#include "prjlibs-c/warn.h"
#include "prjlibs-c/intattr.h"
#include "prjlibs-c/nsec.h"

char const* PROG="statfile";

static void set(char const* const var, char const* const val) {
  if (env_mexec(var, val)==0) DIE0(alloc);
}

static void set64n(char const* const var, time_t t, unsigned int nsec) {
  tain stamp;
  struct timeval tv;
  char buf[TAIN_PACK*2+1];
  tv.tv_sec=t;
  tv.tv_usec=nsec/1000;
  tain_from_timeval_sysclock(&stamp, &tv) ;
  buf[tain_fmt(buf, &stamp)]='\0';
  set(var, buf);
}

int main(int argc, char** argv) {
  struct stat statbuf;
  char buf[((sizeof (unsigned long))*CHAR_BIT+2)/3+1];
  int use_lstat  =0;
  int set_dev    =0;
  int set_ino    =0;
  int set_type   =0;
  int set_mode   =0;
  int set_nlink  =0;
  int set_uid    =0;
  int set_gid    =0;
  int set_rdev   =0;
  int set_size   =0;
  int set_blksize=0;
  int set_blocks =0;
  int set_atime  =0;
  int set_mtime  =0;
  int set_ctime  =0;

  if (argc<3) DIE_USAGE(" -lditmnugrsSbAMC file [command arg ...]");

  {
    char const* x=argv[1];
    if (*x=='-') ++x;
    while (*x!='\0') {
      switch (*x) {
        case 'l': use_lstat  =1; break;
        case 'd': set_dev    =1; break;
        case 'i': set_ino    =1; break;
        case 't': set_type   =1; break;
        case 'm': set_mode   =1; break;
        case 'n': set_nlink  =1; break;
        case 'u': set_uid    =1; break;
        case 'g': set_gid    =1; break;
        case 'r': set_rdev   =1; break;
        case 's': set_size   =1; break;
        case 'S': set_blksize=1; break;
        case 'b': set_blocks =1; break;
        case 'A': set_atime  =1; break;
        case 'M': set_mtime  =1; break;
        case 'C': set_ctime  =1; break;
        default: {
          char opt[2]={ '\0', '\0' };
          opt[0]=*x;
          DIE2X(100, "unrecognized option: ", opt);
        }
      }
      ++x;
    }
  }

  {
    int result;
    if ('0'<=argv[2][0] && argv[2][0]<='9') {
      unsigned int fd;
      unsigned int const len=uint_scan(argv[2], &fd);
      if (len==0 || argv[2][len]!='\0') DIE_MALFORMED_FD(argv[2]);
      if (fd>INT_MAX) DIE_OVERFLOW();
      result=fstat(fd, &statbuf);
    } else
      result=(use_lstat? lstat: stat)(argv[2], &statbuf);
    if (result!=0) {
      if (argc==3) _exit(DIESTAT("stat"));
      DIE1(stat, argv[2]);
    }
  }

  if (argc==3) _exit(0);

#define doit(member, var, type) do { \
    if (INTATTR_NEGATIVE(type, statbuf.st_##member) || \
        statbuf.st_##member!=(type)(unsigned long)statbuf.st_##member) \
      DIE_OVERFLOW(); \
    if (set_##member) { \
      buf[ulong_fmt(buf, statbuf.st_##member)]='\0'; \
      set(var, buf); \
    } \
  } while (0)

  doit(dev,     "DEV",     dev_t);
  doit(ino,     "INO",     ino_t);
  doit(nlink,   "NLINK",   nlink_t);
  doit(uid,     "UID",     uid_t);
  doit(gid,     "GID",     gid_t);
  doit(rdev,    "RDEV",    dev_t);
  doit(size,    "SIZE",    off_t);
  doit(blksize, "BLKSIZE", unsigned long);
  doit(blocks,  "BLOCKS",  unsigned long);
  doit(atime,   "ATIME",   time_t);
  doit(mtime,   "MTIME",   time_t);
  doit(ctime,   "CTIME",   time_t);

#undef doit

  if (set_atime) set64n("ATAI64N", statbuf.st_atime, nsec_atime(&statbuf));
  if (set_mtime) set64n("MTAI64N", statbuf.st_mtime, nsec_mtime(&statbuf));
  if (set_ctime) set64n("CTAI64N", statbuf.st_ctime, nsec_ctime(&statbuf));

  if (set_type) {
    char const* type;
    #ifdef S_ISREG
      if (S_ISREG(statbuf.st_mode)) type="-"; else
    #endif
    #ifdef S_ISDIR
      if (S_ISDIR(statbuf.st_mode)) type="d"; else
    #endif
    #ifdef S_ISLNK
      if (S_ISLNK(statbuf.st_mode)) type="l"; else
    #endif
    #ifdef S_ISCHR
      if (S_ISCHR(statbuf.st_mode)) type="c"; else
    #endif
    #ifdef S_ISBLK
      if (S_ISBLK(statbuf.st_mode)) type="b"; else
    #endif
    #ifdef S_ISFIFO
      if (S_ISFIFO(statbuf.st_mode)) type="p"; else
    #endif
    #ifdef S_ISSOCK
      if (S_ISSOCK(statbuf.st_mode)) type="s"; else
    #endif
    type="unknown";
    set("TYPE", type);
  }

  if (set_mode) {
    unsigned long const mode=statbuf.st_mode&07777;
    buf[0]='0';
    buf[1]='0'+((mode>>9)&07);
    buf[2]='0'+((mode>>6)&07);
    buf[3]='0'+((mode>>3)&07);
    buf[4]='0'+(mode&07);
    buf[5]='\0';
    set("MODE", buf);
    set("MODE_SUID",   (statbuf.st_mode&04000? "1": NULLP));
    set("MODE_SGID",   (statbuf.st_mode&02000? "1": NULLP));
    set("MODE_STICKY", (statbuf.st_mode&01000? "1": NULLP));
    set("MODE_RUSR",   (statbuf.st_mode&00400? "1": NULLP));
    set("MODE_WUSR",   (statbuf.st_mode&00200? "1": NULLP));
    set("MODE_XUSR",   (statbuf.st_mode&00100? "1": NULLP));
    set("MODE_RGRP",   (statbuf.st_mode&00040? "1": NULLP));
    set("MODE_WGRP",   (statbuf.st_mode&00020? "1": NULLP));
    set("MODE_XGRP",   (statbuf.st_mode&00010? "1": NULLP));
    set("MODE_ROTH",   (statbuf.st_mode&00004? "1": NULLP));
    set("MODE_WOTH",   (statbuf.st_mode&00002? "1": NULLP));
    set("MODE_XOTH",   (statbuf.st_mode&00001? "1": NULLP));
  }

  argv+=3;
  mexec((char const**)argv);
  DIE1(exec, argv[0]);
}
