//Written in the D programming language
/**
 * License: $(HTTP boost.org/LICENSE_1_0.txt, Boost License 1.0).
 *
 * Authors: Dmitry Olshansky
 *
 */
// !!! DO NOT EDIT !!!
// !!! Did you even read the comment? !!!
// This module is automatically generated from Unicode Character Database files
// https://github.com/dlang/phobos/blob/master/tools/unicode_table_generator.d
//dfmt off
module std.internal.unicode_tables;

@safe pure nothrow @nogc package(std):


/// Simple Case Entry, wrapper around uint to extract bit fields from simpleCaseTable()
struct SCE
{
    uint x;

    nothrow @nogc pure @safe:

    this(uint x)
    {
        this.x = x;
    }

    this(uint ch, ubyte n, ubyte size)
    {
        this.x = ch | n << 20 | size << 24;
    }

    int ch() const { return this.x & 0x1FFFF; }
    int n() const { return (this.x >> 20) & 0xF; }
    int size() const { return this.x >> 24; }
}

/// Bit backed FullCaseEntry
struct FCE
{
    ulong x; // bit field sizes: 18, 12, 12, 4, 4, 4

nothrow @nogc pure @safe:

    this(ulong x)
    {
        this.x = x;
    }

    this(dchar[3] seq, ubyte n, ubyte size, ubyte entry_len)
    {
        this.x = ulong(seq[0]) << 36 | ulong(seq[1]) << 24 | seq[2] << 12 | n << 8 | size << 4 | entry_len << 0;
    }

    dchar[3] seq() const { return [(x >> 36) & 0x1FFFF, (x >> 24) & 0xFFF, (x >> 12) & 0xFFF]; }
    ubyte n() const { return (x >> 8) & 0xF; }
    ubyte size() const { return (x >> 4) & 0xF; }
    ubyte entry_len() const { return (x >> 0) & 0xF; }
}

struct UnicodeProperty
{
    string name;
    ubyte[] compressed;
}

struct TrieEntry(T...)
{
    immutable(size_t)[] offsets;
    immutable(size_t)[] sizes;
    immutable(size_t)[] data;
}

SCE simpleCaseTable(size_t i)
{
static immutable uint[] t = cast(immutable uint[]) x"
0201E90B0211E92D0201E9110211E93302000496021004970200A7220210A72302001F7902101FF902001F4402101F4C
0200015A0210015B020010FD02101CBD02016E4C02116E6C02001E3802101E390201E9210211E94302001F2302101F2B
020001A0021001A1030003A3031003C2032003C3020004DC021004DD02002CA602102CA70200017B0210017C0201E906
0211E928020010DC02101C9C020104CD021104F502001F6502101F6D0200051E0210051F020010D802101C9802002C1A
02102C4A0200027D02102C640200A69A0210A69B02001F0202101F0A020005220210052302002C6002102C6102002C1E
02102C4E0200A6500210A6510200A65C0210A65D020013C90210AB99020104140211043C020013CD0210AB9D020000CE
021000EE02001E7E02101E7F020013F0021013F802001EC402101EC50200A78B0210A78C02000114021001150200042F
0210044F0201E9020211E924020004D8021004D902002CAA02102CAB0200011002100111020004920210049302001E34
02101E35020013F4021013FC020002240210022502002CA202102CA30200A72E0210A72F02016E4802116E680200004A
0210006A020013A60210AB760200015E0210015F030001C4031001C5032001C602001F4002101F480200A6580210A659
0200022C0210022D02010C9F02110CDF02001FA702101FAF020013AE0210AB7E0200004202100062020013D50210ABA5
02016E4002116E60020000D2021000F202001E8202101E8302010C9702110CD7020010E002101CA00200013D0210013E
02000406021004560200029D0210A7B2020001390210013A020001980210019902001EBC02101EBD020003AB021003CB
0200040E0210045E0200A7B60210A7B70200040A0210045A02001EC002101EC10201057202110599020024B9021024D3
02002C1602102C460200019C0210026F020003A7021003C70200A6920210A693030000C5031000E50320212B02000526
02100527020002660210A7AA0201057A021105A10201057E021105A502001FD002101FD802000046021000660201E90A
0211E92C0200022802100229020013A20210AB720200A7260210A72702010C9302110CD30200049A0210049B020000D6
021000F60200010C0210010D020118A8021118C802000162021001630200019E0210022002002C9E02102C9F03000392
031003B2032003D002001E3002101E3102001FA302101FAB0200216D0210217D0200004E0210006E030000B50310039C
032003BC0200048E0210048F0200A72A0210A72B02010410021104380200038A021003AF02001EFE02101EFF020118AC
021118CC020001EA021001EB02001F0602101F0E02001FC302101FCC020010B402102D140200026A0210A7AE03000412
0310043203201C800200A7600210A76102010C9B02110CDB020024C8021024E2020001A4021001A50200039F021003BF
020010C402102D24020004A2021004A302002C0302102C3302002CE002102CE1020105760211059D020004E0021004E1
02002C2A02102C5A02001ED002101ED1020000C2021000E2020004E8021004E90200A7A60210A7A7020010F902101CB9
02000145021001460200037C021003FE02001EC802101EC90200041A0210043A020000CA021000EA02001F8602101F8E
030004210310044103201C83020104040211042C02010583021105AA0200053F0210056F0200054702100577020013B6
0210AB86020013EC0210ABBC0200FF2C0210FF4C0200A7BA0210A7BB0200A6480210A649020013C50210AB950200FF24
0210FF440201040F021104370201041C0211044402010C8302110CC3020002410210024202001E7602101E77020010BC
02102D1C02002C2202102C52020010D002101C900200A7680210A7690200A7320210A7330200011C0210011D02000427
0210044702001E4002101E41020118B1021118D10200015202100153030003A1031003C1032003F102000397021003B7
020001AC021001AD020001E2021001E302000218021002190201E91A0211E93C020005160210051702001F2702101F2F
0200A7C90210A7CA02001FE002101FE802002165021021750200A66C0210A66D02000393021003B30200053202100562
020000560210007602001F3302101F3B020005540210058403001C880310A64A0320A64B0200A68E0210A68F020003D8
021003D9020024C5021024DF02002CB202102CB302010C8202110CC2020000570210007702016E5302116E730200FF37
0210FF5702001EF602101EF70201E9120211E93402001F1202101F1A020004A6021004A70200016A0210016B02001F34
02101F3C02002CDC02102CDD020001B10210028A020118A4021118C402001E2802101E29020001BF021001F70200018B
0210018C020010A702102D07020001280210012902001E6A02101E6B020010AB02102D0B0200041F0210043F02002C2E
02102C5E02010CAF02110CEF020104C5021104ED020118A5021118C502001FE102101FE9030003A6031003C6032003D5
02002C0A02102C3A0200050E0210050F0200010402100105020003720210037302001E8E02101E8F02000396021003B6
02001EFA02101EFB020003F2021003F90200A6400210A641020010EC02101CAC020118A1021118C10200016602100167
02001F3702101F3F02010C8602110CC6020013DD0210ABAD02001F5102101F5902002C9A02102C9B020001FA021001FB
020021690210217902016E5702116E77020010A402102D0403000053031000730320017F0200037602100377020013B1
0210AB810200046802100469020013E40210ABB402001F9602101F9E020001080210010902010CA002110CE002001E66
02101E6702001E8A02101E8B02001E4C02101E4D02001F7502101FCB02002C0E02102C3E02001E5002101E51020024C0
021024DA0200A7500210A751020000DA021000FA020010E902101CA902002CD802102CD902001EB802101EB90200012C
0210012D0200025202102C70020004C3021004C4020024C4021024DE020104B5021104DD02002C0F02102C3F02000533
0210056302000195021001F602001F1302101F1B020104B8021104E00200055302100583020104250211044D020013D1
0210ABA1020004F4021004F5020004C7021004C802010CAC02110CEC020004D0021004D10200A75C0210A75D02001E2C
02101E2D020010A802102D0802000464021004650200042602100446020004E4021004E5020010ED02101CAD0200A792
0210A79302001EDC02101EDD020104BC021104E4020105770211059E02001EA802101EA9020001B8021001B9020000C6
021000E60200FF300210FF5002016E4402116E640201041B0211044302001E0E02101E0F020004D4021004D502002CAE
02102CAF02002C6B02102C6C0200FF230210FF430200018402100185020105730211059A04000398041003B8042003D1
043003F4020013D40210ABA40201E9190211E93B020001870210018802002C1302102C430200A7A20210A7A302000194
02100263020004160210043602002CBE02102CBF02001FA602101FAE020104C6021104EE02001F8202101F8A020104C9
021104F1020013E00210ABB00200024002102C7F0200A7BE0210A7BF02001F9202101F9A020004020210045202000536
0210056602000550021005800200A6820210A68302000386021003AC02001FE502101FEC020024C9021024E3020000D3
021000F3020010F002101CB002001E3C02101E3D02002C8002102C810201058A021105B1020001560210015702001E7A
02101E7B020013ED0210ABBD0200A73E0210A73F0200011802100119020002140210021503001E6003101E6103201E9B
02002C8A02102C8B0200A64C0210A64D03000395031003B5032003F5020001E6021001E702001F8502101F8D02000512
0210051302001F6102101F690200A6680210A66902010594021105BB02001E1802101E19020013B20210AB8202010415
0211043D0201041802110440020002450210028C02001F7202101FC8020118B5021118D5020010B802102D18020004C0
021004CF0201040B021104330200042302100443020010C502102D2502002C2D02102C5D02001F3002101F38020013C1
0210AB910200A76C0210A76D020104CC021104F402001F7102101FBB0200FF330210FF530200216A0210217A02001ECC
02101ECD020003E8021003E90200A7D00210A7D10200005A0210007A02010CB002110CF0020001D9021001DA02010587
021105AE020001B5021001B6020005430210057302001F2202101F2A02001EA002101EA102010C8F02110CCF02016E54
02116E7402001D7D02102C630200A7D60210A7D70200A74C0210A74D020013CE0210AB9E020000CF021000EF020010D9
02101C99020000450210006502000136021001370200040D0210045D02016E4302116E6302001F7C02101FFA0200024A
0210024B020004B8021004B902002CCA02102CCB02002CED02102CEE0200019F02100275020003A4021003C40200A74E
0210A74F020104C0021104E802001E1602101E17020013AF0210AB7F020005000210050102002C8202102C83020010BD
02102D1D0201E9160211E93802010CA302110CE302010C9402110CD4020104C2021104EA030003A6031003C6032003D5
02016E4D02116E6D020118B4021118D4020013E90210ABB902001F4102101F49020002870210A7B10200054202100572
02001E5802101E5902001F4302101F4B02002CEB02102CEC020013EB0210ABBB02001FD102101FD902001EE802101EE9
020013A30210AB73020003EE021003EF020118B8021118D802010C9E02110CDE0201E9200211E942020000FF02100178
020104170211043F020118A9021118C902001FB002101FB802002C8802102C89020013A90210AB790201E9090211E92B
020013C20210AB92020024B6021024D002001E5E02101E5F02002C6902102C6A0201041F021104470200037F021003F3
02000540021005700200039002101FD30300004B0310006B0320212A020004F8021004F9020013F3021013FB020000C9
021000E90200040F0210045F020010B502102D150200A7460210A747020000CD021000ED02000476021004770200039E
021003BE02001F2002101F28020010FA02101CBA02002C2102102C5102001EA602101EA702001F8302101F8B0200053C
0210056C02001F0102101F0902002CC202102CC30200020802100209020013E30210ABB30200A7960210A79702001F64
02101F6C02010592021105B9020004BE021004BF02001F9102101F99020003E6021003E702001EE002101EE102001FF3
02101FFC02000413021004330200042E0210044E020104CA021104F202002C0002102C30020010DB02101C9B020010F6
02101CB6020118BE021118DE020001A7021001A8020010C302102D23020004150210043502001E1E02101E1F02000506
0210050702001E6402101E65020000C7021000E70200047C0210047D02001E5202101E53020010F402101CB402010586
021105AD0200046A0210046B02002C2702102C5702010C8902110CC9020010A202102D020200FF320210FF5204000398
041003B8042003D1043003F4030003A9031003C90320212602001EDA02101EDB0201E9010211E9230200A79C0210A79D
020010D302101C93020024BC021024D60200FF2F0210FF4F0200025002102C6F02002C2902102C59020004B2021004B3
0201041902110441020002100210021102002C1502102C4502016E4702116E670200A7540210A75502016E4502116E65
030001C7031001C8032001C90200047E0210047F020001FC021001FD0200020E0210020F020002440210028902001F77
02101FDB02002CD602102CD702010589021105B0040004220410044204201C8404301C85020010E702101CA7020004FA
021004FB020010D502101C9502001EAC02101EAD020010B702102D17020013A10210AB710200A7C20210A7C30201E907
0211E929020013F5021013FD02001EE202101EE302001F6202101F6A02001E9202101E93020001D5021001D602002C06
02102C3602010C8B02110CCB02001F7002101FBA020003EC021003ED0201E9150211E9370200216E0210217E020013C0
0210AB9002001EF402101EF50200FF3A0210FF5A020118BC021118DC020004AA021004AB020021630210217302001E1C
02101E1D020013C80210AB98020118AF021118CF02010C9802110CD8020105710211059802010CA702110CE702000051
02100071020004F2021004F302010CAA02110CEA020010AF02102D0F02002C8E02102C8F02002CC402102CC502001E0A
02101E0B02016E5002116E700201058E021105B50201041102110439020104230211044B0200053A0210056A02001E6C
02101E6D020104BE021104E6020104240211044C020010A902102D09020024C2021024DC02010C8102110CC1020013BC
0210AB8C030004620310046303201C87020118A6021118C602001E4A02101E4B020021620210217202001ED402101ED5
020010AA02102D0A020004EC021004ED02002C0C02102C3C0200A79A0210A79B020004CB021004CC020104BD021104E5
02000370021003710201E9140211E936020003DA021003DB02001F3502101F3D02016E5102116E7103000392031003B2
032003D002002C9602102C9702000391021003B102000534021005640200041B0210043B020013DF0210ABAF02002C2C
02102C5C020001240210012502010CA202110CE202001E8C02101E8D02001EB402101EB50200016C0210016D02001E02
02101E030200019A0210023D020001020210010302001F1402101F1C0200FF280210FF4802010C8002110CC0020010EE
02101CAE02001F5502101F5D0200A77B0210A77C02000059021000790200014A0210014B02001E2402101E25020013BD
0210AB8D02002CBC02102CBD0201040D021104350201058F021105B6020118A7021118C70200A6600210A66102000514
021005150200052E0210052F020004A4021004A5020004000210045002016E5802116E780200A64E0210A64F0200FF2E
0210FF4E0201040C02110434020000DB021000FB020001CD021001CE020010E802101CA80200054F0210057F0200A738
0210A73902016E5F02116E7F02001D790210A77D0201E90F0211E93102002C0702102C37020010A302102D030200A7A0
0210A7A1030004210310044103201C830200A6800210A681020002820210A7C5020024CB021024E50200040102100451
0200A7580210A7590200005802100078020001930210026002010CA102110CE10200054E0210057E0200018602100254
020013DC0210ABAC020024BD021024D7020001B3021001B40200A6960210A697020002160210021702001ECE02101ECF
02002168021021780201E90E0211E93002002C1202102C4202002C2F02102C5F02010CAB02110CEB02001E8602101E87
030004120310043203201C80020013D60210ABA602001EBA02101EBB0201E91B0211E93D0201041A0211044202001EEE
02101EEF0200FF310210FF510200A7BC0210A7BD020001DB021001DC020001A60210028002001F0702101F0F020118BD
021118DD020000C8021000E8020004D2021004D3020010E102101CA1020013B70210AB8702010585021105AC02016E4B
02116E6B02000172021001730200049E0210049F020004E6021004E7020010B002102D1002002C9C02102C9D020001EE
021001EF020104B4021104DC02010C8702110CC7020013BA0210AB8A02001E1002101E11020002300210023102000428
0210044802002CD002102CD1020118A0021118C0020003E0021003E102010588021105AF0200A7800210A781020010F5
02101CB50200053B0210056B020000D5021000F5030004140310043403201C8102001FA402101FAC0200012A0210012B
0200005202100072020024B7021024D10201E9080211E92A04000398041003B8042003D1043003F40200020202100203
0200027102102C6E020004700210047102002C1B02102C4B0200048A0210048B0200A7520210A7530200FF250210FF45
0200040702100457020000C1021000E102001E4402101E4502001F9702101F9F02000548021005780200021C0210021D
020013E50210ABB50200051A0210051B02002CB602102CB70200A6540210A655020104030211042B02001E7202101E73
020010D402101C94020104C1021104E902010C8A02110CCA02001E2A02101E2B030004620310046303201C8702001F76
02101FDA03000053031000730320017F02002C2602102C56020013A00210AB70020004C5021004C6020003FA021003FB
0200055502100585020013E80210ABB8030001C7031001C8032001C902001F4202101F4A02001F6702101F6F0201057C
021105A30200A7B40210A7B50200A6980210A6990201E9040211E9260200040C0210045C020010DA02101C9A020000D0
021000F002016E4602116E6602001EC202101EC3020001E4021001E502000044021000640200017D0210017E02001F21
02101F290200FF390210FF59020013AC0210AB7C02001F0002101F08020003A5021003C50200015802100159020104CF
021104F702001E3A02101E3B0200A72C0210A72D020118B7021118D7020001120210011302001E3602101E3702016E4E
02116E6E02001F7D02101FFB020004940210049502001E8002101E8102000116021001170200042D0210044D020013F2
021013FA020001790210017A020004DA021004DB02002CA802102CA9020000CC021000EC020001A2021001A3030003A1
031003C1032003F102001E7C02101E7D020013CF0210AB9F020104120211043A020005200210052102002C1C02102C4C
0200026B02102C6202010591021105B80200015C0210015D0200A6520210A6530200FF260210FF46020001E8021001E9
020010B602102D160200A6560210A6570200051C0210051D020013C30210AB93020104CB021104F302001EC602101EC7
020001E0021001E102001F6302101F6B02002C0102102C310201E9000211E92202001E7802101E79020024CE021024E8
0200A76A0210A76B0200011A0210011B020004290210044902001F8402101F8C020010D602101C9602001FB102101FB9
0200A7620210A76302010CB202110CF202002C2002102C50020010BE02102D1E0201041E02110446020004E2021004E3
020004DE021004DF020013EA0210ABBA0200054102100571020005450210057502001F2502101F2D0200015402100155
02002C2402102C54020013EE0210ABBE0200018102100253030003A3031003C2032003C3020118B3021118D3020010BA
02102D1A020001800210024302001E3E02101E3F020010FF02101CBF02010C8802110CC8020010D202101C920200A76E
0210A76F03001C880310A64A0320A64B020010B202102D12020013C70210AB9702000518021005190200A7660210A767
020001EC021001ED020104C3021104EB020105740211059B0201E91C0211E93E020010E202101CA202002C0502102C35
020004A0021004A102001E4202101E43020013B40210AB84020118BF021118DF020001500210015102002CB002102CB1
020013A40210AB74020004C1021004C2030001CA031001CB032001CC0200004C0210006C020104C7021104EF02016E5A
02116E7A0200022202100223020004900210049102002CF202102CF302010595021105BC0200053D0210056D02000388
021003AD0200054902100579020013E20210ABB202016E4A02116E6A02001EFC02101EFD02001F9002101F98020004D6
021004D70200A7B80210A7B9030004140310043403201C8102010CAD02110CED020010F702101CB702016E5502116E75
0200A7280210A72902001E0002101E010200023202100233020010A502102D0502001F1502101F1D030003A9031003C9
032021260200013B0210013C0200040802100458020000DC021000FC02001EBE02101EBF02002C1802102C480200A7A8
0210A7A90200216B0210217B02002C1002102C4002002CDE02102CDF0200029E0210A7B0020000D4021000F4020010EF
02101CAF0200055102100581020013DA0210ABAA020005350210056502010C9D02110CDD020024C6021024E002000410
0210043002001EB602101EB7020104160211043E02016E4202116E62020001640210016502002CA002102CA102001E2E
02101E2F02010C9502110CD502000498021004990200010A0210010B020118AA021118CA02001E8802101E8902000528
02100529020013D70210ABA70200A65A0210A65B0200022A0210022B02001FA502101FAD020104B3021104DB0201E90C
0211E92E020024BB021024D5020001F4021001F50200A6900210A6910200FF290210FF4902002C7502102C7602002CB8
02102CB9020104010211042902002C0D02102C3D020104D0021104F80201E9130211E935020001AF021001B00200A7C7
0210A7C8020013BF0210AB8F020104020211042A020010EA02101CAA020104BF021104E70201058D021105B402002164
021021740200018E021001DD020001D3021001D402001F1002101F1802001F5702101F5F02001ED202101ED30200FF2A
0210FF4A02001E0402101E05020004EE021004EF02002C9402102C95020000C0021000E0020013DB0210ABAB020001B2
0210028B020104260211044E0200A6620210A663020000540210007402000556021005860300042A0310044A03201C86
020104220211044A02001F3602101F3E0200A7940210A7C4020004CD021004CE020104BB021104E30200023B0210023C
020001CF021001D00200050C0210050D0200A6460210A64702001E4602101E4702010CA602110CE60200A7790210A77A
0200FF360210FF56020118A2021118C2020004EA021004EB0200041D0210043D02016E5202116E720200A6880210A689
02001EB202101EB3020021600210217002001ED602101ED7020024C3021024DD02016E5E02116E7E020004AC021004AD
0201E9100211E9320200023E02102C6602002CB402102CB50200FF2D0210FF4D0200014C0210014D02001E2202101E23
020013BB0210AB8B020001430210014402010581021105A8020001700210017102001E9402101E9502002C2802102C58
0201040E02110436020104B2021104DA0200A7980210A799020013B80210AB8802001E7002101E710400042204100442
04201C8404301C8502010C8502110CC502001EAE02101EAF020001220210012302010C8C02110CCC020104050211042D
0200042002100440020001AE021002880200021A0210021B0200038C021003CC02001F9502101F9D0400034504100399
042003B904301FBE0200018A0210025702002C9002102C91020010B102102D110200011E0210011F02001E0802101E09
0200A7C00210A7C1020013E70210ABB702010580021105A70200041C0210043C0200FB050210FB060200A7A40210A7A5
02001E7402101E7502002C1402102C440201E90D0211E92F020013A50210AB75020010F302101CB3020024BA021024D4
0200013F021001400200039D021003BD02002CE202102CE3020010A102102D0102000508021005090200004D0210006D
02000389021003AE0200048C0210048D02002C0402102C340200216F0210217F020004B0021004B102002CD202102CD3
0200A7340210A73502010584021105AB0200005002100070020001600210016102010CA902110CE902001E3202101E33
0200020002100201020118AB021118CB0200A7860210A787020118BB021118DB020118AE021118CE0200021E0210021F
020003AA021003CA020010C202102D22020013C40210AB9402000539021005690200A74A0210A74B020000C3021000E3
0200046E0210046F0200054D0210057D020010E302101CA3020003DE021003DF020024CA021024E40400034504100399
042003B904301FBE02010C9902110CD90200A75A0210A75B020004190210043902002C0902102C390200037D021003FF
02001F2602101F2E020118B2021118D2020021830210218402002C2302102C53020001320210013302001E5602101E57
020010AE02102D0E0200A7400210A741020000DD021000FD020010E602101CA603001E6003101E6103201E9B02000478
021004790200022E0210022F0200A6940210A6950200052C0210052D02010570021105970200A6660210A667020013A8
0210AB7802010C9602110CD60200020A0210020B020024BF021024D9020013CB0210AB9B02010C9C02110CDC0300041E
0310043E03201C8202001E8402101E8502001EF002101EF102016E5902116E7902002CA402102CA50200A7240210A725
0200049C0210049D0200010E0210010F020104D3021104FB0201E9030211E9250200019102100192020013DE0210ABAE
030003A0031003C0032003D6020013AD0210AB7D020002060210020702001F4502101F4D020003EA021003EB020010BB
02102D1B02000474021004750200FF380210FF58020004FE021004FF02002C8402102C85020104130211043B02002C1F
02102C4F02001E5C02101E5D02001EE402101EE502002CC802102CC9020104CE021104F6020004BA021004BB0201E905
0211E927020000D1021000F10200047202100473020118B6021118D602001F7802101FF8020000430210006302001EA2
02101EA302001F2402101F2C0200019D02100272020004B6021004B7020000DF02101E9E02001F8702101F8F02001E14
02101E1502001EE602101EE70200040B0210045B02000409021004590201057D021105A4020003A8021003C802001E5A
02101E5B0201057F021105A60200054402100574020010FE02101CBE02001E1202101E13020010B902102D1902000047
02100067020004FC021004FD02010C9202110CD202002C8602102C8702001E1A02101E1B02010C9002110CD0020010BF
02102D1F020013CA0210AB9A02016E4902116E690200004102100061020001BC021001BD02001EEA02101EEB02000502
021005030200024802100249020010DF02101C9F030001C4031001C5032001C60200054602100576020013EF0210ABBF
02002CCC02102CCD020010D702101C97020013A70210AB770200047A0210047B02000134021001350300042A0310044A
03201C8602002C1902102C4902000204021002050200025C0210A7AB020024CF021024E902001E5402101E5502010579
021105A00200054A0210057A0200A6840210A6850200FF270210FF4702001F0502101F0D020021320210214E0200024C
0210024D020013B50210AB8502002CC602102CC7020104080211043002002C1D02102C4D0200023F02102C7E0200A742
0210A74302001E6202101E6302010CA802110CE8030000B50310039C032003BC02001F8102101F8902010C9A02110CDA
0200012E0210012F0200040302100453020001FE021001FF0400034504100399042003B904301FBE0200017402100175
020003CF021003D7020024CD021024E702001FB302101FBC020118AD021118CD0200A7440210A745020000D9021000F9
02001E0C02101E0D02002C1702102C47020003E2021003E30200026C0210A7AD0200A7560210A7570200040502100455
020001970210026802001F6002101F6802001EEC02101EED02002C8C02102C8D020010F202101CB2020104B6021104DE
0201E91D0211E93F0201040A02110432020013C60210AB9602016E4102116E610200042402100444020001A902100283
0300039A031003BA032003F0020010E502101CA5020104C8021104F0020104070211042F020013D00210ABA002001FA2
02101FAA0200046C0210046D020010A002102D000200A7820210A783030000C5031000E50320212B020004B4021004B5
02002CCE02102CCF0200A73A0210A73B0200FF210210FF41020104B9021104E102001EA402101EA50200038F021003CE
020024BE021024D8020003B002101FE302000049021000690201E91F0211E941020002460210024702001F7302101FC9
02002C6702102C6802002166021021760200010002100101020003E4021003E5020013D20210ABA20200A79E0210A79F
02001FA002101FA802010CB102110CF102016E5D02116E7D020010C102102D210200A7F50210A7F6020013BE0210AB8E
0200042C0210044C020010DD02101C9D02001F9302101F9B0200024E0210024F020004BC021004BD0200017602100177
020013AB0210AB7B020013E10210ABB1020005040210050502001F0302101F0B0200A6860210A6870200054C0210057C
02016E4F02116E6F020010EB02101CAB020013D90210ABA90200023A02102C65020004A8021004A902002CDA02102CDB
0300041E0310043E03201C8202000394021003B402002C7202102C73020005100210051102001E2602101E270200A642
0210A643020004C9021004CA020001470210014802001EB002101EB10200A75E0210A75F0201058C021105B30200A73C
0210A73D020104D1021104F9020104D2021104FA02010590021105B7020000550210007502001F1102101F190200037B
021003FD02001F3102101F3902002CBA02102CBB02000552021005820200A68C0210A68D02001E6E02101E6F02002167
02102177020000DE021000FE020001900210025B02001EF802101EF9020104B0021104D802010C8E02110CCE02002C08
02102C380200016E0210016F02016E5C02116E7C020010CD02102D2D02001F7B02101FEB02001E9002101E91020013B3
0210AB83020001D1021001D202001F5302101F5B020001260210012702001E4802101E4902002C9802102C9902010400
0211042802010CA402110CE4020118B9021118D9020013B00210AB8002001ED802101ED902001F7402101FCA04000422
0410044204201C8404301C8502001EF202101EF302002161021021710201042002110448030001F1031001F2032001F3
020001D7021001D80200A68A0210A68B020003F7021003F80200FF350210FF55020003DC021003DD02010C8D02110CCD
02000418021004380200021202100213020004F0021004F1020024C1021024DB0200050A0210050B0201E9170211E939
0200A6440210A6450200053702100567020010AC02102D0C020010AD02102D0D020024C7021024E10200046002100461
02002C2B02102C5B02016E5602116E76020010C702102D27020104060211042E02002CD402102CD5020010F102101CB1
02001F7A02101FEA02002C0B02102C3B0200FF340210FF540200041702100437020118BA021118DA030001F1031001F2
032001F303000395031003B5032003F5020001B7021002920200018F0210025902000538021005680200A66A0210A66B
02001F3202101F3A02010421021104490200052A0210052B02000425021004450200A7840210A7850200004F0210006F
0200A6640210A665020104B1021104D9020000C4021000E402010CA502110CE502001EAA02101EAB020013D80210ABA8
020010E402101CA4020000D8021000F8030001CA031001CB032001CC020010D102101C91020105750211059C0200216C
0210217C02002C0202102C3202016E5B02116E7B0200A7D80210A7D90200FF220210FF420200040402100454020010B3
02102D1302001ECA02101ECB02010409021104310201E9180211E93A020004F6021004F702002CC002102CC102001E20
02101E21020104B7021104DF0200039B021003BB020001960210026902002C2502102C550200A7900210A7910200A764
0210A7650200020C0210020D02001F9402101F9C020118B0021118D0020001820210018302002CAC02102CAD02001F80
02101F88020024CC021024E60200025102102C6D0200048002100481020013E60210ABB6020024B8021024D20200054B
0210057B020013AA0210AB7A0300004B0310006B0320212A020002610210A7AC02000141021001420300039A031003BA
032003F0020010F802101CB8020104C4021104EC0200018902100256020010DE02101C9E02001E6802101E69020104BA
021104E202001F0402101F0C020105780211059F020004110210043102010582021105A902000466021004670200A736
0210A737020000CB021000EB020010C002102D20020104270211044F020001DE021001DF02002C9202102C9302000106
0210010702001E4E02101E4F0200042B0210044B0200A7480210A74902002C1102102C410200A7B30210AB530200053E
0210056E0200014E0210014F020013D30210ABA302000524021005250200A65E0210A65F020001F8021001F9020004AE
021004AF0200A77E0210A77F030003A0031003C0032003D6020002650210A78D0200FF2B0210FF4B02001FA102101FA9
02001EDE02101EDF020013B90210AB8902000168021001690200022602100227020013F1021013F902001E0602101E07
02000120021001210201041D0211044502001F6602101F6E020010A602102D0602010C9102110CD102001D8E0210A7C6
02010C8402110CC402010CAE02110CEE020118A3021118C30200004802100068020013CC0210AB9C0201E91E0211E940
0200038E021003CD0200053102100561";
return SCE(t[i]);
}
@property FCE fullCaseTable(size_t index) nothrow @nogc @safe pure
{
static immutable ulong[] t = cast(immutable ulong[]) x"
001E90B000000021001E92D0000001210010CAE0000000210010CEE00000012100004960000000210000497000000121
001E911000000021001E933000000121000A722000000021000A7230000001210001F790000000210001FF9000000121
0001F440000000210001F4C000000121000015A000000021000015B00000012100010FD0000000210001CBD000000121
0016E4C0000000210016E6C0000001210001E380000000210001E39000000121000FB16000000021000057E576000122
001E921000000021001E9430000001210001F230000000210001F2B00000012100001A000000002100001A1000000121
00003A300000003100003C200000013100003C300000023100004DC00000002100004DD0000001210002CA6000000021
0002CA7000000121000017B000000021000017C000000121001E906000000021001E92800000012100010DC000000021
0001C9C00000012100104CD00000002100104F50000001210001F650000000210001F6D000000121000051E000000021
000051F00000012100010D80000000210001C980000001210002C1A0000000210002C4A000000121000027D000000021
0002C64000000121000A69A000000021000A69B0000001210001F020000000210001F0A0000001210000522000000021
00005230000001210002C600000000210002C610000001210001FB700000002100003B13423B9123000A650000000021
000A651000000121000A65C000000021000A65D0000001210002C1E0000000210002C4E0000001210001FB3000000031
0001FBC00000013100003B13B900023200013C9000000021000AB9900000012100013CD000000021000AB9D000000121
00000CE00000002100000EE0000001210001E7E0000000210001E7F0000001210010414000000021001043C000000121
00013F000000002100013F80000001210001EC40000000210001EC5000000121000A78B000000021000A78C000000121
00001140000000210000115000000121000042F000000021000044F000000121001E902000000021001E924000000121
00004D800000002100004D90000001210002CAA0000000210002CAB00000012100001100000000210000111000000121
000049200000002100004930000001210001E340000000210001E3500000012100013F400000002100013FC000000121
000022400000002100002250000001210002CA20000000210002CA3000000121000A72E000000021000A72F000000121
0016E480000000210016E68000000121000004A000000021000006A00000012100013A6000000021000AB76000000121
000015E000000021000015F00000012100001C400000003100001C500000013100001C60000002310001F40000000021
0001F48000000121000A658000000021000A659000000121000022C000000021000022D0000001210010C9F000000021
0010CDF0000001210001F5000000002100003C53130001220001FA70000000310001FAF0000001310001F673B9000232
0000042000000021000006200000012100013AE000000021000AB7E0000001210016E400000000210016E60000000121
00000D200000002100000F20000001210001E820000000210001E8300000012100013D5000000021000ABA5000000121
00010E00000000210001CA0000000121000013D000000021000013E00000012100004060000000210000456000000121
000029D000000021000A7B20000001210000139000000021000013A00000012100001980000000210000199000000121
0001EBC0000000210001EBD00000012100003AB00000002100003CB000000121000040E000000021000045E000000121
000A7B6000000021000A7B7000000121000040A000000021000045A0000001210001EC00000000210001EC1000000121
0010572000000021001059900000012100024B900000002100024D30000001210002C160000000210002C46000000121
000019C000000021000026F00000012100003A700000002100003C7000000121000A692000000021000A693000000121
00000C500000003100000E5000000131000212B000000231000052600000002100005270000001210000266000000021
000A7AA000000121001057A00000002100105A1000000121001057E00000002100105A50000001210001FD0000000021
0001FD800000012100000460000000210000066000000121001E90A000000021001E92C0000001210000228000000021
000022900000012100013A2000000021000AB72000000121000A726000000021000A7270000001210010C93000000021
0010CD3000000121000049A000000021000049B00000012100000D600000002100000F6000000121000010C000000021
000010D0000001210010C970000000210010CD700000012100001620000000210000163000000121000019E000000021
00002200000001210001E970000000210000074308000122000039200000003100003B200000013100003D0000000231
0001E300000000210001E310000001210001FA30000000310001FAB0000001310001F633B900023200118A8000000021
00118C8000000121000216D000000021000217D0000001210002C9E0000000210002C9F000000121000004E000000021
000006E00000012100000B5000000031000039C00000013100003BC000000231000048E000000021000048F000000121
000A72A000000021000A72B00000012100104100000000210010438000000121000038A00000002100003AF000000121
0001EFE0000000210001EFF00000012100118AC00000002100118CC00000012100001EA00000002100001EB000000121
0001F060000000210001F0E0000001210001FC30000000310001FCC00000013100003B73B900023200010B4000000021
0002D14000000121000026A000000021000A7AE000000121000041200000003100004320000001310001C80000000231
0001F860000000310001F8E0000001310001F063B90002320010C9B0000000210010CDB000000121000A760000000021
000A76100000012100024C800000002100024E200000012100001A400000002100001A5000000121000039F000000021
00003BF00000012100010C40000000210002D2400000012100004A200000002100004A30000001210002C03000000021
0002C330000001210002CE00000000210002CE10000001210010576000000021001059D0000001210001F96000000031
0001F9E0000001310001F263B900023200004E000000002100004E10000001210002C2A0000000210002C5A000000121
0001ED00000000210001ED100000012100000C200000002100000E200000012100004E800000002100004E9000000121
000A7A6000000021000A7A700000012100010F90000000210001CB900000012100001450000000210000146000000121
000037C00000002100003FE0000001210001EC80000000210001EC9000000121000041A000000021000043A000000121
00000CA00000002100000EA0000001210001F860000000310001F8E0000001310001F063B90002320000421000000031
00004410000001310001C830000002310010404000000021001042C000000121000FB02000000021000006606C000122
000053F000000021000056F0000001210000547000000021000057700000012100013B6000000021000AB86000000121
00013EC000000021000ABBC000000121000FF2C000000021000FF4C000000121000A7BA000000021000A7BB000000121
000A648000000021000A64900000012100013C5000000021000AB95000000121000FF24000000021000FF44000000121
001040F0000000210010437000000121001041C0000000210010444000000121001058300000002100105AA000000121
0010C830000000210010CC3000000121000024100000002100002420000001210001E760000000210001E77000000121
00010BC0000000210002D1C0000001210002C220000000210002C5200000012100010D00000000210001C90000000121
000A768000000021000A769000000121000A732000000021000A733000000121000011C000000021000011D000000121
000042700000002100004470000001210001E400000000210001E4100000012100118B100000002100118D1000000121
0000152000000021000015300000012100003A100000003100003C100000013100003F10000002310000397000000021
00003B700000012100001AC00000002100001AD00000012100001E200000002100001E30000001210000218000000021
0000219000000121001E91A000000021001E93C000000121000051600000002100005170000001210001F27000000021
0001F2F000000121000A7C9000000021000A7CA0000001210001FE00000000210001FE80000001210002165000000021
0002175000000121000A66C000000021000A66D000000121000039300000002100003B30000001210000532000000021
0000562000000121000005600000002100000760000001210001F330000000210001F3B0000001210000554000000021
00005840000001210001C88000000031000A64A000000131000A64B000000231000A68E000000021000A68F000000121
00003D800000002100003D900000012100024C500000002100024DF0000001210002CB20000000210002CB3000000121
0010C820000000210010CC2000000121000005700000002100000770000001210016E530000000210016E73000000121
000FF37000000021000FF570000001210001EF60000000210001EF7000000121001E912000000021001E934000000121
0001F120000000210001F1A00000012100004A600000002100004A7000000121000016A000000021000016B000000121
0001F340000000210001F3C0000001210002CDC0000000210002CDD00000012100001B1000000021000028A000000121
00118A400000002100118C40000001210001E280000000210001E2900000012100001BF00000002100001F7000000121
000018B000000021000018C00000012100010A70000000210002D0700000012100001280000000210000129000000121
0001E6A0000000210001E6B00000012100010AB0000000210002D0B000000121000041F000000021000043F000000121
0002C2E0000000210002C5E0000001210010CAF0000000210010CEF00000012100104C500000002100104ED000000121
00118A500000002100118C50000001210001FE10000000210001FE900000012100003A600000003100003C6000000131
00003D50000002310002C0A0000000210002C3A000000121000050E000000021000050F0000001210000104000000021
0000105000000121000037200000002100003730000001210001E8E0000000210001E8F0000001210000396000000021
00003B60000001210001EFA0000000210001EFB00000012100003F200000002100003F9000000121000A640000000021
000A64100000012100010EC0000000210001CAC00000012100118A100000002100118C10000001210001FA7000000031
0001FAF0000001310001F673B9000232000016600000002100001670000001210001F370000000210001F3F000000121
0010C860000000210010CC600000012100013DD000000021000ABAD0000001210001F510000000210001F59000000121
0002C9A0000000210002C9B00000012100001FA00000002100001FB00000012100021690000000210002179000000121
0016E570000000210016E7700000012100010A40000000210002D0400000012100000530000000310000073000000131
000017F0000002310000376000000021000037700000012100013B1000000021000AB810000001210000468000000021
000046900000012100013E4000000021000ABB40000001210001F960000000310001F9E0000001310001F263B9000232
000010800000002100001090000001210010CA00000000210010CE00000001210001E660000000210001E67000000121
0001E8A0000000210001E8B0000001210001E4C0000000210001E4D0000001210001F750000000210001FCB000000121
0002C0E0000000210002C3E0000001210001E500000000210001E5100000012100024C000000002100024DA000000121
000A750000000021000A75100000012100000DA00000002100000FA00000012100010E90000000210001CA9000000121
0002CD80000000210002CD90000001210001EB80000000210001EB9000000121000012C000000021000012D000000121
00002520000000210002C7000000012100004C300000002100004C400000012100003B00000000310001FE3000000131
00003C530830123300024C400000002100024DE0000001210002C0F0000000210002C3F0000001210000533000000021
0000563000000121000019500000002100001F60000001210001F130000000210001F1B00000012100104B8000000021
00104E0000000121000055300000002100005830000001210010425000000021001044D00000012100013D1000000021
000ABA100000012100004F400000002100004F500000012100004C700000002100004C800000012100104B5000000021
00104DD00000012100004D000000002100004D1000000121000A75C000000021000A75D0000001210010CAC000000021
0010CEC0000001210001E2C0000000210001E2D00000012100010A80000000210002D080000001210000464000000021
00004650000001210000426000000021000044600000012100004E400000002100004E500000012100010ED000000021
0001CAD000000121000A792000000021000A7930000001210001EDC0000000210001EDD00000012100104BC000000021
00104E40000001210010577000000021001059E0000001210001EA80000000210001EA900000012100001B8000000021
00001B900000012100000C600000002100000E6000000121000FF30000000021000FF500000001210016E44000000021
0016E64000000121001041B00000002100104430000001210001E0E0000000210001E0F00000012100004D4000000021
00004D50000001210002CAE0000000210002CAF0000001210002C6B0000000210002C6C000000121000FF23000000021
000FF43000000121000018400000002100001850000001210010573000000021001059A0000001210000398000000041
00003B800000014100003D100000024100003F400000034100013D4000000021000ABA4000000121001E919000000021
001E93B000000121000018700000002100001880000001210002C130000000210002C43000000121000A7A2000000021
000A7A300000012100001940000000210000263000000121000041600000002100004360000001210002CBE000000021
0002CBF0000001210001FA60000000310001FAE0000001310001F663B900023200104C600000002100104EE000000121
0001F820000000310001F8A0000001310001F023B900023200104C900000002100104F100000012100013E0000000021
000ABB000000012100002400000000210002C7F000000121000A7BE000000021000A7BF0000001210001F92000000031
0001F9A0000001310001F223B90002320000402000000021000045200000012100005360000000210000566000000121
00005500000000210000580000000121000A682000000021000A683000000121000038600000002100003AC000000121
0001FE50000000210001FEC00000012100024C900000002100024E300000012100000D300000002100000F3000000121
00010F00000000210001CB00000001210001E3C0000000210001E3D0000001210002C800000000210002C81000000121
0001F970000000310001F9F0000001310001F273B9000232001058A00000002100105B10000001210000156000000021
00001570000001210001E7A0000000210001E7B00000012100013ED000000021000ABBD000000121000A73E000000021
000A73F00000012100001180000000210000119000000121000021400000002100002150000001210001E60000000031
0001E610000001310001E9B0000002310002C8A0000000210002C8B000000121000A64C000000021000A64D000000121
000039500000003100003B500000013100003F500000023100001E600000002100001E70000001210001F85000000031
0001F8D0000001310001F053B9000232000051200000002100005130000001210001F610000000210001F69000000121
000A668000000021000A669000000121001059400000002100105BB0000001210001E180000000210001E19000000121
00013B2000000021000AB820000001210010415000000021001043D00000012100104180000000210010440000000121
0000245000000021000028C0000001210001F720000000210001FC800000012100118B500000002100118D5000000121
00010B80000000210002D1800000012100004C000000002100004CF000000121001040B0000000210010433000000121
0000423000000021000044300000012100010C50000000210002D250000001210002C2D0000000210002C5D000000121
0001F300000000210001F3800000012100013C1000000021000AB91000000121000A76C000000021000A76D000000121
00104CC00000002100104F40000001210001F710000000210001FBB000000121000FF33000000021000FF53000000121
000216A000000021000217A000000121000014900000002100002BC06E00012200003E800000002100003E9000000121
000A7D0000000021000A7D1000000121000005A000000021000007A0000001210010CB00000000210010CF0000000121
00001D900000002100001DA0000001210001ECC0000000210001ECD00000012100001B500000002100001B6000000121
000054300000002100005730000001210001F220000000210001F2A000000121001058700000002100105AE000000121
0001EA00000000210001EA10000001210010C8F0000000210010CCF0000001210016E540000000210016E74000000121
0001FB400000002100003AC3B90001220001D7D0000000210002C63000000121000A7D6000000021000A7D7000000121
000A74C000000021000A74D00000012100013CE000000021000AB9E00000012100000CF00000002100000EF000000121
00010D90000000210001C990000001210000045000000021000006500000012100001360000000210000137000000121
000040D000000021000045D0000001210016E430000000210016E630000001210001F7C0000000210001FFA000000121
000024A000000021000024B00000012100004B800000002100004B90000001210002CCA0000000210002CCB000000121
0002CED0000000210002CEE000000121000019F000000021000027500000012100003A400000002100003C4000000121
000FB17000000021000057456D000122000A74E000000021000A74F00000012100104C000000002100104E8000000121
0001E160000000210001E1700000012100013AF000000021000AB7F00000012100005000000000210000501000000121
0002C820000000210002C8300000012100010BD0000000210002D1D000000121001E916000000021001E938000000121
0010CA30000000210010CE30000001210010C940000000210010CD400000012100104C200000002100104EA000000121
00003A600000003100003C600000013100003D50000002310001F930000000310001F9B0000001310001F233B9000232
00118B400000002100118D400000012100013E9000000021000ABB90000001210016E4D0000000210016E6D000000121
0001F410000000210001F490000001210000287000000021000A7B100000012100005420000000210000572000000121
0001E580000000210001E590000001210001F430000000210001F4B0000001210001F910000000310001F99000000131
0001F213B90002320002CEB0000000210002CEC0000001210001FD10000000210001FD90000001210001EE8000000021
0001EE900000012100013A3000000021000AB7300000012100003EE00000002100003EF00000012100118B8000000021
00118D80000001210010C9E0000000210010CDE00000012100013EB000000021000ABBB00000012100000FF000000021
00001780000001210010417000000021001043F00000012100118A900000002100118C90000001210001FB0000000021
0001FB80000001210002C880000000210002C8900000012100013A9000000021000AB79000000121001E909000000021
001E92B00000012100013C2000000021000AB9200000012100024B600000002100024D00000001210001E5E000000021
0001E5F0000001210002C690000000210002C6A000000121001041F0000000210010447000000121000037F000000021
00003F30000001210000540000000021000057000000012100003900000000310001FD300000013100003B9308301233
000004B000000031000006B000000131000212A00000023100004F800000002100004F900000012100013F3000000021
00013FB00000012100000C900000002100000E9000000121000040F000000021000045F00000012100010B5000000021
0002D15000000121000A746000000021000A74700000012100000CD00000002100000ED0000001210000476000000021
0000477000000121000039E00000002100003BE0000001210001F200000000210001F2800000012100010FA000000021
0001CBA0000001210002C210000000210002C510000001210001EA60000000210001EA70000001210001F83000000031
0001F8B0000001310001F033B9000232000053C000000021000056C0000001210001F010000000210001F09000000121
0002CC20000000210002CC30000001210000208000000021000020900000012100013E3000000021000ABB3000000121
000A796000000021000A7970000001210001F640000000210001F6C000000121001059200000002100105B9000000121
00004BE00000002100004BF0000001210001F910000000310001F990000001310001F213B900023200003E6000000021
00003E70000001210001EE00000000210001EE10000001210001FF30000000310001FFC00000013100003C93B9000232
00001300000000210000069307000122000041300000002100004330000001210001E98000000021000007730A000122
000042E000000021000044E00000012100010DB0000000210001C9B00000012100010F60000000210001CB6000000121
0002C000000000210002C3000000012100118BE00000002100118DE00000012100001A700000002100001A8000000121
00010C30000000210002D23000000121000041500000002100004350000001210001E1E0000000210001E1F000000121
000050600000002100005070000001210001E640000000210001E6500000012100000C700000002100000E7000000121
000047C000000021000047D0000001210001E520000000210001E5300000012100010F40000000210001CB4000000121
001058600000002100105AD000000121000046A000000021000046B0000001210002C270000000210002C57000000121
00104CA00000002100104F200000012100010A20000000210002D020000001210010C890000000210010CC9000000121
000FF32000000021000FF52000000121000039800000004100003B800000014100003D100000024100003F4000000341
00003A900000003100003C900000013100021260000002310001EDA0000000210001EDB000000121001E901000000021
001E923000000121000A79C000000021000A79D00000012100010D30000000210001C9300000012100024BC000000021
00024D6000000121000FF2F000000021000FF4F00000012100002500000000210002C6F0000001210002C29000000021
0002C59000000121001E920000000021001E94200000012100004B200000002100004B30000001210010419000000021
0010441000000121000021000000002100002110000001210002C150000000210002C450000001210016E47000000021
0016E67000000121000A754000000021000A7550000001210016E450000000210016E6500000012100001C7000000031
00001C800000013100001C9000000231000047E000000021000047F00000012100001FC00000002100001FD000000121
000020E000000021000020F000000121000024400000002100002890000001210001F770000000210001FDB000000121
0002CD60000000210002CD7000000121001058900000002100105B000000012100004220000000410000442000000141
0001C840000002410001C8500000034100010E70000000210001CA700000012100004FA00000002100004FB000000121
00010D50000000210001C950000001210001EAC0000000210001EAD00000012100010B70000000210002D17000000121
00013A1000000021000AB71000000121000A7C2000000021000A7C3000000121001E907000000021001E929000000121
00013F500000002100013FD0000001210001EE20000000210001EE30000001210001F620000000210001F6A000000121
0001E920000000210001E9300000012100001D500000002100001D60000001210002C060000000210002C36000000121
0010C8B0000000210010CCB0000001210001F700000000210001FBA00000012100003EC00000002100003ED000000121
001E915000000021001E937000000121000216E000000021000217E00000012100013C0000000021000AB90000000121
0001EF40000000210001EF50000001210001FB20000000210001F703B9000122000FF3A000000021000FF5A000000121
00118BC00000002100118DC00000012100004AA00000002100004AB00000012100021630000000210002173000000121
0001E1C0000000210001E1D00000012100013C8000000021000AB9800000012100118AF00000002100118CF000000121
0001E9A00000002100000612BE000122001057100000002100105980000001210010CA70000000210010CE7000000121
0010C980000000210010CD80000001210000051000000021000007100000012100004F200000002100004F3000000121
0010CAA0000000210010CEA00000012100010AF0000000210002D0F0000001210002C8E0000000210002C8F000000121
0002CC40000000210002CC50000001210001E0A0000000210001E0B000000121000FB15000000021000057456B000122
0016E500000000210016E70000000121001058E00000002100105B500000012100104110000000210010439000000121
0010423000000021001044B000000121000053A000000021000056A0000001210001E6C0000000210001E6D000000121
00104BE00000002100104E60000001210010424000000021001044C00000012100010A90000000210002D09000000121
00024C200000002100024DC0000001210010C810000000210010CC100000012100013BC000000021000AB8C000000121
000046200000003100004630000001310001C8700000023100118A600000002100118C60000001210001E4A000000021
0001E4B000000121000216200000002100021720000001210001ED40000000210001ED500000012100010AA000000021
0002D0A00000012100004EC00000002100004ED0000001210002C0C0000000210002C3C000000121000A79A000000021
000A79B00000012100004CB00000002100004CC00000012100104BD00000002100104E50000001210000370000000021
00003710000001210001FE700000002100003C5308342123001E914000000021001E93600000012100003DA000000021
00003DB0000001210001F350000000210001F3D0000001210016E510000000210016E710000001210000392000000031
00003B200000013100003D00000002310002C960000000210002C970000001210001FA50000000310001FAD000000131
0001F653B9000232000039100000002100003B100000012100005340000000210000564000000121000041B000000021
000043B00000012100013DF000000021000ABAF0000001210002C2C0000000210002C5C0000001210000124000000021
00001250000001210010CA20000000210010CE20000001210001E8C0000000210001E8D0000001210001EB4000000021
0001EB5000000121000016C000000021000016D0000001210001E020000000210001E03000000121000019A000000021
000023D000000121000010200000002100001030000001210001F140000000210001F1C000000121000FF28000000021
000FF480000001210010C800000000210010CC000000012100010EE0000000210001CAE0000001210001F55000000021
0001F5D000000121000A77B000000021000A77C0000001210001FC700000002100003B73423B91230000059000000021
0000079000000121000014A000000021000014B0000001210001E240000000210001E2500000012100013BD000000021
000AB8D0000001210002CBC0000000210002CBD000000121001040D0000000210010435000000121001058F000000021
00105B600000012100118A700000002100118C7000000121000A660000000021000A6610000001210000514000000021
0000515000000121000052E000000021000052F00000012100004A400000002100004A50000001210000400000000021
00004500000001210016E580000000210016E78000000121000A64E000000021000A64F000000121000FF2E000000021
000FF4E000000121001040C000000021001043400000012100000DB00000002100000FB00000012100001CD000000021
00001CE00000012100010E80000000210001CA8000000121000054F000000021000057F000000121000A738000000021
000A7390000001210016E5F0000000210016E7F0000001210001FA60000000310001FAE0000001310001F663B9000232
001E90F000000021001E9310000001210002C070000000210002C3700000012100010A30000000210002D03000000121
000A7A0000000021000A7A1000000121000042100000003100004410000001310001C83000000231000A680000000021
000A6810000001210000282000000021000A7C50000001210001D79000000021000A77D00000012100024CB000000021
00024E500000012100004010000000210000451000000121000A758000000021000A7590000001210000058000000021
0000078000000121000019300000002100002600000001210010CA10000000210010CE1000000121000054E000000021
000057E0000001210000186000000021000025400000012100013DC000000021000ABAC00000012100024BD000000021
00024D700000012100001B300000002100001B4000000121000A696000000021000A6970000001210000216000000021
00002170000001210001ECE0000000210001ECF000000121000216800000002100021780000001210001FC6000000021
00003B73420001220002C120000000210002C420000001210002C2F0000000210002C5F000000121001E90E000000021
001E9300000001210010CAB0000000210010CEB0000001210001E860000000210001E870000001210000412000000031
00004320000001310001C8000000023100013D6000000021000ABA60000001210001EBA0000000210001EBB000000121
001E91B000000021001E93D000000121001041A00000002100104420000001210001EEE0000000210001EEF000000121
000FF31000000021000FF51000000121000A7BC000000021000A7BD00000012100001DB00000002100001DC000000121
00001A600000002100002800000001210001F070000000210001F0F00000012100118BD00000002100118DD000000121
00000C800000002100000E800000012100004D200000002100004D300000012100010E10000000210001CA1000000121
00013B7000000021000AB87000000121001058500000002100105AC0000001210016E4B0000000210016E6B000000121
00001720000000210000173000000121000049E000000021000049F00000012100004E600000002100004E7000000121
00010B00000000210002D100000001210001E99000000021000007930A00012200001EE00000002100001EF000000121
0002C9C0000000210002C9D00000012100104B400000002100104DC0000001210010C870000000210010CC7000000121
00013BA000000021000AB8A0000001210001E100000000210001E1100000012100002300000000210000231000000121
000042800000002100004480000001210002CD00000000210002CD100000012100118A000000002100118C0000000121
00003E000000002100003E10000001210001FF400000002100003CE3B9000122000A780000000021000A781000000121
00010F50000000210001CB5000000121000053B000000021000056B00000012100000D500000002100000F5000000121
001058800000002100105AF000000121000041400000003100004340000001310001C810000002310001FA4000000031
0001FAC0000001310001F643B9000232000012A000000021000012B00000012100000520000000210000072000000121
00024B700000002100024D1000000121001E908000000021001E92A000000121000039800000004100003B8000000141
00003D100000024100003F4000000341000020200000002100002030000001210001F850000000310001F8D000000131
0001F053B90002320000470000000021000047100000012100002710000000210002C6E0000001210002C1B000000021
0002C4B000000121000FB030000000210000066066069123000048A000000021000048B000000121000A752000000021
000A753000000121000FF25000000021000FF450000001210000407000000021000045700000012100000C1000000021
00000E10000001210001E440000000210001E450000001210001F970000000310001F9F0000001310001F273B9000232
00005480000000210000578000000121000021C000000021000021D00000012100013E5000000021000ABB5000000121
000051A000000021000051B0000001210002CB60000000210002CB7000000121000A654000000021000A655000000121
0010403000000021001042B0000001210001E720000000210001E7300000012100010D40000000210001C94000000121
00104C100000002100104E90000001210010C8A0000000210010CCA0000001210001E2A0000000210001E2B000000121
000046200000003100004630000001310001C870000002310001F760000000210001FDA0000001210000053000000031
0000073000000131000017F0000002310002C260000000210002C5600000012100013A0000000021000AB70000000121
00003FA00000002100003FB00000012100004C500000002100004C60000001210001F920000000310001F9A000000131
0001F223B90002320000555000000021000058500000012100013E8000000021000ABB800000012100001C7000000031
00001C800000013100001C90000002310001F420000000210001F4A0000001210001F670000000210001F6F000000121
001057C00000002100105A3000000121000A7B4000000021000A7B5000000121000A698000000021000A699000000121
001E904000000021001E926000000121000040C000000021000045C00000012100010DA0000000210001C9A000000121
0001FD600000002100003B934200012200000D000000002100000F00000001210016E460000000210016E66000000121
0001EC20000000210001EC300000012100001E400000002100001E500000012100000440000000210000064000000121
000017D000000021000017E0000001210001F210000000210001F29000000121000FF39000000021000FF59000000121
00013AC000000021000AB7C000000121000058700000002100005655820001220001F000000000210001F08000000121
00003A500000002100003C50000001210000158000000021000015900000012100104CF00000002100104F7000000121
0001E3A0000000210001E3B000000121000A72C000000021000A72D00000012100118B700000002100118D7000000121
000011200000002100001130000001210001E360000000210001E370000001210016E4E0000000210016E6E000000121
0001F7D0000000210001FFB000000121000049400000002100004950000001210001E800000000210001E81000000121
00001160000000210000117000000121000042D000000021000044D00000012100013F200000002100013FA000000121
0000179000000021000017A00000012100004DA00000002100004DB0000001210002CA80000000210002CA9000000121
00000CC00000002100000EC00000012100001A200000002100001A300000012100003A100000003100003C1000000131
00003F10000002310001E7C0000000210001E7D00000012100013CF000000021000AB9F0000001210010412000000021
001043A000000121000052000000002100005210000001210002C1C0000000210002C4C000000121000026B000000021
0002C62000000121000FB140000000210000574565000122000015C000000021000015D000000121000A652000000021
000A653000000121000FF26000000021000FF4600000012100001E800000002100001E900000012100010B6000000021
0002D16000000121000A656000000021000A657000000121000051C000000021000051D00000012100013C3000000021
000AB93000000121001059100000002100105B800000012100104CB00000002100104F30000001210001EC6000000021
0001EC700000012100001E000000002100001E10000001210001F630000000210001F6B0000001210002C01000000021
0002C31000000121001E900000000021001E9220000001210001E780000000210001E7900000012100024CE000000021
00024E8000000121000A76A000000021000A76B000000121000011A000000021000011B0000001210000429000000021
00004490000001210001F840000000310001F8C0000001310001F043B900023200010D60000000210001C96000000121
0001FB10000000210001FB9000000121000A762000000021000A7630000001210010CB20000000210010CF2000000121
0002C200000000210002C5000000012100010BE0000000210002D1E000000121001041E0000000210010446000000121
0001FF700000002100003C93423B912300004E200000002100004E30000001210001F900000000310001F98000000131
0001F203B900023200004DE00000002100004DF0000001210001F940000000310001F9C0000001310001F243B9000232
00005410000000210000571000000121000054500000002100005750000001210001F250000000210001F2D000000121
000015400000002100001550000001210002C240000000210002C5400000012100013EA000000021000ABBA000000121
0000181000000021000025300000012100003A300000003100003C200000013100003C300000023100118B3000000021
00118D300000012100010BA0000000210002D1A000000121000018000000002100002430000001210001E3E000000021
0001E3F00000012100010FF0000000210001CBF0000001210001FF30000000310001FFC00000013100003C93B9000232
00010D20000000210001C92000000121000A76E000000021000A76F0000001210010C880000000210010CC8000000121
0001C88000000031000A64A000000131000A64B00000023100010B20000000210002D1200000012100013C7000000021
000AB9700000012100005180000000210000519000000121000A766000000021000A76700000012100001EC000000021
00001ED00000012100104C300000002100104EB0000001210010574000000021001059B00000012100013EE000000021
000ABBE000000121001E91C000000021001E93E00000012100010E20000000210001CA20000001210002C05000000021
0002C3500000012100004A000000002100004A10000001210001E420000000210001E4300000012100013B4000000021
000AB8400000012100118BF00000002100118DF000000121000015000000002100001510000001210001F56000000021
00003C53133421230002CB00000000210002CB100000012100013A4000000021000AB7400000012100004C1000000021
00004C200000012100001CA00000003100001CB00000013100001CC000000231000004C000000021000006C000000121
00104C700000002100104EF0000001210016E5A0000000210016E7A00000012100002220000000210000223000000121
000049000000002100004910000001210002CF20000000210002CF3000000121001059500000002100105BC000000121
000053D000000021000056D000000121000038800000002100003AD00000012100005490000000210000579000000121
00013E2000000021000ABB20000001210016E4A0000000210016E6A0000001210001EFC0000000210001EFD000000121
0001F900000000310001F980000001310001F203B900023200004D600000002100004D7000000121000A7B8000000021
000A7B9000000121000041400000003100004340000001310001C810000002310010CAD0000000210010CED000000121
00010F70000000210001CB70000001210016E550000000210016E75000000121000A728000000021000A729000000121
0001E000000000210001E010000001210000232000000021000023300000012100010A50000000210002D05000000121
0001F150000000210001F1D00000012100003A900000003100003C90000001310002126000000231000013B000000021
000013C0000001210000408000000021000045800000012100000DC00000002100000FC0000001210001EBE000000021
0001EBF0000001210002C180000000210002C48000000121000A7A8000000021000A7A9000000121000216B000000021
000217B0000001210002C100000000210002C400000001210002CDE0000000210002CDF000000121000029E000000021
000A7B000000012100000D400000002100000F400000012100010EF0000000210001CAF0000001210000551000000021
000058100000012100013DA000000021000ABAA000000121000053500000002100005650000001210001FA0000000031
0001FA80000001310001F603B900023200024C600000002100024E000000012100004100000000210000430000000121
0001EB60000000210001EB70000001210001F840000000310001F8C0000001310001F043B90002320010416000000021
001043E0000001210016E420000000210016E62000000121000016400000002100001650000001210002CA0000000021
0002CA10000001210001E2E0000000210001E2F0000001210010C950000000210010CD50000001210000498000000021
0000499000000121000010A000000021000010B0000001210010C9D0000000210010CDD00000012100118AA000000021
00118CA0000001210001E880000000210001E890000001210000528000000021000052900000012100013D7000000021
000ABA7000000121000A65A000000021000A65B000000121000022A000000021000022B0000001210001FA5000000031
0001FAD0000001310001F653B900023200104B300000002100104DB000000121001E90C000000021001E92E000000121
00024BB00000002100024D500000012100001F400000002100001F5000000121000A690000000021000A691000000121
000FF29000000021000FF490000001210002C750000000210002C760000001210002CB80000000210002CB9000000121
001040100000002100104290000001210002C0D0000000210002C3D00000012100104D000000002100104F8000000121
001E913000000021001E93500000012100001AF00000002100001B0000000121000A7C7000000021000A7C8000000121
00013BF000000021000AB8F0000001210010402000000021001042A00000012100010EA0000000210001CAA000000121
00104BF00000002100104E7000000121001058D00000002100105B400000012100021640000000210002174000000121
000018E00000002100001DD00000012100001D300000002100001D40000001210001F100000000210001F18000000121
0001F570000000210001F5F0000001210001ED20000000210001ED3000000121000FF2A000000021000FF4A000000121
0001E040000000210001E0500000012100004EE00000002100004EF0000001210002C940000000210002C95000000121
00000C000000002100000E00000001210001FE600000002100003C534200012200013DB000000021000ABAB000000121
00001B2000000021000028B0000001210010426000000021001044E000000121000A662000000021000A663000000121
0000054000000021000007400000012100005560000000210000586000000121000042A000000031000044A000000131
0001C860000002310010422000000021001044A0000001210001F360000000210001F3E0000001210001FA1000000031
0001FA90000001310001F613B900023200004CD00000002100004CE00000012100104BB00000002100104E3000000121
000023B000000021000023C00000012100001CF00000002100001D0000000121000050C000000021000050D000000121
000A646000000021000A6470000001210001E460000000210001E470000001210010CA60000000210010CE6000000121
000A779000000021000A77A000000121000FF36000000021000FF5600000012100118A200000002100118C2000000121
00004EA00000002100004EB000000121000041D000000021000043D000000121000A794000000021000A7C4000000121
000A688000000021000A6890000001210001EB20000000210001EB300000012100021600000000210002170000000121
0001ED60000000210001ED700000012100024C300000002100024DD0000001210016E5E0000000210016E7E000000121
00004AC00000002100004AD0000001210016E520000000210016E72000000121000023E0000000210002C66000000121
0002CB40000000210002CB5000000121000FF2D000000021000FF4D000000121000014C000000021000014D000000121
0001E220000000210001E2300000012100013BB000000021000AB8B00000012100001430000000210000144000000121
000FB04000000021000006606606C123001058100000002100105A800000012100001700000000210000171000000121
0001E940000000210001E950000001210001FC20000000210001F743B90001220002C280000000210002C58000000121
001E910000000021001E932000000121001040E00000002100104360000001210001F830000000310001F8B000000131
0001F033B9000232000A798000000021000A79900000012100013B8000000021000AB880000001210001E70000000021
0001E71000000121000042200000004100004420000001410001C840000002410001C850000003410010C85000000021
0010CC50000001210001EAE0000000210001EAF0000001210000122000000021000012300000012100104B2000000021
00104DA0000001210010C8C0000000210010CCC0000001210010405000000021001042D0000001210001F87000000031
0001F8F0000001310001F073B90002320000420000000021000044000000012100001AE0000000210000288000000121
000021A000000021000021B000000121000038C00000002100003CC0000001210001F950000000310001F9D000000131
0001F253B90002320000345000000041000039900000014100003B90000002410001FBE000000341000018A000000021
000025700000012100003900000000310001FD300000013100003B930830123300010B10000000210002D11000000121
000011E000000021000011F0000001210001E080000000210001E090000001210002C900000000210002C91000000121
000A7C0000000021000A7C100000012100013E7000000021000ABB7000000121001058000000002100105A7000000121
000041C000000021000043C000000121000FB05000000031000FB060000001310000073074000232000A7A4000000021
000A7A50000001210001E740000000210001E750000001210002C140000000210002C44000000121001E90D000000021
001E92F00000012100013A5000000021000AB7500000012100010F30000000210001CB300000012100024BA000000021
00024D4000000121000013F0000000210000140000000121000039D00000002100003BD0000001210002CE2000000021
0002CE300000012100010A10000000210002D0100000012100005080000000210000509000000121000004D000000021
000006D000000121000038900000002100003AE000000121000048C000000021000048D0000001210002C04000000021
0002C34000000121000216F000000021000217F00000012100004B000000002100004B10000001210002CD2000000021
0002CD3000000121000A734000000021000A735000000121000FB0100000002100000660690001220000050000000021
000007000000012100001600000000210000161000000121001058400000002100105AB0000001210001E32000000021
0001E330000001210010CA90000000210010CE90000001210000200000000021000020100000012100118AB000000021
00118CB000000121000A786000000021000A78700000012100118BB00000002100118DB00000012100118AE000000021
00118CE000000121000021E000000021000021F00000012100003AA00000002100003CA00000012100010C2000000021
0002D2200000012100013C4000000021000AB9400000012100005390000000210000569000000121000A74A000000021
000A74B00000012100000C300000002100000E3000000121000046E000000021000046F000000121000054D000000021
000057D0000001210001FE200000002100003C530830012300010E30000000210001CA30000001210001F52000000021
00003C531330012300003DE00000002100003DF00000012100024CA00000002100024E40000001210000345000000041
000039900000014100003B90000002410001FBE0000003410010C990000000210010CD9000000121000A75A000000021
000A75B000000121000041900000002100004390000001210002C090000000210002C39000000121000037D000000021
00003FF0000001210001F260000000210001F2E00000012100118B200000002100118D20000001210002183000000021
00021840000001210002C230000000210002C53000000121000013200000002100001330000001210001E56000000021
0001E5700000012100010AE0000000210002D0E000000121000A740000000021000A74100000012100000DD000000021
00000FD00000012100010E60000000210001CA60000001210001E600000000310001E610000001310001E9B000000231
00004780000000210000479000000121000022E000000021000022F0000001210001FF600000002100003C9342000122
0001FD200000002100003B9308300123000A694000000021000A695000000121000052C000000021000052D000000121
00105700000000210010597000000121000A666000000021000A66700000012100001F0000000021000006A30C000122
0010C960000000210010CD6000000121000020A000000021000020B00000012100024BF00000002100024D9000000121
00013CB000000021000AB9B0000001210010C9C0000000210010CDC000000121000041E000000031000043E000000131
0001C820000002310001E840000000210001E850000001210001EF00000000210001EF100000012100013A8000000021
000AB780000001210016E590000000210016E790000001210002CA40000000210002CA5000000121000A724000000021
000A725000000121000049C000000021000049D000000121000010E000000021000010F0000001210001FA4000000031
0001FAC0000001310001F643B900023200104D300000002100104FB000000121001E903000000021001E925000000121
000019100000002100001920000001210001F800000000310001F880000001310001F003B900023200013DE000000021
000ABAE00000012100003A000000003100003C000000013100003D600000023100013AD000000021000AB7D000000121
000020600000002100002070000001210001F450000000210001F4D00000012100003EA00000002100003EB000000121
00010BB0000000210002D1B00000012100004740000000210000475000000121000FF38000000021000FF58000000121
00004FE00000002100004FF0000001210002C840000000210002C850000001210010413000000021001043B000000121
0001FD700000002100003B93083421230001E5C0000000210001E5D0000001210001EE40000000210001EE5000000121
0002CC80000000210002CC90000001210002C1F0000000210002C4F00000012100004BA00000002100004BB000000121
00104CE00000002100104F6000000121001E905000000021001E92700000012100000D100000002100000F1000000121
0000472000000021000047300000012100118B600000002100118D60000001210001F780000000210001FF8000000121
000004300000002100000630000001210001EA20000000210001EA30000001210001F240000000210001F2C000000121
000019D000000021000027200000012100004B600000002100004B700000012100000DF0000000310001E9E000000131
00000730730002320001F870000000310001F8F0000001310001F073B90002320001E140000000210001E15000000121
0001EE60000000210001EE7000000121000040B000000021000045B00000012100004090000000210000459000000121
001057D00000002100105A400000012100003A800000002100003C80000001210001E5A0000000210001E5B000000121
001057F00000002100105A60000001210000544000000021000057400000012100010FE0000000210001CBE000000121
0001E120000000210001E1300000012100010B90000000210002D1900000012100000470000000210000067000000121
00004FC00000002100004FD0000001210010C920000000210010CD20000001210002C860000000210002C87000000121
0001E1A0000000210001E1B0000001210010C900000000210010CD000000012100010BF0000000210002D1F000000121
00013CA000000021000AB9A0000001210016E490000000210016E6900000012100000410000000210000061000000121
00001BC00000002100001BD0000001210001EEA0000000210001EEB00000012100005020000000210000503000000121
000024800000002100002490000001210001F950000000310001F9D0000001310001F253B900023200010DF000000021
0001C9F00000012100001C400000003100001C500000013100001C600000023100005460000000210000576000000121
00013EF000000021000ABBF0000001210002CCC0000000210002CCD00000012100010D70000000210001C97000000121
00013A7000000021000AB77000000121000047A000000021000047B00000012100001340000000210000135000000121
000042A000000031000044A0000001310001C860000002310002C190000000210002C490000001210000204000000021
0000205000000121000025C000000021000A7AB00000012100024CF00000002100024E90000001210001E54000000021
0001E55000000121001057900000002100105A0000000121000054A000000021000057A000000121000A684000000021
000A685000000121000FF27000000021000FF470000001210001F050000000210001F0D000000121000FB13000000021
00005745760001220002132000000021000214E000000121000024C000000021000024D00000012100013B5000000021
000AB850000001210002CC60000000210002CC7000000121001040800000002100104300000001210002C1D000000021
0002C4D000000121000023F0000000210002C7E000000121000A742000000021000A7430000001210001E62000000021
0001E630000001210010CA80000000210010CE800000012100000B5000000031000039C00000013100003BC000000231
0001F810000000310001F890000001310001F013B90002320010C9A0000000210010CDA000000121000012E000000021
000012F0000001210000403000000021000045300000012100001FE00000002100001FF0000001210000345000000041
000039900000014100003B90000002410001FBE0000003410000174000000021000017500000012100003CF000000021
00003D700000012100024CD00000002100024E70000001210001FB30000000310001FBC00000013100003B13B9000232
00118AD00000002100118CD000000121000A744000000021000A74500000012100000D900000002100000F9000000121
0001E0C0000000210001E0D0000001210002C170000000210002C4700000012100003E200000002100003E3000000121
000026C000000021000A7AD000000121000A756000000021000A75700000012100004050000000210000455000000121
000019700000002100002680000001210001F600000000210001F680000001210001EEC0000000210001EED000000121
0002C8C0000000210002C8D00000012100010F20000000210001CB200000012100104B600000002100104DE000000121
001E91D000000021001E93F000000121001040A000000021001043200000012100013C6000000021000AB96000000121
0016E410000000210016E610000001210000424000000021000044400000012100001A90000000210000283000000121
000039A00000003100003BA00000013100003F000000023100010E50000000210001CA500000012100104C8000000021
00104F00000001210010407000000021001042F00000012100013D0000000021000ABA00000001210001FA2000000031
0001FAA0000001310001F623B9000232000046C000000021000046D00000012100010A00000000210002D00000000121
000A782000000021000A78300000012100000C500000003100000E5000000131000212B00000023100004B4000000021
00004B50000001210002CCE0000000210002CCF0000001210001FF20000000210001F7C3B9000122000FF21000000021
000FF41000000121000A73A000000021000A73B00000012100104B900000002100104E10000001210001EA4000000021
0001EA5000000121000038F00000002100003CE00000012100024BE00000002100024D800000012100003B0000000031
0001FE300000013100003C530830123300000490000000210000069000000121001E91F000000021001E941000000121
000024600000002100002470000001210001F730000000210001FC90000001210002C670000000210002C68000000121
000216600000002100021760000001210000100000000021000010100000012100003E400000002100003E5000000121
00013D2000000021000ABA2000000121000A79E000000021000A79F0000001210001FA00000000310001FA8000000131
0001F603B90002320010CB10000000210010CF10000001210016E5D0000000210016E7D00000012100010C1000000021
0002D21000000121000A7F5000000021000A7F600000012100013BE000000021000AB8E000000121000042C000000021
000044C0000001210001FC400000002100003AE3B900012200010DD0000000210001C9D0000001210001F93000000031
0001F9B0000001310001F233B9000232000024E000000021000024F00000012100004BC00000002100004BD000000121
0000176000000021000017700000012100013AB000000021000AB7B00000012100013E1000000021000ABB1000000121
000050400000002100005050000001210001F030000000210001F0B000000121000A686000000021000A687000000121
000054C000000021000057C0000001210016E4F0000000210016E6F00000012100010EB0000000210001CAB000000121
00013D9000000021000ABA9000000121000023A0000000210002C6500000012100004A800000002100004A9000000121
0002CDA0000000210002CDB000000121000041E000000031000043E0000001310001C820000002310001F81000000031
0001F890000001310001F013B9000232000039400000002100003B40000001210002C720000000210002C73000000121
000051000000002100005110000001210001E260000000210001E27000000121000A642000000021000A643000000121
00004C900000002100004CA000000121000014700000002100001480000001210001EB00000000210001EB1000000121
000A75E000000021000A75F000000121001058C00000002100105B3000000121000A73C000000021000A73D000000121
00104D100000002100104F900000012100104D200000002100104FA000000121001059000000002100105B7000000121
000005500000002100000750000001210001F110000000210001F19000000121000037B00000002100003FD000000121
0001F310000000210001F390000001210002CBA0000000210002CBB00000012100005520000000210000582000000121
0001FC30000000310001FCC00000013100003B73B90002320001E6E0000000210001E6F0000001210002167000000021
000217700000012100000DE00000002100000FE0000001210000190000000021000025B000000121000A68C000000021
000A68D0000001210001EF80000000210001EF900000012100104B000000002100104D80000001210010C8E000000021
0010CCE0000001210002C080000000210002C38000000121000016E000000021000016F0000001210016E5C000000021
0016E7C00000012100010CD0000000210002D2D0000001210001F7B0000000210001FEB0000001210001E90000000021
0001E9100000012100013B3000000021000AB8300000012100001D100000002100001D20000001210001F53000000021
0001F5B000000121000012600000002100001270000001210001E480000000210001E490000001210002C98000000021
0002C99000000121001040000000002100104280000001210001FA30000000310001FAB0000001310001F633B9000232
00118B900000002100118D900000012100013B0000000021000AB800000001210001ED80000000210001ED9000000121
0010CA40000000210010CE40000001210001F740000000210001FCA00000012100004220000000410000442000000141
0001C840000002410001C850000003410001EF20000000210001EF300000012100021610000000210002171000000121
0010420000000021001044800000012100001F100000003100001F200000013100001F300000023100001D7000000021
00001D8000000121000A68A000000021000A68B00000012100003F700000002100003F8000000121000FF35000000021
000FF5500000012100003DC00000002100003DD0000001210010C8D0000000210010CCD0000001210000418000000021
00004380000001210000212000000021000021300000012100004F000000002100004F100000012100024C1000000021
00024DB000000121000050A000000021000050B000000121001E917000000021001E939000000121000A644000000021
000A6450000001210000537000000021000056700000012100010AC0000000210002D0C00000012100010AD000000021
0002D0D00000012100024C700000002100024E1000000121000046000000002100004610000001210002C2B000000021
0002C5B0000001210016E560000000210016E7600000012100010C70000000210002D27000000121000FB05000000031
000FB0600000013100000730740002320002CD40000000210002CD500000012100010F10000000210001CB1000000121
0001F7A0000000210001FEA0000001210002C0B0000000210002C3B000000121000FF34000000021000FF54000000121
0010406000000021001042E0000001210000417000000021000043700000012100118BA00000002100118DA000000121
0001F820000000310001F8A0000001310001F023B900023200001F100000003100001F200000013100001F3000000231
000039500000003100003B500000013100003F500000023100001B70000000210000292000000121000018F000000021
000025900000012100005380000000210000568000000121000A66A000000021000A66B0000001210001F32000000021
0001F3A00000012100104210000000210010449000000121000052A000000021000052B0000001210000425000000021
0000445000000121000A784000000021000A785000000121000004F000000021000006F000000121000A664000000021
000A6650000001210001FA20000000310001FAA0000001310001F623B900023200104B100000002100104D9000000121
00000C400000002100000E40000001210010CA50000000210010CE50000001210001EAA0000000210001EAB000000121
00013D8000000021000ABA800000012100000DF0000000310001E9E000000131000007307300023200010E4000000021
0001CA400000012100000D800000002100000F800000012100001CA00000003100001CB00000013100001CC000000231
00010D10000000210001C910000001210010575000000021001059C000000121000216C000000021000217C000000121
0002C020000000210002C320000001210016E5B0000000210016E7B000000121000A7D8000000021000A7D9000000121
000FF22000000021000FF420000001210000404000000021000045400000012100010B30000000210002D13000000121
0001ECA0000000210001ECB00000012100104090000000210010431000000121001E918000000021001E93A000000121
00004F600000002100004F70000001210002CC00000000210002CC10000001210001E200000000210001E21000000121
000FB00000000021000006606600012200104B700000002100104DF000000121000039B00000002100003BB000000121
000019600000002100002690000001210002C250000000210002C55000000121000A790000000021000A791000000121
000A764000000021000A765000000121000020C000000021000020D0000001210001F940000000310001F9C000000131
0001F243B900023200118B000000002100118D0000000121000018200000002100001830000001210002CAC000000021
0002CAD0000001210001F800000000310001F880000001310001F003B900023200024CC00000002100024E6000000121
00002510000000210002C6D000000121000048000000002100004810000001210001F5400000002100003C5313301123
00024B800000002100024D2000000121000054B000000021000057B00000012100013AA000000021000AB7A000000121
000004B000000031000006B000000131000212A0000002310000261000000021000A7AC0000001210000141000000021
0000142000000121000039A00000003100003BA00000013100003F000000023100010F80000000210001CB8000000121
00104C400000002100104EC0000001210000189000000021000025600000012100010DE0000000210001C9E000000121
0001E680000000210001E6900000012100013E6000000021000ABB60000001210001F040000000210001F0C000000121
0010578000000021001059F0000001210000411000000021000043100000012100104BA00000002100104E2000000121
00004660000000210000467000000121000A736000000021000A7370000001210001FE400000002100003C1313000122
00000CB00000002100000EB00000012100010C00000000210002D200000001210010427000000021001044F000000121
00001DE00000002100001DF0000001210002C920000000210002C9300000012100001060000000210000107000000121
0001E4E0000000210001E4F000000121000042B000000021000044B000000121000A748000000021000A749000000121
0002C110000000210002C41000000121000A7B3000000021000AB53000000121000053E000000021000056E000000121
000014E000000021000014F00000012100013D3000000021000ABA300000012100005240000000210000525000000121
000A65E000000021000A65F00000012100001F800000002100001F900000012100004AE00000002100004AF000000121
000A77E000000021000A77F00000012100003A000000003100003C000000013100003D60000002310000265000000021
000A78D0000001210001FB600000002100003B13420001220001FA10000000310001FA90000001310001F613B9000232
0001EDE0000000210001EDF00000012100013B9000000021000AB8900000012100001680000000210000169000000121
0000226000000021000022700000012100013F100000002100013F90000001210001E060000000210001E07000000121
00001200000000210000121000000121001041D0000000210010445000000121001058200000002100105A9000000121
0001F660000000210001F6E0000001210001E96000000021000006833100012200010A60000000210002D06000000121
0010C910000000210010CD10000001210001D8E000000021000A7C60000001210010C840000000210010CC4000000121
000FF2B000000021000FF4B00000012100118A300000002100118C300000012100000480000000210000068000000121
00013CC000000021000AB9C000000121001E91E000000021001E940000000121000038E00000002100003CD000000121
00005310000000210000561000000121";
return FCE(t[index]);
}

struct uniProps
{
private alias _U = immutable(UnicodeProperty);
@property static _U[] tab() pure { return _tab; }
static immutable:
private alias _T = ubyte[];
_T ID_Start = x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
_T Cs = x"A0D8008800";
_T Default_Ignorable_Code_Point = x"80AD0182A10182CC018B4202865302550587FB051A05311090F401A0CC9B1080EF0180A0014F09A0BCA70494CF08AC2E859000";
_T Pd = x"2D01855C0133018E41018405018809068E010102011F0204011C0181BE0113016F01A0CD900225010A0180A9018F9F01";
_T Unified_Ideograph = x"A0340099C040A05200A05A0E02010101020A01010101020203A105D6A0A6E020903A0680DE0296820E9D310F826E91A2934B059060";
_T IDS_Binary_Operator = x"A02FF002020A81F101";
_T Sm = x"2B0110033D0101012D01040125011F0182FE01820F039A3B010D0127030D03808B01270506014405050204010201020107011F02020101011F810C20025A011E19280681D501090136086F01815005021F0A1081008083163F042002810230150206A0CFDC01833801010380A40110033D0101018083010604A0D6D40119011F0119011F0119011F0119011F011901972C02";
_T Pi = x"80AB019F6C010202020119018DC8010101040102010F010301";
_T Other_Grapheme_Extend = x"89BE01180181660118016601180180EA0112026701180177010F018D550184D601902102A0CF6E02939E0118018158010C0180F101838001A0B834010805AC2EAD60";
_T Grapheme_Extend = x"83007081130781072D0101010201020101480B30151001650702060202010423011E1B5B0B3A09090118040109010301052B033C082A180120370101010408040103070A021D013A0101010204080109010A021A010202390104020402020303011E0203010B0239010405010204011402160601013A0101020104080107030A021E013B0101010C01090128010301370101030503010407020B021D013A01020102010301050207020B021C02390201010204080109010A021D0148010401020301010801510102070C08620102090B0749021B0101010101370E01050102050B0124090166040106010202021902040310040D01020206010F0182BF0383B2031D021E021E02400201070801020B09012D03010175022201760304020901060380DB0202013A010107010101010208060A0201301F310430070101050128090C0220040202010338010102030101033A080202809803010D010704010601030280C640820C0180C3218BFE03808D016020822A066902A075D404010A2002500281100103010401190205018097021A120D012608190B2E0330010204020227014306020202020C0108012F01330101030202050201012A02080180EE0102010401A04F300182E1101010816E02825D0180E201809505868603010205042803040180A502823D048183025003460B31047B01360F290102020A033104020207013D03240501083E010C0234090A0402015F030201010206010201809D010308150239020101010116010E07030580C30802030101170151010206010102010102010280EB010204060201021B025508020101026A01010102060101650302040105810309010280F5010A020101040180900402020401200A280602040801090602032E0D01028196070106010152160207010201027A060301010201070101480203010101815B020B0234050501010194FD01060FA0369A053B078418013F045101A04CB80292612E0217821E0101030405080802071E0480940387BB37043208010E011605010F8550070111020701020105640180A0078177013D0481FC0483E0076D07AC16D560808080F0";
_T Other_ID_Start = x"98850288910115018F6C02";
_T Pattern_Syntax = x"210F0A071A0401011A042207010101020101010204010401030117011F019F1818080F0213010A813182D080A082761E846C8200808081810304190F01A0CD0D02810502";
_T Lu = x"411A6517010721010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010201010101010101010101010101010102010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010201010101030201010102010302040102010303020102010101010102010102010102010301010102030107010201020102010101010101010101010101010101020101010101010101010101010101010101020102010103010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010702010202010104010101010101010181210101010301080106010103010101020111010923010203030101010101010101010101010101010101010101010101050102010102023330010101010101010101010101010101010101010101010101010101010101010101090101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010201010101010101010101010102010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010102268B49260101050182D256889A2B0203814001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010901010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101090808060A08080808060B01010101010101080848040C040C040C050B0481060104010303020302010305060101010101010402040A0205013D018A7C303001010302010101010101040101020108030101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101080101010401A0794D0101010101010101010101010101010101010101010101010101010101010101010101010101010101010101011301010101010101010101010101010101010101010101010101010180870101010101010101010101010103010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010A0101010102010101010101010104010101020101010301010101010101010101010101010101010101010501050101010101010101010101010101010401010601050101011C01A0572B1A84C528808824809C0B010F0107010286EA338BED20A0558020A065A01A1A1A1A1A1A01010202010202020401081A1A1A020104020801071B0201040105010103071B1A1A1A1A1A1A1A1A1A1A1A1E1921192119211921192101913522";
_T Case_Ignorable = x"270106010B01230101014701040101010401020281F780C0040204010902010180FB0780CF010501312D01010102010201012C010B060A0B010123010A1510016508010A0104210101011E1B5B0B3A0B0401020118182B032C0107020608293A370101010408040103070A020D010F013A010404080114021A010202390104020402020303011E0203010B0239010405010204011402160601013A0102010104080107020B021E013D010C0132010301370101030503010407020B021D013A0102010601050214021C0239020404080114021D014801070301015A0102070B09620102090901010749021B0101010101370E01050102050B0124090166040106010202021902040310040D01020206010F015E0182600383B2031D021E021E02400201070801020B030105012D05330141022201760304020901060380DB0202013A010107010101010208060A02012701081F3104300101050101050128090C0220040202010338010102030101033A08020240065203010D0107040106010302323F0D01226581BD0101030B030D030D030D020C0508020A01020102053105010A01010D01100D33218B8B0271037D010F0160202F0181D5012404030505015D065D03A06F160184E206810E016204010A01011C0450020E224E01170367030302080103010401190205018097021A120D012608190B2E03300102040202110115024206020202020C01080123010B01330101030202050201011B010E020502010164050903790102010401A04F3001809311823D1003010C102201020180A901070106010B01230101012F012D024301150382010180E201809505840506012A0109824603010205042803040180A502823D048183025003460B31047B01360F290102020A0331040202020104010A013203240501083E010C0234090A0402015F030201010206010201809D0103081502390203012507030580C308020301011701540601010402010280EE04060201021B025508020101026A0101010206010165030204010581030901028100020101040180900402020401200A280602040801090602032E0D01028196070106010152160207010201027A060301010201070101480203010101815B020B0234050501010194ED11060FA0369A053B070904840B013F1140020102A0400B04010701028C9E020104925C2E0217822003091002071E0480940387BB37043208010E011605010F8550070111020701020105053E210180A00E8170013D0481FB0583E0076D088AAF05AC0C01011E60808080F0";
_T Lo = x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
_T Pc = x"5F019FDF021301A0DDDE02180380EF01";
_T Bidi_Control = x"861C0199F1021A053704";
_T Co = x"A0E0009900AE0700A0FFFE02A0FFFE";
_T Variation_Selector = x"980B030101A0E5F010AD02F080F0";
_T Soft_Dotted = x"690280C4018119011E01340114018140016201010199090133010D010301808401809D0181A50180D6028B3201A1A7A5023202320232023202320232023202320232023202320232028886018131021A01";
_T Noncharacter_Code_Point = x"A0FDD020820E02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE02A0FFFE";
_T Logical_Order_Exception = x"8E40057B058AF0030201A090FA0202010102";
_T Cased = x"411A061A2F010A0104010517011F0180C301040480D0012407021E0560012A040202020401010601010301010114015301808B0880A6012609298B172601010501022B010482A0560206888209072B02034080C04081160206022602060208010101010101011F0235010701010303010703040206040D0503010774010D01100D65010401020A0101030506010101010101040106040102040505040111200302833134871680E5060403020C2601010501A079122E121E8084660304013B0502010101051805010383352B010E0650A04F40070C0584091A061A84A55060240424740B010F01070102010B010F0107010281C3010203012A010984C5330D338BAD40A0556040A065805501470102020102020204010C010101070141010402080107011C010401050101030701815402190119011F0119011F0119011F0119011F0119010887340A0114060681053E88924487EC1A061A061A";
_T ID_Compat_Math_Continue = x"80B20205019FB601030B010F81730104011601A1B4A20119011F0119011F0119011F0119011F011901";
_T Math = x"2B0110031F011D0101012D01040125011F0182D80302011A020203820F039A0D011B030B0103010D010E0415050B05410D04010302040512010401020A0101020606010302020201030106030E010144180106010204020420010106020E810C080414025A011E1B010118010B0781BD020C0A04060402020203050E01010102060B080502390101011D0409038150408100820030150206A0CFDC01833706010180A20110031D0101011D0101018083010604A0D4135501470102020102020204010C010101070141010402080107011C01040105010103070181540281240232960004011B010201010201010A0104010101010601040101010101010301020101020101010101010101010102010102040107010401040101010A01110503010501113402";
_T No = x"80B20205010203893506817806780380850780D907110981B10A84351484730A81E0018695010306060A80C610290182D63C4E1682761E856901849404808A0A1E08010F200A270FA0757006A058D12D4104110281551B2404853408190727094B05160680A0020210022E400934021E034B05680818082907814A0681601F809E0A2A047007808614817B1485450281AE09836713835315A04B8607831E17A06429140C146C19954E0983A13B010301044C2D010F83C20D";
_T ID_Continue = x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
_T Me = x"848802963401861E040103A0858B03";
_T Ideographic = x"A030060219090E0383C599C040A05200A05900816E026AA0750A011B97F80884D62A09A02467818CA04D04A0A6E020903A0680DE0296820E9D310F826E89A2821E85E2934B059060";
_T Cf = x"80AD01855206160180C001310181800250018F2B0187FC051A053105010AA0DE8F0180F90390C1010F01A0236210A088600494CF08AC2E86011E60";
_T So = x"80A60102010401010183D101810A027F0280CE010A01130280F701820301817501808206010180840180CF0129018187030F010103020614010101010180850801060102050480C50282F00A82D30182D201809D2281610A0909858302010401020A010102060601010101010104010B020E01010201013A0209050204010201020107011F02020101011F810C08041402070251011E19280645190B514E1680B701090136086F0180F82C2C408100820030150206270220016980E5068165022E1A01590C80D61A1004010D020C0115020602815002040A20240B01101F0B1E08010F200A270F814099C040A05690378361040A020101823D03A052C6107F012D0381E401030104020D0281390939110203010D03012F2D867A02824F018C76018895080411A04B4A040501A051560192B3743C80F60A27023C05031602071E043D1542030180BA5784A9820037043208010E010286C8018B5C0180810182D12C04640C0F020F010F01251780A1381D0D2C040907020E06809A80FB0582D80411030D0377045F060C04010F0C0438080A0628081E02024E81540C0E020D0309072E0107080E040907090780930137";
_T White_Space = x"0905120164011A0195DF01897F0B1D0205012F018FA001";
_T Sentence_Terminal = x"21010C01100185490180930380B4012B0380F6013D010101030281250286E402831601040283050180C602809D022D010501813A0281620480AE0202021D0280BC02410283BC0209038DE4010D01160281AD01A074FC01810E0280E3010301817E0256025F0180980280930380900280F901A0526601030280A9010C01100141018AF40284FD052C0480BD0275047F0380810206011002580201026C0181A102817502050F690280F903820501010180FB02570281A40282B4024A02A04B290280850141020B01835301A04E06019DE801";
_T Lt = x"81C5010201020126019D9508080808080C010F012F01";
_T Other_Lowercase = x"80AA010F0181F50907021E05600134018D81018C2F3F0D01222582B1010D01100D80D31083501A879202A07A1E0280D20180810303028362040901A05C16010203012A0109A0D8753E";
_T Other_Alphabetic = x"834501826A0E0101010201020101480B300D01071001650704040202040123011E10660B650203090103010480A70C030706143602020F010205030A021D033A07020202020A010A021D033A050402020204011E0203010B033A08010301021502160304033A070202020209020A021E013B05030301030A01280539070103010308020B021D033A070103010308020B020F010C043A07010301030A010A021D034B060101010812023D0102071201630102060102100180A313090B01246E0C01010204170404030103020703040D0C01010A048674021E021E021E02421380BC022201760C040980DE05390A02144A020B033105300F3C031E090202390B321381B00E86C13488F620A0787408220281620108011705580232100101390126051C0C2D04300C2501430E0C0108022D0332010103020205012C05050180ED08A04F330188570586860301020504831404818302815303350E2D020B032D0909013D03240C12023903300D0E025C09020106010201809D0A17043A07020202020A010A0280D10D01036A1280ED0702071D02520F01016A0B670E81010D80F7060102020203010101808E07020604011C0A2A050104120B2E0E81970801075316010E7A0603010102010301010301420501020104815C040902010130070303A0500E01013707045D02A04CAC01A02361070111020701020105640188B70187E81A061A061A";
_T XID_Continue = x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
_T Cc = x"00205F21";
_T Regional_Indicator = x"A1F1E61A";
_T Sc = x"24017D0484E9017B0181F20281F202070180F501810701824501899B0188C421A0877701A055C3016C01809A0180DB0203029FF604A0C31E0189B001";
_T IDS_Unary_Operator = x"A02FFE02";
_T Pf = x"80BB019F5D0103011C018DC8010101040102010F010301";
_T Pe = x"290133011F018EBD010101875E0189A90137010F01827A0101011E01843E0101010101010101010101010150012001010101010101010181940101010101010101010101010101010101010101014001010121018425010101010101012C0101010101010181AC01010101010101010103010101010101010202A0CD1E0180D9011D010101010101010101010101010101030111010101010180AA0133011F0102010201";
_T Other_Uppercase = x"A021601083461AA1CC601A061A061A";
_T Cn = x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
_T Hyphen = x"2D017F0184DC01927B018809028E050182E301A0CD670180A9015701";
_T Other_Math = x"5E0183710302011A0202029C20011B030B01200418020E02410D04010302040512010401020A0101030506010302020201030106030405054B05020401020102010101050202040204120202010101070101060281220480A80201011801110181BD020C0905050502020203050E01010102061802390101011D0409028156021F0A8193163F042002A0D463010101040180D3010101A0D4C15501470102020102020204010C010101070141010402080107011C010401050101030701815402190119011F0119011F0119011F0119011F011901080232960004011B010201010201010A0104010101010601040101010101010301020101020101010101010101010102010102040107010401040101010A0111050301050111";
_T Ps = x"280132011F018EBE010101875E01897E010301260137010F01827A0101011E01843E0101010101010101010101010150012001010101010101010181940101010101010101010101010101010101010101014001010121018425010101010101011901120101010101010181AC01010101010101010103010101010101010201A0CD210180D7011D010101010101010101010101010101030111010101010180AA0132011F0103010201";
_T Zs = x"20017F0195DF01897F0B24012F018FA001";
_T Alphabetic = x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
_T XID_Start = x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
_T Grapheme_Link = x"894D017F017F017F017F017F017F017F016D0210017C016F017F0180C90180B40286D9021E01809D01828D0180E30165024602918B01A07A86012501809701808E016C0181350180F601A05E510186060129010E0139017902808B01740180B401620180F4017F0180FC017F0176017401810D0181030280A10153011201510181A501810402510181A902";
_T Dash = x"2D01855C0133018E41018405018809063D0127010F018186018C040102011F0204011C0181BE0113016F01A0CD900225010A0180A9018F9F01";
_T Nl = x"96EE038A6F2302048E7E0119090E03A076AB0AA05A503581CC010801808605A0202A6F";
_T IDS_Trinary_Operator = x"A02FF202";
_T Terminal_Punctuation = x"21010A0101010B020301833E010801820101390148010E01010380B4012B0B010180EB02360F1F0181050284F40280AC0104068137028315088305017C034702809D03030127040202813A0281620480AE0201031D0280BC053E0283BC0209038DE4010D0104010A010102030281AC02A074FB02810D0380E305817E0256025F018097038093037F01100280F901A0526403010480A9010A0101010B02030141010201843A01300184860180C7018136028098064406590483B8052C0480BD0770047F038081020601100258056C0181A1030C02816604030F690280F903820501010180FB0257020402819E032D018285024A02852B05A045F90280850141030A01835202A04E06019DE704";
_T Nd = x"300A86260A80860A80C60A819C0A760A760A760A760A760A760A760A760A760A600A760A460A81160A460A87460A260A812C0A80800A80A60A060A80B60A560A80860A060AA089C60A82A60A260A80C60A160A560A81960AA053160A85860A88860A832C0A80800A3C0A80900A81160A81560A760A81760A660A660A81A60A660A82F60A80F60A460A81A60AA04B060A560A80860AA06C743289400A81A60A81F60A84560A92960A";
_T ID_Compat_Math_Start = x"A022020104011601A1B4A20119011F0119011F0119011F0119011F011901";
_T Prepended_Concatenation_Mark = x"86000680D70131018180025001A107DA010F01";
_T ASCII_Hex_Digit = x"300A07061A06";
_T Mc = x"890301370102030804010232023A03060202020A012B013A0342013A030801010235023A010101060202020A0166020102030301030A0129033D043D023A0101050202010209021C010E023A03050301030A012A024B0306081202814A023F0180AB0204010601020219020A0302071502020602010A038678011E0180810107080102815A0402030402010680E0023A01010109010102080680910130010501010501023D011E01040202013C010203010103023008080280AB011501933602A077F302020158023210808E022F013002040202036E02020218012D0101016D010202050180ED02010201020101A064130101017F012D030402730118023B01300309020D015D030302010180AA031F023A0201040202020309010A0280D103080203016A0306010104020180ED03060402017103080201016D010102060169020401810503090180F7060102040102010101808E030804040154011D023E018197010E016A010701020180D50504020101815E020C01300208020101A0500F376802A06173020606";
_T Radical = x"A02E801A01590C80D6";
_T Quotation_Mark = x"220104018083010F019F5C0819028E070181C9040D03A0CE210480BD0104015A02";
_T Other_Default_Ignorable_Code_Point = x"834F018E0F0286530288AF0190FE01A0CE3B014F09AD000701011E60808080F08E10";
_T Join_Control = x"A0200C02";
_T Grapheme_Base = x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
_T Uppercase = x"411A6517010721010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010201010101010101010101010101010102010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010201010101030201010102010302040102010303020102010101010102010102010102010301010102030107010201020102010101010101010101010101010101020101010101010101010101010101010101020102010103010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010702010202010104010101010101010181210101010301080106010103010101020111010923010203030101010101010101010101010101010101010101010101050102010102023330010101010101010101010101010101010101010101010101010101010101010101090101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010201010101010101010101010102010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010102268B49260101050182D256889A2B0203814001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010901010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101090808060A08080808060B01010101010101080848040C040C040C050B0481060104010303020302010305060101010101010402040A0205011A10130183321A8730303001010302010101010101040101020108030101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101080101010401A0794D0101010101010101010101010101010101010101010101010101010101010101010101010101010101010101011301010101010101010101010101010101010101010101010101010180870101010101010101010101010103010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010A0101010102010101010101010104010101020101010301010101010101010101010101010101010101010501050101010101010101010101010101010401010601050101011C01A0572B1A84C528808824809C0B010F0107010286EA338BED20A0558020A065A01A1A1A1A1A1A01010202010202020401081A1A1A020104020801071B0201040105010103071B1A1A1A1A1A1A1A1A1A1A1A1E1921192119211921192101913522880E1A061A061A";
_T Zl = x"A0202801";
_T Hex_Digit = x"300A07061A06A0FEA90A07061A06";
_T Zp = x"A0202901";
_T Pattern_White_Space = x"0905120164019F88021802";
_T Diacritic = x"5E010101470106010401020281F7809F0108050611020401090280FD0580D1013711011B010101020101808608040280860204020303431B5B0B3A0B22027E08290A101C3D01100103041C014A0110016E0110016E0110012F033C011001070177016E0110016E0110016D0210017C017C0601016B010D054B021B01010101010402420301023E017001010228020405190701010A0282C10383B40280B30B0901815B038139080201300F020B68010F0126093602808A0240065219040106010203323F590C250B81BD0101030B030D030D030D028CF003813D0181FA0669045F01A07572010C0201011C0252020E2266036D0280CA011B12390424015F010C0124018095034104330164050903808002A04F3001830110810E0101012F012D02430182FC01849F06012A0109832A02823B0681D503460B310480C0012901480278023E014C010903680280B202510110011807030580CD0103017B0280FB027E0176027301810D028102020401809C0153011201510181A5018102010102510196AF0FA0369A053B078458115002A03FFE04010701029F012E02178220030306080802071E048E823E80C2078177013D0485E0076D030103";
_T Lowercase = x"611A2F010A01040124180108010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010201010101010101010101010101010102010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010201010101030201010102010302040102010303020102010101010102010102010102010301010102020203060102010201010101010101010101010101010101020101010101010101010101010101010101020201010103010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010702010202010104010101010101010145012407021E0560012B0101010301020412011B2301020303010101010101010101010101010101010101010101010105010102010202333001010101010101010101010101010101010101010101010101010101010101010101090101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101020101010101010101010101010201010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010130298B472B010482F8068882097780C0410101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010109010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010908060A08080808060A080808080E020808080808080501020601030301020804020208080A03010279010D01100D6D01030203011B010401040102020804040121100401834B1A87463001010302010101010101040101020108030101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101020701010104010C2601010501A0791301010101010101010101010101010101010101010101010101010101010101010101010101010101010101010113010101010101010101010101010101010101010101010101010103808501010101010101010101010103010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010A010101010201010101010101010104010101020101030101010101010101010101010101010101010101050105010101010101010101010101010101040101010601010101010101010118030101010383352B010E0650A04F40070C0584291A84CD28808824809B0B010F0107010281C3010203012A01098505338BCD20A0558020A0659A1A1A0701121A1A1A0401010107010B1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1C1C1901061A1901061A1901061A1901061A190106010187340A0114060681053E88B422";
_T Ll = x"611A3A0129180108010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010201010101010101010101010101010102010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010201010101030201010102010302040102010303020102010101010102010102010102010301010102020203060102010201010101010101010101010101010101020101010101010101010101010101010101020201010103010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010702010202010104010101010101010145011B80C10101010301030312011B2301020303010101010101010101010101010101010101010101010105010102010202333001010101010101010101010101010101010101010101010101010101010101010101090101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101020101010101010101010101010201010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010130298B472B020382F806888209772C3F0D0122660101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010109010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010908060A08080808060A080808080E020808080808080501020601030301020804020208080A030102811201030203011B010401040102020804040135018AAB3001010302010101010101040101020106050101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101020701010104010C2601010501A07913010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101130101010101010101010101010101010101010101010101010101018087010101010101010101010101030101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010801010101020101010101010101010401010102010103010101010101010101010101010101010101010105010501010101010101010101010101010104010101060101010101010101011C01030183352B05090750A04F40070C0584291A84CD28808824809B0B010F010701028703338BCD20A0558020A0659A1A1A0701121A1A1A0401010107010B1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1C1C1901061A1901061A1901061A1901061A190106010187340A0114060689F722";
_T Extender = x"80B701821802836E0181B901835A0182F0017F018943013801826301818E0144019389012B0567025D03A06F160185F60183C20116018089016C011502A0547B018810028BDA0182680384CF01A050A902849C020101A0715802880603";
_T Lm = x"82B012040C0E0507010101808501050181DE0180E60180A402810D0204011F010901030180A00180A70184D4017F0182350186DA016B0182630181D00680AE3F0D01222582B1010D01100D8BDF0280F10180BF0181D5012B05050161025D03A06F160184E206810E0172011C027909500117016903030281D50116018089016C01150267040901A05406012D0287E006012A0109A0638504844F0D40020101A0400C0401070102A030313E80C90783AD01845F01";
_T Mn = x"83007081130581092D0101010201020101480B30151001650702060202010423011E1B5B0B3A09090118040109010301052B033C082A180120370101010408040103070A021D013A010404080114021A010202390104020402020303011E0203010B0239010405010204011402160601013A0102010104080107020B021E013D010C0132010301370101030503010407020B021D013A0102010601050214021C0239020404080114021D014801070301015A0102070C08620102090B0749021B0101010101370E01050102050B0124090166040106010202021902040310040D01020206010F0182BF0383B2031D021E021E02400201070801020B09012D03010175022201760304020901060380DB0202013A010107010101010208060A0201300E01103104300101050101050128090C0220040202010338010102030101033A080202809803010D010704010601030280C64082D00D0401030C8BFE03808D016020822A046B02A075D401040A2002500281100103010401190205018097021A120D012608190B2E0330010204020227014306020202020C0108012F01330101030202050201012A02080180EE0102010401A04F300182E110101083CD0180E201809505868603010205042803040180A502823D048183025003460B31047B01360F290102020A033104020207013D03240501083E010C0234090A0402015F030201010206010201809D0103081502390203012507030580C308020301011701540601010402010280EE04060201021B025508020101026A0101010206010165030204010581030901028100020101040180900402020401200A280602040801090602032E0D01028196070106010152160207010201027A060301010201070101480203010101815B020B0234050501010194FD01060FA0369A053B078418013F045101A04CB80292612E0217822003110802071E0480940387BB37043208010E011605010F8550070111020701020105640180A0078177013D0481FC0483E0076D07AC17B580F0";
_T Sk = x"5E01010147010601040103018209040C0E05070101011175010E0285020197340101030B030D030D030D02909C02A07663170902670283D0010E02A0504611837B01010180A201A0F41705";
_T Po = x"210301030201010101020A0203021B01440105010E02070182BE01080181D20629013601020102012C02140201020D0101034A0466012B0E80E903360F1F018105020A01808C01780179018186010C01816F015A010A0280A80F010170014A0504026F0680AB018264098305017C034702809D0301032506010481390280D802808007010680AC071C027D043B053E0240080B018342020808080902040203030B0101010A8C9A0401027001808F020403020102090102010102020A05010A02040101010D020381AC03390180BD01A0740202810D0363010A017306817C0456022803010131022F01610D10027C047E02100280F901A052240702011601140202040303010407030601010280950301030201010101020A0203021B0124010202819A03829C013001819E0182E70180C7011F01811009260170074207590483B8052C0480BD076D0201047E0430024F0404010D01010358066B0181A1050A020101680180FA1769031C0D4C0180820380FC01810803809B015C08530301055D0A8137052A028285024A0D80AF018470058B7C02A03A7B0280850141050801835204814701A04CBC019DE7058ED202";
_T Deprecated = x"814901852901890301010188290288C50682B902ADDCD601";
_T Other_ID_Continue = x"80B70182CF018FE10986680186310290ED01A0CE6901";
_U[] _tab = [
_U("Alphabetic", Alphabetic),
_U("ASCII_Hex_Digit", ASCII_Hex_Digit),
_U("Bidi_Control", Bidi_Control),
_U("Cased", Cased),
_U("Case_Ignorable", Case_Ignorable),
_U("Cc", Cc),
_U("Cf", Cf),
_U("Close_Punctuation", Pe),
_U("Cn", Cn),
_U("Co", Co),
_U("Connector_Punctuation", Pc),
_U("Control", Cc),
_U("Cs", Cs),
_U("Currency_Symbol", Sc),
_U("Dash", Dash),
_U("Dash_Punctuation", Pd),
_U("Decimal_Number", Nd),
_U("Default_Ignorable_Code_Point", Default_Ignorable_Code_Point),
_U("Deprecated", Deprecated),
_U("Diacritic", Diacritic),
_U("Enclosing_Mark", Me),
_U("Extender", Extender),
_U("Final_Punctuation", Pf),
_U("Format", Cf),
_U("Grapheme_Base", Grapheme_Base),
_U("Grapheme_Extend", Grapheme_Extend),
_U("Grapheme_Link", Grapheme_Link),
_U("Hex_Digit", Hex_Digit),
_U("Hyphen", Hyphen),
_U("ID_Compat_Math_Continue", ID_Compat_Math_Continue),
_U("ID_Compat_Math_Start", ID_Compat_Math_Start),
_U("ID_Continue", ID_Continue),
_U("Ideographic", Ideographic),
_U("IDS_Binary_Operator", IDS_Binary_Operator),
_U("ID_Start", ID_Start),
_U("IDS_Trinary_Operator", IDS_Trinary_Operator),
_U("IDS_Unary_Operator", IDS_Unary_Operator),
_U("Initial_Punctuation", Pi),
_U("Join_Control", Join_Control),
_U("Letter_Number", Nl),
_U("Line_Separator", Zl),
_U("Ll", Ll),
_U("Lm", Lm),
_U("Lo", Lo),
_U("Logical_Order_Exception", Logical_Order_Exception),
_U("Lowercase", Lowercase),
_U("Lowercase_Letter", Ll),
_U("Lt", Lt),
_U("Lu", Lu),
_U("Math", Math),
_U("Math_Symbol", Sm),
_U("Mc", Mc),
_U("Me", Me),
_U("Mn", Mn),
_U("Modifier_Letter", Lm),
_U("Modifier_Symbol", Sk),
_U("Nd", Nd),
_U("Nl", Nl),
_U("No", No),
_U("Noncharacter_Code_Point", Noncharacter_Code_Point),
_U("Nonspacing_Mark", Mn),
_U("Open_Punctuation", Ps),
_U("Other_Alphabetic", Other_Alphabetic),
_U("Other_Default_Ignorable_Code_Point", Other_Default_Ignorable_Code_Point),
_U("Other_Grapheme_Extend", Other_Grapheme_Extend),
_U("Other_ID_Continue", Other_ID_Continue),
_U("Other_ID_Start", Other_ID_Start),
_U("Other_Letter", Lo),
_U("Other_Lowercase", Other_Lowercase),
_U("Other_Math", Other_Math),
_U("Other_Number", No),
_U("Other_Punctuation", Po),
_U("Other_Symbol", So),
_U("Other_Uppercase", Other_Uppercase),
_U("Paragraph_Separator", Zp),
_U("Pattern_Syntax", Pattern_Syntax),
_U("Pattern_White_Space", Pattern_White_Space),
_U("Pc", Pc),
_U("Pd", Pd),
_U("Pe", Pe),
_U("Pf", Pf),
_U("Pi", Pi),
_U("Po", Po),
_U("Prepended_Concatenation_Mark", Prepended_Concatenation_Mark),
_U("Private_Use", Co),
_U("Ps", Ps),
_U("Quotation_Mark", Quotation_Mark),
_U("Radical", Radical),
_U("Regional_Indicator", Regional_Indicator),
_U("Sc", Sc),
_U("Sentence_Terminal", Sentence_Terminal),
_U("Sk", Sk),
_U("Sm", Sm),
_U("So", So),
_U("Soft_Dotted", Soft_Dotted),
_U("Space_Separator", Zs),
_U("Spacing_Mark", Mc),
_U("Surrogate", Cs),
_U("Terminal_Punctuation", Terminal_Punctuation),
_U("Titlecase_Letter", Lt),
_U("Unassigned", Cn),
_U("Unified_Ideograph", Unified_Ideograph),
_U("Uppercase", Uppercase),
_U("Uppercase_Letter", Lu),
_U("Variation_Selector", Variation_Selector),
_U("White_Space", White_Space),
_U("XID_Continue", XID_Continue),
_U("XID_Start", XID_Start),
_U("Zl", Zl),
_U("Zp", Zp),
_U("Zs", Zs),
];
}

struct blocks
{
private alias _U = immutable(UnicodeProperty);
@property static _U[] tab() pure { return _tab; }
static immutable:
private alias _T = ubyte[];
_T CJK_Unified_Ideographs_Extension_G = x"A300009350";
_T Tai_Le = x"995030";
_T Coptic = x"A02C808080";
_T Halfwidth_and_Fullwidth_Forms = x"A0FF0080F0";
_T Cyrillic = x"84008100";
_T CJK_Radicals_Supplement = x"A02E808080";
_T Thaana = x"878040";
_T Specials = x"A0FFF010";
_T Hangul_Jamo_Extended_B = x"A0D7B050";
_T Linear_B_Syllabary = x"A100008080";
_T Ogham = x"968020";
_T Arabic_Extended_C = x"A10EC040";
_T Nabataean = x"A1088030";
_T Makasar = x"A11EE020";
_T CJK_Unified_Ideographs_Extension_B = x"A20000A0A6E0";
_T Siddham = x"A115808080";
_T Kana_Extended_A = x"A1B10030";
_T Supplemental_Arrows_C = x"A1F8008100";
_T Kaktovik_Numerals = x"A1D2C020";
_T Combining_Diacritical_Marks = x"830070";
_T Vertical_Forms = x"A0FE1010";
_T Old_Sogdian = x"A10F0030";
_T Old_South_Arabian = x"A10A6020";
_T Cyrillic_Extended_C = x"9C8010";
_T Myanmar = x"900080A0";
_T Bassa_Vah = x"A16AD030";
_T Deseret = x"A1040050";
_T Phaistos_Disc = x"A101D030";
_T Common_Indic_Number_Forms = x"A0A83010";
_T Unified_Canadian_Aboriginal_Syllabics_Extended = x"98B050";
_T Kaithi = x"A1108050";
_T Medefaidrin = x"A16E4060";
_T Arabic_Supplement = x"875030";
_T Yi_Radicals = x"A0A49040";
_T Tai_Tham = x"9A208090";
_T Hiragana = x"A0304060";
_T Hangul_Compatibility_Jamo = x"A0313060";
_T Linear_B_Ideograms = x"A100808080";
_T Syriac_Supplement = x"886010";
_T Ahom = x"A1170050";
_T Shorthand_Format_Controls = x"A1BCA010";
_T Phonetic_Extensions = x"9D008080";
_T Nag_Mundari = x"A1E4D030";
_T Devanagari = x"89008080";
_T Tibetan = x"8F008100";
_T Hanifi_Rohingya = x"A10D0040";
_T Old_Uyghur = x"A10F7040";
_T Enclosed_Ideographic_Supplement = x"A1F2008100";
_T Brahmi = x"A110008080";
_T Number_Forms = x"A0215040";
_T Mayan_Numerals = x"A1D2E020";
_T Ornamental_Dingbats = x"A1F65030";
_T Symbols_for_Legacy_Computing = x"A1FB008100";
_T Unified_Canadian_Aboriginal_Syllabics_Extended_A = x"A11AB010";
_T Greek_and_Coptic = x"83708090";
_T CJK_Unified_Ideographs_Extension_I = x"A2EBF08270";
_T Counting_Rod_Numerals = x"A1D36020";
_T Ethiopic_Extended_B = x"A1E7E020";
_T Supplemental_Mathematical_Operators = x"A02A008100";
_T Osage = x"A104B050";
_T Nushu = x"A1B1708190";
_T Chess_Symbols = x"A1FA0070";
_T Cuneiform = x"A120008400";
_T Takri = x"A1168050";
_T Enclosed_Alphanumeric_Supplement = x"A1F1008100";
_T Sutton_SignWriting = x"A1D80082B0";
_T Miscellaneous_Technical = x"A023008100";
_T Old_Italic = x"A1030030";
_T Toto = x"A1E29030";
_T Hanunoo = x"972020";
_T Lisu = x"A0A4D030";
_T Limbu = x"990050";
_T Saurashtra = x"A0A88060";
_T Elbasan = x"A1050030";
_T Variation_Selectors_Supplement = x"AE010080F0";
_T Psalter_Pahlavi = x"A10B8030";
_T Palmyrene = x"A1086020";
_T Tagbanwa = x"976020";
_T CJK_Unified_Ideographs_Extension_F = x"A2CEB09D40";
_T Combining_Diacritical_Marks_for_Symbols = x"A020D030";
_T Dingbats = x"A0270080C0";
_T Nyiakeng_Puachue_Hmong = x"A1E10050";
_T Combining_Half_Marks = x"A0FE2010";
_T Supplementary_Private_Use_Area_A = x"AF0000A10000";
_T Playing_Cards = x"A1F0A060";
_T Malayalam = x"8D008080";
_T Indic_Siyaq_Numbers = x"A1EC7050";
_T Variation_Selectors = x"A0FE0010";
_T Thai = x"8E008080";
_T Supplemental_Arrows_B = x"A029008080";
_T Zanabazar_Square = x"A11A0050";
_T Ideographic_Symbols_and_Punctuation = x"A16FE020";
_T Symbols_and_Pictographs_Extended_A = x"A1FA708090";
_T High_Private_Use_Surrogates = x"A0DB808080";
_T Modifier_Tone_Letters = x"A0A70020";
_T Samaritan = x"880040";
_T Egyptian_Hieroglyph_Format_Controls = x"A1343030";
_T Latin_Extended_Additional = x"9E008100";
_T Coptic_Epact_Numbers = x"A102E020";
_T Small_Form_Variants = x"A0FE5020";
_T Balinese = x"9B008080";
_T Mandaic = x"884020";
_T Block_Elements = x"A0258020";
_T Supplemental_Symbols_and_Pictographs = x"A1F9008100";
_T Miscellaneous_Mathematical_Symbols_A = x"A027C030";
_T Tifinagh = x"A02D3050";
_T Arabic_Extended_A = x"88A060";
_T Cyrillic_Extended_A = x"A02DE020";
_T Syriac = x"870050";
_T Latin_Extended_E = x"A0AB3040";
_T Caucasian_Albanian = x"A1053040";
_T Soyombo = x"A11A5060";
_T Syloti_Nagri = x"A0A80030";
_T NKo = x"87C040";
_T Tai_Viet = x"A0AA8060";
_T Low_Surrogates = x"A0DC008400";
_T Elymaic = x"A10FE020";
_T Hatran = x"A108E020";
_T Sundanese_Supplement = x"9CC010";
_T CJK_Unified_Ideographs_Extension_H = x"A313509060";
_T Chorasmian = x"A10FB030";
_T Kana_Extended_B = x"A1AFF010";
_T Glagolitic = x"A02C0060";
_T Devanagari_Extended_A = x"A11B0060";
_T Hangul_Syllables = x"A0AC00A02BB0";
_T Osmanya = x"A1048030";
_T Enclosed_CJK_Letters_and_Months = x"A032008100";
_T Meetei_Mayek_Extensions = x"A0AAE020";
_T Hangul_Jamo = x"91008100";
_T Mro = x"A16A4030";
_T Currency_Symbols = x"A020A030";
_T Bopomofo_Extended = x"A031A020";
_T Chakma = x"A1110050";
_T Pau_Cin_Hau = x"A11AC040";
_T Latin_Extended_G = x"A1DF008100";
_T CJK_Unified_Ideographs = x"A04E00A05200";
_T Znamenny_Musical_Notation = x"A1CF0080D0";
_T New_Tai_Lue = x"998060";
_T Modi = x"A1160060";
_T Bamum_Supplement = x"A168008240";
_T Cypro_Minoan = x"A12F9070";
_T Basic_Latin = x"008080";
_T Multani = x"A1128030";
_T Alchemical_Symbols = x"A1F7008080";
_T Bopomofo = x"A0310030";
_T Adlam = x"A1E90060";
_T Khitan_Small_Script = x"A18B008200";
_T Old_Turkic = x"A10C0050";
_T Lao = x"8E808080";
_T Cherokee_Supplement = x"A0AB7050";
_T CJK_Compatibility_Ideographs = x"A0F9008200";
_T Latin_Extended_F = x"A1078040";
_T Lisu_Supplement = x"A11FB010";
_T Latin_Extended_A = x"81008080";
_T Spacing_Modifier_Letters = x"82B050";
_T Gothic = x"A1033020";
_T Sinhala_Archaic_Numbers = x"A111E020";
_T Egyptian_Hieroglyphs = x"A130008430";
_T Sinhala = x"8D808080";
_T Rejang = x"A0A93030";
_T Glagolitic_Supplement = x"A1E00030";
_T Lepcha = x"9C0050";
_T Optical_Character_Recognition = x"A0244020";
_T Greek_Extended = x"9F008100";
_T Miscellaneous_Symbols_and_Pictographs = x"A1F3008300";
_T Emoticons = x"A1F60050";
_T Tirhuta = x"A1148060";
_T Marchen = x"A11C7050";
_T Geometric_Shapes_Extended = x"A1F7808080";
_T Arabic_Presentation_Forms_A = x"A0FB5082B0";
_T Cyrillic_Extended_D = x"A1E03060";
_T Phoenician = x"A1090020";
_T Wancho = x"A1E2C040";
_T Meroitic_Hieroglyphs = x"A1098020";
_T Alphabetic_Presentation_Forms = x"A0FB0050";
_T Linear_A = x"A106008180";
_T Mahjong_Tiles = x"A1F00030";
_T Gurmukhi = x"8A008080";
_T Byzantine_Musical_Symbols = x"A1D0008100";
_T Transport_and_Map_Symbols = x"A1F6808080";
_T Unified_Canadian_Aboriginal_Syllabics = x"94008280";
_T Khojki = x"A1120050";
_T Khudawadi = x"A112B050";
_T Newa = x"A114008080";
_T Dogra = x"A1180050";
_T Ancient_Greek_Musical_Notation = x"A1D20050";
_T Cypriot_Syllabary = x"A1080040";
_T Latin_1_Supplement = x"80808080";
_T Hangul_Jamo_Extended_A = x"A0A96020";
_T Imperial_Aramaic = x"A1084020";
_T Tamil_Supplement = x"A11FC040";
_T Oriya = x"8B008080";
_T Tai_Xuan_Jing_Symbols = x"A1D30060";
_T Enclosed_Alphanumerics = x"A0246080A0";
_T Meetei_Mayek = x"A0ABC040";
_T Tagalog = x"970020";
_T Sundanese = x"9B8040";
_T CJK_Compatibility_Ideographs_Supplement = x"A2F8008220";
_T Cuneiform_Numbers_and_Punctuation = x"A124008080";
_T Ethiopic_Extended = x"A02D8060";
_T Domino_Tiles = x"A1F03070";
_T Old_Permic = x"A1035030";
_T Supplementary_Private_Use_Area_B = x"B00000";
_T Shavian = x"A1045030";
_T Anatolian_Hieroglyphs = x"A144008280";
_T Lycian = x"A1028020";
_T Inscriptional_Pahlavi = x"A10B6020";
_T Ancient_Greek_Numbers = x"A1014050";
_T Miao = x"A16F0080A0";
_T CJK_Unified_Ideographs_Extension_E = x"A2B8209690";
_T Pahawh_Hmong = x"A16B008090";
_T Tangut = x"A170009800";
_T Bengali = x"89808080";
_T CJK_Compatibility_Forms = x"A0FE3020";
_T Vithkuqi = x"A1057050";
_T Supplemental_Punctuation = x"A02E008080";
_T Armenian = x"853060";
_T Arabic_Presentation_Forms_B = x"A0FE708090";
_T Tangut_Components = x"A188008300";
_T Sogdian = x"A10F3040";
_T Ottoman_Siyaq_Numbers = x"A1ED0050";
_T Buhid = x"974020";
_T Myanmar_Extended_A = x"A0AA6020";
_T Manichaean = x"A10AC040";
_T Miscellaneous_Mathematical_Symbols_B = x"A029808080";
_T Latin_Extended_D = x"A0A72080E0";
_T Avestan = x"A10B0040";
_T Small_Kana_Extension = x"A1B13040";
_T Ethiopic_Supplement = x"938020";
_T Arabic = x"86008100";
_T Kayah_Li = x"A0A90030";
_T Javanese = x"A0A98060";
_T Lydian = x"A1092020";
_T Aegean_Numbers = x"A1010040";
_T Braille_Patterns = x"A028008100";
_T Buginese = x"9A0020";
_T CJK_Symbols_and_Punctuation = x"A0300040";
_T Batak = x"9BC040";
_T Combining_Diacritical_Marks_Supplement = x"9DC040";
_T High_Surrogates = x"A0D8008380";
_T Vedic_Extensions = x"9CD030";
_T Nandinagari = x"A119A060";
_T Kanbun = x"A0319010";
_T Arrows = x"A0219070";
_T Devanagari_Extended = x"A0A8E020";
_T Inscriptional_Parthian = x"A10B4020";
_T Cham = x"A0AA0060";
_T Box_Drawing = x"A025008080";
_T Ugaritic = x"A1038020";
_T Georgian = x"90A060";
_T Tangut_Supplement = x"A18D008080";
_T Kana_Supplement = x"A1B0008100";
_T Sharada = x"A1118060";
_T Tamil = x"8B808080";
_T Old_Persian = x"A103A040";
_T Geometric_Shapes = x"A025A060";
_T Cherokee = x"93A060";
_T Superscripts_and_Subscripts = x"A0207030";
_T CJK_Strokes = x"A031C030";
_T Letterlike_Symbols = x"A0210050";
_T CJK_Unified_Ideographs_Extension_D = x"A2B74080E0";
_T Yijing_Hexagram_Symbols = x"A04DC040";
_T Georgian_Extended = x"9C9030";
_T Khmer_Symbols = x"99E020";
_T Sora_Sompeng = x"A110D030";
_T Private_Use_Area = x"A0E0009900";
_T Kharoshthi = x"A10A0060";
_T Dives_Akuru = x"A1190060";
_T Ancient_Symbols = x"A1019040";
_T Katakana = x"A030A060";
_T Ideographic_Description_Characters = x"A02FF010";
_T Cyrillic_Extended_B = x"A0A64060";
_T Meroitic_Cursive = x"A109A060";
_T Mende_Kikakui = x"A1E80080E0";
_T Rumi_Numeral_Symbols = x"A10E6020";
_T Combining_Diacritical_Marks_Extended = x"9AB050";
_T Telugu = x"8C008080";
_T Cyrillic_Supplement = x"850030";
_T Ethiopic = x"92008180";
_T Old_North_Arabian = x"A10A8020";
_T IPA_Extensions = x"825060";
_T Vai = x"A0A5008140";
_T Latin_Extended_C = x"A02C6020";
_T Bamum = x"A0A6A060";
_T Control_Pictures = x"A0240040";
_T Mongolian = x"980080B0";
_T Warang_Citi = x"A118A060";
_T Katakana_Phonetic_Extensions = x"A031F010";
_T Mathematical_Operators = x"A022008100";
_T Early_Dynastic_Cuneiform = x"A1248080D0";
_T Ol_Chiki = x"9C5030";
_T Mahajani = x"A1115030";
_T Myanmar_Extended_B = x"A0A9E020";
_T Khmer = x"97808080";
_T Grantha = x"A113008080";
_T Kannada = x"8C808080";
_T Phags_pa = x"A0A84040";
_T Mathematical_Alphanumeric_Symbols = x"A1D4008400";
_T Kawi = x"A11F0060";
_T Musical_Symbols = x"A1D1008100";
_T Yezidi = x"A10E8040";
_T Kangxi_Radicals = x"A02F0080E0";
_T Phonetic_Extensions_Supplement = x"9D8040";
_T CJK_Unified_Ideographs_Extension_A = x"A0340099C0";
_T Mongolian_Supplement = x"A1166020";
_T Bhaiksuki = x"A11C0070";
_T Hebrew = x"859070";
_T Gujarati = x"8A808080";
_T Tangsa = x"A16A7060";
_T General_Punctuation = x"A0200070";
_T CJK_Compatibility = x"A033008100";
_T Yi_Syllables = x"A0A0008490";
_T Carian = x"A102A040";
_T Miscellaneous_Symbols = x"A026008100";
_T CJK_Unified_Ideographs_Extension_C = x"A2A7009040";
_T Miscellaneous_Symbols_and_Arrows = x"A02B008100";
_T Old_Hungarian = x"A10C808080";
_T Latin_Extended_B = x"818080D0";
_T Runic = x"96A060";
_T Tags = x"AE00008080";
_T Supplemental_Arrows_A = x"A027F010";
_T Masaram_Gondi = x"A11D0060";
_T Arabic_Extended_B = x"887030";
_T Ethiopic_Extended_A = x"A0AB0030";
_T Duployan = x"A1BC0080A0";
_T Georgian_Supplement = x"A02D0030";
_T Gunjala_Gondi = x"A11D6050";
_T Arabic_Mathematical_Alphabetic_Symbols = x"A1EE008100";
_U[] _tab = [
_U("Adlam", Adlam),
_U("Aegean Numbers", Aegean_Numbers),
_U("Ahom", Ahom),
_U("Alchemical Symbols", Alchemical_Symbols),
_U("Alphabetic Presentation Forms", Alphabetic_Presentation_Forms),
_U("Anatolian Hieroglyphs", Anatolian_Hieroglyphs),
_U("Ancient Greek Musical Notation", Ancient_Greek_Musical_Notation),
_U("Ancient Greek Numbers", Ancient_Greek_Numbers),
_U("Ancient Symbols", Ancient_Symbols),
_U("Arabic", Arabic),
_U("Arabic Extended-A", Arabic_Extended_A),
_U("Arabic Extended-B", Arabic_Extended_B),
_U("Arabic Extended-C", Arabic_Extended_C),
_U("Arabic Mathematical Alphabetic Symbols", Arabic_Mathematical_Alphabetic_Symbols),
_U("Arabic Presentation Forms-A", Arabic_Presentation_Forms_A),
_U("Arabic Presentation Forms-B", Arabic_Presentation_Forms_B),
_U("Arabic Supplement", Arabic_Supplement),
_U("Armenian", Armenian),
_U("Arrows", Arrows),
_U("Avestan", Avestan),
_U("Balinese", Balinese),
_U("Bamum", Bamum),
_U("Bamum Supplement", Bamum_Supplement),
_U("Basic Latin", Basic_Latin),
_U("Bassa Vah", Bassa_Vah),
_U("Batak", Batak),
_U("Bengali", Bengali),
_U("Bhaiksuki", Bhaiksuki),
_U("Block Elements", Block_Elements),
_U("Bopomofo", Bopomofo),
_U("Bopomofo Extended", Bopomofo_Extended),
_U("Box Drawing", Box_Drawing),
_U("Brahmi", Brahmi),
_U("Braille Patterns", Braille_Patterns),
_U("Buginese", Buginese),
_U("Buhid", Buhid),
_U("Byzantine Musical Symbols", Byzantine_Musical_Symbols),
_U("Carian", Carian),
_U("Caucasian Albanian", Caucasian_Albanian),
_U("Chakma", Chakma),
_U("Cham", Cham),
_U("Cherokee", Cherokee),
_U("Cherokee Supplement", Cherokee_Supplement),
_U("Chess Symbols", Chess_Symbols),
_U("Chorasmian", Chorasmian),
_U("CJK Compatibility", CJK_Compatibility),
_U("CJK Compatibility Forms", CJK_Compatibility_Forms),
_U("CJK Compatibility Ideographs", CJK_Compatibility_Ideographs),
_U("CJK Compatibility Ideographs Supplement", CJK_Compatibility_Ideographs_Supplement),
_U("CJK Radicals Supplement", CJK_Radicals_Supplement),
_U("CJK Strokes", CJK_Strokes),
_U("CJK Symbols and Punctuation", CJK_Symbols_and_Punctuation),
_U("CJK Unified Ideographs", CJK_Unified_Ideographs),
_U("CJK Unified Ideographs Extension A", CJK_Unified_Ideographs_Extension_A),
_U("CJK Unified Ideographs Extension B", CJK_Unified_Ideographs_Extension_B),
_U("CJK Unified Ideographs Extension C", CJK_Unified_Ideographs_Extension_C),
_U("CJK Unified Ideographs Extension D", CJK_Unified_Ideographs_Extension_D),
_U("CJK Unified Ideographs Extension E", CJK_Unified_Ideographs_Extension_E),
_U("CJK Unified Ideographs Extension F", CJK_Unified_Ideographs_Extension_F),
_U("CJK Unified Ideographs Extension G", CJK_Unified_Ideographs_Extension_G),
_U("CJK Unified Ideographs Extension H", CJK_Unified_Ideographs_Extension_H),
_U("CJK Unified Ideographs Extension I", CJK_Unified_Ideographs_Extension_I),
_U("Combining Diacritical Marks", Combining_Diacritical_Marks),
_U("Combining Diacritical Marks Extended", Combining_Diacritical_Marks_Extended),
_U("Combining Diacritical Marks for Symbols", Combining_Diacritical_Marks_for_Symbols),
_U("Combining Diacritical Marks Supplement", Combining_Diacritical_Marks_Supplement),
_U("Combining Half Marks", Combining_Half_Marks),
_U("Common Indic Number Forms", Common_Indic_Number_Forms),
_U("Control Pictures", Control_Pictures),
_U("Coptic", Coptic),
_U("Coptic Epact Numbers", Coptic_Epact_Numbers),
_U("Counting Rod Numerals", Counting_Rod_Numerals),
_U("Cuneiform", Cuneiform),
_U("Cuneiform Numbers and Punctuation", Cuneiform_Numbers_and_Punctuation),
_U("Currency Symbols", Currency_Symbols),
_U("Cypriot Syllabary", Cypriot_Syllabary),
_U("Cypro-Minoan", Cypro_Minoan),
_U("Cyrillic", Cyrillic),
_U("Cyrillic Extended-A", Cyrillic_Extended_A),
_U("Cyrillic Extended-B", Cyrillic_Extended_B),
_U("Cyrillic Extended-C", Cyrillic_Extended_C),
_U("Cyrillic Extended-D", Cyrillic_Extended_D),
_U("Cyrillic Supplement", Cyrillic_Supplement),
_U("Deseret", Deseret),
_U("Devanagari", Devanagari),
_U("Devanagari Extended", Devanagari_Extended),
_U("Devanagari Extended-A", Devanagari_Extended_A),
_U("Dingbats", Dingbats),
_U("Dives Akuru", Dives_Akuru),
_U("Dogra", Dogra),
_U("Domino Tiles", Domino_Tiles),
_U("Duployan", Duployan),
_U("Early Dynastic Cuneiform", Early_Dynastic_Cuneiform),
_U("Egyptian Hieroglyph Format Controls", Egyptian_Hieroglyph_Format_Controls),
_U("Egyptian Hieroglyphs", Egyptian_Hieroglyphs),
_U("Elbasan", Elbasan),
_U("Elymaic", Elymaic),
_U("Emoticons", Emoticons),
_U("Enclosed Alphanumerics", Enclosed_Alphanumerics),
_U("Enclosed Alphanumeric Supplement", Enclosed_Alphanumeric_Supplement),
_U("Enclosed CJK Letters and Months", Enclosed_CJK_Letters_and_Months),
_U("Enclosed Ideographic Supplement", Enclosed_Ideographic_Supplement),
_U("Ethiopic", Ethiopic),
_U("Ethiopic Extended", Ethiopic_Extended),
_U("Ethiopic Extended-A", Ethiopic_Extended_A),
_U("Ethiopic Extended-B", Ethiopic_Extended_B),
_U("Ethiopic Supplement", Ethiopic_Supplement),
_U("General Punctuation", General_Punctuation),
_U("Geometric Shapes", Geometric_Shapes),
_U("Geometric Shapes Extended", Geometric_Shapes_Extended),
_U("Georgian", Georgian),
_U("Georgian Extended", Georgian_Extended),
_U("Georgian Supplement", Georgian_Supplement),
_U("Glagolitic", Glagolitic),
_U("Glagolitic Supplement", Glagolitic_Supplement),
_U("Gothic", Gothic),
_U("Grantha", Grantha),
_U("Greek and Coptic", Greek_and_Coptic),
_U("Greek Extended", Greek_Extended),
_U("Gujarati", Gujarati),
_U("Gunjala Gondi", Gunjala_Gondi),
_U("Gurmukhi", Gurmukhi),
_U("Halfwidth and Fullwidth Forms", Halfwidth_and_Fullwidth_Forms),
_U("Hangul Compatibility Jamo", Hangul_Compatibility_Jamo),
_U("Hangul Jamo", Hangul_Jamo),
_U("Hangul Jamo Extended-A", Hangul_Jamo_Extended_A),
_U("Hangul Jamo Extended-B", Hangul_Jamo_Extended_B),
_U("Hangul Syllables", Hangul_Syllables),
_U("Hanifi Rohingya", Hanifi_Rohingya),
_U("Hanunoo", Hanunoo),
_U("Hatran", Hatran),
_U("Hebrew", Hebrew),
_U("High Private Use Surrogates", High_Private_Use_Surrogates),
_U("High Surrogates", High_Surrogates),
_U("Hiragana", Hiragana),
_U("Ideographic Description Characters", Ideographic_Description_Characters),
_U("Ideographic Symbols and Punctuation", Ideographic_Symbols_and_Punctuation),
_U("Imperial Aramaic", Imperial_Aramaic),
_U("Indic Siyaq Numbers", Indic_Siyaq_Numbers),
_U("Inscriptional Pahlavi", Inscriptional_Pahlavi),
_U("Inscriptional Parthian", Inscriptional_Parthian),
_U("IPA Extensions", IPA_Extensions),
_U("Javanese", Javanese),
_U("Kaithi", Kaithi),
_U("Kaktovik Numerals", Kaktovik_Numerals),
_U("Kana Extended-A", Kana_Extended_A),
_U("Kana Extended-B", Kana_Extended_B),
_U("Kana Supplement", Kana_Supplement),
_U("Kanbun", Kanbun),
_U("Kangxi Radicals", Kangxi_Radicals),
_U("Kannada", Kannada),
_U("Katakana", Katakana),
_U("Katakana Phonetic Extensions", Katakana_Phonetic_Extensions),
_U("Kawi", Kawi),
_U("Kayah Li", Kayah_Li),
_U("Kharoshthi", Kharoshthi),
_U("Khitan Small Script", Khitan_Small_Script),
_U("Khmer", Khmer),
_U("Khmer Symbols", Khmer_Symbols),
_U("Khojki", Khojki),
_U("Khudawadi", Khudawadi),
_U("Lao", Lao),
_U("Latin-1 Supplement", Latin_1_Supplement),
_U("Latin Extended-A", Latin_Extended_A),
_U("Latin Extended Additional", Latin_Extended_Additional),
_U("Latin Extended-B", Latin_Extended_B),
_U("Latin Extended-C", Latin_Extended_C),
_U("Latin Extended-D", Latin_Extended_D),
_U("Latin Extended-E", Latin_Extended_E),
_U("Latin Extended-F", Latin_Extended_F),
_U("Latin Extended-G", Latin_Extended_G),
_U("Lepcha", Lepcha),
_U("Letterlike Symbols", Letterlike_Symbols),
_U("Limbu", Limbu),
_U("Linear A", Linear_A),
_U("Linear B Ideograms", Linear_B_Ideograms),
_U("Linear B Syllabary", Linear_B_Syllabary),
_U("Lisu", Lisu),
_U("Lisu Supplement", Lisu_Supplement),
_U("Low Surrogates", Low_Surrogates),
_U("Lycian", Lycian),
_U("Lydian", Lydian),
_U("Mahajani", Mahajani),
_U("Mahjong Tiles", Mahjong_Tiles),
_U("Makasar", Makasar),
_U("Malayalam", Malayalam),
_U("Mandaic", Mandaic),
_U("Manichaean", Manichaean),
_U("Marchen", Marchen),
_U("Masaram Gondi", Masaram_Gondi),
_U("Mathematical Alphanumeric Symbols", Mathematical_Alphanumeric_Symbols),
_U("Mathematical Operators", Mathematical_Operators),
_U("Mayan Numerals", Mayan_Numerals),
_U("Medefaidrin", Medefaidrin),
_U("Meetei Mayek", Meetei_Mayek),
_U("Meetei Mayek Extensions", Meetei_Mayek_Extensions),
_U("Mende Kikakui", Mende_Kikakui),
_U("Meroitic Cursive", Meroitic_Cursive),
_U("Meroitic Hieroglyphs", Meroitic_Hieroglyphs),
_U("Miao", Miao),
_U("Miscellaneous Mathematical Symbols-A", Miscellaneous_Mathematical_Symbols_A),
_U("Miscellaneous Mathematical Symbols-B", Miscellaneous_Mathematical_Symbols_B),
_U("Miscellaneous Symbols", Miscellaneous_Symbols),
_U("Miscellaneous Symbols and Arrows", Miscellaneous_Symbols_and_Arrows),
_U("Miscellaneous Symbols and Pictographs", Miscellaneous_Symbols_and_Pictographs),
_U("Miscellaneous Technical", Miscellaneous_Technical),
_U("Modi", Modi),
_U("Modifier Tone Letters", Modifier_Tone_Letters),
_U("Mongolian", Mongolian),
_U("Mongolian Supplement", Mongolian_Supplement),
_U("Mro", Mro),
_U("Multani", Multani),
_U("Musical Symbols", Musical_Symbols),
_U("Myanmar", Myanmar),
_U("Myanmar Extended-A", Myanmar_Extended_A),
_U("Myanmar Extended-B", Myanmar_Extended_B),
_U("Nabataean", Nabataean),
_U("Nag Mundari", Nag_Mundari),
_U("Nandinagari", Nandinagari),
_U("Newa", Newa),
_U("New Tai Lue", New_Tai_Lue),
_U("NKo", NKo),
_U("Number Forms", Number_Forms),
_U("Nushu", Nushu),
_U("Nyiakeng Puachue Hmong", Nyiakeng_Puachue_Hmong),
_U("Ogham", Ogham),
_U("Ol Chiki", Ol_Chiki),
_U("Old Hungarian", Old_Hungarian),
_U("Old Italic", Old_Italic),
_U("Old North Arabian", Old_North_Arabian),
_U("Old Permic", Old_Permic),
_U("Old Persian", Old_Persian),
_U("Old Sogdian", Old_Sogdian),
_U("Old South Arabian", Old_South_Arabian),
_U("Old Turkic", Old_Turkic),
_U("Old Uyghur", Old_Uyghur),
_U("Optical Character Recognition", Optical_Character_Recognition),
_U("Oriya", Oriya),
_U("Ornamental Dingbats", Ornamental_Dingbats),
_U("Osage", Osage),
_U("Osmanya", Osmanya),
_U("Ottoman Siyaq Numbers", Ottoman_Siyaq_Numbers),
_U("Pahawh Hmong", Pahawh_Hmong),
_U("Palmyrene", Palmyrene),
_U("Pau Cin Hau", Pau_Cin_Hau),
_U("Phags-pa", Phags_pa),
_U("Phaistos Disc", Phaistos_Disc),
_U("Phoenician", Phoenician),
_U("Phonetic Extensions", Phonetic_Extensions),
_U("Phonetic Extensions Supplement", Phonetic_Extensions_Supplement),
_U("Playing Cards", Playing_Cards),
_U("Private Use Area", Private_Use_Area),
_U("Psalter Pahlavi", Psalter_Pahlavi),
_U("Rejang", Rejang),
_U("Rumi Numeral Symbols", Rumi_Numeral_Symbols),
_U("Runic", Runic),
_U("Samaritan", Samaritan),
_U("Saurashtra", Saurashtra),
_U("Sharada", Sharada),
_U("Shavian", Shavian),
_U("Shorthand Format Controls", Shorthand_Format_Controls),
_U("Siddham", Siddham),
_U("Sinhala", Sinhala),
_U("Sinhala Archaic Numbers", Sinhala_Archaic_Numbers),
_U("Small Form Variants", Small_Form_Variants),
_U("Small Kana Extension", Small_Kana_Extension),
_U("Sogdian", Sogdian),
_U("Sora Sompeng", Sora_Sompeng),
_U("Soyombo", Soyombo),
_U("Spacing Modifier Letters", Spacing_Modifier_Letters),
_U("Specials", Specials),
_U("Sundanese", Sundanese),
_U("Sundanese Supplement", Sundanese_Supplement),
_U("Superscripts and Subscripts", Superscripts_and_Subscripts),
_U("Supplemental Arrows-A", Supplemental_Arrows_A),
_U("Supplemental Arrows-B", Supplemental_Arrows_B),
_U("Supplemental Arrows-C", Supplemental_Arrows_C),
_U("Supplemental Mathematical Operators", Supplemental_Mathematical_Operators),
_U("Supplemental Punctuation", Supplemental_Punctuation),
_U("Supplemental Symbols and Pictographs", Supplemental_Symbols_and_Pictographs),
_U("Supplementary Private Use Area-A", Supplementary_Private_Use_Area_A),
_U("Supplementary Private Use Area-B", Supplementary_Private_Use_Area_B),
_U("Sutton SignWriting", Sutton_SignWriting),
_U("Syloti Nagri", Syloti_Nagri),
_U("Symbols and Pictographs Extended-A", Symbols_and_Pictographs_Extended_A),
_U("Symbols for Legacy Computing", Symbols_for_Legacy_Computing),
_U("Syriac", Syriac),
_U("Syriac Supplement", Syriac_Supplement),
_U("Tagalog", Tagalog),
_U("Tagbanwa", Tagbanwa),
_U("Tags", Tags),
_U("Tai Le", Tai_Le),
_U("Tai Tham", Tai_Tham),
_U("Tai Viet", Tai_Viet),
_U("Tai Xuan Jing Symbols", Tai_Xuan_Jing_Symbols),
_U("Takri", Takri),
_U("Tamil", Tamil),
_U("Tamil Supplement", Tamil_Supplement),
_U("Tangsa", Tangsa),
_U("Tangut", Tangut),
_U("Tangut Components", Tangut_Components),
_U("Tangut Supplement", Tangut_Supplement),
_U("Telugu", Telugu),
_U("Thaana", Thaana),
_U("Thai", Thai),
_U("Tibetan", Tibetan),
_U("Tifinagh", Tifinagh),
_U("Tirhuta", Tirhuta),
_U("Toto", Toto),
_U("Transport and Map Symbols", Transport_and_Map_Symbols),
_U("Ugaritic", Ugaritic),
_U("Unified Canadian Aboriginal Syllabics", Unified_Canadian_Aboriginal_Syllabics),
_U("Unified Canadian Aboriginal Syllabics Extended", Unified_Canadian_Aboriginal_Syllabics_Extended),
_U("Unified Canadian Aboriginal Syllabics Extended-A", Unified_Canadian_Aboriginal_Syllabics_Extended_A),
_U("Vai", Vai),
_U("Variation Selectors", Variation_Selectors),
_U("Variation Selectors Supplement", Variation_Selectors_Supplement),
_U("Vedic Extensions", Vedic_Extensions),
_U("Vertical Forms", Vertical_Forms),
_U("Vithkuqi", Vithkuqi),
_U("Wancho", Wancho),
_U("Warang Citi", Warang_Citi),
_U("Yezidi", Yezidi),
_U("Yijing Hexagram Symbols", Yijing_Hexagram_Symbols),
_U("Yi Radicals", Yi_Radicals),
_U("Yi Syllables", Yi_Syllables),
_U("Zanabazar Square", Zanabazar_Square),
_U("Znamenny Musical Notation", Znamenny_Musical_Notation),
];
}

struct scripts
{
private alias _U = immutable(UnicodeProperty);
@property static _U[] tab() pure { return _tab; }
static immutable:
private alias _T = ubyte[];
_T Old_Hungarian = x"A10C80330D330706";
_T Coptic = x"83E20EA02890740507";
_T Ol_Chiki = x"9C5030";
_T Cyrillic = x"840080850280A997500980A2014C01906720A0784060A0578E02A0E2003E2101";
_T Thaana = x"878032";
_T Inscriptional_Parthian = x"A10B40160208";
_T Nabataean = x"A108801F0809";
_T Ogham = x"96801D";
_T Meroitic_Hieroglyphs = x"A1098020";
_T Makasar = x"A11EE019";
_T Siddham = x"A11580360226";
_T Old_Persian = x"A103A024040E";
_T Imperial_Aramaic = x"A10840160109";
_T Myanmar = x"900080A0A099401F6120";
_T Deseret = x"A1040050";
_T Kaithi = x"A11080430A01";
_T Medefaidrin = x"A16E405B";
_T Kayah_Li = x"A0A9002E0101";
_T Hiragana = x"A03041560603A17F61811F12011D03A040AD01";
_T Ahom = x"A117001B020F0417";
_T Devanagari = x"890051040F021AA09F6020A072000A";
_T Tibetan = x"8F0048012404270124010F01070402";
_T Nko = x"87C03B0203";
_T Brahmi = x"A110004E04240901";
_T Osage = x"A104B0240424";
_T Nushu = x"A16FE101A0418E818C";
_T Cuneiform = x"A12000839A666F01050B80C4";
_T Takri = x"A116803A060A";
_T Toto = x"A1E2901F";
_T Latin = x"411A061A2F010F010517011F0181C127059A1B2606310504050D01464181008171010D01100D808D0206011B0111298AD720A07AA2660340050201010105180E83302B01090104A04F9607841A1A061A882506012A0109A0D7451F0606";
_T Hanunoo = x"972015";
_T Limbu = x"99001F010C040C0401030C";
_T Saurashtra = x"A0A88046080C";
_T Lisu = x"A0A4D030A07AB001";
_T Egyptian_Hieroglyphs = x"A130008456";
_T Elbasan = x"A1050028";
_T Palmyrene = x"A1086020";
_T Tagbanwa = x"97600D01030102";
_T Old_Italic = x"A10300240903";
_T Caucasian_Albanian = x"A10530340B01";
_T Malayalam = x"8D000D01030133010301060410021A";
_T Inherited = x"83007081150281C40B1A0182E004915B1F820103010D010704010601030280C640820C0280C2218F39046B02A0CD6510100E83CF0180E201905A01A0BBC42E0217822003110802071E04AC2F5280F0";
_T Sora_Sompeng = x"A110D019070A";
_T Linear_B = x"A100000C011A01130102010F020E227B";
_T Nyiakeng_Puachue_Hmong = x"A1E1002D030E020A0402";
_T Meroitic_Cursive = x"A109A0180414022E";
_T Thai = x"8E013A051C";
_T Mende_Kikakui = x"A1E80080C50210";
_T Old_Sogdian = x"A10F0028";
_T Old_Turkic = x"A10C0049";
_T Samaritan = x"88002E020F";
_T Old_South_Arabian = x"A10A6020";
_T Hanifi_Rohingya = x"A10D0028080A";
_T Balinese = x"9B004D032F";
_T Mandaic = x"88401C0201";
_T SignWriting = x"A1D800828C0F05010F";
_T Tifinagh = x"A02D303807020E01";
_T Tai_Viet = x"A0AA80431805";
_T Syriac = x"87000E013C020381100B";
_T Soyombo = x"A11A5053";
_T Elymaic = x"A10FE017";
_T Hatran = x"A108E01301020505";
_T Chorasmian = x"A10FB01C";
_T Glagolitic = x"A02C0060A1B3A0070111020701020105";
_T Osmanya = x"A104801E020A";
_T Linear_A = x"A10600813709160A08";
_T Mro = x"A16A401F010A0402";
_T Chakma = x"A11100350112";
_T Modi = x"A11600450B0A";
_T Bassa_Vah = x"A16AD01E0206";
_T Han = x"A02E801A01590C80D62F01010119090E0483C499C040A05200A05900816E026AA07508020C02A0900EA0A6E020903A0680DE0296820E9D310F826E89A2821E85E2934B059060";
_T Multani = x"A112800701010104010F010B";
_T Bopomofo = x"82EA02A02E192B7020";
_T Adlam = x"A1E9004C040A0402";
_T Khitan_Small_Script = x"A16FE4019B1B81D6";
_T Lao = x"8E810201010105011801010117020501010107010A0204";
_T Psalter_Pahlavi = x"A10B801207040C07";
_T Anatolian_Hieroglyphs = x"A144008247";
_T Canadian_Aboriginal = x"94008280823046A101BA10";
_T Common = x"00411A061A2F010F010517011F0181C127050502147401090106010101827D0106010E0103012001809C0182040180810284D90181950481220185EF03470280CB02010184CD010D01070401060103020183050C0257010B030B010F11213F2601030206011B01112903048297190B1583A0810082740220016982005E819215010101190F0804045B0203015A0280931020240B0130401F512F015880A899C040A0590022660380A50A80F40180A001818B010E02A051D20280D00A16230113010480930101201A061A0B0A012D02400701070A05810203042D0309500D332D80E41BA0B9A40492AC743C80F60A27023E03110802071E043D80D5140C140C57091980875501470102020102020204010C010101070141010402080107011C010401050101030701815402812402329471444C3D82C22C04640C0F020F010F01250A80AE381A01020D2C040907020E06809A83D80411030D0377045F060C04010F0C0438080A0628081E02024E81540C0E020D0309072E0107080E040907090780930137250AAC0407011E60";
_T Gothic = x"A103301B";
_T Yi = x"A0A000848D0337";
_T Sinhala = x"8D8103011203180109010102070301040601010108060A0203A103EC14";
_T Rejang = x"A0A930240B01";
_T Lepcha = x"9C0038030F0303";
_T Tai_Tham = x"9A203F011D020B060A060E";
_T Dives_Akuru = x"A1190007020102080102011E0102020C090A";
_T Meetei_Mayek = x"A0AAE01780C92E020A";
_T Tirhuta = x"A1148048080A";
_T Marchen = x"A11C70200216010E";
_T Wancho = x"A1E2C03A0501";
_T Phoenician = x"A109001C0301";
_T Gurmukhi = x"8A0103010604020216010701020102010202010105040202030301070401010711";
_T Khudawadi = x"A112B03B050A";
_T Khojki = x"A1120012012F";
_T Newa = x"A114005C0105";
_T Dogra = x"A118003C";
_T Oriya = x"8B01030108020202160107010201050209020202030703040201050212";
_T Tagalog = x"9700160901";
_T Sundanese = x"9B8040810008";
_T Old_Permic = x"A103502B";
_T Shavian = x"A1045030";
_T Lycian = x"A102801D";
_T Miao = x"A16F004B04390711";
_T Tangut = x"A16FE0011F97F8088300820009";
_T Bengali = x"8980040108020202160107010103040209020202040801040201050219";
_T Inscriptional_Pahlavi = x"A10B60130508";
_T Vithkuqi = x"A105700B010F01070102010B010F01070102";
_T Armenian = x"85312602320203A0F58305";
_T New_Tai_Lue = x"99802C041A060B0302";
_T Sogdian = x"A10F302A";
_T Buhid = x"974014";
_T Manichaean = x"A10AC027040C";
_T Greek = x"83700401030204010104010101010301010114013F0E109926053205040554018140160206022602060208010101010101011F0235010F010E0206011302030109812701A08A3E01A055DA4F1101A0D05F46";
_T Braille = x"A028008100";
_T Avestan = x"A10B00360307";
_T Arabic = x"8600050106010E01030120010A0B1A016C0122503080F01F0102064A011DA0F2507310816B025002360701201070050180878F631F7E03A0DF0004011B010201010201010A0104010101010601040101010101010301020101020101010101010101010102010102040107010401040101010A01110503010501113402";
_T Javanese = x"A0A9804E020A0402";
_T Lydian = x"A109201A0501";
_T Pau_Cin_Hau = x"A11AC039";
_T Cypro_Minoan = x"A12F9063";
_T Buginese = x"9A001C0202";
_T Batak = x"9BC0340804";
_T Nandinagari = x"A119A008022E020B";
_T Cham = x"A0AA0037090E020A0204";
_T Gunjala_Gondi = x"A11D60060102012501020106070A";
_T Cypriot = x"A10800060201012C010203010201";
_T Ugaritic = x"A103801E0101";
_T Georgian = x"90A02601010501022B01048B902B020390402601010501";
_T Sharada = x"A1118060";
_T Tamil = x"8B820201060303010403020101010203020303030C040503030104020106010E15A113C5320D01";
_T Cherokee = x"93A0560206A0977250";
_T Pahawh_Hmong = x"A16B00460A0A010701150513";
_T Syloti_Nagri = x"A0A8002D";
_T Kharoshthi = x"A10A0004010205080103011D0203040A0709";
_T Zanabazar_Square = x"A11A0048";
_T Katakana = x"A030A15A020380F01080D02F0158A0CC0E0A012DA0B05204010701020101811F0332010E04";
_T Telugu = x"8C000D0103011701100209010301040702010302010204020A0709";
_T Ethiopic = x"92004901040207010101040229010402210104020701010104020F0139010402430220031A99E61709070107010701070107010701070107A07D22060206020609070107A13CB10701040102010F";
_T Vai = x"A0A500812C";
_T Bamum = x"A0A6A058A0C1088239";
_T Hangul = x"910081009E2E0281015E711F411FA076E11D8283A02BA40C170431A027A41F0306020602060203";
_T Mongolian = x"980002020101140659072BA0FDB50D";
_T Nag_Mundari = x"A1E4D02A";
_T Old_Uyghur = x"A10F701A";
_T Mahajani = x"A1115027";
_T Khmer = x"97805E020A060A81E620";
_T Grantha = x"A113000401080202021601070102010502090202020302010601050702070305";
_T Kannada = x"8C800D01030117010A0105020901030104070206020104020A0103";
_T Kawi = x"A11F00110129031C";
_T Yezidi = x"A10E802A01030202";
_T Old_North_Arabian = x"A10A8020";
_T Tai_Le = x"99501E0205";
_T Hebrew = x"859137081B0406A0F5281A0105010101020102010A";
_T Gujarati = x"8A8103010901030116010701020105020A0103010302010F04020C0707";
_T Tangsa = x"A16A704F010A";
_T Carian = x"A102A031";
_T Bhaiksuki = x"A11C0009012D010E0A1D";
_T Masaram_Gondi = x"A11D00070102012C030101020109080A";
_T Runic = x"96A04B030B";
_T Duployan = x"A1BC006B050D0309070A0204";
_T Warang_Citi = x"A118A0530C01";
_T Phags_Pa = x"A0A84038";
_U[] _tab = [
_U("Adlam", Adlam),
_U("Ahom", Ahom),
_U("Anatolian_Hieroglyphs", Anatolian_Hieroglyphs),
_U("Arabic", Arabic),
_U("Armenian", Armenian),
_U("Avestan", Avestan),
_U("Balinese", Balinese),
_U("Bamum", Bamum),
_U("Bassa_Vah", Bassa_Vah),
_U("Batak", Batak),
_U("Bengali", Bengali),
_U("Bhaiksuki", Bhaiksuki),
_U("Bopomofo", Bopomofo),
_U("Brahmi", Brahmi),
_U("Braille", Braille),
_U("Buginese", Buginese),
_U("Buhid", Buhid),
_U("Canadian_Aboriginal", Canadian_Aboriginal),
_U("Carian", Carian),
_U("Caucasian_Albanian", Caucasian_Albanian),
_U("Chakma", Chakma),
_U("Cham", Cham),
_U("Cherokee", Cherokee),
_U("Chorasmian", Chorasmian),
_U("Common", Common),
_U("Coptic", Coptic),
_U("Cuneiform", Cuneiform),
_U("Cypriot", Cypriot),
_U("Cypro_Minoan", Cypro_Minoan),
_U("Cyrillic", Cyrillic),
_U("Deseret", Deseret),
_U("Devanagari", Devanagari),
_U("Dives_Akuru", Dives_Akuru),
_U("Dogra", Dogra),
_U("Duployan", Duployan),
_U("Egyptian_Hieroglyphs", Egyptian_Hieroglyphs),
_U("Elbasan", Elbasan),
_U("Elymaic", Elymaic),
_U("Ethiopic", Ethiopic),
_U("Georgian", Georgian),
_U("Glagolitic", Glagolitic),
_U("Gothic", Gothic),
_U("Grantha", Grantha),
_U("Greek", Greek),
_U("Gujarati", Gujarati),
_U("Gunjala_Gondi", Gunjala_Gondi),
_U("Gurmukhi", Gurmukhi),
_U("Han", Han),
_U("Hangul", Hangul),
_U("Hanifi_Rohingya", Hanifi_Rohingya),
_U("Hanunoo", Hanunoo),
_U("Hatran", Hatran),
_U("Hebrew", Hebrew),
_U("Hiragana", Hiragana),
_U("Imperial_Aramaic", Imperial_Aramaic),
_U("Inherited", Inherited),
_U("Inscriptional_Pahlavi", Inscriptional_Pahlavi),
_U("Inscriptional_Parthian", Inscriptional_Parthian),
_U("Javanese", Javanese),
_U("Kaithi", Kaithi),
_U("Kannada", Kannada),
_U("Katakana", Katakana),
_U("Kawi", Kawi),
_U("Kayah_Li", Kayah_Li),
_U("Kharoshthi", Kharoshthi),
_U("Khitan_Small_Script", Khitan_Small_Script),
_U("Khmer", Khmer),
_U("Khojki", Khojki),
_U("Khudawadi", Khudawadi),
_U("Lao", Lao),
_U("Latin", Latin),
_U("Lepcha", Lepcha),
_U("Limbu", Limbu),
_U("Linear_A", Linear_A),
_U("Linear_B", Linear_B),
_U("Lisu", Lisu),
_U("Lycian", Lycian),
_U("Lydian", Lydian),
_U("Mahajani", Mahajani),
_U("Makasar", Makasar),
_U("Malayalam", Malayalam),
_U("Mandaic", Mandaic),
_U("Manichaean", Manichaean),
_U("Marchen", Marchen),
_U("Masaram_Gondi", Masaram_Gondi),
_U("Medefaidrin", Medefaidrin),
_U("Meetei_Mayek", Meetei_Mayek),
_U("Mende_Kikakui", Mende_Kikakui),
_U("Meroitic_Cursive", Meroitic_Cursive),
_U("Meroitic_Hieroglyphs", Meroitic_Hieroglyphs),
_U("Miao", Miao),
_U("Modi", Modi),
_U("Mongolian", Mongolian),
_U("Mro", Mro),
_U("Multani", Multani),
_U("Myanmar", Myanmar),
_U("Nabataean", Nabataean),
_U("Nag_Mundari", Nag_Mundari),
_U("Nandinagari", Nandinagari),
_U("Newa", Newa),
_U("New_Tai_Lue", New_Tai_Lue),
_U("Nko", Nko),
_U("Nushu", Nushu),
_U("Nyiakeng_Puachue_Hmong", Nyiakeng_Puachue_Hmong),
_U("Ogham", Ogham),
_U("Ol_Chiki", Ol_Chiki),
_U("Old_Hungarian", Old_Hungarian),
_U("Old_Italic", Old_Italic),
_U("Old_North_Arabian", Old_North_Arabian),
_U("Old_Permic", Old_Permic),
_U("Old_Persian", Old_Persian),
_U("Old_Sogdian", Old_Sogdian),
_U("Old_South_Arabian", Old_South_Arabian),
_U("Old_Turkic", Old_Turkic),
_U("Old_Uyghur", Old_Uyghur),
_U("Oriya", Oriya),
_U("Osage", Osage),
_U("Osmanya", Osmanya),
_U("Pahawh_Hmong", Pahawh_Hmong),
_U("Palmyrene", Palmyrene),
_U("Pau_Cin_Hau", Pau_Cin_Hau),
_U("Phags_Pa", Phags_Pa),
_U("Phoenician", Phoenician),
_U("Psalter_Pahlavi", Psalter_Pahlavi),
_U("Rejang", Rejang),
_U("Runic", Runic),
_U("Samaritan", Samaritan),
_U("Saurashtra", Saurashtra),
_U("Sharada", Sharada),
_U("Shavian", Shavian),
_U("Siddham", Siddham),
_U("SignWriting", SignWriting),
_U("Sinhala", Sinhala),
_U("Sogdian", Sogdian),
_U("Sora_Sompeng", Sora_Sompeng),
_U("Soyombo", Soyombo),
_U("Sundanese", Sundanese),
_U("Syloti_Nagri", Syloti_Nagri),
_U("Syriac", Syriac),
_U("Tagalog", Tagalog),
_U("Tagbanwa", Tagbanwa),
_U("Tai_Le", Tai_Le),
_U("Tai_Tham", Tai_Tham),
_U("Tai_Viet", Tai_Viet),
_U("Takri", Takri),
_U("Tamil", Tamil),
_U("Tangsa", Tangsa),
_U("Tangut", Tangut),
_U("Telugu", Telugu),
_U("Thaana", Thaana),
_U("Thai", Thai),
_U("Tibetan", Tibetan),
_U("Tifinagh", Tifinagh),
_U("Tirhuta", Tirhuta),
_U("Toto", Toto),
_U("Ugaritic", Ugaritic),
_U("Vai", Vai),
_U("Vithkuqi", Vithkuqi),
_U("Wancho", Wancho),
_U("Warang_Citi", Warang_Citi),
_U("Yezidi", Yezidi),
_U("Yi", Yi),
_U("Zanabazar_Square", Zanabazar_Square),
];
}

struct hangul
{
private alias _U = immutable(UnicodeProperty);
@property static _U[] tab() pure { return _tab; }
static immutable:
private alias _T = ubyte[];
_T V = x"916048A0C60817";
_T T = x"91A858A0C5CB31";
_T LVT = x"A0AC011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B";
_T L = x"910060A098001D";
_T LV = x"A0AC00011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B011B01";
_U[] _tab = [
_U("L", L),
_U("Leading_Jamo", L),
_U("LV", LV),
_U("LV_Syllable", LV),
_U("LVT", LVT),
_U("LVT_Syllable", LVT),
_U("T", T),
_U("Trailing_Jamo", T),
_U("V", V),
_U("Vowel_Jamo", V),
];
}
bool isFormatGen(dchar ch) @safe pure nothrow @nogc
{
    if (ch < 8288)
    {
        if (ch < 2192)
        {
            if (ch < 1564)
            {
                if (ch == 173) return true;
                if (ch < 1536) return false;
                if (ch < 1542) return true;
                return false;
            }
            else if (ch < 1565) return true;
            else
            {
                if (ch == 1757) return true;
                if (ch == 1807) return true;
                return false;
            }
        }
        else if (ch < 2194) return true;
        else
        {
            if (ch < 8203)
            {
                if (ch == 2274) return true;
                if (ch == 6158) return true;
                return false;
            }
            else if (ch < 8208) return true;
            else
            {
                if (ch < 8234) return false;
                if (ch < 8239) return true;
                return false;
            }
        }
    }
    else if (ch < 8293) return true;
    else
    {
        if (ch < 78896)
        {
            if (ch < 65529)
            {
                if (ch < 8294) return false;
                if (ch < 8304) return true;
                if (ch == 65279) return true;
                return false;
            }
            else if (ch < 65532) return true;
            else
            {
                if (ch == 69821) return true;
                if (ch == 69837) return true;
                return false;
            }
        }
        else if (ch < 78912) return true;
        else
        {
            if (ch < 917505)
            {
                if (ch < 113824) return false;
                if (ch < 113828) return true;
                if (ch < 119155) return false;
                if (ch < 119163) return true;
                return false;
            }
            else if (ch < 917506) return true;
            else
            {
                if (ch < 917536) return false;
                if (ch < 917632) return true;
                return false;
            }
        }
    }
}

bool isControlGen(dchar ch) @safe pure nothrow @nogc
{
    if (ch < 32) return true;
    if (ch < 127) return false;
    if (ch < 160) return true;
    return false;
}

bool isSpaceGen(dchar ch) @safe pure nothrow @nogc
{
    if (ch < 160)
    {
        if (ch == 32) return true;
        return false;
    }
    else if (ch < 161) return true;
    else
    {
        if (ch < 8239)
        {
            if (ch == 5760) return true;
            if (ch < 8192) return false;
            if (ch < 8203) return true;
            return false;
        }
        else if (ch < 8240) return true;
        else
        {
            if (ch == 8287) return true;
            if (ch == 12288) return true;
            return false;
        }
    }
}

bool isWhiteGen(dchar ch) @safe pure nothrow @nogc
{
    if (ch < 133)
    {
        if (ch < 9) return false;
        if (ch < 14) return true;
        if (ch == 32) return true;
        return false;
    }
    else if (ch < 134) return true;
    else
    {
        if (ch < 8232)
        {
            if (ch < 5760)
            {
                if (ch == 160) return true;
                return false;
            }
            else if (ch < 5761) return true;
            else
            {
                if (ch < 8192) return false;
                if (ch < 8203) return true;
                return false;
            }
        }
        else if (ch < 8234) return true;
        else
        {
            if (ch < 8287)
            {
                if (ch == 8239) return true;
                return false;
            }
            else if (ch < 8288) return true;
            else
            {
                if (ch == 12288) return true;
                return false;
            }
        }
    }
}

bool isHangL(dchar ch) @safe pure nothrow @nogc
{
    if (ch < 4352) return false;
    if (ch < 4448) return true;
    if (ch < 43360) return false;
    if (ch < 43389) return true;
    return false;
}

bool isHangV(dchar ch) @safe pure nothrow @nogc
{
    if (ch < 4448) return false;
    if (ch < 4520) return true;
    if (ch < 55216) return false;
    if (ch < 55239) return true;
    return false;
}

bool isHangT(dchar ch) @safe pure nothrow @nogc
{
    if (ch < 4520) return false;
    if (ch < 4608) return true;
    if (ch < 55243) return false;
    if (ch < 55292) return true;
    return false;
}


static if (size_t.sizeof == 4)
{
//2080 bytes
enum lowerCaseTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
0000000000000040000000E0",
cast(immutable size_t[]) x"
000001000000014000002500",
cast(immutable size_t[]) x"
020201000402030206020205080702020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000001000000030002000500040006000600060006000600060006000600060006
000600070008000600060006000600060006000600060006000A0009000C000B000E000D000600060006000F00060006
000600060006000600110010000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060013001200060006001400060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006001500060006000600160006
0006000600060006001800170019000600060006000600060006001A0006000600060006000600060006000600060006
0006001B000600060006000600060006000600060006000600060006000600060006000600060006000600060006001C
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006001E001D0020001F00060006000600060006000600210006
000600220006000600060006000600060023000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600000000000000000000000007FFFFFE
000000000420040080000000FF7FFFFFAAAAAAAA55AAAAAAAAAAAB55D4AAAAAA4E243129E6512D2AB5555240AA29AAAA
AAAAAAAA93FAAAAAFFFFAA85FFFFFFFFFFEFFFFF01FFFFFF000000030000001F0000000000000000000000203C8A0000
00010000FFFFF000AAE37FFF192FAAAA00000000FFFF0000FFFFFFFFAAAAAAAAAAAAA802AAAAAAAAAAAAD554AAAAAAAA
AAAAAAAA0000AAAA00000000FFFFFFFF000001FF00000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000FFFF0000F7FFFFFF
000000000000000000000000000000000000000000000000000000003F00000000000000000000000000000000000000
000001FF000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000000000000000
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABFEAAAAAAAAAAAAAAAAAAAAAAAAAAAAA003F00FF00FF00FF00FF003F3FFF00FF
00FF00FF40DF00FF00CF00DC00DC00FF000000000000000000000000800200001FFF0000000000000000000000000000
0008C40032108000000043C0FFFF00000000001000000000000000000000000000000000000000000000000000000000
0000000000000000FFFF0000000003FF00000000FFFF0000FFFFFFFF3FDA1562AAAAAAAAAAAAAAAAAAAAAAAA0008501A
FFFFFFFF000020BF0000000000000000000000000000000000000000000000000000000000000000AAAAAAAA00002AAA
3AAAAAAA00000000000000000000000000000000AAABAAA8AAAAAAAA95FFAAAAAABA50AAAAA082AA02AA050A075C0000
00000000FFFF0000F7FFFFFFFFFF03FFFFFFFFFFFFFFFFFF000000000000000000F8007F000000000000000000000000
00000000000000000000000000000000000000000000000007FFFFFE0000000000000000000000000000000000000000
00000000FFFFFF000000FFFF000000000000000000000000FF0000000FFFFFFF00000000000000000000000000000000
FF8000001BFBFFFB000000000000000000000000000000000000000000000000FFFFFFB907FDFFFF0000000000000000
000000000000000000000000000000000000000000000000FFFFFFFF0007FFFF00000000000000000000000000000000
0000000000000000FFFFFFFF00000000000000000000000000000000FFFFFFFF00000000000000000000000000000000
FC000000000FFFFFFFDFC000000000FF0FFFFFFCEBC000000000FFEFFFFFFC00C000000F00FFFFFFFFFC000000000FFF
FFFFFFC0FC000000000FFFFFFFFFC000000000FF0FFFFFFCFFC000000000FFFFFFFFFC000000003FF7FFFFFCF0000003
0FDFFFFFFFC00000003F7FFFFFFF00000000FDFFFFFFFC0000000BF7000000007FFFFBFF000007E00000000000000000
0000000000000000000000000000000000000000FFFF0000FFFFFFFF00003FFF00000000000000000000000000000000
00000000FFFFFFFC0000000F000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000",
);
//1856 bytes
enum upperCaseTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
0000000000000040000000E0",
cast(immutable size_t[]) x"
000001000000014000001E00",
cast(immutable size_t[]) x"
020201000402030206020205080702020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000001000000030002000500040006000600060006000600060006000600060006
00060007000800060006000600060006000600060006000600060009000B000A000C0006000600060006000D00060006
00060006000600060006000E000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060010000F00060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600110006
000600060006000600130012000600060006000600060006000600140006000600060006000600060006000600060006
000600150006000600060006000600060006000600060006000600060006000600060006000600060006000600060016
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
00060006000600060006000600060006000600060006000600180017001A001900060006000600060006000600060006
00060006000600060006000600060006001B0006000600060006000600060006001C0006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
0006000600060006000600060006000600060006000600060006000600060006000000000000000007FFFFFE00000000
00000000000000007F7FFFFF0000000055555555AA555555555554AA2B555555B1DBCED611AED2D54AAAA49055D25555
555555556C0555550000557A000000000000000000000000000000000000000000000000000000000000000080450000
FFFED74000000FFB551C8000E6905555FFFFFFFF0000FFFF0000000055555555555554015555555555552AAB55555555
55555555FFFE5555007FFFFF000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000FFFFFFFF000020BF00000000
0000000000000000000000000000000000000000FFFFFFFFFFFFFFFF003FFFFF00000000000000000000000000000000
FFFF0000E7FFFFFF00000000000000005555555555555555555555555555555540155555555555555555555555555555
3F00FF00FF00FF00AA003F000000FF00000000000F0000000F000F000F001F003E273884C00F3D50000000200000FFFF
000000080000000000000000000000000000000000000000000000000000000000000000FFC000000000FFFF00000000
FFFFFFFF0000FFFF00000000C025EA9D5555555555555555555555550004280500000000000000005555555500001555
055555550000000000000000000000000000000055545554555555556A00555555452855555F7D55014102F500200000
0000000007FFFFFE000000000000000000000000000000000000000000000000FFFFFFFF000000FF0000000000000000
00000000FFFF0000000FFFFF00000000000000000000000000000000F7FF00000037F7FF000000000000000000000000
00000000000000000000000000000000FFFFFFFF0007FFFF000000000000000000000000000000000000000000000000
00000000FFFFFFFF00000000000000000000000000000000FFFFFFFF0000000000000000000000000000000000000000
03FFFFFFFFF0000000003FFFFFFFFF00D0000003003FDE64FFFF0000000003FF1FDFE7B07B0000000001FC5FFFFFF000
0000003F03FFFFFFFFF0000000003FFFFFFFFF00F0000003003FFFFFFFFF0000000003FFFFFFFF000000000107FFFFFC
F0000000001FFFFFFFC0000000007FFFFFFF0000000001FF0000040000000000FFFFFFFF000000030000000000000000
0000000000000000000000000000000000000000FFFF0000FFFF03FFFFFF03FF000003FF000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000",
);
//11648 bytes
enum simpleCaseTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000004000000280",
cast(immutable size_t[]) x"
0000010000000480000011C0",
cast(immutable size_t[]) x"
020201000402030206020205070202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
0202020202020202020202020202020200010000000300020005000400070006000900080000000A000B0000000D000C
000F000E0011001000130012000000140000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000016001500000000000000000000000000000000000000000018001700000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000019001A00000000001B001D001C001F001E002100200023002200000000000000000025002400000026
000000000000000000000000000000000000000000280027000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000002A0029002C002B0000002D00000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000002E00000000002F003100300033003200000000000000000000000000000000
000000000000000000340000000000350000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000036000000000000000000000000000000000000000000000000000000000038003700000000
0000000000000000000000000000000000000000000000000000000000000000003A0039003C003B003D00000000003E
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000040003F00000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000042004100000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000430000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000450044000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
091DFFFF08E50080031905FB090B00BD09D00B9A0B2B006B082606C70AD200E00467083805D105940A33078501930180
04D9051EFFFF02F9FFFFFFFFFFFFFFFF091EFFFF08E60081031A05FC090C00BE09D10B9B0B2C006C082706C80AD300E1
0468083905D205950A3407860194018104DA051FFFFF02FAFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0965FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05AE077B0856011809BB0AD803DB0257038F05590B4D01280397062B03150047
08DD05F3029F0086058B0711FFFF00CB09820AE204FB02130880070108EF0A4505AF077C0857011909BC0AD903DC0258
0390055A0B4E01290398062C0316004808DE05F402A00087058C0712FFFF00CC09830AE304FC021408810702036A0A46
09DD09DC04CC04CB01CD01CC0B580B57020001FF0732073100CE00CD08AF08AE005C005B06140613005200510620061F
02B202B106630662015A0159080108000B890B8807E707E604BE04BD0A620A6101B401B305930592021C021B096D096C
FFFFFFFF0879087809390938031C031B0094FFFF06FD0095008E06FE081C008F0B30081D07CA0B31012007CB0A230121
FFFF0A2404DC04DB07C507C40B660B6506BD06BC016201610683068202AA02A9060A0609000D000C064106400070006F
083B083A00D200D10728072701DF01DE0B810B8001A201A104C604C50A520A5107CF07CE056605650977097609FB09FA
0625036B0021062605FD002205D305FE068806910B180B17026A0269027505260B39027601AF07FAFFFF01B00AC20769
08B40A47052008B50229027B09900B0900970096FFFF04C908EB00AA032B00D300190018062E062D0105010403CF0553
09A603D0FFFFFFFF01690168075907EE01A7075A052C077F0301052D0AC0030202560255FFFFFFFF0920091F01ADFFFF
FFFFFFFFFFFFFFFF092A092905E0092B05E205E10AE50AE404FD0AE6079804FE0A5D0799076B0A5E0441076C0A800442
02FD0A81055102FE076A05520B540B5306550654016B016A05FA05F902C002BF0647064600F200F106A906A80570056F
0ABAFFFF0ABC0ABB0748074701AE022A0B6E0B6D01EB01EA041A04190971097008410840059F059E0940093F08BE08BD
03AE03AD089808970B120B11041C041B040B040A0A8F0A8E02B402B305310530016D016C07F107F005B705B6084B084A
FFFF00D406CE06CD006400630B830B8200C200C1073E073D00790078088A0889057A057906F506F4FFFFFFFFFFFFFFFF
FFFFFFFF07960A0E04CA0797095D07BE014B028B0692014C02D3041D09D509D409260925032403230954095309F709F6
0B1F04020689021DFFFF052707FB0B3A0AC3FFFF0A48FFFFFFFF0941FFFFFFFF0B2E0521027CFFFF0B76FFFFFFFF00B5
0B0A0991063C00F9FFFF098A00ABFFFF05A0FFFFFFFF08EC032CFFFFFFFFFFFFFFFFFFFFFFFFFFFF0031FFFFFFFFFFFF
FFFF055409A70516FFFFFFFF0350FFFF041E07EF078001A8FFFF02D4FFFFFFFFFFFFFFFFFFFF0AC1FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF0092FFFFFFFF070FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0972FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04A504A401CF01CEFFFFFFFF01F601F5
FFFFFFFF0A37FFFF086E01220382FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0299082806D7FFFF00EBFFFF07F209CA0BA0
04B30386017C04AE0ABD0A17016601D20973059A0B070B32081E09660106039B062F0B73068AFFFF0607032D00AC0345
06FA08FD009A084C06D8029A00EC082904B409CE017D04AF0ABE0A18016701D30974059B0B080B33081F09670107039C
06300B74068C068B0608032E00AD034606FB08FE009B084D0BA107F3097809CB059C04B0FFFFFFFF0347FFFF09790B75
018C018B04A904A80A890A88085F085E058205810989098809DF09DE03BC03BB02F602F508C208C1044A044903630362
06310B34038301D60ABF059D0A84FFFF01D70A8505DB05DA0A38FFFF086F0123051A04F1096E0291098E0AF505AC0090
08F906FF08F700A0031D05EF0391009C0B45071F03C1054003D306E70AB6027D086A0A8C04B7012607AA080A01B90A14
051107EC02DF0A730ACE09A4015B02450664057B0B5B093A062109F0005303C30B46072003C2054103D406E80AB7027E
086B0A8D04B8012707AB080B01BA0A15051207ED02E00A740ACF09A5015C02460665057C0B5C093B062209F1005403C4
051B04F2096F0292098F0AF605AD009108FA070008F800A1031E05F00392009D0AA30AA205CD05CC024402430B4A0B49
01FA01F903E603E509B609B50859085805A305A208E008DF08C608C5039A0399088808870937093603DE03DD04180417
0B220B21FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05A705A6082B082A00E600E506D006CF005E005D061C061B00050004
0730072F00CA00C908AD08AC0568056706B506B4010B010A04F004EF01A0019F0A110A100458045707BB07BA0B700B6F
083108300407040609BF09BE08EE08ED0326032508DA08D909F909F803B803B706C202DB021F06C305D80220023705D9
0A21023804A00A22079204A102DC0793023C023B055C055B0262026106E406E30058005706280627001E001D06790678
011301120677067602480247056A0569011B011A07A907A8049B049A077807770A910A90046A0469023602350B000AFF
038C038B042C042B090E090D08CA08C903380337092409230A010A0003D803D7082508240A950A94079B079A01CB01CA
0A1C0A1B02C402C304EC04EB0171017006A506A405BB05BA064D064C002C002B06390638003800370B6A0B6900B400B3
073807370ACD0ACC088E088D04EE04ED0BA2FFFF0227017E071904B50A9A029308520AC40589047D06D503A701330B63
067C038403030352067E09030135092C06D905B40B270949085A0A060501052407150295022F0A3D05DC0184FFFF0787
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0BA3FFFF0228017F071A04B60A9B029408530AC5058A047E06D603A801340B64
067D038503040353067F09040136092D06DA05B50B28094A085B0A07050205250716029602300A3E05DD0185FFFF0788
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF082209B7050D03EB06F601F001B10B8E
0485024101B704980A9E0A9C046D087C07FE056B0AF706A0039300F704310648090902D908C3068F033B014F09170672
09EA0B4F03D1084E02E101080AA8FFFFFFFFFFFFFFFFFFFF0A55FFFFFFFFFFFF0AE7015303FC0699042D05C409320668
0317002D03C905F109F2002509270B3B055D008C085C06B009AB0AE004290882021504FF0A0A0761024901DA071304D3
0AAE02A108180998058703E106EC03CB011E0B35FFFF039F000EFFFF06950905FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
043305D6036000C3081606C00934006D0374089309FC0B2908BB06030335007E01F70A6D0A5B02CD095506B8055F0137
0B7E07DA07C8057504DF048B075D09EE02E7044F064E03780141085006A209A00041045D089B091900450B9C06340313
023309B10B6709E000820271073905430A0C0ADE077D071701E4052804B908B609FE028903AF06DB05B801FB08060B23
034C05DE035A067A02AD0139092E06860B84004B038D062304390061FFFFFFFF0B85004C038E0624043A0062FFFFFFFF
06E9054205130A160A760A7505CE093CFFFF069DFFFFFFFFFFFFFFFFFFFFFFFF0AE8015403FD069A042E05C509330669
0318002E03CA05F209F3002609280B3C055E008D085D06B109AC0AE1042A0883021605000A0B0762024A01DB071404D4
0AAF02A208190999058803E206ED03CC011F0B36FFFF03A0000FFFFF06960906FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0507FFFFFFFFFFFF030DFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0B92
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
06F306F204C804C7077607750B870B860803080204740473098509840260025F057805770908090708F408F303340333
02CC02CB09140913045C045B03D603D50B040B0307C707C604DE04DD0A1E0A1D01AC01AB05CB05CA0240023F072C072B
00DB00DA083F083E0060005F0616061500130012060E060D02A402A306940693015E015D06B706B605B105B0079F079E
0A640A6304930492020802070B5A0B59020E020D03E003DF09460945087B087A03550354090008FF08D208D1037D037C
088508840962096103DA03D9020402030B3E0B3D01B601B50480047F0A420A4107DD07DC05C305C208110810014E014D
065D065C02AC02AB06330632004A0049061E061D0089008808A308A2053F053E073607350206020504C204C101D101D0
0A5A0A590440043F07D107D0FFFFFFFFFFFFFFFF0886FFFFFFFFFFFFFFFF08F00308030708E808E709C909C803A403A3
025402530ADD0ADC0430042F07E507E40A260A2507B107B004C404C307220721021A0219054605450099009807040703
00A300A205F805F7004E004D06530652012501240AFA0AF902F402F30533053201170116077207710497049607B507B4
0A700A6F03F703F6024E024D0B7D0B7C03BE03BD043C043B08D408D308F608F5035F035E0922092109950994054C054B
08A508A40A780A7704520451019A01990A4A0A4901D501D406E006DF00EE00ED03A906050A020035094F0B41055500F3
03AA06060A03003609500B42055600F40A35076D022B019D06F804CDFFFFFFFF0A36076E022C019E06F904CEFFFFFFFF
05FF039D00160305068008E9017208700600039E00170306068108EA017308710A3902E501820AC804AA01A301E0078E
0A3A02E601830AC904AB01A401E1078F034E0074035605E308BF000AFFFFFFFF034F0075035705E408C0000BFFFFFFFF
01E6FFFF0A5FFFFF04D5FFFF076FFFFF01E7FFFF0A60FFFF04D6FFFF0770FFFF02C509920656043D002903B305E50B8C
02C609930657043E002A03B405E60B8D02ED044709D602D502090A71041F05CF000808E30A570AB006190321FFFFFFFF
09680B1B03A5028502C1066608F1012A09690B1C03A6028602C2066708F2012B03B906E109F4028F07F40B1305B201FD
03BA06E209F5029007F50B1405B301FE0B7A09E400DC09B3073F0590007C02810B7B09E500DD09B407400591007D0282
066A0370097CFFFFFFFFFFFFFFFFFFFF066B037102EE0448FFFF097DFFFF0975FFFFFFFF00F5FFFFFFFFFFFFFFFFFFFF
09D702D6020A0A72FFFF00F6FFFFFFFF035C00BB0387FFFFFFFFFFFFFFFFFFFF035D00BC042005D0FFFFFFFFFFFFFFFF
01C3017609CFFFFF029BFFFFFFFFFFFF01C401770A580AB1FFFF029CFFFFFFFFFFFFFFFF03BFFFFFFFFFFFFFFFFFFFFF
000908E4061A0322FFFF03C0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF06FC
FFFFFFFF09BD0B2DFFFFFFFFFFFFFFFFFFFFFFFFFFFF0951FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0952FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0A7907B204590494017807670A4309DA01EC0534070902F100DE0AEB082E044D
0A7A07B3045A0495017907680A4409DB01ED0535070A02F200DF0AEC082F044EFFFFFFFF0874FFFFFFFF0875FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0596037A00A60B250745081A052A03FE089909CC
0A92020F07B60487018D02210AA0071D029D010205180860097A0B1D0943065E0597037B00A70B260746081B052B03FF
089A09CD0A93021007B70488018E02220AA1071E029E010305190861097B0B1E0944065FFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF065803C7010C0AED06B2082C050B0443
086C0A4F0AB201C80753049C0225020B0B5F070B02770538040C0812098600A8093D070505A4002F095B063A08CF003B
03A10670087601510B0B068403E705D4040407D20AA4011402E304BB053A01BB065903C8010D0AEE06B3082D050C0444
086D0A500AB301C90754049D0226020C0B60070C02780539040D0813098700A9093E070605A50030095C063B08D0003C
03A20671087701520B0C068503E805D5040507D30AA5011502E404BC053B01BC003A0039030E063D0A0F003209D807BF
037E09D90265037F0B200266040305A1FFFF021E0A1A0A19074DFFFFFFFF074EFFFFFFFFFFFFFFFFFFFFFFFF028C095E
02A602A5033A033908CC08CB091209110373037202B902B8099709960470046F07FD07FC0B560B55077A077904B204B1
0A660A6501E901E8056E056D00D600D5072A07290066006508A908A80020001F062A0629005A00590B1A0B1902640263
06BF06BE0190018F07C107C005BD05BC0750074F0A3C0A3B04E204E10280027F0B020B0103AC03AB0472047109580957
08D608D5032803270931093009C109C0057E057D083308320AAD0AAC04220421021802170A130A1201A601A5070E070D
010F010E08210820FFFFFFFFFFFFFFFFFFFFFFFF0358FFFF03290359FFFF032AFFFFFFFF06D206D1FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF082309B8050E03EC06F701F101B20B8F0486024201B804990A9F0A9D046E087D
07FF056C0AF806A1039400F804320649090A02DA08C40690033C01500918067309EB0B5003D2084F02E201090AA9FFFF
FFFFFFFFFFFFFFFF0A56FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01D901D80A200A1F0A990A98079D079C0140013F069F069E02BB02BA04F604F5003E003D0643064205BF05BE064B064A
00770076073C073B0040003F0B6C0B6B04EA04E9078407830AD50AD40892089102C802C70AC70AC6017B017AFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0515051402980297094C094B0A050A04
07AF07AE0A830A820A400A3F018A0189074A074900AF00AE088C088B052F052E05EC05EB00340033FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0007000608AB08AA00C600C5
06F106F000E800E70610060F00680067FFFFFFFF01580157083508340B4C0B4B0504050309C309C20A2C0A2B02B002AF
087F087E0960095F09810980039603950B5E0B5D08550854031203110330032F0212021105A905A804110410098D098C
051D051C08690868023E023D0A280A2700FF00FE066D066C0B100B0F06A706A6015601550661066002EA02E9069C069B
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF07A2FFFF04D707A3050804D80B720B710586058509BA09B90AD10AD008450844
FFFFFFFF004FFFFF0B770050FFFFFFFF0B0E0B0D024C024BFFFF079003B203B107D907D8049F049E03FB03FA09E309E2
0510050F027A0279080F080E011D011C07080707094200B6098B0B2FFFFF00FA035107100B61009305EA05E9009F009E
06E606E5013E013D0550054F028E028D080508040436043505170791075B0B930174075CFFFF0175FFFFFFFFFFFFFFFF
02F802F7FFFFFFFFFFFFFFFF0310030F0AF20AF1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF09ECFFFFFFFF09EDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0B62FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
043405D7036100C4081706C10935006E0375089409FD0B2A08BC06040336007F01F80A6E0A5C02CE095606B905600138
0B7F07DB07C9057604E0048C075E09EF02E80450064F03790142085106A309A10042045E089C091A00460B9D06350314
023409B20B6809E100830272073A05440A0D0ADF077E071801E5052904BA08B709FF028A03B006DC05B901FC08070B24
034D05DF035B067B02AE013A092F0687FFFFFFFFFFFFFFFF080CFFFFFFFF080DFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF09C4FFFF02670AF305AA0143094D0644074B04CF0B78077307C2013B040004F7
054D025902EF03ED0A860AB4019707A4060108C7FFFF0453FFFFFFFFFFFFFFFF09C5FFFF02680AF405AB0144094E0645
074C04D00B79077407C3013C040104F8054E025A02F003EE0A870AB5019807A5060208C8FFFF0454FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF07510A6705C0075F07EA012F09AF0AAA0AFB095902DD099E04E304F9014507D4
047900E908CD063602CF0043036C0723040802D1025D05490B8A0147038006740ACA0A7B047B078C023104830B510781
07520A6805C1076007EB013009B00AAB0AFC095A02DE099F04E404FA014607D5047A00EA08CE063702D00044036D0724
040902D2025E054A0B8B0148038106750ACB0A7C047C078D023204840B520782FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0AD60A4B074107D6022305710B05099A
09C6022D07940B3F04A2024F0763048105C6033106AA034301BF0B3706C90283028709AD065003C5002702EB060B08D7
0A2D075508B00A2FFFFFFFFFFFFFFFFF0AD70A4C074207D7022405720B06099B09C7022E07950B4004A3025007640482
05C7033206AB034401C00B3806CA0284028809AE065103C6002802EC060C08D80A2E075608B10A30FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0463088F026B00A40AE906AC0251011009470B43FFFF00B708FB05E7090100B907CC080801310B470561083602FF03E3
04230583FFFF02A707650A2904E50477063E0A31FFFF03B506D302C90890FFFF00A5046406AD026C01110AEA0B440252
00B8094805E8FFFF00BA08FC080909020B4807CD0837013203E405620584030002A804240A2AFFFF047807660A3204E6
03B6063F02CAFFFFFFFF06D4FFFFFFFF048904D10149019107E20B94057301E203E90697044505C80A8A07E803090A4D
0B90091500C7090F072D0341008A0895086604610100096A071B089D007A036605220201033F04BF0ADA0A69046507A0
083C0963053C046B06EA023901BD0B9609E602FBFFFF066EFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
048A04D2014A019207E30B95057401E303EA0698044605C90A8B07E9030A0A4E0B91091600C80910072E0342008B0896
086704620101096B071C089E007B036705230202034004C00ADB0A6A046607A1083D0964053D046C06EB023A01BE0B97
09E702FCFFFF066FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01DC057F0B9807A601C101A904E70490036E00CF08420733097E00EF045F0848015F0B15068D087202D7034A061108E1
0A6B036408460AB80557045506BA03CD01DD05800B9907A701C201AA04E80491036F00D008430734097F00F004600849
01600B16068E087302D8034B061208E20A6C036508470AB90558045606BB03CEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
09A20084031F07250412025B040E05F5091B0069056306DD034800100A08061704AC0475019507AC06EE030B01EE0AA6
08A604F30AEF06CB09E80A53050507B809A30085032007260413025C040F05F6091C006A056406DE034900110A090618
04AD0476019607AD06EF030C01EF0AA708A704F40AF006CC09E90A54050607B903F8065A08B2005508DB05ED04370023
03760598000000BF0814074305090536000207BC0757019B044B04A60A96033D02730AFD0547016E099C06AE09D20B9E
0014036803F9065B08B3005608DC05EE0438002403770599000100C008150744050A0537000307BD0758019C044C04A7
0A97033E02740AFE0548016F099D06AF09D30B9F00150369FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//11648 bytes
enum fullCaseTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000004000000280",
cast(immutable size_t[]) x"
0000010000000480000011C0",
cast(immutable size_t[]) x"
020201000402030206020205070202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
0202020202020202020202020202020200010000000300020005000400070006000900080000000A000B0000000D000C
000F000E0011001000130012000000140000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000016001500000000000000000000000000000000000000000018001700000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000019001A00000000001B001D001C001F001E002100200023002200000000000000000025002400000026
000000000000000000000000000000000000000000280027000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000002A0029002C002B0000002D00000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000002E00000000002F003100300033003200000000000000000000000000000000
000000000000000000340000000000350000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000036000000000000000000000000000000000000000000000000000000000038003700000000
0000000000000000000000000000000000000000000000000000000000000000003A0039003C003B003D00000000003E
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000040003F00000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000042004100000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000430000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000450044000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
09BEFFFF0984008A0344065809AC00C70A7C0C6A0BF4007408B107360B9100EF04A608C3062905E60AE607FE01AC0199
05210569FFFF0320FFFFFFFFFFFFFFFF09BFFFFF0985008B0345065909AD00C80A7D0C6B0BF5007508B207370B9200F0
04A708C4062A05E70AE707FF01AD019A0522056AFFFF0321FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0A0BFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF060507F208E3012E0A640B9A0412027703C105A60C18013E03C9068A0340004E
097C065002C2009005DA0781FFFF00D50A2A0BA705430230091107710BA20AF9060607F308E4012F0A650B9B04130278
03C205A70C19013F03CA068B0341004F097D065102C3009105DB0782FFFF00D60A2B0BA80544023109120772039B0AFA
0A890A880512051101E601E50C230C22021D021C07A607A500D800D7094609450065006406730672005B005A067F067E
02D802D706C406C301730172088908880C550C5408680867050405030B180B1701CD01CC05E505E4023902380A140A13
FFFF03F6090A090909DD09DC03470346009EFFFF076D009F0098076E08A700990BF908A808440BFA013608450AD60137
031A0AD705240523083F083E0C310C30072A0729017B017A06EB06EA02D002CF06690668000F000E06A0069F00790078
08C608C500DC00DB079C079B01FB01FA0C4D0C4C01BB01BA050C050B0B080B07084B084A05B305B20A1E0A1D0AAB0AAA
0684039C00250685065A0026062B065B06F006F90BE00BDF028A0289029505710C02029601C80881FFFF01C90B8107E0
094E0AFB056B094F0247029B0A380BD000A100A0FFFF050F098A00B4035600DD001D001C068D068C01180117040605A0
0A4E0407FFFFFFFF0182018107D0087401C007D1057707F8032805780B7F032902760275FFFFFFFF09C109C001C6FFFF
FFFFFFFFFFFFFFFF09CE09CD063B09CF063D063C0BAA0BA905450BAB081205460B13081307E20B14047C07E30B39047D
03240B3A059E032507E1059F0C1F0C1E06B606B5018401830657065602E602E506A606A5010101000714071305BD05BC
0B7909280B7B0B7A07BF07BE01C702480C390C3802070206045504540A180A1708CE08CD05F105F009E409E3095B095A
03E103E0092D092C0BD90BD804570456044604450B480B4702DA02D9057C057B0186018508770876060F060E08D808D7
FFFF00DE073D073C006D006C0C4F0C4E00CC00CB07B407B300820081091B091A05C905C807650764FFFFFFFFFFFFFFFF
FFFFFFFF08100ABE051008110A030838016402AD06FA016502FA04580A810A8009C709C6034F034E09FA09F90AA70AA6
0BE8043B06F1023AFFFF057208820C030B82FFFF0AFCFFFFFFFF09E5FFFFFFFF0BF7056C029CFFFF0C41FFFFFFFF00BF
0BD10A39069B0109FFFF0A3200B5FFFF05F7FFFFFFFF098B0357FFFFFFFFFFFFFFFFFFFFFFFFFFFF0035FFFFFFFFFFFF
FFFF05A10A4F055FFFFFFFFF0380FFFF0459087507F901C1FFFF02FBFFFFFFFFFFFFFFFFFFFF0B80FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF009CFFFFFFFF077FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0A19FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04E604E501E801E7FFFFFFFF02120211
FFFFFFFF0AEAFFFF08FF013803B3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02BC08B30746FFFF00FAFFFF08780A750C70
04F90883019504F10B7C0ACA017F01EB0A1A05EC0BCE0BFB08A90A0C011903CD068E0C3E06F2FFFF0666035800B60372
076A099E00A408D9074702BD00FB08B404FA0A79019604F20B7D0ACB018001EC0A1B05ED0BCF0BFC08AA0A0D011A03CE
068F0C3F06F406F30667035900B70373076B099F00A508DA0C7108790A1F0A7605EE04F3FFFFFFFF0374FFFF0A200C40
01A501A404EC04EB0B420B4108F008EF05D105D00A310A300A8B0A8A03F003EF031D031C095F095E0485048403940393
06900BFD03B401EF0B7E05EF0B3DFFFF01F00B3E063106300AEBFFFF09000139056505390A1502B40A360BBA0603009A
099A076F099800AA0348064A03C300A60C0E079003F8058D040A07570B72029D08FB0B4504FD013C0824089401D20AC4
055A087203060B2C0B8D0A4C0174026506C505CA0C2609DE06800A9D005C03FC0C0F079103F9058E040B07580B73029E
08FC0B4604FE013D0825089501D30AC5055B087303070B2D0B8E0A4D0175026606C605CB0C2709DF06810A9E005D03FD
0566053A0A1602B50A370BBB0604009B099B0770099900AB0349064B03C400A70B5C0B5B06250624026402630C130C12
02160215041D041C0A5F0A5E08E608E505F605F5097F097E0963096203CC03CB0919091809DB09DA0415041404530452
0BEB0BEAFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05FE05FD08B608B500F500F4073F073E00670066067B067A00050004
07A407A300D400D30944094305B505B407220721011E011D0538053701B901B80AC10AC004950494083508340C3B0C3A
08BC08BB044204410A680A67098D098C03510350097709760AA90AA803EB03EA07310302023C07320632023D02550633
0AD4025604E10AD5080C04E20303080D025A025905A905A8028202810754075300610060068706860022002106E006DF
0129012806DB06DA0268026705B705B6013101300823082204DC04DB07EF07EE0B4A0B4904A904A8025402530BC50BC4
03BE03BD0467046609AF09AE096709660365036409C509C40AB10AB0040F040E08B008AF0B4E0B4D0815081401E401E3
0ACF0ACE02EB02EA05340533018A01890710070F0613061206AC06AB0030002F06980697003C003B0C350C3400BE00BD
07AE07AD0B8C0B8B09230922053605350C72FFFF02450197078904FB0B5302B608DF0B8305D804BE074403DA014A0C2E
06E403B5032A038206E609A4014C09D00748060C0BF009ED08E70AB60549056F078502B8024D0AF00637019DFFFF0800
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0C73FFFF02460198078A04FC0B5402B708E00B8405D904BF074503DB014B0C2F
06E503B6032B038306E709A5014D09D10749060D0BF109EE08E80AB7054A0570078602B9024E0AF10638019E06620801
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF08AD0A60055604220766020C01CA0C5E
04C6026101D004D90B570B5504AC090D088605B80BBC070B03C50107046C06A709AA0300096006F70368016809B806D4
0A970C1A040808DB0308011B0B61FFFFFFFFFFFFFFFFFFFF0B0BFFFFFFFFFFFF0BAC016C043507020468061C09D606CA
0342003103FE064C0AA1002909CB0C0405AA009608EB071D0A530BA504640913023205470ABA07D8026901F307830519
0B6802C408A30A4005D60418075C040001340BFEFFFF03D10010FFFF06FD09A6FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
046E062E039100CD08A1072F09D8007603A5093B0AAC0BF2095806600362008C02130B240B1102F409FB072505AC014E
0C4A085B084205C4052704CC07D40A9B030E048A06AD03A9015808DD070D0A48004A049A093009BA004C0C6C0693033E
02510A590C320A8C0094029107AF05900ABC0BA007F607870200057304FF09530AAE02AB03E2074A0610021708900C08
037A0639039906EE02D3015009D207190C50005403BF06820474006AFFFFFFFF0C51005503C006830475006BFFFFFFFF
0759058F055C0AC60B2F0B2E062609E0FFFF0708FFFFFFFFFFFFFFFFFFFFFFFF0BAD016D043607030469061D09D706CB
0343003203FF064D0AA2002A09CC0C0505AB009708EC071E0A540BA604650914023305480ABB07D9026A01F40784051A
0B6902C508A40A4105D70419075D040101350BFFFFFF03D20011FFFF06FE09A7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0561FFFFFFFFFFFF0338FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0C62
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
07630762050E050D07ED07EC0C530C52088B088A04B304B20A2D0A2C0280027F05C705C609A909A80995099403610360
02F302F209B509B404990498040D040C0BC90BC808410840052605250AD10AD001C501C4062306220260025F07A0079F
00E500E408CA08C9006900680675067400150014066D066C02C702C606FC06FB01770176072407230608060708190818
0B1A0B1904D404D3022502240C250C24022B022A0417041609EA09E9090C090B0385038409A109A0096F096E03AE03AD
091609150A080A0704110410022102200C070C0601CF01CE04C104C00AF60AF5085E085D061B061A089C089B01670166
06BE06BD02D202D10692069100510050067D067C0093009209380937058C058B07AC07AB022302220508050701EA01E9
0B100B0F047B047A084D084C00DF0C5C05BA03FA0917049EFFFFFFFFFFFF0BA303310330098709860A740A7303D603D5
027402730B9F0B9E046B046A086608650AD90AD8082B082A050A050907930792023702360593059200A300A207740773
00AD00AC065506540057005606B406B3013B013A0BBF0BBE03270326057E057D012D012C07E907E804D804D7082F082E
0B270B260430042F026E026D0C490C4803F203F10477047609710970099709960390038F09C309C20A3D0A3C05990598
093A09390B310B30048D048C01B301B20B000AFF01EE01ED074F074E00FD00FC03DC06640AB2003909F30C0A05A20102
03DD06650AB3003A09F40C0B05A301030AE807E4024901B607680513FFFFFFFF0AE907E5024A01B707690514FFFFFFFF
065C03CF001A032C06E80988018B0901065D03D0001B032D06E90989018C09020AEC030C019B0B8704ED01BC01FC0807
0AED030D019C0B8804EE01BD01FD0808037E007D0386063E095C000CFFFFFFFF037F007E0387063F095D000DFFFFFFFF
020200850B1508ED051B0BEC07E6072B0203FFFF0B16FFFF051CFFFF07E7FFFF02EC0A3A06B70478002D03E606400C5A
02ED0A3B06B80479002E03E706410C5B031404820A8202FC02260B2A045A0627000A09820B0D0B6A0678034CFFFFFFFF
0AC70BE308560B7605F20794099101400AC80BE408570B7705F307950992014103EC07500AA3063409C80BDA06090219
03ED07510AA4063509C90BDB060A021A0C450A900B1F0B9507B5094701F7054F0C460A910B200B9607B6094801F80550
06CC03A10A23048EFFFF0336003F0C4306CD03A203150483FFFF0A24FFFF0A1CFFFFFFFF0AF2084EFFFF0A9F051F0581
0A8302FD02270B2BFFFF0AF3FFFFFFFF038D00C50884091EFFFFFFFF096C064E038E00C6045B0628FFFFFFFFFFFFFFFF
01DC018F0A7A08E902BE0C1604E707F401DD01900B0E0B6BFFFF02BFFFFFFFFFFFFFFFFF06FF0A6BFFFF05D206D8091C
000B09830679034DFFFF0700FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF076C
FFFFFFFF0A660BF6FFFFFFFFFFFFFFFFFFFFFFFFFFFF09F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF09F8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0B32082C049604D5019107DE0AF70A860208057F0779031800EB0BB008B90488
0B33082D049704D6019207DF0AF80A8702090580077A031900EC0BB108BA0489FFFFFFFF0905FFFFFFFF0906FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05E803AB00B00BEE07BC08A505750437092E0A77
0B4B022C083004C801A602410B59078E02C00115056308F10A210BE609E706BF05E903AC00B10BEF07BD08A605760438
092F0A780B4C022D083104C901A702420B5A078F02C10116056408F20A220BE709E806C0FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF06B90402011F0BB2071F08B70554047E
08FD0B050B6C01E107CA04DD024302280C2A077B029705830447089D0A2E00B209E1077505F900330A01069909740045
03D306D20907016A0BD206EC041E062C043D08500B5D012A030A0501058501D406BA040301200BB3072008B80555047F
08FE0B060B6D01E207CB04DE024402290C2B077C029805840448089E0A2F00B309E2077605FA00340A02069A09750046
03D406D30908016B0BD306ED041F062D043E08510B5E012B030B0502058601D5003E003D0339069C0ABF00360A840839
03AF0A85028503B00BE90286043C05F8FFFF023B0ACD0ACC07C4FFFFFFFF07C5FFFFFFFFFFFFFFFFFFFFFFFF02AE0A04
02C902C8036703660969096809B309B203A403A302DF02DE0A3F0A3E04AF04AE088D088C0C210C2007F107F004F504F4
0B1C0B1B0205020405BF05BE00EE00ED079E079D006F006E0940093F0024002306890688006300620BE20BE102840283
072E072D01A901A8083B083A0615061407C707C60AEF0AEE052A052902A0029F0BC70BC603DF03DE04B104B009FE09FD
097309720353035209D509D40A6A0A6905CD05CC08BE08BD0B670B66045D045C023502340AC30AC201BF01BE077E077D
0122012108AC08ABFFFFFFFFFFFFFFFFFFFFFFFF038BFFFF0354038CFFFF0355FFFFFFFF07410740FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF08AE0A61055704230767020D01CB0C5F04C7026201D104DA0B580B5604AD090E
088705B90BBD070C03C60108046D06A809AB0301096106F80369016909B906D50A980C1B040908DC0309011C0B62FFFF
FFFFFFFFFFFFFFFF0B0CFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01F201F10AD30AD20B520B510817081601570156070A070902E102E0053E053D0042004106A206A10617061606AA06A9
0080007F07B207B1004400430C370C360532053107FD07FC0B940B930927092602EF02EE0B860B8501940193FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF055E055D02BB02BA09F009EF0AB50AB4
082908280B3C0B3B0AFE0AFD01A301A207C107C000B900B809210920057A05790647064600380037FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000900080942094100D000CF
0761076000F700F6066F066E00710070FFFFFFFF0171017008C008BF0C150C14054C054B0A700A6F0ADF0ADE02D602D5
0910090F0A060A050A290A2803C803C70C290C2808E208E1033D033C035D035C022F022E060005FF044C044B0A350A34
0568056708FA08F9025C025B0ADB0ADA0114011306CF06CE0BD70BD607120711016F016E06C206C10311031007050704
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF081CFFFF051D081D0562051E0C3D0C3C05D505D40A630A620B900B8F08D208D1
FFFFFFFF0058FFFF0C420059FFFFFFFF0BD50BD4026C026BFFFF082603E503E4085A085904E004DF043404330A8F0A8E
05590558029A0299089A0899013301320778077709E600C00A330BF8FFFF010A038107800C2C009D0645064400A900A8
0756075501550154059D059C02B002AF088F088E047104700560082707D20C63018D07D3FFFF018EFFFFFFFFFFFFFFFF
031F031EFFFFFFFFFFFFFFFF033B033A0BB70BB6FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0A99FFFFFFFF0A9AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0C2DFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
046F062F039200CE08A2073009D9007703A6093C0AAD0BF3095906610363008D02140B250B1202F509FC072605AD014F
0C4B085C084305C5052804CD07D50A9C030F048B06AE03AA015908DE070E0A49004B049B093109BB004D0C6D0694033F
02520A5A0C330A8D0095029207B005910ABD0BA107F7078802010574050009540AAF02AC03E3074B0611021808910C09
037B063A039A06EF02D4015109D3071A08C10BCA05FB01480B630846FFFF0B64FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF09F5FFFF04B4069D035A0016FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0A6DFFFF02870BB80601015A09F106A307C205150C6607EA083C01520439053F
059A0279031604260B3F0B6E01B0081E065E0964FFFF0490FFFFFFFFFFFFFFFF0A6EFFFF02880BB90602015B09F206A4
07C305160C6707EB083D0153043A0540059B027A031704270B400B6F01B1081F065F0965FFFF0491FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF07C80B1D061807D6086D01460A570B700BC009FF03040A46052B0541015C0854
04BA00F8096A069502F60052039D0797044302F8027D05960C56015E03B106D60B890B3404BC0805024F04C40C1C07FA
07C90B1E061907D7086E01470A580B710BC10A0003050A47052C0542015D085504BB00F9096B069602F70053039E0798
044402F9027E05970C57015F03B206D70B8A0B3504BD0806025004C50C1D07FBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0B980B0107B80869025705C00BCC0A42
0A71024B080E0C1004E3026F07DA04C2061E035E0715037001D80C00073802A402A90A5506B10420002B0312066A0978
0AE007CC094A0AE2FFFFFFFFFFFFFFFF0B990B0207B9086A025805C10BCD0A430A72024C080F0C1104E4027007DB04C3
061F035F0716037101D90C01073902A502AA0A5606B20421002C0313066B09790AE107CD094B0AE3FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
04A00924028B00AE0BAE07170271012309EB0C0CFFFF00C1099C064209A200C30848089201600C5805AE08C7032E041A
045E05DCFFFF02CD07DC0ADC052D04B806AF0AE4FFFF03E8074202F00925FFFF00AF04A10718028C01240BAF0C0D0272
00C209EC0643FFFF00C4099D089309A30C59084908C80161041B05AF05DD032F02CE045F0ADDFFFF04B907DD0AE5052E
03E906B002F1FFFFFFFF0743FFFFFFFF04CA0517016201AA08630C6405C201FE04240706048006200B43086B03320B03
0C6009B600D109B007A1036E00D9092A08F704A401110A1107A7093200830397056D021E036C05050B9C0B2804A2081A
08CB0A09058904AA075A025D01D600020A930322FFFF06D0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
04CB0518016301AB08640C6505C301FF04250707048106210B44086C03330B040C6109B700D209B107A2036F00DA092B
08F804A501120A1207A8093300840398056E021F036D05060B9D0B2904A3081B08CC0A0A058A04AB075B025E01D70003
0A940323FFFF06D1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01F505CE0C68082001DA01C2052F04D1039F00E908CF07A90A2600FE049C08D501780BDD06F5090302FE037806700980
0B22039508D30B7405A404920727040401F605CF0C69082101DB01C3053004D203A000EA08D007AA0A2700FF049D08D6
01790BDE06F6090402FF0379067109810B23039608D40B7505A5049307280405FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0A4A008E034A0799044D027B0449065209BC007205B0074C037C00120AB8067604EF04B601AE0836075E0334020A0B5F
093D053B0BB4073A0A950B09054D08320A4B008F034B079A044E027C044A065309BD007305B1074D037D00130AB90677
04F004B701AF0837075F0335020B0B60093E053C0BB5073B0A960B0A054E0833043106BB094C005E097A064804720027
03A705EA000000C9089F07BA055205870006085207CE01B4048604E90B4F036A02930BC2059401870A44071B0A7E0C6E
0018043F043206BC094D005F097B06490473002803A805EB000100CA08A007BB055305880007085307CF01B5048704EA
0B50036B02940BC3059501880A45071C0A7F0C6F00190440FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//5600 bytes
enum alphaTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
0000000000000040000001A0",
cast(immutable size_t[]) x"
00000100000002C000007B00",
cast(immutable size_t[]) x"
03020100060504030A0908070E0D0C0B0303030311100F03141413121414141414141414141414141414141414141414
141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414
141414141414141414141414141414141414141414141414141414141414141414141414141414140000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000001000000030002000500040007000600090008000B000A000D000C000F000E
000100100012001100010013001500140017001600190018001B001A001C0001001E001D001F001F001F0020001F001F
001F001F001F001F00220021001F002300250024001F001F000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001002600010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001002700290028
002B002A002D002C00010001000100010001000100010001000100010001000100010001000100010001000100010001
0001000100010001000100010001000100010001000100010001000100010001000100010001000100010001002E0001
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F0001001F0030002F0031000100330032003500340037003600390038003A0001
003C003B003E003D0040003F00420041004400430046004500480047004A0049004C004B001F004D004F004E00510050
000100010052000100540053001F001F001F001F001F001F001F001F0055001F0001000100010001001F0056001F001F
001F001F001F001F001F001F001F001F001F001F001F001F00010001001F0057001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F
0001000100590058001F001F005B005A0001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001005C00010001000100010001005E005D001F001F001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F005F001F
00600001001F0061001F001F001F001F001F001F001F001F001F0062001F001F001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F0064006300660065001F001F001F001F001F001F0067001F
00690068001F006A001F006B006C001F006E006D001F001F001F001F001F006F0070001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F0001000100010001000100010001007100010001000100010001000100010001
000100010001000100010001007200010001007300010001000100010001000100010001000100010001000100010001
000100010001000100010001000100740001000100010001000100010001000100010001000100010001000100010001
00010001000100010001000100010001000100010075000100010001001F0076001F001F001F001F001F001F001F001F
00010001001F0077001F001F001F001F0001000100010001000100010001000100010001000100010001000100010001
00010001007800010001000100010001000100010001000100010001000100010001000100790001001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F000000000000000007FFFFFE07FFFFFE
0000000004200400FF7FFFFFFF7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0003FFC30000501F000000000000000000000020BCDF0000
FFFFD740FFFFFFFBFFFFFFFFFFBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC03FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFEFFFF027FFFFFFFFFFFFF000001FFBFFF0000FFFF00B6000787FF07FF0000FFFFFFFFFEFFFFFFFFFFC000
FFFFFFFFFFFFFFFF1FEFFFFF9C00E1FEFFFF0000FFFFFFFFFFFFE000FFFFFFFFFFFFFFFF0003FFFFFFFFFC00043007FF
FCFFFFFF00001FFF01FFFFFFFFFF07FF00007EFFFFFFFFFFFFF003FFFFFF03F8FFFFFFFFEFFFFFFFFFE1DFFFFFFE000F
FFF99FEFE3C5FDFFB080599F1003000FFFF987EEC36DFDFF5E021987003F0000FFFBBFEEE3EDFDFF00011BBF1E00000F
FFF99FEEE3EDFDFFB0C0199F0002000FD63DC7ECC3FFC71800811DC700000000FFFDDFFFE3FFFDFF27601DDF0000000F
FFFDDFEFE3EFFDFF60601DDF000E000FFFFDDFFFE7FFFFFF80F05DDFFC00000FFC7FFFEE2FFBFFFFFF5F807F000C0000
FFFFFFFE07FFFFFF0000207F00000000FFFFF7D63BFFFFAFF000205F000000000000000100000000FFFFFEFFFFFE1FFF
FEFFFF0F1FFFFFFF0000000000000000FFFFFFFFF97FFFFFFFFF0000FFFFFFFF3C00FFFFFFFFFFFFFFFF20BFF7FFFFFF
FFFFFFFFFFFFFFFF3D7F3DFFFFFFFFFFFFFF3DFF7F3DFFFFFF7FFF3DFFFFFFFFFF3DFFFFFFFFFFFF07FFFFFF00000000
0000FFFFFFFFFFFFFFFFFFFF3F3FFFFFFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFF9FFF07FFFFFEFFFFFFFFFFFFFFFF01FFC7FF800FFFFF000FFFFF000FFFFF000DDFFF
FFFFFFFFFFCFFFFF108001FF0000000000000000FFFFFFFFFFFFFFFF01FFFFFFFFFFFFFFFFFF07FFFFFFFFFF003FFFFF
7FFFFFFF01FF0FFFFFFF0000001F3FFFFFFFFFFFFFFF0FFF000003FF000000000FFFFFFFFFFFFFFF7FFFFFFF001FFFFE
00000000800000800000700100000000FFFFFFFFFFEFFFFF00001FEF00000000FFFFFFFFFC00F3FFFFFFFFFF0003FFBF
FFFFFFFF007FFFFFFC00E0003FFFFFFFFFFF01FFE7FFFFFF00000000046FDE00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00000000001FFF803F3FFFFFFFFFFFFFAAFF3F3F3FFFFFFFFFFFFFFF5FDFFFFF0FCF1FDC1FDC1FFF
000000000000000000000000800200001FFF00000000000000000000000000003E2FFC84F3FFBD50000043E0FFFFFFFF
000001FF0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000FFC00000FFFFFFFF000003FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF000C781FFFFFFFFFFFFF20BFFFFFFFFF000080FF007FFFFF7F7F7F7F7F7F7F7FFFFFFFFF
0000000000008000000000000000000000000000000000000000000000000000000000E01F3E03FEFFFFFFFEFFFFFFFF
E07FFFFFFFFFFFFEFFFFFFFFF7FFFFFFFFFFFFE0FFFEFFFFFFFFFFFFFFFFFFFF00007FFFFFFFFFFF00000000FFFF0000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00001FFF00000000FFFF00003FFFFFFFFFFF1FFF00000C00FFFFFFFF8FF07FFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFF
FF800000FFFFFFFCFFFFFFFFFFFFFFFFFFFFF9FFFFFFFFFF03EB07FFFFFC0000FFFFFFBF000000FFFFFFFFFF000FFFFF
FFFFFFFFFFFFFFFF0000002FE8FC0000FFFFFC00FFFF07FF0007FFFF1FFFFFFFFFFFFFFFFFF7FFFF000080007C00FFFF
FFFFFFFF007FFFFF00003FFFFC7FFFFFFFFFFFFF7FFFFFFF38000005003CFFFF007E7E7EFFFF7F7FF7FFFFFFFFFF03FF
FFFFFFFFFFFFFFFFFFFFFFFF000007FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000FFFFFF87F0FFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFF3FFFFFFFFFFFFFFFFFFF03FFFFFF00000000E0F8007F5F7FFDFFFFFFFFDBFFFFFFFF
FFFFFFFF0003FFFFFFF80000FFFFFFFFFFFFFFFF3FFFFFFFFFFF0000FFFFFFFFFFFCFFFFFFFFFFFF000000FF0FFF0000
000000000000000000000000FFDF0000FFFFFFFFFFFFFFFFFFFFFFFF1FFFFFFF0000000007FFFFFE07FFFFFEFFFFFFC0
FFFFFFFF7FFFFFFF1CFCFCFC00000000FFFFEFFFB7FFFF7F3FFF3FFF00000000FFFFFFFFFFFFFFFFFFFFFFFF07FFFFFF
0000000000000000FFFFFFFF001FFFFF0000000000000000000000000000000000000000000000000000000000000000
1FFFFFFFFFFFFFFF0001FFFF00000000FFFFFFFFFFFFE000FFFF07FF07FFFFFF3FFFFFFFFFFFFFFF003EFF0F00000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF3FFFFFFFFFFF0000FF0FFFFF0FFFFFFFFFFFFFFFFFFF00FFFFFFFFFFF7FF000F
FFB7F7FF1BFBFFFB0000000000000000FFFFFFFF007FFFFF003FFFFF000000FFFFFFFFBF07FDFFFF0000000000000000
FFFFFD3F91BFFFFF003FFFFF007FFFFF7FFFFFFF00000000000000000037FFFF003FFFFF03FFFFFF0000000000000000
FFFFFFFFC0FFFFFF0000000000000000FEEFF06F003FFFFF000000001FFFFFFF1FFFFFFF00000000FFFFFEFF0000001F
FFFFFFFF003FFFFF003FFFFF0007FFFF0003FFFF000000000000000000000000FFFFFFFFFFFFFFFF000001FF00000000
FFFFFFFF0007FFFFFFFFFFFF0007FFFFFFFFFFFF000000FF000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFF00031BFF00000000000000001FFFFFFFFFFF00800000003FFFFF0000
00000003FFFF00000000001F007FFFFFFFFFFFFFFFFFFFFF0000003F003E0000FFFFFFFF01FFFFFFFFFF0004000001FF
FFFFFFFF0007FFFFFFFF00F00047FFFFFFFFFFFFFFFFFFFF1400C01E00000000FFFBFFFFC09FFFFF0000000300000000
BFFFBD7FFFFF01FFFFFFFFFF000001FFFFF99FEFE3EDFDFFE081199F0000000F00000000000000000000000000000000
FFFFFFFFFFFFFFFF800007BB00000003FFFFFFFFFFFFFFFF000000B30000000000000000000000000000000000000000
FFFFFFFF7F3FFFFF3F00000000000000FFFFFFFF7FFFFFFF0000001100000000FFFFFFFF013FFFFF0000000000000000
E7FFFFFF000007FF0000007F0000000000000000000000000000000000000000FFFFFFFF01FFFFFF0000000000000000
00000000FFFFFFFFFFFFFFFF80000000FF6FF27F99BFFFFF000000070000000000000000FFFFFCFFFCFFFFFF0000001A
FFFFFFFF7FE7FFFFFFFF0000FFFFFFFF20FFFFFFFFFF0000FFFFFFFF01FFFFFFFFFFFDFF7F7FFFFF00000001FFFC0000
FFFCFFFF007FFEFF0000000000000000FFFFFB7FB47FFFFF000000CBFFFFFDBF017B7FFF000000000000000000000000
00000000000000000000000000000000000000000000000000000000007FFFFFFFFDFFFFC7FFFFFF0000000100000000
00000000000100000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03FFFFFF000000000000000000000000
FFFFFFFFFFFFFFFFFFFFFFFF00007FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000000F00000000
0000000000000000000000000000000000000000000000000000000000000000FFFF0000FFFFFFFFFFFFFFFF0001FFFF
FFFFFFFF0000FFFF0000007E0000000000000000000000000000000000000000FFFFFFFFFFFFFFFF0000007F00000000
00000000000000000000000000000000FFFFFFFF01FFFFFF7FFFFFFFFFFF0000FFFFFFFF7FFFFFFFFFFF000000003FFF
FFFFFFFF0000FFFF0000000FE0FFFFF80000FFFF0000000000000000000000000000000000000000FFFFFFFFFFFFFFFF
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFF87FFFFFFFFFFFFFF80FF00000000000000000003000B
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF003FFFFF00000000000001FF00000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000006FEF0000FFFFFFFF0004000700270000FFFF00F0
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0FFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF1FFF07FF43FF01FF000000000000000000000000FFFFFFFFFFFFFFFFFFDFFFFFFFFFFFFF
DFFFFFFFEBFFDE64FFFFFFEFFFFFFFFFDFDFE7BF7BFFFFFFFFFDFC5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF3FF7FFFFFDF7FFFFFFFFDFFFFFFFDFFFFFFFFF7FFFFFFF7FFF
FFFFFDFFFFFFFDFF00000FF7000000007FFFFFFF000007E0000000000000000000000000000000000000000000000000
F9FFFF7FFFFF07DBFFFFFFFF00003FFF00008000000000000000000000000000FFFFFFFF3F801FFF0000400000000000
0000000000000000000000000000000000000000000000000000000000000000FFFF000000003FFFFFFFFFFF00000FFF
000000000000000000000000000000000000000000000000FFFF000000000FFF00000000000000000000000000000000
0000000000000000000000007FFF6F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000001F00000000
FFFFFFFFFFFFFFFF0000088F0000000000000000000000000000000000000000FFFFFFEF0AF7FE96AA96EA845EF7F796
0FFFFBFF0FFFFBEE000000000000000000000000FFFF0000FFFF03FFFFFF03FF000003FF000000000000000000000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000FFFFFFFF03FFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0003FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFF0001FFFFFFFFFFFFFFFF3FFFFFFF0000000000000000000000000000000000000000
3FFFFFFF00000000000000000000000000000000000000000000000000000000FFFFFFFFFFFFFFFFFFFF07FFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFF0000000000000000
0000000000000000000000000000000000000000000000000000000000000000",
);
//3392 bytes
enum markTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
000000000000004000000110",
cast(immutable size_t[]) x"
00000100000001A000004800",
cast(immutable size_t[]) x"
0202010004020302070206050A0908020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020B02020202020202020202020202020202020202020000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000001000000030002000500040007000600090008000B000A000D000C
0000000E000F000000000000001000000012001100140013001600150000000000000017000000000000000000000000
000000000000000000190018000000000000001A00000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000001B001D001C001F001E0000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000002000000000000000000021
0022000000240023000000000000000000000000000000250026000000280027002A0029002C002B002E002D0030002F
003200310000003300350034003700360000000000000000000000000000000000000000000000000000000000000000
000000000000000000000038000000000000000000000000000000000000000000000000000000000000000000000000
00000000003A003900000000003B00000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000003C0000000000000000000000000000000000000000000000000000000000000000003D0000
003E00000000003F000000000000000000000000000000400000000000000000003A0041000000420000004300000000
004500440000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
004600000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFF0000FFFF00000000000000000000000000000000
00000000000000000000000000000000000003F800000000000000000000000000000000000000000000000000000000
FFFE0000BFFFFFFF000000B60000000007FF000000000000FFFFF8000001000000000000000000009FC0000000003D9F
00020000FFFF0000000007FF00000000000000000001FFC000000000200FF800FBC0000000003EEF0E00000000000000
FF00000000000000FFFFFC00FFFFFFFB0000000FDC00000000FEFFFF0000000C0000000ED00000000080399F4000000C
0000000ED000000000023987002300000000000ED000000000003BBFFC00000C0000000ED000000000E0399F0000000C
00000004C000000000803DC7000000000000001FD000000000603DDF0000000C0000000ED000000000603DDF0008000C
0000000FD800000000803DDF0000000C0000000E00000000FF5F8400000C00000000000007F2000000007F8000000000
000000001FF2000000007F000000000003000000C2A0000000000000FFFE0000FEFFE0DF1FFFFFFF0000004000000000
000000007FFFF800C3C00000001E3F9D3C00BFFC0000000000000000000000000000000000000000E000000000000000
00000000000000000000000000000000003C0000001C0000000C0000000C000000000000FFF00000200FFFFF00000000
0000B80000000000000000000000000000000060000002000000000000000000000000000FFF0FFF0000000000000000
000000000000000000000000000000000F800000000000007FE000009FFFFFFF00000000FFFF000000007FFF00000000
0000001FFFF000000000001F000FF8000000000700003FFE00000000000FFFC00000000000FFFFF00000000000000000
0000000000000000FFF70000039021FF000000000000000000000000000000000000000000000000FFFFFFFFFFFFFFFF
000000000000000000000000000000000000000000000000FFFF00000001FFFF00000000000000000000000000000000
0000000000000000000000000003800000000000000000000000000080000000000000000000000000000000FFFFFFFF
000000000000FC000000000000000000060000000000000000000000000000000000000000000000000000003FF78000
C000000000000000000000000003000000000844000010F8000000000000000000000003FFF000000000003F8003FFFF
0000000000003FC0000FFF80000000000000000FFFF80000000000010000002000000000007FFE000000300838000000
00000000C19D0000000000020060F80000000000000000000000000000000000000000000000000000000000000037F8
40000000000000000000000000000000000000000000000000000000000000000000FFFF0000FFFF0000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000020000000
000000000000000000000000000000000000000000000000000000000000000100000000000000000000000007C00000
000000000000000000000000000000000000F06E87000000000000000000000000000000000000000000000000000060
00000000000000F000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000180000000000E000000000000000000000000001FFC0000000000000003C000000000000000000000000
00000007FF0000000000007F801900000000000707FF0000000000040000000000000007001FFF800000006000080000
00000007FFF800000000DE01000000000000000040FFF0000000000200000000000000000000000080000000000007FF
0000000FD80000000080399F001F1FCC0000000000000000000000000000000000000000FFE000004000007F00000000
00000000FFFF00000000000F000000000000000000000000000000000000000000000000FF3F80003000000100000000
00000000FFFF000000000001000000000000000000FFF8000000000000000000E000000000000FFF0000000000000000
000000000000000000000000000000000000000007FFF000000000000000000000000000000000000000000000000000
0000000079BF00000000000D000000000000000000000000FCFE000000000011000007FE7BF800000FFE008000000000
03FFFC0000000000000000000000000000000000FF7F80000000000000000000FFFC0000007FFEFF0000000000000000
00000000B47E0000000000BF0000000000FB7C0000000000000000000000000000000000000000000000000000000000
000000000000000000000000007800000000000BC7F00000000000070000000000000000000000000000000000000000
0000000000000000003FFF81000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000001F000000000000007F0000000000000000000000000000000000000000000000000000
0000000000000000FFFE8000FFFFFFFF000780FF00000000000000000003001000000000000000000000000000000000
60000000000000000000000000000000FFFFFFFFFFFF3FFF0000007F0000000000000000000000000000000000000000
000000000000000000000000F807E3E000000FE700003C00000000000000000000000000000000000000001C00000000
00000000000000000000000000000000FFFFFFFFF87FFFFFFFFFFFFF00201FFFF80000100000FFFE0000000000000000
F9FFFF7F000007DB00000000000000000000800000000000000000000000000000000000000000000000000000000000
0000000000004000000000000000F000000000000000000000000000000000000000000000000000000000000000F000
000000000000000000000000000000000000000000000000007F0000000000000000000000000000000007F000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFF
0000000000000000000000000000000000000000000000000000000000000000",
);
//2848 bytes
enum numberTrieEntries = TrieEntry!(bool, 8, 6, 7)(cast(immutable size_t[]) x"
0000000000000040000001A0",
cast(immutable size_t[]) x"
00000100000002C000002500",
cast(immutable size_t[]) x"
020201000402030207020605090802020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000001000000020002000200020002000200020002000200020004000300050002
000200020007000600060006000900080006000A0006000B000C000C0002000D000E000500020002000200020002000F
000200020002000200100002001100020002000E00130012001400020000000C00020015000200020002000200020002
00170016001900180002000200020002001B001A0002000200020002001D001C00020002000200020002000200020002
001E00020002000200020002000200020002001F00200002002200210002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020006002300020002
000C0024002500170002000C000400020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000E000200020027002600280002002A0029
00230002000200020002000200020002002C002B002E002D0030002F0032003100330002000200000002003400360035
00040037003900380004000200020002000C000C000200020005000C0002003A003B00020002000C0002000200020002
0002003C0023000C00020002003D000C000200020002000200020002000200020002003E000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
00020002000200020002000200020002000200020002000200020002000200020002000200020002000500030002003F
000200020002000200400002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200410002000200420002000200020002000200020043000200020002000200020002000200020002
000200020002000200020002000200020002000200020005000400020002000200040002000200020002000200020002
004400020002000C00020002000200020046004500020047000200020002000200020002000200480002000200020002
000200020002000200020002000200020002000200020002000200020004000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
00020002000200020002000200020002000200020002000200020002000200020000000003FF00000000000000000000
00000000720C0000000000000000000000000000000000000000000000000000000000000000000000000000000003FF
00000000000000000000000003FF00000000000000000000000003FF000000000000000000000000000000000000FFC0
00000000000000000000000003F0FFC000000000000000000000000000FCFFC00000000000000000000000000007FFC0
0000000000000000000000007F00FFC000000000000000007F00000001FFFFC0000000000000000003FF000000000000
00000000000FFFFF000000000000000003FF00000000000000000000000000000000000000000000000000001FFFFE00
0000000000000000000000000001C00000000000000000000000000003FF03FF00000000000000000000FFC000000000
000000000000000007FF00000000000003FF03FF000000000000000000000000000000000000000003FF03FF00000000
00000000000000000000000003F10000000003FF0000000000000000000000000000000000000000FFFF0000FFFFFFFF
000003E7000000000000000000000000000000000000000000000000FFFFFFFF0FFFFFFF0000000000000000FFFFFC00
000000000000000000000000FFC00000000FFFFF00000000000000000000000000000000000000000000000020000000
00000080070003FE0000000000000000003C000000000000000000000000000000000000000003FFFFFEFF0000000000
000003FFFFFE0000000000000000000000000000000003FF000000000000000000000000003F00000000000000000000
000000000000000003FF000003FF0000FFFFFF80000FFFFFFFFFFFFF01FFFFFF00000C00000000000000000000000000
0000000000000000000000000FFFFFFE000000000000000F00000402000000000000000000000000003E000000000000
0000000000000000FF000000FE000000000000000000FF8000000000F80000000FC00000000000000000000000000000
0000000030000000FFFCFFFFFFFFFFFF0000000000000000000001FF60000000E000000000000000000000000000F800
0000000000000000FF000000FF000000000000000000FE000000000000000000000000000000000000000000FC000000
0000000000000000000000007FFFFFFFE00000000000007F001E000000000000000000000000000000000FE000000000
0000000000000000FFFC00000000FFFF00000000FFC000000000000000000000000000000000000003FF0000001FFFFE
000000000FFF000000000000000000000000000000000000000000000007FFFF0000000000000000FFFF000000001FFF
0000000000000000001FFFFF00000000FFFFFFFFFFFFFFFFFFFFFFFF00007FFF0000000000000000FBFF000000000003
007FFFFF0000000000000000000000000000000000000000000FFFFF000FFFFF00000000000000000000000001FFFFFF
0000000000000000FFFFC000FFFFFFFF00000000000000000000FF8000000000000000000000000000000000FFFE0000
FFFFFFFF001EEFFF0000000000000000FFFFFFFE3FFFBFFF000000000000000000001FFF000000000000000000000000
00000000000000000000000003FF0000",
);
//3360 bytes
enum punctuationTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
000000000000004000000100",
cast(immutable size_t[]) x"
000001000000018000004900",
cast(immutable size_t[]) x"
0202010004020302070206050A0908020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000100000002000100030001000500040007000600010008000A0009000C000B
0001000D000E00010001000F001100100013001200150014000100160001000100010017001800010001000100190001
001A000100010001001C001B0001001D0001001E00010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
0001000100010001000100010001000100010001000100010001001F0001002000220021002400230001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010025000100270026
0028000100290001002A000100010001002C002B002E002D000100010030002F00320031000100330035003400370036
00390038003B003A0001003C003E003D00010001000100010001003F0001000100010001000100010001000100400001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010042004100010001004400430001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010045000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010046000100010001000100010001000100010001000100010001
004700010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
00010001000100010001000100010001000000008C00F7EEB8000001280000000000000088C008820000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000040000000
000000800000000000000000000000000000000000000000FC0000000000000000000600400000000000004900180000
E8003600000000000000000000003C000000000000000000001000000000000000003FFF000000000000000000000000
00000000000000000000000003800000000000007FFF0000400000000000000000000000000000000000000000000000
000000000000000000000000000100300000000000000000000000002000000000000000000000000000000000400000
000000000000000000000000000100000000000000000000000000000080000000000010000000000000000000000000
000000000000000000000000000000000000000000000000000000000010000000000000000000000C00800000000000
000000000000000000000000000000000017FFF03C00000000000000000000000000002000000000061F000000000000
00000000000000000000FC000000000000000000000000000000000008000000000000000000000000000000000001FF
000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000
000000000000000000000000000040001800000000000000000000000000380000000000006000000000000000000000
00000000000000000770000000000000000007FF00000000000000000000000000000000000000000000000000000000
0000000000000000000000300000000000000000000000000000000000000000C0000000000000000000000000000000
0000000000003F7F00000000000000000000000000000000FC00000060000001000000000000000000000000F0000000
00000000F800000000000000C00000000000000000000000000800FF00000000FFFF0000FFFF00FF7FFBFFEF60000000
0000600000000000000000000000000000000F0000000600000000000000000000000000000000000000000000000000
000000000000000000000000003FFF000000000000000000000000600000FFC000000000000000000000000000000000
01FFFFF8000000000F0000003000000000000000000000000000000000000000000000000000000000000000DE000000
0000000000000000000000000001000000000000000000000000000000000000FFFFFFFFFFFF7FFF3FFCFFFF00000000
00000000000000000000000000000000FFF3FF0E20010000000000000000000000000000000000010000000008000000
00000000000000000000000000000000000000000000000000000000C00000000000E000000000000000000040080000
00000000000000000000000000FC000000000000000000000000000000F0000000000000000000000000C00017000000
000000000000C00080000000000000000000000000000000C0003FFE000000000000000000000000F000000000000000
0000000000000000C0000000000300000000000000000000000000000000000000000000000000000000000000000800
00000000C000000000000000000000000000000000000000000000000000000003FF0000FFFF0000FFF7FFFF00000D0B
000000000000000000000000000000008C00F7EEB8000001A80000000000003F00000000000000000000000000000000
000000070000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
800000000000000000010000000000000000000000000000000000000000800000000000000000000000000000000000
000000000000000000800000000000000000000000000000000000000000000080000000800000000000000000000000
00000000000000000000000000000000000000000000000001FF000080000000000000000000000000000000007F0000
00000000FE00000000000000000000001E00000000000000000000000000000000000000000000000000000000000000
00000000000020000000000000000000000000000000000003E0000000000000000003C0000000000000000000000000
000000000000000000003F800000000000000000D8000000000000030000000000000000000000000000000F00300000
0000000000000000E80021E000000000000000003F000000000000000000000000000000000002000000000000000000
00000000000000002C00F800000000000000000000000000000000400000000000000000000000000000000000000000
000000000000000000FFFFFE0000000000000000000000000000000E00001FFF00000000020000000000000000000000
000000007000000000000000000000000000000000000000000000000000000000000000080000000000000000000000
000000000000000000000000000000000000000000000000000000700000000000000000000000000000000000000004
00000000800000000000007F00000000DC000000000000070000000000000000000003FF000000000000000000000000
0000000000000000000000000000000000000000000000000000003E0003000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000180000000000000000000000000FFF800000000
00000000000000000000000080000000000000000000000000000000001F000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000600000000000000000000000000000000C000
00000000000000000000000000200000000000000F800000000000100000000000000000000000000000000000000000
000000000000000000000000000000000780000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000040000000000000000000000000000000080000000000000000000000000000000
0000000000000000000000000000000000000F800000000000000000000000000000000000000000C000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000",
);
//3424 bytes
enum symbolTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
000000000000004000000100",
cast(immutable size_t[]) x"
000001000000018000004B00",
cast(immutable size_t[]) x"
0302010005030403070303060A0908030303030303030303030303030303030303030303030303030303030303030303
030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303
030303030303030303030303030303030303030303030303030303030303030303030303030303030000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000001000000030002000500040007000600090008000B000A000D000C000F000E
000100100011000100010001001300120014000100150001000100010016000100180017001A00190019001B001C0019
001D0019001E00190001001F002100200023002200190024000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001002500010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001002600270001
00010028002A002900010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
00010001002B0001002C0001002E002D002F000100010001000100010001000100010030000100310001000100010001
000100010001000100010001000E00010001000100010001000100010032000100010001000100010001000100010001
000100010033000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010034000100010001000100010001000100010001000100010001000100010001000100350001
003700360039003800010001003B003A001900190001003C0001000100010001003D00010001003E0001000100010001
00010001000100010040003F000100410043004200190044001900190046004500190047004900480001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
0001000100010001000100010001000100000000700008104000000050000001000000000113D37C0080000000800000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000FFFC003CFFFFAFE00000000000000000000000000020000000000030000000000000000000400000
000000000000000000000000000000000000000400000000000000000000000000000000000000000000000000000000
0000E0000000000000000000000000000000C9C000000000000000000000000000000000000000004000000060000200
00000000000000000000000000000000000000000000000000000000C040000000000000000000000000000000000000
00000100000000000000000000000000000000000000000000000000000000000000000000000000000000000C0C0000
000000000000000000000000000000000000000000000000000000000002000000000000000000000000000000010000
00000000000000000000000007F800000000000000000000000000008000000000000000000000000000000000000000
000000000000000000008000020000000000000000000000000000000000000000000000800000000000000000000000
00000000000000000000000000000000FCE8000E01500000000000000000000000000000C000000001E0DFBF00000000
00000000000000000000000000000000C000000000000000000000000000000000000000000000000000000000000000
03FF00000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000
000000000000000000000000000000000000000000000000080000000000000000000000000000000000000100000000
0000000000000000C0000000FFFFFFFF0000000000000000000000001FF007FE00000000000000000000000000000000
0000000000000000000000000000000000000000A0000000E000E0036000E0000000000000000000000400101C000000
00001C00FFFFFFFF0000000100000000C1D0037B0C0042AF0000BC1F00000000FFFF0C00FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0FFFFFFF9FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000007F000007FF00000000F0000000FFFFFFFFFFFFFFFF000003FF
FFFFFFFFFFFFFFFFFFFFFFFF000000FFFFF00000FFFFFFFFFFFFFF9FFFFF003FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FE000007FFFFFFFFF0FFFFFFCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFCFFFFFFFBFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00000000000000000000000000000000000000000000000000000000000007E000000000000000000003000000000000
FBFFFFFFFFFFFFFFFFFFFFFF000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF003FFFFFFFFF0000
000C0010C0C0000100000000000000001800000000000000000000000000000000000000000000000000000000000000
FFC3000000000000FFFFFFFF0000800F7FFFFFFFFFFFFC00000100FFFFFFFFFFFFFFFC000001FFFFFFFFFFFFFFFFFFFF
000000000000000000000000000000000000000000000000FFFFFFFFFFFFFFFF00000000000000000000000000000000
FFFF0000FFFFFFFF0000007F00000000007FFFFF00000003000000000000000000000600000000000000000000000000
0000000003C00F0000000000000000000000000000000000000000000000000000000000000000000000000003800000
0000000000000000000000000000000000000000000000000800000000000C0000000000000000000000000000000000
0000000000000200000000000000000000000000FFFC0000000000070000000000000000000000000000FFFF00000000
000000000000000000008000F00000000000000000000000000000000000027400000000000000000000000000000000
7000081040000000500000010000000000000000000000000000000030007F7F00000000FF80000000000000FE000000
1FFF73FF00000001FFFF00001FFFFFFF0000000000000000000000000180000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000
0000000000000000FFE000000003FFFF00000000F0000000000000200000000000000000000000000000000000000000
00000000000000000000000000000000100000000000000000000000000000000000000000000000FFFF0000FFFFFFFF
FFFFFFFFFFFFFFFF0000000F00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF003FFFFF
FFFFFFFFFFFFFE7FFFFFFFFF00001C1FFFFFF018FFFFC3FFFFFFFFFF000007FFFFFFFFFFFFFFFFFF0000002300000000
00000000000000000000000000000000FFFFFFFFFFFFFFFF007FFFFF0000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000080000020800000000200000002000000000800000008000
00000200000002000000000800000000000000000780000000000000FFDFE0000000006F000000000000000000000000
000000000000000000008000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000800000000000000000000000000000000000000000000000000110000000000000000000
000000000000400000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000030000FFFFFFFFFFFF0FFFFFFFFFFFFFFFFFFF000FFFFFFFFE7FFFFFFEFFFE003FFFFF
FFFFE000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00003FFF00000000FFFFFFC0FFFF00070FFFFFFF000301FF0000003F
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0FFFFFF1FFF1FFF
FFFFFFFFFFFFFFFFFFFFFFFFF87FFFFFFFFFFFFFFFFFFFFF03FFFFFF00010FFFFFFF0FFFFFFFFFFF03FF00FFFFFFFFFF
FFFF00FF00033FFF0000000000000000FFFFFFFFFFFFFFFF000FFFFF1FFF3FFFFFFF01FFBFFFFFFF0FFFC03F01FF01FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFFF000007FF0000000000000000000000000000000000000000
00000000000000000000000000000000",
);
//6080 bytes
enum graphicalTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
0000000000000040000001A0",
cast(immutable size_t[]) x"
00000100000002C000008A00",
cast(immutable size_t[]) x"
0202010005040302090807060D0C0B0A02020202100F0E02131312111313131313131313131313131313131313131313
131313131313131313131313131313131313131313131313131313131313131313131313131313131313131313131313
131313131313131313131313131313131313131413131313131313131313131313131313131313130000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000100000002000100030001000500040007000600090008000B000A000D000C
0001000E0010000F000100010012001100140013001600150001001700180001001A0019000100010001001B00010001
00010001001C0001001E001D0020001F0022002100010023000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100240026002500280027002A00290001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
00010001000100010001000100010001000100010001000100010001002B0001002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
0001002C002E002D002F00010031003000330032003500340037003600380001003A0039003C003B003E003D0040003F
00420041004400430046004500480047004A0049004C004B004E004D0050004F000100010051000100530052002C002C
002C002C002C002C002C002C0054002C0001000100010001002C0055002C002C002C002C002C002C002C002C002C002C
002C002C002C002C00010001002C0056002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C0001000100580057002C002C005A0059
0001000100010001000100010001000100010001000100010001000100010001000100010001000100010001005B0001
0001000100010001005D005C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C005E002C005F0001002C0060002C002C002C002C
002C002C002C002C002C0061002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C0062002C
006400630066006500680067006A006900010001002C006B002C002C006C002C006E006D002C006F002C00700071002C
00730072002C002C00750074002C0076007800770001007900010001007B007A0001007C007E007D002C002C002C002C
0001000100010001000100010001007F0001000100010001000100010001000100010001000100010001000100800001
000100810001000100010001000100010001000100010001000100010001000100010001000100010001000100010082
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010083000100010001002C0084002C002C002C002C002C002C002C002C00010001002C0085002C002C002C002C
000100010001000100010001000100010001000100010001000100010001000100010001008600010001000100010001
000100010001000100010001000100010001000100870001002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
0088002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C00000000FFFFFFFFFFFFFFFF7FFFFFFF
00000000FFFFDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFCFFFFFFFFFFD7F0FFFFFFFBFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFE7FFFFFFFFFFFFF
FFFEE7FFFFFFFFFFFFFF00FF001F87FFEFFFFFC0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFDFFFFFFFFFFFFFFF
FFFF3FFFFFFFFFFFFFFFE7FFFFFFFFFFFFFFFFFF0003FFFFFFFFFFFFE7FFFFFFFFFFFFFF7FFF3FFF4FFFFFFFFFFF07FF
FF007FFFFFFFFFFFFFFFFFFFFFFFFFFBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF99FEFF3C5FDFFB080799F7FFFFFCF
FFF987EED36DFDFF5E023987007FFFC0FFFBBFEEF3EDFDFF00013BBFFE03FFCFFFF99FEEF3EDFDFFB0E0399F00FFFFCF
D63DC7ECC3FFC71800813DC707FFFFC0FFFDDFFFF3FFFDFF27603DDFFF80FFCFFFFDDFFFF3EFFDFF60603DDF000EFFCF
FFFDDFFFFFFFFFFFFFF0FDDFFFFFFFCFFC7FFFEE2FFBFFFFFF5F847F001CFFC0FFFFFFFE87FFFFFF0FFFFFFF00000000
FFFFF7D63FFFFFAFF3FF7F5F00000000FFFFFFFFFFFFFFFFFFFFFEFFFFFE1FFFFEFFFFFFDFFFFFFF07FFDFFF00000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF20BFFFFFFFFFFFFFFFFFFFFFFFFF3D7F3DFFFFFFFFFF
FFFF3DFF7F3DFFFFFF7FFF3DFFFFFFFFFF3DFFFFFFFFFFFFE7FFFFFF1FFFFFFF03FFFFFFFFFFFFFFFFFFFFFF3F3FFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF1FFFFFFFFFFFFFFFFFFFFFFF01FFFFFF803FFFFF007FFFFF000FFFFF000DDFFF
FFFFFFFFFFFFFFFF3FFFFFFF03FF03FF03FFBFFFFFFFFFFFFFFFFFFF01FFFFFFFFFFFFFFFFFF07FFFFFFFFFF003FFFFF
7FFFFFFF0FFF0FFFFFFFFFF1001F3FFFFFFFFFFFFFFF0FFFC7FF03FFFFFFFFFFCFFFFFFFFFFFFFFF7FFFFFFF9FFFFFFF
03FF03FFFFFF3FFF00007FFF00000000FFFFFFFFFFFFFFFFFFFF1FFF7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFF
FFFFFFFFF8FFFFFFFFFFE3FFFFFFFFFFFFFF01FFE7FFFFFFFFFF00FF07FFFFFF3F3FFFFFFFFFFFFFAAFF3F3F3FFFFFFF
FFFFFFFFFFDFFFFFEFCFFFDF7FDCFFFFFFFF07FFFFFF80FFFFFFFFFFFFF300001FFF7FFFFFFFFFFFFFFF00010001FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000007F000007FFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFCFFFFFFFBFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFE0FFFFFFFFFFFFFFFFF20BFFFFFFFFF800180FF
007FFFFF7F7F7F7F7F7F7F7FFFFFFFFFFFFFFFFFFFFFFFFF3FFFFFFF00000000FBFFFFFFFFFFFFFFFFFFFFFF000FFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF003FFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF
FE7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFE0FFFEFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFFFFFFFFFFFFFF800F
7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF1FFFFFFFFFFFFFFF007FFFFFFFFFFFFFFFFF00000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03EB07FFFFFC0000FFFFFFFF03FF1FFFFFFFFFFF00FFFFFF
FFFFFFFFFFFFFFFF03FFC03FFFFFFFFFFFFFFFFFFFFFFFFF800FFFFF1FFFFFFFFFFFFFFFFFFFFFFFC3FFBFFF7FFFFFFF
FFFFFFFF007FFFFFF3FF3FFFFFFFFFFFFFFFFFFFFFFFFFFFF8000007007FFFFF007E7E7EFFFF7F7FFFFFFFFFFFFF0FFF
FFFFFFFFFFFFFFFFFFFFFFFF03FF3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000FFFFFF87F0FFFFFFF
0000000000000000000000000000000000000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFF3FFF
FFFFFFFFFFFFFFFF03FFFFFF00000000E0F8007F5F7FFFFFFFFFFFDBFFFFFFFFFFFFFFFFFFFFFFFFFFF80007FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFCFFFFFFFFFFFF000080FFFFFF000003FFFFFFFFFFFFFFFFF7FFFFFFDF0F7F
FFFFFFFFFFFFFFFFFFFFFFFF1FFFFFFFFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFF1CFCFCFC30007F7F
FFFFEFFFB7FFFF7F3FFF3FFF00000000FFFFFFFFFFFFFFFFFFFFFFFF07FFFFFFFFFFFF87FF8FFFFFFFFFFFFFFFFFFFFF
1FFF7FFF00000001FFFF00003FFFFFFF000000000000000000000000000000001FFFFFFFFFFFFFFF0001FFFF0FFFFFFF
FFFFFFFFFFFFE00FFFFF07FF07FFFFFFBFFFFFFFFFFFFFFF003FFF0F00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
3FFFFFFFFFFF03FFFF0FFFFF0FFFFFFFFFFFFFFFFFFF00FFFFFFFFFFF7FF800FFFB7F7FF1BFBFFFB0000000000000000
FFFFFFFF007FFFFF003FFFFF000000FFFFFFFFBF07FDFFFF0000000000000000FFFFFD3F91BFFFFFFFBFFFFFFFFFFFFF
7FFFFFFF0000FF8000000000F837FFFF8FFFFFFF83FFFFFF0000000000000000FFFFFFFFF0FFFFFFFFFCFFFFFFFFFFFF
FEEFF06F873FFFFF01FF01FFFFFFFFFFFFFFFFFF00000000FFFFFFFF007FF87FFFFFFFFFFE3FFFFFFF3FFFFFFF07FFFF
1E03FFFF0000FE000000000000000000FFFFFFFFFFFFFFFF000001FF00000000FFFFFFFF0007FFFFFFFFFFFFFC07FFFF
FFFFFFFF03FF00FF0000000000000000000000000000000000000000000000000000000000000000000000007FFFFFFF
FFFFFFFF00033BFF00000000E0000000FFFFFFFFFFFF00FF03FFFFFFFFFF0000000003FFFFFF000000000FFF007FFFFF
FFFFFFFFFFFFFFFFFFFC3FFF803FFFFFFFFFFFFFDFFFFFFFFFFF000703FF01FFFFFFFFFFFFDFFFFFFFFF00FF007FFFFF
FFFFFFFFFFFFFFFFFFFFFFFF001FFFFEFFFBFFFFFFFFFFFF0000000300000000BFFFBD7FFFFF03FFFFFFFFFF03FF07FF
FFF99FEFFBEDFDFFE081399F001F1FCF00000000000000000000000000000000FFFFFFFFFFFFFFFFEFFFFFFF00000003
FFFFFFFFFFFFFFFF03FF00FF0000000000000000000000000000000000000000FFFFFFFFFF3FFFFF3FFFFFFF00000000
FFFFFFFFFFFFFFFF03FF001F00001FFFFFFFFFFF03FFFFFF000003FF00000000E7FFFFFFFFFF0FFF0000007F00000000
00000000000000000000000000000000FFFFFFFF0FFFFFFF000000000000000000000000FFFFFFFFFFFFFFFF8007FFFF
FF6FF27FF9BFFFFF03FF007F0000000000000000FFFFFCFFFCFFFFFF0000001FFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFF
FFFFFFFFFFFF0007FFFFFFFF01FFFFFF000003FF00000000000000000000000000000000000000000000000000000000
FFFFFDFFFF7FFFFFFFFF003FFFFF1FFFFFFCFFFF007FFEFF0000000000000000FFFFFB7FB47FFFFF03FF00FFFFFFFDBF
01FB7FFF000003FF00000000000000000000000000000000000000000000000000000000000000000000000001FFFFFF
FFFDFFFFC7FFFFFF03FFFFFF000000000000000000010000FFFFFFFF8003FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03FFFFFF000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFF001F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF0000000F000000000000000000000000000000000000000000000000000000000000000000000000
FFFF0000FFFFFFFFFFFFFFFF0007FFFFFFFFFFFF0000FFFF003FFFFF0000000000000000000000000000000000000000
FFFFFFFFFFFFFFFF0000007F0000000000000000000000000000000000000000FFFFFFFF01FFFFFF7FFFFFFFFFFFC3FF
FFFFFFFF7FFFFFFFFFFF03FF003F3FFFFFFFFFFFFFFFFFFFFBFF003FE0FFFFFB0000FFFF000000000000000000000000
0000000000000000FFFFFFFFFFFFFFFF07FFFFFF000000000000000000000000FFFFFFFFFFFFFFFFFFFF87FFFFFFFFFF
FFFF80FF00000000000000000003001FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF003FFFFF00000000000001FF000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000006FEF0000
FFFFFFFF0004000700270000FFFF00F0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF1FFF07FFF3FF01FF000000000000000000000000
FFFFFFFFFFFF3FFFFFFF007FFFFFFFFFFFFFFFFFFFFFFFFF0000000F00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF003FFFFFFFFFFFFFFFFFFE7FFFFFFFFFF807FFFFFFFFFFFFFFFFFFFFFFFFFFFF000007FF
FFFFFFFFFFFFFFFF0000003F000000000000000000000000000FFFFF000FFFFFFFFFFFFFFFFFFFFF007FFFFF01FFFFFF
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFDFFFFFFFFFFFFFDFFFFFFFEBFFDE64FFFFFFEFFFFFFFFF
DFDFE7BF7BFFFFFFFFFDFC5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFF3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFCFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF8000FFF0000FFFE00000000000000007FFFFFFF000007E00000000000000000
00000000000000000000000000000000F9FFFF7FFFFF07DBFFFFFFFF00003FFF00008000000000000000000000000000
FFFFFFFF3FFF1FFF0000C3FF000000000000000000000000000000000000000000000000000000000000000000000000
FFFF000000007FFFFFFFFFFF83FFFFFF000000000000000000000000000000000000000000000000FFFF000003FFFFFF
000000000000000000000000000000000000000000000000000000007FFF6F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF007FFF9F00000000FFFFFFFFFFFFFFFFC3FF0FFF0000000000000000000000000000000000000000
000000000000000000000000FFFE0000FFFFFFFF001FFFFF0000000000000000FFFFFFFE3FFFFFFF0000000000000000
00000000000000000000000000000000FFFFFFEF0AF7FE96AA96EA845EF7F7960FFFFBFF0FFFFBEE0000000000030000
FFFFFFFFFFFF0FFFFFFFFFFFFFFFFFFF000FFFFFFFFE7FFFFFFEFFFE003FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF00003FFF00000000FFFFFFC0FFFF00070FFFFFFF000301FF0000003F00000000000000000000000000000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0FFFFFF1FFF1FFFFFFFFFFFFFFFFFFFFFFFFFFFF87FFFFF
FFFFFFFFFFFFFFFF03FFFFFF00010FFFFFFF0FFFFFFFFFFF03FF00FFFFFFFFFFFFFF00FF00033FFF0000000000000000
FFFFFFFFFFFFFFFF000FFFFF1FFF3FFFFFFF01FFBFFFFFFF0FFFC03F01FF01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFF7FFFFFFFFFFFF000007FF03FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000
FFFFFFFF03FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0003FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0001FFFFFFFFFFFFFFFF3FFFFFFF00000000
000000000000000000000000000000003FFFFFFF00000000000000000000000000000000000000000000000000000000
FFFFFFFFFFFFFFFFFFFF07FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF0000FFFF0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFF
0000000000000000000000000000000000000000000000000000000000000000",
);
//4824 bytes
enum nonCharacterTrieEntries = TrieEntry!(bool, 7, 4, 4, 6)(cast(immutable size_t[]) x"
00000000000000200000009800000298",
cast(immutable size_t[]) x"
00000080000000F000000400000043C0",
cast(immutable size_t[]) x"
0302010007060504090801010B0B0B0A0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B
0B0B0B0B0B0B0B0B0B0B0B0C0D0101010D01010100000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000010000000300020005000400070006
00090008000B000A000D000C000D000D000D000D000D000D000D000D000D000D000D000D000D000D000D000D000D000D
000D000D000D000D000D000D000D000D000E000D000D000F000D000D000D000D000D000D000D000D0010000D000D000D
000D000D000D000D000D000D00120011001400130016001500180017001A0019001C001B001E001D001F000D001D001D
0020001D001D001D001D001D001D001D001D001D00220021000D000D000D000D0023000D0024000D001D001D001D001D
001D001D0025001D001D00260027001D001D001D0028001D002A0029002C002B002E002D0030002F00320031001D0033
000D000D000D000D000D000D000D000D0034000D000D000D0035000D000D0036000D000D0037000D000D000D000D000D
000D000D00390038001D001D001D003A000D000D000D000D000D003B000D000D001D003C001D001D001D001D001D001D
001D001D001D001D001D001D001D001D001D001D001D001D001D001D001D001D001D003D001D001D001D001D001D001D
001D001D001D001D001D001D001D001D000D000D000D000D000D000D000D000D000D000D000D000D000D000D003E000D
000D000D000D000D000D000D000D000D000D000D000D000D000D000D003E000D00000000000000000000000000000000
0000000000000000000100000000000200000000000000000004000300060005000000000000000000080007000A0009
000C000B0000000D00000000000F000E0011001000130012001500140017001600190018001B001A001D001C001F001E
002100200023002200240000002600250000000000270000000000000000000000280000002A0029002C002B002E002D
00000000000000000000000000000000000000000030002F00320031003300000030002D003500340037003600390038
003B003A003D003C003E0000003F00000041004000430042000000000000000000000000000000000045004400470046
00480000004A0049000000000000004B00000000000000000000000000000000004D004C000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000004E00000000004F
000000000050000000520051005400530055000000570056000000000058000000590000000000040000005A005C005B
0000005D0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000005F005E00000000000000000000006000610000000000000062000000610063006400000065000000660000
00680067006900000038006A006B000000000000000000000000000000000000000000000000000000000000006D006C
00000000000000000000000000000000006E0000006F0000007100700072000000000000000000000000000000740073
0075002D007600000000005900780077007A0079007B00000000007C007E007D007F007F0080002F0082008100840083
000000000086008500880087007F00890000000000000000008A0067007F008B004F008C008E008D007F008F00730090
009200910094009300960095007F009700980000009A0099007F009B007F007F009C007F009E009D009F008700A100A0
00A2000000A3000000A4004600A5000000A700A600A900A800AB00AA007F007F00AC000000AD0000007F007F005500AE
00AF000000B100B000B300B2007F007F007F00B400B600B500B800B700BA00B900BB0000003000BC007F00B1007F007F
00BE00BD007F00BF00C100C0007F00C2007F007F00C3007F006F00C400C600C500000000000000000000000000000000
000000000000000000000000007F006F00C700000000000000C80000007F007F007F007F007F007F007F007F007F007F
007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F
007F007F007F007F007F007F009900C900CA0000007F007F007F007F007F007F007F007F007F007F007F007F007F007F
0000000000000000000000000000000000B30000007F007F007F007F007F007F00000000000000000000000000000000
00CB003000CC007700CD0000007F00CE007F007F007F007F007F007F007F007F0000007F007F00CF00D0000000D200D1
00000000000000000000000000000000000000000000000000000000006100000000000000CA0000007F0098007F007F
007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F00D3007F
000000000000000000D500D4000000000000000000B40000007F007F007F007F00D60000007F00D7007F007F007F007F
007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F005F006E00C80000
0000000000350000000000D800D9000000DA000000DB007F00DC0000007F007F00DD000000DF00DE00E100E000000000
00000000000000E20000000000E300000000000000000000000000000000000000000000007F00E4007F007F007F007F
007F007F007F007F007F007F007F007F007F007F007F007F007F00E5007F007F00E700E6007F00E800EA00E9007F007F
007F007F00EC00EB007F007F007F007F007F007F00ED007F007F007F007F007F007F007F007F007F007F007F00EE007F
0000000000EF000000F00000007F007F007F007F007F007F007F007F007F007F00F1007F007F00F2007F00F3007F007F
00F500F400F700F6007F007F007F007F0000003800F900F80000000000FA00E700FC00FB007F007F0000000000000000
000000000000000000000000000000000000000000FD000000FE000000FF00000100004B007F01010000000000000000
010200000104010300000000010601050000000000000000000000000000000000000000009300000000000000000000
000000000000000000000000000000000000000000000000000000B00000000000000107000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000001080000000000000000
000000000000000000000000000000000000000000000000000000000109000000000000000000000000000000000000
00550000007F007F007F007F007F007F00000000000000000000000000000000007F0055007F007F007F007F007F007F
000000000000000000000000000000000000000000000000010A00000000000000000000000000000000000000000000
000000000000000000000000007F010B0000010C007F007F00000000010B0000007F007F007F007F007F007F007F007F
00000000000000000000000000000000000000000000000000000000010D000000000000000000000000000000000000
000000000000000000000000010D0000000000000000000000000000030000000000280F000000040000000000010000
018000000000000000011800000000000000FF00FFE078000000400000000000000018000000000000000000FFFC0000
0000000018000000000000008000C000B00000000000F80000FC800000000000000660100C3A02004F7F866080000030
000678112C920200A1FDC678FF80003F000440110C120200FFFEC44001FC0030000660110C1202004F1FC660FF000030
29C238133C0038E7FF7EC238F800003F000220000C000200D89FC220007F0030000220000C1002009F9FC220FFF10030
0002200000000000000F02200000003003800011D004000000A07B80FFE3003F0000000178000000F0000000FFFFFFFF
00000829C00000500C0080A0FFFFFFFF000001000001E0000100000020000000F8002000FFFFFFFF0000DF4000000000
C280C200000000000000C20080C20000008000C20000000000C200000000000018000000E0000000FC00000000000000
00000000C0C00000E00000000000000000000000FE0000007FC00000FF800000FFF00000FFF22000C0000000FC00FC00
000000000000F80000000000FFC0000080000000F000F0000000000EFFE0C000000000000000F0003800FC0000000000
30000000000000008000000060000000FC00FC000000C000FFFF8000FFFFFFFF0000E00080000000000000000FF00000
000000000700000000001C00000000000000FE00180000000000FF00F8000000C0C00000000000005500C0C0C0000000
0000000000200000103000208023000000000000000C0020E0008000000000000000FFFEFFFE00000000F00000000000
00000000FFFFFF80FFFFF80000000000000000000030000000400000000000000000000001F00000000000000000DF40
000000007FFE7F00FF800000808080808080808000000000C0000000FFFFFFFF040000000000000000000000FFF00000
FFC000000000FFFF00000001000000000000001F0001000000008000000000000000000000007FF08000000000000000
0000E000000000000000FF800000000000000000FFFFF00000000000FF000000FC14F8000003FFFF00000000FC00E000
FC003FC0000000007FF00000E00000003C0040008000000000000000FF8000000C00C0000000000007FFFFF8FF800000
FF8181810000808000000000FC00C000000000000000FFF000000780F0000000000000000000C000FC000000FFFFFFFF
1F07FF80A080000000000024000000000007FFF8000000000003000000000000FFFF7F000000FFFF000800000020F080
00000000600000000000000080000000E3030303C1FF80800000100048000080C000C000FFFFFFFF00000000F8000000
0000007800700000E0008000FFFFFFFE0000FFFFC0000000FFFFFFFFFFFFFFFFFFFE0000F00000000000000000001FF0
0000F800F80000004000000000000000FFC000F0FFFFFFFFC00000000000FC0000F00000F0000000000000000000FF00
0000000008007FF000480800E4040004FFC00000FFFFFF0000000040F8020000000002C06E40000080000000FFFF007F
FFFFFFFF07C80000700000007C000000000000000F00000001100F9078C00000FE00FE000000000000000000FFFFFFFF
00000000FF8007800000000001C0000000C0000000F80000E1FC0000FFFF01FFFFFFFE00FFFFFFFF00000000FFF80000
0000000003F8000000000000FC00FF00FFFFFFFF8000000000000000FFFCC400FFFFFFFF1FFFFFFFFC0000000000FFFF
FFFFFC000000FFFFFFFFF000FF8000000003C0007FC000000000DFF8FC00FE000000FF00FF80000000000000FFE00001
0004000000000000FFFFFFFCFFFFFFFF400042800000FC0000000000FC00F80000066010041202001F7EC660FFE0E030
10000000FFFFFFFCFC00FF00FFFFFFFF0000000000C00000FC00FFE0FFFFE00000000000FC000000FFFFFC00FFFFFFFF
180000000000F000FFFFFF80FFFFFFFF00000000F0000000FFFFFFFF00000000000000007FF8000000900D8006400000
FC00FF80FFFFFFFFFFFFFFFF0000030003000000FFFFFFE00000FF0000000000000000000000FFF80000020000800000
0000FFC00000E00000030000FF800100000004804B800000FC00FF0000000240FE048000FFFFFC00FFFFFFFFFE000000
0002000038000000FFFFFFFFFFFEFFFF000000007FFC000000000000FFE08000FFFFFFF0FFFFFFFF0000FFFF00000000
FFC00000FFFFFFFF8000000000003C000000FC00FFC0C0000400FFC01F000004FFFF0000FFFFFFFFF8000000FFFFFFFF
000078000000000000007F00FFFFFFFFFFFFFFFFFFFCFFE0FFFFFFFF9010FFFF00000000FFFBFFF8FFD8FFFF0000FF0F
00000000E000F8000C00FE00FFFFFFF0000000000000018000000000FFFFF800FFFFFFC0FFFFFFFFFFF00000FFF00000
FF800000FE0000000020000000000000200000001400219B00000010000000002020184084000000000203A000000000
00000000000000C0000030000000000007FFF000FFFF000180000000FFFFF81F060000800000F82400000000FFFFC000
FFFF7FFFFFFFFFFF00000000C000E000FFFF3C00FFFFFFFF0000FFFFFFFF8000000000007C0000000000FFFFFC000000
FFFFFFFF80009080FF800060FFFFFFFF3C00F000FFFFFFFFFFFFFFFF0001FFFF00000000FFE0000000000001C0000000
00000010F50801695569157BA1080869F0000400F0000411FFFFFFFFFFFCFFFFFFF000000001800000010001FFC00000
FFFFFFFF0000003F0000FFF8F0000000FFFCFE00FFFFFFC00F000000E000E0000000000007800000FC000000FFFEF000
FC00FF00000000000000FF00FFFCC000FFF00000E000C0000000FE0040000000F0003FC0FE00FE000008000000000000
FFFFF800FC00FFFFC000000000000000000000000000FFFC000000000000FFFE0000F8000000000000000000FFFF0000
FFFFFFFD0000000000000000C000000000000000C0000000",
);
enum MAX_SIMPLE_LOWER = 1433;
enum MAX_SIMPLE_UPPER = 1450;
enum MAX_SIMPLE_TITLE = 1454;
//10496 bytes
enum toUpperIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000004000000280",
cast(immutable size_t[]) x"
000001000000048000000F80",
cast(immutable size_t[]) x"
020201000402030206020205070202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
0202020202020202020202020202020200010000000300020005000400070006000900080000000A000B0000000D000C
000F000E0011001000130012000000140000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000015000000000000000000000000000000000000000000000016000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000170018000000000019001B001A001D001C001F001E0021002000000000000000000022000000000023
000000000000000000000000000000000000000000240000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000260025002800270000002900000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000002A00000000002B002D002C002F002E00000000000000000000000000000000
000000000000000000300000000000310000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000032000000000000000000000000000000000000000000000000000000000033000000000000
000000000000000000000000000000000000000000000000000000000000000000350034003600000000000000000037
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000038000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000039000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000003A0000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000003C003B000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFF00020001000400030006000500080007000A0009000C000B000E000D
0010000F00120011001400130016001500180017FFFF0019FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001AFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05AAFFFF001C001B001E001D0020001F00220021
002400230026002500280027002A0029002C002B002E002D0030002FFFFF003100330032003500340037003600390038
003AFFFF003BFFFF003CFFFF003DFFFF003EFFFF003FFFFF0040FFFF0041FFFF0042FFFF0043FFFF0044FFFF0045FFFF
0046FFFF0047FFFF0048FFFF0049FFFF004AFFFF004BFFFF004CFFFF004DFFFF004EFFFF004FFFFF0050FFFF0051FFFF
005205AC0053FFFF0054FFFF0055FFFFFFFFFFFFFFFF0056FFFF0057FFFF0058FFFF0059FFFF005AFFFF005BFFFF005C
05C9005D005EFFFF005FFFFF0060FFFF0061FFFF0062FFFF0063FFFF0064FFFF0065FFFF0066FFFF0067FFFF0068FFFF
0069FFFF006AFFFF006BFFFF006CFFFF006DFFFF006EFFFF006FFFFF0070FFFF0071FFFF0072FFFF0073FFFF0074FFFF
FFFFFFFFFFFF0075FFFF007600780077FFFF0079007AFFFF007BFFFFFFFFFFFFFFFF007CFFFFFFFFFFFF007DFFFFFFFF
FFFFFFFFFFFF007E007FFFFFFFFFFFFF0080FFFFFFFF0081FFFFFFFFFFFF00820083FFFF0084FFFF0085FFFFFFFFFFFF
FFFF0086FFFFFFFF0087FFFFFFFFFFFFFFFF0088FFFFFFFFFFFF0089FFFF008A008BFFFFFFFFFFFF008CFFFF008DFFFF
FFFFFFFFFFFFFFFF008EFFFFFFFF008F009100900092FFFFFFFF0093FFFF0094FFFF0095FFFF0096FFFF0097FFFF0098
FFFF0099FFFF009A009C009B009DFFFF009EFFFF009FFFFF00A0FFFF00A1FFFF00A2FFFF00A3FFFF00A4FFFF00A5FFFF
FFFF05D100A700A600A8FFFFFFFFFFFF00A9FFFF00AAFFFF00ABFFFF00ACFFFF00ADFFFF00AEFFFF00AFFFFF00B0FFFF
00B1FFFF00B2FFFF00B3FFFF00B4FFFF00B5FFFF00B6FFFF00B7FFFF00B8FFFF00B9FFFF00BAFFFF00BBFFFF00BCFFFF
FFFFFFFF00BDFFFF00BEFFFF00BFFFFF00C0FFFF00C1FFFF00C2FFFF00C3FFFF00C4FFFF00C5FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF00C600C7FFFFFFFF00C8FFFF00C9FFFFFFFF00CAFFFF00CBFFFF00CCFFFF00CDFFFF00CEFFFF
00D000CF00D200D1FFFF00D300D500D400D6FFFF00D7FFFFFFFF00D8FFFFFFFF00DA00D900DBFFFF00DCFFFFFFFF00DD
00DF00DE00E100E0FFFF00E200E3FFFF00E4FFFFFFFF00E500E6FFFFFFFFFFFFFFFFFFFFFFFFFFFF00E7FFFFFFFFFFFF
FFFF00E800EA00E9FFFFFFFF00EBFFFF00ED00EC00EF00EEFFFF00F0FFFFFFFFFFFFFFFFFFFF00F1FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00F2FFFFFFFF00F3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F4FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F5FFFF00F6FFFFFFFFFFFF00F7FFFF
FFFFFFFF00F8FFFF00FA00F9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF05CBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00FC00FB00FE00FD00FF05CE0101010001030102010501040107010601090108010B010A010D010C
010F010E0111011001130112011501140117011601190118011B011AFFFF011C011E011DFFFFFFFF011FFFFF01210120
0122FFFF0123FFFF0124FFFF0125FFFF0126FFFF0127FFFF0128FFFF0129FFFF012AFFFF012BFFFF012CFFFF012DFFFF
012F012E013101300132FFFFFFFFFFFFFFFF01330134FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0136013501380137013A0139013C013B
013E013D0140013F01420141014401430146014501480147014A0149014C014B014E014D0150014F0152015101540153
0156015501580157015A0159015C015B015E015D0160015F01620161016401630165FFFF0166FFFF0167FFFF0168FFFF
0169FFFF016AFFFF016BFFFF016CFFFF016DFFFF016EFFFF016FFFFF0170FFFF0171FFFF0172FFFF0173FFFF0174FFFF
0175FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0176FFFF0177FFFF0178FFFF0179FFFF017AFFFF017BFFFF017CFFFF
017DFFFF017EFFFF017FFFFF0180FFFF0181FFFF0182FFFF0183FFFF0184FFFF0185FFFF0186FFFF0187FFFF0188FFFF
0189FFFF018AFFFF018BFFFF018CFFFF018DFFFF018EFFFF018FFFFF0190FFFFFFFFFFFFFFFF0191FFFF0192FFFF0193
FFFF0194FFFF0195FFFF0196019801970199FFFF019AFFFF019BFFFF019CFFFF019DFFFF019EFFFF019FFFFF01A0FFFF
01A1FFFF01A2FFFF01A3FFFF01A4FFFF01A5FFFF01A6FFFF01A7FFFF01A8FFFF01A9FFFF01AAFFFF01ABFFFF01ACFFFF
01ADFFFF01AEFFFF01AFFFFF01B0FFFF01B1FFFF01B2FFFF01B3FFFF01B4FFFF01B5FFFF01B6FFFF01B7FFFF01B8FFFF
01B9FFFF01BAFFFF01BBFFFF01BCFFFF01BDFFFF01BEFFFF01BFFFFF01C0FFFF01C1FFFF01C2FFFF01C3FFFF01C4FFFF
01C5FFFF01C6FFFF01C7FFFF01C8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01C9FFFF01CB01CA01CD01CC01CF01CE01D101D001D301D201D501D401D701D6
01D901D801DB01DA01DD01DC01DF01DE01E101E001E301E201E501E401E701E601E901E801EB01EA01ED01EC05BD01EE
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01F001EF01F201F101F401F301F601F501F801F701FA01F901FC01FB01FE01FD020001FF020202010204020302060205
02080207020A0209020C020B020E020D0210020F02120211021402130216021502180217FFFF0219021AFFFF021C021B
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF021E021D0220021F02220221FFFFFFFF022402230226022502280227022A0229
FFFF022BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022CFFFFFFFFFFFF022DFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022EFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022FFFFF0230FFFF0231FFFF0232FFFF
0233FFFF0234FFFF0235FFFF0236FFFF0237FFFF0238FFFF0239FFFF023AFFFF023BFFFF023CFFFF023DFFFF023EFFFF
023FFFFF0240FFFF0241FFFF0242FFFF0243FFFF0244FFFF0245FFFF0246FFFF0247FFFF0248FFFF0249FFFF024AFFFF
024BFFFF024CFFFF024DFFFF024EFFFF024FFFFF0250FFFF0251FFFF0252FFFF0253FFFF0254FFFF0255FFFF0256FFFF
0257FFFF0258FFFF0259FFFF025AFFFF025BFFFF025CFFFF025DFFFF025EFFFF025FFFFF0260FFFF0261FFFF0262FFFF
0263FFFF0264FFFF0265FFFF0266FFFF0267FFFF0268FFFF0269FFFF026AFFFF026BFFFF026CFFFF026DFFFF026EFFFF
026FFFFF0270FFFF0271FFFF0272FFFF0273FFFF0274FFFF0275FFFF0276FFFF0277FFFF0278FFFF0279FFFF05D505D3
05D905D7027A05DBFFFFFFFFFFFFFFFF027BFFFF027CFFFF027DFFFF027EFFFF027FFFFF0280FFFF0281FFFF0282FFFF
0283FFFF0284FFFF0285FFFF0286FFFF0287FFFF0288FFFF0289FFFF028AFFFF028BFFFF028CFFFF028DFFFF028EFFFF
028FFFFF0290FFFF0291FFFF0292FFFF0293FFFF0294FFFF0295FFFF0296FFFF0297FFFF0298FFFF0299FFFF029AFFFF
029BFFFF029CFFFF029DFFFF029EFFFF029FFFFF02A0FFFF02A1FFFF02A2FFFF02A3FFFF02A4FFFF02A5FFFF02A6FFFF
02A7FFFF02A8FFFF02A9FFFF02AAFFFF02AC02AB02AE02AD02B002AF02B202B1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02B402B302B602B502B802B7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02BA02B902BC02BB02BE02BD02C002BF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02C202C102C402C302C602C502C802C7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02CA02C902CC02CB02CE02CDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02CF05DD02D005DF02D105E202D205E5
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D402D302D602D502D802D702DA02D9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02DC02DB02DE02DD02E002DF02E202E102E402E302E602E502E802E7FFFFFFFF06080606060C060A0610060E06140612
06180616061C061A0620061E0624062206280626062C062A0630062E0634063206380636063C063A0640063E06440642
06480646064C064A0650064E0654065206580656065C065A0660065E066406620302030106660672FFFF0674067E05E8
FFFFFFFFFFFFFFFFFFFF0668FFFF0304FFFFFFFF066A0676FFFF0678068105EAFFFFFFFFFFFFFFFFFFFF066CFFFFFFFF
0307030605EF05ECFFFFFFFF05F405F2FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0309030805FA05F7030A05FD060105FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF066E067AFFFF067C06840604FFFFFFFFFFFFFFFFFFFF0670FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030CFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
030E030D0310030F03120311031403130316031503180317031A0319031C031BFFFFFFFFFFFFFFFFFFFF031DFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
031F031E0321032003230322032503240327032603290328032B032A032D032C032F032E033103300333033203350334
03370336FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03390338033B033A033D033C033F033E0341034003430342034503440347034603490348034B034A034D034C034F034E
0351035003530352035503540357035603590358035B035A035D035C035F035E03610360036303620365036403670366
0368FFFFFFFFFFFF0369FFFFFFFF036AFFFF036BFFFF036CFFFF036DFFFFFFFFFFFFFFFF036EFFFFFFFFFFFFFFFF036F
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0370FFFF0371FFFF0372FFFF0373FFFF0374FFFF0375FFFF0376FFFF0377FFFF
0378FFFF0379FFFF037AFFFF037BFFFF037CFFFF037DFFFF037EFFFF037FFFFF0380FFFF0381FFFF0382FFFF0383FFFF
0384FFFF0385FFFF0386FFFF0387FFFF0388FFFF0389FFFF038AFFFF038BFFFF038CFFFF038DFFFF038EFFFF038FFFFF
0390FFFF0391FFFF0392FFFF0393FFFF0394FFFF0395FFFF0396FFFF0397FFFF0398FFFF0399FFFF039AFFFF039BFFFF
039CFFFF039DFFFF039EFFFF039FFFFF03A0FFFF03A1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A2FFFF03A3
FFFFFFFF03A4FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A603A503A803A703AA03A903AC03AB
03AE03AD03B003AF03B203B103B403B303B603B503B803B703BA03B903BC03BB03BE03BD03C003BF03C203C103C403C3
03C603C503C803C703CA03C903CBFFFFFFFFFFFFFFFFFFFF03CCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03CDFFFF03CEFFFF03CFFFFF03D0FFFF03D1FFFF03D2FFFF03D3FFFF03D4FFFF
03D5FFFF03D6FFFF03D7FFFF03D8FFFF03D9FFFF03DAFFFF03DBFFFF03DCFFFF03DDFFFF03DEFFFF03DFFFFF03E0FFFF
03E1FFFF03E2FFFF03E3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03E4FFFF03E5FFFF03E6FFFF03E7FFFF03E8FFFF03E9FFFF03EAFFFF03EBFFFF03ECFFFF03EDFFFF03EEFFFF03EFFFFF
03F0FFFF03F1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF03F2FFFF03F3FFFF03F4FFFF03F5FFFF03F6FFFF03F7FFFF03F8FFFFFFFFFFFF03F9FFFF03FAFFFF03FBFFFF
03FCFFFF03FDFFFF03FEFFFF03FFFFFF0400FFFF0401FFFF0402FFFF0403FFFF0404FFFF0405FFFF0406FFFF0407FFFF
0408FFFF0409FFFF040AFFFF040BFFFF040CFFFF040DFFFF040EFFFF040FFFFF0410FFFF0411FFFF0412FFFF0413FFFF
0414FFFF0415FFFF0416FFFF0417FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0418FFFF0419041AFFFF
041BFFFF041CFFFF041DFFFF041EFFFFFFFFFFFFFFFFFFFFFFFF041FFFFFFFFF0420FFFF0421FFFFFFFF04220423FFFF
0424FFFF0425FFFF0426FFFF0427FFFF0428FFFF0429FFFF042AFFFF042BFFFF042CFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF042DFFFF042EFFFF042FFFFF0430FFFF0431FFFF0432FFFF0433FFFF0434FFFFFFFFFFFFFFFFFFFF
FFFF0435FFFF0436FFFFFFFFFFFFFFFF0437FFFFFFFFFFFFFFFFFFFF0438FFFF0439FFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043A
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF043BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043D043C043F043E0441044004430442044504440447044604490448044B044A
044D044C044F044E0451045004530452045504540457045604590458045B045A045D045C045F045E0461046004630462
046504640467046604690468046B046A046D046C046F046E0471047004730472047504740477047604790478047B047A
047D047C047F047E0481048004830482048504840487048604890488048B048A05AF05AD05B305B105B905B6FFFF05BB
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05BFFFFF05C305C105C705C5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF048CFFFF048E048D0490048F04920491049404930496049504980497049A0499
049C049B049E049D04A0049F04A204A104A404A3FFFF04A5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04A704A604A904A804AB04AA04AD04AC
04AF04AE04B104B004B304B204B504B404B704B604B904B804BB04BA04BD04BC04BF04BE04C104C004C304C204C504C4
04C704C604C904C804CB04CA04CD04CCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04CF04CE04D104D004D304D204D504D404D704D604D904D804DB04DA04DD04DC
04DF04DE04E104E004E304E204E504E404E704E604E904E804EB04EA04ED04EC04EF04EE04F104F0FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04F2FFFF
04F404F304F604F504F804F704FA04F904FC04FB04FDFFFF04FF04FE0501050005030502050505040507050605090508
050B050A050CFFFF050E050D0510050F051205110513FFFFFFFF0514FFFFFFFF0516051505180517051A0519051C051B
051E051D0520051F05220521052405230526052505280527052A0529052C052B052E052D0530052F0532053105340533
0536053505380537053A0539053C053B053E053D0540053F054205410544054305460545FFFF0547FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05490548054B054A054D054C054F054E05510550055305520555055405570556
05590558055B055A055D055C055F055E05610560056305620565056405670566FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05690568056B056A056D056C056F056E05710570057305720575057405770576
05790578057B057A057D057C057F057E05810580058305820585058405870586FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF05890588058B058A058D058C058F058E0591059005930592059505940597059605990598059B059A059D059C
059F059E05A105A005A305A205A505A405A705A605A905A8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//10112 bytes
enum toLowerIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000004000000280",
cast(immutable size_t[]) x"
000001000000048000000EC0",
cast(immutable size_t[]) x"
020201000402030206020205070202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
0202020202020202020202020202020200010000000200000004000300060005000800070000000000090000000B000A
000D000C000F000E00110010000000120000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000014001300000000000000000000000000000000000000000016001500000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000017000000000000000000190018001B001A001D001C001F001E00000000000000000021002000000022
000000000000000000000000000000000000000000240023000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000260025002800270000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000002900000000002A002C002B002E002D00000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000002F000000000000000000000000000000000000000000000000000000000000003000000000
000000000000000000000000000000000000000000000000000000000000000000000031003300320034000000000035
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000003600000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000003700000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000380000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000039000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0000FFFF00020001000400030006000500080007000A0009000C000B000E000D0010000F001200110014001300160015
00180017FFFF0019FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001B001A001D001C001F001E00210020
00230022002500240027002600290028002B002A002D002C002F002EFFFF003000320031003400330036003505990037
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0038FFFF0039FFFF003AFFFF003BFFFF003CFFFF003DFFFF003EFFFF003F
FFFF0040FFFF0041FFFF0042FFFF0043FFFF0044FFFF0045FFFF0046FFFF0047FFFF0048FFFF0049FFFF004AFFFF004B
FFFF004CFFFF004DFFFF004EFFFF004FFFFF059AFFFF0051FFFF0052FFFF00530054FFFF0055FFFF0056FFFF0057FFFF
0058FFFF0059FFFF005AFFFF005BFFFF05A9FFFFFFFF005CFFFF005DFFFF005EFFFF005FFFFF0060FFFF0061FFFF0062
FFFF0063FFFF0064FFFF0065FFFF0066FFFF0067FFFF0068FFFF0069FFFF006AFFFF006BFFFF006CFFFF006DFFFF006E
FFFF006FFFFF0070FFFF0071FFFF0072007400730075FFFF0076FFFFFFFFFFFF0077FFFFFFFF0078FFFF0079007B007A
007CFFFF007E007DFFFFFFFF0080007F008200810083FFFFFFFF008400860085FFFF0087FFFFFFFF00890088008AFFFF
FFFF008BFFFF008CFFFF008D008F008E0090FFFFFFFFFFFFFFFF0091009300920094FFFF009600950097FFFF0098FFFF
FFFF0099FFFFFFFFFFFF009AFFFFFFFFFFFFFFFFFFFFFFFF009C009B009DFFFFFFFF009E00A0009F00A1FFFF00A2FFFF
00A3FFFF00A4FFFF00A5FFFF00A6FFFF00A7FFFF00A8FFFFFFFFFFFFFFFF00A9FFFF00AAFFFF00ABFFFF00ACFFFF00AD
FFFF00AEFFFF00AFFFFF00B0FFFF00B100B205ACFFFF00B3FFFF00B400B600B5FFFF00B7FFFF00B8FFFF00B9FFFF00BA
FFFF00BBFFFF00BCFFFF00BDFFFF00BEFFFF00BFFFFF00C0FFFF00C1FFFF00C2FFFF00C3FFFF00C4FFFF00C5FFFF00C6
FFFF00C7FFFF00C8FFFF00C9FFFF00CAFFFF00CBFFFF00CCFFFF00CDFFFF00CEFFFF00CFFFFF00D0FFFF00D1FFFF00D2
FFFF00D3FFFF00D4FFFFFFFFFFFFFFFFFFFFFFFF00D600D500D7FFFFFFFF00D800D9FFFF00DAFFFF00DC00DBFFFF00DD
FFFF00DEFFFF00DFFFFF00E0FFFF00E1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00E2FFFF00E3FFFFFFFFFFFF00E4FFFFFFFFFFFFFFFFFFFFFFFF00E5FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFF00E600E800E7FFFF00E9FFFF00EA00EC00EB00ED05AA00EF00EE00F100F000F300F2
00F500F400F700F600F900F800FB00FA00FD00FC00FEFFFF010000FF010201010104010301060105FFFFFFFFFFFFFFFF
FFFF05ABFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF0107FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0108FFFF0109FFFF010AFFFF010B
FFFF010CFFFF010DFFFF010EFFFF010FFFFF0110FFFF0111FFFF0112FFFF0113FFFFFFFFFFFFFFFFFFFF01140115FFFF
0116FFFFFFFF01170118FFFF011A0119011C011B011E011D0120011F01220121012401230126012501280127012A0129
012C012B012E012D0130012F01320131013401330136013501380137013A0139013C013B013E013D0140013F01420141
014401430146014501480147014A0149FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF014BFFFF014CFFFF014DFFFF014EFFFF014FFFFF0150FFFF0151FFFF0152
FFFF0153FFFF0154FFFF0155FFFF0156FFFF0157FFFF0158FFFF0159FFFF015AFFFF015BFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF015CFFFF015DFFFF015EFFFF015FFFFF0160FFFF0161FFFF0162FFFF0163FFFF0164FFFF0165FFFF0166
FFFF0167FFFF0168FFFF0169FFFF016AFFFF016BFFFF016CFFFF016DFFFF016EFFFF016FFFFF0170FFFF0171FFFF0172
FFFF0173FFFF0174FFFF0175FFFF0176017801770179FFFF017AFFFF017BFFFF017CFFFF017DFFFF017EFFFFFFFFFFFF
FFFF017FFFFF0180FFFF0181FFFF0182FFFF0183FFFF0184FFFF0185FFFF0186FFFF0187FFFF0188FFFF0189FFFF018A
FFFF018BFFFF018CFFFF018DFFFF018EFFFF018FFFFF0190FFFF0191FFFF0192FFFF0193FFFF0194FFFF0195FFFF0196
FFFF0197FFFF0198FFFF0199FFFF019AFFFF019BFFFF019CFFFF019DFFFF019EFFFF019FFFFF01A0FFFF01A1FFFF01A2
FFFF01A3FFFF01A4FFFF01A5FFFF01A6FFFF01A7FFFF01A8FFFF01A9FFFF01AAFFFF01ABFFFF01ACFFFF01ADFFFF01AE
01AFFFFF01B101B001B301B201B501B401B701B601B901B801BB01BA01BD01BC01BF01BE01C101C001C301C201C501C4
01C701C601C901C801CB01CA01CD01CC01CF01CE01D101D001D301D2FFFF01D4FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05A3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01D601D501D801D701DA01D901DC01DB01DE01DD01E001DF01E201E101E401E3
01E601E501E801E701EA01E901EC01EB01EE01ED01F001EF01F201F101F401F301F601F501F801F701FA01F901FBFFFF
FFFFFFFFFFFFFFFF01FCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01FE01FD020001FF0202020102040203
0206020502080207020A0209020C020B020E020D0210020F02120211021402130216021502180217021A0219021C021B
021E021D0220021F02220221022402230226022502280227022A0229022C022B022E022D0230022F0232023102340233
0236023502380237023A0239023C023B023E023D0240023F02420241024402430246024502480247024A0249024C024B
024E024D0250024F02520251FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF025402530256025502580257025A0259025C025B025E025D0260025F02620261
026402630266026502680267026A0269026C026B026E026D0270026F02720271027402730276027502780277027A0279
027C027BFFFF027D027EFFFF0280027FFFFF0281FFFF0282FFFF0283FFFF0284FFFF0285FFFF0286FFFF0287FFFF0288
FFFF0289FFFF028AFFFF028BFFFF028CFFFF028DFFFF028EFFFF028FFFFF0290FFFF0291FFFF0292FFFF0293FFFF0294
FFFF0295FFFF0296FFFF0297FFFF0298FFFF0299FFFF029AFFFF029BFFFF029CFFFF029DFFFF029EFFFF029FFFFF02A0
FFFF02A1FFFF02A2FFFF02A3FFFF02A4FFFF02A5FFFF02A6FFFF02A7FFFF02A8FFFF02A9FFFF02AAFFFF02ABFFFF02AC
FFFF02ADFFFF02AEFFFF02AFFFFF02B0FFFF02B1FFFF02B2FFFF02B3FFFF02B4FFFF02B5FFFF02B6FFFF02B7FFFF02B8
FFFF02B9FFFF02BAFFFF02BBFFFF02BCFFFF02BDFFFF02BEFFFF02BFFFFF02C0FFFF02C1FFFF02C2FFFF02C3FFFF02C4
FFFF02C5FFFF02C6FFFF02C7FFFF02C8FFFF02C9FFFF02CAFFFF02CB05AE05AD05B005AFFFFF05B1FFFFFFFFFFFF02CC
FFFF02CDFFFF02CEFFFF02CFFFFF02D0FFFF02D1FFFF02D2FFFF02D3FFFF02D4FFFF02D5FFFF02D6FFFF02D7FFFF02D8
FFFF02D9FFFF02DAFFFF02DBFFFF02DCFFFF02DDFFFF02DEFFFF02DFFFFF02E0FFFF02E1FFFF02E2FFFF02E3FFFF02E4
FFFF02E5FFFF02E6FFFF02E7FFFF02E8FFFF02E9FFFF02EAFFFF02EBFFFF02ECFFFF02EDFFFF02EEFFFF02EFFFFF02F0
FFFF02F1FFFF02F2FFFF02F3FFFF02F4FFFF02F5FFFF02F6FFFF02F7FFFF02F8FFFF02F9FFFF02FAFFFF02FBFFFF02FC
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02FE02FD030002FF0302030103040303FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0306030503080307030A0309FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030C030B030E030D0310030F03120311
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF031403130316031503180317031A0319FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
031C031B031E031D0320031FFFFFFFFFFFFF05B2FFFF05B3FFFF05B4FFFF05B50321FFFF0322FFFF0323FFFF0324FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0326032503280327032A0329032C032BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05C305C205C505C405C705C605C905C805CB05CA05CD05CC05CF05CE05D105D0
05D305D205D505D405D705D605D905D805DB05DA05DD05DC05DF05DE05E105E005E305E205E505E405E705E605E905E8
05EB05EA05ED05EC05EF05EE05F105F0FFFFFFFF05F205F8FFFF05F905FE05B60346034503480347FFFF05F3FFFFFFFF
FFFFFFFF05F405FAFFFF05FB05FF05B7034B034A034D034CFFFF05F5FFFFFFFFFFFFFFFF05B905B8FFFFFFFF05BB05BA
0350034F03520351FFFFFFFFFFFFFFFFFFFFFFFF05BD05BCFFFF05BE05C005BF0354035303560355FFFF0357FFFFFFFF
FFFFFFFF05F605FCFFFF05FD060005C103590358035B035AFFFF05F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFF035DFFFFFFFF035F035EFFFFFFFFFFFFFFFFFFFFFFFFFFFF0360FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03620361036403630366036503680367
036A0369036C036B036E036D0370036FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF0371FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03730372
037503740377037603790378037B037A037D037C037F037E0381038003830382038503840387038603890388038B038A
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
038D038C038F038E0391039003930392039503940397039603990398039B039A039D039C039F039E03A103A003A303A2
03A503A403A703A603A903A803AB03AA03AD03AC03AF03AE03B103B003B303B203B503B403B703B603B903B803BB03BA
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF03BC03BE03BDFFFF03BF03C0FFFF03C1FFFF03C2FFFF03C3FFFF03C503C4FFFF03C6FFFF03C703C8FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF03CA03C9FFFF03CBFFFF03CCFFFF03CDFFFF03CEFFFF03CFFFFF03D0FFFF03D1FFFF03D2
FFFF03D3FFFF03D4FFFF03D5FFFF03D6FFFF03D7FFFF03D8FFFF03D9FFFF03DAFFFF03DBFFFF03DCFFFF03DDFFFF03DE
FFFF03DFFFFF03E0FFFF03E1FFFF03E2FFFF03E3FFFF03E4FFFF03E5FFFF03E6FFFF03E7FFFF03E8FFFF03E9FFFF03EA
FFFF03EBFFFF03ECFFFF03EDFFFF03EEFFFF03EFFFFF03F0FFFF03F1FFFF03F2FFFF03F3FFFF03F4FFFF03F5FFFF03F6
FFFF03F7FFFF03F8FFFF03F9FFFF03FAFFFF03FBFFFF03FCFFFFFFFFFFFFFFFFFFFFFFFF03FDFFFF03FEFFFFFFFFFFFF
FFFFFFFFFFFF03FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0400FFFF0401FFFF0402FFFF0403
FFFF0404FFFF0405FFFF0406FFFF0407FFFF0408FFFF0409FFFF040AFFFF040BFFFF040CFFFF040DFFFF040EFFFF040F
FFFF0410FFFF0411FFFF0412FFFF0413FFFF0414FFFF0415FFFF0416FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0417FFFF0418FFFF0419FFFF041AFFFF041BFFFF041CFFFF041DFFFF041E
FFFF041FFFFF0420FFFF0421FFFF0422FFFF0423FFFF0424FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0425FFFF0426FFFF0427FFFF0428FFFF0429FFFF042AFFFF042B
FFFFFFFFFFFF042CFFFF042DFFFF042EFFFF042FFFFF0430FFFF0431FFFF0432FFFF0433FFFF0434FFFF0435FFFF0436
FFFF0437FFFF0438FFFF0439FFFF043AFFFF043BFFFF043CFFFF043DFFFF043EFFFF043FFFFF0440FFFF0441FFFF0442
FFFF0443FFFF0444FFFF0445FFFF0446FFFF0447FFFF0448FFFF0449FFFF044AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
044BFFFF044CFFFF044DFFFFFFFF044EFFFF044FFFFF0450FFFF0451FFFF0452FFFFFFFF0453FFFF0454FFFFFFFFFFFF
FFFF0455FFFF0456FFFFFFFFFFFF0457FFFF0458FFFF0459FFFF045AFFFF045BFFFF045CFFFF045DFFFF045EFFFF045F
FFFF04600462046104640463FFFF04650467046604690468FFFF046AFFFF046BFFFF046CFFFF046DFFFF046EFFFF046F
FFFF0470FFFF047104730472047504740476FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0477FFFFFFFFFFFFFFFFFFFF0478
FFFF0479FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF047AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF059D059C059F059E05A105A0FFFF05A2
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05A4FFFF05A605A505A805A7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF047BFFFF047D047C047F047E04810480
04830482048504840487048604890488048B048A048D048C048F048E0491049004930492FFFF0494FFFFFFFFFFFFFFFF
0496049504980497049A0499049C049B049E049D04A0049F04A204A104A404A304A604A504A804A704AA04A904AC04AB
04AE04AD04B004AF04B204B104B404B304B604B504B804B704BA04B904BC04BBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04BE04BD04C004BF04C204C104C404C3
04C604C504C804C704CA04C904CC04CB04CE04CD04D004CF04D204D104D404D304D604D504D804D704DA04D904DC04DB
04DE04DD04E004DFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
04E204E104E404E304E604E504E804E704EA04E9FFFF04EB04ED04EC04EF04EE04F104F004F304F204F504F404F704F6
04F904F8FFFF04FA04FC04FB04FE04FD050004FFFFFF050105030502FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF050505040507050605090508050B050A050D050C050F050E0511051005130512
051505140517051605190518051B051A051D051C051F051E0521052005230522052505240527052605290528052B052A
052D052C052F052E053105300533053205350534FFFF0536FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05380537053A0539053C053B053E053D0540053F054205410544054305460545
05480547054A0549054C054B054E054D0550054F05520551055405530556055505580557055A0559055C055B055E055D
0560055F05620561056405630566056505680567056A0569056C056B056E056D0570056F057205710574057305760575
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05780577057A0579057C057B057E057D0580057F058205810584058305860585
05880587058A0589058C058B058E058D0590058F05920591059405930596059505980597FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//10496 bytes
enum toTitleIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000004000000280",
cast(immutable size_t[]) x"
000001000000048000000F80",
cast(immutable size_t[]) x"
020201000402030206020205070202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
0202020202020202020202020202020200010000000300020005000400070006000900080000000A000B0000000D000C
000F000E0011001000130012000000140000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000015000000000000000000000000000000000000000000000016000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000170018000000000019001B001A001D001C001F001E0021002000000000000000000022000000000023
000000000000000000000000000000000000000000240000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000260025002800270000002900000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000002A00000000002B002D002C002F002E00000000000000000000000000000000
000000000000000000300000000000310000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000032000000000000000000000000000000000000000000000000000000000033000000000000
000000000000000000000000000000000000000000000000000000000000000000350034003600000000000000000037
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000038000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000039000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000003A0000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000003C003B000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFF00020001000400030006000500080007000A0009000C000B000E000D
0010000F00120011001400130016001500180017FFFF0019FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001AFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05AEFFFF001C001B001E001D0020001F00220021
002400230026002500280027002A0029002C002B002E002D0030002FFFFF003100330032003500340037003600390038
003AFFFF003BFFFF003CFFFF003DFFFF003EFFFF003FFFFF0040FFFF0041FFFF0042FFFF0043FFFF0044FFFF0045FFFF
0046FFFF0047FFFF0048FFFF0049FFFF004AFFFF004BFFFF004CFFFF004DFFFF004EFFFF004FFFFF0050FFFF0051FFFF
005205B00053FFFF0054FFFF0055FFFFFFFFFFFFFFFF0056FFFF0057FFFF0058FFFF0059FFFF005AFFFF005BFFFF005C
05CD005D005EFFFF005FFFFF0060FFFF0061FFFF0062FFFF0063FFFF0064FFFF0065FFFF0066FFFF0067FFFF0068FFFF
0069FFFF006AFFFF006BFFFF006CFFFF006DFFFF006EFFFF006FFFFF0070FFFF0071FFFF0072FFFF0073FFFF0074FFFF
FFFFFFFFFFFF0075FFFF007600780077FFFF0079007AFFFF007BFFFFFFFFFFFFFFFF007CFFFFFFFFFFFF007DFFFFFFFF
FFFFFFFFFFFF007E007FFFFFFFFFFFFF0080FFFFFFFF0081FFFFFFFFFFFF00820083FFFF0084FFFF0085FFFFFFFFFFFF
FFFF0086FFFFFFFF0087FFFFFFFFFFFFFFFF0088FFFFFFFFFFFF0089FFFF008A008BFFFFFFFFFFFF008CFFFF008DFFFF
FFFFFFFFFFFFFFFF008F008E009100900093009200950094FFFF0096FFFF0097FFFF0098FFFF0099FFFF009AFFFF009B
FFFF009CFFFF009D009F009E00A0FFFF00A1FFFF00A2FFFF00A3FFFF00A4FFFF00A5FFFF00A6FFFF00A7FFFF00A8FFFF
00A905D500AB00AA00ACFFFFFFFFFFFF00ADFFFF00AEFFFF00AFFFFF00B0FFFF00B1FFFF00B2FFFF00B3FFFF00B4FFFF
00B5FFFF00B6FFFF00B7FFFF00B8FFFF00B9FFFF00BAFFFF00BBFFFF00BCFFFF00BDFFFF00BEFFFF00BFFFFF00C0FFFF
FFFFFFFF00C1FFFF00C2FFFF00C3FFFF00C4FFFF00C5FFFF00C6FFFF00C7FFFF00C8FFFF00C9FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF00CA00CBFFFFFFFF00CCFFFF00CDFFFFFFFF00CEFFFF00CFFFFF00D0FFFF00D1FFFF00D2FFFF
00D400D300D600D5FFFF00D700D900D800DAFFFF00DBFFFFFFFF00DCFFFFFFFF00DE00DD00DFFFFF00E0FFFFFFFF00E1
00E300E200E500E4FFFF00E600E7FFFF00E8FFFFFFFF00E900EAFFFFFFFFFFFFFFFFFFFFFFFFFFFF00EBFFFFFFFFFFFF
FFFF00EC00EE00EDFFFFFFFF00EFFFFF00F100F000F300F2FFFF00F4FFFFFFFFFFFFFFFFFFFF00F5FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00F6FFFFFFFF00F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F8FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F9FFFF00FAFFFFFFFFFFFF00FBFFFF
FFFFFFFF00FCFFFF00FE00FDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF05CFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF010000FF01020101010305D2010501040107010601090108010B010A010D010C010F010E01110110
01130112011501140117011601190118011B011A011D011C011F011EFFFF012001220121FFFFFFFF0123FFFF01250124
0126FFFF0127FFFF0128FFFF0129FFFF012AFFFF012BFFFF012CFFFF012DFFFF012EFFFF012FFFFF0130FFFF0131FFFF
01330132013501340136FFFFFFFFFFFFFFFF01370138FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF013A0139013C013B013E013D0140013F
01420141014401430146014501480147014A0149014C014B014E014D0150014F01520151015401530156015501580157
015A0159015C015B015E015D0160015F016201610164016301660165016801670169FFFF016AFFFF016BFFFF016CFFFF
016DFFFF016EFFFF016FFFFF0170FFFF0171FFFF0172FFFF0173FFFF0174FFFF0175FFFF0176FFFF0177FFFF0178FFFF
0179FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF017AFFFF017BFFFF017CFFFF017DFFFF017EFFFF017FFFFF0180FFFF
0181FFFF0182FFFF0183FFFF0184FFFF0185FFFF0186FFFF0187FFFF0188FFFF0189FFFF018AFFFF018BFFFF018CFFFF
018DFFFF018EFFFF018FFFFF0190FFFF0191FFFF0192FFFF0193FFFF0194FFFFFFFFFFFFFFFF0195FFFF0196FFFF0197
FFFF0198FFFF0199FFFF019A019C019B019DFFFF019EFFFF019FFFFF01A0FFFF01A1FFFF01A2FFFF01A3FFFF01A4FFFF
01A5FFFF01A6FFFF01A7FFFF01A8FFFF01A9FFFF01AAFFFF01ABFFFF01ACFFFF01ADFFFF01AEFFFF01AFFFFF01B0FFFF
01B1FFFF01B2FFFF01B3FFFF01B4FFFF01B5FFFF01B6FFFF01B7FFFF01B8FFFF01B9FFFF01BAFFFF01BBFFFF01BCFFFF
01BDFFFF01BEFFFF01BFFFFF01C0FFFF01C1FFFF01C2FFFF01C3FFFF01C4FFFF01C5FFFF01C6FFFF01C7FFFF01C8FFFF
01C9FFFF01CAFFFF01CBFFFF01CCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01CDFFFF01CF01CE01D101D001D301D201D501D401D701D601D901D801DB01DA
01DD01DC01DF01DE01E101E001E301E201E501E401E701E601E901E801EB01EA01ED01EC01EF01EE01F101F005C101F2
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01F401F301F601F501F801F701FA01F901FC01FB01FE01FD020001FF02020201020402030206020502080207020A0209
020C020B020E020D0210020F02120211021402130216021502180217021A0219021C021BFFFF021D021EFFFF0220021F
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022202210224022302260225FFFFFFFF02280227022A0229022C022B022E022D
FFFF022FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0230FFFFFFFFFFFF0231FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0232FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0233FFFF0234FFFF0235FFFF0236FFFF
0237FFFF0238FFFF0239FFFF023AFFFF023BFFFF023CFFFF023DFFFF023EFFFF023FFFFF0240FFFF0241FFFF0242FFFF
0243FFFF0244FFFF0245FFFF0246FFFF0247FFFF0248FFFF0249FFFF024AFFFF024BFFFF024CFFFF024DFFFF024EFFFF
024FFFFF0250FFFF0251FFFF0252FFFF0253FFFF0254FFFF0255FFFF0256FFFF0257FFFF0258FFFF0259FFFF025AFFFF
025BFFFF025CFFFF025DFFFF025EFFFF025FFFFF0260FFFF0261FFFF0262FFFF0263FFFF0264FFFF0265FFFF0266FFFF
0267FFFF0268FFFF0269FFFF026AFFFF026BFFFF026CFFFF026DFFFF026EFFFF026FFFFF0270FFFF0271FFFF0272FFFF
0273FFFF0274FFFF0275FFFF0276FFFF0277FFFF0278FFFF0279FFFF027AFFFF027BFFFF027CFFFF027DFFFF05D905D7
05DD05DB027E05DFFFFFFFFFFFFFFFFF027FFFFF0280FFFF0281FFFF0282FFFF0283FFFF0284FFFF0285FFFF0286FFFF
0287FFFF0288FFFF0289FFFF028AFFFF028BFFFF028CFFFF028DFFFF028EFFFF028FFFFF0290FFFF0291FFFF0292FFFF
0293FFFF0294FFFF0295FFFF0296FFFF0297FFFF0298FFFF0299FFFF029AFFFF029BFFFF029CFFFF029DFFFF029EFFFF
029FFFFF02A0FFFF02A1FFFF02A2FFFF02A3FFFF02A4FFFF02A5FFFF02A6FFFF02A7FFFF02A8FFFF02A9FFFF02AAFFFF
02ABFFFF02ACFFFF02ADFFFF02AEFFFF02B002AF02B202B102B402B302B602B5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02B802B702BA02B902BC02BBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02BE02BD02C002BF02C202C102C402C3
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02C602C502C802C702CA02C902CC02CBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02CE02CD02D002CF02D202D1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D305E102D405E302D505E602D605E9
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D802D702DA02D902DC02DB02DE02DDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02E002DF02E202E102E402E302E602E502E802E702EA02E902EC02EBFFFFFFFF060B060A060D060C060F060E06110610
06130612061506140617061606190618061B061A061D061C061F061E0621062006230622062506240627062606290628
062B062A062D062C062F062E063106300633063206350634063706360639063803060305063A0640FFFF0642064C05EC
FFFFFFFFFFFFFFFFFFFF063BFFFF0308FFFFFFFF063C0644FFFF0646064F05EEFFFFFFFFFFFFFFFFFFFF063DFFFFFFFF
030B030A05F305F0FFFFFFFF05F805F6FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030D030C05FE05FB030E060106050603
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF063E0648FFFF064A06520608FFFFFFFFFFFFFFFFFFFF063FFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0310FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03120311031403130316031503180317031A0319031C031B031E031D0320031FFFFFFFFFFFFFFFFFFFFF0321FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03230322032503240327032603290328032B032A032D032C032F032E0331033003330332033503340337033603390338
033B033AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
033D033C033F033E0341034003430342034503440347034603490348034B034A034D034C034F034E0351035003530352
035503540357035603590358035B035A035D035C035F035E0361036003630362036503640367036603690368036B036A
036CFFFFFFFFFFFF036DFFFFFFFF036EFFFF036FFFFF0370FFFF0371FFFFFFFFFFFFFFFF0372FFFFFFFFFFFFFFFF0373
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0374FFFF0375FFFF0376FFFF0377FFFF0378FFFF0379FFFF037AFFFF037BFFFF
037CFFFF037DFFFF037EFFFF037FFFFF0380FFFF0381FFFF0382FFFF0383FFFF0384FFFF0385FFFF0386FFFF0387FFFF
0388FFFF0389FFFF038AFFFF038BFFFF038CFFFF038DFFFF038EFFFF038FFFFF0390FFFF0391FFFF0392FFFF0393FFFF
0394FFFF0395FFFF0396FFFF0397FFFF0398FFFF0399FFFF039AFFFF039BFFFF039CFFFF039DFFFF039EFFFF039FFFFF
03A0FFFF03A1FFFF03A2FFFF03A3FFFF03A4FFFF03A5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A6FFFF03A7
FFFFFFFF03A8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03AA03A903AC03AB03AE03AD03B003AF
03B203B103B403B303B603B503B803B703BA03B903BC03BB03BE03BD03C003BF03C203C103C403C303C603C503C803C7
03CA03C903CC03CB03CE03CD03CFFFFFFFFFFFFFFFFFFFFF03D0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03D1FFFF03D2FFFF03D3FFFF03D4FFFF03D5FFFF03D6FFFF03D7FFFF03D8FFFF
03D9FFFF03DAFFFF03DBFFFF03DCFFFF03DDFFFF03DEFFFF03DFFFFF03E0FFFF03E1FFFF03E2FFFF03E3FFFF03E4FFFF
03E5FFFF03E6FFFF03E7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03E8FFFF03E9FFFF03EAFFFF03EBFFFF03ECFFFF03EDFFFF03EEFFFF03EFFFFF03F0FFFF03F1FFFF03F2FFFF03F3FFFF
03F4FFFF03F5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF03F6FFFF03F7FFFF03F8FFFF03F9FFFF03FAFFFF03FBFFFF03FCFFFFFFFFFFFF03FDFFFF03FEFFFF03FFFFFF
0400FFFF0401FFFF0402FFFF0403FFFF0404FFFF0405FFFF0406FFFF0407FFFF0408FFFF0409FFFF040AFFFF040BFFFF
040CFFFF040DFFFF040EFFFF040FFFFF0410FFFF0411FFFF0412FFFF0413FFFF0414FFFF0415FFFF0416FFFF0417FFFF
0418FFFF0419FFFF041AFFFF041BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF041CFFFF041D041EFFFF
041FFFFF0420FFFF0421FFFF0422FFFFFFFFFFFFFFFFFFFFFFFF0423FFFFFFFF0424FFFF0425FFFFFFFF04260427FFFF
0428FFFF0429FFFF042AFFFF042BFFFF042CFFFF042DFFFF042EFFFF042FFFFF0430FFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF0431FFFF0432FFFF0433FFFF0434FFFF0435FFFF0436FFFF0437FFFF0438FFFFFFFFFFFFFFFFFFFF
FFFF0439FFFF043AFFFFFFFFFFFFFFFF043BFFFFFFFFFFFFFFFFFFFF043CFFFF043DFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043E
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF043FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0441044004430442044504440447044604490448044B044A044D044C044F044E
0451045004530452045504540457045604590458045B045A045D045C045F045E04610460046304620465046404670466
04690468046B046A046D046C046F046E0471047004730472047504740477047604790478047B047A047D047C047F047E
0481048004830482048504840487048604890488048B048A048D048C048F048E05B305B105B705B505BD05BAFFFF05BF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05C3FFFF05C705C505CB05C9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0490FFFF04920491049404930496049504980497049A0499049C049B049E049D
04A0049F04A204A104A404A304A604A504A804A7FFFF04A9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04AB04AA04AD04AC04AF04AE04B104B0
04B304B204B504B404B704B604B904B804BB04BA04BD04BC04BF04BE04C104C004C304C204C504C404C704C604C904C8
04CB04CA04CD04CC04CF04CE04D104D0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04D304D204D504D404D704D604D904D804DB04DA04DD04DC04DF04DE04E104E0
04E304E204E504E404E704E604E904E804EB04EA04ED04EC04EF04EE04F104F004F304F204F504F4FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04F6FFFF
04F804F704FA04F904FC04FB04FE04FD050004FF0501FFFF05030502050505040507050605090508050B050A050D050C
050F050E0510FFFF0512051105140513051605150517FFFFFFFF0518FFFFFFFF051A0519051C051B051E051D0520051F
05220521052405230526052505280527052A0529052C052B052E052D0530052F05320531053405330536053505380537
053A0539053C053B053E053D0540053F05420541054405430546054505480547054A0549FFFF054BFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF054D054C054F054E0551055005530552055505540557055605590558055B055A
055D055C055F055E0561056005630562056505640567056605690568056B056AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF056D056C056F056E0571057005730572057505740577057605790578057B057A
057D057C057F057E0581058005830582058505840587058605890588058B058AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF058D058C058F058E0591059005930592059505940597059605990598059B059A059D059C059F059E05A105A0
05A305A205A505A405A705A605A905A805AB05AA05AD05ACFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//10368 bytes
enum toUpperSimpleIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000004000000280",
cast(immutable size_t[]) x"
000001000000048000000F40",
cast(immutable size_t[]) x"
020201000402030206020205070202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
0202020202020202020202020202020200010000000300020005000400070006000900080000000A000B0000000D000C
000F000E0011001000130012000000140000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000015000000000000000000000000000000000000000000000016000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000170018000000000019001B001A001D001C001F001E0021002000000000000000000022000000000023
000000000000000000000000000000000000000000240000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000260025002800270000002900000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000002A00000000002B002D002C002F002E00000000000000000000000000000000
000000000000000000300000000000310000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000032000000000000
000000000000000000000000000000000000000000000000000000000000000000340033003500000000000000000036
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000037000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000038000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000390000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000003B003A000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFF00020001000400030006000500080007000A0009000C000B000E000D
0010000F00120011001400130016001500180017FFFF0019FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001AFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001C001B001E001D0020001F00220021
002400230026002500280027002A0029002C002B002E002D0030002FFFFF003100330032003500340037003600390038
003AFFFF003BFFFF003CFFFF003DFFFF003EFFFF003FFFFF0040FFFF0041FFFF0042FFFF0043FFFF0044FFFF0045FFFF
0046FFFF0047FFFF0048FFFF0049FFFF004AFFFF004BFFFF004CFFFF004DFFFF004EFFFF004FFFFF0050FFFF0051FFFF
0052FFFF0053FFFF0054FFFF0055FFFFFFFFFFFFFFFF0056FFFF0057FFFF0058FFFF0059FFFF005AFFFF005BFFFF005C
FFFF005D005EFFFF005FFFFF0060FFFF0061FFFF0062FFFF0063FFFF0064FFFF0065FFFF0066FFFF0067FFFF0068FFFF
0069FFFF006AFFFF006BFFFF006CFFFF006DFFFF006EFFFF006FFFFF0070FFFF0071FFFF0072FFFF0073FFFF0074FFFF
FFFFFFFFFFFF0075FFFF007600780077FFFF0079007AFFFF007BFFFFFFFFFFFFFFFF007CFFFFFFFFFFFF007DFFFFFFFF
FFFFFFFFFFFF007E007FFFFFFFFFFFFF0080FFFFFFFF0081FFFFFFFFFFFF00820083FFFF0084FFFF0085FFFFFFFFFFFF
FFFF0086FFFFFFFF0087FFFFFFFFFFFFFFFF0088FFFFFFFFFFFF0089FFFF008A008BFFFFFFFFFFFF008CFFFF008DFFFF
FFFFFFFFFFFFFFFF008EFFFFFFFF008F009100900092FFFFFFFF0093FFFF0094FFFF0095FFFF0096FFFF0097FFFF0098
FFFF0099FFFF009A009C009B009DFFFF009EFFFF009FFFFF00A0FFFF00A1FFFF00A2FFFF00A3FFFF00A4FFFF00A5FFFF
FFFFFFFF00A700A600A8FFFFFFFFFFFF00A9FFFF00AAFFFF00ABFFFF00ACFFFF00ADFFFF00AEFFFF00AFFFFF00B0FFFF
00B1FFFF00B2FFFF00B3FFFF00B4FFFF00B5FFFF00B6FFFF00B7FFFF00B8FFFF00B9FFFF00BAFFFF00BBFFFF00BCFFFF
FFFFFFFF00BDFFFF00BEFFFF00BFFFFF00C0FFFF00C1FFFF00C2FFFF00C3FFFF00C4FFFF00C5FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF00C600C7FFFFFFFF00C8FFFF00C9FFFFFFFF00CAFFFF00CBFFFF00CCFFFF00CDFFFF00CEFFFF
00D000CF00D200D1FFFF00D300D500D400D6FFFF00D7FFFFFFFF00D8FFFFFFFF00DA00D900DBFFFF00DCFFFFFFFF00DD
00DF00DE00E100E0FFFF00E200E3FFFF00E4FFFFFFFF00E500E6FFFFFFFFFFFFFFFFFFFFFFFFFFFF00E7FFFFFFFFFFFF
FFFF00E800EA00E9FFFFFFFF00EBFFFF00ED00EC00EF00EEFFFF00F0FFFFFFFFFFFFFFFFFFFF00F1FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00F2FFFFFFFF00F3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F4FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F5FFFF00F6FFFFFFFFFFFF00F7FFFF
FFFFFFFF00F8FFFF00FA00F9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00FC00FB00FE00FD00FFFFFF0101010001030102010501040107010601090108010B010A010D010C
010F010E0111011001130112011501140117011601190118011B011AFFFF011C011E011DFFFFFFFF011FFFFF01210120
0122FFFF0123FFFF0124FFFF0125FFFF0126FFFF0127FFFF0128FFFF0129FFFF012AFFFF012BFFFF012CFFFF012DFFFF
012F012E013101300132FFFFFFFFFFFFFFFF01330134FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0136013501380137013A0139013C013B
013E013D0140013F01420141014401430146014501480147014A0149014C014B014E014D0150014F0152015101540153
0156015501580157015A0159015C015B015E015D0160015F01620161016401630165FFFF0166FFFF0167FFFF0168FFFF
0169FFFF016AFFFF016BFFFF016CFFFF016DFFFF016EFFFF016FFFFF0170FFFF0171FFFF0172FFFF0173FFFF0174FFFF
0175FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0176FFFF0177FFFF0178FFFF0179FFFF017AFFFF017BFFFF017CFFFF
017DFFFF017EFFFF017FFFFF0180FFFF0181FFFF0182FFFF0183FFFF0184FFFF0185FFFF0186FFFF0187FFFF0188FFFF
0189FFFF018AFFFF018BFFFF018CFFFF018DFFFF018EFFFF018FFFFF0190FFFFFFFFFFFFFFFF0191FFFF0192FFFF0193
FFFF0194FFFF0195FFFF0196019801970199FFFF019AFFFF019BFFFF019CFFFF019DFFFF019EFFFF019FFFFF01A0FFFF
01A1FFFF01A2FFFF01A3FFFF01A4FFFF01A5FFFF01A6FFFF01A7FFFF01A8FFFF01A9FFFF01AAFFFF01ABFFFF01ACFFFF
01ADFFFF01AEFFFF01AFFFFF01B0FFFF01B1FFFF01B2FFFF01B3FFFF01B4FFFF01B5FFFF01B6FFFF01B7FFFF01B8FFFF
01B9FFFF01BAFFFF01BBFFFF01BCFFFF01BDFFFF01BEFFFF01BFFFFF01C0FFFF01C1FFFF01C2FFFF01C3FFFF01C4FFFF
01C5FFFF01C6FFFF01C7FFFF01C8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01C9FFFF01CB01CA01CD01CC01CF01CE01D101D001D301D201D501D401D701D6
01D901D801DB01DA01DD01DC01DF01DE01E101E001E301E201E501E401E701E601E901E801EB01EA01ED01ECFFFF01EE
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01F001EF01F201F101F401F301F601F501F801F701FA01F901FC01FB01FE01FD020001FF020202010204020302060205
02080207020A0209020C020B020E020D0210020F02120211021402130216021502180217FFFF0219021AFFFF021C021B
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF021E021D0220021F02220221FFFFFFFF022402230226022502280227022A0229
FFFF022BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022CFFFFFFFFFFFF022DFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022EFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022FFFFF0230FFFF0231FFFF0232FFFF
0233FFFF0234FFFF0235FFFF0236FFFF0237FFFF0238FFFF0239FFFF023AFFFF023BFFFF023CFFFF023DFFFF023EFFFF
023FFFFF0240FFFF0241FFFF0242FFFF0243FFFF0244FFFF0245FFFF0246FFFF0247FFFF0248FFFF0249FFFF024AFFFF
024BFFFF024CFFFF024DFFFF024EFFFF024FFFFF0250FFFF0251FFFF0252FFFF0253FFFF0254FFFF0255FFFF0256FFFF
0257FFFF0258FFFF0259FFFF025AFFFF025BFFFF025CFFFF025DFFFF025EFFFF025FFFFF0260FFFF0261FFFF0262FFFF
0263FFFF0264FFFF0265FFFF0266FFFF0267FFFF0268FFFF0269FFFF026AFFFF026BFFFF026CFFFF026DFFFF026EFFFF
026FFFFF0270FFFF0271FFFF0272FFFF0273FFFF0274FFFF0275FFFF0276FFFF0277FFFF0278FFFF0279FFFFFFFFFFFF
FFFFFFFF027AFFFFFFFFFFFFFFFFFFFF027BFFFF027CFFFF027DFFFF027EFFFF027FFFFF0280FFFF0281FFFF0282FFFF
0283FFFF0284FFFF0285FFFF0286FFFF0287FFFF0288FFFF0289FFFF028AFFFF028BFFFF028CFFFF028DFFFF028EFFFF
028FFFFF0290FFFF0291FFFF0292FFFF0293FFFF0294FFFF0295FFFF0296FFFF0297FFFF0298FFFF0299FFFF029AFFFF
029BFFFF029CFFFF029DFFFF029EFFFF029FFFFF02A0FFFF02A1FFFF02A2FFFF02A3FFFF02A4FFFF02A5FFFF02A6FFFF
02A7FFFF02A8FFFF02A9FFFF02AAFFFF02AC02AB02AE02AD02B002AF02B202B1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02B402B302B602B502B802B7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02BA02B902BC02BB02BE02BD02C002BF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02C202C102C402C302C602C502C802C7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02CA02C902CC02CB02CE02CDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02CFFFFF02D0FFFF02D1FFFF02D2FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D402D302D602D502D802D702DA02D9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02DC02DB02DE02DD02E002DF02E202E102E402E302E602E502E802E7FFFFFFFF02EA02E902EC02EB02EE02ED02F002EF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02F202F102F402F302F602F502F802F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02FA02F902FC02FB02FE02FD030002FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030203010303FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFF0304FFFFFFFF0305FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03070306FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03090308FFFFFFFF030AFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030CFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
030E030D0310030F03120311031403130316031503180317031A0319031C031BFFFFFFFFFFFFFFFFFFFF031DFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
031F031E0321032003230322032503240327032603290328032B032A032D032C032F032E033103300333033203350334
03370336FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03390338033B033A033D033C033F033E0341034003430342034503440347034603490348034B034A034D034C034F034E
0351035003530352035503540357035603590358035B035A035D035C035F035E03610360036303620365036403670366
0368FFFFFFFFFFFF0369FFFFFFFF036AFFFF036BFFFF036CFFFF036DFFFFFFFFFFFFFFFF036EFFFFFFFFFFFFFFFF036F
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0370FFFF0371FFFF0372FFFF0373FFFF0374FFFF0375FFFF0376FFFF0377FFFF
0378FFFF0379FFFF037AFFFF037BFFFF037CFFFF037DFFFF037EFFFF037FFFFF0380FFFF0381FFFF0382FFFF0383FFFF
0384FFFF0385FFFF0386FFFF0387FFFF0388FFFF0389FFFF038AFFFF038BFFFF038CFFFF038DFFFF038EFFFF038FFFFF
0390FFFF0391FFFF0392FFFF0393FFFF0394FFFF0395FFFF0396FFFF0397FFFF0398FFFF0399FFFF039AFFFF039BFFFF
039CFFFF039DFFFF039EFFFF039FFFFF03A0FFFF03A1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A2FFFF03A3
FFFFFFFF03A4FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A603A503A803A703AA03A903AC03AB
03AE03AD03B003AF03B203B103B403B303B603B503B803B703BA03B903BC03BB03BE03BD03C003BF03C203C103C403C3
03C603C503C803C703CA03C903CBFFFFFFFFFFFFFFFFFFFF03CCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03CDFFFF03CEFFFF03CFFFFF03D0FFFF03D1FFFF03D2FFFF03D3FFFF03D4FFFF
03D5FFFF03D6FFFF03D7FFFF03D8FFFF03D9FFFF03DAFFFF03DBFFFF03DCFFFF03DDFFFF03DEFFFF03DFFFFF03E0FFFF
03E1FFFF03E2FFFF03E3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03E4FFFF03E5FFFF03E6FFFF03E7FFFF03E8FFFF03E9FFFF03EAFFFF03EBFFFF03ECFFFF03EDFFFF03EEFFFF03EFFFFF
03F0FFFF03F1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF03F2FFFF03F3FFFF03F4FFFF03F5FFFF03F6FFFF03F7FFFF03F8FFFFFFFFFFFF03F9FFFF03FAFFFF03FBFFFF
03FCFFFF03FDFFFF03FEFFFF03FFFFFF0400FFFF0401FFFF0402FFFF0403FFFF0404FFFF0405FFFF0406FFFF0407FFFF
0408FFFF0409FFFF040AFFFF040BFFFF040CFFFF040DFFFF040EFFFF040FFFFF0410FFFF0411FFFF0412FFFF0413FFFF
0414FFFF0415FFFF0416FFFF0417FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0418FFFF0419041AFFFF
041BFFFF041CFFFF041DFFFF041EFFFFFFFFFFFFFFFFFFFFFFFF041FFFFFFFFF0420FFFF0421FFFFFFFF04220423FFFF
0424FFFF0425FFFF0426FFFF0427FFFF0428FFFF0429FFFF042AFFFF042BFFFF042CFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF042DFFFF042EFFFF042FFFFF0430FFFF0431FFFF0432FFFF0433FFFF0434FFFFFFFFFFFFFFFFFFFF
FFFF0435FFFF0436FFFFFFFFFFFFFFFF0437FFFFFFFFFFFFFFFFFFFF0438FFFF0439FFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043A
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF043BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043D043C043F043E0441044004430442044504440447044604490448044B044A
044D044C044F044E0451045004530452045504540457045604590458045B045A045D045C045F045E0461046004630462
046504640467046604690468046B046A046D046C046F046E0471047004730472047504740477047604790478047B047A
047D047C047F047E0481048004830482048504840487048604890488048B048A048CFFFF048E048D0490048F04920491
049404930496049504980497049A0499049C049B049E049D04A0049F04A204A104A404A3FFFF04A5FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
04A704A604A904A804AB04AA04AD04AC04AF04AE04B104B004B304B204B504B404B704B604B904B804BB04BA04BD04BC
04BF04BE04C104C004C304C204C504C404C704C604C904C804CB04CA04CD04CCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04CF04CE04D104D004D304D204D504D4
04D704D604D904D804DB04DA04DD04DC04DF04DE04E104E004E304E204E504E404E704E604E904E804EB04EA04ED04EC
04EF04EE04F104F0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF04F2FFFF04F404F304F604F504F804F704FA04F904FC04FB04FDFFFF04FF04FE05010500
05030502050505040507050605090508050B050A050CFFFF050E050D0510050F051205110513FFFFFFFF0514FFFFFFFF
0516051505180517051A0519051C051B051E051D0520051F05220521052405230526052505280527052A0529052C052B
052E052D0530052F05320531053405330536053505380537053A0539053C053B053E053D0540053F0542054105440543
05460545FFFF0547FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05490548054B054A054D054C054F054E
0551055005530552055505540557055605590558055B055A055D055C055F055E05610560056305620565056405670566
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05690568056B056A056D056C056F056E
0571057005730572057505740577057605790578057B057A057D057C057F057E05810580058305820585058405870586
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05890588058B058A058D058C058F058E059105900593059205950594
0597059605990598059B059A059D059C059F059E05A105A005A305A205A505A405A705A605A905A8FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//9856 bytes
enum toLowerSimpleIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000004000000280",
cast(immutable size_t[]) x"
000001000000048000000E40",
cast(immutable size_t[]) x"
020201000402030206020205070202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
0202020202020202020202020202020200010000000200000004000300060005000800070000000000090000000B000A
000D000C000F000E00110010000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000013001200000000000000000000000000000000000000000015001400000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000016000000000000000000180017001A0019001C001B001E001D00000000000000000020001F00000021
000000000000000000000000000000000000000000230022000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000250024002700260000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000028000000000029002B002A002D002C00000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000002E00000000
00000000000000000000000000000000000000000000000000000000000000000000002F003100300032000000000033
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000003400000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000003500000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000360000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000037000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0000FFFF00020001000400030006000500080007000A0009000C000B000E000D0010000F001200110014001300160015
00180017FFFF0019FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001B001A001D001C001F001E00210020
00230022002500240027002600290028002B002A002D002C002F002EFFFF0030003200310034003300360035FFFF0037
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0038FFFF0039FFFF003AFFFF003BFFFF003CFFFF003DFFFF003EFFFF003F
FFFF0040FFFF0041FFFF0042FFFF0043FFFF0044FFFF0045FFFF0046FFFF0047FFFF0048FFFF0049FFFF004AFFFF004B
FFFF004CFFFF004DFFFF004EFFFF004FFFFF0050FFFF0051FFFF0052FFFF00530054FFFF0055FFFF0056FFFF0057FFFF
0058FFFF0059FFFF005AFFFF005BFFFFFFFFFFFFFFFF005CFFFF005DFFFF005EFFFF005FFFFF0060FFFF0061FFFF0062
FFFF0063FFFF0064FFFF0065FFFF0066FFFF0067FFFF0068FFFF0069FFFF006AFFFF006BFFFF006CFFFF006DFFFF006E
FFFF006FFFFF0070FFFF0071FFFF0072007400730075FFFF0076FFFFFFFFFFFF0077FFFFFFFF0078FFFF0079007B007A
007CFFFF007E007DFFFFFFFF0080007F008200810083FFFFFFFF008400860085FFFF0087FFFFFFFF00890088008AFFFF
FFFF008BFFFF008CFFFF008D008F008E0090FFFFFFFFFFFFFFFF0091009300920094FFFF009600950097FFFF0098FFFF
FFFF0099FFFFFFFFFFFF009AFFFFFFFFFFFFFFFFFFFFFFFF009C009B009DFFFFFFFF009E00A0009F00A1FFFF00A2FFFF
00A3FFFF00A4FFFF00A5FFFF00A6FFFF00A7FFFF00A8FFFFFFFFFFFFFFFF00A9FFFF00AAFFFF00ABFFFF00ACFFFF00AD
FFFF00AEFFFF00AFFFFF00B0FFFF00B100B2FFFFFFFF00B3FFFF00B400B600B5FFFF00B7FFFF00B8FFFF00B9FFFF00BA
FFFF00BBFFFF00BCFFFF00BDFFFF00BEFFFF00BFFFFF00C0FFFF00C1FFFF00C2FFFF00C3FFFF00C4FFFF00C5FFFF00C6
FFFF00C7FFFF00C8FFFF00C9FFFF00CAFFFF00CBFFFF00CCFFFF00CDFFFF00CEFFFF00CFFFFF00D0FFFF00D1FFFF00D2
FFFF00D3FFFF00D4FFFFFFFFFFFFFFFFFFFFFFFF00D600D500D7FFFFFFFF00D800D9FFFF00DAFFFF00DC00DBFFFF00DD
FFFF00DEFFFF00DFFFFF00E0FFFF00E1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00E2FFFF00E3FFFFFFFFFFFF00E4FFFFFFFFFFFFFFFFFFFFFFFF00E5FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFF00E600E800E7FFFF00E9FFFF00EA00EC00EB00EDFFFF00EF00EE00F100F000F300F2
00F500F400F700F600F900F800FB00FA00FD00FC00FEFFFF010000FF010201010104010301060105FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF0107FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0108FFFF0109FFFF010AFFFF010B
FFFF010CFFFF010DFFFF010EFFFF010FFFFF0110FFFF0111FFFF0112FFFF0113FFFFFFFFFFFFFFFFFFFF01140115FFFF
0116FFFFFFFF01170118FFFF011A0119011C011B011E011D0120011F01220121012401230126012501280127012A0129
012C012B012E012D0130012F01320131013401330136013501380137013A0139013C013B013E013D0140013F01420141
014401430146014501480147014A0149FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF014BFFFF014CFFFF014DFFFF014EFFFF014FFFFF0150FFFF0151FFFF0152
FFFF0153FFFF0154FFFF0155FFFF0156FFFF0157FFFF0158FFFF0159FFFF015AFFFF015BFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF015CFFFF015DFFFF015EFFFF015FFFFF0160FFFF0161FFFF0162FFFF0163FFFF0164FFFF0165FFFF0166
FFFF0167FFFF0168FFFF0169FFFF016AFFFF016BFFFF016CFFFF016DFFFF016EFFFF016FFFFF0170FFFF0171FFFF0172
FFFF0173FFFF0174FFFF0175FFFF0176017801770179FFFF017AFFFF017BFFFF017CFFFF017DFFFF017EFFFFFFFFFFFF
FFFF017FFFFF0180FFFF0181FFFF0182FFFF0183FFFF0184FFFF0185FFFF0186FFFF0187FFFF0188FFFF0189FFFF018A
FFFF018BFFFF018CFFFF018DFFFF018EFFFF018FFFFF0190FFFF0191FFFF0192FFFF0193FFFF0194FFFF0195FFFF0196
FFFF0197FFFF0198FFFF0199FFFF019AFFFF019BFFFF019CFFFF019DFFFF019EFFFF019FFFFF01A0FFFF01A1FFFF01A2
FFFF01A3FFFF01A4FFFF01A5FFFF01A6FFFF01A7FFFF01A8FFFF01A9FFFF01AAFFFF01ABFFFF01ACFFFF01ADFFFF01AE
01AFFFFF01B101B001B301B201B501B401B701B601B901B801BB01BA01BD01BC01BF01BE01C101C001C301C201C501C4
01C701C601C901C801CB01CA01CD01CC01CF01CE01D101D001D301D2FFFF01D4FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01D601D501D801D701DA01D901DC01DB
01DE01DD01E001DF01E201E101E401E301E601E501E801E701EA01E901EC01EB01EE01ED01F001EF01F201F101F401F3
01F601F501F801F701FA01F901FBFFFFFFFFFFFFFFFFFFFF01FCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01FE01FD020001FF02020201020402030206020502080207020A0209020C020B020E020D0210020F0212021102140213
0216021502180217021A0219021C021B021E021D0220021F02220221022402230226022502280227022A0229022C022B
022E022D0230022F02320231023402330236023502380237023A0239023C023B023E023D0240023F0242024102440243
0246024502480247024A0249024C024B024E024D0250024F02520251FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF025402530256025502580257025A0259
025C025B025E025D0260025F02620261026402630266026502680267026A0269026C026B026E026D0270026F02720271
027402730276027502780277027A0279027C027BFFFF027D027EFFFF0280027FFFFF0281FFFF0282FFFF0283FFFF0284
FFFF0285FFFF0286FFFF0287FFFF0288FFFF0289FFFF028AFFFF028BFFFF028CFFFF028DFFFF028EFFFF028FFFFF0290
FFFF0291FFFF0292FFFF0293FFFF0294FFFF0295FFFF0296FFFF0297FFFF0298FFFF0299FFFF029AFFFF029BFFFF029C
FFFF029DFFFF029EFFFF029FFFFF02A0FFFF02A1FFFF02A2FFFF02A3FFFF02A4FFFF02A5FFFF02A6FFFF02A7FFFF02A8
FFFF02A9FFFF02AAFFFF02ABFFFF02ACFFFF02ADFFFF02AEFFFF02AFFFFF02B0FFFF02B1FFFF02B2FFFF02B3FFFF02B4
FFFF02B5FFFF02B6FFFF02B7FFFF02B8FFFF02B9FFFF02BAFFFF02BBFFFF02BCFFFF02BDFFFF02BEFFFF02BFFFFF02C0
FFFF02C1FFFF02C2FFFF02C3FFFF02C4FFFF02C5FFFF02C6FFFF02C7FFFF02C8FFFF02C9FFFF02CAFFFF02CBFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFF02CCFFFF02CDFFFF02CEFFFF02CFFFFF02D0FFFF02D1FFFF02D2FFFF02D3FFFF02D4
FFFF02D5FFFF02D6FFFF02D7FFFF02D8FFFF02D9FFFF02DAFFFF02DBFFFF02DCFFFF02DDFFFF02DEFFFF02DFFFFF02E0
FFFF02E1FFFF02E2FFFF02E3FFFF02E4FFFF02E5FFFF02E6FFFF02E7FFFF02E8FFFF02E9FFFF02EAFFFF02EBFFFF02EC
FFFF02EDFFFF02EEFFFF02EFFFFF02F0FFFF02F1FFFF02F2FFFF02F3FFFF02F4FFFF02F5FFFF02F6FFFF02F7FFFF02F8
FFFF02F9FFFF02FAFFFF02FBFFFF02FCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02FE02FD030002FF0302030103040303
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0306030503080307030A0309FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
030C030B030E030D0310030F03120311FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF031403130316031503180317031A0319
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF031C031B031E031D0320031FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0321FFFF0322FFFF0323FFFF0324FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0326032503280327032A0329032C032B
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
032E032D0330032F0332033103340333FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0336033503380337033A0339033C033B
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF033E033D0340033F0342034103440343FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0346034503480347FFFF0349FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF034B034A034D034CFFFF034EFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0350034F03520351FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0354035303560355FFFF0357FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03590358035B035AFFFF035CFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF035DFFFFFFFF035F035EFFFFFFFFFFFFFFFF
FFFFFFFFFFFF0360FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03620361036403630366036503680367036A0369036C036B036E036D0370036FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0371FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF03730372037503740377037603790378037B037A037D037C037F037E0381038003830382
038503840387038603890388038B038AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF038D038C038F038E0391039003930392039503940397039603990398039B039A
039D039C039F039E03A103A003A303A203A503A403A703A603A903A803AB03AA03AD03AC03AF03AE03B103B003B303B2
03B503B403B703B603B903B803BB03BAFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03BC03BE03BDFFFF03BF03C0FFFF03C1FFFF03C2FFFF03C3FFFF03C503C4
FFFF03C6FFFF03C703C8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03CA03C9FFFF03CBFFFF03CCFFFF03CDFFFF03CE
FFFF03CFFFFF03D0FFFF03D1FFFF03D2FFFF03D3FFFF03D4FFFF03D5FFFF03D6FFFF03D7FFFF03D8FFFF03D9FFFF03DA
FFFF03DBFFFF03DCFFFF03DDFFFF03DEFFFF03DFFFFF03E0FFFF03E1FFFF03E2FFFF03E3FFFF03E4FFFF03E5FFFF03E6
FFFF03E7FFFF03E8FFFF03E9FFFF03EAFFFF03EBFFFF03ECFFFF03EDFFFF03EEFFFF03EFFFFF03F0FFFF03F1FFFF03F2
FFFF03F3FFFF03F4FFFF03F5FFFF03F6FFFF03F7FFFF03F8FFFF03F9FFFF03FAFFFF03FBFFFF03FCFFFFFFFFFFFFFFFF
FFFFFFFF03FDFFFF03FEFFFFFFFFFFFFFFFFFFFFFFFF03FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF0400FFFF0401FFFF0402FFFF0403FFFF0404FFFF0405FFFF0406FFFF0407FFFF0408FFFF0409FFFF040AFFFF040B
FFFF040CFFFF040DFFFF040EFFFF040FFFFF0410FFFF0411FFFF0412FFFF0413FFFF0414FFFF0415FFFF0416FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0417FFFF0418FFFF0419FFFF041A
FFFF041BFFFF041CFFFF041DFFFF041EFFFF041FFFFF0420FFFF0421FFFF0422FFFF0423FFFF0424FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0425FFFF0426FFFF0427
FFFF0428FFFF0429FFFF042AFFFF042BFFFFFFFFFFFF042CFFFF042DFFFF042EFFFF042FFFFF0430FFFF0431FFFF0432
FFFF0433FFFF0434FFFF0435FFFF0436FFFF0437FFFF0438FFFF0439FFFF043AFFFF043BFFFF043CFFFF043DFFFF043E
FFFF043FFFFF0440FFFF0441FFFF0442FFFF0443FFFF0444FFFF0445FFFF0446FFFF0447FFFF0448FFFF0449FFFF044A
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF044BFFFF044CFFFF044DFFFFFFFF044EFFFF044FFFFF0450FFFF0451FFFF0452
FFFFFFFF0453FFFF0454FFFFFFFFFFFFFFFF0455FFFF0456FFFFFFFFFFFF0457FFFF0458FFFF0459FFFF045AFFFF045B
FFFF045CFFFF045DFFFF045EFFFF045FFFFF04600462046104640463FFFF04650467046604690468FFFF046AFFFF046B
FFFF046CFFFF046DFFFF046EFFFF046FFFFF0470FFFF047104730472047504740476FFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF0477FFFFFFFFFFFFFFFFFFFF0478FFFF0479FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF047AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF047BFFFF047D047C047F047E0481048004830482048504840487048604890488
048B048A048D048C048F048E0491049004930492FFFF0494FFFFFFFFFFFFFFFF0496049504980497049A0499049C049B
049E049D04A0049F04A204A104A404A304A604A504A804A704AA04A904AC04AB04AE04AD04B004AF04B204B104B404B3
04B604B504B804B704BA04B904BC04BBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04BE04BD04C004BF04C204C104C404C304C604C504C804C704CA04C904CC04CB
04CE04CD04D004CF04D204D104D404D304D604D504D804D704DA04D904DC04DB04DE04DD04E004DFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04E204E104E404E304E604E504E804E7
04EA04E9FFFF04EB04ED04EC04EF04EE04F104F004F304F204F504F404F704F604F904F8FFFF04FA04FC04FB04FE04FD
050004FFFFFF050105030502FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
050505040507050605090508050B050A050D050C050F050E0511051005130512051505140517051605190518051B051A
051D051C051F051E0521052005230522052505240527052605290528052B052A052D052C052F052E0531053005330532
05350534FFFF0536FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
05380537053A0539053C053B053E053D0540053F05420541054405430546054505480547054A0549054C054B054E054D
0550054F05520551055405530556055505580557055A0559055C055B055E055D0560055F056205610564056305660565
05680567056A0569056C056B056E056D0570056F057205710574057305760575FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
05780577057A0579057C057B057E057D0580057F05820581058405830586058505880587058A0589058C058B058E058D
0590058F05920591059405930596059505980597FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//10368 bytes
enum toTitleSimpleIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000004000000280",
cast(immutable size_t[]) x"
000001000000048000000F40",
cast(immutable size_t[]) x"
020201000402030206020205070202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
0202020202020202020202020202020200010000000300020005000400070006000900080000000A000B0000000D000C
000F000E0011001000130012000000140000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000015000000000000000000000000000000000000000000000016000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000170018000000000019001B001A001D001C001F001E0021002000000000000000000022000000000023
000000000000000000000000000000000000000000240000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000260025002800270000002900000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000002A00000000002B002D002C002F002E00000000000000000000000000000000
000000000000000000300000000000310000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000032000000000000
000000000000000000000000000000000000000000000000000000000000000000340033003500000000000000000036
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000037000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000038000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000390000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000003B003A000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFF00020001000400030006000500080007000A0009000C000B000E000D
0010000F00120011001400130016001500180017FFFF0019FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001AFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001C001B001E001D0020001F00220021
002400230026002500280027002A0029002C002B002E002D0030002FFFFF003100330032003500340037003600390038
003AFFFF003BFFFF003CFFFF003DFFFF003EFFFF003FFFFF0040FFFF0041FFFF0042FFFF0043FFFF0044FFFF0045FFFF
0046FFFF0047FFFF0048FFFF0049FFFF004AFFFF004BFFFF004CFFFF004DFFFF004EFFFF004FFFFF0050FFFF0051FFFF
0052FFFF0053FFFF0054FFFF0055FFFFFFFFFFFFFFFF0056FFFF0057FFFF0058FFFF0059FFFF005AFFFF005BFFFF005C
FFFF005D005EFFFF005FFFFF0060FFFF0061FFFF0062FFFF0063FFFF0064FFFF0065FFFF0066FFFF0067FFFF0068FFFF
0069FFFF006AFFFF006BFFFF006CFFFF006DFFFF006EFFFF006FFFFF0070FFFF0071FFFF0072FFFF0073FFFF0074FFFF
FFFFFFFFFFFF0075FFFF007600780077FFFF0079007AFFFF007BFFFFFFFFFFFFFFFF007CFFFFFFFFFFFF007DFFFFFFFF
FFFFFFFFFFFF007E007FFFFFFFFFFFFF0080FFFFFFFF0081FFFFFFFFFFFF00820083FFFF0084FFFF0085FFFFFFFFFFFF
FFFF0086FFFFFFFF0087FFFFFFFFFFFFFFFF0088FFFFFFFFFFFF0089FFFF008A008BFFFFFFFFFFFF008CFFFF008DFFFF
FFFFFFFFFFFFFFFF008F008E009100900093009200950094FFFF0096FFFF0097FFFF0098FFFF0099FFFF009AFFFF009B
FFFF009CFFFF009D009F009E00A0FFFF00A1FFFF00A2FFFF00A3FFFF00A4FFFF00A5FFFF00A6FFFF00A7FFFF00A8FFFF
00A9FFFF00AB00AA00ACFFFFFFFFFFFF00ADFFFF00AEFFFF00AFFFFF00B0FFFF00B1FFFF00B2FFFF00B3FFFF00B4FFFF
00B5FFFF00B6FFFF00B7FFFF00B8FFFF00B9FFFF00BAFFFF00BBFFFF00BCFFFF00BDFFFF00BEFFFF00BFFFFF00C0FFFF
FFFFFFFF00C1FFFF00C2FFFF00C3FFFF00C4FFFF00C5FFFF00C6FFFF00C7FFFF00C8FFFF00C9FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF00CA00CBFFFFFFFF00CCFFFF00CDFFFFFFFF00CEFFFF00CFFFFF00D0FFFF00D1FFFF00D2FFFF
00D400D300D600D5FFFF00D700D900D800DAFFFF00DBFFFFFFFF00DCFFFFFFFF00DE00DD00DFFFFF00E0FFFFFFFF00E1
00E300E200E500E4FFFF00E600E7FFFF00E8FFFFFFFF00E900EAFFFFFFFFFFFFFFFFFFFFFFFFFFFF00EBFFFFFFFFFFFF
FFFF00EC00EE00EDFFFFFFFF00EFFFFF00F100F000F300F2FFFF00F4FFFFFFFFFFFFFFFFFFFF00F5FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00F6FFFFFFFF00F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F8FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F9FFFF00FAFFFFFFFFFFFF00FBFFFF
FFFFFFFF00FCFFFF00FE00FDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF010000FF010201010103FFFF010501040107010601090108010B010A010D010C010F010E01110110
01130112011501140117011601190118011B011A011D011C011F011EFFFF012001220121FFFFFFFF0123FFFF01250124
0126FFFF0127FFFF0128FFFF0129FFFF012AFFFF012BFFFF012CFFFF012DFFFF012EFFFF012FFFFF0130FFFF0131FFFF
01330132013501340136FFFFFFFFFFFFFFFF01370138FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF013A0139013C013B013E013D0140013F
01420141014401430146014501480147014A0149014C014B014E014D0150014F01520151015401530156015501580157
015A0159015C015B015E015D0160015F016201610164016301660165016801670169FFFF016AFFFF016BFFFF016CFFFF
016DFFFF016EFFFF016FFFFF0170FFFF0171FFFF0172FFFF0173FFFF0174FFFF0175FFFF0176FFFF0177FFFF0178FFFF
0179FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF017AFFFF017BFFFF017CFFFF017DFFFF017EFFFF017FFFFF0180FFFF
0181FFFF0182FFFF0183FFFF0184FFFF0185FFFF0186FFFF0187FFFF0188FFFF0189FFFF018AFFFF018BFFFF018CFFFF
018DFFFF018EFFFF018FFFFF0190FFFF0191FFFF0192FFFF0193FFFF0194FFFFFFFFFFFFFFFF0195FFFF0196FFFF0197
FFFF0198FFFF0199FFFF019A019C019B019DFFFF019EFFFF019FFFFF01A0FFFF01A1FFFF01A2FFFF01A3FFFF01A4FFFF
01A5FFFF01A6FFFF01A7FFFF01A8FFFF01A9FFFF01AAFFFF01ABFFFF01ACFFFF01ADFFFF01AEFFFF01AFFFFF01B0FFFF
01B1FFFF01B2FFFF01B3FFFF01B4FFFF01B5FFFF01B6FFFF01B7FFFF01B8FFFF01B9FFFF01BAFFFF01BBFFFF01BCFFFF
01BDFFFF01BEFFFF01BFFFFF01C0FFFF01C1FFFF01C2FFFF01C3FFFF01C4FFFF01C5FFFF01C6FFFF01C7FFFF01C8FFFF
01C9FFFF01CAFFFF01CBFFFF01CCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01CDFFFF01CF01CE01D101D001D301D201D501D401D701D601D901D801DB01DA
01DD01DC01DF01DE01E101E001E301E201E501E401E701E601E901E801EB01EA01ED01EC01EF01EE01F101F0FFFF01F2
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01F401F301F601F501F801F701FA01F901FC01FB01FE01FD020001FF02020201020402030206020502080207020A0209
020C020B020E020D0210020F02120211021402130216021502180217021A0219021C021BFFFF021D021EFFFF0220021F
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022202210224022302260225FFFFFFFF02280227022A0229022C022B022E022D
FFFF022FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0230FFFFFFFFFFFF0231FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0232FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0233FFFF0234FFFF0235FFFF0236FFFF
0237FFFF0238FFFF0239FFFF023AFFFF023BFFFF023CFFFF023DFFFF023EFFFF023FFFFF0240FFFF0241FFFF0242FFFF
0243FFFF0244FFFF0245FFFF0246FFFF0247FFFF0248FFFF0249FFFF024AFFFF024BFFFF024CFFFF024DFFFF024EFFFF
024FFFFF0250FFFF0251FFFF0252FFFF0253FFFF0254FFFF0255FFFF0256FFFF0257FFFF0258FFFF0259FFFF025AFFFF
025BFFFF025CFFFF025DFFFF025EFFFF025FFFFF0260FFFF0261FFFF0262FFFF0263FFFF0264FFFF0265FFFF0266FFFF
0267FFFF0268FFFF0269FFFF026AFFFF026BFFFF026CFFFF026DFFFF026EFFFF026FFFFF0270FFFF0271FFFF0272FFFF
0273FFFF0274FFFF0275FFFF0276FFFF0277FFFF0278FFFF0279FFFF027AFFFF027BFFFF027CFFFF027DFFFFFFFFFFFF
FFFFFFFF027EFFFFFFFFFFFFFFFFFFFF027FFFFF0280FFFF0281FFFF0282FFFF0283FFFF0284FFFF0285FFFF0286FFFF
0287FFFF0288FFFF0289FFFF028AFFFF028BFFFF028CFFFF028DFFFF028EFFFF028FFFFF0290FFFF0291FFFF0292FFFF
0293FFFF0294FFFF0295FFFF0296FFFF0297FFFF0298FFFF0299FFFF029AFFFF029BFFFF029CFFFF029DFFFF029EFFFF
029FFFFF02A0FFFF02A1FFFF02A2FFFF02A3FFFF02A4FFFF02A5FFFF02A6FFFF02A7FFFF02A8FFFF02A9FFFF02AAFFFF
02ABFFFF02ACFFFF02ADFFFF02AEFFFF02B002AF02B202B102B402B302B602B5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02B802B702BA02B902BC02BBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02BE02BD02C002BF02C202C102C402C3
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02C602C502C802C702CA02C902CC02CBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02CE02CD02D002CF02D202D1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D3FFFF02D4FFFF02D5FFFF02D6FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D802D702DA02D902DC02DB02DE02DDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02E002DF02E202E102E402E302E602E502E802E702EA02E902EC02EBFFFFFFFF02EE02ED02F002EF02F202F102F402F3
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02F602F502F802F702FA02F902FC02FBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02FE02FD030002FF0302030103040303FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030603050307FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFF0308FFFFFFFF0309FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
030B030AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030D030CFFFFFFFF030EFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0310FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03120311031403130316031503180317031A0319031C031B031E031D0320031FFFFFFFFFFFFFFFFFFFFF0321FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03230322032503240327032603290328032B032A032D032C032F032E0331033003330332033503340337033603390338
033B033AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
033D033C033F033E0341034003430342034503440347034603490348034B034A034D034C034F034E0351035003530352
035503540357035603590358035B035A035D035C035F035E0361036003630362036503640367036603690368036B036A
036CFFFFFFFFFFFF036DFFFFFFFF036EFFFF036FFFFF0370FFFF0371FFFFFFFFFFFFFFFF0372FFFFFFFFFFFFFFFF0373
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0374FFFF0375FFFF0376FFFF0377FFFF0378FFFF0379FFFF037AFFFF037BFFFF
037CFFFF037DFFFF037EFFFF037FFFFF0380FFFF0381FFFF0382FFFF0383FFFF0384FFFF0385FFFF0386FFFF0387FFFF
0388FFFF0389FFFF038AFFFF038BFFFF038CFFFF038DFFFF038EFFFF038FFFFF0390FFFF0391FFFF0392FFFF0393FFFF
0394FFFF0395FFFF0396FFFF0397FFFF0398FFFF0399FFFF039AFFFF039BFFFF039CFFFF039DFFFF039EFFFF039FFFFF
03A0FFFF03A1FFFF03A2FFFF03A3FFFF03A4FFFF03A5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A6FFFF03A7
FFFFFFFF03A8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03AA03A903AC03AB03AE03AD03B003AF
03B203B103B403B303B603B503B803B703BA03B903BC03BB03BE03BD03C003BF03C203C103C403C303C603C503C803C7
03CA03C903CC03CB03CE03CD03CFFFFFFFFFFFFFFFFFFFFF03D0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03D1FFFF03D2FFFF03D3FFFF03D4FFFF03D5FFFF03D6FFFF03D7FFFF03D8FFFF
03D9FFFF03DAFFFF03DBFFFF03DCFFFF03DDFFFF03DEFFFF03DFFFFF03E0FFFF03E1FFFF03E2FFFF03E3FFFF03E4FFFF
03E5FFFF03E6FFFF03E7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03E8FFFF03E9FFFF03EAFFFF03EBFFFF03ECFFFF03EDFFFF03EEFFFF03EFFFFF03F0FFFF03F1FFFF03F2FFFF03F3FFFF
03F4FFFF03F5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF03F6FFFF03F7FFFF03F8FFFF03F9FFFF03FAFFFF03FBFFFF03FCFFFFFFFFFFFF03FDFFFF03FEFFFF03FFFFFF
0400FFFF0401FFFF0402FFFF0403FFFF0404FFFF0405FFFF0406FFFF0407FFFF0408FFFF0409FFFF040AFFFF040BFFFF
040CFFFF040DFFFF040EFFFF040FFFFF0410FFFF0411FFFF0412FFFF0413FFFF0414FFFF0415FFFF0416FFFF0417FFFF
0418FFFF0419FFFF041AFFFF041BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF041CFFFF041D041EFFFF
041FFFFF0420FFFF0421FFFF0422FFFFFFFFFFFFFFFFFFFFFFFF0423FFFFFFFF0424FFFF0425FFFFFFFF04260427FFFF
0428FFFF0429FFFF042AFFFF042BFFFF042CFFFF042DFFFF042EFFFF042FFFFF0430FFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF0431FFFF0432FFFF0433FFFF0434FFFF0435FFFF0436FFFF0437FFFF0438FFFFFFFFFFFFFFFFFFFF
FFFF0439FFFF043AFFFFFFFFFFFFFFFF043BFFFFFFFFFFFFFFFFFFFF043CFFFF043DFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043E
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF043FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0441044004430442044504440447044604490448044B044A044D044C044F044E
0451045004530452045504540457045604590458045B045A045D045C045F045E04610460046304620465046404670466
04690468046B046A046D046C046F046E0471047004730472047504740477047604790478047B047A047D047C047F047E
0481048004830482048504840487048604890488048B048A048D048C048F048E0490FFFF049204910494049304960495
04980497049A0499049C049B049E049D04A0049F04A204A104A404A304A604A504A804A7FFFF04A9FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
04AB04AA04AD04AC04AF04AE04B104B004B304B204B504B404B704B604B904B804BB04BA04BD04BC04BF04BE04C104C0
04C304C204C504C404C704C604C904C804CB04CA04CD04CC04CF04CE04D104D0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04D304D204D504D404D704D604D904D8
04DB04DA04DD04DC04DF04DE04E104E004E304E204E504E404E704E604E904E804EB04EA04ED04EC04EF04EE04F104F0
04F304F204F504F4FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF04F6FFFF04F804F704FA04F904FC04FB04FE04FD050004FF0501FFFF0503050205050504
0507050605090508050B050A050D050C050F050E0510FFFF0512051105140513051605150517FFFFFFFF0518FFFFFFFF
051A0519051C051B051E051D0520051F05220521052405230526052505280527052A0529052C052B052E052D0530052F
05320531053405330536053505380537053A0539053C053B053E053D0540053F05420541054405430546054505480547
054A0549FFFF054BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF054D054C054F054E0551055005530552
055505540557055605590558055B055A055D055C055F055E0561056005630562056505640567056605690568056B056A
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF056D056C056F056E0571057005730572
057505740577057605790578057B057A057D057C057F057E0581058005830582058505840587058605890588058B058A
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF058D058C058F058E0591059005930592059505940597059605990598
059B059A059D059C059F059E05A105A005A305A205A505A405A705A605A905A805AB05AA05AD05ACFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
immutable(uint)[] toUpperTable() nothrow @nogc pure @safe {
static immutable uint[] t =
cast(immutable uint[]) x"
0000004100000042000000430000004400000045000000460000004700000048000000490000004A0000004B0000004C
0000004D0000004E0000004F000000500000005100000052000000530000005400000055000000560000005700000058
000000590000005A0000039C000000C0000000C1000000C2000000C3000000C4000000C5000000C6000000C7000000C8
000000C9000000CA000000CB000000CC000000CD000000CE000000CF000000D0000000D1000000D2000000D3000000D4
000000D5000000D6000000D8000000D9000000DA000000DB000000DC000000DD000000DE000001780000010000000102
0000010400000106000001080000010A0000010C0000010E00000110000001120000011400000116000001180000011A
0000011C0000011E00000120000001220000012400000126000001280000012A0000012C0000012E0000004900000132
0000013400000136000001390000013B0000013D0000013F000001410000014300000145000001470000014A0000014C
0000014E00000150000001520000015400000156000001580000015A0000015C0000015E000001600000016200000164
00000166000001680000016A0000016C0000016E00000170000001720000017400000176000001790000017B0000017D
00000053000002430000018200000184000001870000018B00000191000001F6000001980000023D00000220000001A0
000001A2000001A4000001A7000001AC000001AF000001B3000001B5000001B8000001BC000001F7000001C4000001C4
000001C7000001C7000001CA000001CA000001CD000001CF000001D1000001D3000001D5000001D7000001D9000001DB
0000018E000001DE000001E0000001E2000001E4000001E6000001E8000001EA000001EC000001EE000001F1000001F1
000001F4000001F8000001FA000001FC000001FE00000200000002020000020400000206000002080000020A0000020C
0000020E00000210000002120000021400000216000002180000021A0000021C0000021E000002220000022400000226
000002280000022A0000022C0000022E00000230000002320000023B00002C7E00002C7F000002410000024600000248
0000024A0000024C0000024E00002C6F00002C6D00002C700000018100000186000001890000018A0000018F00000190
0000A7AB000001930000A7AC000001940000A78D0000A7AA00000197000001960000A7AE00002C620000A7AD0000019C
00002C6E0000019D0000019F00002C64000001A60000A7C5000001A90000A7B1000001AE00000244000001B1000001B2
00000245000001B70000A7B20000A7B000000399000003700000037200000376000003FD000003FE000003FF00000386
00000388000003890000038A000003910000039200000393000003940000039500000396000003970000039800000399
0000039A0000039B0000039C0000039D0000039E0000039F000003A0000003A1000003A3000003A3000003A4000003A5
000003A6000003A7000003A8000003A9000003AA000003AB0000038C0000038E0000038F0000039200000398000003A6
000003A0000003CF000003D8000003DA000003DC000003DE000003E0000003E2000003E4000003E6000003E8000003EA
000003EC000003EE0000039A000003A1000003F90000037F00000395000003F7000003FA000004100000041100000412
000004130000041400000415000004160000041700000418000004190000041A0000041B0000041C0000041D0000041E
0000041F000004200000042100000422000004230000042400000425000004260000042700000428000004290000042A
0000042B0000042C0000042D0000042E0000042F00000400000004010000040200000403000004040000040500000406
0000040700000408000004090000040A0000040B0000040C0000040D0000040E0000040F000004600000046200000464
00000466000004680000046A0000046C0000046E00000470000004720000047400000476000004780000047A0000047C
0000047E000004800000048A0000048C0000048E00000490000004920000049400000496000004980000049A0000049C
0000049E000004A0000004A2000004A4000004A6000004A8000004AA000004AC000004AE000004B0000004B2000004B4
000004B6000004B8000004BA000004BC000004BE000004C1000004C3000004C5000004C7000004C9000004CB000004CD
000004C0000004D0000004D2000004D4000004D6000004D8000004DA000004DC000004DE000004E0000004E2000004E4
000004E6000004E8000004EA000004EC000004EE000004F0000004F2000004F4000004F6000004F8000004FA000004FC
000004FE00000500000005020000050400000506000005080000050A0000050C0000050E000005100000051200000514
00000516000005180000051A0000051C0000051E00000520000005220000052400000526000005280000052A0000052C
0000052E0000053100000532000005330000053400000535000005360000053700000538000005390000053A0000053B
0000053C0000053D0000053E0000053F0000054000000541000005420000054300000544000005450000054600000547
00000548000005490000054A0000054B0000054C0000054D0000054E0000054F00000550000005510000055200000553
00000554000005550000055600001C9000001C9100001C9200001C9300001C9400001C9500001C9600001C9700001C98
00001C9900001C9A00001C9B00001C9C00001C9D00001C9E00001C9F00001CA000001CA100001CA200001CA300001CA4
00001CA500001CA600001CA700001CA800001CA900001CAA00001CAB00001CAC00001CAD00001CAE00001CAF00001CB0
00001CB100001CB200001CB300001CB400001CB500001CB600001CB700001CB800001CB900001CBA00001CBD00001CBE
00001CBF000013F0000013F1000013F2000013F3000013F4000013F500000412000004140000041E0000042100000422
000004220000042A000004620000A64A0000A77D00002C630000A7C600001E0000001E0200001E0400001E0600001E08
00001E0A00001E0C00001E0E00001E1000001E1200001E1400001E1600001E1800001E1A00001E1C00001E1E00001E20
00001E2200001E2400001E2600001E2800001E2A00001E2C00001E2E00001E3000001E3200001E3400001E3600001E38
00001E3A00001E3C00001E3E00001E4000001E4200001E4400001E4600001E4800001E4A00001E4C00001E4E00001E50
00001E5200001E5400001E5600001E5800001E5A00001E5C00001E5E00001E6000001E6200001E6400001E6600001E68
00001E6A00001E6C00001E6E00001E7000001E7200001E7400001E7600001E7800001E7A00001E7C00001E7E00001E80
00001E8200001E8400001E8600001E8800001E8A00001E8C00001E8E00001E9000001E9200001E9400001E6000001EA0
00001EA200001EA400001EA600001EA800001EAA00001EAC00001EAE00001EB000001EB200001EB400001EB600001EB8
00001EBA00001EBC00001EBE00001EC000001EC200001EC400001EC600001EC800001ECA00001ECC00001ECE00001ED0
00001ED200001ED400001ED600001ED800001EDA00001EDC00001EDE00001EE000001EE200001EE400001EE600001EE8
00001EEA00001EEC00001EEE00001EF000001EF200001EF400001EF600001EF800001EFA00001EFC00001EFE00001F08
00001F0900001F0A00001F0B00001F0C00001F0D00001F0E00001F0F00001F1800001F1900001F1A00001F1B00001F1C
00001F1D00001F2800001F2900001F2A00001F2B00001F2C00001F2D00001F2E00001F2F00001F3800001F3900001F3A
00001F3B00001F3C00001F3D00001F3E00001F3F00001F4800001F4900001F4A00001F4B00001F4C00001F4D00001F59
00001F5B00001F5D00001F5F00001F6800001F6900001F6A00001F6B00001F6C00001F6D00001F6E00001F6F00001FBA
00001FBB00001FC800001FC900001FCA00001FCB00001FDA00001FDB00001FF800001FF900001FEA00001FEB00001FFA
00001FFB00001F8800001F8900001F8A00001F8B00001F8C00001F8D00001F8E00001F8F00001F9800001F9900001F9A
00001F9B00001F9C00001F9D00001F9E00001F9F00001FA800001FA900001FAA00001FAB00001FAC00001FAD00001FAE
00001FAF00001FB800001FB900001FBC0000039900001FCC00001FD800001FD900001FE800001FE900001FEC00001FFC
00002132000021600000216100002162000021630000216400002165000021660000216700002168000021690000216A
0000216B0000216C0000216D0000216E0000216F00002183000024B6000024B7000024B8000024B9000024BA000024BB
000024BC000024BD000024BE000024BF000024C0000024C1000024C2000024C3000024C4000024C5000024C6000024C7
000024C8000024C9000024CA000024CB000024CC000024CD000024CE000024CF00002C0000002C0100002C0200002C03
00002C0400002C0500002C0600002C0700002C0800002C0900002C0A00002C0B00002C0C00002C0D00002C0E00002C0F
00002C1000002C1100002C1200002C1300002C1400002C1500002C1600002C1700002C1800002C1900002C1A00002C1B
00002C1C00002C1D00002C1E00002C1F00002C2000002C2100002C2200002C2300002C2400002C2500002C2600002C27
00002C2800002C2900002C2A00002C2B00002C2C00002C2D00002C2E00002C2F00002C600000023A0000023E00002C67
00002C6900002C6B00002C7200002C7500002C8000002C8200002C8400002C8600002C8800002C8A00002C8C00002C8E
00002C9000002C9200002C9400002C9600002C9800002C9A00002C9C00002C9E00002CA000002CA200002CA400002CA6
00002CA800002CAA00002CAC00002CAE00002CB000002CB200002CB400002CB600002CB800002CBA00002CBC00002CBE
00002CC000002CC200002CC400002CC600002CC800002CCA00002CCC00002CCE00002CD000002CD200002CD400002CD6
00002CD800002CDA00002CDC00002CDE00002CE000002CE200002CEB00002CED00002CF2000010A0000010A1000010A2
000010A3000010A4000010A5000010A6000010A7000010A8000010A9000010AA000010AB000010AC000010AD000010AE
000010AF000010B0000010B1000010B2000010B3000010B4000010B5000010B6000010B7000010B8000010B9000010BA
000010BB000010BC000010BD000010BE000010BF000010C0000010C1000010C2000010C3000010C4000010C5000010C7
000010CD0000A6400000A6420000A6440000A6460000A6480000A64A0000A64C0000A64E0000A6500000A6520000A654
0000A6560000A6580000A65A0000A65C0000A65E0000A6600000A6620000A6640000A6660000A6680000A66A0000A66C
0000A6800000A6820000A6840000A6860000A6880000A68A0000A68C0000A68E0000A6900000A6920000A6940000A696
0000A6980000A69A0000A7220000A7240000A7260000A7280000A72A0000A72C0000A72E0000A7320000A7340000A736
0000A7380000A73A0000A73C0000A73E0000A7400000A7420000A7440000A7460000A7480000A74A0000A74C0000A74E
0000A7500000A7520000A7540000A7560000A7580000A75A0000A75C0000A75E0000A7600000A7620000A7640000A766
0000A7680000A76A0000A76C0000A76E0000A7790000A77B0000A77E0000A7800000A7820000A7840000A7860000A78B
0000A7900000A7920000A7C40000A7960000A7980000A79A0000A79C0000A79E0000A7A00000A7A20000A7A40000A7A6
0000A7A80000A7B40000A7B60000A7B80000A7BA0000A7BC0000A7BE0000A7C00000A7C20000A7C70000A7C90000A7D0
0000A7D60000A7D80000A7F50000A7B3000013A0000013A1000013A2000013A3000013A4000013A5000013A6000013A7
000013A8000013A9000013AA000013AB000013AC000013AD000013AE000013AF000013B0000013B1000013B2000013B3
000013B4000013B5000013B6000013B7000013B8000013B9000013BA000013BB000013BC000013BD000013BE000013BF
000013C0000013C1000013C2000013C3000013C4000013C5000013C6000013C7000013C8000013C9000013CA000013CB
000013CC000013CD000013CE000013CF000013D0000013D1000013D2000013D3000013D4000013D5000013D6000013D7
000013D8000013D9000013DA000013DB000013DC000013DD000013DE000013DF000013E0000013E1000013E2000013E3
000013E4000013E5000013E6000013E7000013E8000013E9000013EA000013EB000013EC000013ED000013EE000013EF
0000FF210000FF220000FF230000FF240000FF250000FF260000FF270000FF280000FF290000FF2A0000FF2B0000FF2C
0000FF2D0000FF2E0000FF2F0000FF300000FF310000FF320000FF330000FF340000FF350000FF360000FF370000FF38
0000FF390000FF3A00010400000104010001040200010403000104040001040500010406000104070001040800010409
0001040A0001040B0001040C0001040D0001040E0001040F000104100001041100010412000104130001041400010415
000104160001041700010418000104190001041A0001041B0001041C0001041D0001041E0001041F0001042000010421
000104220001042300010424000104250001042600010427000104B0000104B1000104B2000104B3000104B4000104B5
000104B6000104B7000104B8000104B9000104BA000104BB000104BC000104BD000104BE000104BF000104C0000104C1
000104C2000104C3000104C4000104C5000104C6000104C7000104C8000104C9000104CA000104CB000104CC000104CD
000104CE000104CF000104D0000104D1000104D2000104D3000105700001057100010572000105730001057400010575
000105760001057700010578000105790001057A0001057C0001057D0001057E0001057F000105800001058100010582
000105830001058400010585000105860001058700010588000105890001058A0001058C0001058D0001058E0001058F
000105900001059100010592000105940001059500010C8000010C8100010C8200010C8300010C8400010C8500010C86
00010C8700010C8800010C8900010C8A00010C8B00010C8C00010C8D00010C8E00010C8F00010C9000010C9100010C92
00010C9300010C9400010C9500010C9600010C9700010C9800010C9900010C9A00010C9B00010C9C00010C9D00010C9E
00010C9F00010CA000010CA100010CA200010CA300010CA400010CA500010CA600010CA700010CA800010CA900010CAA
00010CAB00010CAC00010CAD00010CAE00010CAF00010CB000010CB100010CB2000118A0000118A1000118A2000118A3
000118A4000118A5000118A6000118A7000118A8000118A9000118AA000118AB000118AC000118AD000118AE000118AF
000118B0000118B1000118B2000118B3000118B4000118B5000118B6000118B7000118B8000118B9000118BA000118BB
000118BC000118BD000118BE000118BF00016E4000016E4100016E4200016E4300016E4400016E4500016E4600016E47
00016E4800016E4900016E4A00016E4B00016E4C00016E4D00016E4E00016E4F00016E5000016E5100016E5200016E53
00016E5400016E5500016E5600016E5700016E5800016E5900016E5A00016E5B00016E5C00016E5D00016E5E00016E5F
0001E9000001E9010001E9020001E9030001E9040001E9050001E9060001E9070001E9080001E9090001E90A0001E90B
0001E90C0001E90D0001E90E0001E90F0001E9100001E9110001E9120001E9130001E9140001E9150001E9160001E917
0001E9180001E9190001E91A0001E91B0001E91C0001E91D0001E91E0001E91F0001E9200001E9210200005300000053
0000013002000046000000460200004600000049020000460000004C0300004600000046000000490300004600000046
0000004C0200005300000054020000530000005402000535000005520200054400000546020005440000053502000544
0000053B0200054E00000546020005440000053D020002BC0000004E030003990000030800000301030003A500000308
000003010200004A0000030C02000048000003310200005400000308020000570000030A020000590000030A02000041
000002BE020003A500000313030003A50000031300000300030003A50000031300000301030003A50000031300000342
020003910000034202000397000003420300039900000308000003000300039900000308000003010200039900000342
030003990000030800000342030003A50000030800000300030003A50000030800000301020003A100000313020003A5
00000342030003A50000030800000342020003A90000034202001F080000039902001F090000039902001F0A00000399
02001F0B0000039902001F0C0000039902001F0D0000039902001F0E0000039902001F0F0000039902001F0800000399
02001F090000039902001F0A0000039902001F0B0000039902001F0C0000039902001F0D0000039902001F0E00000399
02001F0F0000039902001F280000039902001F290000039902001F2A0000039902001F2B0000039902001F2C00000399
02001F2D0000039902001F2E0000039902001F2F0000039902001F280000039902001F290000039902001F2A00000399
02001F2B0000039902001F2C0000039902001F2D0000039902001F2E0000039902001F2F0000039902001F6800000399
02001F690000039902001F6A0000039902001F6B0000039902001F6C0000039902001F6D0000039902001F6E00000399
02001F6F0000039902001F680000039902001F690000039902001F6A0000039902001F6B0000039902001F6C00000399
02001F6D0000039902001F6E0000039902001F6F00000399020003910000039902000391000003990200039700000399
0200039700000399020003A900000399020003A90000039902001FBA00000399020003860000039902001FCA00000399
020003890000039902001FFA000003990200038F00000399030003910000034200000399030003970000034200000399
030003A90000034200000399";
return t;
}
immutable(uint)[] toLowerTable() nothrow @nogc pure @safe {
static immutable uint[] t =
cast(immutable uint[]) x"
0000006100000062000000630000006400000065000000660000006700000068000000690000006A0000006B0000006C
0000006D0000006E0000006F000000700000007100000072000000730000007400000075000000760000007700000078
000000790000007A000000E0000000E1000000E2000000E3000000E4000000E5000000E6000000E7000000E8000000E9
000000EA000000EB000000EC000000ED000000EE000000EF000000F0000000F1000000F2000000F3000000F4000000F5
000000F6000000F8000000F9000000FA000000FB000000FC000000FD000000FE00000101000001030000010500000107
000001090000010B0000010D0000010F00000111000001130000011500000117000001190000011B0000011D0000011F
00000121000001230000012500000127000001290000012B0000012D0000012F00000069000001330000013500000137
0000013A0000013C0000013E00000140000001420000014400000146000001480000014B0000014D0000014F00000151
000001530000015500000157000001590000015B0000015D0000015F0000016100000163000001650000016700000169
0000016B0000016D0000016F00000171000001730000017500000177000000FF0000017A0000017C0000017E00000253
0000018300000185000002540000018800000256000002570000018C000001DD000002590000025B0000019200000260
000002630000026900000268000001990000026F0000027200000275000001A1000001A3000001A500000280000001A8
00000283000001AD00000288000001B00000028A0000028B000001B4000001B600000292000001B9000001BD000001C6
000001C6000001C9000001C9000001CC000001CC000001CE000001D0000001D2000001D4000001D6000001D8000001DA
000001DC000001DF000001E1000001E3000001E5000001E7000001E9000001EB000001ED000001EF000001F3000001F3
000001F500000195000001BF000001F9000001FB000001FD000001FF0000020100000203000002050000020700000209
0000020B0000020D0000020F00000211000002130000021500000217000002190000021B0000021D0000021F0000019E
000002230000022500000227000002290000022B0000022D0000022F000002310000023300002C650000023C0000019A
00002C660000024200000180000002890000028C00000247000002490000024B0000024D0000024F0000037100000373
00000377000003F3000003AC000003AD000003AE000003AF000003CC000003CD000003CE000003B1000003B2000003B3
000003B4000003B5000003B6000003B7000003B8000003B9000003BA000003BB000003BC000003BD000003BE000003BF
000003C0000003C1000003C3000003C4000003C5000003C6000003C7000003C8000003C9000003CA000003CB000003D7
000003D9000003DB000003DD000003DF000003E1000003E3000003E5000003E7000003E9000003EB000003ED000003EF
000003B8000003F8000003F2000003FB0000037B0000037C0000037D0000045000000451000004520000045300000454
00000455000004560000045700000458000004590000045A0000045B0000045C0000045D0000045E0000045F00000430
0000043100000432000004330000043400000435000004360000043700000438000004390000043A0000043B0000043C
0000043D0000043E0000043F000004400000044100000442000004430000044400000445000004460000044700000448
000004490000044A0000044B0000044C0000044D0000044E0000044F0000046100000463000004650000046700000469
0000046B0000046D0000046F00000471000004730000047500000477000004790000047B0000047D0000047F00000481
0000048B0000048D0000048F00000491000004930000049500000497000004990000049B0000049D0000049F000004A1
000004A3000004A5000004A7000004A9000004AB000004AD000004AF000004B1000004B3000004B5000004B7000004B9
000004BB000004BD000004BF000004CF000004C2000004C4000004C6000004C8000004CA000004CC000004CE000004D1
000004D3000004D5000004D7000004D9000004DB000004DD000004DF000004E1000004E3000004E5000004E7000004E9
000004EB000004ED000004EF000004F1000004F3000004F5000004F7000004F9000004FB000004FD000004FF00000501
000005030000050500000507000005090000050B0000050D0000050F0000051100000513000005150000051700000519
0000051B0000051D0000051F00000521000005230000052500000527000005290000052B0000052D0000052F00000561
00000562000005630000056400000565000005660000056700000568000005690000056A0000056B0000056C0000056D
0000056E0000056F00000570000005710000057200000573000005740000057500000576000005770000057800000579
0000057A0000057B0000057C0000057D0000057E0000057F000005800000058100000582000005830000058400000585
0000058600002D0000002D0100002D0200002D0300002D0400002D0500002D0600002D0700002D0800002D0900002D0A
00002D0B00002D0C00002D0D00002D0E00002D0F00002D1000002D1100002D1200002D1300002D1400002D1500002D16
00002D1700002D1800002D1900002D1A00002D1B00002D1C00002D1D00002D1E00002D1F00002D2000002D2100002D22
00002D2300002D2400002D2500002D2700002D2D0000AB700000AB710000AB720000AB730000AB740000AB750000AB76
0000AB770000AB780000AB790000AB7A0000AB7B0000AB7C0000AB7D0000AB7E0000AB7F0000AB800000AB810000AB82
0000AB830000AB840000AB850000AB860000AB870000AB880000AB890000AB8A0000AB8B0000AB8C0000AB8D0000AB8E
0000AB8F0000AB900000AB910000AB920000AB930000AB940000AB950000AB960000AB970000AB980000AB990000AB9A
0000AB9B0000AB9C0000AB9D0000AB9E0000AB9F0000ABA00000ABA10000ABA20000ABA30000ABA40000ABA50000ABA6
0000ABA70000ABA80000ABA90000ABAA0000ABAB0000ABAC0000ABAD0000ABAE0000ABAF0000ABB00000ABB10000ABB2
0000ABB30000ABB40000ABB50000ABB60000ABB70000ABB80000ABB90000ABBA0000ABBB0000ABBC0000ABBD0000ABBE
0000ABBF000013F8000013F9000013FA000013FB000013FC000013FD000010D0000010D1000010D2000010D3000010D4
000010D5000010D6000010D7000010D8000010D9000010DA000010DB000010DC000010DD000010DE000010DF000010E0
000010E1000010E2000010E3000010E4000010E5000010E6000010E7000010E8000010E9000010EA000010EB000010EC
000010ED000010EE000010EF000010F0000010F1000010F2000010F3000010F4000010F5000010F6000010F7000010F8
000010F9000010FA000010FD000010FE000010FF00001E0100001E0300001E0500001E0700001E0900001E0B00001E0D
00001E0F00001E1100001E1300001E1500001E1700001E1900001E1B00001E1D00001E1F00001E2100001E2300001E25
00001E2700001E2900001E2B00001E2D00001E2F00001E3100001E3300001E3500001E3700001E3900001E3B00001E3D
00001E3F00001E4100001E4300001E4500001E4700001E4900001E4B00001E4D00001E4F00001E5100001E5300001E55
00001E5700001E5900001E5B00001E5D00001E5F00001E6100001E6300001E6500001E6700001E6900001E6B00001E6D
00001E6F00001E7100001E7300001E7500001E7700001E7900001E7B00001E7D00001E7F00001E8100001E8300001E85
00001E8700001E8900001E8B00001E8D00001E8F00001E9100001E9300001E95000000DF00001EA100001EA300001EA5
00001EA700001EA900001EAB00001EAD00001EAF00001EB100001EB300001EB500001EB700001EB900001EBB00001EBD
00001EBF00001EC100001EC300001EC500001EC700001EC900001ECB00001ECD00001ECF00001ED100001ED300001ED5
00001ED700001ED900001EDB00001EDD00001EDF00001EE100001EE300001EE500001EE700001EE900001EEB00001EED
00001EEF00001EF100001EF300001EF500001EF700001EF900001EFB00001EFD00001EFF00001F0000001F0100001F02
00001F0300001F0400001F0500001F0600001F0700001F1000001F1100001F1200001F1300001F1400001F1500001F20
00001F2100001F2200001F2300001F2400001F2500001F2600001F2700001F3000001F3100001F3200001F3300001F34
00001F3500001F3600001F3700001F4000001F4100001F4200001F4300001F4400001F4500001F5100001F5300001F55
00001F5700001F6000001F6100001F6200001F6300001F6400001F6500001F6600001F6700001F8000001F8100001F82
00001F8300001F8400001F8500001F8600001F8700001F9000001F9100001F9200001F9300001F9400001F9500001F96
00001F9700001FA000001FA100001FA200001FA300001FA400001FA500001FA600001FA700001FB000001FB100001F70
00001F7100001FB300001F7200001F7300001F7400001F7500001FC300001FD000001FD100001F7600001F7700001FE0
00001FE100001F7A00001F7B00001FE500001F7800001F7900001F7C00001F7D00001FF3000003C90000006B000000E5
0000214E000021700000217100002172000021730000217400002175000021760000217700002178000021790000217A
0000217B0000217C0000217D0000217E0000217F00002184000024D0000024D1000024D2000024D3000024D4000024D5
000024D6000024D7000024D8000024D9000024DA000024DB000024DC000024DD000024DE000024DF000024E0000024E1
000024E2000024E3000024E4000024E5000024E6000024E7000024E8000024E900002C3000002C3100002C3200002C33
00002C3400002C3500002C3600002C3700002C3800002C3900002C3A00002C3B00002C3C00002C3D00002C3E00002C3F
00002C4000002C4100002C4200002C4300002C4400002C4500002C4600002C4700002C4800002C4900002C4A00002C4B
00002C4C00002C4D00002C4E00002C4F00002C5000002C5100002C5200002C5300002C5400002C5500002C5600002C57
00002C5800002C5900002C5A00002C5B00002C5C00002C5D00002C5E00002C5F00002C610000026B00001D7D0000027D
00002C6800002C6A00002C6C0000025100000271000002500000025200002C7300002C760000023F0000024000002C81
00002C8300002C8500002C8700002C8900002C8B00002C8D00002C8F00002C9100002C9300002C9500002C9700002C99
00002C9B00002C9D00002C9F00002CA100002CA300002CA500002CA700002CA900002CAB00002CAD00002CAF00002CB1
00002CB300002CB500002CB700002CB900002CBB00002CBD00002CBF00002CC100002CC300002CC500002CC700002CC9
00002CCB00002CCD00002CCF00002CD100002CD300002CD500002CD700002CD900002CDB00002CDD00002CDF00002CE1
00002CE300002CEC00002CEE00002CF30000A6410000A6430000A6450000A6470000A6490000A64B0000A64D0000A64F
0000A6510000A6530000A6550000A6570000A6590000A65B0000A65D0000A65F0000A6610000A6630000A6650000A667
0000A6690000A66B0000A66D0000A6810000A6830000A6850000A6870000A6890000A68B0000A68D0000A68F0000A691
0000A6930000A6950000A6970000A6990000A69B0000A7230000A7250000A7270000A7290000A72B0000A72D0000A72F
0000A7330000A7350000A7370000A7390000A73B0000A73D0000A73F0000A7410000A7430000A7450000A7470000A749
0000A74B0000A74D0000A74F0000A7510000A7530000A7550000A7570000A7590000A75B0000A75D0000A75F0000A761
0000A7630000A7650000A7670000A7690000A76B0000A76D0000A76F0000A77A0000A77C00001D790000A77F0000A781
0000A7830000A7850000A7870000A78C000002650000A7910000A7930000A7970000A7990000A79B0000A79D0000A79F
0000A7A10000A7A30000A7A50000A7A70000A7A9000002660000025C000002610000026C0000026A0000029E00000287
0000029D0000AB530000A7B50000A7B70000A7B90000A7BB0000A7BD0000A7BF0000A7C10000A7C30000A79400000282
00001D8E0000A7C80000A7CA0000A7D10000A7D70000A7D90000A7F60000FF410000FF420000FF430000FF440000FF45
0000FF460000FF470000FF480000FF490000FF4A0000FF4B0000FF4C0000FF4D0000FF4E0000FF4F0000FF500000FF51
0000FF520000FF530000FF540000FF550000FF560000FF570000FF580000FF590000FF5A00010428000104290001042A
0001042B0001042C0001042D0001042E0001042F00010430000104310001043200010433000104340001043500010436
0001043700010438000104390001043A0001043B0001043C0001043D0001043E0001043F000104400001044100010442
000104430001044400010445000104460001044700010448000104490001044A0001044B0001044C0001044D0001044E
0001044F000104D8000104D9000104DA000104DB000104DC000104DD000104DE000104DF000104E0000104E1000104E2
000104E3000104E4000104E5000104E6000104E7000104E8000104E9000104EA000104EB000104EC000104ED000104EE
000104EF000104F0000104F1000104F2000104F3000104F4000104F5000104F6000104F7000104F8000104F9000104FA
000104FB0001059700010598000105990001059A0001059B0001059C0001059D0001059E0001059F000105A0000105A1
000105A3000105A4000105A5000105A6000105A7000105A8000105A9000105AA000105AB000105AC000105AD000105AE
000105AF000105B0000105B1000105B3000105B4000105B5000105B6000105B7000105B8000105B9000105BB000105BC
00010CC000010CC100010CC200010CC300010CC400010CC500010CC600010CC700010CC800010CC900010CCA00010CCB
00010CCC00010CCD00010CCE00010CCF00010CD000010CD100010CD200010CD300010CD400010CD500010CD600010CD7
00010CD800010CD900010CDA00010CDB00010CDC00010CDD00010CDE00010CDF00010CE000010CE100010CE200010CE3
00010CE400010CE500010CE600010CE700010CE800010CE900010CEA00010CEB00010CEC00010CED00010CEE00010CEF
00010CF000010CF100010CF2000118C0000118C1000118C2000118C3000118C4000118C5000118C6000118C7000118C8
000118C9000118CA000118CB000118CC000118CD000118CE000118CF000118D0000118D1000118D2000118D3000118D4
000118D5000118D6000118D7000118D8000118D9000118DA000118DB000118DC000118DD000118DE000118DF00016E60
00016E6100016E6200016E6300016E6400016E6500016E6600016E6700016E6800016E6900016E6A00016E6B00016E6C
00016E6D00016E6E00016E6F00016E7000016E7100016E7200016E7300016E7400016E7500016E7600016E7700016E78
00016E7900016E7A00016E7B00016E7C00016E7D00016E7E00016E7F0001E9220001E9230001E9240001E9250001E926
0001E9270001E9280001E9290001E92A0001E92B0001E92C0001E92D0001E92E0001E92F0001E9300001E9310001E932
0001E9330001E9340001E9350001E9360001E9370001E9380001E9390001E93A0001E93B0001E93C0001E93D0001E93E
0001E93F0001E9400001E9410001E9420001E943000000DF02000069000003070000FB000000FB010000FB020000FB03
0000FB040000FB050000FB06000005870000FB130000FB140000FB150000FB160000FB170000014900000390000003B0
000001F000001E9600001E9700001E9800001E9900001E9A00001F5000001F5200001F5400001F5600001FB600001FC6
00001FD200001FD300001FD600001FD700001FE200001FE300001FE400001FE600001FE700001FF600001F8000001F81
00001F8200001F8300001F8400001F8500001F8600001F8700001F8000001F8100001F8200001F8300001F8400001F85
00001F8600001F8700001F9000001F9100001F9200001F9300001F9400001F9500001F9600001F9700001F9000001F91
00001F9200001F9300001F9400001F9500001F9600001F9700001FA000001FA100001FA200001FA300001FA400001FA5
00001FA600001FA700001FA000001FA100001FA200001FA300001FA400001FA500001FA600001FA700001FB300001FB3
00001FC300001FC300001FF300001FF300001FB200001FB400001FC200001FC400001FF200001FF400001FB700001FC7
00001FF7";
return t;
}
immutable(uint)[] toTitleTable() nothrow @nogc pure @safe {
static immutable uint[] t =
cast(immutable uint[]) x"
0000004100000042000000430000004400000045000000460000004700000048000000490000004A0000004B0000004C
0000004D0000004E0000004F000000500000005100000052000000530000005400000055000000560000005700000058
000000590000005A0000039C000000C0000000C1000000C2000000C3000000C4000000C5000000C6000000C7000000C8
000000C9000000CA000000CB000000CC000000CD000000CE000000CF000000D0000000D1000000D2000000D3000000D4
000000D5000000D6000000D8000000D9000000DA000000DB000000DC000000DD000000DE000001780000010000000102
0000010400000106000001080000010A0000010C0000010E00000110000001120000011400000116000001180000011A
0000011C0000011E00000120000001220000012400000126000001280000012A0000012C0000012E0000004900000132
0000013400000136000001390000013B0000013D0000013F000001410000014300000145000001470000014A0000014C
0000014E00000150000001520000015400000156000001580000015A0000015C0000015E000001600000016200000164
00000166000001680000016A0000016C0000016E00000170000001720000017400000176000001790000017B0000017D
00000053000002430000018200000184000001870000018B00000191000001F6000001980000023D00000220000001A0
000001A2000001A4000001A7000001AC000001AF000001B3000001B5000001B8000001BC000001F7000001C5000001C5
000001C5000001C8000001C8000001C8000001CB000001CB000001CB000001CD000001CF000001D1000001D3000001D5
000001D7000001D9000001DB0000018E000001DE000001E0000001E2000001E4000001E6000001E8000001EA000001EC
000001EE000001F2000001F2000001F2000001F4000001F8000001FA000001FC000001FE000002000000020200000204
00000206000002080000020A0000020C0000020E00000210000002120000021400000216000002180000021A0000021C
0000021E000002220000022400000226000002280000022A0000022C0000022E00000230000002320000023B00002C7E
00002C7F0000024100000246000002480000024A0000024C0000024E00002C6F00002C6D00002C700000018100000186
000001890000018A0000018F000001900000A7AB000001930000A7AC000001940000A78D0000A7AA0000019700000196
0000A7AE00002C620000A7AD0000019C00002C6E0000019D0000019F00002C64000001A60000A7C5000001A90000A7B1
000001AE00000244000001B1000001B200000245000001B70000A7B20000A7B000000399000003700000037200000376
000003FD000003FE000003FF0000038600000388000003890000038A0000039100000392000003930000039400000395
000003960000039700000398000003990000039A0000039B0000039C0000039D0000039E0000039F000003A0000003A1
000003A3000003A3000003A4000003A5000003A6000003A7000003A8000003A9000003AA000003AB0000038C0000038E
0000038F0000039200000398000003A6000003A0000003CF000003D8000003DA000003DC000003DE000003E0000003E2
000003E4000003E6000003E8000003EA000003EC000003EE0000039A000003A1000003F90000037F00000395000003F7
000003FA000004100000041100000412000004130000041400000415000004160000041700000418000004190000041A
0000041B0000041C0000041D0000041E0000041F00000420000004210000042200000423000004240000042500000426
0000042700000428000004290000042A0000042B0000042C0000042D0000042E0000042F000004000000040100000402
000004030000040400000405000004060000040700000408000004090000040A0000040B0000040C0000040D0000040E
0000040F00000460000004620000046400000466000004680000046A0000046C0000046E000004700000047200000474
00000476000004780000047A0000047C0000047E000004800000048A0000048C0000048E000004900000049200000494
00000496000004980000049A0000049C0000049E000004A0000004A2000004A4000004A6000004A8000004AA000004AC
000004AE000004B0000004B2000004B4000004B6000004B8000004BA000004BC000004BE000004C1000004C3000004C5
000004C7000004C9000004CB000004CD000004C0000004D0000004D2000004D4000004D6000004D8000004DA000004DC
000004DE000004E0000004E2000004E4000004E6000004E8000004EA000004EC000004EE000004F0000004F2000004F4
000004F6000004F8000004FA000004FC000004FE00000500000005020000050400000506000005080000050A0000050C
0000050E00000510000005120000051400000516000005180000051A0000051C0000051E000005200000052200000524
00000526000005280000052A0000052C0000052E00000531000005320000053300000534000005350000053600000537
00000538000005390000053A0000053B0000053C0000053D0000053E0000053F00000540000005410000054200000543
0000054400000545000005460000054700000548000005490000054A0000054B0000054C0000054D0000054E0000054F
00000550000005510000055200000553000005540000055500000556000010D0000010D1000010D2000010D3000010D4
000010D5000010D6000010D7000010D8000010D9000010DA000010DB000010DC000010DD000010DE000010DF000010E0
000010E1000010E2000010E3000010E4000010E5000010E6000010E7000010E8000010E9000010EA000010EB000010EC
000010ED000010EE000010EF000010F0000010F1000010F2000010F3000010F4000010F5000010F6000010F7000010F8
000010F9000010FA000010FD000010FE000010FF000013F0000013F1000013F2000013F3000013F4000013F500000412
000004140000041E0000042100000422000004220000042A000004620000A64A0000A77D00002C630000A7C600001E00
00001E0200001E0400001E0600001E0800001E0A00001E0C00001E0E00001E1000001E1200001E1400001E1600001E18
00001E1A00001E1C00001E1E00001E2000001E2200001E2400001E2600001E2800001E2A00001E2C00001E2E00001E30
00001E3200001E3400001E3600001E3800001E3A00001E3C00001E3E00001E4000001E4200001E4400001E4600001E48
00001E4A00001E4C00001E4E00001E5000001E5200001E5400001E5600001E5800001E5A00001E5C00001E5E00001E60
00001E6200001E6400001E6600001E6800001E6A00001E6C00001E6E00001E7000001E7200001E7400001E7600001E78
00001E7A00001E7C00001E7E00001E8000001E8200001E8400001E8600001E8800001E8A00001E8C00001E8E00001E90
00001E9200001E9400001E6000001EA000001EA200001EA400001EA600001EA800001EAA00001EAC00001EAE00001EB0
00001EB200001EB400001EB600001EB800001EBA00001EBC00001EBE00001EC000001EC200001EC400001EC600001EC8
00001ECA00001ECC00001ECE00001ED000001ED200001ED400001ED600001ED800001EDA00001EDC00001EDE00001EE0
00001EE200001EE400001EE600001EE800001EEA00001EEC00001EEE00001EF000001EF200001EF400001EF600001EF8
00001EFA00001EFC00001EFE00001F0800001F0900001F0A00001F0B00001F0C00001F0D00001F0E00001F0F00001F18
00001F1900001F1A00001F1B00001F1C00001F1D00001F2800001F2900001F2A00001F2B00001F2C00001F2D00001F2E
00001F2F00001F3800001F3900001F3A00001F3B00001F3C00001F3D00001F3E00001F3F00001F4800001F4900001F4A
00001F4B00001F4C00001F4D00001F5900001F5B00001F5D00001F5F00001F6800001F6900001F6A00001F6B00001F6C
00001F6D00001F6E00001F6F00001FBA00001FBB00001FC800001FC900001FCA00001FCB00001FDA00001FDB00001FF8
00001FF900001FEA00001FEB00001FFA00001FFB00001F8800001F8900001F8A00001F8B00001F8C00001F8D00001F8E
00001F8F00001F9800001F9900001F9A00001F9B00001F9C00001F9D00001F9E00001F9F00001FA800001FA900001FAA
00001FAB00001FAC00001FAD00001FAE00001FAF00001FB800001FB900001FBC0000039900001FCC00001FD800001FD9
00001FE800001FE900001FEC00001FFC0000213200002160000021610000216200002163000021640000216500002166
0000216700002168000021690000216A0000216B0000216C0000216D0000216E0000216F00002183000024B6000024B7
000024B8000024B9000024BA000024BB000024BC000024BD000024BE000024BF000024C0000024C1000024C2000024C3
000024C4000024C5000024C6000024C7000024C8000024C9000024CA000024CB000024CC000024CD000024CE000024CF
00002C0000002C0100002C0200002C0300002C0400002C0500002C0600002C0700002C0800002C0900002C0A00002C0B
00002C0C00002C0D00002C0E00002C0F00002C1000002C1100002C1200002C1300002C1400002C1500002C1600002C17
00002C1800002C1900002C1A00002C1B00002C1C00002C1D00002C1E00002C1F00002C2000002C2100002C2200002C23
00002C2400002C2500002C2600002C2700002C2800002C2900002C2A00002C2B00002C2C00002C2D00002C2E00002C2F
00002C600000023A0000023E00002C6700002C6900002C6B00002C7200002C7500002C8000002C8200002C8400002C86
00002C8800002C8A00002C8C00002C8E00002C9000002C9200002C9400002C9600002C9800002C9A00002C9C00002C9E
00002CA000002CA200002CA400002CA600002CA800002CAA00002CAC00002CAE00002CB000002CB200002CB400002CB6
00002CB800002CBA00002CBC00002CBE00002CC000002CC200002CC400002CC600002CC800002CCA00002CCC00002CCE
00002CD000002CD200002CD400002CD600002CD800002CDA00002CDC00002CDE00002CE000002CE200002CEB00002CED
00002CF2000010A0000010A1000010A2000010A3000010A4000010A5000010A6000010A7000010A8000010A9000010AA
000010AB000010AC000010AD000010AE000010AF000010B0000010B1000010B2000010B3000010B4000010B5000010B6
000010B7000010B8000010B9000010BA000010BB000010BC000010BD000010BE000010BF000010C0000010C1000010C2
000010C3000010C4000010C5000010C7000010CD0000A6400000A6420000A6440000A6460000A6480000A64A0000A64C
0000A64E0000A6500000A6520000A6540000A6560000A6580000A65A0000A65C0000A65E0000A6600000A6620000A664
0000A6660000A6680000A66A0000A66C0000A6800000A6820000A6840000A6860000A6880000A68A0000A68C0000A68E
0000A6900000A6920000A6940000A6960000A6980000A69A0000A7220000A7240000A7260000A7280000A72A0000A72C
0000A72E0000A7320000A7340000A7360000A7380000A73A0000A73C0000A73E0000A7400000A7420000A7440000A746
0000A7480000A74A0000A74C0000A74E0000A7500000A7520000A7540000A7560000A7580000A75A0000A75C0000A75E
0000A7600000A7620000A7640000A7660000A7680000A76A0000A76C0000A76E0000A7790000A77B0000A77E0000A780
0000A7820000A7840000A7860000A78B0000A7900000A7920000A7C40000A7960000A7980000A79A0000A79C0000A79E
0000A7A00000A7A20000A7A40000A7A60000A7A80000A7B40000A7B60000A7B80000A7BA0000A7BC0000A7BE0000A7C0
0000A7C20000A7C70000A7C90000A7D00000A7D60000A7D80000A7F50000A7B3000013A0000013A1000013A2000013A3
000013A4000013A5000013A6000013A7000013A8000013A9000013AA000013AB000013AC000013AD000013AE000013AF
000013B0000013B1000013B2000013B3000013B4000013B5000013B6000013B7000013B8000013B9000013BA000013BB
000013BC000013BD000013BE000013BF000013C0000013C1000013C2000013C3000013C4000013C5000013C6000013C7
000013C8000013C9000013CA000013CB000013CC000013CD000013CE000013CF000013D0000013D1000013D2000013D3
000013D4000013D5000013D6000013D7000013D8000013D9000013DA000013DB000013DC000013DD000013DE000013DF
000013E0000013E1000013E2000013E3000013E4000013E5000013E6000013E7000013E8000013E9000013EA000013EB
000013EC000013ED000013EE000013EF0000FF210000FF220000FF230000FF240000FF250000FF260000FF270000FF28
0000FF290000FF2A0000FF2B0000FF2C0000FF2D0000FF2E0000FF2F0000FF300000FF310000FF320000FF330000FF34
0000FF350000FF360000FF370000FF380000FF390000FF3A000104000001040100010402000104030001040400010405
000104060001040700010408000104090001040A0001040B0001040C0001040D0001040E0001040F0001041000010411
00010412000104130001041400010415000104160001041700010418000104190001041A0001041B0001041C0001041D
0001041E0001041F0001042000010421000104220001042300010424000104250001042600010427000104B0000104B1
000104B2000104B3000104B4000104B5000104B6000104B7000104B8000104B9000104BA000104BB000104BC000104BD
000104BE000104BF000104C0000104C1000104C2000104C3000104C4000104C5000104C6000104C7000104C8000104C9
000104CA000104CB000104CC000104CD000104CE000104CF000104D0000104D1000104D2000104D30001057000010571
00010572000105730001057400010575000105760001057700010578000105790001057A0001057C0001057D0001057E
0001057F000105800001058100010582000105830001058400010585000105860001058700010588000105890001058A
0001058C0001058D0001058E0001058F000105900001059100010592000105940001059500010C8000010C8100010C82
00010C8300010C8400010C8500010C8600010C8700010C8800010C8900010C8A00010C8B00010C8C00010C8D00010C8E
00010C8F00010C9000010C9100010C9200010C9300010C9400010C9500010C9600010C9700010C9800010C9900010C9A
00010C9B00010C9C00010C9D00010C9E00010C9F00010CA000010CA100010CA200010CA300010CA400010CA500010CA6
00010CA700010CA800010CA900010CAA00010CAB00010CAC00010CAD00010CAE00010CAF00010CB000010CB100010CB2
000118A0000118A1000118A2000118A3000118A4000118A5000118A6000118A7000118A8000118A9000118AA000118AB
000118AC000118AD000118AE000118AF000118B0000118B1000118B2000118B3000118B4000118B5000118B6000118B7
000118B8000118B9000118BA000118BB000118BC000118BD000118BE000118BF00016E4000016E4100016E4200016E43
00016E4400016E4500016E4600016E4700016E4800016E4900016E4A00016E4B00016E4C00016E4D00016E4E00016E4F
00016E5000016E5100016E5200016E5300016E5400016E5500016E5600016E5700016E5800016E5900016E5A00016E5B
00016E5C00016E5D00016E5E00016E5F0001E9000001E9010001E9020001E9030001E9040001E9050001E9060001E907
0001E9080001E9090001E90A0001E90B0001E90C0001E90D0001E90E0001E90F0001E9100001E9110001E9120001E913
0001E9140001E9150001E9160001E9170001E9180001E9190001E91A0001E91B0001E91C0001E91D0001E91E0001E91F
0001E9200001E92102000053000000730000013002000046000000660200004600000069020000460000006C03000046
000000660000006903000046000000660000006C02000053000000740200005300000074020005350000058202000544
000005760200054400000565020005440000056B0200054E00000576020005440000056D020002BC0000004E03000399
0000030800000301030003A500000308000003010200004A0000030C0200004800000331020000540000030802000057
0000030A020000590000030A02000041000002BE020003A500000313030003A50000031300000300030003A500000313
00000301030003A500000313000003420200039100000342020003970000034203000399000003080000030003000399
00000308000003010200039900000342030003990000030800000342030003A50000030800000300030003A500000308
00000301020003A100000313020003A500000342030003A50000030800000342020003A90000034200001F8800001F89
00001F8A00001F8B00001F8C00001F8D00001F8E00001F8F00001F8800001F8900001F8A00001F8B00001F8C00001F8D
00001F8E00001F8F00001F9800001F9900001F9A00001F9B00001F9C00001F9D00001F9E00001F9F00001F9800001F99
00001F9A00001F9B00001F9C00001F9D00001F9E00001F9F00001FA800001FA900001FAA00001FAB00001FAC00001FAD
00001FAE00001FAF00001FA800001FA900001FAA00001FAB00001FAC00001FAD00001FAE00001FAF00001FBC00001FBC
00001FCC00001FCC00001FFC00001FFC02001FBA00000345020003860000034502001FCA000003450200038900000345
02001FFA000003450200038F00000345030003910000034200000345030003970000034200000345030003A900000342
00000345";
return t;
}

}


static if (size_t.sizeof == 8)
{
//2080 bytes
enum lowerCaseTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000070",
cast(immutable size_t[]) x"
000000000000010000000000000001400000000000002500",
cast(immutable size_t[]) x"
040203020202010008070202060202050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000003000200010000000600060005000400060006000600060006000600060006
000800060006000700060006000600060006000600060006000C000B000A000900060006000E000D000600060006000F
000600060006000600060006001100100006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006001300120006000600140006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600150006000600060016000600060006
000600060006000600190006001800170006000600060006000600060006001A00060006000600060006000600060006
000600060006001B00060006000600060006000600060006000600060006000600060006000600060006001C00060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
0006000600060006000600060006000600060006000600060020001F001E001D00060006000600060021000600060006
000600060006002200060006000600060006000600230006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
0006000600060006000600060006000600060006000600060006000600060006000000000000000007FFFFFE00000000
0420040000000000FF7FFFFF8000000055AAAAAAAAAAAAAAD4AAAAAAAAAAAB55E6512D2A4E243129AA29AAAAB5555240
93FAAAAAAAAAAAAAFFFFFFFFFFFFAA8501FFFFFFFFEFFFFF0000001F0000000300000000000000003C8A000000000020
FFFFF00000010000192FAAAAAAE37FFFFFFF000000000000AAAAAAAAFFFFFFFFAAAAAAAAAAAAA802AAAAAAAAAAAAD554
0000AAAAAAAAAAAAFFFFFFFF0000000000000000000001FF000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000F7FFFFFFFFFF0000
0000000000000000000000000000000000000000000000003F0000000000000000000000000000000000000000000000
00000000000001FF0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000000000000000
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABFEAAAAAAAAAAAAAAAAAAAAA00FF00FF003F00FF3FFF00FF00FF003F
40DF00FF00FF00FF00DC00FF00CF00DC00000000000000008002000000000000000000001FFF00000000000000000000
321080000008C400FFFF0000000043C00000000000000010000000000000000000000000000000000000000000000000
0000000000000000000003FFFFFF0000FFFF0000000000003FDA1562FFFFFFFFAAAAAAAAAAAAAAAA0008501AAAAAAAAA
000020BFFFFFFFFF000000000000000000000000000000000000000000000000000000000000000000002AAAAAAAAAAA
000000003AAAAAAA0000000000000000AAABAAA80000000095FFAAAAAAAAAAAAAAA082AAAABA50AA075C000002AA050A
FFFF000000000000FFFF03FFF7FFFFFFFFFFFFFFFFFFFFFF00000000000000000000000000F8007F0000000000000000
0000000000000000000000000000000000000000000000000000000007FFFFFE00000000000000000000000000000000
FFFFFF0000000000000000000000FFFF00000000000000000FFFFFFFFF00000000000000000000000000000000000000
1BFBFFFBFF80000000000000000000000000000000000000000000000000000007FDFFFFFFFFFFB90000000000000000
0000000000000000000000000000000000000000000000000007FFFFFFFFFFFF00000000000000000000000000000000
000000000000000000000000FFFFFFFF0000000000000000FFFFFFFF0000000000000000000000000000000000000000
000FFFFFFC000000000000FFFFDFC000EBC000000FFFFFFCFFFFFC000000FFEF00FFFFFFC000000F00000FFFFFFC0000
FC000000FFFFFFC0FFFFC000000FFFFF0FFFFFFC000000FF0000FFFFFFC000000000003FFFFFFC00F0000003F7FFFFFC
FFC000000FDFFFFFFFFF0000003F7FFFFFFFFC000000FDFF0000000000000BF7000007E07FFFFBFF0000000000000000
00000000000000000000000000000000FFFF00000000000000003FFFFFFFFFFF00000000000000000000000000000000
FFFFFFFC00000000000000000000000F0000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000",
);
//1856 bytes
enum upperCaseTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000070",
cast(immutable size_t[]) x"
000000000000010000000000000001400000000000001E00",
cast(immutable size_t[]) x"
040203020202010008070202060202050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000003000200010000000600060005000400060006000600060006000600060006
000800060006000700060006000600060006000600060006000B000A0006000900060006000C0006000600060006000D
0006000600060006000600060006000E0006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
0006000600060006000600060006000600060006000600060010000F0006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060011000600060006
000600060006000600060006001300120006000600060006000600060006001400060006000600060006000600060006
000600060006001500060006000600060006000600060006000600060006000600060006000600060006001600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006001A00190018001700060006000600060006000600060006
0006000600060006000600060006000600060006001B0006000600060006000600060006001C00060006000600060006
000600060006000600060006000600060006000600060006000600060006000600060006000600060006000600060006
000600060006000600060006000600060006000600060006000600060006000600000000000000000000000007FFFFFE
0000000000000000000000007F7FFFFFAA555555555555552B555555555554AA11AED2D5B1DBCED655D255554AAAA490
6C05555555555555000000000000557A0000000000000000000000000000000000000000000000008045000000000000
00000FFBFFFED740E6905555551C80000000FFFFFFFFFFFF555555550000000055555555555554015555555555552AAB
FFFE55555555555500000000007FFFFF0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000FFFFFFFF0000000000000000000020BF
00000000000000000000000000000000FFFFFFFF00000000003FFFFFFFFFFFFF00000000000000000000000000000000
E7FFFFFFFFFF000000000000000000005555555555555555555555555555555555555555401555555555555555555555
FF00FF003F00FF000000FF00AA003F000F000000000000000F001F000F000F00C00F3D503E2738840000FFFF00000020
0000000000000008000000000000000000000000000000000000000000000000FFC0000000000000000000000000FFFF
0000FFFFFFFFFFFFC025EA9D000000005555555555555555000428055555555500000000000000000000155555555555
0000000005555555000000000000000055545554000000006A00555555555555555F7D555545285500200000014102F5
07FFFFFE00000000000000000000000000000000000000000000000000000000000000FFFFFFFFFF0000000000000000
FFFF00000000000000000000000FFFFF0000000000000000F7FF000000000000000000000037F7FF0000000000000000
000000000000000000000000000000000007FFFFFFFFFFFF000000000000000000000000000000000000000000000000
FFFFFFFF000000000000000000000000000000000000000000000000FFFFFFFF00000000000000000000000000000000
FFF0000003FFFFFFFFFFFF0000003FFF003FDE64D0000003000003FFFFFF00007B0000001FDFE7B0FFFFF0000001FC5F
03FFFFFF0000003F00003FFFFFF00000F0000003FFFFFF00FFFF0000003FFFFFFFFFFF00000003FF07FFFFFC00000001
001FFFFFF000000000007FFFFFC00000000001FFFFFF0000000000000000040000000003FFFFFFFF0000000000000000
00000000000000000000000000000000FFFF000000000000FFFF03FFFFFF03FF00000000000003FF0000000000000000
0000000000000000000000000000000000000000000000000000000000000000",
);
//11648 bytes
enum simpleCaseTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000140",
cast(immutable size_t[]) x"
0000000000000100000000000000048000000000000011C0",
cast(immutable size_t[]) x"
040203020202010007020202060202050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
02020202020202020202020202020202000300020001000000070006000500040000000A00090008000D000C000B0000
00110010000F000E00000014001300120000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001600150000000000000000000000000000000000000000001800170000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000019000000000000001B001A0000001F001E001D001C002300220021002000000000000000000000002600250024
000000000000000000000000000000000028002700000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000002C002B002A0029000000000000002D
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000002F002E0000003300320031003000000000000000000000000000000000
000000000000000000000035003400000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000360000000000000000000000000000000000000000000000000000000000380037
0000000000000000000000000000000000000000000000000000000000000000003C003B003A00390000003E003D0000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0040003F0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
004200410000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000004300000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000004500440000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
08E50080091DFFFF090B00BD031905FB0B2B006B09D00B9A0AD200E0082606C705D1059404670838019301800A330785
FFFF02F904D9051EFFFFFFFFFFFFFFFF08E60081091EFFFF090C00BE031A05FC0B2C006C09D10B9B0AD300E1082706C8
05D2059504680839019401810A340786FFFF02FA04DA051FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0965FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0856011805AE077B03DB025709BB0AD80B4D0128038F0559031500470397062B
029F008608DD05F3FFFF00CB058B071104FB021309820AE208EF0A45088007010857011905AF077C03DC025809BC0AD9
0B4E01290390055A031600480398062C02A0008708DE05F4FFFF00CC058C071204FC021409830AE3036A0A4608810702
04CC04CB09DD09DC0B580B5701CD01CC07320731020001FF08AF08AE00CE00CD06140613005C005B0620061F00520051
0663066202B202B108010800015A015907E707E60B890B880A620A6104BE04BD0593059201B401B3096D096C021C021B
08790878FFFFFFFF031C031B0939093806FD00950094FFFF081C008F008E06FE07CA0B310B30081D0A230121012007CB
04DC04DBFFFF0A240B660B6507C507C40162016106BD06BC02AA02A906830682000D000C060A06090070006F06410640
00D200D1083B083A01DF01DE0728072701A201A10B810B800A520A5104C604C50566056507CF07CE09FB09FA09770976
002106260625036B05D305FE05FD00220B180B170688069102750526026A026901AF07FA0B3902760AC20769FFFF01B0
052008B508B40A4709900B090229027BFFFF04C900970096032B00D308EB00AA062E062D0019001803CF055301050104
FFFFFFFF09A603D0075907EE01690168052C077F01A7075A0AC003020301052DFFFFFFFF0256025501ADFFFF0920091F
FFFFFFFFFFFFFFFF05E0092B092A09290AE50AE405E205E1079804FE04FD0AE6076B0A5E0A5D07990A8004420441076C
055102FE02FD0A810B540B53076A0552016B016A0655065402C002BF05FA05F900F200F1064706460570056F06A906A8
0ABC0ABB0ABAFFFF01AE022A0748074701EB01EA0B6E0B6D09710970041A0419059F059E0841084008BE08BD0940093F
0898089703AE03AD041C041B0B120B110A8F0A8E040B040A0531053002B402B307F107F0016D016C084B084A05B705B6
06CE06CDFFFF00D40B830B8200640063073E073D00C200C1088A08890079007806F506F4057A0579FFFFFFFFFFFFFFFF
07960A0EFFFFFFFF095D07BE04CA07970692014C014B028B09D509D402D3041D032403230926092509F709F609540953
0689021D0B1F040207FB0B3AFFFF05270A48FFFF0AC3FFFFFFFFFFFFFFFF0941027CFFFF0B2E0521FFFF00B50B76FFFF
063C00F90B0A099100ABFFFFFFFF098AFFFF08EC05A0FFFFFFFFFFFF032CFFFFFFFFFFFFFFFFFFFFFFFFFFFF0031FFFF
09A70516FFFF05540350FFFFFFFFFFFF078001A8041E07EFFFFFFFFFFFFF02D4FFFF0AC1FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF070F0092FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0972FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01CF01CE04A504A401F601F5FFFFFFFF
0A37FFFFFFFFFFFF0382FFFF086E0122FFFFFFFFFFFFFFFFFFFF0299FFFFFFFFFFFF00EB082806D709CA0BA0FFFF07F2
017C04AE04B30386016601D20ABD0A170B070B320973059A0106039B081E0966068AFFFF062F0B7300AC03450607032D
009A084C06FA08FD00EC082906D8029A017D04AF04B409CE016701D30ABE0A180B080B330974059B0107039C081F0967
068C068B06300B7400AD03460608032E009B084D06FB08FE097809CB0BA107F3FFFFFFFF059C04B009790B750347FFFF
04A904A8018C018B085F085E0A890A88098909880582058103BC03BB09DF09DE08C208C102F602F503630362044A0449
038301D606310B340A84FFFF0ABF059D05DB05DA01D70A85086F01230A38FFFF096E0291051A04F105AC0090098E0AF5
08F700A008F906FF0391009C031D05EF03C105400B45071F0AB6027D03D306E704B70126086A0A8C01B90A1407AA080A
02DF0A73051107EC015B02450ACE09A40B5B093A0664057B005303C3062109F003C205410B4607200AB7027E03D406E8
04B80127086B0A8D01BA0A1507AB080B02E00A74051207ED015C02460ACF09A50B5C093B0665057C005403C4062209F1
096F0292051B04F205AD0091098F0AF608F800A108FA07000392009D031E05F005CD05CC0AA30AA20B4A0B4902440243
03E603E501FA01F90859085809B609B508E008DF05A305A2039A039908C608C509370936088808870418041703DE03DD
FFFFFFFF0B220B21FFFFFFFFFFFFFFFF05A705A6FFFFFFFF00E600E5082B082A005E005D06D006CF00050004061C061B
00CA00C90730072F0568056708AD08AC010B010A06B506B401A0019F04F004EF045804570A110A100B700B6F07BB07BA
040704060831083008EE08ED09BF09BE08DA08D90326032503B803B709F909F8021F06C306C202DB023705D905D80220
04A00A220A21023802DC0793079204A1055C055B023C023B06E406E302620261062806270058005706790678001E001D
0677067601130112056A05690248024707A907A8011B011A07780777049B049A046A04690A910A900B000AFF02360235
042C042B038C038B08CA08C9090E090D092409230338033703D803D70A010A000A950A940825082401CB01CA079B079A
02C402C30A1C0A1B0171017004EC04EB05BB05BA06A506A4002C002B064D064C003800370639063800B400B30B6A0B69
0ACD0ACC0738073704EE04ED088E088D0227017E0BA2FFFF0A9A0293071904B50589047D08520AC401330B6306D503A7
03030352067C03840135092C067E09030B27094906D905B405010524085A0A06022F0A3D07150295FFFF078705DC0184
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0228017F0BA3FFFF0A9B0294071A04B6058A047E08530AC501340B6406D603A8
03040353067D03850136092D067F09040B28094A06DA05B505020525085B0A0702300A3E07160296FFFF078805DD0185
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF050D03EB082209B701B10B8E06F601F0
01B7049804850241046D087C0A9E0A9C0AF706A007FE056B04310648039300F708C3068F090902D909170672033B014F
03D1084E09EA0B4F0AA8FFFF02E10108FFFFFFFFFFFFFFFFFFFFFFFF0A55FFFF03FC06990AE7015309320668042D05C4
03C905F10317002D09270B3B09F20025085C06B0055D008C0429088209AB0AE00A0A0761021504FF071304D3024901DA
081809980AAE02A106EC03CB058703E1FFFF039F011E0B3506950905000EFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
036000C3043305D60934006D081606C009FC0B29037408930335007E08BB06030A5B02CD01F70A6D055F0137095506B8
07C805750B7E07DA075D09EE04DF048B064E037802E7044F06A209A001410850089B09190041045D0634031300450B9C
0B6709E0023309B10739054300820271077D07170A0C0ADE04B908B601E4052803AF06DB09FE028908060B2305B801FB
035A067A034C05DE092E068602AD0139038D06230B84004BFFFFFFFF04390061038E06240B85004CFFFFFFFF043A0062
05130A1606E9054205CE093C0A760A75FFFFFFFFFFFF069DFFFFFFFFFFFFFFFF03FD069A0AE8015409330669042E05C5
03CA05F20318002E09280B3C09F30026085D06B1055E008D042A088309AC0AE10A0B076202160500071404D4024A01DB
081909990AAF02A206ED03CC058803E2FFFF03A0011F0B3606960906000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF0507FFFFFFFFFFFF030DFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0B92FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
04C804C706F306F20B870B860776077504740473080308020260025F0985098409080907057805770334033308F408F3
0914091302CC02CB03D603D5045C045B07C707C60B040B030A1E0A1D04DE04DD05CB05CA01AC01AB072C072B0240023F
083F083E00DB00DA061606150060005F060E060D001300120694069302A402A306B706B6015E015D079F079E05B105B0
049304920A640A630B5A0B590208020703E003DF020E020D087B087A09460945090008FF03550354037D037C08D208D1
09620961088508840204020303DA03D901B601B50B3E0B3D0A420A410480047F05C305C207DD07DC014E014D08110810
02AC02AB065D065C004A00490633063200890088061E061D053F053E08A308A2020602050736073501D101D004C204C1
0440043F0A5A0A59FFFFFFFF07D107D00886FFFFFFFFFFFFFFFF08F0FFFFFFFF08E808E70308030703A403A309C909C8
0ADD0ADC0254025307E507E40430042F07B107B00A260A250722072104C404C305460545021A02190704070300990098
05F805F700A300A206530652004E004D0AFA0AF9012501240533053202F402F3077207710117011607B507B404970496
03F703F60A700A6F0B7D0B7C024E024D043C043B03BE03BD08F608F508D408D309220921035F035E054C054B09950994
0A780A7708A508A4019A01990452045101D501D40A4A0A4900EE00ED06E006DF0A02003503A90605055500F3094F0B41
0A03003603AA0606055600F409500B42022B019D0A35076DFFFFFFFF06F804CD022C019E0A36076EFFFFFFFF06F904CE
0016030505FF039D01720870068008E9001703060600039E01730871068108EA01820AC80A3902E501E0078E04AA01A3
01830AC90A3A02E601E1078F04AB01A4035605E3034E0074FFFFFFFF08BF000A035705E4034F0075FFFFFFFF08C0000B
0A5FFFFF01E6FFFF076FFFFF04D5FFFF0A60FFFF01E7FFFF0770FFFF04D6FFFF0656043D02C5099205E50B8C002903B3
0657043E02C6099305E60B8D002A03B409D602D502ED0447041F05CF02090A710A570AB0000808E3FFFFFFFF06190321
03A5028509680B1B08F1012A02C1066603A6028609690B1C08F2012B02C2066709F4028F03B906E105B201FD07F40B13
09F5029003BA06E205B301FE07F50B1400DC09B30B7A09E4007C0281073F059000DD09B40B7B09E5007D028207400591
097CFFFF066A0370FFFFFFFFFFFFFFFF02EE0448066B0371FFFF0975FFFF097D00F5FFFFFFFFFFFFFFFFFFFFFFFFFFFF
020A0A7209D702D6FFFFFFFFFFFF00F60387FFFF035C00BBFFFFFFFFFFFFFFFF042005D0035D00BCFFFFFFFFFFFFFFFF
09CFFFFF01C30176FFFFFFFF029BFFFF0A580AB101C40177FFFFFFFFFFFF029C03BFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
061A0322000908E4FFFFFFFFFFFF03C0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF06FCFFFFFFFF
09BD0B2DFFFFFFFFFFFFFFFFFFFFFFFFFFFF0951FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0952FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF045904940A7907B20A4309DA01780767070902F101EC0534082E044D00DE0AEB
045A04950A7A07B30A4409DB01790768070A02F201ED0535082F044E00DF0AEC0874FFFFFFFFFFFFFFFFFFFFFFFF0875
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0596037AFFFFFFFF0745081A00A60B25089909CC052A03FE
07B604870A92020F0AA0071D018D022105180860029D01020943065E097A0B1D00A70B260597037B052B03FF0746081B
0A930210089A09CD018E022207B70488029E01030AA1071E097B0B1E05190861FFFFFFFF0944065FFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF010C0AED065803C7050B044306B2082C
0AB201C8086C0A4F0225020B0753049C027705380B5F070B098600A8040C081205A4002F093D070508CF003B095B063A
0876015103A1067003E705D40B0B06840AA40114040407D2053A01BB02E304BB010D0AEE065903C8050C044406B3082D
0AB301C9086D0A500226020C0754049D027805390B60070C098700A9040D081305A50030093E070608D0003C095C063B
0877015203A2067103E805D50B0C06850AA50115040507D3053B01BC02E404BC030E063D003A003909D807BF0A0F0032
0265037F037E09D9040305A10B2002660A1A0A19FFFF021EFFFF074E074DFFFFFFFFFFFFFFFFFFFF028C095EFFFFFFFF
033A033902A602A50912091108CC08CB02B902B8037303720470046F099709960B560B5507FD07FC04B204B1077A0779
01E901E80A660A6500D600D5056E056D00660065072A07290020001F08A908A8005A0059062A0629026402630B1A0B19
0190018F06BF06BE05BD05BC07C107C00A3C0A3B0750074F0280027F04E204E103AC03AB0B020B010958095704720471
0328032708D608D509C109C00931093008330832057E057D042204210AAD0AAC0A130A1202180217070E070D01A601A5
08210820010F010EFFFFFFFFFFFFFFFF0358FFFFFFFFFFFFFFFF032A0329035906D206D1FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF050E03EC082309B801B20B8F06F701F101B8049904860242046E087D0A9F0A9D
0AF806A107FF056C04320649039400F808C40690090A02DA09180673033C015003D2084F09EB0B500AA9FFFF02E20109
FFFFFFFFFFFFFFFFFFFFFFFF0A56FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0A200A1F01D901D8079D079C0A990A98069F069E0140013F04F604F502BB02BA06430642003E003D064B064A05BF05BE
073C073B007700760B6C0B6B0040003F0784078304EA04E9089208910AD50AD40AC70AC602C802C7FFFFFFFF017B017A
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02980297051505140A050A04094C094B
0A830A8207AF07AE018A01890A400A3F00AF00AE074A0749052F052E088C088B0034003305EC05EBFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00070006FFFFFFFF00C600C508AB08AA
00E800E706F106F0006800670610060F01580157FFFFFFFF0B4C0B4B0835083409C309C20504050302B002AF0A2C0A2B
0960095F087F087E0396039509810980085508540B5E0B5D0330032F0312031105A905A802120211098D098C04110410
08690868051D051C0A280A27023E023D066D066C00FF00FE06A706A60B100B0F0661066001560155069C069B02EA02E9
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04D707A307A2FFFF0B720B71050804D809BA09B905860585084508440AD10AD0
004FFFFFFFFFFFFFFFFFFFFF0B770050024C024B0B0E0B0D03B203B1FFFF0790049F049E07D907D809E309E203FB03FA
027A02790510050F011D011C080F080E094200B607080707FFFF00FA098B0B2F0B61009303510710009F009E05EA05E9
013E013D06E606E5028E028D0550054F0436043508050804075B0B9305170791FFFF01750174075CFFFFFFFFFFFFFFFF
FFFFFFFF02F802F70310030FFFFFFFFFFFFFFFFF0AF20AF1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF09ED09ECFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0B62FFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
036100C4043405D70935006E081706C109FD0B2A037508940336007F08BC06040A5C02CE01F80A6E05600138095606B9
07C905760B7F07DB075E09EF04E0048C064F037902E8045006A309A101420851089C091A0042045E0635031400460B9D
0B6809E1023409B2073A054400830272077E07180A0D0ADF04BA08B701E5052903B006DC09FF028A08070B2405B901FC
035B067B034D05DF092F068702AE013AFFFFFFFFFFFFFFFFFFFF080D080CFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02670AF309C4FFFF094D064405AA01430B780773074B04CF040004F707C2013B
02EF03ED054D0259019707A40A860AB4FFFF0453060108C7FFFFFFFFFFFFFFFF02680AF409C5FFFF094E064505AB0144
0B790774074C04D0040104F807C3013C02F003EE054E025A019807A50A870AB5FFFF0454060208C8FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05C0075F07510A6709AF0AAA07EA012F02DD099E0AFB0959014507D404E304F9
08CD0636047900E9036C072302CF0043025D0549040802D1038006740B8A0147047B078C0ACA0A7B0B51078102310483
05C1076007520A6809B00AAB07EB013002DE099F0AFC095A014607D504E404FA08CE0637047A00EA036D072402D00044
025E054A040902D2038106750B8B0148047C078D0ACB0A7C0B52078202320484FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF074107D60AD60A4B0B05099A02230571
07940B3F09C6022D0763048104A2024F06AA034305C6033106C9028301BF0B37065003C5028709AD060B08D7002702EB
08B00A2F0A2D0755FFFFFFFFFFFFFFFF074207D70AD70A4C0B06099B0224057207950B4009C7022E0764048204A30250
06AB034405C7033206CA028401C00B38065103C6028809AE060C08D8002802EC08B10A300A2E0756FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
026B00A40463088F025101100AE906ACFFFF00B709470B43090100B908FB05E701310B4707CC080802FF03E305610836
FFFF02A70423058304E5047707650A29FFFF03B5063E0A310890FFFF06D302C906AD026C00A504640B44025201110AEA
05E8FFFF00B809480809090200BA08FC083701320B4807CD0584030003E405620A2AFFFF02A804240A3204E604780766
02CAFFFF03B6063FFFFFFFFFFFFF06D401490191048904D1057301E207E20B94044505C803E9069703090A4D0A8A07E8
00C7090F0B900915008A0895072D03410100096A08660461007A0366071B089D033F04BF05220201046507A00ADA0A69
053C046B083C096301BD0B9606EA0239FFFF066E09E602FBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
014A0192048A04D2057401E307E30B95044605C903EA0698030A0A4E0A8B07E900C809100B910916008B0896072E0342
0101096B08670462007B0367071C089E034004C005230202046607A10ADB0A6A053D046C083D096401BE0B9706EB023A
FFFF066F09E702FCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0B9807A601DC057F04E7049001C101A908420733036E00CF045F0848097E00EF068D0872015F0B15061108E102D7034A
08460AB80A6B036406BA03CD055704550B9907A701DD058004E8049101C201AA08430734036F00D004600849097F00F0
068E087301600B16061208E202D8034B08470AB90A6C036506BB03CE05580456FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
031F072509A20084040E05F50412025B056306DD091B00690A08061703480010019507AC04AC047501EE0AA606EE030B
0AEF06CB08A604F3050507B809E80A530320072609A30085040F05F60413025C056406DE091C006A0A09061803490011
019607AD04AD047601EF0AA706EF030C0AF006CC08A704F4050607B909E90A5408B2005503F8065A0437002308DB05ED
000000BF0376059805090536081407430757019B000207BC0A96033D044B04A60547016E02730AFD09D20B9E099C06AE
03F9065B0014036808DC05EE08B30056037705990438002408150744000100C0000307BD050A0537044C04A70758019C
02740AFE0A97033E099D06AF0548016F0015036909D30B9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//11648 bytes
enum fullCaseTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000140",
cast(immutable size_t[]) x"
0000000000000100000000000000048000000000000011C0",
cast(immutable size_t[]) x"
040203020202010007020202060202050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
02020202020202020202020202020202000300020001000000070006000500040000000A00090008000D000C000B0000
00110010000F000E00000014001300120000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001600150000000000000000000000000000000000000000001800170000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000019000000000000001B001A0000001F001E001D001C002300220021002000000000000000000000002600250024
000000000000000000000000000000000028002700000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000002C002B002A0029000000000000002D
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000002F002E0000003300320031003000000000000000000000000000000000
000000000000000000000035003400000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000360000000000000000000000000000000000000000000000000000000000380037
0000000000000000000000000000000000000000000000000000000000000000003C003B003A00390000003E003D0000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0040003F0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
004200410000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000004300000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000004500440000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0984008A09BEFFFF09AC00C7034406580BF400740A7C0C6A0B9100EF08B10736062905E604A608C301AC01990AE607FE
FFFF032005210569FFFFFFFFFFFFFFFF0985008B09BFFFFF09AD00C8034506590BF500750A7D0C6B0B9200F008B20737
062A05E704A708C401AD019A0AE707FFFFFF03210522056AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0A0BFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF08E3012E060507F2041202770A640B9A0C18013E03C105A60340004E03C9068A
02C20090097C0650FFFF00D505DA0781054302300A2A0BA70BA20AF90911077108E4012F060607F3041302780A650B9B
0C19013F03C205A70341004F03CA068B02C30091097D0651FFFF00D605DB0782054402310A2B0BA8039B0AFA09120772
051205110A890A880C230C2201E601E507A607A5021D021C0946094500D800D70673067200650064067F067E005B005A
06C406C302D802D70889088801730172086808670C550C540B180B170504050305E505E401CD01CC0A140A1302390238
090A0909FFFF03F60347034609DD09DC076D009F009EFFFF08A700990098076E08440BFA0BF908A80AD6013701360845
05240523031A0AD70C310C30083F083E017B017A072A072902D002CF06EB06EA000F000E066906680079007806A0069F
00DC00DB08C608C501FB01FA079C079B01BB01BA0C4D0C4C0B080B07050C050B05B305B2084B084A0AAB0AAA0A1E0A1D
002506850684039C062B065B065A00260BE00BDF06F006F902950571028A028901C808810C0202960B8107E0FFFF01C9
056B094F094E0AFB0A380BD00247029BFFFF050F00A100A0035600DD098A00B4068D068C001D001C040605A001180117
FFFFFFFF0A4E040707D0087401820181057707F801C007D10B7F032903280578FFFFFFFF0276027501C6FFFF09C109C0
FFFFFFFFFFFFFFFF063B09CF09CE09CD0BAA0BA9063D063C0812054605450BAB07E20B140B1308130B39047D047C07E3
059E032503240B3A0C1F0C1E07E1059F0184018306B606B502E602E5065706560101010006A606A505BD05BC07140713
0B7B0B7A0B79092801C7024807BF07BE020702060C390C380A180A170455045405F105F008CE08CD095B095A09E409E3
092D092C03E103E0045704560BD90BD80B480B4704460445057C057B02DA02D9087708760186018508D808D7060F060E
073D073CFFFF00DE0C4F0C4E006D006C07B407B300CC00CB091B091A008200810765076405C905C8FFFFFFFFFFFFFFFF
08100ABEFFFFFFFF0A0308380510081106FA0165016402AD0A810A8002FA0458034F034E09C709C60AA70AA609FA09F9
06F1023A0BE8043B08820C03FFFF05720AFCFFFF0B82FFFFFFFFFFFFFFFF09E5029CFFFF0BF7056CFFFF00BF0C41FFFF
069B01090BD10A3900B5FFFFFFFF0A32FFFF098B05F7FFFFFFFFFFFF0357FFFFFFFFFFFFFFFFFFFFFFFFFFFF0035FFFF
0A4F055FFFFF05A10380FFFFFFFFFFFF07F901C104590875FFFFFFFFFFFF02FBFFFF0B80FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF077F009CFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0A19FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01E801E704E604E502120211FFFFFFFF
0AEAFFFFFFFFFFFF03B3FFFF08FF0138FFFFFFFFFFFFFFFFFFFF02BCFFFFFFFFFFFF00FA08B307460A750C70FFFF0878
019504F104F90883017F01EB0B7C0ACA0BCE0BFB0A1A05EC011903CD08A90A0C06F2FFFF068E0C3E00B6037206660358
00A408D9076A099E00FB08B4074702BD019604F204FA0A79018001EC0B7D0ACB0BCF0BFC0A1B05ED011A03CE08AA0A0D
06F406F3068F0C3F00B703730667035900A508DA076B099F0A1F0A760C710879FFFFFFFF05EE04F30A200C400374FFFF
04EC04EB01A501A408F008EF0B420B410A310A3005D105D003F003EF0A8B0A8A095F095E031D031C0394039304850484
03B401EF06900BFD0B3DFFFF0B7E05EF0631063001F00B3E090001390AEBFFFF0A1502B4056505390603009A0A360BBA
099800AA099A076F03C300A60348064A03F8058D0C0E07900B72029D040A075704FD013C08FB0B4501D20AC408240894
03060B2C055A0872017402650B8D0A4C0C2609DE06C505CA005C03FC06800A9D03F9058E0C0F07910B73029E040B0758
04FE013D08FC0B4601D30AC50825089503070B2D055B0873017502660B8E0A4D0C2709DF06C605CB005D03FD06810A9E
0A1602B50566053A0604009B0A370BBB099900AB099B077003C400A70349064B062506240B5C0B5B0C130C1202640263
041D041C0216021508E608E50A5F0A5E097F097E05F605F503CC03CB0963096209DB09DA091909180453045204150414
FFFFFFFF0BEB0BEAFFFFFFFFFFFFFFFF05FE05FDFFFFFFFF00F500F408B608B500670066073F073E00050004067B067A
00D400D307A407A305B505B409440943011E011D0722072101B901B805380537049504940AC10AC00C3B0C3A08350834
0442044108BC08BB098D098C0A680A67097709760351035003EB03EA0AA90AA8023C073207310302025506330632023D
04E10AD50AD402560303080D080C04E205A905A8025A02590754075302820281068706860061006006E006DF00220021
06DB06DA0129012805B705B602680267082308220131013007EF07EE04DC04DB04A904A80B4A0B490BC50BC402540253
0467046603BE03BD0967096609AF09AE09C509C403650364040F040E0AB10AB00B4E0B4D08B008AF01E401E308150814
02EB02EA0ACF0ACE018A018905340533061306120710070F0030002F06AC06AB003C003B0698069700BE00BD0C350C34
0B8C0B8B07AE07AD0536053509230922024501970C72FFFF0B5302B6078904FB05D804BE08DF0B83014A0C2E074403DA
032A038206E403B5014C09D006E609A40BF009ED0748060C0549056F08E70AB6024D0AF0078502B8FFFF08000637019D
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF024601980C73FFFF0B5402B7078A04FC05D904BF08E00B84014B0C2F074503DB
032B038306E503B6014D09D106E709A50BF109EE0749060D054A057008E80AB7024E0AF1078602B9066208010638019E
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0556042208AD0A6001CA0C5E0766020C
01D004D904C6026104AC090D0B570B550BBC070B088605B8046C06A703C50107096006F709AA030009B806D403680168
040808DB0A970C1A0B61FFFF0308011BFFFFFFFFFFFFFFFFFFFFFFFF0B0BFFFF043507020BAC016C09D606CA0468061C
03FE064C0342003109CB0C040AA1002908EB071D05AA0096046409130A530BA50ABA07D80232054707830519026901F3
08A30A400B6802C4075C040005D60418FFFF03D101340BFE06FD09A60010FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
039100CD046E062E09D8007608A1072F0AAC0BF203A5093B0362008C095806600B1102F402130B2405AC014E09FB0725
084205C40C4A085B07D40A9B052704CC06AD03A9030E048A070D0A48015808DD093009BA004A049A0693033E004C0C6C
0C320A8C02510A5907AF05900094029107F607870ABC0BA004FF09530200057303E2074A0AAE02AB08900C0806100217
039906EE037A063909D2071902D3015003BF06820C500054FFFFFFFF0474006A03C006830C510055FFFFFFFF0475006B
055C0AC60759058F062609E00B2F0B2EFFFFFFFFFFFF0708FFFFFFFFFFFFFFFF043607030BAD016D09D706CB0469061D
03FF064D0343003209CC0C050AA2002A08EC071E05AB0097046509140A540BA60ABB07D9023305480784051A026A01F4
08A40A410B6902C5075D040105D70419FFFF03D201350BFF06FE09A70011FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF0561FFFFFFFFFFFF0338FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0C62FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
050E050D076307620C530C5207ED07EC04B304B2088B088A0280027F0A2D0A2C09A909A805C705C60361036009950994
09B509B402F302F2040D040C04990498084108400BC90BC80AD10AD0052605250623062201C501C407A0079F0260025F
08CA08C900E500E40675067400690068066D066C0015001406FC06FB02C702C607240723017701760819081806080607
04D404D30B1A0B190C250C240225022404170416022B022A090C090B09EA09E909A109A00385038403AE03AD096F096E
0A080A0709160915022102200411041001CF01CE0C070C060AF60AF504C104C0061B061A085E085D01670166089C089B
02D202D106BE06BD005100500692069100930092067D067C058C058B093809370223022207AC07AB01EA01E905080507
047B047A0B100B0F00DF0C5C084D084C0917049E05BA03FAFFFF0BA3FFFFFFFF098709860331033003D603D50A740A73
0B9F0B9E0274027308660865046B046A082B082A0AD90AD807930792050A050905930592023702360774077300A300A2
0655065400AD00AC06B406B3005700560BBF0BBE013B013A057E057D0327032607E907E8012D012C082F082E04D804D7
0430042F0B270B260C490C48026E026D0477047603F203F1099709960971097009C309C20390038F059905980A3D0A3C
0B310B30093A093901B301B2048D048C01EE01ED0B000AFF00FD00FC074F074E0AB2003903DC066405A2010209F30C0A
0AB3003A03DD066505A3010309F40C0B024901B60AE807E4FFFFFFFF07680513024A01B70AE907E5FFFFFFFF07690514
001A032C065C03CF018B090106E80988001B032D065D03D0018C090206E90989019B0B870AEC030C01FC080704ED01BC
019C0B880AED030D01FD080804EE01BD0386063E037E007DFFFFFFFF095C000C0387063F037F007EFFFFFFFF095D000D
0B1508ED0202008507E6072B051B0BEC0B16FFFF0203FFFF07E7FFFF051CFFFF06B7047802EC0A3A06400C5A002D03E6
06B8047902ED0A3B06410C5B002E03E70A8202FC03140482045A062702260B2A0B0D0B6A000A0982FFFFFFFF0678034C
08560B760AC70BE30991014005F2079408570B770AC80BE40992014105F307950AA3063403EC07500609021909C80BDA
0AA4063503ED0751060A021A09C90BDB0B1F0B950C450A9001F7054F07B509470B200B960C460A9101F8055007B60948
0A23048E06CC03A1003F0C43FFFF03360315048306CD03A2FFFF0A1CFFFF0A240AF2084EFFFFFFFF051F0581FFFF0A9F
02270B2B0A8302FDFFFFFFFFFFFF0AF30884091E038D00C5096C064EFFFFFFFF045B0628038E00C6FFFFFFFFFFFFFFFF
0A7A08E901DC018F04E707F402BE0C160B0E0B6B01DD0190FFFFFFFFFFFF02BF06FF0A6BFFFFFFFF06D8091CFFFF05D2
0679034D000B0983FFFFFFFFFFFF0700FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF076CFFFFFFFF
0A660BF6FFFFFFFFFFFFFFFFFFFFFFFFFFFF09F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF09F8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF049604D50B32082C0AF70A86019107DE077903180208057F08B9048800EB0BB0
049704D60B33082D0AF80A87019207DF077A03190209058008BA048900EC0BB10905FFFFFFFFFFFFFFFFFFFFFFFF0906
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05E803ABFFFFFFFF07BC08A500B00BEE092E0A7705750437
083004C80B4B022C0B59078E01A60241056308F102C0011509E706BF0A210BE600B10BEF05E903AC0576043807BD08A6
0B4C022D092F0A7801A70242083104C902C101160B5A078F0A220BE7056408F2FFFFFFFF09E806C0FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF011F0BB206B904020554047E071F08B7
0B6C01E108FD0B050243022807CA04DD029705830C2A077B0A2E00B20447089D05F9003309E10775097400450A010699
0907016A03D306D2041E062C0BD206EC0B5D012A043D0850058501D4030A050101200BB306BA04030555047F072008B8
0B6D01E208FE0B060244022907CB04DE029805840C2B077C0A2F00B30448089E05FA003409E20776097500460A02069A
0908016B03D406D3041F062D0BD306ED0B5E012B043E0851058601D5030B05020339069C003E003D0A8408390ABF0036
028503B003AF0A85043C05F80BE902860ACD0ACCFFFF023BFFFF07C507C4FFFFFFFFFFFFFFFFFFFF02AE0A04FFFFFFFF
0367036602C902C809B309B20969096802DF02DE03A403A304AF04AE0A3F0A3E0C210C20088D088C04F504F407F107F0
020502040B1C0B1B00EE00ED05BF05BE006F006E079E079D002400230940093F0063006206890688028402830BE20BE1
01A901A8072E072D06150614083B083A0AEF0AEE07C707C602A0029F052A052903DF03DE0BC70BC609FE09FD04B104B0
03530352097309720A6A0A6909D509D408BE08BD05CD05CC045D045C0B670B660AC30AC202350234077E077D01BF01BE
08AC08AB01220121FFFFFFFFFFFFFFFF038BFFFFFFFFFFFFFFFF03550354038C07410740FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0557042308AE0A6101CB0C5F0767020D01D104DA04C7026204AD090E0B580B56
0BBD070C088705B9046D06A803C60108096106F809AB030109B906D503690169040908DC0A980C1B0B62FFFF0309011C
FFFFFFFFFFFFFFFFFFFFFFFF0B0CFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0AD30AD201F201F1081708160B520B51070A070901570156053E053D02E102E006A206A10042004106AA06A906170616
07B207B10080007F0C370C360044004307FD07FC05320531092709260B940B930B860B8502EF02EEFFFFFFFF01940193
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02BB02BA055E055D0AB50AB409F009EF
0B3C0B3B0829082801A301A20AFE0AFD00B900B807C107C0057A0579092109200038003706470646FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00090008FFFFFFFF00D000CF09420941
00F700F60761076000710070066F066E01710170FFFFFFFF0C150C1408C008BF0A700A6F054C054B02D602D50ADF0ADE
0A060A050910090F03C803C70A290A2808E208E10C290C28035D035C033D033C060005FF022F022E0A350A34044C044B
08FA08F9056805670ADB0ADA025C025B06CF06CE01140113071207110BD70BD606C206C1016F016E0705070403110310
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF051D081D081CFFFF0C3D0C3C0562051E0A630A6205D505D408D208D10B900B8F
0058FFFFFFFFFFFFFFFFFFFF0C420059026C026B0BD50BD403E503E4FFFF082604E004DF085A08590A8F0A8E04340433
029A02990559055801330132089A089909E600C007780777FFFF010A0A330BF80C2C009D0381078000A900A806450644
015501540756075502B002AF059D059C04710470088F088E07D20C6305600827FFFF018E018D07D3FFFFFFFFFFFFFFFF
FFFFFFFF031F031E033B033AFFFFFFFFFFFFFFFF0BB70BB6FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0A9A0A99FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0C2DFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
039200CE046F062F09D9007708A207300AAD0BF303A6093C0363008D095906610B1202F502140B2505AD014F09FC0726
084305C50C4B085C07D50A9C052804CD06AE03AA030F048B070E0A49015908DE093109BB004B049B0694033F004D0C6D
0C330A8D02520A5A07B005910095029207F707880ABD0BA1050009540201057403E3074B0AAF02AC08910C0906110218
039A06EF037B063A09D3071A02D4015105FB014808C10BCAFFFF0B640B630846FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
09F5FFFFFFFFFFFF035A001604B4069DFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02870BB80A6DFFFF09F106A30601015A0C6607EA07C205150439053F083C0152
03160426059A027901B0081E0B3F0B6EFFFF0490065E0964FFFFFFFFFFFFFFFF02880BB90A6EFFFF09F206A40602015B
0C6707EB07C30516043A0540083D015303170427059B027A01B1081F0B400B6FFFFF0491065F0965FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF061807D607C80B1D0A570B70086D014603040A460BC009FF015C0854052B0541
096A069504BA00F8039D079702F60052027D0596044302F803B106D60C56015E04BC08050B890B340C1C07FA024F04C4
061907D707C90B1E0A580B71086E014703050A470BC10A00015D0855052C0542096B069604BB00F9039E079802F70053
027E0597044402F903B206D70C57015F04BD08060B8A0B350C1D07FB025004C5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF07B808690B980B010BCC0A42025705C0
080E0C100A71024B07DA04C204E3026F07150370061E035E073802A401D80C0006B1042002A90A55066A0978002B0312
094A0AE20AE007CCFFFFFFFFFFFFFFFF07B9086A0B990B020BCD0A43025805C1080F0C110A72024C07DB04C304E40270
07160371061F035F073902A501D90C0106B2042102AA0A56066B0979002C0313094B0AE30AE107CDFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
028B00AE04A00924027101230BAE0717FFFF00C109EB0C0C09A200C3099C064201600C5808480892032E041A05AE08C7
FFFF02CD045E05DC052D04B807DC0ADCFFFF03E806AF0AE40925FFFF074202F00718028C00AF04A10C0D027201240BAF
0643FFFF00C209EC089309A300C4099D08C801610C59084905DD032F041B05AF0ADDFFFF02CE045F0AE5052E04B907DD
02F1FFFF03E906B0FFFFFFFFFFFF0743016201AA04CA051705C201FE08630C64048006200424070603320B030B43086B
00D109B00C6009B600D9092A07A1036E01110A1108F704A40083039707A70932036C0505056D021E04A2081A0B9C0B28
058904AA08CB0A0901D60002075A025DFFFF06D00A930322FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
016301AB04CB051805C301FF08640C65048106210425070703330B040B44086C00D209B10C6109B700DA092B07A2036F
01120A1208F804A50084039807A80933036D0506056E021F04A3081B0B9D0B29058A04AB08CC0A0A01D70003075B025E
FFFF06D10A940323FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0C68082001F505CE052F04D101DA01C208CF07A9039F00E9049C08D50A2600FE06F5090301780BDD0670098002FE0378
08D30B740B2203950727040405A404920C69082101F605CF053004D201DB01C308D007AA03A000EA049D08D60A2700FF
06F6090401790BDE0671098102FF037908D40B750B2303960728040505A50493FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
034A07990A4A008E04490652044D027B05B0074C09BC00720AB80676037C001201AE083604EF04B6020A0B5F075E0334
0BB4073A093D053B054D08320A950B09034B079A0A4B008F044A0653044E027C05B1074D09BD00730AB90677037D0013
01AF083704F004B7020B0B60075F03350BB5073B093E053C054E08330A960B0A094C005E043106BB04720027097A0648
000000C903A705EA05520587089F07BA07CE01B4000608520B4F036A048604E90594018702930BC20A7E0C6E0A44071B
043206BC0018043F097B0649094D005F03A805EB0473002808A007BB000100CA0007085305530588048704EA07CF01B5
02940BC30B50036B0A45071C05950188001904400A7F0C6FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//5600 bytes
enum alphaTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
0000000000000000000000000000002000000000000000D0",
cast(immutable size_t[]) x"
000000000000010000000000000002C00000000000007B00",
cast(immutable size_t[]) x"
06050403030201000E0D0C0B0A09080711100F0303030303141414141414131214141414141414141414141414141414
141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414
141414141414141414141414141414141414141414141414141414141414141414141414141414140000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000030002000100000007000600050004000B000A00090008000F000E000D000C
001200110001001000150014000100130019001800170016001C0001001B001A001F001F001E001D001F001F001F0020
001F001F001F001F001F002300220021001F001F00250024000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010026000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010029002800010027
002D002C002B002A00010001000100010001000100010001000100010001000100010001000100010001000100010001
00010001000100010001000100010001000100010001000100010001000100010001000100010001002E000100010001
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F0030002F0001001F00330032003100010037003600350034003A000100390038
003E003D003C003B004200410040003F0046004500440043004A004900480047001F004D004C004B00510050004F004E
0052000100010001001F001F00540053001F001F001F001F0055001F001F001F0001000100010001001F001F001F0056
001F001F001F001F001F001F001F001F001F001F001F001F001F005700010001001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F
0059005800010001005B005A001F001F0001000100010001000100010001000100010001000100010001000100010001
0001000100010001005C0001000100010001000100010001001F001F005E005D001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F005F001F001F001F
001F006100600001001F001F001F001F001F001F001F001F001F001F001F0062001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F0066006500640063001F001F001F001F0067001F001F001F
001F006A00690068006C001F001F006B001F001F006E006D001F006F001F001F001F001F0070001F001F001F001F001F
001F001F001F001F001F001F001F001F0001000100010001000100710001000100010001000100010001000100010001
000100010001000100720001000100010001000100010073000100010001000100010001000100010001000100010001
000100010001000100010074000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010075000100010001001F007600010001001F001F001F001F001F001F001F001F
001F007700010001001F001F001F001F0001000100010001000100010001000100010001000100010001000100010001
00780001000100010001000100010001000100010001000100010001000100010079000100010001001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F
001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F001F000000000000000007FFFFFE07FFFFFE
0420040000000000FF7FFFFFFF7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000501F0003FFC30000000000000000BCDF000000000020
FFFFFFFBFFFFD740FFBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC03FFFFFFFFFFFFFFFF
FFFEFFFFFFFFFFFFFFFFFFFF027FFFFFBFFF0000000001FF000787FFFFFF00B6FFFFFFFF07FF0000FFFFC000FEFFFFFF
FFFFFFFFFFFFFFFF9C00E1FE1FEFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFE0000003FFFFFFFFFFFF043007FFFFFFFC00
00001FFFFCFFFFFFFFFF07FF01FFFFFFFFFFFFFF00007EFFFFFF03F8FFF003FFEFFFFFFFFFFFFFFFFFFE000FFFE1DFFF
E3C5FDFFFFF99FEF1003000FB080599FC36DFDFFFFF987EE003F00005E021987E3EDFDFFFFFBBFEE1E00000F00011BBF
E3EDFDFFFFF99FEE0002000FB0C0199FC3FFC718D63DC7EC0000000000811DC7E3FFFDFFFFFDDFFF0000000F27601DDF
E3EFFDFFFFFDDFEF000E000F60601DDFE7FFFFFFFFFDDFFFFC00000F80F05DDF2FFBFFFFFC7FFFEE000C0000FF5F807F
07FFFFFFFFFFFFFE000000000000207F3BFFFFAFFFFFF7D600000000F000205F0000000000000001FFFE1FFFFFFFFEFF
1FFFFFFFFEFFFF0F0000000000000000F97FFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFF3C00FFFFF7FFFFFFFFFF20BF
FFFFFFFFFFFFFFFFFFFFFFFF3D7F3DFF7F3DFFFFFFFF3DFFFFFFFFFFFF7FFF3DFFFFFFFFFF3DFFFF0000000007FFFFFF
FFFFFFFF0000FFFF3F3FFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF9FFFFFFFFFFFFFFFFFFF07FFFFFE01FFC7FFFFFFFFFF000FFFFF800FFFFF000DDFFF000FFFFF
FFCFFFFFFFFFFFFF00000000108001FFFFFFFFFF0000000001FFFFFFFFFFFFFFFFFF07FFFFFFFFFF003FFFFFFFFFFFFF
01FF0FFF7FFFFFFF001F3FFFFFFF0000FFFF0FFFFFFFFFFF00000000000003FFFFFFFFFF0FFFFFFF001FFFFE7FFFFFFF
80000080000000000000000000007001FFEFFFFFFFFFFFFF0000000000001FEFFC00F3FFFFFFFFFF0003FFBFFFFFFFFF
007FFFFFFFFFFFFF3FFFFFFFFC00E000E7FFFFFFFFFF01FF046FDE0000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF001FFF8000000000FFFFFFFF3F3FFFFF3FFFFFFFAAFF3F3F5FDFFFFFFFFFFFFF1FDC1FFF0FCF1FDC
00000000000000008002000000000000000000001FFF00000000000000000000F3FFBD503E2FFC84FFFFFFFF000043E0
00000000000001FF00000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFC0000000000000000003FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF000C781FFFFFFFFFFFFF20BFFFFFFFFF000080FFFFFFFFFF7F7F7F7F007FFFFFFFFFFFFF7F7F7F7F
00008000000000000000000000000000000000000000000000000000000000001F3E03FE000000E0FFFFFFFFFFFFFFFE
FFFFFFFEE07FFFFFF7FFFFFFFFFFFFFFFFFEFFFFFFFFFFE0FFFFFFFFFFFFFFFFFFFFFFFF00007FFFFFFF000000000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0000000000001FFF3FFFFFFFFFFF000000000C00FFFF1FFF8FF07FFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFF
FFFFFFFCFF800000FFFFFFFFFFFFFFFFFFFFFFFFFFFFF9FFFFFC000003EB07FF000000FFFFFFFFBF000FFFFFFFFFFFFF
FFFFFFFFFFFFFFFFE8FC00000000002FFFFF07FFFFFFFC001FFFFFFF0007FFFFFFF7FFFFFFFFFFFF7C00FFFF00008000
007FFFFFFFFFFFFFFC7FFFFF00003FFF7FFFFFFFFFFFFFFF003CFFFF38000005FFFF7F7F007E7E7EFFFF03FFF7FFFFFF
FFFFFFFFFFFFFFFF000007FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000FFFFFFFFF0FFFFFFFFFFFF87F
FFFFFFFFFFFFFFFFFFFF3FFFFFFFFFFFFFFFFFFFFFFFFFFF0000000003FFFFFF5F7FFDFFE0F8007FFFFFFFFFFFFFFFDB
0003FFFFFFFFFFFFFFFFFFFFFFF800003FFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFCFFFF0FFF0000000000FF
0000000000000000FFDF000000000000FFFFFFFFFFFFFFFF1FFFFFFFFFFFFFFF07FFFFFE00000000FFFFFFC007FFFFFE
7FFFFFFFFFFFFFFF000000001CFCFCFCB7FFFF7FFFFFEFFF000000003FFF3FFFFFFFFFFFFFFFFFFF07FFFFFFFFFFFFFF
0000000000000000001FFFFFFFFFFFFF0000000000000000000000000000000000000000000000000000000000000000
FFFFFFFF1FFFFFFF000000000001FFFFFFFFE000FFFFFFFF07FFFFFFFFFF07FFFFFFFFFF3FFFFFFF00000000003EFF0F
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00003FFFFFFF0FFFFFFFFF0FFFFFFFFF00FFFFFFFFFFF7FF000FFFFFFFFF
1BFBFFFBFFB7F7FF0000000000000000007FFFFFFFFFFFFF000000FF003FFFFF07FDFFFFFFFFFFBF0000000000000000
91BFFFFFFFFFFD3F007FFFFF003FFFFF000000007FFFFFFF0037FFFF0000000003FFFFFF003FFFFF0000000000000000
C0FFFFFFFFFFFFFF0000000000000000003FFFFFFEEFF06F1FFFFFFF00000000000000001FFFFFFF0000001FFFFFFEFF
003FFFFFFFFFFFFF0007FFFF003FFFFF000000000003FFFF0000000000000000FFFFFFFFFFFFFFFF00000000000001FF
0007FFFFFFFFFFFF0007FFFFFFFFFFFF000000FFFFFFFFFF000000000000000000000000000000000000000000000000
0000000000000000000000000000000000031BFFFFFFFFFF0000000000000000FFFF00801FFFFFFFFFFF00000000003F
FFFF000000000003007FFFFF0000001FFFFFFFFFFFFFFFFF003E00000000003F01FFFFFFFFFFFFFF000001FFFFFF0004
0007FFFFFFFFFFFF0047FFFFFFFF00F0FFFFFFFFFFFFFFFF000000001400C01EC09FFFFFFFFBFFFF0000000000000003
FFFF01FFBFFFBD7F000001FFFFFFFFFFE3EDFDFFFFF99FEF0000000FE081199F00000000000000000000000000000000
FFFFFFFFFFFFFFFF00000003800007BBFFFFFFFFFFFFFFFF00000000000000B300000000000000000000000000000000
7F3FFFFFFFFFFFFF000000003F0000007FFFFFFFFFFFFFFF0000000000000011013FFFFFFFFFFFFF0000000000000000
000007FFE7FFFFFF000000000000007F0000000000000000000000000000000001FFFFFFFFFFFFFF0000000000000000
FFFFFFFF0000000080000000FFFFFFFF99BFFFFFFF6FF27F0000000000000007FFFFFCFF000000000000001AFCFFFFFF
7FE7FFFFFFFFFFFFFFFFFFFFFFFF0000FFFF000020FFFFFF01FFFFFFFFFFFFFF7F7FFFFFFFFFFDFFFFFC000000000001
007FFEFFFFFCFFFF0000000000000000B47FFFFFFFFFFB7FFFFFFDBF000000CB00000000017B7FFF0000000000000000
000000000000000000000000000000000000000000000000007FFFFF00000000C7FFFFFFFFFDFFFF0000000000000001
00010000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000000003FFFFFF0000000000000000
FFFFFFFFFFFFFFFF00007FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000000000000000F
0000000000000000000000000000000000000000000000000000000000000000FFFFFFFFFFFF00000001FFFFFFFFFFFF
0000FFFFFFFFFFFF000000000000007E00000000000000000000000000000000FFFFFFFFFFFFFFFF000000000000007F
0000000000000000000000000000000001FFFFFFFFFFFFFFFFFF00007FFFFFFF7FFFFFFFFFFFFFFF00003FFFFFFF0000
0000FFFFFFFFFFFFE0FFFFF80000000F000000000000FFFF00000000000000000000000000000000FFFFFFFFFFFFFFFF
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFF87FF00000000FFFF80FF0003000B00000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00000000003FFFFF00000000000001FF000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000006FEF00000000000000040007FFFFFFFFFFFF00F000270000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0FFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF1FFF07FFFFFFFFFF0000000043FF01FF0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFDFFFFF
EBFFDE64DFFFFFFFFFFFFFFFFFFFFFEF7BFFFFFFDFDFE7BFFFFFFFFFFFFDFC5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF3FFFFFFFFFF7FFFFFFF7FFFFFDFFDFFFFFFFDFFFFFFFFF7FFFFFFF7FFF
FFFFFDFFFFFFFDFF0000000000000FF7000007E07FFFFFFF000000000000000000000000000000000000000000000000
FFFF07DBF9FFFF7F00003FFFFFFFFFFF000000000000800000000000000000003F801FFFFFFFFFFF0000000000004000
000000000000000000000000000000000000000000000000000000000000000000003FFFFFFF000000000FFFFFFFFFFF
00000000000000000000000000000000000000000000000000000FFFFFFF000000000000000000000000000000000000
00000000000000007FFF6F7F00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000000000000001F
FFFFFFFFFFFFFFFF000000000000088F000000000000000000000000000000000AF7FE96FFFFFFEF5EF7F796AA96EA84
0FFFFBEE0FFFFBFF0000000000000000FFFF000000000000FFFF03FFFFFF03FF00000000000003FF0000000000000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000FFFFFFFF03FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0003FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF0001FFFFFFFFFFFFFFFFFFFFFFFF000000003FFFFFFF00000000000000000000000000000000
000000003FFFFFFF000000000000000000000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFF07FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFF0000000000000000
0000000000000000000000000000000000000000000000000000000000000000",
);
//3392 bytes
enum markTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000088",
cast(immutable size_t[]) x"
000000000000010000000000000001A00000000000004800",
cast(immutable size_t[]) x"
04020302020201000A090802070206050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
02020202020202020202020202020202020202020202020B020202020202020202020202020202020000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000100000000000000050004000300020009000800070006000D000C000B000A
000F00000000000E00100000000000000014001300120011000000000016001500000000000000170000000000000000
00000000000000000000000000190018000000000000001A000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000001B00000000001F001E001D001C0000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000200000000000000000002100000000
0024002300220000000000000000000000000025000000000028002700260000002C002B002A00290030002F002E002D
000000330032003100370036003500340000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000380000000000000000000000000000000000000000000000000000000000000000
003A003900000000003B0000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000003C000000000000000000000000000000000000000000000000003D000000000000
0000003F003E000000000000000000000000004000000000000000000000000000000042003A00410000000000000043
000000000045004400000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000046000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFF0000FFFFFFFFFFFF00000000000000000000000000000000
0000000000000000000000000000000000000000000003F8000000000000000000000000000000000000000000000000
BFFFFFFFFFFE000000000000000000B60000000007FF000000010000FFFFF800000000000000000000003D9F9FC00000
FFFF00000002000000000000000007FF0001FFC000000000200FF8000000000000003EEFFBC00000000000000E000000
00000000FF000000FFFFFFFBFFFFFC00DC0000000000000F0000000C00FEFFFFD00000000000000E4000000C0080399F
D00000000000000E0023000000023987D00000000000000EFC00000C00003BBFD00000000000000E0000000C00E0399F
C0000000000000040000000000803DC7D00000000000001F0000000C00603DDFD00000000000000E0008000C00603DDF
D80000000000000F0000000C00803DDF000000000000000E000C0000FF5F840007F20000000000000000000000007F80
1FF20000000000000000000000007F00C2A0000003000000FFFE0000000000001FFFFFFFFEFFE0DF0000000000000040
7FFFF80000000000001E3F9DC3C00000000000003C00BFFC0000000000000000000000000000000000000000E0000000
00000000000000000000000000000000001C0000003C0000000C0000000C0000FFF000000000000000000000200FFFFF
000000000000B8000000000000000000000002000000006000000000000000000FFF0FFF000000000000000000000000
00000000000000000000000000000000000000000F8000009FFFFFFF7FE00000FFFF0000000000000000000000007FFF
FFF000000000001F000FF8000000001F00003FFE00000007000FFFC00000000000FFFFF0000000000000000000000000
0000000000000000039021FFFFF70000000000000000000000000000000000000000000000000000FFFFFFFFFFFFFFFF
0000000000000000000000000000000000000000000000000001FFFFFFFF000000000000000000000000000000000000
00000000000000000003800000000000000000000000000080000000000000000000000000000000FFFFFFFF00000000
0000FC000000000000000000000000000000000006000000000000000000000000000000000000003FF7800000000000
00000000C00000000003000000000000000010F8000008440000000000000000FFF00000000000038003FFFF0000003F
00003FC00000000000000000000FFF80FFF800000000000F0000002000000001007FFE00000000003800000000003008
C19D0000000000000060F80000000002000000000000000000000000000000000000000000000000000037F800000000
00000000400000000000000000000000000000000000000000000000000000000000FFFF0000FFFF0000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000002000000000000000
0000000000000000000000000000000000000000000000000000000100000000000000000000000007C0000000000000
00000000000000000000000000000000870000000000F06E000000000000000000000000000000000000006000000000
000000F00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000180000000000E0000000000000000000000000000000000000000001FFC0000000000000003C0000000000000000
FF00000000000007801900000000007F07FF0000000000070000000000000004001FFF80000000070008000000000060
FFF8000000000007000000000000DE0140FFF0000000000000000000000000020000000000000000000007FF80000000
D80000000000000F001F1FCC0080399F00000000000000000000000000000000FFE0000000000000000000004000007F
FFFF000000000000000000000000000F00000000000000000000000000000000FF3F8000000000000000000030000001
FFFF000000000000000000000000000100FFF80000000000000000000000000000000FFFE00000000000000000000000
0000000000000000000000000000000007FFF00000000000000000000000000000000000000000000000000000000000
79BF000000000000000000000000000D000000000000000000000011FCFE00007BF80000000007FE000000000FFE0080
0000000003FFFC000000000000000000FF7F8000000000000000000000000000007FFEFFFFFC00000000000000000000
B47E00000000000000000000000000BF0000000000FB7C00000000000000000000000000000000000000000000000000
00000000000000000078000000000000C7F000000000000B000000000000000700000000000000000000000000000000
000000000000000000000000003FFF810000000000000000000000000000000000000000000000000000000000000000
0000000000000000001F000000000000007F000000000000000000000000000000000000000000000000000000000000
0000000000000000FFFFFFFFFFFE800000000000000780FF000300100000000000000000000000000000000000000000
00000000600000000000000000000000FFFF3FFFFFFFFFFF000000000000007F00000000000000000000000000000000
0000000000000000F807E3E00000000000003C0000000FE700000000000000000000000000000000000000000000001C
00000000000000000000000000000000F87FFFFFFFFFFFFF00201FFFFFFFFFFF0000FFFEF80000100000000000000000
000007DBF9FFFF7F00000000000000000000000000008000000000000000000000000000000000000000000000000000
00004000000000000000F000000000000000000000000000000000000000000000000000000000000000F00000000000
00000000000000000000000000000000000000000000000000000000007F0000000000000000000000000000000007F0
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFF
0000000000000000000000000000000000000000000000000000000000000000",
);
//2848 bytes
enum numberTrieEntries = TrieEntry!(bool, 8, 6, 7)(cast(immutable size_t[]) x"
0000000000000000000000000000002000000000000000D0",
cast(immutable size_t[]) x"
000000000000010000000000000002C00000000000002500",
cast(immutable size_t[]) x"
040203020202010009080202070206050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000002000200010000000200020002000200020002000200020005000200040003
000700060002000200090008000600060006000B0006000A0002000D000C000C00020002000E00050002000F00020002
00020002000200020011000200100002001300120002000E0000000C0014000200020002000200150002000200020002
0019001800170016000200020002000200020002001B001A001D001C0002000200020002000200020002000200020002
00020002001E00020002000200020002002000020002001F000200020022002100020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200060023
00250017000C0024000400020002000C0002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
0002000200020002000200020002000200020002000200020002000E000200020027002600020002002A002900280002
00020002002300020002000200020002002E002D002C002B003200310030002F00020000003300020036003500020034
0039003800040037000200020004000200020002000C000C0002003A0005000C0002000C003B00020002000200020002
0023000C0002003C003D000C0002000200020002000200020002000200020002000200020002003E0002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002003F00050003
000200020002000200020002004000020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020042004100020002000200020002004300020002000200020002000200020002000200020002
000200020002000200020002000200020002000500020002000200020004000200020002000400020002000200020002
0002000C0044000200020002000200020002004700460045000200020002000200020048000200020002000200020002
000200020002000200020002000200020002000200020002000400020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000200020002000200020002000200020002
000200020002000200020002000200020002000200020002000200020002000203FF0000000000000000000000000000
720C0000000000000000000000000000000000000000000000000000000000000000000000000000000003FF00000000
000000000000000003FF000000000000000000000000000000000000000003FF00000000000000000000FFC000000000
000000000000000003F0FFC000000000000000000000000000FCFFC00000000000000000000000000007FFC000000000
00000000000000007F00FFC000000000000000000000000001FFFFC07F00000000000000000000000000000003FF0000
000FFFFF0000000000000000000000000000000003FF0000000000000000000000000000000000001FFFFE0000000000
00000000000000000001C00000000000000000000000000003FF03FF000000000000000000000000000000000000FFC0
00000000000000000000000007FF00000000000003FF03FF000000000000000000000000000000000000000003FF03FF
000000000000000003F100000000000000000000000003FF00000000000000000000000000000000FFFFFFFFFFFF0000
00000000000003E700000000000000000000000000000000FFFFFFFF00000000000000000FFFFFFFFFFFFC0000000000
0000000000000000FFC000000000000000000000000FFFFF000000000000000000000000000000002000000000000000
070003FE00000080000000000000000000000000003C00000000000000000000000003FF0000000000000000FFFEFF00
FFFE0000000003FF0000000000000000000003FF000000000000000000000000003F0000000000000000000000000000
000000000000000003FF000003FF0000000FFFFFFFFFFF8001FFFFFFFFFFFFFF0000000000000C000000000000000000
00000000000000000FFFFFFE000000000000000F000000000000000000000402000000000000000000000000003E0000
0000000000000000FE000000FF0000000000FF8000000000F800000000000000000000000FC000000000000000000000
3000000000000000FFFFFFFFFFFCFFFF000000000000000060000000000001FF00000000E00000000000F80000000000
0000000000000000FF000000FF0000000000FE000000000000000000000000000000000000000000FC00000000000000
00000000000000007FFFFFFF000000000000007FE000000000000000001E000000000000000000000000000000000FE0
00000000000000000000FFFFFFFC0000FFC000000000000000000000000000000000000000000000001FFFFE03FF0000
0FFF000000000000000000000000000000000000000000000007FFFF00000000000000000000000000001FFFFFFF0000
000000000000000000000000001FFFFFFFFFFFFFFFFFFFFF00007FFFFFFFFFFF000000000000000000000003FBFF0000
00000000007FFFFF00000000000000000000000000000000000FFFFF000FFFFF000000000000000001FFFFFF00000000
0000000000000000FFFFFFFFFFFFC0000000000000000000000000000000FF800000000000000000FFFE000000000000
001EEFFFFFFFFFFF00000000000000003FFFBFFFFFFFFFFE00000000000000000000000000001FFF0000000000000000
000000000000000003FF000000000000",
);
//3360 bytes
enum punctuationTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000080",
cast(immutable size_t[]) x"
000000000000010000000000000001800000000000004900",
cast(immutable size_t[]) x"
04020302020201000A090802070206050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000200010001000000050004000300010001000800070006000C000B000A0009
000E00010001000D001100100001000F0015001400130012000100010001001600180001000100170019000100010001
00010001001A00010001001D001C001B000100010001001E000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100200001001F00240023002200210001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010027002600250001
002900010028000100010001002A0001002E002D002C002B0030002F0001000100010033003200310037003600350034
003B003A00390038003E003D0001003C0001000100010001000100010001003F00010001000100010040000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
004200410001000100440043000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100450001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001004600010001000100010001000100010001000100010001000100010001
000100010047000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100018C00F7EE0000000028000000B800000188C00882000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000004000000000000000
00000000000000800000000000000000000000000000000000000000FC00000040000000000006000018000000000049
00000000E800360000003C0000000000000000000000000000000000001000000000000000003FFF0000000000000000
000000000000000003800000000000007FFF000000000000000000004000000000000000000000000000000000000000
000000000000000000010030000000000000000000000000200000000000000000000000000000000040000000000000
000000000000000000010000000000000000000000000000008000000000000000000000000000100000000000000000
00000000000000000000000000000000000000000000000000100000000000000000000000000000000000000C008000
000000000000000000000000000000003C0000000017FFF00000000000000000000000000000002000000000061F0000
0000000000000000000000000000FC00000000000000000008000000000000000000000000000000000001FF00000000
000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000
000000000000000000004000000000000000000018000000000038000000000000600000000000000000000000000000
0000000000000000000000000770000000000000000007FF000000000000000000000000000000000000000000000000
000000000000000000000000000000300000000000000000000000000000000000000000C00000000000000000000000
00003F7F000000000000000000000000000000000000000060000001FC0000000000000000000000F000000000000000
F800000000000000C000000000000000000000000000000000000000000800FFFFFF00FFFFFF0000600000007FFBFFEF
000000000000600000000000000000000000060000000F00000000000000000000000000000000000000000000000000
0000000000000000003FFF000000000000000000000000000000FFC00000006000000000000000000000000000000000
0000000001FFFFF8300000000F000000000000000000000000000000000000000000000000000000DE00000000000000
0000000000000000000100000000000000000000000000000000000000000000FFFF7FFFFFFFFFFF000000003FFCFFFF
0000000000000000000000000000000020010000FFF3FF0E000000000000000000000001000000000800000000000000
000000000000000000000000000000000000000000000000C000000000000000000000000000E0004008000000000000
000000000000000000FC000000000000000000000000000000F00000000000000000000000000000170000000000C000
0000C000000000000000000080000000000000000000000000000000C0003FFE000000000000000000000000F0000000
000000000000000000030000C00000000000000000000000000000000000000000000000000000000000080000000000
C000000000000000000000000000000000000000000000000000000000000000FFFF000003FF000000000D0BFFF7FFFF
00000000000000000000000000000000B80000018C00F7EE0000003FA800000000000000000000000000000000000000
000000000000000700000000000000000000000000000000000000000000000000000000000000000000000000000000
000000008000000000000000000100000000000000000000000080000000000000000000000000000000000000000000
000000000000000000000000008000000000000000000000000000000000000080000000800000000000000000000000
0000000000000000000000000000000000000000000000008000000001FF00000000000000000000007F000000000000
FE000000000000000000000000000000000000001E000000000000000000000000000000000000000000000000000000
0000200000000000000000000000000000000000000000000000000003E0000000000000000003C00000000000000000
00000000000000000000000000003F80D80000000000000000000000000000030000000000000000003000000000000F
000000000000000000000000E80021E03F00000000000000000000000000000000000200000000000000000000000000
0000000000000000000000002C00F8000000000000000000000000000000004000000000000000000000000000000000
00000000000000000000000000FFFFFE000000000000000000001FFF0000000E02000000000000000000000000000000
700000000000000000000000000000000000000000000000000000000000000008000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000007000000000000000000000000400000000
8000000000000000000000000000007F00000007DC000000000000000000000000000000000003FF0000000000000000
000000000000000000000000000000000000000000000000000300000000003E00000000000000000000000000000000
00000000000000000000000000000000000000000000000001800000000000000000000000000000000000000000FFF8
000000000000000080000000000000000000000000000000001F00000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000600000000000000000000000000000000C00000000000
000000000000000000200000000000000F80000000000000000000000000001000000000000000000000000000000000
000000000000000000000000000000000000000007800000000000000000000000000000000000000000000000000000
000000000000000000000004000000000000000000000000000000000000000000000000800000000000000000000000
000000000000000000000000000000000000000000000F800000000000000000000000000000000000000000C0000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000",
);
//3424 bytes
enum symbolTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000080",
cast(immutable size_t[]) x"
000000000000010000000000000001800000000000004B00",
cast(immutable size_t[]) x"
05030403030201000A090803070303060303030303030303030303030303030303030303030303030303030303030303
030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303
030303030303030303030303030303030303030303030303030303030303030303030303030303030000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000030002000100000007000600050004000B000A00090008000F000E000D000C
0011000100010010001300120001000100150001001400010016000100010001001A001900180017001C00190019001B
001E0019001D0019002100200001001F0019002400230022000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010025000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010027000100010026
002A00290001002800010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
002B000100010001002E002D002C000100010001002F0001000100010001000100010031000100300001000100010001
0001000100010001000E0001000100010001000100010001003200010001000100010001000100010001000100010001
003300010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100340001000100010001000100010001000100010001000100010035000100010001
0039003800370036003B003A000100010001003C0019001900010001000100010001003E003D00010001000100010001
0001000100010001000100410040003F0019004400430042004600450019001900490048001900470001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
00010001000100010001000100010001700008100000000050000001400000000113D37C000000000080000000800000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000FFFFAFE0FFFC003C0000000000000000002000000000000000000000000000300040000000000000
000000000000000000000000000000000000000000000004000000000000000000000000000000000000000000000000
000000000000E0000000000000000000000000000000C9C0000000000000000000000000000000006000020040000000
000000000000000000000000000000000000000000000000C04000000000000000000000000000000000000000000000
000000000000010000000000000000000000000000000000000000000000000000000000000000000C0C000000000000
000000000000000000000000000000000000000000000000000200000000000000000000000000000001000000000000
000000000000000007F80000000000000000000000000000800000000000000000000000000000000000000000000000
000000000000000002000000000080000000000000000000000000000000000080000000000000000000000000000000
0000000000000000000000000000000001500000FCE8000E0000000000000000C0000000000000000000000001E0DFBF
0000000000000000000000000000000000000000C0000000000000000000000000000000000000000000000000000000
0000000003FF000000000000000000000000000000000000000020000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000800000000000000000000000000000000000001
0000000000000000FFFFFFFFC000000000000000000000001FF007FE0000000000000000000000000000000000000000
00000000000000000000000000000000A0000000000000006000E000E000E00300000000000000001C00000000040010
FFFFFFFF00001C0000000000000000010C0042AFC1D0037B000000000000BC1FFFFFFFFFFFFF0C00FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF9FFFFFFF0FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000007FFFFFFFFF00000000000007FFFFFFFFFFF0000000000003FFFFFFFFFF
FFFFFFFFFFFFFFFF000000FFFFFFFFFFFFFFFFFFFFF00000FFFF003FFFFFFF9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFE000007CFFFFFFFF0FFFFFFFFFFFFFFFFFFFFFFFFCFFFFFFFFFFFFFFFFFFFFFFFBFFFFFFFFFFFFFFFFFFFFF
000000000000000000000000000000000000000000000000000007E00000000000000000000000000000000000030000
FFFFFFFFFBFFFFFF000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000003FFFFF
C0C00001000C001000000000000000000000000018000000000000000000000000000000000000000000000000000000
00000000FFC300000000800FFFFFFFFFFFFFFC007FFFFFFFFFFFFFFF000100FF0001FFFFFFFFFC00FFFFFFFFFFFFFFFF
000000000000000000000000000000000000000000000000FFFFFFFFFFFFFFFF00000000000000000000000000000000
FFFFFFFFFFFF0000000000000000007F00000003007FFFFF000000000000000000000000000006000000000000000000
03C00F000000000000000000000000000000000000000000000000000000000000000000000000000380000000000000
00000000000000000000000000000000000000000000000000000C000800000000000000000000000000000000000000
00000200000000000000000000000000FFFC00000000000000000000000000070000000000000000000000000000FFFF
0000000000000000F0000000000080000000000000000000000002740000000000000000000000000000000000000000
40000000700008100000000050000001000000000000000030007F7F00000000FF80000000000000FE00000000000000
000000011FFF73FF1FFFFFFFFFFF00000000000000000000018000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000
00000000000000000003FFFFFFE00000F000000000000000000000000000002000000000000000000000000000000000
00000000000000000000000000000000000000001000000000000000000000000000000000000000FFFFFFFFFFFF0000
FFFFFFFFFFFFFFFF000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF003FFFFFFFFFFFFF
FFFFFE7FFFFFFFFF00001C1FFFFFFFFFFFFFC3FFFFFFF018000007FFFFFFFFFFFFFFFFFFFFFFFFFF0000000000000023
00000000000000000000000000000000FFFFFFFFFFFFFFFF00000000007FFFFF00000000000000000000000000000000
000000000000000000000000000000000000000000000000080000000800000200200000002000000000800000008000
000002000000020000000000000000080780000000000000FFDFE00000000000000000000000006F0000000000000000
000000000000000000000000000080000000000000000000000000000000000000000000000000000000000000000000
000000000000000080000000000000000000000000000000000000000000000000011000000000000000000000000000
000040000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000003000000000000FFFF0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFE7FFF000FFFFF003FFFFFFFFEFFFE
FFFFFFFFFFFFE000FFFFFFFFFFFFFFFF00003FFFFFFFFFFFFFFFFFC0000000000FFFFFFFFFFF00070000003F000301FF
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF1FFF1FFFF0FFFFFF
FFFFFFFFFFFFFFFFF87FFFFFFFFFFFFFFFFFFFFFFFFFFFFF00010FFF03FFFFFFFFFFFFFFFFFF0FFFFFFFFFFF03FF00FF
00033FFFFFFF00FF0000000000000000FFFFFFFFFFFFFFFF1FFF3FFF000FFFFFBFFFFFFFFFFF01FF01FF01FF0FFFC03F
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFF00000000000007FF00000000000000000000000000000000
00000000000000000000000000000000",
);
//6080 bytes
enum graphicalTrieEntries = TrieEntry!(bool, 8, 5, 8)(cast(immutable size_t[]) x"
0000000000000000000000000000002000000000000000D0",
cast(immutable size_t[]) x"
000000000000010000000000000002C00000000000008A00",
cast(immutable size_t[]) x"
05040302020201000D0C0B0A09080706100F0E0202020202131313131313121113131313131313131313131313131313
131313131313131313131313131313131313131313131313131313131313131313131313131313131313131313131313
131313131313131313131313131313131313131313131314131313131313131313131313131313130000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000200010001000000050004000300010009000800070006000D000C000B000A
0010000F0001000E00120011000100010016001500140013001800010001001700010001001A0019000100010001001B
001C0001000100010020001F001E001D0001002300220021000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
0001000100010001000100010001000100010001000100010026002500010024002A0029002800270001000100010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
000100010001000100010001000100010001000100010001002B000100010001002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002E002D0001002C00310030002F000100350034003300320038000100370036003C003B003A00390040003F003E003D
00440043004200410048004700460045004C004B004A00490050004F004E004D0051000100010001002C002C00530052
002C002C002C002C0054002C002C002C0001000100010001002C002C002C0055002C002C002C002C002C002C002C002C
002C002C002C002C002C005600010001002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C0058005700010001005A0059002C002C
00010001000100010001000100010001000100010001000100010001000100010001000100010001005B000100010001
0001000100010001002C002C005D005C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C005E002C002C002C002C0060005F0001002C002C002C002C
002C002C002C002C002C002C002C0061002C002C002C002C002C002C002C002C002C002C002C002C0062002C002C002C
0066006500640063006A006900680067002C006B00010001006C002C002C002C002C006F006E006D0071002C002C0070
002C002C00730072002C0076007500740001007900780077007B007A00010001007E007D0001007C002C002C002C002C
00010001000100010001007F000100010001000100010001000100010001000100010001000100010080000100010001
000100010001008100010001000100010001000100010001000100010001000100010001000100010001008200010001
000100010001000100010001000100010001000100010001000100010001000100010001000100010001000100010001
0083000100010001002C008400010001002C002C002C002C002C002C002C002C002C008500010001002C002C002C002C
000100010001000100010001000100010001000100010001000100010001000100860001000100010001000100010001
000100010001000100010001000100010087000100010001002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002C002C0088002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C
002C002C002C002C002C002C002C002C002C002C002C002C002C002C002C002CFFFFFFFF000000007FFFFFFFFFFFFFFF
FFFFDFFF00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFCFFFFFFFFFFFFFFFFFFFFFBFFFFD7F0FFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFFFFFFFE7FFFFF
FFFFFFFFFFFEE7FF001F87FFFFFF00FFFFFFFFFFEFFFFFC0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFDFFFFFFF
FFFFFFFFFFFF3FFFFFFFFFFFFFFFE7FF0003FFFFFFFFFFFFE7FFFFFFFFFFFFFF7FFF3FFFFFFFFFFFFFFF07FF4FFFFFFF
FFFFFFFFFF007FFFFFFFFFFBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF3C5FDFFFFF99FEF7FFFFFCFB080799F
D36DFDFFFFF987EE007FFFC05E023987F3EDFDFFFFFBBFEEFE03FFCF00013BBFF3EDFDFFFFF99FEE00FFFFCFB0E0399F
C3FFC718D63DC7EC07FFFFC000813DC7F3FFFDFFFFFDDFFFFF80FFCF27603DDFF3EFFDFFFFFDDFFF000EFFCF60603DDF
FFFFFFFFFFFDDFFFFFFFFFCFFFF0FDDF2FFBFFFFFC7FFFEE001CFFC0FF5F847F87FFFFFFFFFFFFFE000000000FFFFFFF
3FFFFFAFFFFFF7D600000000F3FF7F5FFFFFFFFFFFFFFFFFFFFE1FFFFFFFFEFFDFFFFFFFFEFFFFFF0000000007FFDFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF20BFFFFFFFFFFFFFFFFFFFFFFFFF3D7F3DFF
7F3DFFFFFFFF3DFFFFFFFFFFFF7FFF3DFFFFFFFFFF3DFFFF1FFFFFFFE7FFFFFFFFFFFFFF03FFFFFF3F3FFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF1FFFFFFF01FFFFFFFFFFFFFF007FFFFF803FFFFF000DDFFF000FFFFF
FFFFFFFFFFFFFFFF03FF03FF3FFFFFFFFFFFFFFF03FFBFFF01FFFFFFFFFFFFFFFFFF07FFFFFFFFFF003FFFFFFFFFFFFF
0FFF0FFF7FFFFFFF001F3FFFFFFFFFF1FFFF0FFFFFFFFFFFFFFFFFFFC7FF03FFFFFFFFFFCFFFFFFF9FFFFFFF7FFFFFFF
FFFF3FFF03FF03FF0000000000007FFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF1FFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFF
F8FFFFFFFFFFFFFFFFFFFFFFFFFFE3FFE7FFFFFFFFFF01FF07FFFFFFFFFF00FFFFFFFFFF3F3FFFFF3FFFFFFFAAFF3F3F
FFDFFFFFFFFFFFFF7FDCFFFFEFCFFFDFFFFF80FFFFFF07FFFFF30000FFFFFFFFFFFFFFFF1FFF7FFF0001FFFFFFFF0001
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0FFFFFFFFFFFFFFFFFFF0000007FFFFFFFFFFFFFFFFF000007FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFCFFFFFFFFFFFFFFFFFFFFFFFBFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFE0FFFFFFFFFFFFFFFFF20BFFFFFFFFF800180FFFFFFFFFF
7F7F7F7F007FFFFFFFFFFFFF7F7F7F7FFFFFFFFFFFFFFFFF000000003FFFFFFFFFFFFFFFFBFFFFFF000FFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000003FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFE
FFFFFFFFFE7FFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFE0FFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFF800FFFFFFFFF
FFFFFFFF7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF1FFFFFFFFFFFFFFF007F00000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC000003EB07FF03FF1FFFFFFFFFFF00FFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF03FFC03FFFFFFFFFFFFFFFFF1FFFFFFF800FFFFFFFFFFFFFFFFFFFFF7FFFFFFFC3FFBFFF
007FFFFFFFFFFFFFFFFFFFFFF3FF3FFFFFFFFFFFFFFFFFFF007FFFFFF8000007FFFF7F7F007E7E7EFFFF0FFFFFFFFFFF
FFFFFFFFFFFFFFFF03FF3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000FFFFFFFFF0FFFFFFFFFFFF87F
0000000000000000000000000000000000000000000000000000000000000000FFFFFFFFFFFFFFFFFFFF3FFFFFFFFFFF
FFFFFFFFFFFFFFFF0000000003FFFFFF5F7FFFFFE0F8007FFFFFFFFFFFFFFFDBFFFFFFFFFFFFFFFFFFFFFFFFFFF80007
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFCFFFFFFFF0000000080FFFFFFFFFF03FFFFFFFFDF0F7FFFF7FFFF
FFFFFFFFFFFFFFFF1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFF7FFFFFFFFFFFFFFF30007F7F1CFCFCFC
B7FFFF7FFFFFEFFF000000003FFF3FFFFFFFFFFFFFFFFFFF07FFFFFFFFFFFFFFFF8FFFFFFFFFFF87FFFFFFFFFFFFFFFF
000000011FFF7FFF3FFFFFFFFFFF000000000000000000000000000000000000FFFFFFFF1FFFFFFF0FFFFFFF0001FFFF
FFFFE00FFFFFFFFF07FFFFFFFFFF07FFFFFFFFFFBFFFFFFF00000000003FFF0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF03FF3FFFFFFF0FFFFFFFFF0FFFFFFFFF00FFFFFFFFFFF7FF800FFFFFFFFF1BFBFFFBFFB7F7FF0000000000000000
007FFFFFFFFFFFFF000000FF003FFFFF07FDFFFFFFFFFFBF000000000000000091BFFFFFFFFFFD3FFFFFFFFFFFBFFFFF
0000FF807FFFFFFFF837FFFF0000000083FFFFFF8FFFFFFF0000000000000000F0FFFFFFFFFFFFFFFFFFFFFFFFFCFFFF
873FFFFFFEEFF06FFFFFFFFF01FF01FF00000000FFFFFFFF007FF87FFFFFFFFFFE3FFFFFFFFFFFFFFF07FFFFFF3FFFFF
0000FE001E03FFFF0000000000000000FFFFFFFFFFFFFFFF00000000000001FF0007FFFFFFFFFFFFFC07FFFFFFFFFFFF
03FF00FFFFFFFFFF00000000000000000000000000000000000000000000000000000000000000007FFFFFFF00000000
00033BFFFFFFFFFFE000000000000000FFFF00FFFFFFFFFFFFFF000003FFFFFFFFFF0000000003FF007FFFFF00000FFF
FFFFFFFFFFFFFFFF803FFFFFFFFC3FFFDFFFFFFFFFFFFFFF03FF01FFFFFF0007FFDFFFFFFFFFFFFF007FFFFFFFFF00FF
FFFFFFFFFFFFFFFF001FFFFEFFFFFFFFFFFFFFFFFFFBFFFF0000000000000003FFFF03FFBFFFBD7F03FF07FFFFFFFFFF
FBEDFDFFFFF99FEF001F1FCFE081399F00000000000000000000000000000000FFFFFFFFFFFFFFFF00000003EFFFFFFF
FFFFFFFFFFFFFFFF0000000003FF00FF00000000000000000000000000000000FF3FFFFFFFFFFFFF000000003FFFFFFF
FFFFFFFFFFFFFFFF00001FFF03FF001F03FFFFFFFFFFFFFF00000000000003FFFFFF0FFFE7FFFFFF000000000000007F
000000000000000000000000000000000FFFFFFFFFFFFFFF0000000000000000FFFFFFFF000000008007FFFFFFFFFFFF
F9BFFFFFFF6FF27F0000000003FF007FFFFFFCFF000000000000001FFCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FF
FFFF0007FFFFFFFF01FFFFFFFFFFFFFF00000000000003FF000000000000000000000000000000000000000000000000
FF7FFFFFFFFFFDFFFFFF1FFFFFFF003F007FFEFFFFFCFFFF0000000000000000B47FFFFFFFFFFB7FFFFFFDBF03FF00FF
000003FF01FB7FFF000000000000000000000000000000000000000000000000000000000000000001FFFFFF00000000
C7FFFFFFFFFDFFFF0000000003FFFFFF00010000000000008003FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0000000003FFFFFF0000000000000000FFFFFFFFFFFFFFFF001F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF000000000000000F0000000000000000000000000000000000000000000000000000000000000000
FFFFFFFFFFFF00000007FFFFFFFFFFFF0000FFFFFFFFFFFF00000000003FFFFF00000000000000000000000000000000
FFFFFFFFFFFFFFFF000000000000007F0000000000000000000000000000000001FFFFFFFFFFFFFFFFFFC3FF7FFFFFFF
7FFFFFFFFFFFFFFF003F3FFFFFFF03FFFFFFFFFFFFFFFFFFE0FFFFFBFBFF003F000000000000FFFF0000000000000000
0000000000000000FFFFFFFFFFFFFFFF0000000007FFFFFF0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFF87FF
00000000FFFF80FF0003001F00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000003FFFFF00000000000001FF0000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000006FEF000000000000
00040007FFFFFFFFFFFF00F000270000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF1FFF07FFFFFFFFFF00000000F3FF01FF0000000000000000
FFFF3FFFFFFFFFFFFFFFFFFFFFFF007FFFFFFFFFFFFFFFFF000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF003FFFFFFFFFFFFFFFFFFE7FFFFFFFFFF807FFFFFFFFFFFFFFFFFFFFFFFFFFFF000007FFFFFFFFFF
FFFFFFFFFFFFFFFF000000000000003F0000000000000000000FFFFF000FFFFFFFFFFFFFFFFFFFFF01FFFFFF007FFFFF
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFDFFFFFEBFFDE64DFFFFFFFFFFFFFFFFFFFFFEF
7BFFFFFFDFDFE7BFFFFFFFFFFFFDFC5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFF3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFCFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFEF8000FFF0000000000000000000007E07FFFFFFF0000000000000000
00000000000000000000000000000000FFFF07DBF9FFFF7F00003FFFFFFFFFFF00000000000080000000000000000000
3FFF1FFFFFFFFFFF000000000000C3FF0000000000000000000000000000000000000000000000000000000000000000
00007FFFFFFF000083FFFFFFFFFFFFFF00000000000000000000000000000000000000000000000003FFFFFFFFFF0000
0000000000000000000000000000000000000000000000007FFF6F7F00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00000000007FFF9FFFFFFFFFFFFFFFFF00000000C3FF0FFF00000000000000000000000000000000
0000000000000000FFFE000000000000001FFFFFFFFFFFFF00000000000000003FFFFFFFFFFFFFFE0000000000000000
000000000000000000000000000000000AF7FE96FFFFFFEF5EF7F796AA96EA840FFFFBEE0FFFFBFF0003000000000000
FFFF0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFE7FFF000FFFFF003FFFFFFFFEFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00003FFFFFFFFFFFFFFFFFC0000000000FFFFFFFFFFF00070000003F000301FF00000000000000000000000000000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF1FFF1FFFF0FFFFFFFFFFFFFFFFFFFFFFF87FFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00010FFF03FFFFFFFFFFFFFFFFFF0FFFFFFFFFFF03FF00FF00033FFFFFFF00FF0000000000000000
FFFFFFFFFFFFFFFF1FFF3FFF000FFFFFBFFFFFFFFFFF01FF01FF01FF0FFFC03FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFF7FFFF03FF0000000007FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000FFFFFFFF
03FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF3FFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0003FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0001FFFFFFFFFFFFFFFFFFFFFFFF000000003FFFFFFF
00000000000000000000000000000000000000003FFFFFFF000000000000000000000000000000000000000000000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFF07FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0000FFFFFFFFFFFF0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFF
0000000000000000000000000000000000000000000000000000000000000000",
);
//4824 bytes
enum nonCharacterTrieEntries = TrieEntry!(bool, 7, 4, 4, 6)(cast(immutable size_t[]) x"
00000000000000000000000000000010000000000000004C000000000000014C",
cast(immutable size_t[]) x"
000000000000008000000000000000F0000000000000040000000000000043C0",
cast(immutable size_t[]) x"
07060504030201000B0B0B0A090801010B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B
0B0B0B0B0B0B0B0B0D0101010B0B0B0C000000000D010101000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000030002000100000007000600050004
000B000A00090008000D000D000D000C000D000D000D000D000D000D000D000D000D000D000D000D000D000D000D000D
000D000D000D000D000D000D000D000D000D000F000E000D000D000D000D000D000D000D000D000D000D000D0010000D
000D000D000D000D00120011000D000D0016001500140013001A001900180017001E001D001C001B001D001D001F000D
001D001D0020001D001D001D001D001D00220021001D001D000D000D000D000D0024000D0023000D001D001D001D001D
0025001D001D001D0027001D001D00260028001D001D001D002C002B002A00290030002F002E002D001D003300320031
000D000D000D000D000D000D000D000D000D000D0034000D000D00360035000D0037000D000D000D000D000D000D000D
00390038000D000D001D003A001D001D000D000D000D000D000D000D000D003B001D001D001D003C001D001D001D001D
001D001D001D001D001D001D001D001D001D001D001D001D001D001D001D001D001D001D001D003D001D001D001D001D
001D001D001D001D001D001D001D001D000D000D000D000D000D000D000D000D000D000D000D000D003E000D000D000D
000D000D000D000D000D000D000D000D000D000D000D000D003E000D000D000D00000000000000000000000000000000
00000000000000000000000200010000000000000000000000060005000400030000000000000000000A000900080007
0000000D000C000B000F000E0000000000130012001100100017001600150014001B001A00190018001F001E001D001C
0023002200210020002600250024000000270000000000000000000000000000002A002900280000002E002D002C002B
000000000000000000000000000000000030002F000000000033000000320031003500340030002D0039003800370036
003D003C003B003A003F0000003E00000043004200410040000000000000000000000000000000000047004600450044
004A0049004800000000004B000000000000000000000000000000000000000000000000004D004C0000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000004F004E0000
00500000000000000054005300520051005700560055000000580000000000000000000400590000005C005B0000005A
000000000000005D00000000000000000000000000000000000000000000000000000000000000000000000000000000
005F005E0000000000000000000000000061000000000060006200000000000000640000006100630066000000650000
0069000000680067006B00000038006A000000000000000000000000000000000000000000000000006D006C00000000
00000000000000000000000000000000006F0000006E0000007200000071007000000000000000000074007300000000
007600000075002D0078007700000059007B0000007A0079007E007D0000007C0080002F007F007F0084008300820081
0086008500000000007F0089008800870000000000000000007F008B008A0067008E008D004F008C00730090007F008F
0094009300920091007F009700960095009A009900980000007F007F007F009B009E009D009C007F00A100A0009F0087
00A3000000A2000000A5000000A4004600A900A800A700A6007F007F00AB00AA00AD000000AC0000005500AE007F007F
00B100B000AF0000007F007F00B300B200B600B5007F00B400BA00B900B800B7003000BC00BB0000007F007F007F00B1
007F00BF00BE00BD007F00C200C100C000C3007F007F007F00C600C5006F00C400000000000000000000000000000000
0000000000000000007F006F000000000000000000C70000007F007F00C80000007F007F007F007F007F007F007F007F
007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F
007F007F007F007F009900C9007F007F007F007F00CA0000007F007F007F007F007F007F007F007F007F007F007F007F
00000000000000000000000000000000007F007F00B30000007F007F007F007F00000000000000000000000000000000
00CC007700CB0030007F00CE00CD0000007F007F007F007F007F007F007F007F007F00CF0000007F00D200D100D00000
000000000000000000000000000000000000000000000000006100000000000000CA000000000000007F007F007F0098
007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F00D3007F007F007F
00000000000000000000000000D500D400B4000000000000007F007F007F007F007F00D700D60000007F007F007F007F
007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F00C80000005F006E
003500000000000000D90000000000D800DB007F00DA0000007F007F00DC000000DF00DE00DD00000000000000E100E0
000000E20000000000E300000000000000000000000000000000000000000000007F00E400000000007F007F007F007F
007F007F007F007F007F007F007F007F007F007F007F007F007F007F007F00E5007F00E800E700E6007F007F00EA00E9
00EC00EB007F007F007F007F007F007F00ED007F007F007F007F007F007F007F007F007F007F007F00EE007F007F007F
00EF000000000000007F007F00F00000007F007F007F007F007F007F007F007F007F00F200F1007F007F007F007F00F3
00F700F600F500F4007F007F007F007F00F900F80000003800FA00E700000000007F007F00FC00FB0000000000000000
0000000000000000000000000000000000FD00000000000000FF000000FE0000007F01010100004B0000000000000000
010401030102000001060105000000000000000000000000000000000000000000930000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000B000000000000001070000000000000000
000000000000000000000000000000000000000000000000000000000000000000000108000000000000000000000000
000000000000000000000000000000000000000000000000010900000000000000000000000000000000000000000000
007F007F00550000007F007F007F007F00000000000000000000000000000000007F007F007F0055007F007F007F007F
00000000000000000000000000000000000000000000000000000000010A000000000000000000000000000000000000
0000000000000000007F010B00000000007F007F0000010C010B000000000000007F007F007F007F007F007F007F007F
000000000000000000000000000000000000000000000000010D00000000000000000000000000000000000000000000
0000000000000000010D00000000000000000000000000000300000000000000000000040000280F0001000000000000
00000000018000000000000000011800FFE078000000FF0000000000000040000000000000001800FFFC000000000000
18000000000000008000C000000000000000F800B00000000000000000FC80000C3A020000066010800000304F7F8660
2C92020000067811FF80003FA1FDC6780C1202000004401101FC0030FFFEC4400C12020000066011FF0000304F1FC660
3C0038E729C23813F800003FFF7EC2380C00020000022000007F0030D89FC2200C10020000022000FFF100309F9FC220
000000000002200000000030000F0220D004000003800011FFE3003F00A07B807800000000000001FFFFFFFFF0000000
C000005000000829FFFFFFFF0C0080A00001E000000001002000000001000000FFFFFFFFF8002000000000000000DF40
00000000C280C20080C200000000C20000000000008000C20000000000C20000E00000001800000000000000FC000000
C0C000000000000000000000E0000000FE00000000000000FF8000007FC00000FFF22000FFF00000FC00FC00C0000000
0000F80000000000FFC0000000000000F000F00080000000FFE0C0000000000E0000F00000000000000000003800FC00
000000003000000060000000800000000000C000FC00FC00FFFFFFFFFFFF8000800000000000E0000FF0000000000000
07000000000000000000000000001C00180000000000FE00F80000000000FF0000000000C0C00000C00000005500C0C0
00200000000000008023000010300020000C00200000000000000000E0008000FFFE00000000FFFE000000000000F000
FFFFFF800000000000000000FFFFF8000030000000000000000000000040000001F00000000000000000DF4000000000
7FFE7F000000000080808080FF8000000000000080808080FFFFFFFFC00000000000000004000000FFF0000000000000
0000FFFFFFC000000000000000000001000100000000001F000000000000800000007FF0000000000000000080000000
000000000000E000000000000000FF80FFFFF00000000000FF000000000000000003FFFFFC14F800FC00E00000000000
00000000FC003FC0E00000007FF00000800000003C004000FF80000000000000000000000C00C000FF80000007FFFFF8
00008080FF818181FC00C000000000000000FFF000000000F0000000000007800000C00000000000FFFFFFFFFC000000
A08000001F07FF800000000000000024000000000007FFF800000000000300000000FFFFFFFF7F000020F08000080000
60000000000000008000000000000000C1FF8080E30303034800008000001000FFFFFFFFC000C000F800000000000000
0070000000000078FFFFFFFEE0008000C00000000000FFFFFFFFFFFFFFFFFFFFF0000000FFFE000000001FF000000000
F80000000000F8000000000040000000FFFFFFFFFFC000F00000FC00C0000000F000000000F000000000FF0000000000
08007FF000000000E404000400480800FFFFFF00FFC00000F8020000000000406E400000000002C0FFFF007F80000000
07C80000FFFFFFFF7C000000700000000F0000000000000078C0000001100F9000000000FE00FE00FFFFFFFF00000000
FF8007800000000001C000000000000000F8000000C00000FFFF01FFE1FC0000FFFFFFFFFFFFFE00FFF8000000000000
03F8000000000000FC00FF000000000080000000FFFFFFFFFFFCC400000000001FFFFFFFFFFFFFFF0000FFFFFC000000
0000FFFFFFFFFC00FF800000FFFFF0007FC000000003C000FC00FE000000DFF8FF8000000000FF00FFE0000100000000
0000000000040000FFFFFFFFFFFFFFFC0000FC0040004280FC00F800000000000412020000066010FFE0E0301F7EC660
FFFFFFFC10000000FFFFFFFFFC00FF0000C0000000000000FFFFE000FC00FFE0FC00000000000000FFFFFFFFFFFFFC00
0000F00018000000FFFFFFFFFFFFFF80F00000000000000000000000FFFFFFFF7FF80000000000000640000000900D80
FFFFFFFFFC00FF8000000300FFFFFFFFFFFFFFE003000000000000000000FF000000FFF8000000000080000000000200
0000E0000000FFC0FF800100000300004B8000000000048000000240FC00FF00FFFFFC00FE048000FE000000FFFFFFFF
3800000000020000FFFEFFFFFFFFFFFF7FFC000000000000FFE0800000000000FFFFFFFFFFFFFFF0000000000000FFFF
FFFFFFFFFFC0000000003C0080000000FFC0C0000000FC001F0000040400FFC0FFFFFFFFFFFF0000FFFFFFFFF8000000
0000000000007800FFFFFFFF00007F00FFFCFFE0FFFFFFFF9010FFFFFFFFFFFFFFFBFFF8000000000000FF0FFFD8FFFF
E000F80000000000FFFFFFF00C00FE000000018000000000FFFFF80000000000FFFFFFFFFFFFFFC0FFF00000FFF00000
FE000000FF80000000000000002000001400219B200000000000000000000010840000002020184000000000000203A0
000000C0000000000000000000003000FFFF000107FFF000FFFFF81F800000000000F82406000080FFFFC00000000000
FFFFFFFFFFFF7FFFC000E00000000000FFFFFFFFFFFF3C00FFFF80000000FFFF7C00000000000000FC0000000000FFFF
80009080FFFFFFFFFFFFFFFFFF800060FFFFFFFF3C00F0000001FFFFFFFFFFFFFFE0000000000000C000000000000001
F508016900000010A10808695569157BF0000411F0000400FFFCFFFFFFFFFFFF00018000FFF00000FFC0000000010001
0000003FFFFFFFFFF00000000000FFF8FFFFFFC0FFFCFE00E000E0000F0000000780000000000000FFFEF000FC000000
00000000FC00FF00FFFCC0000000FF00E000C000FFF00000400000000000FE00FE00FE00F0003FC00000000000080000
FC00FFFFFFFFF80000000000C00000000000FFFC000000000000FFFE00000000000000000000F800FFFF000000000000
00000000FFFFFFFDC000000000000000C000000000000000",
);
enum MAX_SIMPLE_LOWER = 1433;
enum MAX_SIMPLE_UPPER = 1450;
enum MAX_SIMPLE_TITLE = 1454;
//10496 bytes
enum toUpperIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000140",
cast(immutable size_t[]) x"
000000000000010000000000000004800000000000000F80",
cast(immutable size_t[]) x"
040203020202010007020202060202050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
02020202020202020202020202020202000300020001000000070006000500040000000A00090008000D000C000B0000
00110010000F000E00000014001300120000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001500000000000000000000000000000000000000000000001600000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000017000000000000001900180000001D001C001B001A00210020001F001E00000000000000000000002300220000
000000000000000000000000000000000024000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000280027002600250000000000000029
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000002B002A0000002F002E002D002C00000000000000000000000000000000
000000000000000000000031003000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000320000000000000000000000000000000000000000000000000000000000330000
000000000000000000000000000000000000000000000000000000000000000000360000003500340000003700000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
003800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
003900000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000003A00000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000003C003B0000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000200010000FFFF0006000500040003000A000900080007000E000D000C000B
001200110010000F0016001500140013FFFF001900180017FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001AFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05AAFFFFFFFFFFFF001E001D001C001B002200210020001F
0026002500240023002A002900280027002E002D002C002BFFFF00310030002F00350034003300320039003800370036
003BFFFF003AFFFF003DFFFF003CFFFF003FFFFF003EFFFF0041FFFF0040FFFF0043FFFF0042FFFF0045FFFF0044FFFF
0047FFFF0046FFFF0049FFFF0048FFFF004BFFFF004AFFFF004DFFFF004CFFFF004FFFFF004EFFFF0051FFFF0050FFFF
0053FFFF005205AC0055FFFF0054FFFFFFFF0056FFFFFFFFFFFF0058FFFF0057FFFF005AFFFF0059FFFF005CFFFF005B
005EFFFF05C9005D0060FFFF005FFFFF0062FFFF0061FFFF0064FFFF0063FFFF0066FFFF0065FFFF0068FFFF0067FFFF
006AFFFF0069FFFF006CFFFF006BFFFF006EFFFF006DFFFF0070FFFF006FFFFF0072FFFF0071FFFF0074FFFF0073FFFF
FFFF0075FFFFFFFF00780077FFFF0076007AFFFFFFFF0079FFFFFFFF007BFFFFFFFFFFFFFFFF007CFFFFFFFFFFFF007D
FFFF007EFFFFFFFFFFFFFFFF007FFFFFFFFF00810080FFFFFFFF0082FFFFFFFF0084FFFF0083FFFFFFFFFFFF0085FFFF
FFFFFFFFFFFF0086FFFFFFFF0087FFFFFFFFFFFFFFFF0088FFFF008AFFFF0089FFFFFFFF008BFFFF008DFFFF008CFFFF
FFFFFFFFFFFFFFFFFFFF008F008EFFFF0092FFFF00910090FFFF0094FFFF0093FFFF0096FFFF0095FFFF0098FFFF0097
FFFF009AFFFF0099009DFFFF009C009B009FFFFF009EFFFF00A1FFFF00A0FFFF00A3FFFF00A2FFFF00A5FFFF00A4FFFF
00A700A6FFFF05D1FFFFFFFF00A8FFFF00AAFFFF00A9FFFF00ACFFFF00ABFFFF00AEFFFF00ADFFFF00B0FFFF00AFFFFF
00B2FFFF00B1FFFF00B4FFFF00B3FFFF00B6FFFF00B5FFFF00B8FFFF00B7FFFF00BAFFFF00B9FFFF00BCFFFF00BBFFFF
00BDFFFFFFFFFFFF00BFFFFF00BEFFFF00C1FFFF00C0FFFF00C3FFFF00C2FFFF00C5FFFF00C4FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00C7FFFFFFFF00C6FFFF00C9FFFF00C800CAFFFFFFFFFFFF00CCFFFF00CBFFFF00CEFFFF00CDFFFF
00D200D100D000CF00D500D4FFFF00D300D7FFFF00D6FFFFFFFFFFFFFFFF00D800DBFFFF00DA00D9FFFF00DD00DCFFFF
00E100E000DF00DE00E3FFFFFFFF00E2FFFF00E500E4FFFFFFFFFFFF00E6FFFFFFFFFFFFFFFFFFFFFFFFFFFF00E7FFFF
00EA00E9FFFF00E800EBFFFFFFFFFFFF00EF00EE00ED00ECFFFFFFFFFFFF00F0FFFF00F1FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF00F300F2FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F4FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F6FFFF00F5FFFF00F7FFFFFFFFFFFF
00F8FFFFFFFFFFFFFFFFFFFF00FA00F9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF05CBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00FE00FD00FC00FB0101010000FF05CE01050104010301020109010801070106010D010C010B010A
01110110010F010E01150114011301120119011801170116FFFF011C011B011AFFFFFFFF011E011D01210120011FFFFF
0123FFFF0122FFFF0125FFFF0124FFFF0127FFFF0126FFFF0129FFFF0128FFFF012BFFFF012AFFFF012DFFFF012CFFFF
01310130012F012EFFFFFFFF0132FFFF0134FFFFFFFF0133FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0138013701360135013C013B013A0139
0140013F013E013D01440143014201410148014701460145014C014B014A01490150014F014E014D0154015301520151
0158015701560155015C015B015A01590160015F015E015D01640163016201610166FFFF0165FFFF0168FFFF0167FFFF
016AFFFF0169FFFF016CFFFF016BFFFF016EFFFF016DFFFF0170FFFF016FFFFF0172FFFF0171FFFF0174FFFF0173FFFF
FFFFFFFF0175FFFFFFFFFFFFFFFFFFFF0176FFFFFFFFFFFF0178FFFF0177FFFF017AFFFF0179FFFF017CFFFF017BFFFF
017EFFFF017DFFFF0180FFFF017FFFFF0182FFFF0181FFFF0184FFFF0183FFFF0186FFFF0185FFFF0188FFFF0187FFFF
018AFFFF0189FFFF018CFFFF018BFFFF018EFFFF018DFFFF0190FFFF018FFFFFFFFF0191FFFFFFFFFFFF0193FFFF0192
FFFF0195FFFF019401980197FFFF0196019AFFFF0199FFFF019CFFFF019BFFFF019EFFFF019DFFFF01A0FFFF019FFFFF
01A2FFFF01A1FFFF01A4FFFF01A3FFFF01A6FFFF01A5FFFF01A8FFFF01A7FFFF01AAFFFF01A9FFFF01ACFFFF01ABFFFF
01AEFFFF01ADFFFF01B0FFFF01AFFFFF01B2FFFF01B1FFFF01B4FFFF01B3FFFF01B6FFFF01B5FFFF01B8FFFF01B7FFFF
01BAFFFF01B9FFFF01BCFFFF01BBFFFF01BEFFFF01BDFFFF01C0FFFF01BFFFFF01C2FFFF01C1FFFF01C4FFFF01C3FFFF
01C6FFFF01C5FFFF01C8FFFF01C7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01CB01CA01C9FFFF01CF01CE01CD01CC01D301D201D101D001D701D601D501D4
01DB01DA01D901D801DF01DE01DD01DC01E301E201E101E001E701E601E501E401EB01EA01E901E805BD01EE01ED01EC
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01F201F101F001EF01F601F501F401F301FA01F901F801F701FE01FD01FC01FB02020201020001FF0206020502040203
020A020902080207020E020D020C020B021202110210020F0216021502140213FFFF021902180217021C021B021AFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0220021F021E021DFFFFFFFF022202210226022502240223022A022902280227
FFFFFFFFFFFF022BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022CFFFFFFFFFFFF022DFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022EFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0230FFFF022FFFFF0232FFFF0231FFFF
0234FFFF0233FFFF0236FFFF0235FFFF0238FFFF0237FFFF023AFFFF0239FFFF023CFFFF023BFFFF023EFFFF023DFFFF
0240FFFF023FFFFF0242FFFF0241FFFF0244FFFF0243FFFF0246FFFF0245FFFF0248FFFF0247FFFF024AFFFF0249FFFF
024CFFFF024BFFFF024EFFFF024DFFFF0250FFFF024FFFFF0252FFFF0251FFFF0254FFFF0253FFFF0256FFFF0255FFFF
0258FFFF0257FFFF025AFFFF0259FFFF025CFFFF025BFFFF025EFFFF025DFFFF0260FFFF025FFFFF0262FFFF0261FFFF
0264FFFF0263FFFF0266FFFF0265FFFF0268FFFF0267FFFF026AFFFF0269FFFF026CFFFF026BFFFF026EFFFF026DFFFF
0270FFFF026FFFFF0272FFFF0271FFFF0274FFFF0273FFFF0276FFFF0275FFFF0278FFFF0277FFFF05D505D30279FFFF
027A05DB05D905D7FFFFFFFFFFFFFFFF027CFFFF027BFFFF027EFFFF027DFFFF0280FFFF027FFFFF0282FFFF0281FFFF
0284FFFF0283FFFF0286FFFF0285FFFF0288FFFF0287FFFF028AFFFF0289FFFF028CFFFF028BFFFF028EFFFF028DFFFF
0290FFFF028FFFFF0292FFFF0291FFFF0294FFFF0293FFFF0296FFFF0295FFFF0298FFFF0297FFFF029AFFFF0299FFFF
029CFFFF029BFFFF029EFFFF029DFFFF02A0FFFF029FFFFF02A2FFFF02A1FFFF02A4FFFF02A3FFFF02A6FFFF02A5FFFF
02A8FFFF02A7FFFF02AAFFFF02A9FFFF02AE02AD02AC02AB02B202B102B002AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02B602B502B402B3FFFFFFFF02B802B7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02BC02BB02BA02B902C002BF02BE02BD
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02C402C302C202C102C802C702C602C5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02CC02CB02CA02C9FFFFFFFF02CE02CDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D005DF02CF05DD02D205E502D105E2
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D602D502D402D302DA02D902D802D7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02DE02DD02DC02DB02E202E102E002DF02E602E502E402E3FFFFFFFF02E802E7060C060A06080606061406120610060E
061C061A06180616062406220620061E062C062A06280626063406320630062E063C063A06380636064406420640063E
064C064A06480646065406520650064E065C065A06580656066406620660065E0666067203020301067E05E8FFFF0674
FFFFFFFFFFFFFFFFFFFF0304FFFF0668066A0676FFFFFFFF068105EAFFFF0678FFFFFFFFFFFFFFFFFFFFFFFFFFFF066C
05EF05EC0307030605F405F2FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05FA05F703090308060105FF030A05FD
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF066E067AFFFFFFFF06840604FFFF067CFFFFFFFFFFFFFFFFFFFFFFFFFFFF0670
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030CFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0310030F030E030D03140313031203110318031703160315031C031B031A0319FFFFFFFFFFFFFFFFFFFFFFFFFFFF031D
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03210320031F031E03250324032303220329032803270326032D032C032B032A03310330032F032E0335033403330332
FFFFFFFF03370336FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
033B033A03390338033F033E033D033C03430342034103400347034603450344034B034A03490348034F034E034D034C
03530352035103500357035603550354035B035A03590358035F035E035D035C03630362036103600367036603650364
FFFFFFFF0368FFFFFFFF036A0369FFFFFFFF036CFFFF036BFFFFFFFFFFFF036D036EFFFFFFFFFFFFFFFF036FFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0371FFFF0370FFFF0373FFFF0372FFFF0375FFFF0374FFFF0377FFFF0376FFFF
0379FFFF0378FFFF037BFFFF037AFFFF037DFFFF037CFFFF037FFFFF037EFFFF0381FFFF0380FFFF0383FFFF0382FFFF
0385FFFF0384FFFF0387FFFF0386FFFF0389FFFF0388FFFF038BFFFF038AFFFF038DFFFF038CFFFF038FFFFF038EFFFF
0391FFFF0390FFFF0393FFFF0392FFFF0395FFFF0394FFFF0397FFFF0396FFFF0399FFFF0398FFFF039BFFFF039AFFFF
039DFFFF039CFFFF039FFFFF039EFFFF03A1FFFF03A0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A3FFFF03A2
03A4FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A803A703A603A503AC03AB03AA03A9
03B003AF03AE03AD03B403B303B203B103B803B703B603B503BC03BB03BA03B903C003BF03BE03BD03C403C303C203C1
03C803C703C603C503CBFFFF03CA03C9FFFFFFFFFFFFFFFFFFFFFFFF03CCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03CEFFFF03CDFFFF03D0FFFF03CFFFFF03D2FFFF03D1FFFF03D4FFFF03D3FFFF
03D6FFFF03D5FFFF03D8FFFF03D7FFFF03DAFFFF03D9FFFF03DCFFFF03DBFFFF03DEFFFF03DDFFFF03E0FFFF03DFFFFF
03E2FFFF03E1FFFFFFFFFFFF03E3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03E5FFFF03E4FFFF03E7FFFF03E6FFFF03E9FFFF03E8FFFF03EBFFFF03EAFFFF03EDFFFF03ECFFFF03EFFFFF03EEFFFF
03F1FFFF03F0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03F2FFFFFFFFFFFF03F4FFFF03F3FFFF03F6FFFF03F5FFFF03F8FFFF03F7FFFF03F9FFFFFFFFFFFF03FBFFFF03FAFFFF
03FDFFFF03FCFFFF03FFFFFF03FEFFFF0401FFFF0400FFFF0403FFFF0402FFFF0405FFFF0404FFFF0407FFFF0406FFFF
0409FFFF0408FFFF040BFFFF040AFFFF040DFFFF040CFFFF040FFFFF040EFFFF0411FFFF0410FFFF0413FFFF0412FFFF
0415FFFF0414FFFF0417FFFF0416FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0418FFFFFFFF041AFFFFFFFF0419
041CFFFF041BFFFF041EFFFF041DFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF041F0421FFFF0420FFFF0423FFFFFFFF0422
0425FFFF0424FFFF0427FFFF0426FFFF0429FFFF0428FFFF042BFFFF042AFFFFFFFFFFFF042CFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF042EFFFF042DFFFF0430FFFF042FFFFF0432FFFF0431FFFF0434FFFF0433FFFFFFFFFFFFFFFFFFFF
FFFF0436FFFF0435FFFFFFFFFFFFFFFFFFFFFFFF0437FFFF0438FFFFFFFFFFFFFFFFFFFF0439FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043AFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
043BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043F043E043D043C04430442044104400447044604450444044B044A04490448
044F044E044D044C04530452045104500457045604550454045B045A04590458045F045E045D045C0463046204610460
0467046604650464046B046A04690468046F046E046D046C04730472047104700477047604750474047B047A04790478
047F047E047D047C04830482048104800487048604850484048B048A0489048805B305B105AF05ADFFFF05BB05B905B6
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05BFFFFFFFFFFFFF05C705C505C305C1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF048E048D048CFFFF049204910490048F0496049504940493049A049904980497
049E049D049C049B04A204A104A0049FFFFF04A504A404A3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04A904A804A704A604AD04AC04AB04AA
04B104B004AF04AE04B504B404B304B204B904B804B704B604BD04BC04BB04BA04C104C004BF04BE04C504C404C304C2
04C904C804C704C604CD04CC04CB04CAFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04D104D004CF04CE04D504D404D304D204D904D804D704D604DD04DC04DB04DA
04E104E004DF04DE04E504E404E304E204E904E804E704E604ED04EC04EB04EA04F104F004EF04EEFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04F2FFFFFFFFFFFF
04F604F504F404F304FA04F904F804F704FDFFFF04FC04FB0501050004FF04FE05050504050305020509050805070506
050CFFFF050B050A0510050F050E050D0513FFFF05120511FFFFFFFFFFFF05140518051705160515051C051B051A0519
0520051F051E051D05240523052205210528052705260525052C052B052A05290530052F052E052D0534053305320531
0538053705360535053C053B053A05390540053F053E053D0544054305420541FFFF054705460545FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF054B054A05490548054F054E054D054C05530552055105500557055605550554
055B055A05590558055F055E055D055C05630562056105600567056605650564FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF056B056A05690568056F056E056D056C05730572057105700577057605750574
057B057A05790578057F057E057D057C05830582058105800587058605850584FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
05890588FFFFFFFF058D058C058B058A05910590058F058E05950594059305920599059805970596059D059C059B059A
05A105A0059F059E05A505A405A305A205A905A805A705A6FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//10112 bytes
enum toLowerIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000140",
cast(immutable size_t[]) x"
000000000000010000000000000004800000000000000EC0",
cast(immutable size_t[]) x"
040203020202010007020202060202050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
02020202020202020202020202020202000200000001000000060005000400030000000000080007000B000A00090000
000F000E000D000C00000012001100100000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001400130000000000000000000000000000000000000000001600150000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000017000000000000000000000000001B001A00190018001F001E001D001C00000000000000000000002200210020
000000000000000000000000000000000024002300000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000280027002600250000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000002A00290000002E002D002C002B00000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000002F0000000000000000000000000000000000000000000000000000000000000030
000000000000000000000000000000000000000000000000000000000000000000330032000000310000003500340000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000360000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000370000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000003800000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000390000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
000200010000FFFF0006000500040003000A000900080007000E000D000C000B001200110010000F0016001500140013
FFFF001900180017FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001D001C001B001A00210020001F001E
00250024002300220029002800270026002D002C002B002AFFFF0030002F002E00340033003200310599003700360035
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0039FFFF0038FFFF003BFFFF003AFFFF003DFFFF003CFFFF003FFFFF003E
FFFF0041FFFF0040FFFF0043FFFF0042FFFF0045FFFF0044FFFF0047FFFF0046FFFF0049FFFF0048FFFF004BFFFF004A
FFFF004DFFFF004CFFFF004FFFFF004EFFFF0051FFFF059AFFFF0053FFFF00520055FFFF0054FFFF0057FFFF0056FFFF
0059FFFF0058FFFF005BFFFF005AFFFFFFFF005C05A9FFFFFFFF005EFFFF005DFFFF0060FFFF005FFFFF0062FFFF0061
FFFF0064FFFF0063FFFF0066FFFF0065FFFF0068FFFF0067FFFF006AFFFF0069FFFF006CFFFF006BFFFF006EFFFF006D
FFFF0070FFFF006FFFFF0072FFFF00710075FFFF00740073FFFFFFFF0076FFFFFFFF00780077FFFF007B007AFFFF0079
007E007D007CFFFF0080007FFFFFFFFF0083FFFF0082008100860085FFFF0084FFFFFFFFFFFF0087008AFFFF00890088
FFFF008CFFFF008B008F008EFFFF008DFFFFFFFF0090FFFF00930092FFFF0091009600950094FFFF0098FFFF0097FFFF
FFFFFFFFFFFF0099FFFFFFFFFFFF009AFFFFFFFFFFFFFFFF009DFFFF009C009B00A0009FFFFF009E00A2FFFF00A1FFFF
00A4FFFF00A3FFFF00A6FFFF00A5FFFF00A8FFFF00A7FFFFFFFF00A9FFFFFFFFFFFF00ABFFFF00AAFFFF00ADFFFF00AC
FFFF00AFFFFF00AEFFFF00B1FFFF00B0FFFF00B300B205AC00B600B5FFFF00B4FFFF00B8FFFF00B7FFFF00BAFFFF00B9
FFFF00BCFFFF00BBFFFF00BEFFFF00BDFFFF00C0FFFF00BFFFFF00C2FFFF00C1FFFF00C4FFFF00C3FFFF00C6FFFF00C5
FFFF00C8FFFF00C7FFFF00CAFFFF00C9FFFF00CCFFFF00CBFFFF00CEFFFF00CDFFFF00D0FFFF00CFFFFF00D2FFFF00D1
FFFF00D4FFFF00D3FFFFFFFFFFFFFFFF00D600D5FFFFFFFFFFFF00D800D7FFFF00DAFFFF00D9FFFFFFFF00DD00DC00DB
FFFF00DFFFFF00DEFFFF00E1FFFF00E0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00E3FFFF00E2FFFF00E4FFFFFFFFFFFFFFFFFFFFFFFF00E5FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF00E6FFFFFFFFFFFF00E900E800E700EC00EBFFFF00EA00EF00EE00ED05AA00F300F200F100F0
00F700F600F500F400FB00FA00F900F800FEFFFF00FD00FC01020101010000FF0106010501040103FFFFFFFFFFFFFFFF
FFFFFFFFFFFF05ABFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF0107FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0109FFFF0108FFFF010BFFFF010A
FFFF010DFFFF010CFFFF010FFFFF010EFFFF0111FFFF0110FFFF0113FFFF0112FFFFFFFFFFFFFFFF0115FFFFFFFF0114
FFFF01170116FFFF011A01190118FFFF011E011D011C011B012201210120011F0126012501240123012A012901280127
012E012D012C012B013201310130012F0136013501340133013A013901380137013E013D013C013B014201410140013F
0146014501440143014A014901480147FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF014CFFFF014BFFFF014EFFFF014DFFFF0150FFFF014FFFFF0152FFFF0151
FFFF0154FFFF0153FFFF0156FFFF0155FFFF0158FFFF0157FFFF015AFFFF0159FFFFFFFFFFFF015BFFFFFFFFFFFFFFFF
FFFF015CFFFFFFFFFFFF015EFFFF015DFFFF0160FFFF015FFFFF0162FFFF0161FFFF0164FFFF0163FFFF0166FFFF0165
FFFF0168FFFF0167FFFF016AFFFF0169FFFF016CFFFF016BFFFF016EFFFF016DFFFF0170FFFF016FFFFF0172FFFF0171
FFFF0174FFFF0173FFFF0176FFFF01750179FFFF01780177017BFFFF017AFFFF017DFFFF017CFFFFFFFFFFFF017EFFFF
FFFF0180FFFF017FFFFF0182FFFF0181FFFF0184FFFF0183FFFF0186FFFF0185FFFF0188FFFF0187FFFF018AFFFF0189
FFFF018CFFFF018BFFFF018EFFFF018DFFFF0190FFFF018FFFFF0192FFFF0191FFFF0194FFFF0193FFFF0196FFFF0195
FFFF0198FFFF0197FFFF019AFFFF0199FFFF019CFFFF019BFFFF019EFFFF019DFFFF01A0FFFF019FFFFF01A2FFFF01A1
FFFF01A4FFFF01A3FFFF01A6FFFF01A5FFFF01A8FFFF01A7FFFF01AAFFFF01A9FFFF01ACFFFF01ABFFFF01AEFFFF01AD
01B101B001AFFFFF01B501B401B301B201B901B801B701B601BD01BC01BB01BA01C101C001BF01BE01C501C401C301C2
01C901C801C701C601CD01CC01CB01CA01D101D001CF01CEFFFF01D401D301D2FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05A3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01D801D701D601D501DC01DB01DA01D901E001DF01DE01DD01E401E301E201E1
01E801E701E601E501EC01EB01EA01E901F001EF01EE01ED01F401F301F201F101F801F701F601F501FBFFFF01FA01F9
FFFFFFFFFFFFFFFFFFFFFFFF01FCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF020001FF01FE01FD0204020302020201
0208020702060205020C020B020A02090210020F020E020D02140213021202110218021702160215021C021B021A0219
0220021F021E021D02240223022202210228022702260225022C022B022A02290230022F022E022D0234023302320231
0238023702360235023C023B023A02390240023F023E023D02440243024202410248024702460245024C024B024A0249
0250024F024E024DFFFFFFFF02520251FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0256025502540253025A025902580257025E025D025C025B026202610260025F
0266026502640263026A026902680267026E026D026C026B027202710270026F0276027502740273027A027902780277
FFFF027D027C027B0280027F027EFFFFFFFF0282FFFF0281FFFF0284FFFF0283FFFF0286FFFF0285FFFF0288FFFF0287
FFFF028AFFFF0289FFFF028CFFFF028BFFFF028EFFFF028DFFFF0290FFFF028FFFFF0292FFFF0291FFFF0294FFFF0293
FFFF0296FFFF0295FFFF0298FFFF0297FFFF029AFFFF0299FFFF029CFFFF029BFFFF029EFFFF029DFFFF02A0FFFF029F
FFFF02A2FFFF02A1FFFF02A4FFFF02A3FFFF02A6FFFF02A5FFFF02A8FFFF02A7FFFF02AAFFFF02A9FFFF02ACFFFF02AB
FFFF02AEFFFF02ADFFFF02B0FFFF02AFFFFF02B2FFFF02B1FFFF02B4FFFF02B3FFFF02B6FFFF02B5FFFF02B8FFFF02B7
FFFF02BAFFFF02B9FFFF02BCFFFF02BBFFFF02BEFFFF02BDFFFF02C0FFFF02BFFFFF02C2FFFF02C1FFFF02C4FFFF02C3
FFFF02C6FFFF02C5FFFF02C8FFFF02C7FFFF02CAFFFF02C905AE05ADFFFF02CBFFFF05B105B005AFFFFF02CCFFFFFFFF
FFFF02CEFFFF02CDFFFF02D0FFFF02CFFFFF02D2FFFF02D1FFFF02D4FFFF02D3FFFF02D6FFFF02D5FFFF02D8FFFF02D7
FFFF02DAFFFF02D9FFFF02DCFFFF02DBFFFF02DEFFFF02DDFFFF02E0FFFF02DFFFFF02E2FFFF02E1FFFF02E4FFFF02E3
FFFF02E6FFFF02E5FFFF02E8FFFF02E7FFFF02EAFFFF02E9FFFF02ECFFFF02EBFFFF02EEFFFF02EDFFFF02F0FFFF02EF
FFFF02F2FFFF02F1FFFF02F4FFFF02F3FFFF02F6FFFF02F5FFFF02F8FFFF02F7FFFF02FAFFFF02F9FFFF02FCFFFF02FB
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030002FF02FE02FD0304030303020301FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0308030703060305FFFFFFFF030A0309FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030E030D030C030B031203110310030F
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0316031503140313031A031903180317FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
031E031D031C031BFFFFFFFF0320031FFFFF05B3FFFF05B2FFFF05B5FFFF05B40322FFFF0321FFFF0324FFFF0323FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0328032703260325032C032B032A0329FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05C505C405C305C205C905C805C705C605CD05CC05CB05CA05D105D005CF05CE
05D505D405D305D205D905D805D705D605DD05DC05DB05DA05E105E005DF05DE05E505E405E305E205E905E805E705E6
05ED05EC05EB05EA05F105F005EF05EE05F205F8FFFFFFFF05FE05B6FFFF05F90348034703460345FFFFFFFFFFFF05F3
05F405FAFFFFFFFF05FF05B7FFFF05FB034D034C034B034AFFFFFFFFFFFF05F505B905B8FFFFFFFF05BB05BAFFFFFFFF
035203510350034FFFFFFFFFFFFFFFFF05BD05BCFFFFFFFF05C005BFFFFF05BE0356035503540353FFFFFFFFFFFF0357
05F605FCFFFFFFFF060005C1FFFF05FD035B035A03590358FFFFFFFFFFFF05F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF035DFFFFFFFF035F035EFFFFFFFFFFFFFFFFFFFFFFFFFFFF0360FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03640363036203610368036703660365
036C036B036A03690370036F036E036DFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0371FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03730372FFFFFFFF
0377037603750374037B037A03790378037F037E037D037C03830382038103800387038603850384038B038A03890388
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
038F038E038D038C03930392039103900397039603950394039B039A03990398039F039E039D039C03A303A203A103A0
03A703A603A503A403AB03AA03A903A803AF03AE03AD03AC03B303B203B103B003B703B603B503B403BB03BA03B903B8
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03BE03BDFFFF03BC03C0FFFFFFFF03BF03C2FFFF03C1FFFF03C503C403C3FFFFFFFF03C7FFFF03C6FFFFFFFF03C8FFFF
FFFFFFFFFFFFFFFF03CA03C9FFFFFFFFFFFF03CCFFFF03CBFFFF03CEFFFF03CDFFFF03D0FFFF03CFFFFF03D2FFFF03D1
FFFF03D4FFFF03D3FFFF03D6FFFF03D5FFFF03D8FFFF03D7FFFF03DAFFFF03D9FFFF03DCFFFF03DBFFFF03DEFFFF03DD
FFFF03E0FFFF03DFFFFF03E2FFFF03E1FFFF03E4FFFF03E3FFFF03E6FFFF03E5FFFF03E8FFFF03E7FFFF03EAFFFF03E9
FFFF03ECFFFF03EBFFFF03EEFFFF03EDFFFF03F0FFFF03EFFFFF03F2FFFF03F1FFFF03F4FFFF03F3FFFF03F6FFFF03F5
FFFF03F8FFFF03F7FFFF03FAFFFF03F9FFFF03FCFFFF03FBFFFFFFFFFFFFFFFF03FDFFFFFFFFFFFFFFFFFFFF03FEFFFF
FFFF03FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0401FFFF0400FFFF0403FFFF0402
FFFF0405FFFF0404FFFF0407FFFF0406FFFF0409FFFF0408FFFF040BFFFF040AFFFF040DFFFF040CFFFF040FFFFF040E
FFFF0411FFFF0410FFFF0413FFFF0412FFFF0415FFFF0414FFFFFFFFFFFF0416FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0418FFFF0417FFFF041AFFFF0419FFFF041CFFFF041BFFFF041EFFFF041D
FFFF0420FFFF041FFFFF0422FFFF0421FFFF0424FFFF0423FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0425FFFFFFFFFFFF0427FFFF0426FFFF0429FFFF0428FFFF042BFFFF042A
FFFF042CFFFFFFFFFFFF042EFFFF042DFFFF0430FFFF042FFFFF0432FFFF0431FFFF0434FFFF0433FFFF0436FFFF0435
FFFF0438FFFF0437FFFF043AFFFF0439FFFF043CFFFF043BFFFF043EFFFF043DFFFF0440FFFF043FFFFF0442FFFF0441
FFFF0444FFFF0443FFFF0446FFFF0445FFFF0448FFFF0447FFFF044AFFFF0449FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
044CFFFF044BFFFFFFFF044E044DFFFFFFFF0450FFFF044FFFFF0452FFFF04510453FFFFFFFFFFFFFFFFFFFF0454FFFF
FFFF0456FFFF0455FFFF0457FFFFFFFFFFFF0459FFFF0458FFFF045BFFFF045AFFFF045DFFFF045CFFFF045FFFFF045E
04620461FFFF0460FFFF0465046404630469046804670466FFFF046BFFFF046AFFFF046DFFFF046CFFFF046FFFFF046E
FFFF0471FFFF04700475047404730472FFFFFFFF0476FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0477FFFF0478FFFFFFFF
FFFFFFFFFFFF0479FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF047AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF059F059E059D059CFFFF05A205A105A0
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05A4FFFFFFFFFFFF05A805A705A605A5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF047D047C047BFFFF04810480047F047E
04850484048304820489048804870486048D048C048B048A04910490048F048EFFFF049404930492FFFFFFFFFFFFFFFF
0498049704960495049C049B049A049904A0049F049E049D04A404A304A204A104A804A704A604A504AC04AB04AA04A9
04B004AF04AE04AD04B404B304B204B104B804B704B604B504BC04BB04BA04B9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04C004BF04BE04BD04C404C304C204C1
04C804C704C604C504CC04CB04CA04C904D004CF04CE04CD04D404D304D204D104D804D704D604D504DC04DB04DA04D9
04E004DF04DE04DDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
04E404E304E204E104E804E704E604E5FFFF04EB04EA04E904EF04EE04ED04EC04F304F204F104F004F704F604F504F4
FFFF04FA04F904F804FE04FD04FC04FBFFFF0501050004FFFFFFFFFF05030502FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0507050605050504050B050A05090508050F050E050D050C0513051205110510
0517051605150514051B051A05190518051F051E051D051C05230522052105200527052605250524052B052A05290528
052F052E052D052C0533053205310530FFFF053605350534FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF053A053905380537053E053D053C053B054205410540053F0546054505440543
054A054905480547054E054D054C054B055205510550054F0556055505540553055A055905580557055E055D055C055B
056205610560055F0566056505640563056A056905680567056E056D056C056B057205710570056F0576057505740573
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF057A057905780577057E057D057C057B058205810580057F0586058505840583
058A058905880587058E058D058C058B059205910590058F0596059505940593FFFFFFFF05980597FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//10496 bytes
enum toTitleIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000140",
cast(immutable size_t[]) x"
000000000000010000000000000004800000000000000F80",
cast(immutable size_t[]) x"
040203020202010007020202060202050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
02020202020202020202020202020202000300020001000000070006000500040000000A00090008000D000C000B0000
00110010000F000E00000014001300120000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001500000000000000000000000000000000000000000000001600000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000017000000000000001900180000001D001C001B001A00210020001F001E00000000000000000000002300220000
000000000000000000000000000000000024000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000280027002600250000000000000029
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000002B002A0000002F002E002D002C00000000000000000000000000000000
000000000000000000000031003000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000320000000000000000000000000000000000000000000000000000000000330000
000000000000000000000000000000000000000000000000000000000000000000360000003500340000003700000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
003800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
003900000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000003A00000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000003C003B0000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000200010000FFFF0006000500040003000A000900080007000E000D000C000B
001200110010000F0016001500140013FFFF001900180017FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001AFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05AEFFFFFFFFFFFF001E001D001C001B002200210020001F
0026002500240023002A002900280027002E002D002C002BFFFF00310030002F00350034003300320039003800370036
003BFFFF003AFFFF003DFFFF003CFFFF003FFFFF003EFFFF0041FFFF0040FFFF0043FFFF0042FFFF0045FFFF0044FFFF
0047FFFF0046FFFF0049FFFF0048FFFF004BFFFF004AFFFF004DFFFF004CFFFF004FFFFF004EFFFF0051FFFF0050FFFF
0053FFFF005205B00055FFFF0054FFFFFFFF0056FFFFFFFFFFFF0058FFFF0057FFFF005AFFFF0059FFFF005CFFFF005B
005EFFFF05CD005D0060FFFF005FFFFF0062FFFF0061FFFF0064FFFF0063FFFF0066FFFF0065FFFF0068FFFF0067FFFF
006AFFFF0069FFFF006CFFFF006BFFFF006EFFFF006DFFFF0070FFFF006FFFFF0072FFFF0071FFFF0074FFFF0073FFFF
FFFF0075FFFFFFFF00780077FFFF0076007AFFFFFFFF0079FFFFFFFF007BFFFFFFFFFFFFFFFF007CFFFFFFFFFFFF007D
FFFF007EFFFFFFFFFFFFFFFF007FFFFFFFFF00810080FFFFFFFF0082FFFFFFFF0084FFFF0083FFFFFFFFFFFF0085FFFF
FFFFFFFFFFFF0086FFFFFFFF0087FFFFFFFFFFFFFFFF0088FFFF008AFFFF0089FFFFFFFF008BFFFF008DFFFF008CFFFF
FFFFFFFFFFFFFFFF00910090008F008E0095009400930092FFFF0097FFFF0096FFFF0099FFFF0098FFFF009BFFFF009A
FFFF009DFFFF009C00A0FFFF009F009E00A2FFFF00A1FFFF00A4FFFF00A3FFFF00A6FFFF00A5FFFF00A8FFFF00A7FFFF
00AB00AA00A905D5FFFFFFFF00ACFFFF00AEFFFF00ADFFFF00B0FFFF00AFFFFF00B2FFFF00B1FFFF00B4FFFF00B3FFFF
00B6FFFF00B5FFFF00B8FFFF00B7FFFF00BAFFFF00B9FFFF00BCFFFF00BBFFFF00BEFFFF00BDFFFF00C0FFFF00BFFFFF
00C1FFFFFFFFFFFF00C3FFFF00C2FFFF00C5FFFF00C4FFFF00C7FFFF00C6FFFF00C9FFFF00C8FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00CBFFFFFFFF00CAFFFF00CDFFFF00CC00CEFFFFFFFFFFFF00D0FFFF00CFFFFF00D2FFFF00D1FFFF
00D600D500D400D300D900D8FFFF00D700DBFFFF00DAFFFFFFFFFFFFFFFF00DC00DFFFFF00DE00DDFFFF00E100E0FFFF
00E500E400E300E200E7FFFFFFFF00E6FFFF00E900E8FFFFFFFFFFFF00EAFFFFFFFFFFFFFFFFFFFFFFFFFFFF00EBFFFF
00EE00EDFFFF00EC00EFFFFFFFFFFFFF00F300F200F100F0FFFFFFFFFFFF00F4FFFF00F5FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF00F700F6FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F8FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FAFFFF00F9FFFF00FBFFFFFFFFFFFF
00FCFFFFFFFFFFFFFFFFFFFF00FE00FDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF05CFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF01020101010000FF01050104010305D20109010801070106010D010C010B010A01110110010F010E
01150114011301120119011801170116011D011C011B011AFFFF0120011F011EFFFFFFFF01220121012501240123FFFF
0127FFFF0126FFFF0129FFFF0128FFFF012BFFFF012AFFFF012DFFFF012CFFFF012FFFFF012EFFFF0131FFFF0130FFFF
0135013401330132FFFFFFFF0136FFFF0138FFFFFFFF0137FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF013C013B013A01390140013F013E013D
01440143014201410148014701460145014C014B014A01490150014F014E014D01540153015201510158015701560155
015C015B015A01590160015F015E015D01640163016201610168016701660165016AFFFF0169FFFF016CFFFF016BFFFF
016EFFFF016DFFFF0170FFFF016FFFFF0172FFFF0171FFFF0174FFFF0173FFFF0176FFFF0175FFFF0178FFFF0177FFFF
FFFFFFFF0179FFFFFFFFFFFFFFFFFFFF017AFFFFFFFFFFFF017CFFFF017BFFFF017EFFFF017DFFFF0180FFFF017FFFFF
0182FFFF0181FFFF0184FFFF0183FFFF0186FFFF0185FFFF0188FFFF0187FFFF018AFFFF0189FFFF018CFFFF018BFFFF
018EFFFF018DFFFF0190FFFF018FFFFF0192FFFF0191FFFF0194FFFF0193FFFFFFFF0195FFFFFFFFFFFF0197FFFF0196
FFFF0199FFFF0198019C019BFFFF019A019EFFFF019DFFFF01A0FFFF019FFFFF01A2FFFF01A1FFFF01A4FFFF01A3FFFF
01A6FFFF01A5FFFF01A8FFFF01A7FFFF01AAFFFF01A9FFFF01ACFFFF01ABFFFF01AEFFFF01ADFFFF01B0FFFF01AFFFFF
01B2FFFF01B1FFFF01B4FFFF01B3FFFF01B6FFFF01B5FFFF01B8FFFF01B7FFFF01BAFFFF01B9FFFF01BCFFFF01BBFFFF
01BEFFFF01BDFFFF01C0FFFF01BFFFFF01C2FFFF01C1FFFF01C4FFFF01C3FFFF01C6FFFF01C5FFFF01C8FFFF01C7FFFF
01CAFFFF01C9FFFF01CCFFFF01CBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01CF01CE01CDFFFF01D301D201D101D001D701D601D501D401DB01DA01D901D8
01DF01DE01DD01DC01E301E201E101E001E701E601E501E401EB01EA01E901E801EF01EE01ED01EC05C101F201F101F0
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01F601F501F401F301FA01F901F801F701FE01FD01FC01FB02020201020001FF0206020502040203020A020902080207
020E020D020C020B021202110210020F0216021502140213021A021902180217FFFF021D021C021B0220021F021EFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0224022302220221FFFFFFFF02260225022A022902280227022E022D022C022B
FFFFFFFFFFFF022FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0230FFFFFFFFFFFF0231FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0232FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0234FFFF0233FFFF0236FFFF0235FFFF
0238FFFF0237FFFF023AFFFF0239FFFF023CFFFF023BFFFF023EFFFF023DFFFF0240FFFF023FFFFF0242FFFF0241FFFF
0244FFFF0243FFFF0246FFFF0245FFFF0248FFFF0247FFFF024AFFFF0249FFFF024CFFFF024BFFFF024EFFFF024DFFFF
0250FFFF024FFFFF0252FFFF0251FFFF0254FFFF0253FFFF0256FFFF0255FFFF0258FFFF0257FFFF025AFFFF0259FFFF
025CFFFF025BFFFF025EFFFF025DFFFF0260FFFF025FFFFF0262FFFF0261FFFF0264FFFF0263FFFF0266FFFF0265FFFF
0268FFFF0267FFFF026AFFFF0269FFFF026CFFFF026BFFFF026EFFFF026DFFFF0270FFFF026FFFFF0272FFFF0271FFFF
0274FFFF0273FFFF0276FFFF0275FFFF0278FFFF0277FFFF027AFFFF0279FFFF027CFFFF027BFFFF05D905D7027DFFFF
027E05DF05DD05DBFFFFFFFFFFFFFFFF0280FFFF027FFFFF0282FFFF0281FFFF0284FFFF0283FFFF0286FFFF0285FFFF
0288FFFF0287FFFF028AFFFF0289FFFF028CFFFF028BFFFF028EFFFF028DFFFF0290FFFF028FFFFF0292FFFF0291FFFF
0294FFFF0293FFFF0296FFFF0295FFFF0298FFFF0297FFFF029AFFFF0299FFFF029CFFFF029BFFFF029EFFFF029DFFFF
02A0FFFF029FFFFF02A2FFFF02A1FFFF02A4FFFF02A3FFFF02A6FFFF02A5FFFF02A8FFFF02A7FFFF02AAFFFF02A9FFFF
02ACFFFF02ABFFFF02AEFFFF02ADFFFF02B202B102B002AF02B602B502B402B3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02BA02B902B802B7FFFFFFFF02BC02BBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02C002BF02BE02BD02C402C302C202C1
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02C802C702C602C502CC02CB02CA02C9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02D002CF02CE02CDFFFFFFFF02D202D1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D405E302D305E102D605E902D505E6
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02DA02D902D802D702DE02DD02DC02DBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02E202E102E002DF02E602E502E402E302EA02E902E802E7FFFFFFFF02EC02EB060D060C060B060A06110610060F060E
06150614061306120619061806170616061D061C061B061A06210620061F061E06250624062306220629062806270626
062D062C062B062A06310630062F062E06350634063306320639063806370636063A064003060305064C05ECFFFF0642
FFFFFFFFFFFFFFFFFFFF0308FFFF063B063C0644FFFFFFFF064F05EEFFFF0646FFFFFFFFFFFFFFFFFFFFFFFFFFFF063D
05F305F0030B030A05F805F6FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05FE05FB030D030C06050603030E0601
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF063E0648FFFFFFFF06520608FFFF064AFFFFFFFFFFFFFFFFFFFFFFFFFFFF063F
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0310FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03140313031203110318031703160315031C031B031A03190320031F031E031DFFFFFFFFFFFFFFFFFFFFFFFFFFFF0321
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03250324032303220329032803270326032D032C032B032A03310330032F032E03350334033303320339033803370336
FFFFFFFF033B033AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
033F033E033D033C03430342034103400347034603450344034B034A03490348034F034E034D034C0353035203510350
0357035603550354035B035A03590358035F035E035D035C03630362036103600367036603650364036B036A03690368
FFFFFFFF036CFFFFFFFF036E036DFFFFFFFF0370FFFF036FFFFFFFFFFFFF03710372FFFFFFFFFFFFFFFF0373FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0375FFFF0374FFFF0377FFFF0376FFFF0379FFFF0378FFFF037BFFFF037AFFFF
037DFFFF037CFFFF037FFFFF037EFFFF0381FFFF0380FFFF0383FFFF0382FFFF0385FFFF0384FFFF0387FFFF0386FFFF
0389FFFF0388FFFF038BFFFF038AFFFF038DFFFF038CFFFF038FFFFF038EFFFF0391FFFF0390FFFF0393FFFF0392FFFF
0395FFFF0394FFFF0397FFFF0396FFFF0399FFFF0398FFFF039BFFFF039AFFFF039DFFFF039CFFFF039FFFFF039EFFFF
03A1FFFF03A0FFFF03A3FFFF03A2FFFF03A5FFFF03A4FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A7FFFF03A6
03A8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03AC03AB03AA03A903B003AF03AE03AD
03B403B303B203B103B803B703B603B503BC03BB03BA03B903C003BF03BE03BD03C403C303C203C103C803C703C603C5
03CC03CB03CA03C903CFFFFF03CE03CDFFFFFFFFFFFFFFFFFFFFFFFF03D0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03D2FFFF03D1FFFF03D4FFFF03D3FFFF03D6FFFF03D5FFFF03D8FFFF03D7FFFF
03DAFFFF03D9FFFF03DCFFFF03DBFFFF03DEFFFF03DDFFFF03E0FFFF03DFFFFF03E2FFFF03E1FFFF03E4FFFF03E3FFFF
03E6FFFF03E5FFFFFFFFFFFF03E7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03E9FFFF03E8FFFF03EBFFFF03EAFFFF03EDFFFF03ECFFFF03EFFFFF03EEFFFF03F1FFFF03F0FFFF03F3FFFF03F2FFFF
03F5FFFF03F4FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03F6FFFFFFFFFFFF03F8FFFF03F7FFFF03FAFFFF03F9FFFF03FCFFFF03FBFFFF03FDFFFFFFFFFFFF03FFFFFF03FEFFFF
0401FFFF0400FFFF0403FFFF0402FFFF0405FFFF0404FFFF0407FFFF0406FFFF0409FFFF0408FFFF040BFFFF040AFFFF
040DFFFF040CFFFF040FFFFF040EFFFF0411FFFF0410FFFF0413FFFF0412FFFF0415FFFF0414FFFF0417FFFF0416FFFF
0419FFFF0418FFFF041BFFFF041AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF041CFFFFFFFF041EFFFFFFFF041D
0420FFFF041FFFFF0422FFFF0421FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04230425FFFF0424FFFF0427FFFFFFFF0426
0429FFFF0428FFFF042BFFFF042AFFFF042DFFFF042CFFFF042FFFFF042EFFFFFFFFFFFF0430FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF0432FFFF0431FFFF0434FFFF0433FFFF0436FFFF0435FFFF0438FFFF0437FFFFFFFFFFFFFFFFFFFF
FFFF043AFFFF0439FFFFFFFFFFFFFFFFFFFFFFFF043BFFFF043CFFFFFFFFFFFFFFFFFFFF043DFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043EFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
043FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04430442044104400447044604450444044B044A04490448044F044E044D044C
04530452045104500457045604550454045B045A04590458045F045E045D045C04630462046104600467046604650464
046B046A04690468046F046E046D046C04730472047104700477047604750474047B047A04790478047F047E047D047C
04830482048104800487048604850484048B048A04890488048F048E048D048C05B705B505B305B1FFFF05BF05BD05BA
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05C3FFFFFFFFFFFF05CB05C905C705C5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF049204910490FFFF0496049504940493049A049904980497049E049D049C049B
04A204A104A0049F04A604A504A404A3FFFF04A904A804A7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04AD04AC04AB04AA04B104B004AF04AE
04B504B404B304B204B904B804B704B604BD04BC04BB04BA04C104C004BF04BE04C504C404C304C204C904C804C704C6
04CD04CC04CB04CA04D104D004CF04CEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04D504D404D304D204D904D804D704D604DD04DC04DB04DA04E104E004DF04DE
04E504E404E304E204E904E804E704E604ED04EC04EB04EA04F104F004EF04EE04F504F404F304F2FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04F6FFFFFFFFFFFF
04FA04F904F804F704FE04FD04FC04FB0501FFFF050004FF05050504050305020509050805070506050D050C050B050A
0510FFFF050F050E05140513051205110517FFFF05160515FFFFFFFFFFFF0518051C051B051A05190520051F051E051D
05240523052205210528052705260525052C052B052A05290530052F052E052D05340533053205310538053705360535
053C053B053A05390540053F053E053D05440543054205410548054705460545FFFF054B054A0549FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF054F054E054D054C05530552055105500557055605550554055B055A05590558
055F055E055D055C05630562056105600567056605650564056B056A05690568FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF056F056E056D056C05730572057105700577057605750574057B057A05790578
057F057E057D057C05830582058105800587058605850584058B058A05890588FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
058D058CFFFFFFFF05910590058F058E05950594059305920599059805970596059D059C059B059A05A105A0059F059E
05A505A405A305A205A905A805A705A605AD05AC05AB05AAFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//10368 bytes
enum toUpperSimpleIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000140",
cast(immutable size_t[]) x"
000000000000010000000000000004800000000000000F40",
cast(immutable size_t[]) x"
040203020202010007020202060202050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
02020202020202020202020202020202000300020001000000070006000500040000000A00090008000D000C000B0000
00110010000F000E00000014001300120000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001500000000000000000000000000000000000000000000001600000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000017000000000000001900180000001D001C001B001A00210020001F001E00000000000000000000002300220000
000000000000000000000000000000000024000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000280027002600250000000000000029
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000002B002A0000002F002E002D002C00000000000000000000000000000000
000000000000000000000031003000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000320000
000000000000000000000000000000000000000000000000000000000000000000350000003400330000003600000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
003700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
003800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000003900000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000003B003A0000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000200010000FFFF0006000500040003000A000900080007000E000D000C000B
001200110010000F0016001500140013FFFF001900180017FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001AFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001E001D001C001B002200210020001F
0026002500240023002A002900280027002E002D002C002BFFFF00310030002F00350034003300320039003800370036
003BFFFF003AFFFF003DFFFF003CFFFF003FFFFF003EFFFF0041FFFF0040FFFF0043FFFF0042FFFF0045FFFF0044FFFF
0047FFFF0046FFFF0049FFFF0048FFFF004BFFFF004AFFFF004DFFFF004CFFFF004FFFFF004EFFFF0051FFFF0050FFFF
0053FFFF0052FFFF0055FFFF0054FFFFFFFF0056FFFFFFFFFFFF0058FFFF0057FFFF005AFFFF0059FFFF005CFFFF005B
005EFFFFFFFF005D0060FFFF005FFFFF0062FFFF0061FFFF0064FFFF0063FFFF0066FFFF0065FFFF0068FFFF0067FFFF
006AFFFF0069FFFF006CFFFF006BFFFF006EFFFF006DFFFF0070FFFF006FFFFF0072FFFF0071FFFF0074FFFF0073FFFF
FFFF0075FFFFFFFF00780077FFFF0076007AFFFFFFFF0079FFFFFFFF007BFFFFFFFFFFFFFFFF007CFFFFFFFFFFFF007D
FFFF007EFFFFFFFFFFFFFFFF007FFFFFFFFF00810080FFFFFFFF0082FFFFFFFF0084FFFF0083FFFFFFFFFFFF0085FFFF
FFFFFFFFFFFF0086FFFFFFFF0087FFFFFFFFFFFFFFFF0088FFFF008AFFFF0089FFFFFFFF008BFFFF008DFFFF008CFFFF
FFFFFFFFFFFFFFFFFFFF008F008EFFFF0092FFFF00910090FFFF0094FFFF0093FFFF0096FFFF0095FFFF0098FFFF0097
FFFF009AFFFF0099009DFFFF009C009B009FFFFF009EFFFF00A1FFFF00A0FFFF00A3FFFF00A2FFFF00A5FFFF00A4FFFF
00A700A6FFFFFFFFFFFFFFFF00A8FFFF00AAFFFF00A9FFFF00ACFFFF00ABFFFF00AEFFFF00ADFFFF00B0FFFF00AFFFFF
00B2FFFF00B1FFFF00B4FFFF00B3FFFF00B6FFFF00B5FFFF00B8FFFF00B7FFFF00BAFFFF00B9FFFF00BCFFFF00BBFFFF
00BDFFFFFFFFFFFF00BFFFFF00BEFFFF00C1FFFF00C0FFFF00C3FFFF00C2FFFF00C5FFFF00C4FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00C7FFFFFFFF00C6FFFF00C9FFFF00C800CAFFFFFFFFFFFF00CCFFFF00CBFFFF00CEFFFF00CDFFFF
00D200D100D000CF00D500D4FFFF00D300D7FFFF00D6FFFFFFFFFFFFFFFF00D800DBFFFF00DA00D9FFFF00DD00DCFFFF
00E100E000DF00DE00E3FFFFFFFF00E2FFFF00E500E4FFFFFFFFFFFF00E6FFFFFFFFFFFFFFFFFFFFFFFFFFFF00E7FFFF
00EA00E9FFFF00E800EBFFFFFFFFFFFF00EF00EE00ED00ECFFFFFFFFFFFF00F0FFFF00F1FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF00F300F2FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F4FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F6FFFF00F5FFFF00F7FFFFFFFFFFFF
00F8FFFFFFFFFFFFFFFFFFFF00FA00F9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00FE00FD00FC00FB0101010000FFFFFF01050104010301020109010801070106010D010C010B010A
01110110010F010E01150114011301120119011801170116FFFF011C011B011AFFFFFFFF011E011D01210120011FFFFF
0123FFFF0122FFFF0125FFFF0124FFFF0127FFFF0126FFFF0129FFFF0128FFFF012BFFFF012AFFFF012DFFFF012CFFFF
01310130012F012EFFFFFFFF0132FFFF0134FFFFFFFF0133FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0138013701360135013C013B013A0139
0140013F013E013D01440143014201410148014701460145014C014B014A01490150014F014E014D0154015301520151
0158015701560155015C015B015A01590160015F015E015D01640163016201610166FFFF0165FFFF0168FFFF0167FFFF
016AFFFF0169FFFF016CFFFF016BFFFF016EFFFF016DFFFF0170FFFF016FFFFF0172FFFF0171FFFF0174FFFF0173FFFF
FFFFFFFF0175FFFFFFFFFFFFFFFFFFFF0176FFFFFFFFFFFF0178FFFF0177FFFF017AFFFF0179FFFF017CFFFF017BFFFF
017EFFFF017DFFFF0180FFFF017FFFFF0182FFFF0181FFFF0184FFFF0183FFFF0186FFFF0185FFFF0188FFFF0187FFFF
018AFFFF0189FFFF018CFFFF018BFFFF018EFFFF018DFFFF0190FFFF018FFFFFFFFF0191FFFFFFFFFFFF0193FFFF0192
FFFF0195FFFF019401980197FFFF0196019AFFFF0199FFFF019CFFFF019BFFFF019EFFFF019DFFFF01A0FFFF019FFFFF
01A2FFFF01A1FFFF01A4FFFF01A3FFFF01A6FFFF01A5FFFF01A8FFFF01A7FFFF01AAFFFF01A9FFFF01ACFFFF01ABFFFF
01AEFFFF01ADFFFF01B0FFFF01AFFFFF01B2FFFF01B1FFFF01B4FFFF01B3FFFF01B6FFFF01B5FFFF01B8FFFF01B7FFFF
01BAFFFF01B9FFFF01BCFFFF01BBFFFF01BEFFFF01BDFFFF01C0FFFF01BFFFFF01C2FFFF01C1FFFF01C4FFFF01C3FFFF
01C6FFFF01C5FFFF01C8FFFF01C7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01CB01CA01C9FFFF01CF01CE01CD01CC01D301D201D101D001D701D601D501D4
01DB01DA01D901D801DF01DE01DD01DC01E301E201E101E001E701E601E501E401EB01EA01E901E8FFFF01EE01ED01EC
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01F201F101F001EF01F601F501F401F301FA01F901F801F701FE01FD01FC01FB02020201020001FF0206020502040203
020A020902080207020E020D020C020B021202110210020F0216021502140213FFFF021902180217021C021B021AFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0220021F021E021DFFFFFFFF022202210226022502240223022A022902280227
FFFFFFFFFFFF022BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022CFFFFFFFFFFFF022DFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF022EFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0230FFFF022FFFFF0232FFFF0231FFFF
0234FFFF0233FFFF0236FFFF0235FFFF0238FFFF0237FFFF023AFFFF0239FFFF023CFFFF023BFFFF023EFFFF023DFFFF
0240FFFF023FFFFF0242FFFF0241FFFF0244FFFF0243FFFF0246FFFF0245FFFF0248FFFF0247FFFF024AFFFF0249FFFF
024CFFFF024BFFFF024EFFFF024DFFFF0250FFFF024FFFFF0252FFFF0251FFFF0254FFFF0253FFFF0256FFFF0255FFFF
0258FFFF0257FFFF025AFFFF0259FFFF025CFFFF025BFFFF025EFFFF025DFFFF0260FFFF025FFFFF0262FFFF0261FFFF
0264FFFF0263FFFF0266FFFF0265FFFF0268FFFF0267FFFF026AFFFF0269FFFF026CFFFF026BFFFF026EFFFF026DFFFF
0270FFFF026FFFFF0272FFFF0271FFFF0274FFFF0273FFFF0276FFFF0275FFFF0278FFFF0277FFFFFFFFFFFF0279FFFF
027AFFFFFFFFFFFFFFFFFFFFFFFFFFFF027CFFFF027BFFFF027EFFFF027DFFFF0280FFFF027FFFFF0282FFFF0281FFFF
0284FFFF0283FFFF0286FFFF0285FFFF0288FFFF0287FFFF028AFFFF0289FFFF028CFFFF028BFFFF028EFFFF028DFFFF
0290FFFF028FFFFF0292FFFF0291FFFF0294FFFF0293FFFF0296FFFF0295FFFF0298FFFF0297FFFF029AFFFF0299FFFF
029CFFFF029BFFFF029EFFFF029DFFFF02A0FFFF029FFFFF02A2FFFF02A1FFFF02A4FFFF02A3FFFF02A6FFFF02A5FFFF
02A8FFFF02A7FFFF02AAFFFF02A9FFFF02AE02AD02AC02AB02B202B102B002AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02B602B502B402B3FFFFFFFF02B802B7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02BC02BB02BA02B902C002BF02BE02BD
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02C402C302C202C102C802C702C602C5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02CC02CB02CA02C9FFFFFFFF02CE02CDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D0FFFF02CFFFFF02D2FFFF02D1FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D602D502D402D302DA02D902D802D7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02DE02DD02DC02DB02E202E102E002DF02E602E502E402E3FFFFFFFF02E802E702EC02EB02EA02E902F002EF02EE02ED
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02F402F302F202F102F802F702F602F5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02FC02FB02FA02F9030002FF02FE02FDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0303FFFF03020301FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF0304FFFFFFFF0305FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF03070306FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03090308FFFFFFFF030AFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030CFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0310030F030E030D03140313031203110318031703160315031C031B031A0319FFFFFFFFFFFFFFFFFFFFFFFFFFFF031D
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03210320031F031E03250324032303220329032803270326032D032C032B032A03310330032F032E0335033403330332
FFFFFFFF03370336FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
033B033A03390338033F033E033D033C03430342034103400347034603450344034B034A03490348034F034E034D034C
03530352035103500357035603550354035B035A03590358035F035E035D035C03630362036103600367036603650364
FFFFFFFF0368FFFFFFFF036A0369FFFFFFFF036CFFFF036BFFFFFFFFFFFF036D036EFFFFFFFFFFFFFFFF036FFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0371FFFF0370FFFF0373FFFF0372FFFF0375FFFF0374FFFF0377FFFF0376FFFF
0379FFFF0378FFFF037BFFFF037AFFFF037DFFFF037CFFFF037FFFFF037EFFFF0381FFFF0380FFFF0383FFFF0382FFFF
0385FFFF0384FFFF0387FFFF0386FFFF0389FFFF0388FFFF038BFFFF038AFFFF038DFFFF038CFFFF038FFFFF038EFFFF
0391FFFF0390FFFF0393FFFF0392FFFF0395FFFF0394FFFF0397FFFF0396FFFF0399FFFF0398FFFF039BFFFF039AFFFF
039DFFFF039CFFFF039FFFFF039EFFFF03A1FFFF03A0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A3FFFF03A2
03A4FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A803A703A603A503AC03AB03AA03A9
03B003AF03AE03AD03B403B303B203B103B803B703B603B503BC03BB03BA03B903C003BF03BE03BD03C403C303C203C1
03C803C703C603C503CBFFFF03CA03C9FFFFFFFFFFFFFFFFFFFFFFFF03CCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03CEFFFF03CDFFFF03D0FFFF03CFFFFF03D2FFFF03D1FFFF03D4FFFF03D3FFFF
03D6FFFF03D5FFFF03D8FFFF03D7FFFF03DAFFFF03D9FFFF03DCFFFF03DBFFFF03DEFFFF03DDFFFF03E0FFFF03DFFFFF
03E2FFFF03E1FFFFFFFFFFFF03E3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03E5FFFF03E4FFFF03E7FFFF03E6FFFF03E9FFFF03E8FFFF03EBFFFF03EAFFFF03EDFFFF03ECFFFF03EFFFFF03EEFFFF
03F1FFFF03F0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03F2FFFFFFFFFFFF03F4FFFF03F3FFFF03F6FFFF03F5FFFF03F8FFFF03F7FFFF03F9FFFFFFFFFFFF03FBFFFF03FAFFFF
03FDFFFF03FCFFFF03FFFFFF03FEFFFF0401FFFF0400FFFF0403FFFF0402FFFF0405FFFF0404FFFF0407FFFF0406FFFF
0409FFFF0408FFFF040BFFFF040AFFFF040DFFFF040CFFFF040FFFFF040EFFFF0411FFFF0410FFFF0413FFFF0412FFFF
0415FFFF0414FFFF0417FFFF0416FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0418FFFFFFFF041AFFFFFFFF0419
041CFFFF041BFFFF041EFFFF041DFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF041F0421FFFF0420FFFF0423FFFFFFFF0422
0425FFFF0424FFFF0427FFFF0426FFFF0429FFFF0428FFFF042BFFFF042AFFFFFFFFFFFF042CFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF042EFFFF042DFFFF0430FFFF042FFFFF0432FFFF0431FFFF0434FFFF0433FFFFFFFFFFFFFFFFFFFF
FFFF0436FFFF0435FFFFFFFFFFFFFFFFFFFFFFFF0437FFFF0438FFFFFFFFFFFFFFFFFFFF0439FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043AFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
043BFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043F043E043D043C04430442044104400447044604450444044B044A04490448
044F044E044D044C04530452045104500457045604550454045B045A04590458045F045E045D045C0463046204610460
0467046604650464046B046A04690468046F046E046D046C04730472047104700477047604750474047B047A04790478
047F047E047D047C04830482048104800487048604850484048B048A04890488048E048D048CFFFF049204910490048F
0496049504940493049A049904980497049E049D049C049B04A204A104A0049FFFFF04A504A404A3FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
04A904A804A704A604AD04AC04AB04AA04B104B004AF04AE04B504B404B304B204B904B804B704B604BD04BC04BB04BA
04C104C004BF04BE04C504C404C304C204C904C804C704C604CD04CC04CB04CAFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04D104D004CF04CE04D504D404D304D2
04D904D804D704D604DD04DC04DB04DA04E104E004DF04DE04E504E404E304E204E904E804E704E604ED04EC04EB04EA
04F104F004EF04EEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF04F2FFFFFFFFFFFF04F604F504F404F304FA04F904F804F704FDFFFF04FC04FB0501050004FF04FE
05050504050305020509050805070506050CFFFF050B050A0510050F050E050D0513FFFF05120511FFFFFFFFFFFF0514
0518051705160515051C051B051A05190520051F051E051D05240523052205210528052705260525052C052B052A0529
0530052F052E052D05340533053205310538053705360535053C053B053A05390540053F053E053D0544054305420541
FFFF054705460545FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF054B054A05490548054F054E054D054C
05530552055105500557055605550554055B055A05590558055F055E055D055C05630562056105600567056605650564
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF056B056A05690568056F056E056D056C
05730572057105700577057605750574057B057A05790578057F057E057D057C05830582058105800587058605850584
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF05890588FFFFFFFF058D058C058B058A05910590058F058E0595059405930592
0599059805970596059D059C059B059A05A105A0059F059E05A505A405A305A205A905A805A705A6FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//9856 bytes
enum toLowerSimpleIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000140",
cast(immutable size_t[]) x"
000000000000010000000000000004800000000000000E40",
cast(immutable size_t[]) x"
040203020202010007020202060202050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
02020202020202020202020202020202000200000001000000060005000400030000000000080007000B000A00090000
000F000E000D000C00000000001100100000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001300120000000000000000000000000000000000000000001500140000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000016000000000000000000000000001A001900180017001E001D001C001B0000000000000000000000210020001F
000000000000000000000000000000000023002200000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000270026002500240000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000002900280000002D002C002B002A00000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002E
0000000000000000000000000000000000000000000000000000000000000000003100300000002F0000003300320000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000340000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000350000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000003600000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000370000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
000200010000FFFF0006000500040003000A000900080007000E000D000C000B001200110010000F0016001500140013
FFFF001900180017FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001D001C001B001A00210020001F001E
00250024002300220029002800270026002D002C002B002AFFFF0030002F002E0034003300320031FFFF003700360035
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0039FFFF0038FFFF003BFFFF003AFFFF003DFFFF003CFFFF003FFFFF003E
FFFF0041FFFF0040FFFF0043FFFF0042FFFF0045FFFF0044FFFF0047FFFF0046FFFF0049FFFF0048FFFF004BFFFF004A
FFFF004DFFFF004CFFFF004FFFFF004EFFFF0051FFFF0050FFFF0053FFFF00520055FFFF0054FFFF0057FFFF0056FFFF
0059FFFF0058FFFF005BFFFF005AFFFFFFFF005CFFFFFFFFFFFF005EFFFF005DFFFF0060FFFF005FFFFF0062FFFF0061
FFFF0064FFFF0063FFFF0066FFFF0065FFFF0068FFFF0067FFFF006AFFFF0069FFFF006CFFFF006BFFFF006EFFFF006D
FFFF0070FFFF006FFFFF0072FFFF00710075FFFF00740073FFFFFFFF0076FFFFFFFF00780077FFFF007B007AFFFF0079
007E007D007CFFFF0080007FFFFFFFFF0083FFFF0082008100860085FFFF0084FFFFFFFFFFFF0087008AFFFF00890088
FFFF008CFFFF008B008F008EFFFF008DFFFFFFFF0090FFFF00930092FFFF0091009600950094FFFF0098FFFF0097FFFF
FFFFFFFFFFFF0099FFFFFFFFFFFF009AFFFFFFFFFFFFFFFF009DFFFF009C009B00A0009FFFFF009E00A2FFFF00A1FFFF
00A4FFFF00A3FFFF00A6FFFF00A5FFFF00A8FFFF00A7FFFFFFFF00A9FFFFFFFFFFFF00ABFFFF00AAFFFF00ADFFFF00AC
FFFF00AFFFFF00AEFFFF00B1FFFF00B0FFFF00B300B2FFFF00B600B5FFFF00B4FFFF00B8FFFF00B7FFFF00BAFFFF00B9
FFFF00BCFFFF00BBFFFF00BEFFFF00BDFFFF00C0FFFF00BFFFFF00C2FFFF00C1FFFF00C4FFFF00C3FFFF00C6FFFF00C5
FFFF00C8FFFF00C7FFFF00CAFFFF00C9FFFF00CCFFFF00CBFFFF00CEFFFF00CDFFFF00D0FFFF00CFFFFF00D2FFFF00D1
FFFF00D4FFFF00D3FFFFFFFFFFFFFFFF00D600D5FFFFFFFFFFFF00D800D7FFFF00DAFFFF00D9FFFFFFFF00DD00DC00DB
FFFF00DFFFFF00DEFFFF00E1FFFF00E0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00E3FFFF00E2FFFF00E4FFFFFFFFFFFFFFFFFFFFFFFF00E5FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF00E6FFFFFFFFFFFF00E900E800E700EC00EBFFFF00EA00EF00EE00EDFFFF00F300F200F100F0
00F700F600F500F400FB00FA00F900F800FEFFFF00FD00FC01020101010000FF0106010501040103FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF0107FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0109FFFF0108FFFF010BFFFF010A
FFFF010DFFFF010CFFFF010FFFFF010EFFFF0111FFFF0110FFFF0113FFFF0112FFFFFFFFFFFFFFFF0115FFFFFFFF0114
FFFF01170116FFFF011A01190118FFFF011E011D011C011B012201210120011F0126012501240123012A012901280127
012E012D012C012B013201310130012F0136013501340133013A013901380137013E013D013C013B014201410140013F
0146014501440143014A014901480147FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF014CFFFF014BFFFF014EFFFF014DFFFF0150FFFF014FFFFF0152FFFF0151
FFFF0154FFFF0153FFFF0156FFFF0155FFFF0158FFFF0157FFFF015AFFFF0159FFFFFFFFFFFF015BFFFFFFFFFFFFFFFF
FFFF015CFFFFFFFFFFFF015EFFFF015DFFFF0160FFFF015FFFFF0162FFFF0161FFFF0164FFFF0163FFFF0166FFFF0165
FFFF0168FFFF0167FFFF016AFFFF0169FFFF016CFFFF016BFFFF016EFFFF016DFFFF0170FFFF016FFFFF0172FFFF0171
FFFF0174FFFF0173FFFF0176FFFF01750179FFFF01780177017BFFFF017AFFFF017DFFFF017CFFFFFFFFFFFF017EFFFF
FFFF0180FFFF017FFFFF0182FFFF0181FFFF0184FFFF0183FFFF0186FFFF0185FFFF0188FFFF0187FFFF018AFFFF0189
FFFF018CFFFF018BFFFF018EFFFF018DFFFF0190FFFF018FFFFF0192FFFF0191FFFF0194FFFF0193FFFF0196FFFF0195
FFFF0198FFFF0197FFFF019AFFFF0199FFFF019CFFFF019BFFFF019EFFFF019DFFFF01A0FFFF019FFFFF01A2FFFF01A1
FFFF01A4FFFF01A3FFFF01A6FFFF01A5FFFF01A8FFFF01A7FFFF01AAFFFF01A9FFFF01ACFFFF01ABFFFF01AEFFFF01AD
01B101B001AFFFFF01B501B401B301B201B901B801B701B601BD01BC01BB01BA01C101C001BF01BE01C501C401C301C2
01C901C801C701C601CD01CC01CB01CA01D101D001CF01CEFFFF01D401D301D2FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01D801D701D601D501DC01DB01DA01D9
01E001DF01DE01DD01E401E301E201E101E801E701E601E501EC01EB01EA01E901F001EF01EE01ED01F401F301F201F1
01F801F701F601F501FBFFFF01FA01F9FFFFFFFFFFFFFFFFFFFFFFFF01FCFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
020001FF01FE01FD02040203020202010208020702060205020C020B020A02090210020F020E020D0214021302120211
0218021702160215021C021B021A02190220021F021E021D02240223022202210228022702260225022C022B022A0229
0230022F022E022D02340233023202310238023702360235023C023B023A02390240023F023E023D0244024302420241
0248024702460245024C024B024A02490250024F024E024DFFFFFFFF02520251FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0256025502540253025A025902580257
025E025D025C025B026202610260025F0266026502640263026A026902680267026E026D026C026B027202710270026F
0276027502740273027A027902780277FFFF027D027C027B0280027F027EFFFFFFFF0282FFFF0281FFFF0284FFFF0283
FFFF0286FFFF0285FFFF0288FFFF0287FFFF028AFFFF0289FFFF028CFFFF028BFFFF028EFFFF028DFFFF0290FFFF028F
FFFF0292FFFF0291FFFF0294FFFF0293FFFF0296FFFF0295FFFF0298FFFF0297FFFF029AFFFF0299FFFF029CFFFF029B
FFFF029EFFFF029DFFFF02A0FFFF029FFFFF02A2FFFF02A1FFFF02A4FFFF02A3FFFF02A6FFFF02A5FFFF02A8FFFF02A7
FFFF02AAFFFF02A9FFFF02ACFFFF02ABFFFF02AEFFFF02ADFFFF02B0FFFF02AFFFFF02B2FFFF02B1FFFF02B4FFFF02B3
FFFF02B6FFFF02B5FFFF02B8FFFF02B7FFFF02BAFFFF02B9FFFF02BCFFFF02BBFFFF02BEFFFF02BDFFFF02C0FFFF02BF
FFFF02C2FFFF02C1FFFF02C4FFFF02C3FFFF02C6FFFF02C5FFFF02C8FFFF02C7FFFF02CAFFFF02C9FFFFFFFFFFFF02CB
FFFFFFFFFFFFFFFFFFFF02CCFFFFFFFFFFFF02CEFFFF02CDFFFF02D0FFFF02CFFFFF02D2FFFF02D1FFFF02D4FFFF02D3
FFFF02D6FFFF02D5FFFF02D8FFFF02D7FFFF02DAFFFF02D9FFFF02DCFFFF02DBFFFF02DEFFFF02DDFFFF02E0FFFF02DF
FFFF02E2FFFF02E1FFFF02E4FFFF02E3FFFF02E6FFFF02E5FFFF02E8FFFF02E7FFFF02EAFFFF02E9FFFF02ECFFFF02EB
FFFF02EEFFFF02EDFFFF02F0FFFF02EFFFFF02F2FFFF02F1FFFF02F4FFFF02F3FFFF02F6FFFF02F5FFFF02F8FFFF02F7
FFFF02FAFFFF02F9FFFF02FCFFFF02FBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030002FF02FE02FD0304030303020301
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0308030703060305FFFFFFFF030A0309FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
030E030D030C030B031203110310030FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0316031503140313031A031903180317
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF031E031D031C031BFFFFFFFF0320031FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0322FFFF0321FFFF0324FFFF0323FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0328032703260325032C032B032A0329
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0330032F032E032D0334033303320331FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0338033703360335033C033B033A0339
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0340033F033E033D0344034303420341FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0348034703460345FFFFFFFFFFFF0349FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF034D034C034B034AFFFFFFFFFFFF034E
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF035203510350034FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0356035503540353FFFFFFFFFFFF0357FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF035B035A03590358FFFFFFFFFFFF035C
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF035DFFFFFFFF035F035EFFFFFFFFFFFFFFFFFFFFFFFF
FFFF0360FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03640363036203610368036703660365036C036B036A03690370036F036E036DFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0371FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF03730372FFFFFFFF0377037603750374037B037A03790378037F037E037D037C0383038203810380
0387038603850384038B038A03890388FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF038F038E038D038C03930392039103900397039603950394039B039A03990398
039F039E039D039C03A303A203A103A003A703A603A503A403AB03AA03A903A803AF03AE03AD03AC03B303B203B103B0
03B703B603B503B403BB03BA03B903B8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03BE03BDFFFF03BC03C0FFFFFFFF03BF03C2FFFF03C1FFFF03C503C403C3FFFF
FFFF03C7FFFF03C6FFFFFFFF03C8FFFFFFFFFFFFFFFFFFFF03CA03C9FFFFFFFFFFFF03CCFFFF03CBFFFF03CEFFFF03CD
FFFF03D0FFFF03CFFFFF03D2FFFF03D1FFFF03D4FFFF03D3FFFF03D6FFFF03D5FFFF03D8FFFF03D7FFFF03DAFFFF03D9
FFFF03DCFFFF03DBFFFF03DEFFFF03DDFFFF03E0FFFF03DFFFFF03E2FFFF03E1FFFF03E4FFFF03E3FFFF03E6FFFF03E5
FFFF03E8FFFF03E7FFFF03EAFFFF03E9FFFF03ECFFFF03EBFFFF03EEFFFF03EDFFFF03F0FFFF03EFFFFF03F2FFFF03F1
FFFF03F4FFFF03F3FFFF03F6FFFF03F5FFFF03F8FFFF03F7FFFF03FAFFFF03F9FFFF03FCFFFF03FBFFFFFFFFFFFFFFFF
03FDFFFFFFFFFFFFFFFFFFFF03FEFFFFFFFF03FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF0401FFFF0400FFFF0403FFFF0402FFFF0405FFFF0404FFFF0407FFFF0406FFFF0409FFFF0408FFFF040BFFFF040A
FFFF040DFFFF040CFFFF040FFFFF040EFFFF0411FFFF0410FFFF0413FFFF0412FFFF0415FFFF0414FFFFFFFFFFFF0416
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0418FFFF0417FFFF041AFFFF0419
FFFF041CFFFF041BFFFF041EFFFF041DFFFF0420FFFF041FFFFF0422FFFF0421FFFF0424FFFF0423FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0425FFFFFFFFFFFF0427FFFF0426
FFFF0429FFFF0428FFFF042BFFFF042AFFFF042CFFFFFFFFFFFF042EFFFF042DFFFF0430FFFF042FFFFF0432FFFF0431
FFFF0434FFFF0433FFFF0436FFFF0435FFFF0438FFFF0437FFFF043AFFFF0439FFFF043CFFFF043BFFFF043EFFFF043D
FFFF0440FFFF043FFFFF0442FFFF0441FFFF0444FFFF0443FFFF0446FFFF0445FFFF0448FFFF0447FFFF044AFFFF0449
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF044CFFFF044BFFFFFFFF044E044DFFFFFFFF0450FFFF044FFFFF0452FFFF0451
0453FFFFFFFFFFFFFFFFFFFF0454FFFFFFFF0456FFFF0455FFFF0457FFFFFFFFFFFF0459FFFF0458FFFF045BFFFF045A
FFFF045DFFFF045CFFFF045FFFFF045E04620461FFFF0460FFFF0465046404630469046804670466FFFF046BFFFF046A
FFFF046DFFFF046CFFFF046FFFFF046EFFFF0471FFFF04700475047404730472FFFFFFFF0476FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF0477FFFF0478FFFFFFFFFFFFFFFFFFFF0479FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF047AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF047D047C047BFFFF04810480047F047E04850484048304820489048804870486
048D048C048B048A04910490048F048EFFFF049404930492FFFFFFFFFFFFFFFF0498049704960495049C049B049A0499
04A0049F049E049D04A404A304A204A104A804A704A604A504AC04AB04AA04A904B004AF04AE04AD04B404B304B204B1
04B804B704B604B504BC04BB04BA04B9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04C004BF04BE04BD04C404C304C204C104C804C704C604C504CC04CB04CA04C9
04D004CF04CE04CD04D404D304D204D104D804D704D604D504DC04DB04DA04D904E004DF04DE04DDFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04E404E304E204E104E804E704E604E5
FFFF04EB04EA04E904EF04EE04ED04EC04F304F204F104F004F704F604F504F4FFFF04FA04F904F804FE04FD04FC04FB
FFFF0501050004FFFFFFFFFF05030502FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0507050605050504050B050A05090508050F050E050D050C05130512051105100517051605150514051B051A05190518
051F051E051D051C05230522052105200527052605250524052B052A05290528052F052E052D052C0533053205310530
FFFF053605350534FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
053A053905380537053E053D053C053B054205410540053F0546054505440543054A054905480547054E054D054C054B
055205510550054F0556055505540553055A055905580557055E055D055C055B056205610560055F0566056505640563
056A056905680567056E056D056C056B057205710570056F0576057505740573FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
057A057905780577057E057D057C057B058205810580057F0586058505840583058A058905880587058E058D058C058B
059205910590058F0596059505940593FFFFFFFF05980597FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
//10368 bytes
enum toTitleSimpleIndexTrieEntries = TrieEntry!(ushort, 8, 7, 6)(cast(immutable size_t[]) x"
000000000000000000000000000000200000000000000140",
cast(immutable size_t[]) x"
000000000000010000000000000004800000000000000F40",
cast(immutable size_t[]) x"
040203020202010007020202060202050202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202020202
02020202020202020202020202020202000300020001000000070006000500040000000A00090008000D000C000B0000
00110010000F000E00000014001300120000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001500000000000000000000000000000000000000000000001600000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000017000000000000001900180000001D001C001B001A00210020001F001E00000000000000000000002300220000
000000000000000000000000000000000024000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000280027002600250000000000000029
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000002B002A0000002F002E002D002C00000000000000000000000000000000
000000000000000000000031003000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000320000
000000000000000000000000000000000000000000000000000000000000000000350000003400330000003600000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
003700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
003800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000003900000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000003B003A0000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000200010000FFFF0006000500040003000A000900080007000E000D000C000B
001200110010000F0016001500140013FFFF001900180017FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001AFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF001E001D001C001B002200210020001F
0026002500240023002A002900280027002E002D002C002BFFFF00310030002F00350034003300320039003800370036
003BFFFF003AFFFF003DFFFF003CFFFF003FFFFF003EFFFF0041FFFF0040FFFF0043FFFF0042FFFF0045FFFF0044FFFF
0047FFFF0046FFFF0049FFFF0048FFFF004BFFFF004AFFFF004DFFFF004CFFFF004FFFFF004EFFFF0051FFFF0050FFFF
0053FFFF0052FFFF0055FFFF0054FFFFFFFF0056FFFFFFFFFFFF0058FFFF0057FFFF005AFFFF0059FFFF005CFFFF005B
005EFFFFFFFF005D0060FFFF005FFFFF0062FFFF0061FFFF0064FFFF0063FFFF0066FFFF0065FFFF0068FFFF0067FFFF
006AFFFF0069FFFF006CFFFF006BFFFF006EFFFF006DFFFF0070FFFF006FFFFF0072FFFF0071FFFF0074FFFF0073FFFF
FFFF0075FFFFFFFF00780077FFFF0076007AFFFFFFFF0079FFFFFFFF007BFFFFFFFFFFFFFFFF007CFFFFFFFFFFFF007D
FFFF007EFFFFFFFFFFFFFFFF007FFFFFFFFF00810080FFFFFFFF0082FFFFFFFF0084FFFF0083FFFFFFFFFFFF0085FFFF
FFFFFFFFFFFF0086FFFFFFFF0087FFFFFFFFFFFFFFFF0088FFFF008AFFFF0089FFFFFFFF008BFFFF008DFFFF008CFFFF
FFFFFFFFFFFFFFFF00910090008F008E0095009400930092FFFF0097FFFF0096FFFF0099FFFF0098FFFF009BFFFF009A
FFFF009DFFFF009C00A0FFFF009F009E00A2FFFF00A1FFFF00A4FFFF00A3FFFF00A6FFFF00A5FFFF00A8FFFF00A7FFFF
00AB00AA00A9FFFFFFFFFFFF00ACFFFF00AEFFFF00ADFFFF00B0FFFF00AFFFFF00B2FFFF00B1FFFF00B4FFFF00B3FFFF
00B6FFFF00B5FFFF00B8FFFF00B7FFFF00BAFFFF00B9FFFF00BCFFFF00BBFFFF00BEFFFF00BDFFFF00C0FFFF00BFFFFF
00C1FFFFFFFFFFFF00C3FFFF00C2FFFF00C5FFFF00C4FFFF00C7FFFF00C6FFFF00C9FFFF00C8FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF00CBFFFFFFFF00CAFFFF00CDFFFF00CC00CEFFFFFFFFFFFF00D0FFFF00CFFFFF00D2FFFF00D1FFFF
00D600D500D400D300D900D8FFFF00D700DBFFFF00DAFFFFFFFFFFFFFFFF00DC00DFFFFF00DE00DDFFFF00E100E0FFFF
00E500E400E300E200E7FFFFFFFF00E6FFFF00E900E8FFFFFFFFFFFF00EAFFFFFFFFFFFFFFFFFFFFFFFFFFFF00EBFFFF
00EE00EDFFFF00EC00EFFFFFFFFFFFFF00F300F200F100F0FFFFFFFFFFFF00F4FFFF00F5FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF00F700F6FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00F8FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FAFFFF00F9FFFF00FBFFFFFFFFFFFF
00FCFFFFFFFFFFFFFFFFFFFF00FE00FDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF01020101010000FF010501040103FFFF0109010801070106010D010C010B010A01110110010F010E
01150114011301120119011801170116011D011C011B011AFFFF0120011F011EFFFFFFFF01220121012501240123FFFF
0127FFFF0126FFFF0129FFFF0128FFFF012BFFFF012AFFFF012DFFFF012CFFFF012FFFFF012EFFFF0131FFFF0130FFFF
0135013401330132FFFFFFFF0136FFFF0138FFFFFFFF0137FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF013C013B013A01390140013F013E013D
01440143014201410148014701460145014C014B014A01490150014F014E014D01540153015201510158015701560155
015C015B015A01590160015F015E015D01640163016201610168016701660165016AFFFF0169FFFF016CFFFF016BFFFF
016EFFFF016DFFFF0170FFFF016FFFFF0172FFFF0171FFFF0174FFFF0173FFFF0176FFFF0175FFFF0178FFFF0177FFFF
FFFFFFFF0179FFFFFFFFFFFFFFFFFFFF017AFFFFFFFFFFFF017CFFFF017BFFFF017EFFFF017DFFFF0180FFFF017FFFFF
0182FFFF0181FFFF0184FFFF0183FFFF0186FFFF0185FFFF0188FFFF0187FFFF018AFFFF0189FFFF018CFFFF018BFFFF
018EFFFF018DFFFF0190FFFF018FFFFF0192FFFF0191FFFF0194FFFF0193FFFFFFFF0195FFFFFFFFFFFF0197FFFF0196
FFFF0199FFFF0198019C019BFFFF019A019EFFFF019DFFFF01A0FFFF019FFFFF01A2FFFF01A1FFFF01A4FFFF01A3FFFF
01A6FFFF01A5FFFF01A8FFFF01A7FFFF01AAFFFF01A9FFFF01ACFFFF01ABFFFF01AEFFFF01ADFFFF01B0FFFF01AFFFFF
01B2FFFF01B1FFFF01B4FFFF01B3FFFF01B6FFFF01B5FFFF01B8FFFF01B7FFFF01BAFFFF01B9FFFF01BCFFFF01BBFFFF
01BEFFFF01BDFFFF01C0FFFF01BFFFFF01C2FFFF01C1FFFF01C4FFFF01C3FFFF01C6FFFF01C5FFFF01C8FFFF01C7FFFF
01CAFFFF01C9FFFF01CCFFFF01CBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF01CF01CE01CDFFFF01D301D201D101D001D701D601D501D401DB01DA01D901D8
01DF01DE01DD01DC01E301E201E101E001E701E601E501E401EB01EA01E901E801EF01EE01ED01ECFFFF01F201F101F0
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
01F601F501F401F301FA01F901F801F701FE01FD01FC01FB02020201020001FF0206020502040203020A020902080207
020E020D020C020B021202110210020F0216021502140213021A021902180217FFFF021D021C021B0220021F021EFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0224022302220221FFFFFFFF02260225022A022902280227022E022D022C022B
FFFFFFFFFFFF022FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0230FFFFFFFFFFFF0231FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0232FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0234FFFF0233FFFF0236FFFF0235FFFF
0238FFFF0237FFFF023AFFFF0239FFFF023CFFFF023BFFFF023EFFFF023DFFFF0240FFFF023FFFFF0242FFFF0241FFFF
0244FFFF0243FFFF0246FFFF0245FFFF0248FFFF0247FFFF024AFFFF0249FFFF024CFFFF024BFFFF024EFFFF024DFFFF
0250FFFF024FFFFF0252FFFF0251FFFF0254FFFF0253FFFF0256FFFF0255FFFF0258FFFF0257FFFF025AFFFF0259FFFF
025CFFFF025BFFFF025EFFFF025DFFFF0260FFFF025FFFFF0262FFFF0261FFFF0264FFFF0263FFFF0266FFFF0265FFFF
0268FFFF0267FFFF026AFFFF0269FFFF026CFFFF026BFFFF026EFFFF026DFFFF0270FFFF026FFFFF0272FFFF0271FFFF
0274FFFF0273FFFF0276FFFF0275FFFF0278FFFF0277FFFF027AFFFF0279FFFF027CFFFF027BFFFFFFFFFFFF027DFFFF
027EFFFFFFFFFFFFFFFFFFFFFFFFFFFF0280FFFF027FFFFF0282FFFF0281FFFF0284FFFF0283FFFF0286FFFF0285FFFF
0288FFFF0287FFFF028AFFFF0289FFFF028CFFFF028BFFFF028EFFFF028DFFFF0290FFFF028FFFFF0292FFFF0291FFFF
0294FFFF0293FFFF0296FFFF0295FFFF0298FFFF0297FFFF029AFFFF0299FFFF029CFFFF029BFFFF029EFFFF029DFFFF
02A0FFFF029FFFFF02A2FFFF02A1FFFF02A4FFFF02A3FFFF02A6FFFF02A5FFFF02A8FFFF02A7FFFF02AAFFFF02A9FFFF
02ACFFFF02ABFFFF02AEFFFF02ADFFFF02B202B102B002AF02B602B502B402B3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02BA02B902B802B7FFFFFFFF02BC02BBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02C002BF02BE02BD02C402C302C202C1
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02C802C702C602C502CC02CB02CA02C9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02D002CF02CE02CDFFFFFFFF02D202D1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02D4FFFF02D3FFFF02D6FFFF02D5FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02DA02D902D802D702DE02DD02DC02DBFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
02E202E102E002DF02E602E502E402E302EA02E902E802E7FFFFFFFF02EC02EB02F002EF02EE02ED02F402F302F202F1
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF02F802F702F602F502FC02FB02FA02F9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
030002FF02FE02FD0304030303020301FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0307FFFF03060305FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF0308FFFFFFFF0309FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF030B030AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030D030CFFFFFFFF030EFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF030FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0310FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03140313031203110318031703160315031C031B031A03190320031F031E031DFFFFFFFFFFFFFFFFFFFFFFFFFFFF0321
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03250324032303220329032803270326032D032C032B032A03310330032F032E03350334033303320339033803370336
FFFFFFFF033B033AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
033F033E033D033C03430342034103400347034603450344034B034A03490348034F034E034D034C0353035203510350
0357035603550354035B035A03590358035F035E035D035C03630362036103600367036603650364036B036A03690368
FFFFFFFF036CFFFFFFFF036E036DFFFFFFFF0370FFFF036FFFFFFFFFFFFF03710372FFFFFFFFFFFFFFFF0373FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0375FFFF0374FFFF0377FFFF0376FFFF0379FFFF0378FFFF037BFFFF037AFFFF
037DFFFF037CFFFF037FFFFF037EFFFF0381FFFF0380FFFF0383FFFF0382FFFF0385FFFF0384FFFF0387FFFF0386FFFF
0389FFFF0388FFFF038BFFFF038AFFFF038DFFFF038CFFFF038FFFFF038EFFFF0391FFFF0390FFFF0393FFFF0392FFFF
0395FFFF0394FFFF0397FFFF0396FFFF0399FFFF0398FFFF039BFFFF039AFFFF039DFFFF039CFFFF039FFFFF039EFFFF
03A1FFFF03A0FFFF03A3FFFF03A2FFFF03A5FFFF03A4FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03A7FFFF03A6
03A8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03AC03AB03AA03A903B003AF03AE03AD
03B403B303B203B103B803B703B603B503BC03BB03BA03B903C003BF03BE03BD03C403C303C203C103C803C703C603C5
03CC03CB03CA03C903CFFFFF03CE03CDFFFFFFFFFFFFFFFFFFFFFFFF03D0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF03D2FFFF03D1FFFF03D4FFFF03D3FFFF03D6FFFF03D5FFFF03D8FFFF03D7FFFF
03DAFFFF03D9FFFF03DCFFFF03DBFFFF03DEFFFF03DDFFFF03E0FFFF03DFFFFF03E2FFFF03E1FFFF03E4FFFF03E3FFFF
03E6FFFF03E5FFFFFFFFFFFF03E7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03E9FFFF03E8FFFF03EBFFFF03EAFFFF03EDFFFF03ECFFFF03EFFFFF03EEFFFF03F1FFFF03F0FFFF03F3FFFF03F2FFFF
03F5FFFF03F4FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
03F6FFFFFFFFFFFF03F8FFFF03F7FFFF03FAFFFF03F9FFFF03FCFFFF03FBFFFF03FDFFFFFFFFFFFF03FFFFFF03FEFFFF
0401FFFF0400FFFF0403FFFF0402FFFF0405FFFF0404FFFF0407FFFF0406FFFF0409FFFF0408FFFF040BFFFF040AFFFF
040DFFFF040CFFFF040FFFFF040EFFFF0411FFFF0410FFFF0413FFFF0412FFFF0415FFFF0414FFFF0417FFFF0416FFFF
0419FFFF0418FFFF041BFFFF041AFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF041CFFFFFFFF041EFFFFFFFF041D
0420FFFF041FFFFF0422FFFF0421FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04230425FFFF0424FFFF0427FFFFFFFF0426
0429FFFF0428FFFF042BFFFF042AFFFF042DFFFF042CFFFF042FFFFF042EFFFFFFFFFFFF0430FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF0432FFFF0431FFFF0434FFFF0433FFFF0436FFFF0435FFFF0438FFFF0437FFFFFFFFFFFFFFFFFFFF
FFFF043AFFFF0439FFFFFFFFFFFFFFFFFFFFFFFF043BFFFF043CFFFFFFFFFFFFFFFFFFFF043DFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF043EFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
043FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04430442044104400447044604450444044B044A04490448044F044E044D044C
04530452045104500457045604550454045B045A04590458045F045E045D045C04630462046104600467046604650464
046B046A04690468046F046E046D046C04730472047104700477047604750474047B047A04790478047F047E047D047C
04830482048104800487048604850484048B048A04890488048F048E048D048C049204910490FFFF0496049504940493
049A049904980497049E049D049C049B04A204A104A0049F04A604A504A404A3FFFF04A904A804A7FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
04AD04AC04AB04AA04B104B004AF04AE04B504B404B304B204B904B804B704B604BD04BC04BB04BA04C104C004BF04BE
04C504C404C304C204C904C804C704C604CD04CC04CB04CA04D104D004CF04CEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF04D504D404D304D204D904D804D704D6
04DD04DC04DB04DA04E104E004DF04DE04E504E404E304E204E904E804E704E604ED04EC04EB04EA04F104F004EF04EE
04F504F404F304F2FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF04F6FFFFFFFFFFFF04FA04F904F804F704FE04FD04FC04FB0501FFFF050004FF0505050405030502
0509050805070506050D050C050B050A0510FFFF050F050E05140513051205110517FFFF05160515FFFFFFFFFFFF0518
051C051B051A05190520051F051E051D05240523052205210528052705260525052C052B052A05290530052F052E052D
05340533053205310538053705360535053C053B053A05390540053F053E053D05440543054205410548054705460545
FFFF054B054A0549FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF054F054E054D054C0553055205510550
0557055605550554055B055A05590558055F055E055D055C05630562056105600567056605650564056B056A05690568
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF056F056E056D056C0573057205710570
0577057605750574057B057A05790578057F057E057D057C05830582058105800587058605850584058B058A05890588
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF058D058CFFFFFFFF05910590058F058E05950594059305920599059805970596
059D059C059B059A05A105A0059F059E05A505A405A305A205A905A805A705A605AD05AC05AB05AAFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
);
immutable(uint)[] toUpperTable() nothrow @nogc pure @safe {
static immutable uint[] t =
cast(immutable uint[]) x"
0000004100000042000000430000004400000045000000460000004700000048000000490000004A0000004B0000004C
0000004D0000004E0000004F000000500000005100000052000000530000005400000055000000560000005700000058
000000590000005A0000039C000000C0000000C1000000C2000000C3000000C4000000C5000000C6000000C7000000C8
000000C9000000CA000000CB000000CC000000CD000000CE000000CF000000D0000000D1000000D2000000D3000000D4
000000D5000000D6000000D8000000D9000000DA000000DB000000DC000000DD000000DE000001780000010000000102
0000010400000106000001080000010A0000010C0000010E00000110000001120000011400000116000001180000011A
0000011C0000011E00000120000001220000012400000126000001280000012A0000012C0000012E0000004900000132
0000013400000136000001390000013B0000013D0000013F000001410000014300000145000001470000014A0000014C
0000014E00000150000001520000015400000156000001580000015A0000015C0000015E000001600000016200000164
00000166000001680000016A0000016C0000016E00000170000001720000017400000176000001790000017B0000017D
00000053000002430000018200000184000001870000018B00000191000001F6000001980000023D00000220000001A0
000001A2000001A4000001A7000001AC000001AF000001B3000001B5000001B8000001BC000001F7000001C4000001C4
000001C7000001C7000001CA000001CA000001CD000001CF000001D1000001D3000001D5000001D7000001D9000001DB
0000018E000001DE000001E0000001E2000001E4000001E6000001E8000001EA000001EC000001EE000001F1000001F1
000001F4000001F8000001FA000001FC000001FE00000200000002020000020400000206000002080000020A0000020C
0000020E00000210000002120000021400000216000002180000021A0000021C0000021E000002220000022400000226
000002280000022A0000022C0000022E00000230000002320000023B00002C7E00002C7F000002410000024600000248
0000024A0000024C0000024E00002C6F00002C6D00002C700000018100000186000001890000018A0000018F00000190
0000A7AB000001930000A7AC000001940000A78D0000A7AA00000197000001960000A7AE00002C620000A7AD0000019C
00002C6E0000019D0000019F00002C64000001A60000A7C5000001A90000A7B1000001AE00000244000001B1000001B2
00000245000001B70000A7B20000A7B000000399000003700000037200000376000003FD000003FE000003FF00000386
00000388000003890000038A000003910000039200000393000003940000039500000396000003970000039800000399
0000039A0000039B0000039C0000039D0000039E0000039F000003A0000003A1000003A3000003A3000003A4000003A5
000003A6000003A7000003A8000003A9000003AA000003AB0000038C0000038E0000038F0000039200000398000003A6
000003A0000003CF000003D8000003DA000003DC000003DE000003E0000003E2000003E4000003E6000003E8000003EA
000003EC000003EE0000039A000003A1000003F90000037F00000395000003F7000003FA000004100000041100000412
000004130000041400000415000004160000041700000418000004190000041A0000041B0000041C0000041D0000041E
0000041F000004200000042100000422000004230000042400000425000004260000042700000428000004290000042A
0000042B0000042C0000042D0000042E0000042F00000400000004010000040200000403000004040000040500000406
0000040700000408000004090000040A0000040B0000040C0000040D0000040E0000040F000004600000046200000464
00000466000004680000046A0000046C0000046E00000470000004720000047400000476000004780000047A0000047C
0000047E000004800000048A0000048C0000048E00000490000004920000049400000496000004980000049A0000049C
0000049E000004A0000004A2000004A4000004A6000004A8000004AA000004AC000004AE000004B0000004B2000004B4
000004B6000004B8000004BA000004BC000004BE000004C1000004C3000004C5000004C7000004C9000004CB000004CD
000004C0000004D0000004D2000004D4000004D6000004D8000004DA000004DC000004DE000004E0000004E2000004E4
000004E6000004E8000004EA000004EC000004EE000004F0000004F2000004F4000004F6000004F8000004FA000004FC
000004FE00000500000005020000050400000506000005080000050A0000050C0000050E000005100000051200000514
00000516000005180000051A0000051C0000051E00000520000005220000052400000526000005280000052A0000052C
0000052E0000053100000532000005330000053400000535000005360000053700000538000005390000053A0000053B
0000053C0000053D0000053E0000053F0000054000000541000005420000054300000544000005450000054600000547
00000548000005490000054A0000054B0000054C0000054D0000054E0000054F00000550000005510000055200000553
00000554000005550000055600001C9000001C9100001C9200001C9300001C9400001C9500001C9600001C9700001C98
00001C9900001C9A00001C9B00001C9C00001C9D00001C9E00001C9F00001CA000001CA100001CA200001CA300001CA4
00001CA500001CA600001CA700001CA800001CA900001CAA00001CAB00001CAC00001CAD00001CAE00001CAF00001CB0
00001CB100001CB200001CB300001CB400001CB500001CB600001CB700001CB800001CB900001CBA00001CBD00001CBE
00001CBF000013F0000013F1000013F2000013F3000013F4000013F500000412000004140000041E0000042100000422
000004220000042A000004620000A64A0000A77D00002C630000A7C600001E0000001E0200001E0400001E0600001E08
00001E0A00001E0C00001E0E00001E1000001E1200001E1400001E1600001E1800001E1A00001E1C00001E1E00001E20
00001E2200001E2400001E2600001E2800001E2A00001E2C00001E2E00001E3000001E3200001E3400001E3600001E38
00001E3A00001E3C00001E3E00001E4000001E4200001E4400001E4600001E4800001E4A00001E4C00001E4E00001E50
00001E5200001E5400001E5600001E5800001E5A00001E5C00001E5E00001E6000001E6200001E6400001E6600001E68
00001E6A00001E6C00001E6E00001E7000001E7200001E7400001E7600001E7800001E7A00001E7C00001E7E00001E80
00001E8200001E8400001E8600001E8800001E8A00001E8C00001E8E00001E9000001E9200001E9400001E6000001EA0
00001EA200001EA400001EA600001EA800001EAA00001EAC00001EAE00001EB000001EB200001EB400001EB600001EB8
00001EBA00001EBC00001EBE00001EC000001EC200001EC400001EC600001EC800001ECA00001ECC00001ECE00001ED0
00001ED200001ED400001ED600001ED800001EDA00001EDC00001EDE00001EE000001EE200001EE400001EE600001EE8
00001EEA00001EEC00001EEE00001EF000001EF200001EF400001EF600001EF800001EFA00001EFC00001EFE00001F08
00001F0900001F0A00001F0B00001F0C00001F0D00001F0E00001F0F00001F1800001F1900001F1A00001F1B00001F1C
00001F1D00001F2800001F2900001F2A00001F2B00001F2C00001F2D00001F2E00001F2F00001F3800001F3900001F3A
00001F3B00001F3C00001F3D00001F3E00001F3F00001F4800001F4900001F4A00001F4B00001F4C00001F4D00001F59
00001F5B00001F5D00001F5F00001F6800001F6900001F6A00001F6B00001F6C00001F6D00001F6E00001F6F00001FBA
00001FBB00001FC800001FC900001FCA00001FCB00001FDA00001FDB00001FF800001FF900001FEA00001FEB00001FFA
00001FFB00001F8800001F8900001F8A00001F8B00001F8C00001F8D00001F8E00001F8F00001F9800001F9900001F9A
00001F9B00001F9C00001F9D00001F9E00001F9F00001FA800001FA900001FAA00001FAB00001FAC00001FAD00001FAE
00001FAF00001FB800001FB900001FBC0000039900001FCC00001FD800001FD900001FE800001FE900001FEC00001FFC
00002132000021600000216100002162000021630000216400002165000021660000216700002168000021690000216A
0000216B0000216C0000216D0000216E0000216F00002183000024B6000024B7000024B8000024B9000024BA000024BB
000024BC000024BD000024BE000024BF000024C0000024C1000024C2000024C3000024C4000024C5000024C6000024C7
000024C8000024C9000024CA000024CB000024CC000024CD000024CE000024CF00002C0000002C0100002C0200002C03
00002C0400002C0500002C0600002C0700002C0800002C0900002C0A00002C0B00002C0C00002C0D00002C0E00002C0F
00002C1000002C1100002C1200002C1300002C1400002C1500002C1600002C1700002C1800002C1900002C1A00002C1B
00002C1C00002C1D00002C1E00002C1F00002C2000002C2100002C2200002C2300002C2400002C2500002C2600002C27
00002C2800002C2900002C2A00002C2B00002C2C00002C2D00002C2E00002C2F00002C600000023A0000023E00002C67
00002C6900002C6B00002C7200002C7500002C8000002C8200002C8400002C8600002C8800002C8A00002C8C00002C8E
00002C9000002C9200002C9400002C9600002C9800002C9A00002C9C00002C9E00002CA000002CA200002CA400002CA6
00002CA800002CAA00002CAC00002CAE00002CB000002CB200002CB400002CB600002CB800002CBA00002CBC00002CBE
00002CC000002CC200002CC400002CC600002CC800002CCA00002CCC00002CCE00002CD000002CD200002CD400002CD6
00002CD800002CDA00002CDC00002CDE00002CE000002CE200002CEB00002CED00002CF2000010A0000010A1000010A2
000010A3000010A4000010A5000010A6000010A7000010A8000010A9000010AA000010AB000010AC000010AD000010AE
000010AF000010B0000010B1000010B2000010B3000010B4000010B5000010B6000010B7000010B8000010B9000010BA
000010BB000010BC000010BD000010BE000010BF000010C0000010C1000010C2000010C3000010C4000010C5000010C7
000010CD0000A6400000A6420000A6440000A6460000A6480000A64A0000A64C0000A64E0000A6500000A6520000A654
0000A6560000A6580000A65A0000A65C0000A65E0000A6600000A6620000A6640000A6660000A6680000A66A0000A66C
0000A6800000A6820000A6840000A6860000A6880000A68A0000A68C0000A68E0000A6900000A6920000A6940000A696
0000A6980000A69A0000A7220000A7240000A7260000A7280000A72A0000A72C0000A72E0000A7320000A7340000A736
0000A7380000A73A0000A73C0000A73E0000A7400000A7420000A7440000A7460000A7480000A74A0000A74C0000A74E
0000A7500000A7520000A7540000A7560000A7580000A75A0000A75C0000A75E0000A7600000A7620000A7640000A766
0000A7680000A76A0000A76C0000A76E0000A7790000A77B0000A77E0000A7800000A7820000A7840000A7860000A78B
0000A7900000A7920000A7C40000A7960000A7980000A79A0000A79C0000A79E0000A7A00000A7A20000A7A40000A7A6
0000A7A80000A7B40000A7B60000A7B80000A7BA0000A7BC0000A7BE0000A7C00000A7C20000A7C70000A7C90000A7D0
0000A7D60000A7D80000A7F50000A7B3000013A0000013A1000013A2000013A3000013A4000013A5000013A6000013A7
000013A8000013A9000013AA000013AB000013AC000013AD000013AE000013AF000013B0000013B1000013B2000013B3
000013B4000013B5000013B6000013B7000013B8000013B9000013BA000013BB000013BC000013BD000013BE000013BF
000013C0000013C1000013C2000013C3000013C4000013C5000013C6000013C7000013C8000013C9000013CA000013CB
000013CC000013CD000013CE000013CF000013D0000013D1000013D2000013D3000013D4000013D5000013D6000013D7
000013D8000013D9000013DA000013DB000013DC000013DD000013DE000013DF000013E0000013E1000013E2000013E3
000013E4000013E5000013E6000013E7000013E8000013E9000013EA000013EB000013EC000013ED000013EE000013EF
0000FF210000FF220000FF230000FF240000FF250000FF260000FF270000FF280000FF290000FF2A0000FF2B0000FF2C
0000FF2D0000FF2E0000FF2F0000FF300000FF310000FF320000FF330000FF340000FF350000FF360000FF370000FF38
0000FF390000FF3A00010400000104010001040200010403000104040001040500010406000104070001040800010409
0001040A0001040B0001040C0001040D0001040E0001040F000104100001041100010412000104130001041400010415
000104160001041700010418000104190001041A0001041B0001041C0001041D0001041E0001041F0001042000010421
000104220001042300010424000104250001042600010427000104B0000104B1000104B2000104B3000104B4000104B5
000104B6000104B7000104B8000104B9000104BA000104BB000104BC000104BD000104BE000104BF000104C0000104C1
000104C2000104C3000104C4000104C5000104C6000104C7000104C8000104C9000104CA000104CB000104CC000104CD
000104CE000104CF000104D0000104D1000104D2000104D3000105700001057100010572000105730001057400010575
000105760001057700010578000105790001057A0001057C0001057D0001057E0001057F000105800001058100010582
000105830001058400010585000105860001058700010588000105890001058A0001058C0001058D0001058E0001058F
000105900001059100010592000105940001059500010C8000010C8100010C8200010C8300010C8400010C8500010C86
00010C8700010C8800010C8900010C8A00010C8B00010C8C00010C8D00010C8E00010C8F00010C9000010C9100010C92
00010C9300010C9400010C9500010C9600010C9700010C9800010C9900010C9A00010C9B00010C9C00010C9D00010C9E
00010C9F00010CA000010CA100010CA200010CA300010CA400010CA500010CA600010CA700010CA800010CA900010CAA
00010CAB00010CAC00010CAD00010CAE00010CAF00010CB000010CB100010CB2000118A0000118A1000118A2000118A3
000118A4000118A5000118A6000118A7000118A8000118A9000118AA000118AB000118AC000118AD000118AE000118AF
000118B0000118B1000118B2000118B3000118B4000118B5000118B6000118B7000118B8000118B9000118BA000118BB
000118BC000118BD000118BE000118BF00016E4000016E4100016E4200016E4300016E4400016E4500016E4600016E47
00016E4800016E4900016E4A00016E4B00016E4C00016E4D00016E4E00016E4F00016E5000016E5100016E5200016E53
00016E5400016E5500016E5600016E5700016E5800016E5900016E5A00016E5B00016E5C00016E5D00016E5E00016E5F
0001E9000001E9010001E9020001E9030001E9040001E9050001E9060001E9070001E9080001E9090001E90A0001E90B
0001E90C0001E90D0001E90E0001E90F0001E9100001E9110001E9120001E9130001E9140001E9150001E9160001E917
0001E9180001E9190001E91A0001E91B0001E91C0001E91D0001E91E0001E91F0001E9200001E9210200005300000053
0000013002000046000000460200004600000049020000460000004C0300004600000046000000490300004600000046
0000004C0200005300000054020000530000005402000535000005520200054400000546020005440000053502000544
0000053B0200054E00000546020005440000053D020002BC0000004E030003990000030800000301030003A500000308
000003010200004A0000030C02000048000003310200005400000308020000570000030A020000590000030A02000041
000002BE020003A500000313030003A50000031300000300030003A50000031300000301030003A50000031300000342
020003910000034202000397000003420300039900000308000003000300039900000308000003010200039900000342
030003990000030800000342030003A50000030800000300030003A50000030800000301020003A100000313020003A5
00000342030003A50000030800000342020003A90000034202001F080000039902001F090000039902001F0A00000399
02001F0B0000039902001F0C0000039902001F0D0000039902001F0E0000039902001F0F0000039902001F0800000399
02001F090000039902001F0A0000039902001F0B0000039902001F0C0000039902001F0D0000039902001F0E00000399
02001F0F0000039902001F280000039902001F290000039902001F2A0000039902001F2B0000039902001F2C00000399
02001F2D0000039902001F2E0000039902001F2F0000039902001F280000039902001F290000039902001F2A00000399
02001F2B0000039902001F2C0000039902001F2D0000039902001F2E0000039902001F2F0000039902001F6800000399
02001F690000039902001F6A0000039902001F6B0000039902001F6C0000039902001F6D0000039902001F6E00000399
02001F6F0000039902001F680000039902001F690000039902001F6A0000039902001F6B0000039902001F6C00000399
02001F6D0000039902001F6E0000039902001F6F00000399020003910000039902000391000003990200039700000399
0200039700000399020003A900000399020003A90000039902001FBA00000399020003860000039902001FCA00000399
020003890000039902001FFA000003990200038F00000399030003910000034200000399030003970000034200000399
030003A90000034200000399";
return t;
}
immutable(uint)[] toLowerTable() nothrow @nogc pure @safe {
static immutable uint[] t =
cast(immutable uint[]) x"
0000006100000062000000630000006400000065000000660000006700000068000000690000006A0000006B0000006C
0000006D0000006E0000006F000000700000007100000072000000730000007400000075000000760000007700000078
000000790000007A000000E0000000E1000000E2000000E3000000E4000000E5000000E6000000E7000000E8000000E9
000000EA000000EB000000EC000000ED000000EE000000EF000000F0000000F1000000F2000000F3000000F4000000F5
000000F6000000F8000000F9000000FA000000FB000000FC000000FD000000FE00000101000001030000010500000107
000001090000010B0000010D0000010F00000111000001130000011500000117000001190000011B0000011D0000011F
00000121000001230000012500000127000001290000012B0000012D0000012F00000069000001330000013500000137
0000013A0000013C0000013E00000140000001420000014400000146000001480000014B0000014D0000014F00000151
000001530000015500000157000001590000015B0000015D0000015F0000016100000163000001650000016700000169
0000016B0000016D0000016F00000171000001730000017500000177000000FF0000017A0000017C0000017E00000253
0000018300000185000002540000018800000256000002570000018C000001DD000002590000025B0000019200000260
000002630000026900000268000001990000026F0000027200000275000001A1000001A3000001A500000280000001A8
00000283000001AD00000288000001B00000028A0000028B000001B4000001B600000292000001B9000001BD000001C6
000001C6000001C9000001C9000001CC000001CC000001CE000001D0000001D2000001D4000001D6000001D8000001DA
000001DC000001DF000001E1000001E3000001E5000001E7000001E9000001EB000001ED000001EF000001F3000001F3
000001F500000195000001BF000001F9000001FB000001FD000001FF0000020100000203000002050000020700000209
0000020B0000020D0000020F00000211000002130000021500000217000002190000021B0000021D0000021F0000019E
000002230000022500000227000002290000022B0000022D0000022F000002310000023300002C650000023C0000019A
00002C660000024200000180000002890000028C00000247000002490000024B0000024D0000024F0000037100000373
00000377000003F3000003AC000003AD000003AE000003AF000003CC000003CD000003CE000003B1000003B2000003B3
000003B4000003B5000003B6000003B7000003B8000003B9000003BA000003BB000003BC000003BD000003BE000003BF
000003C0000003C1000003C3000003C4000003C5000003C6000003C7000003C8000003C9000003CA000003CB000003D7
000003D9000003DB000003DD000003DF000003E1000003E3000003E5000003E7000003E9000003EB000003ED000003EF
000003B8000003F8000003F2000003FB0000037B0000037C0000037D0000045000000451000004520000045300000454
00000455000004560000045700000458000004590000045A0000045B0000045C0000045D0000045E0000045F00000430
0000043100000432000004330000043400000435000004360000043700000438000004390000043A0000043B0000043C
0000043D0000043E0000043F000004400000044100000442000004430000044400000445000004460000044700000448
000004490000044A0000044B0000044C0000044D0000044E0000044F0000046100000463000004650000046700000469
0000046B0000046D0000046F00000471000004730000047500000477000004790000047B0000047D0000047F00000481
0000048B0000048D0000048F00000491000004930000049500000497000004990000049B0000049D0000049F000004A1
000004A3000004A5000004A7000004A9000004AB000004AD000004AF000004B1000004B3000004B5000004B7000004B9
000004BB000004BD000004BF000004CF000004C2000004C4000004C6000004C8000004CA000004CC000004CE000004D1
000004D3000004D5000004D7000004D9000004DB000004DD000004DF000004E1000004E3000004E5000004E7000004E9
000004EB000004ED000004EF000004F1000004F3000004F5000004F7000004F9000004FB000004FD000004FF00000501
000005030000050500000507000005090000050B0000050D0000050F0000051100000513000005150000051700000519
0000051B0000051D0000051F00000521000005230000052500000527000005290000052B0000052D0000052F00000561
00000562000005630000056400000565000005660000056700000568000005690000056A0000056B0000056C0000056D
0000056E0000056F00000570000005710000057200000573000005740000057500000576000005770000057800000579
0000057A0000057B0000057C0000057D0000057E0000057F000005800000058100000582000005830000058400000585
0000058600002D0000002D0100002D0200002D0300002D0400002D0500002D0600002D0700002D0800002D0900002D0A
00002D0B00002D0C00002D0D00002D0E00002D0F00002D1000002D1100002D1200002D1300002D1400002D1500002D16
00002D1700002D1800002D1900002D1A00002D1B00002D1C00002D1D00002D1E00002D1F00002D2000002D2100002D22
00002D2300002D2400002D2500002D2700002D2D0000AB700000AB710000AB720000AB730000AB740000AB750000AB76
0000AB770000AB780000AB790000AB7A0000AB7B0000AB7C0000AB7D0000AB7E0000AB7F0000AB800000AB810000AB82
0000AB830000AB840000AB850000AB860000AB870000AB880000AB890000AB8A0000AB8B0000AB8C0000AB8D0000AB8E
0000AB8F0000AB900000AB910000AB920000AB930000AB940000AB950000AB960000AB970000AB980000AB990000AB9A
0000AB9B0000AB9C0000AB9D0000AB9E0000AB9F0000ABA00000ABA10000ABA20000ABA30000ABA40000ABA50000ABA6
0000ABA70000ABA80000ABA90000ABAA0000ABAB0000ABAC0000ABAD0000ABAE0000ABAF0000ABB00000ABB10000ABB2
0000ABB30000ABB40000ABB50000ABB60000ABB70000ABB80000ABB90000ABBA0000ABBB0000ABBC0000ABBD0000ABBE
0000ABBF000013F8000013F9000013FA000013FB000013FC000013FD000010D0000010D1000010D2000010D3000010D4
000010D5000010D6000010D7000010D8000010D9000010DA000010DB000010DC000010DD000010DE000010DF000010E0
000010E1000010E2000010E3000010E4000010E5000010E6000010E7000010E8000010E9000010EA000010EB000010EC
000010ED000010EE000010EF000010F0000010F1000010F2000010F3000010F4000010F5000010F6000010F7000010F8
000010F9000010FA000010FD000010FE000010FF00001E0100001E0300001E0500001E0700001E0900001E0B00001E0D
00001E0F00001E1100001E1300001E1500001E1700001E1900001E1B00001E1D00001E1F00001E2100001E2300001E25
00001E2700001E2900001E2B00001E2D00001E2F00001E3100001E3300001E3500001E3700001E3900001E3B00001E3D
00001E3F00001E4100001E4300001E4500001E4700001E4900001E4B00001E4D00001E4F00001E5100001E5300001E55
00001E5700001E5900001E5B00001E5D00001E5F00001E6100001E6300001E6500001E6700001E6900001E6B00001E6D
00001E6F00001E7100001E7300001E7500001E7700001E7900001E7B00001E7D00001E7F00001E8100001E8300001E85
00001E8700001E8900001E8B00001E8D00001E8F00001E9100001E9300001E95000000DF00001EA100001EA300001EA5
00001EA700001EA900001EAB00001EAD00001EAF00001EB100001EB300001EB500001EB700001EB900001EBB00001EBD
00001EBF00001EC100001EC300001EC500001EC700001EC900001ECB00001ECD00001ECF00001ED100001ED300001ED5
00001ED700001ED900001EDB00001EDD00001EDF00001EE100001EE300001EE500001EE700001EE900001EEB00001EED
00001EEF00001EF100001EF300001EF500001EF700001EF900001EFB00001EFD00001EFF00001F0000001F0100001F02
00001F0300001F0400001F0500001F0600001F0700001F1000001F1100001F1200001F1300001F1400001F1500001F20
00001F2100001F2200001F2300001F2400001F2500001F2600001F2700001F3000001F3100001F3200001F3300001F34
00001F3500001F3600001F3700001F4000001F4100001F4200001F4300001F4400001F4500001F5100001F5300001F55
00001F5700001F6000001F6100001F6200001F6300001F6400001F6500001F6600001F6700001F8000001F8100001F82
00001F8300001F8400001F8500001F8600001F8700001F9000001F9100001F9200001F9300001F9400001F9500001F96
00001F9700001FA000001FA100001FA200001FA300001FA400001FA500001FA600001FA700001FB000001FB100001F70
00001F7100001FB300001F7200001F7300001F7400001F7500001FC300001FD000001FD100001F7600001F7700001FE0
00001FE100001F7A00001F7B00001FE500001F7800001F7900001F7C00001F7D00001FF3000003C90000006B000000E5
0000214E000021700000217100002172000021730000217400002175000021760000217700002178000021790000217A
0000217B0000217C0000217D0000217E0000217F00002184000024D0000024D1000024D2000024D3000024D4000024D5
000024D6000024D7000024D8000024D9000024DA000024DB000024DC000024DD000024DE000024DF000024E0000024E1
000024E2000024E3000024E4000024E5000024E6000024E7000024E8000024E900002C3000002C3100002C3200002C33
00002C3400002C3500002C3600002C3700002C3800002C3900002C3A00002C3B00002C3C00002C3D00002C3E00002C3F
00002C4000002C4100002C4200002C4300002C4400002C4500002C4600002C4700002C4800002C4900002C4A00002C4B
00002C4C00002C4D00002C4E00002C4F00002C5000002C5100002C5200002C5300002C5400002C5500002C5600002C57
00002C5800002C5900002C5A00002C5B00002C5C00002C5D00002C5E00002C5F00002C610000026B00001D7D0000027D
00002C6800002C6A00002C6C0000025100000271000002500000025200002C7300002C760000023F0000024000002C81
00002C8300002C8500002C8700002C8900002C8B00002C8D00002C8F00002C9100002C9300002C9500002C9700002C99
00002C9B00002C9D00002C9F00002CA100002CA300002CA500002CA700002CA900002CAB00002CAD00002CAF00002CB1
00002CB300002CB500002CB700002CB900002CBB00002CBD00002CBF00002CC100002CC300002CC500002CC700002CC9
00002CCB00002CCD00002CCF00002CD100002CD300002CD500002CD700002CD900002CDB00002CDD00002CDF00002CE1
00002CE300002CEC00002CEE00002CF30000A6410000A6430000A6450000A6470000A6490000A64B0000A64D0000A64F
0000A6510000A6530000A6550000A6570000A6590000A65B0000A65D0000A65F0000A6610000A6630000A6650000A667
0000A6690000A66B0000A66D0000A6810000A6830000A6850000A6870000A6890000A68B0000A68D0000A68F0000A691
0000A6930000A6950000A6970000A6990000A69B0000A7230000A7250000A7270000A7290000A72B0000A72D0000A72F
0000A7330000A7350000A7370000A7390000A73B0000A73D0000A73F0000A7410000A7430000A7450000A7470000A749
0000A74B0000A74D0000A74F0000A7510000A7530000A7550000A7570000A7590000A75B0000A75D0000A75F0000A761
0000A7630000A7650000A7670000A7690000A76B0000A76D0000A76F0000A77A0000A77C00001D790000A77F0000A781
0000A7830000A7850000A7870000A78C000002650000A7910000A7930000A7970000A7990000A79B0000A79D0000A79F
0000A7A10000A7A30000A7A50000A7A70000A7A9000002660000025C000002610000026C0000026A0000029E00000287
0000029D0000AB530000A7B50000A7B70000A7B90000A7BB0000A7BD0000A7BF0000A7C10000A7C30000A79400000282
00001D8E0000A7C80000A7CA0000A7D10000A7D70000A7D90000A7F60000FF410000FF420000FF430000FF440000FF45
0000FF460000FF470000FF480000FF490000FF4A0000FF4B0000FF4C0000FF4D0000FF4E0000FF4F0000FF500000FF51
0000FF520000FF530000FF540000FF550000FF560000FF570000FF580000FF590000FF5A00010428000104290001042A
0001042B0001042C0001042D0001042E0001042F00010430000104310001043200010433000104340001043500010436
0001043700010438000104390001043A0001043B0001043C0001043D0001043E0001043F000104400001044100010442
000104430001044400010445000104460001044700010448000104490001044A0001044B0001044C0001044D0001044E
0001044F000104D8000104D9000104DA000104DB000104DC000104DD000104DE000104DF000104E0000104E1000104E2
000104E3000104E4000104E5000104E6000104E7000104E8000104E9000104EA000104EB000104EC000104ED000104EE
000104EF000104F0000104F1000104F2000104F3000104F4000104F5000104F6000104F7000104F8000104F9000104FA
000104FB0001059700010598000105990001059A0001059B0001059C0001059D0001059E0001059F000105A0000105A1
000105A3000105A4000105A5000105A6000105A7000105A8000105A9000105AA000105AB000105AC000105AD000105AE
000105AF000105B0000105B1000105B3000105B4000105B5000105B6000105B7000105B8000105B9000105BB000105BC
00010CC000010CC100010CC200010CC300010CC400010CC500010CC600010CC700010CC800010CC900010CCA00010CCB
00010CCC00010CCD00010CCE00010CCF00010CD000010CD100010CD200010CD300010CD400010CD500010CD600010CD7
00010CD800010CD900010CDA00010CDB00010CDC00010CDD00010CDE00010CDF00010CE000010CE100010CE200010CE3
00010CE400010CE500010CE600010CE700010CE800010CE900010CEA00010CEB00010CEC00010CED00010CEE00010CEF
00010CF000010CF100010CF2000118C0000118C1000118C2000118C3000118C4000118C5000118C6000118C7000118C8
000118C9000118CA000118CB000118CC000118CD000118CE000118CF000118D0000118D1000118D2000118D3000118D4
000118D5000118D6000118D7000118D8000118D9000118DA000118DB000118DC000118DD000118DE000118DF00016E60
00016E6100016E6200016E6300016E6400016E6500016E6600016E6700016E6800016E6900016E6A00016E6B00016E6C
00016E6D00016E6E00016E6F00016E7000016E7100016E7200016E7300016E7400016E7500016E7600016E7700016E78
00016E7900016E7A00016E7B00016E7C00016E7D00016E7E00016E7F0001E9220001E9230001E9240001E9250001E926
0001E9270001E9280001E9290001E92A0001E92B0001E92C0001E92D0001E92E0001E92F0001E9300001E9310001E932
0001E9330001E9340001E9350001E9360001E9370001E9380001E9390001E93A0001E93B0001E93C0001E93D0001E93E
0001E93F0001E9400001E9410001E9420001E943000000DF02000069000003070000FB000000FB010000FB020000FB03
0000FB040000FB050000FB06000005870000FB130000FB140000FB150000FB160000FB170000014900000390000003B0
000001F000001E9600001E9700001E9800001E9900001E9A00001F5000001F5200001F5400001F5600001FB600001FC6
00001FD200001FD300001FD600001FD700001FE200001FE300001FE400001FE600001FE700001FF600001F8000001F81
00001F8200001F8300001F8400001F8500001F8600001F8700001F8000001F8100001F8200001F8300001F8400001F85
00001F8600001F8700001F9000001F9100001F9200001F9300001F9400001F9500001F9600001F9700001F9000001F91
00001F9200001F9300001F9400001F9500001F9600001F9700001FA000001FA100001FA200001FA300001FA400001FA5
00001FA600001FA700001FA000001FA100001FA200001FA300001FA400001FA500001FA600001FA700001FB300001FB3
00001FC300001FC300001FF300001FF300001FB200001FB400001FC200001FC400001FF200001FF400001FB700001FC7
00001FF7";
return t;
}
immutable(uint)[] toTitleTable() nothrow @nogc pure @safe {
static immutable uint[] t =
cast(immutable uint[]) x"
0000004100000042000000430000004400000045000000460000004700000048000000490000004A0000004B0000004C
0000004D0000004E0000004F000000500000005100000052000000530000005400000055000000560000005700000058
000000590000005A0000039C000000C0000000C1000000C2000000C3000000C4000000C5000000C6000000C7000000C8
000000C9000000CA000000CB000000CC000000CD000000CE000000CF000000D0000000D1000000D2000000D3000000D4
000000D5000000D6000000D8000000D9000000DA000000DB000000DC000000DD000000DE000001780000010000000102
0000010400000106000001080000010A0000010C0000010E00000110000001120000011400000116000001180000011A
0000011C0000011E00000120000001220000012400000126000001280000012A0000012C0000012E0000004900000132
0000013400000136000001390000013B0000013D0000013F000001410000014300000145000001470000014A0000014C
0000014E00000150000001520000015400000156000001580000015A0000015C0000015E000001600000016200000164
00000166000001680000016A0000016C0000016E00000170000001720000017400000176000001790000017B0000017D
00000053000002430000018200000184000001870000018B00000191000001F6000001980000023D00000220000001A0
000001A2000001A4000001A7000001AC000001AF000001B3000001B5000001B8000001BC000001F7000001C5000001C5
000001C5000001C8000001C8000001C8000001CB000001CB000001CB000001CD000001CF000001D1000001D3000001D5
000001D7000001D9000001DB0000018E000001DE000001E0000001E2000001E4000001E6000001E8000001EA000001EC
000001EE000001F2000001F2000001F2000001F4000001F8000001FA000001FC000001FE000002000000020200000204
00000206000002080000020A0000020C0000020E00000210000002120000021400000216000002180000021A0000021C
0000021E000002220000022400000226000002280000022A0000022C0000022E00000230000002320000023B00002C7E
00002C7F0000024100000246000002480000024A0000024C0000024E00002C6F00002C6D00002C700000018100000186
000001890000018A0000018F000001900000A7AB000001930000A7AC000001940000A78D0000A7AA0000019700000196
0000A7AE00002C620000A7AD0000019C00002C6E0000019D0000019F00002C64000001A60000A7C5000001A90000A7B1
000001AE00000244000001B1000001B200000245000001B70000A7B20000A7B000000399000003700000037200000376
000003FD000003FE000003FF0000038600000388000003890000038A0000039100000392000003930000039400000395
000003960000039700000398000003990000039A0000039B0000039C0000039D0000039E0000039F000003A0000003A1
000003A3000003A3000003A4000003A5000003A6000003A7000003A8000003A9000003AA000003AB0000038C0000038E
0000038F0000039200000398000003A6000003A0000003CF000003D8000003DA000003DC000003DE000003E0000003E2
000003E4000003E6000003E8000003EA000003EC000003EE0000039A000003A1000003F90000037F00000395000003F7
000003FA000004100000041100000412000004130000041400000415000004160000041700000418000004190000041A
0000041B0000041C0000041D0000041E0000041F00000420000004210000042200000423000004240000042500000426
0000042700000428000004290000042A0000042B0000042C0000042D0000042E0000042F000004000000040100000402
000004030000040400000405000004060000040700000408000004090000040A0000040B0000040C0000040D0000040E
0000040F00000460000004620000046400000466000004680000046A0000046C0000046E000004700000047200000474
00000476000004780000047A0000047C0000047E000004800000048A0000048C0000048E000004900000049200000494
00000496000004980000049A0000049C0000049E000004A0000004A2000004A4000004A6000004A8000004AA000004AC
000004AE000004B0000004B2000004B4000004B6000004B8000004BA000004BC000004BE000004C1000004C3000004C5
000004C7000004C9000004CB000004CD000004C0000004D0000004D2000004D4000004D6000004D8000004DA000004DC
000004DE000004E0000004E2000004E4000004E6000004E8000004EA000004EC000004EE000004F0000004F2000004F4
000004F6000004F8000004FA000004FC000004FE00000500000005020000050400000506000005080000050A0000050C
0000050E00000510000005120000051400000516000005180000051A0000051C0000051E000005200000052200000524
00000526000005280000052A0000052C0000052E00000531000005320000053300000534000005350000053600000537
00000538000005390000053A0000053B0000053C0000053D0000053E0000053F00000540000005410000054200000543
0000054400000545000005460000054700000548000005490000054A0000054B0000054C0000054D0000054E0000054F
00000550000005510000055200000553000005540000055500000556000010D0000010D1000010D2000010D3000010D4
000010D5000010D6000010D7000010D8000010D9000010DA000010DB000010DC000010DD000010DE000010DF000010E0
000010E1000010E2000010E3000010E4000010E5000010E6000010E7000010E8000010E9000010EA000010EB000010EC
000010ED000010EE000010EF000010F0000010F1000010F2000010F3000010F4000010F5000010F6000010F7000010F8
000010F9000010FA000010FD000010FE000010FF000013F0000013F1000013F2000013F3000013F4000013F500000412
000004140000041E0000042100000422000004220000042A000004620000A64A0000A77D00002C630000A7C600001E00
00001E0200001E0400001E0600001E0800001E0A00001E0C00001E0E00001E1000001E1200001E1400001E1600001E18
00001E1A00001E1C00001E1E00001E2000001E2200001E2400001E2600001E2800001E2A00001E2C00001E2E00001E30
00001E3200001E3400001E3600001E3800001E3A00001E3C00001E3E00001E4000001E4200001E4400001E4600001E48
00001E4A00001E4C00001E4E00001E5000001E5200001E5400001E5600001E5800001E5A00001E5C00001E5E00001E60
00001E6200001E6400001E6600001E6800001E6A00001E6C00001E6E00001E7000001E7200001E7400001E7600001E78
00001E7A00001E7C00001E7E00001E8000001E8200001E8400001E8600001E8800001E8A00001E8C00001E8E00001E90
00001E9200001E9400001E6000001EA000001EA200001EA400001EA600001EA800001EAA00001EAC00001EAE00001EB0
00001EB200001EB400001EB600001EB800001EBA00001EBC00001EBE00001EC000001EC200001EC400001EC600001EC8
00001ECA00001ECC00001ECE00001ED000001ED200001ED400001ED600001ED800001EDA00001EDC00001EDE00001EE0
00001EE200001EE400001EE600001EE800001EEA00001EEC00001EEE00001EF000001EF200001EF400001EF600001EF8
00001EFA00001EFC00001EFE00001F0800001F0900001F0A00001F0B00001F0C00001F0D00001F0E00001F0F00001F18
00001F1900001F1A00001F1B00001F1C00001F1D00001F2800001F2900001F2A00001F2B00001F2C00001F2D00001F2E
00001F2F00001F3800001F3900001F3A00001F3B00001F3C00001F3D00001F3E00001F3F00001F4800001F4900001F4A
00001F4B00001F4C00001F4D00001F5900001F5B00001F5D00001F5F00001F6800001F6900001F6A00001F6B00001F6C
00001F6D00001F6E00001F6F00001FBA00001FBB00001FC800001FC900001FCA00001FCB00001FDA00001FDB00001FF8
00001FF900001FEA00001FEB00001FFA00001FFB00001F8800001F8900001F8A00001F8B00001F8C00001F8D00001F8E
00001F8F00001F9800001F9900001F9A00001F9B00001F9C00001F9D00001F9E00001F9F00001FA800001FA900001FAA
00001FAB00001FAC00001FAD00001FAE00001FAF00001FB800001FB900001FBC0000039900001FCC00001FD800001FD9
00001FE800001FE900001FEC00001FFC0000213200002160000021610000216200002163000021640000216500002166
0000216700002168000021690000216A0000216B0000216C0000216D0000216E0000216F00002183000024B6000024B7
000024B8000024B9000024BA000024BB000024BC000024BD000024BE000024BF000024C0000024C1000024C2000024C3
000024C4000024C5000024C6000024C7000024C8000024C9000024CA000024CB000024CC000024CD000024CE000024CF
00002C0000002C0100002C0200002C0300002C0400002C0500002C0600002C0700002C0800002C0900002C0A00002C0B
00002C0C00002C0D00002C0E00002C0F00002C1000002C1100002C1200002C1300002C1400002C1500002C1600002C17
00002C1800002C1900002C1A00002C1B00002C1C00002C1D00002C1E00002C1F00002C2000002C2100002C2200002C23
00002C2400002C2500002C2600002C2700002C2800002C2900002C2A00002C2B00002C2C00002C2D00002C2E00002C2F
00002C600000023A0000023E00002C6700002C6900002C6B00002C7200002C7500002C8000002C8200002C8400002C86
00002C8800002C8A00002C8C00002C8E00002C9000002C9200002C9400002C9600002C9800002C9A00002C9C00002C9E
00002CA000002CA200002CA400002CA600002CA800002CAA00002CAC00002CAE00002CB000002CB200002CB400002CB6
00002CB800002CBA00002CBC00002CBE00002CC000002CC200002CC400002CC600002CC800002CCA00002CCC00002CCE
00002CD000002CD200002CD400002CD600002CD800002CDA00002CDC00002CDE00002CE000002CE200002CEB00002CED
00002CF2000010A0000010A1000010A2000010A3000010A4000010A5000010A6000010A7000010A8000010A9000010AA
000010AB000010AC000010AD000010AE000010AF000010B0000010B1000010B2000010B3000010B4000010B5000010B6
000010B7000010B8000010B9000010BA000010BB000010BC000010BD000010BE000010BF000010C0000010C1000010C2
000010C3000010C4000010C5000010C7000010CD0000A6400000A6420000A6440000A6460000A6480000A64A0000A64C
0000A64E0000A6500000A6520000A6540000A6560000A6580000A65A0000A65C0000A65E0000A6600000A6620000A664
0000A6660000A6680000A66A0000A66C0000A6800000A6820000A6840000A6860000A6880000A68A0000A68C0000A68E
0000A6900000A6920000A6940000A6960000A6980000A69A0000A7220000A7240000A7260000A7280000A72A0000A72C
0000A72E0000A7320000A7340000A7360000A7380000A73A0000A73C0000A73E0000A7400000A7420000A7440000A746
0000A7480000A74A0000A74C0000A74E0000A7500000A7520000A7540000A7560000A7580000A75A0000A75C0000A75E
0000A7600000A7620000A7640000A7660000A7680000A76A0000A76C0000A76E0000A7790000A77B0000A77E0000A780
0000A7820000A7840000A7860000A78B0000A7900000A7920000A7C40000A7960000A7980000A79A0000A79C0000A79E
0000A7A00000A7A20000A7A40000A7A60000A7A80000A7B40000A7B60000A7B80000A7BA0000A7BC0000A7BE0000A7C0
0000A7C20000A7C70000A7C90000A7D00000A7D60000A7D80000A7F50000A7B3000013A0000013A1000013A2000013A3
000013A4000013A5000013A6000013A7000013A8000013A9000013AA000013AB000013AC000013AD000013AE000013AF
000013B0000013B1000013B2000013B3000013B4000013B5000013B6000013B7000013B8000013B9000013BA000013BB
000013BC000013BD000013BE000013BF000013C0000013C1000013C2000013C3000013C4000013C5000013C6000013C7
000013C8000013C9000013CA000013CB000013CC000013CD000013CE000013CF000013D0000013D1000013D2000013D3
000013D4000013D5000013D6000013D7000013D8000013D9000013DA000013DB000013DC000013DD000013DE000013DF
000013E0000013E1000013E2000013E3000013E4000013E5000013E6000013E7000013E8000013E9000013EA000013EB
000013EC000013ED000013EE000013EF0000FF210000FF220000FF230000FF240000FF250000FF260000FF270000FF28
0000FF290000FF2A0000FF2B0000FF2C0000FF2D0000FF2E0000FF2F0000FF300000FF310000FF320000FF330000FF34
0000FF350000FF360000FF370000FF380000FF390000FF3A000104000001040100010402000104030001040400010405
000104060001040700010408000104090001040A0001040B0001040C0001040D0001040E0001040F0001041000010411
00010412000104130001041400010415000104160001041700010418000104190001041A0001041B0001041C0001041D
0001041E0001041F0001042000010421000104220001042300010424000104250001042600010427000104B0000104B1
000104B2000104B3000104B4000104B5000104B6000104B7000104B8000104B9000104BA000104BB000104BC000104BD
000104BE000104BF000104C0000104C1000104C2000104C3000104C4000104C5000104C6000104C7000104C8000104C9
000104CA000104CB000104CC000104CD000104CE000104CF000104D0000104D1000104D2000104D30001057000010571
00010572000105730001057400010575000105760001057700010578000105790001057A0001057C0001057D0001057E
0001057F000105800001058100010582000105830001058400010585000105860001058700010588000105890001058A
0001058C0001058D0001058E0001058F000105900001059100010592000105940001059500010C8000010C8100010C82
00010C8300010C8400010C8500010C8600010C8700010C8800010C8900010C8A00010C8B00010C8C00010C8D00010C8E
00010C8F00010C9000010C9100010C9200010C9300010C9400010C9500010C9600010C9700010C9800010C9900010C9A
00010C9B00010C9C00010C9D00010C9E00010C9F00010CA000010CA100010CA200010CA300010CA400010CA500010CA6
00010CA700010CA800010CA900010CAA00010CAB00010CAC00010CAD00010CAE00010CAF00010CB000010CB100010CB2
000118A0000118A1000118A2000118A3000118A4000118A5000118A6000118A7000118A8000118A9000118AA000118AB
000118AC000118AD000118AE000118AF000118B0000118B1000118B2000118B3000118B4000118B5000118B6000118B7
000118B8000118B9000118BA000118BB000118BC000118BD000118BE000118BF00016E4000016E4100016E4200016E43
00016E4400016E4500016E4600016E4700016E4800016E4900016E4A00016E4B00016E4C00016E4D00016E4E00016E4F
00016E5000016E5100016E5200016E5300016E5400016E5500016E5600016E5700016E5800016E5900016E5A00016E5B
00016E5C00016E5D00016E5E00016E5F0001E9000001E9010001E9020001E9030001E9040001E9050001E9060001E907
0001E9080001E9090001E90A0001E90B0001E90C0001E90D0001E90E0001E90F0001E9100001E9110001E9120001E913
0001E9140001E9150001E9160001E9170001E9180001E9190001E91A0001E91B0001E91C0001E91D0001E91E0001E91F
0001E9200001E92102000053000000730000013002000046000000660200004600000069020000460000006C03000046
000000660000006903000046000000660000006C02000053000000740200005300000074020005350000058202000544
000005760200054400000565020005440000056B0200054E00000576020005440000056D020002BC0000004E03000399
0000030800000301030003A500000308000003010200004A0000030C0200004800000331020000540000030802000057
0000030A020000590000030A02000041000002BE020003A500000313030003A50000031300000300030003A500000313
00000301030003A500000313000003420200039100000342020003970000034203000399000003080000030003000399
00000308000003010200039900000342030003990000030800000342030003A50000030800000300030003A500000308
00000301020003A100000313020003A500000342030003A50000030800000342020003A90000034200001F8800001F89
00001F8A00001F8B00001F8C00001F8D00001F8E00001F8F00001F8800001F8900001F8A00001F8B00001F8C00001F8D
00001F8E00001F8F00001F9800001F9900001F9A00001F9B00001F9C00001F9D00001F9E00001F9F00001F9800001F99
00001F9A00001F9B00001F9C00001F9D00001F9E00001F9F00001FA800001FA900001FAA00001FAB00001FAC00001FAD
00001FAE00001FAF00001FA800001FA900001FAA00001FAB00001FAC00001FAD00001FAE00001FAF00001FBC00001FBC
00001FCC00001FCC00001FFC00001FFC02001FBA00000345020003860000034502001FCA000003450200038900000345
02001FFA000003450200038F00000345030003910000034200000345030003970000034200000345030003A900000342
00000345";
return t;
}

}

