package io

import (
	"bufio"
	"errors"
	"io"
	"strings"

	json "github.com/json-iterator/go"
)

func init() {
	registerInputHandler("json", func() InputHandler { return &JSONInputHandler{} })
}

// Declare conformance with InputHandler interface.
var _ InputHandler = &JSONInputHandler{}

// JSONInputHandler handles parsing JSON data.
type JSONInputHandler struct{}

// Name implements InputHandler.Name().
func (j *JSONInputHandler) Name() string {
	return "json"
}

// Parse implements InputHandler.Parse().
func (j *JSONInputHandler) Parse(reader io.Reader) (interface{}, error) {
	bufrd := bufio.NewReader(reader)

	_, err := bufrd.Peek(1)
	if errors.Is(err, io.EOF) {
		// Input was empty, replace with an empty object.
		reader = strings.NewReader("{}")
	} else {
		reader = bufrd
	}

	decoder := json.NewDecoder(reader)
	decoder.UseNumber()

	var parsed interface{}
	err = decoder.Decode(&parsed)
	if err != nil {
		return nil, err
	}

	return parsed, nil
}

// SetOption implements InputHandler.SetOption().
func (j *JSONInputHandler) SetOption(name string, value string) error {
	// The JSON handler does not have any options.
	return nil
}
