//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// CLI/GUI 共通のメイン部分
//

#include "mainapp.h"
#include "accel_avx2.h"
#include "config.h"
#include "hostnet.h"
#include "logger.h"
#include "monitor.h"
#include "mystring.h"
#include "mythread.h"
#include "sram.h"
#include "vm_luna.h"
#include "vm_news.h"
#include "vm_virt68k.h"
#include "vm_x68k.h"
#include <fcntl.h>
#include <getopt.h>
#include <pwd.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <algorithm>
#include <thread>
#if defined(__linux__)
#include <sys/prctl.h>
#endif

// グローバルインスタンス
MainApp gMainApp;

// コンストラクタ
MainApp::MainApp()
{
}

// デストラクタ
MainApp::~MainApp()
{
	// 逆順に解放

	pVM.reset();

	pConfig.reset();
	gConfig = NULL;

	pMonitorManager.reset();
	gMonitorManager = NULL;

	hostcpu.reset();

#if 0	// 確認用
	if (objects.empty() == false) {
		printf("~MainApp: undead objects are:");
		for (const auto *obj : objects) {
			printf(" %s", obj->GetIdStr());
		}
		printf("\n");
	}
#endif

	logger.reset();
}

// VM を解放。
void
MainApp::Dispose()
{
	if ((bool)pVM) {
		pVM->Dispose();
	}
}

// ヘルプメッセージ
void
MainApp::ShowHelp(bool all) const
{
	ShowVersion();

#define p(msg)	printf(" " msg "\n")

	printf("usage: %s [<options>]\n", getprogname());
	p("-c <path>           vm directory or configuration file");
	p("-f                  fast mode (same as '-V fast-mode=1')");
	p("--fd[01] <path>     specify floppy image (same as '-V fdN-image=<path>' except");
	p("                     base directory)");
	if (IsGUI())
		p("--fontsize <height> fontsize (same as '-V monitor-fontsize=<height>')");
	p("-h                  show brief help message");
	p("--help              show all help message including for developers");
	if (IsGUI())
		p("-s <scale>          mainview scale (same as '-V mainview-scale=<scale>')");
	p("--show-config       show configuration variables");
	p("--show-hostnet      show list of hostnet drivers");
	p("-v                  show version");
	p("-V <key>=<val>      overwrite config option");
	p("-X <file>           load and execute host binary (a.out or ELF)");
	p("--initrd <img>      initial ramdisk (only for virt68k)");

	if (all) {
		printf("\n(options for developers)\n");
		p("--b [<cpu>,]<addr>[,<skip>] (or -b)");
		p("                    set breakpoint. <cpu> is either 'main'(default) or 'xp'");
		p("--bi [<cpu>,]<inst>[:<mask>][,<skip>]");
		p("                    set instruction breakpoint.");
		p("--bi-exg            set instruction breakpoint on 'exg sp,sp'");
		p("--bv [<cpu>,]<vec1>[-<vec2>][,<skip>]");
		p("                    set vector breakpoint.");
		p("-C                  output log to console");
		p("--console-log <file> specify the console output log");
		p("-d                  debugger prompt on startup");
		p("-D                  same as '-V debugger-driver=stdio'");
		p("-H                  human68k console emulation");
		p("-L <name>=<lv>[,..] set loglevel (-Lhelp displays names)");
		p("-M <name>[,..]      monitors to display at startup (-Mhelp displays names)");
		p("   memdump<N>[=<hex-addr>[.<fmt>]]  fmt := B/W/L/M(MMU)/I(Disasm)/Z(XPDisasm)");
		p("-S                  MSX-DOS console emulation");
		p("--perf              performance measure mode");
	}
}

// long option は enum と struct option をアルファベット順に並べる。
// enum は getopt() の1文字のオプションと衝突しなければいいので適当に
// 0x80 から始めておく。
enum {
	OPTstart = 0x80 - 1,
	OPT_bi,
	OPT_bi_exg,
	OPT_bv,
	OPT_console_log,
	OPT_create_sram,
	OPT_fd0,
	OPT_fd1,
	OPT_fontsize,
	OPT_help,
	OPT_initrd,
	OPT_perf,
	OPT_show_config,
	OPT_show_config_all,
	OPT_show_hostnet,
};
static struct option longopts[] = {
	{ "b",				required_argument,	NULL,	'b' },
	{ "bi",				required_argument,	NULL,	OPT_bi },
	{ "bi-exg",			no_argument,		NULL,	OPT_bi_exg },
	{ "bv",				required_argument,	NULL,	OPT_bv },
	{ "console-log",	required_argument,	NULL,	OPT_console_log },
	{ "create-sram",	no_argument,		NULL,	OPT_create_sram },
	{ "fd0",			required_argument,	NULL,	OPT_fd0 },
	{ "fd1",			required_argument,	NULL,	OPT_fd1 },
	{ "fontsize",		required_argument,	NULL,	OPT_fontsize },
	{ "help",			no_argument,		NULL,	OPT_help },
	{ "initrd",			required_argument,	NULL,	OPT_initrd },
	{ "perf",			no_argument,		NULL,	OPT_perf },
	{ "show-config",	no_argument,		NULL,	OPT_show_config },
	// --show-config-all は開発用なのでヘルプには載せない
	{ "show-config-all",no_argument,		NULL,	OPT_show_config_all },
	{ "show-hostnet",	no_argument,		NULL,	OPT_show_hostnet },
	{ NULL,				0,					NULL,	0 },
};

// VM の初期化、ステージ1。
// VM 作成と設定確定あたりまで。スレッド生成を伴わないもの。
// 所々 CLI と GUI で処理が違う。
// 戻り値は以下のいずれか。
//  MainApp::PASS .. 実行を継続(通常パス)
//  EXIT_SUCCESS / EXIT_FAILURE .. この終了コードでアプリケーションを終了
int
MainApp::Init1(bool is_cli_, int ac, char *av[])
{
	int rv;

	is_cli = is_cli_;

	rv = Init1a(ac, av);
	if (rv != MainApp::PASS) {
		return rv;
	}

	rv = Init1b();
	return rv;
}

// VM の初期化、ステージ1 の前半。
// Config.Fix まで。
int
MainApp::Init1a(int ac, char *av[])
{
	int rv;

#if defined(__linux__)
	// Linux ではケーパビリティが設定されていると coredump しないので
	// 明示的に許可する必要がある
	prctl(PR_SET_DUMPABLE, 1, 0, 0, 0);
#endif

	rv = ParseOpt(ac, av);
	if (rv != MainApp::PASS) {
		return rv;
	}

	// ログ機構は引数処理後なるはや
	// CLI ならログは常に標準出力へ(も)出力。
	// GUI なら -C で指定。
	logger.reset(new Logger());
	logger->UseStdout(IsCLI() ? true : log_to_console);

	// モニタマネージャ (VM 作成より前、もしかしたら Config より前?)
	pMonitorManager.reset(new MonitorManager());
	gMonitorManager = pMonitorManager.get();

	// 設定を作成。
	// c0. コンストラクタで初期値を用意。
	pConfig.reset(new Config());
	gConfig = pConfig.get();

	// c1. ホームディレクトリに設定ファイルがあれば読み込んで設定を更新。
	struct passwd *passwd = getpwuid(getuid());
	// 見付からないことはないはずだが、一応。
	if (passwd != NULL) {
		std::string filename = std::string(passwd->pw_dir) + "/.nono.cfg";
		ConfigFile dotfile(filename, ConfigItem::FromHome);
		if (dotfile.Load()) {
			if (gConfig->Update(dotfile) == false) {
				return EXIT_FAILURE;
			}
		}
	}

	// c2. VM ディレクトリに設定ファイルがあれば読み込んで設定を更新。
	// vmfile が存在するかどうかはここで調べるまで分からない。
	ConfigFile cfgfile(vmfile, ConfigItem::FromConfig);
	if (cfgfile.Load() == false) {
		return EXIT_FAILURE;
	}
	if (gConfig->Update(cfgfile) == false) {
		return EXIT_FAILURE;
	}

	// c3. 最後にコマンドライン引数で更新
	for (const auto& pair : config_options) {
		if (gConfig->Update(pair.first, pair.second) == false) {
			return EXIT_FAILURE;
		}
	}

	// c4. exec-file, exec-arg をここでメンバ変数 exec_file, exec_arg に代入。
	// (c3 までに確定していて、1. VM コンストラクト時には必要)
	const ConfigItem& itemexec = gConfig->Find("exec-file");
	const std::string& execstr = itemexec.AsString();
	if (execstr.empty() == false) {
		// 歴史的経緯で exec_file は const char *。
		exec_file = strdup(NormalizePath(execstr).c_str());
	}
	const ConfigItem& itemarg = gConfig->Find("exec-arg");
	exec_arg = itemarg.AsString();

	// Human68k/MSX-DOS モードならここでいくつかパラメータを強制的に変更。
	if (human_mode) {
		// X68030 最小構成でいい。
		gConfig->Update("vmtype=x68030", "-H");
	}
	if (msxdos_mode) {
		// LUNA-I 最小構成でいい。
		gConfig->Update("vmtype=luna", "-S");
		gConfig->Update("ram-size=16", "-S");
		gConfig->Update("luna-video-plane=1", "-S");
		// ホスト側には干渉しない。
		gConfig->Update("hostcom0-driver=none", "-S");
		gConfig->Update("hostcom1-driver=none", "-S");
		gConfig->Update("hostcom2-driver=none", "-S");
		gConfig->Update("hostnet-driver=none", "-S");
	}

	// ログ表示のためだけのオブジェクト
	hostcpu.reset(new Object(OBJ_HOSTCPU));

	// ホスト CPU の機能チェック。
	if (CheckCPU() == false) {
		return EXIT_FAILURE;
	}

	// 0. VM 種別を決定
	// VM 種別文字列から vmtype を決定
	const ConfigItem& item = gConfig->Find("vmtype");
	vmstr = string_tolower(item.AsString());
	if (vmstr == "x68030") {
		vmtype = VMType::X68030;
	} else if (vmstr == "luna") {
		vmtype = VMType::LUNA1;
	} else if (vmstr == "luna88k") {
		vmtype = VMType::LUNA88K;
	} else if (vmstr == "news") {
		vmtype = VMType::NEWS;
	} else if (vmstr == "virt-m68k" || vmstr == "virt68k") {
		vmtype = VMType::VIRT68K;
	} else {
		if (item.GetFrom() == ConfigItem::FromInitial) {
			// 未指定の時
			warnx("vmtype must be specified");
		} else {
			// ユーザ由来の時
			item.Err("Invalid vmtype");
		}
		return EXIT_FAILURE;
	}

	// 0.5. VM が確定したところで SRAM 作成。
	if (create_sram) {
		if (vmtype == VMType::X68030) {
			return CreateSRAM();
		} else {
			warnx("--create-sram is only for X68030 mode");
			return EXIT_FAILURE;
		}
	}

	// 1. VM 作成
	switch (vmtype) {
	 case VMType::X68030:
		pVM.reset(new VM_X68030());
		break;

	 case VMType::LUNA1:
		pVM.reset(new VM_LUNA1());
		break;

	 case VMType::LUNA88K:
		pVM.reset(new VM_LUNA88K());
		break;

	 case VMType::NEWS:
		pVM.reset(new VM_NEWS());
		break;

	 case VMType::VIRT68K:
		pVM.reset(new VM_Virt68k());
		break;

	 default:
		PANIC("vmtype=%s not configured", vmstr.c_str());
	}

	// 2. 動的なコンストラクションその1
	// 3. ログ出力用の EarlyInit
	if (!pVM->Create(VM::CreationPhase::First)) {
		pVM.reset();
		return EXIT_FAILURE;
	}

	// 4. 設定を確定。
	// (VM コンストラクタで変数の増減があるのでそれより後、
	// Create() でも SCSI パラメータを減らすので、それより後)
	if (gConfig->Fix() == false) {
		pVM.reset();
		return EXIT_SUCCESS;
	}

	// Fix() は配列から削除を行うため、これより前に取得していた ConfigItem
	// のポインタや参照は無効になっている可能性がある。Fix() をまたいで
	// ポインタや参照を使ってしまわないようここで一旦フォーカスを抜ける。
	// なお Fix() 後は要素の削除は出来ないため、以降はこの問題は発生しない。

	return PASS;
}

// VM の初期化、ステージ1 の後半。
// Config.Fix 以降。
int
MainApp::Init1b()
{
	// 4.1 設定が確定したので --show-config なら設定内容を表示して終了。
	if (show_config) {
		gConfig->Show(show_config - 1);
		pVM.reset();
		return EXIT_SUCCESS;
	}

	// 4.2 ログレベルを設定。コンストラクト後すぐに行う。
	// -L help もここで処理。
	if (!ParseLogopt()) {
		pVM.reset();
		return EXIT_FAILURE;
	}

	// 4.3 -X のチェック。
	// -L help より後にしないとヘルプが表示できない。
	const ConfigItem& itemexec = gConfig->Find("exec-file");
	if (exec_file == NULL) {
		// -X が必要なケースで指定されてなければエラー。

		// 実行ファイル指定が必須の VM。
		if (vmtype == VMType::NEWS || vmtype == VMType::VIRT68K) {
			if (itemexec.GetFrom() == ConfigItem::FromInitial) {
				warnx("vmtype=%s requires -X option (or exec-file).",
					vmstr.c_str());
			} else {
				itemexec.Err("filename must be specified.");
			}
			return EXIT_FAILURE;
		}

		// Human モード、MSX-DOS モードでも実行ファイル名が必要。
		if (human_mode) {
			warnx("-H option needs -X");
			return EXIT_FAILURE;
		}
		if (msxdos_mode) {
			warnx("-S option needs -X");
			return EXIT_FAILURE;
		}
	} else {
		// 実行ファイルのファイル名を間違えたくらいならここでエラーに出来る。
		int r = access(exec_file, R_OK);
		if (r != 0) {
			itemexec.Err("%s", strerror(errno));
			return EXIT_FAILURE;
		}
	}

	// 5. 動的コンストラクションその2
	// ログ出力が出来る代わりにログクラスはもう増やせない。
	if (!pVM->Create(VM::CreationPhase::Second)) {
		pVM.reset();
		return EXIT_FAILURE;
	}

	return PASS;
}

// VM の初期化、ステージ2。スレッド生成を伴う。
// Init1() で引数を受け付けてから Init2() でデバッガスレッドなどを起動するが、
// -Mhelp とかが指定された場合はここでスレッド開始する前にプロセスを終了する
// 必要があるので分けてある。wxapp.cpp も参照。
bool
MainApp::Init2()
{
	// 7. VM 初期化。
	if (!pVM->Init()) {
		return false;
	}

	// メインスレッド名を設定
	PTHREAD_SETNAME("Main");

	// 8. スレッド開始
	if (!pVM->StartThread()) {
		return false;
	}

	// 9. 起動時設定の適用
	if (!pVM->Apply()) {
		return false;
	}

	return true;
}

// コマンドライン引数を処理する。
// 知らない引数とかがあればこちらで usage を表示して false を返す。
// 戻り値は、MainApp::PASS なら実行を継続、
// EXIT_SUCCESS/EXIT_FAILURE ならこのコードで終了。
int
MainApp::ParseOpt(int ac, char *av[])
{
	struct stat st;
	const char *cpath;
	int c;

	cpath = NULL;

	while ((c = getopt_long(ac, av, "b:c:CdDfhHL:M:s:SvV:X:",
	                        longopts, NULL)) != -1) {
		switch (c) {
		 case 'b':
			debug_breakaddr.push_back(optarg);
			break;

		 case 'c':
			cpath = optarg;
			// 空文字列なら再び初期値に
			if (cpath[0] == '\0') {
				cpath = NULL;
			}
			break;

		 case 'C':
			log_to_console = true;
			break;

		 case 'd':
			debug_on_start = true;
			break;

		 case 'D':
			// -D は -V debugger-driver=stdio と等価。
			config_options.emplace_back("debugger-driver=stdio", "-D");
			break;

		 case 'f':
			// -f は -V fast-mode=1 と等価。
			config_options.emplace_back("fast-mode=1", "-f");
			break;

		 case OPT_fd0:
		 case OPT_fd1:
		 {
			int fd = c - OPT_fd0;
			// 相対パスはカレントディレクトリを起点とする。
			std::string line = string_format("fd%u-image=%s", fd,
				AbsPath(optarg).c_str());
			std::string from = string_format("--fd%u", fd);
			config_options.emplace_back(line, from);
			break;
		 }

		 case 'H':
			human_mode = true;
			break;

		 case 'L':
			AddLogopt(optarg);
			break;

		 case 'M':
			if (!monitor_opt.empty()) {
				monitor_opt += ",";
			}
			monitor_opt += optarg;
			break;

		 case 's':
		 {
			auto line = string_format("mainview-scale=%s", optarg);
			config_options.emplace_back(line, "-s");
			break;
		 }

		 case 'S':
			msxdos_mode = true;
			break;

		 case 'X':
		 {
			// 相対パスはカレントディレクトリ起点とする。
			std::string line = "exec-file=" + AbsPath(optarg);
			config_options.emplace_back(line, "-X");

			// Human68k モードなら後続を引数にする。
			// UNIX モードでは親和性が悪いのでしない。
			if (human_mode) {
				std::string arg;
				arg = "exec-arg=";
				for (int i = optind; i < ac; i++) {
					if (i != optind) {
						arg += " ";
					}
					arg += av[i];
				}
				optind = ac;
				config_options.emplace_back(arg, "-X");
			}
			break;
		 }

		 case 'v':
			ShowVersion();
			exit(0);

		 case 'V':
			config_options.emplace_back(optarg, "-V");
			break;

		 case OPT_bi:
			debug_breakinst.push_back(optarg);
			break;

		 case OPT_bi_exg:
			debug_breakinst_exg = true;
			break;

		 case OPT_bv:
			debug_breakvec.push_back(optarg);
			break;

		 case OPT_console_log:
			// この時点ではコンソールを持つ VM かどうかが分からないので
			// コンソールのない設定でもエラーを出さない(出せない)。
			console_logfile = optarg;
			break;

		 case OPT_create_sram:
			create_sram = true;
			break;

		 case OPT_fontsize:
		 {
			auto line = string_format("monitor-fontsize=%s", optarg);
			config_options.emplace_back(line, "--fontsize");
			break;
		 }

		 case OPT_help:
			ShowHelp(true);
			return EXIT_SUCCESS;

		 case OPT_initrd:
		 {
			// 相対パスはカレントディレクトリを起点とする。
			std::string line = "exec-initrd=" + AbsPath(optarg);
			config_options.emplace_back(line, "--initrd");
			break;
		 }

		 case OPT_perf:
			// パフォーマンス確認用。(Config::Update() 側で展開する)
			config_options.emplace_back("--perf", "");
			log_to_console = true;
			break;

		 case OPT_show_config:
			show_config = 1;
			break;

		 case OPT_show_config_all:
			show_config = 2;
			break;

		 case OPT_show_hostnet:
			ShowHostnet();
			return EXIT_SUCCESS;

		 case 'h':
			ShowHelp(false);
			return EXIT_SUCCESS;

		 default:
			ShowHelp(false);
			return EXIT_FAILURE;
		}
	}

	if (human_mode) {
		// Human モードで -c ありなら、VM ディレクトリ指定。
		// Human モードで -c 省略なら、VM ディレクトリなし (cpath==NULL)。
	} else {
		// Human モードでなければ、-c 省略は -c . と同義。
		if (cpath == NULL) {
			cpath = ".";
		}
	}

	if (cpath != NULL) {
		// 引数がディレクトリなら、それを VM ディレクトリとし、その中の
		// nono.cfg を設定ファイルとする。
		// 引数がファイルなら、それを設定ファイルとし、そのファイルがある
		// ディレクトリを VM ディレクトリとする。
		// -c DIR       => vmdir = DIR, vmfile = DIR/nono.cfg
		// -c DIR/FILE  => vmdir = DIR, vmfile = FILE
		if (stat(cpath, &st) < 0) {
			warn("stat %s", cpath);
			return EXIT_FAILURE;
		}
		if (S_ISDIR(st.st_mode)) {
			vmdir = std::string(cpath);
			if (vmdir.back() != '/') {
				vmdir += '/';
			}
			vmfile = vmdir + "nono.cfg";
		} else if (S_ISREG(st.st_mode)) {
			vmfile = std::string(cpath);
			auto pos = vmfile.rfind('/');
			if (pos != std::string::npos) {
				vmdir = vmfile.substr(0, pos + 1);
			} else {
				vmdir = "./";
			}
		} else {
			warnx("-c %s: path must be file or directory", cpath);
			return EXIT_FAILURE;
		}
	}

	// CLI 版で -Mhelp つけても黙って起動するのはさすがにどうかと思う。
	// とは言え GUI 版のコマンドラインを CLI 版用に書き換えた時に -M を
	// 消して回らないと起動できないのも面倒なので -M オプション自体は
	// 無意味だけど許容する。うーん。
	if (IsCLI()) {
		if (monitor_opt == "help") {
			warnx("-Mhelp is not available on CLI");
			return EXIT_FAILURE;
		}
		if (!monitor_opt.empty()) {
			warnx("-M option is ignored on CLI");
			return EXIT_FAILURE;
		}
	}

	return MainApp::PASS;
}

// path を絶対パスにして返す。
// 相対パスならカレントディレクトリからのパスとする。
/*static*/ std::string
MainApp::AbsPath(const char *path)
{
	char cwd[PATH_MAX];

	getcwd(cwd, sizeof(cwd));
	return NormalizePath(optarg, std::string(cwd));
}

// CPU の機能とかをチェック。
bool
MainApp::CheckCPU()
{
	const ConfigItem& item_aff = gConfig->Find(".host-cpu-affinity");
	std::string cfg_aff = item_aff.AsString();

	if (cfg_aff.empty() || cfg_aff == "auto") {
		// 指定なし
		// (auto も今の所指定なしと同じ)
		cpu_affinity = CPUAffinity::None;
	} else if (cfg_aff.find(':') == std::string::npos) {
		item_aff.Err();
	} else {
		// 書式は "<key>:<list>"。
		// <key> は "low", "high", "perf" のいずれか。
		// <list> は "0,3-5,7" みたいな感じ。
		std::vector<std::string> vals = string_split(cfg_aff, ':');
		std::string key = string_trim(vals[0]);
		std::string csv = vals[1];
		if (key == "low") {
			cpu_affinity = CPUAffinity::Low;
		} else if (key == "high") {
			cpu_affinity = CPUAffinity::High;
		} else if (key == "perf") {
			cpu_affinity = CPUAffinity::Perf;
		} else {
			item_aff.Err("Invalid affinity keyword");
		}
		cpu_list.resize(std::thread::hardware_concurrency());
		auto errmsg = ParseCPUList(cpu_list, csv);
		if (errmsg.empty() == false) {
			item_aff.Err(errmsg);
			return false;
		}

		// 全部 true/false なら、これは意味がないのでクリアする。
		uint num = std::count(cpu_list.begin(), cpu_list.end(), true);
		if (num == 0) {
			item_aff.Err("No valid cores specified (Ignoring it)");
			cpu_affinity = CPUAffinity::None;
		} else if (num == cpu_list.size()) {
			item_aff.Err("All cores specified (Ignoring it)");
			cpu_affinity = CPUAffinity::None;
		}
	}

#if defined(HAVE_AVX2)
	// AVX2
	detect_avx2 = ::DetectAVX2();

	const ConfigItem& item_avx2 = gConfig->Find("host-avx2");
	std::string cfg_avx2 = item_avx2.AsString();

	if (cfg_avx2 == "auto") {
		enable_avx2 = detect_avx2;
	} else if (cfg_avx2 == "no") {
		enable_avx2 = false;
	} else {
		item_avx2.Err("must be either \"auto\" or \"no\".");
		return false;
	}
#endif

#if defined(HAVE_NEON)
	// NEON は aarch64 なら必ずある?
	detect_neon = true;

	const ConfigItem& item_neon = gConfig->Find("host-neon");
	std::string cfg_neon = item_neon.AsString();

	if (cfg_neon == "auto") {
		enable_neon = detect_neon;
	} else if (cfg_neon == "no") {
		enable_neon = false;
	} else {
		item_neon.Err("must be either \"auto\" or \"no\".");
		return false;
	}
#endif

	// 何もしない。設定のチェックもしないほうがいい。
	return true;
}

// CPU 番号リストの文字列からビット配列を作って返す。
// cpulist は CPU コア数で初期化(resize)しておくこと。
// 成功すれば string.empty を返す。
// 失敗すればエラーメッセージを返す。
std::string
MainApp::ParseCPUList(std::vector<bool>& cpulist, const std::string& input)
{
	if (input.empty()) {
		return "<cpulist> must be specified";
	}

	// 設定は "0,3-5,7,10-" のような書式
	std::vector<std::string> csv = string_split(input, ',');
	for (auto& r : csv) {
		const char *str;
		char *endp;
		uint start, last;

		if (r.empty()) {
			// 空なら無視する?
			continue;
		} else if (r.find('-') == std::string::npos) {
			// '-' がなければ単独
			str = &r[0];
			errno = 0;
			start = strtoul(str, &endp, 10);
			if (endp == str || *endp != '\0' || errno == ERANGE) {
				return "syntax error";
			}
			last = start;
		} else if (r[0] == '-') {
			// "-a" なら範囲 [0, a]
			str = &r[1];
			errno = 0;
			last = strtoul(str, &endp, 10);
			if (endp == str || *endp != '\0' || errno == ERANGE) {
				return "syntax error";
			}
			start = 0;
		} else if (r[r.size() - 1] == '-') {
			// "a-" なら範囲 [a, MAX]
			str = &r[0];
			errno = 0;
			start = strtoul(str, &endp, 10);
			if (endp == str || endp != &r[r.size() - 1] || errno == ERANGE) {
				return "syntax error";
			}
			last = cpulist.size();
		} else {
			// "a-b" なら範囲 [a, b]
			str = &r[0];
			errno = 0;
			start = strtoul(str, &endp, 10);
			if (endp == str || *endp != '-' || errno == ERANGE) {
				return "syntax error";
			}

			str = endp + 1;
			last = strtoul(str, &endp, 10);
			if (endp == str || *endp != '\0' || errno == ERANGE) {
				return "syntax error";
			}

			if (start > last) {
				uint tmp = start;
				start = last;
				last = tmp;
			}
		}

		if (start >= cpulist.size()) {
			return string_format("cpu number %u exceeds number of cores(%zu)",
				start, cpulist.size());
		}
		if (last >= cpulist.size()) {
			return string_format("cpu number %u exceeds number of cores(%zu)",
				start, cpulist.size());
		}
		for (uint n = start; n <= last; n++) {
			cpulist[n] = true;
		}
	}

	return "";
}

// バージョンを表示
void
MainApp::ShowVersion() const
{
	// ここは実行ファイル名によらず nono にする
	fprintf(stderr, "nono version %d.%d.%d (%s)\n",
		NONO_MAJOR_VER, NONO_MINOR_VER, NONO_PATCH_VER, NONO_DATE);
}

// ログレベル指定文字列を logopt に追加する。
void
MainApp::AddLogopt(const char *opt)
{
	if (logopt.empty() == false) {
		logopt += ',';
	}
	logopt += opt;
}

// ログレベル指定文字列を処理する。
// str はログレベル指定文字列を ',' で連結した "foo=1,bar=2" 形式の文字列で、
// これを分解してそれぞれ担当するオブジェクトのログレベルにセットする。
// "help" があれば設定は行わず一覧を表示。
bool
MainApp::ParseLogopt()
{
	std::vector<std::string> items;

	// 分解して..
	items = string_split(logopt.c_str(), ',');

	// "help" があればヘルプを表示して終了
	for (const auto& item : items) {
		if (item == "help") {
			std::vector<std::string> list = GetLogNames();
			// less したいだろうから stderr ではなく stdout に出力する
			for (const auto& name : list) {
				printf(" %s\n", name.c_str());
			}
			return false;
		}
	}

	// ログレベルを設定
	std::string errmsg;
	if (SetLogopt(items, &errmsg) == false) {
		warnx("%s", errmsg.c_str());
		return false;
	}

	return true;
}

// ログレベルを設定する。
// ログレベル指定文字列を 1つずつに分解したリスト items を処理する。
// "help" があるケースはここに来るまでに処理してあるので、ここには来ない。
// 成功なら何も表示せず true を返す。失敗なら *errmsg にエラーメッセージを
// 格納して false を返す。
// MainApp 内と Debugger からも呼ばれる。
/*static*/ bool
MainApp::SetLogopt(const std::vector<std::string>& items, std::string *errmsg)
{
	for (const auto& item : items) {
		if (SetLogopt1(item.c_str(), errmsg) == false) {
			return false;
		}
	}
	if (0) {	// デバッグ用
		for (const auto o : gMainApp.GetObjects()) {
			printf("%-16s %d\n", o->GetName().c_str(), o->loglevel);
		}
	}
	return true;
}

// "logname[=loglevel]" 形式をパースしてオブジェクトにログレベルを設定する。
// loglevel は省略なら 1 とする。
// logname が "all" なら全オブジェクトにセットする。
// そうでない場合は case ignore で完全一致するか前方一致で1つに確定すれば、
// そのオブジェクトにログレベルを設定して true を返す。
// 見付からないか候補が複数ある場合はエラーメッセージを *errmsg に出力して
// false を返す。
// この関数は (このすぐ上の SetLogopt() を経由して)
// MainApp と Debugger から呼ばれることに注意。
/*static*/ bool
MainApp::SetLogopt1(const std::string& item, std::string *errmsg)
{
	std::string name;
	const char *v;
	int level;

	v = strchr(item.c_str(), '=');
	if (v) {
		name = std::string(item.c_str(), v - item.c_str());
		level = atoi(++v);
	} else {
		name = item;
		level = 1;
	}
	// ここで name は変数名、level は値(省略されたら1)

	if (name.empty()) {
		*errmsg = "logname must be specified";
		return false;
	}

	// 今の所、値域は -1 〜 9 ということにしておく。
	if (level < -1) {
		level = -1;
	} else if (level > 9) {
		level = 9;
	}

	// "all" なら全部にセット
	if (name == "all") {
		for (auto obj : gMainApp.GetObjects()) {
			if (obj->GetName().empty() == false) {
				obj->SetLogLevel(level);
			}
		}
		return true;
	}

	// "<key>" なら "<key>*" 全部に展開する。
#define EXPAND(key)	do {	\
	if (name == key) {	\
		for (auto obj : gMainApp.GetObjects()) {	\
			if (strncasecmp(obj->GetName().c_str(), key, strlen(key)) == 0)	\
				obj->SetLogLevel(level);	\
		}	\
		return true;	\
	}	\
} while (0)
	EXPAND("bankram");
	EXPAND("fdd");
	EXPAND("gfpic");
	EXPAND("hostcom");
	EXPAND("hostnet");

	// エイリアスリストを作る
	using aliaslist_t = std::vector<std::pair<std::string, Object *>>;
	aliaslist_t alias_list;
	for (const auto& obj : gMainApp.GetObjects()) {
		const std::vector<std::string>& aliases = obj->GetAliases();

		for (const auto& a : aliases) {
			alias_list.emplace_back(a, obj);
		}
	}

	// エイリアスを完全一致のみのものと部分一致も許容するものに分ける
	aliaslist_t exact_alias;
	aliaslist_t partial_alias;
	for (const auto& a0 : alias_list) {
		bool exact = false;
		for (const auto& a1 : alias_list) {
			if (a0.first == a1.first) {
				continue;
			}
			if (starts_with_ignorecase(a0.first, a1.first)) {
				exact = true;
				break;
			}
		}
		if (exact) {
			exact_alias.push_back(a0);
		} else {
			partial_alias.push_back(a0);
		}
	}

	// 完全一致をまず調べる
	for (const auto& a : exact_alias) {
		if (strcasecmp(a.first.c_str(), name.c_str()) == 0) {
			a.second->SetLogLevel(level);
			return true;
		}
	}

	aliaslist_t found;
	for (const auto& a : partial_alias) {
		// 前方一致したら覚えておく
		if (starts_with_ignorecase(a.first, name)) {
			found.push_back(a);
		}
	}

	// 見付からない場合はエラー
	if (found.empty()) {
		*errmsg = string_format("Unknown logname \"%s\"", name.c_str());
		return false;
	}

	// 1つだけなら確定
	if (found.size() == 1) {
		found[0].second->SetLogLevel(level);
		return true;
	}

	// 複数あれば候補文字列を作成
	*errmsg = string_format("Ambiguous logname \"%s\": candidates are",
		name.c_str());
	for (const auto& cand : found) {
		*errmsg += string_format(" \"%s\"", cand.first.c_str());
	}
	return false;
}

// ログ名の一覧を取得する。
// MainApp と Debugger から呼ばれる。
/*static*/ std::vector<std::string>
MainApp::GetLogNames()
{
	std::vector<Object *> sortobj;

	// エイリアスを持つオブジェクトだけ抜き出す
	for (const auto& obj : gMainApp.GetObjects()) {
		if (obj->GetAliases().empty() == false) {
			sortobj.emplace_back(obj);
		}
	}

	// aliases の一語目で sortobj をソートする
	std::sort(sortobj.begin(), sortobj.end(),
		[](const auto a, const auto b) {
			const auto& sa = a->GetAliases()[0];
			const auto& sb = b->GetAliases()[0];
			return sa < sb;
		}
	);

	std::vector<std::string> list;
	for (const auto *obj : sortobj) {
		const auto& aliases = obj->GetAliases();
		std::string str;

		str = aliases[0];
		if (aliases.size() > 1) {
			str += " (alias:";
			for (int i = 1, sz = aliases.size(); i < sz; i++) {
				str += ' ';
				str += aliases[i];
			}
			str += ')';
		}
		list.emplace_back(str);
	}
	list.emplace_back("all");

	return list;
}

// 関連するファイルのパスを取得。
// 1. name がファイル名のみなら、VM ディレクトリとその親ディレクトリを検索。
//    ファイルが存在すればその時点で戻り値とする。ファイルが存在したけど
//    何らか不都合があったとしても (例えばファイルサイズが 0 だとか
//    パーミッションが足りないとか) 次を試すとかはしない。
// 2. 通常手順(相対パスは VM ディレクトリ起点) でパスを展開。
//    ファイルの有無は不問。
std::string
MainApp::SearchFile(const std::string& name) const
{
	// 1. パス区切りを含んでなければ、ファイル名のみ
	if (name.find('/') == std::string::npos) {
		std::string path;
		struct stat st;

		// 1a. VM ディレクトリ
		path = GetVMDir() + name;
		if (stat(path.c_str(), &st) == 0) {
			return path;
		}

		// 1b. 親ディレクトリ
		path = GetVMDir() + "../" + name;
		if (stat(path.c_str(), &st) == 0) {
			return path;
		}

		// どちらもなければエラー
		return "";
	}

	// 2. 通常手順でパスを展開。
	return NormalizePath(name);
}

// path を必要なら絶対パスに展開して返す。
// 相対パスは VM ディレクトリを起点とする。
std::string
MainApp::NormalizePath(const std::string& path) const
{
	return NormalizePath(path, GetVMDir());
}

// path を必要なら絶対パスに展開して返す。
// 相対パスは basedir を起点とする。
/*static*/ std::string
MainApp::NormalizePath(const std::string& path, const std::string& basedir)
{
	// 絶対パスならそのまま返す。
	if (path[0] == '/') {
		return path;
	}

	std::string newpath;

	if (path[0] == '~') {
		// 先頭が '~' ならホームディレクトリに展開。

		struct passwd *passwd = getpwuid(getuid());
		// 見付からないことはないはずだが、一応。
		if (passwd != NULL) {
			newpath = std::string(passwd->pw_dir);
		}
		newpath += &path[1];
	} else {
		// basedir からの相対パスとして展開。

		newpath = basedir;
		if (newpath.empty() || newpath.back() != '/') {
			newpath += '/';
		}
		newpath += path;
	}

	return newpath;
}

// 初回起動時用に SRAM.DAT を作成する。
// 戻り値は EXIT_SUCCESS / EXIT_FAILURE。
int
MainApp::CreateSRAM()
{
	char buf[16 * 1024];
	std::string filename;
	autofd fd;
	int r;

	filename = GetVMDir() + "SRAM.DAT";

	// 存在確認のため一旦読み込み専用で開いてみる。
	// オープン出来たら何もしない。
	fd = open(filename.c_str(), O_RDONLY);
	if (fd >= 0) {
		warnx("%s: Already exists", filename.c_str());
		return EXIT_FAILURE;
	}

	fd = open(filename.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 0644);
	if (fd < 0) {
		warn("%s: open failed", filename.c_str());
		return EXIT_FAILURE;
	}

	memset(&buf[0], 0, sizeof(buf));
	for (int i = 0; i < SRAMDevice::InitialData.size(); i++) {
		buf[i] = SRAMDevice::InitialData[i];
	}

	r = write(fd, buf, sizeof(buf));
	if (r < 0) {
		warn("%s: write failed", filename.c_str());
		return EXIT_FAILURE;
	}

	fd.Close();
	warnx("created %s", filename.c_str());
	return EXIT_SUCCESS;
}

// 現在の VM が指定のケーパビリティを持っているか?
bool
MainApp::Has(VMCap cap) const
{
	uint32 vmcap = 1U << (int)GetVMType();
	return (vmcap & (uint32)cap) != 0;
}

// VM 機種名を返す。
const char *
MainApp::GetVMName() const
{
	if ((bool)pVM) {
		return pVM->GetVMName();
	} else {
		return NULL; // not configured
	}
}

// コンパイルされている host netdriver の一覧を表示。
void
MainApp::ShowHostnet() const
{
	auto list = HostNetDevice::GetDrivers();
	for (const auto& name : list) {
		printf(" %s\n", name.c_str());
	}
}

// オブジェクト登録
void
MainApp::RegistObject(Object *obj)
{
	// ID が重複していないかチェック (NONE なら重複可)
	auto id = obj->GetId();
	if (id != OBJ_NONE) {
		if (FindObject(id)) {
			PANIC("%s already exists", Object::GetIdStr(id));
		}
	}

	obj->logger = gMainApp.GetLogger();

	objects.push_back(obj);
}

// オブジェクト削除
void
MainApp::UnregistObject(Object *obj)
{
	// 最初に見付かった一つを削除するだけでいい
	for (auto it = objects.begin(); it != objects.end(); ++it) {
		if (*it == obj) {
			objects.erase(it);
			break;
		}
	}
}

// 指定された id を持つオブジェクトを返す。なければ NULL を返す。
Object *
MainApp::FindObject(uint id) const
{
	// NONE はここで検索しない (SCSI デバイスなど、そっちで検索する)
	if (id == OBJ_NONE) {
		return NULL;
	}
	for (auto obj : objects) {
		if (obj->GetId() == id) {
			return obj;
		}
	}
	return NULL;
}

// 指定された id を持つオブジェクトを探す。なければ assert する。
Object *
MainApp::GetObject(uint id) const
{
	Object *obj = FindObject(id);
	if (__predict_false(obj == NULL)) {
		PANIC("objid=%s not found", Object::GetIdStr(id));
	}
	return obj;
}
