//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// ログモニター
//

#include "wxlogmonitor.h"
#include "logger.h"
#include "mainapp.h"
#include "monitor.h"
#include "sjis.h"

//#define WINDOW_DEBUG 1

#if defined(WINDOW_DEBUG)
#define DPRINTF(fmt...) printf(fmt)
#else
#define DPRINTF(fmt...) /**/
#endif

// コンストラクタ
WXLogMonitor::WXLogMonitor(wxWindow *parent)
	: inherited(parent, wxID_ANY, _("Log"), DEFAULT_STYLE | wxRESIZE_BORDER)
{
	DPRINTF("%s begin\n", __method__);
	col = 80;
	row = 40;

	// プライベート利用なので Regist() しない。
	monitor.reset(new Monitor(-1, &log));
	monitor->func = ToMonitorCallback(&LogMonitor::MonitorUpdate);
	monitor->SetSize(col, row);

	log.Init(this);

	// +------------+---------+
	// | TextScreen | VScroll |
	// +------------+---------+

	// テキストスクリーン
	screen = new WXMonitorPanel(this, monitor.get());

	// スクロールバー
	vscroll = new WXScrollBar(this, wxID_ANY, wxSB_VERTICAL);

	// Sizer 使わず自前でレイアウトする。
	SetAutoLayout(true);

	// ウィンドウサイズを確定させる
	FontChanged();

	// スクロールのために (パネル領域での) MouseWheel イベントもここで
	// 受け取りたい。スクロールバー上のマウスホイール操作はスクロールバーが
	// 処理しているので問題ないが、パネル領域でのマウスホイール操作はここ
	// (wxFrame)ではなくパネルに対して飛んでくる。
	// ここで一緒に処理したほうが楽なので、こちらに回す。
	screen->Connect(wxEVT_MOUSEWHEEL,
		wxMouseEventHandler(WXLogMonitor::OnMouseWheel), NULL, this);

	// スクロールバーからの位置変更通知イベントは表示パネルへ回す。
	vscroll->Connect(NONO_EVT_SCROLL,
		wxScrollEventHandler(WXLogMonitor::OnScroll), NULL, this);

	// XXX ログウィンドウのコピーはまだ動いてない
	screen->DisconnectContextMenu();
	DPRINTF("%s done\n", __method__);
}

// デストラクタ
WXLogMonitor::~WXLogMonitor()
{
}

void
WXLogMonitor::FontChanged()
{
	screen->FontChanged();
	vscroll->FontChanged();
	DPRINTF("%s %d\n", __method__, screen->GetFontHeight());

	Fit();
}

void
WXLogMonitor::Fit()
{
	// inherited::Fit() は呼ばずに自力で行う。

	wxSize newsize = DoSizeHints();
#if defined(WINDOW_DEBUG)
	wxSize newwin = ClientToWindowSize(newsize);
	printf("%s (%d,%d) client=(%d,%d)\n", __method__,
		newwin.x, newwin.y, newsize.x, newsize.y);
#endif
	SetClientSize(newsize);
}

bool
WXLogMonitor::Layout()
{
	// inherited::Layout() は(実質何もしないので)、呼ばずに自力で行う。

	wxSize client = GetClientSize();
	const wxSize win = GetSize();
	const wxSize margin = win - client;
	const wxSize vsize = vscroll->GetSize();
	const wxSize monsize = screen->GetSize();

	// wxmonitor.cpp の Layout() 内のコメント参照。
	if (oldmargin != margin) {
		layout_pass2 = true;
		oldmargin = margin;
		CallAfter([this]() {
			DoSizeHints();
		});
		return true;
	} else if (layout_pass2) {
		layout_pass2 = false;
		client.y = monsize.y;
		SetClientSize(client);
		Hide();
		Show();
		return true;
	}

	// コントロールを配置。
	int panel_width  = client.x - vsize.x;
	int panel_height = client.y;
	// 最低でも 1px ないと GTK とかに怒られる。
	if (panel_width < 1) {
		panel_width = 1;
	}
	if (panel_height < 1) {
		panel_height = 1;
	}

	screen->SetSize(0, 0, panel_width, panel_height);
	vscroll->SetSize(panel_width, 0, vsize.x, panel_height);

	// スクロールバーを再設定
	SetScroll();

	DPRINTF("%s done\n", __method__);

	return true;
}

// クライアントサイズと最大/最小サイズを計算する。
// SetSizeHints() を実行し、設定すべきクライアントサイズを返す。
wxSize
WXLogMonitor::DoSizeHints()
{
	auto ssize = screen->GetSize();
	auto vsize = vscroll->GetSize();
	DPRINTF("%s screen=(%d,%d) vscroll=(%d,%d)\n", __method__,
		ssize.x, ssize.y, vsize.x, vsize.y);

	int x = ssize.x + vsize.x;
	int min_y = std::max(vscroll->GetMinSize().y, screen->GetScreenHeight(1));
	int new_y = std::max(ssize.y, min_y);
	wxSize minsize(x, min_y);
	wxSize maxsize(x, GetMaxClientSize().y);
	wxSize incsize(0, screen->GetFontHeight());
	SetSizeHints(ClientToWindowSize(minsize), ClientToWindowSize(maxsize),
		incsize);

	return wxSize(x, new_y);
}

// スクロールバーを再設定
void
WXLogMonitor::SetScroll()
{
	int pos;
	int thumbsize;
	int range;
	int pagesize;

	row = screen->GetRow();

	if (lines <= row) {
		pos = 0;
		range = row;
	} else {
		pos = (lines - row) - vpos;
		range = lines;
	}
	thumbsize = row;
	pagesize = thumbsize - 1;
	vscroll->SetScrollParam(pos, thumbsize, range, pagesize);
}

// マウスホイールイベント (WXTextScreen 上で起きたやつをこっちに回してある)
void
WXLogMonitor::OnMouseWheel(wxMouseEvent& event)
{
	// GetWheelRotation() は上(奥)向きに回すと +120、下(手前)に回すと -120。
	// どこの決まりか知らんけど、スクロールバーのほうはホイール1段階で3行
	// スクロールするのでそれに合わせる。
	int pos = vscroll->GetThumbPosition();
	pos = pos - event.GetWheelRotation() / 40;

	int maxpos = vscroll->GetRange() - vscroll->GetThumbSize();
	if (pos < 0)
		pos = 0;
	if (pos > maxpos)
		pos = maxpos;

	DoScroll(pos);
	// スクロールバーの位置を追従
	vscroll->SetThumbPosition(pos);
}

// スクロールバーからの通知イベント
void
WXLogMonitor::OnScroll(wxScrollEvent& event)
{
	DoScroll(event.GetPosition());
}

// スクロール処理 (OnScroll() と OnMouseWheel() から呼ばれる)
void
WXLogMonitor::DoScroll(int pos)
{
	// pos 即ちスクロールバーの Thumb の位置は上端の位置。
	// 対してログウィンドウでは下端を基準にしたほうが処理しやすい。
	// vpos = 0 だと Thumb が下に接していてパネルの最下行が最新行、
	// vpos = 1 だと Thumb が下から1段階分上にありパネルの最下行は最新行の
	// 一つ前の行、となる。
	//
	// Range=6		Range=6
	// Pos=4		Pos=0
	// ThumbSz=2	ThumbSz=2
	// vpos=0		vpos=4
	// △			△
	// □			■
	// □			■
	// □			□
	// □			□
	// ■			□
	// ■			□
	// ▽			▽

	// パネル最下行に表示する行の最新行からのオフセット
	// (0 なら Thumb が下端に接している)
	vpos = vscroll->GetRange() - vscroll->GetThumbSize() - pos;
}


//
// モニターオブジェクト
//

// コンストラクタ
LogMonitor::LogMonitor()
	: inherited(OBJ_LOGMONITOR)
{
}

// デストラクタ
LogMonitor::~LogMonitor()
{
}

void
LogMonitor::Init(WXLogMonitor *parent_)
{
	parent = parent_;

	logger = gMainApp.GetLogger();

	// バックログは今の所固定長
	logbuf.Init(parent->col, backlog);
	logbuf.Mode = TextScreen::Ring;
}

// モニターテキストを更新
void
LogMonitor::MonitorUpdate(Monitor *, TextScreen& screen)
{
	char buf[1024];

	// どこでどうするかはあるけど、
	// とりあえず row は常に現在の TextScreen の高さということにしておく
	parent->row = screen.GetRow();

	// XXX: 生成時にやるべき
	screen.Mode = TextScreen::Ring;

	// キューから読めるだけバックログに書き込む。
	// 読めるだけと言っても while ループで読めるだけ読んでしまうと、
	// 流速が早すぎた場合にこのループから抜けるタイミングがなくなる可能性が
	// (可能性としては) あるので、適当に1画面分くらいで打ち切る。
	// 取り込み残した分は次回取り込まれるし、その流速も上回るようだと
	// どのみち Logger の固定長キューで落ちる。
	for (int i = 0; i < parent->row; i++) {
		if (logger->Read(buf, sizeof(buf)) == false)
			break;
		Append(buf);
	}

	screen.Clear();

	// スクロールバー(vpos)による表示位置計算
	// vpos は行数、v は logbuf の Y 座標
	int v = cursor - parent->vpos;
	if (v < 0) {
		v += backlog;
	}

	// 表示位置(v)の Row 行分手前から順に表示
	int srcY = v - parent->row;
	if (srcY < 0) {
		srcY += backlog;
	}

	// 画面コピー
	logbuf.Locate(0, srcY);
	for (int i = 0; i < parent->col * parent->row; i++) {
		uint16 ch = logbuf.Getc();
		logbuf.Ascend();
		screen.Putc(ch);
	}
}

// 1行追加された時の共通処理
void
LogMonitor::IncLines()
{
	// スクロールバーが下端にない時(バックログを見ている時)は
	// ログが追加されても現在位置をキープする。
	if (parent->vpos > 0) {
		// 上端に達したら増加させない
		if (parent->vpos < backlog - parent->vscroll->GetThumbSize()) {
			parent->vpos++;
		}
	}

	// 有効行数を増やす
	if (parent->lines < backlog) {
		parent->lines++;

		// 行数が変わったのでスクロールバーを更新
		parent->SetScroll();
	}
}

// バックログに行を追加。
//
// 引数の utfbuf は UTF-8 で改行コードのない1行ずつ。
// これをバックログの最新位置に書き込む。古い行は順次上書きされる。
//
// バックログは col * backlog バイトの Shift_JIS バッファ。
// 属性は持たず文字コードのみ。漢字は2バイト幅をとる。
// 横幅 col が固定なので書き込む時点で行数は表示用のそれと一致する。
//
// cursor はこれから書き込む位置の行頭。cursor は必ず行頭だけをポイント
// しながら移動する。
//
// logbuf    +------------------------
//           | :
//           | 前回書き込んだ行
// cursor -> | 今から書きこむ行 (保存されていた一番古い行)
//           :
//           |
//           +------------------------
void
LogMonitor::Append(const char *utfbuf)
{
	const char *s;
	int x;

	// 文字コードの変換
	wxString utfstr(utfbuf, wxConvUTF8);
	// mb_str() をキャストせずに一度保持しておく必要がある (scan-build 対策)
	auto sjis = utfstr.mb_str(conv);

	logbuf.Locate(0, cursor);

	for (s = (const char *)sjis; *s; ) {
		// 最終桁で2バイト文字の1バイト目が来たら
		// 代わりに1文字空白を入れる (それによってこの後改行が起きる)
		if (SJIS::IsZenkaku(*s)) {
			// 漢字ならば 2 バイトセットで扱う
			if (logbuf.GetX() >= parent->col - 1) {
				// 最終桁には出せないので空白で次の行に送る
				logbuf.Putc(' ');
				IncLines();
			}
			logbuf.Putc(*s++);
			// 万が一に備える
			if (__predict_true(*s)) {
				logbuf.Putc(*s++);
			}
		} else {
			// そうでなければそのまま出力
			logbuf.Putc(*s++);
		}
	}

	// 行の終わりまでは空白でクリアする
	x = logbuf.GetX();
	if (x > 0) {
		for (; x < logbuf.GetCol(); x++) {
			logbuf.Putc(' ');
		}
		IncLines();
	}
	// 次回書き込みの Y 座標
	cursor = logbuf.GetY();
}
