//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// プレーン VRAM モニターウィンドウ
//

#pragma once

#include "wxsubwindow.h"
#include "wxscrollbar.h"
#include "wxtextscreen.h"
#include <array>

class BT45xDevice;
class PlaneVRAMDevice;
class VideoCtlrDevice;

// プレーン VRAM ビットマップパネル
class WXPlaneVRAMBitmapPanel : public WXBitmapPanel
{
	using inherited = WXBitmapPanel;

 public:
	explicit WXPlaneVRAMBitmapPanel(wxWindow *parent);
	~WXPlaneVRAMBitmapPanel() override;

	void EnablePlane(int plane, bool value);
	void EnablePalette(bool value);
	void SetScale(int scale_);

	const wxPoint& GetCursor() const { return cursor; }

	void OnScroll(wxScrollEvent& event);

	Rect view {};				// 表示矩形
	int virtual_width {};		// 拡大後のビットマップ幅   [pixel]
	int virtual_height {};		// 拡大後のビットマップ高さ [pixel]

 private:
	void OnPaletteChanged(wxCommandEvent& event);
	void GenPalette();
	void Draw() override;
	void OnMouse(wxMouseEvent& event);

	std::array<Color, 256> pal {};
	uint32 planemask {};
	bool apply_palette {};
	int scale {};

	int plane_width {};			// 拡大前のビットマップ幅   [pixel]
	int plane_height {};		// 拡大前のビットマップ高さ [pixel]

	// マウスカーソルがこのパネル上にあれば、(拡大前の)仮想画面での座標。
	// マウスカーソルがパネル外なら wxDefaultPosition = (-1, -1)。
	// ステータス表示用。
	wxPoint cursor {};

	BT45xDevice *bt45x {};
	PlaneVRAMDevice *planevram {};
	VideoCtlrDevice *videoctlr {};

	// イベントテーブル
	wxDECLARE_EVENT_TABLE();
};

// ビットマップパネルとスクロールバーを持つ
class WXPlaneVRAMPanel : public WXTextPanel
{
	using inherited = WXTextPanel;
 public:
	explicit WXPlaneVRAMPanel(wxWindow *parent);
	~WXPlaneVRAMPanel() override;

	void Fit() override;
	bool Layout() override;

	// スクロールバーの FontChanged() を呼ぶため(だけ)に
	// これ自身も wxTextPanel を継承している。
	void FontChanged() override;

	void EnablePlane(int plane, bool value);
	void EnablePalette(bool value);
	void SetScale(int scale_);

	const wxPoint& GetCursor() const { return viewctl->GetCursor(); }

 private:
	void SetScroll();

	WXPlaneVRAMBitmapPanel *viewctl {};
	WXScrollBar *vscroll {};
	WXScrollBar *hscroll {};
	WXBitmapPanel *corner {};	// 右下の空き地
};

// プレーン VRAM ウィンドウ
class WXPlaneVRAMWindow : public WXSubWindow
{
	using inherited = WXSubWindow;

 public:
	static const int MAX_PLANES = 8;

 public:
	WXPlaneVRAMWindow(wxWindow *parent, const wxString& name);
	~WXPlaneVRAMWindow() override;

	void Fit() override;
	bool Layout() override;

	// ステータスパネル
	WXTextScreen *statuspanel {};

 private:
	void DoSizeHints();

	void OnPlane(wxCommandEvent& event);
	void DoPlane(int plane, bool value);
	void OnApplyPalette(wxCommandEvent& event);
	void DoApplyPalette(bool value);
	void OnScale(wxCommandEvent& event);
	void DoScale(int value);

	void OnTimer(wxTimerEvent& event);

	// コントロールパネル
	wxPanel *ctrlpanel {};
	std::vector<wxCheckBox*> planesw {};
	wxCheckBox *applysw {};
	wxChoice *scalesw {};

	// ステータスパネル右の空き地用
	WXBitmapPanel *spacer {};

	// グラフィックパネル
	WXPlaneVRAMPanel *viewpanel {};

	// タイマー
	wxTimer timer {};

	PlaneVRAMDevice *planevram {};

	// 謎のリサイズ対策
	int min_height {};
	wxSize correct_clientsize {};

	// イベントテーブル
	wxDECLARE_EVENT_TABLE();

	// 状態の二次記憶域。
	// サブウィンドウは今のところ開いてる時だけオブジェクトが存在している
	// タイプだが、このウィンドウを閉じてまた開くとさっきの状態が継続して
	// いてほしいので、ここでその状況を真似する。
	// ただしこっちはあくまで補助用で、コントロールの値のほうが一次情報。
	static std::array<bool, MAX_PLANES> sticky_plane;
	static bool sticky_apply;
	static int sticky_scale;

	// sticky_used が false ならまだ一度も使われていない (次のコンストラクタ
	// で初期化する)、true なら前回の値が入っている。
	static bool sticky_used;
};
