//
// nono
// Copyright (C) 2024 nono project
// Licensed under nono-license.txt
//

//
// メイン画面をコンソールにする
//

#pragma once

#include "device.h"
#include "keyboard.h"
#include "renderer.h"
#include <bitset>
#include <vector>

class BitmapRGBX;
class COMDriver;
class COMDriverConsole;

class ConsoleDevice : public Device
{
	using inherited = Device;

	static const uint width  = 80;
	static const uint height = 30;
	static const uint font_width  = 8;
	static const uint font_height = 16;

	using bitsetH = std::bitset<height>;

	// 1文字 32ビットのうち下位8ビットが文字コード、上位24ビットが各種属性。
	//
	//
	//    3                   2                   1
	//  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
	// +---------------+---------------+---------------+---------------+
	// |    bgcolor    |    fgcolor    |R| |D|O|S|U|I|B|   文字コード  |
	// +---------------+---------------+---------------+---------------+
	//
	// bgcolor, fgcolor はカラーパレット番号。

	static const uint32 ATTR_BOLD		= 0x0000'0100;	// B: ボールド
	static const uint32 ATTR_ITALIC		= 0x0000'0200;	// I: イタリック
	static const uint32 ATTR_UNDERLINE	= 0x0000'0400;	// U: 下線
	static const uint32 ATTR_STRIKE		= 0x0000'0800;	// S: 打ち消し線
	static const uint32 ATTR_OVERLINE	= 0x0000'1000;	// O: 上線
	static const uint32 ATTR_DECSG		= 0x0000'2000;	// D: DEC 特殊文字
	static const uint32 ATTR_REVERSE	= 0x0000'8000;	// R: 反転(暫定)

	static const uint32 ATTR_LINEMASK	= (ATTR_UNDERLINE |
										   ATTR_STRIKE |
										   ATTR_OVERLINE);

	// フチ [pixel]
	static const uint Padding = ConsoleRenderer::Padding;

 public:
	ConsoleDevice();
	~ConsoleDevice() override;

	bool Init() override;
	void ResetHard(bool poweron) override;

	void Attach(COMDriver *);
	void Detach() { Attach(NULL); }

	void Putchar(uint32);

	bool Render(BitmapRGBX& dst);

 private:
	void VSyncCallback(Event *);
	void SetPalette(bool console_is_active);
	void ResetTerminal();
	void ResetState();
	void PutcharPlain(uint32);
	void PutcharDebug(uint32);
	bool PutcharESC(uint32);
	bool PutcharCSI(uint32);
	bool PutcharCSIQ(uint32);

	void Clear();

	void LocateX(int x);
	void LocateY(int y);
	void Locate(int x, int y) {
		LocateX(x);
		LocateY(y);
	}

	void Ascend();
	void CR();
	void LF();
	void CRLF() {
		CR();
		LF();
	}
	void ScrollUp();
	void ScrollDown();

	// 現在位置に、現在の属性で文字 ch を出力する。出力後カーソルは移動する。
	void Putc(uint32 ch);

	// (x, y) の位置に chattr を出力する。カーソルは移動しない。
	void Setc(int x, int y, uint32 chattr);

	// 現在の属性を設定する。
	void SetAttr(uint32 new_attr);

	std::string Dump() const;
	std::string Dump(uint32 ch) const;
	static std::string Dump(const std::vector<uint8>&);

	// 画面は 1文字分が uint32 の [width * height] の配列。
	std::vector<uint32> screen {};
	// カーソル位置。
	int cur_x {};
	int cur_y {};

	// VM スレッド内での行ごとの更新情報。
	bitsetH dirty {};
	// レンダラスレッドで未処理の行。
	// VM スレッドからレンダラスレッドへの連絡用なので mtx で保護する。
	bitsetH pending {};
	std::mutex mtx {};

	// 画面初期化。
	bool init_screen {};

	// 現在の状態。
	char state {};
	// 現在の属性。
	uint32 cur_attr {};

	// パレット。
	// [0..1] だと通常色、[1..2] だと反転色になる。
	Color palette[3] {};

	// 現在の1文字前までのエスケープシーケンス。(現在の文字は ch)
	std::vector<uint8> seq {};
	// パラメータ部分。
	std::string arg {};

	// カーソル位置保存。
	int save_curx {};
	int save_cury {};

	// スクロール範囲。[0, height) の座標系。
	int scroll_top {};
	int scroll_btm {};

	// デバッグログ。
	FILE *log {};
	bool log_newline {};

	Event *vsync_event {};

	VideoRenderer *renderer {};
	COMDriver *comdriver {};

	static const uint8 decsg8x16[15 * font_height];
};

// 入力担当デバイス
class ConsoleKeyboard : public DumbKeyboard
{
	using inherited = DumbKeyboard;
 public:
	ConsoleKeyboard();
	~ConsoleKeyboard() override;

	void Attach(COMDriverConsole *);
	void Detach() { Attach(NULL); }

	void CharInput(uint charcode) override;

 private:
	COMDriverConsole *comdriver {};
};

static inline ConsoleDevice *GetConsoleDevice() {
	return Object::GetObject<ConsoleDevice>(OBJ_CONSOLE);
}
