//
// nono
// Copyright (C) 2024 nono project
// Licensed under nono-license.txt
//

//
// Goldfish 割り込みコントローラ
//

#pragma once

#include "interrupt.h"

class GFPICDevice : public InterruptDevice
{
	using inherited = InterruptDevice;
	friend class Virt68kInterrupt;

	static const uint32 STATUS		= 0x00U >> 2;
	static const uint32 PENDING		= 0x04U >> 2;	// was NUMBER
	static const uint32 CLEAR_ALL	= 0x08U >> 2;	// was DISABLE_ALL
	static const uint32 DISABLE		= 0x0cU >> 2;
	static const uint32 ENABLE		= 0x10U >> 2;

 public:
	explicit GFPICDevice(uint id_);
	~GFPICDevice() override;

	bool Init() override;
	void ResetHard(bool poweron) override;

	busdata InterruptAcknowledge(int lv) override;

	// IRQ の登録。子デバイスが呼ぶ。irq は 1-32。
	void RegistIRQ(int irq, const char *name, Device *source);

	// Virt68kInterrupt の下請け
	void MonitorScreenSummary(TextScreen& screen, int x, int y);
	int MonitorScreenDetail(TextScreen& screen, int y, int lv);

 protected:
	// BusIO インタフェース
	static const uint32 NPORT = 0x1000 >> 2;
	busdata ReadPort(uint32 offset);
	busdata WritePort(uint32 offset, uint32 data);
	busdata PeekPort(uint32 offset);

 private:
	uint32 GetSTATUS() const;
	uint32 GetPENDING() const;

	void ChangeInterrupt() override;

	InterruptDevice *interrupt {};
};

static inline GFPICDevice *GetGFPICDevice(int id_) {
	return Object::GetObject<GFPICDevice>(OBJ_GFPIC(id_));
}
