//
// nono
// Copyright (C) 2025 nono project
// Licensed under nono-license.txt
//

//
// uint128 のテスト
//

#include "uint128.h"

// 型とクラスの文法の違いを吸収する。
#if defined(UINT128_IS_CLASS)
#define INIT(v, h, l)	v((h), (l))
#define GetH(v)			((v).GetH())
#define GetL(v)			((v).GetL())
#define ToStr(v)		((v).to_str())
#else
#define INIT(v, h, l)	v = (((uint128)(h)) << 64) | ((uint128)(l))
#define GetH(v)			((uint64)(v >> 64))
#define GetL(v)			((uint64)v)
#define ToStr(v)		uint128__to_str(v)
#endif

#if defined(HAVE___INT128)
// __int128 を文字列にして返す。
static std::string
uint128__to_str(unsigned __int128 v)
{
	char buf[48];
	snprintf(buf, sizeof(buf), "%08x'%08x'%08x'%08x",
		(uint32)(v >> 96),
		(uint32)(v >> 64),
		(uint32)(v >> 32),
		(uint32)(v));
	return std::string(buf);
}
#endif

static int total;
static int failed;

#if defined(UINT128_IS_CLASS)
static void
test_ctor_0()
{
	uint128 a;

	total++;
	if (GetH(a) != 0 || GetL(a) != 0) {
		printf("%s: expects 0 but %s\n", __func__, ToStr(a).c_str());
		failed++;
	}
}

static void
test_ctor_1()
{
	constexpr uint64 expected = 0xfedcba98'76543210ULL;
	uint128 a(expected);

	total++;
	if (GetH(a) != 0 || GetL(a) != expected) {
		printf("%s: expects %08x'%08x'%08x'%08x but %s\n", __func__,
			0, 0,
			(uint32)(expected >> 32),
			(uint32)(expected),
			ToStr(a).c_str());
		failed++;
	}
}

static void
test_ctor_2()
{
	constexpr uint64 expectedH = 0xffeeddcc'bbaa9988ULL;
	constexpr uint64 expectedL = 0x77665544'33221100ULL;
	uint128 a(expectedH, expectedL);

	total++;
	if (GetH(a) != expectedH || GetL(a) != expectedL) {
		printf("%s: expects %08x'%08x'%08x'%08x but %s\n", __func__,
			(uint32)(expectedH >> 32),
			(uint32)(expectedH),
			(uint32)(expectedL >> 32),
			(uint32)(expectedL),
			ToStr(a).c_str());
		failed++;
	}
}

static void
test_operator_assign_uint128()
{
	constexpr uint64 expectedH = 0xffeeddcc'bbaa9988ULL;
	constexpr uint64 expectedL = 0x77665544'33221100ULL;
	uint128 exp(expectedH, expectedL);
	uint128 act = exp;

	total++;
	if (GetH(act) != expectedH || GetL(act) != expectedL) {
		printf("%s: expects %s but %s\n", __func__,
			ToStr(exp).c_str(),
			ToStr(act).c_str());
		failed++;
	}
}

static void
test_operator_assign_uint64()
{
	constexpr uint64 expectedL = 0x77665544'33221100ULL;
	uint128 exp(expectedL);
	uint128 act = exp;

	total++;
	if (GetH(act) != 0 || GetL(act) != expectedL) {
		printf("%s: expects %s but %s\n", __func__,
			ToStr(exp).c_str(),
			ToStr(act).c_str());
		failed++;
	}
}

#endif

static void
test_operator_cast_uint64()
{
	constexpr uint64 initialH = 0xffeeddcc'bbaa9988ULL;
	constexpr uint64 initialL = 0x77665544'33221100ULL;
	uint128 INIT(v, initialH, initialL);
	uint64 act = (uint64)v;

	total++;
	if (act != initialL) {
		printf("%s: expects %08x'%08x but %08x'%08x\n", __func__,
			(uint32)(initialL >> 32),
			(uint32)(initialL),
			(uint32)(act >> 32),
			(uint32)(act));
		failed++;
	}
}

static void
test_operator_cast_uint32()
{
	constexpr uint64 initialH = 0xffeeddcc'bbaa9988ULL;
	constexpr uint64 initialL = 0x77665544'33221100ULL;
	constexpr uint32 expected = (uint32)initialL;
	uint128 INIT(v, initialH, initialL);
	uint64 act = (uint32)v;

	total++;
	if (act != expected) {
		printf("%s: expects %08x but %08x\n", __func__,
			act, expected);
		failed++;
	}
}

static void
test_operator_not()
{
	uint128 INIT(val, 0xaaaa5555aaaa5555ULL, 0x0000ffff0000ffffULL);
	uint128 INIT(exp, 0x5555aaaa5555aaaaULL, 0xffff0000ffff0000ULL);
	uint128 act = ~val;

	total++;
	if (GetH(act) != GetH(exp) || GetL(act) != GetL(exp)) {
		printf("%s: expects %s but %s\n", __func__,
			ToStr(exp).c_str(),
			ToStr(act).c_str());
		failed++;
	}
}

static void
test_operator_assign_shift_left()
{
	constexpr uint64 initialH = 0x1122330044556677ULL;
	constexpr uint64 initialL = 0x8899aabbccddeeffULL;
	struct {
		int n;
		uint64 expH, expL;
	} table[] = {
		{ 0,	initialH, initialL },
		{ 4,	0x1223300445566778ULL, 0x899aabbccddeeff0ULL },
		{ 60,	0x78899aabbccddeefULL, 0xf000000000000000ULL },
		{ 64,	initialL, 0 },
		{ 124,	0xf000000000000000ULL, 0 },
	};
	for (const auto& a : table) {
		uint128 INIT(exp, a.expH, a.expL);

		uint128 INIT(act, initialH, initialL);
		act <<= a.n;

		total++;
		if (GetH(act) != GetH(exp) || GetL(act) != GetL(exp)) {
			printf("%s: %u expects %s but %s\n", __func__,
				a.n,
				ToStr(exp).c_str(),
				ToStr(act).c_str());
			failed++;
		}
	}
}

static void
test_operator_assign_shift_right()
{
	constexpr uint64 initialH = 0x1122330044556677ULL;
	constexpr uint64 initialL = 0x8899aabbccddeeffULL;
	struct {
		int n;
		uint64 expH, expL;
	} table[] = {
		{ 0,	initialH, initialL },
		{ 4,	0x0112233004455667ULL, 0x78899aabbccddeefULL },
		{ 60,	0x0000000000000001ULL, 0x1223300445566778ULL },
		{ 64,	0, initialH },
		{ 124,	0, 0x0000000000000001ULL },
	};
	for (const auto& a : table) {
		uint128 INIT(exp, a.expH, a.expL);

		uint128 INIT(act, initialH, initialL);
		act >>= a.n;

		total++;
		if (GetH(act) != GetH(exp) || GetL(act) != GetL(exp)) {
			printf("%s: %u expects %s but %s\n", __func__,
				a.n,
				ToStr(exp).c_str(),
				ToStr(act).c_str());
			failed++;
		}
	}
}

int
main(int ac, char *av[])
{
#if defined(UINT128_IS_CLASS)
	printf("uint128 is class\n");
#else
	printf("uint128 is __int128\n");
#endif

#if defined(UINT128_IS_CLASS)
	test_ctor_0();
	test_ctor_1();
	test_ctor_2();
	test_operator_assign_uint128();
	test_operator_assign_uint64();
#endif
	test_operator_cast_uint64();
	test_operator_cast_uint32();
	test_operator_not();
	test_operator_assign_shift_left();
	test_operator_assign_shift_right();

	printf("%u total, %u passed", total, (total - failed));
	if (failed) {
		printf(", %u FAILED!", failed);
	}
	printf("\n");
	return 0;
}
