//
// nono
// Copyright (C) 2025 nono project
// Licensed under nono-license.txt
//

#pragma once

#include "header.h"

// 時間単位の変換。
constexpr uint64 psec_to_nsec(uint64 ps) { return ps / 1000U; }
constexpr uint64 psec_to_usec(uint64 ps) { return ps / 1000'000U; }
constexpr uint64 psec_to_msec(uint64 ps) { return ps / 1000'000'000U; }
constexpr uint64 psec_to_sec( uint64 ps) { return ps / 1000'000'000'000ULL; }
constexpr uint64 nsec_to_psec(uint64 ns) { return ns * 1000U; }
constexpr uint64 nsec_to_usec(uint64 ns) { return ns / 1000U; }
constexpr uint64 nsec_to_msec(uint64 ns) { return ns / 1000'000U; }
constexpr uint64 nsec_to_sec( uint64 ns) { return ns / 1000'000'000U; }
constexpr uint64 usec_to_psec(uint64 us) { return us * 1000'000U; }
constexpr uint64 usec_to_nsec(uint64 us) { return us * 1000U; }
constexpr uint64 usec_to_msec(uint64 us) { return us / 1000U; }
constexpr uint64 usec_to_sec( uint64 us) { return us / 1000'000U; }
constexpr uint64 msec_to_psec(uint64 ms) { return ms * 1000'000'000U; }
constexpr uint64 msec_to_nsec(uint64 ms) { return ms * 1000'000U; }
constexpr uint64 msec_to_usec(uint64 ms) { return ms * 1000U; }
constexpr uint64 msec_to_sec( uint64 ms) { return ms / 1000U; }
constexpr uint64  sec_to_psec(uint64 ss) { return ss * 1000'000'000'000ULL; }
constexpr uint64  sec_to_nsec(uint64 ss) { return ss * 1000'000'000U; }
constexpr uint64  sec_to_usec(uint64 ss) { return ss * 1000'000U; }
constexpr uint64  sec_to_msec(uint64 ss) { return ss * 1000U; }

// 内部の単位時間は [tsec] と呼ぶ。t は特に意味はないけど tick とか。
static constexpr uint64 TSEC_FREQUENCY = 1000'000'000U;

static constexpr uint64 PSEC_IN_SEC = 1000'000'000'000ULL;
static constexpr uint64 NSEC_IN_SEC = 1000'000'000U;
static constexpr uint64 USEC_IN_SEC = 1000'000U;
static constexpr uint64 MSEC_IN_SEC = 1000U;

constexpr uint64 GCD(uint64 a, uint64 b)
{
	if (a == b) {
		return a;
	}
	if (a < b) {
		uint64 t = b;
		b = a;
		a = t;
	}
	uint64 d = 0;
	do {
		d = a % b;
		a = b;
		b = d;
	} while (d != 0);
	return a;
}

constexpr uint64 tsec_to_xsec(uint64 ts, uint64 base)
{
	auto g = GCD(base, TSEC_FREQUENCY);
	auto n = base / g;
	auto d = TSEC_FREQUENCY / g;
	return ts * n / d;
}
constexpr uint64 xsec_to_tsec(uint64 xs, uint64 base)
{
	auto g = GCD(base, TSEC_FREQUENCY);
	auto n = base / g;
	auto d = TSEC_FREQUENCY / g;
	return xs * d / n;
}

constexpr uint64 tsec_to_psec(uint64 ts) {
	return tsec_to_xsec(ts, PSEC_IN_SEC);
}
constexpr uint64 tsec_to_nsec(uint64 ts) {
	return tsec_to_xsec(ts, NSEC_IN_SEC);
}
constexpr uint64 tsec_to_usec(uint64 ts) {
	return tsec_to_xsec(ts, USEC_IN_SEC);
}
constexpr uint64 tsec_to_msec(uint64 ts) {
	return tsec_to_xsec(ts, MSEC_IN_SEC);
}
constexpr uint64 tsec_to_sec(uint64 ts) {
	return tsec_to_xsec(ts, 1);
}

constexpr uint64 psec_to_tsec(uint64 ps) {
	return xsec_to_tsec(ps, PSEC_IN_SEC);
}
constexpr uint64 nsec_to_tsec(uint64 ns) {
	return xsec_to_tsec(ns, NSEC_IN_SEC);
}
constexpr uint64 usec_to_tsec(uint64 us) {
	return xsec_to_tsec(us, USEC_IN_SEC);
}
constexpr uint64 msec_to_tsec(uint64 ms) {
	return xsec_to_tsec(ms, MSEC_IN_SEC);
}
constexpr uint64 sec_to_tsec(uint64 ss) {
	return xsec_to_tsec(ss, 1);
}

// ユーザ定義リテラルを使ったサフィックス

// 単位時間
constexpr uint64 operator"" _tsec(unsigned long long ts)
{
	return ts;
}
constexpr uint64 operator"" _tsec(long double ts)
{
	return ts;
}

// ナノ秒
constexpr uint64 operator"" _nsec(unsigned long long ns)
{
	return nsec_to_tsec(ns);
}
constexpr uint64 operator"" _nsec(long double ns)
{
	return ns / 1e9 * TSEC_FREQUENCY;
}

// マイクロ秒
constexpr uint64 operator"" _usec(unsigned long long us)
{
	return us * 1000_nsec;
}
constexpr uint64 operator"" _usec(long double us)
{
	return us * 1000_nsec;
}

// ミリ秒
constexpr uint64 operator"" _msec(unsigned long long ms)
{
	return ms * 1000_usec;
}
constexpr uint64 operator"" _msec(long double ms)
{
	return ms * 1000_usec;
}

// 秒
constexpr uint64 operator"" _sec(unsigned long long sec)
{
	return sec * 1000_msec;
}
constexpr uint64 operator"" _sec(long double sec)
{
	return sec * 1000_msec;
}
