//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// SRAM
//

#pragma once

#include "device.h"
#include "mappedfile.h"
#include <array>

class SRAMDevice : public IODevice
{
	using inherited = IODevice;

	static const uint32 baseaddr = 0xed0000;

	// サイズは (今のところ?) 16KB 固定。
	static const uint sramsize = 16 * 1024;

	// Pluto-X によるホストファイル起動アドレス
	static const uint32 plutoaddr = 0xeac000;

 public:
	SRAMDevice();
	~SRAMDevice() override;

	bool Init() override;
	void ResetHard(bool poweron) override;

	busdata Read(busaddr addr) override;
	busdata Write(busaddr addr, uint32 data) override;
	busdata Peek1(uint32 addr) override;
	bool Poke1(uint32 addr, uint32 data) override;

	void WriteEnable(bool value);
	bool IsWriteable() const { return writeable; }

	// SRAM $ed0008.L RAM 容量を返す (ホストファイル起動用)
	uint32 GetRAMSize() const { return Get32(0x0008); }

	// SRAM $ed000c.L ROM 起動アドレスを返す (romemu 用)
	uint32 GetROMAddr() const { return Get32(0x000c); }

 private:
	// SRAM の指定の位置の値を副作用なく読み出す
	uint32 Get16(uint32 offset) const;
	uint32 Get32(uint32 offset) const;

	// RAM 容量欄を設定値に同期させる。
	void SyncRAMSize();

	// アドレスデコーダ
	uint32 Decoder(uint32 addr) const;

	// ファイル
	std::string filename {};
	std::unique_ptr<MappedFile> file {};
	uint8 *mem {};

	bool writeable {};

	// SRAM の初期値
	static std::array<uint8, 0x58> initialdata;
};

inline SRAMDevice *GetSRAMDevice() {
	return Object::GetObject<SRAMDevice>(OBJ_SRAM);
}
