//
// nono
// Copyright (C) 2025 nono project
// Licensed under nono-license.txt
//

//
// ページテーブルモニター
//

#pragma once

#include "wxsubwindow.h"
#include <array>

class MainbusDevice;
class MPU68030Device;
class MPU68040Device;
class WXScrollBar;
class WXTextPanel;

// ページテーブル1段分の構造
class PageTable
{
 public:
	PageTable(wxWindow *parent, int lv_, int width);

	// パラメータをクリアする。
	void Clear();

	// 指定位置のアドレスを返す。
	uint32 GetAddr(int) const;
	// カーソル位置のアドレスを返す。
	uint32 GetAddr() const { return GetAddr(cursor); }

	// 表示開始位置を返す。
	int ViewTop() const;

	// コントロールの表示状態を制御する。
	void Show(bool enable = true);
	void Hide() {
		Show(false);
	}

	WXTextScreen *screen {};
	WXScrollBar *vscroll {};
	int lv {};			// このテーブルの段数 (TIA なら 0)

	// このテーブルの見出し (1行目)
	wxString title {};

	// このテーブルのヘッダ (表の <TH>)
	wxString head {};

	// このテーブルのベースアドレス。0xffffffff なら無効。
	uint32 base {};
	// この段のアドレスマスク。
	uint32 mask {};

	// この段のビット位置と長さ。ビット位置は下から数える。
	// TIA が 10 ビットなら bitlen=10, bitpos=22 (= 32 - 10)。
	int bitpos {};
	int bitlen {};

	// この段のエントリ数。
	int count {};
	// カーソル位置。
	int cursor {};
};

class WXPageTableWindow : public WXSubWindow
{
	using inherited = WXSubWindow;

 public:
	WXPageTableWindow(wxWindow *parent, const wxString& name);
	~WXPageTableWindow() override;

	bool GetMySizeHints(wxSize *, wxSize *, wxSize *, wxSize *) override;
	bool Layout() override;

 protected:
	void ConnectTableEvents();

	void OnTimer(wxTimerEvent&);
	void OnRPChanged(wxCommandEvent&);
	void OnLeftDClick(wxMouseEvent&);
	void OnMouseWheel(wxMouseEvent&);

	void SetRate(int hz);

	virtual void Reconfig(uint32) = 0;
	virtual void DoRefresh() = 0;

	// 全段の最大の行数
	int max_rows {};

	bool show_urp {};

	wxPanel *ctrlpanel {};
	wxRadioButton *radio_srp {};
	wxRadioButton *radio_urp {};
	WXTextScreen *stat {};
	WXTextPanel *bgpanel {};		// テーブル領域全部の下敷き。
	std::vector<PageTable> table {};

	// 現在有効なテーブル数。Reconfig() で決まる。
	// table の要素を増減させるのは大変なので。
	int table_count {};

	// 現在の TC の値。
	// 68030 は 32bit、68040 は 16bit というところだけ違う。
	uint32 tc_value {};

	enum {
		TC_ENABLE = 0,	// TC.E が有効
		TC_DISABLE,		// 構成可能で TC.E が無効
		TC_NOT_CONFIG,	// 構成不能 (68030)
	} tc_status {};

	wxTimer timer {};

	MainbusDevice *mainbus {};

	wxDECLARE_EVENT_TABLE();
};

class WXPageTable68030Window : public WXPageTableWindow
{
	using inherited = WXPageTableWindow;
 public:
	WXPageTable68030Window(wxWindow *, const wxString& name);
	~WXPageTable68030Window() override;

	void Reconfig(uint32) override;
	void DoRefresh() override;

 private:
	MPU68030Device *mpu {};
};

class WXPageTable68040Window : public WXPageTableWindow
{
	using inherited = WXPageTableWindow;
 public:
	WXPageTable68040Window(wxWindow *, const wxString& name);
	~WXPageTable68040Window() override;

	void Reconfig(uint32) override;
	void DoRefresh() override;

 private:
	bool ps4k;

	MPU68040Device *mpu {};
};
