package vulns_test

import (
	"testing"
	"time"

	"github.com/google/osv-scanner/internal/utility/vulns"
	"github.com/google/osv-scanner/pkg/lockfile"
	"github.com/google/osv-scanner/pkg/models"
)

func expectIsAffected(t *testing.T, vuln models.Vulnerability, version string, expectAffected bool) {
	t.Helper()

	pkg := lockfile.PackageDetails{
		Name:      "my-package",
		Version:   version,
		Ecosystem: lockfile.NpmEcosystem,
		CompareAs: lockfile.NpmEcosystem,
	}

	if vulns.IsAffected(vuln, pkg) != expectAffected {
		if expectAffected {
			t.Errorf("Expected OSV to affect package version %s but it did not", version)
		} else {
			t.Errorf("Expected OSV not to affect package version %s but it did", version)
		}
	}
}

func buildOSVWithAffected(affected ...models.Affected) models.Vulnerability {
	return models.Vulnerability{
		ID:        "1",
		Published: time.Time{},
		Modified:  time.Time{},
		Details:   "This is an open source vulnerability!",
		Affected:  affected,
	}
}

func buildEcosystemAffectsRange(events ...models.Event) models.Range {
	return models.Range{Type: models.RangeEcosystem, Events: events}
}

func buildSemverAffectsRange(events ...models.Event) models.Range {
	return models.Range{Type: models.RangeSemVer, Events: events}
}

func TestOSV_AffectsEcosystem(t *testing.T) {
	t.Parallel()

	type AffectsTest struct {
		Affected  []models.Affected
		Ecosystem lockfile.Ecosystem
		Expected  bool
	}

	tests := []AffectsTest{
		{Affected: nil, Ecosystem: "Go", Expected: false},
		{Affected: nil, Ecosystem: "npm", Expected: false},
		{Affected: nil, Ecosystem: "PyPI", Expected: false},
		{Affected: nil, Ecosystem: "", Expected: false},
		{
			Affected: []models.Affected{
				{Package: models.Package{Ecosystem: "crates.io"}},
				{Package: models.Package{Ecosystem: "npm"}},
				{Package: models.Package{Ecosystem: "PyPI"}},
			},
			Ecosystem: "Packagist",
			Expected:  false,
		},
		{
			Affected: []models.Affected{
				{Package: models.Package{Ecosystem: "NuGet"}},
			},
			Ecosystem: "NuGet",
			Expected:  true,
		},
		{
			Affected: []models.Affected{
				{Package: models.Package{Ecosystem: "npm"}},
				{Package: models.Package{Ecosystem: "npm"}},
			},
			Ecosystem: "npm",
			Expected:  true,
		},
	}

	for i, test := range tests {
		vuln := models.Vulnerability{
			ID:        "1",
			Published: time.Time{},
			Modified:  time.Time{},
			Details:   "This is an open source vulnerability!",
			Affected:  test.Affected,
		}

		if vulns.AffectsEcosystem(vuln, test.Ecosystem) != test.Expected {
			t.Errorf(
				"Test #%d: Expected OSV to return %t but it returned %t",
				i,
				test.Expected,
				!test.Expected,
			)
		}
	}

	// test when the OSV doesn't have an "Affected"
	vuln := models.Vulnerability{
		ID:        "1",
		Published: time.Time{},
		Modified:  time.Time{},
		Details:   "This is an open source vulnerability!",
		Affected:  nil,
	}

	if vulns.AffectsEcosystem(vuln, "npm") {
		t.Errorf(
			"Expected OSV to report 'false' when it doesn't have an Affected, but it reported true!",
		)
	}
}

func TestOSV_IsAffected_AffectsWithEcosystem_DifferentEcosystem(t *testing.T) {
	t.Parallel()

	vuln := buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemPyPI, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(models.Event{Introduced: "0"}),
			},
		},
	)

	for _, v := range []string{"1.0.0", "1.1.1", "2.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}
}

func TestOSV_IsAffected_AffectsWithEcosystem_SingleAffected(t *testing.T) {
	t.Parallel()

	var vuln models.Vulnerability

	// "Introduced: 0" means everything is affected
	vuln = buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(models.Event{Introduced: "0"}),
			},
		},
	)

	for _, v := range []string{"1.0.0", "1.1.1", "2.0.0"} {
		expectIsAffected(t, vuln, v, true)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)

	// "Fixed: 1" means all versions after this are not vulnerable
	vuln = buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{Fixed: "1"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.0", "1.1.0", "2.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)

	// multiple fixes and introduced
	vuln = buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{Fixed: "1"},
					models.Event{Introduced: "2.1.0"},
					models.Event{Fixed: "3.2.0"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.0", "1.1.0", "2.0.0rc2", "2.0.1"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"2.1.1", "2.3.4", "3.0.0", "3.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"3.2.0", "3.2.1", "4.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)

	// "LastAffected: 1" means all versions after this are not vulnerable
	vuln = buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{LastAffected: "1"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc", "1.0.0"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.1", "1.1.0", "2.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)

	// mix of fixes, last_known_affected, and introduced
	vuln = buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{Fixed: "1"},
					models.Event{Introduced: "2.1.0"},
					models.Event{LastAffected: "3.1.9"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.0", "1.1.0", "2.0.0rc2", "2.0.1"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"2.1.1", "2.3.4", "3.0.0", "3.0.0-rc", "3.1.9"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"3.2.0", "3.2.1", "4.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)
}

func TestOSV_IsAffected_AffectsWithEcosystem_MultipleAffected(t *testing.T) {
	t.Parallel()

	vuln := buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{Fixed: "1"},
				),
			},
		},
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(
					models.Event{Introduced: "2.1.0"},
					models.Event{Fixed: "3.2.0"},
				),
			},
		},
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(
					models.Event{Introduced: "3.3.0"},
					models.Event{LastAffected: "3.5.0"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.0", "1.1.0", "2.0.0rc2", "2.0.1"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"2.1.1", "2.3.4", "3.0.0", "3.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"3.2.0", "3.2.1", "4.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"3.3.1", "3.4.5"} {
		expectIsAffected(t, vuln, v, true)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)
}

func TestOSV_IsAffected_AffectsWithEcosystem_Unsorted(t *testing.T) {
	t.Parallel()

	vuln := buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{Introduced: "2.1.0"},
					models.Event{Fixed: "1"},
					models.Event{LastAffected: "3.1.9"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.0", "1.1.0", "2.0.0rc2", "2.0.1"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"2.1.1", "2.3.4", "3.0.0", "3.0.0-rc", "3.1.9"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"3.2.0", "3.2.1", "4.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)

	// zeros with build strings
	vuln = buildOSVWithAffected(
		models.Affected{
			// golang.org/x/sys
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(
					models.Event{Fixed: "0.0.0-20220412211240-33da011f77ad"},
					models.Event{Introduced: "0"},
				),
			},
		},
		models.Affected{
			// golang.org/x/net
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildEcosystemAffectsRange(
					models.Event{Introduced: "0.0.0-20180925071336-cf3bd585ca2a"},
					models.Event{Fixed: "0"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.14.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"0.0.0-20180925071336-cf3bd585ca2a"} {
		expectIsAffected(t, vuln, v, true)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)
}

func TestOSV_IsAffected_AffectsWithSemver_DifferentEcosystem(t *testing.T) {
	t.Parallel()

	vuln := buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemPyPI, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(models.Event{Introduced: "0"}),
			},
		},
	)

	for _, v := range []string{"1.0.0", "1.1.1", "2.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}
}

func TestOSV_IsAffected_AffectsWithSemver_SingleAffected(t *testing.T) {
	t.Parallel()

	var vuln models.Vulnerability

	// "Introduced: 0" means everything is affected
	vuln = buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(models.Event{Introduced: "0"}),
			},
		},
	)

	for _, v := range []string{"v1.0.0", "v1.1.1", "v2.0.0"} {
		expectIsAffected(t, vuln, v, true)
	}

	// "Fixed: 1" means all versions after this are not vulnerable
	vuln = buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{Fixed: "1.0.0"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.0", "1.1.0", "2.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	// multiple fixes and introduced
	vuln = buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{Fixed: "1"},
					models.Event{Introduced: "2.1.0"},
					models.Event{Fixed: "3.2.0"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.0", "1.1.0", "2.0.0rc2", "2.0.1"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"2.1.1", "2.3.4", "3.0.0", "3.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"3.2.0", "3.2.1", "4.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)

	// "LastAffected: 1" means all versions after this are not vulnerable
	vuln = buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{LastAffected: "1.0.0"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc", "1.0.0"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.1", "1.1.0", "2.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	// mix of fixes, last_known_affected, and introduced
	vuln = buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{Fixed: "1"},
					models.Event{Introduced: "2.1.0"},
					models.Event{LastAffected: "3.1.9"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.0", "1.1.0", "2.0.0rc2", "2.0.1"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"2.1.1", "2.3.4", "3.0.0", "3.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"3.2.0", "3.2.1", "4.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)
}

func TestOSV_IsAffected_AffectsWithSemver_MultipleAffected(t *testing.T) {
	t.Parallel()

	vuln := buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{Fixed: "1"},
				),
			},
		},
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(
					models.Event{Introduced: "2.1.0"},
					models.Event{Fixed: "3.2.0"},
				),
			},
		},
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(
					models.Event{Introduced: "3.3.0"},
					models.Event{LastAffected: "3.5.0"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.0", "1.1.0", "2.0.0rc2", "2.0.1"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"2.1.1", "2.3.4", "3.0.0", "3.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"3.2.0", "3.2.1", "4.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"3.3.1", "3.4.5", "3.5.0"} {
		expectIsAffected(t, vuln, v, true)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)
}

func TestOSV_IsAffected_AffectsWithSemver_Unsorted(t *testing.T) {
	t.Parallel()

	// mix of fixes, last_known_affected, and introduced
	vuln := buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(
					models.Event{Introduced: "0"},
					models.Event{Introduced: "2.1.0"},
					models.Event{Fixed: "1"},
					models.Event{LastAffected: "3.1.9"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.1.0", "0.0.0.1", "1.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"1.0.0", "1.1.0", "2.0.0rc2", "2.0.1"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"2.1.1", "2.3.4", "3.0.0", "3.0.0-rc"} {
		expectIsAffected(t, vuln, v, true)
	}

	for _, v := range []string{"3.2.0", "3.2.1", "4.0.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)

	// zeros with build strings
	vuln = buildOSVWithAffected(
		models.Affected{
			// golang.org/x/sys
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(
					models.Event{Fixed: "0.0.0-20220412211240-33da011f77ad"},
					models.Event{Introduced: "0"},
				),
			},
		},
		models.Affected{
			// golang.org/x/net
			Package: models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(
					models.Event{Introduced: "0.0.0-20180925071336-cf3bd585ca2a"},
					models.Event{Fixed: "0"},
				),
			},
		},
	)

	for _, v := range []string{"0.0.0", "0.14.0"} {
		expectIsAffected(t, vuln, v, false)
	}

	for _, v := range []string{"0.0.0-20180925071336-cf3bd585ca2a"} {
		expectIsAffected(t, vuln, v, true)
	}

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)
}

func TestOSV_IsAffected_OnlyVersions(t *testing.T) {
	t.Parallel()

	vuln := buildOSVWithAffected(
		models.Affected{
			Package:  models.Package{Ecosystem: models.EcosystemNPM, Name: "my-package"},
			Versions: []string{"1.0.0"},
		},
	)

	expectIsAffected(t, vuln, "0.0.0", false)
	expectIsAffected(t, vuln, "1.0.0", true)
	expectIsAffected(t, vuln, "1.0.0-beta1", false)
	expectIsAffected(t, vuln, "1.1.0", false)

	// an empty version should always be treated as affected
	expectIsAffected(t, vuln, "", true)
}

func TestOSV_EcosystemsWithSuffix(t *testing.T) {
	t.Parallel()

	vuln := buildOSVWithAffected(
		models.Affected{
			Package: models.Package{Ecosystem: "Debian:12", Name: "my-package"},
			Ranges: []models.Range{
				buildSemverAffectsRange(
					models.Event{Introduced: "0"},
				),
			},
		},
	)

	pkg := lockfile.PackageDetails{
		Name:      "my-package",
		Version:   "0.0.0",
		Ecosystem: "Debian:12",
		CompareAs: "Debian:12",
	}

	if !vulns.IsAffected(vuln, pkg) {
		t.Errorf("Expected OSV to affect package version %s but it did not", "0.0.0")
	}
}
