#!/usr/bin/env python3
# Gimp3 plugins have a shebang


'''
A script to test the resynthesizer.

Lloyd Konneker

Suite of several test cases.
Most test cases are the outer plugins that call the resynthesizer engine plugin.
A few are direct calls to the outer resynthesizer engine plugin.
No tests of the outer, control panel plugin.
Note the plugins must be installed, this calls them through Gimp.

Uses diff to compare output image with known good results (reference images).
Output is to a temp directory.

For comparisons to succeed:
   must be built non-threaded
   PRNG seed must be the same as previously
   may need EXPORT BABL_TOLERANCE=0.0 so floating point by BABL is more exact?
   also might depend on platform and build
   also might depend on version of GIMP: 2 gives different results from 3

Also, the context should be the default
especially the background color white.

The test program leaves the result images visible.
You can do a human comparison for perceptual identical.
You must allow for subjective interpretation: the results are plausible if not the same.
You must know at what each test is supposed to do, from a high level outlook.

Result type enumeration:
FAILED: output does not match reference image
SUCCESS: does match
UNKNOWN: no reference image exists, one was created, manually judge the reference and run testsuite again
EXCEPTION: a called plugin threw a RuntimeError exception
IMPROPER: a test file could not be opened, or similar

Reference images are created on the first run.
Delete a reference image and run testsuite again to create a new reference image.
OR review the output in temp and when OK, paste it over the reference image.

Note these call a version of the resynthesizer that is repeatable especially its random seed.
Note that anytime you change the code
so that the prng pseudo random number generator sequence is changed
this automated test will usually fail but the functional result will be as expected.

The test harness is itself a plugin.
Invoke: Test>Resynthesizer

It takes several minutes.
It creates and displays result images.
Then review the log, where a summary is printed last.  Everything should pass.

Note save PPM because PPM format is large but repeatable,
whereas PNG is not repeatable because of timestamp.

Note that the test harness creates a new image from a file and
saves the result separately from the input.

Note uses the logging module, mainly to avoid issues with print to stdout on some platforms.
when you want to see the results in real time, use 'tail -f resynth-test.log'
'''

# Since Gimp3, use gi
import gi
gi.require_version('Gimp', '3.0')
from gi.repository import Gimp
gi.require_version('GimpUi', '3.0')
from gi.repository import GimpUi
from gi.repository import GObject
from gi.repository import GLib
from gi.repository import Gio



import sys
import os
import subprocess
import time
import uuid
import logging



'''
paths
'''

'''
testdir is *not* the working dir

testdir contains in images, and reference out images

use absolute path to the test dir of the repo
'''
testdir = "/work/resynthesizer/test/"


'''
Put test results in their own directory.
The results are not temporary, but must be deleted manually.

The directory name is unique.

Names of the files are generated by the test, not random, temporary names.
'''
tmpName = "testResynth" + str(uuid.uuid4())
tmpDirName = "/work/" + tmpName


'''
Not called at top, only from the run func of the plugin.
'''
def makeTempDir():
  os.mkdir(tmpDirName)


'''
We don't use tempfile module, because it deletes the files.

import tempfile.
tmpdir = tempfile.TemporaryDirectory(delete=False)
tmpDirName = tmpdir.name
'''

# test images must be in this subdir of the working dir
indir = testdir + 'in_images/'


'''
Return a string for path to reference dir.
Ends with /

The reference images for GIMP 2 and GIMP 3 are different.
The reference files are differ by PRNG pseudo random number generator.
The reference files don't work except for a non-threaded build.
Uncomment one of these lines.

Since GIMP 3, most comparisons to reference files from GIMP 2 fail,
without extreme care to ensure sameness.
GIMP 3 support of colorspaces and color models
means subtle difference in algorithms.

The invariant is:
if you create the reference files,
then retest, all tests should pass,
but only on  a same reproducible build for the same host computer/container.
(Same versions of all tools and dependency libraries.)

You must create the directory before testing.
The directory is under version control, in the repo.
A testing session will create reference images if they don't exist.
A testing session does not overwrite reference images with new versions.
'''
def referenceDir():

  # Testing GIMP 2 using glib with GRand
  #return testdir + 'reference_out_images/'

  # Testing GIMP3 with glib GRand
  return testdir + 'reference_out_images_3/'

  # Testing GIMP 2 with stdc rand():
  ## return testdir + 'reference_out_images_rand/'



'''
Other globals
'''

test_summary = ""

# selection rects
select1 = (100, 90, 100, 50)  # ufo
select2 = (90, 175, 135, 100) # donkey
selectUfoSmall = (100, 14, 100, 50)  # ufo from ufo-small

# Names of input files
# They are all in the input directory
# They all have suffix .png
# This abstraction lets us change the input files later,
# and lets us use shorter text in the test program.
grass =      'grass-input'
grassAlpha = 'grass-input-alpha'
brick  =     'brick'
donkey  =    'donkey_original'
# Not image of angel, but another subject, with colors
angelTexture = 'angel_texture'
# Image of angel
angelTarget =  'angel_target'
wander  =      'wander-texture'
zap  =         'zap-texture'
zapOutputMap = 'zap-output-map'
ufo  =         'ufo-input'
ufoAlpha =     'ufo-input-w-alpha'
ufoAlphaGray = 'ufo-input-w-alpha-gray'
ufoSmall =                'ufo-input-small'
ufoSmallIndexedSepia=     'ufo-small-indexed-sepia'
ufoSmallHighBitDepth=     'ufo-small-high-bit-depth'
wilber =                  'wilber-small'
wilberSmallIndexedAlpha = 'wilber-small-indexed-alpha'



'''
Other test data.
We need a GimpPattern.
# !!! You can't call Gimp until after Gimp.main()
so this is initialized with memoization later.
'''
testPattern = None

def initTestPattern():
  global testPattern

  # Once, when not memoized, it crashes after four calls??? TODO
  if testPattern is None:
    # pattern must be an object, not a string name.
    testPattern = Gimp.Resource.get_by_name(Gimp.Pattern, 'Maple Leaves')
    print (testPattern.get_name())


def record_test_result(testname, result, outfilename = ""):
  global test_summary
  logging.info(result + " " + testname)
  test_summary += result + " " + testname + " " + outfilename + "\n"



'''
Opens test input image file at name.

This knows the path to in test files, and their suffix.

Returns image, drawables.
Returns list of drawables.
'''
def openTestFilename(filename, select=None, invert=False):
  infilepath = indir + filename + '.png'
  return openTestFilepath (infilepath, select, invert)



'''
Opens test image file.
Optionally selects area in the image.
Optionally inverts the selection.
Returns image, drawables.
'''
def openTestFilepath(filepath, select=None, invert=False):

  # v3 file load takes a GFile
  infile = Gio.file_new_for_path(filepath)
  image = Gimp.file_load(Gimp.RunMode.NONINTERACTIVE, infile)
  assert image is not None

  # require each test image have exactly one image, need not be chosen
  # TODO get_selected_layers and get_layers are not in the GI, marked in image.pdb to $skip-GI ?
  # count, layers = image.get_layers()
  layers = image.get_layers()
  # layers is a GimpCoreObjectArray, a list in Python
  assert len(layers) == 1
  drawables = layers
  assert drawables is not None

  # TODO optionally flatten
  # i.e. remove all layers but one
  # TODO does flatten remove the alpha?

  if select is not None:
    # Make selection    x,y, width, height
    # Gimp.rect_select(image, 100, 90, 100, 50, 0, False, 0)
    image.select_rectangle(Gimp.ChannelOps.ADD, select[0], select[1], select[2], select[3])
  if invert:
    Gimp.Selection.invert(image)

  return image, drawables


'''
Wrapper functions.

Since v3 a call to the PDB is more complicated than in v2 GimpFu
In v2 you could directly call "pdb.plug-in-foo"
Now the test harness calls the wrapper instead of directly.

Since GIMP v3, a plugin takes an array of drawables,
binding from a Python list.
So we wrap a single drawable => [drawable]

No resynthesizer plugin returns a meaningful value
They return just a "status" of the call, which is not useful.
The outer plugins have side effects only:
may affect the target image, or create a new image.
'''


'''
Special

Lists won't bind to the C type GimpCoreObjectArray.
A deficiency of PyGObject?
So there is a special function for this in the GIMP API.
Otherwise, would just use: config.set_property('drawables', targetDrawables)

Assert value is-a Python list
'''
def setConfigObjectArrayProperty (config, property_name, value):
   config.set_core_object_array(property_name, value)

'''
When config is for a filter plugin, it takes usual arguments.
'''
def setStandardArgsToConfig (config, image, drawables):
  config.set_property('run-mode', Gimp.RunMode.NONINTERACTIVE)
  config.set_property('image', image)
  setConfigObjectArrayProperty (config, 'drawables', drawables)


'''
Get the PDB procedure by name.
Raise when not found.
This lets testing continue despite some plugins fail to register in PDB.
'''
def findTestedPluginProcedure(name):
  result = Gimp.get_pdb().lookup_procedure(name)
  if result is None:
    raise RuntimeError
  return result


def callHealSelection(targetImage, targetDrawables, corpusWidth, sampleFrom, synthDirection):
  pdb_proc   = findTestedPluginProcedure('plug-in-heal-selection')
  config = pdb_proc.create_config()
  setStandardArgsToConfig (config, targetImage, targetDrawables)
  config.set_property('adjustment', corpusWidth)
  config.set_property('option', sampleFrom)
  config.set_property('option-2', synthDirection)
  result = pdb_proc.run(config)
  # Side effect: the target image is modified
  # The plugin (unless it is in debug mode) has deleted any temporary images

  # display the result image
  Gimp.Display.new(targetImage)
  

def callHealTransparency(targetImage, targetDrawables, corpusWidth, synthDirection):
  pdb_proc   = findTestedPluginProcedure('plug-in-heal-transparency')
  config = pdb_proc.create_config()
  setStandardArgsToConfig (config, targetImage, targetDrawables)
  config.set_property('adjustment', corpusWidth)
  # No sampleFrom, the target and corpus are not compact
  # but usually spread over the entire image.
  config.set_property('option', synthDirection)
  result = pdb_proc.run(config)
  # Side effect: the target image is modified

  # display the result image
  Gimp.Display.new(targetImage)

def callUncrop(targetImage, targetDrawables, percentSizeIncrease, shouldAntierase):
  pdb_proc   = findTestedPluginProcedure('plug-in-uncrop')
  config = pdb_proc.create_config()
  setStandardArgsToConfig (config, targetImage, targetDrawables)
  # No sampleFrom, corpus is always just inside the image edge
  # No corpusWidth corpus is always just tens of pixels deep
  # No direction, always outward
  config.set_property('adjustment', percentSizeIncrease)
  config.set_property('toggle',     shouldAntierase)
  result = pdb_proc.run(config)
  # Side effect: the target image is modified

  # display the result image
  Gimp.Display.new(targetImage)


def callFillPatternResynth(targetImage, targetDrawables, pattern):
  pdb_proc   = findTestedPluginProcedure('plug-in-fill-pattern-resynth')
  config = pdb_proc.create_config()
  setStandardArgsToConfig (config, targetImage, targetDrawables)
  config.set_property('pattern', pattern)
  result = pdb_proc.run(config)
  # Side effect: targetImage is modified

  # display the result image
  Gimp.Display.new(targetImage)


def callRenderTexture(targetImage, targetDrawables, sizeRatio, isTileable):
  pdb_proc   = findTestedPluginProcedure('plug-in-render-texture')
  config = pdb_proc.create_config()
  setStandardArgsToConfig (config, targetImage, targetDrawables)
  config.set_property('adjustment', sizeRatio)
  config.set_property('toggle', isTileable)
  result = pdb_proc.run(config)
  # Side effect: plugin created new image and displayed it
  # The plugin has already displayed the result image

  # clean up
  targetImage.delete()



def callMapStyle(targetImage, targetDrawables, corpusFilename, percentTransfer=50, mapBy=0):

  # open the corpus
  # open with no selection and no inversion of selection
  corpusImage, corpusDrawables = openTestFilename (corpusFilename)
  # single drawable from a drawables list
  corpusDrawable = corpusDrawables[0]

  pdb_proc   = findTestedPluginProcedure('plug-in-map-style')
  config = pdb_proc.create_config()
  setStandardArgsToConfig (config, targetImage, targetDrawables)
  # Arg for target is standard, named "drawables", arg for corpus is just "drawable"
  # In general, when there are two args of same type, the second has a numeric suffix like "-2"
  config.set_property('drawable', corpusDrawable)
  config.set_property('adjustment', percentTransfer)
  config.set_property('option', mapBy)
  result = pdb_proc.run(config)
  # Side effect: the target image is modified

  '''
  clean up images created by test harness.
  The tested plugin created and should have already delete two map images.
  '''
  corpusImage.delete()

  # display the result image
  Gimp.Display.new(targetImage)

  


def callSharpen(targetImage, targetDrawables, factor=1.0):
  pdb_proc   = findTestedPluginProcedure('plug-in-sharpen-resynthesized')
  config = pdb_proc.create_config()
  setStandardArgsToConfig (config, targetImage, targetDrawables)
  config.set_property('adjustment', factor)
  result = pdb_proc.run(config)
  # Side effect: the target image is modified

  # display the result image
  Gimp.Display.new(targetImage)


def callEnlarge(targetImage, targetDrawables, factor=1.0):
  pdb_proc   = findTestedPluginProcedure('plug-in-enlarge-resynthesized')
  config = pdb_proc.create_config()
  setStandardArgsToConfig (config, targetImage, targetDrawables)
  config.set_property('adjustment', factor)
  result = pdb_proc.run(config)
  # Side effect: the target image is modified

  # display the result image
  Gimp.Display.new(targetImage)

'''
Note passing a single drawable.
'''
def callResynthesizer(
    targetImage,
    # !!! The signature of a callFoo method takes a list of drawables.
    targetDrawables,
    # resynth engine plugin requires corpus drawable
    corpusFilename,
    # These are defaults for testing.
    # The plugin defaults are not effective,
    # i.e. they are for a GUI dialog and otherwise don't contribute

    # a group, usually both the same
    hTile          = False,
    vTile          = False,

    # TODO encoded
    useBorder      = 1,

    # A group, either all nil or all non-nil
    input_map      = None,
    output_map     = None,
    map_weight     = 0.0,

    # AKA tweaks, basic engine params
    autism         = 0.117,
    neighbours     = 16,
    trys           = 500,

    # affect the corpus
    select         = None,
    invert         = False
    ):

  # open the corpus
  corpusImage, corpusDrawables = openTestFilename (corpusFilename, select=select, invert=invert)
  # drawable_of_file_with_anti_selection('"+ zappath + "', select1)


  pdb_proc   = findTestedPluginProcedure('plug-in-resynthesizer')
  config = pdb_proc.create_config()
  # Takes a single drawable, not many.
  # Pass the first of a list.
  config.set_property('drawable', targetDrawables[0])
  config.set_property('h-tile', hTile)
  config.set_property('v-tile', vTile),
  config.set_property('use-border', useBorder),
  # Again, pass first of list of drawables
  config.set_property('corpus-drawable', corpusDrawables[0]),
  config.set_property('input-map', input_map),
  config.set_property('output-map', output_map),
  config.set_property('map-weight', map_weight),
  config.set_property('autism', autism),
  config.set_property('neighbours', neighbours),
  config.set_property('trys', trys),
  result = pdb_proc.run(config)
  # Side effect: the target image is modified

  # display the result image
  Gimp.Display.new(targetImage)

  '''
  clean up images created by test harness.
  The tested plugin does not create any new image, only modifies target.
  '''
  corpusImage.delete()



'''
Export image to a .ppm file.

Formerly called Gimp.file_save
but that now seems to export in PPM "raw" P6 format.
We want the "ascii" P2 (gray) or P3 (RGB) format, 
for backward compatibility with existing reference images.
'''
def exportPPMImage (image, outfilepath):

  # Require image is flattened, only one layer
  # since PPM format does not handle transparency

  # need a GFile, from string
  outfile = Gio.file_new_for_path(outfilepath)

  '''
  Prior code:
  Gimp.file_save(
    Gimp.RunMode.NONINTERACTIVE, 
    image, 
    outfile,
    None)   # export options
  '''
  
  '''
  Call a specific file export plugin.
  Since GIMP 3, it has "raw" option, we want "ascii" option.
  '''
  exportProc   = findTestedPluginProcedure('file-pnm-export')
  config = exportProc.create_config()
  config.set_property('run-mode', Gimp.RunMode.NONINTERACTIVE)
  config.set_property('image',    image)
  config.set_property('file',     outfile)
  config.set_property('options',  None)
  config.set_property('raw',      "ascii-output")  # !!! Want P2 or P3, plain format
  exportProc.run(config)



'''
About the diffing algorithms.

Images are in ppm (color) or pgm (gray) format, which can be diffed.

The ppm format is also stable, and should not change.

Note however, the file can have a comment "# CREATOR" or "# Created by"
that can change when the GIMP file save plugin changes version.
'''

'''
diff two PPM files, 
returning boolean indicating similarity.

PPM is ASCII, not UTF-8

This just chooses a similarity measure.
'''
def diffPPM(file1, file2):
  print("Comparing " + file1 + " and " + file2)
  # return diffPPMSmallDiff (file1, file2)
  return diffPPMAbsDiff (file1, file2)
  

'''
Allowing small Absolute Difference over entire image

Absolute difference is the sum of the absolute differences of each pixel.
'''
def diffPPMAbsDiff (file1, file2):
  absDiff = 0
  try:
    with open(file1, 'r', encoding='ascii', errors='replace') as fileFirst:
      linesFirst = fileFirst.readlines()
      with open(file2, 'r', encoding='ascii', errors='replace') as fileSecond:
        linesSecond = fileSecond.readlines()
        # Compare the two files line by line
        for index, line in enumerate(linesFirst):
          # Ignore the first four lines (header)
          if index in [0, 1, 2, 3]:
            continue
          # Each line is a pixelel
          # Sum the absolute differences
          absDiff += abs(int(line.split()[0]) - int(linesSecond[index].split()[0]))
  except Exception as inst:
    print(f"Error reading files {inst}")
    return False
  
  logging.info("Total absolute difference: " + str(absDiff))

  # If the abs difference is greater than 1000, return False
  # This is a magic number, but it is not too strict
  if absDiff > 10000:
    print("Difference found: " + str(absDiff))
    return False
  return True

'''
Allowing small difference of pixelels, say 2
'''
def diffPPMSmallDiff (file1, file2):
  try:
    with open(file1, 'r', encoding='ascii', errors='replace') as fileFirst:
      linesFirst = fileFirst.readlines()
      with open(file2, 'r', encoding='ascii', errors='replace') as fileSecond:
        linesSecond = fileSecond.readlines()
        # Compare the two files line by line
        for index, line in enumerate(linesFirst):
          # Ignore the first four lines (header)
          if index in [0, 1, 2, 3]:
            continue
          # Compare the lines, allowing for one pixelel difference
          # Each line is a pixelel
          if line != linesSecond[index]:
            # Allow for two pixelel difference
            if abs(int(line.split()[0]) - int(linesSecond[index].split()[0])) > 2:
              print(f"Difference found at line {index} {line} {linesSecond[index]}")
              return False
  except:
    print("Error reading files")
    return False
  return True


'''
Write an out test result image to a temp file in .ppm format

And name the image in the GIMP app (instead of untitled.)
'''
def exportImage (image, outfilepath):

  '''
  Since the file type is .ppm, this is an export.
  The displayed image remains untitled, in .xcf format,
  with a larger file size including the undo steps.
  '''
  exportPPMImage (image, outfilepath)
 
  '''
  Set the filename so we know what test the viewed image is for.
  Note the suffix must be .xcf, but the image is still dirty
  and that file does not exist.
  The display image should show same view as the exported file.
  '''
  # Just append ".xcf"
  image.set_file(
    Gio.file_new_for_path(
      outfilepath + ".xcf")
    )

  '''
  Alternatively, close the image and open the exported file,
  returning the new image reference.
  resultImage =

  TODO some plugins do not alter the targetImage but return a new image.
  We should always delete the targetImage and display the result image.
  The selection in the result image won't show the selection we started with?
  '''
  return image



'''
Find the unnamed image in the GIMP app.
This is the image that is the result of the test.
It is the only image without a display and without a name (is untitled.)
The plugin has side effects, creating a new image or modifying the target image.,
and does not return which image is the result.

When there is more than one unnamed image, declare an error:
the tested plugin should have cleaned up its working images
and the test harness should have deleted any input images.
'''
def findUnnamedImage():
  result = None
  images = Gimp.get_images()
  for image in images:
    # find the unsaved image
    print (image.get_name())
    if image.get_name().startswith("[Untitled]"):
      # check if it is the only unsaved image
      if result is not None:
        raise RuntimeError("More than one unsaved image found")
      result = image

  # No unsaved image found
  if result is not None:
    # one unsaved image found
    return result
  else:
   raise RuntimeError("No unsaved image found")



'''
Post process the result image.
It is the only image without a name (is untitled.)
The passed image is the target image: it may be the result image, but not always.

Some plugins have displayed, but not named, the result image. 

flatten and save to .ppm format
'''
def postProcessOutImage (image, outfilename, outfilepath):

  resultImage = findUnnamedImage()

  # flatten always, even if image might already be flat (no alpha and one layer)
  resultImage.flatten()

  resultImage = exportImage (resultImage, outfilepath)

  # Log actual exported filepath
  # OLD outfile.get_path()
  logging.info("Test exported file name " + outfilepath)

  return resultImage

'''
Delete temp images.

The output images of tests have all been named and saved to file.
Delete any temp images that are not saved.
'''
def cleanupTempImages (outImage):
  # TODO targetImage.delete()
  pass


'''
On Linux, you can compare two PPM files
using the "diff" utility.
They are ASCII when the PPM format is "plain" i.e. P3.

This is susceptible to changes in underlying image algorithms.
This is *exact* comparison,
and catches subtle differences that a human can't see.
So not appropriate when subtle differences are acceptable test results.
'''
def diffPPMFilesUsingDiff(outfilepath, referencefilepath):
  # ignore comment lines by arg to diff: --ignore-matching-lines
  result = subprocess.run(['diff', '--ignore-matching-lines="^#"', outfilepath, referencefilepath], stdout=subprocess.PIPE)
  # result is-a CompletedProcess, having attribute returncode

  # diff returns zero when files are same
  return result.returncode == 0


'''
Return whether the test result image and the reference result image are the same.
IOW, whether the test result is same as on prior runs.

This hides the actual diffing algorithm.

Returns a boolean.
May log a numeric measure, zero if exactly the same.

Testing depends on a deterministic result:
the resynthesizer must be built-non threaded.
'''
def areImagesSame(outfilepath, referencefilepath):
  # diff reference image with recent image out

  # OLD return diffPPMFilesUsingDiff(outfilepath, referencefilepath)
  return diffPPM (outfilepath, referencefilepath)


'''
Run a single test case

testname: identify test
filename: image file to target
wrapperName: wrapper of outer plugin to test
testparameter: a string of params e.g. "50, 0, 0"
select: optional rect to select in target
'''
def runtest(filename, testname, wrapperName, testparameters, select=None):

  global test_summary

  logging.info("Running test " + testname + " " + filename + " " + wrapperName + " " + testparameters)
  print(">>>>>>>>>>>>>>> Running test " + testname + " " + filename + " " + wrapperName + " " + testparameters)


  '''
  Define some path names and files.
  '''
  # since some tests use the same input file, cat the testname to make an outfilename
  outfilename = testname + '-' + filename
  # out and reference have same name, different directories
  outfilepath = tmpDirName + '/' + outfilename + '.ppm'

  
  referencefilepath = referenceDir() + outfilename + '.ppm'


  # open test input file
  try:
    targetImage, targetDrawables = openTestFilename(filename, select)
  except Exception as inst:
    record_test_result(testname, "IMPROPER preprocessing")
    print (inst)
    return

  # Build an executable Python string that is call to wrapper of tested plugin
  # E.G. callHealSelection(targetImage, targetDrawables, 0, 0)
  teststring = wrapperName + "(" + "targetImage, targetDrawables, " + testparameters + ")"
  logging.info("Test string:" + teststring)


  start = time.time()
  try:
    # Invoke the test
    # Not eval, it only takes expressions, not statements
    # All python vars names in the teststring must be defined in the current scope
    # i.e. targetImage, targetDrawables, testPattern
    exec (teststring)
  except RuntimeError as inst:
    record_test_result(testname, "EXCEPTION")
    print(inst)
    return
  logging.info("Elapsed time: " + str(time.time() - start))

  '''
  !!! Note that the test string can assign to image when the test returns a new image
  (rather than alter the image.)  Special handling here.
  '''
  logging.info("Test target image name: " + filename)
  # TODO formerly logged the return image name
  # logging.info("Test image name" + targetImage.name)


  # post process output image, saving to /tmp
  try:
    resultImage = postProcessOutImage (targetImage, outfilename, outfilepath)
  except Exception as inst:
    # log to the test result file
    record_test_result(testname, f"IMPROPER post processing {inst}")
    # log to the console
    print(inst)
    return

  '''
  Assert: resultImage is named and saved. Keep it for visual comparison.
  Assert: the test itself has cleaned up working images.
  
  resultImage is same as targetImage, for plugins which alter targetImage.
  resultImage is a new image, for plugins which generate new image.

  The wrapper function is responsible for displaying the result image.
  '''

  # targetImage should not be used again, next test will overwrite it
  targetImage = None

  # when not reference out exists
  if not os.path.exists(referencefilepath) :
    logging.debug(f"Reference image file not exist: {referencefilepath}")

    # make a new reference image (the first test run)

    # Reference images are usually in a repo.
    # Thus you must commit changes to actually perpetuate new ref images
    logging.debug("Making reference output image file.")
    subprocess.run(['cp', outfilepath, referencefilepath], stdout=subprocess.PIPE)
    # status, output = getstatusoutput('cp ' + outfilepath + ' ' + referencefilepath)
    record_test_result(testname, "UNKNOWN, NO REFERENCE ", outfilename)
    # test inconclusive, no reference data, must rerun
  else:
    if not os.path.exists(outfilepath) :
       raise RuntimeError

    if not areImagesSame(outfilepath, referencefilepath):
      record_test_result(testname, "FAILED", outfilename)
    else:
      record_test_result(testname, "PASSED", outfilename)



'''
Groups of tests of an outer plugin.
'''

def testEnginePlugin():
  '''
  Engine plugin tests (without using an outer plugin)
  '''

  pluginWrapperName = 'callResynthesizer'

  '''
  Similar to heal selection:
  synthesize a small selection from the inverted selection of the same image.

  But the results are worse since the corpus is the full image
  and the user doesn't choose things like direction
  that tend to produce better results.
  '''

  # This is historical, matches v2 reference image.
  # parameters = "0,0, True, drawable_of_file_with_anti_selection('"+ zap  + "', select1).ID, -1, -1, 0.0, 0.117, 16, 500"

  # Name of all tests is "resynthfull", not e.g. resynthfullzap
  # The tests are all the same process, just different images.
  # Which matches existing reference file names.

  # New call, defaulting most parameters
  # corpus is zap , selection corpus is same i.e. select1
  # Keep the corpus name wrapped in single quotes
  parameters = "'" + zap + "'"  + " , select=select1, invert=True"
  runtest(zap, 'resynthfull', pluginWrapperName, parameters, select1)

  # Resynthesis of ufo from full context
  # parameters = "0,0, True, drawable_of_file_with_anti_selection('"+ ufo  + "', select1).ID, -1, -1, 0.0, 0.117, 16, 500"
  parameters = "'" + ufo + "'"  + " , select=select1, invert=True"
  runtest(ufo, 'resynthfull', pluginWrapperName, parameters, select1)

  # Resynthesis of rect selection in bricks from full context
  # parameters = "0,0, True, drawable_of_file_with_anti_selection('"+ brick  + "', select1).ID, -1, -1, 0.0, 0.117, 16, 500"
  parameters = "'" + brick + "'"  + " , select=select1, invert=True"
  runtest(brick, 'resynthfull', pluginWrapperName, parameters, select1)

  # Resynthesis of rect selection in donkey from full context
  # parameters = "0,0, True, drawable_of_file_with_anti_selection('"+ donkey  + "', select2).ID, -1, -1, 0.0, 0.117, 16, 500"
  # select1 in corpus fails?
  parameters = "'" + donkey + "'"  + " , select=select2, invert=True"
  runtest(donkey, 'resynthfull', pluginWrapperName, parameters, select2)


  '''
  Unusual engine tests.

  No user would usually want to do these things, but it tests edge cases.
  '''

  # Straight resynthesis, from selection in drawable into same selection in drawable
  # !!! Note the target image is not in this parameter string, it is the corpus.
  # parameters = "0,0, True, drawable.ID, -1, -1, 0.0, 0.117, 16, 500"
  # Not invert selection !!!
  parameters = "'" + ufo + "'"  + " , select=select1"
  runtest(ufo, 'resynthSelf', pluginWrapperName, parameters, select1)

  # Straight resynthesis, from separate image with no selection into selection in drawable
  #parameters = "0,0, True, drawable_of_file('"+ grass + "').ID, -1, -1, 0.0, 0.117, 16, 500"
  # corpus is an unrelated image !!!!
  parameters = "'" + grass + "'"  + " , select=select1"
  runtest(ufo, 'resynthtwoimages', pluginWrapperName, parameters, select1)

  # Straight resynthesis, from separate image with no selection into selection in drawable
  # Tileable without using context: 1,1,0,...
  #parameters = "1, 1, 0, drawable_of_file('"+ grass + "').ID, -1, -1, 0.0, 0.117, 16, 500"
  parameters = "'" + ufo + "'"  + " , select=select1, useBorder=0, hTile=True, vTile=True"
  runtest(ufo, 'resynthtileable', pluginWrapperName, parameters, select1)


def testMapStyle():
  '''
  "Map Style" outer

  These tests also test engine's treatment of differences in bpp (RGB vs RGBA)
  from corpus to target.
  '''
  pluginWrapperName = "callMapStyle"

  # with corpus having alpha
  # map style: RGBA to RGB  50% w/ color maps
  # parameters = "drawable_of_file('"+ grass + "'), 50.0, 0"
  parameters = "'" + grassAlpha + "'"  + ", 50.0, 0"
  runtest(ufo, 'mapstylealpha', pluginWrapperName, parameters, select=None)


  # with target having alpha
  # map style: RGB to RGBA 50% w/ color maps
  # TODO this is not a pluginWrapperName of the resynth since the plug-in is adding an alpha to texture
  # parameters = "drawable_of_file('"+ grass + "'), 50.0, 0"
  parameters = "'" + grass + "'"  + ", 50.0, 0"
  runtest(ufoAlpha, 'mapstyletoalpha', pluginWrapperName, parameters, select=None)

  # map style: RGB to RGB 50% w/ color maps
  # parameters = "drawable_of_file('"+ grass + "'), 50.0, 0"
  parameters = "'" + grass + "'"  + ", 50.0, 0"
  runtest(ufo, 'mapstyle', pluginWrapperName, parameters, select=None)

  # map style: RGB to RGB w/ color maps
  # This is conceptually little different from the above test,
  # but it is a popular documented example: style of a color swatch to image of angels
  # parameters = "drawable_of_file('"+ angel  + "'), 10.0, 0"
  parameters = "'" + angelTexture + "'"  + ", 10.0, 0"
  runtest(angelTarget, 'mapstyleangel', pluginWrapperName, parameters, select=None)

  # map style: RGB to RGB w/ GRAY maps and map by brightness
  # parameters = "drawable_of_file('"+ grass + "'), 50.0, 1"
  # source image, fifty percent transfer, map by brightness
  parameters = "'" + grass + "'"  + ", 50.0, 1"
  runtest(ufo, 'mapstylebrightness', pluginWrapperName, parameters, select=None)

  # mapstyle: GRAY to GRAY, 50%, map by color
  # maps an image to itself?
  # parameters = "drawable_of_file('"+ wander  + "'), 50.0, 0"
  parameters = "'" + wander + "'"  + ", 50.0, 0"
  runtest('wander', 'mapstylegraygray', pluginWrapperName, parameters, select=None)

  # mapstyle: RGB to GRAY, 50%, map by colors
  # parameters = "drawable_of_file('"+ zap  + "'), 10.0, 0"
  parameters = "'" + zap + "'"  + ", 50.0, 0"
  runtest(zapOutputMap, 'mapstylergbgray', pluginWrapperName, parameters, select=None)


def testHealSelection():
  ''' "Heal selection" outer plugin. '''

  pluginWrapperName = 'callHealSelection'

  # Heal from donut corpus
  parameters = "50, 1, 1" # 1 = sample from sides, 1 = direction inward
  runtest(ufo, 'heal', pluginWrapperName, parameters, select1)

  # This is a common example: the donkey
  # Heal from corpus on the sides, nearby
  # select2 selects the donkey
  # This usually gives better results than corpus entire image, direction random
  parameters = "50, 1, 1" # 1 = sample from all around, 1 = direction inward
  runtest(donkey, 'heal', pluginWrapperName, parameters, select2)

  # Heal from donut corpus
  # also randomly all around
  parameters = "50, 0, 0" # 0 = sample from all around, 0 = randomly
  runtest(ufo, 'healaroundrandom', pluginWrapperName, parameters, select1)

  # Heal where target includes alpha layer.
  # This selects from an area having a transparent area.
  # The transparent area should NOT be resynthesized, but yield white upon flattening.
  parameters = "50, 1, 1"
  runtest('ufo-input-w-alpha', 'healincludedalpha', pluginWrapperName, parameters, select1)

  # Heal where target has distant alpha layer.
  # This selects from a black sky area with a nearby transparent area.
  # The result should be black resynthesized, NOT white of transparent flattened.
  parameters = "50, 1, 1"
  runtest('apollo11_w_alpha', 'healalpha', pluginWrapperName, parameters, select1)

  # Heal a grayscale
  parameters = "50, 1, 1"
  runtest('wander', 'healgray', pluginWrapperName, parameters, select1)

  # Heal a grayscale w alpha
  parameters = "50, 0, 1"
  runtest('ufo-input-w-alpha-gray', 'healalphagray', pluginWrapperName, parameters, select1)


def testHealTransparency():
  ''' Heal transparency outer plugin. '''
  
  pluginWrapperName = 'callHealTransparency'

  # Heal transparency outward
  parameters = "50, 2" # pixels of context width, outward
  runtest('apollo11_w_alpha', 'healtransparency', pluginWrapperName, parameters, select=None)

  # Heal transparency inward
  parameters = "50, 1" # pixels of context width, 1=inward
  runtest('apollo11_w_alpha', 'healtransparencyinward', pluginWrapperName, parameters, select=None)


def testRenderTexture():

  # without alpha, suitable for seamless tiling
  runtest(grass, 'rendertexture', "callRenderTexture", "2.0, True")

  # source with alpha, without seamless tiling
  runtest(wilber, 'rendertexture', "callRenderTexture", "1.5, False")
  # assert result image has alpha component
  # assert result image has no transparent pixels.


def testUncrop():
  ''' Uncrop outer plugin '''

  # Test image w/o alpha
  pluginWrapperName = "callUncrop"
  parameters = "20, False"
  runtest(ufo, 'uncrop', pluginWrapperName, parameters, select=None)

  # Test image with alpha, and w/o anti-erase
  # Wilber has transparent edges, yields transparent enlargement.
  # Which is plausible.
  pluginWrapperName = "callUncrop"
  parameters = "20, False"
  runtest(wilber, 'uncrop-alpha-no-anti-erase', pluginWrapperName, parameters, select=None)

  # Test image with alpha, and with anti-erase
  # Wilber has transparent edges, yields opaque enlargement.
  # But enlargment is weird, not plausible since it surrounds a transparent image.
  pluginWrapperName = "callUncrop"
  parameters = "20, True"
  runtest(wilber, 'uncrop-alpha-anti-erase', pluginWrapperName, parameters, select=None)

  # Note the reference images are flattened before saving,
  # you can't see, only infer (where white), resultant transparency.
  # You can Undo the flattening to see the resultant transparency in the displayed images.



def testFillPattern():
  ''' Fill Pattern outer plugin '''

  pluginName = "plug-in-fill-pattern-resynth"
  pluginWrapperName = "callFillPatternResynth"

  # parameter refer to pattern object in scope of runtest()
  # The pattern is a Python object, not a string.
  # parameter is name of Python pattern object
  parameters = "testPattern"
  
  runtest(ufo, 'fillpattern', pluginWrapperName, parameters, select1)



'''
Sharpen with resynthesis
Too slow to test regularly.
Image stays the same resolution (count of pixels.)
Image might appear more sharp (more grainy?)
Count of distinct colors might be reduced? Eliminating some colors that interpolate?
'''
def testSharpen():
  pluginName = "plug-in-sharpen-resynthesized"
  pluginWrapperName = "callSharpen"
  parameters = "factor=2"
  runtest(ufoSmall, 'sharpen', pluginWrapperName, parameters, select=None)


def testEnlarge():
  '''
  Enlarge with resynthesis.
  Image dimensions increase by factor.
  Image count of pixels increase by factor squared.
  The image looks the same as the original, in gross.
  The image detail MIGHT be better than the usual scale,
  which interpolates.

  !!! Very slow. Too slow to test regularly?
  '''
  pluginWrapperName = "callEnlarge"
  parameters = "factor=2"
  runtest(ufoSmall, 'enlarge', pluginWrapperName, parameters, select=None)


'''
Test plugins on image of mode INDEXED.

The resynthesizer engine works on several pixel formats.
It can work on pixels of one byte (the palette index.)

Works best when the palette is ordered by value
E.G. the palette is monochrome: gray, sepia, etc.
Otherwise, the results are not as good, not plausible.
Ordering: values with near index should have near colors/values.

These tests are designed to show plausible results.
So they use an ordered palette, i.e. sepia, which is monochrome.
Obtain such an image using Image>Mode>Indexed and Color>Desaturate>Sepia
'''
def testIndexedMode():

  # heal selection
  parameters = "50, 2, 1" # 1 = sample from above and below, 1 = direction inward
  runtest(ufoSmallIndexedSepia, 'healUfoIndexedSepia', 'callHealSelection', parameters, selectUfoSmall)

  # healTransparency
  runtest(wilberSmallIndexedAlpha, 'healTransparencyUfoIndexedSepia', 'callHealTransparency', 
          "50, 2", # pixels of context width, outward
          select=None)
  # Result should be wilber with synthesized background

  # uncrop
  runtest(ufoSmallIndexedSepia, 'uncropUfoIndexedSepia', 'callUncrop', 
          "20, True", # 20%, True = anti-erase
          select=None)

  # map style
  # Note the plugin synchronizes the mode of the target image to the corpus image.

  # Here the target is indexed and corpus is RGB, yielding RGB.
  # The plugin will convert the target to RGB.
  runtest(ufoSmallIndexedSepia, 'mapStyleUfoIndexedSepiaFromRGB', 'callMapStyle', 
          "'" + grass + "'"  + ", 50.0, 0",
          select=None)
  
  # Here the target is indexed and corpus is indexed, yielding indexed
  runtest(ufoSmallIndexedSepia, 'mapStyleUfoIndexedSepiaFromIndexed', 'callMapStyle', 
          "'" + wilberSmallIndexedAlpha + "'"  + ", 50.0, 0",
          select=None)

  # render texture
  runtest(ufoSmallIndexedSepia, 'renderTextureUfoIndexedSepia', 'callRenderTexture',
          "2.0, True",
          select=None)
  # TODO the ufo object seems to dominate the texture improperly
  # I think the texture should show other portions of the ufo image
  
  # fill pattern
  runtest(ufoSmallIndexedSepia, 'fillPatternUfoIndexedSepia', 'callFillPatternResynth', 
          "testPattern",
          select=None)
  # Result looks like maple leaves, sepia.

  # Not testing the resynthesizer engine directly, or Sharpen, or Enlarge


'''
Test adaption of high-bit depth images to working format of the resynthesizer engine.
The engine works on 8-bit depth images, but the input image may be higher.

For now, the adaptor writes back to the target image with loss of precision.
Converting from 8-bit to higher bit depth, e.g. 16-bit.
Instead, it could use the source coords of the best match
to get the precise pixel value from the source image.

TODO Get an image with more precision (non-zero values in the lower bits)
The one used is a 16-bit image with all zeroes in the lower bits.
Then you can see the difference between the input and output images.
The output image, where synthesized, for now will have zeroes in the lower bits.

TODO test conversion from a floating format.
I can not forsee a reason in the resynthesizer it would not work; 
it would only test that Babl is correct?
'''
def testBitDepth():
  # heal selection
  parameters = "50, 2, 1" # 1 = sample from above and below, 1 = direction inward
  runtest(ufoSmallHighBitDepth, 'healUfoHighBitDepth', 'callHealSelection', parameters, selectUfoSmall)



def configureLog():
  logFilename = tmpDirName + '/resynth-test.log'
  logging.basicConfig(filename=logFilename, level=logging.DEBUG)

  print("Running testResynth.py, log file: " + logFilename)


def testAll():
  '''
  Run test cases for resynthesizer plugin and outer plugins that call it.

  Notes on the test string:  optionally use names "image" and "drawable" to match the variable names used
  in the runtest() function; the names will be eval'ed.
  '''

  # log and result images to the temp dir
  makeTempDir()
  configureLog()

  initTestPattern()

  # These are the plugins that typically exist.
  # Some no longer exist or haven't been ported.

  # This may test each plugin more than once, varying args for varying use cases.

  # Temporarily disabled, to run one at a time
  if True:
    testEnginePlugin()
    testHealSelection()
    testHealTransparency()
    testRenderTexture()
    testMapStyle()
    testUncrop()
    testFillPattern()

    testIndexedMode()

    testBitDepth()


  # Temporarily disabled, until ported to 3.0 API
  # Doubt these are worth porting.
  # testSharpen()
  # testEnlarge()

  # TODO a much harder test with many layers and alpha channels

  logging.info("\nTemp directory: " + tmpDirName)
  logging.info( "\n" + test_summary)





'''
The testing plugin itself.
Just a standard shell for a plugin.
'''
def test_resynth(procedure, args, data):
    testAll()
    return procedure.new_return_values(Gimp.PDBStatusType.SUCCESS, GLib.Error())

class TestResynthPlugin (Gimp.PlugIn):

    # Copied from pallette-offset.py
    # Must have a run-mode arg, even when is not an image procedure

    ## Parameter: run-mode ##
    @GObject.Property(type=Gimp.RunMode,
                      default=Gimp.RunMode.NONINTERACTIVE,
                      nick="Run mode", blurb="The run mode")
    def run_mode(self):
        """Read-write integer property."""
        return self._run_mode

    @run_mode.setter
    def run_mode(self, run_mode):
        self._run_mode = run_mode



    ## GimpPlugIn virtual methods ##
    def do_set_i18n(self, procname):
        print("query")
        return True, 'gimp30-python', None

    def do_query_procedures(self):
        print("query")
        return [ 'plug-in-test-resynth' ]

    def do_create_procedure(self, name):
        print("create")
        procedure = Gimp.Procedure.new(self, name,
                                            Gimp.PDBProcType.PLUGIN,
                                            test_resynth, # run_func
                                            None)

        # not require image to be open
        procedure.set_sensitivity_mask (Gimp.ProcedureSensitivityMask.NO_IMAGE)

        procedure.set_documentation ("Test Resynthesizer",
                                     "Test Resynthesizer",
                                     name)
        procedure.set_attribution("Lloyd Konneker",
                                  "Lloyd Konneker",
                                  "2023")

        procedure.add_enum_argument ("run-mode", "Run mode",
                                         "The run mode", Gimp.RunMode,
                                         Gimp.RunMode.INTERACTIVE,
                                         GObject.ParamFlags.READWRITE)

        procedure.set_menu_label("Test Resynthesizer...")
        # Top level menu "Test"
        procedure.add_menu_path ("<Image>/Test")

        return procedure


Gimp.main(TestResynthPlugin.__gtype__, sys.argv)
