/*--------------------------------------------------------
  rtsp.c - the RTSP state engine

--------------------------------------------------------*/


/*--------------------------------------------------------

REALNETWORKS LICENSE AGREEMENT AND WARRANTY 
             DISCLAIMER

_____________________________________________

Free Real-Time Streaming Protocol (RTSP) Firewall 
             Proxy License

IMPORTANT -- READ CAREFULLY: This RealNetworks 
License Agreement ("License Agreement") is a legal 
agreement between you (either an individual or an 
entity) and RealNetworks, Inc.  and its suppliers and 
licensors collectively ("RN") for the software product 
listed above, which includes computer software and 
associated media and printed material, whether provided 
in a physical form or received on-line form ("Software").  
By clicking on the "Accept" button or opening the 
package, you are consenting to be bound by this Agreement.  
If you do not agree to all of the terms of this agreement, 
click the "Do Not Accept" button and, if you received the 
Software by package, return the product to the place of 
purchase.

__________________________________________________________

1. GRANT OF LICENSE.

Subject to the provisions contained in this License Agreement, 
RN hereby grants you a non-exclusive, non-transferable, 
perpetual, worldwide license to use, modify or redistribute the 
Software subject to the following terms and conditions:

(a) The copyright notice (" 1998 RealNetworks, 
Inc.") and this copy of  this License Agreement shall 
appear on all copies and/or any derivative versions 
of the Software you create or distribute.

(b)	You acknowledge and agree that RN is and shall be 
the exclusive owner of all right, title and interest, 
including copyright, in the Software.

All rights not expressly granted to you are reserved to RN.

2.  SOFTWARE MAINTENANCE AND UPGRADES. 

RN is not obligated to provide maintenance or updates to you 
for the Software. However, any maintenance or updates 
provided by RN shall be covered by this Agreement.

3.  DISCLAIMER OF WARRANTY.

The Software is deemed accepted by you.  Because RN is 
providing you the Software for free, the Software is provided 
to you AS IS, WITHOUT WARRANTY OF ANY KIND. TO 
THE MAXIMUM EXTENT PERMITTED BY 
APPLICABLE LAW, REALNETWORKS FURTHER 
DISCLAIMS ALL
WARRANTIES, INCLUDING WITHOUT LIMITATION 
ANY IMPLIED WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE, AND NONINFRINGEMENT. THE ENTIRE 
RISK ARISING OUT OF THE USE OR PERFORMANCE 
OF THE SOFTWARE REMAINS WITH YOU. TO THE 
MAXIMUM EXTENT PERMITTED BY APPLICABLE 
LAW, IN NO
EVENT SHALL REALNETWORKS OR ITS SUPPLIERS 
BE LIABLE FOR ANY CONSEQUENTIAL, INCIDENTAL, 
DIRECT, INDIRECT, SPECIAL, PUNITIVE, OR OTHER 
DAMAGES WHATSOEVER (INCLUDING, WITHOUT 
LIMITATION, DAMAGES FOR LOSS OF BUSINESS 
PROFITS, BUSINESS INTERRUPTION, LOSS OF 
BUSINESS INFORMATION, OR OTHER PECUNIARY 
LOSS) ARISING OUT OF THIS AGREEMENT OR THE 
USE OF OR INABILITY TO USE THE SOFTWARE, EVEN 
IF REALNETWORKS HAS BEEN ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGES. BECAUSE SOME 
STATES/JURISDICTIONS DO NOT ALLOW THE 
EXCLUSION OR LIMITATION OF LIABILITY FOR 
CONSEQUENTIAL OR INCIDENTAL DAMAGES, THE 
ABOVE LIMITATION MAY NOT APPLY TO YOU.

4. INDEMNIFICATION. 

You hereby agree to defend, indemnify, and hold RN, its 
directors, officers, employees and agents, harmless from any 
and all claims, damages, and expenses (including attorneys 
fees and costs) of any nature arising out of the use, 
modification, or redistribution the Software or any derivative 
versions thereof.

5. U.S. GOVERNMENT RESTRICTED RIGHTS AND 
EXPORT RESTRICTIONS. 

The Software is provided with RESTRICTED RIGHTS. Use, 
duplication, or disclosure by the Government is subject to 
restrictions as set forth in subparagraph (a) through (d) of the 
of the Commercial Computer Software-Restricted Rights at 
FAR 52.227-19, as applicable, or subparagraph (c)(1)(ii) of 
The Rights in Technical Data and Computer Software clause 
of DFARS 252.227-7013, and in similar clauses in the NASA 
FAR supplement, as applicable.  Manufacturer is 
RealNetworks, Inc., 1111 Third Avenue, Suite 500, Seattle, 
Washington 98101.  You acknowledge that none of the 
Software or underlying information or technology may be 
downloaded or otherwise exported or re-exported (i) into (or 
to a national or resident of) Cuba, Iraq, Libya, Yugoslavia, 
North Korea, Iran, Syria, Sudan or Angola or any other 
country to which the U.S. has embargoed goods; or (ii) to 
anyone on the U.S. Treasury Department's list of Specially 
Designated Nationals or the U.S. Commerce Department's 
Table of Denial Orders.  By using the Software, you are 
agreeing to the foregoing and you are representing and 
warranting that you are not located in, under the control of, or 
a national or resident or resident of any such country or on any 
such list.

6. GOVERNING LAW; ATTORNEYS FEES. 

This agreement shall be governed by the laws of the State of 
Washington and you further consent to jurisdiction by the state 
and federal courts sitting in the State of Washington. If either 
RN or you employs
attorneys to enforce any rights arising out of or relating to this 
Agreement, the prevailing party shall be entitled to recover 
reasonable attorneys' fees.

8.  ENTIRE AGREEMENT. 

This agreement constitutes the complete and exclusive 
agreement between RN and you with respect to the subject 
matter hereof, and supersedes all prior oral or written 
understandings, communications or agreements not 
specifically incorporated herein.  This agreement may not be 
modified except in a writing duly signed by an authorized 
representative of RN and you.    

Copyright  1997-1998 RealNetworks, Inc. and or its 
suppliers.  1111 Third Avenue, Suite 2900, Seattle, 
Washington 98101 U.S.A.  All rights are reserved.



RTSP Proxy License Agreement 8-98

--------------------------------------------------------*/


#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#ifdef NEED_STRING_H
#include <string.h>
#endif /* NEED_STRING_H */
#include <ctype.h>
#include <time.h>
#ifdef USE_SYS_TYPES_H
#include <sys/types.h>
#endif /* USE_SYS_TYPES_H */
#include <netinet/in.h>

#include "rtsp.h"
#include "util.h"
#include "simplib.h"
#include "setup.h"


/* some convenience macros */
#define TOLOWER(c)      (((c) >= 'A' && (c) <= 'Z') ? ((c) - 'A' + 'a') : (c))
#define CHARTODIGIT(c)  ((c) - '0')
#define ISDIGIT(c)      (((c) >= '0') && ((c) <= '9'))
#define ISUPPER(c)      (((c) >= 'A') && ((c) <= 'Z'))
#define ISLOWER(c)      (((c) >= 'a') && ((c) <= 'z'))
#define ISWHITE(c)      (((c) == ' ') || ((c) == '\t'))
#define ISEND(c)        (((c) == '\n') || ((c) == '\r'))

/* local port range */
#define MIN_LOCAL_PORT  1024
#define MAX_LOCAL_PORT  30000

/* RTSPFromType - the direction an RTSP message is coming from (client/server)*/
typedef enum
{
  FROM_CLIENT,
  FROM_SERVER
} RTSPFromType;



/* RTSPStateType - various states the RTSP state engine may enter */
typedef enum
{
  RTSP_STATE_EXPECTING,
  RTSP_STATE_METHOD,
  RTSP_STATE_CONTENT,
  RTSP_STATE_FORWARD
} RTSPStateType;



/* RTSPMethodType - a list of all the RTSP methods we handle (see 
                    RTSPHandlerTable and new_method_handler_table() below.)
*/
typedef enum
{
  RTSP_METHOD_OPTIONS,
  RTSP_METHOD_DESCRIBE,
  RTSP_METHOD_ANNOUNCE,
  RTSP_METHOD_SETUP,
  RTSP_METHOD_PLAY,
  RTSP_METHOD_PAUSE,
  RTSP_METHOD_TEARDOWN,
  RTSP_METHOD_GET_PARAMETER,
  RTSP_METHOD_SET_PARAMETER,
  RTSP_METHOD_REDIRECT,
  RTSP_METHOD_RECORD,

  /* non-methods */
  RTSP_METHOD_RESPONSE,


  /* placeholder! */
  RTSP_METHOD_LAST
} RTSPMethodType;



  /* TransportType - enumerates all the transports we support.
     We will check for the transports in the order listed.
     See rtsp_class_init() and rtsp_transports[] below.
   */
typedef enum
{
  TRANSPORT_X_PN_TNG_TCP,
  TRANSPORT_X_REAL_RDT_TCP,
  TRANSPORT_RTP_AVP_TCP,

  TRANSPORT_X_PN_TNG_UDP,
  TRANSPORT_X_REAL_RDT_UDP,
  TRANSPORT_RTP_AVP_UDP,

  /* placeholder! */ 
  TRANSPORT_LAST
} TransportType;




/* structures */

typedef struct _RTSP        RTSP;
typedef struct _CSeq        CSeq;
typedef struct _Transport   Transport;
typedef struct _PortPair    PortPair;
typedef struct _SessionIDMapping SessionIDMapping;




/* rtsp method handlers  */

typedef int (*RTSPMethodHandler) (RTSP * rtsp, RTSPFromType from, char * buff, int * n);

/* these method handlers are used by the client RTSP object */

int rtsp_options_handler (RTSP * rtsp, RTSPFromType from, char * buff, int * n);
int rtsp_setup_handler   (RTSP * rtsp, RTSPFromType from, char * buff, int * n);
int rtsp_teardown_handler(RTSP * rtsp, RTSPFromType from, char * buff, int * n);
int rtsp_response_handler(RTSP * rtsp, RTSPFromType from, char * buff, int * n);
int rtsp_play_handler    (RTSP * rtsp, RTSPFromType from, char * buff, int * n);

/* these method handlers are used by the server RTSP object(s) */

int rtsp_response_handler_sv  (RTSP * rtsp, RTSPFromType from, char * buff, int * n);
int rtsp_set_param_handler_sv (RTSP * rtsp, RTSPFromType from, char * buff, int * n);



/* default method handlers for RTSP objects */
typedef struct _RTSPHandlerTable RTSPHandlerTable;
struct _RTSPHandlerTable {
  char *text;                 /* string to watch for */
  int len;                    /* string's length */
  RTSPMethodHandler handler;  /* call this routine when match found (or NULL) */
};


/* RTSP - a structure for holding information related to an RTSP connection.
   Each side of the RTSP connection (client and server) gets its own
   RTSP object.  Since the client may connect to multiple servers
   while in TCP-only mode, a separate RTSP object is needed for each
   server.  At present, the method handlers for the client object do
   most of the work, the server object existing primarily to provide
   additional buffering so messages from various servers get interlaced
   properly.
*/
struct _RTSP
{
  int             rtsp_id;          /* a unique identifier */
  char           *rtsp_idstr;       /* a unique identifier ("server:port") */

  int             rtsp_seen_first_option_request;
  int             rtsp_objtype;     /* server or client object */

  char           *rtsp_cbuff;       /* client-side buffer */
  int             rtsp_cbufflen;    /* client-side buffer length */
  int             rtsp_ccontentlen; /* client-side Content-Length: value */
  int             rtsp_cmethodlen;  /* client-side method **header** length */
  int             rtsp_cembededlen; /* client-side $-embedded data length */
  RTSPStateType   rtsp_cstate;      /* client-side state */

  char           *rtsp_sbuff;       /* server-side buffer */
  int             rtsp_sbufflen;    /* server-side buffer length */
  int             rtsp_scontentlen; /* server-side Content-Length: value */
  int             rtsp_smethodlen;  /* server-side method **header** length */
  int             rtsp_sembededlen; /* server-side $-embedded data length */
  RTSPStateType   rtsp_sstate;      /* server-side state */

  GHashTable     *rtsp_cseq;        /* a table of CSeq data */
  GHashTable     *rtsp_session;     /* a table of Session data */
  GHashTable     *rtsp_cseq_map;    /* for mapping client/server CSeq numbers*/
  GHashTable     *rtsp_session_map; /* for remapping session ids */

  GList          *rtsp_list; /* other RTSP structures related to this one. */

  /* these are related to remapping CSeq ids; see rtsp_remap_cseq_id(). */
  int             rtsp_last_cseq_id;/* last client->server CSeq id seen */
  int             rtsp_last_server_cseq_id; /* last server->client CSeq seen */
                                               


  /* callbacks to child-main.c for network-level operations */
  RTSPServerConnectFunc   connect_to_server;
  RTSPWriteFunc           write_client;
  RTSPWriteServersFunc    write_all_servers;
  RTSPAppendServerFunc    append_server;
  RTSPUDPTunnelOpenFunc   udp_client_tunnel_open;
  RTSPUDPTunnelOpenFunc   udp_server_tunnel_open;
  RTSPUDPTunnelCloseFunc  udp_tunnel_close;

  RTSPHandlerTable *rtsp_method;   /* a table of method handler routines */

};


/* CSeq - a structure for holding information related to an RTSP Sequence ID */
struct _CSeq
{
  int    cseq_id;         /* The CSeq identifier */
  char  *cseq_session;    /* The session associated with this CSeq */
  GList *cseq_transports; /* the transports being configured with this CSeq */
  int    cseq_srtspid;    /* the RTSP id of the associated server */
  int    cseq_ping;       /* the server sent a "ping" message */

};


/* Transport - a structure for holding information about transports */
struct _Transport
{
  TransportType  transport_type;      /* The type of transport */
  GList         *transport_portpairs; /* The associated ports */
};


/* PortPair - a structure for holding a "portpair" which associates a port
   on the proxy with a port on a remote server or client (a "tunnel").
 */
struct _PortPair
{
  int   portpair_udpid;  /* id of allocated UDP tunnel */

  unsigned short portpair_rport;  /* remote port */
  unsigned short portpair_pport;  /* proxy port */
};



/* SessionIDMapping - a structure for remapping Session IDs */
struct _SessionIDMapping
{
  char *client_session;  /* The client's version of the session id */
  int   client_rtspid;   /* The client's RTSP id */
  char *server_session;  /* The server's version of the session id */
  int   server_rtspid;   /* The server's RTSP id */
};



/* prototypes */

static void clear_cseq_hash_node (void * key, void * data, void * user_data);
static void clear_session_hash_node (void * key, void * data, void * user_data);
static void clear_session_map_node (void * key, void * data, void * user_data);
static int  rtsp_read (RTSPFromType from, int rtspid, char *buff, int n);
static int  extract_serverid (char * buff, int len);
static int  extract_host_and_port (char * buff, int len, char * hbuff, int hmaxlen, unsigned short * portp);
static void append_output_buffer (RTSP * rtsp, RTSPFromType from, char * buff, int len, char * obuff, int * olenp);
static char rtsp_get_channel_id (RTSP * rtsp, char old_cid);
static void rtsp_map_channel_ids (RTSP * rtsp, int num1, int num2, int * newnum1, int * newnum2);
static int rtsp_remove_session (RTSP * rtsp, char * session);
static RTSPHandlerTable * new_method_handler_table(void);
static int next_rtsp_id (void);
static int rtsp_remap_cseq_id (RTSP *rtsp, RTSPFromType from, RTSPMethodType method, char * buff, int * len);
static void replace_cseq_number (char * str, int slen, char * buff, int * len);



extern void close_udp_tunnel (int idx, int sibflag);
extern int create_rtp_udp_tunnel (int port, int toserver, int count, int rtspid, int local);
extern int rtp_udp_tunnel_port_ok (int idx, unsigned short int *pport);
extern int rtp_rtcp_tunnel_port_ok (int idx2, int idx1, unsigned short int *pport);
extern void set_tunnel_siblings (int idx1, int idx2);
extern int get_sibling_index (int idx);
extern void append_rtsp_conn_buff (int id, char *buff, int len);
extern void show_rtsp_conn_buffs (void);




/* string parsing */

static RTSPMethodType method_type  (RTSP * rtsp, char * buff, int len);
static int get_cseq (char * data, int n);
static char *get_session (char * data, int n);
static int get_num_range (char * data, int data_length, unsigned short *num1, unsigned short *num2);
static int cistrncmp (char * str1, char * str2, int len);
static int cistrnfind (char * str1, int len1, char * str2, int len2);

#define get_port_range(str,len,num1,num2) get_num_range(str, len, num1, num2)


/* allocation/destruction of basic objects */

static PortPair   *portpair_new      (RTSP * rtsp);
static void        portpair_destroy  (RTSP * rtsp, PortPair * portpair);
static Transport  *transport_new     (RTSP * rtsp, TransportType type);
static void        transport_destroy (RTSP * rtsp, Transport * transport);
static CSeq       *cseq_new          (RTSP * rtsp);
static void        cseq_destroy      (RTSP * rtsp, CSeq * cseq);




/* transports we proxy */

typedef Transport * (*GetPortsFunc) (RTSP * rtsp, TransportType type, char * buff, int n);
typedef int (*SetPortsFunc) (RTSP * rtsp, Transport * transport, RTSPFromType from, char * buff, int * n);


static void rtsp_remove_transports (RTSP * rtsp, char * buff, int * n);
static Transport *x_real_rdt_get_ports (RTSP * rtsp, TransportType type, char * buff, int n);
static int x_real_rdt_set_ports (RTSP * rtsp, Transport * transport, RTSPFromType from, char * buff, int * n);

static Transport *rtp_get_ports (RTSP * rtsp, TransportType type, char * buff, int n);
static int rtp_set_ports (RTSP * rtsp, Transport * transport, RTSPFromType from, char * buff, int * n);
static int rtp_set_ports_client (RTSP * rtsp, Transport * transport, RTSPFromType from, char * buff, int * n);
static int rtp_set_ports_server (RTSP * rtsp, Transport * transport, RTSPFromType from, char * buff, int * n);
static void rtsp_interleave_update (RTSP * rtsp, char * buff, int *n);
static void rtsp_session_update (RTSP * rtsp, RTSPFromType from, char * buff, int *n);
static SessionIDMapping * session_map_new (char * server_session, int server_session_len, char * client_session, int client_session_len, int server_rtspid, int client_rtspid);
static void session_map_destroy (SessionIDMapping *sess);
static char * generate_session_id (RTSP *rtsp, char *str, int len);
static int mutate_session_id (char *str, int len);




/* rtsp_transports[] - a list of supported transports & associated routines */
struct {
  char *text;             /* a transport name */
  int   len;              /* the length of this name */
  GetPortsFunc get_ports; /* the routine to extract port nums from SETUP msgs */
  SetPortsFunc set_ports; /* the routine to remap port nums in SETUP msgs */
} rtsp_transports[TRANSPORT_LAST];


/* the rtsp id/pointer and idstr/pointer hashes */
static GHashTable *rtsp_id_hash = NULL;
static GHashTable *rtsp_idstr_hash = NULL;
int max_rtsp_id=0; /* maximum allocated RTSP id */


/* A table for remapping channel ids (used for $-encapsulation) */
static GHashTable *channel_id_hash = NULL;

/* A table for remapping CSeq IDs so that when connecting to multiple
   servers, each one sees IDs that increment by one.
*/
/*static GHashTable *cseq_map_hash=NULL;*/


/* connect to server/port */
static RTSPServerConnectFunc connect_to_server = NULL;


/* write a buffer to the client */
static RTSPWriteFunc write_client = NULL;


/* flush all server buffers */
static RTSPWriteServersFunc write_all_servers = NULL;


/* append to a server output buffer (without writing) */
static RTSPAppendServerFunc append_server = NULL;

  
/* Create a new UDP tunnel forwarding to port 
 * cport on the client, returning a ID for the
 * tunnel, and also assigning pport the value of
 * the port on the proxy the server needs to send
 * packets to.
 */
static RTSPUDPTunnelOpenFunc udp_client_tunnel_open = NULL;
static RTSPUDPTunnelOpenFunc udp_server_tunnel_open = NULL;

/* close a UDP tunnel */
static RTSPUDPTunnelCloseFunc udp_tunnel_close = NULL;


/* memory chunks -- fast fixed size memory allocation */
#define RTSP_CHUNK_SIZE        50
#define CSEQ_CHUNK_SIZE        50
#define TRANSPORT_CHUNK_SIZE   (CSEQ_CHUNK_SIZE * 2)
#define PORTPAIR_CHUNK_SIZE    (TRANSPORT_CHUNK_SIZE * 2)

static GMemChunk *rtsp_mem_chunk      = NULL;
static GMemChunk *cseq_mem_chunk      = NULL; 
static GMemChunk *transport_mem_chunk = NULL;
static GMemChunk *portpair_mem_chunk  = NULL;

/* this is the length of the string used as a key in the content-id
   hash table.  It is currently the content-id expressed as an ASCII
   number, followed by a colon and the RTSP object id, again expressed
   as an ASCII number.  This is related to remapping channel ids for
   multiplexing multiple P->S server RTSP connections over a single
   C->P RTSP connection.
 */ 
#define MAX_CID_STR_LEN 32




/*----------------------------------------------------------------------*/


/*----------------------------------------------------------------------
   rtsp_class_init() -- Initialization needed before using RTSP objects.

   Paramaters:
     sc   - A function for connecting to a server.
     wc   - A function for writing to a client.
     ws   - A function for writing to the servers (flushes output buffers).
     as   - A function for appending to a server output buffer.
     ucto - A function for opening a UDP tunnel to a client.
     usto - A function for opening a UDP tunnel to a server.
     utc  - A function for closing a UDP tunnel.

   Return:  nothing

   This is initial setup needed before using rtsp_new() and other routines.
   It sets up the default callbacks and some related hash tables.
----------------------------------------------------------------------*/
void
rtsp_class_init (RTSPServerConnectFunc sc,
                 RTSPWriteFunc wc,
                 RTSPWriteServersFunc ws,
                 RTSPAppendServerFunc as,
                 RTSPUDPTunnelOpenFunc ucto,
                 RTSPUDPTunnelOpenFunc usto,
                 RTSPUDPTunnelCloseFunc utc)
{
  connect_to_server      = sc;
  write_client           = wc;
  write_all_servers      = ws;
  append_server          = as;
  udp_client_tunnel_open = ucto;
  udp_server_tunnel_open = usto;
  udp_tunnel_close       = utc;

  /* rtsp id-to-pointer hash */
  rtsp_id_hash    = g_hash_table_new (g_int_hash, g_int_equal);
  rtsp_idstr_hash = g_hash_table_new (g_str_hash, g_str_equal);

  /* A table for remapping channel ids */
  channel_id_hash = g_hash_table_new (g_str_hash, g_str_equal);

  /* initalize memory chunks */
  rtsp_mem_chunk = 
    g_mem_chunk_new ("rtsp mem chunk",
                     sizeof (RTSP),
                     sizeof (RTSP) * RTSP_CHUNK_SIZE,
                     G_ALLOC_AND_FREE);

  cseq_mem_chunk = 
    g_mem_chunk_new ("cseq mem chunk",
                     sizeof (CSeq),
                     sizeof (CSeq) * CSEQ_CHUNK_SIZE,
                     G_ALLOC_AND_FREE);

  transport_mem_chunk = 
    g_mem_chunk_new ("transport mem chunk",
                     sizeof (Transport),
                     sizeof (Transport) * TRANSPORT_CHUNK_SIZE,
                     G_ALLOC_AND_FREE);

  portpair_mem_chunk =
    g_mem_chunk_new ("ports mem chunk",
                     sizeof (PortPair),
                     sizeof (PortPair) * PORTPAIR_CHUNK_SIZE,
                     G_ALLOC_AND_FREE);

rtsp_transports[TRANSPORT_X_PN_TNG_TCP].text        = "x-pn-tng/tcp";
rtsp_transports[TRANSPORT_X_PN_TNG_TCP].len         = 12;
rtsp_transports[TRANSPORT_X_PN_TNG_TCP].get_ports   = NULL;
rtsp_transports[TRANSPORT_X_PN_TNG_TCP].set_ports   = NULL;

rtsp_transports[TRANSPORT_X_REAL_RDT_TCP].text      = "x-real-rdt/tcp";
rtsp_transports[TRANSPORT_X_REAL_RDT_TCP].len       = 14;
rtsp_transports[TRANSPORT_X_REAL_RDT_TCP].get_ports = NULL;
rtsp_transports[TRANSPORT_X_REAL_RDT_TCP].set_ports = NULL;

rtsp_transports[TRANSPORT_RTP_AVP_TCP].text         = "rtp/avp/tcp";
rtsp_transports[TRANSPORT_RTP_AVP_TCP].len          = 11;
rtsp_transports[TRANSPORT_RTP_AVP_TCP].get_ports    = NULL;
rtsp_transports[TRANSPORT_RTP_AVP_TCP].set_ports    = NULL;

rtsp_transports[TRANSPORT_X_PN_TNG_UDP].text        = "x-pn-tng";
rtsp_transports[TRANSPORT_X_PN_TNG_UDP].len         = 8;
rtsp_transports[TRANSPORT_X_PN_TNG_UDP].get_ports   = x_real_rdt_get_ports;
rtsp_transports[TRANSPORT_X_PN_TNG_UDP].set_ports   = x_real_rdt_set_ports;

rtsp_transports[TRANSPORT_X_REAL_RDT_UDP].text      = "x-real-rdt";
rtsp_transports[TRANSPORT_X_REAL_RDT_UDP].len       = 10;
rtsp_transports[TRANSPORT_X_REAL_RDT_UDP].get_ports = x_real_rdt_get_ports;
rtsp_transports[TRANSPORT_X_REAL_RDT_UDP].set_ports = x_real_rdt_set_ports;

rtsp_transports[TRANSPORT_RTP_AVP_UDP].text         = "rtp/avp";
rtsp_transports[TRANSPORT_RTP_AVP_UDP].len          = 7;
rtsp_transports[TRANSPORT_RTP_AVP_UDP].get_ports    = rtp_get_ports;
rtsp_transports[TRANSPORT_RTP_AVP_UDP].set_ports    = rtp_set_ports;

}



/*----------------------------------------------------------------------
   rtsp_new() -- Allocate and initialize a new RTSP structure.

   Paramaters:
     idstr   - A unique string identifying this structure/object.
     parent  - A "parent" RTSP object, possibly NULL.
     objtype - The type of RTSP object this is, CLIENT_OBJECT or SERVER_OBJECT.

   Return:
     The id of the allocated RTSP structure/object.

   Currently, have one CLIENT_OBJECT which is the parent of all SERVER_OBJECTS.
----------------------------------------------------------------------*/
int
rtsp_new (char *idstr, void *parent, RTSPObjectType objtype)
{
  RTSP *rtsp=NULL;
  RTSP *parent_rtsp=(RTSP *)parent;

  rtsp = g_chunk_new (RTSP, rtsp_mem_chunk);

  rtsp->rtsp_id          = next_rtsp_id();
  rtsp->rtsp_idstr       = idstr;

  rtsp->rtsp_seen_first_option_request = 0;

  rtsp->rtsp_cbuff       = NULL;
  rtsp->rtsp_cbufflen    = 0;
  rtsp->rtsp_ccontentlen = 0;
  rtsp->rtsp_cmethodlen  = 0;
  rtsp->rtsp_cstate      = RTSP_STATE_EXPECTING;

  rtsp->rtsp_sbuff       = NULL;
  rtsp->rtsp_sbufflen    = 0;
  rtsp->rtsp_scontentlen = 0;
  rtsp->rtsp_smethodlen  = 0;
  rtsp->rtsp_sstate      = RTSP_STATE_EXPECTING;

  rtsp->rtsp_session     = g_hash_table_new (g_str_hash, g_str_equal);

  rtsp->rtsp_last_cseq_id = 0;
  rtsp->rtsp_last_server_cseq_id = 0;

  /* here's where we associate client/server RTSP objects */
  if (parent_rtsp)
      rtsp->rtsp_list    = parent_rtsp->rtsp_list;
  else
      rtsp->rtsp_list    = NULL;
  rtsp->rtsp_list        = g_list_append (rtsp->rtsp_list, rtsp);

  rtsp->rtsp_cseq        = g_hash_table_new (g_int_hash, g_int_equal);
  rtsp->rtsp_cseq_map    = g_hash_table_new (g_str_hash, g_str_equal);
  rtsp->rtsp_session_map = g_hash_table_new (g_str_hash, g_str_equal);

  /* flag indicating whether this RTSP object talks directly to a
     client or to a server (they no longer talk to both; it takes two
     RTSP objects to complete a connection.)  Possible values:
     SERVER_OBJECT or CLIENT_OBJECT.
   */
  rtsp->rtsp_objtype     = objtype;

  /* define default callbacks */
  rtsp->connect_to_server      = connect_to_server;
  rtsp->write_client           = write_client;
  rtsp->write_all_servers      = write_all_servers;
  rtsp->append_server          = append_server;
  rtsp->udp_client_tunnel_open = udp_client_tunnel_open;
  rtsp->udp_server_tunnel_open = udp_server_tunnel_open;
  rtsp->udp_tunnel_close       = udp_tunnel_close;

  /* define the method handlers */
  if (objtype == CLIENT_OBJECT)
    {
      rtsp->rtsp_method = new_method_handler_table(); /* assigns NULL handlers*/
      rtsp->rtsp_method[RTSP_METHOD_OPTIONS].handler  = rtsp_options_handler;
      rtsp->rtsp_method[RTSP_METHOD_SETUP].handler    = rtsp_setup_handler;
      rtsp->rtsp_method[RTSP_METHOD_TEARDOWN].handler = rtsp_teardown_handler;
      rtsp->rtsp_method[RTSP_METHOD_RESPONSE].handler = rtsp_response_handler;
#ifdef LOG_URLS_PLAYED
      rtsp->rtsp_method[RTSP_METHOD_PLAY].handler     = rtsp_play_handler;
#endif /* LOG_URLS_PLAYED */
    }
  else if (objtype == SERVER_OBJECT)
    {
      rtsp->rtsp_method = new_method_handler_table(); /* assigns NULL handlers*/
      rtsp->rtsp_method[RTSP_METHOD_RESPONSE].handler =
                rtsp_response_handler_sv;
      rtsp->rtsp_method[RTSP_METHOD_SET_PARAMETER].handler =
                rtsp_set_param_handler_sv;
    }
  else
    rtsp->rtsp_method = NULL;

  /* it helps to be able to look up the rtsp structure by id or idstr */
  g_hash_table_insert (rtsp_id_hash,    &rtsp->rtsp_id,   rtsp);
  g_hash_table_insert (rtsp_idstr_hash, rtsp->rtsp_idstr, rtsp);

  return rtsp->rtsp_id;
}



/*----------------------------------------------------------------------
   rtsp_destroy() -- Deallocate/destroy an RTSP structure/object.

   Paramaters:
     id    - A unique id indicating which RTSP object to destroy.
     idstr - A unique id string indicating which RTSP object to destroy.

   Return:  nothing
----------------------------------------------------------------------*/
void
rtsp_destroy (int id, char *idstr)
{
  RTSP *rtsp = g_hash_table_lookup (rtsp_id_hash, &id);

  g_hash_table_remove (rtsp_id_hash, &id);
  g_hash_table_remove (rtsp_idstr_hash, &idstr);

  /* free overflow buffers */
  g_free (rtsp->rtsp_cbuff);
  g_free (rtsp->rtsp_sbuff);

  /* free the cseq hash table */
  g_hash_table_foreach (rtsp->rtsp_cseq, clear_cseq_hash_node, rtsp);
  g_hash_table_destroy (rtsp->rtsp_cseq);

  /* free the session hash table */
  g_hash_table_foreach (rtsp->rtsp_session, clear_session_hash_node, rtsp);
  g_hash_table_destroy (rtsp->rtsp_session);
 
  /* free the session remapping hash table */
  g_hash_table_foreach (rtsp->rtsp_session_map, clear_session_map_node, rtsp);
  g_hash_table_destroy (rtsp->rtsp_session_map);

  g_chunk_free (rtsp, rtsp_mem_chunk);
}


/*----------------------------------------------------------------------
   next_rtsp_id() -- Allocate and return the next available RTSP id.

   Paramaters: none

   Return: The next available RTSP id.
----------------------------------------------------------------------*/
static int
next_rtsp_id (void)
{
  return ++max_rtsp_id;
}


/*----------------------------------------------------------------------
   clear_cseq_hash_node() -- A wrapper for cseq_destroy().

   Paramaters:
     key       - unused
     data      - A pointer to the CSeq structure to destroy.
     user_data - The related RTSP structure.

   Return:  nothing

   This is used by a g_hash_table_foreach() call.
----------------------------------------------------------------------*/
static void
clear_cseq_hash_node (void * key, void * data, void * user_data)
{
  cseq_destroy ((RTSP *) user_data, (CSeq *) data); 
}



/*----------------------------------------------------------------------
   clear_session_hash_node() -- A wrapper for cseq_destroy().

   Paramaters:
     key - unused
     data - A pointer to the CSeq structure to destroy.
     user_data - The related RTSP structure.

   Return:  nothing

   This is used by a g_hash_table_foreach() call.
----------------------------------------------------------------------*/
static void
clear_session_hash_node (void * key, void * data, void * user_data)
{
  cseq_destroy ((RTSP *) user_data, (CSeq *) data);
}



/*----------------------------------------------------------------------
   clear_session_map_node() -- A wrapper for session_map_destroy().

   Paramaters:
     key - unused
     data - A pointer to the SessionIDMapping structure to destroy.
     user_data - The related RTSP structure.

   Return:  nothing

   This is used by a g_hash_table_foreach() call.
----------------------------------------------------------------------*/
static void
clear_session_map_node (void * key, void * data, void * user_data)
{
  session_map_destroy ((SessionIDMapping *) data);
}



/*======================================================================*/

/*----------------------------------------------------------------------
   rtsp_client_read() -- A wrapper for rtsp_read().

   Paramaters:
     rtspid - The id of the RTSP structure processing the data.
     buff   - A buffer of data to process.
     n      - The size of the buffer.

   Return:
     -1 on error
      0 on success

   This is for processing data coming from the client.
   The RTSP structure/object may be a CLIENT_OBJECT or a SERVER_OBJECT.
----------------------------------------------------------------------*/
int
rtsp_client_read (int rtspid, char *buff, int n)
{
  return rtsp_read (FROM_CLIENT, rtspid, buff, n);
}



/*----------------------------------------------------------------------
   rtsp_server_read() -- A wrapper for rtsp_read().

   Paramaters:
     rtspid - The id of the RTSP structure processing the data.
     buff   - A buffer of data to process.
     n      - The size of the buffer.

   Return:
     -1 on error
      0 on success

   This is for processing data coming from the server.
   The RTSP structure/object may be a CLIENT_OBJECT or a SERVER_OBJECT.
----------------------------------------------------------------------*/
int
rtsp_server_read (int rtspid, char *buff, int n)
{
  return rtsp_read (FROM_SERVER, rtspid, buff, n);
}



/*----------------------------------------------------------------------
   rtsp_read() -- This processes a buffer of RTSP Control data.

   Paramaters:
     from   - The direction the data is coming from, FROM_CLIENT or FROM_SERVER.
     rtspid - The id of the RTSP structure processing the data.
     buff   - A buffer of data to process. 
     n      - The size of the buffer.

   Return:
     -1 on error
      0 on success

   This is the heart of the RTSP state engine.  It processes the RTSP
   data in the buffer, calling method handlers, and either writes the
   (possibly) transmogrified RTSP data to the other party, or hands it
   off to another RTSP object which will complete the job.  Currently,
   most of the method-handler processing happens in the client RTSP
   object, the server objects serve primarily as a buffering mechanism
   to ensure we don't scramble RTSP data to/from multiple servers,
   and to remap certain values.

   Note that this uses "content length" in two ways:  the content of an
   RTSP message (with length indicated by the Content-Length: header) and
   the length of content embedded in the TCP stream using '$'-encoding.

   This state engine stores any state information it needs within the RTSP
   object itself, so the processing of multiple RTSP Control conversations
   may be interleaved without causing conflicts.  This is important for
   handling multiple client<-->server connections over a single
   client<-->proxy connection.
----------------------------------------------------------------------*/
static int
rtsp_read (RTSPFromType from, int rtspid, char *buff, int n)
{
  char mbuff[MAX_TCP_BUFF];   /* holds an individual RTSP method */
  char obuff[MAX_TCP_BUFF];   /* output buffer for writing to client */
  char rbuff[MAX_TCP_BUFF];   /* read buffer, where incoming data is located */
  int rlen=0, olen=0, mlen=0; /* lengths of data in three buffers above */
  int pos=0;                  /* location of substring within a string */
  RTSP *rtsp=NULL;            /* The RTSP object processing this buffer */
  RTSPMethodType method;      /* The type of RTSP method in the buffer */
  char *rtsp_buff=NULL;       /* method data carried over from previous call */
  int rtsp_bufflen=0;         /* length of data in rtsp_buff */
  int rtsp_contentlen=0;      /* length of content associates with this method*/
  int rtsp_methodlen=0;       /* length of method **header** */
  int rtsp_embededlen=0;      /* length of $-embedded data */
  RTSPStateType rtsp_state;   /* current state */

  rtsp = g_hash_table_lookup (rtsp_id_hash, &rtspid);
  if (rtsp == NULL)
    {  log_message (ERROR_LOG, "rtcp_read: rtsp object not found for id=%d", rtspid);
       return -1;
    }


  if (from == FROM_CLIENT)
    {
      rtsp_buff       = rtsp->rtsp_cbuff;
      rtsp_bufflen    = rtsp->rtsp_cbufflen;
      rtsp_contentlen = rtsp->rtsp_ccontentlen;
      rtsp_methodlen  = rtsp->rtsp_cmethodlen;
      rtsp_embededlen = rtsp->rtsp_cembededlen;
      rtsp_state      = rtsp->rtsp_cstate;
    }
  else
    {
      rtsp_buff       = rtsp->rtsp_sbuff;
      rtsp_bufflen    = rtsp->rtsp_sbufflen;
      rtsp_contentlen = rtsp->rtsp_scontentlen;
      rtsp_methodlen  = rtsp->rtsp_smethodlen;
      rtsp_embededlen = rtsp->rtsp_sembededlen;
      rtsp_state      = rtsp->rtsp_sstate;
    }
 
  /* Append incoming buffer to current buffer, fail on overflow;
   * this check may not be sufficent, because the rtsp method
   * handlers are free to extend this buffer later on.
   *
   * If this overflow happens with the current MAX_TCP_BUFF limit,
   * someone is most likely messing with us.
   */
  if (rtsp_bufflen + n > MAX_TCP_BUFF)
    {  log_message (ERROR_LOG, "rtcp_read: TCP buffer limit exceeded; %d > %d", rtsp_bufflen + n, MAX_TCP_BUFF);
       return -1;
    }


  /* Copy the overflow information from the last read into
   * the read working buffer, then append the newly read buffer
   * into the working buffer.
   */
  if ( (rlen = rtsp_bufflen) > 0)
    memcpy (rbuff, rtsp_buff, rlen);


  /* copy the incoming buffer into the read buffer */
  memcpy (rbuff + rlen, buff, n);
  rlen += n;


  /* set output, method buffer length */
  olen = 0;
  mlen = 0;


  while (1)
    {    /* we'll break out of this loop when we can't process any more data */

      char tbuff[MAX_TCP_BUFF]; /* a temporary buffer */
      int tlen=0;               /* the length of tbuff */

      /* log_message (DEBUG_LOG,
                      "BUFFER-READ ID:%d STATE:%d FROM:%d OBJTYPE:%d",
                      rtsp->rtsp_id, rtsp_state, from, rtsp->rtsp_objtype); */

      /* BUFFER-READ SWITCH */
      switch (rtsp_state)
	{

	/* readline state */
	case RTSP_STATE_EXPECTING:

          /*log_message (DEBUG_LOG, "BUFFER-READ CASE: RTSP_STATE_EXPECTING");*/

	  /* look for binary data (encoded with preceeding '$') */
	  if (rlen > 0 && rbuff[0] == '$')
	    {


	      if (rlen >= 4)
		{
                  rtsp_embededlen = ntohs (*(unsigned short *)(&rbuff[2]));
		}

	      if (rlen >= rtsp_embededlen + 4)
		{
                  log_message (DEBUG_LOG,
                               "PROCESSING EMBEDDED BINARY DATA OF LENGTH %d",
                               rtsp_embededlen);

                  if (from == FROM_SERVER && rtsp->rtsp_objtype == SERVER_OBJECT)
                      rbuff[1] = rtsp_get_channel_id (rtsp, rbuff[1]);

		  tlen = rtsp_embededlen + 4;
		  memcpy (tbuff, rbuff, tlen);
		  
		  rlen -= tlen;
		  memmove (rbuff, rbuff + tlen, rlen);
               	 
		}
	      else
		{

		  /* we don't have the full inline-data yet,
		   * so we break out of this read until we recieve
		   * our next chunk of data; we also need to store
		   * what we've got for next time
		   */
		  tlen = 0;
		  
                  log_message (DEBUG_LOG,
                               "SAVING %d BYTES OF EMBEDDED DATA (SIZE %d) FOR LATER",
                               rlen, rtsp_embededlen);

		}


	      break;

	    }
	  


	/* notice no break from RTSP_STATE_EXPECTING */
	case RTSP_STATE_METHOD:

          /* log_message (DEBUG_LOG, "BUFFER-READ CASE: RTSP_STATE_METHOD"); */

	  if ((pos = cistrnfind ("\r\n", 2, rbuff, rlen)) >= 0)
	    {

	      tlen = pos + 2;
	      memcpy (tbuff, rbuff, tlen);

	      rlen -= tlen;
	      memmove (rbuff, rbuff + tlen , rlen);

	    }
	  else
	    {
	      tlen = 0;
	    }

	  break;



	     /* for buffering data when a method has a Content-Length header */
	case RTSP_STATE_CONTENT:

           /* log_message (DEBUG_LOG,
                           "BUFFER-READ CASE: RTSP_STATE_CONTENT %d %d %d",
                           rtsp_contentlen, rtsp_methodlen, rlen); */

	  if ( rtsp_contentlen + rtsp_methodlen <= rlen)
            {
              tlen = rtsp_contentlen + rtsp_methodlen;
              memcpy (tbuff, rbuff, tlen);
              rlen -= tlen;
              if (rlen > 0)
                memmove (rbuff, rbuff + tlen, rlen);
            }
          else
            {
	      /* otherwise we keep buffering */
              tlen = 0;
            }

          break;




	/* read buffer n */
	case RTSP_STATE_FORWARD:

           /* log_message (DEBUG_LOG,
                           "BUFFER-READ CASE: RTSP_STATE_FORWARD %d %d %d",
                           rtsp_contentlen, rtsp_methodlen, rlen); */

	  if (rtsp_contentlen <= rlen)
	    {
              tlen = rtsp_contentlen;

	      memcpy (tbuff, rbuff, tlen);
	      rlen -= tlen;

	      if (rlen > 0)
		memmove (rbuff, rbuff + tlen , rlen);
	    }

	  break;

	}


      /* log_message (DEBUG_LOG, "BUFFER-WRITE STATE: %d", rtsp_state); */

      /* we can't read any more from the buffer */      
      if (tlen <= 0)
	  break;


      /* BUFFER-WRITE SWITCH */
      switch (rtsp_state)
	{
	  
	case RTSP_STATE_EXPECTING:

           /* log_message (DEBUG_LOG,
                           "BUFFER-WRITE CASE: RTSP_STATE_EXPECTING"); */

	  /* binary inline data (encoded with preceeding '$') */
	  if (tbuff[0] == '$')
	    {
              /* we only append the embedded data to the output buffer
                 when we have a complete $-buffer to avoid writing garbage
                 when merging traffic from multiple servers
               */
	      if (tlen >= rtsp_embededlen + 4)
                {
                  log_message (DEBUG_LOG,
                               "COPYING %d+4 BYTES OF EMBEDDED DATA "
                               "TO OUTPUT BUFFER",
                               rtsp_embededlen);

                  append_output_buffer (rtsp, from, tbuff, tlen, obuff, &olen);

                }

	      break;

	    }


	  /* RTSP methods */

          /* log_message (DEBUG_LOG, "rtsp_method=0x%x", rtsp->rtsp_method); */

          if (! rtsp->rtsp_method ||
              (method = method_type (rtsp, tbuff, tlen)) != -1)
	    {
	      rtsp_state = RTSP_STATE_METHOD;
	      mlen = tlen;
	      memcpy (mbuff, tbuff, tlen);
	    }
	  else
	    {
#ifdef DEBUG
	      fprintf (stderr, "** UNKNOWN METHOD ** [olen=%d tlen=%d]\n", olen, tlen);
	      write(1, obuff, olen);
	      fprintf (stderr, "\n----\n");
	      write(1, tbuff, tlen);
#endif /* DEBUG */
	      return -1;
	    }
	  
          break;



	case RTSP_STATE_METHOD:

          /* log_message (DEBUG_LOG, "BUFFER-WRITE CASE: RTSP_STATE_METHOD"); */

	  memcpy (mbuff + mlen, tbuff, tlen);
	  mlen += tlen;

	  if (tlen == 2 && cistrncmp ("\r\n", tbuff, 2)) /* END OF METHOD HDR */
	    {	 
	      /* Call handler for method; errors from the rtsp methods
	       * are returned, so if a handler returns an error,
	       * we return -1 which shuts the connection down.
               *
               * Notice that we call the handler before we've read the
               * associated content (if any).  It might be better to wait.
	       */
	      if (rtsp->rtsp_method)
                if ((method = method_type (rtsp, mbuff, mlen)) != -1)
		  {

                  /* for server RTSP Responses we remap the CSeq and session
                     IDs on the way in, before calling the method handler.
                   */
                    if (rtsp->rtsp_objtype == SERVER_OBJECT &&
                        from == FROM_SERVER)
                      {
                        rtsp_remap_cseq_id (rtsp, from, method, mbuff, &mlen);
                        rtsp_session_update (rtsp, from, mbuff, &mlen);
                      }

                    if (rtsp->rtsp_method[method].handler != NULL)
                      {
                        log_message (DEBUG_LOG,
				     "calling %s object's handler for %s "
                                     "mbuff=0x%x mlen=%d rtspid=%d",
				     ((rtsp->rtsp_objtype == SERVER_OBJECT) ?
                                       "server" : "client") ,
				     rtsp->rtsp_method[method].text, mbuff,
                                     mlen, rtsp->rtsp_id);

		        if ((* rtsp->rtsp_method[method].handler) (rtsp, from, mbuff, &mlen) != 0)
		          return -1;

                        log_message (DEBUG_LOG,
                              "returned from handler for %s mbuff=0x%x mlen=%d",
                              rtsp->rtsp_method[method].text, mbuff, mlen);
		      }
                    else
                      log_message (DEBUG_LOG,
				   "ignoring %s object's NULL handler for %s",
                                   ((rtsp->rtsp_objtype == SERVER_OBJECT) ?
                                       "server" : "client") ,
				   rtsp->rtsp_method[method].text);


                  /* for client RTSP Requests we remap the CSeq and session
                     IDs on the way out, after calling the method handler.
                     This keeps the remapping symetrical, and the CSeq
                     numbers and session IDs we normally use internally are
                     the (unique) client versions.
                   */
                    if (rtsp->rtsp_objtype == SERVER_OBJECT &&
                        from == FROM_CLIENT)
                      {
                        rtsp_remap_cseq_id (rtsp, from, method, mbuff, &mlen);
                        rtsp_session_update (rtsp, from, mbuff, &mlen);
                      }


		  }


	      /* check for content-length in rtsp method */
	      if ( (pos = cistrnfind ("content-length:", 15, mbuff, mlen)) >= 0)
		{
		  rtsp_contentlen = atoi (&mbuff[pos + 16]);
		  rtsp_state      = RTSP_STATE_FORWARD;
		  rtsp_methodlen  = mlen;
                  /* log_message (DEBUG_LOG, "CONTENT-LEN: %d mlen=%d rlen=%d",
                                  rtsp_contentlen, mlen, rlen); */
                  if (rtsp_contentlen > rlen)
		      rtsp_state  = RTSP_STATE_CONTENT;
		}
	      else
		{
		  rtsp_state = RTSP_STATE_EXPECTING;
		}

              if (rtsp_state != RTSP_STATE_CONTENT)
                {
                  append_output_buffer (rtsp, from, mbuff, mlen, obuff, &olen);
	          mlen = 0;
                }
	    }

            /* log_message (DEBUG_LOG, "id:%d state:%d mlen:%d", rtsp->rtsp_id,
                            rtsp_state, mlen); */


	  break;
	  

	  
	case RTSP_STATE_CONTENT:
          /* log_message (DEBUG_LOG, "BUFFER-WRITE CASE: RTSP_STATE_CONTENT");*/

          append_output_buffer (rtsp, from, tbuff, tlen, obuff, &olen);

	  rtsp_state = RTSP_STATE_EXPECTING;

	  break;



	case RTSP_STATE_FORWARD:
	  
          /* log_message (DEBUG_LOG,
                          "BUFFER-WRITE CASE: RTSP_STATE_FORWARD %d %d",
                          rtsp_contentlen, tlen);*/

          append_output_buffer (rtsp, from, tbuff, tlen, obuff, &olen);

	  rtsp_contentlen -= tlen;

	  if (rtsp_contentlen == 0)
	    rtsp_state = RTSP_STATE_EXPECTING;
	  
	  break;
	  
	}

        /* log_message (DEBUG_LOG, "LOOP STATE: %d", rtsp_state); */

    }
     

  /* if we're in the middle of a method we store
   * what we've got of the method and wait for the next
   * read
   */
  if ( (mlen + rlen) > 0)
    {
      rtsp_buff    = g_malloc (mlen + rlen);
      rtsp_bufflen = 0;

      if (mlen > 0)
	{
	  memcpy (rtsp_buff + rtsp_bufflen, mbuff, mlen);
	  rtsp_bufflen += mlen;
	}
      
      if (rlen > 0)
	{
	  memcpy (rtsp_buff + rtsp_bufflen, rbuff, rlen);
	  rtsp_bufflen += rlen;
	}
    }
  else
    {
      rtsp_buff    = NULL;
      rtsp_bufflen = 0;
    }


  /* log_message (DEBUG_LOG, "olen:%d mlen:%d rlen:%d buff=0x%x", olen, mlen,
                  rlen, rtsp_buff); */


  if (from == FROM_CLIENT)
    {
      /* buffer comes from the client side of the universe */

      g_free  (rtsp->rtsp_cbuff);
      
      rtsp->rtsp_cbuff       = rtsp_buff;
      rtsp->rtsp_cbufflen    = rtsp_bufflen;
      rtsp->rtsp_ccontentlen = rtsp_contentlen;
      rtsp->rtsp_cmethodlen  = rtsp_methodlen;
      rtsp->rtsp_cembededlen = rtsp_embededlen;
      rtsp->rtsp_cstate      = rtsp_state;

      /* show_rtsp_conn_buffs();*/ /* for debugging */
      (* rtsp->write_all_servers) ();

      return (olen);

    }
  else
    {
      /* buffer comes from the server side of the universe */

      g_free (rtsp->rtsp_sbuff);

      rtsp->rtsp_sbuff       = rtsp_buff;
      rtsp->rtsp_sbufflen    = rtsp_bufflen;
      rtsp->rtsp_scontentlen = rtsp_contentlen;
      rtsp->rtsp_smethodlen  = rtsp_methodlen;
      rtsp->rtsp_sembededlen = rtsp_embededlen;
      rtsp->rtsp_sstate      = rtsp_state;

      /* for server objects we've already appended the data to the client
       * objects input buffer.
       */
      if (rtsp->rtsp_objtype == CLIENT_OBJECT)
        return ((* rtsp->write_client) (rtsp->rtsp_id, obuff, olen));
    }

  return 0;
}



/*----------------------------------------------------------------------
   append_output_buffer() -- Pass a buffer of data coming from an RTSP object.

   Paramaters:
     rtsp   - The RTSP structure/object the buffer of data is coming from.
     from   - The direction the data is coming from, FROM_CLIENT or FROM_SERVER.
     buff   - A buffer of data to process. 
     len    - The size of the buffer.
     obuff  - When writing to the client we simply append to this buffer
                (which will be written to the network elsewhere).
     olen   - The length of the output buffer.

   Return:  nothing

   Depending on the context, this either appends the buffer to obuff
   (which is data ready to write to the client) or appends the buffer
   to the output buffer for the appropriate server (the CtrlConn buffers 
   in child-main.c).
   
   We have messages coming from the client and from the server, and
   we have an object connected to the client and at least one object
   connected to a server.  The "from" indicates the direction the
   message is going (from the client or from the server) and
   rtsp->rtsp_objtype indicates which side of the proxy we're processing
   the message on, the server side or the client side.  Each message
   gets processed by both sides of the proxy by passing through exactly
   one client RTSP object and one server RTSP object.  Depending on which
   combination of these we're currently in, this routine either passes
   the buffer to the other side of the proxy or writes it to the network,
   either to the client or the server.

   MESSAGE FROM      OBJECT TYPE       ACTION
   client            client            pass buffer to server object
   client            server            write buffer to server
   server            server            pass buffer to client object
   server            client            write buffer to client

----------------------------------------------------------------------*/
static void
append_output_buffer (RTSP *rtsp, RTSPFromType from, char *buff, int len, char *obuff, int *olenp)
{

  /* log_message (DEBUG_LOG,
                  "append_output_buffer: id:%d from:%d buff:0x%x len:%d "
                  "obuff:0x%x olen:%d",
                  rtsp->rtsp_id, from, buff, len, obuff, olenp ? *olenp : 0); */

  if (len <= 0)
    return;

  if (from == FROM_CLIENT)  /* the RTSP message is coming from the client */
    {
      if (rtsp->rtsp_objtype == SERVER_OBJECT)
        {
          /* if this is a server RTSP object, append the buffer to the real
             output buffer space which will get written at a later point (with
             no further processing of the data)
           */
          log_message (DEBUG_LOG, "WRITING %d BYTES TO SERVER ID=%d",
                       len, rtsp->rtsp_id);
          (* rtsp->append_server) (rtsp->rtsp_id, buff, len);
        }
      else
        {
          int id = extract_serverid (buff, len); /* determine dest. server */
          if (id)
            {  /* if this is a client RTSP object we find the appropriate server
                  object and append the buffer to it's client-input buffer.
                */
               log_message (DEBUG_LOG,
                            "PASSING %d BYTES TO SERVER OBJECT ID=%d", len, id);

               /* now, force the message to be processed by the server RTSP
                  object, which will cause it to be written to the server.
                */
               rtsp_read (from, id, buff, len); /* recursion */

            }
          else
            {
               /* don't have any other way to determine which server
                  this is for.
                */
              log_message (DEBUG_LOG,
                           "couldn't determine server's RTSP object buffer "
                           "so can't send to server");
            }
        }
    }
  else  /* RTSP message is coming from the server */
    {
      if (rtsp->rtsp_objtype == SERVER_OBJECT)
        {
           /* if this is a server object we append the buffer to
              the client object's input buffer
            */
               RTSP *rtsp2=NULL;

               rtsp2 = g_hash_table_lookup (rtsp_idstr_hash, CLIENT_IDSTR);
               log_message (DEBUG_LOG,
                            "PASSING %d BYTES TO CLIENT OBJECT ID=%d",
                            len, rtsp2->rtsp_id);

               /* force the message to be processed by the client RTSP
                  object, which will cause it to be written to the client.
                */
               rtsp_read (from, rtsp2->rtsp_id, buff, len); /* recursion */

        }
      else
        { /* if this is a client object we append the buffer to the
             the real output buffer which will get written at a later
             point without further processing
           */
          log_message (DEBUG_LOG, "WRITING %d BYTES TO CLIENT ID=%d",
                       len, rtsp->rtsp_id);
          memcpy (obuff + *olenp, buff, len);
          *olenp += len;
        }
    }
}



/*----------------------------------------------------------------------
   extract_serverid() -- Extract an RTSP id from a RTSP method buffer.

   Paramaters:
     buff - The buffer containing an RTSP method to examine.
     len  - The length of the buffer.

   Return:
     The RTSP id for the server indicated by the buffer.
     Zero is returned if a "rtsp://server:port" is not found in
     the first line of the buffer.
----------------------------------------------------------------------*/
int
extract_serverid (char *buff, int len)
{
   char *idstr=NULL;
   RTSP *rtsp=NULL;
   char hbuff[HOSTNAME_MAX]="";
   unsigned short port=0;
   int ret=0, i=0;

   ret = extract_host_and_port (buff, len, hbuff, HOSTNAME_MAX, &port);

   /* log_message (DEBUG_LOG, "extract_serverid: ret=%d host=%s port=%d",
                   ret, hbuff, port); */

   /* If we couldn't extract the host:port but there's only one server,
      assume that's it.
    */
   if (ret == 1 && last_rtsp_id() == 2)
     return 2;

   /* If we couldn't extract the host:port return an error */
   if (ret == 1)
     return 0;

   /* Generate a "host:port" string for looking up the rtsp object */
   idstr = g_malloc (HOSTNAME_MAX);
   snprintf (idstr, HOSTNAME_MAX, "%s:%d", hbuff, port);
   idstr[HOSTNAME_MAX-1] = '\0';
   for (i=0; i < HOSTNAME_MAX && idstr[i]; ++i)
     idstr[i] = tolower(idstr[i]);

   /* Look for the rtsp object in the hash rtsp_idstr_hash */
   rtsp = g_hash_table_lookup (rtsp_idstr_hash, idstr);
   log_message (DEBUG_LOG, "extract_serverid: idstr=%s rtsp=0x%x", idstr, rtsp);

   /* cleanup */
   g_free (idstr);

   /* return the RTSP id */
   if (rtsp)
     return (rtsp->rtsp_id);
   else
     return 0;
}


/*----------------------------------------------------------------------
   extract_host_and_port() -- Extract the host and port from the first line
                              of a method.

   Paramaters:
     buff    - The buffer containing an RTSP method to examine.
     len     - The length of the buffer.
     hbuff   - A buffer in which to copy the embedded hostname.
     hmaxlen - The maximum length of the hostname to copy into hbuff.
     portp   - A pointer to the integer which will be assigned the
               port value found.

   Return:
     1 on error
     0 on success

   Methods passing through a proxy normally have a "rtsp://host:port" 
   in the first line.
----------------------------------------------------------------------*/
int
extract_host_and_port (char *buff, int len, char *hbuff, int hmaxlen, unsigned short *portp)
{
  int pos=0, tlen=0, i=0, eol=0;
  char *tbuff=NULL;

  /* find the end of the first line of the method */
  if ((eol = cistrnfind ("\r\n", 2, buff, len)) < 0)
    return 1;

  /* find the rtsp://host:port string within this first line */
  if ((pos = cistrnfind ("rtsp://", 7, buff, eol)) < 0)
    return 1;

  /* skip the "rtsp://" */
  pos   += 7;
  tbuff = buff + pos;
  tlen  = len - pos;

  /* this is a bad RTSP message, return error */
  if (tlen == 0)
    return 1;

  /* read until the end of the hostname */
  for (i = 0; i < tlen; i++)
    if (ISWHITE(tbuff[i]) || ISEND(tbuff[i]) || tbuff[i] == ':')
      break;

  /* buffer overrun control on hbuff */
  if (i >= hmaxlen - 2)
    return 1;

  /* copy the hostname into hbuff, add a null termination on the end */
  memcpy (hbuff, tbuff, i);
  hbuff[i] = '\0';

  /* reset the tbuff pointer to the end of the hostname portion
   * of the URL; set it to point where the ":" should be.
   */
  tbuff += i;
  tlen  -= i;

  /* extract the port if present */
  if (tlen > 0 && tbuff[0] == ':')
    {
      tbuff++; tlen--;

      /* check for atoi short casting overrun */
      if (tlen > 0)
        *portp = (unsigned short) atoi(tbuff);
    }
  else  /* otherwise, use the default RTSP port */
    *portp = DEFAULT_SERVER_PORT;

  /* log_message (DEBUG_LOG, "extract_host_and_port(): host=%s port=%d",
                  hbuff, *portp); */

  return 0;

}



/*----------------------------------------------------------------------*/


/* RTSP METHOD HANDLERS */

/* METHOD: OPTIONS */

/*----------------------------------------------------------------------
   rtsp_options_handler() -- A handler for the OPTIONS method.

   Paramaters:
     rtsp - The related rtsp object.
     from - The direction the message is traveling, FROM_CLIENT or FROM_SERVER.
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.

   Return:
     1 on error
     0 on success

   This is only intended to be called with a CLIENT_OBJECT.
----------------------------------------------------------------------*/
int
rtsp_options_handler (RTSP * rtsp, RTSPFromType from, char * buff, int * n)
{
  char hbuff[HOSTNAME_MAX];
  unsigned short sport = DEFAULT_SERVER_PORT;
  int i=0, new_rtspid=0;
  char *idstr=NULL;
  RTSP *rtsp2=NULL;

  if (rtsp->rtsp_objtype != CLIENT_OBJECT)
    {
      log_message (ERROR_LOG,
          "error: client OPTIONS method handler called with a server object");
      return 0;
    }

  hbuff[0] = '\0';
  if (extract_host_and_port (buff, *n, hbuff, HOSTNAME_MAX, &sport) == 1)
    return 0;

  idstr = g_malloc (HOSTNAME_MAX);
  snprintf (idstr, HOSTNAME_MAX, "%s:%d", hbuff, sport);
  idstr[HOSTNAME_MAX-1] = '\0';
  for (i=0; i < HOSTNAME_MAX && idstr[i]; ++i)
    idstr[i] = tolower(idstr[i]);
  if ((rtsp2 = g_hash_table_lookup (rtsp_idstr_hash, idstr)))
    {
      log_message (DEBUG_LOG,
                  "OPTIONS handler found existing server object id=%d idstr=%s",
                  rtsp2->rtsp_id, idstr);

      return 0;

    }
  else
    {
      /* we haven't seen this server before, so we need to 
         create an RTSP object for it
       */
      new_rtspid = rtsp_new (idstr, rtsp, SERVER_OBJECT);
      rtsp2 = g_hash_table_lookup (rtsp_idstr_hash, idstr);
      log_message (DEBUG_LOG,
                 "OPTIONS handler created server object id=%d idstr=%s rtsp=%x",
                 rtsp2->rtsp_id, rtsp2->rtsp_idstr, rtsp2);
      rtsp->rtsp_seen_first_option_request = 1;
      rtsp = rtsp2;
      rtsp->rtsp_seen_first_option_request = 1;

      /* note: now rtsp points to the new server object! */

    }


  /* make the callback */
  (* rtsp->connect_to_server) (hbuff, sport, rtsp->rtsp_id);


  /* we've now seen the first options request, ignore all others
     at least as far as setting up ports is concerned
   */
  rtsp->rtsp_seen_first_option_request = 1;

  return 0;
}


/* METHOD: SETUP */

/*----------------------------------------------------------------------
   rtsp_setup_handler() -- A handler for the SETUP method.

   Paramaters:
     rtsp - The related rtsp object.
     from - The direction the message is traveling, FROM_CLIENT or FROM_SERVER.
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.

   Return:
     1 on error
     0 on success

   This is only intended to be called with a CLIENT_OBJECT.
----------------------------------------------------------------------*/
int
rtsp_setup_handler (RTSP * rtsp, RTSPFromType from, char * buff, int * n)
{
  int i=0;
  CSeq *cseq=NULL;
  GList *list=NULL, *transport_list=NULL;
  Transport *transport=NULL;

  if (rtsp->rtsp_objtype != CLIENT_OBJECT)
    {
      log_message (ERROR_LOG,
          "error: client SETUP method handler called with a server object");
      return 0;
    }

  /* filter out multicast transports and anything else we don't handle */
  rtsp_remove_transports (rtsp, buff, n);

  for (i = 0; i < TRANSPORT_LAST; i++)
    {
      if (rtsp_transports[i].get_ports)
        transport = (* rtsp_transports[i].get_ports) (rtsp, i, buff, *n);

      if (transport)
	transport_list = g_list_append (transport_list, transport);
    }

  /* no supported transports */
  if (!transport_list)
    return 0;

  /* allocate a sequence structure for this setup message */
  cseq = cseq_new (rtsp);
  cseq->cseq_transports = transport_list;
  cseq->cseq_srtspid = extract_serverid (buff, *n);


  if (! (cseq->cseq_id = get_cseq (buff, *n)))
    goto cleanup;
  
  g_hash_table_insert (rtsp->rtsp_cseq, &cseq->cseq_id, cseq);

  /* modifify the SETUP method text to contain the
   * allocated ports on the proxy instead of the ones
   * on the client
   */
  list = transport_list;
  while (list && list->prev)  /* ensure we start at the head of the list... */
      list = list->prev;
  while (list)
    {
      transport = list->data;
      list      = list->next;

      if (rtsp_transports[transport->transport_type].set_ports)
        (* rtsp_transports[transport->transport_type].set_ports) (rtsp, transport, FROM_CLIENT, buff, n);
    }


  return 0;



  /* cleanup on error */
 cleanup:

  if (cseq)
    cseq_destroy (rtsp, cseq);

  return 1;
}



/* METHOD: SET_PARAMATER */

/*----------------------------------------------------------------------
   rtsp_set_param_handler_sv() -- A handler for the SET_PARAMATER method.

   Paramaters:
     rtsp - The related rtsp object.
     from - The direction the message is traveling, FROM_CLIENT or FROM_SERVER.
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.

   Return:
     1 on error
     0 on success

   This is only intended to be called with a SERVER_OBJECT.
   The server sends keep-alive "Ping: Pong" messages to test that the
   client is still listening.  We need to record that a given server
   has pinged the client so that when the client replies we can
   tell where to send the response (in case there are multiple servers).
----------------------------------------------------------------------*/
int
rtsp_set_param_handler_sv (RTSP * rtsp, RTSPFromType from, char * buff, int * n)
{
  CSeq *cseq=NULL;

  if (from == FROM_SERVER)
    {
      if (cistrnfind ("\r\nping: pong\r\n", 14, buff, *n))
        {
          cseq = cseq_new (rtsp);
          if ( (cseq->cseq_id = get_cseq (buff, *n)) <= 0)
            cseq_destroy (rtsp, cseq);
          else
            {
              cseq->cseq_ping = 1;
              g_hash_table_insert (rtsp->rtsp_cseq, &cseq->cseq_id, cseq);
            }

          log_message (DEBUG_LOG, "CSEQ_ID:%d CSEQ:%x", cseq->cseq_id, cseq);
        }

    }

  return 0;

}



/* METHOD: TEARDOWN */

/*----------------------------------------------------------------------
   rtsp_teardown_handler() -- A handler for the TEARDOWN method.

   Paramaters:
     rtsp - The related rtsp object.
     from - The direction the message is traveling, FROM_CLIENT or FROM_SERVER.
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.

   Return:
     1 on error
     0 on success

   This is only intended to be called with a CLIENT_OBJECT.
----------------------------------------------------------------------*/
int
rtsp_teardown_handler (RTSP * rtsp, RTSPFromType from, char * buff, int * n)
{
  char *session=NULL;

  if (rtsp->rtsp_objtype != CLIENT_OBJECT)
    {
      log_message (ERROR_LOG,
          "error: client TEARDOWN method handler called with a server object");
      return 0;
    }

  if (! (session = get_session (buff, *n)))
    return 1;

  rtsp_remove_session (rtsp, session);

  /* free the returned session buffer */
  g_free (session);

  return 0;
}



/* RTSP RESPONSE HANDLER */

/*----------------------------------------------------------------------
   rtsp_response_handler() -- A handler for RTSP responses.

   Paramaters:
     rtsp - The related rtsp object.
     from - The direction the message is traveling, FROM_CLIENT or FROM_SERVER.
     buff - A buffer containing the message being processed.
     n    - The size of the buffer.

   Return:
     1 on error
     0 on success

   This is only intended to be called with a CLIENT_OBJECT.
   We process RTSP responses coming from the server(s) in the client object.
----------------------------------------------------------------------*/
int
rtsp_response_handler (RTSP * rtsp, RTSPFromType from, char * buff, int * n)
{
  int cseq_id=0;
  GList *list=NULL, *oldlist=NULL;
  CSeq *cseq=NULL;
  Transport *transport=NULL;

  if (rtsp->rtsp_objtype != CLIENT_OBJECT)
    {
      log_message (ERROR_LOG,
          "error: client object handler called with a server object:  "
          "RTSP response");
      return 0;
    }

  /* parse out sequence ID */
  if (! (cseq_id = get_cseq (buff, *n)))
    return 0;
  

  /* Handle the "Ping: Pong" messages */

  if (from == FROM_CLIENT)
    {
      if (cistrnfind ("451 parameter not understood\r\n", 30, buff, *n) >= 0)
        {

          log_message (DEBUG_LOG,
                       "CLIENT SENT A 'PARAMATER NOT UNDERSTOOD' MESSAGE");

          list = rtsp->rtsp_list;
          while (list->prev)
            list = list->prev;
          while (list)
            {
              RTSP *rtsp2 = list->data;
              CSeq *cseq2 = g_hash_table_lookup (rtsp2->rtsp_cseq, &cseq_id);

              /* log_message (DEBUG_LOG, "list:%x rtsp:%x cseq:%x cseqid:%d hash:%x", list, rtsp2, cseq2, cseq_id, rtsp2->rtsp_cseq);*/

              if (cseq2)
                {

                  /* log_message (DEBUG_LOG, "ping:%d", cseq2->cseq_ping); */

                  if (cseq2->cseq_ping > 0)
                    {
                      log_message (DEBUG_LOG,
                                   "SERVER WITH ID=%d SENT THE PING MESSAGE",
                                   rtsp2->rtsp_id);
                      cseq_destroy (rtsp2, cseq2);
                      cseq2->cseq_ping = 0;

                      /* since the message doesn't indicate which
                         server to send this too, we must use the fact
                         that a server pinged the client to know where
                         the reply should go.  Therefore, we do this here
                         rather in the usual place (since there we wouldn't
                         know where to send it).
                       */
                      rtsp_read (from, rtsp2->rtsp_id, buff, *n);/* recursion */
                      *n = 0;

                      list = NULL;
                    }
                }
              if (list) 
                list = list->next;
            }

          return 0;

        }
    }


  /* look for the sequence entry in the sequence hash table */
  if (! (cseq = g_hash_table_lookup (rtsp->rtsp_cseq, &cseq_id)))
    return 0;


  /* only one transport will be selected by the server,
   * so we'll run the set_ports method on all the transports
   * setup by the client, and remove the ones not used by the
   * server.
   */
  list = cseq->cseq_transports;
  while (list && list->prev)  /* ensure we start at the head of the list... */
      list = list->prev;
  while (list)
    {
      int res=0;

      transport = list->data;
      oldlist   = list;
      list      = list->next;

      /* log_message (DEBUG_LOG,
                      "checking transport 0x%x %d", transport,
                      transport->transport_type); */

      if (rtsp_transports[transport->transport_type].set_ports)
        res = (* rtsp_transports[transport->transport_type].set_ports) (rtsp, transport, FROM_SERVER, buff, n);

      if (res == -1) /* error */
	return 1;

      if (res == 0) /* transport unused */
	{
	  /* log_message (DEBUG_LOG, "removing transport 0x%x %d", transport,
                          transport->transport_type); */

	  transport_destroy (rtsp, transport);

          if (oldlist)
            cseq->cseq_transports =
                g_list_remove_link (cseq->cseq_transports, oldlist);

	}
    }


  /* get the session id string which is used to refer
   * to this connection in the TEARDOWN method, also remove
   * the cseq structure from the cseq hash table (this RTSP 
   * response completes the sequence), and use the session id
   * string to refer to this in the session hash table
   */
  if (! (cseq->cseq_session = get_session (buff, *n))) /* no session id found */
    return 1;

  g_hash_table_remove (rtsp->rtsp_cseq, &cseq->cseq_id);
  g_hash_table_insert (rtsp->rtsp_session, cseq->cseq_session, cseq);

  return 0;
}



/*----------------------------------------------------------------------
   rtsp_response_handler_sv() -- A handler for RTSP responses.

   Paramaters:
     rtsp - The related rtsp object.
     from - The direction the message is traveling, FROM_CLIENT or FROM_SERVER.
     buff - A buffer containing the message being processed.
     n    - The size of the buffer.

   Return:
     1 on error
     0 on success

   This is only intended to be called with a SERVER_OBJECT.
   We process RTSP responses coming from the client in the server object.
----------------------------------------------------------------------*/
int
rtsp_response_handler_sv (RTSP * rtsp, RTSPFromType from, char * buff, int * n)
{

  if (rtsp->rtsp_objtype != SERVER_OBJECT)
    {
      log_message (ERROR_LOG,
                   "error: server object handler called with a client object:  "
                   "RTSP response");
      return 0;
    }

  if (from == FROM_SERVER)
    {
      /* if we have an interleave paramater for a Transports: header, we may
         need to remap the value in order to support interleaving multiple
         back-end servers over a TCP-only connection.
       */
      rtsp_interleave_update (rtsp, buff, n);

    }

  return 0;
}



/* RTSP PLAY HANDLER */

/*----------------------------------------------------------------------
   rtsp_play_handler() -- A handler for the PLAY method.

   Paramaters:
     rtsp - The related rtsp object.
     from - The direction the message is traveling, FROM_CLIENT or FROM_SERVER.
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.

   Return:
     1 on error
     0 on success

   This is only intended to be called with a CLIENT_OBJECT.
----------------------------------------------------------------------*/
int
rtsp_play_handler (RTSP * rtsp, RTSPFromType from, char * buff, int * n)
{

#ifdef LOG_URLS_PLAYED

  int pos=0;
  char *msg=NULL;

  if (rtsp->rtsp_objtype != CLIENT_OBJECT)
    {
      log_message (ERROR_LOG,
          "error: client PLAY method handler called with a server object");
      return 0;
    }

  /* first line looks like "PLAY rtsp://server:port/path... RTSP/1.0\r\n" */

  /* find the end of the first line of the method */
  if ((pos = cistrnfind ("\r\n", 2, buff, *n)) < 0)
    return 1;

  msg = g_malloc (pos + 1);
  if (msg)
  {
    memcpy (msg, buff, pos);
    msg[pos] = '\0';
    log_message (INFO_LOG, "%s", msg);
  }
  g_free (msg);

#endif /* LOG_URLS_PLAYED */

  return 0;
}



/*======================================================================
 * Because RTSP does not define the UDP transport types, anyone
 * can come up with their own; right now this proxy supports x-real-rdt/udp
 * and rtp/avp/udp;  each UDP transport protocol has it's own requirements
 * for ports and how they need to be allocated, so each transport needs
 * it's own set of functions for parsing ports, allocating tunnels through
 * the proxy, and replacing the specified ports for both the C->S SETUP
 * method and the corresponding S->C response.
 *====================================================================*/


/* PROTOCOL: x-real-rdt/udp */

/*----------------------------------------------------------------------
   x_real_rdt_get_ports() -- Extract port numbers for x-real-rdt/udp.

   Paramaters:
     rtsp - The related rtsp object.
     type - Flag indicating which transport type is being used.
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.

   Return:
     A pointer to an allocated Transport structure.
     NULL on error.

   This works with x_real_rdt_set_ports() to remap ports at the proxy.
----------------------------------------------------------------------*/
static Transport*
x_real_rdt_get_ports (RTSP * rtsp, TransportType type, char * buff, int n)
{
  int pos=0;
  unsigned short port=0;
  Transport *transport=NULL;
  PortPair *portpair=NULL;
  int server_rtspid = extract_serverid (buff, n);

  /* look for transport */
  if ( (pos = cistrnfind (rtsp_transports[type].text, rtsp_transports[type].len,  buff, n)) < 0)
    return NULL;

  buff += pos;
  n    -= pos;

  if ( (pos = cistrnfind ("client_port", 11, buff, n)) < 0)
    return NULL;

  buff += pos + 11;
  n    -= pos - 11;

  /* get the numeric port values */
  if (!get_port_range (buff, n, &port, &port))
    return NULL;

  /* create a new transport structure, allocate portpairs
   * and assign proper port values
   */
  transport = transport_new (rtsp, type);

  portpair = portpair_new (rtsp);
  portpair->portpair_rport = port;
  
  transport->transport_portpairs =
    g_list_append (transport->transport_portpairs, portpair);
  
  /* allocate UDP tunnels */
  portpair->portpair_udpid =
    (* rtsp->udp_client_tunnel_open) (server_rtspid, portpair->portpair_rport, &portpair->portpair_pport);
	  
  if (portpair->portpair_udpid == -1)
    {
      transport_destroy (rtsp, transport);
      return NULL;
    }

  log_message (DEBUG_LOG, "X-REAL-RDT/UDP REWROTE CLIENT_PORT %d -> %d",
	       portpair->portpair_rport,
	       portpair->portpair_pport);

  return transport;
}



/*----------------------------------------------------------------------
   x_real_rdt_set_ports() -- Set outgoing port numbers for x-real-rdt/udp.

   Paramaters:
     rtsp    - The related rtsp object.
     transport - The related transport structure allocated in the get_ports rtn.
     from    - The direction the message is passing, FROM_SERVER or FROM_CLIENT.
     buff    - A buffer containing the method being processed.
     n       - The size of the buffer.

   Return:
     1 if the transport was found and ports sucessfully set
    -1 if there was any kind of error
     0 if the transport isn't listed in the buffer

   This works with x_real_rdt_get_ports() to remap ports at the proxy.
----------------------------------------------------------------------*/
static int
x_real_rdt_set_ports (RTSP * rtsp, Transport * transport, RTSPFromType from, char * buff, int * n)
{
  int i=0, pos=0, diff=0, old_plen=0, new_plen=0, cudp_id=0;
  unsigned short old_port=0, new_port=0;
  int trans_str_len=0, client_port_len=0, server_port_len=0;
  char *trans_str=NULL, *client_port=NULL, *server_port=NULL;
  char old_pbuff[12], new_pbuff[12], *c=NULL;
  PortPair *portpair=NULL;

  /* log_message (DEBUG_LOG, "x_real_rdt_set_ports() called"); */

  /* search for the transport */
  pos = cistrnfind (rtsp_transports[transport->transport_type].text,
		    rtsp_transports[transport->transport_type].len, 
		    buff, *n);
  if (pos < 0)
    return 0;

  trans_str     = buff + pos;
  trans_str_len = *n - pos;


  /* parse forward until we encounter a ',' or a carriage return;
   * that will be the end of this particular transport string
   */
  for (i = 0; i < trans_str_len; i++)
    if (trans_str[i] == ',' || trans_str[i] == '\r')
      break;

  trans_str_len = i; /* set length at previous charactor */

  /* find the client port field */
  pos = cistrnfind ("client_port", 11, trans_str, trans_str_len);
  if (pos < 0)
    goto cleanup;

  client_port     = trans_str + pos;
  client_port_len = trans_str_len - pos;


  /* parse forward until we encounter a ';' or a '\r' that
   * delimits the end of the client_port field
   */
  for (i = 0; i < client_port_len; i++)
    if (client_port[i] == ';' || client_port[i] == '\r')
      break;
 
  client_port_len = i; /* set length at previous charactor */


  /* find the server port field, this field only exists 
   * in the response message, so we have special handling
   * for that case
   */
  pos = cistrnfind ("server_port", 11, trans_str, trans_str_len);
  if (pos >= 0)
    {
      server_port     = trans_str + pos;
      server_port_len = trans_str_len - pos;
      
      /* parse forward until we encounter a ';' or a '\r' that
       * delimits the end of the server_port field
       */
      
      for (i = 0; i < server_port_len; i++)
	if (server_port[i] == ';' || server_port[i] == '\r')
	  break;
      
      server_port_len = i; /* set length at previous charactor */
    }


  /* in x-real-rdt, there will only be one portpair */
  portpair = transport->transport_portpairs->data;

  if (!portpair)
    goto cleanup;

  cudp_id = portpair->portpair_udpid;

      
  if (from == FROM_CLIENT)
    {
      old_port = portpair->portpair_rport;
      new_port = portpair->portpair_pport;
    }
  else
    {
      old_port = portpair->portpair_pport;
      new_port = portpair->portpair_rport;
    }
  
  
  snprintf (old_pbuff, 12, "%d", old_port);
  snprintf (new_pbuff, 12, "%d", new_port);
      
  old_plen = strlen (old_pbuff);
  new_plen = strlen (new_pbuff);
  diff     = new_plen - old_plen;

  /* log_message (DEBUG_LOG, "old_pbuff=%s new_pbuff=%s", old_pbuff, new_pbuff); */


  if ( (pos = cistrnfind (old_pbuff, old_plen, client_port, client_port_len)) < 0)
    goto cleanup;

  c = client_port + pos;
  
  if (diff != 0)
    {
      memmove (c + new_plen, c + old_plen, (buff + *n) - (c + old_plen)); 
      *n += diff;
      server_port += diff;
    }
  
  memcpy (c, new_pbuff, new_plen);



  /* do the repacement on the server_port field, this is the 
   * port the client checks to see that UDP content is coming
   * from, therefore this is currently the same as 
   * portpair->portpair_pport 
   */
  if (from == FROM_SERVER)
    {
      unsigned short sport=0;

      if (!get_port_range (server_port + 11, server_port_len - 11, &sport, &sport))
	goto cleanup;

      old_port = sport;

      /* allocate UDP tunnel to the server */
      portpair = portpair_new (rtsp);
      portpair->portpair_rport = old_port;
      
      transport->transport_portpairs =
	g_list_append (transport->transport_portpairs, portpair);
      
      portpair->portpair_udpid =
	(* rtsp->udp_server_tunnel_open) (cudp_id, portpair->portpair_rport, &portpair->portpair_pport);
      
      if (portpair->portpair_udpid == -1)
	{
	  portpair_destroy (rtsp, portpair);
	  goto cleanup;
	}
      new_port = portpair->portpair_pport;

      snprintf (old_pbuff, 12, "%d", old_port);
      snprintf (new_pbuff, 12, "%d", new_port);
      
      old_plen = strlen (old_pbuff);
      new_plen = strlen (new_pbuff);
      diff     = new_plen - old_plen;

      log_message (DEBUG_LOG, "X-REAL-RDT/UDP REWROTE SERVER_PORT %s -> %s", old_pbuff, new_pbuff);

      if ( (pos = cistrnfind (old_pbuff, old_plen, server_port, server_port_len)) < 0)
	goto cleanup;
      
      c = server_port + pos;
      
      
      if (diff != 0)
	{
      memmove (c + new_plen, c + old_plen, (buff + *n) - (c + old_plen)); 
	  *n += diff;
	}
      
      memcpy (c, new_pbuff, new_plen);

    }

  return 1;
 
  /* this is really an error condition */
 cleanup:
  return -1;
}




/* PROTOCOL: rtp/avp/udp */

/*----------------------------------------------------------------------
   rtp_get_ports() -- Extract port numbers for rtp/avp/udp.

   Paramaters:
     rtsp - The related rtsp object.
     type - Flag indicating which transport type is being used.
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.

   Return:
     A pointer to an allocated Transport structure.
     NULL on error.

   This works with rtp_set_ports() to remap ports at the proxy.
----------------------------------------------------------------------*/
static Transport*
rtp_get_ports (RTSP * rtsp, TransportType type, char * buff, int n)
{
  int pos=0, done1=0, done2=0, local1=0, local2=0;
  unsigned short port1=0, port2=0;
  int cliidx1=0, cliidx2=0;
  Transport *transport=NULL;
  PortPair *cportpair_udp=NULL,
	   *cportpair_rtcp=NULL;

  if ( (pos = cistrnfind (rtsp_transports[type].text,
                          rtsp_transports[type].len, buff, n)) < 0)
    return NULL;

  buff += pos;
  n    -= pos;

  if ( (pos = cistrnfind ("client_port", 11, buff, n)) < 0)
    return NULL;

  buff += pos + 11;
  n    -= pos - 11;

  /* get the client_port numeric port values */
  if (!get_port_range (buff, n, &port1, &port2))
    return NULL;

  /* create a new transport structure, allocate client portpairs
   * and assign proper port values
   */
  transport = transport_new (rtsp, type);

  cportpair_udp = portpair_new (rtsp);
  cportpair_udp->portpair_rport = port1;
  transport->transport_portpairs =
    g_list_append (transport->transport_portpairs, cportpair_udp);

  cportpair_rtcp = portpair_new (rtsp);
  cportpair_rtcp->portpair_rport = port2;
  transport->transport_portpairs =
    g_list_append (transport->transport_portpairs, cportpair_rtcp);


	/* here's where we try to convince the OS to allocate two ports
           for us, numbered 2N and 2N+1.  These are for the RTP Data
           and RTCP Reports connections between the proxy and the client.
         */
  local1 = MIN_LOCAL_PORT;
  local2 = local1 + 1;

  while (!done1 || !done2)
  {
    done1 = 0;
    done2 = 0;
    while (! done1)
    {
      cliidx1 = create_rtp_udp_tunnel (port1, 0, 1, rtsp->rtsp_id, local1);

      /* ensure we allocate 2N ...  if not, destroy this tunnel and retry.  */
  
      if (cliidx1 != -1 &&
	  rtp_udp_tunnel_port_ok (cliidx1, &cportpair_udp->portpair_pport))
        {  done1 = 1;
        }
      else
        {
           log_message (DEBUG_LOG, "RTP/AVP/UDP port1 was odd...retrying");
           close_udp_tunnel (cliidx1, 0);
           local1 += 2;
           local2 = local1 + 1;
        }
      if (local2 > MAX_LOCAL_PORT)
        {  log_message (ERROR_LOG, "RTP/AVP/UDP error allocating port1 to client: no available port found");
           exit_proxy (EXIT_PORT_ALLOCATION_ERROR);
        }
    }
  
    cliidx2 = create_rtp_udp_tunnel (port2, 0, 2, rtsp->rtsp_id, local2);
    /* ensure we alloate 2N and 2N+1 ...
       if not, destroy both of these tunnels and retry.
     */
    if (cliidx2 != -1 &&
	rtp_rtcp_tunnel_port_ok (cliidx2, cliidx1, &cportpair_rtcp->portpair_pport))
      {  done2 = 1;
      }
    else
      {
        log_message (DEBUG_LOG, "RTP/AVP/UDP port2 was not 2N+1 (aka port1+1)...retrying");
        close_udp_tunnel (cliidx1, 0);
        close_udp_tunnel (cliidx2, 0);
        done1 = 0;
        local1 += 2;
        local2 = local1 + 1;
      }
    if (local2 > MAX_LOCAL_PORT)
      { 
        log_message (ERROR_LOG, "RTP/AVP/UDP error allocating port2 to client: no available port found");
        exit_proxy (EXIT_PORT_ALLOCATION_ERROR);
      }
  }

  log_message (DEBUG_LOG,
               "RTP/AVP/UDP REWROTE CLIENT_PORTS %d -> %d AND %d -> %d",
               cportpair_udp->portpair_rport,
               cportpair_udp->portpair_pport,
               cportpair_rtcp->portpair_rport,
               cportpair_rtcp->portpair_pport);


  cportpair_udp->portpair_udpid = cliidx1;
  cportpair_rtcp->portpair_udpid = cliidx2;

  return transport;
}



/*----------------------------------------------------------------------
   rtp_set_ports() -- Set outgoing port numbers for rtp/avp/udp.

   Paramaters:
     rtsp    - The related rtsp object.
     transport - The related transport structure allocated in the get_ports rtn.
     from    - The direction the message is passing, FROM_SERVER or FROM_CLIENT.
     buff    - A buffer containing the method being processed.
     n       - The size of the buffer.

   Return:
     1 if the transport was found and ports sucessfully set
    -1 if there was any kind of error
     0 if the transport isn't listed in the buffer

   This works with rtp_get_ports() to remap ports at the proxy.
   This is just a wrapper for rtp_set_ports_client() and
   rtp_set_ports_server().
----------------------------------------------------------------------*/
static int
rtp_set_ports (RTSP * rtsp, Transport * transport, RTSPFromType from, char * buff, int * n)
{

    if (from == FROM_CLIENT)
      return (rtp_set_ports_client (rtsp, transport, from, buff, n));
    else
      return (rtp_set_ports_server (rtsp, transport, from, buff, n));
}




/*----------------------------------------------------------------------
   rtsp_remove_transports() -- Remove transports from the Transport header.

   Paramaters:
     rtsp    - The related rtsp object.
     buff    - A buffer containing the method header being processed.
     n       - The size of the buffer.

   Return:  nothing

   This is for tossing any multicast transports the client specifies
   so the server can't choose it, since this proxy doesn't support multicast.
----------------------------------------------------------------------*/
static void
rtsp_remove_transports (RTSP * rtsp, char * buff, int * n)
{
  int i=0, pos=0;
  char *trans_str=NULL, *trans_hdr_str=NULL;
  int trans_str_len=0, trans_hdr_str_len=0;

  if ( (pos = cistrnfind ("transport:", 10, buff, *n)) < 0)
    return;

  trans_str     = buff + pos + 11;
  trans_str_len = *n - pos - 11; /* length of first transport string */

  /* find the \r\n at the end of the header -- Transport:...\r\n */
  if ( (pos = cistrnfind ("\r\n", 2, trans_str, *n - (trans_str - buff))) < 0)
    return;

  trans_hdr_str = trans_str;
  trans_hdr_str_len = pos; /* length of the Transport: header line */

  /* loop across all transport strings */
  while (trans_str < trans_hdr_str + trans_hdr_str_len - 1)
    {
      /* parse forward until we encounter a ',' or a carriage return;
       * that will be the end of this particular transport string
       */
      for (i = 0; i < trans_str_len; i++)
        if (trans_str[i] == ',' || trans_str[i] == '\r')
          break;

      trans_str_len = i; /* set length at previous charactor */

      /* if it contains 'unicast' it should be ok, so go to the next one */
      if ( (cistrnfind ("unicast", 7, trans_str, trans_str_len)) < 0)
        {
          /* Look for 'mcast' or 'multicast' in the transport, if there,
             remove it.
           */
          if ( cistrnfind ("mcast",     5, trans_str, trans_str_len) >= 0 ||
               cistrnfind ("multicast", 9, trans_str, trans_str_len) >= 0 )
            {
              memmove (trans_str, trans_str + trans_str_len + 1, *n - trans_str_len - 1);
              *n = *n - trans_str_len - 1;
              trans_hdr_str_len = trans_hdr_str_len - trans_str_len - 1;
            }
        }

      trans_str += trans_str_len + 1;
    }

}




/*----------------------------------------------------------------------
   rtp_set_ports_client() -- Set outgoing port numbers for rtp/avp/udp
                             for messages coming from the client.

   Paramaters:
     rtsp    - The related rtsp object.
     transport - The related transport structure allocated in the get_ports rtn.
     from    - The direction the message is passing, FROM_SERVER or FROM_CLIENT.
     buff    - A buffer containing the method being processed.
     n       - The size of the buffer.

   Return:
     1 if the transport was found and ports sucessfully set
    -1 if there was any kind of error
     0 if the transport isn't listed in the buffer

   This works with rtp_get_ports() to remap ports at the proxy.
----------------------------------------------------------------------*/
static int
rtp_set_ports_client (RTSP * rtsp, Transport * transport, RTSPFromType from, char * buff, int * n)
{
  int i=0, pos=0, diff=0, old_plen=0, new_plen=0;
  unsigned short old_port1=0, old_port2=0, new_port1=0, new_port2=0;
  int trans_str_len=0, client_port_len=0;
  char *trans_str=NULL, *client_port=NULL, *server_port=NULL;
  char old_pbuff[24], new_pbuff[24], *c=NULL;
  int cliidx1=0, cliidx2=0;
  CSeq *cseq=NULL;
  int cseq_id=0;
  GList *trlist=NULL;


  /* search for the transport */
  pos = cistrnfind (rtsp_transports[transport->transport_type].text,
                    rtsp_transports[transport->transport_type].len,
                    buff, *n);
  if (pos < 0)
    return 0;


  trans_str     =  buff + pos;
  trans_str_len = *n - pos;

  /* parse forward until we encounter a ',' or a carriage return;
   * that will be the end of this particular transport string
   */
  for (i = 0; i < trans_str_len; i++)
    if (trans_str[i] == ',' || trans_str[i] == '\r')
      break;

  trans_str_len = i; /* set length at previous charactor */



  /* use the sequence ID to find the ports we've already allocated
   * for the P->C connection.  Once found, allocate ports to the
   * server, and associate the sets of ports with each other.
   */
  cseq_id = get_cseq (buff, *n);
  cseq = g_hash_table_lookup (rtsp->rtsp_cseq, &cseq_id);

  trlist = cseq->cseq_transports;
  while (trlist->prev)
      trlist = trlist->prev;
  while (trlist)
    {
      Transport *tr = (Transport *)trlist->data;
      GList *pplist = tr->transport_portpairs;
      int tt = tr->transport_type;
      if (tt == TRANSPORT_RTP_AVP_UDP)
        {
          while (pplist->prev) /* ensure we're at the front of the list */
              pplist = pplist->prev;
          while (pplist)
            {
              PortPair *pp = (PortPair *)pplist->data;
              /* log_message (DEBUG_LOG,
                     "CSEQ LIST: cseq_id: %d tt:%d rport:%d pport:%d pp_id:%d",
  	             cseq_id, tt, pp->portpair_rport, pp->portpair_pport,
                     pp->portpair_udpid); */

              if (pp->portpair_pport % 2 == 0)
                {
                  cliidx1 = pp->portpair_udpid;
                  old_port1 = pp->portpair_rport;
                  new_port1 = pp->portpair_pport;
                }
              else
                {
                  cliidx2 = pp->portpair_udpid;
                  old_port2 = pp->portpair_rport;
                  new_port2 = pp->portpair_pport;
                }
  
              pplist = pplist->next;
            }
        }
      trlist = trlist->next;
    }


  /* find the client port field */
  pos = cistrnfind ("client_port", 11, trans_str, trans_str_len);
  if (pos < 0)
    goto cleanup;

  client_port     = trans_str + pos;
  client_port_len = trans_str_len - pos;

  /* parse forward until we encounter a ';' or a '\r' that
   * delimits the end of the client_port field
   */     
  for (i = 0; i < client_port_len; i++)
    if (client_port[i] == ';' || client_port[i] == '\r')
      break;

  client_port_len = i; /* set length at previous charactor */

  snprintf (old_pbuff, 24, "%d-%d", old_port1, old_port2);
  snprintf (new_pbuff, 24, "%d-%d", new_port1, new_port2);

  old_plen = strlen (old_pbuff);
  new_plen = strlen (new_pbuff);
  diff     = new_plen - old_plen;

  if ( (pos = cistrnfind (old_pbuff, old_plen, client_port, client_port_len)) < 0)
    goto cleanup;

  c = client_port + pos;

  if (diff != 0)
    {
      memmove (c + new_plen, c + old_plen, (buff + *n) - (c + old_plen));
      *n += diff;
      server_port += diff;
    }

  memcpy (c, new_pbuff, new_plen);

  return 1;

  /* this is really an error condition */
 cleanup:
  return -1;

}



/*----------------------------------------------------------------------
   rtp_set_ports_server() -- Set outgoing port numbers for RTP/AVP/UDP
                             for messages coming from the server.

   Paramaters:
     rtsp      - The related rtsp object.
     transport - The related transport structure allocated in the get_ports rtn.
     from      - The direction the msg is passing, FROM_SERVER or FROM_CLIENT.
     buff      - A buffer containing the method being processed.
     n         - The size of the buffer.

   Return:
     1 if the transport was found and ports sucessfully set
    -1 if there was any kind of error
     0 if the transport isn't listed in the buffer

   This works with rtp_get_ports() to remap ports at the proxy.
----------------------------------------------------------------------*/
static int
rtp_set_ports_server (RTSP * rtsp, Transport * transport, RTSPFromType from, char * buff, int * n)
{
  int pos=0, i=0, done1=0, done2=0, local1=0, local2=0;
  int cliidx1=0, cliidx2=0, seridx1=0, seridx2=0;
  char *trans_str=NULL, *client_port=NULL, *server_port=NULL;
  int trans_str_len=0, client_port_len=0, server_port_len=0;
  unsigned short sport1=0, sport2=0;
  PortPair *sportpair_udp=NULL, *sportpair_rtcp=NULL;
  PortPair *cportpair_udp=NULL, *cportpair_rtcp=NULL;
  char old_pbuff[24], new_pbuff[24], *c=NULL;
  int old_plen=0, diff=0, new_plen=0;
  CSeq *cseq=NULL;
  int cseq_id=0;
  GList *trlist=NULL;


  cseq_id = get_cseq (buff, *n);
  cseq = g_hash_table_lookup (rtsp->rtsp_cseq, &cseq_id);
  trlist = cseq->cseq_transports;
  if (trlist)
    while (trlist->prev)
      trlist = trlist->prev;
  while (trlist)
    {
      Transport *tr = (Transport *)trlist->data;
      GList *pplist = tr->transport_portpairs;
      int tt = tr->transport_type;
      if (tt == TRANSPORT_RTP_AVP_UDP)
        {
          while (pplist->prev) /* ensure we're at the front of the list */
              pplist = pplist->prev;
          while (pplist)
          {
            PortPair *pp = (PortPair *)pplist->data;
            /* log_message (DEBUG_LOG,
              "CSEQ LIST: cseq_id: %d tt:%d rport:%d pport:%d pp_id:%d",
	      cseq_id, tt, pp->portpair_rport, pp->portpair_pport, pp->portpair_udpid); */


            if (pp->portpair_pport % 2 == 0)
              {
                cliidx1 = pp->portpair_udpid;
                cportpair_udp = pp;
              }
            else
              {
                cliidx2 = pp->portpair_udpid;
                cportpair_rtcp = pp;
              }

            pplist = pplist->next;
          }
        }
      trlist = trlist->next;
    }


  /* search for the transport */
  pos = cistrnfind (rtsp_transports[transport->transport_type].text,
                    rtsp_transports[transport->transport_type].len,
                    buff, *n);
  if (pos < 0)
    return 0;

  trans_str     = buff + pos;
  trans_str_len = *n - pos;

  /* parse forward until we encounter a ',' or a carriage return;
   * that will be the end of this particular transport string
   */
  for (i = 0; i < trans_str_len; i++)
    if (trans_str[i] == ',' || trans_str[i] == '\r')
      break;

  trans_str_len = i; /* set length at previous charactor */


  /* find the client_port range */
  pos = cistrnfind ("client_port", 11, trans_str, trans_str_len);
  if (pos >= 0)
    {
      client_port     = trans_str + pos;
      client_port_len = trans_str_len - pos;
      
      /* parse forward until we encounter a ';' or a '\r' that
       * delimits the end of the client_port field
       */
      
      for (i = 0; i < client_port_len; i++)
        if (client_port[i] == ';' || client_port[i] == '\r')
          break;
  
      client_port_len = i; /* set length at previous charactor */
    }


  /* find the server port field */
  pos = cistrnfind ("server_port", 11, trans_str, trans_str_len);
  if (pos >= 0)
    {
      server_port     = trans_str + pos;
      server_port_len = trans_str_len - pos;
      
      /* parse forward until we encounter a ';' or a '\r' that
       * delimits the end of the server_port field
       */
      
      for (i = 0; i < server_port_len; i++)
        if (server_port[i] == ';' || server_port[i] == '\r')
          break;
  
      server_port_len = i; /* set length at previous charactor */

      /* get the numeric port values */
      if (!get_port_range (server_port+11, server_port_len-11, &sport1, &sport2))
        return 0;
    }


  /* Allocate the portpair structures for the P->S tunnels */
  sportpair_udp = portpair_new (rtsp);
  sportpair_udp->portpair_rport = sport1;
  transport->transport_portpairs =
    g_list_append (transport->transport_portpairs, sportpair_udp);
  sportpair_rtcp = portpair_new (rtsp);
  sportpair_rtcp->portpair_rport = sport2;
  transport->transport_portpairs =
    g_list_append (transport->transport_portpairs, sportpair_rtcp);


	/* here's where we try to convince the OS to allocate two ports
           for us, numbered 2N and 2N+1.  These are for the RTP Data
           and RTCP Reports connections between the proxy and the server.
         */
  local1 = MIN_LOCAL_PORT;
  local2 = local1 + 1;

  while (!done1 || !done2)
    {
      done1 = 0;
      done2 = 0;
      while (! done1)
        {
          seridx1 = create_rtp_udp_tunnel (sport1, 1, 1, cseq->cseq_srtspid, local1);
          /* ensure we alloate 2N ...  If not, destroy this tunnel and retry. */
          if (seridx1 != -1 &&
	      rtp_udp_tunnel_port_ok (seridx1, &sportpair_udp->portpair_pport))
            {  done1 = 1;
            }
          else
            {
              log_message (DEBUG_LOG, "RTP/AVP/UDP sport1 was odd...retrying");
              close_udp_tunnel (seridx1, 0);
              local1 += 2;
              local2 = local1 + 1;
            }
          if (local2 > MAX_LOCAL_PORT)
            {  log_message (ERROR_LOG, "RTP/AVP/UDP error allocating port1 to server: no available port found");
               exit_proxy (EXIT_PORT_ALLOCATION_ERROR);
            }
        }
    
      seridx2 = create_rtp_udp_tunnel (sport2, 1, 2, cseq->cseq_srtspid, local2);
      /* ensure we alloate 2N and 2N+1 ...  If not, destroy both and retry. */
      if (seridx2 != -1 &&
	  rtp_rtcp_tunnel_port_ok (seridx2, seridx1, &sportpair_rtcp->portpair_pport))
        {  done2 = 1;
        }
      else
        {
          log_message (DEBUG_LOG, "RTP/AVP/UDP sport2 was not 2N+1 (aka sport1+1)...retrying");
          close_udp_tunnel (seridx1, 0);
          close_udp_tunnel (seridx2, 0);
          done1 = 0;
          local1 += 2;
          local2 = local1 + 1;
        }
      if (local2 > MAX_LOCAL_PORT)
        { log_message (ERROR_LOG, "RTP/AVP/UDP error allocating port2 to server: no available port found");
          exit_proxy (EXIT_PORT_ALLOCATION_ERROR);
        }
    }
  

  /* Since we couldn't set the udpt_siblings tunnels value until now,
     set it for all four connections (the UDP and RTCP connections to
     the client, and the UDP and RTCP connections to the server.
   */
  set_tunnel_siblings (cliidx1, seridx1); /* RTP Data ports */
  set_tunnel_siblings (cliidx2, seridx2); /* RTCP Control ports */


  log_message (DEBUG_LOG,
               "RTP/AVP/UDP REWROTE SERVER_PORTS %d -> %d AND %d -> %d",
               sportpair_udp->portpair_rport,
               sportpair_udp->portpair_pport,
               sportpair_rtcp->portpair_rport,
               sportpair_rtcp->portpair_pport);


/* substitute in the proxy's ports to the server */
  snprintf (old_pbuff, 24, "%d-%d",
            sportpair_udp->portpair_rport,
            sportpair_rtcp->portpair_rport);
  snprintf (new_pbuff, 24, "%d-%d",
            sportpair_udp->portpair_pport,
            sportpair_rtcp->portpair_pport);

  old_plen = strlen (old_pbuff);
  new_plen = strlen (new_pbuff);
  diff     = new_plen - old_plen;


  if ( (pos = cistrnfind (old_pbuff, old_plen, server_port,
                          server_port_len)) < 0)
    goto cleanup;


  c = server_port + pos;

  if (diff != 0)
    {
      memmove (c + new_plen, c + old_plen, (buff + *n) - (c + old_plen) + 1);
      *n += diff;
      server_port += diff;
    }

  memcpy (c, new_pbuff, new_plen);


/* substitute in the real client's ports */
  snprintf (old_pbuff, 24, "%d-%d",
            cportpair_udp->portpair_pport,
            cportpair_rtcp->portpair_pport);
  snprintf (new_pbuff, 24, "%d-%d",
            cportpair_udp->portpair_rport,
            cportpair_rtcp->portpair_rport);

  old_plen = strlen (old_pbuff);
  new_plen = strlen (new_pbuff);
  diff     = new_plen - old_plen;

  if ( (pos = cistrnfind (old_pbuff, old_plen, client_port,
                          client_port_len)) < 0)
    goto cleanup;

  c = client_port + pos;

  if (diff != 0)
    {
      memmove (c + new_plen, c + old_plen, (buff + *n) - (c + old_plen) + 1);
      *n += diff;
      client_port += diff;
    }

  memcpy (c, new_pbuff, new_plen);



  return 1;

  /* this is really an error condition */
 cleanup:
  return -1;

}




/*----------------------------------------------------------------------
   rtsp_remove_session() -- Remove resources related to a session from an
                            RTSP structure.

   Paramaters:
     rtsp    - The related rtsp object.
     session - The name of the session to remove

   Return:
     0 on error
     1 on success
----------------------------------------------------------------------*/
static int
rtsp_remove_session (RTSP *rtsp, char *session)
{
  CSeq *cseq=NULL;
  GList *list=NULL;
  RTSP *rtsp2=NULL;

  log_message (DEBUG_LOG, "removing session %s from RTSP object %d",
               session, rtsp->rtsp_id);

  list = rtsp->rtsp_list;
  while (list->prev)
    list = list->prev;
  rtsp2 = list->data;


  /* look for the session entry in the session hash table,
   * it's not a error if it's not here
   */
  if (! (cseq = g_hash_table_lookup (rtsp->rtsp_session, session)))
    {
      log_message (DEBUG_LOG, "session %s not found in object %d",
                   session, rtsp->rtsp_id);
      return 0;
    }


  /* teardown the connection by removing the cseq structure
   * from the session hash table and calling it's destroy function
   */
  g_hash_table_remove (rtsp->rtsp_session, cseq->cseq_session);
  cseq_destroy (rtsp, cseq);

  log_message (DEBUG_LOG, "REMOVED SESSION %s", session);

  return 1;
}



/*----------------------------------------------------------------------
   method_type() -- Determine which RTSP method a buffer contains

   Paramaters:
     rtsp - The related rtsp object.
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.

   Return:
     The enum type of method the buffer contains.
     -1  on error.
----------------------------------------------------------------------*/
static RTSPMethodType
method_type (RTSP *rtsp, char * buff, int len)
{
  int i=0;

  if ( ! rtsp->rtsp_method)
    return -1;

  for (i = 0; i < RTSP_METHOD_LAST; i++)
    {
      /* log_message (DEBUG_LOG, "testing for method_type:%s",
                      rtsp->rtsp_method[i].text); */

      if (len >= rtsp->rtsp_method[i].len)
	if (cistrncmp (rtsp->rtsp_method[i].text, buff,
                       rtsp->rtsp_method[i].len))
          {
            /* log_message (DEBUG_LOG, "method_type:%s handler:0x%x",
                            rtsp->rtsp_method[i].text,
                            rtsp->rtsp_method[i].handler); */
	    return i;
          }
    }

  log_message (DEBUG_LOG,
               "FAILED TO FIND A METHOD IN BUFFER: <<<len:%d buff=%s...>>",
               len, buff);

  return -1;
}



/*----------------------------------------------------------------------
   get_cseq() -- Extract the sequence number of an RTSP method.

   Paramaters:
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.

   Return:
     The sequence number of a rtsp method
     0 on failure
----------------------------------------------------------------------*/
static int
get_cseq (char * data, int n)
{
  int i=0, state=0, cseq=0, cseq_offset=0;


  if ((cseq_offset = cistrnfind ("\r\ncseq:", 7, data, n)) < 0)
    return -1;

  data += cseq_offset + 7;
  n    -= cseq_offset + 7;


  for (i = 0; i < n && state >= 0; i++)
    switch(state)
      {
	
      /* ignore leading white space */
      case 0:
	if (!ISWHITE(data[i]))
	  {
	    if(ISDIGIT(data[i]))
	      {
		cseq = CHARTODIGIT(data[i]);
		state = 1;
	      }
	    else
	      {
		return -1;
	      }
	  }
	break;
	
	
      /* get the cseq number, converting to int */
      case 1:
	if(ISDIGIT(data[i]))
	  {
	    cseq *= 10;
	    cseq += CHARTODIGIT(data[i]);
	  }
	else
	  {
	    state = -1;
	  }
	break;
      }


  return cseq;
}



/*----------------------------------------------------------------------
   get_session() -- Extact a session ID from an RTSP method.

   Paramaters:
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.
   
   Return:
     A malloc'd, null-terminated string
     NULL if no session ID found
----------------------------------------------------------------------*/
static char *
get_session (char * data, int n)
{
  int i=0;
  int state=0, session_length=0, session_offset=0;
  char *session=NULL, *str=NULL;


  if ((session_offset = cistrnfind ("\r\nsession:", 10, data, n)) < 0)
    return NULL;

  data += session_offset + 10;
  n    -= session_offset + 10;


  for (i = 0; i < n && state >= 0; i++)
    switch(state)
      {
	
      /* ignore leading white space */
      case 0:
	if (!ISWHITE(data[i]))
	  {
	    session = &data[i];
	    session_length++;
	    state = 1;
	  }
	break;
	
	
      /* get the string */
      case 1:
	if(ISWHITE(data[i]) || ISEND(data[i]))
	  {
	    state = -1;
	  }
	else
	  {
	    session_length++;
	  }
	break;

      }
  

  if (!session || session_length < 1)
    return NULL;


  str = g_malloc (session_length + 1);
  memcpy (str, session, session_length);
  str[session_length] = '\0';


  return str;
}



/*----------------------------------------------------------------------
   get_num_range() -- Parse out a number range from a string.

   Paramaters:
     buff - A buffer containing the string being processed.
     n    - The size of the buffer.
     num1 - A pointer to a short int to assign a value to.
     num2 - A pointer to a short int to assign a value to.

   Return:
     0 on error
     1 on success

   Given that the data buffer begins at the end of a
   client_port or server_port, parse out the port range and
   return them in num1 and num2; if there is only one port
   specified, then both num1 and num2 are set to this value.
   When called, data points at the "=" after "client_port=xxx"
   or "server_port=xxx".  This is also used to extract other
   similarly-encoded "name=num1-num2" values.
   These are unsigned shorts since this is typically used
   as get_port_range(), which needs unsigned short ints.
----------------------------------------------------------------------*/
static int
get_num_range (char * data, int n, unsigned short *num1, unsigned short *num2)
{
  int i=0;
  int state=0;
  
  *num1 = 0;
  *num2 = 0;
  
  for (i = 0; i < n && state >= 0; i++)
    switch(state)
      {
	
	/*
	 * parse leading white space up to '='
	 */
      case 0:
	if(data[i] == '=')
	  {
	    state = 1;
	  }
	else if (!ISWHITE(data[i]))
	  {
	    return 0;
	  }
	break;
	

	/*
	 * parse any white space after the '=' up to the first
	 * digit
	 */
      case 1:
	if(ISDIGIT(data[i]))
	  {
	    *num1 = CHARTODIGIT(data[i]);
	    state = 2;
	  }
	else if(!ISWHITE(data[i]))
	  {
	    return 0;
	  }
	break;
	
	/*
	 * now get the first number
	 */    
      case 2:
	if(ISDIGIT(data[i]))
	  {
	    *num1 *= 10;
	    *num1 += CHARTODIGIT(data[i]);
	  }
	else
	  {
	    if(data[i] == '-')
	      state = 4;
	    else
	      state = 3;
	  }
	break;
	

	/*
	 * if the next non-whitespace charactor is a '-', then
	 * this is a number range and we go on to parse the second
	 * number, otherwise this was a single number message
	 */
      case 3:
	if(data[i] == '-')
	  {
	    state = 4;
	  }
	else if(!ISWHITE(data[i]))
	  {
	    state = -1;
	  }
	break;
	

	/*
	 * parse any white space after the '-' up to the first
	 * digit
	 */
      case 4:
	if(ISDIGIT(data[i]))
	  {
	    *num2 = CHARTODIGIT(data[i]);
	    state = 5;
	  }
	else if(!ISWHITE(data[i]))
	  {
	    return 0;
	  }
	break;
	

	/*
	 * now get the second number
	 */    
      case 5:
	if(!ISDIGIT(data[i]))
	  {
	    state = -1;
	  }
	else
	  {
	    *num2 *= 10;
	    *num2 += CHARTODIGIT(data[i]);
	  }
	break;
      }

  
  /* 
   * found one number
   */
  if(!*num1)
    return 0;

  if (!*num2)
    *num2 = *num1;

  
  return 1;
}



/*----------------------------------------------------------------------
   cistrncmp() -- Case-insensitive string compare.

   Paramaters:
     str1 - First string to examine.
     str2 - Second string to examine.
     len  - Number of characters to compare.

   Return:
     0 if strings are different
     1 is they are the same (ignoring case)
----------------------------------------------------------------------*/
static int
cistrncmp (char * str1, char * str2, int len)
{
  int i=0;

  for (i = 0; i < len; i++)
    if (TOLOWER(str1[i]) != TOLOWER(str2[i]))
      return 0;
  
  return 1;
}



/*----------------------------------------------------------------------
   cistrnfind() -- Case-insensitive substring find.

   Paramaters:
     str1 - String to look for in str2.
     len1 - Number of characters in str1.
     str2 - String to examine.
     len2 - Number of characters in str2.

   Return:
     beginning index of string 1 in string 2
     -1 if string 1 not found in string2
----------------------------------------------------------------------*/
static int
cistrnfind (char * str1, int len1, char * str2, int len2)
{
  int i=0, j=0, match=0;

  for (i = 0; i < len2; i++)
    {
      if (TOLOWER(str1[0]) != TOLOWER(str2[i]))
	continue;

      match = 1;

      for (j = 1; j < len1; j++)
	if (TOLOWER(str1[j]) != TOLOWER(str2[j+i]))
	  {
	    match = 0;
	    break;
	  }

      if (match)
	return i;
    }

  return -1;
}



/* MEMORY ALLOCATION/DESTRUCTION */

/*----------------------------------------------------------------------
   portpair_new() -- Allocate a new PortPair structure/object.

   Paramaters:
     rtsp - A related RTSP structure/object.

   Return:  A pointer to the allocated object.
----------------------------------------------------------------------*/
static PortPair*
portpair_new (RTSP * rtsp)
{
  PortPair *portpair = g_chunk_new (PortPair, portpair_mem_chunk); 

  portpair->portpair_udpid = -1;
  portpair->portpair_rport = 0;
  portpair->portpair_pport = 0;

  return portpair;
}


/*----------------------------------------------------------------------
   portpair_destroy() -- Deallocate a PortPair structure/object.

   Paramaters:
     rtsp     - A related RTSP structure/object.
     portpair - The PortPair structure/object to remove.

   Return:  nothing.

   This uses the RTSP object's callback to shutdown the associated tunnel.
----------------------------------------------------------------------*/
static void
portpair_destroy (RTSP * rtsp, PortPair * portpair)
{
  (* rtsp->udp_tunnel_close) (rtsp->rtsp_id, portpair->portpair_udpid);
	      
  g_chunk_free (portpair, portpair_mem_chunk);
}



/*----------------------------------------------------------------------
   transport_new() -- Allocate a new Transport structure/object.

   Paramaters:
     rtsp - A related RTSP structure/object.
     type - A flag indicating what type of transport this is.

   Return:  A pointer to the allocated object.
----------------------------------------------------------------------*/
static Transport*
transport_new (RTSP * rtsp, TransportType type)
{
  Transport *transport = g_chunk_new (Transport, transport_mem_chunk);


  transport->transport_type      = type;
  transport->transport_portpairs = NULL;

  return transport;
}



/*----------------------------------------------------------------------
   transport_destroy() -- Deallocate a Transport structure/object.

   Paramaters:
     rtsp      - A related RTSP structure/object.
     transport - The Transport structure/object to remove.

   Return:  nothing.
----------------------------------------------------------------------*/
static void
transport_destroy (RTSP * rtsp, Transport * transport)
{
  GList *list=NULL, *next=NULL;

  /* free all the portspairs in the transport */
  list = transport->transport_portpairs;
  if (list) /* ensure we start at beginning of list */
    while (list->prev)
      list = list->prev;
  while (list)
    {
      next = list->next;
      portpair_destroy (rtsp, list->data);
      list = next;
    }

  /* free the list nodes */
  g_list_free (transport->transport_portpairs);

  g_chunk_free (transport, transport_mem_chunk);

}



/*----------------------------------------------------------------------
   cseq_new() -- Allocate a new CSeq structure/object.

   Paramaters:
     rtsp - A related RTSP structure/object.

   Return:  A pointer to the allocated object.
----------------------------------------------------------------------*/
static CSeq *
cseq_new (RTSP * rtsp)
{
  CSeq *cseq = g_chunk_new (CSeq, cseq_mem_chunk);

  cseq->cseq_id         = -1;
  cseq->cseq_session    = NULL;
  cseq->cseq_transports = NULL;
  cseq->cseq_srtspid    = 0;
  cseq->cseq_ping       = 0;

  return cseq;
}



/*----------------------------------------------------------------------
   cseq_destroy() -- Deallocate a Transport structure/object.

   Paramaters:
     rtsp - A related RTSP structure/object.
     cseq - The CSeq structure/object to remove.

   Return:  nothing.
----------------------------------------------------------------------*/
static void
cseq_destroy (RTSP * rtsp, CSeq * cseq)
{
  GList *list=NULL, *next=NULL;
  RTSP *main_rtsp=NULL;

  list = rtsp->rtsp_list;
  while (list->prev)
    list = list->prev;
  main_rtsp = (RTSP *)(list->data);

  /* destroy all the transports in the cseq */
  list = cseq->cseq_transports;
  if (list) /* ensure we start at beginning of list */
    while (list->prev)
      list = list->prev;
  while (list)
    {
      /* log_message (DEBUG_LOG,
                      "cseq_destroy: destroying transport: %x %x %x %x",
                      list, list->data, list->next, list->prev); */
      next = list->next;
      transport_destroy (main_rtsp, list->data);
      list = next;
    }


  /* free the list nodes */
  g_list_free (cseq->cseq_transports);


  /* free the session ID string */
  g_free (cseq->cseq_session);


  g_chunk_free (cseq, cseq_mem_chunk);
}



/*----------------------------------------------------------------------
   new_method_handler_table() -- Create a default method handler table.

   Paramaters: none
   Return:  A pointer to the allocated object.

   This routine is so each RTSP object can set up it's own method handlers.
   This is so RTSP objects attached to servers can behave differently
   than those attached to clients by assigning different handlers
   for the different methods.
----------------------------------------------------------------------*/
static RTSPHandlerTable *
new_method_handler_table (void)
{
  RTSPHandlerTable *ht=NULL;

  ht = g_malloc (RTSP_METHOD_LAST * sizeof(RTSPHandlerTable));
  if (ht)
    {
        /* Assign the default method handlers for each possible method,
           the default handlers being simply NULLed out. */

      ht[RTSP_METHOD_OPTIONS].text    = "OPTIONS ";
      ht[RTSP_METHOD_OPTIONS].len     = 8;
      ht[RTSP_METHOD_OPTIONS].handler = NULL;

      ht[RTSP_METHOD_DESCRIBE].text    = "DESCRIBE ";
      ht[RTSP_METHOD_DESCRIBE].len     = 9;
      ht[RTSP_METHOD_DESCRIBE].handler = NULL;

      ht[RTSP_METHOD_ANNOUNCE].text    = "ANNOUNCE ";
      ht[RTSP_METHOD_ANNOUNCE].len     = 9;
      ht[RTSP_METHOD_ANNOUNCE].handler = NULL;

      ht[RTSP_METHOD_SETUP].text    = "SETUP ";
      ht[RTSP_METHOD_SETUP].len     = 6;
      ht[RTSP_METHOD_SETUP].handler = NULL;

      ht[RTSP_METHOD_PLAY].text    = "PLAY ";
      ht[RTSP_METHOD_PLAY].len     = 5;
      ht[RTSP_METHOD_PLAY].handler = NULL;

      ht[RTSP_METHOD_PAUSE].text    = "PAUSE ";
      ht[RTSP_METHOD_PAUSE].len     = 6;
      ht[RTSP_METHOD_PAUSE].handler = NULL;

      ht[RTSP_METHOD_TEARDOWN].text    = "TEARDOWN ";
      ht[RTSP_METHOD_TEARDOWN].len     = 9;
      ht[RTSP_METHOD_TEARDOWN].handler = NULL;

      ht[RTSP_METHOD_GET_PARAMETER].text    = "GET_PARAMETER ";
      ht[RTSP_METHOD_GET_PARAMETER].len     = 14;
      ht[RTSP_METHOD_GET_PARAMETER].handler = NULL;

      ht[RTSP_METHOD_SET_PARAMETER].text    = "SET_PARAMETER ";
      ht[RTSP_METHOD_SET_PARAMETER].len     = 14;
      ht[RTSP_METHOD_SET_PARAMETER].handler = NULL;

      ht[RTSP_METHOD_REDIRECT].text    = "REDIRECT ";
      ht[RTSP_METHOD_REDIRECT].len     = 9;
      ht[RTSP_METHOD_REDIRECT].handler = NULL;

      ht[RTSP_METHOD_RECORD].text    = "RECORD ";
      ht[RTSP_METHOD_RECORD].len     = 7;
      ht[RTSP_METHOD_RECORD].handler = NULL;

        /* non-methods (not true "methods", but we match them just the same) */
      ht[RTSP_METHOD_RESPONSE].text    = "RTSP/1.0";
      ht[RTSP_METHOD_RESPONSE].len     = 8;
      ht[RTSP_METHOD_RESPONSE].handler = NULL;

    }
  
  return ht;
}





/*----------------------------------------------------------------------
   rtsp_get_channel_id() -- Remap a channel id from a $-encoded buffer.

   Paramaters:
     rtsp    - A related RTSP structure/object.
     old_cid - A content-id handed to us by a server.

   Return:
     A new channel id to hand to the client in place of old_cid.

   When stream data is encapsulated within the TCP Control connection
   using the "$" encoding, "channel IDs" are used to differentiate
   the various streams.  The servers allocate these IDs, starting at 1
   and incrementing by one for each stream.  If the client is talking
   to two or more servers, the IDs from the different servers will
   conflict.  We need to remap them in the proxy so the client can
   tell the different streams from one another.  We do this by creating
   a hash table entry for each server/id combination we encounter,
   and mapping it to a new id.  This routine creates one of these
   hash table entries if not already present, and returns the new id.
----------------------------------------------------------------------*/
static char
rtsp_get_channel_id (RTSP *rtsp, char old_cid)
{
  char *cidstr=NULL;
  char *cid_p=NULL;
  static int max_channel_id=0;

  cidstr = g_malloc (MAX_CID_STR_LEN);
  snprintf (cidstr, MAX_CID_STR_LEN, "%d:%d", old_cid, rtsp->rtsp_id);
  cidstr[MAX_CID_STR_LEN-1] = '\0';

  cid_p = g_hash_table_lookup (channel_id_hash, cidstr);
  if (! cid_p)
    {  /* haven't mapped it yet (should have, though), so do so now */
      cid_p = g_malloc (1);
      *cid_p = ++max_channel_id;
      g_hash_table_insert (channel_id_hash, cidstr, cid_p);
      log_message (DEBUG_LOG,
                   "rtsp_get_channel_id: MAPPING CID %d TO NEWCID %d KEY=%s",
                   old_cid, *cid_p, cidstr);
    }
  else
    {
      log_message (DEBUG_LOG,
                   "rtsp_get_channel_id: FOUND CID %d MAPPED TO NEWCID %d KEY=%s",
                   old_cid, *cid_p, cidstr);
    }

  return *cid_p;
}



/*----------------------------------------------------------------------
   rtsp_interleave_update() -- Update an "interleaved=" string.

   Paramaters:
     rtsp - The related rtsp object.
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.

   Return:  nothing

   If we have an interleave paramater for a Transports: header, we may
   need to remap the value in order to support interleaving multiple
   back-end servers over a TCP-only connection.  The interleave numbers
   indicate which channel ids the server is going to use for encapsulating
   stream data in the RTSP Control stream (see rtsp_map_channel_ids()
   and rtsp_get_channel_id().)
----------------------------------------------------------------------*/
static void
rtsp_interleave_update (RTSP *rtsp, char *buff, int *n)
{
  int pos=0, i=0;
  char *trans_str=NULL;
  int trans_str_len=0;

  if ( (pos = cistrnfind ("transport:", 10, buff, *n)) < 0)
    return;

  trans_str     = buff + pos;
  trans_str_len = *n - pos;


  /* parse forward until we encounter a ',' or a carriage return;
   * that will be the end of this particular transport string
   */
  for (i = 0; i < trans_str_len; i++)
    if (trans_str[i] == ',' || trans_str[i] == '\r')
      break;

  trans_str_len = i; /* set length at previous charactor */


  if ( (pos = cistrnfind ("interleaved=", 12, trans_str, trans_str_len)) >= 0)
    {
      char cid_str[MAX_CID_STR_LEN]="";
      int  cid_str_len=0;
      char *inter_str=NULL;
      int inter_str_len=0;
      int num1=0, num2=0, newnum1=0, newnum2=0;
      unsigned short int snum1=0, snum2=0;

      inter_str = trans_str + pos + 11;
      inter_str_len = trans_str_len - 11;

      for (i = 0; i < inter_str_len; ++i)
        if (inter_str[i] == ';' || inter_str[i] == ',' || inter_str[i] == '\r')
          break;
      inter_str_len = i;

      get_num_range (inter_str, inter_str_len, &snum1, &snum2);
      num1=snum1;
      num2=snum2;

      /* note that we're passing the rtsp object for the server side of
         the universe.)
       */
      rtsp_map_channel_ids (rtsp, num1, num2, &newnum1, &newnum2);
      if (num1 == num2)
          snprintf (cid_str, MAX_CID_STR_LEN, "%d", newnum1);
      else
          snprintf (cid_str, MAX_CID_STR_LEN, "%d-%d", newnum1, newnum2);
      cid_str[MAX_CID_STR_LEN-1] = '\0';
      cid_str_len = strlen (cid_str);


      /* make room for the new cidstr */
      memmove (inter_str + cid_str_len + 1,
               inter_str + inter_str_len,
               *n - (inter_str - buff + inter_str_len) );
      /* insert the new cidstr */
      memcpy (inter_str + 1, cid_str, cid_str_len);
      trans_str_len = trans_str_len - inter_str_len + cid_str_len + 1;
      *n = *n - inter_str_len + cid_str_len + 1;
    }
  else
    {
      /* log_message (DEBUG_LOG, "no 'interleaved' found"); */
    }

  return;
}



/*----------------------------------------------------------------------
   rtsp_session_update() -- Update/remap a Session ID.

   Paramaters:
     rtsp - The related rtsp object.
     from - The direction the message is traveling, FROM_CLIENT or FROM_SERVER.
     buff - A buffer containing the method being processed.
     n    - The size of the buffer.

   Return:  nothing

   If we have a "Session:", header, we may need to remap the value in
   order to support multiple back-end servers over a single client<->proxy
   RTSP Control connection.  Multiple servers could potentially choose
   the same session id string, so we do this to eliminate this (remote)
   possibility.
----------------------------------------------------------------------*/
static void
rtsp_session_update (RTSP *rtsp, RTSPFromType from, char *buff, int *n)
{
  int pos=0, i=0;
  char *session_str=NULL;
  int session_str_len=0;

  if (rtsp->rtsp_objtype != SERVER_OBJECT)
    return;

  if (from == FROM_SERVER)
    {
      if ( (pos = cistrnfind ("session:", 8, buff, *n)) >= 0)
        {
          session_str     = buff + pos + 9;
          session_str_len = *n - pos - 9;
        }
      else
        return;
    }
  else if (from == FROM_CLIENT)
    {
      if ( (pos = cistrnfind ("session:", 8, buff, *n)) >= 0)
        {
          session_str     = buff + pos + 9;
          session_str_len = *n - pos - 9;
        }
      else
        return;
    }
  else
    return;


  /* parse forward until we encounter a carriage return;
   * that will be the end of this particular header
   */
  for (i = 0; i < session_str_len; i++)
    if (session_str[i] == '\r')
      break;

  session_str_len = i; /* set length at previous charactor */

  if (session_str_len > 0)
    {
       GList *list = rtsp->rtsp_list;
       RTSP *crtsp=NULL;

       if (list) /* get the RTSP id for the client */
         {
           while (list->prev)
             list = list->prev;
           crtsp = list->data;
         }

      if (from == FROM_SERVER)
        {
          char * sess_key=g_malloc (session_str_len + 1);
          SessionIDMapping *sess_map=NULL;

          strncpy (sess_key, session_str, session_str_len);
          sess_key[session_str_len] = '\0';
          sess_map = g_hash_table_lookup (rtsp->rtsp_session_map, sess_key);
          if (!sess_map)
            {
              char * new_id=NULL;
              /* note that if the session id is not already in use, we
                 simply get back what we already have.
               */
              new_id = generate_session_id (crtsp, session_str, session_str_len);
              sess_map = session_map_new (session_str, session_str_len,
                                          new_id, strlen (new_id),
                                          rtsp->rtsp_id, crtsp->rtsp_id);
              g_hash_table_insert (rtsp->rtsp_session_map,
                                   sess_map->server_session,
                                   sess_map);
              g_hash_table_insert (crtsp->rtsp_session_map,
                                   sess_map->client_session,
                                   sess_map);

            }
#ifdef DEBUG
          if (strncmp (sess_key, sess_map->client_session, session_str_len) != 0)
            log_message (DEBUG_LOG, "SERVER SESSION ID: WAS %s NOW %s",
                         sess_key, sess_map->client_session);
#endif /* DEBUG */
          g_free (sess_key);

          /* remap the session id (possibly identical to what we have) */
          strncpy (session_str, sess_map->client_session, session_str_len);

        }
      else
        {
          char * sess_key=g_malloc (session_str_len + 1);
          SessionIDMapping *sess_map=NULL;

          strncpy (sess_key, session_str, session_str_len);
          sess_key[session_str_len] = '\0';
          sess_map = g_hash_table_lookup (crtsp->rtsp_session_map, sess_key);
          if (!sess_map)
            {
              char * new_id=NULL;
              log_message (DEBUG_LOG, "YO! MAPPING CLIENT SESSION ID: %s", sess_key);
              /* note that if the session id is not already in use, we
                 simply get back what we already have.
               */
              new_id = generate_session_id (rtsp, session_str, session_str_len);
              sess_map = session_map_new (new_id, strlen (new_id),
                                          session_str, session_str_len,
                                          rtsp->rtsp_id, crtsp->rtsp_id);
              g_hash_table_insert (rtsp->rtsp_session_map,
                                   sess_map->server_session,
                                   sess_map);
              g_hash_table_insert (crtsp->rtsp_session_map,
                                   sess_map->client_session,
                                   sess_map);
            }

#ifdef DEBUG
          if (strncmp (sess_key, sess_map->server_session, session_str_len) != 0)
            log_message (DEBUG_LOG, "CLIENT SESSION ID: WAS %s NOW %s",
                         sess_key, sess_map->server_session);
#endif /* DEBUG */

          /* remap the session id (possibly identical to what we have) */
          strncpy (session_str, sess_map->server_session, session_str_len);

          g_free (sess_key);
        }       
    }

  return;
}



/*----------------------------------------------------------------------
   session_map_new -- allocate a new SessionIDMapping structure.

   Paramaters:
     server_session     - The server's idea of the session ID.
     server_session_len - Length of server_session.
     client_session     - The client's idea of the session ID.
     client_session_len - Length of client_session.
     server_rtspid      - The RTSP id for the server.
     client_rtspid      - The RTSP id for the client.

   Return:  The allocated structure.
----------------------------------------------------------------------*/
static SessionIDMapping *
session_map_new (char *server_session, int server_session_len,
                 char *client_session, int client_session_len,
                 int server_rtspid,
                 int client_rtspid)
{
  SessionIDMapping * sess_map = g_malloc(sizeof(SessionIDMapping));

  sess_map->server_session = g_malloc (server_session_len + 1);
  strncpy (sess_map->server_session, server_session, server_session_len);
  sess_map->server_session[server_session_len] = '\0';

  sess_map->client_session = g_malloc (client_session_len + 1);
  strncpy (sess_map->client_session, client_session, client_session_len);
  sess_map->client_session[client_session_len] = '\0';

  sess_map->server_rtspid = server_rtspid;
  sess_map->client_rtspid = client_rtspid;

  return sess_map;
}



/*----------------------------------------------------------------------
   session_map_destroy - deallocate a SessionIDMapping structure.

   Paramaters:
     sess - The structure to destroy.

   Return:  nothing
----------------------------------------------------------------------*/
static void
session_map_destroy (SessionIDMapping *sess)
{
  if (sess)
    {
      if (sess->server_session)
        g_free (sess->server_session);
      if (sess->client_session)
        g_free (sess->client_session);

      g_free (sess);
    }
}



/*----------------------------------------------------------------------
   generate_session_id - Generate a unique Session ID.

   Paramaters:
     rtsp - the RTSP object in which to check for the session id.
     str  - The original session id we're remapping.
     len  - The length of str.

   Return:
     The replacement session id, a character string of length len.

   This takes the original session id string, checks if it already
   exists, if it does, it mutates it and checks again, looping until success.
----------------------------------------------------------------------*/
static char *
generate_session_id (RTSP *rtsp, char *str, int len)
{
  char * new_session_id = g_malloc (len + 1);

  strncpy (new_session_id, str, len);
  new_session_id[len] = '\0';

  /* look and see if the new id exists, if so regenerate */
  while (g_hash_table_lookup (rtsp->rtsp_session_map, new_session_id) != NULL)
    {
      if (! mutate_session_id (new_session_id, len))
        break;
    }

  return new_session_id;
}




/*----------------------------------------------------------------------
   mutate_session_id - modify a session id string (in-place)

   Paramaters:
     str - The session id string to alter.
     len - The length of the session id string.

   Return:
     1 - string was altered
     0 - string was not altered

   This should almost never happen, if the servers are really
   selecting session IDs at random.  We really only mutate letters
   and numbers, although we should be able to mutate anything.
----------------------------------------------------------------------*/
static int
mutate_session_id (char *str, int len)
{
  int i=0, count=0;
  static int seeded=0;

  if (! str || len <= 0)  /* can't mutate what doesn't exist */
    return 0;

    /* this should be moved elsewhere if we use randomness in other places */
  if (!seeded)
    {
      SRANDOM(time(NULL));
      seeded = 1;
    }
  
	/* pick a starting character from the session id at random */
  i = (int) ( 1.0 * len * RANDOM() / (RAND_MAX + 1.0) );

  while (count < len)
    {
	/* mutate it if it is a number or a letter */
      if (ISDIGIT(str[i]))
        {
          ++str[i];
          if (!ISDIGIT(str[i]))    /* > '9' */
            str[i] = '0';
          return 1; /* mutation suceeded */
        }
      if (ISLOWER(str[i]))
        {
          ++str[i];
          if (str[i] > 'z')
          if (!ISLOWER(str[i]))    /* > 'z' */
            str[i] = 'a';
          return 1; /* mutation suceeded */
        }
      if (ISUPPER(str[i]))
        {
          ++str[i];
          if (!ISUPPER(str[i]))    /* > 'Z' */
            str[i] = 'A';
          return 1; /* mutation suceeded */
        }

	/* try the next character, wrapping to start if necessary */
      ++i;
      if (i == len)
        i = 0;

	/* increase counter so we don't loop forever on a strange session id */
      ++count;
    }

  return 0; /* mutation failed, perhaps no letters or numbers in string? */
}




/*----------------------------------------------------------------------
   rtsp_map_channel_ids() -- Remap channel ids.

   Paramaters:
     rtsp - The related rtsp object.
     num1 - The first channel id number to remap.
     num2 - The last channel id number to remap.
     newnum1 - A replacement channel id for num1.
     newnum2 - A replacement channel id for num2.

   Return:  nothing

   All numbers between num1 and num2 will be remapped to better
   support ranges.  It can safely be assumed that they will be
   mapped in numerical order without gaps.  To map only one id,
   call with num1=num2.  See rtsp_get_channel_id().
----------------------------------------------------------------------*/
static void
rtsp_map_channel_ids (RTSP *rtsp, int num1, int num2, int *newnum1, int *newnum2)
{
  char c='\0';
  int  n=0;

  /* if it's a range we should remap all the numbers between */
  for (c=num1; c <= num2 && num1 <= num2; ++c)
    {
      /* rtsp_get_channel_id() will allocate a new id if not found which
         in this case it won't be
       */
      n = rtsp_get_channel_id (rtsp, c);

      if ((int)c == num1)
        *newnum1 = n;

      if ((int)c == num2)
        *newnum2 = n;
    }
}



/*----------------------------------------------------------------------
   rtsp_remap_cseq_id() - Remap the CSeq header in an RTSP method.

   Paramaters:
     rtsp   - The related rtsp object (should be a server object).
     from   - The direction the message is passing, FROM_SERVER or FROM_CLIENT.
     method - The type of RTSP method in the buffer
     buff   - A buffer containing the method being processed.
     len    - The size of the buffer.

   Return:
     -1 on error
      0 on did nothing
      1 on success

   We want to give each server a monotonically incrementing series of
   CSeq numbers.  The client gives us numbers that increase monotonically,
   but across all servers (CSeq 1 might go to server 1 while CSeq 2 might
   go to server 2; both server 1 and server 2 want to see a 1).
   We toss the mappings into a hash table, hashing it twice so it can
   be looked up from whichever direction we're traveling in the future,
   client->server or server->client.

   We use the client's idea of the CSeq number everywhere internally,
   since they're unique, so we remap the client's RTSP requests at the
   end of processing, just before writing the message to the wire.
   Conversely, we remap the server's RTSP responses immediatly after
   we have a complete RTSP method (header), before we do any real
   processing, and before calling the method handler.

   Note that we may have CSeq numbers generated by the servers.
   These are the "Ping" messages, and potentially other things.
   For this reason, each RTSP object has it's own hash table of
   CSeq mappings.
----------------------------------------------------------------------*/
static int
rtsp_remap_cseq_id (RTSP *rtsp, RTSPFromType from, RTSPMethodType method, char * buff, int * len)
{
  char *old_cseq_str=NULL; /* what is being remapped */
  char *new_cseq_str=NULL; /* what old_cseq_str was remapped to */
  char *seqnum=NULL;
  int old_cseq=0;
  int *p_next_cseq=NULL;
  RTSP *crtsp=NULL;
  GList *list=NULL;
  GHashTable *cseq_map_hash=NULL;

  log_message (DEBUG_LOG,
               "rtsp_remap_cseq_id() called, from=%d method=%d objtype=%d",
               from, method, rtsp->rtsp_objtype);

  /* only remap the CSeq number when on the server side of the universe */
  if (rtsp->rtsp_objtype != SERVER_OBJECT)
    return -1;

  /* extract the CSeq number from the buffer */
  old_cseq = get_cseq (buff, *len);
  if (old_cseq <= 0)  /* bail if no CSeq: header found */
    return -1;

  log_message (DEBUG_LOG,
               "rtsp_remap_cseq_id: server object %d, cseq=%d",
               rtsp->rtsp_id, old_cseq);

  /* look up the client's RTSP object */
  list = rtsp->rtsp_list;
  if (list)
    {
      while (list->prev)
        list = list->prev;
      crtsp = list->data;
    }
  
  if (crtsp == NULL)
    return -1;

  if ( (from == FROM_CLIENT && method != RTSP_METHOD_RESPONSE) ||
       (from == FROM_SERVER && method == RTSP_METHOD_RESPONSE)  )
    {
      cseq_map_hash = crtsp->rtsp_cseq_map;
      p_next_cseq = &(crtsp->rtsp_last_cseq_id);
    }
  else
    {
      cseq_map_hash = rtsp->rtsp_cseq_map;
      p_next_cseq = &(crtsp->rtsp_last_server_cseq_id);
    }



  /* for RTSP requests coming from either the client or server,
     we remap the CSeq number.
   */
  if (method != RTSP_METHOD_RESPONSE)
    {
      /* create old hash string for this mapping (what is being remapped) */
      old_cseq_str = g_malloc (MAX_CSEQ_STR_LEN);
      snprintf (old_cseq_str, MAX_CSEQ_STR_LEN, "%d", old_cseq);
      old_cseq_str[MAX_CSEQ_STR_LEN-1] = '\0';

      log_message (DEBUG_LOG, "rtsp_remap_cseq_id: old_cseq_str=%s",
                   old_cseq_str);

      /* see if it already exist */
      new_cseq_str = g_hash_table_lookup (cseq_map_hash, old_cseq_str);
      if (! new_cseq_str)
        {
          /* create the new hash string for this mapping */
          new_cseq_str = g_malloc (MAX_CSEQ_STR_LEN); 
          ++ (*p_next_cseq);
          snprintf (new_cseq_str, MAX_CSEQ_STR_LEN, "%d:%d", rtsp->rtsp_id,
                    *p_next_cseq);
          new_cseq_str[MAX_CSEQ_STR_LEN-1] = '\0';

          /* insert the key/value pair_s_  into the hash table.
             We map "x", where x is the original CSeq number,  to "y:z",
             where y is the destination RTSP id and z is the destination CSeq
             number.  We also map "y:z" to  "x" so we can look it up
             when traffic is going the other direction.
          */
          g_hash_table_insert (cseq_map_hash, old_cseq_str, new_cseq_str);
          g_hash_table_insert (cseq_map_hash, new_cseq_str, old_cseq_str);
          log_message (DEBUG_LOG,
               "rtsp_remap_cseq_id: MAPPING CLIENT'S CSEQ %s TO %d (SERVER=%d)",
               old_cseq_str, *p_next_cseq, rtsp->rtsp_id);

          /* replace CSeq number in outgoing buffer */
          seqnum = g_malloc (MAX_CSEQ_STR_LEN); 
          snprintf (seqnum, MAX_CSEQ_STR_LEN, "%d", *p_next_cseq);
          seqnum[MAX_CSEQ_STR_LEN-1] = '\0';
          replace_cseq_number (seqnum, strlen(seqnum), buff, len);
          g_free (seqnum);
          seqnum = NULL;

          return 1;
        }
      else /* already exists (shouldn't though) */
        {
          log_message (DEBUG_LOG,
                       "warning:  RTSP Request from client contains "
                       "duplicate CSeq number");
          g_free (old_cseq_str);
        }
    }


  /* for RTSP responses coming from either the client or server,
     we lookup the CSeq number (which we should have saved earlier above).
   */
  if (method == RTSP_METHOD_RESPONSE)
    {
      /* create the server hash string for this mapping */
      new_cseq_str = g_malloc (MAX_CSEQ_STR_LEN); 
      snprintf (new_cseq_str, MAX_CSEQ_STR_LEN, "%d:%d",
                rtsp->rtsp_id, old_cseq);

      /* see if it exist (it should) */
      old_cseq_str = g_hash_table_lookup (cseq_map_hash, new_cseq_str);
      if (old_cseq_str)
        {
          replace_cseq_number (old_cseq_str, strlen(old_cseq_str), buff, len);
          g_free (new_cseq_str);
        }
      else
        {
          /* We've seen an RTSP response before the corresponding RTSP request,
             which is wrong */
          log_message (ERROR_LOG, "warning:  RTSP Response seen before "
                       "corresponding RTSP Request");
        }


    }
 
  return -1;
}




/*----------------------------------------------------------------------
   replace_cseq_number - Replace the CSeq: value in an RTSP method header.

   Paramaters:
     str    - The new CSeq value as a string.
     slen   - The length of str.
     buff   - A buffer containing the method being processed.
     len    - The size of the buffer.

   Return:  nothing

   This does the actual buffer substitution needed by rtsp_remap_cseq_id().
----------------------------------------------------------------------*/
static void
replace_cseq_number (char * str, int slen, char * buff, int * len)
{
  int pos=0, i=0;
  char *cseq_str=NULL;
  int cseq_str_len=0;

  if ( (pos = cistrnfind ("\r\ncseq:", 7, buff, *len)) < 0)
    return;

  cseq_str = buff + pos + 8;
  cseq_str_len = *len - pos - 8;

  for (i = 0; i < cseq_str_len; ++i)
    if (! ISDIGIT(cseq_str[i]))
      break;
  cseq_str_len = i;

  memmove (cseq_str + slen,
           cseq_str + cseq_str_len,
           *len - (cseq_str - buff) - cseq_str_len);
  memcpy (cseq_str, str, slen);
  *len = *len - cseq_str_len + slen;

}
