# MD060 - Table Format

**Aliases:** `table-format`

**Enabled by default:** No (opt-in)

**Fixable:** Yes

## Rationale

Tables in Markdown are significantly easier to read and maintain when their columns are properly aligned in source form. While Markdown tables render the same regardless of spacing, aligned tables in
source code:

- Improve readability during code review
- Make editing cells more intuitive
- Provide better developer experience
- Ensure consistency across documents

This rule enforces consistent column alignment and padding in Markdown tables, matching the behavior of popular formatters like Prettier.

## Configuration

```toml
[MD060]
enabled = false              # Default: opt-in for conservative adoption
style = "aligned"            # Options: "aligned", "aligned-no-space", "compact", "tight", "any"
max-width = 0                # Default: inherit from MD013's line-length
column-align = "auto"        # Options: "auto", "left", "center", "right"
column-align-header = "auto" # Override alignment for header row only
column-align-body = "auto"   # Override alignment for body rows only
loose-last-column = false    # Allow unpadded last column in body rows
```

### Options

#### `enabled`

**Type:** `boolean`
**Default:** `false`

Whether to enable MD060 table formatting. This rule is disabled by default to allow gradual adoption, as it can make significant changes to existing tables.

#### `style`

**Type:** `string`
**Default:** `"any"`
**Values:** `"aligned"`, `"aligned-no-space"`, `"compact"`, `"tight"`, `"any"`

Controls the table formatting style:

- **`aligned`**: Columns are padded with spaces for visual alignment (recommended for readability)
- **`aligned-no-space`**: Like `aligned`, but delimiter row has no spaces around dashes (see below)
- **`compact`**: Minimal spacing with single spaces between pipes and content
- **`tight`**: No spacing, pipes directly adjacent to content
- **`any`**: Preserve existing formatting style (no enforcement)

#### `max-width`

**Type:** `number`
**Default:** `0` (inherit from MD013)

Maximum table width before auto-switching to compact mode.

- **`0`** (default): Smart inheritance from MD013's settings (see below)
- **Non-zero**: Explicit max width threshold, independent of MD013

When a table's aligned width would exceed this limit, MD060 automatically uses compact formatting instead to prevent excessively long lines. This intelligent auto-compacting matches Prettier's table
formatting behavior.

**Smart Inheritance Behavior (when `max-width = 0`):**

When `max-width` is not explicitly set, MD060 intelligently determines whether to apply width limits:

| MD013 Configuration                         | Effective max-width             |
| ------------------------------------------- | ------------------------------- |
| MD013 disabled                              | **Unlimited** (no auto-compact) |
| `MD013.tables = false`                      | **Unlimited** (no auto-compact) |
| `MD013.line-length = 0`                     | **Unlimited** (no auto-compact) |
| `MD013.tables = true` and `line-length = N` | **Inherits N**                  |

This means if you don't care about table line length (by disabling MD013 or setting `tables = false`), MD060 won't force tables to compact. Your aligned tables will stay aligned regardless of width.

**Why this matters:** Wide tables with many columns or long content can produce extremely long lines when aligned. Auto-compacting ensures tables don't violate line length limits while still
maintaining alignment where practical. However, if you've explicitly opted out of line length checks for tables, MD060 respects that choice.

#### `column-align`

**Type:** `string`
**Default:** `"auto"`
**Values:** `"auto"`, `"left"`, `"center"`, `"right"`

Controls how cell text is aligned within the padded column width:

- **`auto`** (default): Respects alignment indicators from the delimiter row (`:---` for left, `:---:` for center, `---:` for right)
- **`left`**: Forces all columns to left-align text (content on left, padding on right)
- **`center`**: Forces all columns to center text (padding split on both sides)
- **`right`**: Forces all columns to right-align text (padding on left, content on right)

**Note:** This option only applies when `style = "aligned"` or `style = "aligned-no-space"`. It has no effect on `compact` or `tight` styles (which have minimal or no padding).

**Example:**

```toml
[MD060]
enabled = true
style = "aligned"
column-align = "right"  # Right-align all cell content
```

#### `column-align-header`

**Type:** `string` (optional)
**Default:** Not set (falls back to `column-align`)
**Values:** `"auto"`, `"left"`, `"center"`, `"right"`

Overrides `column-align` specifically for the header row. When set, the header row uses this alignment while body rows use `column-align` (unless overridden by `column-align-body`).

**Example:**

```toml
[MD060]
enabled = true
style = "aligned"
column-align-header = "center"  # Center header text
column-align-body = "left"      # Left-align body text
```

#### `column-align-body`

**Type:** `string` (optional)
**Default:** Not set (falls back to `column-align`)
**Values:** `"auto"`, `"left"`, `"center"`, `"right"`

Overrides `column-align` specifically for body rows (non-header, non-delimiter). When set, body rows use this alignment while the header row uses `column-align` (unless overridden by `column-align-header`).

**Example:**

```toml
[MD060]
enabled = true
style = "aligned"
column-align = "left"           # Default for both
column-align-body = "right"     # Override body to right-align
```

#### `loose-last-column`

**Type:** `boolean`
**Default:** `false`

When enabled, the last column in body rows is not padded to match the header width. This is useful for tables where the last column contains descriptions or other variable-length content that would waste space if padded.

The header and delimiter rows remain fully aligned, but body rows can have shorter or longer last columns.

**Example:**

```toml
[MD060]
enabled = true
style = "aligned"
loose-last-column = true
```

**Result:**

```markdown
| Name   | Status   | Description                                        |
| ------ | -------- | -------------------------------------------------- |
| Foo    | Enabled  | Short |
| Bar    | Disabled | A much longer description that would waste space if padded |
```

Without `loose-last-column`, the "Short" cell would be padded to match the header width, creating unnecessary whitespace.

## Examples

### ❌ Incorrect (unaligned)

```markdown
| Name | Age | City |
|---|---|---|
| Alice | 30 | Seattle |
| Bob | 25 | Portland |
```

### ✅ Correct (aligned)

```markdown
| Name  | Age | City     |
| ----- | --- | -------- |
| Alice | 30  | Seattle  |
| Bob   | 25  | Portland |
```

### Compact Style

When `style = "compact"`:

```markdown
| Name | Age | City |
| --- | --- | --- |
| Alice | 30 | Seattle |
| Bob | 25 | Portland |
```

### Tight Style

When `style = "tight"`:

```markdown
|Name|Age|City|
|---|---|---|
|Alice|30|Seattle|
|Bob|25|Portland|
```

### Column Alignment Examples

When `column-align = "center"`:

```markdown
| Name  | Age |   City   |
| ----- | --- | -------- |
| Alice | 30  | Seattle  |
|  Bob  | 25  | Portland |
```

When `column-align = "right"`:

```markdown
|  Name | Age |     City |
| ----- | --- | -------- |
| Alice |  30 |  Seattle |
|   Bob |  25 | Portland |
```

This is useful for consistently formatting tables with numeric data or when you want a specific visual style across all columns.

### Aligned No-Space Style

When `style = "aligned-no-space"`:

```markdown
| Name  | Age | City     |
|-------|-----|----------|
| Alice | 30  | Seattle  |
| Bob   | 25  | Portland |
```

This style is identical to `aligned` except the delimiter row has no spaces around the dashes. Compare:

- **`aligned`**: `| ----- | --- | -------- |`
- **`aligned-no-space`**: `|-------|-----|----------|`

Both styles produce aligned columns with equal-length rows. The only difference is the delimiter row formatting.
This style is popular in hand-written tables and used by projects like GitLab and Obsidian.

## Auto-Compact Threshold

### Example: Inheriting from MD013

```toml
[MD013]
line-length = 100

[MD060]
enabled = true
style = "aligned"
max-width = 0  # Tables exceeding 100 chars will auto-compact
```

**Behavior:**

- Tables ≤ 100 chars wide: Aligned with full padding
- Tables > 100 chars wide: Automatically compact to stay under limit

### Example: Explicit Threshold

```toml
[MD060]
enabled = true
style = "aligned"
max-width = 120  # Independent of MD013, uses 120 char threshold
```

## Unicode Support

MD060 properly handles various Unicode characters:

### ✅ Supported

**CJK Characters** (Chinese, Japanese, Korean):

```markdown
| Name | City |
| ---- | ---- |
| 中文 | 東京 |
```

**Basic Emoji:**

```markdown
| Status | Name |
| ------ | ---- |
| ✅     | Pass |
| ❌     | Fail |
```

The rule correctly measures CJK characters as double-width and aligns columns accordingly.

### ⚠️ Automatically Skipped

Tables containing complex Unicode sequences are automatically skipped to prevent alignment corruption:

- **Zero-Width Joiner (ZWJ) emoji:** 👨‍👩‍👧‍👦, 👩‍💻
- **Zero-Width Space (ZWS):** Invisible word break opportunities
- **Zero-Width Non-Joiner (ZWNJ):** Ligature prevention marks
- **Word Joiner (WJ):** Non-breaking invisible characters

These characters have inconsistent or zero display widths across terminals and fonts, making accurate alignment impossible. The rule preserves these tables as-is rather than risk corrupting them.

**Example (automatically skipped):**

```markdown
| Emoji    | Name      |
| -------- | --------- |
| 👨‍👩‍👧‍👦 | Family    |
| 👩‍💻     | Developer |
```

This is an honest limitation of terminal display technology, similar to what other tools like markdownlint experience.

## Fix Behavior

When applying automatic fixes (`rumdl fmt`), this rule:

1. **Calculates proper display width** for each column using Unicode width measurements
2. **Pads cells** with trailing spaces to align columns
3. **Preserves cell content** exactly (only spacing is modified)
4. **Respects alignment indicators** in delimiter rows:
   - `:---` → Left-aligned
   - `:---:` → Center-aligned
   - `---:` → Right-aligned
5. **Auto-compacts** tables exceeding `max-width` to prevent line length violations
6. **Skips tables with ZWJ emoji** to prevent alignment corruption
7. **Masks inline code blocks** to avoid treating code pipes as table delimiters

## Common Use Cases

### Recommended Configuration (Conservative)

For most projects, start with this conservative configuration:

```toml
[MD013]
line-length = 100

[MD060]
enabled = true        # Enable table formatting
style = "aligned"     # Align columns for readability
max-width = 0         # Auto-compact tables exceeding 100 chars
```

This provides readable tables without creating excessively long lines.

### Strict Compact Mode

For projects preferring minimal formatting:

```toml
[MD060]
enabled = true
style = "compact"
```

### Allow Any Style

To check without enforcing:

```toml
[MD060]
enabled = true
style = "any"  # No enforcement, preserves existing formatting
```

### Unlimited Table Width

For projects that want aligned tables without any width limits:

```toml
[global]
disable = ["MD013"]  # No line length checks = no table compacting

[MD060]
enabled = true
style = "aligned"
max-width = 0  # Will be unlimited since MD013 is disabled
```

Or, if you want MD013 for code but not tables:

```toml
[MD013]
line-length = 80
tables = false  # Skip table line length checks

[MD060]
enabled = true
style = "aligned"
max-width = 0  # Will be unlimited since MD013.tables = false
```

## Integration with Other Rules

**MD013 (Line Length):**

- MD060's `max-width = 0` uses smart inheritance from MD013
- If MD013 is disabled or `tables = false`, tables have **unlimited width** (no auto-compact)
- If MD013 is enabled with `tables = true`, tables are auto-compacted at `line-length`
- Can override with explicit `max-width` value to ignore MD013 entirely

**MD056 (Table Column Count):**

- MD056 validates column count consistency
- MD060 formats tables after MD056 validates structure
- Use both rules together for comprehensive table linting

## Performance Notes

Table formatting is computationally intensive due to Unicode width calculations. MD060 is optimized with:

- Efficient Unicode width caching
- Early bailout for complex Unicode sequences
- Parallel processing in fix mode (when multiple tables exist)

For large documents with many tables, expect formatting to take a few hundred milliseconds.

## Rationale for Default Disabled

MD060 is disabled by default because:

1. **Large diffs:** Enabling on existing codebases can produce massive formatting changes
2. **Opt-in adoption:** Teams should consciously decide to adopt table formatting
3. **Performance:** Unicode width calculations add overhead
4. **Conservative approach:** Matches rumdl's philosophy of opt-in for aggressive formatting

We recommend enabling MD060 for new projects or when doing a formatting cleanup pass.

## See Also

- [MD056](md056.md) - Table Column Count (validates table structure)
- [MD013](md013.md) - Line Length (works with MD060's auto-compact)
- [MD055](md055.md) - Table Pipe Style (validates pipe placement)
