use crate::counter::{fetch_new_count_for_group, update_new_counts};
use crate::db;
use crate::help;
use crate::links::extract_links;
use crate::nntp_client::NntpClient;
use crate::refresh::manual_refresh;
use crate::threads;
use crate::threads::ThreadedArticleView;
use crate::vimode::handle_key_event;
use crate::vimode::render_body;
use crate::vimode::VisualState;

use crossterm::cursor::MoveTo;
use crossterm::event::{self, Event, KeyCode, KeyModifiers};
use crossterm::execute;
use crossterm::terminal::{Clear, ClearType};
use std::io::stdout;
use std::process::Stdio;

use ratatui::{
    backend::CrosstermBackend,
    layout::{Constraint, Direction, Layout, Rect},
    style::{Color, Modifier, Style},
    text::{Line, Span},
    widgets::{Block, Borders, List, ListItem, Paragraph},
    Terminal,
};
use rusqlite::Connection;
use std::error::Error;
use std::sync::Arc;
use tokio::sync::Mutex;

use crate::compose::{compose_post, compose_reply};

#[derive(PartialEq, Eq)]
pub enum View {
    Groups,
    Articles,
    Article,
}

pub struct AppState {
    pub subscriptions: Vec<String>,
    pub selected_group: usize,
    pub subjects: Vec<(String, String, String, String)>,
    pub read_flags: Vec<bool>,
    pub selected_article: usize,
    pub scroll_offset: usize,
    pub current_body: Option<String>,
    pub current_view: View,
    pub status: Option<String>,
    pub article_scroll: u16,
    pub links_in_article: Vec<String>,
    pub show_link_menu: bool,
    pub selected_link: usize,
    pub group_scroll: usize,
    pub threaded_subjects: Vec<ThreadedArticleView>,
    pub last_seen_id: u32,
    pub threaded_mode: bool,
    pub new_counts: Vec<u32>,
    pub visual: VisualState,
}

impl AppState {
    pub fn new(subs: Vec<String>, threaded_mode: bool) -> Self {
        AppState {
            subscriptions: subs,
            selected_group: 0,
            subjects: Vec::new(),
            read_flags: Vec::new(),
            selected_article: 0,
            scroll_offset: 0,
            current_body: None,
            current_view: View::Groups,
            status: None,
            article_scroll: 0,
            links_in_article: Vec::new(),
            show_link_menu: false,
            selected_link: 0,
            group_scroll: 0,
            threaded_subjects: Vec::new(),
            last_seen_id: 0,
            threaded_mode,
            new_counts: Vec::new(),
            visual: VisualState::new(),
        }
    }
}

pub async fn run_ui(
    terminal: &mut Terminal<CrosstermBackend<std::io::Stdout>>,
    mut state: AppState,
    client: Arc<Mutex<NntpClient>>,
    db: Arc<Mutex<Connection>>,
    username: String,
    email: String,
) -> Result<(), Box<dyn Error>> {
    // here lets get the new articles from the helper at conunter.rs
    state.new_counts = vec![0; state.subscriptions.len()];
    update_new_counts(&client, &db, &state.subscriptions, &mut state.new_counts).await;
    loop {
        terminal.draw(|f| {
            let area = f.area();
            let chunks = Layout::default()
                .direction(Direction::Vertical)
                .constraints([
                    Constraint::Length(3),
                    Constraint::Min(1),
                    Constraint::Length(3),
                ])
                .split(area);

            let header = Paragraph::new("Usenet Reborn - The OG social network")
                .style(
                    Style::default()
                        .fg(Color::Yellow)
                        .add_modifier(Modifier::BOLD),
                )
                .block(Block::default().borders(Borders::ALL));
            f.render_widget(header, chunks[0]);

            match state.current_view {
                View::Groups => {
                    // split the screen: left is our newsgroups panel
                    let cols = Layout::default()
                        .direction(Direction::Horizontal)
                        .constraints([Constraint::Length(45), Constraint::Min(10)])
                        .split(chunks[1]);

                    // Compute visible window in the sidebar
                    let sidebar = cols[0];
                    // Reserve 2 rows for the top/bottom borders
                    let max_rows = sidebar.height.saturating_sub(2) as usize;
                    // start/end indices into subscriptions
                    let start = state.group_scroll.min(state.subscriptions.len());
                    let end = (start + max_rows).min(state.subscriptions.len());
                    let window = &state.subscriptions[start..end];

                    // Build the ListItems, marking the real selected_group
                    let items: Vec<ListItem> = window
                        .iter()
                        .enumerate()
                        .map(|(i, grp)| {
                            let idx = start + i; // real index in subscriptions[]
                            let symbol = if idx == state.selected_group {
                                "> "
                            } else {
                                "  "
                            };
                            let count = state.new_counts[idx];
                            let count_label = if count > 0 {
                                format!(" ({})", count)
                            } else {
                                String::new()
                            };
                            let content = format!("{}{}{}", symbol, grp, count_label);

                            let style = if idx == state.selected_group {
                                Style::default()
                                    .fg(Color::LightGreen)
                                    .add_modifier(Modifier::BOLD)
                            } else {
                                Style::default().fg(Color::White)
                            };
                            ListItem::new(Span::styled(content, style))
                        })
                        .collect();

                    let grp_list = List::new(items)
                        .block(Block::default().borders(Borders::ALL).title("Newsgroups"));
                    f.render_widget(grp_list, sidebar);

                    help::render_help(f, cols[1]);
                }

                View::Articles => {
                    let cols = Layout::default()
                        .direction(Direction::Horizontal)
                        .constraints([Constraint::Length(45), Constraint::Min(10)])
                        .split(chunks[1]);

                    let grp_items: Vec<ListItem> = state
                        .subscriptions
                        .iter()
                        .enumerate()
                        .map(|(i, grp)| {
                            let selected = i == state.selected_group;
                            let symbol = if selected { "> " } else { "  " };
                            let count = state.new_counts[i];
                            let count_label = if count > 0 {
                                format!(" ({})", count)
                            } else {
                                String::new()
                            };
                            let content = format!("{}{}{}", symbol, grp, count_label);

                            let span = if selected {
                                Span::styled(
                                    content,
                                    Style::default()
                                        .fg(Color::LightGreen)
                                        .add_modifier(Modifier::BOLD),
                                )
                            } else {
                                Span::styled(content, Style::default().fg(Color::White))
                            };
                            ListItem::new(span)
                        })
                        .collect();
                    let grp_list = List::new(grp_items)
                        .block(Block::default().borders(Borders::ALL).title("Newsgroups"));
                    f.render_widget(grp_list, cols[0]);

                    let visible_rows = (cols[1].height as usize).saturating_sub(2);

                    let list_len = if state.threaded_mode {
                        state.threaded_subjects.len()
                    } else {
                        state.subjects.len()
                    };

                    let start = state.scroll_offset.min(list_len);
                    let end = (start + visible_rows).min(list_len);

                    let subj_items = if state.threaded_mode {
                        let window = &state.threaded_subjects[start..end];
                        threads::render_threaded_articles(
                            window,
                            start,
                            state.selected_article,
                            &state.read_flags,
                            cols[1].width as usize - 2,
                            state.last_seen_id,
                        )
                    } else {
                        let window = &state.subjects[start..end];
                        threads::render_articles(
                            state.threaded_mode,
                            &[],
                            window,
                            start,
                            state.selected_article,
                            &state.read_flags,
                            cols[1].width as usize - 2,
                            state.last_seen_id,
                        )
                    };

                    let subj_list =
                        List::new(subj_items).block(Block::default().borders(Borders::ALL).title(
                            format!("Articles in {}", state.subscriptions[state.selected_group]),
                        ));
                    f.render_widget(subj_list, cols[1]);
                }

                View::Article => {
                    if let Some(body) = &state.current_body {
                        // let lines = highlight_links_in_body(body);
                        let lines = render_body(
                            &state.visual,
                            body,
                            state.article_scroll,
                            chunks[1].height as usize - 2, // subtract borders
                            chunks[1].width as usize,
                        );
                        let paragraph = Paragraph::new(lines)
                            .block(Block::default().borders(Borders::ALL).title("Body"));
                        // .scroll((state.article_scroll, 0));
                        f.render_widget(paragraph, chunks[1]);
                    }
                }
            }
            // draw link‐selection popup if active
            if state.current_view == View::Article && state.show_link_menu {
                let area = f.area();
                let width = 60;
                let height = (state.links_in_article.len().min(10) as u16) + 2;
                let popup = Rect {
                    x: (area.width.saturating_sub(width)) / 2,
                    y: (area.height.saturating_sub(height)) / 2,
                    width,
                    height,
                };
                let items: Vec<ListItem> = state
                    .links_in_article
                    .iter()
                    .enumerate()
                    .map(|(i, url)| {
                        let style = if i == state.selected_link {
                            Style::default()
                                .fg(Color::Yellow)
                                .add_modifier(Modifier::BOLD)
                        } else {
                            Style::default().fg(Color::White)
                        };
                        ListItem::new(Span::styled(url.clone(), style))
                    })
                    .collect();
                let list = List::new(items)
                    .block(Block::default().borders(Borders::ALL).title("Open link"));
                f.render_widget(list, popup);
            }

            // start
            // Always show group & selected‐article in the footer when in Articles view.
            let footer_text = match state.current_view {
                View::Articles => {
                    let selected_subject = if state.threaded_mode {
                        state
                            .threaded_subjects
                            .get(state.selected_article)
                            .map(|s| &s.subject)
                    } else {
                        state
                            .subjects
                            .get(state.selected_article)
                            .map(|(_, s, _, _)| s)
                    };
                    format!(
                        "Articles view | Group: {} | Selected: {}  (Esc to go back)",
                        state.subscriptions[state.selected_group],
                        selected_subject.unwrap_or(&"None".to_string())
                    )
                }
                // All other views still respect `state.status`, falling back to their defaults.
                _ => state
                    .status
                    .clone()
                    .unwrap_or_else(|| match state.current_view {
                        View::Groups => format!(
                            "Groups view | Selected: {}",
                            state.subscriptions[state.selected_group]
                        ),
                        View::Article => {
                            // grab subject + author and date
                            let (subj, from, date) = if state.threaded_mode {
                                let art = &state.threaded_subjects[state.selected_article];
                                (art.subject.clone(), art.sender.clone(), art.date.clone())
                            } else {
                                let (_id, s, sender, date) =
                                    &state.subjects[state.selected_article];
                                (s.clone(), sender.clone(), date.clone())
                            };
                            format!("Article view | {} | From: {}, | {}", subj, from, date)
                        }
                        _ => String::new(),
                    }),
            };
            //
            let mut parts = footer_text.splitn(2, " | ");
            let left = parts.next().unwrap_or("");
            let right = parts.next().unwrap_or("");

            let footer_line = Line::from(vec![
                Span::styled(
                    left,
                    Style::default()
                        .fg(Color::LightGreen)
                        .add_modifier(Modifier::BOLD),
                ),
                Span::raw(" | "),
                Span::raw(right),
            ]);

            // meow with a yellow border
            let footer = Paragraph::new(vec![footer_line]).block(
                Block::default()
                    .borders(Borders::ALL)
                    .style(Style::default().fg(Color::Yellow)),
            );
            // end test
            f.render_widget(footer, chunks[2]);
        })?;

        if event::poll(std::time::Duration::from_millis(200))? {
            if let Event::Key(key) = event::read()? {
                if state.current_view == View::Article {
                    if state.visual.insert_active || state.visual.visual_active {
                        // If inside Insert or Visual mode: handle hjkl etc
                        if handle_key_event(
                            &mut state.visual,
                            key.code,
                            &state.current_body,
                            &mut state.article_scroll,
                            terminal,
                        ) {
                            continue;
                        }
                    } else if key.code == KeyCode::Char('i') {
                        // If outside modes, allow 'i' to enter Insert
                        if handle_key_event(
                            &mut state.visual,
                            key.code,
                            &state.current_body,
                            &mut state.article_scroll,
                            terminal,
                        ) {
                            continue;
                        }
                    }
                }

                //

                if state.show_link_menu {
                    match key.code {
                        KeyCode::Char('q') | KeyCode::Esc => state.show_link_menu = false,
                        KeyCode::Char('j')
                            if state.selected_link + 1 < state.links_in_article.len() =>
                        {
                            state.selected_link += 1
                        }
                        KeyCode::Char('k') if state.selected_link > 0 => state.selected_link -= 1,
                        KeyCode::Enter => {
                            if let Some(url) = state.links_in_article.get(state.selected_link) {
                                let _ = std::process::Command::new("xdg-open")
                                    .arg(url)
                                    .stdout(Stdio::null())
                                    .stderr(Stdio::null())
                                    .spawn();
                                state.status = Some(format!("🌐 Opened: {}", url));
                            }
                            state.show_link_menu = false;
                        }
                        _ => {}
                    }
                    continue;
                }
                if key.modifiers.contains(KeyModifiers::CONTROL) {
                    if key.code == KeyCode::Char('h') {
                        state.current_view = View::Groups;
                        // meow update meow new group counter
                        // fast, single‐group refresh:
                        let g = &state.subscriptions[state.selected_group];
                        state.new_counts[state.selected_group] =
                            fetch_new_count_for_group(&client, &db, g).await;

                        continue;
                    }
                    if key.code == KeyCode::Char('l') {
                        state.current_view = View::Articles;
                        continue;
                    }
                    // Ctrl+R (in Articles view): mark all visible articles read
                    if key.code == KeyCode::Char('r') && state.current_view == View::Articles {
                        // Mark in-memory flags
                        for flag in state.read_flags.iter_mut() {
                            *flag = true;
                        }
                        // Persist to SQLite to tired now but will need to refactor outside ui.rs
                        let group = state.subscriptions[state.selected_group].clone();
                        let ids: Vec<String> = if state.threaded_mode {
                            state
                                .threaded_subjects
                                .iter()
                                .map(|a| a.id.clone())
                                .collect()
                        } else {
                            state
                                .subjects
                                .iter()
                                .map(|(id, _, _, _)| id.clone())
                                .collect()
                        };
                        let db2 = db.clone();
                        tokio::task::spawn_blocking(move || {
                            let conn = db2.blocking_lock();
                            for id in ids {
                                let _ = db::mark_as_read(&conn, &group, &id);
                            }
                        })
                        .await
                        .ok();
                        state.status = Some("✅ All articles marked read".into());
                        continue;
                    }
                    // Meow refresh for new articles and refresh counters of new articles
                    if key.code == KeyCode::Char('u') {
                        // Call the manual refresh helper
                        manual_refresh(terminal, &mut state, client.clone(), db.clone()).await;
                        continue;
                    }
                }

                match state.current_view {
                    View::Groups => match key.code {
                        KeyCode::Char('q') => return Ok(()),
                        KeyCode::Char('j') => {
                            if state.selected_group + 1 < state.subscriptions.len() {
                                state.selected_group += 1;
                                let max_rows = (terminal.size()?.height as usize).saturating_sub(8);
                                if state.selected_group >= state.group_scroll + max_rows {
                                    state.group_scroll += 1;
                                }
                                state.status = None;
                            }
                        }
                        KeyCode::Char('k') => {
                            if state.selected_group > 0 {
                                state.selected_group -= 1;
                                if state.selected_group < state.group_scroll {
                                    state.group_scroll = state.group_scroll.saturating_sub(1);
                                }
                                state.status = None;
                            }
                        }
                        KeyCode::Enter => {
                            let group = state.subscriptions[state.selected_group].clone();

                            let last_seen: u32 = tokio::task::spawn_blocking({
                                let db = db.clone();
                                let group = group.clone();
                                move || db::get_last_seen(&db.blocking_lock(), &group).unwrap_or(0)
                            })
                            .await
                            .unwrap_or(0);

                            state.last_seen_id = last_seen;

                            let loading_msg = if last_seen == 0 {
                                "Loading articles for the first time… this may take a while"
                            } else {
                                "Checking for new articles…"
                            };
                            state.status = Some(loading_msg.to_string());

                            if let Ok(area) = terminal.size() {
                                let popup = Rect {
                                    x: (area.width.saturating_sub(60)) / 2,
                                    y: (area.height.saturating_sub(3)) / 2,
                                    width: 60,
                                    height: 3,
                                };
                                let _ = terminal.draw(|f| {
                                    let p = Paragraph::new(state.status.clone().unwrap())
                                        .style(Style::default().fg(Color::White).bg(Color::Blue))
                                        .block(
                                            Block::default().borders(Borders::ALL).title("Loading"),
                                        );
                                    f.render_widget(p, popup);
                                });
                            }

                            let new_subjs = client
                                .lock()
                                .await
                                .fetch_new_subjects(&group, last_seen)
                                .await?;
                            //println!("📥 Received {} articles from NNTP", new_subjs.len());

                            if new_subjs.is_empty() {
                                state.status = Some("No new articles.".to_string());

                                let cached = tokio::task::spawn_blocking({
                                    let db = db.clone();
                                    let group = group.clone();
                                    move || {
                                        db::load_threadable_articles(&db.blocking_lock(), &group)
                                            .unwrap_or_default()
                                    }
                                })
                                .await
                                .unwrap_or_default();

                                let tree = threads::build_thread_tree(cached.clone());

                                let mut flat = Vec::new();
                                threads::flatten_thread_tree(&tree, 0, &mut flat);
                                state.threaded_subjects = flat;

                                // state.read_flags = vec![false; state.subjects.len()];
                                state.read_flags = if state.threaded_mode {
                                    vec![false; state.threaded_subjects.len()]
                                } else {
                                    vec![false; state.subjects.len()]
                                };

                                state.selected_article = 0;
                                state.scroll_offset = 0;
                                // Clear the “new” badge for this group now that we’re opening it
                                state.new_counts[state.selected_group] = 0;
                                state.current_view = View::Articles;

                                continue;
                            }
                            for (id, subj, sender, date, msgid, refs) in &new_subjs {
                                let db = db.clone();
                                let group = group.clone();
                                let id2 = id.clone();
                                let subj2 = subj.clone();
                                let sender2 = sender.clone();
                                let date2 = date.clone();
                                let msgid2 = msgid.clone();
                                let refs2 = refs.clone();

                                tokio::task::spawn_blocking(move || {
                                    let conn = db.blocking_lock();
                                    let _ = db::cache_subject(
                                        &conn,
                                        &group,
                                        &id2,
                                        &subj2,
                                        &sender2,
                                        &date2,
                                        msgid2.as_deref(),
                                        refs2.as_deref(),
                                    );
                                })
                                .await
                                .ok();
                            }

                            if let Some((highest_id, _, _, _, _, _)) = new_subjs.last() {
                                if let Ok(high_id) = highest_id.parse::<u32>() {
                                    let db = db.clone();
                                    let group = group.clone();
                                    tokio::task::spawn_blocking(move || {
                                        let conn = db.blocking_lock();
                                        let _ = db::set_last_seen(&conn, &group, high_id);
                                    })
                                    .await
                                    .ok();
                                }
                            }
                            let unread_flat = tokio::task::spawn_blocking({
                                let db = db.clone();
                                let group = group.clone();
                                move || {
                                    db::load_unread_subjects(&db.blocking_lock(), &group)
                                        .unwrap_or_default()
                                }
                            })
                            .await
                            .unwrap_or_default();

                            if state.threaded_mode {
                                let threadable = tokio::task::spawn_blocking({
                                    let db = db.clone();
                                    let group = group.clone();
                                    move || {
                                        db::load_threadable_articles(&db.blocking_lock(), &group)
                                            .unwrap_or_default()
                                    }
                                })
                                .await
                                .unwrap_or_default();

                                let tree = threads::build_thread_tree(threadable);
                                let mut flat = Vec::new();
                                threads::flatten_thread_tree(&tree, 0, &mut flat);
                                state.threaded_subjects = flat;
                                state.read_flags = vec![false; state.threaded_subjects.len()];
                            } else {
                                state.subjects = unread_flat;
                                state.read_flags = vec![false; state.subjects.len()];
                            }

                            //
                            let threadable = tokio::task::spawn_blocking({
                                let db = db.clone();
                                let group = group.clone();
                                move || {
                                    db::load_threadable_articles(&db.blocking_lock(), &group)
                                        .unwrap_or_default()
                                }
                            })
                            .await
                            .unwrap_or_default();

                            let tree = threads::build_thread_tree(threadable);
                            let mut flat = Vec::new();
                            threads::flatten_thread_tree(&tree, 0, &mut flat);
                            state.threaded_subjects = flat;

                            //
                            //state.read_flags = vec![false; state.subjects.len()];
                            state.selected_article = 0;
                            state.scroll_offset = 0;
                            state.current_view = View::Articles;
                            let count = if state.threaded_mode {
                                state.threaded_subjects.len()
                            } else {
                                state.subjects.len()
                            };
                            state.status = Some(format!("{} Articles loaded", count));
                        }
                        _ => {}
                    },

                    View::Articles => match key.code {
                        KeyCode::Char('q') => {
                            state.current_view = View::Articles;
                            state.current_body = None;
                            state.status = None;
                        }
                        KeyCode::Char('j') => {
                            let list_len = if state.threaded_mode {
                                state.threaded_subjects.len()
                            } else {
                                state.subjects.len()
                            };
                            if state.selected_article + 1 < list_len {
                                state.selected_article += 1;
                                if let Ok(area) = terminal.size() {
                                    let vis = (area.height as usize).saturating_sub(5);
                                    if state.selected_article >= state.scroll_offset + vis {
                                        state.scroll_offset += 1;
                                    }
                                }
                            }
                        }

                        KeyCode::Char('k') => {
                            if state.selected_article > 0 {
                                state.selected_article -= 1;
                                if state.selected_article < state.scroll_offset {
                                    state.scroll_offset = state.selected_article;
                                }
                            }
                        }
                        KeyCode::Char(' ') => {
                            if let Some(flag) = state.read_flags.get_mut(state.selected_article) {
                                *flag = !*flag;
                                let group = state.subscriptions[state.selected_group].clone();
                                let id = if state.threaded_mode {
                                    state.threaded_subjects[state.selected_article].id.clone()
                                } else {
                                    state.subjects[state.selected_article].0.clone()
                                };

                                let db = db.clone();
                                let is_read = *flag;

                                tokio::task::spawn_blocking(move || {
                                    let conn = db.blocking_lock();
                                    if is_read {
                                        let _ = db::mark_as_read(&conn, &group, &id);
                                    } else {
                                        let _ = db::mark_as_unread(&conn, &group, &id);
                                    }
                                })
                                .await
                                .ok();
                            }
                        }
                        KeyCode::Enter => {
                            state.status = Some("Loading body…".to_string());
                            let id = if state.threaded_mode {
                                &state.threaded_subjects[state.selected_article].id
                            } else {
                                &state.subjects[state.selected_article].0
                            };

                            let mut cl = client.lock().await;
                            if let Ok(body) = cl.fetch_article_body(id).await {
                                let links = extract_links(&body);
                                state.current_body = Some(body);
                                state.article_scroll = 0;
                                state.links_in_article = links;
                                state.current_view = View::Article;
                                state.status = None;
                            } else {
                                state.status = Some("Error loading body".to_string());
                            }
                        }
                        KeyCode::Esc => {
                            state.current_view = View::Groups;
                            // meow this to clear article text
                            state.current_body = None;
                            state.status = None;

                            // Refresh the new-article badges now that we're back
                            update_new_counts(
                                &client,
                                &db,
                                &state.subscriptions,
                                &mut state.new_counts,
                            )
                            .await;
                        }

                        KeyCode::Char('n') => {
                            // Popup to get newsgroups & subject
                            let mut newsgroups_input =
                                state.subscriptions[state.selected_group].clone();
                            let mut subject_input = String::new();
                            let mut field = 0; // 0=newsgroups, 1=subject

                            loop {
                                let area = terminal.size()?;
                                let popup = Rect {
                                    x: (area.width.saturating_sub(60)) / 2,
                                    y: (area.height.saturating_sub(7)) / 2,
                                    width: 60,
                                    height: 7,
                                };

                                terminal.draw(|f| {
                                    let block = Block::default()
                                        .borders(Borders::ALL)
                                        .title("Compose New Article");
                                    f.render_widget(block, popup);

                                    let inner = Rect {
                                        x: popup.x + 1,
                                        y: popup.y + 1,
                                        width: popup.width - 2,
                                        height: popup.height - 2,
                                    };

                                    let lines = vec![
                                        Line::from(vec![
                                            Span::raw("Newsgroups: "),
                                            Span::styled(
                                                newsgroups_input.as_str(),
                                                if field == 0 {
                                                    Style::default().fg(Color::Green)
                                                } else {
                                                    Style::default().fg(Color::White)
                                                },
                                            ),
                                        ]),
                                        Line::from(vec![
                                            Span::raw("Subject:  "),
                                            Span::styled(
                                                subject_input.as_str(),
                                                if field == 1 {
                                                    Style::default().fg(Color::Green)
                                                } else {
                                                    Style::default().fg(Color::White)
                                                },
                                            ),
                                        ]),
                                        Line::from(""),
                                        Line::from(
                                            "Tab to switch, Enter to confirm, Esc to cancel",
                                        ),
                                    ];

                                    let paragraph = Paragraph::new(lines).block(Block::default());
                                    f.render_widget(paragraph, inner);
                                })?;

                                if let Event::Key(ev) = event::read()? {
                                    match ev.code {
                                        KeyCode::Char(c) => {
                                            if field == 0 {
                                                newsgroups_input.push(c);
                                            } else {
                                                subject_input.push(c);
                                            }
                                        }
                                        KeyCode::Backspace => {
                                            if field == 0 {
                                                newsgroups_input.pop();
                                            } else {
                                                subject_input.pop();
                                            }
                                        }
                                        KeyCode::Tab => field = (field + 1) % 2,
                                        KeyCode::Enter => break,
                                        KeyCode::Esc => {
                                            subject_input.clear();
                                            break;
                                        }
                                        _ => {}
                                    }
                                }
                            }

                            if subject_input.trim().is_empty() {
                                state.status = Some("🚫 Post cancelled.".into());
                                continue;
                            }

                            let from = format!("{} <{}>", username, email);
                            let mut cl = client.lock().await;
                            match compose_post(
                                &mut cl,
                                newsgroups_input.trim(),
                                subject_input.trim(),
                                from,
                            )
                            .await
                            {
                                Ok(Some(article)) => match cl.post_article_raw(&article).await {
                                    Ok(_) => {
                                        let mut stdout = stdout();
                                        execute!(stdout, Clear(ClearType::All), MoveTo(0, 0)).ok();
                                        terminal.clear()?;
                                        state.status = Some("✅ Article posted!".into());
                                        state.current_view = View::Articles;
                                        state.current_body = None;
                                    }
                                    Err(e) => {
                                        state.status = Some(format!("❌ Failed to post: {}", e));
                                    }
                                },
                                Ok(None) => {
                                    state.status = Some("🚫 Post cancelled.".into());
                                }
                                Err(e) => {
                                    state.status = Some(format!("💥 Editor failed: {}", e));
                                }
                            }
                        }

                        _ => {}
                    },

                    View::Article => {
                        // 🚨 If Insert or Visual mode is active, do not handle normal j/k here
                        if state.visual.insert_active || state.visual.visual_active {
                            // Do nothing — handled by handle_key_event()
                        } else {
                            match key.code {
                                KeyCode::Char('j') => {
                                    if let Some(body) = &state.current_body {
                                        let total_lines = body.lines().count() as u16;
                                        if let Ok(area) = terminal.size() {
                                            let visible_rows = area.height.saturating_sub(8);
                                            let max_scroll =
                                                total_lines.saturating_sub(visible_rows);
                                            state.article_scroll =
                                                (state.article_scroll + 1).min(max_scroll);
                                        }
                                    }
                                }
                                KeyCode::Char('k') => {
                                    state.article_scroll = state.article_scroll.saturating_sub(1);
                                }
                                KeyCode::Char('o') => {
                                    if state.links_in_article.is_empty() {
                                        state.status =
                                            Some("No URLs found in this article.".into());
                                    } else {
                                        state.show_link_menu = true;
                                        state.selected_link = 0;
                                    }
                                }
                                KeyCode::Char('r') => {
                                    // When replying, prefer the real Message-ID header if we have it,
                                    // otherwise fall back to the numeric article ID.
                                    let (id, subject) = if state.threaded_mode {
                                        let art = &state.threaded_subjects[state.selected_article];
                                        let ref_id = art
                                            .message_id
                                            .clone()
                                            .unwrap_or_else(|| art.id.clone());
                                        (ref_id, art.subject.clone())
                                    } else {
                                        let (id, subject, _sender, _) =
                                            &state.subjects[state.selected_article];
                                        (id.clone(), subject.clone())
                                    };
                                    let newsgroup = &state.subscriptions[state.selected_group];
                                    let from = format!("{} <{}>", username, email);
                                    let mut client = client.lock().await;
                                    match compose_reply(&mut client, newsgroup, &id, &subject, from)
                                        .await
                                    {
                                        Ok(Some(article)) => {
                                            match client.post_article(&article, newsgroup).await {
                                                Ok(_) => {
                                                    let mut stdout = stdout();
                                                    execute!(
                                                        stdout,
                                                        Clear(ClearType::All),
                                                        MoveTo(0, 0)
                                                    )
                                                    .ok();
                                                    terminal.clear()?;
                                                    state.status = Some("✅ Reply posted!".into());
                                                    state.current_view = View::Articles;
                                                    state.current_body = None;
                                                }
                                                Err(e) => {
                                                    state.status =
                                                        Some(format!("❌ Failed to post: {}", e));
                                                }
                                            }
                                        }
                                        Ok(None) => {
                                            state.status =
                                                Some("🚫 Reply cancelled or unchanged.".into());
                                        }
                                        Err(e) => {
                                            state.status = Some(format!("💥 Editor failed: {}", e));
                                        }
                                    }
                                }
                                KeyCode::Char('q') | KeyCode::Esc => {
                                    state.current_view = View::Articles;
                                    state.current_body = None;
                                    state.status = None;
                                }
                                _ => {}
                            }
                        }
                    }
                }
            }
        }
    }
}
