/*
 * Decompiled with CFR 0.152.
 */
package com.google.android.apps.common.testing.accessibility.framework.utils.contrast;

import com.google.android.apps.common.testing.accessibility.framework.utils.contrast.ContrastUtils;
import com.google.android.apps.common.testing.accessibility.framework.utils.contrast.Image;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.UnmodifiableIterator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import org.checkerframework.checker.nullness.qual.Nullable;

public class ContrastSwatch {
    public static final int MAX_FOREGROUND_COLOR = 5;
    private static final double COLOR_DIFFERENCE_LIMIT = 2.0;
    private static final double COLOR_SIGNIFICANCE_PERCENTAGE = 0.03;
    private static final double COLOR_CUTOFF_PERCENTAGE = 0.01;
    private final SeparatedColors separatedColors;

    public ContrastSwatch(Image image, boolean multipleForegroundColors) {
        this.separatedColors = ContrastSwatch.processSwatch(image, multipleForegroundColors);
    }

    private static SeparatedColors processSwatch(Image image, boolean multipleForegroundColors) {
        int[] pixels = image.getPixels();
        ColorHistogram colorHistogram = new ColorHistogram(pixels);
        int imageSize = pixels.length;
        return ContrastSwatch.separateColors(colorHistogram, imageSize, multipleForegroundColors);
    }

    private static SeparatedColors separateColors(ColorHistogram colorHistogram, int imageSize, boolean multipleForegroundColors) {
        if (colorHistogram.getColors().isEmpty()) {
            return new SeparatedColors(-16777216);
        }
        if (colorHistogram.getColors().size() == 1) {
            int singleColor = (Integer)colorHistogram.getColors().iterator().next();
            return new SeparatedColors(singleColor);
        }
        double averageLuminance = colorHistogram.calculateAverageLuminance();
        if (multipleForegroundColors) {
            return ContrastSwatch.separateColorsUsingMultipleForegroundMethod(colorHistogram, averageLuminance, imageSize);
        }
        return ContrastSwatch.separateColorsUsingSingleForegroundMethod(colorHistogram, averageLuminance);
    }

    private static SeparatedColors separateColorsUsingMultipleForegroundMethod(ColorHistogram colorHistogram, double averageLuminance, int imageSize) {
        ColorHistogram dominantColorHistogram = ContrastSwatch.reduceColors(colorHistogram, imageSize, 0.01);
        if (dominantColorHistogram.getColors().size() < 2) {
            int singleColor = (Integer)dominantColorHistogram.getColors().iterator().next();
            return new SeparatedColors(singleColor);
        }
        PriorityQueue<Map.Entry<Integer, Integer>> frequencyMaxHeap = new PriorityQueue<Map.Entry<Integer, Integer>>(dominantColorHistogram.getColors().size(), (a, b) -> (Integer)b.getValue() - (Integer)a.getValue());
        for (Map.Entry entry : dominantColorHistogram.entrySet()) {
            frequencyMaxHeap.offer(entry);
        }
        int backgroundColor = (Integer)((Map.Entry)Preconditions.checkNotNull((Object)((Map.Entry)frequencyMaxHeap.poll()))).getKey();
        List<Integer> foregroundColors = ContrastSwatch.extractDominantForegroundColors(backgroundColor, frequencyMaxHeap, averageLuminance, imageSize);
        if (foregroundColors.isEmpty()) {
            return new SeparatedColors(backgroundColor);
        }
        return new SeparatedColors(backgroundColor, foregroundColors);
    }

    private static SeparatedColors separateColorsUsingSingleForegroundMethod(ColorHistogram colorHistogram, double averageLuminance) {
        int foregroundColor;
        int backgroundColor;
        int lowLuminanceColor = -1;
        int highLuminanceColor = -1;
        int maxLowLuminanceFrequency = 0;
        int maxHighLuminanceFrequency = 0;
        for (Map.Entry entry : colorHistogram.entrySet()) {
            int color2 = (Integer)entry.getKey();
            double luminanceValue = ContrastUtils.calculateLuminance(color2);
            int frequency = (Integer)entry.getValue();
            if (luminanceValue < averageLuminance && frequency > maxLowLuminanceFrequency) {
                maxLowLuminanceFrequency = frequency;
                lowLuminanceColor = color2;
                continue;
            }
            if (!(luminanceValue >= averageLuminance) || frequency <= maxHighLuminanceFrequency) continue;
            maxHighLuminanceFrequency = frequency;
            highLuminanceColor = color2;
        }
        if (maxHighLuminanceFrequency > maxLowLuminanceFrequency) {
            backgroundColor = highLuminanceColor;
            foregroundColor = lowLuminanceColor;
        } else {
            backgroundColor = lowLuminanceColor;
            foregroundColor = highLuminanceColor;
        }
        return new SeparatedColors(backgroundColor, foregroundColor);
    }

    private static ColorHistogram reduceColors(ColorHistogram colorHistogram, int imageSize, double cutoff) {
        HashMap<Integer, Integer> dominantColorHistogram = new HashMap<Integer, Integer>();
        ArrayList<Map.Entry> dominantColorsList = new ArrayList<Map.Entry>();
        for (Map.Entry entry : colorHistogram.entrySet()) {
            if (!((double)((Integer)entry.getValue()).intValue() >= (double)imageSize * cutoff)) continue;
            dominantColorsList.add(entry);
        }
        Collections.sort(dominantColorsList, (a, b) -> (Integer)b.getValue() - (Integer)a.getValue());
        for (Map.Entry entry : dominantColorsList) {
            int color2 = (Integer)entry.getKey();
            int colorCount = (Integer)entry.getValue();
            Iterator iterator = dominantColorHistogram.keySet().iterator();
            while (iterator.hasNext()) {
                int dominantColor = (Integer)iterator.next();
                if (!(ContrastUtils.colorDifference(dominantColor, (Integer)entry.getKey()) < 2.0)) continue;
                color2 = dominantColor;
                colorCount += ((Integer)dominantColorHistogram.get(dominantColor)).intValue();
                break;
            }
            dominantColorHistogram.put(color2, colorCount);
        }
        return new ColorHistogram(dominantColorHistogram);
    }

    private static List<Integer> extractDominantForegroundColors(int backgroundColor, PriorityQueue<Map.Entry<Integer, Integer>> frequencyMaxHeap, double averageLuminance, int imageSize) {
        double backgroundLuminance = ContrastUtils.calculateLuminance(backgroundColor);
        boolean backgroundLuminanceBelowAverage = backgroundLuminance < averageLuminance;
        ArrayList<Integer> foregroundColors = new ArrayList<Integer>();
        int priorityIndex = 0;
        while (!frequencyMaxHeap.isEmpty() && foregroundColors.size() < 5) {
            boolean oppositeLuminance;
            Map.Entry<Integer, Integer> entry = frequencyMaxHeap.poll();
            int newColor = (Integer)((Map.Entry)Preconditions.checkNotNull(entry)).getKey();
            double newLuminance = ContrastUtils.calculateLuminance(newColor);
            boolean newLuminanceBelowAverage = newLuminance <= averageLuminance;
            boolean bl = oppositeLuminance = backgroundLuminanceBelowAverage != newLuminanceBelowAverage;
            if (oppositeLuminance) {
                foregroundColors.add(priorityIndex++, newColor);
                continue;
            }
            if (!((double)entry.getValue().intValue() > (double)imageSize * 0.03)) continue;
            foregroundColors.add(newColor);
        }
        return foregroundColors;
    }

    public int getBackgroundColor() {
        return this.separatedColors.getBackgroundColor();
    }

    public ImmutableList<Integer> getForegroundColors() {
        return this.separatedColors.getForegroundColors();
    }

    public double getBackgroundLuminance() {
        return ContrastUtils.calculateLuminance(this.getBackgroundColor());
    }

    public ImmutableList<Double> getForegroundLuminances() {
        ImmutableList.Builder foregroundLuminances = ImmutableList.builder();
        for (Integer color2 : this.getForegroundColors()) {
            foregroundLuminances.add((Object)ContrastUtils.calculateLuminance((Integer)Preconditions.checkNotNull((Object)color2)));
        }
        return foregroundLuminances.build();
    }

    public ImmutableList<Double> getContrastRatios() {
        ImmutableList.Builder contrastRatios = ImmutableList.builder();
        double backgroundLuminance = this.getBackgroundLuminance();
        ImmutableList<Double> foregroundLuminances = this.getForegroundLuminances();
        for (Double element : foregroundLuminances) {
            contrastRatios.add((Object)((double)Math.round(ContrastUtils.calculateContrastRatio(backgroundLuminance, element) * 100.0) / 100.0));
        }
        return contrastRatios.build();
    }

    public String toString() {
        return "{contrast:1:" + this.getContrastRatios().get(0) + ", background:" + ContrastUtils.colorToHexString(this.separatedColors.getBackgroundColor()) + ", foreground:" + ContrastUtils.colorToHexString((Integer)this.separatedColors.getForegroundColors().get(0)) + "}";
    }

    private static class SeparatedColors {
        private final int backgroundColor;
        private final ImmutableList<Integer> foregroundColors;

        SeparatedColors(int singleColor) {
            this(singleColor, singleColor);
        }

        SeparatedColors(int backgroundColor, int foregroundColor) {
            this(backgroundColor, (List<Integer>)ImmutableList.of((Object)foregroundColor));
        }

        SeparatedColors(int backgroundColor, List<Integer> foregroundColors) {
            this.backgroundColor = backgroundColor;
            this.foregroundColors = ImmutableList.copyOf(foregroundColors);
        }

        int getBackgroundColor() {
            return this.backgroundColor;
        }

        ImmutableList<Integer> getForegroundColors() {
            return this.foregroundColors;
        }
    }

    private static class ColorHistogram {
        private final ImmutableMap<Integer, Integer> colorHistogram;

        ColorHistogram(int[] pixels) {
            this.colorHistogram = ColorHistogram.processLuminanceData(pixels);
        }

        private ColorHistogram(Map<Integer, Integer> map2) {
            this.colorHistogram = ImmutableMap.copyOf(map2);
        }

        ImmutableSet<Integer> getColors() {
            return this.colorHistogram.keySet();
        }

        ImmutableSet<Map.Entry<Integer, Integer>> entrySet() {
            return this.colorHistogram.entrySet();
        }

        @Nullable Integer getCount(int color2) {
            return (Integer)this.colorHistogram.get((Object)color2);
        }

        double calculateAverageLuminance() {
            double luminanceSum = 0.0;
            UnmodifiableIterator unmodifiableIterator = this.getColors().iterator();
            while (unmodifiableIterator.hasNext()) {
                int color2 = (Integer)unmodifiableIterator.next();
                luminanceSum += ContrastUtils.calculateLuminance(color2);
            }
            return luminanceSum / (double)this.getColors().size();
        }

        private static ImmutableMap<Integer, Integer> processLuminanceData(int[] pixels) {
            if (pixels.length == 0) {
                return ImmutableMap.of();
            }
            Arrays.sort(pixels);
            ImmutableMap.Builder colorHistogram = ImmutableMap.builder();
            int currentColor = pixels[0];
            int currentColorCount = 1;
            for (int i = 1; i < pixels.length; ++i) {
                int color2 = pixels[i];
                if (color2 == currentColor) {
                    ++currentColorCount;
                    continue;
                }
                colorHistogram.put((Object)currentColor, (Object)currentColorCount);
                currentColor = color2;
                currentColorCount = 1;
            }
            colorHistogram.put((Object)currentColor, (Object)currentColorCount);
            return colorHistogram.buildOrThrow();
        }
    }
}

