/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.hppc;

import java.util.Arrays;
import java.util.IllegalFormatException;
import java.util.Iterator;
import java.util.Locale;
import java.util.NoSuchElementException;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.lucene.util.BitUtil;
import org.apache.lucene.util.hppc.BitMixer;

public class IntIntHashMap
implements Iterable<IntIntCursor>,
Cloneable {
    public static final int DEFAULT_EXPECTED_ELEMENTS = 4;
    public static final float DEFAULT_LOAD_FACTOR = 0.75f;
    private static final AtomicInteger ITERATION_SEED = new AtomicInteger();
    public static final float MIN_LOAD_FACTOR = 0.01f;
    public static final float MAX_LOAD_FACTOR = 0.99f;
    public static final int MIN_HASH_ARRAY_LENGTH = 4;
    public static final int MAX_HASH_ARRAY_LENGTH = 0x40000000;
    public int[] keys;
    public int[] values;
    protected int assigned;
    protected int mask;
    protected int resizeAt;
    protected boolean hasEmptyKey;
    protected double loadFactor;
    protected int iterationSeed;

    public IntIntHashMap() {
        this(4);
    }

    public IntIntHashMap(int expectedElements) {
        this(expectedElements, 0.75);
    }

    public IntIntHashMap(int expectedElements, double loadFactor) {
        this.loadFactor = this.verifyLoadFactor(loadFactor);
        this.iterationSeed = ITERATION_SEED.incrementAndGet();
        this.ensureCapacity(expectedElements);
    }

    public IntIntHashMap(Iterable<? extends IntIntCursor> container) {
        this();
        this.putAll(container);
    }

    public int put(int key, int value) {
        int existing;
        assert (this.assigned < this.mask + 1);
        int mask = this.mask;
        if (key == 0) {
            this.hasEmptyKey = true;
            int previousValue = this.values[mask + 1];
            this.values[mask + 1] = value;
            return previousValue;
        }
        int[] keys = this.keys;
        int slot = this.hashKey(key) & mask;
        while ((existing = keys[slot]) != 0) {
            if (existing == key) {
                int previousValue = this.values[slot];
                this.values[slot] = value;
                return previousValue;
            }
            slot = slot + 1 & mask;
        }
        if (this.assigned == this.resizeAt) {
            this.allocateThenInsertThenRehash(slot, key, value);
        } else {
            keys[slot] = key;
            this.values[slot] = value;
        }
        ++this.assigned;
        return 0;
    }

    public int putAll(Iterable<? extends IntIntCursor> iterable) {
        int count = this.size();
        for (IntIntCursor intIntCursor : iterable) {
            this.put(intIntCursor.key, intIntCursor.value);
        }
        return this.size() - count;
    }

    public boolean putIfAbsent(int key, int value) {
        int keyIndex = this.indexOf(key);
        if (!this.indexExists(keyIndex)) {
            this.indexInsert(keyIndex, key, value);
            return true;
        }
        return false;
    }

    public int putOrAdd(int key, int putValue, int incrementValue) {
        assert (this.assigned < this.mask + 1);
        int keyIndex = this.indexOf(key);
        if (this.indexExists(keyIndex)) {
            putValue = this.values[keyIndex] + incrementValue;
            this.indexReplace(keyIndex, putValue);
        } else {
            this.indexInsert(keyIndex, key, putValue);
        }
        return putValue;
    }

    public int addTo(int key, int incrementValue) {
        return this.putOrAdd(key, incrementValue, incrementValue);
    }

    public int remove(int key) {
        int existing;
        int mask = this.mask;
        if (key == 0) {
            this.hasEmptyKey = false;
            int previousValue = this.values[mask + 1];
            this.values[mask + 1] = 0;
            return previousValue;
        }
        int[] keys = this.keys;
        int slot = this.hashKey(key) & mask;
        while ((existing = keys[slot]) != 0) {
            if (existing == key) {
                int previousValue = this.values[slot];
                this.shiftConflictingKeys(slot);
                return previousValue;
            }
            slot = slot + 1 & mask;
        }
        return 0;
    }

    public int get(int key) {
        int existing;
        if (key == 0) {
            return this.hasEmptyKey ? this.values[this.mask + 1] : 0;
        }
        int[] keys = this.keys;
        int mask = this.mask;
        int slot = this.hashKey(key) & mask;
        while ((existing = keys[slot]) != 0) {
            if (existing == key) {
                return this.values[slot];
            }
            slot = slot + 1 & mask;
        }
        return 0;
    }

    public int getOrDefault(int key, int defaultValue) {
        int existing;
        if (key == 0) {
            return this.hasEmptyKey ? this.values[this.mask + 1] : defaultValue;
        }
        int[] keys = this.keys;
        int mask = this.mask;
        int slot = this.hashKey(key) & mask;
        while ((existing = keys[slot]) != 0) {
            if (existing == key) {
                return this.values[slot];
            }
            slot = slot + 1 & mask;
        }
        return defaultValue;
    }

    public boolean containsKey(int key) {
        int existing;
        if (key == 0) {
            return this.hasEmptyKey;
        }
        int[] keys = this.keys;
        int mask = this.mask;
        int slot = this.hashKey(key) & mask;
        while ((existing = keys[slot]) != 0) {
            if (existing == key) {
                return true;
            }
            slot = slot + 1 & mask;
        }
        return false;
    }

    public int indexOf(int key) {
        int existing;
        int mask = this.mask;
        if (key == 0) {
            return this.hasEmptyKey ? mask + 1 : ~(mask + 1);
        }
        int[] keys = this.keys;
        int slot = this.hashKey(key) & mask;
        while ((existing = keys[slot]) != 0) {
            if (existing == key) {
                return slot;
            }
            slot = slot + 1 & mask;
        }
        return ~slot;
    }

    public boolean indexExists(int index) {
        assert (index < 0 || index >= 0 && index <= this.mask || index == this.mask + 1 && this.hasEmptyKey);
        return index >= 0;
    }

    public int indexGet(int index) {
        assert (index >= 0) : "The index must point at an existing key.";
        assert (index <= this.mask || index == this.mask + 1 && this.hasEmptyKey);
        return this.values[index];
    }

    public int indexReplace(int index, int newValue) {
        assert (index >= 0) : "The index must point at an existing key.";
        assert (index <= this.mask || index == this.mask + 1 && this.hasEmptyKey);
        int previousValue = this.values[index];
        this.values[index] = newValue;
        return previousValue;
    }

    public void indexInsert(int index, int key, int value) {
        assert (index < 0) : "The index must not point at an existing key.";
        index ^= 0xFFFFFFFF;
        if (key == 0) {
            assert (index == this.mask + 1);
            this.values[index] = value;
            this.hasEmptyKey = true;
        } else {
            assert (this.keys[index] == 0);
            if (this.assigned == this.resizeAt) {
                this.allocateThenInsertThenRehash(index, key, value);
            } else {
                this.keys[index] = key;
                this.values[index] = value;
            }
            ++this.assigned;
        }
    }

    public int indexRemove(int index) {
        assert (index >= 0) : "The index must point at an existing key.";
        assert (index <= this.mask || index == this.mask + 1 && this.hasEmptyKey);
        int previousValue = this.values[index];
        if (index > this.mask) {
            this.hasEmptyKey = false;
            this.values[index] = 0;
        } else {
            this.shiftConflictingKeys(index);
        }
        return previousValue;
    }

    public void clear() {
        this.assigned = 0;
        this.hasEmptyKey = false;
        Arrays.fill(this.keys, 0);
    }

    public void release() {
        this.assigned = 0;
        this.hasEmptyKey = false;
        this.keys = null;
        this.values = null;
        this.ensureCapacity(4);
    }

    public int size() {
        return this.assigned + (this.hasEmptyKey ? 1 : 0);
    }

    public boolean isEmpty() {
        return this.size() == 0;
    }

    public int hashCode() {
        int h = this.hasEmptyKey ? -559038737 : 0;
        for (IntIntCursor c : this) {
            h += BitMixer.mix(c.key) + BitMixer.mix(c.value);
        }
        return h;
    }

    public boolean equals(Object obj) {
        return obj != null && this.getClass() == obj.getClass() && this.equalElements((IntIntHashMap)this.getClass().cast(obj));
    }

    protected boolean equalElements(IntIntHashMap other) {
        if (other.size() != this.size()) {
            return false;
        }
        for (IntIntCursor c : other) {
            int key = c.key;
            if (this.containsKey(key) && this.get(key) == c.value) continue;
            return false;
        }
        return true;
    }

    public void ensureCapacity(int expectedElements) {
        if (expectedElements > this.resizeAt || this.keys == null) {
            int[] prevKeys = this.keys;
            int[] prevValues = this.values;
            this.allocateBuffers(IntIntHashMap.minBufferSize(expectedElements, this.loadFactor));
            if (prevKeys != null && !this.isEmpty()) {
                this.rehash(prevKeys, prevValues);
            }
        }
    }

    protected int nextIterationSeed() {
        this.iterationSeed = BitMixer.mixPhi(this.iterationSeed);
        return this.iterationSeed;
    }

    @Override
    public Iterator<IntIntCursor> iterator() {
        return new EntryIterator();
    }

    public KeysContainer keys() {
        return new KeysContainer();
    }

    public IntContainer values() {
        return new ValuesContainer();
    }

    public IntIntHashMap clone() {
        try {
            IntIntHashMap cloned = (IntIntHashMap)super.clone();
            cloned.keys = (int[])this.keys.clone();
            cloned.values = (int[])this.values.clone();
            cloned.hasEmptyKey = this.hasEmptyKey;
            cloned.iterationSeed = this.nextIterationSeed();
            return cloned;
        }
        catch (CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }

    public String toString() {
        StringBuilder buffer = new StringBuilder();
        buffer.append("[");
        boolean first = true;
        for (IntIntCursor cursor : this) {
            if (!first) {
                buffer.append(", ");
            }
            buffer.append(cursor.key);
            buffer.append("=>");
            buffer.append(cursor.value);
            first = false;
        }
        buffer.append("]");
        return buffer.toString();
    }

    public static IntIntHashMap from(int[] keys, int[] values) {
        if (keys.length != values.length) {
            throw new IllegalArgumentException("Arrays of keys and values must have an identical length.");
        }
        IntIntHashMap map2 = new IntIntHashMap(keys.length);
        for (int i = 0; i < keys.length; ++i) {
            map2.put(keys[i], values[i]);
        }
        return map2;
    }

    protected int hashKey(int key) {
        assert (key != 0);
        return BitMixer.mixPhi(key);
    }

    protected double verifyLoadFactor(double loadFactor) {
        IntIntHashMap.checkLoadFactor(loadFactor, 0.01f, 0.99f);
        return loadFactor;
    }

    protected void rehash(int[] fromKeys, int[] fromValues) {
        assert (fromKeys.length == fromValues.length && IntIntHashMap.checkPowerOfTwo(fromKeys.length - 1));
        int[] keys = this.keys;
        int[] values = this.values;
        int mask = this.mask;
        int from = fromKeys.length - 1;
        keys[keys.length - 1] = fromKeys[from];
        values[values.length - 1] = fromValues[from];
        while (--from >= 0) {
            int existing = fromKeys[from];
            if (existing == 0) continue;
            int slot = this.hashKey(existing) & mask;
            while (keys[slot] != 0) {
                slot = slot + 1 & mask;
            }
            keys[slot] = existing;
            values[slot] = fromValues[from];
        }
    }

    protected void allocateBuffers(int arraySize) {
        assert (Integer.bitCount(arraySize) == 1);
        int[] prevKeys = this.keys;
        int[] prevValues = this.values;
        try {
            int emptyElementSlot = 1;
            this.keys = new int[arraySize + emptyElementSlot];
            this.values = new int[arraySize + emptyElementSlot];
        }
        catch (OutOfMemoryError e) {
            this.keys = prevKeys;
            this.values = prevValues;
            throw new BufferAllocationException("Not enough memory to allocate buffers for rehashing: %,d -> %,d", (Throwable)e, this.mask + 1, arraySize);
        }
        this.resizeAt = IntIntHashMap.expandAtCount(arraySize, this.loadFactor);
        this.mask = arraySize - 1;
    }

    protected void allocateThenInsertThenRehash(int slot, int pendingKey, int pendingValue) {
        assert (this.assigned == this.resizeAt && this.keys[slot] == 0 && pendingKey != 0);
        int[] prevKeys = this.keys;
        int[] prevValues = this.values;
        this.allocateBuffers(IntIntHashMap.nextBufferSize(this.mask + 1, this.size(), this.loadFactor));
        assert (this.keys.length > prevKeys.length);
        prevKeys[slot] = pendingKey;
        prevValues[slot] = pendingValue;
        this.rehash(prevKeys, prevValues);
    }

    static int nextBufferSize(int arraySize, int elements, double loadFactor) {
        assert (IntIntHashMap.checkPowerOfTwo(arraySize));
        if (arraySize == 0x40000000) {
            throw new BufferAllocationException("Maximum array size exceeded for this load factor (elements: %d, load factor: %f)", elements, loadFactor);
        }
        return arraySize << 1;
    }

    static int expandAtCount(int arraySize, double loadFactor) {
        assert (IntIntHashMap.checkPowerOfTwo(arraySize));
        return Math.min(arraySize - 1, (int)Math.ceil((double)arraySize * loadFactor));
    }

    static boolean checkPowerOfTwo(int arraySize) {
        assert (arraySize > 1);
        assert (BitUtil.nextHighestPowerOfTwo(arraySize) == arraySize);
        return true;
    }

    static int minBufferSize(int elements, double loadFactor) {
        if (elements < 0) {
            throw new IllegalArgumentException("Number of elements must be >= 0: " + elements);
        }
        long length = (long)Math.ceil((double)elements / loadFactor);
        if (length == (long)elements) {
            ++length;
        }
        if ((length = Math.max(4L, BitUtil.nextHighestPowerOfTwo(length))) > 0x40000000L) {
            throw new BufferAllocationException("Maximum array size exceeded for this load factor (elements: %d, load factor: %f)", elements, loadFactor);
        }
        return (int)length;
    }

    static void checkLoadFactor(double loadFactor, double minAllowedInclusive, double maxAllowedInclusive) {
        if (loadFactor < minAllowedInclusive || loadFactor > maxAllowedInclusive) {
            throw new BufferAllocationException("The load factor should be in range [%.2f, %.2f]: %f", minAllowedInclusive, maxAllowedInclusive, loadFactor);
        }
    }

    static int iterationIncrement(int seed) {
        return 29 + ((seed & 7) << 1);
    }

    protected void shiftConflictingKeys(int gapSlot) {
        int slot;
        int existing;
        int[] keys = this.keys;
        int[] values = this.values;
        int mask = this.mask;
        int distance = 0;
        while ((existing = keys[slot = gapSlot + ++distance & mask]) != 0) {
            int idealSlot = this.hashKey(existing);
            int shift = slot - idealSlot & mask;
            if (shift < distance) continue;
            keys[gapSlot] = existing;
            values[gapSlot] = values[slot];
            gapSlot = slot;
            distance = 0;
        }
        keys[gapSlot] = 0;
        values[gapSlot] = 0;
        --this.assigned;
    }

    public static class BufferAllocationException
    extends RuntimeException {
        public BufferAllocationException(String message) {
            super(message);
        }

        public BufferAllocationException(String message, Object ... args) {
            this(message, null, args);
        }

        public BufferAllocationException(String message, Throwable t, Object ... args) {
            super(BufferAllocationException.formatMessage(message, t, args), t);
        }

        private static String formatMessage(String message, Throwable t, Object ... args) {
            try {
                return String.format(Locale.ROOT, message, args);
            }
            catch (IllegalFormatException e) {
                BufferAllocationException substitute = new BufferAllocationException(message + " [ILLEGAL FORMAT, ARGS SUPPRESSED]");
                if (t != null) {
                    substitute.addSuppressed(t);
                }
                substitute.addSuppressed(e);
                throw substitute;
            }
        }
    }

    public final class IntCursor {
        public int index;
        public int value;

        public String toString() {
            return "[cursor, index: " + this.index + ", value: " + this.value + "]";
        }
    }

    public final class IntIntCursor {
        public int index;
        public int key;
        public int value;

        public String toString() {
            return "[cursor, index: " + this.index + ", key: " + this.key + ", value: " + this.value + "]";
        }
    }

    public static abstract class AbstractIterator<E>
    implements Iterator<E> {
        private static final int NOT_CACHED = 0;
        private static final int CACHED = 1;
        private static final int AT_END = 2;
        private int state = 0;
        private E nextElement;

        @Override
        public boolean hasNext() {
            if (this.state == 0) {
                this.state = 1;
                this.nextElement = this.fetch();
            }
            return this.state == 1;
        }

        @Override
        public E next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            this.state = 0;
            return this.nextElement;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }

        protected abstract E fetch();

        protected final E done() {
            this.state = 2;
            return null;
        }
    }

    private final class ValuesIterator
    extends AbstractIterator<IntCursor> {
        private final IntCursor cursor;
        private final int increment;
        private int index;
        private int slot;

        public ValuesIterator() {
            this.cursor = new IntCursor();
            int seed = IntIntHashMap.this.nextIterationSeed();
            this.increment = IntIntHashMap.iterationIncrement(seed);
            this.slot = seed & IntIntHashMap.this.mask;
        }

        @Override
        protected IntCursor fetch() {
            int mask = IntIntHashMap.this.mask;
            while (this.index <= mask) {
                ++this.index;
                this.slot = this.slot + this.increment & mask;
                if (IntIntHashMap.this.keys[this.slot] == 0) continue;
                this.cursor.index = this.slot;
                this.cursor.value = IntIntHashMap.this.values[this.slot];
                return this.cursor;
            }
            if (this.index == mask + 1 && IntIntHashMap.this.hasEmptyKey) {
                this.cursor.index = this.index;
                this.cursor.value = IntIntHashMap.this.values[this.index++];
                return this.cursor;
            }
            return (IntCursor)this.done();
        }
    }

    public abstract class IntContainer
    implements Iterable<IntCursor> {
        public int size() {
            return IntIntHashMap.this.size();
        }

        public int[] toArray() {
            int[] array = new int[this.size()];
            int i = 0;
            for (IntCursor cursor : this) {
                array[i++] = cursor.value;
            }
            return array;
        }
    }

    private final class ValuesContainer
    extends IntContainer {
        private ValuesContainer() {
        }

        @Override
        public Iterator<IntCursor> iterator() {
            return new ValuesIterator();
        }
    }

    private final class KeysIterator
    extends AbstractIterator<IntCursor> {
        private final IntCursor cursor;
        private final int increment;
        private int index;
        private int slot;

        public KeysIterator() {
            this.cursor = new IntCursor();
            int seed = IntIntHashMap.this.nextIterationSeed();
            this.increment = IntIntHashMap.iterationIncrement(seed);
            this.slot = seed & IntIntHashMap.this.mask;
        }

        @Override
        protected IntCursor fetch() {
            int mask = IntIntHashMap.this.mask;
            while (this.index <= mask) {
                ++this.index;
                this.slot = this.slot + this.increment & mask;
                int existing = IntIntHashMap.this.keys[this.slot];
                if (existing == 0) continue;
                this.cursor.index = this.slot;
                this.cursor.value = existing;
                return this.cursor;
            }
            if (this.index == mask + 1 && IntIntHashMap.this.hasEmptyKey) {
                this.cursor.index = this.index++;
                this.cursor.value = 0;
                return this.cursor;
            }
            return (IntCursor)this.done();
        }
    }

    public final class KeysContainer
    extends IntContainer {
        @Override
        public Iterator<IntCursor> iterator() {
            return new KeysIterator();
        }
    }

    private final class EntryIterator
    extends AbstractIterator<IntIntCursor> {
        private final IntIntCursor cursor;
        private final int increment;
        private int index;
        private int slot;

        public EntryIterator() {
            this.cursor = new IntIntCursor();
            int seed = IntIntHashMap.this.nextIterationSeed();
            this.increment = IntIntHashMap.iterationIncrement(seed);
            this.slot = seed & IntIntHashMap.this.mask;
        }

        @Override
        protected IntIntCursor fetch() {
            int mask = IntIntHashMap.this.mask;
            while (this.index <= mask) {
                ++this.index;
                this.slot = this.slot + this.increment & mask;
                int existing = IntIntHashMap.this.keys[this.slot];
                if (existing == 0) continue;
                this.cursor.index = this.slot;
                this.cursor.key = existing;
                this.cursor.value = IntIntHashMap.this.values[this.slot];
                return this.cursor;
            }
            if (this.index == mask + 1 && IntIntHashMap.this.hasEmptyKey) {
                this.cursor.index = this.index;
                this.cursor.key = 0;
                this.cursor.value = IntIntHashMap.this.values[this.index++];
                return this.cursor;
            }
            return (IntIntCursor)this.done();
        }
    }
}

