/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import java.io.IOException;
import java.nio.ByteOrder;
import java.util.Map;
import java.util.TreeMap;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.io.sstable.Downsampling;
import org.apache.cassandra.io.sstable.IndexSummary;
import org.apache.cassandra.io.util.Memory;
import org.apache.cassandra.io.util.SafeMemoryWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexSummaryBuilder
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(IndexSummaryBuilder.class);
    static final String defaultExpectedKeySizeName = "cassandra.index_summary_expected_key_size";
    static long defaultExpectedKeySize = Long.valueOf(System.getProperty("cassandra.index_summary_expected_key_size", "64"));
    private final SafeMemoryWriter offsets;
    private final SafeMemoryWriter entries;
    private final int minIndexInterval;
    private final int samplingLevel;
    private final int[] startPoints;
    private long keysWritten = 0L;
    private long indexIntervalMatches = 0L;
    private long nextSamplePosition;
    private TreeMap<Long, ReadableBoundary> lastReadableByData = new TreeMap();
    private TreeMap<Long, ReadableBoundary> lastReadableByIndex = new TreeMap();
    private long dataSyncPosition;
    private long indexSyncPosition;
    private ReadableBoundary lastReadableBoundary;

    public IndexSummaryBuilder(long expectedKeys, int minIndexInterval, int samplingLevel) {
        this.samplingLevel = samplingLevel;
        this.startPoints = Downsampling.getStartPoints(128, samplingLevel);
        long expectedEntrySize = IndexSummaryBuilder.getEntrySize(defaultExpectedKeySize);
        long maxExpectedEntries = expectedKeys / (long)minIndexInterval;
        long maxExpectedEntriesSize = maxExpectedEntries * expectedEntrySize;
        if (maxExpectedEntriesSize > Integer.MAX_VALUE) {
            int effectiveMinInterval = (int)Math.ceil((double)(expectedKeys * expectedEntrySize) / 2.147483647E9);
            maxExpectedEntries = expectedKeys / (long)effectiveMinInterval;
            maxExpectedEntriesSize = maxExpectedEntries * expectedEntrySize;
            assert (maxExpectedEntriesSize <= Integer.MAX_VALUE) : maxExpectedEntriesSize;
            logger.warn("min_index_interval of {} is too low for {} expected keys of avg size {}; using interval of {} instead", new Object[]{minIndexInterval, expectedKeys, defaultExpectedKeySize, effectiveMinInterval});
            this.minIndexInterval = effectiveMinInterval;
        } else {
            this.minIndexInterval = minIndexInterval;
        }
        maxExpectedEntries = Math.max(1L, maxExpectedEntries * (long)samplingLevel / 128L);
        this.offsets = new SafeMemoryWriter(4L * maxExpectedEntries).order(ByteOrder.nativeOrder());
        this.entries = new SafeMemoryWriter(expectedEntrySize * maxExpectedEntries).order(ByteOrder.nativeOrder());
        this.nextSamplePosition = 0L;
        ++this.indexIntervalMatches;
    }

    private static long getEntrySize(DecoratedKey key) {
        return IndexSummaryBuilder.getEntrySize(key.getKey().remaining());
    }

    private static long getEntrySize(long keySize) {
        return keySize + (long)TypeSizes.sizeof(0L);
    }

    public void markIndexSynced(long upToPosition) {
        this.indexSyncPosition = upToPosition;
        this.refreshReadableBoundary();
    }

    public void markDataSynced(long upToPosition) {
        this.dataSyncPosition = upToPosition;
        this.refreshReadableBoundary();
    }

    private void refreshReadableBoundary() {
        Map.Entry<Long, ReadableBoundary> byData = this.lastReadableByData.floorEntry(this.dataSyncPosition);
        Map.Entry<Long, ReadableBoundary> byIndex = this.lastReadableByIndex.floorEntry(this.indexSyncPosition);
        if (byData == null || byIndex == null) {
            return;
        }
        this.lastReadableBoundary = byIndex.getValue().indexLength < byData.getValue().indexLength ? byIndex.getValue() : byData.getValue();
        this.lastReadableByData.headMap(this.lastReadableBoundary.dataLength, false).clear();
        this.lastReadableByIndex.headMap(this.lastReadableBoundary.indexLength, false).clear();
    }

    public ReadableBoundary getLastReadableBoundary() {
        return this.lastReadableBoundary;
    }

    public IndexSummaryBuilder maybeAddEntry(DecoratedKey decoratedKey, long indexStart) throws IOException {
        return this.maybeAddEntry(decoratedKey, indexStart, 0L, 0L);
    }

    public IndexSummaryBuilder maybeAddEntry(DecoratedKey decoratedKey, long indexStart, long indexEnd, long dataEnd) throws IOException {
        if (this.keysWritten == this.nextSamplePosition) {
            if (this.entries.length() + IndexSummaryBuilder.getEntrySize(decoratedKey) <= Integer.MAX_VALUE) {
                this.offsets.writeInt((int)this.entries.length());
                this.entries.write(decoratedKey.getKey());
                this.entries.writeLong(indexStart);
                this.setNextSamplePosition(this.keysWritten);
            } else {
                logger.error("Memory capacity of index summary exceeded (2GB), index summary will not cover full sstable, you should increase min_sampling_level");
            }
        } else if (dataEnd != 0L && this.keysWritten + 1L == this.nextSamplePosition) {
            ReadableBoundary boundary = new ReadableBoundary(decoratedKey, indexEnd, dataEnd, (int)(this.offsets.length() / 4L), this.entries.length());
            this.lastReadableByData.put(dataEnd, boundary);
            this.lastReadableByIndex.put(indexEnd, boundary);
        }
        ++this.keysWritten;
        return this;
    }

    private void setNextSamplePosition(long position) {
        block0: while (true) {
            position += (long)this.minIndexInterval;
            ++this.indexIntervalMatches;
            for (int start : this.startPoints) {
                long test;
                if ((test - (long)start) % 128L == 0L) continue block0;
            }
            break;
        }
        this.nextSamplePosition = position;
    }

    public void prepareToCommit() {
        this.entries.setCapacity(this.entries.length());
        this.offsets.setCapacity(this.offsets.length());
    }

    public IndexSummary build(IPartitioner partitioner) {
        return this.build(partitioner, null);
    }

    public IndexSummary build(IPartitioner partitioner, ReadableBoundary boundary) {
        assert (this.entries.length() > 0L);
        int count = (int)(this.offsets.length() / 4L);
        long entriesLength = this.entries.length();
        if (boundary != null) {
            count = boundary.summaryCount;
            entriesLength = boundary.entriesLength;
        }
        int sizeAtFullSampling = (int)Math.ceil((double)this.keysWritten / (double)this.minIndexInterval);
        assert (count > 0);
        return new IndexSummary(partitioner, this.offsets.currentBuffer().sharedCopy(), count, this.entries.currentBuffer().sharedCopy(), entriesLength, sizeAtFullSampling, this.minIndexInterval, this.samplingLevel);
    }

    @Override
    public void close() {
        this.entries.close();
        this.offsets.close();
    }

    public Throwable close(Throwable accumulate) {
        accumulate = this.entries.close(accumulate);
        accumulate = this.offsets.close(accumulate);
        return accumulate;
    }

    static int entriesAtSamplingLevel(int samplingLevel, int maxSummarySize) {
        return (int)Math.ceil((double)(samplingLevel * maxSummarySize) / 128.0);
    }

    static int calculateSamplingLevel(int currentSamplingLevel, int currentNumEntries, long targetNumEntries, int minIndexInterval, int maxIndexInterval) {
        int effectiveMinSamplingLevel = Math.max(1, (int)Math.ceil((double)(128 * minIndexInterval) / (double)maxIndexInterval));
        int newSamplingLevel = (int)(targetNumEntries * (long)currentSamplingLevel) / currentNumEntries;
        return Math.min(128, Math.max(effectiveMinSamplingLevel, newSamplingLevel));
    }

    public static IndexSummary downsample(IndexSummary existing, int newSamplingLevel, int minIndexInterval, IPartitioner partitioner) {
        int currentSamplingLevel = existing.getSamplingLevel();
        assert (currentSamplingLevel > newSamplingLevel);
        assert (minIndexInterval == existing.getMinIndexInterval());
        int[] startPoints = Downsampling.getStartPoints(currentSamplingLevel, newSamplingLevel);
        int newKeyCount = existing.size();
        long newEntriesLength = existing.getEntriesLength();
        int[] nArray = startPoints;
        int n = nArray.length;
        for (int i = 0; i < n; ++i) {
            int start;
            for (int j = start = nArray[i]; j < existing.size(); j += currentSamplingLevel) {
                --newKeyCount;
                long length = existing.getEndInSummary(j) - (long)existing.getPositionInSummary(j);
                newEntriesLength -= length;
            }
        }
        Memory oldEntries = existing.getEntries();
        Memory newOffsets = Memory.allocate(newKeyCount * 4);
        Memory newEntries = Memory.allocate(newEntriesLength);
        int i = 0;
        int newEntriesOffset = 0;
        block2: for (int oldSummaryIndex = 0; oldSummaryIndex < existing.size(); ++oldSummaryIndex) {
            for (int start : startPoints) {
                if ((oldSummaryIndex - start) % currentSamplingLevel == 0) continue block2;
            }
            newOffsets.setInt(i * 4, newEntriesOffset);
            ++i;
            long start = existing.getPositionInSummary(oldSummaryIndex);
            long length = existing.getEndInSummary(oldSummaryIndex) - start;
            newEntries.put(newEntriesOffset, oldEntries, start, length);
            newEntriesOffset = (int)((long)newEntriesOffset + length);
        }
        assert ((long)newEntriesOffset == newEntriesLength);
        return new IndexSummary(partitioner, newOffsets, newKeyCount, newEntries, newEntriesLength, existing.getMaxNumberOfEntries(), minIndexInterval, newSamplingLevel);
    }

    public static class ReadableBoundary {
        public final DecoratedKey lastKey;
        public final long indexLength;
        public final long dataLength;
        public final int summaryCount;
        public final long entriesLength;

        public ReadableBoundary(DecoratedKey lastKey, long indexLength, long dataLength, int summaryCount, long entriesLength) {
            this.lastKey = lastKey;
            this.indexLength = indexLength;
            this.dataLength = dataLength;
            this.summaryCount = summaryCount;
            this.entriesLength = entriesLength;
        }
    }
}

