# MantisBT PGP Keys
 
This file contains the [PGP](https://en.wikipedia.org/wiki/Pretty_Good_Privacy)
keys of MantisBT Developers, which were used to sign official release packages 
and the corresponding tags in the Git repository. 

Do not use these keys for email or other purposes without asking their owner
first, as some may be used for code signing only.

Please note that this file as well public key servers may be compromised.
You are encouraged to confirm the keys' authenticity with their respective
owners.


## Usage

### For End-Users

Add the keys to your keyring
```
gpg --import KEYS.md
```

After importing the keys, you may want to review them and eventually decide to
sign and trust them.
```
gpg --sign-key KEY
```

Check a release's integrity

```
gpg --verify $SIGNATURE $RELEASE_FILE
```

Where _$SIGNATURE_ is the ASCII-armored signature file you downloaded (e.g.
`mantisbt-2.24.0.tar.gz.asc`) and _$RELEASE_FILE_ is the corresponding release's 
zip or tarball (e.g. `mantisbt-2.24.0.tar.gz`).

The above command's output should be similar to the following example, the 
**Good signature** mention confirming that the downloaded file has not been 
tampered with.

```
gpg: Signature made 2020-03-15T20:15:09 CET
gpg:                using RSA key 5769AA4978E571A7BADB2A4DD4EAE2390A45E2D6
gpg: Good signature from "keybase.io/vboctor <vboctor@keybase.io>" [full]
```

### For Developers

To append your Public Key to this file, follow the instructions below.

1. Edit this file
2. Go to the end and add a level 3 header (`###`) with your **username** on the
[mantisbt.org bug tracker](https://mantisbt.org/bugs) 
3. Execute the following command
    ```
    (echo '```' && 
         gpg --fingerprint $YOUR_KEY && 
         gpg --export --armor $YOUR_KEY &&
         echo '```'
    ) >> KEYS.md
    ```

Please also submit your key on a **public keyserver**, preferably one that is part 
of the [SKS pool](https://sks-keyservers.net/), such as:
- http://keys.gnupg.net/
- https://keyserver.ubuntu.com/
- https://pgp.mit.edu/


## Keys

### vboctor
```
pub   rsa4096 2015-06-01 [SCEA]
      8079 96A5 AAFD C2F3 466D  D672 9839 9BB0 8EBC 1D10
uid           [  full  ] keybase.io/vboctor <vboctor@keybase.io>
sub   rsa2048 2015-06-01 [S] [expires: 2023-05-30]
sub   rsa2048 2015-06-01 [E] [expires: 2023-05-30]

-----BEGIN PGP PUBLIC KEY BLOCK-----
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=9aN+
-----END PGP PUBLIC KEY BLOCK-----
```

```
pub   rsa2048 2014-12-08 [SC]
      4EA1 D867 7022 073E A38D  3914 0223 93FF F8CD 7E90
uid           [  full  ] Victor Boctor <vboctor@gmail.com>
sub   rsa2048 2014-12-08 [E]

-----BEGIN PGP PUBLIC KEY BLOCK-----
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=2jbn
-----END PGP PUBLIC KEY BLOCK-----
```

### dregad
```
pub   rsa2048 2010-11-20 [SC]
      EAA6 3F1F 284C A8F5 1537  4907 0FFE 77FB 7E08 105F
uid           [ultimate] Damien Regad (MantisBT dev) <dregad@mantisbt.org>
uid           [ultimate] [jpeg image of size 7697]
sub   rsa2048 2010-11-20 [E]

-----BEGIN PGP PUBLIC KEY BLOCK-----
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=iLrs
-----END PGP PUBLIC KEY BLOCK-----
```

### jreese
```
pub   dsa1024 2008-02-23 [SC]
      1D5C CEEA 92DF 2E3F 7267  BA40 D53E A311 DE61 84DC
uid           [  full  ] John Reese <jreese@leetcode.net>
uid           [  full  ] John Reese (LeetCode.net) <jreese@leetcode.net>
sub   elg2048 2008-02-23 [E]

-----BEGIN PGP PUBLIC KEY BLOCK-----
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=oym2
-----END PGP PUBLIC KEY BLOCK-----
```

```
pub   rsa4096 2011-04-05 [SC]
      9A24 B14A 6239 AA5A 4507  08B6 71FC FA26 C45D 960E
uid           [  full  ] John Reese <john@noswap.com>
sub   rsa4096 2011-04-05 [E]

-----BEGIN PGP PUBLIC KEY BLOCK-----
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=jTOG
-----END PGP PUBLIC KEY BLOCK-----
```

### dhx
```
pub   rsa4096 2011-03-12 [SC]
      2442 14B5 2E51 CB0F CA3B  9DB8 59E0 E7B7 728F 3435
uid           [  full  ] David Hicks <d@hx.id.au>
uid           [  full  ] David Hicks <david@hx.id.au>
uid           [  full  ] David Hicks <david@hicks.id.au>
sub   rsa4096 2011-03-12 [E]

-----BEGIN PGP PUBLIC KEY BLOCK-----
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=aw2m
-----END PGP PUBLIC KEY BLOCK-----
```
